#ifndef __TUNER_MT2063_H
#define __TUNER_MT2063_H

/**

@file

@brief   MT2063 tuner module declaration

One can manipulate MT2063 tuner through MT2063 module.
MT2063 module is derived from tunerd module.



@par Example:
@code

// The example is the same as the tuner example in tuner_base.h except the listed lines.



#include "tuner_mt2063.h"


...



int main(void)
{
	TUNER_MODULE        *pTuner;
	MT2063_EXTRA_MODULE *pTunerExtra;

	TUNER_MODULE          TunerModuleMemory;
	BASE_INTERFACE_MODULE BaseInterfaceModuleMemory;
	I2C_BRIDGE_MODULE     I2cBridgeModuleMemory;

	unsigned long IfFreqHz;


	...



	// Build MT2063 tuner module.
	BuildMt2063Module(
		&pTuner,
		&TunerModuleMemory,
		&BaseInterfaceModuleMemory,
		&I2cBridgeModuleMemory,
		0xc0								// I2C device address is 0xc0 in 8-bit format.
		);





	// Get MT2063 tuner extra module.
	pTunerExtra = &(pTuner->pExtra.Mt2063);

	// Open MT2063 handle.
	pTunerExtra->OpenHandle(pTuner);





	// ==== Initialize tuner and set its parameters =====

	...

	// Set MT2063 IF frequency.
	pTunerExtra->SetIfFreqHz(pTuner, IF_FREQ_36125000HZ);





	// ==== Get tuner information =====

	...

	// Get MT2063 IF frequency.
	pTunerExtra->SetIfFreqHz(pTuner, &IfFreqHz);





	// Close MT2063 handle.
	pTunerExtra->CloseHandle(pTuner);



	// See the example for other tuner functions in tuner_base.h


	return 0;
}


@endcode

*/





// The following context is source code provided by Microtune.





// Microtune source code - mt_errordef.h



/*****************************************************************************
**
**  Name: mt_errordef.h
**
**  Copyright 2004-2007 Microtune, Inc. All Rights Reserved
**
**  This source code file contains confidential information and/or trade
**  secrets of Microtune, Inc. or its affiliates and is subject to the
**  terms of your confidentiality agreement with Microtune, Inc. or one of
**  its affiliates, as applicable.
**
*****************************************************************************/

/*****************************************************************************
**
**  Name: mt_errordef.h
**
**  Description:    Definition of bits in status/error word used by various
**                  MicroTuner control programs.
**
**  References:     None
**
**  Exports:        None
**
**  CVS ID:         $Id: mt_errordef.h,v 1.2 2008/11/05 13:46:19 software Exp $
**  CVS Source:     $Source: /export/vol0/cvsroot/software/tuners/MT2063/mt_errordef.h,v $
**
**  Revision History:
**
**   SCR      Date      Author  Description
**  -------------------------------------------------------------------------
**   N/A   09-09-2004    JWS    Original
**   088   01-26-2005    DAD    Added MT_TUNER_INIT_ERR.
**   N/A   12-09-2005    DAD    Added MT_TUNER_TIMEOUT (info).
**   N/A   10-17-2006    JWS    Updated copyright notice.
**   N/A L 02-25-2008    RSK    Correcting Comment Block to match constants.
**
*****************************************************************************/

/*
** Note to users:  DO NOT EDIT THIS FILE
**
** If you wish to rename any of the "user defined" bits,
** it should be done in the user file that includes this
** source file (e.g. mt_userdef.h)
**
*/



/*
**  3 3 2 2 2 2 2 2 2 2 2 2 1 1 1 1 1 1 1 1 1 1
**  1 0 9 8 7 6 5 4 3 2 1 0 9 8 7 6 5 4 3 2 1 0 9 8 7 6 5 4 3 2 1 0
**  M U <- Info Codes --> <# Spurs> < User> <----- Err Codes ----->
**
**  31 = MT_ERROR - Master Error Flag.  If set, check Err Codes for reason.
**  30 = MT_USER_ERROR - User-declared error flag.
**  29 = Unused
**  28 = Unused
**
**  27 = MT_DNC_RANGE
**  26 = MT_UPC_RANGE
**  25 = MT_FOUT_RANGE
**  24 = MT_FIN_OUT_OF_RANGE
**
**  23 = MT_SPUR_PRESENT - Unavoidable spur in output
**  22 = MT_TUNER_TIMEOUT
**  21 = Unused
**  20 = MT_SPUR_CNT_MASK (MSB) - Count of avoided spurs
**
**  19 = MT_SPUR_CNT_MASK
**  18 = MT_SPUR_CNT_MASK
**  17 = MT_SPUR_CNT_MASK
**  16 = MT_SPUR_CNT_MASK (LSB)
**
**  15 = MT_USER_DEFINED4 - User-definable bit (see MT_Userdef.h)
**  14 = MT_USER_DEFINED3 - User-definable bit (see MT_Userdef.h)
**  13 = MT_USER_DEFINED2 - User-definable bit (see MT_Userdef.h)
**  12 = MT_USER_DEFINED1 - User-definable bit (see MT_Userdef.h)
**
**  11 = Unused
**  10 = Unused
**   9 = MT_TUNER_INIT_ERR - Tuner initialization error
**   8 = MT_TUNER_ID_ERR - Tuner Part Code / Rev Code mismatches expected value
**
**   7 = MT_TUNER_CNT_ERR - Attempt to open more than MT_TUNER_CNT tuners
**   6 = MT_ARG_NULL - Null pointer passed as argument
**   5 = MT_ARG_RANGE - Argument value out of range
**   4 = MT_INV_HANDLE - Tuner handle is invalid
**
**   3 = MT_COMM_ERR - Serial bus communications error
**   2 = MT_DNC_UNLOCK - Downconverter PLL is unlocked
**   1 = MT_UPC_UNLOCK - Upconverter PLL is unlocked
**   0 = MT_UNKNOWN - Unknown error
*/
#define MT_ERROR (1 << 31)
#define MT_USER_ERROR (1 << 30)

/*  Macro to be used to check for errors  */
#define MT_IS_ERROR(s) (((s) >> 30) != 0)
#define MT_NO_ERROR(s) (((s) >> 30) == 0)


#define MT_OK                           (0x00000000)

/*  Unknown error  */
#define MT_UNKNOWN                      (0x80000001)

/*  Error:  Upconverter PLL is not locked  */
#define MT_UPC_UNLOCK                   (0x80000002)

/*  Error:  Downconverter PLL is not locked  */
#define MT_DNC_UNLOCK                   (0x80000004)

/*  Error:  Two-wire serial bus communications error  */
#define MT_COMM_ERR                     (0x80000008)

/*  Error:  Tuner handle passed to function was invalid  */
#define MT_INV_HANDLE                   (0x80000010)

/*  Error:  Function argument is invalid (out of range)  */
#define MT_ARG_RANGE                    (0x80000020)

/*  Error:  Function argument (ptr to return value) was NULL  */
#define MT_ARG_NULL                     (0x80000040)

/*  Error: Attempt to open more than MT_TUNER_CNT tuners  */
#define MT_TUNER_CNT_ERR                (0x80000080)

/*  Error: Tuner Part Code / Rev Code mismatches expected value  */
#define MT_TUNER_ID_ERR                 (0x80000100)

/*  Error: Tuner Initialization failure  */
#define MT_TUNER_INIT_ERR               (0x80000200)

/*  User-definable fields (see mt_userdef.h)  */
#define MT_USER_DEFINED1                (0x00001000)
#define MT_USER_DEFINED2                (0x00002000)
#define MT_USER_DEFINED3                (0x00004000)
#define MT_USER_DEFINED4                (0x00008000)
#define MT_USER_MASK                    (0x4000f000)
#define MT_USER_SHIFT                   (12)

/*  Info: Mask of bits used for # of LO-related spurs that were avoided during tuning  */
#define MT_SPUR_CNT_MASK                (0x001f0000)
#define MT_SPUR_SHIFT                   (16)

/*  Info: Tuner timeout waiting for condition  */
#define MT_TUNER_TIMEOUT                (0x00400000)

/*  Info: Unavoidable LO-related spur may be present in the output  */
#define MT_SPUR_PRESENT                 (0x00800000)

/*  Info: Tuner input frequency is out of range */
#define MT_FIN_RANGE                    (0x01000000)

/*  Info: Tuner output frequency is out of range */
#define MT_FOUT_RANGE                   (0x02000000)

/*  Info: Upconverter frequency is out of range (may be reason for MT_UPC_UNLOCK) */
#define MT_UPC_RANGE                    (0x04000000)

/*  Info: Downconverter frequency is out of range (may be reason for MT_DPC_UNLOCK) */
#define MT_DNC_RANGE                    (0x08000000)























// Microtune source code - mt_userdef.h



/*****************************************************************************
**
**  Name: mt_userdef.h
**
**  Copyright 2006-2008 Microtune, Inc. All Rights Reserved
**
**  This source code file contains confidential information and/or trade
**  secrets of Microtune, Inc. or its affiliates and is subject to the
**  terms of your confidentiality agreement with Microtune, Inc. or one of
**  its affiliates, as applicable and the terms of the end-user software
**  license agreement (EULA).
**
**  Protected by US Patent 7,035,614 and additional patents pending or filed.
**
**  BY INSTALLING, COPYING, OR USING THE MICROTUNE SOFTWARE, YOU AGREE TO BE
**  BOUND BY ALL OF THE TERMS AND CONDITIONS OF THE MICROTUNE, INC. END-USER
**  SOFTWARE LICENSE AGREEMENT (EULA), INCLUDING BUT NOT LIMITED TO, THE
**  CONFIDENTIALITY OBLIGATIONS DESCRIBED THEREIN. IF YOU DO NOT AGREE, DO NOT
**  INSTALL, COPY, OR USE THE MICROTUNE SOFTWARE AND IMMEDIATELY RETURN OR
**  DESTROY ANY COPIES OF MICROTUNE SOFTWARE OR CONFIDENTIAL INFORMATION THAT
**  YOU HAVE IN YOUR POSSESSION.
*****************************************************************************/

/*****************************************************************************
**
**  Name: mt_userdef.h
**
**  Description:    User-defined data types needed by MicroTuner source code.
**
**                  Customers must provide the code for these functions
**                  in the file "mt_userdef.c".
**
**                  Customers must verify that the typedef's in the 
**                  "Data Types" section are correct for their platform.
**
**  Functions
**  Requiring
**  Implementation: MT_WriteSub
**                  MT_ReadSub
**                  MT_Sleep
**
**  References:     None
**
**  Exports:        None
**
**  CVS ID:         $Id: mt_userdef.h,v 1.2 2008/11/05 13:46:20 software Exp $
**  CVS Source:     $Source: /export/vol0/cvsroot/software/tuners/MT2063/mt_userdef.h,v $
**	               
**  Revision History:
**
**   SCR      Date      Author  Description
**  -------------------------------------------------------------------------
**   N/A   03-25-2004    DAD    Original
**   082   12-06-2004    JWS    Multi-tuner support - requires MTxxxx_CNT 
**                              declarations
**   N/A   04-13-2007    JWS    Added Signed 8-bit type - S8Data
**   200   08-07-2008    JWS    Included definition of DECT avoid constants.
**
*****************************************************************************/
#if !defined( __MT_USERDEF_H )
#define __MT_USERDEF_H

//#include "mt_errordef.h"

#if defined( __cplusplus )     
extern "C"                     /* Use "C" external linkage                  */
{
#endif

#if !defined( __MT_DATA_DEF_H )
#define __MT_DATA_DEF_H
/*
**  Data Types
*/
typedef unsigned char   U8Data;         /*  type corresponds to 8 bits      */
typedef signed char     S8Data;         /*  type corresponds to 8 bits      */
typedef unsigned int    UData_t;        /*  type must be at least 32 bits   */
typedef int             SData_t;        /*  type must be at least 32 bits   */
typedef void *          Handle_t;       /*  memory pointer type             */
typedef double          FData_t;        /*  floating point data type        */
#endif
#define MAX_UDATA         (4294967295U)  /*  max value storable in UData_t   */

/*
** Define an MTxxxx_CNT macro for each type of tuner that will be built
** into your application (e.g., MT2121, MT2060). MT_TUNER_CNT
** must be set to the SUM of all of the MTxxxx_CNT macros.
**
** #define MT2050_CNT  (1)
** #define MT2060_CNT  (1)
** #define MT2111_CNT  (1)
** #define MT2121_CNT  (3)
*/


#if !defined( MT_TUNER_CNT )
#define MT_TUNER_CNT               (1)  /*  total num of MicroTuner tuners  */
#endif

#ifndef _MT_DECT_Avoid_Type_DEFINE_
#define _MT_DECT_Avoid_Type_DEFINE_
//
// DECT Frequency Avoidance:
// These constants are used to avoid interference from DECT frequencies.
//
typedef enum
{
    MT_NO_DECT_AVOIDANCE = 0,              /* Do not create DECT exclusion zones.     */
    MT_AVOID_US_DECT     = 0x00000001,     /* Avoid US DECT frequencies.              */
    MT_AVOID_EURO_DECT   = 0x00000002,     /* Avoid European DECT frequencies.        */
    MT_AVOID_BOTH                          /* Avoid both regions. Not typically used. */
} MT_DECT_Avoid_Type;
#endif
/*
**  Optional user-defined Error/Info Codes  (examples below)
**
**  This is the area where you can define user-specific error/info return
**  codes to be returned by any of the functions you are responsible for
**  writing such as MT_WriteSub() and MT_ReadSub.  There are four bits
**  available in the status field for your use.  When set, these
**  bits will be returned in the status word returned by any tuner driver
**  call.  If you OR in the MT_ERROR bit as well, the tuner driver code
**  will treat the code as an error.
**
**  The following are a few examples of errors you can provide.
**
**  Example 1:
**  You might check to see the hUserData handle is correct and issue 
**  MY_USERDATA_INVALID which would be defined like this:
**
**  #define MY_USERDATA_INVALID  (MT_USER_ERROR | MT_USER_DEFINED1)
**
**
**  Example 2:
**  You might be able to provide more descriptive two-wire bus errors:
**
**  #define NO_ACK   (MT_USER_ERROR | MT_USER_DEFINED1)
**  #define NO_NACK  (MT_USER_ERROR | MT_USER_DEFINED2)
**  #define BUS_BUSY (MT_USER_ERROR | MT_USER_DEFINED3)
**
**
**  Example 3:
**  You can also provide information (non-error) feedback:
**
**  #define MY_INFO_1   (MT_USER_DEFINED1)
**
**
**  Example 4:
**  You can combine the fields together to make a multi-bit field.
**  This one can provide the tuner number based off of the addr
**  passed to MT_WriteSub or MT_ReadSub.  It assumes that
**  MT_USER_DEFINED4 through MT_USER_DEFINED1 are contiguously. If
**  TUNER_NUM were OR'ed into the status word on an error, you could
**  use this to identify which tuner had the problem (and whether it
**  was during a read or write operation).
**
**  #define TUNER_NUM  ((addr & 0x07) >> 1) << MT_USER_SHIFT
**
*/

/*****************************************************************************
**
**  Name: MT_WriteSub
**
**  Description:    Write values to device using a two-wire serial bus.
**
**  Parameters:     hUserData  - User-specific I/O parameter that was
**                               passed to tuner's Open function.
**                  addr       - device serial bus address  (value passed
**                               as parameter to MTxxxx_Open)
**                  subAddress - serial bus sub-address (Register Address)
**                  pData      - pointer to the Data to be written to the 
**                               device 
**                  cnt        - number of bytes/registers to be written
**
**  Returns:        status:
**                      MT_OK            - No errors
**                      MT_COMM_ERR      - Serial bus communications error
**                      user-defined
**
**  Notes:          This is a callback function that is called from the
**                  the tuning algorithm.  You MUST provide code for this
**                  function to write data using the tuner's 2-wire serial 
**                  bus.
**
**                  The hUserData parameter is a user-specific argument.
**                  If additional arguments are needed for the user's
**                  serial bus read/write functions, this argument can be
**                  used to supply the necessary information.
**                  The hUserData parameter is initialized in the tuner's Open
**                  function.
**
**  Revision History:
**
**   SCR      Date      Author  Description
**  -------------------------------------------------------------------------
**   N/A   03-25-2004    DAD    Original
**
*****************************************************************************/
UData_t MT2063_WriteSub(Handle_t hUserData, 
                    UData_t addr, 
                    U8Data subAddress, 
                    U8Data *pData, 
                    UData_t cnt);


/*****************************************************************************
**
**  Name: MT_ReadSub
**
**  Description:    Read values from device using a two-wire serial bus.
**
**  Parameters:     hUserData  - User-specific I/O parameter that was
**                               passed to tuner's Open function.
**                  addr       - device serial bus address  (value passed
**                               as parameter to MTxxxx_Open)
**                  subAddress - serial bus sub-address (Register Address)
**                  pData      - pointer to the Data to be written to the 
**                               device 
**                  cnt        - number of bytes/registers to be written
**
**  Returns:        status:
**                      MT_OK            - No errors
**                      MT_COMM_ERR      - Serial bus communications error
**                      user-defined
**
**  Notes:          This is a callback function that is called from the
**                  the tuning algorithm.  You MUST provide code for this
**                  function to read data using the tuner's 2-wire serial 
**                  bus.
**
**                  The hUserData parameter is a user-specific argument.
**                  If additional arguments are needed for the user's
**                  serial bus read/write functions, this argument can be
**                  used to supply the necessary information.
**                  The hUserData parameter is initialized in the tuner's Open
**                  function.
**
**  Revision History:
**
**   SCR      Date      Author  Description
**  -------------------------------------------------------------------------
**   N/A   03-25-2004    DAD    Original
**
*****************************************************************************/
UData_t MT2063_ReadSub(Handle_t hUserData, 
                   UData_t addr, 
                   U8Data subAddress, 
                   U8Data *pData, 
                   UData_t cnt);


/*****************************************************************************
**
**  Name: MT_Sleep
**
**  Description:    Delay execution for "nMinDelayTime" milliseconds
**
**  Parameters:     hUserData     - User-specific I/O parameter that was
**                                  passed to tuner's Open function.
**                  nMinDelayTime - Delay time in milliseconds
**
**  Returns:        None.
**
**  Notes:          This is a callback function that is called from the
**                  the tuning algorithm.  You MUST provide code that
**                  blocks execution for the specified period of time. 
**
**  Revision History:
**
**   SCR      Date      Author  Description
**  -------------------------------------------------------------------------
**   N/A   03-25-2004    DAD    Original
**
*****************************************************************************/
void MT2063_Sleep(Handle_t hUserData,
              UData_t nMinDelayTime);


#if defined(MT2060_CNT)
#if MT2060_CNT > 0
/*****************************************************************************
**
**  Name: MT_TunerGain  (for MT2060 only)
**
**  Description:    Measure the relative tuner gain using the demodulator
**
**  Parameters:     hUserData  - User-specific I/O parameter that was
**                               passed to tuner's Open function.
**                  pMeas      - Tuner gain (1/100 of dB scale).
**                               ie. 1234 = 12.34 (dB)
**
**  Returns:        status:
**                      MT_OK  - No errors
**                      user-defined errors could be set
**
**  Notes:          This is a callback function that is called from the
**                  the 1st IF location routine.  You MUST provide
**                  code that measures the relative tuner gain in a dB
**                  (not linear) scale.  The return value is an integer
**                  value scaled to 1/100 of a dB.
**
**  Revision History:
**
**   SCR      Date      Author  Description
**  -------------------------------------------------------------------------
**   N/A   06-16-2004    DAD    Original
**   N/A   11-30-2004    DAD    Renamed from MT_DemodInputPower.  This name
**                              better describes what this function does.
**
*****************************************************************************/
UData_t MT_TunerGain(Handle_t hUserData,
                     SData_t* pMeas);
#endif
#endif

#if defined( __cplusplus )     
}
#endif

#endif























// Microtune source code - mt2063.h



/*****************************************************************************
**
**  Name: mt2063.h
**
**  Copyright 2007 Microtune, Inc. All Rights Reserved
**
**  This source code file contains confidential information and/or trade
**  secrets of Microtune, Inc. or its affiliates and is subject to the
**  terms of your confidentiality agreement with Microtune, Inc. or one of
**  its affiliates, as applicable.
**
*****************************************************************************/

/*****************************************************************************
**
**  Name: mt2063.h
**
**  Description:    Microtune MT2063 B0/B1 Tuner software interface
**                  Supports tuners with Part/Rev codes: 0x9B, 0x9C.
**
**  Functions
**  Implemented:    UData_t  MT2063_Open
**                  UData_t  MT2063_Close
**                  UData_t  MT2063_GetGPIO
**                  UData_t  MT2063_GetLocked
**                  UData_t  MT2063_GetParam
**                  UData_t  MT2063_GetPowerMaskBits
**                  UData_t  MT2063_GetReg
**                  UData_t  MT2063_GetTemp
**                  UData_t  MT2063_GetUserData
**                  UData_t  MT2063_ReInit
**                  UData_t  MT2063_SetGPIO
**                  UData_t  MT2063_SetParam
**                  UData_t  MT2063_SetPowerMaskBits
**                  UData_t  MT2063_ClearPowerMaskBits
**                  UData_t  MT2063_SetReg
**                  UData_t  MT2063_Tune
**
**  References:     AN-xxxxx: MT2063 Programming Procedures Application Note
**                  MicroTune, Inc.
**                  AN-00010: MicroTuner Serial Interface Application Note
**                  MicroTune, Inc.
**
**  Enumerations
**  Defined:        MT2063_Ext_SRO
**                  MT2063_Param
**                  MT2063_Mask_Bits
**                  MT2063_GPIO_Attr;
**                  MT2063_Temperature
**
**  Exports:        None
**
**  Dependencies:   MT_ReadSub(hUserData, IC_Addr, subAddress, *pData, cnt);
**                  - Read byte(s) of data from the two-wire bus.
**
**                  MT_WriteSub(hUserData, IC_Addr, subAddress, *pData, cnt);
**                  - Write byte(s) of data to the two-wire bus.
**
**                  MT_Sleep(hUserData, nMinDelayTime);
**                  - Delay execution for nMinDelayTime milliseconds
**
**  CVS ID:         $Id: mt2063.h,v 1.5 2009/04/29 09:58:14 software Exp $
**  CVS Source:     $Source: /export/vol0/cvsroot/software/tuners/MT2063/mt2063.h,v $
**
**  Revision History:
**
**   SCR      Date      Author  Description
**  -------------------------------------------------------------------------
**   138   06-19-2007    DAD    Ver 1.00: Initial, derived from mt2067_b.
**   N/A   08-01-2007    PINZ   Changed Analog & DVB-T settings and added
**                              SAW-less receiver mode.
**   155   10-01-2007    DAD    Ver 1.06: Add receiver mode for SECAM positive
**                                        modulation
**                                        (MT2063_ANALOG_TV_POS_NO_RFAGC_MODE)
**   N/A   10-22-2007    PINZ   Ver 1.07: Changed some Registers at init to have
**                                        the same settings as with MT Launcher
**   N/A   10-30-2007    PINZ             Add SetParam VGAGC & VGAOI
**                                        Add SetParam DNC_OUTPUT_ENABLE
**                                        Removed VGAGC from receiver mode,
**                                        default now 1
**   N/A   10-31-2007    PINZ   Ver 1.08: Add SetParam TAGC, removed from rcvr-mode
**                                        Add SetParam AMPGC, removed from rcvr-mode
**                                        Corrected names of GCU values
**                                        Actualized Receiver-Mode values
**   N/A   04-18-2008    PINZ   Ver 1.09: Add SetParam LNA_TGT
**                                        Split SetParam up to ACLNA / ACLNA_MAX
**                                        removed ACLNA_INRC/DECR (+RF & FIF)
**                                        removed GCUAUTO / BYPATNDN/UP
**   175 I 06-06-2008    PINZ   Ver 1.10: Add control to avoid US DECT freqs.
**   175 I 06-19-2008    RSK    Ver 1.17: Refactor DECT control to SpurAvoid.
**   175 I 06-24-2008    PINZ   Ver 1.18: Add Get/SetParam CTFILT_SW
**         07-10-2008    PINZ   Ver 1.19: Documentation Updates, Add a GetParam
**                                        for each SetParam (LNA_RIN, TGTs)
**         01-07-2009    PINZ   Ver 1.27: Changed value of MT2063_ALL_SD in .h
**         02-16-2009    PINZ   Ver 1.29: Several fixes to improve compiler behavior
**   N/A I 03-19-2009    PINZ   Ver 1.32: Add GetParam VERSION (of API)
**                                        Add GetParam IC_REV
*****************************************************************************/
#if !defined( __MT2063_H )
#define __MT2063_H

//#include "mt_userdef.h"

#if defined( __cplusplus )
extern "C"                     /* Use "C" external linkage                  */
{
#endif


/*
** Chip revision
**
*/
typedef enum
{
    MT2063_XX = 0,
    MT2063_B0 = 206310,
    MT2063_B1 = 206311,
    MT2063_B3 = 206313
} MT2063_REVCODE;


/*
**  Values returned by the MT2063's on-chip temperature sensor
**  to be read/written.
*/
typedef enum
{
    MT2063_T_0C = 0,                    /*  Temperature approx 0C           */
    MT2063_T_10C,                       /*  Temperature approx 10C          */
    MT2063_T_20C,                       /*  Temperature approx 20C          */
    MT2063_T_30C,                       /*  Temperature approx 30C          */
    MT2063_T_40C,                       /*  Temperature approx 40C          */
    MT2063_T_50C,                       /*  Temperature approx 50C          */
    MT2063_T_60C,                       /*  Temperature approx 60C          */
    MT2063_T_70C,                       /*  Temperature approx 70C          */
    MT2063_T_80C,                       /*  Temperature approx 80C          */
    MT2063_T_90C,                       /*  Temperature approx 90C          */
    MT2063_T_100C,                      /*  Temperature approx 100C         */
    MT2063_T_110C,                      /*  Temperature approx 110C         */
    MT2063_T_120C,                      /*  Temperature approx 120C         */
    MT2063_T_130C,                      /*  Temperature approx 130C         */
    MT2063_T_140C,                      /*  Temperature approx 140C         */
    MT2063_T_150C                       /*  Temperature approx 150C         */
} MT2063_Temperature;


/*
** Parameters for selecting GPIO bits
*/
typedef enum
{
    MT2063_GPIO_IN,
    MT2063_GPIO_DIR,
    MT2063_GPIO_OUT
} MT2063_GPIO_Attr;

typedef enum
{
    MT2063_GPIO0,
    MT2063_GPIO1,
    MT2063_GPIO2
} MT2063_GPIO_ID;


/*
**  Parameter for function MT2063_SetExtSRO that specifies the external
**  SRO drive frequency.
**
**  MT2063_EXT_SRO_OFF is the power-up default value.
*/
typedef enum
{
    MT2063_EXT_SRO_OFF,                 /*  External SRO drive off          */
    MT2063_EXT_SRO_BY_4,                /*  External SRO drive divide by 4  */
    MT2063_EXT_SRO_BY_2,                /*  External SRO drive divide by 2  */
    MT2063_EXT_SRO_BY_1                 /*  External SRO drive divide by 1  */
} MT2063_Ext_SRO;


/*
**  Parameter for function MT2063_SetPowerMask that specifies the power down
**  of various sections of the MT2063.
*/
typedef enum
{
    MT2063_REG_SD   = 0x0040,   /* Shutdown regulator                 */
    MT2063_SRO_SD   = 0x0020,   /* Shutdown SRO                       */
    MT2063_AFC_SD   = 0x0010,   /* Shutdown AFC A/D                   */
    MT2063_PD_SD    = 0x0002,   /* Enable power detector shutdown     */
    MT2063_PDADC_SD = 0x0001,   /* Enable power detector A/D shutdown */
    MT2063_VCO_SD   = 0x8000,   /* Enable VCO shutdown                */
    MT2063_LTX_SD   = 0x4000,   /* Enable LTX shutdown                */
    MT2063_LT1_SD   = 0x2000,   /* Enable LT1 shutdown                */
    MT2063_LNA_SD   = 0x1000,   /* Enable LNA shutdown                */
    MT2063_UPC_SD   = 0x0800,   /* Enable upconverter shutdown        */
    MT2063_DNC_SD   = 0x0400,   /* Enable downconverter shutdown      */
    MT2063_VGA_SD   = 0x0200,   /* Enable VGA shutdown                */
    MT2063_AMP_SD   = 0x0100,   /* Enable AMP shutdown                */
    MT2063_ALL_SD   = 0xFF13,   /* All shutdown bits for this tuner   */
    MT2063_NONE_SD  = 0x0000    /* No shutdown bits                   */
} MT2063_Mask_Bits;


/*
**  Parameter for function MT2063_GetParam & MT2063_SetParam that
**  specifies the tuning algorithm parameter to be read/written.
*/
typedef enum
{
    /*  version of the API, e.g. 10302 = 1.32                               */
    MT2063_VERSION,

    /*  tuner address                                  set by MT2063_Open() */
    MT2063_IC_ADDR,

    /*  tuner revision code  (see enum  MT2063_REVCODE for values)          */
    MT2063_IC_REV,

    /*  max number of MT2063 tuners     set by MT_TUNER_CNT in mt_userdef.h */
    MT2063_MAX_OPEN,

    /*  current number of open MT2063 tuners           set by MT2063_Open() */
    MT2063_NUM_OPEN,

    /*  crystal frequency                            (default: 16000000 Hz) */
    MT2063_SRO_FREQ,

    /*  min tuning step size                            (default: 50000 Hz) */
    MT2063_STEPSIZE,

    /*  input center frequency                         set by MT2063_Tune() */
    MT2063_INPUT_FREQ,

    /*  LO1 Frequency                                  set by MT2063_Tune() */
    MT2063_LO1_FREQ,

    /*  LO1 minimum step size                          (default: 250000 Hz) */
    MT2063_LO1_STEPSIZE,

    /*  LO1 FracN keep-out region                      (default: 999999 Hz) */
    MT2063_LO1_FRACN_AVOID,

    /*  Current 1st IF in use                          set by MT2063_Tune() */
    MT2063_IF1_ACTUAL,

    /*  Requested 1st IF                               set by MT2063_Tune() */
    MT2063_IF1_REQUEST,

    /*  Center of 1st IF SAW filter                (default: 1218000000 Hz) */
    MT2063_IF1_CENTER,

    /*  Bandwidth of 1st IF SAW filter               (default: 20000000 Hz) */
    MT2063_IF1_BW,

    /*  zero-IF bandwidth                             (default: 2000000 Hz) */
    MT2063_ZIF_BW,

    /*  LO2 Frequency                                  set by MT2063_Tune() */
    MT2063_LO2_FREQ,

    /*  LO2 minimum step size                           (default: 50000 Hz) */
    MT2063_LO2_STEPSIZE,

    /*  LO2 FracN keep-out region                      (default: 374999 Hz) */
    MT2063_LO2_FRACN_AVOID,

    /*  output center frequency                        set by MT2063_Tune() */
    MT2063_OUTPUT_FREQ,

    /*  output bandwidth                               set by MT2063_Tune() */
    MT2063_OUTPUT_BW,

    /*  min inter-tuner LO separation                 (default: 1000000 Hz) */
    MT2063_LO_SEPARATION,

    /*  ID of avoid-spurs algorithm in use            compile-time constant */
    MT2063_AS_ALG,

    /*  max # of intra-tuner harmonics                       (default: 15)  */
    MT2063_MAX_HARM1,

    /*  max # of inter-tuner harmonics                        (default: 7)  */
    MT2063_MAX_HARM2,

    /*  # of 1st IF exclusion zones used               set by MT2063_Tune() */
    MT2063_EXCL_ZONES,

    /*  # of spurs found/avoided                       set by MT2063_Tune() */
    MT2063_NUM_SPURS,

    /*  >0 spurs avoided                               set by MT2063_Tune() */
    MT2063_SPUR_AVOIDED,

    /*  >0 spurs in output (mathematically)            set by MT2063_Tune() */
    MT2063_SPUR_PRESENT,

    /* Receiver Mode for some parameters. 1 is DVB-T                        */
    MT2063_RCVR_MODE,

    /* directly set LNA attenuation, parameter is value to set              */
    MT2063_ACLNA,

    /* maximum LNA attenuation, parameter is value to set                   */
    MT2063_ACLNA_MAX,

    /* directly set ATN attenuation.  Paremeter is value to set.            */
    MT2063_ACRF,

    /* maxium ATN attenuation.  Paremeter is value to set.                  */
    MT2063_ACRF_MAX,

    /* directly set FIF attenuation.  Paremeter is value to set.            */
    MT2063_ACFIF,

    /* maxium FIF attenuation.  Paremeter is value to set.                  */
    MT2063_ACFIF_MAX,

    /*  LNA Rin                                                             */
    MT2063_LNA_RIN,

    /*  Power Detector LNA level target                                     */
    MT2063_LNA_TGT,

    /*  Power Detector 1 level                                              */
    MT2063_PD1,

    /*  Power Detector 1 level target                                       */
    MT2063_PD1_TGT,

    /*  Power Detector 2 level                                              */
    MT2063_PD2,

    /*  Power Detector 2 level target                                       */
    MT2063_PD2_TGT,

    /*  Selects, which DNC is activ                                         */
    MT2063_DNC_OUTPUT_ENABLE,

    /*  VGA gain code                                                       */
    MT2063_VGAGC,

    /*  VGA bias current                                                    */
    MT2063_VGAOI,

    /*  TAGC, determins the speed of the AGC                                */
    MT2063_TAGC,

    /*  AMP gain code                                                       */
    MT2063_AMPGC,

    /* Control setting to avoid DECT freqs         (default: MT_AVOID_BOTH) */
    MT2063_AVOID_DECT,

    /* Cleartune filter selection: 0 - by IC (default), 1 - by software     */
    MT2063_CTFILT_SW,

    MT2063_EOP                    /*  last entry in enumerated list         */

} MT2063_Param;


/*
**  Parameter for selecting tuner mode
*/
typedef enum
{
    MT2063_CABLE_QAM = 0,            /* Digital cable              */
    MT2063_CABLE_ANALOG,             /* Analog cable               */
    MT2063_OFFAIR_COFDM,             /* Digital offair             */
    MT2063_OFFAIR_COFDM_SAWLESS,     /* Digital offair without SAW */
    MT2063_OFFAIR_ANALOG,            /* Analog offair              */
    MT2063_OFFAIR_8VSB,              /* Analog offair              */
    MT2063_NUM_RCVR_MODES
} MT2063_RCVR_MODES;


/*
**  Possible values for MT2063_DNC_OUTPUT
*/
typedef enum {
    MT2063_DNC_NONE = 0,
    MT2063_DNC_1,
    MT2063_DNC_2,
    MT2063_DNC_BOTH
} MT2063_DNC_Output_Enable;


/* ====== Functions which are declared in MT2063.c File ======= */

/******************************************************************************
**
**  Name: MT2063_Open
**
**  Description:    Initialize the tuner's register values.
**
**  Usage:          status = MT2063_Open(0xC0, &hMT2063, NULL);
**                  if (MT_IS_ERROR(status))
**                      //  Check error codes for reason
**
**  Parameters:     MT2063_Addr  - Serial bus address of the tuner.
**                  hMT2063      - Tuner handle passed back.
**                  hUserData    - User-defined data, if needed for the
**                                 MT_ReadSub() & MT_WriteSub functions.
**
**  Returns:        status:
**                      MT_OK             - No errors
**                      MT_TUNER_ID_ERR   - Tuner Part/Rev code mismatch
**                      MT_TUNER_INIT_ERR - Tuner initialization failed
**                      MT_COMM_ERR       - Serial bus communications error
**                      MT_ARG_NULL       - Null pointer argument passed
**                      MT_TUNER_CNT_ERR  - Too many tuners open
**
**  Dependencies:   MT_ReadSub  - Read byte(s) of data from the two-wire bus
**                  MT_WriteSub - Write byte(s) of data to the two-wire bus
**
**  Revision History:
**
**   SCR      Date      Author  Description
**  -------------------------------------------------------------------------
**   138   06-19-2007    DAD    Ver 1.00: Initial, derived from mt2067_b.
**
******************************************************************************/
UData_t MT2063_Open(UData_t MT2063_Addr,
                    Handle_t* hMT2063,
                    Handle_t hUserData);


/******************************************************************************
**
**  Name: MT2063_Close
**
**  Description:    Release the handle to the tuner.
**
**  Parameters:     hMT2063      - Handle to the MT2063 tuner
**
**  Usage:          status = MT2063_Close(hMT2063);
**
**  Returns:        status:
**                      MT_OK         - No errors
**                      MT_INV_HANDLE - Invalid tuner handle
**
**  Dependencies:   mt_errordef.h - definition of error codes
**
**  Revision History:
**
**   SCR      Date      Author  Description
**  -------------------------------------------------------------------------
**   138   06-19-2007    DAD    Ver 1.00: Initial, derived from mt2067_b.
**
******************************************************************************/
UData_t MT2063_Close(Handle_t hMT2063);


/****************************************************************************
**
**  Name: MT2063_Tune
**
**  Description:    Change the tuner's tuned frequency to f_in.
**
**  Parameters:     h           - Tuner handle (returned by MT2063_Open)
**                  f_in        - RF input center frequency (in Hz).
**
**  Usage:          status = MT2063_Tune(hMT2063,
**                                       f_in)
**
**  Returns:        status:
**                      MT_OK            - No errors
**                      MT_INV_HANDLE    - Invalid tuner handle
**                      MT_UPC_UNLOCK    - Upconverter PLL unlocked
**                      MT_DNC_UNLOCK    - Downconverter PLL unlocked
**                      MT_COMM_ERR      - Serial bus communications error
**                      MT_SPUR_CNT_MASK - Count of avoided LO spurs
**                      MT_SPUR_PRESENT  - LO spur possible in output
**                      MT_FIN_RANGE     - Input freq out of range
**                      MT_FOUT_RANGE    - Output freq out of range
**                      MT_UPC_RANGE     - Upconverter freq out of range
**                      MT_DNC_RANGE     - Downconverter freq out of range
**
**  Dependencies:   MUST CALL MT2063_Open BEFORE MT2063_Tune!
**
**                  MT_ReadSub   - Read data from the two-wire serial bus
**                  MT_WriteSub  - Write data to the two-wire serial bus
**                  MT_Sleep     - Delay execution for x milliseconds
**
**  Revision History:
**
**   SCR      Date      Author  Description
**  -------------------------------------------------------------------------
**   138   06-19-2007    DAD    Ver 1.00: Initial, derived from mt2067_b.
**
****************************************************************************/
UData_t MT2063_Tune(Handle_t h,
                    UData_t f_in);    /* RF input center frequency   */


/******************************************************************************
**
**  Name: MT2063_GetGPIO
**
**  Description:    Get the current MT2063 GPIO value.
**
**  Parameters:     h            - Open handle to the tuner (from MT2063_Open).
**                  attr         - Selects input readback, I/O direction or
**                                 output value
**                  bit          - Selects GPIO0, GPIO1, or GPIO2
**                  *value       - current setting of GPIO pin
**
**  Usage:          status = MT2063_GetGPIO(hMT2063, MT2063_GPIO2, &value);
**
**  Returns:        status:
**                      MT_OK            - No errors
**                      MT_COMM_ERR      - Serial bus communications error
**                      MT_INV_HANDLE    - Invalid tuner handle
**                      MT_ARG_NULL      - Null pointer argument passed
**
**  Dependencies:   MT_ReadSub  - Read byte(s) of data from the serial bus
**
**  Revision History:
**
**   SCR      Date      Author  Description
**  -------------------------------------------------------------------------
**   138   06-19-2007    DAD    Ver 1.00: Initial, derived from mt2067_b.
**
******************************************************************************/
UData_t MT2063_GetGPIO(Handle_t h, MT2063_GPIO_ID gpio_id, MT2063_GPIO_Attr attr, UData_t* value);


/****************************************************************************
**
**  Name: MT2063_GetLocked
**
**  Description:    Checks to see if LO1 and LO2 are locked.
**
**  Parameters:     h            - Open handle to the tuner (from MT2063_Open).
**
**  Usage:          status = MT2063_GetLocked(hMT2063);
**                  if (status & (MT_UPC_UNLOCK | MT_DNC_UNLOCK))
**                      //  error!, one or more PLL's unlocked
**
**  Returns:        status:
**                      MT_OK            - No errors
**                      MT_UPC_UNLOCK    - Upconverter PLL unlocked
**                      MT_DNC_UNLOCK    - Downconverter PLL unlocked
**                      MT_COMM_ERR      - Serial bus communications error
**                      MT_INV_HANDLE    - Invalid tuner handle
**
**  Dependencies:   MT_ReadSub    - Read byte(s) of data from the serial bus
**                  MT_Sleep      - Delay execution for x milliseconds
**
**  Revision History:
**
**   SCR      Date      Author  Description
**  -------------------------------------------------------------------------
**   138   06-19-2007    DAD    Ver 1.00: Initial, derived from mt2067_b.
**
****************************************************************************/
UData_t MT2063_GetLocked(Handle_t h);


/****************************************************************************
**
**  Name: MT2063_GetParam
**
**  Description:    Gets a tuning algorithm parameter.
**
**                  This function provides access to the internals of the
**                  tuning algorithm - mostly for testing purposes.
**
**  Parameters:     h           - Tuner handle (returned by MT2063_Open)
**                  param       - Tuning algorithm parameter
**                                (see enum MT2063_Param)
**                  pValue      - ptr to returned value
**
**                  param                   Description
**                  ----------------------  --------------------------------
**                  MT2063_IC_ADDR          Serial Bus address of this tuner
**                  MT2063_MAX_OPEN         Max # of MT2063's allowed open
**                  MT2063_NUM_OPEN         # of MT2063's open
**                  MT2063_SRO_FREQ         crystal frequency
**                  MT2063_STEPSIZE         minimum tuning step size
**                  MT2063_INPUT_FREQ       input center frequency
**                  MT2063_LO1_FREQ         LO1 Frequency
**                  MT2063_LO1_STEPSIZE     LO1 minimum step size
**                  MT2063_LO1_FRACN_AVOID  LO1 FracN keep-out region
**                  MT2063_IF1_ACTUAL       Current 1st IF in use
**                  MT2063_IF1_REQUEST      Requested 1st IF
**                  MT2063_IF1_CENTER       Center of 1st IF SAW filter
**                  MT2063_IF1_BW           Bandwidth of 1st IF SAW filter
**                  MT2063_ZIF_BW           zero-IF bandwidth
**                  MT2063_LO2_FREQ         LO2 Frequency
**                  MT2063_LO2_STEPSIZE     LO2 minimum step size
**                  MT2063_LO2_FRACN_AVOID  LO2 FracN keep-out region
**                  MT2063_OUTPUT_FREQ      output center frequency
**                  MT2063_OUTPUT_BW        output bandwidth
**                  MT2063_LO_SEPARATION    min inter-tuner LO separation
**                  MT2063_AS_ALG           ID of avoid-spurs algorithm in use
**                  MT2063_MAX_HARM1        max # of intra-tuner harmonics
**                  MT2063_MAX_HARM2        max # of inter-tuner harmonics
**                  MT2063_EXCL_ZONES       # of 1st IF exclusion zones
**                  MT2063_NUM_SPURS        # of spurs found/avoided
**                  MT2063_SPUR_AVOIDED     >0 spurs avoided
**                  MT2063_SPUR_PRESENT     >0 spurs in output (mathematically)
**                  MT2063_RCVR_MODE          Predefined modes.
**                  MT2063_LNA_RIN            Get LNA RIN value
**                  MT2063_LNA_TGT            Get target power level at LNA
**                  MT2063_PD1_TGT            Get target power level at PD1
**                  MT2063_PD2_TGT            Get target power level at PD2
**                  MT2063_ACLNA              LNA attenuator gain code
**                  MT2063_ACRF               RF attenuator gain code
**                  MT2063_ACFIF              FIF attenuator gain code
**                  MT2063_ACLNA_MAX          LNA attenuator limit
**                  MT2063_ACRF_MAX           RF attenuator limit
**                  MT2063_ACFIF_MAX          FIF attenuator limit
**                  MT2063_PD1                Actual value of PD1
**                  MT2063_PD2                Actual value of PD2
**                  MT2063_DNC_OUTPUT_ENABLE  DNC output selection
**                  MT2063_VGAGC              VGA gain code
**                  MT2063_VGAOI              VGA output current
**                  MT2063_TAGC               TAGC setting
**                  MT2063_AMPGC              AMP gain code
**                  MT2063_AVOID_DECT         Avoid DECT Frequencies
**                  MT2063_CTFILT_SW          Cleartune filter selection
**
**  Usage:          status |= MT2063_GetParam(hMT2063,
**                                            MT2063_IF1_ACTUAL,
**                                            &f_IF1_Actual);
**
**  Returns:        status:
**                      MT_OK            - No errors
**                      MT_INV_HANDLE    - Invalid tuner handle
**                      MT_ARG_NULL      - Null pointer argument passed
**                      MT_ARG_RANGE     - Invalid parameter requested
**
**  Dependencies:   USERS MUST CALL MT2063_Open() FIRST!
**
**  See Also:       MT2063_SetParam, MT2063_Open
**
**  Revision History:
**
**   SCR      Date      Author  Description
**  -------------------------------------------------------------------------
**   138   06-19-2007    DAD    Ver 1.00: Initial, derived from mt2067_b.
**   154   09-13-2007    RSK    Ver 1.05: Get/SetParam changes for LOx_FREQ
**         10-31-2007    PINZ   Ver 1.08: Get/SetParam add VGAGC, VGAOI, AMPGC, TAGC
**   173 M 01-23-2008    RSK    Ver 1.12: Read LO1C and LO2C registers from HW
**                                        in GetParam.
**         04-18-2008    PINZ   Ver 1.15: Add SetParam LNARIN & PDxTGT
**                                        Split SetParam up to ACLNA / ACLNA_MAX
**                                        removed ACLNA_INRC/DECR (+RF & FIF)
**                                        removed GCUAUTO / BYPATNDN/UP
**   175 I 16-06-2008    PINZ   Ver 1.16: Add control to avoid US DECT freqs.
**   175 I 06-19-2008    RSK    Ver 1.17: Refactor DECT control to SpurAvoid.
**         06-24-2008    PINZ   Ver 1.18: Add Get/SetParam CTFILT_SW
**         07-10-2008    PINZ   Ver 1.19: Documentation Updates, Add a GetParam
**                                        for each SetParam (LNA_RIN, TGTs)
**
****************************************************************************/
UData_t MT2063_GetParam(Handle_t     h,
                        MT2063_Param param,
                        UData_t*     pValue);


/****************************************************************************
**
**  Name: MT2063_GetReg
**
**  Description:    Gets an MT2063 register.
**
**  Parameters:     h           - Tuner handle (returned by MT2063_Open)
**                  reg         - MT2063 register/subaddress location
**                  *val        - MT2063 register/subaddress value
**
**  Returns:        status:
**                      MT_OK            - No errors
**                      MT_COMM_ERR      - Serial bus communications error
**                      MT_INV_HANDLE    - Invalid tuner handle
**                      MT_ARG_NULL      - Null pointer argument passed
**                      MT_ARG_RANGE     - Argument out of range
**
**  Dependencies:   USERS MUST CALL MT2063_Open() FIRST!
**
**                  Use this function if you need to read a register from
**                  the MT2063.
**
**  Revision History:
**
**   SCR      Date      Author  Description
**  -------------------------------------------------------------------------
**   138   06-19-2007    DAD    Ver 1.00: Initial, derived from mt2067_b.
**
****************************************************************************/
UData_t MT2063_GetReg(Handle_t h,
                      U8Data   reg,
                      U8Data*  val);


/******************************************************************************
**
**  Name: MT2063_GetTemp
**
**  Description:    Get the MT2063 Temperature register.
**
**  Parameters:     h            - Open handle to the tuner (from MT2063_Open).
**                  *value       - value read from the register
**
**                                    Binary
**                  Value Returned    Value    Approx Temp
**                  ---------------------------------------------
**                  MT2063_T_0C       0000         0C
**                  MT2063_T_10C      0001        10C
**                  MT2063_T_20C      0010        20C
**                  MT2063_T_30C      0011        30C
**                  MT2063_T_40C      0100        40C
**                  MT2063_T_50C      0101        50C
**                  MT2063_T_60C      0110        60C
**                  MT2063_T_70C      0111        70C
**                  MT2063_T_80C      1000        80C
**                  MT2063_T_90C      1001        90C
**                  MT2063_T_100C     1010       100C
**                  MT2063_T_110C     1011       110C
**                  MT2063_T_120C     1100       120C
**                  MT2063_T_130C     1101       130C
**                  MT2063_T_140C     1110       140C
**                  MT2063_T_150C     1111       150C
**
**  Returns:        status:
**                      MT_OK            - No errors
**                      MT_COMM_ERR      - Serial bus communications error
**                      MT_INV_HANDLE    - Invalid tuner handle
**                      MT_ARG_NULL      - Null pointer argument passed
**                      MT_ARG_RANGE     - Argument out of range
**
**  Dependencies:   MT_ReadSub  - Read byte(s) of data from the two-wire bus
**                  MT_WriteSub - Write byte(s) of data to the two-wire bus
**
**  Revision History:
**
**   SCR      Date      Author  Description
**  -------------------------------------------------------------------------
**   138   06-19-2007    DAD    Ver 1.00: Initial, derived from mt2067_b.
**
******************************************************************************/
UData_t MT2063_GetTemp(Handle_t h, MT2063_Temperature* value);


/****************************************************************************
**
**  Name: MT2063_GetUserData
**
**  Description:    Gets the user-defined data item.
**
**  Parameters:     h           - Tuner handle (returned by MT2063_Open)
**
**  Usage:          status = MT2063_GetUserData(hMT2063, &hUserData);
**
**  Returns:        status:
**                      MT_OK            - No errors
**                      MT_INV_HANDLE    - Invalid tuner handle
**                      MT_ARG_NULL      - Null pointer argument passed
**
**  Dependencies:   USERS MUST CALL MT2063_Open() FIRST!
**
**                  The hUserData parameter is a user-specific argument
**                  that is stored internally with the other tuner-
**                  specific information.
**
**                  For example, if additional arguments are needed
**                  for the user to identify the device communicating
**                  with the tuner, this argument can be used to supply
**                  the necessary information.
**
**                  The hUserData parameter is initialized in the tuner's
**                  Open function to NULL.
**
**  See Also:       MT2063_Open
**
**  Revision History:
**
**   SCR      Date      Author  Description
**  -------------------------------------------------------------------------
**   138   06-19-2007    DAD    Ver 1.00: Initial, derived from mt2067_b.
**
****************************************************************************/
UData_t MT2063_GetUserData(Handle_t h,
                           Handle_t* hUserData);


/******************************************************************************
**
**  Name: MT2063_ReInit
**
**  Description:    Initialize the tuner's register values.
**
**  Parameters:     h           - Tuner handle (returned by MT2063_Open)
**
**  Returns:        status:
**                      MT_OK            - No errors
**                      MT_TUNER_ID_ERR   - Tuner Part/Rev code mismatch
**                      MT_INV_HANDLE    - Invalid tuner handle
**                      MT_COMM_ERR      - Serial bus communications error
**
**  Dependencies:   MT_ReadSub  - Read byte(s) of data from the two-wire bus
**                  MT_WriteSub - Write byte(s) of data to the two-wire bus
**
**  Revision History:
**
**   SCR      Date      Author  Description
**  -------------------------------------------------------------------------
**   138   06-19-2007    DAD    Ver 1.00: Initial, derived from mt2067_b.
**
******************************************************************************/
UData_t MT2063_ReInit(Handle_t h);


/******************************************************************************
**
**  Name: MT2063_SetGPIO
**
**  Description:    Modify the MT2063 GPIO value.
**
**  Parameters:     h            - Open handle to the tuner (from MT2063_Open).
**                  gpio_id      - Selects GPIO0, GPIO1, or GPIO2
**                  attr         - Selects input readback, I/O direction or
**                                 output value
**                  value        - value to set GPIO pin
**
**  Usage:          status = MT2063_SetGPIO(hMT2063, MT2063_GPIO1, MT2063_GPIO_OUT, 1);
**
**  Returns:        status:
**                      MT_OK            - No errors
**                      MT_COMM_ERR      - Serial bus communications error
**                      MT_INV_HANDLE    - Invalid tuner handle
**
**  Dependencies:   MT_WriteSub - Write byte(s) of data to the two-wire-bus
**
**  Revision History:
**
**   SCR      Date      Author  Description
**  -------------------------------------------------------------------------
**   138   06-19-2007    DAD    Ver 1.00: Initial, derived from mt2067_b.
**
******************************************************************************/
UData_t MT2063_SetGPIO(Handle_t h, MT2063_GPIO_ID gpio_id, MT2063_GPIO_Attr attr, UData_t value);


/****************************************************************************
**
**  Name: MT2063_SetParam
**
**  Description:    Sets a tuning algorithm parameter.
**
**                  This function provides access to the internals of the
**                  tuning algorithm.  You can override many of the tuning
**                  algorithm defaults using this function.
**
**  Parameters:     h           - Tuner handle (returned by MT2063_Open)
**                  param       - Tuning algorithm parameter
**                                (see enum MT2063_Param)
**                  nValue      - value to be set
**
**                  param                   Description
**                  ----------------------  --------------------------------
**                  MT2063_SRO_FREQ         crystal frequency
**                  MT2063_STEPSIZE         minimum tuning step size
**                  MT2063_LO1_FREQ           LO1 frequency
**                  MT2063_LO1_STEPSIZE     LO1 minimum step size
**                  MT2063_LO1_FRACN_AVOID  LO1 FracN keep-out region
**                  MT2063_IF1_REQUEST      Requested 1st IF
**                  MT2063_ZIF_BW           zero-IF bandwidth
**                  MT2063_LO2_FREQ           LO2 frequency
**                  MT2063_LO2_STEPSIZE     LO2 minimum step size
**                  MT2063_LO2_FRACN_AVOID  LO2 FracN keep-out region
**                  MT2063_OUTPUT_FREQ      output center frequency
**                  MT2063_OUTPUT_BW        output bandwidth
**                  MT2063_LO_SEPARATION    min inter-tuner LO separation
**                  MT2063_MAX_HARM1        max # of intra-tuner harmonics
**                  MT2063_MAX_HARM2        max # of inter-tuner harmonics
**                  MT2063_RCVR_MODE          Predefined modes
**                  MT2063_LNA_RIN            Set LNA Rin (*)
**                  MT2063_LNA_TGT            Set target power level at LNA (*)
**                  MT2063_PD1_TGT            Set target power level at PD1 (*)
**                  MT2063_PD2_TGT            Set target power level at PD2 (*)
**                  MT2063_ACLNA_MAX          LNA attenuator limit (*)
**                  MT2063_ACRF_MAX           RF attenuator limit (*)
**                  MT2063_ACFIF_MAX          FIF attenuator limit (*)
**                  MT2063_DNC_OUTPUT_ENABLE  DNC output selection
**                  MT2063_VGAGC              VGA gain code
**                  MT2063_VGAOI              VGA output current
**                  MT2063_TAGC               TAGC setting
**                  MT2063_AMPGC              AMP gain code
**                  MT2063_AVOID_DECT         Avoid DECT Frequencies
**                  MT2063_CTFILT_SW          Cleartune filter selection
**
**                  (*) This parameter is set by MT2063_RCVR_MODE, do not call
**                      additionally.
**
**  Usage:          status |= MT2063_SetParam(hMT2063,
**                                            MT2063_STEPSIZE,
**                                            50000);
**
**  Returns:        status:
**                      MT_OK            - No errors
**                      MT_INV_HANDLE    - Invalid tuner handle
**                      MT_ARG_NULL      - Null pointer argument passed
**                      MT_ARG_RANGE     - Invalid parameter requested
**                                         or set value out of range
**                                         or non-writable parameter
**
**  Dependencies:   USERS MUST CALL MT2063_Open() FIRST!
**
**  See Also:       MT2063_GetParam, MT2063_Open
**
**  Revision History:
**
**   SCR      Date      Author  Description
**  -------------------------------------------------------------------------
**   138   06-19-2007    DAD    Ver 1.00: Initial, derived from mt2067_b.
**   154   09-13-2007    RSK    Ver 1.05: Get/SetParam changes for LOx_FREQ
**         10-31-2007    PINZ   Ver 1.08: Get/SetParam add VGAGC, VGAOI, AMPGC, TAGC
**         04-18-2008    PINZ   Ver 1.15: Add SetParam LNARIN & PDxTGT
**                                        Split SetParam up to ACLNA / ACLNA_MAX
**                                        removed ACLNA_INRC/DECR (+RF & FIF)
**                                        removed GCUAUTO / BYPATNDN/UP
**   175 I 06-06-2008    PINZ   Ver 1.16: Add control to avoid US DECT freqs.
**   175 I 06-19-2008    RSK    Ver 1.17: Refactor DECT control to SpurAvoid.
**         06-24-2008    PINZ   Ver 1.18: Add Get/SetParam CTFILT_SW
**
****************************************************************************/
UData_t MT2063_SetParam(Handle_t     h,
                        MT2063_Param param,
                        UData_t      nValue);


/****************************************************************************
**
**  Name: MT2063_SetPowerMaskBits
**
**  Description:    Sets the power-down mask bits for various sections of
**                  the MT2063
**
**  Parameters:     h           - Tuner handle (returned by MT2063_Open)
**                  Bits        - Mask bits to be set.
**
**                    MT2063_ALL_SD   All shutdown bits for this tuner
**
**                    MT2063_REG_SD   Shutdown regulator
**                    MT2063_SRO_SD   Shutdown SRO
**                    MT2063_AFC_SD   Shutdown AFC A/D
**                    MT2063_PD_SD    Enable power detector shutdown
**                    MT2063_PDADC_SD Enable power detector A/D shutdown
**                    MT2063_VCO_SD   Enable VCO shutdown VCO
**                    MT2063_UPC_SD   Enable upconverter shutdown
**                    MT2063_DNC_SD   Enable downconverter shutdown
**                    MT2063_VGA_SD   Enable VGA shutdown
**
**  Returns:        status:
**                      MT_OK            - No errors
**                      MT_INV_HANDLE    - Invalid tuner handle
**                      MT_COMM_ERR      - Serial bus communications error
**
**  Dependencies:   USERS MUST CALL MT2063_Open() FIRST!
**
**  Revision History:
**
**   SCR      Date      Author  Description
**  -------------------------------------------------------------------------
**   138   06-19-2007    DAD    Ver 1.00: Initial, derived from mt2067_b.
**
****************************************************************************/
UData_t MT2063_SetPowerMaskBits(Handle_t h, MT2063_Mask_Bits Bits);


/****************************************************************************
**
**  Name: MT2063_ClearPowerMaskBits
**
**  Description:    Clears the power-down mask bits for various sections of
**                  the MT2063
**
**  Parameters:     h           - Tuner handle (returned by MT2063_Open)
**                  Bits        - Mask bits to be cleared.
**
**                    MT2063_ALL_SD   All shutdown bits for this tuner
**
**                    MT2063_REG_SD   Shutdown regulator
**                    MT2063_SRO_SD   Shutdown SRO
**                    MT2063_AFC_SD   Shutdown AFC A/D
**                    MT2063_PD_SD    Enable power detector shutdown
**                    MT2063_PDADC_SD Enable power detector A/D shutdown
**                    MT2063_VCO_SD   Enable VCO shutdown VCO
**                    MT2063_UPC_SD   Enable upconverter shutdown
**                    MT2063_DNC_SD   Enable downconverter shutdown
**                    MT2063_VGA_SD   Enable VGA shutdown
**
**  Returns:        status:
**                      MT_OK            - No errors
**                      MT_INV_HANDLE    - Invalid tuner handle
**                      MT_COMM_ERR      - Serial bus communications error
**
**  Dependencies:   USERS MUST CALL MT2063_Open() FIRST!
**
**  Revision History:
**
**   SCR      Date      Author  Description
**  -------------------------------------------------------------------------
**   138   06-19-2007    DAD    Ver 1.00: Initial, derived from mt2067_b.
**
****************************************************************************/
UData_t MT2063_ClearPowerMaskBits(Handle_t h, MT2063_Mask_Bits Bits);


/****************************************************************************
**
**  Name: MT2063_GetPowerMaskBits
**
**  Description:    Returns a mask of the enabled power shutdown bits
**
**  Parameters:     h           - Tuner handle (returned by MT2063_Open)
**                  Bits        - Mask bits to currently set.
**
**                    MT2063_REG_SD   Shutdown regulator
**                    MT2063_SRO_SD   Shutdown SRO
**                    MT2063_AFC_SD   Shutdown AFC A/D
**                    MT2063_PD_SD    Enable power detector shutdown
**                    MT2063_PDADC_SD Enable power detector A/D shutdown
**                    MT2063_VCO_SD   Enable VCO shutdown VCO
**                    MT2063_UPC_SD   Enable upconverter shutdown
**                    MT2063_DNC_SD   Enable downconverter shutdown
**                    MT2063_VGA_SD   Enable VGA shutdown
**
**  Returns:        status:
**                      MT_OK            - No errors
**                      MT_INV_HANDLE    - Invalid tuner handle
**                      MT_ARG_NULL      - Output argument is NULL
**                      MT_COMM_ERR      - Serial bus communications error
**
**  Dependencies:   USERS MUST CALL MT2063_Open() FIRST!
**
**  Revision History:
**
**   SCR      Date      Author  Description
**  -------------------------------------------------------------------------
**   138   06-19-2007    DAD    Ver 1.00: Initial, derived from mt2067_b.
**
****************************************************************************/
UData_t MT2063_GetPowerMaskBits(Handle_t h, MT2063_Mask_Bits *Bits);


/****************************************************************************
**
**  Name: MT2063_EnableExternalShutdown
**
**  Description:    Enables or disables the operation of the external
**                  shutdown pin
**
**  Parameters:     h           - Tuner handle (returned by MT2063_Open)
**                  Enabled     - 0 = disable the pin, otherwise enable it
**
**  Returns:        status:
**                      MT_OK            - No errors
**                      MT_INV_HANDLE    - Invalid tuner handle
**                      MT_COMM_ERR      - Serial bus communications error
**
**  Dependencies:   USERS MUST CALL MT2063_Open() FIRST!
**
**  Revision History:
**
**   SCR      Date      Author  Description
**  -------------------------------------------------------------------------
**   138   06-19-2007    DAD    Ver 1.00: Initial, derived from mt2067_b.
**
****************************************************************************/
UData_t MT2063_EnableExternalShutdown(Handle_t h, U8Data Enabled);


/****************************************************************************
**
**  Name: MT2063_SoftwareShutdown
**
**  Description:    Enables or disables software shutdown function.  When
**                  Shutdown==1, any section whose power mask is set will be
**                  shutdown.
**
**  Parameters:     h           - Tuner handle (returned by MT2063_Open)
**                  Shutdown    - 1 = shutdown the masked sections, otherwise
**                                power all sections on
**
**  Returns:        status:
**                      MT_OK            - No errors
**                      MT_INV_HANDLE    - Invalid tuner handle
**                      MT_COMM_ERR      - Serial bus communications error
**
**  Dependencies:   USERS MUST CALL MT2063_Open() FIRST!
**
**  Revision History:
**
**   SCR      Date      Author  Description
**  -------------------------------------------------------------------------
**   138   06-19-2007    DAD    Ver 1.00: Initial, derived from mt2067_b.
**
****************************************************************************/
UData_t MT2063_SoftwareShutdown(Handle_t h, U8Data Shutdown);


/****************************************************************************
**
**  Name: MT2063_SetExtSRO
**
**  Description:    Sets the external SRO driver.
**
**  Parameters:     h           - Tuner handle (returned by MT2063_Open)
**                  Ext_SRO_Setting - external SRO drive setting
**
**       (default)    MT2063_EXT_SRO_OFF  - ext driver off
**                    MT2063_EXT_SRO_BY_1 - ext driver = SRO frequency
**                    MT2063_EXT_SRO_BY_2 - ext driver = SRO/2 frequency
**                    MT2063_EXT_SRO_BY_4 - ext driver = SRO/4 frequency
**
**  Returns:        status:
**                      MT_OK            - No errors
**                      MT_COMM_ERR      - Serial bus communications error
**                      MT_INV_HANDLE    - Invalid tuner handle
**
**  Dependencies:   USERS MUST CALL MT2063_Open() FIRST!
**
**                  The Ext_SRO_Setting settings default to OFF
**                  Use this function if you need to override the default
**
**  Revision History:
**
**   SCR      Date      Author  Description
**  -------------------------------------------------------------------------
**   138   06-19-2007    DAD    Ver 1.00: Initial, derived from mt2067_b.
**
****************************************************************************/
UData_t MT2063_SetExtSRO(Handle_t h, MT2063_Ext_SRO Ext_SRO_Setting);


/****************************************************************************
**
**  Name: MT2063_SetReg
**
**  Description:    Sets an MT2063 register.
**
**  Parameters:     h           - Tuner handle (returned by MT2063_Open)
**                  reg         - MT2063 register/subaddress location
**                  val         - MT2063 register/subaddress value
**
**  Returns:        status:
**                      MT_OK            - No errors
**                      MT_COMM_ERR      - Serial bus communications error
**                      MT_INV_HANDLE    - Invalid tuner handle
**                      MT_ARG_RANGE     - Argument out of range
**
**  Dependencies:   USERS MUST CALL MT2063_Open() FIRST!
**
**                  Use this function if you need to override a default
**                  register value
**
**  Revision History:
**
**   SCR      Date      Author  Description
**  -------------------------------------------------------------------------
**   138   06-19-2007    DAD    Ver 1.00: Initial, derived from mt2067_b.
**
****************************************************************************/
UData_t MT2063_SetReg(Handle_t h,
                      U8Data   reg,
                      U8Data   val);


#if defined( __cplusplus )
}
#endif

#endif























// Microtune source code - mt_spuravoid.h



/*****************************************************************************
**
**  Name: mt_spuravoid.h
**
**  Copyright 2006-2008 Microtune, Inc. All Rights Reserved
**
**  This source code file contains confidential information and/or trade
**  secrets of Microtune, Inc. or its affiliates and is subject to the
**  terms of your confidentiality agreement with Microtune, Inc. or one of
**  its affiliates, as applicable.
**
*****************************************************************************/

/*****************************************************************************
**
**  Name: mt_spuravoid.h
**
**  Description:    Implements spur avoidance for MicroTuners
**
**  References:     None
**
**  Exports:        None
**
**  CVS ID:         $Id: mt_spuravoid.h,v 1.4 2008/11/05 13:46:20 software Exp $
**  CVS Source:     $Source: /export/vol0/cvsroot/software/tuners/MT2063/mt_spuravoid.h,v $
**
**  Revision History:
**
**   SCR      Date      Author  Description
**  -------------------------------------------------------------------------
**   N/A   03-25-2004    DAD    Original
**   083   02-08-2005    JWS    Added separate version number for expected
**                              version of MT_SpurAvoid.h
**   083   02-08-2005    JWS    Added function to return the version of the
**                              MT_AvoidSpurs source file.
**   098   04-12-2005    DAD    Increased MAX_ZONES from 32 to 48.  32 is
**                              sufficient for the default avoidance params.
**   103   01-31-2005    DAD    In MT_AddExclZone(), if the range
**                              (f_min, f_max) < 0, ignore the entry.
**   195 M 06-20-2008    RSK    Ver 1.21: Refactoring to place DECT frequency
**                              avoidance (US and Euro) in 'SpurAvoidance'.
**   199   08-06-2008    JWS    Ver 1.22: Added 0x1x1 spur frequency calculations
**                              and indicate success of AddExclZone operation.
**   200   08-07-2008    JWS    Ver 1.23: Moved definition of DECT avoid constants
**                              so users could access them more easily.
**
*****************************************************************************/
#if !defined(__MT2063_SPURAVOID_H)
#define __MT2063_SPURAVOID_H

//#include "mt_userdef.h"

#if defined( __cplusplus )
extern "C"                     /* Use "C" external linkage                  */
{
#endif

/*
**  Constant defining the version of the following structure
**  and therefore the API for this code.
**
**  When compiling the tuner driver, the preprocessor will
**  check against this version number to make sure that
**  it matches the version that the tuner driver knows about.
*/
/* Version 010201 => 1.21 */
#define MT2063_AVOID_SPURS_INFO_VERSION 010201


#define MT2063_MAX_ZONES 48

struct MT2063_ExclZone_t;

struct MT2063_ExclZone_t
{
    UData_t         min_;
    UData_t         max_;
    struct MT2063_ExclZone_t*  next_;
};
#ifndef _MT_DECT_Avoid_Type_DEFINE_
#define _MT_DECT_Avoid_Type_DEFINE_
//
// DECT Frequency Avoidance:
// These constants are used to avoid interference from DECT frequencies.
//
typedef enum
{
    MT_NO_DECT_AVOIDANCE = 0x00000000,     // Do not create DECT exclusion zones.     
    MT_AVOID_US_DECT     = 0x00000001,     // Avoid US DECT frequencies.              
    MT_AVOID_EURO_DECT   = 0x00000002,     // Avoid European DECT frequencies.        
    MT_AVOID_BOTH                          // Avoid both regions. Not typically used. 
} MT_DECT_Avoid_Type;
#endif

/*
**  Structure of data needed for Spur Avoidance
*/
typedef struct
{
    UData_t nAS_Algorithm;
    UData_t f_ref;
    UData_t f_in;
    UData_t f_LO1;
    UData_t f_if1_Center;
    UData_t f_if1_Request;
    UData_t f_if1_bw;
    UData_t f_LO2;
    UData_t f_out;
    UData_t f_out_bw;
    UData_t f_LO1_Step;
    UData_t f_LO2_Step;
    UData_t f_LO1_FracN_Avoid;
    UData_t f_LO2_FracN_Avoid;
    UData_t f_zif_bw;
    UData_t f_min_LO_Separation;
    UData_t maxH1;
    UData_t maxH2;
    MT_DECT_Avoid_Type  avoidDECT;
    UData_t bSpurPresent;
    UData_t bSpurAvoided;
    UData_t nSpursFound;
    UData_t nZones;
    struct MT2063_ExclZone_t* freeZones;
    struct MT2063_ExclZone_t* usedZones;
    struct MT2063_ExclZone_t MT_ExclZones[MT2063_MAX_ZONES];
} MT2063_AvoidSpursData_t;

UData_t MT2063_RegisterTuner(MT2063_AvoidSpursData_t* pAS_Info);

void MT2063_UnRegisterTuner(MT2063_AvoidSpursData_t* pAS_Info);

void MT2063_ResetExclZones(MT2063_AvoidSpursData_t* pAS_Info);

UData_t MT2063_AddExclZone(MT2063_AvoidSpursData_t* pAS_Info,
                       UData_t f_min,
                       UData_t f_max);

UData_t MT2063_ChooseFirstIF(MT2063_AvoidSpursData_t* pAS_Info);

UData_t MT2063_AvoidSpurs(Handle_t h,
                     MT2063_AvoidSpursData_t* pAS_Info);

UData_t MT2063_AvoidSpursVersion(void);

#if defined( __cplusplus )
}
#endif

#endif























// The following context is MT2063 tuner API source code





/**

@file

@brief   MT2063 tuner module declaration

One can manipulate MT2063 tuner through MT2063 module.
MT2063 module is derived from tuner module.

*/


#include "tuner_base.h"





// Definitions

// MT2063 API option
#define MT2063_CNT		4

// MT2063 macro
//#define abs(x) ((x) < 0 ? -(x) : (x))

// MT2063 bandwidth shift
#define MT2063_BANDWIDTH_SHIFT_HZ		1000000



// Bandwidth modes
enum MT2063_BANDWIDTH_MODE
{
	MT2063_BANDWIDTH_6MHZ = 6000000,
	MT2063_BANDWIDTH_7MHZ = 7000000,
	MT2063_BANDWIDTH_8MHZ = 8000000,
};


// Standard modes
enum MT2063_STANDARD_MODE
{
	MT2063_STANDARD_DVBT,
	MT2063_STANDARD_QAM,
};


// VGAGC settings
enum MT2063_VGAGC_SETTINGS
{
	MT2063_VGAGC_0X0 = 0x0,
	MT2063_VGAGC_0X1 = 0x1,
	MT2063_VGAGC_0X3 = 0x3,
};





// Builder
void
BuildMt2063Module(
	TUNER_MODULE **ppTuner,
	TUNER_MODULE *pTunerModuleMemory,
	BASE_INTERFACE_MODULE *pBaseInterfaceModuleMemory,
	I2C_BRIDGE_MODULE *pI2cBridgeModuleMemory,
	unsigned char DeviceAddr,
	int StandardMode,
	unsigned long IfVgaGainControl
	);





// Manipulaing functions
void
mt2063_GetTunerType(
	TUNER_MODULE *pTuner,
	int *pTunerType
	);

void
mt2063_GetDeviceAddr(
	TUNER_MODULE *pTuner,
	unsigned char *pDeviceAddr
	);

int
mt2063_Initialize(
	TUNER_MODULE *pTuner
	);

int
mt2063_SetRfFreqHz(
	TUNER_MODULE *pTuner,
	unsigned long RfFreqHz
	);

int
mt2063_GetRfFreqHz(
	TUNER_MODULE *pTuner,
	unsigned long *pRfFreqHz
	);





// Extra manipulaing functions
int
mt2063_OpenHandle(
	TUNER_MODULE *pTuner
	);

int
mt2063_CloseHandle(
	TUNER_MODULE *pTuner
	);

void
mt2063_GetHandle(
	TUNER_MODULE *pTuner,
	void **pDeviceHandle
	);

int
mt2063_SetIfFreqHz(
	TUNER_MODULE *pTuner,
	unsigned long IfFreqHz
	);

int
mt2063_GetIfFreqHz(
	TUNER_MODULE *pTuner,
	unsigned long *pIfFreqHz
	);

int
mt2063_SetBandwidthHz(
	TUNER_MODULE *pTuner,
	unsigned long BandwidthHz
	);

int
mt2063_GetBandwidthHz(
	TUNER_MODULE *pTuner,
	unsigned long *pBandwidthHz
	);















#endif
