/*
 * Decompiled with CFR 0.152.
 */
package org.elasticsearch.common.geo;

import java.io.IOException;
import org.apache.lucene.spatial.prefix.tree.GeohashPrefixTree;
import org.apache.lucene.util.SloppyMath;
import org.elasticsearch.ElasticsearchParseException;
import org.elasticsearch.common.geo.GeoPoint;
import org.elasticsearch.common.unit.DistanceUnit;
import org.elasticsearch.common.xcontent.XContentParser;

public class GeoUtils {
    public static final double MAX_LAT = 90.0;
    public static final double MIN_LAT = -90.0;
    public static final double MAX_LON = 180.0;
    public static final double MIN_LON = -180.0;
    public static final String LATITUDE = "lat";
    public static final String LONGITUDE = "lon";
    public static final String GEOHASH = "geohash";
    public static final double EARTH_SEMI_MAJOR_AXIS = 6378137.0;
    public static final double EARTH_SEMI_MINOR_AXIS = 6356752.314245;
    public static final double EARTH_MEAN_RADIUS = 6371008.7714;
    public static final double EARTH_AXIS_RATIO = 0.9966471893352243;
    public static final double EARTH_EQUATOR = 4.007501668557849E7;
    public static final double EARTH_POLAR_DISTANCE = 1.9970326371122006E7;
    public static final double TOLERANCE = 1.0E-6;

    public static double maxRadialDistance(GeoPoint center, double initialRadius) {
        double maxRadius = GeoUtils.maxRadialDistanceMeters(center.lat(), center.lon());
        return Math.min(initialRadius, maxRadius);
    }

    public static boolean isValidLatitude(double latitude) {
        return !Double.isNaN(latitude) && !Double.isInfinite(latitude) && !(latitude < -90.0) && !(latitude > 90.0);
    }

    public static boolean isValidLongitude(double longitude) {
        return !Double.isNaN(longitude) && !Double.isNaN(longitude) && !(longitude < -180.0) && !(longitude > 180.0);
    }

    public static double geoHashCellWidth(int level) {
        assert (level >= 0);
        return 4.007501668557849E7 / (double)(1L << (level + 1) / 2 * 3 + level / 2 * 2);
    }

    public static double quadTreeCellWidth(int level) {
        assert (level >= 0);
        return 4.007501668557849E7 / (double)(1L << level);
    }

    public static double geoHashCellHeight(int level) {
        assert (level >= 0);
        return 1.9970326371122006E7 / (double)(1L << (level + 1) / 2 * 2 + level / 2 * 3);
    }

    public static double quadTreeCellHeight(int level) {
        assert (level >= 0);
        return 1.9970326371122006E7 / (double)(1L << level);
    }

    public static double geoHashCellSize(int level) {
        assert (level >= 0);
        double w = GeoUtils.geoHashCellWidth(level);
        double h = GeoUtils.geoHashCellHeight(level);
        return Math.sqrt(w * w + h * h);
    }

    public static double quadTreeCellSize(int level) {
        assert (level >= 0);
        return Math.sqrt(2.0048208977185252E15) / (double)(1L << level);
    }

    public static int quadTreeLevelsForPrecision(double meters) {
        int level;
        assert (meters >= 0.0);
        if (meters == 0.0) {
            return 50;
        }
        double ratio = 1.4983235946676121;
        double width = Math.sqrt(meters * meters / 2.244973594337675);
        long part = Math.round(Math.ceil(4.007501668557849E7 / width));
        return part <= 1L << (level = 64 - Long.numberOfLeadingZeros(part) - 1) ? level : level + 1;
    }

    public static int quadTreeLevelsForPrecision(String distance) {
        return GeoUtils.quadTreeLevelsForPrecision(DistanceUnit.METERS.parse(distance, DistanceUnit.DEFAULT));
    }

    public static int geoHashLevelsForPrecision(double meters) {
        int full;
        assert (meters >= 0.0);
        if (meters == 0.0) {
            return GeohashPrefixTree.getMaxLevelsPossible();
        }
        double ratio = 1.4983235946676121;
        double width = Math.sqrt(meters * meters / 2.244973594337675);
        double part = Math.ceil(4.007501668557849E7 / width);
        if (part == 1.0) {
            return 1;
        }
        int bits = (int)Math.round(Math.ceil(Math.log(part) / Math.log(2.0)));
        int left = bits - (full = bits / 5) * 5;
        int even = full + (left > 0 ? 1 : 0);
        int odd = full + (left > 3 ? 1 : 0);
        return even + odd;
    }

    public static int geoHashLevelsForPrecision(String distance) {
        return GeoUtils.geoHashLevelsForPrecision(DistanceUnit.METERS.parse(distance, DistanceUnit.DEFAULT));
    }

    public static double normalizeLon(double lon) {
        return GeoUtils.centeredModulus(lon, 360.0);
    }

    public static double normalizeLat(double lat) {
        if ((lat = GeoUtils.centeredModulus(lat, 360.0)) < -90.0) {
            lat = -180.0 - lat;
        } else if (lat > 90.0) {
            lat = 180.0 - lat;
        }
        return lat;
    }

    public static void normalizePoint(GeoPoint point) {
        GeoUtils.normalizePoint(point, true, true);
    }

    public static void normalizePoint(GeoPoint point, boolean normLat, boolean normLon) {
        double[] pt = new double[]{point.lon(), point.lat()};
        GeoUtils.normalizePoint(pt, normLon, normLat);
        point.reset(pt[1], pt[0]);
    }

    public static void normalizePoint(double[] lonLat) {
        GeoUtils.normalizePoint(lonLat, true, true);
    }

    public static void normalizePoint(double[] lonLat, boolean normLon, boolean normLat) {
        assert (lonLat != null && lonLat.length == 2);
        normLat = normLat && (lonLat[1] > 90.0 || lonLat[1] < -90.0);
        boolean bl = normLon = normLon && (lonLat[0] > 180.0 || lonLat[0] < -180.0);
        if (normLat) {
            lonLat[1] = GeoUtils.centeredModulus(lonLat[1], 360.0);
            boolean shift = true;
            if (lonLat[1] < -90.0) {
                lonLat[1] = -180.0 - lonLat[1];
            } else if (lonLat[1] > 90.0) {
                lonLat[1] = 180.0 - lonLat[1];
            } else {
                shift = false;
            }
            if (shift) {
                lonLat[0] = normLon ? lonLat[0] + 180.0 : lonLat[0] + (GeoUtils.normalizeLon(lonLat[0]) > 0.0 ? -180.0 : 180.0);
            }
        }
        if (normLon) {
            lonLat[0] = GeoUtils.centeredModulus(lonLat[0], 360.0);
        }
    }

    private static double centeredModulus(double dividend, double divisor) {
        double rtn = dividend % divisor;
        if (rtn <= 0.0) {
            rtn += divisor;
        }
        if (rtn > divisor / 2.0) {
            rtn -= divisor;
        }
        return rtn;
    }

    public static GeoPoint parseGeoPoint(XContentParser parser) throws IOException, ElasticsearchParseException {
        return GeoUtils.parseGeoPoint(parser, new GeoPoint());
    }

    public static GeoPoint parseGeoPoint(XContentParser parser, GeoPoint point) throws IOException, ElasticsearchParseException {
        double lat = Double.NaN;
        double lon = Double.NaN;
        String geohash = null;
        NumberFormatException numberFormatException = null;
        if (parser.currentToken() == XContentParser.Token.START_OBJECT) {
            block10: while (parser.nextToken() != XContentParser.Token.END_OBJECT) {
                if (parser.currentToken() == XContentParser.Token.FIELD_NAME) {
                    String field = parser.currentName();
                    if (LATITUDE.equals(field)) {
                        parser.nextToken();
                        switch (parser.currentToken()) {
                            case VALUE_NUMBER: 
                            case VALUE_STRING: {
                                try {
                                    lat = parser.doubleValue(true);
                                }
                                catch (NumberFormatException e) {
                                    numberFormatException = e;
                                }
                                continue block10;
                            }
                        }
                        throw new ElasticsearchParseException("latitude must be a number", new Object[0]);
                    }
                    if (LONGITUDE.equals(field)) {
                        parser.nextToken();
                        switch (parser.currentToken()) {
                            case VALUE_NUMBER: 
                            case VALUE_STRING: {
                                try {
                                    lon = parser.doubleValue(true);
                                }
                                catch (NumberFormatException e) {
                                    numberFormatException = e;
                                }
                                continue block10;
                            }
                        }
                        throw new ElasticsearchParseException("longitude must be a number", new Object[0]);
                    }
                    if (GEOHASH.equals(field)) {
                        if (parser.nextToken() == XContentParser.Token.VALUE_STRING) {
                            geohash = parser.text();
                            continue;
                        }
                        throw new ElasticsearchParseException("geohash must be a string", new Object[0]);
                    }
                    throw new ElasticsearchParseException("field must be either [{}], [{}] or [{}]", LATITUDE, LONGITUDE, GEOHASH);
                }
                throw new ElasticsearchParseException("token [{}] not allowed", new Object[]{parser.currentToken()});
            }
            if (geohash != null) {
                if (!Double.isNaN(lat) || !Double.isNaN(lon)) {
                    throw new ElasticsearchParseException("field must be either lat/lon or geohash", new Object[0]);
                }
                return point.resetFromGeoHash(geohash);
            }
            if (numberFormatException != null) {
                throw new ElasticsearchParseException("[{}] and [{}] must be valid double values", numberFormatException, LATITUDE, LONGITUDE);
            }
            if (Double.isNaN(lat)) {
                throw new ElasticsearchParseException("field [{}] missing", LATITUDE);
            }
            if (Double.isNaN(lon)) {
                throw new ElasticsearchParseException("field [{}] missing", LONGITUDE);
            }
            return point.reset(lat, lon);
        }
        if (parser.currentToken() == XContentParser.Token.START_ARRAY) {
            int element = 0;
            while (parser.nextToken() != XContentParser.Token.END_ARRAY) {
                if (parser.currentToken() == XContentParser.Token.VALUE_NUMBER) {
                    if (++element == 1) {
                        lon = parser.doubleValue();
                        continue;
                    }
                    if (element == 2) {
                        lat = parser.doubleValue();
                        continue;
                    }
                    throw new ElasticsearchParseException("only two values allowed", new Object[0]);
                }
                throw new ElasticsearchParseException("numeric value expected", new Object[0]);
            }
            return point.reset(lat, lon);
        }
        if (parser.currentToken() == XContentParser.Token.VALUE_STRING) {
            String data = parser.text();
            return GeoUtils.parseGeoPoint(data, point);
        }
        throw new ElasticsearchParseException("geo_point expected", new Object[0]);
    }

    public static GeoPoint parseGeoPoint(String data, GeoPoint point) {
        int comma = data.indexOf(44);
        if (comma > 0) {
            double lat = Double.parseDouble(data.substring(0, comma).trim());
            double lon = Double.parseDouble(data.substring(comma + 1).trim());
            return point.reset(lat, lon);
        }
        return point.resetFromGeoHash(data);
    }

    public static double maxRadialDistanceMeters(double centerLat, double centerLon) {
        if (Math.abs(centerLat) == 90.0) {
            return SloppyMath.haversinMeters((double)centerLat, (double)centerLon, (double)0.0, (double)centerLon);
        }
        return SloppyMath.haversinMeters((double)centerLat, (double)centerLon, (double)centerLat, (double)((180.0 + centerLon) % 360.0));
    }

    public static double arcDistance(double lat1, double lon1, double lat2, double lon2) {
        return SloppyMath.haversinMeters((double)lat1, (double)lon1, (double)lat2, (double)lon2);
    }

    public static double planeDistance(double lat1, double lon1, double lat2, double lon2) {
        double x = (lon2 - lon1) * (Math.PI / 180) * Math.cos((lat2 + lat1) / 2.0 * (Math.PI / 180));
        double y = (lat2 - lat1) * (Math.PI / 180);
        return Math.sqrt(x * x + y * y) * 6371008.7714;
    }

    private GeoUtils() {
    }
}

