/*
 * Decompiled with CFR 0.152.
 */
package org.elasticsearch.indices.recovery;

import java.io.IOException;
import java.util.concurrent.atomic.AtomicLong;
import java.util.concurrent.atomic.AtomicReference;
import org.apache.logging.log4j.message.ParameterizedMessage;
import org.apache.lucene.index.IndexNotFoundException;
import org.apache.lucene.store.AlreadyClosedException;
import org.apache.lucene.store.RateLimiter;
import org.elasticsearch.ElasticsearchException;
import org.elasticsearch.ElasticsearchTimeoutException;
import org.elasticsearch.ExceptionsHelper;
import org.elasticsearch.action.support.PlainActionFuture;
import org.elasticsearch.cluster.ClusterState;
import org.elasticsearch.cluster.ClusterStateObserver;
import org.elasticsearch.cluster.node.DiscoveryNode;
import org.elasticsearch.cluster.service.ClusterService;
import org.elasticsearch.common.Nullable;
import org.elasticsearch.common.component.AbstractComponent;
import org.elasticsearch.common.inject.Inject;
import org.elasticsearch.common.settings.Settings;
import org.elasticsearch.common.unit.ByteSizeValue;
import org.elasticsearch.common.unit.TimeValue;
import org.elasticsearch.common.util.CancellableThreads;
import org.elasticsearch.common.util.concurrent.AbstractRunnable;
import org.elasticsearch.index.engine.RecoveryEngineException;
import org.elasticsearch.index.mapper.MapperException;
import org.elasticsearch.index.shard.IllegalIndexShardStateException;
import org.elasticsearch.index.shard.IndexEventListener;
import org.elasticsearch.index.shard.IndexShard;
import org.elasticsearch.index.shard.ShardId;
import org.elasticsearch.index.shard.ShardNotFoundException;
import org.elasticsearch.index.shard.TranslogRecoveryPerformer;
import org.elasticsearch.index.store.Store;
import org.elasticsearch.indices.recovery.DelayRecoveryException;
import org.elasticsearch.indices.recovery.RecoveriesCollection;
import org.elasticsearch.indices.recovery.RecoveryCleanFilesRequest;
import org.elasticsearch.indices.recovery.RecoveryFailedException;
import org.elasticsearch.indices.recovery.RecoveryFileChunkRequest;
import org.elasticsearch.indices.recovery.RecoveryFilesInfoRequest;
import org.elasticsearch.indices.recovery.RecoveryFinalizeRecoveryRequest;
import org.elasticsearch.indices.recovery.RecoveryPrepareForTranslogOperationsRequest;
import org.elasticsearch.indices.recovery.RecoveryResponse;
import org.elasticsearch.indices.recovery.RecoverySettings;
import org.elasticsearch.indices.recovery.RecoveryState;
import org.elasticsearch.indices.recovery.RecoveryTarget;
import org.elasticsearch.indices.recovery.RecoveryTranslogOperationsRequest;
import org.elasticsearch.indices.recovery.RecoveryWaitForClusterStateRequest;
import org.elasticsearch.indices.recovery.StartRecoveryRequest;
import org.elasticsearch.node.NodeClosedException;
import org.elasticsearch.threadpool.ThreadPool;
import org.elasticsearch.transport.ConnectTransportException;
import org.elasticsearch.transport.FutureTransportResponseHandler;
import org.elasticsearch.transport.TransportChannel;
import org.elasticsearch.transport.TransportRequestHandler;
import org.elasticsearch.transport.TransportResponse;
import org.elasticsearch.transport.TransportService;

public class PeerRecoveryTargetService
extends AbstractComponent
implements IndexEventListener {
    private final ThreadPool threadPool;
    private final TransportService transportService;
    private final RecoverySettings recoverySettings;
    private final ClusterService clusterService;
    private final RecoveriesCollection onGoingRecoveries;

    @Inject
    public PeerRecoveryTargetService(Settings settings, ThreadPool threadPool, TransportService transportService, RecoverySettings recoverySettings, ClusterService clusterService) {
        super(settings);
        this.threadPool = threadPool;
        this.transportService = transportService;
        this.recoverySettings = recoverySettings;
        this.clusterService = clusterService;
        this.onGoingRecoveries = new RecoveriesCollection(this.logger, threadPool, this::waitForClusterState);
        transportService.registerRequestHandler("internal:index/shard/recovery/filesInfo", RecoveryFilesInfoRequest::new, "generic", new FilesInfoRequestHandler());
        transportService.registerRequestHandler("internal:index/shard/recovery/file_chunk", RecoveryFileChunkRequest::new, "generic", new FileChunkTransportRequestHandler());
        transportService.registerRequestHandler("internal:index/shard/recovery/clean_files", RecoveryCleanFilesRequest::new, "generic", new CleanFilesRequestHandler());
        transportService.registerRequestHandler("internal:index/shard/recovery/prepare_translog", RecoveryPrepareForTranslogOperationsRequest::new, "generic", new PrepareForTranslogOperationsRequestHandler());
        transportService.registerRequestHandler("internal:index/shard/recovery/translog_ops", RecoveryTranslogOperationsRequest::new, "generic", new TranslogOperationsRequestHandler());
        transportService.registerRequestHandler("internal:index/shard/recovery/finalize", RecoveryFinalizeRecoveryRequest::new, "generic", new FinalizeRecoveryRequestHandler());
        transportService.registerRequestHandler("internal:index/shard/recovery/wait_clusterstate", RecoveryWaitForClusterStateRequest::new, "generic", new WaitForClusterStateRequestHandler());
    }

    @Override
    public void beforeIndexShardClosed(ShardId shardId, @Nullable IndexShard indexShard, Settings indexSettings) {
        if (indexShard != null) {
            this.onGoingRecoveries.cancelRecoveriesForShard(shardId, "shard closed");
        }
    }

    public boolean cancelRecoveriesForShard(ShardId shardId, String reason) {
        return this.onGoingRecoveries.cancelRecoveriesForShard(shardId, reason);
    }

    public void startRecovery(IndexShard indexShard, DiscoveryNode sourceNode, RecoveryListener listener) {
        long recoveryId = this.onGoingRecoveries.startRecovery(indexShard, sourceNode, listener, this.recoverySettings.activityTimeout());
        this.threadPool.generic().execute(new RecoveryRunner(recoveryId));
    }

    protected void retryRecovery(RecoveryTarget recoveryTarget, Throwable reason, TimeValue retryAfter, StartRecoveryRequest currentRequest) {
        this.logger.trace(() -> new ParameterizedMessage("will retry recovery with id [{}] in [{}]", (Object)recoveryTarget.recoveryId(), (Object)retryAfter), reason);
        this.retryRecovery(recoveryTarget, retryAfter, currentRequest);
    }

    protected void retryRecovery(RecoveryTarget recoveryTarget, String reason, TimeValue retryAfter, StartRecoveryRequest currentRequest) {
        this.logger.trace("will retry recovery with id [{}] in [{}] (reason [{}])", (Object)recoveryTarget.recoveryId(), (Object)retryAfter, (Object)reason);
        this.retryRecovery(recoveryTarget, retryAfter, currentRequest);
    }

    private void retryRecovery(RecoveryTarget recoveryTarget, TimeValue retryAfter, StartRecoveryRequest currentRequest) {
        try {
            this.onGoingRecoveries.resetRecovery(recoveryTarget.recoveryId(), recoveryTarget.shardId());
        }
        catch (Exception e) {
            this.onGoingRecoveries.failRecovery(recoveryTarget.recoveryId(), new RecoveryFailedException(currentRequest, (Throwable)e), true);
        }
        this.threadPool.schedule(retryAfter, "generic", new RecoveryRunner(recoveryTarget.recoveryId()));
    }

    private void doRecovery(RecoveryTarget recoveryTarget) {
        assert (recoveryTarget.sourceNode() != null) : "can't do a recovery without a source node";
        this.logger.trace("collecting local files for {}", (Object)recoveryTarget);
        Store.MetadataSnapshot metadataSnapshot = null;
        try {
            metadataSnapshot = recoveryTarget.indexShard().indexSettings().isOnSharedFilesystem() ? Store.MetadataSnapshot.EMPTY : recoveryTarget.indexShard().snapshotStoreMetadata();
        }
        catch (IndexNotFoundException e) {
            metadataSnapshot = Store.MetadataSnapshot.EMPTY;
        }
        catch (IOException e) {
            this.logger.warn("error while listing local files, recover as if there are none", (Throwable)e);
            metadataSnapshot = Store.MetadataSnapshot.EMPTY;
        }
        catch (Exception e) {
            this.logger.trace("unexpected error while listing local files, failing recovery", (Throwable)e);
            this.onGoingRecoveries.failRecovery(recoveryTarget.recoveryId(), new RecoveryFailedException(recoveryTarget.state(), "failed to list local files", (Throwable)e), true);
            return;
        }
        this.logger.trace("{} local file count: [{}]", (Object)recoveryTarget, (Object)metadataSnapshot.size());
        StartRecoveryRequest request = new StartRecoveryRequest(recoveryTarget.shardId(), recoveryTarget.sourceNode(), this.clusterService.localNode(), metadataSnapshot, recoveryTarget.state().getPrimary(), recoveryTarget.recoveryId());
        AtomicReference responseHolder = new AtomicReference();
        try {
            this.logger.trace("[{}][{}] starting recovery from {}", (Object)request.shardId().getIndex().getName(), (Object)request.shardId().id(), (Object)request.sourceNode());
            recoveryTarget.indexShard().prepareForIndexRecovery();
            recoveryTarget.CancellableThreads().execute(() -> responseHolder.set(this.transportService.submitRequest(request.sourceNode(), "internal:index/shard/recovery/start_recovery", request, new FutureTransportResponseHandler<RecoveryResponse>(){

                @Override
                public RecoveryResponse newInstance() {
                    return new RecoveryResponse();
                }
            }).txGet()));
            RecoveryResponse recoveryResponse = (RecoveryResponse)responseHolder.get();
            assert (responseHolder != null);
            TimeValue recoveryTime = new TimeValue(recoveryTarget.state().getTimer().time());
            this.onGoingRecoveries.markRecoveryAsDone(recoveryTarget.recoveryId());
            if (this.logger.isTraceEnabled()) {
                StringBuilder sb = new StringBuilder();
                sb.append('[').append(request.shardId().getIndex().getName()).append(']').append('[').append(request.shardId().id()).append("] ");
                sb.append("recovery completed from ").append(request.sourceNode()).append(", took[").append(recoveryTime).append("]\n");
                sb.append("   phase1: recovered_files [").append(recoveryResponse.phase1FileNames.size()).append("]").append(" with total_size of [").append(new ByteSizeValue(recoveryResponse.phase1TotalSize)).append("]").append(", took [").append(TimeValue.timeValueMillis(recoveryResponse.phase1Time)).append("], throttling_wait [").append(TimeValue.timeValueMillis(recoveryResponse.phase1ThrottlingWaitTime)).append(']').append("\n");
                sb.append("         : reusing_files   [").append(recoveryResponse.phase1ExistingFileNames.size()).append("] with total_size of [").append(new ByteSizeValue(recoveryResponse.phase1ExistingTotalSize)).append("]\n");
                sb.append("   phase2: start took [").append(TimeValue.timeValueMillis(recoveryResponse.startTime)).append("]\n");
                sb.append("         : recovered [").append(recoveryResponse.phase2Operations).append("]").append(" transaction log operations").append(", took [").append(TimeValue.timeValueMillis(recoveryResponse.phase2Time)).append("]").append("\n");
                this.logger.trace("{}", (Object)sb);
            } else {
                this.logger.debug("{} recovery done from [{}], took [{}]", (Object)request.shardId(), (Object)recoveryTarget.sourceNode(), (Object)recoveryTime);
            }
        }
        catch (CancellableThreads.ExecutionCancelledException e) {
            this.logger.trace("recovery cancelled", (Throwable)e);
        }
        catch (Exception e) {
            Throwable cause;
            if (this.logger.isTraceEnabled()) {
                this.logger.trace(() -> new ParameterizedMessage("[{}][{}] Got exception on recovery", (Object)request.shardId().getIndex().getName(), (Object)request.shardId().id()), (Throwable)e);
            }
            if ((cause = ExceptionsHelper.unwrapCause(e)) instanceof CancellableThreads.ExecutionCancelledException) {
                this.onGoingRecoveries.failRecovery(recoveryTarget.recoveryId(), new RecoveryFailedException(request, "source has canceled the recovery", cause), false);
                return;
            }
            if (cause instanceof RecoveryEngineException) {
                cause = cause.getCause();
            }
            if ((cause = ExceptionsHelper.unwrapCause(cause)) instanceof RecoveryEngineException) {
                cause = cause.getCause();
            }
            if (cause instanceof IllegalIndexShardStateException || cause instanceof org.elasticsearch.index.IndexNotFoundException || cause instanceof ShardNotFoundException) {
                this.retryRecovery(recoveryTarget, "remote shard not ready", this.recoverySettings.retryDelayStateSync(), request);
                return;
            }
            if (cause instanceof DelayRecoveryException) {
                this.retryRecovery(recoveryTarget, cause, this.recoverySettings.retryDelayStateSync(), request);
                return;
            }
            if (cause instanceof ConnectTransportException) {
                this.logger.debug("delaying recovery of {} for [{}] due to networking error [{}]", (Object)recoveryTarget.shardId(), (Object)this.recoverySettings.retryDelayNetwork(), (Object)cause.getMessage());
                this.retryRecovery(recoveryTarget, cause.getMessage(), this.recoverySettings.retryDelayNetwork(), request);
                return;
            }
            if (cause instanceof AlreadyClosedException) {
                this.onGoingRecoveries.failRecovery(recoveryTarget.recoveryId(), new RecoveryFailedException(request, "source shard is closed", cause), false);
                return;
            }
            this.onGoingRecoveries.failRecovery(recoveryTarget.recoveryId(), new RecoveryFailedException(request, (Throwable)e), true);
        }
    }

    private void waitForClusterState(final long clusterStateVersion) {
        ClusterStateObserver observer = new ClusterStateObserver(this.clusterService, TimeValue.timeValueMinutes(5L), this.logger, this.threadPool.getThreadContext());
        ClusterState clusterState = observer.observedState();
        if (clusterState.getVersion() >= clusterStateVersion) {
            this.logger.trace("node has cluster state with version higher than {} (current: {})", (Object)clusterStateVersion, (Object)clusterState.getVersion());
            return;
        }
        this.logger.trace("waiting for cluster state version {} (current: {})", (Object)clusterStateVersion, (Object)clusterState.getVersion());
        final PlainActionFuture future = new PlainActionFuture();
        observer.waitForNextChange(new ClusterStateObserver.Listener(){

            @Override
            public void onNewClusterState(ClusterState state) {
                future.onResponse(null);
            }

            @Override
            public void onClusterServiceClose() {
                future.onFailure(new NodeClosedException(PeerRecoveryTargetService.this.clusterService.localNode()));
            }

            @Override
            public void onTimeout(TimeValue timeout) {
                future.onFailure(new IllegalStateException("cluster state never updated to version " + clusterStateVersion));
            }
        }, new ClusterStateObserver.ValidationPredicate(){

            @Override
            protected boolean validate(ClusterState newState) {
                return newState.getVersion() >= clusterStateVersion;
            }
        });
        try {
            future.get();
            this.logger.trace("successfully waited for cluster state with version {} (current: {})", (Object)clusterStateVersion, (Object)observer.observedState().getVersion());
        }
        catch (Exception e) {
            this.logger.debug(() -> new ParameterizedMessage("failed waiting for cluster state with version {} (current: {})", (Object)clusterStateVersion, (Object)observer.observedState()), (Throwable)e);
            throw ExceptionsHelper.convertToRuntime(e);
        }
    }

    class RecoveryRunner
    extends AbstractRunnable {
        final long recoveryId;

        RecoveryRunner(long recoveryId) {
            this.recoveryId = recoveryId;
        }

        @Override
        public void onFailure(Exception e) {
            try (RecoveriesCollection.RecoveryRef recoveryRef = PeerRecoveryTargetService.this.onGoingRecoveries.getRecovery(this.recoveryId);){
                if (recoveryRef != null) {
                    PeerRecoveryTargetService.this.logger.error(() -> new ParameterizedMessage("unexpected error during recovery [{}], failing shard", (Object)this.recoveryId), (Throwable)e);
                    PeerRecoveryTargetService.this.onGoingRecoveries.failRecovery(this.recoveryId, new RecoveryFailedException(recoveryRef.status().state(), "unexpected error", (Throwable)e), true);
                } else {
                    PeerRecoveryTargetService.this.logger.debug(() -> new ParameterizedMessage("unexpected error during recovery, but recovery id [{}] is finished", (Object)this.recoveryId), (Throwable)e);
                }
            }
        }

        @Override
        public void doRun() {
            RecoveriesCollection.RecoveryRef recoveryRef = PeerRecoveryTargetService.this.onGoingRecoveries.getRecovery(this.recoveryId);
            if (recoveryRef == null) {
                PeerRecoveryTargetService.this.logger.trace("not running recovery with id [{}] - can't find it (probably finished)", (Object)this.recoveryId);
                return;
            }
            try {
                PeerRecoveryTargetService.this.doRecovery(recoveryRef.status());
            }
            finally {
                recoveryRef.close();
            }
        }
    }

    class FileChunkTransportRequestHandler
    implements TransportRequestHandler<RecoveryFileChunkRequest> {
        final AtomicLong bytesSinceLastPause = new AtomicLong();

        FileChunkTransportRequestHandler() {
        }

        @Override
        public void messageReceived(RecoveryFileChunkRequest request, TransportChannel channel) throws Exception {
            try (RecoveriesCollection.RecoveryRef recoveryRef = PeerRecoveryTargetService.this.onGoingRecoveries.getRecoverySafe(request.recoveryId(), request.shardId());){
                long bytes;
                RateLimiter rateLimiter;
                RecoveryTarget status = recoveryRef.status();
                RecoveryState.Index indexState = status.state().getIndex();
                if (request.sourceThrottleTimeInNanos() != -1L) {
                    indexState.addSourceThrottling(request.sourceThrottleTimeInNanos());
                }
                if ((rateLimiter = PeerRecoveryTargetService.this.recoverySettings.rateLimiter()) != null && (bytes = this.bytesSinceLastPause.addAndGet(request.content().length())) > rateLimiter.getMinPauseCheckBytes()) {
                    this.bytesSinceLastPause.addAndGet(-bytes);
                    long throttleTimeInNanos = rateLimiter.pause(bytes);
                    indexState.addTargetThrottling(throttleTimeInNanos);
                    status.indexShard().recoveryStats().addThrottleTime(throttleTimeInNanos);
                }
                status.writeFileChunk(request.metadata(), request.position(), request.content(), request.lastChunk(), request.totalTranslogOps());
            }
            channel.sendResponse(TransportResponse.Empty.INSTANCE);
        }
    }

    class CleanFilesRequestHandler
    implements TransportRequestHandler<RecoveryCleanFilesRequest> {
        CleanFilesRequestHandler() {
        }

        @Override
        public void messageReceived(RecoveryCleanFilesRequest request, TransportChannel channel) throws Exception {
            try (RecoveriesCollection.RecoveryRef recoveryRef = PeerRecoveryTargetService.this.onGoingRecoveries.getRecoverySafe(request.recoveryId(), request.shardId());){
                recoveryRef.status().cleanFiles(request.totalTranslogOps(), request.sourceMetaSnapshot());
                channel.sendResponse(TransportResponse.Empty.INSTANCE);
            }
        }
    }

    class FilesInfoRequestHandler
    implements TransportRequestHandler<RecoveryFilesInfoRequest> {
        FilesInfoRequestHandler() {
        }

        @Override
        public void messageReceived(RecoveryFilesInfoRequest request, TransportChannel channel) throws Exception {
            try (RecoveriesCollection.RecoveryRef recoveryRef = PeerRecoveryTargetService.this.onGoingRecoveries.getRecoverySafe(request.recoveryId(), request.shardId());){
                recoveryRef.status().receiveFileInfo(request.phase1FileNames, request.phase1FileSizes, request.phase1ExistingFileNames, request.phase1ExistingFileSizes, request.totalTranslogOps);
                channel.sendResponse(TransportResponse.Empty.INSTANCE);
            }
        }
    }

    class TranslogOperationsRequestHandler
    implements TransportRequestHandler<RecoveryTranslogOperationsRequest> {
        TranslogOperationsRequestHandler() {
        }

        @Override
        public void messageReceived(final RecoveryTranslogOperationsRequest request, final TransportChannel channel) throws IOException {
            try (RecoveriesCollection.RecoveryRef recoveryRef = PeerRecoveryTargetService.this.onGoingRecoveries.getRecoverySafe(request.recoveryId(), request.shardId());){
                ClusterStateObserver observer = new ClusterStateObserver(PeerRecoveryTargetService.this.clusterService, null, PeerRecoveryTargetService.this.logger, PeerRecoveryTargetService.this.threadPool.getThreadContext());
                RecoveryTarget recoveryTarget = recoveryRef.status();
                try {
                    recoveryTarget.indexTranslogOperations(request.operations(), request.totalTranslogOps());
                    channel.sendResponse(TransportResponse.Empty.INSTANCE);
                }
                catch (TranslogRecoveryPerformer.BatchOperationException exception) {
                    MapperException mapperException = (MapperException)ExceptionsHelper.unwrap(exception, MapperException.class);
                    if (mapperException == null) {
                        throw exception;
                    }
                    PeerRecoveryTargetService.this.logger.trace(() -> new ParameterizedMessage("delaying recovery due to missing mapping changes (rolling back stats for [{}] ops)", (Object)exception.completedOperations()), (Throwable)exception);
                    RecoveryState.Translog translog = recoveryTarget.state().getTranslog();
                    translog.decrementRecoveredOperations(exception.completedOperations());
                    observer.waitForNextChange(new ClusterStateObserver.Listener(){

                        @Override
                        public void onNewClusterState(ClusterState state) {
                            try {
                                TranslogOperationsRequestHandler.this.messageReceived(request, channel);
                            }
                            catch (Exception e) {
                                this.onFailure(e);
                            }
                        }

                        protected void onFailure(Exception e) {
                            try {
                                channel.sendResponse(e);
                            }
                            catch (IOException e1) {
                                PeerRecoveryTargetService.this.logger.warn("failed to send error back to recovery source", (Throwable)e1);
                            }
                        }

                        @Override
                        public void onClusterServiceClose() {
                            this.onFailure(new ElasticsearchException("cluster service was closed while waiting for mapping updates", new Object[0]));
                        }

                        @Override
                        public void onTimeout(TimeValue timeout) {
                            this.onFailure(new ElasticsearchTimeoutException("timed out waiting for mapping updates (timeout [" + timeout + "])", new Object[0]));
                        }
                    });
                }
            }
        }
    }

    class WaitForClusterStateRequestHandler
    implements TransportRequestHandler<RecoveryWaitForClusterStateRequest> {
        WaitForClusterStateRequestHandler() {
        }

        @Override
        public void messageReceived(RecoveryWaitForClusterStateRequest request, TransportChannel channel) throws Exception {
            try (RecoveriesCollection.RecoveryRef recoveryRef = PeerRecoveryTargetService.this.onGoingRecoveries.getRecoverySafe(request.recoveryId(), request.shardId());){
                recoveryRef.status().ensureClusterStateVersion(request.clusterStateVersion());
            }
            channel.sendResponse(TransportResponse.Empty.INSTANCE);
        }
    }

    class FinalizeRecoveryRequestHandler
    implements TransportRequestHandler<RecoveryFinalizeRecoveryRequest> {
        FinalizeRecoveryRequestHandler() {
        }

        @Override
        public void messageReceived(RecoveryFinalizeRecoveryRequest request, TransportChannel channel) throws Exception {
            try (RecoveriesCollection.RecoveryRef recoveryRef = PeerRecoveryTargetService.this.onGoingRecoveries.getRecoverySafe(request.recoveryId(), request.shardId());){
                recoveryRef.status().finalizeRecovery();
            }
            channel.sendResponse(TransportResponse.Empty.INSTANCE);
        }
    }

    class PrepareForTranslogOperationsRequestHandler
    implements TransportRequestHandler<RecoveryPrepareForTranslogOperationsRequest> {
        PrepareForTranslogOperationsRequestHandler() {
        }

        @Override
        public void messageReceived(RecoveryPrepareForTranslogOperationsRequest request, TransportChannel channel) throws Exception {
            try (RecoveriesCollection.RecoveryRef recoveryRef = PeerRecoveryTargetService.this.onGoingRecoveries.getRecoverySafe(request.recoveryId(), request.shardId());){
                recoveryRef.status().prepareForTranslogOperations(request.totalTranslogOps(), request.getMaxUnsafeAutoIdTimestamp());
            }
            channel.sendResponse(TransportResponse.Empty.INSTANCE);
        }
    }

    public static interface RecoveryListener {
        public void onRecoveryDone(RecoveryState var1);

        public void onRecoveryFailure(RecoveryState var1, RecoveryFailedException var2, boolean var3);
    }

    public static class Actions {
        public static final String FILES_INFO = "internal:index/shard/recovery/filesInfo";
        public static final String FILE_CHUNK = "internal:index/shard/recovery/file_chunk";
        public static final String CLEAN_FILES = "internal:index/shard/recovery/clean_files";
        public static final String TRANSLOG_OPS = "internal:index/shard/recovery/translog_ops";
        public static final String PREPARE_TRANSLOG = "internal:index/shard/recovery/prepare_translog";
        public static final String FINALIZE = "internal:index/shard/recovery/finalize";
        public static final String WAIT_CLUSTERSTATE = "internal:index/shard/recovery/wait_clusterstate";
    }
}

