/*
 * Decompiled with CFR 0.152.
 */
package org.elasticsearch.index.translog;

import java.io.IOException;
import java.nio.channels.Channels;
import java.nio.channels.FileChannel;
import java.nio.file.DirectoryStream;
import java.nio.file.Files;
import java.nio.file.LinkOption;
import java.nio.file.Path;
import java.nio.file.StandardCopyOption;
import java.nio.file.StandardOpenOption;
import java.util.Arrays;
import java.util.HashSet;
import java.util.List;
import java.util.Map;
import java.util.Set;
import joptsimple.OptionParser;
import joptsimple.OptionSet;
import joptsimple.OptionSpec;
import org.apache.lucene.index.DirectoryReader;
import org.apache.lucene.index.IndexCommit;
import org.apache.lucene.store.Directory;
import org.apache.lucene.store.FSDirectory;
import org.apache.lucene.store.Lock;
import org.apache.lucene.store.LockFactory;
import org.apache.lucene.store.LockObtainFailedException;
import org.apache.lucene.store.NativeFSLockFactory;
import org.apache.lucene.store.OutputStreamDataOutput;
import org.apache.lucene.util.BytesRef;
import org.apache.lucene.util.IOUtils;
import org.elasticsearch.ElasticsearchException;
import org.elasticsearch.cli.SettingCommand;
import org.elasticsearch.cli.Terminal;
import org.elasticsearch.common.SuppressForbidden;
import org.elasticsearch.common.io.PathUtils;
import org.elasticsearch.index.IndexNotFoundException;
import org.elasticsearch.index.translog.Checkpoint;
import org.elasticsearch.index.translog.TranslogWriter;

public class TruncateTranslogCommand
extends SettingCommand {
    private final OptionSpec<String> translogFolder;
    private final OptionSpec<Void> batchMode;

    public TruncateTranslogCommand() {
        super("Truncates a translog to create a new, empty translog");
        this.translogFolder = this.parser.acceptsAll(Arrays.asList("d", "dir"), "Translog Directory location on disk").withRequiredArg().required();
        this.batchMode = this.parser.acceptsAll(Arrays.asList("b", "batch"), "Enable batch mode explicitly, automatic confirmation of warnings");
    }

    public OptionParser getParser() {
        return this.parser;
    }

    @Override
    protected void printAdditionalHelp(Terminal terminal) {
        terminal.println("This tool truncates the translog and translog");
        terminal.println("checkpoint files to create a new translog");
    }

    @SuppressForbidden(reason="Necessary to use the path passed in")
    private Path getTranslogPath(OptionSet options) {
        return PathUtils.get((String)this.translogFolder.value(options), "", "");
    }

    @Override
    protected void execute(Terminal terminal, OptionSet options, Map<String, String> settings) throws Exception {
        boolean batch = options.has(this.batchMode);
        Path translogPath = this.getTranslogPath(options);
        Path idxLocation = translogPath.getParent().resolve("index");
        if (!Files.exists(translogPath, new LinkOption[0]) || !Files.isDirectory(translogPath, new LinkOption[0])) {
            throw new ElasticsearchException("translog directory [" + translogPath + "], must exist and be a directory", new Object[0]);
        }
        if (!Files.exists(idxLocation, new LinkOption[0]) || !Files.isDirectory(idxLocation, new LinkOption[0])) {
            throw new ElasticsearchException("unable to find a shard at [" + idxLocation + "], which must exist and be a directory", new Object[0]);
        }
        try (FSDirectory dir = FSDirectory.open((Path)idxLocation, (LockFactory)NativeFSLockFactory.INSTANCE);
             Lock writeLock = dir.obtainLock("write.lock");){
            List commits;
            Set<Path> translogFiles;
            try {
                terminal.println("Checking existing translog files");
                translogFiles = TruncateTranslogCommand.filesInDirectory(translogPath);
            }
            catch (IOException e) {
                terminal.println("encountered IOException while listing directory, aborting...");
                throw new ElasticsearchException("failed to find existing translog files", (Throwable)e, new Object[0]);
            }
            TruncateTranslogCommand.warnAboutDeletingFiles(terminal, translogFiles, batch);
            try {
                terminal.println("Reading translog UUID information from Lucene commit from shard at [" + idxLocation + "]");
                commits = DirectoryReader.listCommits((Directory)dir);
            }
            catch (IndexNotFoundException infe) {
                throw new ElasticsearchException("unable to find a valid shard at [" + idxLocation + "]", (Throwable)infe, new Object[0]);
            }
            Map commitData = ((IndexCommit)commits.get(commits.size() - 1)).getUserData();
            String translogGeneration = (String)commitData.get("translog_generation");
            String translogUUID = (String)commitData.get("translog_uuid");
            if (translogGeneration == null || translogUUID == null) {
                throw new ElasticsearchException("shard must have a valid translog generation and UUID but got: [{}] and: [{}]", translogGeneration, translogUUID);
            }
            terminal.println("Translog Generation: " + translogGeneration);
            terminal.println("Translog UUID      : " + translogUUID);
            Path tempEmptyCheckpoint = translogPath.resolve("temp-translog.ckp");
            Path realEmptyCheckpoint = translogPath.resolve("translog.ckp");
            Path tempEmptyTranslog = translogPath.resolve("temp-translog-" + translogGeneration + ".tlog");
            Path realEmptyTranslog = translogPath.resolve("translog-" + translogGeneration + ".tlog");
            long gen = Long.parseLong(translogGeneration);
            int translogLen = TruncateTranslogCommand.writeEmptyTranslog(tempEmptyTranslog, translogUUID);
            TruncateTranslogCommand.writeEmptyCheckpoint(tempEmptyCheckpoint, translogLen, gen);
            terminal.println("Removing existing translog files");
            IOUtils.rm((Path[])translogFiles.toArray(new Path[0]));
            terminal.println("Creating new empty checkpoint at [" + realEmptyCheckpoint + "]");
            Files.move(tempEmptyCheckpoint, realEmptyCheckpoint, StandardCopyOption.ATOMIC_MOVE);
            terminal.println("Creating new empty translog at [" + realEmptyTranslog + "]");
            Files.move(tempEmptyTranslog, realEmptyTranslog, StandardCopyOption.ATOMIC_MOVE);
            IOUtils.fsync((Path)translogPath, (boolean)true);
        }
        catch (LockObtainFailedException lofe) {
            throw new ElasticsearchException("Failed to lock shard's directory at [" + idxLocation + "], is Elasticsearch still running?", new Object[0]);
        }
        terminal.println("Done.");
    }

    public static void writeEmptyCheckpoint(Path filename, int translogLength, long translogGeneration) throws IOException {
        Checkpoint emptyCheckpoint = new Checkpoint(translogLength, 0, translogGeneration);
        Checkpoint.write(FileChannel::open, filename, emptyCheckpoint, StandardOpenOption.WRITE, StandardOpenOption.READ, StandardOpenOption.CREATE_NEW);
        IOUtils.fsync((Path)filename, (boolean)false);
    }

    public static int writeEmptyTranslog(Path filename, String translogUUID) throws IOException {
        BytesRef translogRef = new BytesRef((CharSequence)translogUUID);
        try (FileChannel fc = FileChannel.open(filename, StandardOpenOption.WRITE, StandardOpenOption.READ, StandardOpenOption.CREATE_NEW);
             OutputStreamDataOutput out = new OutputStreamDataOutput(Channels.newOutputStream(fc));){
            TranslogWriter.writeHeader(out, translogRef);
            fc.force(true);
        }
        return TranslogWriter.getHeaderLength(translogRef.length);
    }

    public static void warnAboutDeletingFiles(Terminal terminal, Set<Path> files, boolean batchMode) {
        String text;
        terminal.println("!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!");
        terminal.println("!   WARNING: Elasticsearch MUST be stopped before running this tool   !");
        terminal.println("!                                                                     !");
        terminal.println("!   WARNING:    Documents inside of translog files will be lost       !");
        terminal.println("!                                                                     !");
        terminal.println("!   WARNING:          The following files will be DELETED!            !");
        terminal.println("!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!");
        for (Path file : files) {
            terminal.println("--> " + file);
        }
        terminal.println("");
        if (!batchMode && !(text = terminal.readText("Continue and DELETE files? [y/N] ")).equalsIgnoreCase("y")) {
            throw new ElasticsearchException("aborted by user", new Object[0]);
        }
    }

    public static Set<Path> filesInDirectory(Path directory) throws IOException {
        HashSet<Path> files = new HashSet<Path>();
        try (DirectoryStream<Path> stream = Files.newDirectoryStream(directory);){
            for (Path file : stream) {
                files.add(file);
            }
        }
        return files;
    }
}

