/*
 * Decompiled with CFR 0.152.
 */
package org.elasticsearch.bootstrap;

import java.io.BufferedReader;
import java.io.IOException;
import java.nio.file.Files;
import java.nio.file.Path;
import java.util.ArrayList;
import java.util.Arrays;
import java.util.Collections;
import java.util.List;
import java.util.Locale;
import java.util.regex.Matcher;
import java.util.regex.Pattern;
import org.apache.logging.log4j.Logger;
import org.apache.logging.log4j.message.ParameterizedMessage;
import org.apache.lucene.util.Constants;
import org.elasticsearch.bootstrap.BootstrapInfo;
import org.elasticsearch.bootstrap.BootstrapSettings;
import org.elasticsearch.bootstrap.JNACLibrary;
import org.elasticsearch.bootstrap.JNANatives;
import org.elasticsearch.bootstrap.JavaVersion;
import org.elasticsearch.bootstrap.Natives;
import org.elasticsearch.common.SuppressForbidden;
import org.elasticsearch.common.io.PathUtils;
import org.elasticsearch.common.logging.Loggers;
import org.elasticsearch.common.settings.Settings;
import org.elasticsearch.common.transport.BoundTransportAddress;
import org.elasticsearch.common.transport.TransportAddress;
import org.elasticsearch.monitor.jvm.JvmInfo;
import org.elasticsearch.monitor.process.ProcessProbe;
import org.elasticsearch.node.Node;
import org.elasticsearch.node.NodeValidationException;

final class BootstrapCheck {
    private BootstrapCheck() {
    }

    static void check(Settings settings, BoundTransportAddress boundTransportAddress) throws NodeValidationException {
        BootstrapCheck.check(BootstrapCheck.enforceLimits(boundTransportAddress), BootstrapCheck.checks(settings), Node.NODE_NAME_SETTING.get(settings));
    }

    static void check(boolean enforceLimits, List<Check> checks, String nodeName) throws NodeValidationException {
        BootstrapCheck.check(enforceLimits, checks, Loggers.getLogger(BootstrapCheck.class, nodeName));
    }

    static void check(boolean enforceLimits, List<Check> checks, Logger logger) throws NodeValidationException {
        ArrayList<String> errors = new ArrayList<String>();
        ArrayList<String> ignoredErrors = new ArrayList<String>();
        if (enforceLimits) {
            logger.info("bound or publishing to a non-loopback or non-link-local address, enforcing bootstrap checks");
        }
        for (Check check : checks) {
            if (!check.check()) continue;
            if (!enforceLimits && !check.alwaysEnforce()) {
                ignoredErrors.add(check.errorMessage());
                continue;
            }
            errors.add(check.errorMessage());
        }
        if (!ignoredErrors.isEmpty()) {
            ignoredErrors.forEach(error -> BootstrapCheck.log(logger, error));
        }
        if (!errors.isEmpty()) {
            ArrayList<String> messages = new ArrayList<String>(1 + errors.size());
            messages.add("bootstrap checks failed");
            messages.addAll(errors);
            NodeValidationException ne = new NodeValidationException(String.join((CharSequence)"\n", messages));
            errors.stream().map(IllegalStateException::new).forEach(ne::addSuppressed);
            throw ne;
        }
    }

    static void log(Logger logger, String error) {
        logger.warn(error);
    }

    static boolean enforceLimits(BoundTransportAddress boundTransportAddress) {
        return !Arrays.stream(boundTransportAddress.boundAddresses()).allMatch(TransportAddress::isLoopbackOrLinkLocalAddress) || !boundTransportAddress.publishAddress().isLoopbackOrLinkLocalAddress();
    }

    static List<Check> checks(Settings settings) {
        ArrayList<Check> checks = new ArrayList<Check>();
        checks.add(new HeapSizeCheck());
        FileDescriptorCheck fileDescriptorCheck = Constants.MAC_OS_X ? new OsXFileDescriptorCheck() : new FileDescriptorCheck();
        checks.add(fileDescriptorCheck);
        checks.add(new MlockallCheck(BootstrapSettings.MEMORY_LOCK_SETTING.get(settings)));
        if (Constants.LINUX) {
            checks.add(new MaxNumberOfThreadsCheck());
        }
        if (Constants.LINUX || Constants.MAC_OS_X) {
            checks.add(new MaxSizeVirtualMemoryCheck());
        }
        if (Constants.LINUX) {
            checks.add(new MaxMapCountCheck());
        }
        checks.add(new ClientJvmCheck());
        checks.add(new UseSerialGCCheck());
        checks.add(new OnErrorCheck());
        checks.add(new OnOutOfMemoryErrorCheck());
        checks.add(new G1GCCheck());
        return Collections.unmodifiableList(checks);
    }

    static class G1GCCheck
    implements Check {
        G1GCCheck() {
        }

        @Override
        public boolean check() {
            if ("Oracle Corporation".equals(this.jvmVendor()) && this.isJava8() && this.isG1GCEnabled()) {
                String jvmVersion = this.jvmVersion();
                Pattern pattern = Pattern.compile("(\\d+)\\.(\\d+)-b\\d+");
                Matcher matcher = pattern.matcher(jvmVersion);
                boolean matches = matcher.matches();
                assert (matches) : jvmVersion;
                int major = Integer.parseInt(matcher.group(1));
                int update = Integer.parseInt(matcher.group(2));
                return major == 25 && update < 40;
            }
            return false;
        }

        String jvmVendor() {
            return Constants.JVM_VENDOR;
        }

        boolean isG1GCEnabled() {
            assert ("Oracle Corporation".equals(this.jvmVendor()));
            return JvmInfo.jvmInfo().useG1GC().equals("true");
        }

        String jvmVersion() {
            assert ("Oracle Corporation".equals(this.jvmVendor()));
            return Constants.JVM_VERSION;
        }

        boolean isJava8() {
            assert ("Oracle Corporation".equals(this.jvmVendor()));
            return JavaVersion.current().equals(JavaVersion.parse("1.8"));
        }

        @Override
        public String errorMessage() {
            return String.format(Locale.ROOT, "JVM version [%s] can cause data corruption when used with G1GC; upgrade to at least Java 8u40", this.jvmVersion());
        }
    }

    static class OnOutOfMemoryErrorCheck
    extends MightForkCheck {
        OnOutOfMemoryErrorCheck() {
        }

        @Override
        boolean mightFork() {
            String onOutOfMemoryError = this.onOutOfMemoryError();
            return onOutOfMemoryError != null && !onOutOfMemoryError.equals("");
        }

        String onOutOfMemoryError() {
            return JvmInfo.jvmInfo().onOutOfMemoryError();
        }

        @Override
        public String errorMessage() {
            return String.format(Locale.ROOT, "OnOutOfMemoryError [%s] requires forking but is prevented by system call filters ([%s=true]); upgrade to at least Java 8u92 and use ExitOnOutOfMemoryError", this.onOutOfMemoryError(), BootstrapSettings.SECCOMP_SETTING.getKey());
        }
    }

    static class OnErrorCheck
    extends MightForkCheck {
        OnErrorCheck() {
        }

        @Override
        boolean mightFork() {
            String onError = this.onError();
            return onError != null && !onError.equals("");
        }

        String onError() {
            return JvmInfo.jvmInfo().onError();
        }

        @Override
        public String errorMessage() {
            return String.format(Locale.ROOT, "OnError [%s] requires forking but is prevented by system call filters ([%s=true]); upgrade to at least Java 8u92 and use ExitOnOutOfMemoryError", this.onError(), BootstrapSettings.SECCOMP_SETTING.getKey());
        }
    }

    static abstract class MightForkCheck
    implements Check {
        MightForkCheck() {
        }

        @Override
        public boolean check() {
            return this.isSeccompInstalled() && this.mightFork();
        }

        boolean isSeccompInstalled() {
            return Natives.isSeccompInstalled();
        }

        abstract boolean mightFork();

        @Override
        public final boolean alwaysEnforce() {
            return true;
        }
    }

    static class UseSerialGCCheck
    implements Check {
        UseSerialGCCheck() {
        }

        @Override
        public boolean check() {
            return this.getUseSerialGC().equals("true");
        }

        String getUseSerialGC() {
            return JvmInfo.jvmInfo().useSerialGC();
        }

        @Override
        public String errorMessage() {
            return String.format(Locale.ROOT, "JVM is using the serial collector but should not be for the best performance; either it's the default for the VM [%s] or -XX:+UseSerialGC was explicitly specified", JvmInfo.jvmInfo().getVmName());
        }
    }

    static class ClientJvmCheck
    implements Check {
        ClientJvmCheck() {
        }

        @Override
        public boolean check() {
            return this.getVmName().toLowerCase(Locale.ROOT).contains("client");
        }

        String getVmName() {
            return JvmInfo.jvmInfo().getVmName();
        }

        @Override
        public String errorMessage() {
            return String.format(Locale.ROOT, "JVM is using the client VM [%s] but should be using a server VM for the best performance", this.getVmName());
        }
    }

    static class MaxMapCountCheck
    implements Check {
        private final long limit = 262144L;

        MaxMapCountCheck() {
        }

        @Override
        public boolean check() {
            return this.getMaxMapCount() != -1L && this.getMaxMapCount() < 262144L;
        }

        @Override
        public String errorMessage() {
            return String.format(Locale.ROOT, "max virtual memory areas vm.max_map_count [%d] is too low, increase to at least [%d]", this.getMaxMapCount(), 262144L);
        }

        long getMaxMapCount() {
            return this.getMaxMapCount(Loggers.getLogger(BootstrapCheck.class));
        }

        /*
         * Enabled aggressive exception aggregation
         */
        long getMaxMapCount(Logger logger) {
            block19: {
                Path path = this.getProcSysVmMaxMapCountPath();
                try {
                    Throwable throwable = null;
                    try (BufferedReader bufferedReader = this.getBufferedReader(path);){
                        String rawProcSysVmMaxMapCount = this.readProcSysVmMaxMapCount(bufferedReader);
                        if (rawProcSysVmMaxMapCount == null) break block19;
                        try {
                            long l = this.parseProcSysVmMaxMapCount(rawProcSysVmMaxMapCount);
                            return l;
                        }
                        catch (NumberFormatException e) {
                            try {
                                logger.warn(() -> new ParameterizedMessage("unable to parse vm.max_map_count [{}]", (Object)rawProcSysVmMaxMapCount), (Throwable)e);
                            }
                            catch (Throwable throwable2) {
                                throwable = throwable2;
                                throw throwable2;
                            }
                            catch (Throwable throwable3) {
                                throw throwable3;
                            }
                        }
                    }
                }
                catch (IOException e) {
                    logger.warn(() -> new ParameterizedMessage("I/O exception while trying to read [{}]", (Object)path), (Throwable)e);
                }
            }
            return -1L;
        }

        @SuppressForbidden(reason="access /proc/sys/vm/max_map_count")
        private Path getProcSysVmMaxMapCountPath() {
            return PathUtils.get("/proc/sys/vm/max_map_count", new String[0]);
        }

        BufferedReader getBufferedReader(Path path) throws IOException {
            return Files.newBufferedReader(path);
        }

        String readProcSysVmMaxMapCount(BufferedReader bufferedReader) throws IOException {
            return bufferedReader.readLine();
        }

        long parseProcSysVmMaxMapCount(String procSysVmMaxMapCount) throws NumberFormatException {
            return Long.parseLong(procSysVmMaxMapCount);
        }
    }

    static class MaxSizeVirtualMemoryCheck
    implements Check {
        MaxSizeVirtualMemoryCheck() {
        }

        @Override
        public boolean check() {
            return this.getMaxSizeVirtualMemory() != Long.MIN_VALUE && this.getMaxSizeVirtualMemory() != this.getRlimInfinity();
        }

        @Override
        public String errorMessage() {
            return String.format(Locale.ROOT, "max size virtual memory [%d] for user [%s] is too low, increase to [unlimited]", this.getMaxSizeVirtualMemory(), BootstrapInfo.getSystemProperties().get("user.name"));
        }

        long getRlimInfinity() {
            return JNACLibrary.RLIM_INFINITY;
        }

        long getMaxSizeVirtualMemory() {
            return JNANatives.MAX_SIZE_VIRTUAL_MEMORY;
        }
    }

    static class MaxNumberOfThreadsCheck
    implements Check {
        private final long maxNumberOfThreadsThreshold = 2048L;

        MaxNumberOfThreadsCheck() {
        }

        @Override
        public boolean check() {
            return this.getMaxNumberOfThreads() != -1L && this.getMaxNumberOfThreads() < 2048L;
        }

        @Override
        public String errorMessage() {
            return String.format(Locale.ROOT, "max number of threads [%d] for user [%s] is too low, increase to at least [%d]", this.getMaxNumberOfThreads(), BootstrapInfo.getSystemProperties().get("user.name"), 2048L);
        }

        long getMaxNumberOfThreads() {
            return JNANatives.MAX_NUMBER_OF_THREADS;
        }
    }

    static class MlockallCheck
    implements Check {
        private final boolean mlockallSet;

        public MlockallCheck(boolean mlockAllSet) {
            this.mlockallSet = mlockAllSet;
        }

        @Override
        public boolean check() {
            return this.mlockallSet && !this.isMemoryLocked();
        }

        @Override
        public String errorMessage() {
            return "memory locking requested for elasticsearch process but memory is not locked";
        }

        boolean isMemoryLocked() {
            return Natives.isMemoryLocked();
        }
    }

    static class FileDescriptorCheck
    implements Check {
        private final int limit;

        FileDescriptorCheck() {
            this(65536);
        }

        protected FileDescriptorCheck(int limit) {
            if (limit <= 0) {
                throw new IllegalArgumentException("limit must be positive but was [" + limit + "]");
            }
            this.limit = limit;
        }

        @Override
        public final boolean check() {
            long maxFileDescriptorCount = this.getMaxFileDescriptorCount();
            return maxFileDescriptorCount != -1L && maxFileDescriptorCount < (long)this.limit;
        }

        @Override
        public final String errorMessage() {
            return String.format(Locale.ROOT, "max file descriptors [%d] for elasticsearch process is too low, increase to at least [%d]", this.getMaxFileDescriptorCount(), this.limit);
        }

        long getMaxFileDescriptorCount() {
            return ProcessProbe.getInstance().getMaxFileDescriptorCount();
        }
    }

    static class OsXFileDescriptorCheck
    extends FileDescriptorCheck {
        public OsXFileDescriptorCheck() {
            super(10240);
        }
    }

    static class HeapSizeCheck
    implements Check {
        HeapSizeCheck() {
        }

        @Override
        public boolean check() {
            long initialHeapSize = this.getInitialHeapSize();
            long maxHeapSize = this.getMaxHeapSize();
            return initialHeapSize != 0L && maxHeapSize != 0L && initialHeapSize != maxHeapSize;
        }

        @Override
        public String errorMessage() {
            return String.format(Locale.ROOT, "initial heap size [%d] not equal to maximum heap size [%d]; this can cause resize pauses and prevents mlockall from locking the entire heap", this.getInitialHeapSize(), this.getMaxHeapSize());
        }

        long getInitialHeapSize() {
            return JvmInfo.jvmInfo().getConfiguredInitialHeapSize();
        }

        long getMaxHeapSize() {
            return JvmInfo.jvmInfo().getConfiguredMaxHeapSize();
        }
    }

    static interface Check {
        public boolean check();

        public String errorMessage();

        default public boolean alwaysEnforce() {
            return false;
        }
    }
}

