/*
 * Decompiled with CFR 0.152.
 */
package org.elasticsearch.common.cache;

import java.util.Arrays;
import java.util.HashMap;
import java.util.Iterator;
import java.util.Map;
import java.util.Objects;
import java.util.concurrent.CompletableFuture;
import java.util.concurrent.CompletionStage;
import java.util.concurrent.ExecutionException;
import java.util.concurrent.atomic.LongAdder;
import java.util.concurrent.locks.ReadWriteLock;
import java.util.concurrent.locks.ReentrantLock;
import java.util.concurrent.locks.ReentrantReadWriteLock;
import java.util.function.BiFunction;
import java.util.function.ToLongBiFunction;
import org.elasticsearch.common.cache.CacheLoader;
import org.elasticsearch.common.cache.RemovalListener;
import org.elasticsearch.common.cache.RemovalNotification;
import org.elasticsearch.common.collect.Tuple;
import org.elasticsearch.common.util.concurrent.ReleasableLock;

public class Cache<K, V> {
    private long expireAfterAccessNanos = -1L;
    private boolean entriesExpireAfterAccess;
    private long expireAfterWriteNanos = -1L;
    private boolean entriesExpireAfterWrite;
    private int count = 0;
    private long weight = 0L;
    private long maximumWeight = -1L;
    private ToLongBiFunction<K, V> weigher = (k, v) -> 1L;
    private RemovalListener<K, V> removalListener = notification -> {};
    public static final int NUMBER_OF_SEGMENTS = 256;
    private final CacheSegment<K, V>[] segments = new CacheSegment[256];
    Entry<K, V> head;
    Entry<K, V> tail;
    private ReleasableLock lruLock;

    Cache() {
        for (int i = 0; i < this.segments.length; ++i) {
            this.segments[i] = new CacheSegment();
        }
        this.lruLock = new ReleasableLock(new ReentrantLock());
    }

    void setExpireAfterAccessNanos(long expireAfterAccessNanos) {
        if (expireAfterAccessNanos <= 0L) {
            throw new IllegalArgumentException("expireAfterAccessNanos <= 0");
        }
        this.expireAfterAccessNanos = expireAfterAccessNanos;
        this.entriesExpireAfterAccess = true;
    }

    long getExpireAfterAccessNanos() {
        return this.expireAfterAccessNanos;
    }

    void setExpireAfterWriteNanos(long expireAfterWriteNanos) {
        if (expireAfterWriteNanos <= 0L) {
            throw new IllegalArgumentException("expireAfterWriteNanos <= 0");
        }
        this.expireAfterWriteNanos = expireAfterWriteNanos;
        this.entriesExpireAfterWrite = true;
    }

    long getExpireAfterWriteNanos() {
        return this.expireAfterWriteNanos;
    }

    void setMaximumWeight(long maximumWeight) {
        if (maximumWeight < 0L) {
            throw new IllegalArgumentException("maximumWeight < 0");
        }
        this.maximumWeight = maximumWeight;
    }

    void setWeigher(ToLongBiFunction<K, V> weigher) {
        Objects.requireNonNull(weigher);
        this.weigher = weigher;
    }

    void setRemovalListener(RemovalListener<K, V> removalListener) {
        Objects.requireNonNull(removalListener);
        this.removalListener = removalListener;
    }

    protected long now() {
        return this.entriesExpireAfterAccess || this.entriesExpireAfterWrite ? System.nanoTime() : 0L;
    }

    public V get(K key) {
        return this.get(key, this.now());
    }

    private V get(K key, long now) {
        CacheSegment<K, V> segment = this.getCacheSegment(key);
        Entry<K, V> entry = segment.get(key, now);
        if (entry == null || this.isExpired(entry, now)) {
            return null;
        }
        this.promote(entry, now);
        return entry.value;
    }

    public V computeIfAbsent(K key, CacheLoader<K, V> loader) throws ExecutionException {
        long now = this.now();
        Object value = this.get(key, now);
        if (value == null) {
            CompletionStage completableValue;
            CompletableFuture<Entry<K, V>> future;
            CacheSegment<K, V> segment = this.getCacheSegment(key);
            CompletableFuture completableFuture = new CompletableFuture();
            try (ReleasableLock ignored = segment.writeLock.acquire();){
                future = segment.map.putIfAbsent(key, completableFuture);
            }
            BiFunction<Entry, Throwable, Object> handler = (ok, ex) -> {
                if (ok != null) {
                    try (ReleasableLock ignored = this.lruLock.acquire();){
                        this.promote((Entry<K, V>)ok, now);
                    }
                    return ok.value;
                }
                try (ReleasableLock ignored = segment.writeLock.acquire();){
                    CompletableFuture sanity = segment.map.get(key);
                    if (sanity != null && sanity.isCompletedExceptionally()) {
                        segment.map.remove(key);
                    }
                }
                return null;
            };
            if (future == null) {
                V loaded;
                future = completableFuture;
                completableValue = future.handle(handler);
                try {
                    loaded = loader.load(key);
                }
                catch (Exception e) {
                    future.completeExceptionally(e);
                    throw new ExecutionException(e);
                }
                if (loaded == null) {
                    NullPointerException npe = new NullPointerException("loader returned a null value");
                    future.completeExceptionally(npe);
                    throw new ExecutionException(npe);
                }
                future.complete(new Entry<K, V>(key, loaded, now));
            } else {
                completableValue = future.handle(handler);
            }
            try {
                value = ((CompletableFuture)completableValue).get();
                if (future.isCompletedExceptionally()) {
                    future.get();
                    throw new IllegalStateException("the future was completed exceptionally but no exception was thrown");
                }
            }
            catch (InterruptedException e) {
                throw new IllegalStateException(e);
            }
        }
        return value;
    }

    public void put(K key, V value) {
        long now = this.now();
        this.put(key, value, now);
    }

    private void put(K key, V value, long now) {
        CacheSegment<K, V> segment = this.getCacheSegment(key);
        Tuple<Entry<K, V>, Entry<K, V>> tuple = segment.put(key, value, now);
        boolean replaced = false;
        try (ReleasableLock ignored = this.lruLock.acquire();){
            if (tuple.v2() != null && tuple.v2().state == State.EXISTING && this.unlink(tuple.v2())) {
                replaced = true;
            }
            this.promote(tuple.v1(), now);
        }
        if (replaced) {
            this.removalListener.onRemoval(new RemovalNotification(tuple.v2().key, tuple.v2().value, RemovalNotification.RemovalReason.REPLACED));
        }
    }

    public void invalidate(K key) {
        CacheSegment<K, V> segment = this.getCacheSegment(key);
        Entry<K, V> entry = segment.remove(key);
        if (entry != null) {
            try (ReleasableLock ignored = this.lruLock.acquire();){
                this.delete(entry, RemovalNotification.RemovalReason.INVALIDATED);
            }
        }
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    public void invalidateAll() {
        Entry<K, V> h;
        boolean[] haveSegmentLock = new boolean[256];
        try {
            for (int i = 0; i < 256; ++i) {
                this.segments[i].segmentLock.writeLock().lock();
                haveSegmentLock[i] = true;
            }
            try (ReleasableLock ignored = this.lruLock.acquire();){
                h = this.head;
                Arrays.stream(this.segments).forEach(segment -> {
                    segment.map = new HashMap();
                });
                Entry<K, V> current = this.head;
                while (current != null) {
                    current.state = State.DELETED;
                    current = current.after;
                }
                this.tail = null;
                this.head = null;
                this.count = 0;
                this.weight = 0L;
            }
        }
        catch (Throwable throwable) {
            for (int i = 255; i >= 0; --i) {
                if (!haveSegmentLock[i]) continue;
                this.segments[i].segmentLock.writeLock().unlock();
            }
            throw throwable;
        }
        for (int i = 255; i >= 0; --i) {
            if (!haveSegmentLock[i]) continue;
            this.segments[i].segmentLock.writeLock().unlock();
        }
        while (h != null) {
            this.removalListener.onRemoval(new RemovalNotification(h.key, h.value, RemovalNotification.RemovalReason.INVALIDATED));
            h = h.after;
        }
    }

    public void refresh() {
        long now = this.now();
        try (ReleasableLock ignored = this.lruLock.acquire();){
            this.evict(now);
        }
    }

    public int count() {
        return this.count;
    }

    public long weight() {
        return this.weight;
    }

    public Iterable<K> keys() {
        return () -> new Iterator<K>(){
            private CacheIterator iterator;
            {
                this.iterator = new CacheIterator(Cache.this.head);
            }

            @Override
            public boolean hasNext() {
                return this.iterator.hasNext();
            }

            @Override
            public K next() {
                return ((Entry)this.iterator.next()).key;
            }

            @Override
            public void remove() {
                this.iterator.remove();
            }
        };
    }

    public Iterable<V> values() {
        return () -> new Iterator<V>(){
            private CacheIterator iterator;
            {
                this.iterator = new CacheIterator(Cache.this.head);
            }

            @Override
            public boolean hasNext() {
                return this.iterator.hasNext();
            }

            @Override
            public V next() {
                return ((Entry)this.iterator.next()).value;
            }
        };
    }

    public CacheStats stats() {
        long hits = 0L;
        long misses = 0L;
        long evictions = 0L;
        for (int i = 0; i < this.segments.length; ++i) {
            hits += this.segments[i].segmentStats.hits.longValue();
            misses += this.segments[i].segmentStats.misses.longValue();
            evictions += this.segments[i].segmentStats.evictions.longValue();
        }
        return new CacheStats(hits, misses, evictions);
    }

    private boolean promote(Entry<K, V> entry, long now) {
        boolean promoted = true;
        try (ReleasableLock ignored = this.lruLock.acquire();){
            switch (entry.state) {
                case DELETED: {
                    promoted = false;
                    break;
                }
                case EXISTING: {
                    this.relinkAtHead(entry);
                    break;
                }
                case NEW: {
                    this.linkAtHead(entry);
                }
            }
            if (promoted) {
                this.evict(now);
            }
        }
        return promoted;
    }

    private void evict(long now) {
        assert (this.lruLock.isHeldByCurrentThread().booleanValue());
        while (this.tail != null && this.shouldPrune(this.tail, now)) {
            CacheSegment segment = this.getCacheSegment(this.tail.key);
            Entry<K, V> entry = this.tail;
            if (segment != null) {
                segment.remove(this.tail.key);
            }
            this.delete(entry, RemovalNotification.RemovalReason.EVICTED);
        }
    }

    private void delete(Entry<K, V> entry, RemovalNotification.RemovalReason removalReason) {
        assert (this.lruLock.isHeldByCurrentThread().booleanValue());
        if (this.unlink(entry)) {
            this.removalListener.onRemoval(new RemovalNotification(entry.key, entry.value, removalReason));
        }
    }

    private boolean shouldPrune(Entry<K, V> entry, long now) {
        return this.exceedsWeight() || this.isExpired(entry, now);
    }

    private boolean exceedsWeight() {
        return this.maximumWeight != -1L && this.weight > this.maximumWeight;
    }

    private boolean isExpired(Entry<K, V> entry, long now) {
        return this.entriesExpireAfterAccess && now - entry.accessTime > this.expireAfterAccessNanos || this.entriesExpireAfterWrite && now - entry.writeTime > this.expireAfterWriteNanos;
    }

    private boolean unlink(Entry<K, V> entry) {
        assert (this.lruLock.isHeldByCurrentThread().booleanValue());
        if (entry.state == State.EXISTING) {
            Entry before = entry.before;
            Entry after = entry.after;
            if (before == null) {
                assert (this.head == entry);
                this.head = after;
                if (this.head != null) {
                    this.head.before = null;
                }
            } else {
                before.after = after;
                entry.before = null;
            }
            if (after == null) {
                assert (this.tail == entry);
                this.tail = before;
                if (this.tail != null) {
                    this.tail.after = null;
                }
            } else {
                after.before = before;
                entry.after = null;
            }
            --this.count;
            this.weight -= this.weigher.applyAsLong(entry.key, entry.value);
            entry.state = State.DELETED;
            return true;
        }
        return false;
    }

    private void linkAtHead(Entry<K, V> entry) {
        assert (this.lruLock.isHeldByCurrentThread().booleanValue());
        Entry<K, V> h = this.head;
        entry.before = null;
        entry.after = this.head;
        this.head = entry;
        if (h == null) {
            this.tail = entry;
        } else {
            h.before = entry;
        }
        ++this.count;
        this.weight += this.weigher.applyAsLong(entry.key, entry.value);
        entry.state = State.EXISTING;
    }

    private void relinkAtHead(Entry<K, V> entry) {
        assert (this.lruLock.isHeldByCurrentThread().booleanValue());
        if (this.head != entry) {
            this.unlink(entry);
            this.linkAtHead(entry);
        }
    }

    private CacheSegment<K, V> getCacheSegment(K key) {
        return this.segments[key.hashCode() & 0xFF];
    }

    public static class CacheStats {
        private long hits;
        private long misses;
        private long evictions;

        public CacheStats(long hits, long misses, long evictions) {
            this.hits = hits;
            this.misses = misses;
            this.evictions = evictions;
        }

        public long getHits() {
            return this.hits;
        }

        public long getMisses() {
            return this.misses;
        }

        public long getEvictions() {
            return this.evictions;
        }
    }

    private class CacheIterator
    implements Iterator<Entry<K, V>> {
        private Entry<K, V> current = null;
        private Entry<K, V> next;

        CacheIterator(Entry<K, V> head) {
            this.next = head;
        }

        @Override
        public boolean hasNext() {
            return this.next != null;
        }

        @Override
        public Entry<K, V> next() {
            this.current = this.next;
            this.next = this.next.after;
            return this.current;
        }

        @Override
        public void remove() {
            Entry entry = this.current;
            if (entry != null) {
                CacheSegment segment = Cache.this.getCacheSegment(entry.key);
                segment.remove(entry.key);
                try (ReleasableLock ignored = Cache.this.lruLock.acquire();){
                    this.current = null;
                    Cache.this.delete(entry, RemovalNotification.RemovalReason.INVALIDATED);
                }
            }
        }
    }

    private static class CacheSegment<K, V> {
        ReadWriteLock segmentLock = new ReentrantReadWriteLock();
        ReleasableLock readLock = new ReleasableLock(this.segmentLock.readLock());
        ReleasableLock writeLock = new ReleasableLock(this.segmentLock.writeLock());
        Map<K, CompletableFuture<Entry<K, V>>> map = new HashMap<K, CompletableFuture<Entry<K, V>>>();
        SegmentStats segmentStats = new SegmentStats();

        private CacheSegment() {
        }

        Entry<K, V> get(K key, long now) {
            CompletableFuture<Entry<K, V>> future;
            Entry entry = null;
            try (ReleasableLock ignored = this.readLock.acquire();){
                future = this.map.get(key);
            }
            if (future != null) {
                try {
                    entry = (Entry)((CompletableFuture)future.handle((ok, ex) -> {
                        if (ok != null) {
                            this.segmentStats.hit();
                            ok.accessTime = now;
                            return ok;
                        }
                        this.segmentStats.miss();
                        return null;
                    })).get();
                }
                catch (InterruptedException | ExecutionException e) {
                    throw new IllegalStateException(e);
                }
            } else {
                this.segmentStats.miss();
            }
            return entry;
        }

        Tuple<Entry<K, V>, Entry<K, V>> put(K key, V value, long now) {
            Entry<K, V> entry = new Entry<K, V>(key, value, now);
            Entry existing = null;
            try (ReleasableLock ignored = this.writeLock.acquire();){
                try {
                    CompletableFuture<Entry<K, V>> future = this.map.put(key, CompletableFuture.completedFuture(entry));
                    if (future != null) {
                        existing = (Entry)((CompletableFuture)future.handle((ok, ex) -> {
                            if (ok != null) {
                                return ok;
                            }
                            return null;
                        })).get();
                    }
                }
                catch (InterruptedException | ExecutionException e) {
                    throw new IllegalStateException(e);
                }
            }
            return Tuple.tuple(entry, existing);
        }

        Entry<K, V> remove(K key) {
            CompletableFuture<Entry<K, V>> future;
            Entry entry = null;
            try (ReleasableLock ignored = this.writeLock.acquire();){
                future = this.map.remove(key);
            }
            if (future != null) {
                try {
                    entry = (Entry)((CompletableFuture)future.handle((ok, ex) -> {
                        if (ok != null) {
                            this.segmentStats.eviction();
                            return ok;
                        }
                        return null;
                    })).get();
                }
                catch (InterruptedException | ExecutionException e) {
                    throw new IllegalStateException(e);
                }
            }
            return entry;
        }

        private static class SegmentStats {
            private final LongAdder hits = new LongAdder();
            private final LongAdder misses = new LongAdder();
            private final LongAdder evictions = new LongAdder();

            private SegmentStats() {
            }

            void hit() {
                this.hits.increment();
            }

            void miss() {
                this.misses.increment();
            }

            void eviction() {
                this.evictions.increment();
            }
        }
    }

    static class Entry<K, V> {
        final K key;
        final V value;
        long writeTime;
        volatile long accessTime;
        Entry<K, V> before;
        Entry<K, V> after;
        State state = State.NEW;

        public Entry(K key, V value, long writeTime) {
            this.key = key;
            this.value = value;
            this.writeTime = this.accessTime = writeTime;
        }
    }

    static enum State {
        NEW,
        EXISTING,
        DELETED;

    }
}

