/*
 * Decompiled with CFR 0.152.
 */
package org.elasticsearch.indices.recovery;

import java.io.IOException;
import java.util.concurrent.ConcurrentMap;
import java.util.concurrent.atomic.AtomicBoolean;
import org.apache.logging.log4j.Logger;
import org.apache.logging.log4j.message.ParameterizedMessage;
import org.elasticsearch.ElasticsearchTimeoutException;
import org.elasticsearch.cluster.node.DiscoveryNode;
import org.elasticsearch.common.unit.TimeValue;
import org.elasticsearch.common.util.Callback;
import org.elasticsearch.common.util.concurrent.AbstractRunnable;
import org.elasticsearch.common.util.concurrent.ConcurrentCollections;
import org.elasticsearch.index.shard.IndexShard;
import org.elasticsearch.index.shard.IndexShardClosedException;
import org.elasticsearch.index.shard.ShardId;
import org.elasticsearch.indices.recovery.PeerRecoveryTargetService;
import org.elasticsearch.indices.recovery.RecoveryFailedException;
import org.elasticsearch.indices.recovery.RecoveryTarget;
import org.elasticsearch.threadpool.ThreadPool;

public class RecoveriesCollection {
    private final ConcurrentMap<Long, RecoveryTarget> onGoingRecoveries = ConcurrentCollections.newConcurrentMap();
    private final Logger logger;
    private final ThreadPool threadPool;
    private final Callback<Long> ensureClusterStateVersionCallback;

    public RecoveriesCollection(Logger logger, ThreadPool threadPool, Callback<Long> ensureClusterStateVersionCallback) {
        this.logger = logger;
        this.threadPool = threadPool;
        this.ensureClusterStateVersionCallback = ensureClusterStateVersionCallback;
    }

    public long startRecovery(IndexShard indexShard, DiscoveryNode sourceNode, PeerRecoveryTargetService.RecoveryListener listener, TimeValue activityTimeout) {
        RecoveryTarget status = new RecoveryTarget(indexShard, sourceNode, listener, this.ensureClusterStateVersionCallback);
        RecoveryTarget existingStatus = this.onGoingRecoveries.putIfAbsent(status.recoveryId(), status);
        assert (existingStatus == null) : "found two RecoveryStatus instances with the same id";
        this.logger.trace("{} started recovery from {}, id [{}]", (Object)indexShard.shardId(), (Object)sourceNode, (Object)status.recoveryId());
        this.threadPool.schedule(activityTimeout, "generic", new RecoveryMonitor(status.recoveryId(), status.lastAccessTime(), activityTimeout));
        return status.recoveryId();
    }

    public void resetRecovery(long id, ShardId shardId) throws IOException {
        try (RecoveryRef ref = this.getRecoverySafe(id, shardId);){
            RecoveryTarget status = ref.status();
            RecoveryTarget resetRecovery = status.resetRecovery();
            if (!this.onGoingRecoveries.replace(id, status, resetRecovery)) {
                resetRecovery.cancel("replace failed");
                throw new IllegalStateException("failed to replace recovery target");
            }
        }
    }

    public RecoveryRef getRecovery(long id) {
        RecoveryTarget status = (RecoveryTarget)this.onGoingRecoveries.get(id);
        if (status != null && status.tryIncRef()) {
            return new RecoveryRef(status);
        }
        return null;
    }

    public RecoveryRef getRecoverySafe(long id, ShardId shardId) {
        RecoveryRef recoveryRef = this.getRecovery(id);
        if (recoveryRef == null) {
            throw new IndexShardClosedException(shardId);
        }
        assert (recoveryRef.status().shardId().equals(shardId));
        return recoveryRef;
    }

    public boolean cancelRecovery(long id, String reason) {
        RecoveryTarget removed = (RecoveryTarget)this.onGoingRecoveries.remove(id);
        boolean cancelled = false;
        if (removed != null) {
            this.logger.trace("{} canceled recovery from {}, id [{}] (reason [{}])", (Object)removed.shardId(), (Object)removed.sourceNode(), (Object)removed.recoveryId(), (Object)reason);
            removed.cancel(reason);
            cancelled = true;
        }
        return cancelled;
    }

    public void failRecovery(long id, RecoveryFailedException e, boolean sendShardFailure) {
        RecoveryTarget removed = (RecoveryTarget)this.onGoingRecoveries.remove(id);
        if (removed != null) {
            this.logger.trace("{} failing recovery from {}, id [{}]. Send shard failure: [{}]", (Object)removed.shardId(), (Object)removed.sourceNode(), (Object)removed.recoveryId(), (Object)sendShardFailure);
            removed.fail(e, sendShardFailure);
        }
    }

    public void markRecoveryAsDone(long id) {
        RecoveryTarget removed = (RecoveryTarget)this.onGoingRecoveries.remove(id);
        if (removed != null) {
            this.logger.trace("{} marking recovery from {} as done, id [{}]", (Object)removed.shardId(), (Object)removed.sourceNode(), (Object)removed.recoveryId());
            removed.markAsDone();
        }
    }

    public int size() {
        return this.onGoingRecoveries.size();
    }

    public boolean cancelRecoveriesForShard(ShardId shardId, String reason) {
        boolean cancelled = false;
        for (RecoveryTarget status : this.onGoingRecoveries.values()) {
            if (!status.shardId().equals(shardId)) continue;
            cancelled |= this.cancelRecovery(status.recoveryId(), reason);
        }
        return cancelled;
    }

    private class RecoveryMonitor
    extends AbstractRunnable {
        private final long recoveryId;
        private final TimeValue checkInterval;
        private long lastSeenAccessTime;

        private RecoveryMonitor(long recoveryId, long lastSeenAccessTime, TimeValue checkInterval) {
            this.recoveryId = recoveryId;
            this.checkInterval = checkInterval;
            this.lastSeenAccessTime = lastSeenAccessTime;
        }

        @Override
        public void onFailure(Exception e) {
            RecoveriesCollection.this.logger.error(() -> new ParameterizedMessage("unexpected error while monitoring recovery [{}]", (Object)this.recoveryId), (Throwable)e);
        }

        @Override
        protected void doRun() throws Exception {
            RecoveryTarget status = (RecoveryTarget)RecoveriesCollection.this.onGoingRecoveries.get(this.recoveryId);
            if (status == null) {
                RecoveriesCollection.this.logger.trace("[monitor] no status found for [{}], shutting down", (Object)this.recoveryId);
                return;
            }
            long accessTime = status.lastAccessTime();
            if (accessTime == this.lastSeenAccessTime) {
                String message = "no activity after [" + this.checkInterval + "]";
                RecoveriesCollection.this.failRecovery(this.recoveryId, new RecoveryFailedException(status.state(), message, (Throwable)new ElasticsearchTimeoutException(message, new Object[0])), true);
                return;
            }
            this.lastSeenAccessTime = accessTime;
            RecoveriesCollection.this.logger.trace("[monitor] rescheduling check for [{}]. last access time is [{}]", (Object)this.recoveryId, (Object)this.lastSeenAccessTime);
            RecoveriesCollection.this.threadPool.schedule(this.checkInterval, "generic", this);
        }
    }

    public static class RecoveryRef
    implements AutoCloseable {
        private final RecoveryTarget status;
        private final AtomicBoolean closed = new AtomicBoolean(false);

        public RecoveryRef(RecoveryTarget status) {
            this.status = status;
            this.status.setLastAccessTime();
        }

        @Override
        public void close() {
            if (this.closed.compareAndSet(false, true)) {
                this.status.decRef();
            }
        }

        public RecoveryTarget status() {
            return this.status;
        }
    }
}

