<?php

namespace Drupal\Tests\Component\Utility;

use Drupal\Component\Utility\Rectangle;
use Drupal\Tests\UnitTestCase;

/**
 * @coversDefaultClass \Drupal\Component\Utility\Rectangle
 * @group Image
 */
class RectangleTest extends UnitTestCase {

  /**
   * Tests wrong rectangle width.
   *
   * @covers ::rotate
   *
   * @expectedException \InvalidArgumentException
   */
  public function testWrongWidth() {
    $rect = new Rectangle(-40, 20);
  }

  /**
   * Tests wrong rectangle height.
   *
   * @covers ::rotate
   *
   * @expectedException \InvalidArgumentException
   */
  public function testWrongHeight() {
    $rect = new Rectangle(40, 0);
  }

  /**
   * Tests getting rectangle dimensions after a rotation operation.
   *
   * @param int $width
   *   The width of the rectangle.
   * @param int $height
   *   The height of the rectangle.
   * @param float $angle
   *   The angle for rotation.
   * @param int $exp_width
   *   The expected width of the rotated rectangle.
   * @param int $exp_height
   *   The expected height of the rotated rectangle.
   *
   * @covers ::rotate
   * @covers ::getBoundingWidth
   * @covers ::getBoundingHeight
   *
   * @dataProvider providerPhp55RotateDimensions
   */
  public function testRotateDimensions($width, $height, $angle, $exp_width, $exp_height) {
    $rect = new Rectangle($width, $height);
    $rect->rotate($angle);
    $this->assertEquals($exp_width, $rect->getBoundingWidth());
    $this->assertEquals($exp_height, $rect->getBoundingHeight());
  }

  /**
   * Provides data for image dimension rotation tests.
   *
   * This dataset sample was generated by running on PHP 5.5 the function below
   * - first, for all integer rotation angles (-360 to 360) on a rectangle
   *   40x20;
   * - second, for 500 random float rotation angle in the range -360 to 360 on
   *   a rectangle 40x20;
   * - third, on 1000 rectangles of random WxH rotated to a random float angle
   *   in the range -360 to 360
   * - fourth, on 2000 rectangles of random WxH rotated to a random integer
   *   angle multiple of 30 degrees in the range -360 to 360 (which is the most
   *   tricky case).
   * Using the GD toolkit operations gives us true data coming from the GD
   * library that can be used to match against the Rectangle class under test.
   * @code
   *   protected function rotateResults($width, $height, $angle, &$new_width, &$new_height) {
   *     $image = \Drupal::service('image.factory')->get(NULL, 'gd');
   *     $image->createNew($width, $height);
   *     $old_res = $image->getToolkit()->getResource();
   *     $image->rotate($angle);
   *     $new_width = $image->getWidth();
   *     $new_height = $image->getHeight();
   *     if (is_resource($old_res)) {
   *       imagedestroy($old_res);
   *     }
   *   }
   * @endcode
   *
   * @return array[]
   *   A simple array of simple arrays, each having the following elements:
   *   - original image width
   *   - original image height
   *   - rotation angle in degrees
   *   - expected image width after rotation
   *   - expected image height after rotation
   *
   * @see testRotateDimensions()
   */
  public function providerPhp55RotateDimensions() {
    return [
      [ 40, 20, -360, 40, 20 ],
      [ 40, 20, -359, 39, 20 ],
      [ 40, 20, -358, 40, 21 ],
      [ 40, 20, -357, 40, 22 ],
      [ 40, 20, -356, 40, 22 ],
      [ 40, 20, -355, 41, 23 ],
      [ 40, 20, -354, 41, 24 ],
      [ 40, 20, -353, 41, 24 ],
      [ 40, 20, -352, 42, 25 ],
      [ 40, 20, -351, 42, 26 ],
      [ 40, 20, -350, 42, 26 ],
      [ 40, 20, -349, 43, 27 ],
      [ 40, 20, -348, 43, 28 ],
      [ 40, 20, -347, 42, 27 ],
      [ 40, 20, -346, 43, 28 ],
      [ 40, 20, -345, 43, 29 ],
      [ 40, 20, -344, 44, 30 ],
      [ 40, 20, -343, 44, 30 ],
      [ 40, 20, -342, 44, 31 ],
      [ 40, 20, -341, 44, 32 ],
      [ 40, 20, -340, 44, 32 ],
      [ 40, 20, -339, 44, 33 ],
      [ 40, 20, -338, 44, 33 ],
      [ 40, 20, -337, 44, 33 ],
      [ 40, 20, -336, 44, 34 ],
      [ 40, 20, -335, 44, 34 ],
      [ 40, 20, -334, 44, 35 ],
      [ 40, 20, -333, 44, 36 ],
      [ 40, 20, -332, 44, 36 ],
      [ 40, 20, -331, 44, 36 ],
      [ 40, 20, -330, 44, 36 ],
      [ 40, 20, -329, 44, 37 ],
      [ 40, 20, -328, 44, 38 ],
      [ 40, 20, -327, 44, 38 ],
      [ 40, 20, -326, 44, 39 ],
      [ 40, 20, -325, 43, 38 ],
      [ 40, 20, -324, 44, 39 ],
      [ 40, 20, -323, 43, 40 ],
      [ 40, 20, -322, 43, 40 ],
      [ 40, 20, -321, 44, 41 ],
      [ 40, 20, -320, 43, 40 ],
      [ 40, 20, -319, 43, 41 ],
      [ 40, 20, -318, 42, 41 ],
      [ 40, 20, -317, 43, 42 ],
      [ 40, 20, -316, 42, 41 ],
      [ 40, 20, -315, 42, 42 ],
      [ 40, 20, -314, 41, 42 ],
      [ 40, 20, -313, 42, 43 ],
      [ 40, 20, -312, 41, 42 ],
      [ 40, 20, -311, 41, 43 ],
      [ 40, 20, -310, 40, 43 ],
      [ 40, 20, -309, 41, 44 ],
      [ 40, 20, -308, 40, 43 ],
      [ 40, 20, -307, 40, 43 ],
      [ 40, 20, -306, 39, 44 ],
      [ 40, 20, -305, 38, 43 ],
      [ 40, 20, -304, 39, 44 ],
      [ 40, 20, -303, 38, 44 ],
      [ 40, 20, -302, 38, 44 ],
      [ 40, 20, -301, 37, 44 ],
      [ 40, 20, -300, 36, 44 ],
      [ 40, 20, -299, 36, 44 ],
      [ 40, 20, -298, 36, 44 ],
      [ 40, 20, -297, 36, 44 ],
      [ 40, 20, -296, 35, 44 ],
      [ 40, 20, -295, 34, 44 ],
      [ 40, 20, -294, 34, 44 ],
      [ 40, 20, -293, 33, 44 ],
      [ 40, 20, -292, 33, 44 ],
      [ 40, 20, -291, 33, 44 ],
      [ 40, 20, -290, 32, 44 ],
      [ 40, 20, -289, 32, 44 ],
      [ 40, 20, -288, 31, 44 ],
      [ 40, 20, -287, 30, 44 ],
      [ 40, 20, -286, 30, 44 ],
      [ 40, 20, -285, 29, 43 ],
      [ 40, 20, -284, 28, 43 ],
      [ 40, 20, -283, 27, 42 ],
      [ 40, 20, -282, 28, 43 ],
      [ 40, 20, -281, 27, 43 ],
      [ 40, 20, -280, 26, 42 ],
      [ 40, 20, -279, 26, 42 ],
      [ 40, 20, -278, 25, 42 ],
      [ 40, 20, -277, 24, 41 ],
      [ 40, 20, -276, 24, 41 ],
      [ 40, 20, -275, 23, 41 ],
      [ 40, 20, -274, 22, 40 ],
      [ 40, 20, -273, 22, 40 ],
      [ 40, 20, -272, 21, 40 ],
      [ 40, 20, -271, 20, 39 ],
      [ 40, 20, -270, 20, 40 ],
      [ 40, 20, -269, 20, 39 ],
      [ 40, 20, -268, 21, 39 ],
      [ 40, 20, -267, 22, 39 ],
      [ 40, 20, -266, 22, 39 ],
      [ 40, 20, -265, 23, 40 ],
      [ 40, 20, -264, 24, 40 ],
      [ 40, 20, -263, 24, 40 ],
      [ 40, 20, -262, 25, 41 ],
      [ 40, 20, -261, 26, 41 ],
      [ 40, 20, -260, 26, 41 ],
      [ 40, 20, -259, 27, 42 ],
      [ 40, 20, -258, 28, 42 ],
      [ 40, 20, -257, 27, 41 ],
      [ 40, 20, -256, 28, 42 ],
      [ 40, 20, -255, 29, 42 ],
      [ 40, 20, -254, 30, 43 ],
      [ 40, 20, -253, 30, 43 ],
      [ 40, 20, -252, 31, 43 ],
      [ 40, 20, -251, 32, 43 ],
      [ 40, 20, -250, 32, 43 ],
      [ 40, 20, -249, 33, 43 ],
      [ 40, 20, -248, 33, 43 ],
      [ 40, 20, -247, 33, 43 ],
      [ 40, 20, -246, 34, 43 ],
      [ 40, 20, -245, 34, 43 ],
      [ 40, 20, -244, 35, 43 ],
      [ 40, 20, -243, 36, 43 ],
      [ 40, 20, -242, 36, 43 ],
      [ 40, 20, -241, 36, 43 ],
      [ 40, 20, -240, 36, 43 ],
      [ 40, 20, -239, 37, 43 ],
      [ 40, 20, -238, 38, 43 ],
      [ 40, 20, -237, 38, 43 ],
      [ 40, 20, -236, 39, 43 ],
      [ 40, 20, -235, 38, 42 ],
      [ 40, 20, -234, 39, 43 ],
      [ 40, 20, -233, 40, 42 ],
      [ 40, 20, -232, 40, 42 ],
      [ 40, 20, -231, 41, 43 ],
      [ 40, 20, -230, 40, 42 ],
      [ 40, 20, -229, 41, 42 ],
      [ 40, 20, -228, 41, 41 ],
      [ 40, 20, -227, 42, 42 ],
      [ 40, 20, -226, 41, 41 ],
      [ 40, 20, -225, 42, 41 ],
      [ 40, 20, -224, 42, 40 ],
      [ 40, 20, -223, 43, 41 ],
      [ 40, 20, -222, 42, 40 ],
      [ 40, 20, -221, 43, 40 ],
      [ 40, 20, -220, 43, 39 ],
      [ 40, 20, -219, 44, 40 ],
      [ 40, 20, -218, 43, 39 ],
      [ 40, 20, -217, 43, 39 ],
      [ 40, 20, -216, 44, 38 ],
      [ 40, 20, -215, 43, 37 ],
      [ 40, 20, -214, 44, 38 ],
      [ 40, 20, -213, 44, 37 ],
      [ 40, 20, -212, 44, 37 ],
      [ 40, 20, -211, 44, 36 ],
      [ 40, 20, -210, 44, 35 ],
      [ 40, 20, -209, 44, 35 ],
      [ 40, 20, -208, 44, 35 ],
      [ 40, 20, -207, 44, 35 ],
      [ 40, 20, -206, 44, 34 ],
      [ 40, 20, -205, 44, 33 ],
      [ 40, 20, -204, 44, 33 ],
      [ 40, 20, -203, 44, 32 ],
      [ 40, 20, -202, 44, 32 ],
      [ 40, 20, -201, 44, 32 ],
      [ 40, 20, -200, 44, 31 ],
      [ 40, 20, -199, 44, 31 ],
      [ 40, 20, -198, 44, 30 ],
      [ 40, 20, -197, 44, 29 ],
      [ 40, 20, -196, 44, 29 ],
      [ 40, 20, -195, 43, 28 ],
      [ 40, 20, -194, 43, 27 ],
      [ 40, 20, -193, 42, 26 ],
      [ 40, 20, -192, 43, 27 ],
      [ 40, 20, -191, 43, 26 ],
      [ 40, 20, -190, 42, 25 ],
      [ 40, 20, -189, 42, 25 ],
      [ 40, 20, -188, 42, 24 ],
      [ 40, 20, -187, 41, 23 ],
      [ 40, 20, -186, 41, 23 ],
      [ 40, 20, -185, 41, 22 ],
      [ 40, 20, -184, 40, 21 ],
      [ 40, 20, -183, 40, 21 ],
      [ 40, 20, -182, 40, 20 ],
      [ 40, 20, -181, 39, 19 ],
      [ 40, 20, -180, 40, 20 ],
      [ 40, 20, -179, 39, 19 ],
      [ 40, 20, -178, 39, 20 ],
      [ 40, 20, -177, 39, 21 ],
      [ 40, 20, -176, 39, 21 ],
      [ 40, 20, -175, 40, 22 ],
      [ 40, 20, -174, 40, 23 ],
      [ 40, 20, -173, 40, 23 ],
      [ 40, 20, -172, 41, 24 ],
      [ 40, 20, -171, 41, 25 ],
      [ 40, 20, -170, 41, 25 ],
      [ 40, 20, -169, 42, 26 ],
      [ 40, 20, -168, 42, 27 ],
      [ 40, 20, -167, 41, 26 ],
      [ 40, 20, -166, 42, 27 ],
      [ 40, 20, -165, 42, 28 ],
      [ 40, 20, -164, 43, 29 ],
      [ 40, 20, -163, 43, 29 ],
      [ 40, 20, -162, 43, 30 ],
      [ 40, 20, -161, 43, 31 ],
      [ 40, 20, -160, 43, 31 ],
      [ 40, 20, -159, 43, 32 ],
      [ 40, 20, -158, 43, 32 ],
      [ 40, 20, -157, 43, 32 ],
      [ 40, 20, -156, 43, 33 ],
      [ 40, 20, -155, 43, 33 ],
      [ 40, 20, -154, 43, 34 ],
      [ 40, 20, -153, 43, 35 ],
      [ 40, 20, -152, 43, 35 ],
      [ 40, 20, -151, 43, 35 ],
      [ 40, 20, -150, 43, 36 ],
      [ 40, 20, -149, 43, 36 ],
      [ 40, 20, -148, 43, 37 ],
      [ 40, 20, -147, 43, 37 ],
      [ 40, 20, -146, 43, 38 ],
      [ 40, 20, -145, 42, 37 ],
      [ 40, 20, -144, 43, 38 ],
      [ 40, 20, -143, 42, 39 ],
      [ 40, 20, -142, 42, 39 ],
      [ 40, 20, -141, 43, 40 ],
      [ 40, 20, -140, 42, 39 ],
      [ 40, 20, -139, 42, 40 ],
      [ 40, 20, -138, 41, 40 ],
      [ 40, 20, -137, 42, 41 ],
      [ 40, 20, -136, 41, 40 ],
      [ 40, 20, -135, 41, 41 ],
      [ 40, 20, -134, 40, 41 ],
      [ 40, 20, -133, 41, 42 ],
      [ 40, 20, -132, 40, 41 ],
      [ 40, 20, -131, 40, 42 ],
      [ 40, 20, -130, 39, 42 ],
      [ 40, 20, -129, 40, 43 ],
      [ 40, 20, -128, 39, 42 ],
      [ 40, 20, -127, 39, 42 ],
      [ 40, 20, -126, 38, 43 ],
      [ 40, 20, -125, 37, 42 ],
      [ 40, 20, -124, 38, 43 ],
      [ 40, 20, -123, 37, 43 ],
      [ 40, 20, -122, 37, 43 ],
      [ 40, 20, -121, 36, 43 ],
      [ 40, 20, -120, 36, 43 ],
      [ 40, 20, -119, 35, 43 ],
      [ 40, 20, -118, 35, 43 ],
      [ 40, 20, -117, 35, 43 ],
      [ 40, 20, -116, 34, 43 ],
      [ 40, 20, -115, 33, 43 ],
      [ 40, 20, -114, 33, 43 ],
      [ 40, 20, -113, 32, 43 ],
      [ 40, 20, -112, 32, 43 ],
      [ 40, 20, -111, 32, 43 ],
      [ 40, 20, -110, 31, 43 ],
      [ 40, 20, -109, 31, 43 ],
      [ 40, 20, -108, 30, 43 ],
      [ 40, 20, -107, 29, 43 ],
      [ 40, 20, -106, 29, 43 ],
      [ 40, 20, -105, 28, 42 ],
      [ 40, 20, -104, 27, 42 ],
      [ 40, 20, -103, 26, 41 ],
      [ 40, 20, -102, 27, 42 ],
      [ 40, 20, -101, 26, 42 ],
      [ 40, 20, -100, 25, 41 ],
      [ 40, 20, -99, 25, 41 ],
      [ 40, 20, -98, 24, 41 ],
      [ 40, 20, -97, 23, 40 ],
      [ 40, 20, -96, 23, 40 ],
      [ 40, 20, -95, 22, 40 ],
      [ 40, 20, -94, 21, 39 ],
      [ 40, 20, -93, 21, 39 ],
      [ 40, 20, -92, 20, 39 ],
      [ 40, 20, -91, 19, 39 ],
      [ 40, 20, -90, 20, 40 ],
      [ 40, 20, -89, 19, 39 ],
      [ 40, 20, -88, 20, 40 ],
      [ 40, 20, -87, 21, 40 ],
      [ 40, 20, -86, 21, 40 ],
      [ 40, 20, -85, 22, 41 ],
      [ 40, 20, -84, 23, 41 ],
      [ 40, 20, -83, 23, 41 ],
      [ 40, 20, -82, 24, 42 ],
      [ 40, 20, -81, 25, 42 ],
      [ 40, 20, -80, 25, 42 ],
      [ 40, 20, -79, 26, 43 ],
      [ 40, 20, -78, 27, 43 ],
      [ 40, 20, -77, 26, 42 ],
      [ 40, 20, -76, 27, 43 ],
      [ 40, 20, -75, 28, 43 ],
      [ 40, 20, -74, 29, 44 ],
      [ 40, 20, -73, 29, 44 ],
      [ 40, 20, -72, 30, 44 ],
      [ 40, 20, -71, 31, 44 ],
      [ 40, 20, -70, 31, 44 ],
      [ 40, 20, -69, 32, 44 ],
      [ 40, 20, -68, 32, 44 ],
      [ 40, 20, -67, 32, 44 ],
      [ 40, 20, -66, 33, 44 ],
      [ 40, 20, -65, 33, 44 ],
      [ 40, 20, -64, 34, 44 ],
      [ 40, 20, -63, 35, 44 ],
      [ 40, 20, -62, 35, 44 ],
      [ 40, 20, -61, 35, 44 ],
      [ 40, 20, -60, 35, 44 ],
      [ 40, 20, -59, 36, 44 ],
      [ 40, 20, -58, 37, 44 ],
      [ 40, 20, -57, 37, 44 ],
      [ 40, 20, -56, 38, 44 ],
      [ 40, 20, -55, 37, 43 ],
      [ 40, 20, -54, 38, 44 ],
      [ 40, 20, -53, 39, 43 ],
      [ 40, 20, -52, 39, 43 ],
      [ 40, 20, -51, 40, 44 ],
      [ 40, 20, -50, 39, 43 ],
      [ 40, 20, -49, 40, 43 ],
      [ 40, 20, -48, 40, 42 ],
      [ 40, 20, -47, 41, 43 ],
      [ 40, 20, -46, 40, 42 ],
      [ 40, 20, -45, 41, 42 ],
      [ 40, 20, -44, 41, 41 ],
      [ 40, 20, -43, 42, 42 ],
      [ 40, 20, -42, 41, 41 ],
      [ 40, 20, -41, 42, 41 ],
      [ 40, 20, -40, 42, 40 ],
      [ 40, 20, -39, 43, 41 ],
      [ 40, 20, -38, 42, 40 ],
      [ 40, 20, -37, 42, 40 ],
      [ 40, 20, -36, 43, 39 ],
      [ 40, 20, -35, 42, 38 ],
      [ 40, 20, -34, 43, 39 ],
      [ 40, 20, -33, 43, 38 ],
      [ 40, 20, -32, 43, 38 ],
      [ 40, 20, -31, 43, 37 ],
      [ 40, 20, -30, 43, 37 ],
      [ 40, 20, -29, 43, 36 ],
      [ 40, 20, -28, 43, 36 ],
      [ 40, 20, -27, 43, 36 ],
      [ 40, 20, -26, 43, 35 ],
      [ 40, 20, -25, 43, 34 ],
      [ 40, 20, -24, 43, 34 ],
      [ 40, 20, -23, 43, 33 ],
      [ 40, 20, -22, 43, 33 ],
      [ 40, 20, -21, 43, 33 ],
      [ 40, 20, -20, 43, 32 ],
      [ 40, 20, -19, 43, 32 ],
      [ 40, 20, -18, 43, 31 ],
      [ 40, 20, -17, 43, 30 ],
      [ 40, 20, -16, 43, 30 ],
      [ 40, 20, -15, 42, 29 ],
      [ 40, 20, -14, 42, 28 ],
      [ 40, 20, -13, 41, 27 ],
      [ 40, 20, -12, 42, 28 ],
      [ 40, 20, -11, 42, 27 ],
      [ 40, 20, -10, 41, 26 ],
      [ 40, 20, -9, 41, 26 ],
      [ 40, 20, -8, 41, 25 ],
      [ 40, 20, -7, 40, 24 ],
      [ 40, 20, -6, 40, 24 ],
      [ 40, 20, -5, 40, 23 ],
      [ 40, 20, -4, 39, 22 ],
      [ 40, 20, -3, 39, 22 ],
      [ 40, 20, -2, 39, 21 ],
      [ 40, 20, -1, 39, 20 ],
      [ 40, 20, 0, 40, 20 ],
      [ 40, 20, 1, 39, 20 ],
      [ 40, 20, 2, 40, 21 ],
      [ 40, 20, 3, 40, 22 ],
      [ 40, 20, 4, 40, 22 ],
      [ 40, 20, 5, 41, 23 ],
      [ 40, 20, 6, 41, 24 ],
      [ 40, 20, 7, 41, 24 ],
      [ 40, 20, 8, 42, 25 ],
      [ 40, 20, 9, 42, 26 ],
      [ 40, 20, 10, 42, 26 ],
      [ 40, 20, 11, 43, 27 ],
      [ 40, 20, 12, 43, 28 ],
      [ 40, 20, 13, 42, 27 ],
      [ 40, 20, 14, 43, 28 ],
      [ 40, 20, 15, 43, 29 ],
      [ 40, 20, 16, 44, 30 ],
      [ 40, 20, 17, 44, 30 ],
      [ 40, 20, 18, 44, 31 ],
      [ 40, 20, 19, 44, 32 ],
      [ 40, 20, 20, 44, 32 ],
      [ 40, 20, 21, 44, 33 ],
      [ 40, 20, 22, 44, 33 ],
      [ 40, 20, 23, 44, 33 ],
      [ 40, 20, 24, 44, 34 ],
      [ 40, 20, 25, 44, 34 ],
      [ 40, 20, 26, 44, 35 ],
      [ 40, 20, 27, 44, 36 ],
      [ 40, 20, 28, 44, 36 ],
      [ 40, 20, 29, 44, 36 ],
      [ 40, 20, 30, 44, 36 ],
      [ 40, 20, 31, 44, 37 ],
      [ 40, 20, 32, 44, 38 ],
      [ 40, 20, 33, 44, 38 ],
      [ 40, 20, 34, 44, 39 ],
      [ 40, 20, 35, 43, 38 ],
      [ 40, 20, 36, 44, 39 ],
      [ 40, 20, 37, 43, 40 ],
      [ 40, 20, 38, 43, 40 ],
      [ 40, 20, 39, 44, 41 ],
      [ 40, 20, 40, 43, 40 ],
      [ 40, 20, 41, 43, 41 ],
      [ 40, 20, 42, 42, 41 ],
      [ 40, 20, 43, 43, 42 ],
      [ 40, 20, 44, 42, 41 ],
      [ 40, 20, 45, 42, 42 ],
      [ 40, 20, 46, 41, 42 ],
      [ 40, 20, 47, 42, 43 ],
      [ 40, 20, 48, 41, 42 ],
      [ 40, 20, 49, 41, 43 ],
      [ 40, 20, 50, 40, 43 ],
      [ 40, 20, 51, 41, 44 ],
      [ 40, 20, 52, 40, 43 ],
      [ 40, 20, 53, 40, 43 ],
      [ 40, 20, 54, 39, 44 ],
      [ 40, 20, 55, 38, 43 ],
      [ 40, 20, 56, 39, 44 ],
      [ 40, 20, 57, 38, 44 ],
      [ 40, 20, 58, 38, 44 ],
      [ 40, 20, 59, 37, 44 ],
      [ 40, 20, 60, 36, 44 ],
      [ 40, 20, 61, 36, 44 ],
      [ 40, 20, 62, 36, 44 ],
      [ 40, 20, 63, 36, 44 ],
      [ 40, 20, 64, 35, 44 ],
      [ 40, 20, 65, 34, 44 ],
      [ 40, 20, 66, 34, 44 ],
      [ 40, 20, 67, 33, 44 ],
      [ 40, 20, 68, 33, 44 ],
      [ 40, 20, 69, 33, 44 ],
      [ 40, 20, 70, 32, 44 ],
      [ 40, 20, 71, 32, 44 ],
      [ 40, 20, 72, 31, 44 ],
      [ 40, 20, 73, 30, 44 ],
      [ 40, 20, 74, 30, 44 ],
      [ 40, 20, 75, 29, 43 ],
      [ 40, 20, 76, 28, 43 ],
      [ 40, 20, 77, 27, 42 ],
      [ 40, 20, 78, 28, 43 ],
      [ 40, 20, 79, 27, 43 ],
      [ 40, 20, 80, 26, 42 ],
      [ 40, 20, 81, 26, 42 ],
      [ 40, 20, 82, 25, 42 ],
      [ 40, 20, 83, 24, 41 ],
      [ 40, 20, 84, 24, 41 ],
      [ 40, 20, 85, 23, 41 ],
      [ 40, 20, 86, 22, 40 ],
      [ 40, 20, 87, 22, 40 ],
      [ 40, 20, 88, 21, 40 ],
      [ 40, 20, 89, 20, 39 ],
      [ 40, 20, 90, 20, 40 ],
      [ 40, 20, 91, 20, 39 ],
      [ 40, 20, 92, 21, 39 ],
      [ 40, 20, 93, 22, 39 ],
      [ 40, 20, 94, 22, 39 ],
      [ 40, 20, 95, 23, 40 ],
      [ 40, 20, 96, 24, 40 ],
      [ 40, 20, 97, 24, 40 ],
      [ 40, 20, 98, 25, 41 ],
      [ 40, 20, 99, 26, 41 ],
      [ 40, 20, 100, 26, 41 ],
      [ 40, 20, 101, 27, 42 ],
      [ 40, 20, 102, 28, 42 ],
      [ 40, 20, 103, 27, 41 ],
      [ 40, 20, 104, 28, 42 ],
      [ 40, 20, 105, 29, 42 ],
      [ 40, 20, 106, 30, 43 ],
      [ 40, 20, 107, 30, 43 ],
      [ 40, 20, 108, 31, 43 ],
      [ 40, 20, 109, 32, 43 ],
      [ 40, 20, 110, 32, 43 ],
      [ 40, 20, 111, 33, 43 ],
      [ 40, 20, 112, 33, 43 ],
      [ 40, 20, 113, 33, 43 ],
      [ 40, 20, 114, 34, 43 ],
      [ 40, 20, 115, 34, 43 ],
      [ 40, 20, 116, 35, 43 ],
      [ 40, 20, 117, 36, 43 ],
      [ 40, 20, 118, 36, 43 ],
      [ 40, 20, 119, 36, 43 ],
      [ 40, 20, 120, 36, 43 ],
      [ 40, 20, 121, 37, 43 ],
      [ 40, 20, 122, 38, 43 ],
      [ 40, 20, 123, 38, 43 ],
      [ 40, 20, 124, 39, 43 ],
      [ 40, 20, 125, 38, 42 ],
      [ 40, 20, 126, 39, 43 ],
      [ 40, 20, 127, 40, 42 ],
      [ 40, 20, 128, 40, 42 ],
      [ 40, 20, 129, 41, 43 ],
      [ 40, 20, 130, 40, 42 ],
      [ 40, 20, 131, 41, 42 ],
      [ 40, 20, 132, 41, 41 ],
      [ 40, 20, 133, 42, 42 ],
      [ 40, 20, 134, 41, 41 ],
      [ 40, 20, 135, 42, 41 ],
      [ 40, 20, 136, 42, 40 ],
      [ 40, 20, 137, 43, 41 ],
      [ 40, 20, 138, 42, 40 ],
      [ 40, 20, 139, 43, 40 ],
      [ 40, 20, 140, 43, 39 ],
      [ 40, 20, 141, 44, 40 ],
      [ 40, 20, 142, 43, 39 ],
      [ 40, 20, 143, 43, 39 ],
      [ 40, 20, 144, 44, 38 ],
      [ 40, 20, 145, 43, 37 ],
      [ 40, 20, 146, 44, 38 ],
      [ 40, 20, 147, 44, 37 ],
      [ 40, 20, 148, 44, 37 ],
      [ 40, 20, 149, 44, 36 ],
      [ 40, 20, 150, 44, 35 ],
      [ 40, 20, 151, 44, 35 ],
      [ 40, 20, 152, 44, 35 ],
      [ 40, 20, 153, 44, 35 ],
      [ 40, 20, 154, 44, 34 ],
      [ 40, 20, 155, 44, 33 ],
      [ 40, 20, 156, 44, 33 ],
      [ 40, 20, 157, 44, 32 ],
      [ 40, 20, 158, 44, 32 ],
      [ 40, 20, 159, 44, 32 ],
      [ 40, 20, 160, 44, 31 ],
      [ 40, 20, 161, 44, 31 ],
      [ 40, 20, 162, 44, 30 ],
      [ 40, 20, 163, 44, 29 ],
      [ 40, 20, 164, 44, 29 ],
      [ 40, 20, 165, 43, 28 ],
      [ 40, 20, 166, 43, 27 ],
      [ 40, 20, 167, 42, 26 ],
      [ 40, 20, 168, 43, 27 ],
      [ 40, 20, 169, 43, 26 ],
      [ 40, 20, 170, 42, 25 ],
      [ 40, 20, 171, 42, 25 ],
      [ 40, 20, 172, 42, 24 ],
      [ 40, 20, 173, 41, 23 ],
      [ 40, 20, 174, 41, 23 ],
      [ 40, 20, 175, 41, 22 ],
      [ 40, 20, 176, 40, 21 ],
      [ 40, 20, 177, 40, 21 ],
      [ 40, 20, 178, 40, 20 ],
      [ 40, 20, 179, 39, 19 ],
      [ 40, 20, 180, 40, 20 ],
      [ 40, 20, 181, 39, 19 ],
      [ 40, 20, 182, 39, 20 ],
      [ 40, 20, 183, 39, 21 ],
      [ 40, 20, 184, 39, 21 ],
      [ 40, 20, 185, 40, 22 ],
      [ 40, 20, 186, 40, 23 ],
      [ 40, 20, 187, 40, 23 ],
      [ 40, 20, 188, 41, 24 ],
      [ 40, 20, 189, 41, 25 ],
      [ 40, 20, 190, 41, 25 ],
      [ 40, 20, 191, 42, 26 ],
      [ 40, 20, 192, 42, 27 ],
      [ 40, 20, 193, 41, 26 ],
      [ 40, 20, 194, 42, 27 ],
      [ 40, 20, 195, 42, 28 ],
      [ 40, 20, 196, 43, 29 ],
      [ 40, 20, 197, 43, 29 ],
      [ 40, 20, 198, 43, 30 ],
      [ 40, 20, 199, 43, 31 ],
      [ 40, 20, 200, 43, 31 ],
      [ 40, 20, 201, 43, 32 ],
      [ 40, 20, 202, 43, 32 ],
      [ 40, 20, 203, 43, 32 ],
      [ 40, 20, 204, 43, 33 ],
      [ 40, 20, 205, 43, 33 ],
      [ 40, 20, 206, 43, 34 ],
      [ 40, 20, 207, 43, 35 ],
      [ 40, 20, 208, 43, 35 ],
      [ 40, 20, 209, 43, 35 ],
      [ 40, 20, 210, 43, 36 ],
      [ 40, 20, 211, 43, 36 ],
      [ 40, 20, 212, 43, 37 ],
      [ 40, 20, 213, 43, 37 ],
      [ 40, 20, 214, 43, 38 ],
      [ 40, 20, 215, 42, 37 ],
      [ 40, 20, 216, 43, 38 ],
      [ 40, 20, 217, 42, 39 ],
      [ 40, 20, 218, 42, 39 ],
      [ 40, 20, 219, 43, 40 ],
      [ 40, 20, 220, 42, 39 ],
      [ 40, 20, 221, 42, 40 ],
      [ 40, 20, 222, 41, 40 ],
      [ 40, 20, 223, 42, 41 ],
      [ 40, 20, 224, 41, 40 ],
      [ 40, 20, 225, 41, 41 ],
      [ 40, 20, 226, 40, 41 ],
      [ 40, 20, 227, 41, 42 ],
      [ 40, 20, 228, 40, 41 ],
      [ 40, 20, 229, 40, 42 ],
      [ 40, 20, 230, 39, 42 ],
      [ 40, 20, 231, 40, 43 ],
      [ 40, 20, 232, 39, 42 ],
      [ 40, 20, 233, 39, 42 ],
      [ 40, 20, 234, 38, 43 ],
      [ 40, 20, 235, 37, 42 ],
      [ 40, 20, 236, 38, 43 ],
      [ 40, 20, 237, 37, 43 ],
      [ 40, 20, 238, 37, 43 ],
      [ 40, 20, 239, 36, 43 ],
      [ 40, 20, 240, 36, 43 ],
      [ 40, 20, 241, 35, 43 ],
      [ 40, 20, 242, 35, 43 ],
      [ 40, 20, 243, 35, 43 ],
      [ 40, 20, 244, 34, 43 ],
      [ 40, 20, 245, 33, 43 ],
      [ 40, 20, 246, 33, 43 ],
      [ 40, 20, 247, 32, 43 ],
      [ 40, 20, 248, 32, 43 ],
      [ 40, 20, 249, 32, 43 ],
      [ 40, 20, 250, 31, 43 ],
      [ 40, 20, 251, 31, 43 ],
      [ 40, 20, 252, 30, 43 ],
      [ 40, 20, 253, 29, 43 ],
      [ 40, 20, 254, 29, 43 ],
      [ 40, 20, 255, 28, 42 ],
      [ 40, 20, 256, 27, 42 ],
      [ 40, 20, 257, 26, 41 ],
      [ 40, 20, 258, 27, 42 ],
      [ 40, 20, 259, 26, 42 ],
      [ 40, 20, 260, 25, 41 ],
      [ 40, 20, 261, 25, 41 ],
      [ 40, 20, 262, 24, 41 ],
      [ 40, 20, 263, 23, 40 ],
      [ 40, 20, 264, 23, 40 ],
      [ 40, 20, 265, 22, 40 ],
      [ 40, 20, 266, 21, 39 ],
      [ 40, 20, 267, 21, 39 ],
      [ 40, 20, 268, 20, 39 ],
      [ 40, 20, 269, 19, 39 ],
      [ 40, 20, 270, 20, 40 ],
      [ 40, 20, 271, 19, 39 ],
      [ 40, 20, 272, 20, 40 ],
      [ 40, 20, 273, 21, 40 ],
      [ 40, 20, 274, 21, 40 ],
      [ 40, 20, 275, 22, 41 ],
      [ 40, 20, 276, 23, 41 ],
      [ 40, 20, 277, 23, 41 ],
      [ 40, 20, 278, 24, 42 ],
      [ 40, 20, 279, 25, 42 ],
      [ 40, 20, 280, 25, 42 ],
      [ 40, 20, 281, 26, 43 ],
      [ 40, 20, 282, 27, 43 ],
      [ 40, 20, 283, 26, 42 ],
      [ 40, 20, 284, 27, 43 ],
      [ 40, 20, 285, 28, 43 ],
      [ 40, 20, 286, 29, 44 ],
      [ 40, 20, 287, 29, 44 ],
      [ 40, 20, 288, 30, 44 ],
      [ 40, 20, 289, 31, 44 ],
      [ 40, 20, 290, 31, 44 ],
      [ 40, 20, 291, 32, 44 ],
      [ 40, 20, 292, 32, 44 ],
      [ 40, 20, 293, 32, 44 ],
      [ 40, 20, 294, 33, 44 ],
      [ 40, 20, 295, 33, 44 ],
      [ 40, 20, 296, 34, 44 ],
      [ 40, 20, 297, 35, 44 ],
      [ 40, 20, 298, 35, 44 ],
      [ 40, 20, 299, 35, 44 ],
      [ 40, 20, 300, 35, 44 ],
      [ 40, 20, 301, 36, 44 ],
      [ 40, 20, 302, 37, 44 ],
      [ 40, 20, 303, 37, 44 ],
      [ 40, 20, 304, 38, 44 ],
      [ 40, 20, 305, 37, 43 ],
      [ 40, 20, 306, 38, 44 ],
      [ 40, 20, 307, 39, 43 ],
      [ 40, 20, 308, 39, 43 ],
      [ 40, 20, 309, 40, 44 ],
      [ 40, 20, 310, 39, 43 ],
      [ 40, 20, 311, 40, 43 ],
      [ 40, 20, 312, 40, 42 ],
      [ 40, 20, 313, 41, 43 ],
      [ 40, 20, 314, 40, 42 ],
      [ 40, 20, 315, 41, 42 ],
      [ 40, 20, 316, 41, 41 ],
      [ 40, 20, 317, 42, 42 ],
      [ 40, 20, 318, 41, 41 ],
      [ 40, 20, 319, 42, 41 ],
      [ 40, 20, 320, 42, 40 ],
      [ 40, 20, 321, 43, 41 ],
      [ 40, 20, 322, 42, 40 ],
      [ 40, 20, 323, 42, 40 ],
      [ 40, 20, 324, 43, 39 ],
      [ 40, 20, 325, 42, 38 ],
      [ 40, 20, 326, 43, 39 ],
      [ 40, 20, 327, 43, 38 ],
      [ 40, 20, 328, 43, 38 ],
      [ 40, 20, 329, 43, 37 ],
      [ 40, 20, 330, 43, 37 ],
      [ 40, 20, 331, 43, 36 ],
      [ 40, 20, 332, 43, 36 ],
      [ 40, 20, 333, 43, 36 ],
      [ 40, 20, 334, 43, 35 ],
      [ 40, 20, 335, 43, 34 ],
      [ 40, 20, 336, 43, 34 ],
      [ 40, 20, 337, 43, 33 ],
      [ 40, 20, 338, 43, 33 ],
      [ 40, 20, 339, 43, 33 ],
      [ 40, 20, 340, 43, 32 ],
      [ 40, 20, 341, 43, 32 ],
      [ 40, 20, 342, 43, 31 ],
      [ 40, 20, 343, 43, 30 ],
      [ 40, 20, 344, 43, 30 ],
      [ 40, 20, 345, 42, 29 ],
      [ 40, 20, 346, 42, 28 ],
      [ 40, 20, 347, 41, 27 ],
      [ 40, 20, 348, 42, 28 ],
      [ 40, 20, 349, 42, 27 ],
      [ 40, 20, 350, 41, 26 ],
      [ 40, 20, 351, 41, 26 ],
      [ 40, 20, 352, 41, 25 ],
      [ 40, 20, 353, 40, 24 ],
      [ 40, 20, 354, 40, 24 ],
      [ 40, 20, 355, 40, 23 ],
      [ 40, 20, 356, 39, 22 ],
      [ 40, 20, 357, 39, 22 ],
      [ 40, 20, 358, 39, 21 ],
      [ 40, 20, 359, 39, 20 ],
      [ 40, 20, 360, 40, 20 ],
      [ 40, 20, 110.1, 32, 43 ],
      [ 40, 20, 344.9, 42, 29 ],
      [ 40, 20, 142.85, 43, 39 ],
      [ 40, 20, -106.52, 29, 43 ],
      [ 40, 20, -295.78, 35, 45 ],
      [ 40, 20, 287.58, 30, 44 ],
      [ 40, 20, -3.62, 39, 22 ],
      [ 40, 20, -33.96, 43, 39 ],
      [ 40, 20, 136.46, 42, 40 ],
      [ 40, 20, -293.72, 34, 44 ],
      [ 40, 20, 10.71, 43, 27 ],
      [ 40, 20, -80.02, 25, 42 ],
      [ 40, 20, 258.53, 26, 42 ],
      [ 40, 20, -110.41, 31, 43 ],
      [ 40, 20, 340.87, 43, 32 ],
      [ 40, 20, 347.4, 42, 28 ],
      [ 40, 20, -277.97, 25, 42 ],
      [ 40, 20, 126.74, 39, 43 ],
      [ 40, 20, 70.99, 32, 44 ],
      [ 40, 20, 57.54, 38, 44 ],
      [ 40, 20, 251.66, 30, 42 ],
      [ 40, 20, -91.15, 19, 39 ],
      [ 40, 20, 164.92, 43, 28 ],
      [ 40, 20, -27.14, 43, 36 ],
      [ 40, 20, 230.28, 39, 42 ],
      [ 40, 20, 271.37, 19, 39 ],
      [ 40, 20, -207.19, 44, 35 ],
      [ 40, 20, -58.19, 37, 44 ],
      [ 40, 20, -339.34, 44, 33 ],
      [ 40, 20, 352.45, 41, 25 ],
      [ 40, 20, -245.62, 34, 43 ],
      [ 40, 20, 130.77, 41, 42 ],
      [ 40, 20, 337.34, 43, 33 ],
      [ 40, 20, 257.24, 27, 42 ],
      [ 40, 20, -335.75, 44, 34 ],
      [ 40, 20, -318.44, 42, 41 ],
      [ 40, 20, 184.82, 40, 22 ],
      [ 40, 20, 20.64, 44, 33 ],
      [ 40, 20, 7.6, 42, 25 ],
      [ 40, 20, -38.72, 43, 41 ],
      [ 40, 20, 86.92, 22, 40 ],
      [ 40, 20, -341.69, 43, 31 ],
      [ 40, 20, 241.26, 36, 44 ],
      [ 40, 20, -14.56, 42, 29 ],
      [ 40, 20, -92.1, 20, 39 ],
      [ 40, 20, 222.12, 41, 40 ],
      [ 40, 20, -27.16, 43, 36 ],
      [ 40, 20, -10.06, 41, 26 ],
      [ 40, 20, -11.15, 42, 27 ],
      [ 40, 20, -316.18, 42, 41 ],
      [ 40, 20, -312.53, 42, 43 ],
      [ 40, 20, -119.49, 35, 43 ],
      [ 40, 20, -47.32, 41, 43 ],
      [ 40, 20, 212.4, 43, 37 ],
      [ 40, 20, 213.37, 43, 38 ],
      [ 40, 20, -177.05, 39, 21 ],
      [ 40, 20, 123.76, 39, 43 ],
      [ 40, 20, -353.82, 41, 24 ],
      [ 40, 20, 124.76, 38, 42 ],
      [ 40, 20, 144.43, 44, 38 ],
      [ 40, 20, 358.63, 39, 20 ],
      [ 40, 20, 239.15, 36, 43 ],
      [ 40, 20, -84.81, 22, 41 ],
      [ 40, 20, 335.97, 43, 34 ],
      [ 40, 20, 136.38, 42, 40 ],
      [ 40, 20, -60.55, 35, 44 ],
      [ 40, 20, -342.47, 44, 31 ],
      [ 40, 20, -38.8, 43, 41 ],
      [ 40, 20, 320.09, 42, 40 ],
      [ 40, 20, 25.13, 44, 34 ],
      [ 40, 20, 282.48, 27, 43 ],
      [ 40, 20, 47, 42, 43 ],
      [ 40, 20, 43.45, 43, 42 ],
      [ 40, 20, 163.73, 44, 29 ],
      [ 40, 20, -327.56, 44, 38 ],
      [ 40, 20, 311.35, 40, 43 ],
      [ 40, 20, 25.85, 44, 35 ],
      [ 40, 20, 5.28, 41, 23 ],
      [ 40, 20, -58.72, 36, 44 ],
      [ 40, 20, -345.3, 43, 29 ],
      [ 40, 20, 49.11, 41, 43 ],
      [ 40, 20, -11.25, 42, 27 ],
      [ 40, 20, -104.78, 28, 42 ],
      [ 40, 20, -358.22, 40, 21 ],
      [ 40, 20, -158.86, 43, 32 ],
      [ 40, 20, -251.42, 31, 42 ],
      [ 40, 20, -175.27, 40, 22 ],
      [ 40, 20, 324.91, 42, 38 ],
      [ 40, 20, -245.24, 34, 43 ],
      [ 40, 20, 309.5, 39, 43 ],
      [ 40, 20, 109.34, 32, 43 ],
      [ 40, 20, -246.61, 33, 43 ],
      [ 40, 20, 188.64, 41, 25 ],
      [ 40, 20, -335.47, 44, 34 ],
      [ 40, 20, -270.64, 20, 39 ],
      [ 40, 20, -34.98, 42, 38 ],
      [ 40, 20, -36.02, 43, 39 ],
      [ 40, 20, -253.11, 30, 43 ],
      [ 40, 20, 286.22, 29, 44 ],
      [ 40, 20, -75.94, 27, 43 ],
      [ 40, 20, 132.03, 41, 41 ],
      [ 40, 20, 208.69, 44, 36 ],
      [ 40, 20, 331.07, 44, 37 ],
      [ 40, 20, -184.53, 41, 22 ],
      [ 40, 20, 12.42, 43, 28 ],
      [ 40, 20, -356.5, 40, 22 ],
      [ 40, 20, -233.19, 39, 43 ],
      [ 40, 20, -321.73, 43, 40 ],
      [ 40, 20, 8.78, 42, 26 ],
      [ 40, 20, 68.1, 33, 44 ],
      [ 40, 20, -307.02, 40, 43 ],
      [ 40, 20, -302.11, 38, 44 ],
      [ 40, 20, -303.15, 38, 44 ],
      [ 40, 20, -51.8, 39, 43 ],
      [ 40, 20, -300.33, 37, 44 ],
      [ 40, 20, -102, 27, 42 ],
      [ 40, 20, 56.78, 38, 44 ],
      [ 40, 20, -115.59, 34, 44 ],
      [ 40, 20, -137.09, 42, 41 ],
      [ 40, 20, 171.55, 42, 24 ],
      [ 40, 20, -166.1, 42, 27 ],
      [ 40, 20, 332.25, 43, 36 ],
      [ 40, 20, 284.95, 28, 43 ],
      [ 40, 20, -337.46, 44, 33 ],
      [ 40, 20, 356.79, 39, 22 ],
      [ 40, 20, -345.7, 43, 28 ],
      [ 40, 20, -12.44, 42, 28 ],
      [ 40, 20, -39.24, 42, 40 ],
      [ 40, 20, -238.81, 37, 43 ],
      [ 40, 20, -86.22, 21, 40 ],
      [ 40, 20, 244.83, 34, 44 ],
      [ 40, 20, 253.22, 29, 43 ],
      [ 40, 20, -237.53, 38, 43 ],
      [ 40, 20, 215.89, 43, 38 ],
      [ 40, 20, -291.31, 33, 44 ],
      [ 40, 20, 134.89, 42, 41 ],
      [ 40, 20, 219.39, 42, 39 ],
      [ 40, 20, -164.5, 42, 28 ],
      [ 40, 20, 173.17, 41, 23 ],
      [ 40, 20, -131.83, 40, 41 ],
      [ 40, 20, 263.61, 23, 40 ],
      [ 40, 20, 226.15, 40, 41 ],
      [ 40, 20, -73.94, 29, 44 ],
      [ 40, 20, 320.47, 42, 40 ],
      [ 40, 20, -185.65, 41, 22 ],
      [ 40, 20, -14.26, 42, 28 ],
      [ 40, 20, -141.54, 42, 39 ],
      [ 40, 20, 231.14, 40, 43 ],
      [ 40, 20, 230.15, 39, 42 ],
      [ 40, 20, 81.38, 25, 42 ],
      [ 40, 20, 42.68, 43, 42 ],
      [ 40, 20, -295.95, 35, 44 ],
      [ 40, 20, 53.63, 39, 44 ],
      [ 40, 20, -32.37, 43, 38 ],
      [ 40, 20, -273.41, 22, 40 ],
      [ 40, 20, 50.41, 40, 43 ],
      [ 40, 20, -18.07, 43, 31 ],
      [ 40, 20, 74.16, 29, 43 ],
      [ 40, 20, -348.83, 43, 27 ],
      [ 40, 20, 103.13, 28, 42 ],
      [ 40, 20, 347.94, 42, 28 ],
      [ 40, 20, 256.01, 27, 42 ],
      [ 40, 20, -3.66, 39, 22 ],
      [ 40, 20, -249.59, 32, 43 ],
      [ 40, 20, 111.89, 33, 43 ],
      [ 40, 20, 65.03, 34, 44 ],
      [ 40, 20, 245.3, 33, 43 ],
      [ 40, 20, -28.72, 44, 37 ],
      [ 40, 20, 260.54, 25, 41 ],
      [ 40, 20, 58.47, 37, 44 ],
      [ 40, 20, 199.45, 43, 31 ],
      [ 40, 20, 164.15, 43, 28 ],
      [ 40, 20, -75.38, 28, 43 ],
      [ 40, 20, -234.49, 39, 43 ],
      [ 40, 20, 124.61, 38, 42 ],
      [ 40, 20, 98.97, 26, 41 ],
      [ 40, 20, 111.25, 33, 43 ],
      [ 40, 20, 343.08, 43, 30 ],
      [ 40, 20, -29.9, 43, 36 ],
      [ 40, 20, -18.6, 42, 31 ],
      [ 40, 20, 64.45, 35, 45 ],
      [ 40, 20, -347.22, 43, 28 ],
      [ 40, 20, 45.46, 42, 42 ],
      [ 40, 20, -241.93, 36, 43 ],
      [ 40, 20, -19.59, 43, 32 ],
      [ 40, 20, 132.05, 41, 41 ],
      [ 40, 20, 168.49, 43, 26 ],
      [ 40, 20, 322.34, 42, 40 ],
      [ 40, 20, -153.79, 43, 34 ],
      [ 40, 20, 179.66, 39, 19 ],
      [ 40, 20, 65.47, 34, 44 ],
      [ 40, 20, -165.85, 42, 27 ],
      [ 40, 20, 75.66, 28, 43 ],
      [ 40, 20, -298.19, 36, 44 ],
      [ 40, 20, -55.44, 37, 43 ],
      [ 40, 20, -172.45, 41, 24 ],
      [ 40, 20, 126.84, 39, 43 ],
      [ 40, 20, -170.14, 41, 25 ],
      [ 40, 20, 158.84, 44, 32 ],
      [ 40, 20, 27.38, 44, 36 ],
      [ 40, 20, 248.33, 32, 43 ],
      [ 40, 20, -1.71, 39, 21 ],
      [ 40, 20, -168.47, 42, 26 ],
      [ 40, 20, -187.06, 41, 23 ],
      [ 40, 20, 123.8, 39, 43 ],
      [ 40, 20, 316.15, 41, 41 ],
      [ 40, 20, 271.92, 20, 40 ],
      [ 40, 20, -124.95, 37, 42 ],
      [ 40, 20, 299.22, 35, 44 ],
      [ 40, 20, -117.99, 35, 43 ],
      [ 40, 20, 216.46, 43, 38 ],
      [ 40, 20, 3.67, 40, 22 ],
      [ 40, 20, -105.2, 28, 42 ],
      [ 40, 20, -98.09, 24, 41 ],
      [ 40, 20, 121.74, 38, 44 ],
      [ 40, 20, 235.21, 37, 42 ],
      [ 40, 20, -326.05, 44, 39 ],
      [ 40, 20, -69.77, 31, 44 ],
      [ 40, 20, 197.55, 43, 30 ],
      [ 40, 20, -119.83, 35, 43 ],
      [ 40, 20, -250.11, 32, 43 ],
      [ 40, 20, -96.99, 23, 40 ],
      [ 40, 20, 74.32, 29, 43 ],
      [ 40, 20, 185.56, 40, 22 ],
      [ 40, 20, -35.18, 43, 39 ],
      [ 40, 20, -341.13, 43, 31 ],
      [ 40, 20, -346.9, 43, 28 ],
      [ 40, 20, -268.34, 21, 39 ],
      [ 40, 20, -151.26, 44, 36 ],
      [ 40, 20, 171.95, 42, 24 ],
      [ 40, 20, 119.05, 36, 43 ],
      [ 40, 20, -262.94, 24, 40 ],
      [ 40, 20, -189.77, 42, 25 ],
      [ 40, 20, 310.58, 40, 43 ],
      [ 40, 20, -89.99, 19, 39 ],
      [ 40, 20, 294.04, 33, 44 ],
      [ 40, 20, 266.72, 21, 39 ],
      [ 40, 20, -178.07, 39, 20 ],
      [ 40, 20, -190.92, 43, 26 ],
      [ 40, 20, 205.94, 43, 34 ],
      [ 40, 20, 63.94, 35, 44 ],
      [ 40, 20, -334.47, 45, 35 ],
      [ 40, 20, -150.4, 43, 35 ],
      [ 40, 20, 318.74, 42, 41 ],
      [ 40, 20, -72.56, 29, 44 ],
      [ 40, 20, 331.35, 44, 37 ],
      [ 40, 20, 193.95, 42, 27 ],
      [ 40, 20, -38.6, 42, 40 ],
      [ 40, 20, -98.43, 24, 41 ],
      [ 40, 20, 31.49, 44, 37 ],
      [ 40, 20, 201.57, 43, 32 ],
      [ 40, 20, 11.46, 43, 27 ],
      [ 40, 20, 294.51, 33, 44 ],
      [ 40, 20, -84.11, 23, 41 ],
      [ 40, 20, -162.99, 43, 29 ],
      [ 40, 20, -100.68, 26, 42 ],
      [ 40, 20, -65.23, 33, 44 ],
      [ 40, 20, -149.88, 43, 36 ],
      [ 40, 20, -9.01, 41, 26 ],
      [ 40, 20, 143.51, 44, 38 ],
      [ 40, 20, -337.94, 45, 34 ],
      [ 40, 20, -249.97, 32, 43 ],
      [ 40, 20, 240.58, 35, 43 ],
      [ 40, 20, -167.7, 42, 27 ],
      [ 40, 20, -299.39, 36, 44 ],
      [ 40, 20, -209.42, 44, 35 ],
      [ 40, 20, -233.67, 39, 43 ],
      [ 40, 20, 327.34, 43, 38 ],
      [ 40, 20, -27.49, 43, 36 ],
      [ 40, 20, -64.59, 34, 45 ],
      [ 40, 20, 173.27, 41, 23 ],
      [ 40, 20, -323.55, 44, 39 ],
      [ 40, 20, -39.05, 43, 41 ],
      [ 40, 20, -337.14, 44, 33 ],
      [ 40, 20, 355.2, 40, 23 ],
      [ 40, 20, 248.4, 32, 43 ],
      [ 40, 20, 354.22, 40, 24 ],
      [ 40, 20, 189.14, 41, 25 ],
      [ 40, 20, -150.21, 43, 35 ],
      [ 40, 20, -104.22, 27, 42 ],
      [ 40, 20, -139.37, 42, 40 ],
      [ 40, 20, -308.64, 40, 43 ],
      [ 40, 20, 267.25, 20, 39 ],
      [ 40, 20, -204.87, 44, 33 ],
      [ 40, 20, -32.74, 43, 38 ],
      [ 40, 20, -255.74, 28, 42 ],
      [ 40, 20, 54.45, 39, 44 ],
      [ 40, 20, 262.03, 24, 41 ],
      [ 40, 20, -45.61, 40, 42 ],
      [ 40, 20, -314.56, 42, 42 ],
      [ 40, 20, 45.53, 42, 42 ],
      [ 40, 20, -23.54, 43, 33 ],
      [ 40, 20, -204.53, 44, 33 ],
      [ 40, 20, -73.9, 29, 44 ],
      [ 40, 20, 168.76, 43, 26 ],
      [ 40, 20, -143.92, 43, 38 ],
      [ 40, 20, 76.69, 28, 43 ],
      [ 40, 20, 295.09, 33, 44 ],
      [ 40, 20, -176.59, 39, 21 ],
      [ 40, 20, -310.8, 41, 43 ],
      [ 40, 20, -129.5, 39, 42 ],
      [ 40, 20, 356.69, 39, 22 ],
      [ 40, 20, -274.35, 23, 41 ],
      [ 40, 20, 191.45, 42, 26 ],
      [ 40, 20, -340.45, 44, 32 ],
      [ 40, 20, -279.15, 26, 42 ],
      [ 40, 20, 79.84, 27, 43 ],
      [ 40, 20, -346.24, 43, 28 ],
      [ 40, 20, 269.99, 19, 39 ],
      [ 40, 20, 289.64, 31, 44 ],
      [ 40, 20, -90.45, 19, 39 ],
      [ 40, 20, -229.38, 41, 42 ],
      [ 40, 20, 341.01, 43, 32 ],
      [ 40, 20, -183.2, 40, 21 ],
      [ 40, 20, -74.25, 28, 43 ],
      [ 40, 20, -51.74, 39, 43 ],
      [ 40, 20, -78.93, 26, 43 ],
      [ 40, 20, 340.21, 43, 32 ],
      [ 40, 20, -149.72, 43, 36 ],
      [ 40, 20, 235.46, 37, 42 ],
      [ 40, 20, -334.36, 45, 35 ],
      [ 40, 20, 255.82, 27, 42 ],
      [ 40, 20, -148.08, 43, 37 ],
      [ 40, 20, -178.88, 39, 19 ],
      [ 40, 20, -178.08, 39, 20 ],
      [ 40, 20, -339.33, 44, 33 ],
      [ 40, 20, 37.2, 43, 40 ],
      [ 40, 20, 258.61, 26, 42 ],
      [ 40, 20, 315.77, 41, 41 ],
      [ 40, 20, 220.62, 42, 40 ],
      [ 40, 20, 307.81, 39, 43 ],
      [ 40, 20, -173.74, 40, 23 ],
      [ 40, 20, 217.3, 42, 39 ],
      [ 40, 20, -326.54, 44, 39 ],
      [ 40, 20, -342.29, 44, 31 ],
      [ 40, 20, 236.86, 37, 43 ],
      [ 40, 20, -245.68, 34, 43 ],
      [ 40, 20, 97.55, 25, 41 ],
      [ 40, 20, 250.62, 31, 43 ],
      [ 40, 20, -335.69, 44, 34 ],
      [ 40, 20, 27.19, 44, 36 ],
      [ 40, 20, -199.84, 44, 31 ],
      [ 40, 20, -205.07, 44, 33 ],
      [ 40, 20, 8.19, 42, 25 ],
      [ 40, 20, -23.03, 43, 33 ],
      [ 40, 20, 80.68, 26, 42 ],
      [ 40, 20, 316.45, 41, 41 ],
      [ 40, 20, 258.04, 27, 42 ],
      [ 40, 20, 60.89, 36, 44 ],
      [ 40, 20, -193.27, 43, 27 ],
      [ 40, 20, 133.49, 42, 42 ],
      [ 40, 20, 86.53, 22, 40 ],
      [ 40, 20, -297.46, 36, 44 ],
      [ 40, 20, 345.41, 42, 29 ],
      [ 40, 20, 267.65, 20, 39 ],
      [ 40, 20, -115.54, 34, 44 ],
      [ 40, 20, -353.92, 41, 24 ],
      [ 40, 20, -55.15, 37, 43 ],
      [ 40, 20, -216.94, 43, 39 ],
      [ 40, 20, 321.85, 42, 40 ],
      [ 40, 20, -194.53, 43, 28 ],
      [ 40, 20, -269.13, 20, 39 ],
      [ 40, 20, -211.89, 44, 37 ],
      [ 40, 20, -337.23, 44, 33 ],
      [ 40, 20, -235.66, 39, 43 ],
      [ 40, 20, -194.18, 43, 27 ],
      [ 40, 20, 259.63, 26, 42 ],
      [ 40, 20, -121.34, 36, 43 ],
      [ 40, 20, 263.37, 23, 40 ],
      [ 40, 20, 150.24, 44, 35 ],
      [ 40, 20, -97.02, 23, 40 ],
      [ 40, 20, -69.44, 32, 44 ],
      [ 40, 20, 310.41, 39, 43 ],
      [ 40, 20, 57.91, 38, 44 ],
      [ 40, 20, 298.75, 36, 45 ],
      [ 40, 20, -72.63, 29, 44 ],
      [ 40, 20, -221.41, 42, 40 ],
      [ 40, 20, 255.19, 28, 42 ],
      [ 40, 20, -174.6, 40, 22 ],
      [ 40, 20, 199.48, 43, 31 ],
      [ 40, 20, -298.08, 36, 44 ],
      [ 40, 20, 318.9, 42, 41 ],
      [ 40, 20, -73.99, 29, 44 ],
      [ 40, 20, -235.54, 38, 42 ],
      [ 40, 20, 304.31, 38, 44 ],
      [ 40, 20, -166.34, 42, 27 ],
      [ 40, 20, 8.92, 42, 26 ],
      [ 40, 20, 310.39, 39, 43 ],
      [ 40, 20, 138.52, 42, 40 ],
      [ 40, 20, 151.99, 44, 35 ],
      [ 40, 20, 272.24, 20, 40 ],
      [ 40, 20, 303.99, 38, 44 ],
      [ 40, 20, 242.87, 35, 43 ],
      [ 40, 20, -299.66, 36, 44 ],
      [ 40, 20, 326.77, 43, 38 ],
      [ 40, 20, -352.79, 42, 25 ],
      [ 40, 20, -133.84, 40, 41 ],
      [ 40, 20, 226.39, 40, 41 ],
      [ 40, 20, -114.12, 33, 43 ],
      [ 40, 20, -230.47, 40, 42 ],
      [ 40, 20, 16.63, 44, 30 ],
      [ 40, 20, 148.86, 44, 36 ],
      [ 40, 20, 60.09, 36, 44 ],
      [ 40, 20, -32.96, 43, 38 ],
      [ 40, 20, -153.24, 43, 35 ],
      [ 40, 20, -1.16, 39, 20 ],
      [ 40, 20, 254.41, 28, 42 ],
      [ 40, 20, -14.65, 42, 29 ],
      [ 40, 20, -105.97, 29, 43 ],
      [ 40, 20, -280.19, 27, 43 ],
      [ 40, 20, -175.17, 40, 22 ],
      [ 40, 20, -44.05, 41, 41 ],
      [ 40, 20, -321.29, 44, 41 ],
      [ 40, 20, 110.85, 33, 43 ],
      [ 40, 20, 80.41, 26, 42 ],
      [ 40, 20, 343.02, 43, 30 ],
      [ 40, 20, 304.51, 37, 43 ],
      [ 40, 20, -270.67, 20, 39 ],
      [ 40, 20, 293.41, 32, 44 ],
      [ 40, 20, 83.02, 24, 41 ],
      [ 40, 20, 241.33, 36, 44 ],
      [ 40, 20, 205.64, 44, 34 ],
      [ 40, 20, 27.01, 44, 36 ],
      [ 40, 20, 124.19, 39, 43 ],
      [ 40, 20, 265.98, 21, 39 ],
      [ 40, 20, -6.23, 40, 24 ],
      [ 40, 20, 131.41, 41, 41 ],
      [ 40, 20, -227.86, 41, 41 ],
      [ 40, 20, -139.85, 42, 39 ],
      [ 40, 20, -342.72, 44, 30 ],
      [ 40, 20, -98.33, 24, 41 ],
      [ 40, 20, 236.79, 37, 43 ],
      [ 40, 20, 166.14, 43, 27 ],
      [ 40, 20, 321.77, 42, 40 ],
      [ 40, 20, -156.18, 43, 33 ],
      [ 40, 20, -347.1, 42, 27 ],
      [ 40, 20, -39.4, 42, 40 ],
      [ 40, 20, -261.77, 25, 41 ],
      [ 40, 20, -1.74, 39, 21 ],
      [ 40, 20, 214.63, 42, 37 ],
      [ 40, 20, -181.95, 40, 20 ],
      [ 40, 20, 183.09, 39, 21 ],
      [ 40, 20, -189.43, 42, 25 ],
      [ 40, 20, -143.24, 43, 38 ],
      [ 40, 20, -66.07, 33, 44 ],
      [ 40, 20, 250.99, 31, 43 ],
      [ 40, 20, -160.22, 43, 31 ],
      [ 40, 20, -121.56, 36, 43 ],
      [ 40, 20, 340.33, 43, 32 ],
      [ 40, 20, -226.82, 42, 42 ],
      [ 40, 20, 321.46, 42, 40 ],
      [ 40, 20, 221.65, 41, 40 ],
      [ 40, 20, 338.83, 43, 33 ],
      [ 40, 20, -11.53, 42, 27 ],
      [ 40, 20, -14.16, 42, 28 ],
      [ 40, 20, 244.81, 34, 44 ],
      [ 40, 20, 342.24, 43, 31 ],
      [ 40, 20, -242.76, 36, 43 ],
      [ 40, 20, -343.06, 44, 30 ],
      [ 40, 20, -157.61, 43, 32 ],
      [ 40, 20, -225.48, 42, 41 ],
      [ 40, 20, -81.38, 24, 42 ],
      [ 40, 20, -280.83, 27, 43 ],
      [ 40, 20, 300.67, 36, 44 ],
      [ 40, 20, -119.62, 35, 43 ],
      [ 40, 20, -77, 26, 42 ],
      [ 40, 20, 313.57, 40, 42 ],
      [ 40, 20, 200.99, 43, 32 ],
      [ 40, 20, 21.23, 44, 33 ],
      [ 40, 20, -48.18, 40, 42 ],
      [ 40, 20, 55.62, 39, 44 ],
      [ 40, 20, 199.29, 43, 31 ],
      [ 40, 20, -225.09, 42, 41 ],
      [ 40, 20, 226.2, 40, 41 ],
      [ 40, 20, -303.96, 39, 44 ],
      [ 40, 20, 68.84, 33, 44 ],
      [ 40, 20, 117.18, 36, 43 ],
      [ 40, 20, -104.18, 27, 42 ],
      [ 40, 20, 307.29, 39, 43 ],
      [ 40, 20, 97.5, 25, 41 ],
      [ 40, 20, 29.01, 44, 36 ],
      [ 40, 20, 268.74, 19, 39 ],
      [ 40, 20, -40.85, 42, 41 ],
      [ 40, 20, 7.83, 42, 25 ],
      [ 40, 20, -102.79, 27, 42 ],
      [ 40, 20, 304.99, 37, 43 ],
      [ 40, 20, -107.37, 29, 43 ],
      [ 40, 20, -120.56, 36, 43 ],
      [ 40, 20, -297.78, 36, 44 ],
      [ 40, 20, -90.42, 19, 39 ],
      [ 40, 20, 81.84, 25, 42 ],
      [ 40, 20, -163.25, 43, 29 ],
      [ 40, 20, 188.2, 41, 24 ],
      [ 40, 20, 161.02, 44, 31 ],
      [ 40, 20, -222.59, 43, 41 ],
      [ 40, 20, -291.42, 33, 44 ],
      [ 40, 20, -275.99, 24, 41 ],
      [ 40, 20, -269.03, 20, 39 ],
      [ 40, 20, 269.58, 19, 39 ],
      [ 40, 20, 105.25, 29, 42 ],
      [ 573, 347, -55.47, 609, 669 ],
      [ 765, 205, 0.58, 766, 212 ],
      [ 351, 714, 256.41, 775, 508 ],
      [ 276, 434, -74.59, 490, 381 ],
      [ 146, 6, 293.24, 62, 136 ],
      [ 512, 715, -174.13, 581, 762 ],
      [ 853, 14, 95.45, 95, 849 ],
      [ 457, 224, -76.35, 324, 497 ],
      [ 174, 370, -7.76, 221, 390 ],
      [ 294, 467, -98.19, 502, 357 ],
      [ 955, 129, 227, 744, 785 ],
      [ 364, 702, -301.62, 788, 677 ],
      [ 877, 197, -353.05, 894, 302 ],
      [ 730, 404, 53.02, 762, 826 ],
      [ 112, 514, 58.24, 495, 366 ],
      [ 17, 129, 8.44, 35, 130 ],
      [ 281, 768, 20.87, 536, 818 ],
      [ 929, 343, 230.05, 858, 931 ],
      [ 308, 473, -143.37, 528, 562 ],
      [ 809, 693, 294.19, 962, 1021 ],
      [ 158, 641, -305.47, 613, 500 ],
      [ 993, 146, 352.59, 1002, 273 ],
      [ 52, 36, -182.82, 53, 37 ],
      [ 61, 583, 195.61, 214, 576 ],
      [ 649, 666, -43.4, 928, 929 ],
      [ 220, 678, 77.36, 710, 362 ],
      [ 744, 121, 77.35, 280, 751 ],
      [ 261, 49, -321.61, 234, 200 ],
      [ 77, 280, 121.51, 279, 210 ],
      [ 385, 119, 50.72, 335, 373 ],
      [ 291, 237, -68.73, 325, 357 ],
      [ 369, 213, 67.7, 337, 422 ],
      [ 358, 251, 100.59, 312, 396 ],
      [ 610, 297, -73.68, 455, 668 ],
      [ 376, 15, -169.68, 371, 81 ],
      [ 827, 179, -254.18, 397, 843 ],
      [ 425, 737, -141.29, 791, 839 ],
      [ 23, 164, -95.58, 164, 37 ],
      [ 77, 222, 166.62, 125, 232 ],
      [ 762, 510, 277.54, 604, 822 ],
      [ 322, 728, -221.02, 720, 759 ],
      [ 736, 236, -22.05, 770, 495 ],
      [ 321, 504, 212.83, 541, 597 ],
      [ 976, 193, -229.44, 781, 865 ],
      [ 359, 475, -215.81, 569, 594 ],
      [ 630, 326, -48.07, 662, 686 ],
      [ 780, 644, -77.76, 793, 899 ],
      [ 67, 661, 75.98, 657, 225 ],
      [ 443, 718, 283.2, 799, 595 ],
      [ 168, 522, 40.53, 466, 506 ],
      [ 51, 763, 2.14, 78, 763 ],
      [ 249, 547, 222.59, 552, 570 ],
      [ 729, 19, -25.54, 664, 331 ],
      [ 520, 750, 154.94, 789, 898 ],
      [ 706, 251, -120.38, 573, 735 ],
      [ 911, 723, 185.11, 970, 800 ],
      [ 330, 540, 68.77, 622, 503 ],
      [ 732, 590, -32.05, 932, 888 ],
      [ 328, 154, -78.98, 212, 350 ],
      [ 219, 279, -310.49, 354, 347 ],
      [ 788, 317, -315.86, 786, 775 ],
      [ 279, 503, 196.18, 406, 559 ],
      [ 84, 282, 213.32, 224, 281 ],
      [ 559, 671, 196.16, 722, 798 ],
      [ 268, 432, -288.99, 495, 394 ],
      [ 609, 347, -331.79, 700, 593 ],
      [ 360, 594, 173.57, 424, 629 ],
      [ 970, 375, 5.79, 1003, 470 ],
      [ 412, 620, 149.31, 670, 742 ],
      [ 812, 16, -309.25, 525, 638 ],
      [ 882, 606, -12.36, 990, 780 ],
      [ 945, 47, -261.03, 193, 939 ],
      [ 712, 110, 2.68, 716, 143 ],
      [ 503, 529, -88.4, 542, 517 ],
      [ 270, 730, 316.29, 698, 714 ],
      [ 371, 418, -79.07, 479, 443 ],
      [ 412, 687, -242.53, 800, 681 ],
      [ 146, 646, 108.15, 659, 338 ],
      [ 667, 187, -31.06, 666, 504 ],
      [ 367, 27, -16.94, 358, 132 ],
      [ 439, 689, -168.99, 561, 758 ],
      [ 934, 629, -125.04, 1050, 1124 ],
      [ 50, 395, -22.51, 196, 384 ],
      [ 566, 4, -246.96, 225, 521 ],
      [ 952, 206, -283.43, 421, 973 ],
      [ 890, 484, 107.7, 731, 993 ],
      [ 266, 25, 31.52, 239, 160 ],
      [ 433, 134, -83.19, 183, 445 ],
      [ 75, 634, 91.53, 636, 90 ],
      [ 543, 140, 116.69, 368, 547 ],
      [ 6, 469, 41.81, 317, 353 ],
      [ 278, 401, -88.97, 404, 284 ],
      [ 612, 439, 280.54, 542, 681 ],
      [ 68, 95, 283.63, 107, 88 ],
      [ 229, 41, -243.53, 139, 221 ],
      [ 338, 708, 210.11, 646, 780 ],
      [ 1003, 139, 232.72, 717, 881 ],
      [ 536, 463, 357.89, 551, 482 ],
      [ 930, 519, 231.4, 985, 1049 ],
      [ 548, 157, -357.44, 554, 181 ],
      [ 202, 162, 82.14, 187, 222 ],
      [ 773, 370, -262.23, 471, 814 ],
      [ 135, 61, 148.43, 147, 121 ],
      [ 22, 112, 236.79, 105, 78 ],
      [ 937, 283, 274.47, 354, 956 ],
      [ 79, 536, 217.74, 389, 471 ],
      [ 890, 520, 347.55, 980, 699 ],
      [ 708, 154, 61.36, 474, 695 ],
      [ 705, 82, -172.68, 708, 169 ],
      [ 523, 493, -25.61, 683, 671 ],
      [ 526, 644, 125.61, 830, 801 ],
      [ 131, 455, -248, 471, 290 ],
      [ 270, 556, -191.46, 374, 597 ],
      [ 993, 572, -87.31, 616, 1018 ],
      [ 818, 506, 177.51, 839, 540 ],
      [ 696, 565, -40.43, 894, 881 ],
      [ 494, 464, -273.43, 492, 521 ],
      [ 476, 227, -129.96, 478, 509 ],
      [ 52, 755, -53.22, 635, 493 ],
      [ 318, 379, -311.68, 494, 489 ],
      [ 794, 5, 291.36, 293, 741 ],
      [ 460, 103, -2.3, 462, 121 ],
      [ 619, 305, -201.62, 687, 511 ],
      [ 859, 282, 333.95, 894, 630 ],
      [ 223, 127, 88.01, 134, 226 ],
      [ 987, 649, -102.96, 852, 1106 ],
      [ 418, 251, 331.17, 486, 421 ],
      [ 541, 608, -176.92, 572, 635 ],
      [ 868, 646, -189.3, 960, 777 ],
      [ 281, 116, 165.4, 300, 181 ],
      [ 349, 711, 169.74, 470, 761 ],
      [ 252, 288, 265.89, 304, 271 ],
      [ 760, 752, -168.4, 894, 888 ],
      [ 986, 628, 95.31, 716, 1038 ],
      [ 949, 27, 213.8, 802, 548 ],
      [ 562, 767, 101.53, 864, 702 ],
      [ 927, 312, 336.26, 973, 659 ],
      [ 886, 717, 185.51, 949, 798 ],
      [ 123, 600, 70.64, 606, 315 ],
      [ 365, 42, 178.97, 365, 47 ],
      [ 89, 304, 124.91, 299, 245 ],
      [ 842, 492, -325.96, 972, 879 ],
      [ 708, 294, -341.61, 764, 502 ],
      [ 981, 265, 246.51, 633, 1004 ],
      [ 604, 208, 271.05, 218, 607 ],
      [ 396, 629, 269.55, 631, 399 ],
      [ 29, 556, -158.42, 229, 526 ],
      [ 1019, 453, -206.3, 1114, 856 ],
      [ 771, 544, 355.98, 806, 597 ],
      [ 359, 50, -324.83, 322, 247 ],
      [ 102, 116, -40.39, 151, 154 ],
      [ 791, 747, -299.35, 1038, 1055 ],
      [ 840, 510, -23.99, 973, 807 ],
      [ 866, 477, 224.07, 953, 944 ],
      [ 705, 162, -301.25, 503, 686 ],
      [ 578, 459, 288.1, 614, 692 ],
      [ 450, 481, 89.13, 487, 456 ],
      [ 736, 476, -207, 871, 757 ],
      [ 955, 286, 302.83, 756, 957 ],
      [ 949, 555, 349.29, 1034, 721 ],
      [ 999, 631, -262.23, 760, 1073 ],
      [ 430, 457, -281.97, 536, 515 ],
      [ 516, 318, 195.96, 582, 446 ],
      [ 994, 199, -76.86, 418, 1012 ],
      [ 800, 727, 74.89, 910, 962 ],
      [ 884, 393, -215, 949, 828 ],
      [ 782, 729, 235.01, 1044, 1057 ],
      [ 397, 513, -38.84, 630, 648 ],
      [ 614, 461, 228.97, 750, 765 ],
      [ 533, 404, 29.07, 661, 611 ],
      [ 518, 385, -99.58, 465, 573 ],
      [ 657, 707, 327.94, 930, 947 ],
      [ 768, 326, -94.35, 382, 789 ],
      [ 534, 303, 91.98, 321, 542 ],
      [ 937, 135, 53.41, 666, 832 ],
      [ 532, 29, -299.01, 282, 479 ],
      [ 738, 615, -335.72, 925, 864 ],
      [ 560, 144, 144.86, 540, 439 ],
      [ 1017, 604, -143.59, 1176, 1088 ],
      [ 830, 236, 234.95, 668, 814 ],
      [ 360, 624, -124.39, 717, 648 ],
      [ 730, 348, -181.98, 741, 372 ],
      [ 684, 155, 122.85, 501, 657 ],
      [ 38, 555, -313.91, 426, 412 ],
      [ 681, 490, -188.19, 744, 581 ],
      [ 245, 120, -161.22, 269, 191 ],
      [ 332, 673, -304.93, 742, 657 ],
      [ 366, 325, -170.16, 415, 381 ],
      [ 60, 319, -323.93, 236, 293 ],
      [ 368, 173, -103.4, 252, 396 ],
      [ 190, 443, -239.3, 478, 388 ],
      [ 525, 222, 86.17, 257, 538 ],
      [ 779, 735, 231.02, 1060, 1066 ],
      [ 441, 763, 31.22, 772, 881 ],
      [ 507, 506, -230.09, 713, 712 ],
      [ 751, 689, -118.02, 959, 985 ],
      [ 10, 170, -207.58, 87, 154 ],
      [ 88, 444, 96.25, 450, 134 ],
      [ 358, 489, -325.38, 572, 605 ],
      [ 410, 765, -163.49, 609, 848 ],
      [ 774, 139, 250.99, 382, 775 ],
      [ 946, 533, -261.54, 666, 1012 ],
      [ 557, 348, -293.13, 538, 649 ],
      [ 373, 679, -298.13, 774, 648 ],
      [ 930, 291, 175.45, 950, 362 ],
      [ 90, 86, 101, 101, 103 ],
      [ 435, 93, 259.53, 169, 443 ],
      [ 651, 159, -44.7, 573, 570 ],
      [ 276, 427, -306.82, 507, 476 ],
      [ 325, 734, -310.41, 769, 723 ],
      [ 605, 546, -180.54, 609, 550 ],
      [ 450, 487, 318.4, 658, 662 ],
      [ 590, 137, -75.57, 279, 605 ],
      [ 685, 416, -159.6, 786, 627 ],
      [ 773, 345, 112.41, 613, 845 ],
      [ 511, 412, 192.27, 586, 510 ],
      [ 142, 738, 278.75, 749, 252 ],
      [ 1023, 458, -293.08, 822, 1121 ],
      [ 448, 665, 106.65, 765, 619 ],
      [ 523, 141, 74, 280, 541 ],
      [ 594, 594, -134.45, 838, 839 ],
      [ 849, 163, 322.03, 768, 650 ],
      [ 790, 605, -270.42, 610, 793 ],
      [ 171, 351, 119.26, 389, 320 ],
      [ 456, 162, -277.76, 222, 473 ],
      [ 103, 545, 108.48, 549, 269 ],
      [ 889, 651, 79.93, 796, 989 ],
      [ 773, 650, -210.99, 997, 954 ],
      [ 868, 218, -308.01, 706, 817 ],
      [ 508, 610, -176.53, 543, 638 ],
      [ 101, 287, -340.03, 192, 304 ],
      [ 422, 155, -187.53, 438, 208 ],
      [ 368, 747, -340.56, 596, 826 ],
      [ 496, 107, -12.25, 506, 210 ],
      [ 153, 448, 139.26, 407, 437 ],
      [ 270, 525, -70.57, 583, 429 ],
      [ 936, 424, -180.74, 940, 435 ],
      [ 538, 235, -291.72, 417, 586 ],
      [ 750, 118, -88.21, 140, 753 ],
      [ 824, 499, -237.04, 867, 961 ],
      [ 61, 575, 31.41, 352, 522 ],
      [ 89, 123, 176.59, 95, 127 ],
      [ 335, 399, 156.89, 465, 497 ],
      [ 362, 2, 256.74, 84, 352 ],
      [ 857, 117, -43.45, 701, 674 ],
      [ 543, 319, -271.27, 331, 549 ],
      [ 955, 253, 125.56, 761, 922 ],
      [ 186, 656, 345.76, 340, 681 ],
      [ 283, 450, -263.73, 477, 329 ],
      [ 669, 300, 203.75, 732, 543 ],
      [ 844, 345, 22.18, 911, 637 ],
      [ 377, 412, 137.13, 556, 557 ],
      [ 116, 663, -209.4, 426, 633 ],
      [ 851, 166, -207.8, 829, 542 ],
      [ 704, 40, -98.81, 146, 700 ],
      [ 130, 447, 199.8, 272, 464 ],
      [ 257, 395, -283.5, 443, 341 ],
      [ 947, 534, 331.25, 1086, 923 ],
      [ 927, 745, 33.05, 1183, 1129 ],
      [ 40, 479, 313.89, 371, 360 ],
      [ 842, 343, -82.76, 445, 878 ],
      [ 361, 626, 303.08, 721, 644 ],
      [ 44, 712, 203.88, 327, 667 ],
      [ 258, 582, 359.09, 265, 586 ],
      [ 475, 342, -323.57, 585, 557 ],
      [ 846, 439, 95.2, 513, 881 ],
      [ 618, 631, -255.2, 767, 757 ],
      [ 727, 573, 244.51, 828, 902 ],
      [ 686, 500, 222.61, 842, 831 ],
      [ 221, 530, -49.12, 544, 514 ],
      [ 156, 392, 272.31, 397, 171 ],
      [ 550, 663, 138.28, 851, 860 ],
      [ 469, 696, 85.4, 731, 523 ],
      [ 247, 121, -89.45, 122, 247 ],
      [ 246, 477, 230.48, 523, 492 ],
      [ 298, 343, -78.27, 395, 361 ],
      [ 945, 39, -206.96, 860, 462 ],
      [ 70, 584, 329.99, 351, 541 ],
      [ 930, 330, 78.29, 511, 977 ],
      [ 734, 496, -145.75, 884, 822 ],
      [ 153, 612, 221.61, 519, 558 ],
      [ 28, 256, 122.53, 231, 160 ],
      [ 736, 608, 54.16, 923, 952 ],
      [ 346, 25, 167.09, 343, 100 ],
      [ 731, 209, -106.47, 406, 759 ],
      [ 546, 431, 214.67, 693, 663 ],
      [ 947, 372, 250.3, 668, 1015 ],
      [ 141, 424, 76.93, 444, 233 ],
      [ 98, 353, -334.21, 242, 360 ],
      [ 721, 135, 129.95, 565, 638 ],
      [ 2, 249, -16.73, 72, 238 ],
      [ 829, 270, 223.01, 789, 761 ],
      [ 492, 54, 72.23, 201, 484 ],
      [ 57, 313, 94.83, 316, 81 ],
      [ 806, 92, 289.88, 360, 788 ],
      [ 143, 502, -26.17, 348, 514 ],
      [ 960, 444, 321.87, 1028, 941 ],
      [ 804, 549, -1.2, 813, 565 ],
      [ 402, 622, 329.13, 663, 740 ],
      [ 438, 395, -264.53, 434, 473 ],
      [ 111, 396, -31.3, 299, 395 ],
      [ 599, 250, 221.13, 614, 581 ],
      [ 404, 618, 271.02, 624, 414 ],
      [ 1019, 661, -156.22, 1198, 1014 ],
      [ 642, 497, -70.04, 685, 773 ],
      [ 542, 604, -319.97, 803, 810 ],
      [ 1017, 555, 95.73, 653, 1065 ],
      [ 963, 390, -109.91, 693, 1037 ],
      [ 449, 691, -246.89, 812, 682 ],
      [ 405, 251, 123.08, 431, 475 ],
      [ 541, 334, -225.77, 616, 619 ],
      [ 1008, 15, 8.18, 999, 158 ],
      [ 811, 318, -132.55, 781, 811 ],
      [ 684, 314, -233.79, 657, 735 ],
      [ 974, 27, 231.55, 625, 778 ],
      [ 363, 434, 76.98, 504, 451 ],
      [ 420, 428, -122.82, 586, 583 ],
      [ 44, 382, 242.46, 358, 215 ],
      [ 399, 718, 169.72, 520, 776 ],
      [ 560, 254, -315.23, 576, 574 ],
      [ 223, 659, -2.44, 249, 667 ],
      [ 414, 647, 11.35, 532, 715 ],
      [ 938, 487, 309.16, 969, 1035 ],
      [ 237, 232, -116.8, 312, 315 ],
      [ 417, 194, -91.67, 205, 421 ],
      [ 234, 466, 314.41, 495, 493 ],
      [ 856, 12, -4.73, 853, 82 ],
      [ 169, 45, 352.78, 172, 66 ],
      [ 2, 344, 305.73, 279, 202 ],
      [ 755, 764, -176.75, 795, 804 ],
      [ 819, 163, -279.19, 291, 834 ],
      [ 303, 473, 327.13, 510, 561 ],
      [ 832, 408, 63.3, 737, 926 ],
      [ 759, 586, 280.36, 711, 851 ],
      [ 81, 130, -257.92, 143, 105 ],
      [ 463, 305, 178.43, 470, 316 ],
      [ 398, 179, 189.44, 420, 241 ],
      [ 903, 305, 230.84, 805, 892 ],
      [ 893, 306, -167.29, 937, 494 ],
      [ 816, 104, -171.82, 821, 218 ],
      [ 52, 718, -19.67, 289, 693 ],
      [ 167, 177, -296.63, 232, 228 ],
      [ 120, 32, 85.5, 41, 122 ],
      [ 722, 601, -85.84, 650, 764 ],
      [ 609, 662, 35.57, 880, 892 ],
      [ 754, 240, 321.29, 737, 658 ],
      [ 496, 538, -231.74, 729, 721 ],
      [ 254, 447, 53.71, 510, 469 ],
      [ 70, 348, 340.06, 183, 350 ],
      [ 344, 192, -282.97, 264, 378 ],
      [ 612, 231, -330.59, 646, 501 ],
      [ 72, 356, -165.26, 159, 361 ],
      [ 162, 446, -135.64, 426, 431 ],
      [ 795, 219, -292.56, 507, 818 ],
      [ 161, 676, 327.18, 500, 655 ],
      [ 724, 473, -168.11, 804, 611 ],
      [ 669, 77, 336.01, 641, 342 ],
      [ 851, 267, 34.78, 850, 704 ],
      [ 416, 319, -359.02, 420, 326 ],
      [ 387, 576, -179.64, 389, 577 ],
      [ 497, 267, 36.17, 559, 509 ],
      [ 539, 320, -350.8, 583, 402 ],
      [ 816, 441, 66.6, 729, 923 ],
      [ 111, 269, 271.66, 271, 118 ],
      [ 207, 390, 184.72, 237, 405 ],
      [ 160, 164, -92.13, 168, 164 ],
      [ 433, 666, -20.46, 637, 775 ],
      [ 399, 536, 229.18, 665, 650 ],
      [ 960, 79, -192.4, 954, 282 ],
      [ 962, 369, 347.54, 1018, 567 ],
      [ 194, 741, -122.85, 727, 563 ],
      [ 758, 377, 176.79, 777, 417 ],
      [ 771, 220, -130.18, 664, 730 ],
      [ 353, 303, 121.74, 443, 458 ],
      [ 228, 458, -233.29, 503, 455 ],
      [ 1002, 577, -79.69, 746, 1088 ],
      [ 359, 133, -176.18, 366, 155 ],
      [ 842, 432, 325.52, 938, 832 ],
      [ 656, 384, -320.92, 751, 711 ],
      [ 895, 337, 24.6, 953, 678 ],
      [ 877, 482, -0.9, 883, 495 ],
      [ 190, 282, 351.99, 226, 305 ],
      [ 954, 91, -162, 934, 380 ],
      [ 256, 356, 121.52, 436, 403 ],
      [ 942, 526, -170.02, 1017, 680 ],
      [ 98, 510, -349.45, 189, 518 ],
      [ 496, 11, -224.86, 359, 356 ],
      [ 758, 642, 179.75, 760, 644 ],
      [ 709, 366, -180.86, 713, 375 ],
      [ 764, 269, 134.32, 725, 733 ],
      [ 287, 158, -134.37, 312, 314 ],
      [ 798, 300, 129.27, 737, 806 ],
      [ 787, 247, 214.41, 788, 647 ],
      [ 44, 439, -172.34, 101, 439 ],
      [ 729, 377, 320.69, 802, 753 ],
      [ 999, 450, 78.19, 644, 1069 ],
      [ 1014, 54, 88.04, 88, 1015 ],
      [ 183, 622, -30.22, 470, 629 ],
      [ 950, 385, 312.59, 924, 960 ],
      [ 181, 190, -155.94, 241, 245 ],
      [ 884, 405, -8.25, 931, 527 ],
      [ 181, 236, 272.79, 243, 191 ],
      [ 877, 57, -216.03, 743, 560 ],
      [ 670, 90, 195.13, 668, 260 ],
      [ 937, 637, -172.02, 1014, 760 ],
      [ 881, 618, 249.25, 889, 1041 ],
      [ 480, 610, 299.25, 765, 716 ],
      [ 93, 747, 161.79, 321, 738 ],
      [ 562, 691, -197.6, 744, 827 ],
      [ 495, 59, -19.85, 484, 223 ],
      [ 785, 721, 359.66, 787, 725 ],
      [ 261, 89, -139.99, 255, 234 ],
      [ 137, 746, -86.8, 751, 178 ],
      [ 342, 480, -2.77, 363, 495 ],
      [ 107, 414, -126.38, 395, 331 ],
      [ 375, 306, -267.64, 321, 386 ],
      [ 217, 666, 304.01, 672, 552 ],
      [ 131, 735, 283.69, 744, 301 ],
      [ 873, 388, 211.15, 947, 782 ],
      [ 80, 759, 265.54, 762, 137 ],
      [ 564, 579, 221.17, 804, 806 ],
      [ 563, 7, 303.67, 317, 472 ],
      [ 876, 109, 282.32, 291, 878 ],
      [ 241, 365, 11.53, 309, 406 ],
      [ 749, 446, -321.05, 862, 817 ],
      [ 57, 726, -34.7, 458, 629 ],
      [ 188, 121, -131.26, 213, 220 ],
      [ 109, 219, -162.69, 168, 240 ],
      [ 1024, 105, 200.79, 993, 460 ],
      [ 812, 165, 191.77, 827, 326 ],
      [ 678, 587, 14.3, 801, 736 ],
      [ 481, 241, 20.14, 534, 391 ],
      [ 401, 130, 122.08, 322, 407 ],
      [ 290, 310, -255.92, 371, 355 ],
      [ 818, 103, 161.27, 807, 359 ],
      [ 874, 145, 121.63, 581, 819 ],
      [ 313, 286, 234.41, 414, 419 ],
      [ 638, 367, -281.16, 483, 696 ],
      [ 918, 367, -183.26, 937, 417 ],
      [ 692, 208, -29.51, 703, 521 ],
      [ 453, 716, -199.75, 668, 826 ],
      [ 985, 309, 25.67, 1021, 705 ],
      [ 502, 609, 146.76, 753, 783 ],
      [ 163, 58, -283.29, 93, 171 ],
      [ 311, 671, -187.24, 393, 704 ],
      [ 29, 558, -51.47, 454, 370 ],
      [ 714, 24, 216.39, 587, 441 ],
      [ 535, 502, -159.67, 674, 655 ],
      [ 647, 423, 184.07, 674, 466 ],
      [ 899, 174, -341.28, 907, 453 ],
      [ 345, 513, 329.85, 555, 617 ],
      [ 573, 483, -101.39, 585, 655 ],
      [ 97, 91, -251.33, 117, 119 ],
      [ 818, 213, -197.64, 844, 449 ],
      [ 927, 447, -288.8, 721, 1021 ],
      [ 149, 468, 234.16, 465, 393 ],
      [ 587, 235, 256.52, 364, 624 ],
      [ 383, 636, 7.03, 458, 677 ],
      [ 668, 353, -317, 729, 713 ],
      [ 418, 258, -154.77, 487, 410 ],
      [ 444, 516, 325.47, 656, 676 ],
      [ 402, 178, 58.01, 363, 434 ],
      [ 769, 250, 143.01, 764, 661 ],
      [ 924, 95, 342.54, 909, 368 ],
      [ 131, 22, 40.69, 113, 102 ],
      [ 232, 133, -241.28, 228, 266 ],
      [ 53, 573, -21.74, 260, 551 ],
      [ 930, 92, -146.28, 823, 592 ],
      [ 428, 593, 183.95, 466, 620 ],
      [ 489, 138, -294.32, 327, 502 ],
      [ 781, 471, 189.43, 846, 591 ],
      [ 732, 4, 355.43, 729, 62 ],
      [ 302, 580, -130.66, 635, 606 ],
      [ 1018, 505, -42.07, 1092, 1057 ],
      [ 993, 603, -22.08, 1146, 932 ],
      [ 539, 8, 102.05, 120, 528 ],
      [ 708, 48, -81.14, 155, 706 ],
      [ 165, 745, 4.73, 225, 755 ],
      [ 469, 298, -160.01, 541, 439 ],
      [ 218, 664, -31.28, 530, 680 ],
      [ 312, 482, -310.38, 569, 549 ],
      [ 69, 262, -307.45, 249, 213 ],
      [ 63, 489, 236.14, 440, 323 ],
      [ 389, 484, -10.97, 472, 549 ],
      [ 841, 460, -166.4, 924, 643 ],
      [ 298, 96, -158.96, 311, 195 ],
      [ 955, 626, -114.48, 964, 1127 ],
      [ 327, 749, -136.45, 752, 767 ],
      [ 846, 332, 142.16, 872, 779 ],
      [ 107, 496, 44.36, 423, 429 ],
      [ 574, 729, -224.62, 920, 921 ],
      [ 645, 13, 20.69, 608, 239 ],
      [ 719, 59, -241.73, 392, 660 ],
      [ 543, 351, 211.16, 645, 579 ],
      [ 16, 213, -78.13, 210, 59 ],
      [ 291, 436, 11.09, 369, 483 ],
      [ 577, 383, -122.45, 631, 691 ],
      [ 926, 629, -140.49, 1113, 1073 ],
      [ 220, 495, 170.69, 297, 522 ],
      [ 934, 574, -84.97, 652, 980 ],
      [ 485, 236, -121.77, 455, 535 ],
      [ 678, 720, -110.28, 909, 884 ],
      [ 195, 491, -55.09, 513, 440 ],
      [ 363, 130, 273.14, 148, 369 ],
      [ 152, 141, -247.66, 187, 193 ],
      [ 553, 359, 160.46, 641, 521 ],
      [ 56, 23, -200.49, 60, 40 ],
      [ 394, 717, -331.63, 687, 818 ],
      [ 706, 114, 131.79, 555, 601 ],
      [ 437, 46, -50.28, 313, 365 ],
      [ 828, 410, 170.92, 882, 534 ],
      [ 143, 150, 124.98, 204, 202 ],
      [ 498, 296, -135.53, 561, 558 ],
      [ 932, 568, -14.46, 1043, 782 ],
      [ 808, 681, 117.2, 975, 1028 ],
      [ 968, 327, -266.95, 378, 982 ],
      [ 684, 369, -245.86, 616, 774 ],
      [ 911, 665, -294.01, 977, 1103 ],
      [ 952, 426, -187.84, 1001, 550 ],
      [ 627, 753, -145.19, 943, 974 ],
      [ 44, 606, 238.3, 538, 354 ],
      [ 799, 713, -342, 979, 924 ],
      [ 464, 319, -65.38, 482, 554 ],
      [ 784, 249, -253.58, 460, 821 ],
      [ 251, 87, -335.43, 264, 183 ],
      [ 930, 45, -29.07, 833, 490 ],
      [ 38, 558, 316.6, 409, 431 ],
      [ 200, 473, 219.12, 452, 492 ],
      [ 294, 418, -102, 469, 373 ],
      [ 539, 120, -116.07, 343, 536 ],
      [ 845, 153, -268.05, 181, 848 ],
      [ 672, 751, -319.72, 998, 1007 ],
      [ 697, 331, -21.35, 769, 561 ],
      [ 92, 151, 211.96, 157, 175 ],
      [ 244, 339, 293.35, 406, 358 ],
      [ 279, 267, 335.17, 364, 359 ],
      [ 749, 296, 137.87, 754, 721 ],
      [ 687, 446, -139.04, 809, 786 ],
      [ 487, 666, 252.84, 778, 660 ],
      [ 854, 302, -354.74, 878, 379 ],
      [ 177, 167, -212.02, 239, 234 ],
      [ 308, 670, -228.12, 704, 675 ],
      [ 365, 425, 81.96, 472, 420 ],
      [ 847, 494, 222.68, 956, 936 ],
      [ 636, 676, -180.42, 640, 679 ],
      [ 541, 116, 69.88, 295, 547 ],
      [ 506, 678, 346.6, 648, 777 ],
      [ 190, 425, 43.89, 431, 437 ],
      [ 504, 22, -52.24, 324, 411 ],
      [ 352, 662, 230.37, 733, 692 ],
      [ 359, 27, -333.47, 333, 184 ],
      [ 570, 257, 294.55, 469, 625 ],
      [ 757, 530, -27.59, 914, 820 ],
      [ 362, 397, -285.26, 478, 453 ],
      [ 167, 106, 347.98, 184, 138 ],
      [ 422, 511, -263.32, 557, 477 ],
      [ 9, 122, -348.21, 33, 120 ],
      [ 1014, 264, 49.91, 855, 945 ],
      [ 565, 642, 70.15, 795, 749 ],
      [ 1003, 138, -29.48, 940, 613 ],
      [ 818, 407, -5.06, 849, 477 ],
      [ 856, 66, 235.38, 539, 740 ],
      [ 763, 634, 12.1, 879, 779 ],
      [ 212, 137, -336.25, 249, 210 ],
      [ 318, 262, -237.7, 390, 407 ],
      [ 301, 578, 241.1, 650, 541 ],
      [ 438, 584, 355.3, 483, 617 ],
      [ 455, 577, -117.24, 720, 667 ],
      [ 14, 232, -235.53, 198, 141 ],
      [ 626, 216, -106.74, 386, 660 ],
      [ 72, 61, 274.4, 65, 76 ],
      [ 577, 702, 335.87, 812, 876 ],
      [ 399, 506, 209.55, 596, 635 ],
      [ 929, 665, 337.4, 1112, 971 ],
      [ 962, 135, -137.9, 803, 743 ],
      [ 112, 360, -316.76, 328, 338 ],
      [ 967, 688, 230.71, 1143, 1183 ],
      [ 961, 261, 51.08, 806, 911 ],
      [ 282, 271, 268.25, 278, 288 ],
      [ 459, 740, -250.03, 852, 683 ],
      [ 819, 26, -193.37, 802, 213 ],
      [ 697, 458, -255.25, 620, 790 ],
      [ 663, 757, 218.88, 990, 1004 ],
      [ 449, 685, 121.42, 819, 739 ],
      [ 417, 639, 247.11, 750, 632 ],
      [ 733, 723, -136.06, 1028, 1028 ],
      [ 794, 680, -211.41, 1031, 992 ],
      [ 610, 632, 32.85, 855, 861 ],
      [ 171, 75, 286.64, 119, 184 ],
      [ 40, 419, 260.1, 418, 110 ],
      [ 197, 264, 283.76, 301, 254 ],
      [ 434, 19, -7.45, 431, 75 ],
      [ 583, 516, -18.13, 714, 671 ],
      [ 382, 84, -96.2, 124, 387 ],
      [ 43, 397, -217.9, 277, 338 ],
      [ 906, 178, -260.93, 318, 921 ],
      [ 201, 5, -344.2, 194, 59 ],
      [ 412, 462, 247.6, 583, 555 ],
      [ 971, 590, 317.22, 1112, 1092 ],
      [ 866, 620, -10.88, 966, 772 ],
      [ 724, 768, 236.57, 1038, 1026 ],
      [ 616, 325, 253.97, 481, 681 ],
      [ 93, 761, 17.18, 313, 754 ],
      [ 580, 280, 95.75, 337, 604 ],
      [ 955, 311, -252.94, 577, 1002 ],
      [ 133, 223, -86.78, 229, 145 ],
      [ 274, 373, -82.53, 404, 319 ],
      [ 296, 682, 350.52, 402, 721 ],
      [ 136, 641, 183.43, 172, 647 ],
      [ 76, 523, 44.45, 420, 426 ],
      [ 572, 298, 43.57, 619, 610 ],
      [ 397, 759, 347.43, 551, 827 ],
      [ 246, 61, 340.77, 251, 139 ],
      [ 782, 495, -117.54, 799, 921 ],
      [ 406, 443, 173.94, 450, 482 ],
      [ 559, 542, -337.71, 723, 713 ],
      [ 947, 747, 11.49, 1077, 920 ],
      [ 318, 201, -69.91, 297, 367 ],
      [ 304, 303, -189.04, 348, 345 ],
      [ 53, 359, 300.74, 335, 229 ],
      [ 628, 20, -140.49, 496, 413 ],
      [ 178, 317, -148.96, 314, 362 ],
      [ 161, 501, -92.13, 505, 178 ],
      [ 133, 319, -348.12, 196, 339 ],
      [ 478, 624, 66.87, 761, 684 ],
      [ 213, 274, -145.07, 330, 345 ],
      [ 245, 216, -164.86, 291, 271 ],
      [ 773, 454, 23.31, 889, 722 ],
      [ 162, 682, 307.04, 640, 540 ],
      [ 405, 722, -76.47, 795, 562 ],
      [ 321, 425, -58.1, 529, 497 ],
      [ 633, 558, 238.73, 804, 830 ],
      [ 933, 678, -12.03, 1052, 857 ],
      [ 290, 10, 286.84, 93, 280 ],
      [ 307, 368, 151.05, 446, 469 ],
      [ 914, 527, -312.32, 1005, 1030 ],
      [ 196, 711, -110.04, 734, 427 ],
      [ 473, 522, 315.35, 702, 703 ],
      [ 1019, 643, 234.38, 1115, 1201 ],
      [ 943, 179, 190.45, 958, 346 ],
      [ 322, 419, -131.97, 526, 518 ],
      [ 752, 126, -329.31, 710, 491 ],
      [ 579, 57, 305.87, 384, 502 ],
      [ 50, 274, 314.31, 229, 226 ],
      [ 970, 504, -61.53, 904, 1092 ],
      [ 673, 422, -287.69, 606, 769 ],
      [ 741, 568, -341.9, 880, 770 ],
      [ 72, 155, 147.35, 144, 168 ],
      [ 9, 150, 30.13, 82, 134 ],
      [ 854, 89, 197.41, 840, 339 ],
      [ 613, 331, -130.07, 646, 681 ],
      [ 937, 126, -12.79, 940, 330 ],
      [ 981, 560, 40.29, 1110, 1061 ],
      [ 904, 598, 297.04, 942, 1077 ],
      [ 839, 557, 49.32, 968, 999 ],
      [ 239, 293, -275.98, 315, 268 ],
      [ 342, 80, 256.08, 159, 349 ],
      [ 368, 134, -319.39, 366, 341 ],
      [ 66, 141, -179.05, 66, 141 ],
      [ 620, 13, -95.86, 75, 616 ],
      [ 389, 472, 213.71, 584, 607 ],
      [ 716, 407, 331.54, 822, 699 ],
      [ 186, 374, 136.35, 392, 398 ],
      [ 755, 284, -23.86, 804, 565 ],
      [ 666, 145, -222.27, 590, 553 ],
      [ 224, 324, 52.16, 393, 375 ],
      [ 343, 580, 127.12, 668, 622 ],
      [ 196, 88, 252.56, 141, 211 ],
      [ 253, 137, -336.17, 286, 227 ],
      [ 511, 602, -324.69, 764, 786 ],
      [ 886, 125, 117.44, 519, 843 ],
      [ 678, 662, -221.74, 946, 944 ],
      [ 638, 33, 128.46, 422, 519 ],
      [ 320, 599, -326.1, 599, 675 ],
      [ 798, 330, -191.1, 847, 476 ],
      [ 994, 498, 111.94, 833, 1107 ],
      [ 556, 755, -64.69, 919, 825 ],
      [ 225, 133, 17.12, 254, 193 ],
      [ 72, 323, 144.73, 245, 304 ],
      [ 106, 706, -11.75, 246, 712 ],
      [ 156, 602, 105.1, 621, 306 ],
      [ 835, 343, 5.2, 862, 417 ],
      [ 8, 53, 35.77, 37, 47 ],
      [ 702, 292, -122.9, 625, 747 ],
      [ 751, 122, 186.23, 758, 201 ],
      [ 991, 99, -67.53, 468, 953 ],
      [ 638, 516, -80.16, 616, 716 ],
      [ 34, 684, 44.43, 503, 511 ],
      [ 570, 737, 346.85, 722, 847 ],
      [ 264, 48, 288.03, 126, 266 ],
      [ 760, 165, 132.14, 631, 673 ],
      [ 397, 23, -267.13, 42, 396 ],
      [ 916, 28, -218.24, 736, 587 ],
      [ 455, 555, 55.17, 715, 690 ],
      [ 792, 349, 169.23, 843, 489 ],
      [ 545, 324, 261.67, 398, 585 ],
      [ 961, 34, 24.65, 887, 431 ],
      [ 335, 60, -54.7, 241, 308 ],
      [ 910, 487, -83.82, 580, 956 ],
      [ 892, 685, -38.88, 1123, 1092 ],
      [ 789, 486, 115.43, 777, 920 ],
      [ 465, 15, 136.55, 347, 329 ],
      [ 597, 702, 162.75, 778, 846 ],
      [ 799, 275, -37.53, 800, 704 ],
      [ 365, 101, 289.64, 216, 377 ],
      [ 94, 510, -126.88, 463, 380 ],
      [ 978, 462, -95.12, 546, 1014 ],
      [ 501, 713, -39.78, 840, 868 ],
      [ 936, 628, -303.76, 1042, 1127 ],
      [ 304, 528, 337.71, 480, 604 ],
      [ 761, 352, 72.83, 560, 831 ],
      [ 413, 701, 86.73, 723, 452 ],
      [ 95, 380, 24.59, 244, 385 ],
      [ 839, 211, 282.2, 382, 865 ],
      [ 274, 485, -343.38, 401, 543 ],
      [ 173, 555, 134.13, 518, 509 ],
      [ 505, 520, -152.84, 685, 692 ],
      [ 882, 128, -204.43, 856, 480 ],
      [ 313, 61, -336.21, 311, 182 ],
      [ 393, 289, 158.67, 471, 410 ],
      [ 361, 92, -231.78, 295, 339 ],
      [ 977, 401, -295.42, 781, 1054 ],
      [ 588, 472, 60.64, 699, 743 ],
      [ 111, 333, 258.13, 347, 175 ],
      [ 1, 538, -7.75, 72, 533 ],
      [ 24, 668, -207.79, 332, 601 ],
      [ 727, 278, 279.69, 395, 763 ],
      [ 1022, 171, -320.66, 898, 779 ],
      [ 219, 406, -263.54, 427, 262 ],
      [ 253, 700, 7.25, 338, 725 ],
      [ 991, 203, 92.81, 251, 998 ],
      [ 149, 167, -251.44, 205, 193 ],
      [ 241, 608, 191.06, 352, 642 ],
      [ 839, 691, -216.96, 1085, 1055 ],
      [ 694, 691, 287.01, 863, 865 ],
      [ 171, 709, 192.56, 319, 728 ],
      [ 388, 486, -267.17, 504, 410 ],
      [ 273, 484, -107.27, 542, 403 ],
      [ 329, 648, 272.46, 660, 356 ],
      [ 466, 69, 208.26, 442, 280 ],
      [ 989, 44, -322.16, 808, 641 ],
      [ 609, 155, -165.97, 627, 296 ],
      [ 763, 335, -316.43, 783, 768 ],
      [ 523, 196, 331.05, 551, 425 ],
      [ 726, 716, 258.65, 843, 851 ],
      [ 623, 76, 203, 602, 312 ],
      [ 384, 366, -61.76, 502, 511 ],
      [ 516, 571, -328.46, 738, 756 ],
      [ 876, 49, 278.47, 176, 873 ],
      [ 751, 399, 342.82, 834, 602 ],
      [ 536, 372, -336.75, 639, 553 ],
      [ 589, 60, -191.98, 588, 180 ],
      [ 865, 632, 121.69, 992, 1067 ],
      [ 927, 256, 304.9, 739, 906 ],
      [ 886, 32, 255.78, 247, 865 ],
      [ 742, 499, 326.58, 893, 824 ],
      [ 519, 19, -50.51, 344, 412 ],
      [ 943, 406, -235.77, 866, 1006 ],
      [ 988, 294, -189.88, 1023, 458 ],
      [ 872, 89, 183.64, 875, 143 ],
      [ 847, 491, -187.97, 906, 602 ],
      [ 880, 164, -131.9, 708, 763 ],
      [ 95, 45, -259.59, 61, 100 ],
      [ 780, 740, -19.97, 985, 962 ],
      [ 702, 636, 10, 801, 747 ],
      [ 554, 424, -242.69, 631, 686 ],
      [ 506, 45, -225.48, 386, 391 ],
      [ 946, 752, 154.49, 1177, 1085 ],
      [ 99, 724, -290.08, 713, 341 ],
      [ 341, 610, -206.71, 578, 697 ],
      [ 90, 477, 252.85, 481, 225 ],
      [ 335, 368, -313.57, 497, 496 ],
      [ 659, 440, -271.91, 461, 673 ],
      [ 802, 257, -298.62, 610, 827 ],
      [ 262, 15, 297.37, 132, 239 ],
      [ 788, 430, -25.32, 895, 726 ],
      [ 955, 41, 16.33, 928, 307 ],
      [ 122, 750, 0.56, 128, 751 ],
      [ 854, 56, -41.09, 679, 603 ],
      [ 953, 312, -190.15, 993, 473 ],
      [ 147, 379, 256.2, 402, 231 ],
      [ 1019, 630, -118.81, 1042, 1195 ],
      [ 61, 356, 292.86, 350, 194 ],
      [ 186, 189, -187.11, 207, 210 ],
      [ 274, 385, -173.82, 312, 411 ],
      [ 184, 208, 228.5, 276, 274 ],
      [ 660, 155, 266.41, 195, 667 ],
      [ 172, 247, 249.42, 290, 247 ],
      [ 684, 119, 301.76, 460, 644 ],
      [ 114, 65, -126.46, 118, 129 ],
      [ 356, 175, 228.43, 366, 381 ],
      [ 208, 171, 98.43, 199, 229 ],
      [ 551, 216, -288.35, 378, 590 ],
      [ 456, 356, -111.48, 496, 553 ],
      [ 701, 560, 249.07, 772, 853 ],
      [ 966, 698, -276.73, 806, 1041 ],
      [ 779, 425, -131.54, 833, 864 ],
      [ 646, 554, 99.15, 649, 724 ],
      [ 489, 299, 209.84, 572, 501 ],
      [ 406, 384, 270.44, 386, 408 ],
      [ 738, 650, -285.75, 826, 886 ],
      [ 551, 38, -125.87, 352, 467 ],
      [ 179, 451, 76.5, 480, 279 ],
      [ 281, 24, -310.54, 200, 229 ],
      [ 634, 550, 214.41, 833, 811 ],
      [ 477, 506, 148.69, 670, 678 ],
      [ 595, 322, -173, 628, 391 ],
      [ 920, 38, 345.67, 899, 264 ],
      [ 549, 404, -94.74, 447, 579 ],
      [ 335, 708, 264.69, 735, 398 ],
      [ 208, 494, 153.81, 404, 533 ],
      [ 314, 138, 188.81, 330, 183 ],
      [ 647, 272, -109.12, 467, 699 ],
      [ 243, 482, -87.11, 492, 266 ],
      [ 314, 190, -292.29, 295, 362 ],
      [ 107, 547, 181.54, 120, 548 ],
      [ 830, 225, -237.15, 639, 818 ],
      [ 72, 146, -201.94, 121, 160 ],
      [ 52, 557, 176.6, 84, 558 ],
      [ 429, 41, 120.19, 250, 390 ],
      [ 293, 196, -137.62, 347, 341 ],
      [ 1024, 431, -8.38, 1075, 575 ],
      [ 781, 147, 246.32, 447, 773 ],
      [ 113, 330, -21.87, 226, 348 ],
      [ 502, 564, 155.6, 690, 720 ],
      [ 598, 644, -52.24, 874, 866 ],
      [ 344, 498, 157.9, 505, 589 ],
      [ 519, 552, 294.48, 716, 701 ],
      [ 744, 590, 96.63, 671, 806 ],
      [ 483, 144, 134.21, 439, 445 ],
      [ 142, 363, 317.65, 348, 363 ],
      [ 458, 363, 1.19, 465, 372 ],
      [ 958, 180, 138.96, 840, 764 ],
      [ 796, 265, -272.53, 300, 807 ],
      [ 253, 641, 256.2, 681, 397 ],
      [ 987, 321, 139.53, 958, 883 ],
      [ 400, 578, -113.84, 689, 598 ],
      [ 113, 199, -316.9, 218, 222 ],
      [ 20, 756, 236.02, 637, 438 ],
      [ 669, 350, -349.85, 720, 462 ],
      [ 348, 456, -9.77, 418, 508 ],
      [ 288, 31, 330.03, 263, 170 ],
      [ 801, 749, -221.64, 1096, 1091 ],
      [ 487, 578, 26.13, 692, 733 ],
      [ 611, 768, -93.8, 805, 659 ],
      [ 464, 739, 206.31, 742, 866 ],
      [ 150, 271, 27.98, 259, 309 ],
      [ 500, 355, 214.08, 612, 573 ],
      [ 562, 370, 202.66, 660, 556 ],
      [ 385, 103, -189.69, 396, 165 ],
      [ 400, 364, 237.25, 521, 532 ],
      [ 898, 579, 265.83, 641, 936 ],
      [ 855, 412, 247.68, 704, 945 ],
      [ 28, 8, 69.11, 16, 29 ],
      [ 577, 467, 68.26, 647, 708 ],
      [ 956, 46, 40.72, 754, 658 ],
      [ 737, 158, 294.26, 445, 736 ],
      [ 265, 533, -93.23, 545, 293 ],
      [ 57, 186, 252.73, 193, 108 ],
      [ 858, 475, 348.77, 933, 633 ],
      [ 76, 6, -30.9, 67, 44 ],
      [ 925, 679, -208.65, 1137, 1038 ],
      [ 791, 552, 176.74, 820, 594 ],
      [ 632, 573, 184.2, 671, 616 ],
      [ 218, 237, -98.78, 266, 250 ],
      [ 827, 186, -56.01, 615, 789 ],
      [ 373, 738, 92.05, 751, 397 ],
      [ 279, 169, -128.47, 304, 322 ],
      [ 659, 211, 45.56, 612, 618 ],
      [ 506, 86, -229.86, 392, 440 ],
      [ 490, 143, -224.56, 449, 444 ],
      [ 958, 68, -308.34, 647, 793 ],
      [ 150, 661, 208.86, 449, 650 ],
      [ 913, 367, 25.37, 981, 723 ],
      [ 663, 530, 247.34, 743, 814 ],
      [ 10, 381, -299.04, 337, 193 ],
      [ 443, 661, -327.45, 729, 795 ],
      [ 62, 102, -169.91, 78, 109 ],
      [ 391, 596, 27.65, 623, 709 ],
      [ 968, 207, 107.88, 494, 984 ],
      [ 129, 574, 241.1, 564, 388 ],
      [ 321, 524, 304.55, 613, 561 ],
      [ 395, 636, 204.06, 618, 741 ],
      [ 180, 553, -172.71, 247, 570 ],
      [ 728, 281, 323.36, 751, 659 ],
      [ 568, 289, -39.48, 621, 584 ],
      [ 655, 620, -140.52, 898, 894 ],
      [ 701, 666, -45.48, 965, 966 ],
      [ 971, 191, -207.62, 949, 618 ],
      [ 499, 149, -14.17, 518, 266 ],
      [ 140, 245, -196.4, 203, 273 ],
      [ 995, 485, 294.81, 856, 1107 ],
      [ 916, 13, 170.8, 906, 158 ],
      [ 694, 148, -31.69, 667, 490 ],
      [ 961, 694, 231.56, 1140, 1182 ],
      [ 908, 351, -218.24, 930, 837 ],
      [ 340, 74, -357.1, 343, 91 ],
      [ 652, 599, 267.04, 630, 681 ],
      [ 76, 559, -274.31, 562, 117 ],
      [ 293, 164, -135.55, 323, 321 ],
      [ 784, 269, 93.44, 316, 797 ],
      [ 1017, 247, -171.9, 1040, 387 ],
      [ 924, 165, -159.9, 923, 471 ],
      [ 655, 686, -21.69, 861, 879 ],
      [ 98, 638, -91.92, 640, 117 ],
      [ 939, 551, 236.78, 974, 1086 ],
      [ 117, 38, 305.36, 97, 117 ],
      [ 121, 527, 146.37, 392, 505 ],
      [ 1013, 583, -50.09, 1095, 1151 ],
      [ 111, 34, 103.63, 59, 114 ],
      [ 430, 622, 355.41, 477, 654 ],
      [ 51, 616, -133.67, 480, 460 ],
      [ 318, 541, 20.97, 490, 618 ],
      [ 603, 264, -56.65, 551, 648 ],
      [ 60, 337, -179.21, 63, 336 ],
      [ 441, 274, 336.64, 512, 426 ],
      [ 266, 361, -348.6, 331, 406 ],
      [ 189, 452, 144.59, 416, 476 ],
      [ 909, 443, -29.3, 1008, 830 ],
      [ 325, 526, 2.12, 343, 538 ],
      [ 985, 80, -135.32, 755, 748 ],
      [ 978, 118, -278.03, 253, 984 ],
      [ 276, 356, 228.81, 448, 440 ],
      [ 818, 40, -244.03, 394, 752 ],
      [ 225, 85, 71.79, 151, 240 ],
      [ 483, 416, 327.8, 629, 609 ],
      [ 449, 615, -54.23, 760, 723 ],
      [ 466, 756, -351.4, 573, 816 ],
      [ 159, 669, 63.34, 669, 442 ],
      [ 630, 145, -164.34, 644, 309 ],
      [ 121, 115, -89.96, 114, 120 ],
      [ 440, 80, 19.91, 440, 224 ],
      [ 557, 287, 353.35, 585, 349 ],
      [ 370, 132, -329.54, 385, 301 ],
      [ 535, 301, -250.42, 463, 604 ],
      [ 125, 663, 138.77, 531, 580 ],
      [ 79, 231, -5.19, 98, 237 ],
      [ 514, 580, -16.94, 659, 704 ],
      [ 526, 699, -110.11, 835, 732 ],
      [ 105, 403, 25.22, 266, 409 ],
      [ 383, 493, 132.67, 621, 614 ],
      [ 767, 55, 207.66, 704, 404 ],
      [ 283, 472, -243.5, 548, 463 ],
      [ 274, 749, -119.81, 785, 608 ],
      [ 317, 382, 162.07, 419, 459 ],
      [ 473, 476, 62.93, 639, 638 ],
      [ 158, 535, 279.41, 552, 242 ],
      [ 663, 152, 102.91, 296, 679 ],
      [ 127, 547, 38.04, 437, 509 ],
      [ 482, 625, -304.52, 788, 751 ],
      [ 6, 143, 157.38, 60, 133 ],
      [ 707, 718, 208.39, 961, 967 ],
      [ 490, 162, -69.52, 322, 516 ],
      [ 656, 367, -87.51, 394, 671 ],
      [ 997, 605, 270.38, 610, 1000 ],
      [ 716, 191, -4.08, 727, 241 ],
      [ 293, 309, -223.07, 425, 425 ],
      [ 179, 38, -80.84, 65, 182 ],
      [ 837, 133, -288.78, 395, 835 ],
      [ 379, 494, 296.46, 609, 559 ],
      [ 458, 498, -289.49, 621, 597 ],
      [ 170, 260, -336.57, 258, 306 ],
      [ 978, 627, -184.85, 1027, 706 ],
      [ 367, 351, 158.99, 468, 458 ],
      [ 755, 330, 5.41, 782, 400 ],
      [ 628, 99, 184.02, 632, 142 ],
      [ 110, 319, -246.44, 335, 227 ],
      [ 304, 452, -211.14, 494, 543 ],
      [ 702, 312, -87.03, 347, 717 ],
      [ 803, 596, -344.23, 934, 792 ],
      [ 712, 171, 122.5, 526, 691 ],
      [ 813, 298, -354.2, 838, 378 ],
      [ 846, 263, 233.36, 714, 834 ],
      [ 71, 539, -158.35, 263, 526 ],
      [ 809, 337, 150.89, 870, 686 ],
      [ 305, 39, 243.13, 171, 289 ],
      [ 55, 121, -178.71, 56, 121 ],
      [ 216, 349, 244.81, 406, 343 ],
      [ 428, 106, -183.44, 433, 130 ],
      [ 816, 708, -345.39, 968, 890 ],
      [ 839, 474, -185.56, 881, 552 ],
      [ 501, 315, 93.42, 343, 518 ],
      [ 509, 181, 339.95, 539, 344 ],
      [ 329, 234, 124.48, 379, 402 ],
      [ 616, 73, -280.47, 183, 618 ],
      [ 318, 301, -244.54, 408, 415 ],
      [ 152, 342, -131.81, 355, 340 ],
      [ 410, 504, 194.66, 523, 590 ],
      [ 246, 56, 294.02, 150, 247 ],
      [ 497, 668, 237.51, 828, 777 ],
      [ 518, 529, -38.43, 733, 735 ],
      [ 766, 136, 256.73, 306, 775 ],
      [ 387, 517, -293.6, 628, 561 ],
      [ 358, 763, -74.25, 830, 551 ],
      [ 23, 457, -6.24, 71, 456 ],
      [ 137, 695, 275.85, 703, 207 ],
      [ 301, 41, -123.98, 201, 271 ],
      [ 625, 348, 347.85, 682, 471 ],
      [ 390, 532, -319.83, 641, 658 ],
      [ 296, 137, 306, 283, 320 ],
      [ 122, 525, 90, 525, 122 ],
      [ 579, 331, 210, 666, 575 ],
      [ 433, 621, -30, 684, 754 ],
      [ 527, 121, -30, 516, 368 ],
      [ 933, 138, -330, 877, 586 ],
      [ 413, 240, 330, 476, 414 ],
      [ 293, 466, -360, 293, 466 ],
      [ 629, 166, -30, 626, 458 ],
      [ 611, 458, -270, 458, 611 ],
      [ 668, 680, -120, 922, 917 ],
      [ 592, 3, -360, 592, 3 ],
      [ 195, 438, -30, 386, 476 ],
      [ 1003, 762, -180, 1003, 762 ],
      [ 451, 389, -60, 561, 584 ],
      [ 917, 320, 60, 735, 954 ],
      [ 967, 629, 300, 1027, 1151 ],
      [ 920, 80, 0, 920, 80 ],
      [ 926, 551, 150, 1076, 938 ],
      [ 372, 242, -150, 442, 395 ],
      [ 526, 742, -240, 905, 825 ],
      [ 862, 418, -210, 955, 791 ],
      [ 873, 563, 210, 1037, 923 ],
      [ 300, 547, 210, 532, 623 ],
      [ 554, 117, -180, 554, 117 ],
      [ 969, 36, 150, 857, 514 ],
      [ 544, 761, 0, 544, 761 ],
      [ 438, 683, 90, 683, 438 ],
      [ 953, 609, -120, 1002, 1129 ],
      [ 664, 120, 90, 120, 664 ],
      [ 986, 514, 90, 514, 986 ],
      [ 167, 392, -270, 392, 167 ],
      [ 375, 279, 300, 428, 463 ],
      [ 142, 503, 60, 506, 373 ],
      [ 923, 151, 180, 923, 151 ],
      [ 175, 109, 210, 205, 180 ],
      [ 876, 517, 210, 1016, 885 ],
      [ 368, 78, 120, 251, 356 ],
      [ 990, 24, -30, 868, 516 ],
      [ 299, 522, 90, 522, 299 ],
      [ 954, 493, -150, 1072, 903 ],
      [ 549, 618, 210, 783, 808 ],
      [ 701, 130, -240, 463, 671 ],
      [ 579, 236, 240, 492, 618 ],
      [ 162, 160, -360, 162, 160 ],
      [ 925, 291, -270, 291, 925 ],
      [ 711, 180, 240, 510, 704 ],
      [ 488, 455, 330, 649, 638 ],
      [ 150, 429, 330, 343, 447 ],
      [ 625, 653, 90, 653, 625 ],
      [ 235, 600, -180, 235, 600 ],
      [ 529, 244, -300, 475, 580 ],
      [ 322, 1, -180, 322, 1 ],
      [ 487, 435, 30, 638, 620 ],
      [ 298, 556, 210, 535, 630 ],
      [ 309, 482, -270, 482, 309 ],
      [ 466, 246, -90, 246, 466 ],
      [ 287, 612, 360, 287, 612 ],
      [ 211, 724, 30, 544, 732 ],
      [ 167, 425, -90, 425, 167 ],
      [ 818, 600, -270, 600, 818 ],
      [ 77, 229, -30, 180, 236 ],
      [ 160, 470, -30, 372, 487 ],
      [ 416, 67, -330, 393, 265 ],
      [ 987, 289, 210, 998, 742 ],
      [ 747, 521, -90, 521, 747 ],
      [ 894, 101, 150, 824, 532 ],
      [ 256, 316, 0, 256, 316 ],
      [ 225, 474, -30, 430, 522 ],
      [ 765, 599, -360, 765, 599 ],
      [ 128, 444, 210, 331, 448 ],
      [ 291, 502, -300, 580, 503 ],
      [ 778, 621, 150, 983, 925 ],
      [ 241, 165, 210, 290, 262 ],
      [ 284, 137, -270, 137, 284 ],
      [ 44, 697, 240, 625, 386 ],
      [ 446, 599, 360, 446, 599 ],
      [ 152, 23, -90, 23, 152 ],
      [ 673, 191, 360, 673, 191 ],
      [ 110, 764, 210, 476, 716 ],
      [ 113, 92, -120, 135, 142 ],
      [ 916, 309, -360, 916, 309 ],
      [ 999, 124, 240, 605, 926 ],
      [ 683, 305, -360, 683, 305 ],
      [ 456, 517, -210, 652, 674 ],
      [ 614, 550, -300, 782, 806 ],
      [ 441, 116, 300, 319, 439 ],
      [ 403, 230, 300, 399, 464 ],
      [ 786, 734, -240, 1028, 1046 ],
      [ 777, 48, -270, 48, 777 ],
      [ 543, 133, -180, 543, 133 ],
      [ 899, 52, 150, 804, 493 ],
      [ 900, 32, 270, 32, 900 ],
      [ 704, 544, -210, 881, 821 ],
      [ 728, 118, 300, 464, 689 ],
      [ 935, 577, 90, 577, 935 ],
      [ 22, 140, 210, 88, 131 ],
      [ 931, 441, -330, 1026, 847 ],
      [ 846, 263, -360, 846, 263 ],
      [ 1015, 77, -300, 574, 917 ],
      [ 154, 484, -180, 154, 484 ],
      [ 439, 390, -120, 556, 574 ],
      [ 112, 297, -90, 297, 112 ],
      [ 974, 56, -300, 534, 871 ],
      [ 194, 602, -210, 469, 616 ],
      [ 103, 535, 360, 103, 535 ],
      [ 744, 551, -240, 848, 918 ],
      [ 517, 481, 180, 517, 481 ],
      [ 595, 348, -300, 598, 689 ],
      [ 627, 341, -240, 608, 711 ],
      [ 723, 456, 240, 755, 853 ],
      [ 996, 17, -150, 870, 512 ],
      [ 312, 101, 150, 320, 241 ],
      [ 695, 63, 210, 632, 401 ],
      [ 779, 208, 60, 569, 778 ],
      [ 1018, 285, -180, 1018, 285 ],
      [ 1003, 75, 360, 1003, 75 ],
      [ 149, 462, 90, 462, 149 ],
      [ 907, 140, -330, 855, 574 ],
      [ 991, 610, 0, 991, 610 ],
      [ 153, 384, -300, 409, 324 ],
      [ 984, 363, -300, 805, 1033 ],
      [ 287, 597, -180, 287, 597 ],
      [ 1008, 350, -120, 806, 1046 ],
      [ 780, 165, 60, 532, 757 ],
      [ 331, 161, 360, 331, 161 ],
      [ 594, 144, -330, 586, 421 ],
      [ 568, 256, 120, 505, 618 ],
      [ 160, 168, 270, 168, 160 ],
      [ 214, 143, 90, 143, 214 ],
      [ 722, 257, -270, 257, 722 ],
      [ 813, 227, 90, 227, 813 ],
      [ 926, 441, -90, 441, 926 ],
      [ 149, 429, 240, 445, 343 ],
      [ 480, 246, -330, 538, 452 ],
      [ 64, 493, -180, 64, 493 ],
      [ 26, 171, -30, 107, 161 ],
      [ 87, 597, 210, 373, 559 ],
      [ 764, 716, 360, 764, 716 ],
      [ 602, 108, -240, 394, 574 ],
      [ 229, 649, 0, 229, 649 ],
      [ 367, 490, 240, 606, 561 ],
      [ 989, 416, -90, 416, 989 ],
      [ 357, 528, 330, 572, 635 ],
      [ 190, 119, -180, 190, 119 ],
      [ 243, 167, 300, 265, 293 ],
      [ 510, 186, -270, 186, 510 ],
      [ 968, 251, 300, 699, 963 ],
      [ 743, 55, 240, 418, 670 ],
      [ 741, 506, 360, 741, 506 ],
      [ 928, 678, 240, 1050, 1141 ],
      [ 434, 185, -30, 467, 377 ],
      [ 241, 158, 360, 241, 158 ],
      [ 636, 425, 150, 762, 684 ],
      [ 578, 568, 240, 780, 783 ],
      [ 847, 750, -330, 1108, 1073 ],
      [ 750, 364, -150, 830, 689 ],
      [ 880, 322, 90, 322, 880 ],
      [ 780, 111, 150, 730, 484 ],
      [ 610, 666, -120, 881, 860 ],
      [ 584, 594, -210, 802, 804 ],
      [ 398, 151, -30, 419, 330 ],
      [ 864, 331, 120, 718, 912 ],
      [ 861, 39, -210, 764, 463 ],
      [ 536, 472, 360, 536, 472 ],
      [ 370, 340, 330, 489, 479 ],
      [ 425, 134, -60, 327, 435 ],
      [ 728, 25, 240, 385, 642 ],
      [ 341, 610, 360, 341, 610 ],
      [ 27, 299, 270, 299, 27 ],
      [ 388, 737, 90, 737, 388 ],
      [ 628, 267, 240, 544, 676 ],
      [ 90, 147, -180, 90, 147 ],
      [ 785, 24, -150, 690, 412 ],
      [ 1023, 426, 300, 879, 1098 ],
      [ 994, 703, -120, 1105, 1211 ],
      [ 939, 253, 30, 939, 688 ],
      [ 344, 31, 60, 198, 312 ],
      [ 179, 287, -120, 337, 298 ],
      [ 160, 307, 180, 160, 307 ],
      [ 6, 597, 150, 303, 518 ],
      [ 643, 300, -330, 706, 581 ],
      [ 456, 367, -210, 577, 544 ],
      [ 710, 188, -180, 710, 188 ],
      [ 1016, 187, 240, 669, 972 ],
      [ 927, 164, 180, 927, 164 ],
      [ 267, 99, -330, 280, 219 ],
      [ 808, 357, -300, 712, 877 ],
      [ 358, 491, -30, 555, 604 ],
      [ 721, 611, 270, 611, 721 ],
      [ 458, 615, 120, 761, 702 ],
      [ 153, 328, -330, 296, 360 ],
      [ 204, 670, 30, 511, 681 ],
      [ 449, 434, 210, 604, 599 ],
      [ 725, 428, -360, 725, 428 ],
      [ 545, 355, -210, 648, 578 ],
      [ 277, 554, -120, 617, 515 ],
      [ 345, 392, 240, 510, 493 ],
      [ 455, 660, -30, 723, 799 ],
      [ 946, 432, -180, 946, 432 ],
      [ 822, 7, -330, 714, 416 ],
      [ 469, 122, -30, 466, 340 ],
      [ 516, 274, -120, 494, 582 ],
      [ 27, 611, 300, 541, 328 ],
      [ 812, 386, -30, 895, 740 ],
      [ 822, 27, 330, 724, 434 ],
      [ 24, 234, 120, 214, 136 ],
      [ 383, 492, -270, 492, 383 ],
      [ 194, 65, -360, 194, 65 ],
      [ 658, 7, 60, 334, 572 ],
      [ 912, 623, 60, 995, 1100 ],
      [ 961, 206, 180, 961, 206 ],
      [ 424, 593, -300, 725, 663 ],
      [ 778, 613, 300, 918, 979 ],
      [ 687, 454, -60, 735, 821 ],
      [ 142, 302, -60, 331, 273 ],
      [ 69, 320, 180, 69, 320 ],
      [ 734, 607, -90, 607, 734 ],
      [ 898, 752, -30, 1152, 1100 ],
      [ 916, 477, -30, 1031, 871 ],
      [ 487, 393, -150, 617, 582 ],
      [ 66, 346, 30, 230, 332 ],
      [ 832, 663, -120, 989, 1051 ],
      [ 938, 478, -300, 882, 1051 ],
      [ 834, 225, 150, 834, 610 ],
      [ 221, 331, -300, 397, 356 ],
      [ 666, 382, 0, 666, 382 ],
      [ 398, 165, -150, 426, 341 ],
      [ 786, 70, -150, 714, 453 ],
      [ 236, 757, 300, 772, 582 ],
      [ 718, 353, -60, 663, 797 ],
      [ 1006, 403, 270, 403, 1006 ],
      [ 545, 258, 180, 545, 258 ],
      [ 874, 193, -120, 603, 852 ],
      [ 996, 51, 120, 541, 886 ],
      [ 699, 216, -330, 713, 536 ],
      [ 805, 715, 60, 1021, 1054 ],
      [ 312, 245, 210, 392, 367 ],
      [ 628, 66, 270, 66, 628 ],
      [ 922, 242, 270, 242, 922 ],
      [ 827, 12, -120, 422, 721 ],
      [ 231, 766, 270, 766, 231 ],
      [ 95, 407, -240, 399, 284 ],
      [ 842, 294, -60, 674, 876 ],
      [ 178, 272, 0, 178, 272 ],
      [ 838, 28, 210, 738, 442 ],
      [ 914, 631, 150, 1106, 1001 ],
      [ 476, 97, -360, 476, 97 ],
      [ 257, 567, -300, 619, 505 ],
      [ 131, 490, -60, 488, 358 ],
      [ 1013, 342, -60, 801, 1048 ],
      [ 317, 515, -60, 603, 531 ],
      [ 157, 585, 360, 157, 585 ],
      [ 341, 448, -120, 557, 518 ],
      [ 781, 581, 150, 966, 892 ],
      [ 265, 441, 180, 265, 441 ],
      [ 35, 359, 30, 209, 328 ],
      [ 757, 716, 150, 1013, 997 ],
      [ 782, 141, -60, 511, 747 ],
      [ 893, 238, -330, 892, 652 ],
      [ 302, 230, 0, 302, 230 ],
      [ 752, 467, -240, 779, 883 ],
      [ 174, 585, 330, 442, 594 ],
      [ 138, 72, 0, 138, 72 ],
      [ 492, 658, -180, 492, 658 ],
      [ 185, 88, 270, 88, 185 ],
      [ 939, 113, -150, 869, 566 ],
      [ 487, 681, -180, 487, 681 ],
      [ 163, 499, -60, 512, 390 ],
      [ 594, 400, 180, 594, 400 ],
      [ 655, 626, -360, 655, 626 ],
      [ 148, 422, 120, 438, 338 ],
      [ 325, 552, -60, 639, 557 ],
      [ 259, 655, 0, 259, 655 ],
      [ 789, 255, -120, 614, 810 ],
      [ 46, 394, -30, 235, 364 ],
      [ 915, 330, 90, 330, 915 ],
      [ 214, 695, 0, 214, 695 ],
      [ 466, 48, -270, 48, 466 ],
      [ 906, 494, 120, 880, 1030 ],
      [ 638, 217, 0, 638, 217 ],
      [ 676, 327, -360, 676, 327 ],
      [ 312, 571, 180, 312, 571 ],
      [ 727, 765, 90, 765, 727 ],
      [ 214, 588, 330, 478, 616 ],
      [ 576, 622, -30, 808, 827 ],
      [ 31, 540, 270, 540, 31 ],
      [ 661, 701, 210, 922, 936 ],
      [ 149, 282, 270, 282, 149 ],
      [ 302, 193, 0, 302, 193 ],
      [ 987, 671, 210, 1189, 1073 ],
      [ 459, 410, -240, 584, 601 ],
      [ 492, 644, 330, 747, 804 ],
      [ 236, 421, 330, 414, 483 ],
      [ 853, 581, 150, 1028, 928 ],
      [ 785, 245, 0, 785, 245 ],
      [ 217, 268, -210, 321, 339 ],
      [ 90, 763, -300, 705, 458 ],
      [ 888, 107, 0, 888, 107 ],
      [ 726, 333, 180, 726, 333 ],
      [ 198, 305, 90, 305, 198 ],
      [ 983, 649, -270, 649, 983 ],
      [ 155, 250, 360, 155, 250 ],
      [ 89, 427, 30, 290, 414 ],
      [ 17, 298, -150, 162, 265 ],
      [ 729, 118, 90, 118, 729 ],
      [ 223, 280, 330, 332, 353 ],
      [ 436, 347, 330, 550, 519 ],
      [ 558, 244, -300, 489, 605 ],
      [ 32, 20, 0, 32, 20 ],
      [ 787, 169, 300, 538, 765 ],
      [ 387, 137, 180, 387, 137 ],
      [ 534, 253, -330, 588, 485 ],
      [ 514, 320, 90, 320, 514 ],
      [ 31, 332, -360, 31, 332 ],
      [ 321, 111, -240, 256, 331 ],
      [ 936, 277, 30, 948, 707 ],
      [ 884, 604, 360, 884, 604 ],
      [ 825, 254, -150, 840, 631 ],
      [ 907, 278, -120, 693, 923 ],
      [ 408, 100, 30, 403, 290 ],
      [ 315, 390, 180, 315, 390 ],
      [ 62, 22, -330, 64, 49 ],
      [ 141, 407, -30, 325, 422 ],
      [ 789, 429, 300, 765, 897 ],
      [ 808, 670, -330, 1034, 983 ],
      [ 984, 603, -60, 1012, 1153 ],
      [ 509, 498, -240, 685, 688 ],
      [ 496, 349, 30, 603, 549 ],
      [ 808, 260, 300, 627, 829 ],
      [ 122, 566, -360, 122, 566 ],
      [ 684, 33, 30, 608, 370 ],
      [ 405, 79, 30, 389, 270 ],
      [ 463, 185, -330, 492, 391 ],
      [ 947, 8, 90, 8, 947 ],
      [ 850, 614, 0, 850, 614 ],
      [ 899, 584, -150, 1069, 954 ],
      [ 294, 197, 330, 352, 318 ],
      [ 493, 568, -90, 568, 493 ],
      [ 7, 406, 180, 7, 406 ],
      [ 914, 497, -30, 1039, 887 ],
      [ 930, 242, 0, 930, 242 ],
      [ 440, 546, 90, 546, 440 ],
      [ 1002, 125, 270, 125, 1002 ],
      [ 58, 67, 270, 67, 58 ],
      [ 710, 703, 120, 963, 964 ],
      [ 205, 609, -60, 628, 481 ],
      [ 505, 61, 120, 305, 466 ],
      [ 445, 431, -60, 594, 600 ],
      [ 848, 435, 330, 951, 801 ],
      [ 574, 352, 60, 591, 673 ],
      [ 29, 281, 300, 256, 165 ],
      [ 552, 611, 90, 611, 552 ],
      [ 155, 594, 210, 430, 590 ],
      [ 4, 637, 270, 637, 4 ],
      [ 887, 401, 210, 968, 789 ],
      [ 539, 555, 60, 750, 743 ],
      [ 968, 165, 120, 626, 919 ],
      [ 635, 498, -210, 798, 747 ],
      [ 36, 366, 210, 213, 334 ],
      [ 1001, 28, -210, 880, 523 ],
      [ 580, 49, 60, 331, 526 ],
      [ 481, 463, -90, 463, 481 ],
      [ 85, 579, -240, 543, 361 ],
      [ 878, 581, 360, 878, 581 ],
      [ 735, 478, 0, 735, 478 ],
      [ 506, 113, -210, 494, 349 ],
      [ 64, 71, -30, 90, 93 ],
      [ 728, 547, -300, 837, 903 ],
      [ 941, 574, 60, 967, 1101 ],
      [ 710, 556, 90, 556, 710 ],
      [ 948, 223, 120, 666, 930 ],
      [ 536, 583, -180, 536, 583 ],
      [ 938, 646, -360, 938, 646 ],
      [ 83, 536, 210, 338, 504 ],
      [ 76, 319, -60, 312, 224 ],
      [ 603, 698, 30, 871, 905 ],
      [ 846, 746, 120, 1068, 1104 ],
      [ 284, 523, -120, 594, 506 ],
      [ 385, 461, -300, 591, 563 ],
      [ 974, 225, 240, 681, 955 ],
      [ 575, 167, -300, 432, 580 ],
      [ 241, 569, 270, 569, 241 ],
      [ 524, 504, 150, 705, 696 ],
      [ 554, 566, -60, 765, 762 ],
      [ 334, 622, 240, 705, 599 ],
      [ 750, 306, 180, 750, 306 ],
      [ 294, 172, 150, 340, 294 ],
      [ 955, 385, -90, 385, 955 ],
      [ 297, 673, 360, 297, 673 ],
      [ 403, 635, -180, 403, 635 ],
      [ 226, 298, 0, 226, 298 ],
      [ 345, 478, -210, 537, 585 ],
      [ 216, 103, 300, 195, 238 ],
      [ 948, 518, 90, 518, 948 ],
      [ 346, 768, -60, 836, 683 ],
      [ 169, 562, 240, 570, 426 ],
      [ 921, 14, -330, 804, 472 ],
      [ 618, 730, 30, 900, 940 ],
      [ 1013, 184, -60, 664, 969 ],
      [ 998, 486, -180, 998, 486 ],
      [ 257, 655, 120, 695, 548 ],
      [ 764, 146, -180, 764, 146 ],
      [ 1004, 308, -60, 767, 1023 ],
      [ 892, 250, -330, 897, 662 ],
      [ 510, 510, -330, 696, 696 ],
      [ 956, 636, 240, 1028, 1144 ],
      [ 787, 558, 240, 875, 959 ],
      [ 846, 253, 210, 858, 641 ],
      [ 395, 245, -360, 395, 245 ],
      [ 840, 225, 120, 614, 838 ],
      [ 83, 417, 0, 83, 417 ],
      [ 746, 222, 150, 757, 563 ],
      [ 1022, 206, -300, 688, 988 ],
      [ 412, 107, -60, 297, 409 ],
      [ 491, 489, -330, 669, 668 ],
      [ 569, 438, 300, 662, 711 ],
      [ 372, 260, 90, 260, 372 ],
      [ 194, 126, 330, 230, 206 ],
      [ 988, 422, -180, 988, 422 ],
      [ 1003, 283, 60, 746, 1009 ],
      [ 641, 345, -300, 619, 727 ],
      [ 129, 136, -90, 136, 129 ],
      [ 835, 134, 120, 533, 789 ],
      [ 926, 443, 240, 846, 1022 ],
      [ 326, 42, -30, 302, 199 ],
      [ 405, 469, -360, 405, 469 ],
      [ 307, 747, -120, 799, 638 ],
      [ 952, 124, 0, 952, 124 ],
      [ 911, 96, -330, 836, 538 ],
      [ 171, 80, -60, 153, 188 ],
      [ 754, 561, 270, 561, 754 ],
      [ 868, 657, -330, 1079, 1002 ],
      [ 254, 515, -210, 476, 571 ],
      [ 937, 441, 240, 849, 1031 ],
      [ 740, 685, 270, 685, 740 ],
      [ 169, 220, -30, 255, 275 ],
      [ 183, 449, -30, 382, 480 ],
      [ 541, 395, 90, 395, 541 ],
      [ 44, 310, 180, 44, 310 ],
      [ 108, 438, 240, 432, 311 ],
      [ 549, 235, 30, 592, 478 ],
      [ 425, 118, -60, 313, 427 ],
      [ 481, 308, -300, 507, 570 ],
      [ 715, 243, 120, 567, 739 ],
      [ 515, 29, 30, 460, 282 ],
      [ 370, 156, 240, 319, 397 ],
      [ 849, 292, -60, 676, 881 ],
      [ 275, 698, 330, 586, 741 ],
      [ 891, 731, -120, 1077, 1136 ],
      [ 610, 43, 300, 340, 549 ],
      [ 434, 455, -210, 602, 609 ],
      [ 1006, 6, -90, 6, 1006 ],
      [ 405, 366, 210, 532, 518 ],
      [ 491, 134, -330, 492, 361 ],
      [ 139, 520, -300, 519, 380 ],
      [ 699, 29, -150, 619, 373 ],
      [ 528, 665, 150, 789, 838 ],
      [ 955, 103, 90, 103, 955 ],
      [ 884, 3, 30, 766, 444 ],
      [ 203, 460, 90, 460, 203 ],
      [ 104, 17, -210, 98, 65 ],
      [ 319, 3, -210, 277, 161 ],
      [ 691, 307, 150, 751, 610 ],
      [ 448, 674, 270, 674, 448 ],
      [ 529, 10, 30, 463, 273 ],
      [ 647, 535, 60, 786, 827 ],
      [ 973, 162, -30, 922, 626 ],
      [ 663, 110, 60, 426, 629 ],
      [ 259, 5, 60, 133, 226 ],
      [ 829, 157, -240, 550, 794 ],
      [ 432, 235, -240, 419, 490 ],
      [ 641, 473, -210, 791, 729 ],
      [ 857, 223, 60, 621, 853 ],
      [ 537, 559, -30, 744, 752 ],
      [ 396, 187, 0, 396, 187 ],
      [ 947, 672, -240, 1055, 1155 ],
      [ 512, 633, -90, 633, 512 ],
      [ 963, 362, 30, 1014, 795 ],
      [ 527, 556, 30, 734, 745 ],
      [ 95, 410, 180, 95, 410 ],
      [ 274, 733, -330, 603, 771 ],
      [ 476, 446, 120, 623, 634 ],
      [ 681, 321, 330, 749, 618 ],
      [ 271, 723, 150, 595, 760 ],
      [ 759, 252, 330, 782, 597 ],
      [ 236, 194, 240, 285, 300 ],
      [ 426, 578, 90, 578, 426 ],
      [ 831, 532, -300, 876, 985 ],
      [ 358, 159, 240, 316, 389 ],
      [ 915, 229, -120, 654, 906 ],
      [ 657, 435, -150, 785, 704 ],
      [ 711, 23, 300, 374, 626 ],
      [ 371, 534, -150, 587, 646 ],
      [ 327, 736, -180, 327, 736 ],
      [ 4, 537, 60, 466, 271 ],
      [ 953, 714, 240, 1093, 1181 ],
      [ 749, 265, 60, 603, 780 ],
      [ 365, 119, -180, 365, 119 ],
      [ 464, 388, -30, 594, 568 ],
      [ 280, 306, 210, 394, 404 ],
      [ 641, 30, -120, 345, 569 ],
      [ 956, 563, -90, 563, 956 ],
      [ 840, 72, -330, 763, 481 ],
      [ 128, 317, -360, 128, 317 ],
      [ 379, 320, 150, 488, 465 ],
      [ 966, 266, 120, 712, 968 ],
      [ 786, 60, 360, 786, 60 ],
      [ 352, 333, -270, 333, 352 ],
      [ 626, 681, 120, 902, 881 ],
      [ 87, 122, -330, 136, 149 ],
      [ 878, 603, -300, 960, 1061 ],
      [ 199, 551, 240, 575, 447 ],
      [ 575, 413, 300, 644, 703 ],
      [ 638, 509, -120, 759, 806 ],
      [ 658, 25, 180, 658, 25 ],
      [ 369, 749, -300, 833, 693 ],
      [ 7, 570, -240, 497, 290 ],
      [ 1022, 66, 60, 567, 918 ],
      [ 147, 535, 0, 147, 535 ],
      [ 812, 600, 120, 925, 1002 ],
      [ 861, 490, -60, 853, 990 ],
      [ 949, 640, -270, 640, 949 ],
      [ 763, 303, 120, 643, 810 ],
      [ 673, 13, -120, 346, 588 ],
      [ 1006, 506, -90, 506, 1006 ],
      [ 740, 14, -120, 381, 646 ],
      [ 829, 19, -300, 430, 726 ],
      [ 996, 17, -240, 512, 869 ],
      [ 583, 127, 270, 127, 583 ],
      [ 53, 736, 120, 663, 412 ],
      [ 709, 613, -150, 920, 884 ],
      [ 121, 556, -270, 556, 121 ],
      [ 266, 360, 0, 266, 360 ],
      [ 962, 96, 30, 881, 563 ],
      [ 287, 82, -240, 214, 288 ],
      [ 668, 637, -210, 896, 884 ],
      [ 1024, 490, -210, 1131, 934 ],
      [ 91, 468, -180, 91, 468 ],
      [ 269, 137, -60, 252, 300 ],
      [ 137, 177, -90, 177, 137 ],
      [ 804, 708, -240, 1014, 1049 ],
      [ 76, 30, 300, 62, 80 ],
      [ 200, 229, -90, 229, 200 ],
      [ 728, 182, 0, 728, 182 ],
      [ 248, 397, 60, 467, 412 ],
      [ 442, 130, -60, 332, 447 ],
      [ 654, 130, -330, 631, 439 ],
      [ 892, 197, 120, 616, 869 ],
      [ 128, 399, 270, 399, 128 ],
      [ 557, 501, -330, 732, 712 ],
      [ 942, 335, -360, 942, 335 ],
      [ 95, 392, -360, 95, 392 ],
      [ 1014, 542, -120, 975, 1148 ],
      [ 364, 320, 60, 458, 475 ],
      [ 866, 505, -300, 869, 1001 ],
      [ 453, 68, -180, 453, 68 ],
      [ 888, 558, -60, 925, 1048 ],
      [ 952, 459, 150, 1053, 872 ],
      [ 616, 554, -210, 810, 786 ],
      [ 462, 203, 270, 203, 462 ],
      [ 543, 141, -150, 540, 392 ],
      [ 544, 212, 240, 455, 576 ],
      [ 585, 204, 0, 585, 204 ],
      [ 931, 477, 330, 1044, 878 ],
      [ 495, 358, 60, 557, 607 ],
      [ 588, 697, 120, 897, 856 ],
      [ 854, 594, -90, 594, 854 ],
      [ 268, 540, 360, 268, 540 ],
      [ 968, 234, 150, 955, 685 ],
      [ 152, 580, 360, 152, 580 ],
      [ 3, 219, -270, 219, 3 ],
      [ 301, 627, -60, 692, 573 ],
      [ 96, 297, 150, 231, 303 ],
      [ 614, 227, -150, 644, 503 ],
      [ 533, 598, 210, 759, 783 ],
      [ 101, 271, 150, 222, 284 ],
      [ 782, 143, -30, 748, 515 ],
      [ 160, 344, -240, 377, 309 ],
      [ 148, 301, -30, 278, 335 ],
      [ 874, 415, -210, 963, 794 ],
      [ 847, 417, 0, 847, 417 ],
      [ 1007, 642, -120, 1058, 1192 ],
      [ 426, 714, 150, 725, 829 ],
      [ 117, 406, -360, 117, 406 ],
      [ 443, 37, 210, 401, 252 ],
      [ 222, 113, -270, 113, 222 ],
      [ 928, 699, -120, 1068, 1152 ],
      [ 378, 50, 210, 351, 231 ],
      [ 548, 161, -270, 161, 548 ],
      [ 4, 48, 150, 27, 42 ],
      [ 233, 682, -180, 233, 682 ],
      [ 754, 670, -330, 987, 956 ],
      [ 61, 221, -360, 61, 221 ],
      [ 789, 309, 30, 837, 662 ],
      [ 795, 641, 60, 952, 1008 ],
      [ 573, 39, 180, 573, 39 ],
      [ 712, 734, 90, 734, 712 ],
      [ 18, 250, 150, 140, 224 ],
      [ 805, 660, 300, 973, 1027 ],
      [ 968, 663, 360, 968, 663 ],
      [ 661, 69, 270, 69, 661 ],
      [ 908, 634, 180, 908, 634 ],
      [ 986, 680, -360, 986, 680 ],
      [ 992, 503, -60, 930, 1110 ],
      [ 514, 331, -180, 514, 331 ],
      [ 85, 760, -150, 452, 699 ],
      [ 830, 526, -180, 830, 526 ],
      [ 458, 539, 60, 695, 665 ],
      [ 152, 374, -30, 317, 400 ],
      [ 59, 332, -150, 216, 316 ],
      [ 29, 59, -90, 59, 29 ],
      [ 909, 740, -210, 1157, 1094 ],
      [ 633, 711, -300, 932, 903 ],
      [ 651, 686, 180, 651, 686 ],
      [ 56, 303, -240, 289, 198 ],
      [ 314, 366, -240, 473, 453 ],
      [ 624, 220, 270, 220, 624 ],
      [ 888, 563, 60, 931, 1050 ],
      [ 459, 676, -330, 735, 814 ],
      [ 911, 720, 0, 911, 720 ],
      [ 197, 742, 30, 541, 741 ],
      [ 599, 655, 30, 845, 866 ],
      [ 823, 361, -30, 892, 724 ],
      [ 929, 81, -120, 533, 844 ],
      [ 681, 122, 180, 681, 122 ],
      [ 874, 358, -210, 935, 745 ],
      [ 31, 57, 0, 31, 57 ],
      [ 914, 722, -210, 1152, 1080 ],
      [ 468, 298, -300, 491, 554 ],
      [ 520, 213, -330, 556, 443 ],
      [ 1014, 360, -360, 1014, 360 ],
      [ 563, 42, 270, 42, 563 ],
      [ 73, 658, -120, 605, 391 ],
      [ 530, 587, -60, 771, 751 ],
      [ 877, 329, 60, 723, 923 ],
      [ 604, 216, -330, 631, 488 ],
      [ 818, 239, -270, 239, 818 ],
      [ 302, 156, -330, 339, 285 ],
      [ 534, 507, -30, 715, 706 ],
      [ 644, 128, 180, 644, 128 ],
      [ 690, 121, -210, 657, 448 ],
      [ 699, 542, -180, 699, 542 ],
      [ 557, 597, -300, 795, 780 ],
      [ 896, 226, 270, 226, 896 ],
      [ 318, 116, -150, 332, 258 ],
      [ 926, 568, 60, 954, 1085 ],
      [ 987, 414, 300, 851, 1061 ],
      [ 98, 639, -300, 601, 403 ],
      [ 173, 272, 180, 173, 272 ],
      [ 644, 755, 330, 934, 976 ],
      [ 375, 504, -300, 623, 576 ],
      [ 585, 259, 210, 635, 515 ],
      [ 851, 677, 60, 1011, 1074 ],
      [ 970, 580, 270, 580, 970 ],
      [ 846, 51, -360, 846, 51 ],
      [ 136, 745, 180, 136, 745 ],
      [ 737, 716, -150, 995, 987 ],
      [ 632, 22, -300, 334, 558 ],
      [ 734, 151, -30, 710, 498 ],
      [ 488, 633, -30, 738, 792 ],
      [ 412, 146, -300, 331, 429 ],
      [ 497, 585, -30, 722, 755 ],
      [ 280, 455, -120, 533, 469 ],
      [ 857, 413, -330, 948, 786 ],
      [ 712, 279, -270, 279, 712 ],
      [ 719, 381, -300, 689, 812 ],
      [ 460, 165, -330, 480, 372 ],
      [ 751, 639, -330, 969, 928 ],
      [ 870, 421, -180, 870, 421 ],
      [ 327, 18, -300, 179, 292 ],
      [ 790, 327, -150, 847, 677 ],
      [ 900, 699, -330, 1128, 1054 ],
      [ 330, 141, 90, 141, 330 ],
      [ 663, 16, -240, 345, 581 ],
      [ 744, 549, 30, 918, 846 ],
      [ 892, 320, -360, 892, 320 ],
      [ 1009, 664, -180, 1009, 664 ],
      [ 33, 459, -330, 257, 414 ],
      [ 108, 344, 90, 344, 108 ],
      [ 385, 589, 120, 702, 626 ],
      [ 481, 413, -330, 622, 598 ],
      [ 771, 320, 360, 771, 320 ],
      [ 817, 567, -240, 899, 989 ],
      [ 444, 296, -240, 477, 531 ],
      [ 622, 86, 300, 383, 581 ],
      [ 148, 755, -150, 505, 727 ],
      [ 181, 743, -240, 733, 526 ],
      [ 434, 767, 210, 758, 880 ],
      [ 514, 80, -210, 485, 324 ],
      [ 131, 368, 360, 131, 368 ],
      [ 796, 728, 30, 1053, 1027 ],
      [ 872, 539, 330, 1024, 903 ],
      [ 856, 383, 150, 932, 758 ],
      [ 3, 716, -330, 360, 621 ],
      [ 194, 414, -240, 455, 374 ],
      [ 93, 525, -240, 501, 341 ],
      [ 417, 660, -270, 660, 417 ],
      [ 602, 218, -270, 218, 602 ],
      [ 376, 603, -210, 626, 708 ],
      [ 607, 701, 150, 875, 909 ],
      [ 599, 529, 120, 757, 781 ],
      [ 117, 415, -120, 416, 308 ],
      [ 61, 289, 270, 289, 61 ],
      [ 760, 291, 210, 803, 631 ],
      [ 830, 436, -120, 792, 935 ],
      [ 1013, 505, -360, 1013, 505 ],
      [ 153, 50, 270, 50, 153 ],
      [ 283, 501, -270, 501, 283 ],
      [ 412, 15, 330, 363, 219 ],
      [ 646, 469, 270, 469, 646 ],
      [ 347, 150, 30, 375, 303 ],
      [ 1019, 237, -330, 1000, 714 ],
      [ 360, 283, -30, 452, 425 ],
      [ 211, 85, 240, 178, 224 ],
      [ 1016, 707, -90, 707, 1016 ],
      [ 323, 698, -330, 628, 765 ],
      [ 330, 44, -300, 202, 307 ],
      [ 192, 256, 210, 293, 317 ],
      [ 343, 564, 210, 578, 658 ],
      [ 274, 281, -90, 281, 274 ],
      [ 115, 541, 60, 526, 369 ],
      [ 661, 537, 300, 794, 840 ],
      [ 736, 38, -180, 736, 38 ],
      [ 171, 196, -90, 196, 171 ],
      [ 1018, 190, -150, 975, 673 ],
      [ 397, 432, -210, 559, 571 ],
      [ 450, 679, -180, 450, 679 ],
      [ 568, 54, 60, 330, 518 ],
      [ 330, 311, -120, 433, 440 ],
      [ 111, 517, 150, 354, 502 ],
      [ 517, 602, -90, 602, 517 ],
      [ 99, 330, -300, 335, 250 ],
      [ 1021, 113, -270, 113, 1021 ],
      [ 271, 241, -90, 241, 271 ],
      [ 657, 236, 120, 532, 685 ],
      [ 961, 534, -210, 1099, 941 ],
      [ 147, 103, -300, 162, 178 ],
      [ 415, 529, -270, 529, 415 ],
      [ 1023, 8, 60, 518, 889 ],
      [ 335, 91, -210, 335, 245 ],
      [ 20, 479, -360, 20, 479 ],
      [ 426, 553, -30, 644, 692 ],
      [ 523, 550, 60, 737, 727 ],
      [ 671, 752, 300, 985, 957 ],
      [ 55, 477, -210, 285, 439 ],
      [ 715, 429, 300, 728, 833 ],
      [ 927, 539, -330, 1071, 930 ],
      [ 1018, 82, 180, 1018, 82 ],
      [ 158, 81, 180, 158, 81 ],
      [ 736, 332, 270, 332, 736 ],
      [ 978, 347, 0, 978, 347 ],
      [ 999, 666, -210, 1198, 1075 ],
      [ 435, 290, 300, 467, 521 ],
      [ 21, 25, 300, 31, 30 ],
      [ 951, 66, 0, 951, 66 ],
      [ 148, 602, -330, 429, 594 ],
      [ 55, 528, 210, 310, 483 ],
      [ 99, 524, 300, 502, 347 ],
      [ 847, 642, 360, 847, 642 ],
      [ 581, 425, -60, 657, 715 ],
      [ 436, 390, 270, 390, 436 ],
      [ 929, 371, 180, 929, 371 ],
      [ 111, 697, -300, 659, 444 ],
      [ 24, 712, -270, 712, 24 ],
      [ 941, 657, -210, 1142, 1038 ],
      [ 444, 768, -360, 444, 768 ],
      [ 519, 41, 150, 469, 294 ],
      [ 288, 115, -90, 115, 288 ],
      [ 166, 749, -210, 517, 730 ],
      [ 152, 417, 210, 339, 436 ],
      [ 579, 744, -180, 579, 744 ],
      [ 444, 673, 180, 444, 673 ],
      [ 173, 755, 120, 740, 525 ],
      [ 288, 5, 60, 147, 251 ],
      [ 436, 710, -30, 731, 833 ],
      [ 670, 275, -30, 717, 573 ],
      [ 683, 664, 0, 683, 664 ],
      [ 375, 111, 120, 283, 378 ],
      [ 765, 236, 90, 236, 765 ],
      [ 986, 349, -240, 794, 1026 ],
      [ 749, 15, -270, 15, 749 ],
      [ 8, 348, -360, 8, 348 ],
      [ 786, 477, 360, 786, 477 ],
      [ 443, 693, 360, 443, 693 ],
      [ 26, 251, 330, 147, 230 ],
      [ 483, 753, -180, 483, 753 ],
      [ 940, 497, -270, 497, 940 ],
      [ 426, 10, -150, 372, 221 ],
      [ 64, 583, 60, 536, 346 ],
      [ 701, 555, -330, 884, 831 ],
      [ 870, 348, -330, 927, 735 ],
      [ 1005, 354, 0, 1005, 354 ],
      [ 1013, 175, -270, 175, 1013 ],
      [ 1004, 507, -330, 1122, 940 ],
      [ 1001, 526, -90, 526, 1001 ],
      [ 921, 120, -120, 563, 856 ],
      [ 183, 56, 360, 183, 56 ],
      [ 329, 376, -360, 329, 376 ],
      [ 623, 423, 210, 750, 676 ],
      [ 207, 181, 0, 207, 181 ],
      [ 256, 65, 330, 253, 184 ],
      [ 324, 50, -60, 203, 305 ],
      [ 855, 42, 90, 42, 855 ],
      [ 998, 27, -150, 877, 521 ],
      [ 16, 9, 360, 16, 9 ],
      [ 392, 700, -270, 700, 392 ],
      [ 749, 69, -240, 434, 681 ],
      [ 743, 316, 150, 801, 644 ],
      [ 749, 15, -210, 655, 386 ],
      [ 509, 170, -30, 524, 401 ],
      [ 1007, 362, 30, 1053, 817 ],
      [ 945, 604, 90, 604, 945 ],
      [ 331, 477, 120, 578, 523 ],
      [ 973, 458, 150, 1071, 882 ],
      [ 243, 469, 180, 243, 469 ],
      [ 214, 763, 90, 763, 214 ],
      [ 344, 557, 180, 344, 557 ],
      [ 549, 345, -270, 345, 549 ],
      [ 229, 139, -270, 139, 229 ],
      [ 474, 520, -90, 520, 474 ],
      [ 959, 506, 270, 506, 959 ],
      [ 506, 447, 90, 447, 506 ],
      [ 119, 694, -180, 119, 694 ],
      [ 811, 656, 270, 656, 811 ],
      [ 514, 70, -30, 479, 318 ],
      [ 229, 230, -30, 312, 313 ],
      [ 877, 487, -240, 860, 1001 ],
      [ 592, 131, 90, 131, 592 ],
      [ 727, 302, 240, 624, 779 ],
      [ 882, 657, -30, 1091, 1010 ],
      [ 240, 608, -270, 608, 240 ],
      [ 78, 219, 150, 176, 227 ],
      [ 722, 308, 90, 308, 722 ],
      [ 977, 147, -30, 919, 615 ],
      [ 818, 533, 60, 870, 974 ],
      [ 261, 704, 270, 704, 261 ],
      [ 721, 593, 0, 721, 593 ],
      [ 898, 268, 120, 680, 910 ],
      [ 510, 46, -330, 464, 294 ],
      [ 307, 707, 300, 764, 618 ],
      [ 796, 118, 150, 748, 498 ],
      [ 935, 177, 360, 935, 177 ],
      [ 646, 718, -90, 718, 646 ],
      [ 258, 682, 60, 719, 564 ],
      [ 744, 528, -180, 744, 528 ],
      [ 298, 723, -240, 774, 618 ],
      [ 158, 495, 360, 158, 495 ],
      [ 668, 400, -150, 777, 679 ],
      [ 327, 14, -90, 14, 327 ],
      [ 389, 244, -150, 457, 404 ],
      [ 302, 73, -30, 297, 214 ],
      [ 1, 6, 120, 5, 2 ],
      [ 1016, 490, 90, 490, 1016 ],
      [ 392, 683, 0, 392, 683 ],
      [ 989, 472, -240, 903, 1091 ],
      [ 246, 695, -270, 695, 246 ],
      [ 442, 46, 210, 404, 260 ],
      [ 404, 547, -150, 622, 675 ],
      [ 723, 24, -150, 637, 381 ],
      [ 78, 315, 90, 315, 78 ],
      [ 375, 540, 150, 594, 654 ],
      [ 829, 541, 180, 829, 541 ],
      [ 494, 534, -120, 708, 693 ],
      [ 92, 60, -180, 92, 60 ],
      [ 598, 33, 270, 33, 598 ],
      [ 783, 218, 210, 786, 579 ],
      [ 907, 549, 240, 927, 1059 ],
      [ 667, 83, 30, 618, 405 ],
      [ 960, 625, 60, 1020, 1143 ],
      [ 247, 683, 360, 247, 683 ],
      [ 883, 196, 120, 611, 861 ],
      [ 590, 50, -90, 50, 590 ],
      [ 305, 420, -330, 474, 516 ],
      [ 672, 489, -270, 489, 672 ],
      [ 924, 168, -240, 606, 883 ],
      [ 782, 755, -30, 1054, 1045 ],
      [ 542, 668, -240, 849, 802 ],
      [ 363, 400, -150, 513, 526 ],
      [ 912, 351, -300, 759, 964 ],
      [ 467, 536, -360, 467, 536 ],
      [ 442, 430, -180, 442, 430 ],
      [ 112, 104, -120, 145, 147 ],
      [ 503, 332, 270, 332, 503 ],
      [ 581, 68, 0, 581, 68 ],
      [ 739, 761, 180, 739, 761 ],
      [ 941, 579, 150, 1103, 970 ],
      [ 409, 217, 60, 392, 462 ],
      [ 607, 489, -300, 726, 769 ],
      [ 916, 404, 30, 995, 807 ],
      [ 11, 754, -180, 11, 754 ],
      [ 15, 317, 240, 281, 170 ],
      [ 280, 401, 330, 442, 487 ],
      [ 611, 10, -90, 10, 611 ],
      [ 478, 446, -30, 635, 625 ],
      [ 995, 232, -30, 976, 698 ],
      [ 713, 169, -210, 701, 501 ],
      [ 413, 475, 0, 413, 475 ],
      [ 1002, 162, -270, 162, 1002 ],
      [ 75, 81, 120, 107, 103 ],
      [ 640, 89, 90, 89, 640 ],
      [ 896, 100, -330, 825, 534 ],
      [ 699, 310, 60, 617, 760 ],
      [ 641, 768, 60, 985, 939 ],
      [ 1, 358, -270, 358, 1 ],
      [ 475, 336, -30, 578, 528 ],
      [ 938, 103, 120, 557, 862 ],
      [ 125, 412, -150, 313, 418 ],
      [ 623, 395, 0, 623, 395 ],
      [ 749, 450, 90, 450, 749 ],
      [ 390, 162, 180, 390, 162 ],
      [ 11, 66, 270, 66, 11 ],
      [ 55, 590, -180, 55, 590 ],
      [ 633, 302, -180, 633, 302 ],
      [ 199, 303, 180, 199, 303 ],
      [ 359, 659, -240, 750, 638 ],
      [ 827, 595, -150, 1013, 927 ],
      [ 496, 688, 240, 843, 772 ],
      [ 799, 387, -120, 733, 884 ],
      [ 293, 180, 330, 342, 302 ],
      [ 920, 472, -270, 472, 920 ],
      [ 640, 480, -210, 794, 734 ],
      [ 493, 521, 360, 493, 521 ],
      [ 759, 227, -90, 227, 759 ],
      [ 1023, 376, 210, 1072, 836 ],
      [ 741, 645, 90, 645, 741 ],
      [ 906, 497, -60, 881, 1032 ],
      [ 183, 101, -150, 208, 177 ],
      [ 9, 700, 240, 609, 356 ],
      [ 360, 152, -330, 387, 311 ],
      [ 287, 73, 120, 206, 283 ],
      [ 429, 553, -150, 647, 692 ],
      [ 659, 154, 330, 646, 462 ],
      [ 651, 723, -180, 651, 723 ],
      [ 21, 722, 180, 21, 722 ],
      [ 819, 509, 60, 850, 963 ],
      [ 447, 420, -210, 597, 586 ],
      [ 867, 557, -90, 557, 867 ],
      [ 155, 563, -180, 155, 563 ],
      [ 982, 64, -30, 881, 546 ],
      [ 1, 279, 60, 242, 139 ],
      [ 674, 600, -150, 882, 856 ],
      [ 962, 326, 0, 962, 326 ],
      [ 920, 46, -60, 498, 819 ],
      [ 157, 61, -90, 61, 157 ],
      [ 918, 675, -360, 918, 675 ],
      [ 492, 242, 60, 455, 547 ],
      [ 728, 124, -150, 691, 470 ],
      [ 74, 240, -360, 74, 240 ],
      [ 353, 208, -300, 356, 409 ],
      [ 833, 209, -30, 825, 597 ],
      [ 386, 714, -180, 386, 714 ],
      [ 676, 667, 120, 915, 917 ],
      [ 146, 588, 180, 146, 588 ],
      [ 580, 705, 240, 900, 854 ],
      [ 950, 626, 150, 1135, 1015 ],
      [ 975, 226, -360, 975, 226 ],
      [ 534, 4, -240, 269, 463 ],
      [ 835, 59, 0, 835, 59 ],
      [ 861, 323, 180, 861, 323 ],
      [ 972, 179, -360, 972, 179 ],
      [ 430, 469, 330, 606, 621 ],
      [ 688, 207, 240, 522, 698 ],
      [ 356, 316, 60, 451, 466 ],
      [ 85, 751, 0, 85, 751 ],
      [ 920, 695, -150, 1143, 1061 ],
      [ 606, 658, 90, 658, 606 ],
      [ 614, 290, 90, 290, 614 ],
      [ 786, 148, 150, 754, 519 ],
      [ 254, 26, -300, 149, 232 ],
      [ 1022, 755, -120, 1164, 1262 ],
      [ 20, 309, 330, 171, 278 ],
      [ 994, 57, -210, 888, 544 ],
      [ 807, 324, 90, 324, 807 ],
      [ 381, 387, 90, 387, 381 ],
      [ 894, 309, 0, 894, 309 ],
      [ 193, 763, -90, 763, 193 ],
      [ 817, 455, 180, 817, 455 ],
      [ 421, 276, 330, 501, 449 ],
      [ 102, 466, 360, 102, 466 ],
      [ 213, 464, 330, 415, 508 ],
      [ 562, 479, -120, 695, 725 ],
      [ 512, 456, -60, 649, 671 ],
      [ 737, 294, 270, 294, 737 ],
      [ 359, 579, -120, 679, 599 ],
      [ 982, 481, 180, 982, 481 ],
      [ 483, 626, 180, 483, 626 ],
      [ 861, 470, -120, 836, 979 ],
      [ 601, 17, 150, 528, 314 ],
      [ 538, 93, -150, 511, 349 ],
      [ 509, 252, 300, 471, 566 ],
      [ 462, 674, 30, 737, 814 ],
      [ 827, 289, -270, 289, 827 ],
      [ 242, 73, 0, 242, 73 ],
      [ 89, 343, -180, 89, 343 ],
      [ 451, 311, 300, 493, 545 ],
      [ 200, 673, 150, 509, 681 ],
      [ 965, 550, -150, 1109, 957 ],
      [ 288, 232, -120, 344, 364 ],
      [ 1003, 635, -30, 1185, 1051 ],
      [ 290, 249, 210, 375, 360 ],
      [ 196, 595, 120, 612, 465 ],
      [ 740, 447, -360, 740, 447 ],
      [ 868, 628, -270, 628, 868 ],
      [ 363, 694, 60, 782, 661 ],
      [ 629, 264, 360, 629, 264 ],
      [ 512, 413, 270, 413, 512 ],
      [ 204, 369, 60, 421, 360 ],
      [ 522, 584, 270, 584, 522 ],
      [ 862, 568, 150, 1030, 921 ],
      [ 302, 17, -360, 302, 17 ],
      [ 78, 164, 210, 148, 180 ],
      [ 751, 718, -90, 718, 751 ],
      [ 785, 601, -240, 912, 978 ],
      [ 916, 105, -300, 548, 845 ],
      [ 480, 577, -60, 738, 703 ],
      [ 457, 192, 360, 457, 192 ],
      [ 307, 345, -30, 437, 452 ],
      [ 888, 736, -210, 1137, 1079 ],
      [ 755, 615, 330, 960, 910 ],
      [ 443, 73, 360, 443, 73 ],
      [ 462, 131, -240, 343, 464 ],
      [ 250, 694, -270, 694, 250 ],
      [ 632, 514, 300, 759, 804 ],
      [ 827, 433, -360, 827, 433 ],
      [ 922, 25, 210, 810, 482 ],
      [ 345, 367, -360, 345, 367 ],
      [ 317, 597, 0, 317, 597 ],
      [ 780, 495, -30, 922, 819 ],
      [ 998, 293, -180, 998, 293 ],
      [ 948, 625, -120, 1014, 1132 ],
      [ 921, 203, 0, 921, 203 ],
      [ 87, 390, -60, 380, 270 ],
      [ 211, 96, -300, 188, 230 ],
      [ 111, 716, 120, 675, 453 ],
      [ 152, 639, 120, 628, 449 ],
      [ 961, 130, -270, 130, 961 ],
      [ 1001, 367, 330, 1049, 818 ],
      [ 477, 183, 60, 396, 504 ],
      [ 933, 163, 330, 889, 607 ],
      [ 160, 106, 210, 190, 171 ],
      [ 508, 28, -360, 508, 28 ],
      [ 6, 93, 30, 51, 83 ],
      [ 428, 251, 120, 430, 494 ],
      [ 511, 334, 60, 544, 609 ],
      [ 147, 448, -60, 460, 351 ],
      [ 840, 400, 60, 765, 927 ],
      [ 997, 383, -330, 1054, 830 ],
      [ 926, 740, -150, 1170, 1103 ],
      [ 490, 671, 30, 759, 825 ],
      [ 442, 23, 120, 240, 392 ],
      [ 203, 403, 150, 376, 449 ],
      [ 234, 407, 240, 468, 405 ],
      [ 785, 728, -270, 728, 785 ],
      [ 439, 343, 60, 516, 551 ],
      [ 23, 453, -270, 453, 23 ],
      [ 459, 314, 120, 501, 553 ],
      [ 42, 294, -240, 275, 182 ],
      [ 115, 220, -270, 220, 115 ],
      [ 431, 587, -360, 431, 587 ],
      [ 965, 150, -330, 910, 612 ],
      [ 615, 303, 60, 569, 683 ],
      [ 303, 478, -300, 565, 501 ],
      [ 114, 298, -330, 247, 314 ],
      [ 260, 627, 0, 260, 627 ],
      [ 851, 644, -330, 1058, 983 ],
      [ 1014, 219, 0, 1014, 219 ],
      [ 686, 251, 270, 251, 686 ],
      [ 868, 336, -270, 336, 868 ],
      [ 1011, 659, 300, 1075, 1204 ],
      [ 1024, 615, -300, 1044, 1193 ],
      [ 43, 307, 0, 43, 307 ],
      [ 623, 534, -270, 534, 623 ],
      [ 722, 619, 0, 722, 619 ],
      [ 767, 46, -120, 422, 686 ],
      [ 243, 684, -240, 713, 551 ],
      [ 323, 676, -30, 616, 746 ],
      [ 821, 423, 210, 922, 775 ],
      [ 687, 305, -210, 746, 606 ],
      [ 845, 295, -330, 878, 677 ],
      [ 762, 295, 270, 295, 762 ],
      [ 878, 327, -180, 878, 327 ],
      [ 373, 26, 360, 373, 26 ],
      [ 505, 567, 210, 720, 742 ],
      [ 10, 374, 240, 328, 194 ],
      [ 350, 556, 180, 350, 556 ],
      [ 524, 30, 90, 30, 524 ],
      [ 989, 645, -270, 645, 989 ],
      [ 765, 392, 30, 858, 721 ],
      [ 988, 257, 330, 983, 717 ],
      [ 41, 60, -150, 64, 71 ],
      [ 938, 718, 180, 938, 718 ],
      [ 196, 230, 210, 283, 296 ],
      [ 165, 608, 0, 165, 608 ],
      [ 959, 615, 360, 959, 615 ],
      [ 790, 109, 180, 790, 109 ],
      [ 508, 502, 210, 689, 688 ],
      [ 104, 475, 60, 462, 327 ],
      [ 262, 281, -300, 373, 366 ],
      [ 827, 253, -30, 842, 632 ],
      [ 760, 284, 30, 800, 625 ],
      [ 62, 219, -30, 162, 221 ],
      [ 822, 365, 180, 822, 365 ],
      [ 593, 488, 30, 757, 719 ],
      [ 95, 439, -120, 426, 301 ],
      [ 95, 263, 0, 95, 263 ],
      [ 835, 644, -270, 644, 835 ],
      [ 590, 722, 180, 590, 722 ],
      [ 180, 150, -270, 150, 180 ],
      [ 292, 2, -30, 252, 148 ],
      [ 746, 571, 240, 867, 931 ],
      [ 256, 617, -300, 661, 529 ],
      [ 723, 466, 60, 765, 859 ],
      [ 472, 142, -210, 479, 357 ],
      [ 7, 213, 210, 112, 186 ],
      [ 362, 284, -270, 284, 362 ],
      [ 862, 141, 360, 862, 141 ],
      [ 1006, 583, 300, 1006, 1162 ],
      [ 760, 718, -300, 1001, 1017 ],
      [ 887, 169, -300, 589, 852 ],
      [ 326, 728, 270, 728, 326 ],
      [ 144, 152, 120, 203, 199 ],
      [ 252, 694, -180, 252, 694 ],
      [ 846, 279, -60, 663, 871 ],
      [ 43, 284, 150, 179, 266 ],
      [ 848, 555, -300, 904, 1011 ],
      [ 980, 433, -180, 980, 433 ],
      [ 946, 419, -360, 946, 419 ],
      [ 849, 221, 330, 845, 615 ],
      [ 952, 117, -240, 576, 881 ],
      [ 32, 361, -270, 361, 32 ],
      [ 897, 469, -120, 853, 1010 ],
      [ 536, 657, -210, 792, 835 ],
      [ 795, 524, 60, 851, 950 ],
      [ 219, 556, 330, 466, 591 ],
      [ 950, 424, 120, 841, 1033 ],
      [ 11, 390, -180, 11, 390 ],
      [ 283, 331, 210, 410, 427 ],
      [ 308, 199, -330, 365, 325 ],
      [ 267, 145, -210, 303, 258 ],
      [ 450, 168, 150, 473, 368 ],
      [ 578, 73, -150, 536, 351 ],
      [ 909, 474, -240, 864, 1023 ],
      [ 115, 303, 270, 303, 115 ],
      [ 717, 467, 60, 762, 853 ],
      [ 673, 411, -300, 692, 787 ],
      [ 344, 419, 90, 419, 344 ],
      [ 592, 630, -330, 827, 841 ],
      [ 374, 93, -150, 369, 267 ],
      [ 450, 293, 0, 450, 293 ],
      [ 681, 630, 150, 904, 885 ],
      [ 369, 295, 240, 438, 466 ],
      [ 681, 208, -60, 519, 693 ],
      [ 846, 295, 240, 677, 879 ],
      [ 684, 64, -60, 395, 624 ],
      [ 238, 569, 360, 238, 569 ],
      [ 357, 58, 0, 357, 58 ],
      [ 994, 502, -120, 931, 1110 ],
      [ 25, 14, -30, 27, 24 ],
      [ 345, 351, 240, 475, 473 ],
      [ 857, 93, 120, 509, 787 ],
      [ 569, 370, -360, 569, 370 ],
      [ 378, 112, -150, 382, 285 ],
      [ 818, 747, 150, 1081, 1054 ],
      [ 637, 491, 210, 796, 742 ],
      [ 54, 669, 0, 54, 669 ],
      [ 17, 169, 60, 154, 98 ],
      [ 539, 146, -210, 539, 394 ],
      [ 876, 164, -180, 876, 164 ],
      [ 312, 422, 150, 481, 519 ],
      [ 139, 297, 240, 325, 268 ],
      [ 805, 723, -150, 1058, 1027 ],
      [ 839, 238, -30, 844, 625 ],
      [ 127, 83, -60, 134, 150 ],
      [ 832, 561, -330, 1000, 901 ],
      [ 597, 601, 330, 817, 818 ],
      [ 96, 613, -240, 578, 388 ],
      [ 696, 249, -120, 563, 726 ],
      [ 941, 137, 60, 589, 882 ],
      [ 180, 371, -300, 410, 340 ],
      [ 911, 475, 0, 911, 475 ],
      [ 743, 310, -30, 797, 639 ],
      [ 42, 171, 180, 42, 171 ],
      [ 515, 266, 270, 266, 515 ],
      [ 959, 122, 60, 585, 891 ],
      [ 9, 569, -90, 569, 9 ],
      [ 975, 640, -240, 1041, 1163 ],
      [ 117, 394, 0, 117, 394 ],
      [ 477, 331, 120, 525, 577 ],
      [ 31, 465, -270, 465, 31 ],
      [ 148, 381, 210, 318, 403 ],
      [ 660, 169, -240, 475, 654 ],
      [ 87, 201, -90, 201, 87 ],
      [ 855, 587, 180, 855, 587 ],
      [ 710, 538, 300, 819, 883 ],
      [ 288, 544, 90, 544, 288 ],
      [ 665, 507, -30, 828, 771 ],
      [ 836, 594, 360, 836, 594 ],
      [ 314, 183, -60, 313, 362 ],
      [ 997, 206, -360, 997, 206 ],
      [ 126, 316, 0, 126, 316 ],
      [ 912, 43, 180, 912, 43 ],
      [ 62, 108, 360, 62, 108 ],
      [ 464, 749, 180, 464, 749 ],
      [ 197, 513, -30, 426, 542 ],
      [ 91, 728, -240, 675, 441 ],
      [ 743, 459, 240, 768, 872 ],
      [ 201, 318, 60, 375, 333 ],
      [ 183, 553, 240, 569, 434 ],
      [ 605, 532, -300, 763, 789 ],
      [ 623, 627, 0, 623, 627 ],
      [ 123, 542, 60, 530, 377 ],
      [ 873, 589, 150, 1050, 945 ],
      [ 866, 169, 150, 833, 577 ],
      [ 616, 316, -120, 581, 690 ],
      [ 59, 384, -150, 242, 361 ],
      [ 227, 173, 300, 262, 282 ],
      [ 45, 323, -150, 199, 301 ],
      [ 655, 460, -360, 655, 460 ],
      [ 484, 145, 150, 491, 366 ],
      [ 587, 611, 30, 813, 822 ],
      [ 87, 704, -210, 427, 652 ],
      [ 668, 590, -360, 668, 590 ],
      [ 368, 471, -210, 553, 590 ],
      [ 41, 165, 90, 165, 41 ],
      [ 398, 209, -270, 209, 398 ],
      [ 702, 379, -120, 678, 796 ],
      [ 593, 412, 210, 718, 652 ],
      [ 907, 135, -90, 135, 907 ],
      [ 934, 498, 60, 897, 1057 ],
      [ 646, 170, -120, 469, 643 ],
      [ 168, 235, -180, 168, 235 ],
      [ 413, 736, -330, 725, 843 ],
      [ 419, 243, 120, 419, 482 ],
      [ 648, 274, 270, 274, 648 ],
      [ 274, 572, -270, 572, 274 ],
      [ 411, 330, 90, 330, 411 ],
      [ 778, 6, -240, 393, 675 ],
      [ 551, 686, -120, 868, 819 ],
      [ 936, 618, 360, 936, 618 ],
      [ 490, 334, -210, 591, 532 ],
      [ 858, 460, 30, 973, 826 ],
      [ 115, 1, -30, 99, 58 ],
      [ 159, 315, 210, 294, 351 ],
      [ 830, 33, -270, 33, 830 ],
      [ 696, 238, 300, 552, 721 ],
      [ 841, 546, -120, 892, 1000 ],
      [ 466, 362, -120, 546, 583 ],
      [ 641, 7, -210, 558, 325 ],
      [ 995, 708, -360, 995, 708 ],
      [ 989, 307, -30, 1009, 760 ],
      [ 185, 182, -330, 251, 250 ],
      [ 717, 268, -330, 754, 590 ],
      [ 183, 387, -30, 351, 426 ],
      [ 995, 241, 0, 995, 241 ],
      [ 124, 763, 240, 722, 488 ],
      [ 15, 625, 30, 324, 548 ],
      [ 345, 207, -360, 345, 207 ],
      [ 683, 687, -360, 683, 687 ],
      [ 910, 665, 330, 1120, 1031 ],
      [ 938, 638, -120, 1021, 1130 ],
      [ 385, 8, 60, 199, 337 ],
      [ 445, 546, 300, 694, 658 ],
      [ 506, 682, -60, 842, 779 ],
      [ 986, 660, 180, 986, 660 ],
      [ 485, 753, 180, 485, 753 ],
      [ 302, 764, 30, 643, 812 ],
      [ 846, 254, 240, 642, 858 ],
      [ 847, 765, 150, 1115, 1085 ],
      [ 857, 680, 60, 1017, 1082 ],
      [ 786, 615, -90, 615, 786 ],
      [ 100, 135, -60, 165, 153 ],
      [ 681, 469, -300, 746, 823 ],
      [ 595, 80, -360, 595, 80 ],
      [ 1023, 51, 270, 51, 1023 ],
      [ 749, 415, 270, 415, 749 ],
      [ 467, 641, 240, 787, 724 ],
      [ 1017, 507, -240, 947, 1132 ],
      [ 819, 374, -240, 733, 895 ],
      [ 511, 248, -330, 566, 470 ],
      [ 66, 69, 270, 69, 66 ],
      [ 469, 144, -360, 469, 144 ],
      [ 883, 655, 90, 655, 883 ],
      [ 1000, 332, 180, 1000, 332 ],
      [ 1003, 331, 210, 1033, 787 ],
      [ 861, 125, -120, 537, 807 ],
      [ 697, 474, -240, 758, 839 ],
      [ 528, 469, 240, 669, 691 ],
      [ 696, 315, -150, 759, 620 ],
      [ 860, 698, 120, 1033, 1092 ],
      [ 906, 747, 180, 906, 747 ],
      [ 747, 330, 300, 658, 811 ],
      [ 768, 224, 210, 776, 577 ],
      [ 389, 206, -210, 439, 371 ],
      [ 116, 190, 90, 190, 116 ],
      [ 935, 67, 210, 842, 524 ],
      [ 259, 590, -60, 639, 519 ],
      [ 437, 217, -360, 437, 217 ],
      [ 265, 739, -60, 771, 598 ],
      [ 592, 616, -120, 828, 819 ],
      [ 225, 527, -150, 457, 567 ],
      [ 974, 319, 180, 974, 319 ],
      [ 891, 126, -360, 891, 126 ],
      [ 656, 417, -150, 776, 688 ],
      [ 864, 504, 30, 1000, 867 ],
      [ 489, 437, 90, 437, 489 ],
      [ 279, 630, -90, 630, 279 ],
      [ 701, 190, 150, 702, 514 ],
      [ 723, 388, 120, 697, 819 ],
      [ 140, 64, -30, 152, 125 ],
      [ 486, 233, -270, 233, 486 ],
      [ 803, 195, 30, 792, 570 ],
      [ 536, 95, 150, 511, 348 ],
      [ 567, 587, -180, 567, 587 ],
      [ 873, 466, 300, 839, 989 ],
      [ 408, 65, -30, 385, 260 ],
      [ 31, 274, -150, 162, 251 ],
      [ 440, 31, 30, 396, 246 ],
      [ 114, 573, -330, 384, 552 ],
      [ 772, 678, -270, 678, 772 ],
      [ 203, 274, -60, 337, 312 ],
      [ 336, 108, 150, 344, 260 ],
      [ 894, 510, 240, 888, 1028 ],
      [ 594, 167, 60, 441, 597 ],
      [ 850, 54, -240, 471, 762 ],
      [ 753, 359, -180, 753, 359 ],
      [ 214, 382, 90, 382, 214 ],
      [ 514, 712, 120, 873, 800 ],
      [ 43, 29, -60, 45, 51 ],
      [ 89, 608, -150, 380, 570 ],
      [ 219, 760, 90, 760, 219 ],
      [ 659, 244, 210, 691, 539 ],
      [ 334, 146, -30, 361, 293 ],
      [ 134, 591, 120, 578, 410 ],
      [ 717, 460, 180, 717, 460 ],
      [ 896, 257, -240, 670, 902 ],
      [ 137, 417, 150, 326, 428 ],
      [ 767, 34, 90, 34, 767 ],
      [ 414, 66, 120, 263, 390 ],
      [ 825, 132, -30, 779, 526 ],
      [ 91, 296, -60, 300, 226 ],
      [ 745, 22, 180, 745, 22 ],
      [ 520, 272, 330, 585, 496 ],
      [ 974, 372, 150, 1029, 807 ],
      [ 626, 141, -150, 612, 434 ],
      [ 350, 45, 120, 213, 324 ],
      [ 552, 147, -240, 402, 550 ],
      [ 239, 722, -210, 567, 743 ],
      [ 873, 265, -150, 888, 664 ],
      [ 521, 115, 0, 521, 115 ],
      [ 979, 183, 270, 183, 979 ],
      [ 401, 741, 300, 841, 717 ],
      [ 147, 363, -180, 147, 363 ],
      [ 86, 325, 180, 86, 325 ],
      [ 813, 26, 300, 428, 717 ],
      [ 105, 288, 360, 105, 288 ],
      [ 763, 702, -240, 989, 1010 ],
      [ 952, 112, -300, 572, 880 ],
      [ 163, 767, -30, 524, 745 ],
      [ 485, 389, 90, 389, 485 ],
      [ 983, 355, 240, 797, 1028 ],
      [ 852, 655, 240, 992, 1064 ],
      [ 749, 765, -180, 749, 765 ],
      [ 1007, 61, 150, 902, 555 ],
      [ 737, 671, 180, 737, 671 ],
      [ 654, 749, -300, 975, 940 ],
      [ 631, 553, -360, 631, 553 ],
      [ 803, 498, -240, 832, 943 ],
      [ 914, 620, -240, 993, 1100 ],
      [ 353, 215, 120, 362, 411 ],
      [ 968, 185, -270, 185, 968 ],
      [ 803, 55, 360, 803, 55 ],
      [ 601, 616, 360, 601, 616 ],
      [ 882, 604, -330, 1065, 963 ],
      [ 571, 388, 330, 687, 621 ],
      [ 295, 111, 300, 242, 310 ],
      [ 402, 583, 90, 583, 402 ],
      [ 420, 417, -180, 420, 417 ],
      [ 588, 334, -330, 676, 582 ],
      [ 753, 598, -120, 893, 950 ],
      [ 412, 556, 60, 687, 634 ],
      [ 545, 390, 120, 610, 665 ],
      [ 526, 72, -30, 490, 325 ],
      [ 502, 733, -180, 502, 733 ],
      [ 560, 393, 210, 680, 619 ],
      [ 487, 613, 300, 773, 727 ],
      [ 388, 146, 120, 319, 408 ],
      [ 1, 461, -210, 230, 398 ],
      [ 278, 133, 90, 133, 278 ],
      [ 357, 698, -60, 781, 658 ],
      [ 723, 238, -270, 238, 723 ],
      [ 310, 646, 120, 713, 590 ],
      [ 994, 272, 180, 994, 272 ],
      [ 451, 649, 150, 714, 786 ],
      [ 713, 300, -210, 767, 615 ],
      [ 467, 665, -360, 467, 665 ],
      [ 369, 187, -240, 346, 411 ],
      [ 24, 188, 210, 113, 174 ],
      [ 234, 396, 360, 234, 396 ],
      [ 889, 663, 270, 663, 889 ],
      [ 293, 437, -240, 524, 470 ],
      [ 438, 669, -360, 438, 669 ],
      [ 78, 646, -90, 646, 78 ],
      [ 836, 216, -210, 831, 603 ],
      [ 524, 750, 90, 750, 524 ],
      [ 733, 332, -30, 799, 654 ],
      [ 736, 608, 180, 736, 608 ],
      [ 933, 626, 360, 933, 626 ],
      [ 721, 33, -30, 640, 389 ],
      [ 686, 699, -120, 947, 943 ],
      [ 556, 151, 300, 407, 556 ],
      [ 743, 479, 210, 882, 785 ],
      [ 767, 537, 90, 537, 767 ],
      [ 130, 396, 300, 406, 310 ],
      [ 357, 20, 270, 20, 357 ],
      [ 984, 570, -150, 1136, 985 ],
      [ 449, 353, -300, 530, 564 ],
      [ 164, 285, 300, 327, 284 ],
      [ 127, 57, 330, 137, 112 ],
      [ 618, 571, 270, 571, 618 ],
      [ 968, 220, -330, 948, 674 ],
      [ 875, 9, 150, 761, 444 ],
      [ 649, 584, -90, 584, 649 ],
      [ 258, 681, 300, 717, 563 ],
      [ 155, 181, 330, 224, 234 ],
      [ 27, 151, 120, 144, 97 ],
      [ 340, 487, -270, 487, 340 ],
      [ 441, 610, 0, 441, 610 ],
      [ 350, 705, 60, 785, 655 ],
      [ 303, 400, -150, 461, 496 ],
      [ 164, 357, 60, 390, 320 ],
      [ 225, 245, 90, 245, 225 ],
      [ 924, 731, -90, 731, 924 ],
      [ 291, 156, -180, 291, 156 ],
      [ 184, 272, 0, 184, 272 ],
      [ 103, 292, 150, 235, 303 ],
      [ 782, 547, -150, 950, 864 ],
      [ 906, 109, -300, 546, 838 ],
      [ 387, 371, -360, 387, 371 ],
      [ 966, 598, 30, 1135, 1000 ],
      [ 259, 721, -360, 259, 721 ],
      [ 868, 121, -120, 538, 811 ],
      [ 464, 46, -180, 464, 46 ],
      [ 838, 264, 0, 838, 264 ],
      [ 72, 402, 240, 383, 262 ],
      [ 570, 479, -210, 732, 698 ],
      [ 244, 297, 330, 359, 379 ],
      [ 567, 208, -330, 595, 463 ],
      [ 679, 497, 30, 836, 769 ],
      [ 706, 454, -120, 745, 837 ],
      [ 242, 648, -180, 242, 648 ],
      [ 253, 531, -60, 585, 484 ],
      [ 589, 111, 0, 589, 111 ],
      [ 876, 739, 240, 1077, 1127 ],
      [ 346, 25, -120, 194, 311 ],
      [ 178, 452, 360, 178, 452 ],
      [ 399, 634, -120, 747, 661 ],
      [ 324, 292, 90, 292, 324 ],
      [ 394, 33, -180, 394, 33 ],
      [ 959, 562, 270, 562, 959 ],
      [ 272, 743, 150, 606, 777 ],
      [ 547, 164, -90, 164, 547 ],
      [ 982, 606, 30, 1153, 1015 ],
      [ 455, 494, 0, 455, 494 ],
      [ 279, 753, 30, 617, 791 ],
      [ 638, 118, -270, 118, 638 ],
      [ 610, 417, 330, 736, 666 ],
      [ 977, 660, -120, 1059, 1175 ],
      [ 597, 187, -90, 187, 597 ],
      [ 880, 138, -300, 559, 831 ],
      [ 743, 342, -330, 814, 667 ],
      [ 445, 752, -150, 760, 872 ],
      [ 854, 720, -330, 1099, 1050 ],
      [ 387, 292, 150, 481, 445 ],
      [ 905, 501, 150, 1033, 885 ],
      [ 431, 211, 270, 211, 431 ],
      [ 558, 668, -90, 668, 558 ],
      [ 507, 633, -180, 507, 633 ],
      [ 843, 312, 0, 843, 312 ],
      [ 199, 203, 120, 275, 272 ],
      [ 303, 760, -300, 809, 642 ],
      [ 374, 325, -300, 467, 485 ],
      [ 662, 198, -360, 662, 198 ],
      [ 734, 488, -60, 788, 879 ],
      [ 440, 398, -330, 580, 564 ],
      [ 126, 721, -120, 686, 469 ],
      [ 993, 371, -210, 1044, 816 ],
      [ 367, 750, -360, 367, 750 ],
      [ 621, 615, -60, 842, 844 ],
      [ 100, 764, 150, 468, 710 ],
      [ 787, 223, 150, 792, 585 ],
      [ 906, 503, -300, 888, 1035 ],
      [ 1001, 231, -90, 231, 1001 ],
      [ 8, 13, -360, 8, 13 ],
      [ 428, 343, 0, 428, 343 ],
      [ 491, 436, -30, 642, 623 ],
      [ 835, 412, 330, 928, 774 ],
      [ 20, 687, 300, 603, 360 ],
      [ 49, 385, 150, 234, 356 ],
      [ 493, 460, 150, 656, 643 ],
      [ 183, 282, 360, 183, 282 ],
      [ 886, 193, 120, 609, 862 ],
      [ 999, 175, 330, 952, 651 ],
      [ 350, 181, 360, 350, 181 ],
      [ 352, 502, -60, 609, 555 ],
      [ 883, 102, 360, 883, 102 ],
      [ 327, 728, 0, 327, 728 ],
      [ 264, 743, -60, 773, 599 ],
      [ 177, 11, 300, 97, 158 ],
      [ 909, 380, 0, 909, 380 ],
      [ 610, 517, 270, 517, 610 ],
      [ 608, 413, -270, 413, 608 ],
      [ 251, 394, -120, 465, 413 ],
      [ 202, 656, 60, 668, 502 ],
      [ 170, 152, -180, 170, 152 ],
      [ 594, 46, -90, 46, 594 ],
      [ 575, 290, -120, 537, 641 ],
      [ 82, 488, -150, 314, 463 ],
      [ 504, 620, -150, 745, 788 ],
      [ 415, 533, 210, 625, 668 ],
      [ 939, 222, -30, 923, 661 ],
      [ 813, 678, -360, 813, 678 ],
      [ 944, 98, 0, 944, 98 ],
      [ 284, 249, -90, 249, 284 ],
      [ 888, 377, 60, 769, 957 ],
      [ 113, 54, 90, 54, 113 ],
      [ 498, 485, -360, 498, 485 ],
      [ 828, 546, 90, 546, 828 ],
      [ 99, 156, -30, 162, 184 ],
      [ 409, 467, -270, 467, 409 ],
      [ 200, 403, -60, 447, 374 ],
      [ 679, 244, -150, 709, 549 ],
      [ 685, 184, -60, 500, 685 ],
      [ 191, 396, 210, 362, 437 ],
      [ 572, 294, -180, 572, 294 ],
      [ 130, 378, -120, 391, 300 ],
      [ 773, 751, 330, 1044, 1036 ],
      [ 778, 604, 120, 911, 974 ],
      [ 409, 678, 270, 678, 409 ],
      [ 867, 216, -30, 857, 620 ],
      [ 1010, 366, 360, 1010, 366 ],
      [ 425, 107, -150, 421, 304 ],
      [ 743, 620, 30, 953, 908 ],
      [ 167, 763, -330, 525, 744 ],
      [ 946, 424, -30, 1030, 840 ],
      [ 179, 521, 330, 415, 540 ],
      [ 507, 332, 300, 540, 605 ],
      [ 457, 147, 150, 468, 354 ],
      [ 110, 453, 60, 446, 321 ],
      [ 994, 335, 270, 335, 994 ],
      [ 453, 324, -120, 506, 553 ],
      [ 448, 642, -30, 707, 780 ],
      [ 768, 431, -150, 880, 756 ],
      [ 307, 556, -150, 542, 634 ],
      [ 374, 497, 240, 616, 571 ],
      [ 831, 631, 0, 831, 631 ],
      [ 768, 243, 330, 786, 594 ],
      [ 683, 585, -270, 585, 683 ],
      [ 377, 667, 150, 659, 765 ],
      [ 976, 645, -270, 645, 976 ],
      [ 838, 216, 60, 605, 833 ],
      [ 163, 552, -60, 558, 417 ],
      [ 654, 359, 360, 654, 359 ],
      [ 947, 589, 150, 1114, 982 ],
      [ 210, 101, -120, 191, 231 ],
      [ 37, 724, -240, 645, 393 ],
      [ 557, 532, -30, 747, 739 ],
      [ 495, 276, -180, 495, 276 ],
      [ 628, 558, -300, 796, 822 ],
      [ 341, 522, 330, 555, 622 ],
      [ 500, 382, -210, 624, 579 ],
      [ 66, 503, 330, 308, 469 ],
      [ 488, 225, -60, 437, 534 ],
      [ 460, 168, -240, 374, 481 ],
      [ 148, 325, -150, 290, 354 ],
      [ 499, 352, -180, 499, 352 ],
      [ 667, 2, 330, 577, 335 ],
      [ 133, 373, -150, 301, 388 ],
      [ 379, 76, -360, 379, 76 ],
      [ 490, 332, 150, 590, 531 ],
      [ 435, 706, -210, 729, 827 ],
      [ 668, 755, 270, 755, 668 ],
      [ 612, 353, -270, 353, 612 ],
      [ 10, 697, -90, 697, 10 ],
      [ 193, 40, 210, 186, 130 ],
      [ 509, 414, -180, 509, 414 ],
      [ 766, 146, -180, 766, 146 ],
      [ 707, 245, 180, 707, 245 ],
      [ 1016, 529, 240, 965, 1143 ],
      [ 20, 128, -180, 20, 128 ],
      [ 743, 454, -240, 764, 869 ],
      [ 950, 187, -240, 636, 914 ],
      [ 804, 646, 90, 646, 804 ],
      [ 957, 653, 30, 1154, 1044 ],
      [ 309, 30, 60, 180, 282 ],
      [ 94, 411, -270, 411, 94 ],
      [ 347, 217, -150, 408, 360 ],
      [ 603, 747, 90, 747, 603 ],
      [ 331, 741, -150, 656, 806 ],
      [ 160, 755, 0, 160, 755 ],
      [ 431, 544, -330, 645, 686 ],
      [ 618, 488, -150, 778, 731 ],
      [ 788, 323, -270, 323, 788 ],
      [ 404, 272, -360, 404, 272 ],
      [ 948, 504, -330, 1072, 909 ],
      [ 522, 574, 60, 757, 739 ],
      [ 648, 66, 270, 66, 648 ],
      [ 967, 518, 240, 931, 1095 ],
      [ 589, 766, 210, 892, 956 ],
      [ 915, 118, 210, 850, 558 ],
      [ 387, 441, 0, 387, 441 ],
      [ 464, 136, -270, 136, 464 ],
      [ 790, 727, 30, 1047, 1024 ],
      [ 952, 262, 300, 701, 955 ],
      [ 960, 204, 30, 933, 656 ],
      [ 1008, 596, -150, 1169, 1019 ],
      [ 579, 313, -90, 313, 579 ],
      [ 438, 270, -330, 514, 452 ],
      [ 270, 711, -330, 588, 750 ],
      [ 64, 629, -210, 369, 575 ],
      [ 865, 151, 210, 824, 562 ],
      [ 343, 499, 330, 546, 603 ],
      [ 471, 323, 300, 514, 568 ],
      [ 5, 268, -180, 5, 268 ],
      [ 926, 220, 0, 926, 220 ],
      [ 470, 208, -150, 510, 414 ],
      [ 778, 641, 150, 993, 942 ],
      [ 150, 202, -330, 230, 249 ],
      [ 212, 404, 360, 212, 404 ],
      [ 270, 451, 210, 458, 525 ],
      [ 485, 332, 360, 485, 332 ],
      [ 262, 588, 90, 588, 262 ],
      [ 220, 173, -330, 276, 259 ],
      [ 124, 176, -60, 212, 195 ],
      [ 375, 102, 150, 375, 274 ],
      [ 898, 454, 330, 1003, 842 ],
      [ 166, 269, 210, 277, 315 ],
      [ 876, 381, -330, 948, 767 ],
      [ 921, 540, 60, 928, 1067 ],
      [ 891, 742, -240, 1088, 1141 ],
      [ 674, 337, 60, 628, 751 ],
      [ 659, 533, -90, 533, 659 ],
      [ 284, 698, 60, 745, 594 ],
      [ 339, 22, 240, 187, 303 ],
      [ 750, 303, 330, 800, 637 ],
      [ 431, 208, 30, 477, 395 ],
      [ 388, 332, 300, 480, 502 ],
      [ 176, 221, -90, 221, 176 ],
      [ 231, 144, -300, 240, 272 ],
      [ 824, 43, -330, 734, 448 ],
      [ 995, 548, 0, 995, 548 ],
      [ 582, 274, -240, 527, 640 ],
      [ 953, 486, -300, 897, 1068 ],
      [ 529, 740, -300, 905, 828 ],
      [ 340, 535, 0, 340, 535 ],
      [ 285, 90, 210, 290, 219 ],
      [ 835, 380, -210, 913, 745 ],
      [ 719, 512, 0, 719, 512 ],
      [ 86, 685, 150, 416, 634 ],
      [ 172, 535, 180, 172, 535 ],
      [ 222, 512, -30, 447, 554 ],
      [ 721, 180, 240, 515, 713 ],
      [ 906, 127, -30, 847, 563 ],
      [ 997, 523, -60, 950, 1124 ],
      [ 94, 10, -300, 55, 86 ],
      [ 617, 223, -210, 645, 500 ],
      [ 393, 81, 150, 380, 265 ],
      [ 612, 620, -90, 620, 612 ],
      [ 100, 684, -180, 100, 684 ],
      [ 804, 44, 360, 804, 44 ],
      [ 540, 211, 90, 211, 540 ],
      [ 1007, 751, 270, 751, 1007 ],
      [ 815, 663, -360, 815, 663 ],
      [ 247, 642, 150, 534, 678 ],
      [ 665, 712, 180, 665, 712 ],
      [ 771, 407, -360, 771, 407 ],
      [ 995, 701, -300, 1104, 1211 ],
      [ 702, 392, 330, 802, 690 ],
      [ 67, 467, 240, 436, 291 ],
      [ 343, 301, 270, 301, 343 ],
      [ 308, 705, -270, 705, 308 ],
      [ 955, 692, -270, 692, 955 ],
      [ 817, 535, 360, 817, 535 ],
      [ 847, 720, 240, 1046, 1092 ],
      [ 551, 450, 180, 551, 450 ],
      [ 267, 259, -180, 267, 259 ],
      [ 279, 237, -240, 344, 358 ],
      [ 400, 764, 150, 728, 860 ],
      [ 322, 45, -150, 300, 199 ],
      [ 131, 302, 150, 264, 326 ],
      [ 1023, 533, 90, 533, 1023 ],
      [ 147, 480, 0, 147, 480 ],
      [ 271, 324, -210, 396, 415 ],
      [ 255, 192, -270, 192, 255 ],
      [ 71, 604, 180, 71, 604 ],
      [ 835, 36, -330, 741, 448 ],
      [ 93, 245, -90, 245, 93 ],
      [ 286, 545, -90, 545, 286 ],
      [ 1000, 17, -60, 513, 874 ],
      [ 288, 115, 240, 243, 306 ],
      [ 1000, 114, 0, 1000, 114 ],
      [ 604, 223, -270, 223, 604 ],
      [ 106, 427, 60, 422, 304 ],
      [ 322, 618, 240, 695, 586 ],
      [ 472, 670, 60, 815, 743 ],
      [ 197, 528, 120, 555, 433 ],
      [ 267, 597, 360, 267, 597 ],
      [ 653, 43, 120, 363, 585 ],
      [ 9, 25, 150, 19, 25 ],
      [ 448, 240, 270, 240, 448 ],
      [ 265, 222, 360, 265, 222 ],
      [ 792, 674, -180, 792, 674 ],
      [ 934, 753, 240, 1118, 1184 ],
      [ 484, 226, 90, 226, 484 ],
      [ 289, 580, 0, 289, 580 ],
      [ 900, 727, -240, 1079, 1141 ],
      [ 533, 159, 360, 533, 159 ],
      [ 493, 649, -360, 493, 649 ],
      [ 155, 655, -330, 461, 644 ],
      [ 863, 223, -90, 223, 863 ],
      [ 700, 421, 120, 714, 815 ],
      [ 688, 247, 30, 718, 557 ],
      [ 973, 179, 0, 973, 179 ],
      [ 803, 542, 240, 869, 965 ],
      [ 431, 758, 60, 871, 752 ],
      [ 952, 665, 0, 952, 665 ],
      [ 128, 297, 150, 258, 319 ],
      [ 100, 666, 60, 626, 419 ],
      [ 129, 14, -60, 75, 118 ],
      [ 190, 661, 150, 494, 665 ],
      [ 570, 417, -180, 570, 417 ],
      [ 221, 165, 60, 253, 273 ],
      [ 769, 126, 240, 492, 727 ],
      [ 274, 728, 0, 274, 728 ],
      [ 80, 282, 0, 80, 282 ],
      [ 658, 228, -90, 228, 658 ],
      [ 157, 324, 180, 157, 324 ],
      [ 892, 399, 90, 399, 892 ],
      [ 442, 495, 90, 495, 442 ],
      [ 866, 637, 0, 866, 637 ],
      [ 561, 296, -360, 561, 296 ],
      [ 818, 461, -180, 818, 461 ],
      [ 379, 270, -60, 422, 463 ],
      [ 178, 475, -120, 499, 391 ],
      [ 699, 535, 180, 699, 535 ],
      [ 182, 260, -360, 182, 260 ],
      [ 551, 378, -30, 665, 602 ],
      [ 291, 278, 360, 291, 278 ],
      [ 919, 609, 90, 609, 919 ],
      [ 541, 490, -60, 693, 713 ],
      [ 19, 143, 240, 132, 87 ],
      [ 53, 756, -60, 680, 423 ],
      [ 305, 272, 210, 399, 387 ],
      [ 725, 405, -90, 405, 725 ],
      [ 66, 161, -300, 171, 137 ],
      [ 807, 297, -60, 659, 846 ],
      [ 827, 710, 300, 1027, 1071 ],
      [ 255, 160, -180, 255, 160 ],
      [ 214, 81, -330, 225, 176 ],
      [ 833, 487, 150, 964, 837 ],
      [ 253, 501, 300, 559, 469 ],
      [ 66, 540, 270, 540, 66 ],
      [ 494, 1, -210, 427, 246 ],
      [ 257, 544, 180, 257, 544 ],
      [ 652, 593, 360, 652, 593 ],
      [ 737, 430, -120, 739, 852 ],
      [ 143, 282, -180, 143, 282 ],
      [ 53, 472, -30, 280, 435 ],
      [ 333, 633, 60, 714, 604 ],
      [ 401, 489, -210, 591, 622 ],
      [ 98, 678, 270, 678, 98 ],
      [ 1009, 727, 60, 1134, 1236 ],
      [ 879, 329, 60, 724, 925 ],
      [ 87, 522, -180, 87, 522 ],
      [ 858, 243, -330, 864, 638 ],
      [ 819, 27, 90, 27, 819 ],
      [ 151, 134, 360, 151, 134 ],
      [ 429, 174, 60, 365, 458 ],
      [ 921, 423, -60, 825, 1008 ],
      [ 496, 724, -330, 791, 874 ],
      [ 719, 29, 330, 636, 384 ],
      [ 585, 17, 270, 17, 585 ],
      [ 147, 676, -150, 464, 657 ],
      [ 733, 740, 360, 733, 740 ],
      [ 1020, 615, -150, 1190, 1042 ],
      [ 48, 461, -120, 422, 271 ],
      [ 672, 574, 0, 672, 574 ],
      [ 647, 128, 180, 647, 128 ],
      [ 227, 50, -150, 220, 155 ],
      [ 651, 421, -210, 773, 689 ],
      [ 701, 192, -180, 701, 192 ],
      [ 631, 631, -150, 861, 860 ],
      [ 506, 741, -240, 894, 807 ],
      [ 820, 522, -270, 522, 820 ],
      [ 805, 519, 330, 956, 851 ],
      [ 89, 554, 30, 354, 524 ],
      [ 432, 290, -180, 432, 290 ],
      [ 953, 7, -60, 481, 828 ],
      [ 680, 177, 0, 680, 177 ],
      [ 972, 665, -330, 1173, 1061 ],
      [ 179, 422, -150, 365, 453 ],
      [ 448, 127, -270, 127, 448 ],
      [ 738, 507, -300, 807, 892 ],
      [ 905, 354, 210, 959, 758 ],
      [ 10, 189, -30, 102, 169 ],
      [ 959, 255, -240, 700, 956 ],
      [ 474, 578, 30, 699, 737 ],
      [ 755, 524, 30, 915, 831 ],
      [ 181, 266, 210, 288, 319 ],
      [ 698, 227, 120, 545, 716 ],
      [ 751, 361, -240, 688, 829 ],
      [ 37, 696, -120, 620, 379 ],
      [ 163, 482, -360, 163, 482 ],
      [ 252, 392, -30, 413, 465 ],
      [ 13, 399, 150, 210, 351 ],
      [ 465, 350, -330, 577, 535 ],
      [ 631, 706, 240, 925, 898 ],
      [ 159, 503, 0, 159, 503 ],
      [ 719, 639, 240, 911, 941 ],
      [ 491, 394, -270, 394, 491 ],
      [ 123, 189, 60, 225, 200 ],
      [ 318, 217, 0, 318, 217 ],
      [ 682, 338, -270, 338, 682 ],
      [ 697, 527, 90, 527, 697 ],
      [ 159, 536, -270, 536, 159 ],
      [ 896, 116, 90, 116, 896 ],
      [ 949, 589, 30, 1115, 984 ],
      [ 748, 708, -240, 986, 1000 ],
      [ 222, 479, -360, 222, 479 ],
      [ 50, 78, 30, 82, 92 ],
      [ 180, 170, 210, 239, 236 ],
      [ 790, 409, -330, 888, 748 ],
      [ 280, 152, 0, 280, 152 ],
      [ 411, 675, -240, 790, 691 ],
      [ 41, 26, 270, 26, 41 ],
      [ 202, 698, -360, 202, 698 ],
      [ 829, 641, 210, 1037, 968 ],
      [ 348, 433, 150, 517, 547 ],
      [ 538, 599, -120, 787, 764 ],
      [ 555, 636, -60, 827, 798 ],
      [ 73, 2, 120, 38, 63 ],
      [ 867, 594, -240, 947, 1046 ],
      [ 925, 36, -90, 36, 925 ],
      [ 409, 344, -180, 409, 344 ],
      [ 595, 374, -150, 701, 620 ],
      [ 470, 525, -210, 669, 688 ],
      [ 499, 378, -330, 621, 576 ],
      [ 289, 638, 90, 638, 289 ],
      [ 1021, 273, -90, 273, 1021 ],
      [ 343, 689, -210, 641, 767 ],
      [ 793, 743, -210, 1057, 1038 ],
      [ 445, 625, 360, 445, 625 ],
      [ 641, 550, -360, 641, 550 ],
      [ 16, 89, 0, 16, 89 ],
      [ 289, 534, 360, 289, 534 ],
      [ 596, 119, 120, 400, 574 ],
      [ 808, 493, -270, 493, 808 ],
      [ 850, 709, 360, 850, 709 ],
      [ 445, 707, -120, 833, 738 ],
      [ 838, 195, -210, 822, 586 ],
      [ 30, 22, -240, 33, 35 ],
      [ 249, 355, -360, 249, 355 ],
      [ 235, 68, 150, 237, 175 ],
      [ 269, 79, 240, 201, 271 ],
      [ 761, 295, 30, 806, 635 ],
      [ 726, 742, 150, 999, 1004 ],
      [ 367, 580, -120, 684, 606 ],
      [ 512, 449, -180, 512, 449 ],
      [ 483, 15, -240, 254, 424 ],
      [ 817, 643, -60, 964, 1028 ],
      [ 22, 665, -30, 351, 587 ],
      [ 217, 84, 300, 180, 229 ],
      [ 221, 260, 360, 221, 260 ],
      [ 958, 461, -300, 877, 1059 ],
      [ 788, 263, 0, 788, 263 ],
      [ 307, 40, -30, 284, 188 ],
      [ 1007, 315, -210, 1029, 775 ],
      [ 316, 698, 210, 621, 761 ],
      [ 569, 292, 240, 536, 637 ],
      [ 739, 137, 120, 488, 706 ],
      [ 146, 153, 0, 146, 153 ],
      [ 605, 315, 90, 315, 605 ],
      [ 512, 480, 330, 682, 672 ],
      [ 509, 430, 30, 655, 626 ],
      [ 610, 253, 300, 522, 654 ],
      [ 80, 483, 330, 310, 458 ],
      [ 539, 470, -120, 675, 700 ],
      [ 747, 707, -180, 747, 707 ],
      [ 528, 365, 120, 579, 638 ],
      [ 329, 151, 240, 294, 359 ],
      [ 987, 260, -360, 987, 260 ],
      [ 483, 713, -60, 857, 774 ],
      [ 90, 329, -330, 241, 329 ],
      [ 43, 710, 90, 710, 43 ],
      [ 610, 400, 330, 727, 651 ],
      [ 504, 460, 60, 649, 666 ],
      [ 450, 96, -240, 307, 436 ],
      [ 815, 655, -300, 974, 1032 ],
      [ 63, 283, 60, 276, 195 ],
      [ 724, 530, 210, 891, 820 ],
      [ 542, 501, -300, 704, 719 ],
      [ 563, 95, -360, 563, 95 ],
      [ 1004, 162, -30, 949, 642 ],
      [ 36, 194, -90, 194, 36 ],
      [ 665, 652, 300, 896, 901 ],
      [ 607, 261, 0, 607, 261 ],
      [ 169, 598, 90, 598, 169 ],
      [ 357, 441, 0, 357, 441 ],
      [ 462, 488, 270, 488, 462 ],
      [ 29, 263, 30, 156, 242 ],
      [ 820, 669, -240, 988, 1043 ],
      [ 934, 323, -150, 969, 746 ],
      [ 974, 308, 30, 997, 753 ],
      [ 429, 334, 210, 537, 502 ],
      [ 830, 65, 90, 65, 830 ],
      [ 739, 520, 360, 739, 520 ],
      [ 237, 646, 180, 237, 646 ],
      [ 885, 145, -120, 567, 838 ],
      [ 359, 491, 330, 555, 604 ],
      [ 210, 512, -150, 436, 547 ],
      [ 767, 359, -240, 694, 842 ],
      [ 967, 291, 60, 735, 982 ],
      [ 270, 253, 360, 270, 253 ],
      [ 813, 574, -60, 902, 991 ],
      [ 589, 428, 0, 589, 428 ],
      [ 210, 214, -240, 289, 287 ],
      [ 179, 391, -360, 179, 391 ],
      [ 944, 287, -210, 960, 719 ],
      [ 249, 556, 270, 556, 249 ],
      [ 204, 713, 0, 204, 713 ],
      [ 509, 520, 360, 509, 520 ],
      [ 682, 477, -90, 477, 682 ],
      [ 262, 679, 150, 565, 717 ],
      [ 251, 520, -60, 574, 477 ],
      [ 686, 194, 360, 686, 194 ],
      [ 182, 351, -180, 182, 351 ],
      [ 371, 485, 210, 563, 604 ],
      [ 396, 425, -270, 425, 396 ],
      [ 614, 611, 270, 611, 614 ],
      [ 461, 763, 240, 890, 780 ],
      [ 991, 377, 0, 991, 377 ],
      [ 974, 120, -300, 590, 903 ],
      [ 320, 316, 360, 320, 316 ],
      [ 3, 504, 120, 437, 253 ],
      [ 451, 251, 300, 441, 515 ],
      [ 444, 387, -90, 387, 444 ],
      [ 723, 664, -360, 723, 664 ],
      [ 498, 193, 30, 527, 415 ],
      [ 656, 653, -120, 893, 894 ],
      [ 529, 230, -120, 462, 572 ],
      [ 329, 205, 240, 341, 386 ],
      [ 821, 168, 360, 821, 168 ],
      [ 925, 407, -90, 407, 925 ],
      [ 909, 409, -330, 991, 808 ],
      [ 562, 747, -90, 747, 562 ],
      [ 473, 312, 300, 505, 565 ],
      [ 852, 85, 180, 852, 85 ],
      [ 852, 459, 360, 852, 459 ],
      [ 394, 182, 270, 182, 394 ],
      [ 750, 579, -270, 579, 750 ],
      [ 74, 57, -60, 84, 92 ],
      [ 924, 672, 90, 672, 924 ],
      [ 910, 597, -270, 597, 910 ],
      [ 292, 511, 150, 507, 587 ],
      [ 346, 164, 120, 314, 380 ],
      [ 734, 518, -330, 894, 815 ],
      [ 613, 389, -240, 643, 723 ],
      [ 353, 260, 210, 434, 400 ],
      [ 349, 555, -360, 349, 555 ],
      [ 191, 349, 210, 339, 396 ],
      [ 340, 404, 240, 519, 495 ],
      [ 762, 329, 150, 823, 664 ],
      [ 383, 243, 0, 383, 243 ],
      [ 546, 462, -270, 462, 546 ],
      [ 229, 721, -90, 721, 229 ],
      [ 908, 503, -330, 1037, 889 ],
      [ 977, 194, 30, 943, 656 ],
      [ 136, 458, 300, 463, 346 ],
      [ 930, 720, 90, 720, 930 ],
      [ 943, 95, -330, 863, 553 ],
      [ 702, 349, 60, 652, 781 ],
      [ 536, 152, -360, 536, 152 ],
      [ 243, 439, -120, 500, 429 ],
      [ 745, 80, 330, 684, 441 ],
      [ 903, 251, 270, 251, 903 ],
      [ 256, 164, 30, 303, 269 ],
      [ 299, 129, 210, 322, 260 ],
      [ 859, 230, -90, 230, 859 ],
      [ 742, 160, -150, 721, 509 ],
      [ 341, 98, -60, 254, 344 ],
      [ 404, 624, 300, 740, 661 ],
      [ 1006, 258, -300, 725, 1000 ],
      [ 1021, 440, 120, 891, 1103 ],
      [ 336, 230, 180, 336, 230 ],
      [ 266, 139, -300, 252, 299 ],
      [ 132, 331, -150, 279, 352 ],
      [ 669, 554, -30, 855, 814 ],
      [ 439, 430, 210, 594, 590 ],
      [ 820, 219, 360, 820, 219 ],
      [ 112, 474, -300, 465, 333 ],
      [ 553, 8, 300, 282, 482 ],
    ];
  }

}
