/*
 * Decompiled with CFR 0.152.
 */
package org.apache.lucene.util.bkd;

import java.io.IOException;
import java.util.Arrays;
import org.apache.lucene.codecs.CodecUtil;
import org.apache.lucene.index.CorruptIndexException;
import org.apache.lucene.index.PointValues;
import org.apache.lucene.store.IndexInput;
import org.apache.lucene.util.Accountable;
import org.apache.lucene.util.BytesRef;
import org.apache.lucene.util.RamUsageEstimator;
import org.apache.lucene.util.StringHelper;
import org.apache.lucene.util.bkd.DocIdsWriter;

public class BKDReader
implements Accountable {
    private final byte[] splitPackedValues;
    final long[] leafBlockFPs;
    private final int leafNodeOffset;
    final int numDims;
    final int bytesPerDim;
    final int bytesPerIndexEntry;
    final IndexInput in;
    final int maxPointsInLeafNode;
    final byte[] minPackedValue;
    final byte[] maxPackedValue;
    final long pointCount;
    final int docCount;
    final int version;
    protected final int packedBytesLength;

    public BKDReader(IndexInput in) throws IOException {
        this.version = CodecUtil.checkHeader(in, "BKD", 0, 3);
        this.numDims = in.readVInt();
        this.maxPointsInLeafNode = in.readVInt();
        this.bytesPerDim = in.readVInt();
        this.bytesPerIndexEntry = this.numDims == 1 && this.version >= 3 ? this.bytesPerDim : this.bytesPerDim + 1;
        this.packedBytesLength = this.numDims * this.bytesPerDim;
        int numLeaves = in.readVInt();
        assert (numLeaves > 0);
        this.leafNodeOffset = numLeaves;
        this.minPackedValue = new byte[this.packedBytesLength];
        this.maxPackedValue = new byte[this.packedBytesLength];
        in.readBytes(this.minPackedValue, 0, this.packedBytesLength);
        in.readBytes(this.maxPackedValue, 0, this.packedBytesLength);
        for (int dim = 0; dim < this.numDims; ++dim) {
            if (StringHelper.compare(this.bytesPerDim, this.minPackedValue, dim * this.bytesPerDim, this.maxPackedValue, dim * this.bytesPerDim) <= 0) continue;
            throw new CorruptIndexException("minPackedValue " + new BytesRef(this.minPackedValue) + " is > maxPackedValue " + new BytesRef(this.maxPackedValue) + " for dim=" + dim, in);
        }
        this.pointCount = in.readVLong();
        this.docCount = in.readVInt();
        this.splitPackedValues = new byte[this.bytesPerIndexEntry * numLeaves];
        in.readBytes(this.splitPackedValues, 0, this.splitPackedValues.length);
        long[] leafBlockFPs = new long[numLeaves];
        long lastFP = 0L;
        for (int i = 0; i < numLeaves; ++i) {
            long delta = in.readVLong();
            leafBlockFPs[i] = lastFP + delta;
            lastFP += delta;
        }
        if (this.numDims == 1 && numLeaves > 1) {
            int levelCount = 2;
            while (true) {
                if (numLeaves >= levelCount && numLeaves <= 2 * levelCount) {
                    int lastLevel = 2 * (numLeaves - levelCount);
                    assert (lastLevel >= 0);
                    if (lastLevel == 0) break;
                    long[] newLeafBlockFPs = new long[numLeaves];
                    System.arraycopy(leafBlockFPs, lastLevel, newLeafBlockFPs, 0, leafBlockFPs.length - lastLevel);
                    System.arraycopy(leafBlockFPs, 0, newLeafBlockFPs, leafBlockFPs.length - lastLevel, lastLevel);
                    leafBlockFPs = newLeafBlockFPs;
                    break;
                }
                levelCount *= 2;
            }
        }
        this.leafBlockFPs = leafBlockFPs;
        this.in = in;
    }

    protected BKDReader(IndexInput in, int numDims, int maxPointsInLeafNode, int bytesPerDim, long[] leafBlockFPs, byte[] splitPackedValues, byte[] minPackedValue, byte[] maxPackedValue, long pointCount, int docCount) throws IOException {
        this.in = in;
        this.numDims = numDims;
        this.maxPointsInLeafNode = maxPointsInLeafNode;
        this.bytesPerDim = bytesPerDim;
        this.bytesPerIndexEntry = numDims == 1 ? bytesPerDim : bytesPerDim + 1;
        this.packedBytesLength = numDims * bytesPerDim;
        this.leafNodeOffset = leafBlockFPs.length;
        this.leafBlockFPs = leafBlockFPs;
        this.splitPackedValues = splitPackedValues;
        this.minPackedValue = minPackedValue;
        this.maxPackedValue = maxPackedValue;
        this.pointCount = pointCount;
        this.docCount = docCount;
        this.version = 3;
        assert (minPackedValue.length == this.packedBytesLength);
        assert (maxPackedValue.length == this.packedBytesLength);
    }

    public void verify(int maxDoc) throws IOException {
        byte[] rootMinPacked = new byte[this.packedBytesLength];
        byte[] rootMaxPacked = new byte[this.packedBytesLength];
        Arrays.fill(rootMaxPacked, (byte)-1);
        this.verify(this.getIntersectState(new VerifyVisitor(this.numDims, this.bytesPerDim, maxDoc)), 1, rootMinPacked, rootMaxPacked);
    }

    private void verify(IntersectState state, int nodeID, byte[] cellMinPacked, byte[] cellMaxPacked) throws IOException {
        if (nodeID >= this.leafNodeOffset) {
            int leafID = nodeID - this.leafNodeOffset;
            if (leafID < this.leafBlockFPs.length) {
                VerifyVisitor visitor = (VerifyVisitor)state.visitor;
                visitor.cellMinPacked = cellMinPacked;
                visitor.cellMaxPacked = cellMaxPacked;
                int count = this.readDocIDs(state.in, this.leafBlockFPs[leafID], state.scratchDocIDs);
                this.visitDocValues(state.commonPrefixLengths, state.scratchPackedValue, state.in, state.scratchDocIDs, count, state.visitor);
            }
        } else {
            int splitDim;
            int address = nodeID * this.bytesPerIndexEntry;
            if (this.numDims == 1) {
                splitDim = 0;
                if (this.version < 3) {
                    assert (this.splitPackedValues[address] == 0);
                    ++address;
                }
            } else {
                splitDim = this.splitPackedValues[address++] & 0xFF;
            }
            assert (splitDim < this.numDims);
            byte[] splitPackedValue = new byte[this.packedBytesLength];
            System.arraycopy(cellMaxPacked, 0, splitPackedValue, 0, this.packedBytesLength);
            System.arraycopy(this.splitPackedValues, address, splitPackedValue, splitDim * this.bytesPerDim, this.bytesPerDim);
            this.verify(state, 2 * nodeID, cellMinPacked, splitPackedValue);
            System.arraycopy(cellMinPacked, 0, splitPackedValue, 0, this.packedBytesLength);
            System.arraycopy(this.splitPackedValues, address, splitPackedValue, splitDim * this.bytesPerDim, this.bytesPerDim);
            this.verify(state, 2 * nodeID + 1, splitPackedValue, cellMaxPacked);
        }
    }

    public void intersect(PointValues.IntersectVisitor visitor) throws IOException {
        this.intersect(this.getIntersectState(visitor), 1, this.minPackedValue, this.maxPackedValue);
    }

    private void addAll(IntersectState state, int nodeID) throws IOException {
        if (nodeID >= this.leafNodeOffset) {
            this.visitDocIDs(state.in, this.leafBlockFPs[nodeID - this.leafNodeOffset], state.visitor);
        } else {
            this.addAll(state, 2 * nodeID);
            this.addAll(state, 2 * nodeID + 1);
        }
    }

    public IntersectState getIntersectState(PointValues.IntersectVisitor visitor) {
        return new IntersectState(this.in.clone(), this.numDims, this.packedBytesLength, this.maxPointsInLeafNode, visitor);
    }

    public void visitLeafBlockValues(int nodeID, IntersectState state) throws IOException {
        int leafID = nodeID - this.leafNodeOffset;
        int count = this.readDocIDs(state.in, this.leafBlockFPs[leafID], state.scratchDocIDs);
        this.visitDocValues(state.commonPrefixLengths, state.scratchPackedValue, state.in, state.scratchDocIDs, count, state.visitor);
    }

    protected void visitDocIDs(IndexInput in, long blockFP, PointValues.IntersectVisitor visitor) throws IOException {
        in.seek(blockFP);
        int count = in.readVInt();
        visitor.grow(count);
        if (this.version < 1) {
            DocIdsWriter.readInts32(in, count, visitor);
        } else {
            DocIdsWriter.readInts(in, count, visitor);
        }
    }

    protected int readDocIDs(IndexInput in, long blockFP, int[] docIDs) throws IOException {
        in.seek(blockFP);
        int count = in.readVInt();
        if (this.version < 1) {
            DocIdsWriter.readInts32(in, count, docIDs);
        } else {
            DocIdsWriter.readInts(in, count, docIDs);
        }
        return count;
    }

    protected void visitDocValues(int[] commonPrefixLengths, byte[] scratchPackedValue, IndexInput in, int[] docIDs, int count, PointValues.IntersectVisitor visitor) throws IOException {
        int compressedDim;
        visitor.grow(count);
        this.readCommonPrefixes(commonPrefixLengths, scratchPackedValue, in);
        int n = compressedDim = this.version < 2 ? -1 : this.readCompressedDim(in);
        if (compressedDim == -1) {
            this.visitRawDocValues(commonPrefixLengths, scratchPackedValue, in, docIDs, count, visitor);
        } else {
            this.visitCompressedDocValues(commonPrefixLengths, scratchPackedValue, in, docIDs, count, visitor, compressedDim);
        }
    }

    private void visitRawDocValues(int[] commonPrefixLengths, byte[] scratchPackedValue, IndexInput in, int[] docIDs, int count, PointValues.IntersectVisitor visitor) throws IOException {
        for (int i = 0; i < count; ++i) {
            for (int dim = 0; dim < this.numDims; ++dim) {
                int prefix = commonPrefixLengths[dim];
                in.readBytes(scratchPackedValue, dim * this.bytesPerDim + prefix, this.bytesPerDim - prefix);
            }
            visitor.visit(docIDs[i], scratchPackedValue);
        }
    }

    private void visitCompressedDocValues(int[] commonPrefixLengths, byte[] scratchPackedValue, IndexInput in, int[] docIDs, int count, PointValues.IntersectVisitor visitor, int compressedDim) throws IOException {
        int i;
        int runLen;
        int compressedByteOffset = compressedDim * this.bytesPerDim + commonPrefixLengths[compressedDim];
        int n = compressedDim;
        commonPrefixLengths[n] = commonPrefixLengths[n] + 1;
        for (i = 0; i < count; i += runLen) {
            scratchPackedValue[compressedByteOffset] = in.readByte();
            runLen = Byte.toUnsignedInt(in.readByte());
            for (int j = 0; j < runLen; ++j) {
                for (int dim = 0; dim < this.numDims; ++dim) {
                    int prefix = commonPrefixLengths[dim];
                    in.readBytes(scratchPackedValue, dim * this.bytesPerDim + prefix, this.bytesPerDim - prefix);
                }
                visitor.visit(docIDs[i + j], scratchPackedValue);
            }
        }
        if (i != count) {
            throw new CorruptIndexException("Sub blocks do not add up to the expected count: " + count + " != " + i, in);
        }
    }

    private int readCompressedDim(IndexInput in) throws IOException {
        byte compressedDim = in.readByte();
        if (compressedDim < -1 || compressedDim >= this.numDims) {
            throw new CorruptIndexException("Got compressedDim=" + compressedDim, in);
        }
        return compressedDim;
    }

    private void readCommonPrefixes(int[] commonPrefixLengths, byte[] scratchPackedValue, IndexInput in) throws IOException {
        for (int dim = 0; dim < this.numDims; ++dim) {
            int prefix;
            commonPrefixLengths[dim] = prefix = in.readVInt();
            if (prefix <= 0) continue;
            in.readBytes(scratchPackedValue, dim * this.bytesPerDim, prefix);
        }
    }

    private void intersect(IntersectState state, int nodeID, byte[] cellMinPacked, byte[] cellMaxPacked) throws IOException {
        PointValues.Relation r = state.visitor.compare(cellMinPacked, cellMaxPacked);
        if (r == PointValues.Relation.CELL_OUTSIDE_QUERY) {
            return;
        }
        if (r == PointValues.Relation.CELL_INSIDE_QUERY) {
            this.addAll(state, nodeID);
            return;
        }
        if (nodeID >= this.leafNodeOffset) {
            int leafID = nodeID - this.leafNodeOffset;
            if (leafID < this.leafBlockFPs.length) {
                int count = this.readDocIDs(state.in, this.leafBlockFPs[leafID], state.scratchDocIDs);
                this.visitDocValues(state.commonPrefixLengths, state.scratchPackedValue, state.in, state.scratchDocIDs, count, state.visitor);
            }
        } else {
            int splitDim;
            int address = nodeID * this.bytesPerIndexEntry;
            if (this.numDims == 1) {
                splitDim = 0;
                if (this.version < 3) {
                    assert (this.splitPackedValues[address] == 0);
                    ++address;
                }
            } else {
                splitDim = this.splitPackedValues[address++] & 0xFF;
            }
            assert (splitDim < this.numDims);
            byte[] splitPackedValue = new byte[this.packedBytesLength];
            System.arraycopy(cellMaxPacked, 0, splitPackedValue, 0, this.packedBytesLength);
            System.arraycopy(this.splitPackedValues, address, splitPackedValue, splitDim * this.bytesPerDim, this.bytesPerDim);
            this.intersect(state, 2 * nodeID, cellMinPacked, splitPackedValue);
            System.arraycopy(cellMinPacked, 0, splitPackedValue, 0, this.packedBytesLength);
            System.arraycopy(this.splitPackedValues, address, splitPackedValue, splitDim * this.bytesPerDim, this.bytesPerDim);
            this.intersect(state, 2 * nodeID + 1, splitPackedValue, cellMaxPacked);
        }
    }

    public void copySplitValue(int nodeID, byte[] splitPackedValue) {
        int splitDim;
        int address = nodeID * this.bytesPerIndexEntry;
        if (this.numDims == 1) {
            splitDim = 0;
            if (this.version < 3) {
                assert (this.splitPackedValues[address] == 0);
                ++address;
            }
        } else {
            splitDim = this.splitPackedValues[address++] & 0xFF;
        }
        assert (splitDim < this.numDims);
        System.arraycopy(this.splitPackedValues, address, splitPackedValue, splitDim * this.bytesPerDim, this.bytesPerDim);
    }

    @Override
    public long ramBytesUsed() {
        return RamUsageEstimator.sizeOf(this.splitPackedValues) + RamUsageEstimator.sizeOf(this.leafBlockFPs);
    }

    public byte[] getMinPackedValue() {
        return (byte[])this.minPackedValue.clone();
    }

    public byte[] getMaxPackedValue() {
        return (byte[])this.maxPackedValue.clone();
    }

    public int getNumDimensions() {
        return this.numDims;
    }

    public int getBytesPerDimension() {
        return this.bytesPerDim;
    }

    public long getPointCount() {
        return this.pointCount;
    }

    public int getDocCount() {
        return this.docCount;
    }

    public boolean isLeafNode(int nodeID) {
        return nodeID >= this.leafNodeOffset;
    }

    public static final class IntersectState {
        final IndexInput in;
        final int[] scratchDocIDs;
        final byte[] scratchPackedValue;
        final int[] commonPrefixLengths;
        final PointValues.IntersectVisitor visitor;

        public IntersectState(IndexInput in, int numDims, int packedBytesLength, int maxPointsInLeafNode, PointValues.IntersectVisitor visitor) {
            this.in = in;
            this.visitor = visitor;
            this.commonPrefixLengths = new int[numDims];
            this.scratchDocIDs = new int[maxPointsInLeafNode];
            this.scratchPackedValue = new byte[packedBytesLength];
        }
    }

    private static class VerifyVisitor
    implements PointValues.IntersectVisitor {
        byte[] cellMinPacked;
        byte[] cellMaxPacked;
        byte[] lastPackedValue;
        final int numDims;
        final int bytesPerDim;
        final int maxDoc;

        public VerifyVisitor(int numDims, int bytesPerDim, int maxDoc) {
            this.numDims = numDims;
            this.bytesPerDim = bytesPerDim;
            this.maxDoc = maxDoc;
        }

        @Override
        public void visit(int docID) {
            throw new UnsupportedOperationException();
        }

        @Override
        public void visit(int docID, byte[] packedValue) {
            if (docID < 0 || docID >= this.maxDoc) {
                throw new RuntimeException("docID=" + docID + " is out of bounds of 0.." + this.maxDoc);
            }
            for (int dim = 0; dim < this.numDims; ++dim) {
                if (StringHelper.compare(this.bytesPerDim, this.cellMinPacked, dim * this.bytesPerDim, packedValue, dim * this.bytesPerDim) > 0) {
                    throw new RuntimeException("value=" + new BytesRef(packedValue, dim * this.bytesPerDim, this.bytesPerDim) + " for docID=" + docID + " dim=" + dim + " is less than this leaf block's minimum=" + new BytesRef(this.cellMinPacked, dim * this.bytesPerDim, this.bytesPerDim));
                }
                if (StringHelper.compare(this.bytesPerDim, this.cellMaxPacked, dim * this.bytesPerDim, packedValue, dim * this.bytesPerDim) >= 0) continue;
                throw new RuntimeException("value=" + new BytesRef(packedValue, dim * this.bytesPerDim, this.bytesPerDim) + " for docID=" + docID + " dim=" + dim + " is greater than this leaf block's maximum=" + new BytesRef(this.cellMaxPacked, dim * this.bytesPerDim, this.bytesPerDim));
            }
            if (this.numDims == 1) {
                if (this.lastPackedValue == null) {
                    this.lastPackedValue = Arrays.copyOf(packedValue, packedValue.length);
                } else {
                    if (StringHelper.compare(this.bytesPerDim, this.lastPackedValue, 0, packedValue, 0) > 0) {
                        throw new RuntimeException("value=" + new BytesRef(packedValue) + " for docID=" + docID + " dim=0 sorts before last value=" + new BytesRef(this.lastPackedValue));
                    }
                    System.arraycopy(packedValue, 0, this.lastPackedValue, 0, this.bytesPerDim);
                }
            }
        }

        @Override
        public PointValues.Relation compare(byte[] minPackedValue, byte[] maxPackedValue) {
            throw new UnsupportedOperationException();
        }
    }
}

