"""
Pympc is a client for Music Player Daemon.
Copyright (C) 2004  Magnus Bjernstad <bjernstad@gmail.com>

This file is part of Pympc.

Pympc is free software; you can redistribute it and/or modify
it under the terms of the GNU General Public License as published by
the Free Software Foundation; either version 2 of the License, or
(at your option) any later version.

Pympc is distributed in the hope that it will be useful,
but WITHOUT ANY WARRANTY; without even the implied warranty of
MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
GNU General Public License for more details.

You should have received a copy of the GNU General Public License
along with Pympc; if not, write to the Free Software
Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
"""
import gtk, mpdclient

# These are the metadata fields returned by mpd
METADATA = [
    'Artist',
    'Album',
    'Title',
    'Genre',
    'Date',
    'Track',
    'Composer',
    'Performer',
    'Comment']

class Find:
    def __init__(self, app, gui_file, incremental):
        self.app = app
        self.incremental = incremental

        self.gui_popup = gtk.glade.XML(gui_file, 'find_popup')
        self.gui_popup.signal_autoconnect(self)

        self.gui_main = gtk.glade.XML(gui_file, 'find_root')
        self.gui_main.signal_autoconnect(self)
        self.w = self.gui_main.get_widget
        self.contents = self.w('find_root')

        self.hits = gtk.ListStore(str)
        column = gtk.TreeViewColumn('Results', gtk.CellRendererText(), text=0)
        self.w('view').set_model(self.hits)
        self.w('view').get_selection().set_mode(gtk.SELECTION_MULTIPLE)
        self.w('view').append_column(column)

        self.w('combo').append_text('Filename')
        for x in METADATA:
            self.w('combo').append_text(x)
        self.w('combo').set_active(0)

    def on_combo_changed(self, widget):
        self.w('entry').emit('changed')
    def play(self):
        try:
            self.app.c.clear()
            self.enqueue()
            self.app.c.play()
        except mpdclient.MpdError:
            self.app.c = False

    def enqueue(self):
        indices = self.w('view').get_selection().get_selected_rows()[1]
        self.app.lib.enqueue([self.hits[x][0] for x in indices])

    def find(self, pattern):
        if not self.app.c:
            return
        if len(pattern) < 1:
            return
        self.hits.clear()
        entry_type = self.w('combo').get_active_text().lower()
        try:
            matches = self.app.c.search(entry_type, pattern.encode('utf-8'))
        except mpdclient.MpdError:
            self.app.c = False
        map(lambda x: self.hits.append([x]), matches)

    def on_play(self, action): self.play()
    def on_add(self, action): self.enqueue()
    def on_enter(self, widget): self.find(widget.get_text())
    def on_row_select(self, treeview, path, view_column):
        self.enqueue()
    def on_new_pattern(self, widget):
        if self.incremental:
            pattern = widget.get_text().decode('utf-8')
            if len(pattern) >= 2:
                self.find(pattern)
    def on_popup_menu(self, widget):
        self.popup_menu(widget, 0,0)
        return True
    def on_button_press(self, widget, event):
        if event.button == 3:
            return widget.get_selection().count_selected_rows() > 1
    def on_button_release(self, widget, event):
        if event.button == 3:
            self.popup_menu(widget, event.button, event.time)
    def popup_menu(self, view, button, time):
        if button in (0, 3):
            pathlist = view.get_selection().get_selected_rows()[1]
            w = self.gui_popup.get_widget
            w('play').set_sensitive(len(pathlist) > 0)
            w('add').set_sensitive(len(pathlist) > 0)
            w('find_popup').popup(None, None, None, button, time)
            return True
