#############################################################################
##
#W  database.g                GAP4 Package `RCWA'                 Stefan Kohl
##
##  Data library of the groups generated by 3 class transpositions which
##  interchange residue classes with moduli <= 6.
##
##  `grps' is the list of the groups in the database, and the following
##  relations hold:
##
##  1. mods  = List(grps,Mod);
##  2. sizes = List(grps,Size);
##  3. TriangleTypes() = List(grps,G->List([[1,2],[1,3],[2,3]],
##           ind->CTPairProductType(GeneratorsOfGroup(G){ind})));
##     where CTPairProductType is the component of the record returned by
##     the function `LoadDatabaseOfProductsOf2ClassTranspositions'.
##
##  Given a group G which is generated by 3 class transpositions with moduli
##  <= 6, Id3CTsGroup(G,grps) is the position of the group G in the list
##  grps of groups in this database.
##
##  The list `freeproductcandidates' is a list of the indices of those groups
##  which are possibly isomorphic to the free product of 3 cyclic groups of
##  order 2, thus whose generators possibly do not satisfy any further
##  relations. Only one index per class in 'conjugacyclasses' is given.
##
##  The lists `stabilize_digitsum_base2_mod2', 
##  `stabilize_digitsum_base2_mod3' and `stabilize_digitsum_base3_mod2'
##  contain indices of groups which stabilize the digit sum of an integer
##  in base 2 modulo 2, in base 2 modulo 3 or in base 3 modulo 2,
##  respectively.
##
##  The list `abc_torsion' contains an entry [ <group number>, <order of the
##  product of all 3 generators> ] for every infinite group in the database
##  for which the product of all 3 generators has finite order.
##
##  Descriptions of the following components can be found in other files in
##  this directory:
##
##    - `equalityclasses': equalityclasses.g,
##    - `samegroups': samegroups.g,
##    - `conjugacyclasses': conjugacyclasses.g,
##    - `subgroups': subgroups.g,
##    - `supergroups': supergroups.g,
##    - `chains': chains.g,
##    - `intransitivemodulo': intransitivemodulo.g,
##    - `partitionlengths': partitionlengths.g,
##    - `degrees': degrees.g,
##    - `orbitlengths': orbitlengths.g,
##    - `permgroupgens': permgroupgens.g,
##    - `respectedpartitions': respectedpartitions.g,
##    - `finiteorbits': finite-orbits.g,
##    - `shortresidueclassorbitlengths': residueclassorbitlengths.g,
##    - `cyclist': spheresizecycles.g,
##    - `trsstatus', `trsstatuspos' and `trsstatusset': trsstatuspos.g, and
##    - `orbitgrowthtype': orbitgrowthtype.g.
##
#############################################################################

return rec(

#############################################################################
##
#S  Utility functions. //////////////////////////////////////////////////////
##
#############################################################################

Id3CTsGroup := function ( G, grps )

  local  gens;

  if not IsRcwaGroupOverZ(G) then return fail; fi;

  gens := Set(GeneratorsOfGroup(G));
  if Length(gens) <> 3 or not ForAll(gens,IsClassTransposition)
    or not ForAll(gens,g->Maximum(List(TransposedClasses(g),Mod))<=6)
  then return fail; fi;

  return First([1..Length(grps)],i->gens=GeneratorsOfGroup(grps[i]));
end,

TriangleTypes := function ( )

  local  triangles, triangle, cts, ctpairs, cttriples,
         data, ctptypes, mods, sizes, i;

  cts       := List(ClassPairs(6),ClassTransposition);
  ctpairs   := Combinations(cts,2);
  cttriples := Combinations(cts,3);

  data     := LoadDatabaseOfProductsOf2ClassTranspositions();
  ctptypes := data.CTPairsProductType;

  data  := LoadDatabaseOfGroupsGeneratedBy3ClassTranspositions();
  mods  := data.mods;
  sizes := data.sizes;

  triangles := [];
  for i in [1..Length(cttriples)] do
    triangle := List([[1,2],[1,3],[2,3]],ind->Set(cttriples[i]{ind}));
    triangle := List(triangle,edge->ctptypes[PositionSorted(ctpairs,edge)]);
    Add(triangles,triangle);
  od;

  return triangles;
end,

3CTsGroupsWithGivenOrbit := function ( orbit, maxmod )

  local  cts, stab, perms, gens, gensp, indcombs,
         grps, grpsp, trsinds, start;

  start := Runtime();
  orbit := AsSet(orbit);

  Info(InfoRCWA,2,"Setting up the list of class transpositions ...");
  cts   := List(ClassPairs(maxmod),ClassTransposition);
  Info(InfoRCWA,2,"Elapsed time: ",Runtime()-start); start := Runtime();

  Info(InfoRCWA,2,"Determining which class transpositions stabilize ...");
  stab  := Filtered(cts,ct->orbit^ct=orbit);
  perms := List(stab,s->Permutation(s,orbit));
  Info(InfoRCWA,2,"Found ",Length(stab),
                  " class transpositions which stabilize.");
  Info(InfoRCWA,2,"Elapsed time: ",Runtime()-start); start := Runtime();

  Info(InfoRCWA,2,"Forming the list of triples of class transpositions ",
                  "which stabilize ...");
  indcombs := Combinations([1..Length(stab)],3);
  gens     := List(indcombs,ind->stab{ind});
  gensp    := List(indcombs,ind->perms{ind});

  Info(InfoRCWA,2,"Forming the list of groups ...");
  grpsp    := List(gensp,Group);
  Info(InfoRCWA,2,"There are ",Length(grpsp)," groups to check in total.");
  Info(InfoRCWA,2,"Elapsed time: ",Runtime()-start); start := Runtime();

  Info(InfoRCWA,2,"Filtering the groups for transitivity ...");
  trsinds  := Filtered([1..Length(grpsp)],
                       i->IsTransitive(grpsp[i],[1..Length(orbit)]));
  grps     := List(gens{trsinds},Group);
  Info(InfoRCWA,2,"Found ",Length(grps),
                  " groups which have the given orbit.");
  Info(InfoRCWA,2,"Elapsed time: ",Runtime()-start); start := Runtime();

  return grps;
end,

ProbablyFixesDigitSumsModulo := function ( g, b, m )

  local  n, B;

  if b <= 1 then return fail; fi;
  if IsOne(g) or m = 1 then return true; fi;
  B := 100 * Mod(g);
  n := 0;
  repeat
    if   Sum(CoefficientsQadic(n  ,b)) mod m
      <> Sum(CoefficientsQadic(n^g,b)) mod m
    then return false; fi;
    if n >= B then return true; fi;
    n := n + 1;
  until false;
end,

ProbablyStabilizesDigitSumsModulo := function ( G, b, m )
  return ForAll(GeneratorsOfGroup(G),
                g->~.ProbablyFixesDigitSumsModulo(g,b,m));
end,

#############################################################################
##
#S  The groups. /////////////////////////////////////////////////////////////
##
#############################################################################

cts  := List(ClassPairs(6),ClassTransposition),

grps := List(Combinations(~.cts,3),Group),

groups := ~.grps, # synonym for `grps'

#############################################################################
##
#S  Data lists. /////////////////////////////////////////////////////////////
##
#############################################################################

mods :=
[ 6,6,12,12,12,0,12,12,0,12,0,0,30,30,30,30,30,
  30,30,30,30,30,6,6,6,6,6,6,0,0,6,6,6,0,6,0,0,0,0,
  0,6,6,0,6,0,6,0,6,0,0,12,12,0,0,0,0,0,0,0,0,0,0,
  0,0,0,0,0,6,12,12,12,0,12,12,0,12,0,0,30,30,30,30,
  30,30,30,30,30,30,6,6,6,6,6,6,0,0,6,6,6,12,6,0,0,
  0,0,0,6,6,0,6,0,6,12,6,0,0,0,0,0,0,0,0,0,0,0,0,
  0,0,0,0,0,0,0,12,12,12,0,12,12,0,12,0,0,30,30,30,
  30,30,30,30,30,30,30,6,6,6,6,6,6,0,0,6,6,6,0,6,0,
  0,0,0,0,6,6,0,6,0,6,0,6,0,0,0,0,0,0,0,0,0,0,0,0,
  0,0,0,0,0,0,0,4,4,0,4,4,0,4,0,0,20,20,20,20,20,20,
  20,20,20,20,12,12,12,12,12,12,0,0,12,12,12,24,12,24,
  24,0,0,0,12,12,0,12,0,12,24,12,0,0,24,24,0,0,0,0,0,
  0,0,0,0,0,0,0,0,0,0,4,0,4,4,0,4,0,0,20,20,20,20,
  20,20,20,20,20,20,12,12,12,12,12,12,0,0,12,12,12,0,
  12,0,0,0,0,0,12,12,0,12,0,12,0,12,0,0,0,0,0,0,0,0,
  0,0,0,0,0,0,0,0,0,0,0,0,4,4,0,4,0,0,20,20,20,20,
  20,20,20,20,20,20,12,12,12,12,12,12,0,0,12,12,12,0,
  12,0,0,0,0,0,12,12,0,12,0,12,0,12,0,0,0,0,0,0,0,0,
  0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,
  0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,
  0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,
  4,0,4,0,0,20,20,20,20,20,20,20,20,20,20,12,12,12,12,
  12,12,0,0,12,12,12,24,12,24,24,0,0,0,12,12,0,12,0,
  12,24,12,0,0,24,24,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,
  0,4,0,0,20,20,20,20,20,20,20,20,20,20,12,12,12,12,12,
  12,0,0,12,12,12,0,12,0,0,0,0,0,12,12,0,12,0,12,0,
  12,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,
  0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,
  0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,
  0,0,0,0,0,0,0,0,0,20,20,20,20,20,20,20,20,20,20,12,
  12,12,12,12,12,0,0,12,12,12,0,12,0,0,0,0,0,12,12,0,
  12,0,12,0,12,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,
  0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,
  0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,
  0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,
  0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,
  0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,10,10,10,10,10,10,
  10,10,10,30,30,30,30,30,30,0,0,30,30,30,0,30,120,120,
  0,0,0,30,30,0,30,0,30,0,30,0,0,0,0,0,0,0,0,0,0,0,
  0,0,0,0,0,0,0,0,10,10,10,10,10,10,10,10,30,30,30,30,
  30,30,0,0,30,30,30,120,30,0,0,0,0,0,30,30,0,30,0,30,
  120,30,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,10,
  10,10,10,10,10,10,30,30,30,30,30,30,0,0,30,30,30,0,
  30,0,0,0,0,0,30,30,0,30,0,30,0,30,0,0,0,0,0,0,0,0,
  0,0,0,0,0,0,0,0,0,0,0,10,10,10,10,10,10,30,30,30,
  30,30,30,0,0,30,30,30,0,30,0,0,0,0,0,30,30,0,30,0,
  30,0,30,0,0,120,120,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,
  10,10,10,10,10,30,30,30,30,30,30,0,0,30,30,30,0,30,0,
  0,0,0,0,30,30,0,30,0,30,0,30,0,0,0,0,0,0,0,0,0,0,
  0,0,0,0,0,0,0,0,0,10,10,10,10,30,30,30,30,30,30,0,
  0,30,30,30,480,30,0,0,0,0,0,30,30,0,30,0,30,480,30,
  0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,10,10,10,
  30,30,30,30,30,30,0,0,30,30,30,0,30,120,120,0,0,0,30,
  30,0,30,0,30,0,30,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,
  0,0,0,0,10,10,30,30,30,30,30,30,0,0,30,30,30,0,30,0,
  0,0,0,0,30,30,0,30,0,30,0,30,0,0,0,0,0,0,0,0,0,0,
  0,0,0,0,0,0,0,0,0,10,30,30,30,30,30,30,0,0,30,30,
  30,0,30,0,0,0,0,0,30,30,0,30,0,30,0,30,0,0,480,480,
  0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,30,30,30,30,30,30,0,
  0,30,30,30,0,30,0,0,0,0,0,30,30,0,30,0,30,0,30,0,0,
  0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,6,6,6,6,6,0,0,
  6,6,6,12,6,12,12,0,0,0,6,6,0,6,0,6,12,6,0,0,12,12,
  0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,6,6,6,6,0,0,6,6,6,
  12,6,0,0,0,0,0,6,6,0,6,0,6,12,6,0,0,0,0,0,0,0,0,
  0,0,0,0,0,0,0,0,0,0,0,6,6,6,0,0,6,6,6,12,6,0,0,
  0,0,0,6,6,0,6,0,6,12,6,0,0,0,0,0,0,0,0,0,0,0,0,
  0,0,0,0,0,0,0,6,6,0,0,6,6,6,12,6,12,12,0,0,0,6,6,
  0,6,0,6,12,6,0,0,12,12,0,0,0,0,0,0,0,0,0,0,0,0,0,
  0,0,6,0,0,6,6,6,12,6,0,0,0,0,0,6,6,0,6,0,6,12,6,
  0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,6,6,6,
  12,6,0,0,0,0,0,6,6,0,6,0,6,12,6,0,0,0,0,0,0,0,0,
  0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,
  0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,
  0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,
  0,0,0,0,0,0,0,0,0,0,0,0,0,0,6,6,0,6,0,0,0,0,0,6,
  6,0,6,0,6,0,6,0,0,12,12,0,0,0,0,0,0,0,0,0,0,0,0,
  0,0,0,6,0,6,0,0,0,0,0,6,6,0,6,0,6,0,6,0,0,12,12,
  0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,6,12,12,0,0,0,6,
  6,0,6,0,6,0,6,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,
  0,0,0,0,0,0,0,0,12,0,0,0,0,0,12,0,0,0,0,0,0,0,0,
  0,0,0,0,0,0,0,0,0,0,0,0,12,12,0,0,0,6,6,0,6,0,6,
  0,6,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,12,0,
  0,0,12,0,0,0,0,12,0,12,0,0,0,0,0,0,0,0,0,0,0,0,0,
  0,0,0,0,0,0,0,0,0,12,0,0,0,0,12,0,12,0,0,0,0,0,0,
  0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,
  0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,
  0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,
  0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,
  0,6,0,6,0,6,12,6,0,0,12,12,0,0,0,0,0,0,0,0,0,0,0,
  0,0,0,0,0,6,0,6,0,6,0,0,12,12,0,0,0,0,0,0,0,0,0,
  0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,
  0,0,0,0,0,0,0,6,0,6,0,0,12,12,0,0,0,0,0,0,0,0,0,
  0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,
  0,0,0,0,0,6,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,
  0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,
  0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,
  0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,
  0,0,0,12,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,
  0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,
  0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,
  0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,
  0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,
  0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,3,12,12,
  12,0,12,12,0,12,0,0,15,15,15,15,15,15,15,15,15,15,6,
  6,6,6,6,6,0,0,6,6,6,0,6,0,0,0,0,0,6,6,0,6,0,6,0,
  6,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,12,12,12,
  0,12,12,0,12,0,0,15,15,15,15,15,15,15,15,15,15,6,6,
  6,6,6,6,0,0,6,6,6,0,6,0,0,0,0,0,6,6,0,6,0,6,0,6,
  0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,12,12,0,12,
  12,96,12,96,0,60,60,60,60,60,60,60,60,60,60,12,12,12,
  12,12,12,0,0,12,12,12,0,12,0,0,0,0,0,12,12,48,12,0,
  12,48,12,0,0,24,24,0,108,0,0,0,0,0,0,0,0,72,0,0,72,
  0,12,0,12,12,96,12,0,0,60,60,60,60,60,60,60,60,60,60,
  12,12,12,12,12,12,0,0,12,12,12,0,12,0,0,0,0,0,12,12,
  0,12,108,12,0,12,24,24,0,0,0,0,0,0,0,0,0,0,0,0,0,
  72,0,72,0,0,12,12,0,12,0,0,60,60,60,60,60,60,60,60,
  60,60,12,12,12,12,12,12,0,0,12,12,12,0,12,0,0,0,0,0,
  12,12,24,12,36,12,24,12,24,24,0,0,0,0,0,0,0,0,0,0,
  0,0,36,36,0,72,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,
  0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,
  0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,12,0,
  12,0,0,60,60,60,60,60,60,60,60,60,60,12,12,12,12,12,
  12,48,0,12,12,12,0,12,0,48,0,0,0,12,12,0,12,0,12,0,
  12,0,0,24,24,0,108,0,0,0,72,0,0,0,0,0,72,0,0,0,0,
  12,0,0,60,60,60,60,60,60,60,60,60,60,12,12,12,12,12,
  12,24,0,12,12,12,0,12,0,24,0,0,0,12,12,0,12,0,12,0,
  12,0,0,0,0,0,36,0,0,0,36,0,0,0,0,0,72,0,36,0,0,0,
  0,0,7680,0,0,0,0,0,0,0,0,48,0,0,0,0,0,0,0,48,0,0,
  0,0,0,0,0,0,0,0,0,48,0,0,0,48,0,0,0,0,0,0,0,0,0,
  0,144,0,0,0,0,0,0,0,0,0,0,0,60,60,60,60,60,60,60,
  60,60,60,12,12,12,12,12,12,24,0,12,12,12,0,12,0,24,0,
  0,0,12,12,24,12,108,12,24,12,24,24,24,24,0,0,0,0,0,
  72,0,0,0,0,72,0,0,0,0,0,0,0,0,1920,0,0,0,0,0,0,48,
  0,0,0,0,0,48,0,48,0,0,0,0,0,48,0,0,0,0,0,0,0,0,0,
  0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,144,0,0,0,0,0,
  0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,
  0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,
  0,0,0,0,0,15,15,15,15,15,15,15,15,15,30,30,30,30,30,
  30,120,0,30,30,30,0,30,0,120,0,0,0,30,30,60,30,810,
  30,60,30,0,0,0,0,0,270,0,0,0,1080,0,0,0,0,6480,6480,
  0,6480,0,15,15,15,15,15,15,15,15,30,30,30,30,30,30,0,
  0,30,30,30,0,30,0,0,0,0,0,30,30,60,30,0,30,60,30,60,
  60,0,0,0,0,0,0,0,0,0,0,0,0,6480,1080,0,1080,0,15,15,
  15,15,15,15,15,30,30,30,30,30,30,0,0,30,30,30,0,30,0,
  0,0,0,0,30,30,0,30,270,30,0,30,60,60,0,0,0,0,0,0,0,
  0,0,0,0,0,1080,1080,0,1080,0,15,15,15,15,15,15,30,30,
  30,30,30,30,120,0,30,30,30,0,30,0,120,0,0,0,30,30,
  120,30,270,30,120,30,0,0,60,60,0,810,0,0,0,1080,0,0,
  0,0,1080,0,0,0,0,15,15,15,15,15,30,30,30,30,30,30,0,
  0,30,30,30,0,30,0,0,0,0,0,30,30,60,30,0,30,60,30,
  120,120,120,120,0,270,0,0,0,0,0,0,0,0,1080,0,0,0,0,
  15,15,15,15,30,30,30,30,30,30,30,0,30,30,30,0,30,0,
  30,0,0,0,30,30,30,30,810,30,30,30,30,30,30,30,0,810,
  0,0,0,0,0,0,0,0,1080,6480,0,6480,0,15,15,15,30,30,30,
  30,30,30,60,0,30,30,30,0,30,0,60,0,0,0,30,30,0,30,0,
  30,0,30,0,0,0,0,0,810,0,0,0,6480,0,0,0,0,0,1080,0,
  1080,0,15,15,30,30,30,30,30,30,60,0,30,30,30,0,30,0,
  60,0,0,0,30,30,0,30,0,30,0,30,0,0,60,60,0,270,0,0,
  0,1080,0,0,0,0,0,0,0,0,0,15,30,30,30,30,30,30,30,0,
  30,30,30,0,30,0,30,0,0,0,30,30,30,30,270,30,30,30,30,
  30,30,30,0,0,0,0,0,6480,0,0,0,0,6480,0,0,0,0,30,30,
  30,30,30,30,0,0,30,30,30,0,30,0,0,0,0,0,30,30,60,30,
  0,30,60,30,0,0,0,0,0,0,0,0,0,0,0,0,0,0,1080,0,0,0,
  0,6,6,6,6,6,12,0,6,6,6,0,6,0,12,0,0,0,6,6,12,6,
  18,6,12,6,6,6,6,6,0,18,0,0,0,36,0,0,0,0,36,36,0,
  36,0,6,6,6,6,0,0,6,6,6,0,6,0,0,0,0,0,6,6,12,6,18,
  6,12,6,12,12,0,0,0,0,0,0,0,0,0,0,0,0,36,36,0,0,0,
  6,6,6,0,0,6,6,6,0,6,0,0,0,0,0,6,6,12,6,18,6,12,6,
  12,12,0,0,0,0,0,0,0,0,0,0,0,0,36,36,0,0,0,6,6,0,
  0,6,6,6,0,6,0,0,0,0,0,6,6,12,6,18,6,12,6,0,0,12,
  12,0,0,0,0,0,0,0,0,0,0,36,36,0,0,0,6,12,0,6,6,6,
  0,6,0,12,0,0,0,6,6,12,6,18,6,12,6,6,6,6,6,0,18,0,
  0,0,36,0,0,0,0,36,36,0,36,0,12,0,6,6,6,0,6,0,12,0,
  0,0,6,6,12,6,18,6,12,6,6,6,6,6,0,18,0,0,0,36,0,0,
  0,0,36,36,0,36,0,0,12,0,12,0,12,0,6,0,0,0,12,12,6,
  6,36,12,6,12,0,0,0,0,0,0,0,0,0,0,0,0,0,0,72,72,0,
  0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,
  0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,6,6,0,6,0,12,0,0,
  0,6,6,12,6,18,6,12,6,6,6,6,6,0,18,0,0,0,36,0,0,0,
  0,36,36,0,36,0,6,0,6,0,0,0,0,0,6,6,12,6,18,6,12,6,
  0,0,12,12,0,0,0,0,0,0,0,0,0,0,36,36,0,0,0,0,6,0,
  12,0,0,0,6,6,12,6,18,6,12,6,6,6,6,6,0,18,0,0,0,36,
  0,0,0,0,36,36,0,36,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,
  0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,12,0,0,
  0,6,6,12,6,18,6,12,6,6,6,6,6,0,18,0,0,0,36,0,0,0,
  0,36,36,0,36,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,
  0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,12,12,6,6,36,
  12,6,12,0,0,0,0,0,0,0,0,0,0,0,0,0,0,72,72,0,0,0,
  0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,
  0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,
  0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,
  0,0,0,0,0,0,0,0,0,0,0,0,6,0,6,0,6,0,6,0,0,12,12,
  0,18,0,0,0,36,0,0,0,0,0,0,0,36,0,0,6,0,6,0,6,0,0,
  12,12,0,18,0,0,0,36,0,0,0,0,0,0,0,36,0,6,0,0,6,0,
  0,0,0,0,0,36,0,0,0,72,0,0,0,0,0,0,0,72,0,0,6,6,6,
  12,12,12,12,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,
  0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,6,12,12,
  0,0,0,18,0,0,0,36,0,0,0,0,0,0,0,36,0,0,0,0,0,0,0,
  36,0,0,0,72,0,0,0,0,0,0,0,72,0,12,12,0,0,0,18,0,0,
  0,36,0,0,0,0,0,0,0,36,0,6,6,6,0,0,0,0,0,72,0,0,0,
  0,0,72,0,0,0,6,6,0,0,0,0,0,72,0,0,0,0,0,72,0,0,0,
  6,0,0,0,0,0,0,0,0,0,0,0,72,0,72,0,0,0,0,0,0,0,0,
  0,0,0,0,72,0,72,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,
  0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,
  0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,
  0,0,0,0,36,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,
  0,0,0,0,0,0,0,0,0,36,0,0,0,0,0,0,0,0,0,12,12,12,
  0,12,12,0,12,0,0,15,15,15,15,15,15,15,15,15,15,6,6,
  6,6,6,6,0,0,6,6,6,0,6,0,0,0,0,0,6,6,0,6,0,6,0,6,
  0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,12,12,0,12,
  12,0,12,0,0,60,60,60,60,60,60,60,60,60,60,12,12,12,
  12,12,12,0,0,12,12,12,24,12,48,0,0,108,0,12,12,0,12,
  0,12,24,12,0,48,0,0,0,0,0,0,0,0,0,0,72,0,0,0,0,0,
  72,12,0,12,12,0,12,0,96,60,60,60,60,60,60,60,60,60,
  60,12,12,12,12,12,12,24,24,12,12,12,0,12,0,0,0,0,0,
  12,12,0,12,0,12,0,12,0,0,0,0,0,0,108,0,72,0,0,0,0,
  0,0,0,0,0,72,0,12,12,0,12,0,0,60,60,60,60,60,60,60,
  60,60,60,12,12,12,12,12,12,0,0,12,12,12,0,12,24,0,0,
  36,0,12,12,0,12,0,12,0,12,0,24,0,0,0,0,0,0,36,0,0,
  0,36,0,0,0,0,0,72,96,0,0,0,0,0,0,1920,0,0,0,0,0,0,
  0,0,0,48,0,0,0,0,0,0,0,0,0,0,0,48,0,0,0,0,48,0,0,
  0,0,0,0,0,0,48,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,
  144,12,0,12,0,96,60,60,60,60,60,60,60,60,60,60,12,12,
  12,12,12,12,0,0,12,12,12,24,12,0,0,48,108,0,12,12,0,
  12,0,12,24,12,0,0,0,48,0,0,0,0,72,0,72,0,0,0,0,0,
  0,0,0,0,12,0,0,60,60,60,60,60,60,60,60,60,60,12,12,
  12,12,12,12,24,24,12,12,12,0,12,0,0,24,0,0,12,12,0,
  12,0,12,0,12,0,0,0,24,0,0,36,0,72,0,36,0,0,0,0,0,
  0,0,36,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,
  0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,
  0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,60,60,60,60,60,60,
  60,60,60,60,12,12,12,12,12,12,24,24,12,12,12,24,12,24,
  0,24,0,0,12,12,0,12,0,12,24,12,0,24,0,24,0,0,108,0,
  0,0,72,0,72,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,
  0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,
  0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,
  7680,0,0,0,0,0,0,0,48,0,0,0,0,0,0,0,0,0,0,0,0,0,
  48,0,0,48,0,0,0,0,0,0,0,0,0,0,48,0,0,0,0,0,0,0,0,
  144,0,0,0,0,0,0,15,15,15,15,15,15,15,15,15,30,30,30,
  30,30,30,0,0,30,30,30,0,30,60,0,0,810,0,30,30,0,30,
  0,30,0,30,0,60,0,0,0,0,0,0,1080,0,0,0,6480,0,0,0,0,
  0,1080,15,15,15,15,15,15,15,15,30,30,30,30,30,30,0,0,
  30,30,30,60,30,120,0,120,810,0,30,30,0,30,0,30,60,30,
  0,120,0,120,0,0,270,0,0,0,1080,0,1080,0,0,0,0,0,0,
  15,15,15,15,15,15,15,30,30,30,30,30,30,0,0,30,30,30,
  60,30,60,0,0,270,0,30,30,0,30,0,30,60,30,0,60,0,0,0,
  0,0,0,0,0,0,0,1080,0,0,0,0,0,0,15,15,15,15,15,15,
  30,30,30,30,30,30,60,60,30,30,30,0,30,0,0,60,0,0,30,
  30,0,30,0,30,0,30,0,0,0,60,0,0,0,0,1080,0,6480,0,0,
  0,0,0,0,0,1080,15,15,15,15,15,30,30,30,30,30,30,120,
  120,30,30,30,120,30,0,0,60,270,0,30,30,0,30,0,30,120,
  30,0,0,0,60,0,0,0,0,0,0,1080,0,0,0,0,0,0,0,0,15,
  15,15,15,30,30,30,30,30,30,30,30,30,30,30,30,30,30,0,
  30,0,0,30,30,0,30,0,30,30,30,0,30,0,30,0,0,270,0,0,
  0,6480,0,6480,0,0,0,0,0,0,15,15,15,30,30,30,30,30,30,
  0,0,30,30,30,0,30,120,0,60,270,0,30,30,0,30,0,30,0,
  30,0,120,0,60,0,0,810,0,6480,0,6480,0,1080,0,0,0,0,0,
  6480,15,15,30,30,30,30,30,30,60,60,30,30,30,0,30,0,0,
  0,0,0,30,30,0,30,0,30,0,30,0,0,0,0,0,0,270,0,1080,
  0,1080,0,0,0,0,0,0,0,1080,15,30,30,30,30,30,30,30,30,
  30,30,30,30,30,30,0,30,810,0,30,30,0,30,0,30,30,30,0,
  30,0,30,0,0,810,0,6480,0,1080,0,0,0,0,0,0,0,6480,30,
  30,30,30,30,30,0,0,30,30,30,0,30,0,0,60,0,0,30,30,0,
  30,0,30,0,30,0,0,0,60,0,0,0,0,0,0,1080,0,0,0,0,0,
  0,0,0,6,6,6,6,6,0,0,6,6,6,12,6,12,0,0,18,0,6,6,0,
  6,0,6,12,6,0,12,0,0,0,0,0,0,36,0,0,0,36,0,0,0,0,
  0,0,6,6,6,6,6,6,6,6,6,6,6,12,0,12,18,0,6,6,0,6,0,
  6,6,6,0,12,0,12,0,0,18,0,36,0,36,0,36,0,0,0,0,0,
  36,6,6,6,0,0,6,6,6,12,6,12,0,0,18,0,6,6,0,6,0,6,
  12,6,0,12,0,0,0,0,0,0,36,0,0,0,36,0,0,0,0,0,0,6,
  6,0,0,6,6,6,12,6,0,0,12,0,0,6,6,0,6,0,6,12,6,0,0,
  0,12,0,0,18,0,0,0,36,0,0,0,0,0,0,0,36,6,0,0,6,6,
  6,12,6,0,0,12,0,0,6,6,0,6,0,6,12,6,0,0,0,12,0,0,
  18,0,0,0,36,0,0,0,0,0,0,0,36,12,12,6,6,6,12,6,6,0,
  6,0,0,6,6,0,6,0,6,12,6,0,6,0,6,0,0,0,0,0,0,0,0,
  0,0,0,0,0,0,0,6,12,6,6,6,12,0,0,0,0,0,6,12,0,6,0,
  6,6,12,0,0,0,0,0,0,0,0,0,0,0,0,72,0,0,0,0,0,72,
  12,6,6,6,12,0,0,0,0,0,6,12,0,6,0,6,6,12,0,0,0,0,
  0,0,0,0,0,0,0,0,72,0,0,0,0,0,72,6,6,0,6,0,0,12,0,
  0,6,6,0,6,0,6,0,6,0,0,0,12,0,0,18,0,0,0,36,0,0,0,
  0,0,0,0,36,6,6,6,12,0,12,18,0,6,6,0,6,0,6,6,6,0,
  12,0,12,0,0,18,0,36,0,36,0,36,0,0,0,0,0,36,6,6,12,
  0,12,18,0,6,6,0,6,0,6,6,6,0,12,0,12,0,0,18,0,36,0,
  36,0,36,0,0,0,0,0,36,0,0,0,0,0,0,6,0,0,6,0,6,6,0,
  0,0,0,0,0,0,0,0,72,0,0,0,0,0,0,0,0,0,72,12,0,0,
  18,0,6,6,0,6,0,6,0,6,0,12,0,0,0,0,0,0,36,0,0,0,
  36,0,0,0,0,0,0,0,6,0,0,12,0,0,12,0,12,0,12,0,6,0,
  6,0,0,36,0,0,0,72,0,0,0,0,0,0,0,72,0,0,0,0,0,0,0,
  0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,
  36,0,12,12,0,12,0,12,0,0,0,6,0,6,0,0,0,0,72,0,0,0,
  72,0,0,0,0,0,0,0,18,0,0,18,0,18,0,18,0,0,0,36,0,0,
  0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,
  0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,6,0,6,0,6,6,6,0,
  12,0,12,0,0,18,0,36,0,36,0,36,0,0,0,0,0,36,0,6,0,
  6,0,6,0,0,0,12,0,0,18,0,0,0,36,0,0,0,0,0,0,0,36,
  0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,
  0,6,6,6,0,12,0,12,0,0,18,0,36,0,36,0,36,0,0,0,0,0,
  36,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,
  6,6,0,12,0,12,0,0,18,0,36,0,36,0,36,0,0,0,0,0,36,
  0,0,0,0,0,0,0,0,0,72,0,0,0,0,0,0,0,0,0,72,0,12,0,
  0,0,0,0,0,36,0,0,0,36,0,0,0,0,0,0,0,0,0,0,0,0,0,
  0,0,0,0,0,0,0,0,0,0,0,0,6,0,0,36,0,0,0,72,0,0,0,
  0,0,0,0,72,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,
  0,0,72,0,0,0,72,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,
  0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,
  0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,36,0,0,0,
  0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,36,0,0,0,
  0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,
  0,12,12,0,12,12,0,12,0,0,60,60,60,60,60,60,60,60,60,
  60,12,12,12,12,12,12,0,0,12,12,12,48,12,48,0,0,0,0,
  12,12,48,12,0,12,0,12,48,0,0,0,0,0,0,0,0,0,0,0,0,
  0,0,0,0,0,0,12,0,12,12,0,12,0,0,60,60,60,60,60,60,
  60,60,60,60,12,12,12,12,12,12,0,0,12,12,12,0,12,0,0,
  0,0,0,12,12,0,12,0,12,0,12,0,0,0,0,0,0,0,0,0,0,0,
  0,0,0,0,0,0,0,0,24,12,12,24,12,12,12,60,60,60,60,60,
  60,60,60,60,60,12,12,12,12,12,12,0,24,12,12,12,0,12,
  24,0,0,36,36,12,12,24,12,36,12,0,12,0,0,24,0,36,36,
  36,36,36,72,72,72,36,36,36,36,72,72,72,0,12,12,0,0,0,
  0,0,60,0,60,0,0,60,0,60,0,0,12,0,12,0,0,24,0,12,24,
  0,12,0,0,0,0,0,0,12,0,0,0,12,0,0,0,0,24,0,0,0,0,
  0,0,0,0,0,0,0,0,0,0,0,0,12,0,12,0,0,60,60,60,60,
  60,60,60,60,60,60,12,12,12,12,12,12,0,48,12,12,12,0,
  12,0,48,48,0,0,12,12,0,12,0,12,0,12,0,0,48,0,0,0,0,
  0,0,0,0,0,0,0,0,0,0,0,0,12,12,24,24,60,60,60,60,60,
  60,60,60,60,60,12,12,12,12,12,12,0,0,12,12,12,24,12,
  0,24,24,36,36,12,12,0,12,36,12,0,12,24,0,0,0,36,36,
  36,72,72,36,36,36,72,72,72,72,36,36,36,0,0,0,0,960,
  60,0,60,0,0,60,0,60,0,0,12,0,12,0,0,0,0,12,24,0,12,
  0,0,0,0,0,0,12,0,0,0,12,0,0,0,0,0,0,0,0,0,0,0,0,
  0,0,0,0,0,0,0,0,0,0,0,60,60,60,60,60,60,60,60,60,
  60,12,12,12,12,12,12,0,24,12,12,12,24,12,24,24,24,0,
  108,12,12,24,12,0,12,0,12,24,0,24,0,108,0,0,0,0,0,0,
  72,0,72,0,0,0,0,0,12,0,0,60,0,60,0,0,60,0,60,0,0,
  12,0,12,0,0,0,0,12,24,24,12,0,0,0,0,0,0,12,0,0,0,
  12,0,0,24,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,
  0,60,960,60,0,0,60,0,60,0,0,12,0,12,0,0,0,0,12,24,
  0,12,0,0,0,0,0,0,12,0,0,0,12,0,0,0,0,0,0,0,0,0,0,
  0,0,0,0,0,0,0,0,0,0,0,15,15,15,15,15,15,15,15,15,
  30,30,30,30,30,30,0,0,30,30,30,60,30,60,0,0,0,0,30,
  30,60,30,0,30,0,30,60,0,0,0,0,0,0,0,38880,0,0,0,0,
  0,0,38880,0,38880,38880,15,15,15,15,15,15,15,15,30,30,
  30,30,30,30,0,0,30,30,30,120,30,120,0,0,0,0,30,30,
  120,30,0,30,0,30,120,0,0,0,0,0,0,0,0,0,0,0,38880,0,
  38880,0,0,0,0,15,15,15,15,15,15,15,30,30,30,30,30,30,
  0,0,30,30,30,120,30,60,0,0,90,90,30,30,60,30,90,30,0,
  30,120,0,0,0,90,90,90,180,180,180,180,180,180,180,180,
  180,180,180,180,15,15,15,15,15,15,30,30,30,30,30,30,0,
  120,30,30,30,0,30,0,120,120,0,0,30,30,0,30,0,30,0,30,
  0,0,120,0,0,0,0,0,0,38880,38880,0,0,0,0,0,0,0,0,15,
  15,15,15,15,30,30,30,30,30,30,0,60,30,30,30,60,30,0,
  0,0,90,90,30,30,0,30,90,30,0,30,60,0,60,0,90,90,90,
  180,180,180,180,180,180,180,180,180,180,180,180,15,15,15,
  15,30,30,30,30,30,30,0,30,30,30,30,30,30,30,30,30,0,
  0,30,30,30,30,0,30,0,30,30,0,30,0,0,0,0,6480,0,0,0,
  6480,0,0,0,0,1080,0,0,15,15,15,30,30,30,30,30,30,0,
  60,30,30,30,0,30,0,60,60,0,0,30,30,0,30,0,30,0,30,0,
  0,60,0,0,0,0,0,38880,0,0,0,0,0,0,38880,0,38880,38880,
  15,15,30,30,30,30,30,30,0,120,30,30,30,0,30,0,60,60,
  90,90,30,30,0,30,90,30,0,30,0,0,120,0,90,90,90,180,
  180,180,180,180,180,180,180,180,180,180,180,15,30,30,30,
  30,30,30,0,30,30,30,30,30,30,30,30,30,0,0,30,30,30,
  30,0,30,0,30,30,0,30,0,0,0,0,1080,0,0,0,0,0,6480,0,
  0,6480,0,0,30,30,30,30,30,30,0,0,30,30,30,0,30,60,60,
  60,90,90,30,30,60,30,90,30,0,30,0,0,0,0,90,90,90,180,
  180,180,180,180,180,180,180,180,180,180,180,6,6,6,6,6,
  0,0,6,6,6,12,6,12,0,0,18,0,6,6,12,6,18,6,0,6,12,0,
  0,0,18,0,0,0,36,0,0,36,36,0,36,36,36,0,0,6,6,6,6,
  0,0,6,6,6,12,6,12,0,0,18,0,6,6,12,6,18,6,0,6,12,0,
  0,0,18,0,0,0,36,0,0,36,36,0,36,36,36,0,0,6,6,6,0,
  12,6,6,6,12,6,6,6,6,18,18,6,6,6,6,18,6,0,6,12,0,
  12,0,18,18,18,36,36,36,36,36,36,36,36,36,36,36,36,6,
  6,0,12,6,6,6,12,6,6,6,6,0,18,6,6,6,6,0,6,0,6,12,
  0,12,0,18,0,0,36,0,0,0,36,0,36,0,0,36,0,0,6,0,0,6,
  6,6,12,6,0,12,12,18,18,6,6,0,6,18,6,0,6,12,0,0,0,
  18,18,18,36,36,36,36,36,36,36,36,36,36,36,36,0,12,6,
  6,6,12,6,6,6,6,0,18,6,6,6,6,0,6,0,6,12,0,12,0,18,
  0,0,36,0,0,0,36,0,36,0,0,36,0,0,0,0,0,0,0,0,0,0,
  0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,
  0,0,0,0,0,0,12,12,6,6,12,0,0,0,36,0,12,12,0,12,36,
  0,0,12,6,0,6,0,36,0,0,0,72,0,0,72,72,0,72,72,72,72,
  72,6,6,0,6,0,12,12,0,18,6,6,0,6,0,6,0,6,0,0,12,0,
  0,18,18,36,0,36,36,0,0,36,0,0,0,36,36,6,0,6,12,0,0,
  18,18,6,6,12,6,18,6,0,6,0,0,12,0,18,18,18,36,36,36,
  36,36,36,36,36,36,36,36,36,6,6,12,12,12,18,0,6,6,12,
  6,18,6,0,6,6,0,6,0,0,18,18,0,36,36,36,0,36,0,36,36,
  0,36,36,0,0,0,0,0,36,0,12,0,12,0,12,0,12,6,0,6,0,
  0,36,36,72,72,72,72,0,0,72,0,72,0,72,72,12,0,0,18,
  18,6,6,12,6,18,6,0,6,0,0,12,0,18,18,18,36,36,36,36,
  36,36,36,36,36,36,36,36,6,6,0,36,0,6,6,6,0,0,0,6,0,
  0,0,0,0,36,36,72,0,72,72,0,0,72,0,0,0,72,72,6,36,0,
  12,6,6,6,36,12,0,6,0,0,0,0,36,0,0,0,72,0,0,72,72,
  0,72,72,72,0,0,36,0,12,6,6,6,36,12,0,6,0,0,0,0,36,
  0,0,0,72,0,0,72,72,0,72,72,72,0,0,18,0,18,0,0,18,
  18,0,0,0,0,36,0,0,0,0,36,0,0,0,0,0,36,0,0,0,0,0,
  18,18,36,18,18,18,0,18,36,0,0,0,0,0,0,0,36,0,0,0,
  36,0,36,36,0,0,0,6,0,6,0,6,0,6,0,0,12,0,0,18,18,
  36,0,36,36,0,0,36,0,0,0,36,36,6,6,18,6,0,6,12,0,12,
  0,18,18,18,36,36,36,36,36,36,36,36,36,36,36,36,6,0,0,
  0,6,0,0,0,0,0,36,36,72,0,72,72,0,0,72,0,0,0,72,72,
  0,6,0,6,12,0,12,0,18,0,0,36,0,0,0,36,0,36,0,0,36,
  0,0,18,0,0,0,0,36,0,0,0,0,36,0,0,0,0,0,36,0,0,0,
  0,0,0,6,12,0,0,0,18,18,18,36,36,36,36,36,36,36,36,
  36,36,36,36,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,
  0,0,12,0,12,0,18,0,0,36,0,0,0,36,0,36,0,0,36,0,0,
  0,6,0,0,36,36,72,72,72,72,0,0,72,0,72,0,72,72,0,0,
  0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,36,0,0,0,72,0,0,
  72,72,0,72,72,72,72,72,0,0,0,0,0,0,0,0,0,0,0,0,0,
  0,0,18,18,0,0,36,36,0,0,0,0,0,0,36,36,18,0,0,0,0,
  36,0,0,0,0,36,0,0,0,0,0,0,36,0,0,0,0,36,0,0,36,0,
  0,0,36,36,36,36,0,0,0,0,0,0,36,36,36,36,0,0,0,36,
  36,0,0,0,0,36,36,36,36,0,0,0,0,36,36,36,0,0,0,0,36,
  36,36,36,36,36,0,0,0,36,36,0,0,0,36,0,0,0,0,0,0,36,
  36,36,4,0,4,4,8,4,0,0,20,20,20,20,20,20,20,20,20,20,
  12,12,12,12,12,12,0,0,12,12,12,0,12,0,0,0,0,0,12,12,
  48,12,0,12,0,12,0,48,0,0,0,0,0,0,0,0,0,0,0,0,0,0,
  24,24,24,0,4,4,8,4,0,0,20,20,20,20,20,20,20,20,20,
  20,12,12,12,12,12,12,0,0,12,12,12,0,12,0,0,0,0,0,12,
  12,48,12,0,12,0,12,0,48,0,0,0,0,0,0,0,0,0,0,0,0,0,
  0,24,24,24,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,
  0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,
  0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,4,0,4,0,0,
  20,20,20,20,20,20,20,20,20,20,12,12,12,12,12,12,0,0,
  12,12,12,24,12,24,24,0,0,0,12,12,0,12,0,12,24,12,0,
  0,24,24,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,4,8,0,20,
  20,20,20,20,20,20,20,20,20,12,12,12,12,12,12,0,0,12,
  12,12,24,12,0,24,0,0,0,12,12,0,12,0,12,0,12,48,0,0,
  24,0,0,0,0,0,0,0,0,24,24,24,0,0,0,0,0,0,0,0,160,0,
  160,0,0,0,160,0,40,48,0,24,48,0,0,0,0,48,24,48,96,
  24,96,0,0,0,0,0,96,96,48,0,24,48,48,0,96,96,96,0,0,
  0,0,0,0,0,0,0,0,0,0,0,0,0,8,0,20,20,20,20,20,20,
  20,20,20,20,12,12,12,12,12,12,0,0,12,12,12,24,12,0,
  24,0,0,0,12,12,0,12,0,12,0,12,48,0,0,24,0,0,0,0,0,
  0,0,0,24,24,24,0,0,0,0,0,160,0,0,80,160,160,0,80,80,
  40,48,0,0,48,24,0,0,0,48,96,48,0,24,96,96,0,0,0,0,
  24,0,48,0,24,96,48,96,96,0,0,0,0,0,0,0,0,0,0,0,0,
  0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,
  0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,
  0,0,0,0,0,0,0,0,0,0,0,0,20,20,20,20,20,20,20,20,20,
  60,60,60,60,60,60,0,0,60,60,60,480,60,240,240,0,4860,
  0,60,60,0,60,0,60,0,60,960,960,240,0,0,1620,0,0,0,0,
  0,0,4320,720,4320,0,4320,4320,720,20,20,20,20,20,20,20,
  20,60,60,60,60,60,60,0,0,60,60,60,240,60,240,0,0,0,0,
  60,60,960,60,0,60,240,60,0,0,0,0,0,0,0,0,0,0,0,0,
  720,0,4320,0,4320,720,0,20,20,20,20,20,20,20,60,60,60,
  60,60,60,0,0,60,60,60,240,60,240,0,0,0,0,60,60,0,60,
  0,60,0,60,0,960,0,0,0,0,0,0,0,0,0,0,4320,0,720,0,
  720,720,4320,20,20,20,20,20,20,60,60,60,60,60,60,0,0,
  60,60,60,0,60,0,0,0,0,0,60,60,0,60,0,60,240,60,960,
  0,240,240,0,0,0,0,0,0,0,0,0,720,720,0,720,0,720,20,
  20,20,20,20,60,60,60,60,60,60,0,0,60,60,60,0,60,0,0,
  0,0,0,60,60,960,60,0,60,240,60,960,960,480,960,0,0,0,
  0,0,0,0,0,0,4320,4320,0,720,4320,0,20,20,20,20,60,60,
  60,60,60,60,0,0,60,60,60,480,60,0,0,0,0,0,60,60,960,
  60,0,60,480,60,480,0,480,480,0,4860,0,0,0,0,0,0,4320,
  0,720,0,720,4320,0,20,20,20,60,60,60,60,60,60,0,0,60,
  60,60,0,60,240,240,0,1620,0,60,60,0,60,0,60,0,60,0,
  960,0,480,0,4860,0,0,0,0,0,0,720,4320,0,0,0,720,4320,
  20,20,60,60,60,60,60,60,0,0,60,60,60,0,60,480,960,0,
  0,0,60,60,960,60,0,60,0,60,0,0,480,480,0,0,0,0,0,0,
  0,0,4320,720,0,0,4320,0,4320,20,60,60,60,60,60,60,0,0,
  60,60,60,0,60,480,480,0,4860,0,60,60,0,60,0,60,0,60,
  960,0,240,480,0,0,0,0,0,0,0,0,0,4320,4320,0,4320,0,
  4320,60,60,60,60,60,60,0,0,60,60,60,0,60,0,0,0,180,0,
  60,60,0,60,0,60,0,60,0,0,0,960,0,180,0,0,0,0,0,0,
  120,120,120,0,120,120,120,12,12,12,12,12,0,0,12,12,12,
  24,12,24,24,0,108,0,12,12,48,12,0,12,24,12,48,48,24,
  24,0,108,0,0,0,0,0,0,72,0,72,0,72,72,0,12,12,12,12,
  0,0,12,12,12,24,12,48,0,0,108,0,12,12,48,12,0,12,24,
  12,48,48,0,0,0,0,0,0,0,0,0,0,72,0,72,0,72,0,72,12,
  12,12,0,0,12,12,12,24,12,24,0,0,0,0,12,12,48,12,0,
  12,24,12,48,48,0,0,0,0,0,0,0,0,0,0,0,0,72,0,24,24,
  24,12,12,0,0,12,12,12,24,12,24,24,0,0,0,12,12,48,12,
  0,12,24,12,48,0,24,24,0,0,0,0,0,0,0,0,0,72,72,0,72,
  0,72,12,0,0,12,12,12,24,12,0,48,0,0,0,12,12,48,12,0,
  12,24,12,48,48,48,24,0,0,0,0,0,0,0,0,24,24,24,0,72,
  0,0,0,0,12,12,12,24,12,0,24,0,0,0,12,12,48,12,0,12,
  24,12,48,48,24,48,0,108,0,0,0,0,0,0,0,72,72,0,72,72,
  0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,
  0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,
  0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,
  0,0,0,0,12,12,0,12,0,0,0,0,0,12,12,48,12,0,12,48,
  12,0,0,24,24,0,108,0,0,0,0,0,0,0,72,72,0,0,72,72,
  12,0,12,0,0,0,0,0,12,12,48,12,0,12,24,12,0,48,24,24,
  0,0,0,0,0,0,0,0,0,72,0,0,24,24,24,0,12,24,24,0,108,
  0,12,12,0,12,0,12,0,12,48,48,48,48,0,108,0,0,0,0,0,
  0,72,0,72,0,0,72,72,0,0,0,0,0,0,24,48,48,24,0,0,24,
  0,48,192,96,0,0,216,0,0,0,0,0,0,0,144,0,0,0,144,144,
  24,24,0,108,0,12,12,0,12,0,12,0,12,48,48,48,48,0,36,
  0,0,0,0,0,0,72,72,72,0,24,24,24,24,0,0,0,24,48,48,
  24,0,24,96,24,0,48,96,96,0,432,0,0,0,0,0,0,0,144,
  144,0,144,288,0,0,0,0,24,24,48,48,0,24,96,24,0,48,0,
  0,0,0,0,0,0,0,0,0,0,0,144,0,144,0,144,0,0,0,0,0,0,
  0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,
  108,0,432,108,0,36,432,108,0,0,216,432,0,0,0,0,0,0,0,
  0,0,72,72,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,
  0,0,0,0,0,0,0,0,0,0,0,0,12,0,12,0,12,24,12,0,48,
  24,24,0,108,0,0,0,0,0,0,72,72,0,0,0,72,72,48,12,0,
  12,0,12,48,0,24,24,0,0,0,0,0,0,0,0,24,24,24,0,0,0,
  72,48,0,0,48,0,0,192,0,192,0,432,0,0,0,0,0,0,144,
  144,0,0,0,288,0,0,12,0,12,0,0,24,24,0,0,0,0,0,0,0,
  0,72,72,0,0,72,0,72,0,0,0,0,0,0,0,0,0,0,0,0,0,0,
  0,0,0,0,0,0,0,0,0,12,48,48,0,48,0,108,0,0,0,0,0,0,
  24,24,24,0,72,72,72,0,48,0,96,0,0,0,0,0,0,0,0,0,
  144,144,0,0,288,0,144,48,48,48,0,0,108,0,0,0,0,0,0,
  72,72,0,0,72,72,0,0,48,48,0,0,0,0,0,0,0,0,864,288,
  0,0,0,144,144,48,48,0,0,0,0,0,0,0,0,0,144,144,0,864,
  0,288,24,0,0,0,0,0,0,0,0,144,0,144,0,432,144,288,0,
  0,0,0,0,0,0,0,144,0,0,0,144,144,0,0,0,0,0,0,0,0,0,
  0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,72,0,72,0,0,0,0,
  0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,
  0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,
  0,0,0,0,0,24,24,0,0,0,0,24,0,0,0,0,0,0,0,0,0,0,0,
  24,24,24,0,4,4,8,4,0,0,20,20,20,20,20,20,20,20,20,
  20,12,12,12,12,12,12,0,0,12,12,12,0,12,0,0,0,0,0,12,
  12,48,12,0,12,0,12,0,48,0,0,0,0,0,0,0,0,0,0,0,0,0,
  0,24,24,24,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,
  0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,
  0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,4,0,4,0,8,
  20,20,20,20,20,20,20,20,20,20,12,12,12,12,12,12,48,0,
  12,12,12,0,12,0,0,48,0,0,12,12,0,12,0,12,0,12,0,0,
  0,0,0,0,0,24,24,0,0,0,0,0,0,24,0,0,0,0,4,0,8,20,
  20,20,20,20,20,20,20,20,20,12,12,12,12,12,12,48,0,12,
  12,12,0,12,0,0,48,0,0,12,12,0,12,0,12,0,12,0,0,0,0,
  0,0,0,24,24,0,0,0,0,0,0,24,0,0,0,0,0,0,0,160,160,
  160,0,0,0,0,0,40,48,0,24,48,96,0,96,48,48,24,48,0,
  24,0,0,96,0,0,0,0,96,48,0,24,0,48,0,0,0,0,0,0,0,0,
  0,0,0,0,0,0,0,0,0,0,0,0,0,20,20,20,20,20,20,20,20,
  20,20,12,12,12,12,12,12,24,24,12,12,12,0,12,0,0,24,0,
  0,12,12,24,12,0,12,0,12,24,24,0,0,0,0,0,0,0,0,0,0,
  0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,
  0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,
  0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,160,0,160,0,0,0,
  160,0,40,0,48,0,48,24,48,0,0,0,96,48,0,24,0,0,96,0,
  0,48,24,96,0,0,24,0,48,48,96,0,0,0,0,0,0,0,0,0,0,
  0,0,0,0,0,0,0,20,20,20,20,20,20,20,20,20,60,60,60,
  60,60,60,0,0,60,60,60,0,60,0,0,0,0,0,60,60,240,60,
  4860,60,0,60,0,480,0,0,0,0,0,0,720,0,0,0,0,0,0,4320,
  4320,4320,720,20,20,20,20,20,20,20,20,60,60,60,60,60,
  60,0,240,60,60,60,0,60,0,0,240,0,0,60,60,480,60,0,60,
  0,60,240,240,0,0,4860,0,1620,720,0,0,0,0,0,0,0,720,
  4320,720,0,20,20,20,20,20,20,20,60,60,60,60,60,60,0,0,
  60,60,60,0,60,0,0,0,0,0,60,60,480,60,1620,60,0,60,0,
  240,0,0,1620,0,0,0,4320,0,0,0,0,0,0,720,720,4320,0,
  20,20,20,20,20,20,60,60,60,60,60,60,240,240,60,60,60,
  0,60,0,0,480,0,4860,60,60,240,60,1620,60,0,60,240,0,0,
  0,0,0,0,4320,720,0,0,0,0,0,0,0,720,0,720,20,20,20,
  20,20,60,60,60,60,60,60,0,240,60,60,60,0,60,0,0,240,
  0,0,60,60,240,60,1620,60,0,60,240,0,0,0,0,0,1620,4320,
  0,0,0,0,0,0,0,4320,4320,0,4320,20,20,20,20,60,60,60,
  60,60,60,0,0,60,60,60,0,60,0,0,480,0,0,60,60,240,60,
  0,60,0,60,480,480,0,0,0,0,4860,4320,0,0,0,0,0,0,0,
  4320,720,4320,0,20,20,20,60,60,60,60,60,60,480,0,60,60,
  60,0,60,0,0,240,0,0,60,60,0,60,0,60,0,60,0,0,0,0,0,
  0,4860,4320,4320,0,0,0,0,0,0,720,0,720,4320,20,20,60,
  60,60,60,60,60,240,0,60,60,60,0,60,0,0,480,0,1620,60,
  60,0,60,0,60,0,60,0,0,0,0,0,0,1620,720,4320,0,0,0,0,
  0,0,0,0,4320,720,20,60,60,60,60,60,60,480,480,60,60,
  60,0,60,0,0,240,0,0,60,60,480,60,4860,60,0,60,0,0,0,
  0,0,0,0,720,4320,0,0,0,0,0,0,0,4320,0,4320,60,60,60,
  60,60,60,0,0,60,60,60,0,60,0,0,960,0,180,60,60,960,
  60,180,60,0,60,0,0,0,0,180,0,180,120,120,0,0,0,0,0,
  0,120,120,120,120,12,12,12,12,12,0,0,12,12,12,0,12,0,
  0,0,0,0,12,12,0,12,108,12,0,12,24,24,0,0,108,0,0,0,
  72,0,0,0,0,0,0,72,72,72,0,12,12,12,12,24,24,12,12,
  12,0,12,0,0,24,0,0,12,12,24,12,108,12,0,12,24,24,0,
  0,108,0,108,0,72,0,0,0,0,0,0,72,72,0,72,12,12,12,0,
  0,12,12,12,0,12,0,0,0,0,0,12,12,24,12,0,12,0,12,24,
  24,0,0,36,0,0,0,72,0,0,0,0,0,0,0,24,24,24,12,12,0,
  48,12,12,12,0,12,0,0,24,0,108,12,12,24,12,108,12,0,
  12,48,0,0,0,108,0,108,72,0,0,0,0,0,0,0,72,72,0,72,
  12,48,0,12,12,12,0,12,0,0,24,0,0,12,12,24,12,36,12,
  0,12,24,0,0,0,0,0,108,24,24,0,0,0,0,0,0,24,0,0,72,
  24,24,12,12,12,0,12,0,0,0,0,108,12,12,48,12,108,12,0,
  12,0,0,0,0,108,0,0,72,0,0,0,0,0,0,0,72,72,72,0,24,
  24,0,24,0,24,0,0,0,0,0,24,24,0,0,432,0,0,24,96,0,0,
  0,216,0,0,0,0,0,0,0,0,0,0,144,144,0,144,24,24,0,0,
  24,0,0,0,0,0,24,24,0,0,216,0,0,24,48,96,0,0,0,0,0,
  0,144,0,0,0,0,0,0,144,432,288,144,12,12,0,12,0,0,24,
  0,108,12,12,24,12,108,12,0,12,24,24,0,0,0,0,108,72,0,
  0,0,0,0,0,0,72,0,72,72,12,0,12,0,0,24,0,0,12,12,24,
  12,108,12,0,12,0,48,0,0,0,0,36,0,0,0,0,0,0,0,0,72,
  24,24,24,0,12,0,0,24,0,0,12,12,24,12,108,12,0,12,0,
  24,0,0,0,0,108,0,72,0,0,0,0,0,0,72,0,72,72,0,0,0,
  0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,
  0,0,0,0,0,0,0,0,0,0,108,12,12,48,12,108,12,0,12,0,
  0,0,0,36,0,36,72,72,0,0,0,0,0,0,72,24,24,24,0,0,0,
  0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,
  0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,
  0,0,0,0,0,0,0,0,0,0,0,0,0,24,24,48,0,48,0,24,0,0,
  0,0,0,0,0,0,144,0,0,0,0,0,0,0,144,144,0,0,0,0,0,0,
  0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,
  108,36,0,108,36,36,0,108,0,216,0,0,0,0,0,0,72,0,0,0,
  0,0,0,72,0,0,0,12,0,12,0,12,0,12,0,0,0,0,0,0,108,
  72,72,0,0,0,0,0,0,0,0,72,72,0,12,0,12,0,12,0,0,0,
  0,0,0,0,24,24,0,0,0,0,0,0,24,0,72,0,0,0,0,0,24,0,
  0,0,0,0,0,0,144,144,0,0,0,0,0,0,0,0,144,0,0,12,0,
  12,24,24,0,0,108,0,108,72,72,0,0,0,0,0,0,0,72,0,72,
  36,0,0,0,0,0,0,0,0,0,72,72,0,0,0,0,0,0,0,0,0,0,0,
  12,24,24,0,0,108,0,108,24,24,0,0,0,0,0,0,24,72,72,
  72,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,24,24,
  0,0,108,0,0,72,72,0,0,0,0,0,0,0,72,72,0,24,0,0,0,
  0,216,432,288,0,0,0,0,0,0,144,0,144,144,0,0,0,0,432,
  144,0,0,0,0,0,0,0,144,0,0,144,0,0,0,0,0,0,0,0,0,0,
  0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,36,
  0,0,0,0,0,0,0,0,0,0,72,72,0,0,0,0,0,0,0,0,0,0,0,
  0,0,0,0,0,0,0,0,0,0,0,72,72,0,24,0,0,0,0,0,0,24,
  0,0,0,0,0,0,0,0,0,24,0,0,0,0,0,0,0,0,0,0,0,0,0,
  0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,
  0,0,0,0,0,0,0,0,24,24,24,8,8,8,8,0,0,0,0,40,0,40,
  80,160,40,0,40,0,0,24,0,24,0,0,48,0,24,0,0,24,48,0,
  0,0,0,48,24,0,0,0,24,48,0,0,48,48,0,0,0,0,0,0,0,0,
  0,0,0,0,0,24,24,24,4,0,4,0,0,20,20,20,20,20,20,20,
  20,20,20,12,12,12,12,12,12,0,48,12,12,12,0,12,24,0,0,
  0,0,12,12,0,12,0,12,24,12,0,0,24,0,0,0,0,0,0,24,24,
  24,0,0,0,0,0,0,0,8,4,8,8,20,20,20,20,20,20,20,20,
  20,20,12,12,12,12,12,12,0,48,12,12,12,0,12,0,0,0,12,
  12,12,12,0,12,12,12,0,12,48,0,0,0,12,12,12,24,24,24,
  24,24,24,24,24,24,24,24,24,0,0,0,0,0,40,0,40,0,0,40,
  0,40,48,0,24,0,24,0,0,0,0,24,0,0,24,48,0,0,0,0,0,
  24,48,0,0,24,0,0,0,48,0,0,0,0,0,0,0,24,24,24,0,0,
  0,0,0,0,0,0,0,20,20,20,20,20,20,20,20,20,20,12,12,
  12,12,12,12,0,48,12,12,12,0,12,24,0,0,0,0,12,12,0,
  12,0,12,24,12,0,0,24,0,0,0,0,0,0,24,24,24,0,0,0,0,
  0,0,0,4,0,0,20,80,20,0,0,20,0,20,0,0,12,24,12,0,0,
  0,0,12,24,0,12,0,0,0,0,0,0,12,0,0,0,12,48,24,24,48,
  0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,20,80,20,0,
  0,20,0,20,0,0,12,24,12,0,0,0,0,12,24,0,12,0,0,0,0,
  0,0,12,0,0,0,12,48,24,24,48,0,0,0,0,0,0,0,0,0,0,0,
  0,0,0,0,0,0,20,20,20,20,20,20,20,20,20,60,60,60,60,
  60,60,0,0,60,60,60,0,60,480,0,0,1620,0,60,60,480,60,
  1620,60,240,60,0,240,0,0,0,0,0,0,2160,0,0,0,2160,0,0,
  0,4320,4320,0,20,20,20,20,20,20,20,20,60,60,60,60,60,
  60,0,0,60,60,60,0,60,0,0,0,0,0,60,60,240,60,0,60,
  480,60,0,240,0,0,0,0,0,0,0,0,0,0,0,0,2160,2160,4320,
  0,0,20,20,20,20,20,20,20,60,60,60,60,60,60,0,0,60,60,
  60,0,60,240,0,0,60,60,60,60,240,60,60,60,240,60,240,
  240,0,0,60,60,60,60,60,120,120,120,60,60,60,60,120,120,
  120,20,20,20,20,20,20,60,60,60,60,60,60,0,480,60,60,
  60,0,60,0,0,0,0,1620,60,60,240,60,1620,60,0,60,480,0,
  480,0,0,0,0,2160,0,0,0,0,0,2160,0,0,0,0,0,20,20,20,
  20,20,60,60,60,60,60,60,0,240,60,60,60,0,60,0,0,0,60,
  60,60,60,240,60,60,60,240,60,240,240,240,0,60,60,60,60,
  60,120,120,120,60,60,60,60,120,120,120,20,20,20,20,60,
  60,60,60,60,60,0,240,60,60,60,0,60,240,0,0,0,0,60,60,
  240,60,0,60,480,60,0,0,0,0,0,1620,1620,0,0,0,0,0,0,
  0,2160,2160,4320,0,0,20,20,20,60,60,60,60,60,60,0,240,
  60,60,60,0,60,480,0,0,1620,0,60,60,0,60,0,60,0,60,0,
  240,240,0,0,0,0,0,2160,0,0,0,2160,0,0,0,0,0,4320,20,
  20,60,60,60,60,60,60,0,240,60,60,60,0,60,240,0,0,60,
  60,60,60,240,60,60,60,0,60,0,0,240,0,60,60,60,60,60,
  120,120,120,60,60,60,60,120,120,120,20,60,60,60,60,60,
  60,0,480,60,60,60,0,60,0,0,0,0,0,60,60,0,60,0,60,
  240,60,0,240,240,0,0,0,0,2160,0,0,720,0,0,2160,0,0,0,
  0,0,60,60,60,60,60,60,0,240,60,60,60,0,60,240,0,0,60,
  60,60,60,240,60,60,60,240,60,0,0,0,0,60,60,60,60,60,
  120,120,120,60,60,60,60,120,120,120,12,12,12,12,12,0,0,
  12,12,12,0,12,24,0,0,36,0,12,12,24,12,36,12,24,12,24,
  24,0,0,108,0,0,0,36,0,0,72,36,0,36,36,0,72,0,12,12,
  12,12,0,0,12,12,12,0,12,24,0,0,36,0,12,12,24,12,36,
  12,24,12,24,24,0,0,0,0,0,0,36,0,0,72,36,0,36,36,0,
  0,72,12,12,12,0,24,12,12,12,0,12,24,0,0,36,36,12,12,
  24,12,36,12,24,12,0,24,24,0,12,12,12,36,36,24,24,24,
  36,36,36,36,24,24,24,12,12,0,24,12,12,12,0,12,0,0,0,
  0,36,12,12,24,12,36,12,24,12,24,0,24,0,108,0,108,36,
  0,0,72,0,0,36,36,36,72,0,0,12,0,0,12,12,12,0,12,0,
  0,0,12,12,12,12,24,12,12,12,24,12,24,24,24,0,36,36,
  36,12,12,72,72,72,12,12,12,12,72,72,72,0,24,12,12,12,
  0,12,0,0,0,0,36,12,12,24,12,36,12,24,12,24,0,24,0,0,
  108,0,36,0,72,0,0,0,36,36,36,72,0,0,0,0,0,0,0,0,0,
  0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,
  0,0,0,0,0,0,0,24,24,0,0,24,0,0,0,72,0,24,24,48,24,
  72,24,24,24,48,48,24,0,0,0,0,0,72,0,0,432,72,0,72,
  72,144,144,144,12,12,0,12,0,0,0,0,36,12,12,24,12,36,
  12,24,12,24,24,24,0,0,108,0,36,0,0,72,0,0,36,36,36,
  0,72,0,12,0,12,24,0,0,36,36,12,12,24,12,36,12,24,12,
  0,24,24,0,12,12,12,36,36,24,24,24,36,36,36,36,24,24,
  24,0,12,24,0,0,36,0,12,12,24,12,36,12,24,12,24,24,24,
  0,0,0,108,0,36,72,0,0,36,0,36,36,0,0,72,0,0,0,0,0,
  0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,
  0,0,0,0,24,0,0,36,36,12,12,24,12,36,12,24,12,0,24,
  24,0,12,12,12,36,36,24,24,24,36,36,36,36,24,24,24,0,
  0,0,72,24,24,24,24,72,24,48,24,48,24,48,0,0,432,432,
  72,0,0,432,0,0,72,72,72,144,144,144,0,0,0,0,0,0,0,0,
  0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,
  0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,
  0,0,0,12,36,12,72,36,12,12,72,36,72,0,72,0,0,0,0,12,
  0,0,0,0,0,12,12,12,0,0,0,36,12,72,36,12,12,72,36,72,
  72,0,0,0,0,0,0,12,0,0,0,12,0,12,12,0,0,0,12,0,12,
  0,12,0,12,0,24,24,0,0,0,108,36,36,72,0,0,36,36,0,0,
  0,0,72,24,12,12,12,0,12,24,0,24,0,36,36,36,12,12,72,
  72,72,12,12,12,12,72,72,72,0,0,0,24,0,0,48,0,0,0,0,
  0,72,72,144,144,144,72,72,0,0,0,0,432,0,12,0,12,0,24,
  24,0,0,0,0,36,36,0,0,72,36,36,0,0,0,0,72,12,0,0,0,
  72,72,0,0,0,0,12,12,0,0,0,12,12,0,0,0,0,0,24,12,24,
  24,0,0,36,36,36,12,12,72,72,72,12,12,12,12,72,72,72,
  0,0,0,48,0,0,0,0,72,72,432,0,0,72,72,0,0,144,144,
  144,0,24,24,0,108,0,0,36,36,0,0,72,36,36,0,0,0,72,0,
  24,24,0,0,0,0,72,72,0,0,0,72,72,0,0,0,0,0,24,0,0,
  0,432,72,0,144,144,144,0,72,72,72,0,0,0,0,432,0,0,0,
  72,0,0,0,72,0,72,72,144,144,144,0,0,0,0,0,0,0,0,0,
  0,0,0,0,0,0,12,12,0,0,24,0,0,0,0,0,0,0,0,24,12,0,
  0,0,0,24,0,0,0,0,0,0,24,0,0,0,0,24,0,0,0,0,0,24,
  0,12,0,0,0,12,12,12,12,0,0,0,0,0,0,12,12,12,12,0,0,
  0,24,24,0,0,0,0,24,24,24,24,0,0,0,0,24,24,24,0,0,0,
  0,24,24,24,12,12,12,0,0,0,12,12,0,0,0,12,0,0,0,0,0,
  0,24,24,24,0,0,8,0,0,0,80,80,0,160,80,160,0,160,40,
  0,48,24,48,96,48,96,96,0,24,48,0,24,96,96,0,0,0,48,
  0,0,0,0,24,0,48,0,0,0,96,0,0,0,0,0,0,0,0,0,0,0,0,
  0,0,0,4,0,0,0,0,80,20,0,20,0,0,20,0,20,0,0,12,24,
  12,0,48,24,0,12,24,0,12,0,0,0,0,0,0,12,0,0,0,12,0,
  24,0,0,0,48,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,
  0,80,20,0,20,0,0,20,0,20,0,0,12,24,12,0,48,24,0,12,
  24,0,12,0,0,0,0,0,0,12,0,0,0,12,0,24,0,0,0,48,0,0,
  0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,80,160,0,0,80,160,
  0,160,40,0,48,24,48,0,48,96,0,0,24,48,0,24,0,96,0,0,
  0,48,96,0,0,0,24,96,48,96,48,96,96,0,0,0,0,0,0,0,0,
  0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,
  0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,
  0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,
  0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,
  0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,
  0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,
  0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,
  0,0,0,0,0,0,0,0,0,80,80,0,0,480,0,480,240,480,0,480,
  0,0,480,0,0,480,0,0,0,0,480,480,480,0,0,0,0,0,3840,
  480,0,0,0,0,0,0,0,0,0,0,0,0,0,0,2160,360,0,0,20,0,
  80,20,80,20,0,0,60,240,60,240,0,0,0,60,0,0,60,240,0,
  0,0,0,240,60,0,0,0,60,0,0,0,480,0,0,0,0,0,0,0,0,0,
  0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,
  0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,
  0,0,0,0,0,0,0,0,0,0,0,0,0,20,0,20,0,240,60,0,60,0,
  0,240,0,60,240,0,60,0,0,0,0,0,240,60,240,0,0,60,240,
  0,960,240,240,960,0,0,0,0,0,0,0,0,0,0,0,0,0,180,0,
  80,40,0,80,0,480,480,480,0,1920,0,0,0,480,1920,0,0,0,
  0,0,0,0,480,240,960,0,0,480,1920,480,960,480,480,480,0,
  0,0,0,0,0,0,0,0,0,0,0,360,2160,0,0,0,80,0,960,0,
  1920,480,3840,30720,960,0,960,480,0,480,480,0,0,0,0,0,
  0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,
  720,2160,80,20,0,480,60,240,60,0,960,240,0,60,0,0,60,
  240,0,0,0,0,0,60,480,0,0,60,0,0,0,0,0,3840,0,0,0,0,
  0,0,0,0,0,0,0,0,0,540,540,0,0,960,480,480,0,1920,0,
  1920,0,480,480,0,0,1920,960,0,0,0,480,1920,0,0,0,480,
  0,1920,960,960,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,
  2160,0,0,60,0,60,480,0,240,0,60,0,0,60,240,0,0,0,0,
  0,60,0,0,0,60,240,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,
  0,0,60,60,60,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,
  0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,
  0,0,0,24,24,48,48,48,0,24,24,0,24,24,0,0,0,0,48,24,
  24,0,0,0,0,0,96,48,0,0,0,0,0,0,0,0,0,0,0,0,0,0,
  72,0,72,24,12,24,0,24,0,12,24,0,12,24,0,0,0,0,24,12,
  0,0,0,12,24,0,48,24,24,0,0,0,0,0,0,0,0,0,0,0,0,0,
  12,12,12,24,24,0,24,0,0,48,0,24,48,48,0,0,0,0,0,24,
  0,0,24,24,48,0,0,48,96,0,0,0,0,0,0,0,0,0,0,0,0,72,
  0,72,0,48,0,0,12,0,0,12,0,48,0,0,0,24,12,24,0,0,12,
  24,24,48,24,24,48,0,0,0,0,0,0,0,0,0,0,0,0,0,36,0,
  48,24,0,24,0,0,0,0,96,0,0,0,48,24,48,0,0,24,24,24,
  96,48,48,48,0,0,0,0,0,0,0,0,0,0,0,0,72,72,0,48,0,
  0,96,0,48,96,0,0,0,0,0,0,48,0,0,0,0,0,0,0,0,0,0,
  0,0,0,0,0,0,0,0,0,0,0,144,0,144,0,24,0,0,24,0,96,
  0,0,0,48,24,48,0,0,24,24,24,96,48,24,48,0,0,0,0,0,
  0,0,0,0,0,0,0,0,72,72,0,0,0,0,0,0,0,0,0,0,0,0,0,
  0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,
  24,0,12,24,0,0,0,0,0,12,24,0,0,12,24,24,0,0,24,48,
  0,0,0,0,0,0,0,0,0,0,0,0,12,12,12,0,0,24,48,0,0,0,
  24,24,24,0,0,24,48,48,48,24,48,96,0,0,0,0,0,0,0,0,
  0,0,0,0,0,72,72,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,
  0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,24,48,0,0,0,24,
  12,24,0,0,12,24,24,48,24,24,48,0,0,0,0,0,0,0,0,0,0,
  0,0,12,12,12,48,0,0,0,24,24,24,0,0,24,48,48,48,48,
  48,96,0,0,0,0,0,0,0,0,0,0,0,0,72,72,0,0,0,0,0,48,
  0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,144,
  0,144,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,
  0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,
  0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,
  0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,24,0,24,0,
  24,24,48,0,0,0,0,0,0,0,0,0,0,0,0,0,72,72,24,0,0,
  12,0,24,0,0,24,48,0,0,0,0,0,0,0,0,0,0,0,0,0,0,36,
  0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,
  72,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,
  0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,
  24,0,48,24,0,0,0,0,0,0,0,0,0,0,0,0,0,0,36,36,36,0,
  0,0,48,0,0,0,0,0,0,0,0,0,0,0,0,0,0,72,72,48,24,0,
  0,0,0,0,0,0,0,0,0,0,0,0,0,72,72,0,0,48,48,0,0,0,
  0,0,0,0,0,0,0,0,0,0,144,144,24,48,0,0,0,0,0,0,0,0,
  0,0,0,0,0,0,72,0,0,0,0,0,0,0,0,0,0,0,0,0,72,0,0,
  0,0,0,0,0,0,0,0,0,0,0,0,144,144,0,0,0,0,0,0,0,0,
  0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,
  0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,
  0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,
  0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,
  12,12,12,0,4,0,8,20,20,20,20,20,20,20,20,20,20,12,12,
  12,12,12,12,48,0,12,12,12,0,12,0,0,48,0,0,12,12,0,
  12,0,12,0,12,0,0,0,0,0,0,0,24,24,0,0,0,0,0,0,24,0,
  0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,
  0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,
  0,0,0,0,0,0,0,0,0,0,0,0,0,0,20,20,20,20,20,20,20,
  20,20,20,12,12,12,12,12,12,0,48,12,12,12,0,12,24,0,0,
  0,0,12,12,0,12,0,12,24,12,0,0,24,0,0,0,0,0,0,24,24,
  24,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,
  0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,
  0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,160,160,160,0,
  0,0,0,0,40,0,48,96,48,24,48,96,0,0,0,48,96,24,0,96,
  96,0,0,48,24,0,0,0,24,96,48,0,0,96,48,0,0,0,0,0,0,
  0,0,0,0,0,0,0,0,0,20,20,20,20,20,20,20,20,20,60,60,
  60,60,60,60,960,0,60,60,60,0,60,240,240,0,4860,0,60,
  60,0,60,0,60,480,60,0,0,0,0,0,1620,0,0,720,720,0,
  4320,0,0,0,4320,0,0,0,20,20,20,20,20,20,20,20,60,60,
  60,60,60,60,0,960,60,60,60,240,60,0,0,0,0,0,60,60,0,
  60,0,60,240,60,0,0,0,240,0,0,0,720,0,0,720,720,0,0,
  0,720,0,0,0,20,20,20,20,20,20,20,60,60,60,60,60,60,0,
  0,60,60,60,480,60,960,480,960,0,0,60,60,0,60,0,60,480,
  60,0,0,0,0,0,0,0,4320,0,4320,0,720,0,0,0,4320,0,0,0,
  20,20,20,20,20,20,60,60,60,60,60,60,0,0,60,60,60,0,
  60,0,240,960,0,0,60,60,0,60,0,60,0,60,0,0,240,240,0,
  0,0,4320,720,720,4320,0,0,0,0,0,0,0,0,20,20,20,20,20,
  60,60,60,60,60,60,960,960,60,60,60,480,60,0,0,960,0,0,
  60,60,0,60,0,60,960,60,0,0,0,240,0,0,0,720,4320,0,
  4320,4320,0,0,0,0,0,0,0,20,20,20,20,60,60,60,60,60,
  60,0,960,60,60,60,240,60,480,480,0,0,0,60,60,0,60,0,
  60,480,60,0,0,0,0,0,4860,0,4320,0,0,4320,4320,0,0,0,
  4320,0,0,0,20,20,20,60,60,60,60,60,60,960,960,60,60,
  60,240,60,240,240,0,1620,0,60,60,0,60,0,60,0,60,0,0,
  480,0,0,4860,0,4320,4320,4320,4320,720,0,0,0,720,0,0,0,
  20,20,60,60,60,60,60,60,960,0,60,60,60,0,60,0,240,0,
  0,0,60,60,0,60,0,60,0,60,0,0,240,0,0,0,0,720,720,
  4320,720,0,0,0,0,4320,0,0,0,20,60,60,60,60,60,60,0,
  480,60,60,60,480,60,0,0,960,4860,0,60,60,0,60,0,60,
  480,60,0,0,480,480,0,0,0,720,4320,4320,720,0,0,0,0,0,
  0,0,0,60,60,60,60,60,60,0,0,60,60,60,0,60,0,0,0,180,
  0,60,60,0,60,0,60,960,60,0,0,0,0,0,180,0,120,120,120,
  120,120,0,0,0,120,0,0,0,12,12,12,12,12,48,0,12,12,12,
  24,12,24,24,0,108,0,12,12,0,12,0,12,24,12,0,0,24,24,
  0,108,0,0,72,72,0,72,0,0,0,72,0,0,0,12,12,12,12,0,
  0,12,12,12,24,12,0,0,48,108,0,12,12,0,12,0,12,24,12,
  0,0,0,48,0,0,0,0,72,0,72,72,0,0,0,72,0,0,0,12,12,
  12,0,48,12,12,12,24,12,24,48,48,0,0,12,12,0,12,0,12,
  24,12,0,0,48,0,0,0,0,0,0,24,24,24,0,0,0,72,0,0,0,
  12,12,0,48,12,12,12,24,12,24,24,48,0,0,12,12,0,12,0,
  12,24,12,0,0,24,24,0,0,0,72,0,0,72,72,0,0,0,72,0,0,
  0,12,48,0,12,12,12,24,12,0,0,48,0,0,12,12,0,12,0,12,
  24,12,0,0,0,24,0,0,0,24,24,0,0,72,0,0,0,24,0,0,0,
  0,0,12,12,12,24,12,48,24,48,0,0,12,12,0,12,0,12,24,
  12,0,0,24,0,0,108,0,72,0,72,0,72,0,0,0,72,0,0,0,0,
  48,48,48,48,48,48,48,192,0,0,48,48,0,48,0,48,48,48,0,
  0,192,0,0,0,0,864,0,0,144,144,0,0,0,288,0,0,0,48,48,
  48,48,48,0,0,0,0,0,48,48,0,48,0,48,48,48,0,0,48,48,
  0,0,0,0,144,864,0,288,0,0,0,144,0,0,0,12,12,0,12,0,
  48,48,0,0,12,12,0,12,0,12,0,12,0,0,24,24,0,108,0,72,
  0,72,72,0,0,0,0,72,0,0,0,12,48,12,48,0,48,0,0,12,
  12,0,12,0,12,24,12,0,0,24,24,0,0,0,72,0,24,24,24,0,
  0,0,0,0,0,0,48,12,24,24,0,108,0,12,12,0,12,0,12,48,
  12,0,0,0,0,0,108,0,0,72,72,72,0,0,0,0,72,0,0,0,0,
  0,96,0,0,0,24,0,0,24,0,48,24,48,0,0,96,96,0,216,0,
  432,144,144,144,0,0,0,0,288,0,0,0,24,24,0,108,0,12,
  12,0,12,0,12,48,12,0,0,0,0,0,36,0,72,72,24,24,24,0,
  0,0,72,0,0,0,24,48,0,0,24,0,0,24,0,24,0,24,0,0,0,
  96,0,0,0,144,0,0,144,0,0,0,0,144,0,0,0,48,432,0,24,
  24,0,0,0,24,96,24,0,0,0,96,0,0,0,144,288,0,144,144,
  0,0,0,0,0,0,0,432,0,48,48,0,48,0,0,192,0,0,0,48,48,
  0,432,0,0,288,144,0,144,0,0,0,0,0,0,0,0,108,0,0,108,
  0,36,0,108,0,0,216,0,0,0,0,72,0,0,0,0,0,0,0,72,0,
  0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,
  0,0,0,0,0,12,0,12,0,12,24,12,0,0,24,24,0,108,0,72,
  72,72,72,0,0,0,0,0,0,0,0,0,12,0,12,0,12,0,0,24,24,
  0,0,0,24,24,0,72,0,0,0,0,24,0,0,0,0,0,0,0,0,0,0,
  0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,12,48,12,0,0,
  24,24,0,0,0,72,72,0,72,72,0,0,0,0,0,0,0,0,0,0,0,0,
  0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,48,12,0,0,0,0,
  0,108,0,24,24,72,72,72,0,0,0,24,0,0,0,0,0,0,0,0,0,
  432,0,144,144,144,0,0,0,0,0,0,0,0,0,0,0,0,0,0,108,
  0,72,72,72,0,72,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,
  0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,
  0,0,24,0,0,0,0,144,0,0,144,0,0,0,144,0,0,0,0,432,0,
  288,0,144,0,144,0,0,0,144,0,0,0,0,0,0,0,0,0,0,0,0,
  0,0,0,0,0,0,0,0,0,72,72,0,0,0,0,0,0,0,0,0,0,0,0,
  0,0,0,0,0,0,0,24,0,0,0,0,0,0,24,0,0,0,0,0,0,0,0,
  0,24,0,0,0,24,24,0,0,0,0,0,0,0,24,0,0,0,0,0,0,0,
  0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,
  0,0,0,0,0,0,0,0,80,20,0,20,0,0,20,0,20,0,0,12,24,
  12,0,48,24,0,12,24,0,12,0,0,0,0,0,0,12,0,0,0,12,0,
  24,0,0,0,48,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,8,0,20,
  20,20,20,20,20,20,20,20,20,12,12,12,12,12,12,0,0,12,
  12,12,24,12,0,24,0,0,0,12,12,0,12,0,12,0,12,48,0,0,
  24,0,0,0,0,0,0,0,0,24,24,24,0,0,0,0,8,160,0,40,0,
  40,0,0,40,80,40,48,0,24,0,24,0,48,0,0,24,0,48,24,0,
  48,0,0,0,0,24,0,0,0,24,0,0,48,0,0,48,0,0,0,24,24,
  0,0,0,0,0,0,24,0,0,0,0,0,40,0,40,0,0,40,0,40,0,0,
  24,0,24,0,48,0,0,24,0,0,24,0,48,48,0,0,48,24,0,0,0,
  24,0,0,0,0,0,0,0,0,0,0,0,0,0,0,24,24,24,0,0,0,0,
  20,20,20,20,20,20,20,20,20,60,60,60,60,60,60,240,0,60,
  60,60,240,60,0,480,0,0,0,60,60,0,60,0,60,0,60,240,0,
  0,0,0,1620,0,0,0,2160,0,0,0,0,0,4320,0,2160,0,20,20,
  20,20,20,20,20,20,60,60,60,60,60,60,0,480,60,60,60,
  480,60,0,0,240,0,0,60,60,0,60,0,60,0,60,480,0,0,0,
  1620,0,1620,0,0,0,0,2160,0,0,0,0,2160,0,0,20,20,20,
  20,20,20,20,60,60,60,60,60,60,0,0,60,60,60,240,60,0,
  240,240,60,60,60,60,0,60,60,60,0,60,240,0,0,0,60,60,
  60,120,120,60,60,60,120,120,120,120,60,60,60,20,20,20,
  20,20,20,60,60,60,60,60,60,240,0,60,60,60,0,60,0,0,
  240,0,0,60,60,0,60,0,60,0,60,0,0,0,480,0,0,0,4320,0,
  0,2160,0,0,0,0,0,0,0,2160,20,20,20,20,20,60,60,60,60,
  60,60,240,240,60,60,60,240,60,0,0,240,60,60,60,60,0,
  60,60,60,0,60,240,0,0,240,60,60,60,120,120,60,60,60,
  120,120,120,120,60,60,60,20,20,20,20,60,60,60,60,60,60,
  240,0,60,60,60,240,60,0,0,0,0,0,60,60,0,60,0,60,0,
  60,480,0,0,240,0,0,0,0,0,0,0,2160,0,0,720,0,2160,0,
  0,20,20,20,60,60,60,60,60,60,240,0,60,60,60,0,60,0,
  480,480,0,0,60,60,0,60,0,60,0,60,0,0,0,240,0,1620,
  1620,4320,4320,2160,0,0,0,0,0,0,0,2160,0,20,20,60,60,
  60,60,60,60,240,240,60,60,60,0,60,0,240,240,60,60,60,
  60,0,60,60,60,0,60,0,0,0,240,60,60,60,120,120,60,60,
  60,120,120,120,120,60,60,60,20,60,60,60,60,60,60,0,0,
  60,60,60,0,60,0,240,240,1620,0,60,60,0,60,1620,60,0,
  60,240,0,0,480,0,0,0,4320,0,0,2160,0,0,0,0,0,0,0,
  2160,60,60,60,60,60,60,0,0,60,60,60,0,60,0,240,240,60,
  60,60,60,0,60,60,60,0,60,240,0,0,240,60,60,60,120,120,
  60,60,60,120,120,120,120,60,60,60,12,12,12,12,12,24,0,
  12,12,12,24,12,0,24,0,0,0,12,12,0,12,108,12,0,12,24,
  0,0,0,36,36,0,0,0,36,0,36,72,0,0,72,36,36,0,12,12,
  12,12,24,24,12,12,12,24,12,0,0,24,108,0,12,12,0,12,0,
  12,0,12,24,0,0,24,36,0,36,0,72,0,36,36,0,0,72,0,36,
  0,36,12,12,12,0,24,12,12,12,24,12,0,24,24,36,36,12,
  12,0,12,36,12,0,12,24,0,0,0,12,12,12,72,72,12,12,12,
  72,72,72,72,12,12,12,12,12,0,24,12,12,12,24,12,0,0,
  24,0,108,12,12,0,12,108,12,0,12,24,0,0,24,36,0,36,72,
  0,0,36,36,0,0,72,0,36,0,36,12,24,0,12,12,12,24,12,0,
  24,24,12,12,12,12,0,12,12,12,0,12,24,0,0,24,36,36,36,
  24,24,36,36,36,24,24,24,24,36,36,36,24,0,12,12,12,24,
  12,0,24,0,0,0,12,12,0,12,0,12,0,12,24,0,0,0,36,36,
  0,0,0,36,0,36,0,72,0,72,36,36,0,24,24,24,24,24,24,0,
  24,48,0,0,24,24,0,0,432,24,0,24,48,0,0,0,72,0,72,0,
  0,0,72,72,144,144,144,0,72,0,72,24,24,24,24,24,0,48,
  0,0,0,24,0,0,24,0,0,0,0,48,0,0,0,72,72,0,0,0,72,0,
  72,0,0,0,0,72,72,0,12,12,0,12,0,24,24,0,0,12,12,0,
  12,0,12,0,12,0,0,0,24,0,36,36,0,0,36,36,0,0,72,0,
  72,0,36,36,12,24,12,0,0,24,36,36,12,12,0,12,36,12,0,
  12,0,0,0,24,12,12,12,72,72,12,12,12,72,72,72,72,12,
  12,12,24,12,0,24,24,0,0,12,12,0,12,108,12,0,12,0,0,
  0,24,0,36,36,0,0,36,36,0,72,0,0,72,0,36,36,0,0,48,
  0,0,432,0,24,0,24,0,24,0,24,24,0,0,48,0,72,72,144,
  144,72,72,0,0,0,0,144,0,72,72,0,24,0,36,36,12,12,0,
  12,36,12,0,12,24,0,0,24,12,12,12,72,72,12,12,12,72,
  72,72,72,12,12,12,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,
  0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,24,432,0,24,24,0,0,
  432,24,0,24,0,0,0,48,72,0,72,144,144,0,72,72,0,0,432,
  144,72,0,72,0,0,24,24,0,24,0,24,0,0,48,0,0,24,72,72,
  0,0,0,72,0,72,144,144,144,432,72,72,0,12,0,12,0,108,
  12,12,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,24,0,24,0,0,
  0,108,12,0,0,12,12,0,108,0,0,0,0,0,0,0,0,0,0,0,0,
  24,0,0,24,0,0,0,12,0,12,0,12,0,12,0,0,0,24,0,36,36,
  0,72,36,36,0,0,72,0,0,0,36,36,0,12,12,12,0,12,24,0,
  0,24,36,36,36,24,24,36,36,36,24,24,24,24,36,36,36,0,
  0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,
  12,0,12,24,0,0,24,36,0,36,72,0,0,36,36,72,0,0,0,36,
  0,36,12,0,0,0,0,0,0,0,0,0,0,24,0,0,0,0,24,0,0,0,
  0,0,0,12,24,0,0,24,36,36,36,24,24,36,36,36,24,24,24,
  24,36,36,36,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,
  0,0,24,0,0,0,36,36,0,0,72,36,0,36,0,72,0,0,36,36,0,
  0,0,24,0,72,72,144,144,72,72,0,0,432,0,144,0,72,72,0,
  0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,
  0,0,0,0,0,0,0,0,72,72,0,144,144,72,0,72,432,0,0,144,
  72,72,0,12,12,0,0,12,12,0,0,0,0,0,0,12,12,12,0,0,0,
  12,12,0,0,0,0,12,0,12,0,0,12,0,12,0,0,0,0,12,12,0,
  24,0,0,0,24,24,24,24,0,0,0,0,0,0,24,24,24,24,0,0,0,
  12,12,0,0,0,0,12,12,12,12,0,0,0,0,12,12,12,0,0,0,0,
  12,12,12,24,24,24,0,0,0,24,24,0,0,0,24,0,0,0,0,0,0,
  12,12,12,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,
  0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,
  0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,
  0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,
  0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,
  0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,
  0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,
  0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,
  0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,
  0,0,0,0,0,0,0,0,0,0,0,80,80,0,0,0,0,80,1920,0,960,
  0,0,0,0,0,480,1920,960,7680,480,1920,0,0,0,0,0,480,
  960,960,0,1920,480,960,0,1920,0,0,0,0,0,0,0,0,720,720,
  0,0,0,0,0,0,0,0,20,0,0,20,0,20,240,0,60,0,60,0,0,
  0,240,60,960,960,60,240,0,3840,0,0,0,60,240,0,0,60,
  240,0,0,240,0,0,0,0,0,0,0,0,540,0,0,0,0,0,0,0,0,
  80,0,0,0,0,80,960,0,0,480,480,0,7680,480,0,1920,960,0,
  0,0,0,7680,0,0,0,0,480,960,0,1920,0,0,0,0,1920,0,0,
  0,0,0,0,720,0,0,0,0,0,0,0,0,0,0,0,20,0,20,0,0,60,
  480,60,0,0,480,480,60,0,0,60,0,0,0,0,0,0,60,240,0,0,
  60,480,480,0,240,0,1920,0,0,0,0,0,540,0,540,0,0,0,0,
  0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,
  0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,
  0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,
  0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,
  0,0,0,0,0,20,480,0,60,240,60,0,960,240,0,60,0,0,60,
  480,0,3840,0,0,0,60,240,960,0,60,0,0,0,0,0,3840,0,0,
  0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,
  0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,
  0,0,0,0,0,0,0,0,0,0,0,240,0,60,0,60,0,0,0,0,60,0,
  0,60,240,0,0,0,0,0,60,240,0,0,60,0,480,0,0,0,7680,0,
  0,0,0,0,60,60,60,0,0,0,0,0,0,0,0,0,0,0,0,0,0,48,
  24,24,96,24,24,0,0,0,0,0,24,24,48,0,24,48,24,0,0,48,
  48,0,0,0,0,0,72,0,72,0,0,0,0,0,0,0,0,0,0,0,0,0,0,
  0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,
  0,0,0,0,0,0,0,0,0,0,0,0,0,12,0,0,0,24,12,24,48,12,
  0,0,0,0,0,0,12,24,24,0,12,24,24,0,24,24,0,0,0,0,0,
  0,12,12,12,0,0,0,0,0,0,0,0,0,0,24,24,24,48,48,24,0,
  0,0,0,0,0,24,48,24,0,24,24,48,0,0,24,0,0,0,0,0,0,
  0,72,72,0,0,0,0,0,0,0,0,0,0,24,12,24,48,12,0,0,0,
  0,0,0,12,24,24,0,12,24,24,0,24,24,48,0,0,0,0,0,0,0,
  36,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,
  0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,
  0,48,48,96,48,48,0,0,0,0,0,0,48,96,48,0,48,48,96,0,
  0,48,0,0,0,0,0,0,0,144,144,0,0,0,0,0,0,0,24,24,48,
  48,24,0,0,0,0,0,0,24,48,24,0,24,24,48,0,0,0,0,0,0,
  0,0,0,0,0,72,0,0,0,0,0,0,0,24,24,0,0,0,0,0,0,0,0,
  0,24,48,0,24,24,0,0,0,48,0,0,0,0,0,0,72,72,0,0,0,
  0,0,0,0,0,0,48,12,24,0,48,0,0,0,12,24,0,0,12,24,24,
  0,24,0,48,0,0,0,0,0,12,12,12,0,0,0,0,0,0,0,48,24,
  24,0,48,0,0,0,24,24,0,0,0,24,48,0,48,0,96,0,0,0,0,
  0,72,72,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,48,96,0,
  48,0,0,0,0,96,0,0,0,0,0,0,144,144,0,0,0,0,0,0,0,0,
  24,0,0,0,0,0,12,24,24,0,12,0,0,0,24,24,0,0,0,0,0,
  0,12,12,12,0,0,0,0,0,0,0,0,0,0,0,0,24,24,48,0,24,
  0,24,0,0,48,48,0,0,0,0,0,72,0,0,0,0,0,0,0,0,0,0,
  0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,
  0,0,0,0,0,0,0,0,48,48,96,0,48,0,48,0,0,96,0,0,0,0,
  0,0,144,0,144,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,
  0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,
  0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,
  0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,
  24,24,0,12,0,0,0,0,24,0,0,0,0,0,0,36,0,0,0,0,0,0,
  0,0,0,0,0,0,48,48,0,48,0,96,0,0,0,0,0,0,72,72,0,0,
  0,0,0,0,0,0,0,48,24,0,24,24,48,0,0,0,0,0,0,72,72,
  0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,
  0,0,0,0,0,24,24,0,24,0,48,0,0,0,0,0,36,36,36,0,0,
  0,0,0,0,0,0,0,0,48,0,0,0,0,0,0,72,0,0,0,0,0,0,0,
  0,0,0,24,24,0,0,0,0,0,0,72,0,72,0,0,0,0,0,0,0,0,
  0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,24,0,0,0,0,0,
  0,72,72,0,0,0,0,0,0,0,0,48,0,0,0,0,0,0,72,0,0,0,
  0,0,0,0,0,0,0,0,0,0,144,0,144,0,0,0,0,0,0,0,0,0,
  0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,
  0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,
  0,0,0,0,0,0,0,0,0,0,12,12,0,0,0,0,0,0,0,12,0,0,0,
  0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,
  0,0,0,0,0,0,0,0,0,160,0,0,80,0,160,0,160,80,40,48,
  0,96,48,24,0,48,96,48,0,48,96,24,96,0,0,0,0,0,24,0,
  48,0,24,96,48,0,96,0,96,0,0,0,0,0,0,0,0,0,0,0,0,0,
  0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,
  0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,
  0,0,0,0,0,0,0,0,0,20,20,20,20,20,20,20,20,20,60,60,
  60,60,60,60,480,0,60,60,60,480,60,0,240,0,0,0,60,60,
  240,60,0,60,480,60,0,480,0,0,0,0,0,0,0,720,0,4320,
  4320,720,4320,0,0,0,0,20,20,20,20,20,20,20,20,60,60,
  60,60,60,60,0,480,60,60,60,240,60,0,0,0,0,4860,60,60,
  480,60,0,60,0,60,480,0,0,0,0,0,14580,0,0,0,720,720,
  720,0,4320,0,0,0,0,20,20,20,20,20,20,20,60,60,60,60,
  60,60,0,0,60,60,60,240,60,240,0,480,0,0,60,60,960,60,
  0,60,480,60,480,0,0,0,0,0,0,0,0,720,4320,4320,0,4320,
  0,0,0,0,0,20,20,20,20,20,20,60,60,60,60,60,60,0,480,
  60,60,60,0,60,0,0,480,0,0,60,60,0,60,14580,60,0,60,
  480,0,240,0,4860,0,0,0,0,720,4320,0,0,720,720,0,0,0,
  0,20,20,20,20,20,60,60,60,60,60,60,480,0,60,60,60,0,
  60,0,0,0,0,0,60,60,0,60,0,60,240,60,0,480,0,240,0,0,
  0,0,0,4320,0,720,4320,720,0,0,0,0,0,20,20,20,20,60,
  60,60,60,60,60,120,120,60,60,60,120,60,120,120,120,0,0,
  60,60,120,60,0,60,120,60,120,120,120,120,0,0,1620,0,0,
  0,4320,4320,4320,0,720,0,0,0,0,20,20,20,60,60,60,60,
  60,60,480,0,60,60,60,0,60,240,0,240,0,0,60,60,0,60,0,
  60,0,60,0,480,480,480,0,0,0,0,0,4320,4320,720,720,4320,
  0,0,0,0,0,20,20,60,60,60,60,60,60,0,480,60,60,60,0,
  60,0,240,960,0,0,60,60,480,60,0,60,0,60,0,0,240,480,
  0,0,0,0,0,0,0,4320,720,4320,4320,0,0,0,0,20,60,60,60,
  60,60,60,120,120,60,60,60,120,60,120,120,120,0,0,60,60,
  120,60,1620,60,120,60,120,120,120,120,0,0,0,0,0,4320,
  720,0,0,4320,4320,0,0,0,0,60,60,60,60,60,60,0,0,60,
  60,60,0,60,480,480,960,0,180,60,60,960,60,180,60,480,
  60,0,0,0,480,180,0,180,0,0,120,120,120,120,120,120,0,
  0,0,0,12,12,12,12,12,24,0,12,12,12,0,12,0,24,0,0,0,
  12,12,24,12,108,12,24,12,24,24,24,24,108,0,0,0,0,72,
  0,72,72,0,72,0,0,0,0,12,12,12,12,24,24,12,12,12,24,
  12,24,0,24,0,0,12,12,0,12,108,12,24,12,0,24,0,24,108,
  0,108,0,0,0,72,72,72,0,72,0,0,0,0,12,12,12,0,24,12,
  12,12,24,12,24,24,24,0,108,12,12,24,12,0,12,48,12,24,
  0,24,0,36,0,0,0,0,24,24,24,0,72,0,0,0,0,0,12,12,0,
  24,12,12,12,24,12,24,24,24,0,108,12,12,24,12,108,12,0,
  12,24,0,24,0,108,0,108,0,0,0,72,72,0,72,72,0,0,0,0,
  12,24,0,12,12,12,48,12,0,24,0,0,0,12,12,24,12,36,12,
  24,12,24,24,24,24,0,0,0,0,0,72,0,0,24,24,24,0,0,0,
  0,24,24,12,12,12,24,12,24,24,24,0,108,12,12,24,12,108,
  12,24,12,24,24,24,24,108,0,0,0,0,72,0,72,0,72,72,0,
  0,0,0,24,24,24,24,24,24,48,24,48,0,216,24,0,24,24,0,
  24,24,48,48,0,48,0,216,0,432,0,0,0,144,144,0,144,864,
  0,0,0,0,0,24,24,24,0,0,48,0,0,0,24,24,48,24,216,24,
  24,24,24,48,24,48,0,0,0,0,0,144,0,0,144,0,144,0,0,0,
  0,12,12,0,12,0,24,0,0,108,12,12,24,12,108,12,24,12,
  24,24,24,24,0,0,108,0,0,72,72,0,0,72,72,0,0,0,0,12,
  24,12,24,0,24,0,0,12,12,0,12,0,12,24,12,0,24,48,24,
  0,0,36,0,0,24,24,24,72,0,0,0,0,0,0,24,12,24,24,24,
  0,0,12,12,24,12,108,12,24,12,24,24,24,24,0,0,108,0,0,
  72,72,0,72,0,72,0,0,0,0,0,0,48,0,0,0,24,24,48,24,
  216,24,24,24,24,48,24,48,0,0,0,0,0,144,144,0,0,0,144,
  0,0,0,0,0,24,0,0,108,12,12,24,12,108,12,24,12,24,24,
  24,24,36,0,36,0,0,24,24,24,72,72,72,0,0,0,0,24,24,0,
  0,24,24,24,24,216,24,48,24,48,24,48,24,0,0,0,0,0,144,
  0,0,0,144,144,0,0,0,0,24,0,0,0,24,24,24,0,0,24,24,
  0,48,0,48,0,0,216,0,0,0,144,144,144,0,0,0,0,0,0,0,
  0,24,24,24,24,216,24,48,24,48,24,48,24,432,0,0,0,0,
  144,0,144,864,0,144,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,
  0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,108,36,432,108,
  36,36,216,108,0,216,0,216,0,0,0,0,0,0,0,0,72,0,72,0,
  0,0,0,12,0,12,0,12,24,12,0,24,0,24,0,0,108,0,0,72,
  72,0,72,72,0,0,0,0,0,24,12,0,12,0,12,24,0,24,48,108,
  0,0,0,0,0,0,72,24,24,24,0,0,0,0,24,0,0,24,24,0,48,
  0,48,0,0,216,0,0,864,144,0,144,144,0,0,0,0,0,0,12,
  24,12,24,24,24,24,108,0,108,0,0,0,72,72,72,72,0,0,0,
  0,0,36,0,0,0,432,0,0,0,0,0,0,0,0,0,0,72,72,0,0,0,
  0,0,24,12,0,24,0,24,108,0,108,0,0,72,72,72,24,24,24,
  0,0,0,0,0,48,0,48,0,216,0,0,0,0,0,0,144,144,144,0,
  0,0,0,0,24,48,24,0,108,0,0,0,0,72,0,72,72,72,0,0,0,
  0,0,24,24,24,0,0,216,0,0,144,144,0,144,0,0,0,0,0,0,
  24,24,216,0,0,0,0,0,864,144,0,144,144,0,0,0,0,24,0,
  0,216,0,0,0,144,0,144,0,144,0,0,0,0,216,0,0,0,0,144,
  0,144,0,144,0,0,0,0,0,0,36,0,0,72,72,0,0,0,0,0,0,
  0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,72,0,72,0,0,0,
  0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,
  0,24,24,0,0,0,0,0,0,0,24,0,0,0,0,0,0,0,0,0,0,0,0,
  0,0,24,24,0,0,0,0,24,0,0,0,0,0,0,0,0,0,0,0,0,0,0,
  0,0,20,80,20,0,0,20,0,20,0,0,12,24,12,0,0,0,0,12,
  24,0,12,0,0,0,0,0,0,12,0,0,0,12,48,24,24,48,0,0,0,
  0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,80,80,80,
  0,0,0,1920,960,0,0,0,960,480,480,0,0,0,480,0,0,0,0,
  0,0,3840,0,480,0,0,960,30720,0,0,0,0,0,0,720,0,0,0,
  0,0,0,2160,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,
  0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,
  0,0,0,0,0,0,0,0,0,0,0,0,0,80,20,80,0,20,40,20,0,0,
  60,240,60,0,0,0,480,60,0,0,60,0,240,480,0,0,0,60,0,
  0,0,60,3840,0,240,960,0,0,0,0,0,0,540,0,0,0,0,0,0,
  540,0,0,0,0,80,0,0,0,0,480,0,480,480,0,0,480,3840,
  480,240,480,0,0,0,480,480,0,0,0,0,0,480,0,0,0,0,480,
  0,0,0,0,0,0,2160,360,0,0,0,0,0,0,0,0,0,0,0,0,20,0,
  20,240,0,60,0,60,0,240,960,240,60,240,240,60,0,0,240,
  0,0,0,60,0,0,0,60,960,0,240,0,0,240,0,0,0,0,180,0,
  0,0,0,0,0,0,0,0,0,0,80,0,0,480,0,1920,480,480,0,960,
  960,960,0,480,0,480,960,1920,0,0,0,0,480,0,1920,0,0,0,
  1920,1920,0,0,0,0,0,0,0,0,0,0,0,0,0,0,2160,0,0,0,0,
  0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,
  0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,
  20,240,0,60,240,60,0,480,0,0,60,0,0,60,0,240,0,0,0,
  0,60,0,240,0,60,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,
  0,0,0,0,0,80,480,0,240,480,480,0,480,960,480,0,1920,
  480,480,0,0,960,0,0,0,480,0,1920,0,0,480,480,0,0,0,
  1920,0,0,0,360,2160,0,0,0,0,0,0,0,0,0,0,0,0,60,0,
  60,0,0,0,0,60,0,0,60,0,240,0,0,0,0,60,0,480,0,60,0,
  0,240,0,0,240,0,0,0,60,60,0,0,0,0,0,0,60,0,0,0,0,
  0,0,0,0,24,0,48,24,24,24,24,0,24,0,0,0,0,24,0,48,0,
  24,48,24,48,0,0,0,0,0,0,0,72,0,0,0,0,0,0,72,0,0,0,
  0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,
  0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,12,0,0,
  0,24,12,24,24,12,48,24,24,0,0,0,12,0,24,0,12,48,24,
  24,0,0,0,0,0,0,0,0,0,0,0,0,0,0,36,0,0,0,0,0,0,0,
  24,24,48,48,24,48,48,24,0,0,0,24,0,24,0,24,96,48,24,
  0,0,24,0,0,0,72,0,0,0,0,0,0,0,72,0,0,0,0,0,0,24,
  12,24,24,12,0,24,24,0,0,0,12,0,24,0,12,48,24,24,0,0,
  24,0,0,0,12,12,0,0,0,0,0,0,12,0,0,0,0,0,0,0,0,0,
  0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,
  0,0,0,0,0,0,0,0,0,0,48,24,24,24,24,0,48,0,0,0,0,
  24,0,48,0,24,48,24,48,0,0,0,0,0,0,0,0,0,0,0,0,0,0,
  72,0,0,0,96,48,48,48,48,0,48,0,0,0,0,48,0,96,0,48,
  48,48,96,0,0,0,0,0,0,0,144,0,0,0,0,0,0,144,0,0,0,
  24,24,0,0,0,24,24,0,0,0,0,0,48,0,24,0,0,48,48,0,0,
  0,0,0,72,0,0,0,0,0,0,0,72,0,0,0,0,24,12,48,0,24,0,
  0,0,12,0,0,0,12,48,24,0,48,0,24,0,0,0,0,36,0,0,0,
  0,0,0,0,0,0,0,48,24,48,24,24,0,0,0,24,0,0,0,0,96,
  48,0,96,0,48,0,0,0,0,72,0,0,0,0,0,0,72,0,0,0,0,0,
  48,0,0,0,0,24,0,48,0,24,0,0,24,0,0,48,0,0,0,72,0,
  0,0,0,0,0,0,0,0,0,0,0,24,0,0,0,0,12,0,24,0,12,0,
  0,24,0,0,24,0,0,0,36,36,0,0,0,0,0,0,36,0,0,0,48,0,
  0,0,0,0,0,96,0,48,0,0,96,0,0,0,0,0,0,144,0,0,0,0,
  0,0,0,144,0,0,0,24,0,0,0,24,0,0,0,24,96,48,0,96,0,
  48,0,0,0,72,72,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,48,
  0,24,0,0,48,48,0,0,0,0,0,0,0,0,0,0,0,0,0,72,0,0,
  0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,
  0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,
  0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,
  0,0,0,0,0,0,0,0,0,0,0,24,0,12,0,0,24,0,0,24,0,0,
  0,12,12,0,0,0,0,0,0,12,0,0,0,0,0,0,0,0,0,0,0,0,0,
  0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,48,24,24,48,0,24,
  0,0,0,72,72,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,
  0,0,0,0,0,0,0,0,0,0,0,0,0,0,48,24,24,48,0,24,0,0,
  0,12,12,0,0,0,0,0,0,12,0,0,0,0,48,0,0,0,0,0,0,144,
  144,0,0,0,0,0,0,0,0,0,0,24,0,0,0,0,0,0,72,72,0,0,
  0,0,0,0,0,0,0,0,48,0,24,0,0,0,0,72,0,0,0,0,0,0,
  72,0,0,0,0,0,0,0,0,144,0,0,0,0,0,0,0,144,0,0,0,0,
  0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,72,0,0,0,
  0,0,0,72,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,
  0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,
  12,0,0,0,0,0,0,12,0,0,0,0,0,0,0,0,0,12,0,0,0,0,0,
  0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,
  0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,
  0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,
  0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,
  0,0,80,80,0,0,0,0,80,0,0,0,480,1920,960,0,0,0,480,
  960,1920,1920,0,0,480,0,0,960,0,7680,0,0,0,0,0,480,
  7680,0,0,0,0,0,0,0,0,0,0,720,0,0,0,0,0,0,0,20,0,0,
  20,0,20,0,0,60,240,60,960,0,0,0,60,0,0,60,0,480,240,
  0,0,480,60,3840,0,0,60,3840,0,240,960,0,0,0,0,0,0,0,
  0,0,0,0,0,0,0,0,0,0,80,0,0,0,0,80,0,480,480,0,1920,
  960,1920,0,0,0,960,0,1920,0,1920,960,0,0,1920,960,0,0,
  0,480,0,960,0,0,7680,480,0,0,0,0,0,0,0,0,0,720,720,
  0,0,0,0,0,0,20,0,20,0,480,60,480,60,0,240,0,0,60,0,
  0,60,0,0,240,0,0,0,60,0,0,0,60,1920,480,480,0,0,480,
  0,0,0,0,0,0,0,0,540,540,0,0,0,0,0,0,0,0,0,0,0,0,
  0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,
  0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,
  0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,
  0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,20,0,240,
  60,0,60,0,240,0,0,60,960,0,60,0,240,240,0,0,240,60,
  3840,0,0,60,0,0,0,0,960,240,0,0,0,0,0,0,0,0,0,0,
  540,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,
  0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,
  0,0,0,0,0,0,60,0,60,0,0,0,0,60,0,0,60,0,240,240,0,
  0,240,60,0,0,0,60,7680,480,0,0,0,0,0,0,0,0,0,0,0,0,
  60,60,60,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,
  0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,
  0,0,0,0,0,24,24,48,0,0,0,24,24,48,24,0,0,24,0,0,48,
  24,0,0,0,0,0,0,24,48,0,24,0,0,0,0,0,0,0,0,72,0,72,
  0,0,0,0,24,12,24,0,0,0,12,24,24,12,0,24,24,0,0,24,
  12,48,0,0,12,0,0,24,48,0,0,0,0,0,0,0,0,0,0,36,0,0,
  0,0,0,0,24,24,0,0,0,0,48,24,24,0,0,48,0,0,0,0,0,0,
  0,24,0,48,24,0,48,24,0,0,0,0,0,0,0,0,0,72,72,0,0,
  0,0,0,24,0,0,12,0,0,12,0,24,24,0,0,24,12,48,0,0,12,
  48,24,24,48,48,24,0,0,0,0,0,0,0,0,12,12,12,0,0,0,0,
  24,0,0,24,0,24,0,0,48,0,0,0,48,24,96,0,0,24,48,24,
  24,48,96,48,0,0,0,0,0,0,0,0,0,72,72,0,0,0,0,0,0,
  24,48,24,24,0,0,48,0,0,0,24,0,0,0,24,0,24,0,0,0,0,
  0,0,0,0,0,0,0,0,72,0,72,0,0,0,0,0,0,0,0,0,0,0,0,
  0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,
  0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,
  0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,24,24,12,0,0,
  24,0,0,0,12,0,0,0,12,48,24,0,0,48,24,0,0,0,0,0,0,
  0,0,0,36,0,0,0,0,0,0,0,0,24,24,0,0,24,24,48,0,0,
  24,96,48,48,96,48,24,0,0,0,0,0,0,0,0,72,0,72,0,0,0,
  0,0,0,48,0,0,0,48,24,96,0,0,24,48,24,0,48,96,48,0,
  0,0,0,0,0,0,0,0,0,72,0,0,0,0,0,24,0,0,0,24,12,48,
  0,0,12,48,24,24,48,48,24,0,0,0,0,0,0,0,0,36,36,36,
  0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,
  0,0,0,0,0,0,0,0,0,0,0,24,0,0,24,0,0,0,0,24,48,24,
  0,0,0,0,0,0,0,0,0,0,0,0,72,0,0,0,0,0,0,0,0,24,24,
  48,0,0,24,96,48,48,96,48,48,0,0,0,0,0,0,0,0,0,72,
  72,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,
  0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,
  0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,24,48,24,0,0,
  96,48,0,0,0,0,0,0,0,0,72,72,0,0,0,0,0,0,0,0,12,0,
  24,0,0,48,24,0,0,0,0,0,0,0,0,12,12,12,0,0,0,0,0,0,
  0,0,48,0,0,0,0,0,0,0,0,0,0,0,0,144,144,0,0,0,0,0,
  0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,
  0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,
  24,48,0,0,0,0,0,0,0,0,0,0,12,12,12,0,0,0,0,0,0,0,
  0,0,0,0,0,0,0,0,0,0,144,144,0,0,0,0,0,48,96,0,0,0,
  0,0,0,0,0,0,0,72,72,0,0,0,0,0,0,48,24,0,0,0,0,0,
  0,0,0,0,72,0,0,0,0,0,48,48,0,0,0,0,0,0,0,0,0,144,
  144,0,0,0,0,0,0,0,0,0,0,0,0,0,144,0,144,0,0,0,0,0,
  0,0,0,0,0,0,0,72,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,
  0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,
  0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,
  0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,
  0,0,0,12,12,0,0,0,0,12,0,0,0,0,0,0,0,0,0,0,0,0,0,
  0,5,5,5,5,5,5,5,5,30,30,30,30,30,30,0,0,30,30,30,
  120,30,240,0,0,270,0,30,30,0,30,0,30,0,30,0,120,0,0,
  0,0,0,0,0,0,0,2160,2160,0,0,2160,0,2160,0,5,5,5,5,5,
  5,5,30,30,30,30,30,30,0,0,30,30,30,120,30,240,0,0,
  270,0,30,30,0,30,0,30,0,30,0,120,0,0,0,0,0,0,0,0,0,
  2160,2160,0,0,2160,0,2160,0,5,5,5,5,5,5,30,30,30,30,
  30,30,60,0,30,30,30,0,30,0,60,0,0,0,30,30,120,30,0,
  30,120,30,120,0,60,0,0,0,0,0,0,0,0,0,0,0,77760,0,
  77760,0,0,5,5,5,5,5,30,30,30,30,30,30,60,0,30,30,30,
  60,30,0,0,0,0,0,30,30,60,30,0,30,60,30,60,60,60,0,0,
  270,0,0,0,0,0,2160,2160,0,2160,2160,2160,2160,0,5,5,5,
  5,30,30,30,30,30,30,60,0,30,30,30,60,30,0,0,0,0,0,
  30,30,60,30,0,30,60,30,60,60,60,0,0,270,0,0,0,0,0,
  2160,2160,0,2160,2160,2160,2160,0,5,5,5,30,30,30,30,30,
  30,120,0,30,30,30,120,30,120,120,0,0,0,30,30,0,30,0,
  30,0,30,0,120,120,0,0,0,0,0,77760,77760,0,77760,77760,
  77760,0,77760,0,77760,77760,5,5,30,30,30,30,30,30,120,0,
  30,30,30,0,30,120,60,0,270,0,30,30,120,30,270,30,0,30,
  0,0,120,0,0,90,0,0,360,360,0,2160,2160,360,2160,2160,
  2160,2160,360,5,30,30,30,30,30,30,0,0,30,30,30,0,30,
  60,60,0,0,0,30,30,60,30,270,30,60,30,60,0,0,0,0,0,0,
  0,2160,2160,0,0,0,2160,2160,0,2160,0,2160,30,30,30,30,
  30,30,0,0,30,30,30,0,30,60,60,0,0,0,30,30,60,30,270,
  30,60,30,60,0,0,0,0,0,0,0,2160,2160,0,0,0,2160,2160,
  0,2160,0,2160,30,30,30,30,30,240,0,30,30,30,120,30,120,
  120,0,810,0,30,30,120,30,810,30,120,30,120,120,0,0,0,
  270,0,0,1080,1080,0,6480,6480,0,6480,6480,6480,6480,0,30,
  30,30,30,0,0,30,30,30,120,30,120,0,0,810,0,30,30,120,
  30,810,30,120,30,120,120,0,0,0,0,0,0,1080,0,0,6480,
  6480,0,6480,6480,6480,0,1080,30,30,30,0,0,30,30,30,120,
  30,60,0,0,810,0,30,30,60,30,810,30,120,30,120,120,0,0,
  0,0,0,0,6480,0,0,2160,6480,0,6480,6480,2160,2160,2160,
  30,30,0,0,30,30,30,120,30,60,60,0,0,0,30,30,60,30,
  810,30,120,30,120,0,120,0,0,0,0,0,0,0,0,6480,0,1080,
  6480,6480,6480,0,1080,30,0,0,30,30,30,120,30,0,240,0,0,
  0,30,30,120,30,270,30,120,30,120,120,120,0,0,0,0,0,0,
  0,0,6480,2160,2160,2160,2160,6480,0,0,120,0,30,30,30,
  120,30,0,60,0,0,0,30,30,60,30,810,30,120,30,120,0,60,
  0,0,270,0,0,0,1080,0,6480,0,1080,6480,6480,6480,6480,0,
  0,120,120,60,120,120,240,120,0,0,0,60,120,60,60,3240,
  60,60,120,240,0,0,0,0,0,0,0,0,0,0,25920,0,0,12960,
  25920,12960,0,2160,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,
  0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,30,30,
  0,30,0,240,0,0,0,30,30,120,30,810,30,120,30,0,0,60,0,
  0,270,0,0,0,1080,0,0,0,1080,6480,6480,0,6480,1080,30,
  120,30,240,0,0,0,0,30,30,120,30,810,30,120,30,0,120,
  120,0,0,0,0,0,0,2160,0,2160,0,0,6480,6480,0,0,360,0,
  30,120,0,0,810,0,30,30,120,30,810,30,120,30,60,120,60,
  0,0,270,0,0,1080,1080,0,0,6480,0,6480,6480,0,6480,1080,
  0,0,960,0,0,0,0,120,0,120,6480,120,0,240,60,480,60,0,
  0,1080,0,0,4320,2160,0,0,0,2160,25920,12960,0,12960,2160,
  120,120,0,810,0,30,30,120,30,0,30,120,30,0,120,60,0,0,
  90,0,0,6480,360,0,0,6480,1080,0,0,2160,2160,2160,60,0,
  0,0,120,60,60,60,3240,120,480,60,0,60,480,0,0,2160,0,
  0,0,4320,0,0,0,4320,25920,12960,0,0,0,0,0,0,120,60,60,
  60,3240,120,120,60,0,240,0,0,0,0,0,0,0,0,0,12960,
  25920,0,12960,12960,12960,0,4320,0,0,0,0,0,0,0,0,0,0,
  0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,810,0,
  3240,810,0,270,0,810,6480,0,3240,0,0,0,0,0,0,0,0,0,0,
  1080,2160,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,
  0,0,0,0,0,0,0,0,0,0,0,30,0,30,0,30,0,30,0,120,120,
  0,0,270,0,0,1080,1080,0,0,6480,1080,0,0,0,6480,1080,60,
  30,0,30,0,30,120,0,120,0,0,270,0,0,2160,0,0,0,0,360,
  0,2160,0,6480,6480,60,0,0,60,0,0,480,0,0,0,1080,0,0,
  2160,6480,0,0,25920,2160,0,0,0,25920,4320,0,30,60,30,
  120,120,0,0,0,0,0,0,1080,0,0,6480,6480,1080,0,0,6480,
  0,1080,0,0,0,0,0,3240,0,0,0,0,0,1080,0,0,0,2160,1080,
  0,0,0,0,0,0,30,120,120,0,0,0,0,0,0,360,1080,0,6480,
  2160,2160,2160,0,0,0,1080,0,480,0,0,0,0,0,0,0,2160,
  2160,0,25920,12960,2160,0,0,12960,12960,4320,0,120,120,0,
  0,270,0,0,1080,1080,0,0,6480,1080,0,0,6480,6480,0,0,60,
  0,0,1080,0,0,0,2160,0,0,25920,2160,0,25920,0,12960,2160,
  120,0,0,0,0,0,0,4320,0,12960,0,2160,12960,12960,25920,0,
  2160,0,0,0,0,0,2160,0,0,25920,25920,0,0,0,12960,0,0,0,
  0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,
  0,0,0,0,0,0,0,0,0,2160,0,0,0,0,2160,0,360,0,0,0,0,
  0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,2160,
  360,2160,2160,0,0,0,0,2160,0,0,0,0,2160,2160,360,0,0,
  0,0,0,0,0,0,0,0,0,0,0,2160,2160,2160,0,2160,0,0,0,
  2160,2160,0,0,0,0,0,0,0,0,0,0,0,2160,2160,5,5,5,5,5,
  5,5,30,30,30,30,30,30,0,0,30,30,30,120,30,240,0,0,
  270,0,30,30,0,30,0,30,0,30,0,120,0,0,0,0,0,0,0,0,0,
  2160,2160,0,0,2160,0,2160,0,5,5,5,5,5,5,30,30,30,30,
  30,30,0,240,30,30,30,0,30,0,0,120,0,270,30,30,120,30,
  0,30,240,30,240,0,0,240,270,0,0,2160,0,0,2160,0,0,0,
  2160,0,2160,0,0,5,5,5,5,5,30,30,30,30,30,30,0,240,30,
  30,30,0,30,0,0,120,0,270,30,30,120,30,0,30,240,30,240,
  0,0,240,270,0,0,2160,0,0,2160,0,0,0,2160,0,2160,0,0,
  5,5,5,5,30,30,30,30,30,30,0,60,30,30,30,60,30,60,0,
  60,0,0,30,30,60,30,0,30,60,30,60,60,0,60,0,0,0,77760,
  0,0,77760,77760,77760,0,77760,77760,77760,77760,0,5,5,5,
  30,30,30,30,30,30,0,120,30,30,30,60,30,60,0,120,0,0,
  30,30,0,30,0,30,0,30,0,60,0,120,0,0,0,77760,0,0,
  77760,0,0,0,0,0,0,0,0,5,5,30,30,30,30,30,30,0,60,30,
  30,30,0,30,60,0,60,0,0,30,30,60,30,0,30,0,30,0,0,0,
  60,0,0,270,2160,0,0,2160,0,0,0,2160,0,2160,0,0,5,30,
  30,30,30,30,30,0,60,30,30,30,0,30,60,0,60,0,0,30,30,
  60,30,0,30,0,30,0,0,0,60,0,0,270,2160,0,0,2160,0,0,
  0,2160,0,2160,0,0,30,30,30,30,30,30,0,60,30,30,30,0,
  30,120,0,120,270,270,30,30,60,30,0,30,60,30,60,0,0,60,
  270,0,90,360,0,0,360,360,360,0,2160,360,2160,360,0,30,
  30,30,30,30,0,0,30,30,30,120,30,120,0,0,810,0,30,30,
  120,30,0,30,120,30,60,60,0,0,810,0,0,0,0,0,0,1080,
  1080,0,6480,1080,6480,1080,0,30,30,30,30,0,0,30,30,30,
  60,30,240,0,120,810,0,30,30,120,30,0,30,60,30,120,120,
  0,240,810,0,270,0,0,0,1080,1080,1080,0,6480,1080,6480,0,
  0,30,30,30,0,0,30,30,30,120,30,60,0,0,810,0,30,30,
  120,30,0,30,120,30,120,120,0,0,270,0,0,0,0,0,0,360,
  6480,0,6480,0,2160,0,0,30,30,0,240,30,30,30,120,30,0,
  0,60,0,810,30,30,120,30,0,30,120,30,120,0,0,120,810,0,
  270,1080,0,0,1080,1080,0,0,6480,1080,6480,0,0,30,0,0,
  30,30,30,120,30,0,0,120,0,0,30,30,120,30,0,30,120,30,
  60,0,0,60,810,0,270,0,0,0,6480,6480,0,0,2160,360,6480,
  0,0,0,120,30,30,30,120,30,60,0,60,0,810,30,30,120,30,
  0,30,120,30,60,60,0,60,810,0,0,1080,0,0,0,1080,0,0,
  6480,1080,6480,1080,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,
  0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,
  120,60,60,60,120,0,0,0,3240,0,60,120,240,60,0,60,60,
  120,60,480,0,0,3240,0,0,0,0,0,0,4320,12960,0,12960,
  4320,12960,8640,0,30,30,0,30,0,0,120,0,810,30,30,120,
  30,0,30,120,30,60,60,0,60,0,0,270,1080,0,0,1080,0,0,
  0,6480,1080,0,1080,0,30,0,30,0,0,120,0,810,30,30,120,
  30,0,30,60,30,0,240,0,120,0,0,90,6480,0,0,360,0,0,0,
  6480,6480,2160,2160,0,60,30,120,0,120,810,0,30,30,120,
  30,0,30,60,30,0,60,0,60,0,0,270,0,0,0,1080,0,1080,0,
  6480,1080,0,1080,0,0,0,0,0,0,3240,60,240,240,60,0,60,
  60,0,120,480,0,480,0,0,2160,4320,0,0,4320,0,0,0,25920,
  0,0,4320,0,120,0,0,0,0,30,30,120,30,0,30,120,30,60,
  60,0,60,270,0,270,6480,0,0,360,360,6480,0,0,1080,0,360,
  0,0,60,0,3240,120,120,120,60,0,120,480,60,240,120,0,60,
  0,0,3240,12960,0,0,4320,0,0,0,25920,2160,0,0,0,0,0,0,
  0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,
  0,0,0,3240,0,240,60,0,60,0,0,240,0,0,0,0,120,0,0,0,
  0,0,0,0,4320,4320,0,0,0,12960,4320,0,0,810,0,0,810,0,
  270,0,810,0,0,0,0,0,0,0,1080,0,0,0,0,0,0,2160,1080,
  0,0,0,810,270,0,810,0,0,3240,0,3240,3240,0,0,0,0,0,0,
  0,0,0,0,1080,0,0,1080,0,0,0,30,0,30,0,30,60,30,0,
  240,0,120,0,0,270,1080,0,0,1080,0,1080,0,0,0,0,1080,0,
  120,30,0,30,0,30,240,0,0,120,0,0,270,360,0,0,0,0,
  2160,0,2160,0,0,0,0,0,0,0,60,0,0,960,0,0,0,0,12960,
  2160,0,0,0,4320,2160,0,0,0,0,2160,0,0,30,0,30,120,0,
  0,0,810,0,270,1080,0,0,1080,1080,1080,0,0,0,6480,0,0,
  0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,60,
  30,120,120,0,120,810,0,270,360,0,0,1080,6480,360,0,0,
  360,6480,1080,0,0,240,0,0,0,0,0,0,2160,0,0,0,4320,
  4320,0,0,0,0,0,0,120,120,0,0,810,0,0,1080,0,0,0,1080,
  1080,0,0,0,0,1080,0,60,0,60,0,0,1080,6480,0,0,2160,0,
  4320,0,0,0,0,4320,0,0,60,0,0,1080,4320,0,0,4320,0,0,
  0,12960,0,25920,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,
  0,3240,0,0,0,0,0,0,4320,4320,0,0,25920,12960,12960,0,0,
  270,0,0,0,1080,0,0,0,0,0,0,1080,0,0,0,0,0,0,0,0,0,
  0,0,0,0,0,0,0,0,0,360,0,0,0,0,2160,360,0,0,0,0,0,
  360,0,2160,360,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,
  0,0,0,0,0,360,0,0,0,0,2160,360,0,0,0,0,0,2160,360,0,
  0,2160,360,0,0,0,0,0,0,0,0,2160,0,0,0,0,0,0,2160,0,
  0,5,5,5,5,5,5,30,30,30,30,30,30,0,0,30,30,30,0,30,
  0,60,60,0,0,30,30,60,30,270,30,60,30,60,0,0,0,0,0,0,
  2160,0,0,2160,0,0,0,2160,0,2160,0,0,5,5,5,5,5,30,30,
  30,30,30,30,0,0,30,30,30,120,30,0,0,120,30,30,30,30,
  60,30,30,30,120,30,120,120,0,0,30,30,30,60,60,60,60,
  60,60,60,60,60,60,60,60,5,5,5,5,30,30,30,30,30,30,0,
  0,30,30,30,0,30,0,60,60,0,0,30,30,60,30,270,30,60,30,
  60,0,0,0,0,0,0,2160,0,0,2160,0,0,0,2160,0,2160,0,0,
  5,5,5,30,30,30,30,30,30,0,0,30,30,30,120,30,60,120,
  120,90,0,30,30,0,30,0,30,0,30,0,120,0,0,0,270,270,
  2160,2160,2160,2160,360,360,2160,0,360,0,360,2160,5,5,30,
  30,30,30,30,30,0,0,30,30,30,0,30,0,0,120,30,30,30,30,
  120,30,30,30,0,30,0,0,0,0,30,30,30,60,60,60,60,60,60,
  60,60,60,60,60,60,5,30,30,30,30,30,30,0,0,30,30,30,
  60,30,60,60,60,270,0,30,30,60,30,90,30,60,30,60,60,0,
  0,0,0,270,360,2160,2160,360,0,0,2160,2160,0,2160,0,2160,
  30,30,30,30,30,30,0,0,30,30,30,0,30,0,0,120,30,30,30,
  30,120,30,30,30,0,30,0,0,0,0,30,30,30,60,60,60,60,60,
  60,60,60,60,60,60,60,30,30,30,30,30,0,0,30,30,30,120,
  30,120,0,0,270,0,30,30,120,30,270,30,0,30,60,60,0,0,
  0,0,0,0,1080,0,0,1080,1080,0,1080,1080,1080,1080,0,30,
  30,30,30,0,0,30,30,30,60,30,120,0,0,270,0,30,30,120,
  30,270,30,60,30,120,120,0,0,270,0,0,0,0,0,0,1080,1080,
  0,1080,1080,1080,0,1080,30,30,30,0,0,30,30,30,240,30,
  60,0,0,90,90,30,30,60,30,90,30,120,30,240,120,0,0,30,
  30,30,180,180,60,60,60,180,180,180,180,60,60,60,30,30,
  0,0,30,30,30,120,30,120,120,120,0,270,30,30,60,30,0,
  30,120,30,120,0,0,0,270,0,270,0,0,0,1080,1080,0,1080,
  1080,1080,1080,0,0,30,0,0,30,30,30,120,30,0,120,120,30,
  30,30,30,240,30,30,30,240,30,60,0,0,0,90,90,90,60,60,
  180,180,180,60,60,60,60,180,180,180,0,0,30,30,30,120,
  30,60,120,60,0,270,30,30,120,30,270,30,120,30,120,0,0,
  0,270,270,0,1080,0,1080,0,0,0,1080,0,1080,0,0,0,0,0,
  0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,
  0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,
  0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,
  0,30,30,0,30,0,120,120,0,0,30,30,0,30,270,30,120,30,
  60,60,0,0,0,270,270,1080,0,1080,0,0,0,0,1080,1080,0,
  1080,1080,30,0,30,120,0,120,90,90,30,30,120,30,90,30,
  60,30,0,120,0,0,30,30,30,180,180,60,60,60,180,180,180,
  180,60,60,60,0,30,0,120,0,0,0,30,30,120,30,0,30,120,
  30,0,60,0,0,0,270,0,0,1080,1080,1080,0,0,0,0,0,0,
  1080,1080,0,0,240,0,0,0,60,120,480,60,0,60,60,120,120,
  0,0,0,0,1080,0,720,6480,8640,4320,0,0,0,2160,720,0,
  4320,4320,120,120,0,90,90,30,30,120,30,90,30,120,30,0,
  60,0,0,30,30,30,180,180,60,60,60,180,180,180,180,60,60,
  60,120,60,0,0,120,60,60,60,1080,240,0,120,240,60,0,0,
  0,0,1080,4320,0,2160,2160,0,0,0,0,8640,6480,2160,720,0,
  2160,0,120,120,60,60,1080,120,120,120,0,0,0,0,0,0,0,0,
  2160,0,0,4320,2160,0,2160,720,0,0,4320,0,0,120,120,120,
  120,0,120,960,120,240,60,0,0,1080,0,0,0,2160,0,0,0,
  720,0,2160,2160,0,4320,0,30,270,30,2160,0,30,30,1080,
  270,0,0,0,0,0,0,0,120,0,0,0,0,0,0,120,120,0,0,0,
  270,30,0,0,30,30,0,0,0,2160,0,0,0,0,0,0,120,0,0,0,
  0,0,0,0,0,0,0,30,0,30,0,30,60,30,0,120,0,0,0,270,
  270,0,1080,0,1080,0,1080,1080,0,0,0,1080,0,60,30,30,30,
  0,30,120,0,0,0,90,90,90,60,60,180,180,180,60,60,60,60,
  180,180,180,60,0,0,0,120,0,480,0,0,0,0,2160,4320,8640,
  2160,720,2160,0,0,0,0,0,2160,2160,0,30,0,30,120,120,0,
  0,270,0,270,1080,1080,0,1080,0,1080,1080,0,0,1080,0,
  1080,30,0,0,0,1080,0,0,0,0,0,120,120,0,0,0,120,120,0,
  0,0,0,0,0,30,120,0,0,0,90,90,90,60,60,180,180,180,60,
  60,60,60,180,180,180,0,480,0,0,0,1080,1080,0,8640,4320,
  2160,0,2160,4320,0,0,0,720,2160,6480,240,120,0,0,270,
  270,0,1080,1080,1080,0,0,0,1080,0,0,0,1080,0,60,0,0,0,
  1080,0,2160,720,4320,8640,0,2160,2160,0,6480,0,4320,0,0,
  0,1080,0,1080,4320,0,6480,720,720,0,8640,4320,0,2160,0,
  0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,
  0,0,0,0,0,0,0,0,30,30,0,0,120,120,0,0,0,0,0,0,0,
  120,30,0,0,0,0,120,0,0,0,0,0,0,120,0,0,0,0,0,0,0,
  0,0,0,120,0,60,0,0,0,60,60,60,60,0,0,0,0,0,0,60,60,
  60,60,0,0,0,60,60,0,0,0,0,60,60,60,60,0,0,0,0,60,
  60,60,0,0,0,0,60,60,60,60,60,60,0,0,0,60,60,0,0,0,
  60,0,0,0,0,0,0,60,60,60,5,5,5,5,5,30,30,30,30,30,
  30,0,240,30,30,30,0,30,0,0,120,0,270,30,30,120,30,0,
  30,240,30,240,0,0,240,270,0,0,2160,0,0,2160,0,0,0,
  2160,0,2160,0,0,5,5,5,5,30,30,30,30,30,30,0,0,30,30,
  30,0,30,0,60,60,0,0,30,30,60,30,270,30,60,30,60,0,0,
  0,0,0,0,2160,0,0,2160,0,0,0,2160,0,2160,0,0,5,5,5,
  30,30,30,30,30,30,120,0,30,30,30,0,30,0,240,0,0,0,30,
  30,0,30,0,30,0,30,0,0,120,0,0,270,0,0,2160,2160,0,0,
  0,2160,0,0,0,0,2160,5,5,30,30,30,30,30,30,120,0,30,
  30,30,0,30,0,240,0,0,0,30,30,0,30,0,30,0,30,0,0,120,
  0,0,270,0,0,2160,2160,0,0,0,2160,0,0,0,0,2160,5,30,
  30,30,30,30,30,60,60,30,30,30,0,30,0,60,60,0,0,30,30,
  60,30,0,30,60,30,60,0,60,60,0,0,0,77760,77760,77760,
  77760,0,0,77760,77760,0,77760,0,77760,30,30,30,30,30,30,
  0,60,30,30,30,0,30,0,120,60,0,270,30,30,120,30,90,30,
  60,30,60,0,0,60,270,270,0,2160,360,360,2160,0,0,360,
  360,0,360,0,360,30,30,30,30,30,240,0,30,30,30,0,30,0,
  120,0,0,0,30,30,240,30,270,30,120,30,60,0,60,60,810,
  810,0,0,1080,1080,0,0,0,0,1080,0,1080,0,0,30,30,30,30,
  60,60,30,30,30,0,30,0,0,120,0,0,30,30,120,30,270,30,
  60,30,120,0,0,120,810,0,0,0,1080,0,6480,0,0,0,1080,0,
  1080,0,1080,30,30,30,0,240,30,30,30,0,30,0,120,120,0,
  0,30,30,60,30,270,30,120,30,120,0,240,0,270,0,0,0,0,
  2160,2160,0,0,0,0,0,360,0,0,30,30,0,120,30,30,30,0,
  30,0,0,120,0,810,30,30,60,30,270,30,120,30,240,0,120,
  120,810,0,0,6480,0,0,6480,0,0,1080,1080,0,1080,0,1080,
  30,240,0,30,30,30,0,30,0,0,120,0,0,30,30,120,30,90,
  30,120,30,60,0,0,60,810,0,0,2160,2160,0,6480,0,0,0,
  360,0,6480,0,6480,0,120,30,30,30,0,30,0,60,0,0,810,30,
  30,60,30,270,30,0,30,60,0,60,0,810,810,0,6480,0,1080,
  0,0,0,1080,1080,0,1080,0,0,60,120,0,60,0,120,0,120,
  960,0,0,60,120,0,60,1080,60,60,120,480,0,480,0,3240,0,
  0,25920,0,0,12960,0,0,0,4320,0,4320,0,0,120,60,0,0,
  120,0,240,0,0,0,60,120,0,60,1080,60,60,120,60,0,120,
  240,3240,0,0,0,4320,4320,0,0,0,0,2160,0,6480,0,0,30,
  30,0,30,0,240,120,0,810,30,30,120,30,270,30,240,30,0,
  0,60,60,0,810,0,6480,0,1080,6480,0,0,1080,1080,0,0,0,
  1080,30,0,30,0,0,120,0,810,30,30,120,30,270,30,60,30,
  0,0,120,120,0,0,0,6480,0,0,2160,0,0,6480,6480,0,0,0,
  360,0,30,0,120,120,0,0,30,30,120,30,270,30,60,30,60,0,
  60,60,0,810,0,0,1080,1080,6480,0,0,0,1080,0,0,0,1080,
  0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,
  0,0,0,0,0,0,0,0,0,0,120,0,0,810,30,30,0,30,270,30,
  120,30,60,0,60,60,0,270,0,6480,1080,360,0,0,0,6480,
  1080,0,360,0,360,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,
  0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,120,0,0,120,60,60,
  60,3240,120,60,60,0,0,0,480,3240,0,0,0,0,0,25920,0,0,
  0,4320,0,12960,0,2160,0,0,120,120,0,120,12960,120,0,0,
  240,0,240,60,0,0,0,0,2160,2160,0,0,0,4320,0,0,2160,0,
  0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,
  0,0,0,0,0,810,270,0,810,270,270,3240,810,3240,0,0,0,0,
  0,0,0,1080,0,0,0,0,0,1080,0,0,0,0,30,0,30,0,30,0,
  30,0,0,120,120,0,810,0,6480,1080,1080,6480,0,0,1080,0,
  0,0,0,1080,0,30,0,30,0,30,0,0,120,120,0,810,0,2160,0,
  6480,6480,0,0,360,0,0,0,0,0,60,0,0,120,0,0,0,0,240,
  0,3240,0,12960,4320,4320,0,0,0,4320,0,0,0,0,0,0,30,60,
  30,120,0,120,120,810,0,0,6480,1080,0,6480,0,0,1080,0,0,
  1080,0,1080,270,0,0,0,0,2160,0,0,0,0,2160,360,0,0,0,
  0,360,0,0,0,0,0,60,30,120,0,0,120,0,0,0,0,360,6480,
  0,0,0,360,360,0,6480,0,1080,0,0,0,480,0,0,0,0,12960,
  12960,4320,0,0,0,4320,0,0,0,0,25920,120,0,0,0,0,810,0,
  0,1080,1080,0,0,0,1080,0,0,1080,0,0,0,60,60,0,3240,0,
  12960,8640,12960,12960,0,0,4320,0,0,0,0,4320,0,0,0,0,0,
  0,0,0,0,0,0,0,0,0,0,0,0,60,3240,0,0,0,4320,0,25920,
  0,0,0,4320,0,4320,0,0,3240,0,0,0,0,4320,0,0,0,4320,0,
  0,2160,0,0,0,0,0,0,1080,0,0,0,0,0,0,0,0,1080,0,0,0,
  0,2160,0,0,0,0,0,1080,0,1080,0,0,0,0,0,0,0,0,0,0,0,
  0,2160,0,0,0,0,2160,2160,0,0,0,0,0,0,0,0,360,360,0,
  0,0,0,2160,0,0,0,0,0,360,0,360,0,0,0,0,0,2160,0,
  2160,0,0,0,0,0,0,0,0,0,0,0,0,0,360,0,0,0,0,0,0,0,
  0,0,0,0,0,360,0,5,5,5,5,30,30,30,30,30,30,60,0,30,
  30,30,60,30,0,0,0,0,0,30,30,60,30,0,30,60,30,60,60,
  60,0,0,270,0,0,0,0,0,2160,2160,0,2160,2160,2160,2160,0,
  5,5,5,30,30,30,30,30,30,60,60,30,30,30,60,30,0,0,60,
  270,0,30,30,0,30,0,30,0,30,0,60,60,60,0,0,0,2160,
  2160,2160,2160,0,0,2160,0,0,0,0,2160,5,5,30,30,30,30,
  30,30,120,120,30,30,30,0,30,0,0,60,30,30,30,30,120,30,
  30,30,0,30,0,0,120,120,30,30,30,60,60,60,60,60,60,60,
  60,60,60,60,60,5,30,30,30,30,30,30,60,60,30,30,30,60,
  30,0,0,60,270,0,30,30,0,30,0,30,0,30,0,60,60,60,0,0,
  0,2160,2160,2160,2160,0,0,2160,0,0,0,0,2160,30,30,30,
  30,30,30,0,120,30,30,30,0,30,0,0,120,30,30,30,30,120,
  30,30,30,120,30,120,0,0,120,30,30,30,60,60,60,60,60,
  60,60,60,60,60,60,60,30,30,30,30,30,120,0,30,30,30,
  120,30,0,0,0,0,0,30,30,120,30,0,30,120,30,60,120,120,
  120,270,270,0,0,0,1080,0,0,1080,0,1080,1080,1080,0,0,
  30,30,30,30,120,120,30,30,30,120,30,0,0,120,270,0,30,
  30,120,30,270,30,60,30,120,0,0,120,270,0,270,0,1080,0,
  1080,1080,0,0,1080,0,1080,0,0,30,30,30,0,120,30,30,30,
  120,30,0,0,0,90,90,30,30,60,30,90,30,120,30,120,120,
  120,0,30,30,30,180,180,60,60,60,180,180,180,180,60,60,
  60,30,30,0,120,30,30,30,0,30,0,0,60,0,270,30,30,60,
  30,270,30,120,30,120,0,0,120,270,0,270,1080,0,0,1080,0,
  0,0,1080,1080,1080,0,1080,30,120,0,30,30,30,120,30,0,0,
  120,30,30,30,30,240,30,30,30,120,30,60,0,0,60,90,90,
  90,60,60,180,180,180,60,60,60,60,180,180,180,120,0,30,
  30,30,0,30,0,0,0,0,0,30,30,60,30,0,30,120,30,0,60,
  120,60,270,270,0,0,0,1080,0,1080,0,1080,1080,1080,0,
  1080,0,120,0,0,120,0,120,0,0,240,0,2160,120,120,60,60,
  1080,120,60,120,0,0,480,0,1080,0,0,2160,0,0,4320,0,0,
  2160,720,2160,4320,0,4320,120,60,60,60,120,0,0,0,0,0,
  60,120,240,0,1080,60,60,0,60,0,60,240,1080,0,0,0,0,
  2160,0,2160,4320,0,0,4320,720,2160,0,30,30,0,30,0,0,
  120,0,270,30,30,120,30,270,30,120,30,120,120,120,60,0,
  270,270,1080,0,0,1080,0,0,1080,1080,0,0,1080,0,30,0,30,
  0,0,240,90,90,30,30,120,30,90,30,60,30,0,120,120,120,
  30,30,30,180,180,60,60,60,180,180,180,180,60,60,60,60,
  30,0,0,120,270,0,30,30,120,30,270,30,0,30,60,120,60,0,
  0,270,270,0,1080,1080,0,0,1080,0,0,1080,0,1080,1080,0,
  0,0,0,0,0,120,120,0,120,0,120,120,0,60,480,60,0,0,
  1080,1080,720,0,0,0,0,0,2160,2160,2160,0,4320,4320,0,0,
  0,90,90,30,30,120,30,90,30,120,30,60,120,120,60,30,30,
  30,180,180,60,60,60,180,180,180,180,60,60,60,0,0,0,0,
  0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,
  0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,
  0,0,0,0,0,0,0,0,0,0,0,120,60,60,60,1080,120,480,60,
  240,60,0,60,0,1080,0,0,0,8640,0,4320,720,6480,720,0,
  4320,4320,0,30,270,30,1080,270,30,30,1080,0,0,0,1080,
  1080,0,0,0,0,0,0,0,0,0,120,120,120,0,0,0,270,30,0,
  270,30,30,2160,270,1080,1080,0,1080,0,0,0,0,0,0,0,0,
  120,0,120,0,0,0,0,30,0,30,0,30,120,30,0,120,120,120,
  0,0,0,1080,0,1080,1080,0,1080,0,0,0,0,0,1080,0,30,30,
  30,0,30,120,0,120,120,90,90,90,60,60,180,180,180,60,60,
  60,60,180,180,180,0,0,0,60,60,0,240,0,480,0,0,1080,
  4320,4320,720,720,6480,8640,4320,0,0,0,0,0,0,30,60,30,
  0,120,0,120,0,0,0,0,1080,0,1080,1080,1080,1080,0,0,0,
  0,1080,30,0,0,0,0,1080,0,0,0,0,120,120,0,0,0,120,120,
  0,0,0,0,0,60,30,120,120,0,120,90,90,90,60,60,180,180,
  180,60,60,60,60,180,180,180,0,240,0,0,0,1080,1080,0,0,
  4320,2160,0,2160,0,4320,0,0,2160,720,2160,0,120,0,0,270,
  0,0,0,1080,1080,0,1080,1080,1080,0,0,0,1080,0,120,60,
  60,0,0,1080,720,2160,4320,0,0,2160,2160,0,0,0,0,4320,0,
  60,2160,0,1080,4320,0,0,720,2160,0,0,4320,4320,2160,0,
  2160,120,0,0,0,0,4320,0,2160,2160,0,0,0,4320,720,0,
  2160,2160,1080,0,2160,720,0,0,4320,2160,2160,0,2160,0,
  4320,0,30,30,0,0,120,120,0,0,0,0,0,0,0,0,30,0,0,0,
  120,120,0,0,0,0,0,0,120,0,0,120,0,120,0,0,0,0,120,
  120,0,60,0,0,0,60,60,60,60,0,0,0,0,0,0,60,60,60,60,
  0,0,0,60,60,0,0,0,0,60,60,60,60,0,0,0,0,60,60,60,0,
  0,0,0,60,60,60,60,60,60,0,0,0,60,60,0,0,0,60,0,0,0,
  0,0,0,60,60,60,5,5,5,30,30,30,30,30,30,0,60,30,30,
  30,0,30,60,60,60,0,0,30,30,0,30,0,30,0,30,0,0,0,60,
  0,0,270,2160,0,0,2160,2160,2160,0,0,2160,0,2160,0,5,5,
  30,30,30,30,30,30,60,60,30,30,30,0,30,60,60,60,0,0,
  30,30,60,30,270,30,0,30,0,0,60,60,0,270,90,2160,0,0,
  2160,2160,2160,0,360,2160,360,2160,0,5,30,30,30,30,30,
  30,30,30,30,30,30,30,30,30,30,30,0,0,30,30,30,30,0,
  30,30,30,30,30,30,30,0,0,0,77760,0,0,77760,0,0,0,
  77760,0,77760,0,0,30,30,30,30,30,30,0,60,30,30,30,0,
  30,60,60,60,0,0,30,30,0,30,0,30,0,30,0,0,0,60,0,0,
  270,2160,0,0,2160,2160,2160,0,0,2160,0,2160,0,30,30,30,
  30,30,60,0,30,30,30,240,30,0,60,0,0,0,30,30,60,30,
  810,30,60,30,30,30,30,30,0,810,0,0,0,0,0,6480,6480,0,
  1080,6480,1080,6480,0,30,30,30,30,30,30,30,30,30,30,30,
  60,0,60,0,0,30,30,120,30,810,30,30,30,240,60,0,60,0,
  0,270,0,0,0,6480,6480,6480,0,1080,6480,1080,0,0,30,30,
  30,0,60,30,30,30,60,30,30,30,30,0,0,30,30,30,30,0,30,
  240,30,60,0,60,0,0,270,270,6480,0,0,0,2160,0,0,1080,
  6480,360,0,0,30,30,0,60,30,30,30,60,30,30,30,30,0,0,
  30,30,30,30,810,30,240,30,60,0,60,0,0,0,270,6480,0,0,
  6480,6480,0,0,1080,6480,1080,0,0,30,60,0,30,30,30,0,30,
  0,60,0,0,0,30,30,60,30,270,30,60,30,30,30,30,30,0,
  810,0,0,0,0,0,0,0,0,360,2160,6480,6480,0,60,60,30,30,
  30,60,30,30,30,30,0,0,30,30,30,30,810,30,60,30,30,30,
  30,30,0,810,0,6480,0,0,0,6480,0,0,1080,6480,1080,6480,
  0,30,60,30,30,30,60,60,30,60,0,0,30,0,30,30,0,30,30,
  0,60,0,60,0,0,0,1080,12960,0,0,12960,12960,0,0,0,0,
  2160,0,0,0,30,30,30,0,0,60,0,0,0,30,60,60,30,1620,30,
  30,60,30,60,30,60,0,1620,0,0,0,0,0,0,0,0,2160,12960,
  51840,12960,0,30,30,0,30,0,60,240,0,0,30,30,60,30,810,
  30,60,30,30,30,30,30,0,810,270,6480,0,0,6480,0,0,0,
  1080,6480,0,6480,0,30,30,30,60,0,60,0,0,30,30,240,30,
  0,30,30,30,0,60,0,60,0,0,90,0,0,0,2160,0,6480,0,6480,
  0,2160,2160,0,30,30,60,60,60,0,0,30,30,60,30,810,30,
  30,30,30,30,30,30,0,810,270,0,0,0,6480,0,6480,0,1080,
  6480,0,6480,0,0,0,60,0,0,0,30,60,60,30,1620,30,30,60,
  30,60,30,60,0,1620,0,0,0,0,12960,0,0,0,2160,12960,0,
  12960,0,0,60,0,0,0,30,30,60,30,810,30,60,30,30,30,30,
  30,0,270,0,0,0,0,0,0,0,0,1080,6480,0,2160,0,30,30,0,
  0,60,30,30,30,1620,60,60,30,60,30,60,30,0,1620,0,0,0,
  0,0,0,0,0,2160,12960,0,12960,0,30,0,0,0,30,30,30,0,
  240,30,30,0,60,0,60,0,0,540,0,0,0,12960,12960,12960,0,
  51840,51840,2160,0,0,0,0,60,30,30,30,1620,60,60,30,60,
  30,60,30,0,1620,0,0,0,0,0,12960,0,0,2160,12960,2160,
  12960,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,
  0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,
  0,0,0,0,0,0,0,0,0,0,0,30,0,30,0,30,30,30,0,60,0,
  60,0,810,270,6480,0,0,6480,0,6480,0,0,0,0,6480,0,30,
  30,0,30,0,30,60,0,60,240,0,0,0,2160,0,0,6480,6480,
  2160,0,2160,0,1080,6480,0,30,0,0,30,30,0,60,0,60,0,
  6480,540,0,0,0,12960,12960,12960,0,0,0,0,0,0,0,30,30,
  30,60,60,60,60,0,0,270,6480,0,0,6480,6480,6480,0,0,0,
  1080,0,0,0,0,0,0,6480,0,0,0,0,0,0,0,0,0,0,2160,0,0,
  0,0,0,0,30,30,0,60,0,60,0,810,270,0,0,0,6480,0,2160,
  0,0,0,0,6480,0,0,60,0,60,0,0,0,0,12960,0,0,0,12960,
  12960,0,0,0,2160,51840,0,60,0,60,0,0,0,0,6480,0,0,0,
  6480,0,0,0,0,1080,0,0,30,30,30,0,0,540,0,0,0,12960,0,
  12960,0,0,12960,0,12960,0,30,30,0,0,6480,12960,0,0,0,
  12960,0,0,0,12960,2160,0,0,30,0,0,540,0,0,0,12960,0,
  12960,0,0,12960,0,0,0,0,3240,0,12960,0,0,0,12960,0,0,
  0,0,2160,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,270,0,0,
  0,2160,0,0,0,0,0,1080,0,0,0,0,0,0,2160,0,0,0,0,360,
  2160,0,0,0,0,0,0,0,2160,0,0,0,0,0,0,0,0,0,0,0,0,0,
  0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,2160,0,0,0,0,0,0,
  2160,2160,0,0,2160,2160,0,0,0,0,0,0,0,0,2160,0,0,0,0,
  0,0,2160,0,0,5,5,30,30,30,30,30,30,120,0,30,30,30,0,
  30,0,240,0,0,0,30,30,0,30,0,30,0,30,0,0,120,0,0,270,
  0,0,2160,2160,0,0,0,2160,0,0,0,0,2160,5,30,30,30,30,
  30,30,60,60,30,30,30,60,30,0,0,60,270,0,30,30,0,30,0,
  30,0,30,0,60,60,60,0,0,0,2160,2160,2160,2160,0,0,2160,
  0,0,0,0,2160,30,30,30,30,30,30,0,60,30,30,30,0,30,60,
  60,60,0,0,30,30,0,30,0,30,0,30,0,0,0,60,0,0,270,
  2160,0,0,2160,2160,2160,0,0,2160,0,2160,0,30,30,30,30,
  30,120,0,30,30,30,120,30,120,120,0,270,0,30,30,0,30,0,
  30,0,30,0,60,0,0,0,810,0,0,6480,6480,0,1080,1080,0,0,
  1080,0,1080,0,30,30,30,30,0,0,30,30,30,60,30,240,0,
  120,270,0,30,30,0,30,0,30,0,30,0,120,0,120,0,0,810,0,
  6480,0,6480,1080,1080,0,0,1080,0,0,6480,30,30,30,0,120,
  30,30,30,120,30,60,60,60,270,0,30,30,0,30,0,30,0,30,
  0,120,120,0,0,0,0,0,6480,0,0,360,0,0,0,6480,0,2160,
  2160,30,30,0,120,30,30,30,120,30,60,60,60,0,0,30,30,0,
  30,0,30,0,30,0,0,120,120,0,0,810,6480,0,0,6480,1080,0,
  6480,0,1080,0,0,6480,30,120,0,30,30,30,120,30,0,240,
  120,0,0,30,30,0,30,0,30,0,30,0,120,120,120,0,0,810,0,
  0,0,6480,0,2160,2160,0,360,0,0,6480,120,120,30,30,30,0,
  30,60,60,60,0,0,30,30,0,30,0,30,0,30,0,60,120,60,0,
  810,0,6480,0,6480,0,1080,0,6480,0,1080,0,1080,0,0,120,
  120,120,120,120,240,60,480,0,0,120,120,0,0,0,120,0,120,
  0,0,480,0,0,0,0,25920,0,0,12960,2160,4320,12960,0,2160,
  0,0,12960,120,120,60,60,120,0,0,0,1080,0,120,120,0,120,
  0,0,0,0,0,240,60,480,0,6480,0,0,12960,25920,0,2160,
  2160,0,0,2160,0,0,25920,30,30,0,30,0,120,120,0,0,30,
  30,0,30,0,30,0,30,0,0,120,120,0,810,810,6480,0,6480,
  6480,0,0,6480,0,1080,0,1080,6480,30,120,30,240,0,120,0,
  0,30,30,0,30,0,30,0,30,0,0,120,120,0,0,270,6480,0,
  2160,2160,2160,0,6480,0,0,0,0,2160,60,30,0,120,120,270,
  0,30,30,0,30,0,30,0,30,0,60,0,120,0,810,810,0,6480,
  6480,6480,0,1080,0,0,1080,0,1080,6480,0,0,480,0,0,0,0,
  0,0,60,0,60,0,120,0,0,60,0,0,3240,3240,12960,0,25920,
  0,0,0,25920,0,0,0,2160,0,120,120,0,0,0,30,30,0,30,0,
  30,0,30,0,60,120,0,0,270,0,0,0,2160,2160,2160,1080,
  6480,0,1080,0,360,0,60,60,0,0,120,0,0,60,0,120,0,60,
  0,120,960,120,0,0,3240,12960,0,25920,12960,0,0,12960,0,
  4320,0,0,12960,60,2160,0,120,60,0,0,0,120,0,60,0,240,
  0,480,0,0,3240,0,0,0,25920,4320,4320,0,0,0,0,0,12960,
  1080,0,120,60,0,60,0,120,0,0,0,60,0,60,0,3240,0,0,
  25920,25920,0,2160,6480,0,0,4320,0,2160,0,0,270,0,0,270,
  0,90,0,270,0,0,1080,0,0,0,0,2160,0,0,0,0,0,2160,0,
  360,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,
  0,0,0,0,0,0,0,0,30,0,30,0,30,0,30,0,240,120,120,0,
  810,810,6480,6480,6480,6480,0,1080,6480,0,0,0,1080,6480,
  0,30,0,30,0,30,0,0,120,120,0,810,810,2160,2160,6480,
  6480,0,0,2160,0,2160,0,6480,6480,0,0,0,0,0,0,0,0,0,0,
  0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,30,0,30,0,120,120,
  120,0,0,810,6480,6480,0,6480,1080,1080,6480,0,0,0,0,
  6480,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,
  0,30,0,120,0,120,0,810,0,2160,2160,6480,0,1080,360,0,0,
  2160,0,6480,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,
  0,0,0,120,240,0,0,810,0,6480,6480,6480,0,1080,1080,0,0,
  0,0,1080,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,
  120,60,0,0,3240,12960,0,0,12960,0,0,25920,0,2160,0,0,
  25920,0,0,0,6480,0,12960,0,25920,2160,2160,0,0,2160,0,
  4320,12960,0,0,0,12960,12960,12960,0,2160,2160,25920,0,
  4320,0,2160,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,
  2160,1080,0,0,0,0,0,0,0,0,0,2160,0,1080,0,0,0,0,0,
  1080,0,0,0,0,0,2160,0,0,2160,0,0,0,0,0,0,2160,2160,0,
  2160,0,0,0,0,2160,0,0,0,0,0,2160,2160,2160,0,0,0,0,0,
  2160,0,0,0,0,0,0,360,2160,2160,0,360,0,0,0,0,2160,0,
  0,0,0,0,0,0,0,0,0,0,0,2160,5,30,30,30,30,30,30,0,
  60,30,30,30,0,30,60,0,60,0,0,30,30,60,30,0,30,0,30,
  0,0,0,60,0,0,270,2160,0,0,2160,0,0,0,2160,0,2160,0,0,
  30,30,30,30,30,30,0,0,30,30,30,0,30,0,0,120,30,30,30,
  30,120,30,30,30,0,30,0,0,0,0,30,30,30,60,60,60,60,60,
  60,60,60,60,60,60,60,30,30,30,30,30,120,0,30,30,30,0,
  30,120,120,0,270,0,30,30,120,30,270,30,0,30,0,0,60,60,
  270,270,0,0,1080,1080,0,1080,0,0,1080,0,0,1080,0,30,30,
  30,30,60,60,30,30,30,0,30,120,0,0,270,0,30,30,120,30,
  270,30,0,30,0,0,0,120,0,0,270,0,1080,0,1080,0,1080,0,
  0,1080,1080,0,1080,30,30,30,0,120,30,30,30,0,30,120,60,
  60,90,90,30,30,120,30,90,30,0,30,0,0,120,0,30,30,30,
  180,180,60,60,60,180,180,180,180,60,60,60,30,30,0,120,
  30,30,30,0,30,120,120,60,0,270,30,30,120,30,270,30,0,
  30,0,0,120,120,270,0,0,1080,0,0,1080,1080,0,1080,1080,
  0,0,0,1080,30,120,0,30,30,30,0,30,0,120,120,30,30,30,
  30,120,30,30,30,0,30,0,0,0,60,90,90,90,60,60,180,180,
  180,60,60,60,60,180,180,180,120,120,30,30,30,0,30,60,
  60,60,0,270,30,30,120,30,270,30,0,30,0,0,60,0,0,0,0,
  1080,0,1080,0,1080,0,0,1080,1080,1080,1080,0,60,120,0,
  60,0,0,0,60,480,0,1080,60,120,60,0,1080,60,0,120,0,0,
  0,0,2160,0,1080,2160,0,0,4320,8640,6480,720,720,0,4320,
  0,0,120,60,0,0,120,0,240,0,1080,0,60,240,240,120,0,0,
  0,240,0,0,120,480,0,0,0,0,4320,2160,0,2160,4320,0,8640,
  0,2160,720,6480,30,30,0,30,0,120,120,0,270,30,30,0,30,
  0,30,0,30,0,0,60,60,0,270,270,1080,0,1080,1080,0,0,
  1080,0,1080,0,0,1080,30,0,30,120,0,240,90,90,30,30,120,
  30,90,30,0,30,0,0,120,240,30,30,30,180,180,60,60,60,
  180,180,180,180,60,60,60,0,30,120,0,120,270,0,30,30,0,
  30,0,30,0,30,0,0,0,120,0,0,0,0,1080,0,0,0,1080,0,
  1080,1080,0,1080,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,
  0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,120,240,0,90,90,
  30,30,120,30,90,30,0,30,0,0,60,0,30,30,30,180,180,60,
  60,60,180,180,180,180,60,60,60,120,60,0,0,0,0,0,120,0,
  120,0,120,0,0,240,120,0,2160,1080,0,0,2160,2160,0,0,
  4320,0,4320,0,2160,720,60,0,0,120,60,60,60,1080,120,0,
  120,0,0,0,0,0,0,1080,6480,2160,0,0,0,2160,0,2160,720,
  4320,0,8640,0,0,120,60,120,120,2160,120,0,120,0,0,480,
  0,0,2160,0,0,2160,0,0,0,2160,2160,720,2160,4320,8640,0,
  30,0,30,0,270,30,30,0,270,0,0,1080,1080,0,0,0,0,0,0,
  0,0,0,120,0,120,0,0,0,0,30,1080,270,30,30,0,270,0,0,
  0,1080,0,0,0,0,0,0,0,0,120,0,120,120,0,0,0,30,0,30,
  0,30,0,30,0,0,120,0,0,270,270,1080,1080,1080,1080,0,0,
  1080,0,0,0,1080,1080,0,30,30,30,0,30,0,0,240,120,90,
  90,90,60,60,180,180,180,60,60,60,60,180,180,180,60,0,0,
  0,120,0,0,0,960,0,0,0,0,4320,720,2160,0,0,0,0,0,0,
  2160,2160,0,30,0,30,0,0,120,120,270,0,270,0,0,0,0,
  1080,1080,0,0,0,1080,0,1080,30,0,0,0,0,0,0,0,0,0,0,
  120,0,0,0,0,0,0,0,0,0,0,0,30,0,0,0,120,90,90,90,60,
  60,180,180,180,60,60,60,60,180,180,180,0,0,0,0,0,0,0,
  0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,120,0,0,270,0,0,
  1080,0,0,1080,1080,0,0,0,1080,1080,0,0,0,0,0,0,0,0,0,
  0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,
  0,0,0,60,0,0,0,0,4320,0,2160,0,8640,0,4320,4320,720,
  6480,720,0,1080,0,720,2160,4320,0,0,2160,2160,0,6480,
  8640,4320,0,30,30,0,0,0,0,0,0,0,0,0,0,120,0,30,0,0,
  0,120,0,0,0,0,0,120,0,120,0,0,120,0,120,0,0,0,0,120,
  120,0,60,0,0,0,60,60,60,60,0,0,0,0,0,0,60,60,60,60,
  0,0,0,60,60,0,0,0,0,60,60,60,60,0,0,0,0,60,60,60,0,
  0,0,0,60,60,60,60,60,60,0,0,0,60,60,0,0,0,60,0,0,0,
  0,0,0,60,60,60,30,30,30,30,30,30,0,0,30,30,30,0,30,
  60,60,0,0,0,30,30,60,30,270,30,60,30,60,0,0,0,0,0,0,
  0,2160,2160,0,0,0,2160,2160,0,2160,0,2160,30,30,30,30,
  30,60,0,30,30,30,0,30,0,60,0,810,0,30,30,60,30,270,
  30,60,30,30,30,30,30,0,0,0,0,6480,6480,0,0,0,0,6480,
  0,6480,0,0,30,30,30,30,30,30,30,30,30,30,30,60,0,60,
  810,0,30,30,240,30,270,30,30,30,0,60,0,60,0,0,810,0,
  6480,0,1080,0,0,0,6480,0,6480,0,6480,30,30,30,0,60,30,
  30,30,60,30,30,30,30,0,0,30,30,30,30,0,30,240,30,60,
  0,60,0,0,0,0,1080,6480,2160,2160,0,0,6480,6480,0,2160,
  0,0,30,30,0,60,30,30,30,60,30,30,30,30,0,0,30,30,30,
  30,270,30,0,30,60,0,60,240,0,0,810,1080,0,0,1080,0,0,
  6480,6480,0,6480,0,6480,30,60,0,30,30,30,0,30,0,60,240,
  0,0,30,30,60,30,90,30,60,30,30,30,30,30,0,0,0,2160,
  2160,6480,6480,0,0,0,2160,0,0,0,0,60,60,30,30,30,60,
  30,30,30,30,0,0,30,30,30,30,270,30,60,30,30,30,30,30,
  0,0,0,1080,0,6480,0,0,0,6480,6480,0,6480,0,0,30,60,30,
  30,30,60,60,30,60,0,0,30,0,30,30,6480,30,30,0,60,0,
  60,0,0,0,6480,2160,0,0,0,0,0,12960,0,0,12960,0,12960,
  240,30,30,30,0,0,60,0,0,0,30,60,60,30,540,30,30,60,
  30,60,30,60,0,0,0,0,12960,12960,0,0,0,0,12960,0,0,0,
  12960,30,30,0,30,0,60,120,0,0,30,30,60,30,270,30,60,
  30,30,30,30,30,0,0,810,1080,0,6480,1080,0,0,6480,6480,
  0,0,0,6480,30,30,30,60,0,60,810,0,30,30,0,30,0,30,30,
  30,0,60,0,60,0,0,270,6480,6480,0,360,0,0,0,0,0,0,0,
  2160,30,30,60,60,60,810,0,30,30,60,30,270,30,30,30,30,
  30,30,30,0,0,810,0,6480,6480,1080,0,0,0,6480,0,0,0,
  6480,0,0,60,0,0,0,30,60,60,30,540,30,30,60,30,60,30,
  60,0,0,0,0,0,12960,0,0,0,0,12960,0,0,0,12960,240,60,
  0,810,0,30,30,60,30,270,30,60,30,30,30,30,30,0,0,0,0,
  6480,2160,0,0,0,0,6480,0,0,0,0,30,30,0,0,60,30,30,30,
  540,60,60,30,60,30,60,30,0,0,0,2160,0,12960,51840,0,0,
  12960,12960,0,0,0,51840,30,1620,0,0,30,30,30,0,0,30,30,
  0,60,0,60,0,0,1620,0,12960,0,2160,0,0,0,0,0,0,0,
  12960,6480,0,60,30,30,30,540,60,60,30,60,30,60,30,0,0,
  0,0,0,12960,0,0,0,12960,12960,0,0,0,0,0,810,270,1620,
  810,270,270,3240,0,1620,0,1620,0,0,0,0,1080,0,0,0,0,0,
  0,2160,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,
  0,0,0,0,0,0,0,0,0,0,30,0,30,0,30,30,30,0,60,240,60,
  0,0,810,1080,6480,6480,1080,0,0,6480,0,0,0,0,6480,30,
  30,270,30,0,30,60,0,60,240,0,0,0,360,0,0,1080,0,0,
  2160,0,0,6480,0,6480,30,0,0,30,30,0,60,0,60,0,0,1620,
  2160,12960,0,2160,0,0,12960,0,0,0,0,12960,0,30,30,30,
  60,60,60,60,0,0,810,1080,6480,0,1080,0,0,6480,0,0,6480,
  0,6480,0,0,0,0,1080,0,0,0,0,0,360,2160,0,0,0,0,2160,
  0,0,0,0,0,30,30,0,60,0,60,0,0,810,0,2160,0,1080,0,0,
  0,0,0,0,0,6480,0,60,0,60,0,0,0,0,2160,0,0,0,0,0,
  12960,0,0,12960,0,0,60,0,60,0,0,0,0,1080,0,0,0,0,0,
  6480,0,0,6480,0,0,30,30,30,0,0,1620,51840,12960,0,2160,
  0,0,0,0,0,0,0,12960,30,30,0,0,0,2160,0,0,0,0,0,
  12960,12960,0,12960,0,0,30,0,0,1620,0,12960,0,2160,0,0,
  0,12960,0,0,0,12960,0,0,0,2160,51840,12960,0,0,0,12960,
  0,0,12960,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,
  0,0,0,0,0,0,0,0,0,0,0,0,2160,0,0,0,0,0,0,0,0,0,
  2160,0,0,0,0,2160,2160,0,0,0,0,0,0,0,0,2160,0,0,0,0,
  0,2160,0,0,0,0,0,0,0,2160,0,0,0,0,0,2160,0,2160,0,0,
  0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,
  0,0,30,30,30,30,30,0,0,30,30,30,0,30,120,0,0,270,0,
  30,30,120,30,270,30,120,30,120,0,0,0,0,270,0,0,0,1080,
  0,0,1080,0,1080,1080,1080,0,0,30,30,30,30,0,0,30,30,
  30,0,30,120,0,120,0,0,30,30,120,30,270,30,120,30,120,
  0,0,120,0,0,0,0,1080,0,1080,1080,0,0,1080,0,1080,0,0,
  30,30,30,0,120,30,30,30,0,30,60,60,60,90,90,30,30,60,
  30,90,30,120,30,0,0,0,0,30,30,30,180,180,60,60,60,180,
  180,180,180,60,60,60,30,30,0,120,30,30,30,0,30,60,60,
  60,0,270,30,30,60,30,270,30,120,30,120,0,0,120,270,0,
  270,1080,0,0,1080,0,0,0,1080,1080,1080,0,1080,30,0,0,
  30,30,30,0,30,0,120,120,30,30,30,30,120,30,30,30,0,30,
  120,0,0,120,90,90,90,60,60,180,180,180,60,60,60,60,180,
  180,180,0,120,30,30,30,0,30,60,60,60,0,270,30,30,60,
  30,270,30,120,30,120,0,0,0,270,0,0,0,0,1080,0,1080,0,
  1080,1080,1080,0,1080,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,
  0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,
  120,120,0,0,120,0,960,0,0,0,120,0,0,120,1080,0,0,240,
  60,0,0,480,0,2160,0,0,4320,2160,0,0,0,0,4320,4320,720,
  2160,2160,30,30,0,30,0,120,120,0,0,30,30,120,30,0,30,
  120,30,0,0,0,120,0,0,270,1080,0,0,1080,0,0,1080,1080,
  0,0,1080,0,30,0,30,120,0,120,90,90,30,30,120,30,90,30,
  120,30,0,0,0,0,30,30,30,180,180,60,60,60,180,180,180,
  180,60,60,60,0,30,0,0,0,270,0,30,30,0,30,0,30,120,30,
  0,0,0,120,0,270,0,0,1080,1080,0,0,1080,0,0,1080,0,
  1080,1080,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,
  0,0,0,0,0,0,0,0,0,0,0,0,0,120,240,0,90,90,30,30,
  120,30,90,30,120,30,0,0,0,0,30,30,30,180,180,60,60,60,
  180,180,180,180,60,60,60,60,60,0,1080,0,60,60,60,0,240,
  240,0,960,0,0,120,0,2160,1080,0,0,2160,2160,0,0,4320,0,
  0,0,720,720,60,2160,0,120,60,60,60,1080,120,120,0,0,0,
  0,240,1080,0,0,0,720,0,0,4320,2160,0,2160,720,0,0,0,
  1080,0,120,60,60,60,0,120,240,60,0,0,0,60,0,1080,0,0,
  2160,0,0,0,720,2160,720,2160,4320,4320,0,30,270,30,1080,
  0,30,30,1080,270,2160,0,0,1080,0,0,0,120,0,0,0,0,0,0,
  0,120,0,0,0,0,30,0,270,30,30,1080,270,0,0,0,2160,0,0,
  0,0,120,0,0,0,120,0,0,120,0,0,0,30,0,30,0,30,0,30,
  0,0,0,120,0,270,270,1080,0,1080,1080,0,1080,0,0,0,0,0,
  1080,60,30,30,30,0,30,120,0,0,120,90,90,90,60,60,180,
  180,180,60,60,60,60,180,180,180,60,0,0,60,60,0,0,0,
  240,0,1080,0,4320,4320,720,720,2160,0,0,0,0,0,2160,2160,
  0,30,0,30,120,0,0,120,270,0,270,0,1080,0,1080,1080,
  1080,1080,0,0,0,0,1080,30,0,0,0,0,0,0,0,0,0,120,0,0,
  0,0,120,0,0,0,0,0,0,0,30,120,0,0,120,90,90,90,60,60,
  180,180,180,60,60,60,60,180,180,180,0,480,0,0,0,2160,
  1080,0,4320,0,0,0,2160,4320,4320,0,0,0,720,0,240,0,0,
  0,270,270,0,1080,0,1080,0,1080,1080,1080,0,0,1080,0,0,
  0,0,0,0,0,1080,720,2160,0,4320,0,2160,0,0,2160,0,4320,
  4320,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,
  0,0,0,0,0,0,0,0,0,0,0,1080,1080,0,0,720,4320,0,4320,
  0,2160,0,0,4320,0,0,30,30,0,0,120,0,0,0,0,0,0,0,120,
  120,30,0,0,0,0,0,0,0,0,0,120,0,120,0,0,120,0,0,0,0,
  0,0,120,0,0,60,0,0,0,60,60,60,60,0,0,0,0,0,0,60,60,
  60,60,0,0,0,60,60,0,0,0,0,60,60,60,60,0,0,0,0,60,
  60,60,0,0,0,0,60,60,60,60,60,60,0,0,0,60,60,0,0,0,
  60,0,0,0,0,0,0,60,60,60,6,6,6,6,0,0,6,6,6,12,6,12,
  0,0,18,0,6,6,12,6,18,6,12,6,12,12,0,0,18,0,0,0,36,
  0,0,36,36,0,36,36,36,0,0,6,6,6,0,0,6,6,6,12,6,12,
  0,0,18,0,6,6,12,6,18,6,12,6,12,12,0,0,18,0,0,0,36,
  0,0,36,36,0,36,36,36,0,0,6,6,0,0,6,6,6,12,6,12,12,
  0,0,0,6,6,12,6,18,6,12,6,12,0,12,12,18,0,0,0,0,0,
  0,36,0,0,36,36,36,0,0,6,12,0,6,6,6,12,6,0,12,0,0,
  0,6,6,12,6,18,6,12,6,6,6,6,6,18,18,0,0,0,36,0,36,
  0,0,36,36,36,36,0,12,0,6,6,6,12,6,0,12,0,0,0,6,6,
  12,6,18,6,12,6,6,6,6,6,18,18,0,0,0,36,0,36,0,0,36,
  36,36,36,0,0,12,12,12,12,12,24,12,0,0,0,12,12,12,12,
  36,12,12,12,24,0,0,0,36,0,0,0,0,0,0,72,72,0,72,72,
  72,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,
  0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,6,6,0,6,0,12,
  0,0,0,6,6,12,6,18,6,12,6,6,6,6,6,0,18,0,0,0,36,0,
  0,0,0,36,36,0,36,0,6,12,6,12,0,0,18,0,6,6,12,6,18,
  6,12,6,0,12,12,12,18,18,0,0,36,36,0,36,36,0,36,36,
  36,36,0,0,6,12,12,0,18,0,6,6,12,6,18,6,12,6,6,6,6,
  6,0,18,0,0,36,36,0,0,36,0,36,36,0,36,0,0,0,24,0,0,
  0,12,12,24,12,36,12,12,12,12,12,12,12,0,36,0,0,72,72,
  0,0,0,0,72,72,0,72,0,12,12,0,18,0,6,6,12,6,18,6,12,
  6,6,6,6,6,0,18,0,0,36,36,0,0,36,0,36,36,0,36,0,12,
  0,0,0,12,12,12,12,36,12,24,12,12,12,12,12,0,36,0,0,
  0,72,0,0,0,0,72,72,72,72,0,0,36,0,12,12,12,12,36,12,
  12,12,0,24,0,0,36,0,0,0,72,0,0,72,72,0,72,72,72,0,
  0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,
  0,0,0,0,0,0,0,18,18,36,18,18,18,36,18,36,0,36,36,0,
  0,0,0,0,0,0,0,0,0,36,36,0,0,0,0,0,0,0,0,0,0,0,0,
  0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,6,0,6,0,6,
  12,6,0,12,12,12,0,18,0,0,36,36,0,0,36,0,0,0,0,36,0,
  12,6,18,6,0,6,12,0,12,12,18,18,0,0,36,36,0,36,36,0,
  36,36,36,36,0,12,0,0,12,0,0,24,0,0,0,36,0,0,72,72,
  0,72,72,0,0,0,0,72,0,0,6,12,6,12,12,12,12,18,0,0,0,
  36,0,0,36,36,0,0,0,36,0,0,0,0,0,0,36,0,0,0,0,0,0,
  36,0,0,0,36,0,0,0,0,0,0,0,6,12,12,0,0,18,18,0,0,
  36,36,0,36,36,0,0,0,36,36,0,0,24,0,0,0,36,36,0,0,
  72,72,0,72,72,0,0,0,72,72,0,12,12,0,0,18,18,0,0,36,
  36,0,36,36,0,0,0,36,36,0,6,6,6,0,36,0,0,72,72,0,0,
  72,0,0,72,0,72,0,6,6,36,0,0,0,0,72,0,72,0,0,72,72,
  72,0,0,6,36,0,0,0,72,0,0,72,72,0,0,72,72,72,0,36,0,
  0,0,72,0,0,72,72,0,0,72,72,72,0,18,0,0,0,36,0,0,0,
  0,0,0,0,36,0,0,0,0,0,0,36,0,0,0,0,36,0,0,0,0,0,0,
  0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,36,
  0,36,36,0,0,0,0,36,0,0,0,0,36,36,0,0,0,0,0,0,0,0,
  0,0,0,0,0,36,36,0,0,36,36,0,0,0,0,0,0,0,0,36,0,0,
  0,0,0,0,36,0,0,6,6,6,0,0,6,6,6,12,6,12,0,0,18,0,
  6,6,12,6,18,6,12,6,12,12,0,0,18,0,0,0,36,0,0,36,36,
  0,36,36,36,0,0,6,6,0,0,6,6,6,12,6,0,0,12,0,0,6,6,
  12,6,18,6,12,6,12,0,0,12,18,0,18,0,0,0,36,36,0,0,
  36,36,36,0,36,6,0,0,6,6,6,12,6,0,0,12,0,0,6,6,12,
  6,18,6,12,6,12,0,0,12,18,0,18,0,0,0,36,36,0,0,36,
  36,36,0,36,12,12,6,6,6,12,6,12,0,12,0,0,6,6,12,6,
  18,6,12,6,12,12,0,12,18,0,0,0,0,0,0,36,0,0,36,36,
  36,0,0,6,12,6,6,6,12,0,0,0,0,0,6,12,12,6,36,6,6,
  12,12,0,0,0,36,0,0,0,0,0,0,72,72,0,72,72,72,0,72,
  12,6,6,6,12,0,0,0,0,0,6,12,12,6,36,6,6,12,12,0,0,
  0,36,0,0,0,0,0,0,72,72,0,72,72,72,0,72,6,6,0,6,0,
  0,12,0,0,6,6,12,6,18,6,12,6,12,12,0,12,0,0,18,0,0,
  0,36,0,0,0,36,36,0,0,36,6,6,6,12,0,12,18,0,6,6,12,
  6,18,6,6,6,0,12,0,12,0,0,18,0,36,0,36,0,36,0,36,36,
  0,0,36,6,6,12,0,12,18,0,6,6,12,6,18,6,6,6,0,12,0,
  12,0,0,18,0,36,0,36,0,36,0,36,36,0,0,36,0,0,0,0,0,
  0,6,12,12,6,36,6,6,12,12,24,0,24,0,0,36,0,72,0,72,
  0,0,0,72,72,0,0,72,12,0,0,18,0,6,6,12,6,18,6,12,6,
  12,12,0,12,18,0,18,0,36,0,36,36,36,0,36,36,36,0,36,
  0,12,0,0,12,12,12,12,36,12,24,12,24,12,0,12,0,0,36,
  0,0,0,72,0,0,0,72,72,72,0,72,0,0,0,0,0,0,0,0,0,0,
  0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,36,0,12,
  12,12,12,36,12,24,12,24,12,0,12,36,0,0,0,72,0,0,72,
  72,0,72,72,72,0,0,0,18,18,36,18,18,18,36,18,36,0,0,
  36,0,0,0,0,0,0,0,0,0,0,36,36,0,0,0,0,0,0,0,0,0,0,
  0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,6,0,6,0,
  6,6,6,0,12,0,12,0,0,18,0,36,0,36,0,36,0,0,0,0,0,
  36,12,6,18,6,0,6,12,0,0,12,18,0,18,0,36,0,36,36,36,
  0,36,36,36,0,36,0,0,0,12,12,0,24,0,24,0,0,36,0,72,
  0,72,72,72,0,0,0,0,0,72,0,6,6,6,12,12,0,12,18,0,18,
  0,36,0,36,36,36,0,0,0,36,0,36,0,0,0,0,36,0,0,0,0,
  0,0,36,0,0,0,36,0,0,0,0,0,0,6,6,12,12,0,12,18,0,
  18,0,36,0,36,36,36,0,0,0,36,0,36,0,12,0,0,0,36,0,0,
  0,72,0,0,72,72,0,0,0,72,0,72,12,12,0,0,18,0,0,0,36,
  0,0,36,36,0,0,0,36,0,0,12,0,12,0,0,36,0,72,0,72,0,
  72,0,0,72,0,0,72,0,12,36,0,36,0,0,0,72,72,0,0,72,
  72,72,0,72,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,36,0,
  0,0,72,0,0,72,72,0,0,72,72,0,0,0,18,0,0,0,36,0,0,
  0,0,0,0,0,36,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,
  36,0,0,0,0,36,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,36,
  0,36,36,0,0,0,0,0,0,0,0,0,0,0,0,36,0,0,0,0,36,0,
  36,0,0,0,0,36,0,36,0,36,36,0,0,0,0,0,0,0,0,36,0,0,
  0,0,0,0,0,36,0,6,6,0,12,6,6,6,12,6,6,6,6,0,18,6,
  6,6,6,18,6,12,6,12,0,12,0,18,0,0,36,0,0,0,36,0,36,
  36,36,36,0,0,6,0,0,6,6,6,12,6,0,12,12,18,18,6,6,12,
  6,18,6,12,6,12,12,12,0,18,18,18,36,36,36,36,36,36,36,
  36,36,36,36,36,0,12,6,6,6,12,6,6,6,6,0,18,6,6,6,6,
  18,6,12,6,12,0,12,0,18,0,0,36,0,0,0,36,0,36,36,36,
  36,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,
  0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,12,12,12,12,
  12,0,0,0,36,0,12,12,24,12,36,12,12,12,12,24,12,0,36,
  0,0,0,72,0,0,72,72,0,72,72,72,72,72,6,6,0,6,0,12,
  12,0,18,6,6,12,6,18,6,12,6,12,12,12,0,0,18,18,36,0,
  36,36,0,0,36,36,36,0,36,36,6,0,6,12,0,0,18,18,6,6,
  12,6,18,6,12,6,0,12,12,0,6,6,6,36,36,12,12,12,36,36,
  36,36,12,12,12,12,6,12,12,12,18,0,6,6,12,6,18,6,12,
  6,12,12,12,0,0,18,18,0,36,36,36,0,36,0,36,36,0,36,
  36,0,0,0,0,0,36,12,12,24,12,36,12,12,12,12,24,12,0,
  0,36,36,72,72,72,72,0,0,72,72,72,0,72,72,12,0,0,18,
  18,6,6,12,6,18,6,12,6,0,12,12,0,6,6,6,36,36,12,12,
  12,36,36,36,36,12,12,12,6,6,0,36,12,6,6,6,36,12,12,
  6,24,12,24,0,0,36,36,72,0,72,72,0,0,72,72,72,72,72,
  72,6,36,0,12,6,6,6,36,12,12,6,0,12,0,0,36,0,0,0,72,
  0,0,72,72,0,72,72,72,0,0,36,0,12,6,6,6,36,12,12,6,
  0,12,0,0,36,0,0,0,72,0,0,72,72,0,72,72,72,0,0,18,
  18,18,36,18,18,18,36,18,36,0,36,0,0,0,0,36,0,0,0,0,
  0,36,36,36,0,0,0,18,18,36,18,18,18,36,18,36,36,0,0,
  0,0,0,0,36,0,0,0,36,0,36,36,0,0,0,6,0,6,0,6,12,6,
  0,12,12,0,0,18,18,36,36,36,36,0,36,36,0,0,0,36,36,6,
  6,18,6,0,6,12,0,12,0,18,18,18,36,36,36,36,36,36,36,
  36,36,36,36,36,6,0,0,12,6,0,12,0,0,0,36,36,72,72,72,
  72,72,72,72,0,0,0,72,72,0,6,0,6,12,12,12,0,18,0,0,
  36,36,0,0,36,36,36,0,0,36,0,0,18,0,0,0,36,36,0,0,0,
  0,36,36,0,0,0,36,36,0,0,0,0,0,12,6,12,12,0,0,18,18,
  18,36,36,36,36,36,36,36,36,36,36,36,36,0,24,0,24,0,
  36,0,0,72,72,0,0,72,72,72,0,0,72,72,72,12,12,12,0,
  18,0,0,36,36,0,0,36,36,36,0,0,36,0,0,12,12,0,0,36,
  36,72,72,72,72,0,72,72,0,72,0,72,72,12,0,36,0,0,72,
  0,72,72,72,0,72,72,72,72,0,0,0,36,0,0,0,72,0,0,72,
  72,0,72,72,72,72,72,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,
  6,6,0,0,12,12,0,0,0,0,0,0,12,12,6,0,0,0,0,12,0,0,
  0,0,12,0,0,0,0,0,0,12,0,0,0,0,12,0,0,36,0,0,0,36,
  36,36,36,0,0,0,0,0,0,36,36,36,36,0,0,0,12,12,0,0,0,
  0,12,12,12,12,0,0,0,0,12,12,12,0,0,0,0,12,12,12,36,
  36,36,0,0,0,36,36,0,0,0,36,0,0,0,0,0,0,12,12,12,6,
  0,0,6,6,6,12,6,0,0,12,0,0,6,6,12,6,18,6,12,6,12,0,
  0,12,18,0,18,0,0,0,36,36,0,0,36,36,36,0,36,0,12,6,
  6,6,12,6,6,6,6,0,18,6,6,6,6,18,6,12,6,12,0,12,0,
  18,0,0,36,0,0,0,36,0,36,36,36,36,0,0,0,0,0,0,0,0,
  0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,
  0,0,0,0,0,0,0,0,12,12,12,12,12,0,0,0,0,0,12,12,24,
  12,36,12,12,12,12,0,12,24,36,0,0,0,0,0,0,72,0,0,72,
  72,72,0,72,6,6,0,6,0,0,12,0,18,6,6,12,6,18,6,12,6,
  0,0,12,12,0,0,18,36,0,0,36,0,0,36,36,36,0,0,36,6,0,
  6,0,0,12,0,18,6,6,12,6,18,6,12,6,0,0,12,12,0,0,18,
  36,0,0,36,0,0,36,36,36,0,0,36,12,6,12,12,12,0,0,6,
  6,12,6,18,6,12,6,12,0,12,12,0,0,18,0,0,0,36,0,0,0,
  36,36,0,0,36,0,0,0,0,0,36,12,12,24,12,36,12,12,12,
  12,0,12,24,0,0,36,72,0,0,72,0,0,72,72,72,0,0,72,12,
  12,0,0,18,6,6,12,6,18,6,12,6,12,0,12,12,18,0,18,36,
  0,0,36,36,0,36,36,36,36,0,36,6,6,0,0,12,6,6,6,36,
  12,12,6,0,0,0,12,0,0,36,72,0,0,72,0,0,0,72,72,72,0,
  72,6,0,0,12,6,6,6,36,12,12,6,0,0,0,12,0,0,36,72,0,
  0,72,0,0,0,72,72,72,0,72,0,0,12,6,6,6,36,12,12,6,
  24,0,24,12,36,0,0,0,0,0,0,72,0,72,72,72,72,0,0,0,0,
  0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,
  0,0,18,18,36,18,18,18,36,18,36,0,0,36,0,0,0,0,0,0,
  0,0,0,0,36,36,0,0,0,6,0,6,0,6,12,6,0,0,12,12,0,0,
  18,36,0,0,36,0,0,36,0,0,0,0,36,6,6,0,6,0,6,12,0,
  12,12,18,0,18,36,0,0,36,36,0,36,0,0,36,0,36,6,0,0,
  12,6,0,0,0,12,0,0,36,72,0,0,72,72,0,72,0,0,0,0,72,
  0,6,0,6,12,0,12,12,18,0,18,36,0,0,36,36,0,36,0,0,
  36,0,36,18,0,0,0,0,36,0,0,0,0,36,0,0,0,0,0,36,0,0,
  0,0,0,12,6,12,0,0,12,18,0,18,36,0,0,36,36,0,36,36,
  36,36,0,36,0,24,0,24,0,36,0,0,72,0,0,0,72,0,72,0,0,
  72,0,72,12,0,12,0,18,0,0,36,0,0,0,36,0,36,0,0,36,0,
  0,0,12,12,0,0,36,72,0,0,72,0,0,72,0,72,0,0,72,0,0,
  0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,12,36,0,36,0,0,0,
  72,72,0,0,72,72,72,0,72,36,0,0,72,0,0,0,72,0,72,0,
  72,72,0,0,0,18,0,0,0,36,0,0,0,0,0,0,0,36,0,0,0,0,
  0,0,0,0,0,0,0,0,0,0,0,0,0,36,0,0,0,0,36,0,0,0,0,
  0,0,0,36,36,36,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,
  0,0,0,0,0,36,0,0,0,0,36,0,36,0,0,0,0,36,0,36,0,0,
  0,0,0,0,36,36,0,0,0,36,0,0,0,0,0,0,0,36,0,12,0,6,
  6,6,12,6,0,12,0,0,0,6,6,12,6,18,6,12,6,6,6,6,6,18,
  18,0,0,0,36,0,36,0,0,36,36,36,36,0,0,12,12,12,12,12,
  0,12,24,0,0,12,12,12,12,36,12,12,12,24,0,0,0,36,0,
  36,0,0,0,72,72,72,72,72,72,72,0,72,0,0,0,0,0,0,0,0,
  0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,
  0,0,0,0,0,6,6,0,6,0,12,12,0,0,6,6,12,6,18,6,12,6,
  6,6,6,6,0,18,18,0,0,36,36,0,0,0,36,36,0,36,36,6,12,
  6,0,0,12,18,18,6,6,12,6,18,6,12,6,0,12,12,12,18,18,
  18,36,36,36,36,36,36,36,36,36,36,36,36,0,6,0,12,12,0,
  0,6,6,12,6,18,6,12,6,6,6,6,6,0,18,18,0,0,36,36,0,
  0,0,36,36,0,36,36,0,0,24,0,0,0,12,12,24,12,36,12,12,
  12,12,12,12,12,0,36,36,72,72,72,72,0,0,0,72,72,0,72,
  72,0,12,0,18,18,6,6,12,6,18,6,12,6,6,6,6,6,18,18,
  18,36,36,36,36,36,36,36,36,36,36,36,36,0,0,0,0,0,0,
  0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,
  0,12,0,0,12,12,12,12,36,12,12,12,0,0,0,24,36,0,36,
  72,72,0,72,72,0,0,72,72,72,0,72,0,0,12,12,12,12,36,
  12,24,12,12,12,12,12,36,36,0,0,0,72,0,72,72,72,72,72,
  72,72,0,6,18,6,36,18,6,6,36,18,36,0,0,36,0,0,0,0,0,
  0,0,0,0,0,12,12,0,0,0,18,6,36,18,6,6,36,18,36,0,0,
  36,0,0,0,0,0,0,0,0,0,0,12,12,0,0,0,6,0,6,0,6,12,
  6,0,12,12,12,0,18,18,36,36,36,36,0,36,36,0,0,0,36,
  36,0,6,6,6,0,6,12,0,0,12,18,18,18,12,12,36,36,36,12,
  12,12,12,36,36,36,12,0,0,12,12,0,0,0,24,0,36,36,72,
  72,72,72,72,72,72,0,0,0,72,72,0,6,12,6,12,12,12,12,
  18,0,18,36,36,0,36,36,36,36,0,0,36,0,36,6,0,0,0,36,
  36,0,0,0,0,12,12,0,0,0,12,12,0,0,0,0,0,0,6,12,0,0,
  12,18,18,18,12,12,36,36,36,12,12,12,12,36,36,36,0,24,
  0,0,0,36,36,0,72,72,72,0,72,72,72,0,0,72,72,72,12,
  12,12,0,18,18,0,36,36,36,0,36,36,36,0,0,36,36,0,6,6,
  6,0,36,36,72,72,72,72,0,72,72,0,72,0,72,72,6,6,36,0,
  36,0,0,0,72,72,0,0,72,72,72,0,72,6,36,0,36,0,0,0,
  72,72,0,0,72,72,72,0,72,36,36,0,72,72,72,0,72,72,72,
  0,72,72,72,0,18,18,0,0,36,36,0,0,0,0,0,0,36,36,18,
  0,0,0,36,36,0,0,0,0,36,0,36,0,0,36,0,36,0,0,0,0,
  36,36,0,12,0,0,0,12,12,12,12,0,0,0,0,0,0,12,12,12,
  12,0,0,0,36,36,0,0,0,0,36,36,36,36,0,0,0,0,36,36,
  36,0,0,0,0,36,36,36,12,12,12,0,0,0,12,12,0,0,0,12,
  0,0,0,0,0,0,36,36,36,12,12,12,12,12,12,12,12,12,0,
  36,12,12,12,12,36,12,12,12,24,0,24,0,36,0,0,72,0,0,
  0,72,0,72,72,72,72,0,0,12,12,12,12,12,0,24,0,0,0,12,
  12,24,12,36,12,12,12,12,12,12,12,36,36,0,0,0,72,0,72,
  0,0,72,72,72,72,0,6,6,0,6,0,12,0,0,18,6,6,12,6,18,
  6,12,6,6,6,6,6,0,18,0,36,0,36,0,0,0,36,36,36,0,36,
  0,6,12,6,12,0,12,0,18,6,6,12,6,18,6,12,6,0,12,12,
  12,18,18,0,36,0,36,0,36,0,36,36,36,36,36,0,12,6,12,
  12,12,0,0,6,6,12,6,18,6,12,6,6,6,6,6,0,18,0,0,0,
  36,0,0,0,0,36,36,0,36,0,0,0,24,0,0,36,12,12,24,12,
  36,12,12,12,12,12,12,12,0,36,0,72,0,72,0,0,0,72,72,
  72,0,72,0,0,12,0,0,18,6,6,12,6,18,6,12,6,6,6,6,6,
  0,18,0,36,0,36,0,0,0,36,36,36,0,36,0,6,6,0,0,12,6,
  6,6,36,12,12,6,12,6,12,6,0,36,0,0,0,72,0,0,0,72,72,
  72,72,72,0,6,0,0,12,6,6,6,36,12,12,6,0,12,0,12,36,
  0,0,72,0,0,0,72,0,0,72,72,72,0,0,0,0,12,6,6,6,36,
  12,12,6,12,6,12,6,0,36,0,0,0,72,0,0,0,72,72,72,72,
  72,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,
  0,0,0,0,0,0,0,18,18,36,18,18,18,36,18,36,36,0,36,0,
  0,0,0,0,0,0,0,0,0,36,36,0,0,0,6,0,6,0,6,12,6,0,
  12,12,12,0,18,0,36,0,36,0,0,0,36,0,0,0,36,0,6,6,0,
  6,0,6,12,0,12,0,18,18,0,36,0,36,0,36,0,36,0,0,36,
  36,0,6,0,0,12,6,0,12,0,12,0,36,0,72,0,72,0,72,0,72,
  0,0,0,72,0,0,6,12,6,12,12,12,12,18,0,0,36,0,0,0,36,
  0,36,0,0,36,0,0,18,0,0,0,0,36,0,0,0,0,36,0,0,0,0,
  0,36,0,0,0,0,0,12,6,12,12,0,12,18,18,0,36,0,36,0,
  36,0,36,36,36,36,36,0,0,24,0,24,0,36,36,0,72,0,72,0,
  72,0,72,0,0,72,72,0,12,0,12,0,18,18,0,36,0,36,0,36,
  0,36,0,0,36,36,0,6,6,6,0,36,0,72,0,72,0,0,0,72,0,
  72,0,72,0,6,6,36,0,0,72,0,72,0,72,0,72,0,72,72,0,0,
  6,36,0,0,0,0,0,0,72,0,0,72,72,72,72,0,36,0,0,72,0,
  72,0,72,0,72,0,72,72,0,0,18,0,0,0,36,0,0,0,0,0,0,
  0,36,0,0,0,0,0,0,36,0,0,0,0,36,0,0,0,0,0,0,0,0,0,
  0,0,0,0,0,0,0,0,0,0,36,36,36,0,0,0,0,0,0,0,0,0,0,
  0,0,0,0,36,0,0,0,0,36,36,0,0,0,0,0,0,0,0,0,0,0,0,
  0,36,36,0,0,0,0,0,0,0,36,36,0,0,0,36,0,0,0,0,0,0,
  36,0,0,12,6,6,6,12,0,0,0,0,0,6,12,12,6,36,6,6,12,
  12,0,0,0,36,0,0,0,0,0,0,72,72,0,72,72,72,0,72,0,12,
  0,12,0,12,24,0,36,12,12,12,12,36,12,12,12,0,0,24,0,
  0,0,36,72,0,0,72,0,0,72,72,72,0,0,72,6,6,0,0,0,0,
  0,36,6,12,12,6,36,6,6,12,0,0,12,0,0,0,36,72,0,0,72,
  0,0,72,72,72,0,0,72,6,12,24,12,24,0,0,6,12,12,6,36,
  6,6,12,12,0,12,0,0,0,36,0,0,0,72,0,72,0,72,72,0,0,
  72,0,0,0,0,0,36,6,12,12,6,36,6,6,12,0,0,12,0,0,0,
  36,72,0,0,72,0,0,72,72,72,0,0,72,24,12,0,0,36,12,12,
  12,12,36,12,12,12,0,0,24,0,36,0,36,72,0,0,72,72,72,
  72,72,72,72,0,72,0,12,0,0,24,12,12,12,72,24,12,12,0,
  0,0,0,0,0,72,0,0,0,144,0,0,144,144,144,144,0,144,0,
  0,0,12,12,6,6,36,12,6,12,0,0,0,0,0,0,0,0,0,0,0,0,
  0,0,72,72,0,0,0,0,0,24,12,12,12,72,24,12,12,0,0,0,
  0,72,0,0,0,0,0,0,144,144,0,144,144,144,0,0,0,0,0,0,
  0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,
  36,36,72,36,36,36,36,36,72,0,0,0,0,0,0,0,0,0,0,0,
  216,0,72,72,0,0,0,12,0,6,0,6,6,12,0,0,12,0,0,0,36,
  72,0,0,72,0,72,72,0,0,0,0,72,12,0,0,12,0,12,24,0,
  24,0,36,0,36,72,0,0,72,72,72,72,72,0,72,0,72,6,0,0,
  6,12,0,0,0,0,0,0,72,72,0,0,72,72,72,72,0,0,0,0,72,
  0,6,6,0,12,0,12,0,36,0,0,72,0,0,0,72,72,72,0,0,72,
  0,72,36,0,0,0,0,72,0,0,0,0,72,0,0,0,0,216,72,0,0,
  0,0,0,6,12,12,0,0,0,36,0,36,72,0,0,72,72,72,72,72,
  72,72,0,72,0,12,0,12,0,36,0,0,72,0,0,0,72,72,72,0,
  0,72,0,72,24,0,24,0,36,0,0,72,0,0,0,72,72,72,0,0,
  72,0,0,0,12,0,0,0,72,144,0,0,144,0,144,144,0,144,0,
  0,144,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,72,0,0,
  0,0,0,0,144,144,0,144,144,144,0,144,0,0,0,0,0,0,0,0,
  0,0,0,0,0,0,0,0,36,0,0,0,72,0,0,0,0,0,0,0,72,0,0,
  0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,72,0,0,0,0,72,0,0,
  0,0,0,0,216,0,72,72,0,0,0,0,0,0,0,0,0,0,0,0,0,0,
  0,0,0,0,0,0,0,0,72,0,0,0,0,72,0,72,0,0,0,0,72,0,
  72,72,72,216,0,0,0,72,72,0,0,0,0,0,0,0,0,0,0,0,72,
  0,12,0,0,12,0,24,0,0,0,12,12,24,12,36,12,12,12,12,
  12,12,12,0,36,0,0,0,72,0,0,0,0,72,72,0,72,72,6,6,
  12,0,0,0,36,0,6,12,12,6,36,6,6,12,0,24,12,24,36,36,
  0,0,72,72,0,72,72,0,72,72,72,72,72,6,0,0,12,0,0,0,
  6,12,12,6,36,6,6,12,6,12,6,12,0,36,0,0,72,72,0,0,0,
  0,72,72,0,72,72,0,0,12,0,0,0,6,12,12,6,36,6,6,12,6,
  12,6,12,0,36,0,0,72,72,0,0,0,0,72,72,0,72,72,0,24,
  0,36,0,12,12,24,12,36,12,12,12,12,12,12,12,0,36,0,0,
  72,72,0,0,72,0,72,72,72,72,72,0,0,0,0,0,0,0,0,0,0,
  0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,
  12,24,12,12,72,12,12,24,0,0,0,0,72,0,0,0,144,0,0,
  144,0,0,144,144,144,0,144,0,0,0,0,0,0,0,0,0,0,0,0,
  0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,36,36,72,36,
  36,36,36,36,72,0,36,72,0,0,0,0,0,0,0,0,0,0,72,72,0,
  0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,
  0,0,0,0,0,12,0,6,0,6,6,12,0,24,12,24,0,36,0,0,72,
  72,0,0,72,0,0,0,0,72,72,0,12,36,0,0,12,12,0,12,24,
  36,36,0,0,72,72,0,72,72,0,72,72,72,72,72,12,0,0,0,0,
  0,0,0,0,0,72,0,0,144,144,0,144,144,0,0,0,0,144,144,
  0,6,6,12,12,24,12,24,36,0,0,0,72,0,0,72,72,0,0,0,
  72,0,72,0,0,0,0,72,36,0,0,0,0,0,72,0,0,0,72,0,0,0,
  0,0,0,6,0,12,0,0,0,36,36,0,0,72,72,0,72,72,0,0,0,
  72,72,72,0,12,0,0,0,36,36,0,0,72,72,0,72,72,0,0,0,
  72,72,72,12,24,12,0,36,36,0,0,72,72,0,72,72,0,0,0,
  72,72,0,12,6,12,0,72,0,0,72,72,0,0,72,0,0,72,0,72,
  72,0,12,72,0,0,0,0,144,0,144,0,0,144,144,144,0,144,0,
  36,0,0,0,72,0,0,72,72,0,72,72,72,72,72,72,0,0,0,144,
  0,0,144,144,0,0,144,144,144,0,36,0,0,0,72,0,0,0,0,0,
  0,0,72,216,0,0,0,0,0,72,0,0,0,0,72,0,216,0,0,0,0,
  0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,72,
  0,72,72,0,0,0,0,72,0,0,0,0,72,72,216,0,0,0,0,0,0,
  0,0,0,0,0,0,72,72,216,0,72,72,0,0,0,0,0,0,0,0,72,
  0,0,0,0,0,0,72,72,72,6,0,6,0,0,12,0,18,6,6,12,6,
  18,6,12,6,0,0,12,12,0,0,18,36,0,0,36,0,0,36,36,36,
  0,0,36,0,6,0,12,12,0,0,6,6,12,6,18,6,12,6,6,6,6,6,
  0,18,18,0,0,36,36,0,0,0,36,36,0,36,36,0,0,0,0,0,0,
  0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,
  0,0,0,0,12,0,0,18,6,6,12,6,18,6,12,6,6,6,6,6,0,18,
  0,36,0,36,0,0,0,36,36,36,0,36,0,0,0,0,0,0,0,0,0,0,
  0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,12,
  0,0,12,12,12,12,36,12,12,12,0,0,0,24,0,0,36,72,0,0,
  72,0,0,0,72,72,0,0,72,0,0,12,12,12,12,36,12,24,12,
  12,12,12,12,0,36,0,0,0,72,0,0,0,72,72,72,0,72,0,0,
  0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,
  0,0,0,18,18,36,18,18,18,36,18,36,36,0,36,0,0,0,0,0,
  0,0,0,0,0,36,36,0,0,0,6,0,6,0,6,0,6,0,0,12,12,0,
  18,18,36,0,36,36,0,0,36,0,0,0,36,36,0,6,0,6,0,6,0,
  0,12,12,0,18,18,36,0,36,36,0,0,36,0,0,0,36,36,12,0,
  0,12,12,0,0,0,24,0,36,36,72,0,72,72,0,0,72,0,0,0,
  72,72,0,6,12,6,12,12,12,12,0,0,18,36,0,0,36,0,0,36,
  0,0,0,0,36,18,0,0,0,0,36,0,0,0,0,36,0,0,0,0,0,36,
  0,0,0,0,0,12,6,12,12,0,12,0,18,18,36,0,36,36,0,0,
  36,36,36,0,36,36,0,0,0,24,0,0,36,0,72,0,72,0,0,0,
  72,0,0,0,72,72,12,12,12,0,0,18,0,36,0,36,0,0,0,36,
  0,0,0,36,0,6,6,6,0,0,36,72,0,72,72,0,0,72,0,72,0,
  0,72,6,6,0,0,36,72,0,72,72,0,0,72,0,72,0,0,72,6,0,
  0,36,0,0,0,72,0,0,0,72,72,0,72,72,0,36,0,72,0,72,0,
  0,0,72,0,72,0,72,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,
  18,0,0,0,36,0,0,0,0,0,0,0,36,0,0,36,0,0,0,0,0,0,
  0,36,0,0,0,0,0,0,36,36,36,0,0,0,0,0,0,0,0,0,0,0,
  0,0,36,0,0,0,0,0,0,36,36,0,0,0,0,0,0,36,36,0,0,0,
  0,0,0,0,0,0,0,0,0,0,36,36,0,0,0,36,0,0,0,0,0,0,0,
  0,36,6,6,12,0,12,18,0,6,6,12,6,18,6,6,6,0,12,0,12,
  0,0,18,0,36,0,36,0,36,0,36,36,0,0,36,0,0,0,0,0,36,
  6,12,12,6,36,6,6,12,0,0,12,0,0,0,36,72,0,0,72,0,0,
  72,72,72,0,0,72,12,0,0,18,18,6,6,12,6,18,6,12,6,0,
  12,12,0,6,6,6,36,36,12,12,12,36,36,36,36,12,12,12,0,
  12,0,36,12,12,12,12,36,12,24,12,0,12,24,12,0,0,36,72,
  0,0,72,0,0,72,72,72,72,72,72,0,0,0,0,0,0,0,0,0,0,
  0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,36,0,12,
  12,12,12,36,12,24,12,0,12,24,12,36,36,0,0,72,72,0,72,
  72,72,72,72,72,72,0,18,18,18,36,18,18,18,36,18,0,0,
  36,36,0,0,0,36,0,0,0,0,0,36,36,36,0,0,0,18,18,36,
  18,18,18,36,18,0,36,0,36,0,0,0,0,36,0,0,0,36,0,36,
  36,0,0,0,6,0,6,0,6,6,6,0,12,12,12,0,0,18,36,36,0,
  36,0,36,36,0,0,0,0,36,12,6,18,6,0,6,0,0,12,12,18,
  18,18,36,36,36,36,36,36,36,36,36,36,36,36,0,0,0,12,
  12,0,24,0,24,0,0,36,72,72,72,72,72,72,72,0,0,0,0,72,
  0,6,6,6,0,12,12,12,0,0,18,36,36,0,36,0,36,36,0,0,0,
  0,36,18,0,0,0,36,36,0,0,0,0,36,36,0,0,0,36,36,0,0,
  0,0,0,6,6,0,12,0,12,18,18,18,36,36,36,36,36,36,36,
  36,36,36,36,36,0,0,0,12,0,36,36,0,72,72,72,0,72,72,
  72,0,0,72,72,72,0,12,12,0,18,18,0,36,36,36,0,36,36,
  36,0,0,36,36,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,
  0,12,12,0,0,36,72,0,72,72,72,0,72,72,72,0,0,72,12,0,
  0,36,0,72,0,72,0,72,0,72,72,72,72,72,36,36,0,72,72,
  72,0,72,72,72,0,72,72,72,0,6,6,0,0,0,12,0,0,0,0,0,
  0,0,12,6,0,0,0,12,0,0,0,0,0,0,0,12,0,0,12,0,12,0,
  0,0,0,12,12,0,36,0,0,0,36,36,36,36,0,0,0,0,0,0,36,
  36,36,36,0,0,0,12,12,0,0,0,0,12,12,12,12,0,0,0,0,
  12,12,12,0,0,0,0,12,12,12,36,36,36,0,0,0,36,36,0,0,
  0,36,0,0,0,0,0,0,12,12,12,0,0,12,0,0,0,6,12,12,6,
  36,6,6,12,6,12,6,12,0,36,0,0,72,72,0,0,0,0,72,72,0,
  72,72,12,12,0,18,0,6,6,12,6,18,6,12,6,6,6,6,6,0,18,
  0,0,36,36,0,0,36,0,36,36,0,36,0,12,12,0,0,12,12,12,
  12,36,12,24,12,12,12,12,12,0,36,36,0,0,72,72,0,0,0,
  72,72,0,72,72,12,36,0,12,12,12,12,36,12,12,12,0,24,0,
  24,0,0,36,0,72,0,72,0,72,0,72,72,0,0,72,36,0,12,12,
  12,12,36,12,24,12,12,12,12,12,0,36,0,0,72,72,0,0,72,
  0,72,72,0,72,0,0,18,18,36,18,18,18,36,18,36,0,36,36,
  0,0,0,0,0,0,0,0,0,0,36,36,0,0,0,0,0,0,0,0,0,0,0,
  0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,6,0,6,0,6,
  6,6,0,12,0,12,0,18,18,0,36,36,36,0,36,0,0,0,0,36,
  36,12,6,18,6,0,6,12,0,12,12,0,18,18,0,36,36,36,0,36,
  0,36,36,0,36,36,12,0,0,12,12,0,24,0,24,0,36,36,0,72,
  72,72,0,72,0,0,0,0,72,72,0,6,6,6,12,12,12,12,0,0,
  18,0,36,0,36,0,36,0,0,0,0,0,36,0,0,0,0,36,0,0,0,0,
  0,0,36,0,0,0,36,0,0,0,0,0,0,6,6,0,12,0,12,0,18,18,
  0,36,36,36,0,36,0,0,0,0,36,36,0,12,0,12,0,0,36,0,0,
  72,72,0,0,72,0,0,0,0,72,72,12,12,12,0,0,18,0,0,36,
  36,0,0,36,0,0,0,0,36,0,6,6,6,0,0,36,0,72,0,72,0,
  72,0,0,72,0,72,72,6,6,0,0,36,0,0,72,72,0,0,0,72,72,
  0,0,72,6,0,0,36,0,72,0,72,0,72,0,0,72,0,72,72,0,36,
  0,0,72,72,0,0,72,0,0,72,0,72,0,0,0,0,0,0,0,0,0,0,
  0,0,0,0,0,18,0,0,0,36,0,0,0,0,0,0,0,36,0,0,36,0,
  0,0,0,0,0,0,36,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,36,
  0,36,36,0,0,0,36,0,0,0,0,0,0,36,36,0,0,0,0,0,0,36,
  36,0,0,0,0,0,0,0,0,36,36,0,0,0,0,0,0,0,0,36,0,0,
  0,0,0,0,0,0,36,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,
  0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,
  0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,
  0,0,12,24,12,12,72,12,12,24,0,0,0,0,0,0,72,144,0,0,
  144,0,0,0,144,144,0,0,144,0,0,0,0,0,0,0,0,0,0,0,0,
  0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,
  0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,36,36,
  72,36,36,36,36,36,36,72,0,72,0,0,0,0,216,0,0,0,0,0,
  72,72,0,0,0,0,0,6,0,6,6,0,0,0,12,0,0,36,0,72,72,
  72,0,0,0,72,0,0,0,72,72,0,12,0,12,0,12,12,0,12,24,
  0,36,36,72,72,72,72,0,0,72,0,72,0,72,72,12,0,0,0,0,
  0,0,0,0,0,72,0,144,144,144,144,0,0,144,0,0,0,144,0,
  0,6,6,12,12,24,12,24,0,0,36,72,72,0,72,0,0,72,0,0,
  0,0,72,36,0,0,0,0,72,0,0,0,0,72,216,0,0,0,0,72,0,
  0,0,0,0,6,12,12,24,0,24,0,36,36,72,72,72,72,0,0,72,
  72,72,0,72,72,0,0,0,12,0,0,36,0,72,72,72,0,0,0,72,
  0,0,0,72,72,12,24,12,0,0,36,0,72,72,72,0,0,0,72,0,
  0,0,72,0,0,6,0,0,36,36,72,72,72,72,0,0,72,0,72,0,
  72,72,12,12,0,0,72,144,0,144,144,0,0,144,0,144,0,0,
  144,12,0,0,72,0,72,0,72,0,0,0,72,72,0,72,72,0,72,0,
  144,144,144,0,0,0,144,0,0,0,144,0,0,0,0,0,0,0,0,0,
  0,0,0,0,0,0,36,0,0,0,72,0,0,0,0,0,0,0,72,0,0,72,
  0,0,0,0,0,0,0,72,0,72,0,0,0,0,0,72,72,0,0,0,0,0,
  0,0,216,216,72,0,0,0,72,0,0,0,0,0,0,72,72,0,0,0,0,
  0,0,72,72,0,0,0,0,0,0,0,0,0,0,0,0,0,72,72,0,0,0,
  0,0,0,0,0,0,0,0,0,72,12,0,0,36,12,12,12,12,36,12,
  24,12,12,12,12,12,0,36,0,72,0,72,0,0,0,72,72,72,72,
  72,72,0,36,0,12,12,12,12,36,12,12,12,0,24,0,0,36,0,
  36,72,72,0,72,72,72,0,72,72,72,0,72,0,0,0,0,0,0,0,
  0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,18,
  18,18,36,18,18,18,36,18,36,0,36,36,0,0,0,36,0,0,0,0,
  0,36,36,36,0,0,0,18,18,36,18,18,18,36,18,36,36,0,36,
  0,0,0,0,36,0,0,0,36,0,36,36,0,0,0,6,0,6,0,6,0,6,
  0,12,12,0,0,18,0,36,36,36,0,0,36,36,0,0,0,36,0,12,
  6,18,6,0,6,12,0,12,12,18,18,18,36,36,36,36,36,36,36,
  36,36,36,36,36,12,0,0,12,0,0,24,0,0,0,36,0,72,72,72,
  72,72,72,72,0,0,0,72,0,0,6,12,6,12,12,12,12,18,0,18,
  36,36,0,36,36,36,36,0,0,36,0,36,18,0,0,0,36,36,0,0,
  0,0,36,36,0,0,0,36,36,0,0,0,0,0,12,6,12,12,0,12,18,
  18,18,36,36,36,36,36,36,36,36,36,36,36,36,0,0,0,24,0,
  0,36,0,72,72,72,0,0,72,72,0,0,72,72,72,0,12,12,0,0,
  18,0,36,36,36,0,0,36,36,0,0,0,36,0,6,6,6,0,36,0,72,
  72,72,0,0,72,72,0,72,0,72,0,6,6,36,0,36,72,0,72,72,
  72,0,72,72,72,72,0,72,6,36,0,36,0,72,0,72,72,72,0,
  72,72,72,72,72,0,36,0,72,72,72,0,0,72,72,0,72,0,72,
  0,6,6,0,0,12,0,0,0,0,0,0,0,12,0,6,0,0,0,12,12,0,
  0,0,0,12,0,12,0,0,12,0,0,0,0,0,0,0,12,0,36,0,0,0,
  36,36,36,36,0,0,0,0,0,0,36,36,36,36,0,0,0,12,12,0,
  0,0,0,12,12,12,12,0,0,0,0,12,12,12,0,0,0,0,12,12,
  12,36,36,36,0,0,0,36,36,0,0,0,36,0,0,0,0,0,0,12,12,
  12,6,0,0,12,6,6,6,36,12,12,6,0,0,0,12,0,0,36,72,0,
  0,72,0,0,0,72,72,72,0,72,0,0,12,6,6,6,36,12,12,6,
  12,6,12,6,0,36,0,0,0,72,0,0,0,72,72,72,72,72,0,0,0,
  0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,
  0,0,36,36,36,36,36,36,72,36,72,0,0,72,0,0,0,0,0,0,
  0,0,0,0,72,72,0,0,0,0,0,12,0,12,0,12,0,12,24,12,0,
  36,36,72,0,72,72,0,0,72,0,0,0,72,72,6,6,0,0,0,6,0,
  0,0,12,0,36,36,72,0,72,72,0,0,72,0,0,0,72,72,6,0,0,
  0,6,0,0,0,12,0,36,36,72,0,72,72,0,0,72,0,0,0,72,72,
  0,12,12,6,24,12,24,12,0,0,36,72,0,0,72,0,0,72,0,0,
  72,0,72,36,0,0,0,0,72,0,0,0,0,72,0,0,0,0,0,72,0,0,
  0,0,0,0,12,24,12,0,12,0,36,36,72,0,72,72,0,0,72,72,
  72,72,72,72,0,0,0,0,0,0,72,0,144,0,144,0,0,0,144,0,
  0,0,144,144,24,12,24,0,0,36,0,72,0,72,0,0,0,72,0,0,
  72,72,0,0,12,12,0,0,72,144,0,0,144,0,0,144,0,144,0,
  144,144,0,6,0,0,36,0,0,0,72,0,0,0,0,0,0,0,72,12,0,
  0,72,0,0,0,144,0,0,0,144,144,144,0,144,0,72,0,72,0,
  72,0,0,0,72,0,72,72,72,0,0,0,0,0,0,0,0,0,0,0,0,0,
  0,0,36,0,0,0,72,0,0,0,0,0,216,0,72,0,0,72,0,0,0,0,
  0,0,216,72,0,0,0,0,0,0,72,72,72,0,0,0,0,0,0,0,0,0,
  0,0,0,0,72,0,0,0,0,0,216,0,72,0,0,0,0,0,216,72,0,
  0,0,0,0,0,0,0,0,0,0,0,0,0,72,72,0,0,0,72,0,0,0,0,
  0,0,72,72,72,36,0,12,6,6,6,36,12,12,6,0,12,0,0,36,
  0,0,0,72,0,0,72,72,0,72,72,72,0,0,0,36,36,36,36,36,
  36,72,36,0,0,0,72,0,0,0,216,0,0,0,0,0,0,72,72,0,0,
  0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,
  0,0,0,0,12,0,0,0,12,12,12,0,24,0,24,0,0,36,72,72,0,
  72,0,72,0,0,0,0,0,72,6,6,36,12,0,6,0,0,0,12,36,0,
  36,72,72,0,72,72,72,0,72,72,72,0,72,6,0,0,6,6,0,12,
  0,12,0,0,36,72,72,0,72,72,72,0,0,0,0,0,72,0,0,6,6,
  0,12,0,12,0,0,36,72,72,0,72,72,72,0,0,0,0,0,72,0,0,
  0,0,72,0,0,0,0,0,216,72,0,0,0,72,0,0,0,0,0,0,12,
  12,0,24,0,24,36,0,36,72,72,0,72,72,72,0,0,72,72,0,
  72,0,0,0,0,0,72,0,0,72,72,0,0,72,72,0,0,0,72,0,72,
  0,12,0,0,36,0,0,72,72,0,0,72,72,0,0,0,72,0,0,0,0,
  0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,12,0,0,72,144,
  0,0,144,144,0,0,144,144,0,0,144,0,0,0,0,0,0,0,0,0,
  0,0,0,0,0,0,0,72,0,0,0,144,0,0,144,144,0,0,144,144,
  0,0,0,36,0,0,0,72,0,0,0,0,0,0,0,72,0,0,0,0,0,0,0,
  0,0,0,0,0,0,0,0,0,0,72,0,0,0,0,72,0,0,72,0,0,0,
  216,0,216,72,0,0,0,0,0,0,0,0,72,72,0,0,0,0,0,0,0,
  0,0,0,0,0,72,0,0,0,0,72,0,72,0,0,0,0,72,0,72,0,72,
  72,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,72,0,0,36,36,
  36,36,36,36,72,36,72,0,72,36,0,0,0,0,0,0,0,0,0,216,
  72,72,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,
  0,0,0,0,0,0,0,0,0,12,0,12,0,12,0,0,0,12,24,12,0,
  36,0,0,72,72,0,0,72,72,0,0,0,72,0,6,6,36,12,0,6,24,
  0,24,12,36,36,0,0,72,72,0,72,72,72,72,72,72,72,0,6,
  0,0,0,6,0,12,0,0,0,36,0,0,72,72,0,72,72,72,0,0,0,
  72,0,0,12,12,6,24,12,24,12,36,0,0,0,72,0,0,72,72,72,
  0,0,72,0,0,0,0,0,0,72,0,0,0,0,0,0,72,0,0,0,72,216,
  0,0,0,0,0,0,0,24,12,0,12,36,36,0,0,72,72,0,72,72,
  72,72,0,72,72,0,0,0,0,0,0,0,72,0,0,144,144,0,0,144,
  144,0,0,144,144,0,0,12,0,0,0,36,0,0,72,72,0,72,72,
  72,0,0,0,72,0,12,12,0,0,72,0,0,144,144,0,0,144,144,
  0,144,0,144,0,12,6,72,0,0,0,0,72,0,72,0,72,72,72,72,
  0,0,0,72,0,0,0,144,0,0,144,144,0,144,0,144,144,0,0,
  0,0,0,72,0,0,0,72,0,0,0,0,0,0,36,0,0,0,72,0,0,0,
  0,0,0,0,72,0,0,0,0,0,0,72,0,0,0,0,72,0,0,0,0,0,0,
  0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,
  216,72,72,0,0,0,0,72,0,0,0,0,72,72,0,0,0,0,0,0,0,
  0,0,0,0,0,0,72,72,0,72,72,72,0,0,0,72,216,0,0,0,0,
  0,0,0,0,0,0,72,0,0,18,6,36,18,6,6,36,18,36,0,0,36,
  0,0,0,0,0,0,0,0,0,0,12,12,0,0,0,0,0,18,0,18,0,18,
  0,0,36,36,0,0,0,36,0,0,0,0,0,36,0,0,0,0,0,0,0,6,
  6,0,0,0,0,36,0,0,0,0,12,0,0,0,0,0,12,0,0,0,0,0,
  36,0,0,0,36,0,0,0,72,0,0,0,72,0,0,0,0,0,72,0,0,0,
  0,0,0,18,0,18,36,0,36,36,0,0,0,36,0,0,0,0,0,36,0,
  0,0,0,0,6,0,0,0,0,36,0,0,0,0,12,0,0,0,0,0,12,0,0,
  0,0,0,36,18,36,0,0,36,0,0,0,12,0,0,0,0,0,12,12,12,
  0,0,0,0,0,0,72,0,0,0,0,72,0,0,0,0,0,72,0,0,0,0,0,
  36,0,36,0,0,0,0,36,0,0,0,0,0,36,0,0,0,0,0,0,36,36,
  0,0,0,72,0,0,0,0,0,72,0,216,0,0,0,0,0,0,0,0,0,0,
  0,0,0,0,0,0,0,0,0,0,36,0,0,0,0,0,0,0,0,0,0,72,72,
  0,0,0,0,0,0,72,0,0,0,0,0,72,0,216,0,0,0,0,0,0,0,
  0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,
  0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,12,12,12,0,0,0,0,
  0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,
  0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,12,12,0,0,0,12,0,0,
  0,0,0,0,0,0,0,18,0,18,0,0,36,18,0,36,0,36,0,0,0,0,
  36,0,0,0,36,0,0,0,0,0,0,36,18,6,6,0,18,36,0,0,36,
  0,0,0,0,12,0,0,0,12,0,12,12,0,0,0,0,0,0,0,0,0,72,
  0,0,0,0,0,0,72,0,0,0,72,0,0,0,0,0,0,0,0,36,18,36,
  36,0,36,0,0,0,0,36,0,0,0,36,0,0,0,0,0,0,6,0,0,0,
  36,0,0,0,0,0,0,12,0,0,0,12,0,0,0,0,0,0,0,0,0,36,
  0,0,0,0,0,0,12,0,0,0,12,0,0,0,0,0,0,0,72,0,0,0,0,
  0,0,0,72,0,0,0,72,0,0,0,0,0,0,36,36,0,0,0,0,0,0,
  36,0,0,0,36,0,0,0,0,0,0,36,0,36,0,0,0,0,72,0,0,0,
  72,0,0,216,0,0,0,0,36,0,0,0,0,0,0,0,0,0,0,72,72,0,
  0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,72,
  0,0,0,72,0,0,216,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,
  0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,
  0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,12,0,12,12,0,0,0,
  0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,
  0,12,12,0,0,0,0,0,0,0,0,12,0,0,0,0,0,0,0,0,0,0,6,
  0,6,0,6,0,0,12,12,0,18,18,36,0,36,36,0,0,36,0,0,0,
  36,36,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,
  0,0,0,0,6,6,6,0,12,12,12,0,0,18,36,36,0,36,0,36,36,
  0,0,0,0,36,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,
  0,0,0,0,6,6,0,12,0,12,0,18,18,0,36,36,36,0,36,0,0,
  0,0,36,36,0,0,0,12,0,0,36,0,72,72,72,0,0,0,72,0,0,
  0,72,72,0,12,12,0,0,18,0,36,36,36,0,0,36,36,0,0,0,
  36,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,12,12,0,
  0,36,72,0,72,72,0,0,72,0,0,0,0,72,12,0,0,36,0,72,0,
  72,0,72,0,0,0,0,72,72,0,36,0,72,72,72,0,0,72,72,0,
  0,0,72,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,18,0,0,0,36,
  0,0,0,0,0,0,0,36,0,0,36,0,0,0,0,0,0,0,36,0,36,0,
  0,0,36,36,0,0,0,0,0,0,0,0,36,36,0,0,0,0,0,36,0,0,
  0,0,0,0,36,36,0,0,0,0,0,0,36,36,0,0,0,0,0,0,0,36,
  0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,36,6,0,0,
  0,6,0,0,0,12,0,36,36,72,0,72,72,0,0,72,0,0,0,72,72,
  0,6,0,6,12,0,12,12,18,0,18,36,0,0,36,36,0,36,0,0,
  36,0,36,6,0,0,0,0,36,0,0,0,0,12,0,0,0,0,0,12,0,0,
  0,0,0,0,6,12,0,0,12,18,18,18,12,12,36,36,36,12,12,
  12,12,36,36,36,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,
  0,0,0,12,0,12,0,18,18,0,36,0,36,0,36,0,36,0,0,36,
  36,0,0,12,12,0,36,36,72,72,72,72,0,0,72,0,72,0,72,
  72,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,12,36,0,36,
  0,72,0,72,72,72,0,72,72,72,72,72,36,36,0,72,72,72,0,
  72,0,72,0,72,72,72,0,18,18,0,0,36,36,0,0,0,0,0,0,
  36,36,18,0,0,0,36,36,0,0,0,0,36,0,36,0,0,36,0,36,0,
  0,0,0,36,36,0,12,0,0,0,12,12,12,12,0,0,0,0,0,0,12,
  12,12,12,0,0,0,36,36,0,0,0,0,36,36,36,36,0,0,0,0,
  36,36,36,0,0,0,0,36,36,36,12,12,12,0,0,0,12,12,0,0,
  0,12,0,0,0,0,0,0,36,36,36,0,0,6,6,0,12,0,12,0,0,
  36,72,72,0,72,72,72,0,0,0,0,0,72,0,0,0,0,0,0,0,0,
  0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,
  0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,36,0,0,0,72,0,
  0,0,0,0,0,0,72,0,0,12,0,0,0,36,0,0,72,72,0,72,72,
  72,0,0,0,72,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,
  0,0,12,0,0,72,144,0,144,144,0,0,144,0,0,0,0,144,0,0,
  0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,72,0,144,144,144,0,
  144,144,0,0,0,0,144,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,
  36,0,0,0,72,216,0,0,0,0,0,0,72,0,0,72,0,216,0,0,0,
  0,0,72,0,72,0,0,0,72,72,0,0,0,0,0,0,0,0,72,72,0,0,
  0,0,0,72,72,0,0,0,0,0,0,72,72,0,0,0,0,0,72,0,0,0,
  0,0,0,216,216,72,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,
  0,0,0,72,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,
  0,0,0,6,6,12,12,0,12,18,0,18,0,36,0,36,36,36,0,0,0,
  36,0,36,0,12,0,12,0,36,0,0,72,0,0,0,72,72,72,0,0,
  72,0,72,12,12,12,0,18,0,0,36,36,0,0,36,36,36,0,0,36,
  0,0,12,12,12,0,0,36,72,72,0,72,0,72,72,0,0,0,0,72,
  12,12,36,0,36,72,0,0,72,72,0,72,0,0,72,0,72,12,36,0,
  36,0,72,0,72,72,72,0,0,0,72,0,72,36,0,0,72,72,0,0,
  72,72,72,0,0,72,0,0,0,18,0,0,0,36,0,0,0,0,0,0,0,
  36,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,36,0,0,0,0,
  36,0,0,36,0,0,0,36,36,0,0,0,0,0,0,0,0,36,36,0,0,0,
  0,0,0,0,0,0,0,0,0,0,0,36,0,0,0,0,36,0,36,0,0,0,0,
  36,0,36,36,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,
  36,0,0,0,0,36,0,0,0,0,0,0,12,0,0,0,12,0,0,0,0,0,
  0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,
  0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,
  0,0,0,0,0,0,0,0,0,0,36,0,0,0,0,72,0,0,0,0,0,72,0,
  0,0,0,0,0,0,0,0,0,72,0,0,0,72,0,0,0,0,0,0,0,0,0,
  0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,
  0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,
  0,0,0,12,0,0,0,12,12,0,0,0,0,0,0,0,0,12,12,0,0,0,
  0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,
  0,0,12,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,
  0,12,0,0,0,36,36,0,0,72,72,0,72,72,0,0,0,72,72,72,
  12,12,0,0,18,18,0,0,36,36,0,36,36,0,0,0,36,36,0,12,
  0,12,0,36,36,72,72,72,72,0,72,0,0,72,0,72,72,0,12,
  36,0,36,72,0,72,72,72,0,72,72,72,72,0,72,0,0,0,0,0,
  0,0,0,0,0,0,0,0,0,0,0,36,36,0,72,72,72,0,72,72,0,
  0,72,72,72,0,18,18,0,0,36,36,0,0,0,0,0,0,36,36,18,
  0,0,0,36,36,0,0,0,0,36,0,36,0,0,36,0,36,0,0,0,0,
  36,36,0,12,0,0,0,12,12,12,12,0,0,0,0,0,0,12,12,12,
  12,0,0,0,36,36,0,0,0,0,36,36,36,36,0,0,0,0,36,36,
  36,0,0,0,0,36,36,36,12,12,12,0,0,0,12,12,0,0,0,12,
  0,0,0,0,0,0,36,36,36,0,0,0,0,0,0,0,0,0,0,0,0,0,0,
  0,0,0,0,0,0,12,0,0,72,0,144,144,144,0,0,0,144,0,0,
  0,144,144,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,72,
  0,0,0,144,0,0,144,144,0,0,0,144,144,0,0,0,0,0,0,0,
  0,0,0,0,0,0,0,0,0,36,0,0,0,72,0,0,0,0,0,0,0,72,
  216,0,0,0,0,0,72,0,0,0,0,72,0,216,0,0,0,0,0,0,0,0,
  0,0,0,0,72,0,0,0,72,72,0,0,0,0,0,0,0,0,72,72,0,0,
  0,0,0,0,72,0,0,0,0,72,0,216,0,0,0,0,0,0,0,0,0,0,
  0,0,0,72,216,72,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,
  0,72,72,72,12,12,0,0,36,0,72,72,72,0,0,72,72,0,0,0,
  72,0,12,0,36,0,0,72,0,72,0,72,0,72,0,0,72,0,0,0,36,
  0,0,0,72,0,0,72,72,0,0,0,72,72,0,0,0,0,0,0,0,0,0,
  0,0,0,0,0,0,0,18,0,0,0,36,0,0,0,0,0,0,0,36,0,0,0,
  0,0,0,36,0,0,0,0,36,0,0,0,0,0,0,0,0,0,0,0,0,0,0,
  36,0,0,0,36,36,0,0,0,0,0,0,0,0,36,36,0,0,0,0,0,0,
  36,0,0,0,0,36,36,0,0,0,0,0,0,0,0,0,0,0,0,0,36,36,
  0,36,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,36,0,0,6,
  6,0,0,36,72,0,72,72,0,0,72,0,72,0,0,72,6,0,0,36,0,
  72,0,72,0,72,0,0,72,0,72,72,0,36,0,72,72,72,0,0,72,
  72,0,72,0,72,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,36,0,
  0,0,72,0,0,0,0,0,0,0,72,0,0,72,0,0,0,0,0,0,0,72,
  0,72,0,0,0,72,72,0,72,0,0,0,0,0,0,72,72,0,72,0,0,
  0,72,0,0,0,0,0,0,72,72,0,0,0,0,0,0,72,72,0,0,0,0,
  0,0,0,72,0,216,0,0,0,0,216,0,0,0,0,0,0,0,0,0,0,0,
  0,72,6,36,0,36,0,0,0,72,72,0,0,72,72,72,0,72,36,0,
  0,72,0,72,0,72,0,72,0,72,72,0,0,0,36,0,0,216,72,0,
  0,0,0,0,0,0,72,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,216,
  0,72,0,0,0,0,72,0,0,0,0,0,0,0,72,72,72,0,0,0,0,0,
  0,0,0,0,0,0,0,0,72,72,0,0,0,0,216,0,216,72,0,0,0,
  0,72,0,0,0,0,0,0,0,0,72,0,0,0,0,0,0,72,72,0,0,0,
  72,0,0,0,0,0,0,0,72,0,36,0,0,0,72,0,0,72,72,0,0,
  72,72,72,0,0,36,0,0,0,72,0,0,0,0,0,0,216,72,0,0,0,
  0,0,0,0,0,0,0,0,0,0,0,0,0,0,72,0,0,0,0,72,216,0,
  0,0,0,0,0,0,0,0,0,0,0,0,0,0,72,0,72,72,0,0,0,0,0,
  0,0,0,0,0,0,0,72,0,0,0,0,72,216,0,0,0,0,0,0,216,
  72,0,72,72,0,0,0,0,0,0,0,0,72,0,0,0,0,0,0,72,72,
  72,36,0,0,0,72,0,0,0,0,0,0,0,72,0,0,0,0,0,0,72,0,
  0,0,0,72,0,0,0,0,0,0,0,0,0,0,0,0,0,0,72,0,0,0,72,
  0,0,72,0,0,0,0,0,0,0,72,0,72,0,0,0,0,72,0,0,0,0,
  72,72,0,0,0,0,0,0,0,0,0,0,0,0,0,72,72,0,72,0,216,
  0,0,0,0,216,0,0,0,0,0,0,0,0,0,0,72,0,0,6,0,0,0,
  12,0,0,0,0,0,0,0,12,0,0,12,0,0,0,0,0,0,0,12,0,0,
  0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,12,0,0,
  0,0,0,0,12,12,0,0,0,0,0,0,12,12,0,0,0,0,0,0,0,0,
  0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,12,0,0,0,
  0,12,0,0,0,0,12,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,
  0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,12,0,0,0,0,12,0,
  12,0,0,0,0,12,0,12,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,
  0,0,0,0,12,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,
  0,0,0,0,0,12,0,0,0,0,12,12,0,0,0,0,0,0,0,0,0,0,0,
  0,0,12,12,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,
  12,0,0,0,0,0,12,12,12,12,0,0,0,0,0,0,0,0,0,0,0,0,
  0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,12,12,12,0,0,0,12,
  12,0,0,0,12,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,
  0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,12,12,12,0,0,0,12,
  12,0,0,0,12,0,0,0,0,0,0,0,0,0,12,0,0,0,0,12,12,12,
  0,0,0,0,12,12,12,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,
  0,0,12,12,12,0,0,0,0,12,12,12,0,0,0,0,0,0,0,0,0,0,
  0,0,0,0,0,0,0,0,12,12,12,0,0,0,0,0,0,0,0,0,0,0,0,
  0,0,0,0,0,0,12,12,12,12,12,0,0,0,12,0,0,0,0,0,0,0,
  0,0,12,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,12 ],

sizesset := 
[ infinity,4,6,8,12,16,24,32,36,48,72,80,96,120,128,144,192,
  200,240,288,336,384,432,480,512,576,648,672,720,768,864,960,
  1008,1152,1296,1440,1536,1728,2160,2304,2400,2688,2880,3072,
  3456,3840,4032,4320,4608,4800,5184,5760,6480,6912,7200,7680,
  8064,8192,8640,9216,10080,10368,11520,12288,12960,13824,14400,
  15360,15552,17280,18432,21504,21600,23040,24192,25920,27648,
  28224,28800,30240,34560,36864,38880,40320,41472,43200,46080,
  50400,51840,55296,56448,57600,60480,62208,69120,73728,77760,
  80640,86400,90720,92160,103680,110592,120960,124416,129600,138240,
  155520,161280,172800,184320,193536,207360,233280,241920,259200,
  276480,294912,302400,311040,322560,345600,362880,373248,387072,
  414720,483840,518400,552960,604800,622080,645120,691200,725760,
  777600,829440,921600,967680,1036800,1088640,1105920,1119744,
  1161216,1209600,1244160,1451520,1555200,1658880,1814400,1935360,
  2073600,2177280,2211840,2332800,2419200,2488320,2903040,3110400,
  3265920,3628800,3732480,3870720,4064256,4147200,4354560,4423680,
  4665600,4838400,4976640,5443200,5806080,6220800,6350400,6531840,
  6773760,7257600,7464960,7962624,8709120,8847360,9676800,9953280,
  10077696,10368000,11612160,12441600,12700800,13996800,14515200,
  17418240,18662400,19353600,19906560,21772800,23224320,24883200,
  26127360,29030400,32659200,34836480,36288000,37324800,38102400,
  40642560,43545600,49766400,52254720,58060800,60963840,62208000,
  65318400,69672960,74649600,76204800,87091200,93312000,101606400,
  104509440,116121600,121927680,124416000,145152000,149299200,
  152409600,154828800,162570240,174182400,185794560,195955200,
  203212800,207360000,209018880,223948800,248832000,261273600,
  265420800,279936000,290304000,313528320,318504960,348364800,
  373248000,398131200,435456000,447897600,522547200,559872000,
  597196800,609638400,668860416,696729600,746496000,783820800,
  1045094400,1119744000,1161216000,1219276800,1306368000,1321205760,
  1343692800,1393459200,1524096000,1567641600,1592524800,1625702400,
  1741824000,1828915200,1866240000,2090188800,2239488000,2286144000,
  2395008000,2438553600,2612736000,2786918400,2799360000,2874009600,
  3135283200,3483648000,3657830400,3919104000,3981312000,4478976000,
  4702924800,4877107200,5016453120,5225472000,5486745600,5643509760,
  6227020800,6270566400,6718464000,6967296000,7315660800,7838208000,
  8360755200,9405849600,10450944000,11147673600,11197440000,12541132800,
  13168189440,14370048000,14631321600,14929920000,15676416000,
  16721510400,18811699200,19160064000,19595520000,19906560000,
  20901888000,23514624000,24385536000,26127360000,26873856000,
  29262643200,31352832000,36578304000,37623398400,39191040000,
  41803776000,43893964800,47029248000,48771072000,50164531200,
  54867456000,56435097600,57480192000,62705664000,65840947200,
  67184640000,75246796800,78033715200,94058496000,100329062400,
  104509440000,109734912000,131681894400,134369280000,141087744000,
  146313216000,156764160000,188116992000,200658124800,201553920000,
  219469824000,235146240000,250822656000,258660864000,263363788800,
  268738560000,275904921600,282175488000,313528320000,322486272000,
  329204736000,376233984000,413857382400,431101440000,438939648000,
  468202291200,470292480000,526727577600,548674560000,627056640000,
  658409472000,877879296000,940584960000,1053455155200,1128701952000,
  1207084032000,1316818944000,1343692800000,1404606873600,1410877440000,
  1551965184000,1609445376000,1755758592000,1843546521600,1881169920000,
  2149908480000,2257403904000,2351462400000,2508226560000,2615348736000,
  2633637888000,2809213747200,3009871872000,3160365465600,3950456832000,
  4389396480000,4514807808000,5267275776000,5643509760000,5794003353600,
  6449725440000,6584094720000,6772211712000,8427641241600,8465264640000,
  8778792960000,9217732608000,10534551552000,11466178560000,
  13436928000000,13544423424000,14485008384000,15049359360000,
  15801827328000,17557585920000,18059231232000,18106260480000,
  21069103104000,22574039040000,23044331520000,23410114560000,
  23702740992000,26336378880000,28443289190400,29496744345600,
  31603654656000,39504568320000,45148078080000,47087773286400,
  47405481984000,52438656614400,52672757760000,56435097600000,
  57940033536000,60197437440000,63207309312000,64497254400000,
  67722117120000,70230343680000,79009136640000,90296156160000,
  96745881600000,101131694899200,105345515520000,110361968640000,
  110612791296000,142216445952000,180592312320000,193491763200000,
  210691031040000,217275125760000,225740390400000,252829237248000,
  253957939200000,260730150912000,269684519731200,270888468480000,
  276531978240000,284432891904000,288947699712000,316036546560000,
  372471644160000,379243855872000,386983526400000,421382062080000,
  474054819840000,526727577600000,564915326976000,651825377280000,
  737418608640000,1083553873920000,1106127912960000,1160950579200000,
  1218998108160000,1264146186240000,1516975423488000,1580182732800000,
  1757514350592000,1769804660736000,1883051089920000,1896219279360000,
  2031663513600000,2370274099200000,2708884684800000,2844328919040000,
  2902376448000000,3539609321472000,3687093043200000,4424511651840000,
  4977575608320000,5214603018240000,5688657838080000,6083703521280000,
  6320730931200000,7964120973312000,8090535591936000,8126654054400000,
  8532986757120000,9481096396800000,10113169489920000,11377315676160000,
  12189981081600000,13273534955520000,14158437285888000,14221644595200000,
  16591918694400000,17382010060800000,17698046607360000,18962192793600000,
  19910302433280000,20858412072960000,22754631352320000,28443289190400000,
  29865453649920000,30474952704000000,30971581562880000,32506616217600000,
  34764020121600000,34828517376000000,36502221127680000,41716824145920000,
  45509262704640000,50565847449600000,56886578380800000,59730907299840000,
  63207309312000000,63712967786496000,72814820327424000,79641209733120000,
  85329867571200000,97339256340480000,113773156761600000,
  142216445952000000,156438090547200000,158176291553280000,
  159282419466240000,176980466073600000,187725708656640000,
  204791682170880000,208584120729600000,227546313523200000,
  292017769021440000,344163799203840000,417168241459200000,
  424753118576640000,426649337856000000,477847258398720000,
  485432135516160000,573416710078464000,573606332006400000,
  682638940569600000,796412097331200000,849506237153280000,
  910185254092800000,1365277881139200000,1462797729792000000,
  2184444609822720000,2336142152171520000,2389236291993600000,
  2730555762278400000,2925595459584000000,3716589787545600000,
  4095833643417600000,6257523621888000000,6371296778649600000,
  7008426456514560000,7509028346265600000,7592461994557440000,
  7747410198528000000,7964120973312000000,10922223049113600000,
  11468334201569280000,13652778811392000000,15768959527157760000,
  16686729658368000000,16724654043955200000,17521066141286400000,
  17839630980218880000,18772570865664000000,21503126627942400000,
  27305557622784000000,34131947028480000000,38227780671897600000,
  40958336434176000000,41299655904460800000,43688892196454400000,
  47784725839872000000,56317712596992000000,57341671007846400000,
  65825897840640000000,75260943197798400000,92156256976896000000,
  100347924263731200000,122875009302528000000,123898967713382400000,
  131066676589363200000,131407996059648000000,153593761628160000000,
  163833345736704000000,170830394877542400000,184312513953792000000,
  210252793695436800000,210642873090048000000,229366684031385600000,
  245750018605056000000,491500037210112000000,567682542977679360000,
  585704211008716800000,1070377858813132800000,1228750093025280000000,
  1238989677133824000000,1337972323516416000000,1351625102327808000000,
  1638333457367040000000,1720250130235392000000,2064300156282470400000,
  2293666840313856000000,2890020218795458560000,4013916970549248000000,
  4281511435252531200000,4816700364659097600000,5529375418613760000000,
  8257200625129881600000,8410111747817472000000,10321500781412352000000,
  11009600833506508800000,14745001116303360000000,16588126255841280000000,
  18062626367471616000000,19266801458636390400000,26491852005625036800000,
  26541002009346048000000,30964502344237056000000,37157402813084467200000,
  41286003125649408000000,44153086676041728000000,81097506139668480000000,
  144501010939772928000000,176940013395640320000000,
  208150265758482432000000,275240020837662720000000,
  330288025005195264000000,385336029172727808000000,
  619290046884741120000000,778536058940817408000000,
  851523814466519040000000,1040751328792412160000000,
  1061640080373841920000000,1083757582048296960000000,
  1105875083722752000000000,1618411322525456793600000,
  1816584137528573952000000,2312016175036366848000000,
  2890020218795458560000000,2998805560364630016000000,
  4768533361012506624000000,4932301173410915942400000,
  6192900468847411200000000,6502545492289781760000000,
  6936048525109100544000000,7077600535825612800000000,
  8859492052209185587200000,9289350703271116800000000,
  9537066722025013248000000,10260073516345196544000000,
  11147220843925340160000000,13872097050218201088000000,
  14862961125233786880000000,15259306755240021196800000,
  15606109181495476224000000,18165841375285739520000000,
  20808145575327301632000000,24276169837881851904000000,
  27248762062928609280000000,30964502344237056000000000,
  31212218362990952448000000,33441662531776020480000000,
  36331682750571479040000000,37157402813084467200000000,
  44588883375701360640000000,45517818446028472320000000,
  53506660050841632768000000,55736104219626700800000000,
  71528000415187599360000000,73470736228933435392000000,
  96430341300475133952000000,110206104343400153088000000,
  124848873451963809792000000,166465164602618413056000000,
  175170613261683916800000000,177116953409035960320000000,
  207817225333268860108800000,208081455753273016320000000,
  238426668050625331200000000,247963734772650344448000000,
  277441941004364021760000000,408731430943929139200000000,
  462403235007273369600000000,490477717132714967040000000,
  495432037507792896000000000,501624937976640307200000000,
  572224003321500794880000000,624244367259819048960000000,
  712100981911200989184000000,873942114163746668544000000,
  1062701720454215761920000000,1486296112523378688000000000,
  1487782408635902066688000000,1618411322525456793600000000,
  1634925723775716556800000000,1747884228327493337088000000,
  1820712737841138892800000000,1948286487499395563520000000,
  1969934115138277736448000000,2179900965034288742400000000,
  2247279722135348576256000000,2663442633641894608896000000,
  2675333002542081638400000000,3329303292052368261120000000,
  3433344019929004769280000000,3932739513736860008448000000,
  3939868230276555472896000000,4046441166344898478080000000,
  4993954938078552391680000000,5785820478028508037120000000,
  8025999007626244915200000000,10195991331910377799680000000,
  11652561522183288913920000000,12588928073073017487360000000,
  13638005412495768944640000000,13733376079716019077120000000,
  16646516460261841305600000000,23970983702777051480064000000,
  35069156774989120143360000000,35956475554165577220096000000,
  37454662035589142937600000000,46818327544486428672000000000,
  48066816279006066769920000000,48940758393169813438464000000,
  58262807610916444569600000000,62424436725981904896000000000,
  67418391664060457287680000000,70138313549978240286720000000,
  79348395127248110223360000000,80259990076262449152000000000,
  84272989580075571609600000000,112363986106767428812800000000,
  136380054124957689446400000000,160519980152524898304000000000,
  168545979160151143219200000000,262182634249124000563200000000,
  294286630279628980224000000000,409899235032340365312000000000,
  441429945419443470336000000000,453201410630628629544960000000,
  481559940457574694912000000000,491592439217107501056000000000,
  602197641590722265088000000000,629238322197897601351680000000,
  840818943656082800640000000000,873942114163746668544000000000,
  1041447686045131446681600000000,1123639861067674288128000000000,
  1310913171245620002816000000000,1412575825342219105075200000000,
  1573095805494744003379200000000,1648005129565922289254400000000,
  1785338890363082480025600000000,1887714966593692804055040000000,
  2060006411957402861568000000000,2400288952599255334256640000000,
  2499474446508315472035840000000,4017012503316935580057600000000,
  4443510127125894172508160000000,4500541786123603751731200000000,
  6696655764585335016652800000000,7855491117597562912112640000000,
  11783236676396344368168960000000,14099599441841779585843200000000,
  14832046166093300603289600000000,16833195251994777668812800000000,
  16919519330210135503011840000000,17674855014594516552253440000000,
  19262397618302987796480000000000,19776061554791067471052800000000,
  26587816090330212933304320000000,33709195832030228643840000000000,
  36256112850450290363596800000000,39655123430180005085184000000000,
  47192874164842320101376000000000,51387796312828304110387200000000,
  63825865330480198660915200000000,74160230830466503016448000000000,
  80718618590983948861440000000000,80902069996872548745216000000000,
  97082483996247058494259200000000,197760615547910674710528000000000,
  207648646325306208446054400000000,224952700185748392483225600000000,
  294580916909908609204224000000000,302034394654990848648806400000000,
  332237834120489933513687040000000,339788693986864704729907200000000,
  424910655906413630246092800000000,466150022362932304674816000000000,
  475704171581610586621870080000000,666526519068884125876224000000000,
  733179170975772538463846400000000,830594585301224833784217600000000,
  1211283770230952882601984000000000,2067958036707558436613652480000000,
  2553034613219207946436608000000000,2610440125232420906178969600000000,
  2855168886972960366133248000000000,3028209425577382206504960000000000,
  3640593149859264693534720000000000,3779205363120572993718190080000000,
  5663144899781078412165120000000000,6398654583061287608411750400000000,
  8198077813559456628001996800000000,11420675547891841464532992000000000,
  16967860814010735890163302400000000,17321357914302626221208371200000000,
  17798455399311960723947520000000000,59084515334501669617532928000000000,
  63955783068194312201384755200000000,65776379280420229261295616000000000,
  68524053287351048787197952000000000,101978557417539271259062272000000000,
  126854445755096156432498688000000000,144976509434395607351427072000000000,
  152710747326096623011469721600000000,186053187107474362767664742400000000,
  203873216392118822837944320000000000,288453633838182510132776140800000000,
  315726620546017100454218956800000000,323931888267477685175844864000000000,
  356325077094225453693429350400000000,523274588739771645284057088000000000,
  747535126771102350405795840000000000,791833504653834341540954112000000000,
  900710611543736563502835302400000000,1269284133619504399056371712000000000,
  1511682145248229197487276032000000000,1903445924648640244088832000000000000
    ,2159545921783184567838965760000000000,
  2616372943698858226420285440000000000,2687434924885740795532935168000000000
    ,2848939427583201179879866368000000000,
  3266313206697066658856435712000000000,9977102158638312703416021811200000000
    ,11337616089361718981154570240000000000,
  14678871580232555244331991040000000000,
  15983660165142115710494834688000000000,
  17125350176937981118387322880000000000,
  19571828773643406992442654720000000000,
  23921124056675275212985466880000000000,
  25332485949667590848517242880000000000,
  39195758480364799906277228544000000000,
  40815417921702188332156452864000000000,
  40828915083713333235705446400000000000,
  54420557228936251109541937152000000000,
  64246491173049740893209231360000000000,
  74421274842989745107065896960000000000,
  79145281546890461349213634560000000000,
  82201680849302309368259149824000000000,
  88886910140595876812251830681600000000,
  89508496924795847978771074252800000000,
  112130269015665352560869376000000000000,
  137002801415503848947098583040000000000,
  181401857429787503698473123840000000000,
  182670401887338465262798110720000000000,
  195718287736434069924426547200000000000,
  274005602831007697894197166080000000000,
  383734698409165873208308531200000000000,
  384831083261763489988903698432000000000,
  720568489234989250802268241920000000000,
  922671927900332043929439436800000000000,
  1130273111677906753813563310080000000000,
  1473541241891196953119904759808000000000,
  1924155416308817449944518492160000000000,
  1932024261761314928713581723648000000000,
  1989280676553115886711871425740800000000,
  2232638245289692353211976908800000000000,
  2938710090362557559915264606208000000000,
  4219686283597518547570636357632000000000,
  6850140070775192447354929152000000000000,
  7263330371488691648086863878553600000000,
  13124640339637304211041314406400000000000,
  15286345807342271963448119132160000000000,
  33154677942551931445197857095680000000000,
  34524705956706969934668842926080000000000,
  41100840424651154684129574912000000000000,
  45713268072306450932015227207680000000000,
  88777815317246494117719881809920000000000,
  102140583349059726301221523292160000000000,
  122290766458738175707584953057280000000000,
  136187444465412968401628697722880000000000,
  162813150610746091918382333952000000000000,
  179121376936384460794835175997440000000000,
  212742516798041560106686249697280000000000,
  231788037349485280725761018717798400000000,
  291830238140170646574918637977600000000000,
  396889123870632079341889347649536000000000,
  538635239538714964821135543238656000000000,
  690494119134139398693376858521600000000000,
  755979283563108722555979709808640000000000,
  7062291762992129647113031039057920000000000,
  7957122706212463546847485702963200000000000,
  9051227078316677284539014987120640000000000,
  10894995557233037472130295817830400000000000,
  12930324397595253263627164267315200000000000,
  17639516616472536859639526562201600000000000,
  22858966345748751305340879686860800000000000,
  23274583915671455874528895681167360000000000,
  40227675903629676820173399942758400000000000,
  47411189457849261966632935646822400000000000,
  63920027028417475764758314903142400000000000,
  75331112138582716235872331083284480000000000,
  79544922524252858729477014101688320000000000,
  89862312463042279964326113130315776000000000,
  615483441325534055348653019124203520000000000,
  775819463855715195817629856038912000000000000,
  782284626054512822449443438172569600000000000,
  853401410241286715399392841642803200000000000,
  1432282087118243438432547426533376000000000000,
  2514486298032362643587496765554688000000000000,
  2982934594659482202355388028813312000000000000,
  3509091113439696424159741195006771200000000000,
  8506887547732597391902402896986112000000000000,
  21085738601646531402062089313996242944000000000,
  25612578127118123520196592568316723200000000000,
  29296679054691343058847560997273600000000000000,
  31090706407529834567326243937474052096000000000,
  41911491480738747578503515111791001600000000000,
  76837734381354370560589777704950169600000000000,
  135596001849448889224570195949912064000000000000,
  182472737898864214056306542140352102400000000000,
  263119171166665808661549165675597004800000000000,
  281622465379624616081799637742125056000000000000,
  282398284843480331277617267598163968000000000000,
  364979238311433260162801444098513305600000000000,
  612495903436747012216973008583000064000000000000,
  1354063570916174921767036642073247744000000000000,
  1564052039133121834768341789774446592000000000000,
  2800449658031179951836371237015126016000000000000,
  3625756999808600616962974149022580736000000000000,
  4640593962263108740521453661968138240000000000000,
  4823851471388873158795068037385945088000000000000,
  5120822231828443340500793119113373286400000000000,
  5415161929859590840072435345455688187904000000000,
  11648929249793563665201712288424263680000000000000,
  15770858061258978500580779713559003136000000000000,
  17972633699681498226311213244997435392000000000000,
  33921681975398857393067386183891455836160000000000,
  35335085391040476451111860608220266496000000000000,
  47249683979406198085309346376402862080000000000000,
  75796368050297442761850409812146257920000000000000,
  77961978566020226840760421521064722432000000000000,
  454200712164258580816726455750499290316800000000000,
  857865356079471907449174281122165555200000000000000,
  1142539659926419166257041303656584760524800000000000,
  1488365045351295239687244410856690155520000000000000,
  1563300091037384756963164702375516569600000000000000,
  3795432948296375652371176076519323729920000000000000,
  6444042696532913099686198061092268943605760000000000,
  7786297922815861385429596384294273548288000000000000,
  84458810113188579077490456647820115968000000000000000,
  97596847241906802489544527681925467340800000000000000,
  168564665816212783660934857498414465155072000000000000,
  216057091502526702866991900682491556528128000000000000,
  310694915725694024449155977945520109780992000000000000,
  447766202075264917588489497627367217037312000000000000,
  1047132734300268252072280599147516644032512000000000000,
  1367722217248081930088477010934503499313971200000000000,
  1786144300585946869061776787238838459170816000000000000,
  8930190272107771438123466465140140933120000000000000000,
  10314369764284476011032603767236862777753600000000000000,
  12274952446751773104038801177538084628070400000000000000,
  34067806143908267855683676462837449799761920000000000000,
  46175889589011694626312979442822244256972800000000000000,
  63110610881590066202654010647406375754058956800000000000,
  228935657287063559990194305830267662654400102400000000000,
  229957691471380808025864816124152786148392960000000000000,
  294747358512920638858548950825799007642583040000000000000,
  510806883564564526260662281806016061374464000000000000000,
  517404805810606818058195836279343768833884160000000000000,
  690156972798674992976391812342982003860176896000000000000,
  1363964738629268518192629546618749368821350400000000000000,
  1485760406521930473017791733137690947747840000000000000000,
  1605393062631488511104938599961764764319744000000000000000,
  11833239540298137412997626996388695453886054400000000000000,
  53045330216320162342453390802932437142732800000000000000000,
  72802597552691397798061590853972354887718010880000000000000,
  207937694447975036382417291947495153599512576000000000000000,
  241455576044949848427158056930360425455812608000000000000000,
  322314905573834980963554411520682561886801100800000000000000,
  1282954865963512744413393069594477862356988723200000000000000,
  1576204097856370538175757898144278132241203200000000000000000,
  1930850019874053909265303425226740711995473920000000000000000,
  3229934632697798967129513681765155778065627873280000000000000,
  7586095189194060692402347679557505730684439756800000000000000,
  14316895332547614542033842434459018168203476992000000000000000,
  95273270910708562124307559219579597649231216640000000000000000,
  150606301550176204922693667167685775535646965760000000000000000,
  203309856010601506784756974959897543417772847923200000000000000,
  367137339493195107604588408425255983952282255360000000000000000,
  4602656851594363528159556996515346983743519719424000000000000000,
  9019644059504997161481320733709181445968190504960000000000000000,
  40892835873780691346340679469040198201721271353344000000000000000,
  414069301492744803877398730844932063579189385625600000000000000000,
  680768353403510937841938168684593013872464593577574400000000000000,
  718903650853486927990900663458197713525244790256435200000000000000,
  5662748757492081648174479072163034297306543578635304960000000000000,
  15256745311454177935034740776449179401805827057647616000000000000000,
  28655640449971978674261065748448779945672889136578560000000000000000,
  40820331972159758582246891549129552601181753599590400000000000000000,
  729963707020463649959991442896016139887171633183457280000000000000000,
  2338296965629187691937956060053684758845187810098413568000000000000000,
  2473198713646256212626684294287551187240102491450245120000000000000000,
  29920085278378455955430818088688800077193521270672588800000000000000000,
  99726314446395706650897740034921186820221957304372690944000000000000000,
  123656465633097235526954253805291186522329149938914557952000000000000000,
  606181929786572414946366339504573856127549040953917440000000000000000000,
  746044378095708569778227724976284260145274325926234030080000000000000000,
  1025253284929720757234334580177384855801351578273919795200000000000000000,
  5999160088288458920039914416517824698032676417763803136000000000000000000,
  505656759763632489467760588203398691653063824995726131200000000000000000000,
  4622251909777153061915274021271721883894813455490140004679680000000000000000
    ,
  3194581314442596224725635569245344117534295586156452616601600000000000000000\
0,
  1131746512366968991284185526785755425980983079853291491819520000000000000000\
00,
  2643469901303426538526387888494414274941118536723442213847040000000000000000\
00,
  1096791358138976952791117509843842061499746685999560822829875200000000000000\
000,
  7156462691890231601481118341613398138747418481416086749184000000000000000000\
000,
  2093089677222789046440236424969549465808470468049707754397368320000000000000\
0000,
  6486757921410452314359132711818673300992511954894920379714974515200000000000\
000000,
  1855568540825824336669747112861188231689509220538599635681280000000000000000\
0000000,
  2797081193180344584086019518304159778919113988503519992813660930048000000000\
0000000,
  2431537015898160210772036616693301058805932882593780962596749312000000000000\
000000000,
  1300082850775315916477872492979237259483130817065255621366032787046400000000\
0000000000,
  3627342905136471469983198081211543622726109874916290402658990161920000000000\
0000000000,
  1174054220765499889262645006348562510234506061097595122625692390517637120000\
000000000000000,
  5531835195610958887067794419326748149270595184868435132170851295111938048000\
0000000000000000,
  5250696046263516092648170987414794689920778272106519961077281857522892800000\
00000000000000000,
  1274840690433857032419929995270250529386172567373975523344266007490750054400\
0000000000000000000,
  1266510491555829687838099172928283249894580032083586792234364527268226662400\
00000000000000000000,
  8957147548733264629940172723773870603298947723338970166011042417045250087321\
60000000000000000000,
  5184259261705608042824431380989661746964670604319535906397749155347672705531\
904000000000000000000000,
  7046403825746979717790151762110084990322572542137773789522100824437770158080\
000000000000000000000000,
  1245907893097207968456281395656302939049053196982972451492656167551971596907\
3152000000000000000000000,
  1806405821099057657625083163216568566706783862903218647666842205749424999628\
8000000000000000000000000,
  1959631333324497649212172216839366354533597776141513874700990752612321808560\
9472000000000000000000000,
  1533746864272984559336263031653558909370031214023943145154280516815744569180\
16000000000000000000000000,
  1158593030939257111194763446896099949300292515915388894364778767380931149824\
000000000000000000000000000,
  4145980273981389153508649245319313335495884320904923764117608120335024032361\
48019200000000000000000000000,
  6218970410972083730262973867978970003243826481357385646176412180502536048542\
22028800000000000000000000000,
  4859136937852978343853298304306116819318132901045030305518791898446326180099\
420009267200000000000000000000,
  3064650049008667159126739690516710727952538530174789215681494517974144286569\
9278028800000000000000000000000,
  1579760477436815558200288076989831408287186817989230475700860826308758898329\
2636535193600000000000000000000000,
  5932397347653089233075558846167571046393009978246902048226886947495715263034\
7923456000000000000000000000000000,
  3614660057009566039293823585327592555235030662561570980158052623494858627623\
2340225130496000000000000000000000000,
  6274176916566953685848959150207403326259616452643266277336548120253731177999\
0400131249902199480569036800000000000000000000000000000000,
  4574037988963764811994905915998328669681764183711672043658607811791123975868\
4309958174081947270879023862906880000000000000000000000000000,
  4794962122703154283780823138328748958614818005487963158725675558725364909900\
7732639759894263022235540258816000000000000000000000000000000,
  5775752629901609642380832200293019921982815338004535532650810056291088317331\
274781843001583802779807034898851700296617295872000000000000000000000000000000\
000,
  2768668094367193327342537963950285098848312852505394350744979631565176762621\
436932021290092862497226231606797746451741933568000000000000000000000000000000\
00000000,
  1651369432361955908463145725927301423425176806747045465707409579192756037770\
415756478460527001118221058623871480092777821767238028727627920616944245536408\
256258873753600000000000000000000000000000000000000000000,
  7065115985045946673536986139001118402705538174395734072025359108433701126993\
558961870245980549651087353428258861053147367587934265517055066894825367817997\
032901067145216000000000000000000000000000000000000000000000000000000,
  1422904478107155693691365040582043426558877618339840148771540855404829317884\
017469243797791886922736473977159826561752756549178369558679621864132661490617\
58424860652994560000000000000000000000000000000000000000000000000000000,
  7165033589793852697531456980706732548435609645091822296777976465116824959213\
549917461783791175492101413818415520493496100407385332345831553946154344805152\
60818409913846161473536000000000000000000000000000000000000000000000000 ],

sizespos :=
[ 14,14,13,37,138,1,13,138,1,37,1,1,95,95,141,95,141,111,
  95,141,111,259,6,29,29,10,29,29,1,1,6,10,29,1,29,1,1,1,
  1,1,10,10,1,6,1,29,1,29,1,1,22,22,1,1,1,1,1,1,1,1,1,
  1,1,1,1,1,1,14,13,37,138,1,13,138,1,37,1,1,95,95,141,
  95,141,111,95,141,111,259,10,6,10,10,10,6,1,1,29,29,29,
  22,29,1,1,1,1,1,6,29,1,29,1,29,22,29,1,1,1,1,1,1,1,1,
  1,1,1,1,1,1,1,1,1,1,1,34,87,79,1,34,79,1,87,1,1,243,
  243,197,243,197,425,243,197,425,386,29,29,11,10,11,29,1,1,
  29,11,10,1,11,1,1,1,1,1,29,11,1,29,1,11,1,10,1,1,1,1,
  1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,7,7,1,2,7,1,7,1,1,13,
  13,13,13,13,22,13,13,22,11,4,13,13,4,13,13,1,1,13,13,8,
  22,8,22,22,1,1,1,4,13,1,13,1,8,22,8,1,1,22,22,1,1,1,
  1,1,1,1,1,1,1,1,1,1,1,1,7,1,7,7,1,4,1,1,37,37,90,37,
  90,118,37,90,118,92,17,15,77,17,77,15,1,1,15,77,49,1,51,
  1,1,1,1,1,17,77,1,15,1,51,1,49,1,1,1,1,1,1,1,1,1,1,
  1,1,1,1,1,1,1,1,1,1,7,4,1,7,1,1,103,103,62,103,62,334,
  103,62,334,231,17,77,15,17,15,77,1,1,77,15,139,1,20,1,1,
  1,1,1,17,15,1,77,1,20,1,139,1,1,1,1,1,1,1,1,1,1,1,1,
  1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,
  1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,
  1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,7,1,7,1,1,13,
  13,13,13,13,22,13,13,22,11,4,13,13,4,13,13,1,1,13,13,8,
  22,8,22,22,1,1,1,4,13,1,13,1,8,22,8,1,1,22,22,1,1,1,
  1,1,1,1,1,1,1,1,1,1,1,1,1,7,1,1,103,103,62,103,62,334,
  103,62,334,231,17,77,15,17,15,77,1,1,77,15,139,1,20,1,1,
  1,1,1,17,15,1,77,1,20,1,139,1,1,1,1,1,1,1,1,1,1,1,1,
  1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,
  1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,
  1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,37,37,90,37,90,118,
  37,90,118,92,17,15,77,17,77,15,1,1,15,77,49,1,51,1,1,1,
  1,1,17,77,1,15,1,51,1,49,1,1,1,1,1,1,1,1,1,1,1,1,1,
  1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,
  1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,
  1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,
  1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,
  1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,17,17,16,31,31,16,
  15,42,42,17,38,94,17,94,38,1,1,38,94,77,1,207,250,456,1,
  1,1,17,94,1,38,1,207,1,77,1,1,1,1,1,1,1,1,1,1,1,1,1,
  1,1,1,1,1,1,17,17,17,8,16,31,31,34,17,38,94,17,94,38,1,
  1,38,94,77,456,207,1,1,1,1,1,17,94,1,38,1,207,250,77,1,
  1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,31,11,31,15,67,
  18,67,17,66,124,17,124,66,1,1,66,124,96,1,370,1,1,1,1,1,
  17,124,1,66,1,370,1,96,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,
  1,1,1,1,17,31,17,17,8,34,17,38,94,17,94,38,1,1,38,94,
  77,1,207,1,1,1,1,1,17,94,1,38,1,207,1,77,1,1,456,250,1,
  1,1,1,1,1,1,1,1,1,1,1,1,1,1,31,31,11,31,19,17,66,124,
  17,124,66,1,1,66,124,96,1,370,1,1,1,1,1,17,124,1,66,1,
  370,1,96,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,42,
  18,18,42,17,49,183,17,183,49,1,1,49,183,25,710,178,1,1,1,
  1,1,17,183,1,49,1,178,718,25,1,1,1,1,1,1,1,1,1,1,1,1,
  1,1,1,1,1,1,1,17,31,42,17,38,94,17,94,38,1,1,38,94,77,
  1,207,456,250,1,1,1,17,94,1,38,1,207,1,77,1,1,1,1,1,1,
  1,1,1,1,1,1,1,1,1,1,1,1,1,31,67,17,66,124,17,124,66,1,
  1,66,124,96,1,370,1,1,1,1,1,17,124,1,66,1,370,1,96,1,1,
  1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,42,17,49,183,17,
  183,49,1,1,49,183,25,1,178,1,1,1,1,1,17,183,1,49,1,178,
  1,25,1,1,710,718,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,81,
  180,417,81,417,180,1,1,180,417,122,1,167,1,1,1,1,1,81,
  417,1,180,1,167,1,122,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,
  1,1,1,1,10,10,4,10,10,1,1,6,6,10,17,10,17,17,1,1,1,4,
  6,1,6,1,10,17,10,1,1,7,7,1,1,1,1,1,1,1,1,1,1,1,1,1,
  1,1,10,10,10,4,1,1,11,11,11,15,11,1,1,1,1,1,6,11,1,11,
  1,11,15,11,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,10,
  4,10,1,1,11,11,11,34,11,1,1,1,1,1,6,11,1,11,1,11,34,
  11,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,10,10,1,1,
  10,10,6,17,6,7,7,1,1,1,4,10,1,10,1,6,17,6,1,1,17,17,
  1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,10,1,1,11,11,11,34,11,
  1,1,1,1,1,6,11,1,11,1,11,34,11,1,1,1,1,1,1,1,1,1,1,
  1,1,1,1,1,1,1,1,1,1,1,11,11,11,15,11,1,1,1,1,1,6,11,
  1,11,1,11,15,11,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,
  1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,
  1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,
  1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,
  1,10,11,1,11,1,1,1,1,1,10,10,1,4,1,11,1,11,1,1,15,15,
  1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,11,1,11,1,1,1,1,1,10,
  4,1,10,1,11,1,11,1,1,34,34,1,1,1,1,1,1,1,1,1,1,1,1,
  1,1,1,1,10,15,15,1,1,1,10,11,1,11,1,10,1,4,1,1,1,1,1,
  1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,7,1,1,1,1,
  1,7,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,34,34,
  1,1,1,10,11,1,11,1,4,1,10,1,1,1,1,1,1,1,1,1,1,1,1,1,
  1,1,1,1,1,1,7,1,1,1,17,1,1,1,1,34,1,15,1,1,1,1,1,1,
  1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,17,1,1,1,1,34,1,15,
  1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,
  1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,
  1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,
  1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,
  1,1,1,10,1,10,1,10,7,10,1,1,17,17,1,1,1,1,1,1,1,1,1,
  1,1,1,1,1,1,1,10,1,11,1,11,1,1,34,34,1,1,1,1,1,1,1,
  1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,
  1,1,1,1,1,1,1,1,11,1,11,1,1,15,15,1,1,1,1,1,1,1,1,1,
  1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,
  1,1,1,1,10,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,
  1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,
  1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,
  1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,7,1,
  1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,
  1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,
  1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,
  1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,
  1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,
  1,1,1,1,1,1,1,1,1,3,65,65,152,1,65,152,1,54,1,1,83,83,
  147,83,147,35,83,147,35,277,9,9,9,29,29,10,1,1,9,29,10,
  1,29,1,1,1,1,1,9,9,1,10,1,29,1,29,1,1,1,1,1,1,1,1,1,
  1,1,1,1,1,1,1,1,1,1,65,65,152,1,65,152,1,54,1,1,83,83,
  147,83,147,35,83,147,35,277,9,9,9,29,29,10,1,1,9,29,10,
  1,29,1,1,1,1,1,9,9,1,10,1,29,1,29,1,1,1,1,1,1,1,1,1,
  1,1,1,1,1,1,1,1,1,1,39,39,1,13,104,398,104,408,1,89,89,
  210,89,210,311,89,210,311,169,7,48,48,13,67,67,1,1,7,13,
  67,1,67,1,1,1,1,1,13,13,172,13,1,48,172,48,1,1,116,116,
  1,298,1,1,1,1,1,1,1,1,213,1,1,213,1,39,1,39,39,408,30,
  1,1,89,89,210,89,210,311,89,210,311,169,7,13,13,48,67,67,
  1,1,7,48,67,1,67,1,1,1,1,1,48,48,1,13,298,28,1,28,116,
  116,1,1,1,1,1,1,1,1,1,1,1,1,1,213,1,213,1,1,104,138,1,
  104,1,1,248,248,349,248,349,362,248,349,362,427,31,80,80,
  80,46,160,1,1,31,80,160,1,46,1,1,1,1,1,70,70,113,17,52,
  70,113,70,230,230,1,1,1,1,1,1,1,1,1,1,1,1,81,81,1,101,
  1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,
  1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,
  1,1,1,1,1,1,1,1,1,1,1,1,39,1,104,1,1,89,89,210,89,210,
  311,89,210,311,169,7,48,48,13,67,67,172,1,7,13,67,1,67,1,
  172,1,1,1,13,13,1,13,1,48,1,48,1,1,116,116,1,298,1,1,1,
  213,1,1,1,1,1,213,1,1,1,1,104,1,1,248,248,349,248,349,
  362,248,349,362,427,31,70,70,70,17,70,113,1,31,70,70,1,17,
  1,113,1,1,1,80,80,1,17,1,160,1,160,1,1,1,1,1,52,1,1,1,
  81,1,1,1,1,1,101,1,81,1,1,1,1,1,1049,1,1,1,1,1,1,1,1,
  76,1,1,1,1,1,1,1,76,1,1,1,1,1,1,1,1,1,1,1,149,1,1,1,
  149,1,1,1,1,1,1,1,1,1,1,516,1,1,1,1,1,1,1,1,1,1,1,
  364,364,452,364,452,145,364,452,145,441,13,134,134,104,42,
  17,205,1,13,104,17,1,42,1,205,1,1,1,104,104,205,7,309,
  134,205,134,327,327,265,265,1,1,1,1,1,323,1,1,1,1,323,1,
  1,1,1,1,1,1,1,1036,1,1,1,1,1,1,76,1,1,1,1,1,149,1,76,
  1,1,1,1,1,149,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,
  1,1,1,1,1,1,1,1,516,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,
  1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,
  1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,39,39,31,81,81,
  31,20,57,57,11,108,108,108,95,210,302,1,11,108,210,1,95,1,
  302,1,1,1,108,108,233,13,691,190,233,190,1,1,1,1,1,318,1,
  1,1,453,1,1,1,1,924,924,1,924,1,39,39,39,24,31,81,81,43,
  11,108,108,108,95,210,1,1,11,108,210,1,95,1,1,1,1,1,108,
  108,233,13,1,190,233,190,300,300,1,1,1,1,1,1,1,1,1,1,1,
  1,924,453,1,453,1,81,110,81,20,168,144,168,31,210,210,210,
  137,355,1,1,31,210,355,1,137,1,1,1,1,1,210,210,1,40,656,
  267,1,267,463,463,1,1,1,1,1,1,1,1,1,1,1,1,574,868,1,
  868,1,39,81,39,39,24,43,11,108,108,108,95,210,302,1,11,
  108,210,1,95,1,302,1,1,1,108,108,302,13,318,190,302,190,1,
  1,300,300,1,691,1,1,1,453,1,1,1,1,453,1,1,1,1,81,81,
  110,81,13,31,210,210,210,137,355,1,1,31,210,355,1,137,1,1,
  1,1,1,210,210,320,40,1,267,320,267,593,593,593,593,1,656,
  1,1,1,1,1,1,1,1,868,1,1,1,1,57,144,16,57,13,221,221,
  221,90,62,203,1,13,221,62,1,90,1,203,1,1,1,221,221,203,8,
  555,128,203,128,343,343,238,238,1,555,1,1,1,1,1,1,1,1,
  613,889,1,889,1,39,81,57,11,108,108,108,95,210,233,1,11,
  108,210,1,95,1,233,1,1,1,108,108,1,13,1,190,1,190,1,1,1,
  1,1,691,1,1,1,924,1,1,1,1,1,453,1,453,1,81,168,31,210,
  210,210,137,355,320,1,31,210,355,1,137,1,320,1,1,1,210,
  210,1,40,1,267,1,267,1,1,463,463,1,282,1,1,1,868,1,1,1,
  1,1,1,1,1,1,57,13,221,221,221,90,62,146,1,13,221,62,1,
  90,1,146,1,1,1,221,221,146,8,293,128,146,128,238,238,343,
  343,1,1,1,1,1,889,1,1,1,1,889,1,1,1,1,22,335,335,335,
  87,182,1,1,22,335,182,1,87,1,1,1,1,1,335,335,280,17,1,
  219,280,219,1,1,1,1,1,1,1,1,1,1,1,1,1,1,694,1,1,1,1,
  5,5,5,7,7,9,1,2,5,7,1,7,1,9,1,1,1,5,5,9,4,11,5,9,5,
  5,5,5,5,1,11,1,1,1,11,1,1,1,1,11,11,1,11,1,5,14,14,
  14,1,1,5,14,14,1,14,1,1,1,1,1,9,9,16,10,29,10,16,10,
  29,29,1,1,1,1,1,1,1,1,1,1,1,1,45,45,1,1,1,14,14,14,1,
  1,5,14,14,1,14,1,1,1,1,1,9,9,16,10,29,10,16,10,29,29,
  1,1,1,1,1,1,1,1,1,1,1,1,45,45,1,1,1,14,14,1,1,5,5,
  14,1,14,1,1,1,1,1,10,10,16,10,29,9,16,9,1,1,29,29,1,1,
  1,1,1,1,1,1,1,1,45,45,1,1,1,7,29,1,7,14,7,1,4,1,29,
  1,1,1,14,14,29,6,13,14,29,14,14,14,14,14,1,13,1,1,1,19,
  1,1,1,1,17,17,1,19,1,10,1,7,14,4,1,7,1,10,1,1,1,14,
  14,10,6,36,14,10,14,14,14,14,14,1,36,1,1,1,67,1,1,1,1,
  52,52,1,67,1,1,9,1,10,1,29,1,3,1,1,1,16,16,7,7,55,16,
  7,16,1,1,1,1,1,1,1,1,1,1,1,1,1,1,102,102,1,1,1,1,1,
  1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,
  1,1,1,1,1,1,1,1,1,1,5,7,1,7,1,9,1,1,1,5,5,9,4,11,5,
  9,5,5,5,5,5,1,11,1,1,1,11,1,1,1,1,11,11,1,11,1,14,1,
  14,1,1,1,1,1,10,10,16,10,29,9,16,9,1,1,29,29,1,1,1,1,
  1,1,1,1,1,1,45,45,1,1,1,1,7,1,10,1,1,1,14,14,10,6,36,
  14,10,14,14,14,14,14,1,36,1,1,1,67,1,1,1,1,52,52,1,67,
  1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,
  1,1,1,1,1,1,1,1,1,1,29,1,1,1,14,14,29,6,13,14,29,14,
  14,14,14,14,1,13,1,1,1,19,1,1,1,1,17,17,1,19,1,1,1,1,
  1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,
  1,1,1,1,1,1,16,16,7,7,55,16,7,16,1,1,1,1,1,1,1,1,1,
  1,1,1,1,1,102,102,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,
  1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,
  1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,
  1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,5,1,10,1,
  14,1,14,1,1,29,29,1,29,1,1,1,45,1,1,1,1,1,1,1,45,1,1,
  10,1,14,1,14,1,1,29,29,1,29,1,1,1,45,1,1,1,1,1,1,1,
  45,1,7,1,1,3,1,1,1,1,1,1,55,1,1,1,102,1,1,1,1,1,1,1,
  102,1,1,10,7,10,13,13,13,13,1,1,1,1,1,1,1,1,1,1,1,1,
  1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,
  1,5,29,29,1,1,1,29,1,1,1,48,1,1,1,1,1,1,1,48,1,1,1,
  1,1,1,1,55,1,1,1,102,1,1,1,1,1,1,1,102,1,29,29,1,1,1,
  29,1,1,1,48,1,1,1,1,1,1,1,48,1,5,14,14,1,1,1,1,1,318,
  1,1,1,1,1,318,1,1,1,14,14,1,1,1,1,1,318,1,1,1,1,1,
  318,1,1,1,5,1,1,1,1,1,1,1,1,1,1,1,318,1,318,1,1,1,1,
  1,1,1,1,1,1,1,1,318,1,318,1,1,1,1,1,1,1,1,1,1,1,1,1,
  1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,
  1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,
  1,1,1,1,1,81,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,
  1,1,1,1,1,1,1,1,81,1,1,1,1,1,1,1,1,1,65,65,152,1,65,
  152,1,54,1,1,83,83,147,83,147,35,83,147,35,277,9,9,9,29,
  29,10,1,1,9,29,10,1,29,1,1,1,1,1,9,9,1,10,1,29,1,29,
  1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,39,39,1,13,
  104,1,104,1,1,89,89,210,89,210,311,89,210,311,169,13,7,13,
  13,13,13,1,1,48,67,67,116,48,172,1,1,298,1,7,48,1,67,1,
  67,116,48,1,172,1,1,1,1,1,1,1,1,1,1,213,1,1,1,1,1,213,
  39,1,39,39,1,30,1,408,89,89,210,89,210,311,89,210,311,169,
  48,7,48,48,48,13,116,116,13,67,67,1,28,1,1,1,1,1,7,13,
  1,67,1,67,1,28,1,1,1,1,1,1,298,1,213,1,1,1,1,1,1,1,1,
  1,213,1,104,138,1,104,1,1,248,248,349,248,349,362,248,349,
  362,427,80,31,80,70,70,17,1,1,70,17,70,1,160,113,1,1,52,
  1,31,70,1,70,1,17,1,160,1,113,1,1,1,1,1,1,81,1,1,1,81,
  1,1,1,1,1,101,408,1,1,1,1,1,1,1036,1,1,1,1,1,1,1,1,1,
  76,1,1,1,1,1,1,1,1,1,1,1,149,1,1,1,1,76,1,1,1,1,1,1,
  1,1,149,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,516,39,1,104,
  1,398,89,89,210,89,210,311,89,210,311,169,13,7,13,13,13,
  13,1,1,48,67,67,116,48,1,1,172,298,1,7,48,1,67,1,67,116,
  48,1,1,1,172,1,1,1,1,213,1,213,1,1,1,1,1,1,1,1,1,104,
  1,1,248,248,349,248,349,362,248,349,362,427,70,31,70,80,80,
  17,230,230,80,46,160,1,70,1,1,113,1,1,31,80,1,160,1,46,
  1,70,1,1,1,113,1,1,52,1,101,1,81,1,1,1,1,1,1,1,81,1,
  1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,
  1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,
  1,1,1,1,1,1,1,1,1,364,364,452,364,452,145,364,452,145,
  441,104,13,104,104,104,7,327,327,134,42,17,265,134,205,1,
  205,1,1,13,134,1,17,1,42,265,134,1,205,1,205,1,1,309,1,
  1,1,323,1,323,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,
  1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,
  1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1049,1,1,
  1,1,1,1,1,76,1,1,1,1,1,1,1,1,1,1,1,1,1,149,1,1,76,1,
  1,1,1,1,1,1,1,1,1,149,1,1,1,1,1,1,1,1,516,1,1,1,1,1,
  1,39,39,31,81,81,31,20,57,57,108,11,108,108,108,13,1,1,
  108,95,210,1,190,233,1,1,691,1,11,108,1,210,1,95,1,190,1,
  233,1,1,1,1,1,1,453,1,1,1,924,1,1,1,1,1,453,39,39,39,
  24,31,81,81,43,108,11,108,108,108,13,1,1,108,95,210,300,
  190,302,1,302,691,1,11,108,1,210,1,95,300,190,1,302,1,302,
  1,1,318,1,1,1,453,1,453,1,1,1,1,1,1,81,110,81,20,168,
  144,168,210,31,210,210,210,40,1,1,210,137,355,463,267,320,
  1,1,282,1,31,210,1,355,1,137,463,267,1,320,1,1,1,1,1,1,
  1,1,1,1,868,1,1,1,1,1,1,39,81,39,39,24,43,108,11,108,
  108,108,13,300,300,108,95,210,1,190,1,1,233,1,1,11,108,1,
  210,1,95,1,190,1,1,1,233,1,1,1,1,453,1,924,1,1,1,1,1,
  1,1,453,81,81,110,81,13,210,31,210,210,210,40,593,593,210,
  137,355,593,267,1,1,320,656,1,31,210,1,355,1,137,593,267,
  1,1,1,320,1,1,1,1,1,1,868,1,1,1,1,1,1,1,1,57,144,16,
  57,221,13,221,221,221,8,238,238,221,90,62,343,128,146,1,
  146,1,1,13,221,1,62,1,90,343,128,1,146,1,146,1,1,293,1,
  1,1,889,1,889,1,1,1,1,1,1,39,81,57,108,11,108,108,108,
  13,1,1,108,95,210,1,190,302,1,233,318,1,11,108,1,210,1,
  95,1,190,1,302,1,233,1,1,691,1,924,1,924,1,453,1,1,1,1,
  1,924,81,168,210,31,210,210,210,40,463,463,210,137,355,1,
  267,1,1,1,1,1,31,210,1,355,1,137,1,267,1,1,1,1,1,1,656,
  1,868,1,574,1,1,1,1,1,1,1,868,57,221,13,221,221,221,8,
  343,343,221,90,62,238,128,203,1,203,555,1,13,221,1,62,1,
  90,238,128,1,203,1,203,1,1,555,1,889,1,613,1,1,1,1,1,1,
  1,889,335,22,335,335,335,17,1,1,335,87,182,1,219,1,1,280,
  1,1,22,335,1,182,1,87,1,219,1,1,1,280,1,1,1,1,1,1,694,
  1,1,1,1,1,1,1,1,5,5,10,10,10,1,1,9,14,14,29,14,16,1,
  1,29,1,5,9,1,14,1,14,29,14,1,16,1,1,1,1,1,1,45,1,1,1,
  45,1,1,1,1,1,1,5,5,5,4,5,5,5,7,7,5,5,9,1,9,11,1,2,
  5,1,7,1,7,5,5,1,9,1,9,1,1,11,1,11,1,11,1,11,1,1,1,1,
  1,11,10,10,10,1,1,9,14,14,29,14,16,1,1,29,1,5,9,1,14,
  1,14,29,14,1,16,1,1,1,1,1,1,45,1,1,1,45,1,1,1,1,1,1,
  5,10,1,1,14,14,14,29,9,1,1,16,1,1,5,14,1,14,1,14,29,9,
  1,1,1,16,1,1,29,1,1,1,45,1,1,1,1,1,1,1,45,10,1,1,14,
  14,14,29,9,1,1,16,1,1,5,14,1,14,1,14,29,9,1,1,1,16,1,
  1,29,1,1,1,45,1,1,1,1,1,1,1,45,13,13,10,6,6,13,10,7,
  1,7,1,1,4,10,1,6,1,6,13,10,1,7,1,7,1,1,1,1,1,1,1,1,
  1,1,1,1,1,1,1,5,29,14,14,14,29,1,1,1,1,1,5,29,1,14,1,
  14,14,29,1,1,1,1,1,1,1,1,1,1,1,1,318,1,1,1,1,1,318,
  29,14,14,14,29,1,1,1,1,1,5,29,1,14,1,14,14,29,1,1,1,1,
  1,1,1,1,1,1,1,1,318,1,1,1,1,1,318,14,14,1,10,1,1,16,
  1,1,5,5,1,14,1,14,1,10,1,1,1,16,1,1,29,1,1,1,45,1,1,
  1,1,1,1,1,45,7,14,14,29,1,29,13,1,7,14,1,7,1,4,14,14,
  1,29,1,29,1,1,13,1,19,1,17,1,19,1,1,1,1,1,17,14,14,10,
  1,10,36,1,7,14,1,4,1,7,14,14,1,10,1,10,1,1,36,1,67,1,
  52,1,67,1,1,1,1,1,52,1,1,1,1,1,1,5,1,1,14,1,14,5,1,
  1,1,1,1,1,1,1,1,318,1,1,1,1,1,1,1,1,1,318,16,1,1,29,
  1,5,10,1,14,1,14,1,5,1,16,1,1,1,1,1,1,48,1,1,1,48,1,
  1,1,1,1,1,1,7,1,1,9,1,1,10,1,29,1,16,1,3,1,7,1,1,55,
  1,1,1,102,1,1,1,1,1,1,1,102,1,1,1,1,1,1,1,1,1,1,1,1,
  1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,55,1,9,16,1,10,
  1,29,1,1,1,7,1,3,1,1,1,1,102,1,1,1,102,1,1,1,1,1,1,
  1,11,1,1,36,1,13,1,29,1,1,1,55,1,1,1,1,1,1,1,1,1,1,
  1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,
  1,1,1,1,1,1,1,5,1,7,1,7,5,5,1,9,1,9,1,1,11,1,11,1,
  11,1,11,1,1,1,1,1,11,1,14,1,14,1,10,1,1,1,16,1,1,29,
  1,1,1,45,1,1,1,1,1,1,1,45,1,1,1,1,1,1,1,1,1,1,1,1,
  1,1,1,1,1,1,1,1,1,1,1,1,1,7,14,14,1,10,1,10,1,1,36,
  1,67,1,52,1,67,1,1,1,1,1,52,1,1,1,1,1,1,1,1,1,1,1,1,
  1,1,1,1,1,1,1,1,1,1,14,14,1,29,1,29,1,1,13,1,19,1,17,
  1,19,1,1,1,1,1,17,1,1,1,1,1,1,1,1,1,318,1,1,1,1,1,1,
  1,1,1,318,1,16,1,1,1,1,1,1,48,1,1,1,48,1,1,1,1,1,1,
  1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,7,1,1,55,1,1,
  1,102,1,1,1,1,1,1,1,102,1,1,1,1,1,1,1,1,1,1,1,1,1,1,
  1,1,1,1,1,1,102,1,1,1,102,1,1,1,1,1,1,1,1,1,1,1,1,1,
  1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,
  1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,81,1,1,1,
  1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,81,1,1,1,1,1,
  1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,115,
  115,1,34,47,1,47,1,1,269,269,318,269,318,404,269,318,404,
  544,48,48,31,17,99,70,1,1,48,99,17,209,99,411,1,1,1,1,
  48,31,411,70,1,99,1,17,209,1,1,1,1,1,1,1,1,1,1,1,1,1,
  1,1,1,1,1,115,1,115,115,1,46,1,1,269,269,318,269,318,404,
  269,318,404,544,48,48,31,46,104,160,1,1,48,104,17,1,104,1,
  1,1,1,1,48,31,1,160,1,104,1,46,1,1,1,1,1,1,1,1,1,1,1,
  1,1,1,1,1,1,1,1,272,47,8,272,47,39,39,299,299,95,299,95,
  336,299,95,336,153,36,36,6,70,10,70,1,39,30,19,13,1,19,
  40,1,1,30,15,30,6,40,70,30,10,1,70,1,1,39,1,44,198,198,
  48,138,172,172,150,138,48,138,138,150,172,172,1,39,39,1,1,
  1,1,1,560,1,566,1,1,501,1,681,1,1,29,1,43,1,1,36,1,84,
  127,1,84,1,1,1,1,1,1,29,1,1,1,43,1,1,1,1,36,1,1,1,1,
  1,1,1,1,1,1,1,1,1,1,1,1,115,1,47,1,1,269,269,318,269,
  318,404,269,318,404,544,48,48,31,17,99,70,1,209,48,99,17,
  1,99,1,411,411,1,1,48,31,1,70,1,99,1,17,1,1,209,1,1,1,
  1,1,1,1,1,1,1,1,1,1,1,1,1,39,47,272,272,299,299,95,299,
  95,336,299,95,336,153,30,30,6,70,19,70,1,1,36,10,13,39,
  10,1,40,40,198,44,36,6,1,70,198,19,1,70,39,1,1,1,15,30,
  30,150,172,138,138,48,172,150,172,172,48,138,138,1,1,1,1,
  1053,579,1,604,1,1,501,1,683,1,1,29,1,123,1,1,1,1,84,
  127,1,84,1,1,1,1,1,1,29,1,1,1,123,1,1,1,1,1,1,1,1,1,
  1,1,1,1,1,1,1,1,1,1,1,1,1,1,405,405,311,405,311,82,405,
  311,82,442,99,99,10,10,99,6,1,173,99,99,8,173,99,315,315,
  315,1,109,99,10,315,6,1,99,1,10,173,1,173,1,109,1,1,1,1,
  1,1,116,1,116,1,1,1,1,1,39,1,1,501,1,566,1,1,560,1,681,
  1,1,29,1,84,1,1,1,1,43,127,36,43,1,1,1,1,1,1,29,1,1,
  1,84,1,1,36,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,
  1,501,1053,604,1,1,579,1,683,1,1,29,1,84,1,1,1,1,123,
  127,1,123,1,1,1,1,1,1,29,1,1,1,84,1,1,1,1,1,1,1,1,1,
  1,1,1,1,1,1,1,1,1,1,1,1,86,86,146,104,104,146,17,165,
  165,154,154,31,137,298,355,1,1,154,298,17,359,298,512,1,1,
  1,1,154,31,512,355,1,298,1,137,359,1,1,1,1,1,1,1,1064,1,
  1,1,1,1,1,1065,1,1065,1064,86,86,86,37,146,104,104,123,
  154,154,31,137,298,355,1,1,154,298,17,396,298,490,1,1,1,1,
  154,31,490,355,1,298,1,137,396,1,1,1,1,1,1,1,1,1,1,1,
  1064,1,1065,1,1,1,1,104,31,104,17,39,80,39,95,95,11,92,
  114,231,1,1,95,114,34,302,114,300,1,1,680,550,95,11,300,
  231,680,114,1,92,302,1,1,1,550,636,636,318,551,496,496,509,
  580,318,508,556,258,556,551,86,104,86,86,37,123,154,154,31,
  137,298,355,1,396,154,298,17,1,298,1,490,490,1,1,154,31,1,
  355,1,298,1,137,1,1,396,1,1,1,1,1,1,1064,1065,1,1,1,1,
  1,1,1,1,104,104,31,104,20,95,95,11,92,114,231,1,233,95,
  114,34,233,114,1,1,1,668,261,95,11,1,231,668,114,1,92,233,
  1,233,1,261,668,668,509,496,551,556,318,551,318,556,496,509,
  496,496,165,80,79,165,285,285,34,72,339,64,1,235,285,339,
  10,235,339,372,368,368,1,1,285,34,372,64,1,339,1,72,235,1,
  235,1,1,1,1,974,1,1,1,974,1,1,1,1,619,1,1,86,104,165,
  154,154,31,137,298,355,1,359,154,298,17,1,298,1,512,512,1,
  1,154,31,1,355,1,298,1,137,1,1,359,1,1,1,1,1,1065,1,1,
  1,1,1,1,1064,1,1064,1065,104,39,95,95,11,92,114,231,1,302,
  95,114,34,1,114,1,300,300,636,550,95,11,1,231,636,114,1,
  92,1,1,302,1,550,680,680,258,556,580,508,318,496,509,496,
  551,318,551,556,165,285,285,34,72,339,64,1,235,285,339,10,
  235,339,368,372,372,1,1,285,34,368,64,1,339,1,72,235,1,
  235,1,1,1,1,619,1,1,1,1,1,974,1,1,974,1,1,301,301,20,
  179,191,418,1,1,301,191,17,1,191,471,471,471,800,240,301,
  20,471,418,800,191,1,179,1,1,1,1,240,800,800,431,650,829,
  829,605,829,605,829,650,431,650,650,5,5,14,14,14,1,1,9,10,
  10,16,10,29,1,1,80,1,9,5,29,14,80,14,1,14,16,1,1,1,17,
  1,1,1,86,1,1,31,86,1,86,86,31,1,1,5,14,14,14,1,1,9,10,
  10,16,10,29,1,1,80,1,9,5,29,14,80,14,1,14,16,1,1,1,17,
  1,1,1,86,1,1,31,86,1,86,86,31,1,1,7,5,7,1,9,5,5,4,9,
  5,5,5,5,10,6,5,2,5,7,10,5,1,7,9,1,9,1,6,10,10,7,16,
  16,16,7,16,7,16,16,7,16,16,14,7,1,29,14,14,6,29,14,14,
  14,14,1,36,14,7,14,7,1,14,1,4,29,1,29,1,36,1,1,67,1,1,
  1,67,1,67,1,1,67,1,1,14,1,1,10,9,10,16,9,1,29,29,22,
  22,10,5,1,14,22,5,1,14,16,1,1,1,16,35,35,43,43,130,130,
  52,43,43,43,43,52,130,130,1,10,14,14,6,10,14,14,14,14,1,
  36,14,7,14,4,1,14,1,7,10,1,10,1,36,1,1,67,1,1,1,67,1,
  67,1,1,67,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,
  1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,16,16,7,7,
  16,1,1,1,179,1,16,9,1,10,179,1,1,29,7,1,3,1,31,1,1,1,
  113,1,1,34,113,1,113,113,73,331,331,14,10,1,14,1,29,29,1,
  17,5,5,1,14,1,10,1,14,1,1,16,1,1,80,80,31,1,86,86,1,1,
  31,1,1,1,86,86,10,1,5,29,1,1,35,16,14,5,29,14,35,9,1,
  14,1,1,16,1,22,22,22,52,130,43,43,43,130,52,130,130,43,
  43,43,7,10,13,13,13,20,1,10,4,13,6,20,10,1,6,7,1,7,1,
  1,20,20,1,17,17,17,1,17,1,17,17,1,17,17,1,1,1,1,1,31,
  1,9,1,10,1,16,1,29,3,1,7,1,1,179,179,73,331,113,113,1,
  1,34,1,331,1,113,113,29,1,1,35,16,14,5,29,14,35,9,1,14,
  1,1,16,1,22,22,22,52,130,43,43,43,130,52,130,130,43,43,
  43,14,14,1,60,1,5,5,14,1,1,1,14,1,1,1,1,1,159,159,52,
  1,201,201,1,1,52,1,1,1,201,201,5,159,1,29,5,14,14,159,
  29,1,14,1,1,1,1,60,1,1,1,201,1,1,52,201,1,201,201,52,1,
  1,159,1,29,5,14,14,159,29,1,14,1,1,1,1,60,1,1,1,201,1,
  1,52,201,1,201,201,52,1,1,22,1,10,1,1,10,22,1,1,1,1,
  179,1,1,1,1,134,1,1,1,1,1,134,1,1,1,1,1,17,6,60,36,22,
  22,1,36,31,1,1,1,1,1,1,1,81,1,1,1,81,1,81,81,1,1,1,5,
  1,14,1,10,1,14,1,1,16,1,1,80,80,31,1,86,86,1,1,31,1,1,
  1,86,86,5,7,10,5,1,7,9,1,9,1,6,10,10,7,16,16,16,7,16,
  7,16,16,7,16,16,14,1,1,1,14,1,1,1,1,1,159,159,52,1,201,
  201,1,1,52,1,1,1,201,201,1,14,1,7,10,1,10,1,36,1,1,67,
  1,1,1,67,1,67,1,1,67,1,1,22,1,1,1,1,179,1,1,1,1,134,
  1,1,1,1,1,134,1,1,1,1,1,1,14,16,1,1,1,16,35,35,43,43,
  130,130,52,43,43,43,43,52,130,130,1,1,1,1,1,1,1,1,1,1,
  1,1,1,1,1,1,1,1,1,1,29,1,29,1,36,1,1,67,1,1,1,67,1,
  67,1,1,67,1,1,1,7,1,1,179,179,73,331,113,113,1,1,34,1,
  331,1,113,113,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,31,
  1,1,1,113,1,1,34,113,1,113,113,73,331,331,1,1,1,1,1,1,
  1,1,1,1,1,1,1,1,1,22,22,1,1,81,81,1,1,1,1,1,1,81,81,
  10,1,1,1,1,134,1,1,1,1,134,1,1,1,1,1,1,134,1,1,1,1,
  134,1,1,43,1,1,1,36,48,36,43,1,1,1,1,1,1,138,36,109,16,
  1,1,1,16,43,1,1,1,1,36,138,109,43,1,1,1,1,36,109,138,1,
  1,1,1,48,36,36,43,16,109,1,1,1,43,36,1,1,1,138,1,1,1,
  1,1,1,43,43,16,3,1,7,7,14,7,1,1,38,38,70,38,70,218,38,
  70,218,54,16,16,48,16,16,70,1,1,16,48,70,1,27,1,1,1,1,
  1,16,16,176,70,1,10,1,33,1,176,1,1,1,1,1,1,1,1,1,1,1,
  1,1,1,38,38,38,1,7,7,14,7,1,1,38,38,70,38,70,218,38,70,
  218,54,16,16,48,16,16,70,1,1,16,48,70,1,27,1,1,1,1,1,
  16,16,176,70,1,10,1,33,1,176,1,1,1,1,1,1,1,1,1,1,1,1,
  1,1,38,38,38,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,
  1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,
  1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,7,1,7,1,1,13,13,
  13,13,13,22,13,13,22,11,4,13,13,4,13,13,1,1,13,13,8,22,
  8,22,22,1,1,1,4,13,1,13,1,8,22,8,1,1,22,22,1,1,1,1,1,
  1,1,1,1,1,1,1,1,1,1,1,3,14,1,113,113,75,113,75,135,113,
  75,135,35,16,70,16,16,48,16,1,1,70,16,33,134,10,1,134,1,
  1,1,16,48,1,16,1,27,1,70,189,1,1,134,1,1,1,1,1,1,1,1,
  38,38,38,1,1,1,1,1,1,1,1,403,1,419,1,1,1,546,1,329,59,
  1,36,59,1,1,1,1,184,36,122,314,17,273,1,1,1,1,1,379,184,
  198,1,168,184,138,1,476,474,198,1,1,1,1,1,1,1,1,1,1,1,
  1,1,1,1,14,1,113,113,75,113,75,135,113,75,135,35,16,70,
  16,16,48,16,1,1,70,16,33,134,10,1,134,1,1,1,16,48,1,16,
  1,27,1,70,189,1,1,134,1,1,1,1,1,1,1,1,38,38,38,1,1,1,
  1,1,403,1,1,298,546,531,1,435,355,372,59,1,1,59,36,1,1,
  1,198,379,138,1,168,227,235,1,1,1,1,36,1,184,1,17,342,
  122,184,325,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,
  1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,
  1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,
  1,1,26,26,31,81,81,31,20,123,123,20,62,156,20,156,117,1,
  1,62,156,151,678,369,675,679,1,838,1,20,156,1,117,1,369,1,
  176,841,841,644,1,1,739,1,1,1,1,1,1,640,407,662,1,640,
  662,397,26,26,26,17,31,81,81,93,20,62,156,20,156,117,1,1,
  62,156,151,679,369,644,1,1,1,1,20,156,841,117,1,369,675,
  176,1,1,1,1,1,1,1,1,1,1,1,1,407,1,640,1,662,397,1,81,
  31,81,20,138,45,138,20,113,233,20,233,161,1,1,113,233,153,
  766,330,828,1,1,1,1,20,233,1,161,1,330,1,111,1,949,1,1,
  1,1,1,1,1,1,1,1,792,1,323,1,323,556,649,26,81,26,26,17,
  93,20,62,156,20,156,117,1,1,62,156,151,1,369,1,1,1,1,1,
  20,156,1,117,1,369,644,176,806,1,638,592,1,1,1,1,1,1,1,
  1,1,397,407,1,397,1,407,81,81,31,81,40,20,113,233,20,233,
  161,1,1,113,233,153,1,330,1,1,1,1,1,20,233,949,161,1,330,
  828,111,994,943,863,898,1,1,1,1,1,1,1,1,1,649,792,1,556,
  649,1,123,45,146,123,54,236,444,54,444,148,1,1,236,444,205,
  930,462,1,1,1,1,1,54,444,988,148,1,462,887,222,934,1,879,
  853,1,985,1,1,1,1,1,1,724,1,557,1,580,745,1,26,81,123,
  20,62,156,20,156,117,1,1,62,156,151,1,369,638,592,1,739,1,
  20,156,1,117,1,369,1,176,1,806,1,678,1,838,1,1,1,1,1,1,
  397,662,1,1,1,407,640,81,138,20,113,233,20,233,161,1,1,
  113,233,153,1,330,863,898,1,1,1,20,233,943,161,1,330,1,
  111,1,1,842,919,1,1,1,1,1,1,1,1,649,556,1,1,649,1,792,
  123,54,236,444,54,444,148,1,1,236,444,205,1,462,879,853,1,
  985,1,54,444,1,148,1,462,1,222,983,1,894,905,1,1,1,1,1,
  1,1,1,1,745,724,1,745,1,724,113,286,554,113,554,228,1,1,
  286,554,139,1,188,1,1,1,767,1,113,554,1,228,1,188,1,133,
  1,1,1,1011,1,767,1,1,1,1,1,1,521,510,416,1,486,510,521,
  10,10,4,10,10,1,1,7,7,10,45,10,23,23,1,59,1,4,7,79,7,
  1,10,45,10,12,12,26,26,1,59,1,1,1,1,1,1,31,1,31,1,31,
  31,1,10,10,10,10,1,1,31,43,43,70,36,151,1,1,198,1,7,31,
  128,43,1,43,70,36,128,151,1,1,1,1,1,1,1,1,1,1,126,1,89,
  1,89,1,126,10,10,10,1,1,31,43,36,99,43,52,1,1,1,1,7,31,
  70,43,1,36,99,43,128,151,1,1,1,1,1,1,1,1,1,1,1,1,70,1,
  11,26,26,10,10,1,1,10,10,6,23,6,26,26,1,1,1,4,10,12,10,
  1,6,23,6,79,1,45,45,1,1,1,1,1,1,1,1,1,31,31,1,31,1,
  31,10,1,1,43,31,43,89,36,1,151,1,1,1,7,43,128,31,1,43,
  89,36,70,226,226,45,1,1,1,1,1,1,1,1,26,26,11,1,70,1,1,
  1,1,43,31,36,116,43,1,52,1,1,1,7,43,70,31,1,36,116,43,
  70,226,45,226,1,198,1,1,1,1,1,1,1,126,89,1,89,126,1,1,
  1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,
  1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,
  1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,10,
  43,1,43,1,1,1,1,1,10,10,151,10,1,36,151,36,1,1,70,70,1,
  198,1,1,1,1,1,1,1,89,126,1,1,126,89,43,1,43,1,1,1,1,1,
  10,10,151,10,1,36,52,36,1,181,99,99,1,1,1,1,1,1,1,1,1,
  70,1,1,26,26,11,1,10,119,119,1,122,1,10,36,1,36,1,10,1,
  6,204,84,204,115,1,122,1,1,1,1,1,1,38,1,128,1,1,38,128,
  1,1,1,1,1,1,26,151,139,52,1,1,26,1,209,422,195,1,1,196,
  1,1,1,1,1,1,1,287,1,1,1,102,242,104,104,1,37,1,10,36,1,
  36,1,6,1,10,204,84,115,204,1,17,1,1,1,1,1,1,43,144,144,
  1,29,10,29,26,1,1,1,45,226,226,45,1,100,308,93,1,172,308,
  195,1,620,1,1,1,1,1,1,1,186,205,1,414,449,1,1,1,1,45,
  45,226,226,1,100,195,93,1,139,1,1,1,1,1,1,1,1,1,1,1,1,
  205,1,242,1,102,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,
  1,1,1,1,1,1,1,1,1,1,1,1,59,1,222,198,1,17,440,122,1,1,
  361,608,1,1,1,1,1,1,1,1,1,149,149,1,1,1,1,1,1,1,1,1,
  1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,10,1,
  10,1,10,26,10,1,79,23,23,1,59,1,1,1,1,1,1,31,31,1,1,1,
  31,31,226,10,1,43,1,43,181,1,89,89,1,1,1,1,1,1,1,1,26,
  11,26,1,1,1,70,226,1,1,172,1,1,503,1,422,1,514,1,1,1,1,
  1,1,59,295,1,1,1,278,1,1,43,1,43,1,1,116,116,1,1,1,1,
  1,1,1,1,126,89,1,1,126,1,89,1,1,1,1,1,1,1,1,1,1,1,1,
  1,1,1,1,1,1,1,1,1,1,1,10,194,130,1,194,1,37,1,1,1,1,
  1,1,10,29,29,1,144,43,144,1,215,1,308,1,1,1,1,1,1,1,1,
  1,186,205,1,1,449,1,345,194,130,194,1,1,122,1,1,1,1,1,1,
  38,128,1,1,128,38,1,1,204,204,1,1,1,1,1,1,1,1,527,323,
  1,1,1,295,59,226,226,1,1,1,1,1,1,1,1,1,59,295,1,527,1,
  278,26,1,1,1,1,1,1,1,1,205,1,186,1,219,414,449,1,1,1,1,
  1,1,1,1,205,1,1,1,102,242,1,1,1,1,1,1,1,1,1,1,1,1,1,
  1,1,1,1,1,1,1,1,1,1,1,1,149,1,149,1,1,1,1,1,1,1,1,1,
  1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,
  1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,26,
  26,1,1,1,1,29,1,1,1,1,1,1,1,1,1,1,1,26,29,26,1,7,7,
  14,7,1,1,38,38,70,38,70,218,38,70,218,54,16,16,48,16,16,
  70,1,1,16,48,70,1,27,1,1,1,1,1,16,16,176,70,1,10,1,33,
  1,176,1,1,1,1,1,1,1,1,1,1,1,1,1,1,38,38,38,1,1,1,1,
  1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,
  1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,
  1,1,1,1,1,1,1,3,1,7,1,14,38,38,70,38,70,218,38,70,218,
  54,16,16,16,16,48,70,176,1,16,16,70,1,10,1,1,176,1,1,16,
  48,1,70,1,27,1,33,1,1,1,1,1,1,1,38,38,1,1,1,1,1,1,38,
  1,1,1,1,7,1,14,38,38,70,38,70,218,38,70,218,54,16,16,16,
  16,48,70,176,1,16,16,70,1,10,1,1,176,1,1,16,48,1,70,1,
  27,1,33,1,1,1,1,1,1,1,38,38,1,1,1,1,1,1,38,1,1,1,1,
  1,1,1,419,546,403,1,1,1,1,1,329,198,1,36,184,379,1,314,
  184,59,36,198,1,17,1,1,198,1,1,1,1,235,122,1,168,1,249,
  1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,17,17,34,
  17,34,17,17,34,17,10,13,6,13,13,13,6,95,95,6,13,13,1,8,
  1,1,95,1,1,13,13,95,6,1,8,1,13,95,95,1,1,1,1,1,1,1,1,
  1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,
  1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,
  1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,403,1,419,1,1,1,546,
  1,329,1,59,1,184,36,122,1,1,1,379,198,1,168,1,1,235,1,1,
  198,36,198,1,1,17,1,249,184,314,1,1,1,1,1,1,1,1,1,1,1,
  1,1,1,1,1,1,26,26,31,81,81,31,20,123,123,62,20,156,62,
  156,151,1,1,20,156,117,1,369,1,1,1,1,1,62,156,592,151,
  838,369,1,166,1,678,1,1,1,1,1,1,397,1,1,1,1,1,1,662,
  662,640,407,26,26,26,17,31,81,81,93,62,20,156,62,156,151,
  1,644,20,156,117,1,369,1,1,679,1,1,62,156,678,151,1,369,
  1,166,675,679,1,1,838,1,739,407,1,1,1,1,1,1,1,397,640,
  407,1,81,31,81,20,138,45,138,113,20,233,113,233,153,1,1,
  20,233,161,1,330,1,1,1,1,1,113,233,919,153,873,330,1,57,
  1,766,1,1,873,1,1,1,649,1,1,1,1,1,1,556,323,792,1,26,
  81,26,26,17,93,62,20,156,62,156,151,679,675,20,156,117,1,
  369,1,1,678,1,838,62,156,679,151,739,369,1,166,644,1,1,1,
  1,1,1,640,407,1,1,1,1,1,1,1,407,1,397,81,81,31,81,40,
  113,20,233,113,233,153,1,828,20,233,161,1,330,1,1,766,1,1,
  113,233,766,153,884,330,1,57,828,1,1,1,1,1,884,792,1,1,1,
  1,1,1,1,649,792,1,649,123,45,146,123,236,54,444,236,444,
  205,1,1,54,444,148,1,462,1,1,930,1,1,236,444,875,205,1,
  462,1,251,887,930,1,1,1,1,907,724,1,1,1,1,1,1,1,745,557,
  724,1,26,81,123,62,20,156,62,156,151,678,1,20,156,117,1,
  369,1,1,592,1,1,62,156,1,151,1,369,1,166,1,1,1,1,1,1,
  838,662,640,1,1,1,1,1,1,407,1,397,662,81,138,113,20,233,
  113,233,153,766,1,20,233,161,1,330,1,1,919,1,873,113,233,
  1,153,1,330,1,57,1,1,1,1,1,1,873,323,792,1,1,1,1,1,1,
  1,1,649,556,123,236,54,444,236,444,205,930,887,54,444,148,
  1,462,1,1,875,1,1,236,444,930,205,907,462,1,251,1,1,1,1,
  1,1,1,557,724,1,1,1,1,1,1,1,724,1,745,286,113,554,286,
  554,139,1,1,113,554,228,1,188,1,1,1038,1,771,286,554,1038,
  139,767,188,1,275,1,1,1,1,771,1,767,416,521,1,1,1,1,1,1,
  510,416,521,510,10,10,31,43,43,1,1,7,31,43,1,43,1,1,1,1,
  1,31,31,1,13,198,21,1,21,70,70,1,1,127,1,1,1,89,1,1,1,
  1,1,1,126,89,126,1,10,10,10,6,26,26,4,10,10,1,6,1,1,23,
  1,1,7,7,45,10,59,10,1,6,23,23,1,1,59,1,59,1,31,1,1,1,
  1,1,1,31,31,1,31,43,31,43,1,1,7,43,31,1,43,1,1,1,1,1,
  31,31,45,21,1,13,1,21,89,89,1,1,17,1,1,1,70,1,1,1,1,1,
  1,1,11,26,26,10,43,1,151,10,10,10,1,36,1,1,70,1,198,31,
  43,70,43,127,36,1,10,151,1,1,1,198,1,127,126,1,1,1,1,1,
  1,1,89,126,1,89,43,151,1,10,10,10,1,36,1,1,89,1,1,31,
  43,99,36,17,43,1,10,52,1,1,1,1,1,84,26,26,1,1,1,1,1,1,
  11,1,1,70,119,119,10,36,36,1,10,1,1,1,1,122,13,21,115,
  16,257,16,1,21,1,1,1,1,257,1,1,38,1,1,1,1,1,1,1,128,
  128,38,1,26,23,1,52,1,104,1,1,1,1,1,70,89,1,1,608,1,1,
  115,278,1,1,1,196,1,1,1,1,1,1,1,1,1,1,287,102,1,242,23,
  52,1,1,104,1,1,1,1,1,70,89,1,1,361,1,1,115,150,278,1,1,
  1,1,1,1,186,1,1,1,1,1,1,205,219,495,345,10,10,1,10,1,1,
  45,1,59,10,10,23,6,59,6,1,6,26,26,1,1,1,1,59,31,1,1,1,
  1,1,1,1,31,1,31,31,10,1,43,1,1,99,1,1,43,31,89,43,84,
  36,1,10,1,151,1,1,1,1,17,1,1,1,1,1,1,1,1,70,26,26,11,
  1,43,1,1,116,1,1,43,31,116,36,127,43,1,10,1,52,1,1,1,1,
  127,1,126,1,1,1,1,1,1,89,1,126,89,1,1,1,1,1,1,1,1,1,
  1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,
  1,1,1,37,21,13,194,16,162,16,1,21,1,1,1,1,179,1,179,43,
  144,1,1,1,1,1,1,144,29,10,29,1,1,1,1,1,1,1,1,1,1,1,1,
  1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,
  1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,
  1,1,45,78,115,1,194,1,78,1,1,1,1,1,1,1,1,287,1,1,1,1,
  1,1,1,242,102,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,
  1,1,1,1,1,1,1,1,1,1,127,17,1,257,179,179,1,127,1,196,1,
  1,1,1,1,1,149,1,1,1,1,1,1,149,1,1,1,10,1,43,1,43,1,
  21,1,1,1,1,1,1,198,89,126,1,1,1,1,1,1,1,1,89,126,1,43,
  1,43,1,21,1,1,1,1,1,1,1,11,26,1,1,1,1,1,1,26,1,70,1,
  1,1,1,1,78,1,1,1,1,1,1,1,242,102,1,1,1,1,1,1,1,1,287,
  1,1,10,1,21,119,119,1,1,122,1,257,128,38,1,1,1,1,1,1,1,
  38,1,128,179,1,1,1,1,1,1,1,1,1,149,149,1,1,1,1,1,1,1,
  1,1,1,1,21,104,104,1,1,37,1,162,29,10,1,1,1,1,1,1,29,
  43,144,144,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,
  115,115,1,1,127,1,1,116,116,1,1,1,1,1,1,1,116,116,1,26,
  1,1,1,1,361,219,495,1,1,1,1,1,1,345,1,186,205,1,1,1,1,
  608,102,1,1,1,1,1,1,1,242,1,1,287,1,1,1,1,1,1,1,1,1,
  1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,179,
  1,1,1,1,1,1,1,1,1,1,149,149,1,1,1,1,1,1,1,1,1,1,1,1,
  1,1,1,1,1,1,1,1,1,1,149,149,1,26,1,1,1,1,1,1,29,1,1,
  1,1,1,1,1,1,1,26,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,
  1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,
  1,1,1,26,29,26,14,10,10,14,1,1,1,1,337,1,337,290,658,
  274,1,64,1,1,15,1,52,1,1,110,1,15,1,1,20,110,1,1,1,1,
  232,52,1,1,1,15,215,1,1,367,205,1,1,1,1,1,1,1,1,1,1,1,
  1,1,32,32,32,7,1,3,1,1,113,113,75,113,75,135,113,75,135,
  35,16,70,48,16,16,16,1,189,70,48,33,1,27,134,1,1,1,1,16,
  16,1,16,1,10,134,70,1,1,134,1,1,1,1,1,1,38,38,38,1,1,
  1,1,1,1,1,10,7,10,10,13,13,8,13,8,150,13,8,150,8,13,13,
  6,13,6,13,1,52,13,6,13,1,7,1,1,1,13,13,13,6,1,13,13,7,
  1,13,52,1,1,1,13,13,13,10,10,10,10,10,10,10,10,10,10,10,
  10,1,1,1,1,1,274,1,68,1,1,274,1,58,232,1,15,1,52,1,1,
  1,1,15,1,1,20,293,1,1,1,1,1,52,110,1,1,15,1,1,1,215,1,
  1,1,1,1,1,1,32,32,32,1,1,1,1,1,1,1,1,1,113,113,75,113,
  75,135,113,75,135,35,16,70,48,16,16,16,1,189,70,48,33,1,
  27,134,1,1,1,1,16,16,1,16,1,10,134,70,1,1,134,1,1,1,1,
  1,1,38,38,38,1,1,1,1,1,1,1,3,1,1,65,426,174,1,1,116,1,
  35,1,1,48,116,10,1,1,1,1,48,116,1,27,1,1,1,1,1,1,10,1,
  1,1,10,235,116,86,235,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,
  1,1,1,1,65,426,174,1,1,116,1,35,1,1,48,116,10,1,1,1,1,
  48,116,1,27,1,1,1,1,1,1,10,1,1,1,10,235,116,86,235,1,1,
  1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,43,43,31,57,57,31,13,
  127,127,131,131,172,131,20,333,1,1,131,172,333,1,388,601,1,
  1,825,1,131,20,686,333,825,34,627,455,1,625,1,1,1,1,1,1,
  614,1,1,1,614,1,1,1,975,975,1,43,43,43,26,31,57,57,52,
  131,131,172,131,20,333,1,1,131,172,333,1,388,1,1,1,1,1,
  131,20,625,333,1,34,601,455,1,704,1,1,1,1,1,1,1,1,1,1,
  1,1,614,614,975,1,1,57,16,57,13,17,43,17,205,205,95,205,
  10,330,1,1,205,95,330,1,243,409,1,1,68,337,205,10,340,330,
  337,13,340,107,804,371,1,1,312,410,410,70,75,205,205,330,
  75,70,57,57,107,330,205,43,57,43,43,26,52,131,131,172,131,
  20,333,1,601,131,172,333,1,388,1,1,1,1,825,131,20,704,333,
  764,34,1,455,957,1,688,1,1,1,1,614,1,1,1,1,1,614,1,1,1,
  1,1,57,57,16,57,15,205,205,95,205,10,330,1,409,205,95,330,
  1,243,1,1,1,337,68,205,10,371,330,274,13,409,107,740,343,
  343,1,95,312,410,75,70,205,330,205,70,75,75,75,330,205,205,
  127,43,45,127,383,383,186,383,179,326,1,640,383,186,326,1,
  388,640,1,1,1,1,383,179,689,326,1,113,665,432,1,1,1,1,1,
  977,952,1,1,1,1,1,1,1,647,647,664,1,1,43,57,127,131,131,
  172,131,20,333,1,627,131,172,333,1,388,688,1,1,764,1,131,
  20,1,333,1,34,1,455,1,528,627,1,1,1,1,1,614,1,1,1,614,
  1,1,1,1,1,975,57,17,205,205,95,205,10,330,1,340,205,95,
  330,1,243,343,1,1,274,337,205,10,343,330,274,13,1,107,1,1,
  371,1,312,95,312,57,75,330,107,205,75,57,70,70,205,205,330,
  127,383,383,186,383,179,326,1,665,383,186,326,1,388,1,1,1,
  1,1,383,179,1,326,1,113,640,432,1,730,774,1,1,1,1,647,1,
  1,731,1,1,647,1,1,1,1,1,480,480,141,480,13,358,1,529,480,
  141,358,1,197,495,1,1,428,190,480,13,461,358,436,10,529,
  215,1,1,1,1,287,540,575,146,214,448,536,448,214,146,146,
  146,536,448,448,29,29,31,10,10,1,1,31,13,84,1,84,70,1,1,
  32,1,31,7,70,31,32,10,13,10,81,31,1,1,227,1,1,1,38,1,1,
  172,38,1,38,38,1,172,1,29,10,10,31,1,1,31,29,29,1,22,70,
  1,1,32,1,31,7,70,10,32,10,31,31,99,13,1,1,1,1,1,1,38,
  1,1,172,38,1,38,38,1,1,172,29,6,29,1,36,13,14,13,1,14,
  13,1,1,15,15,13,6,13,29,15,6,36,29,1,36,36,1,13,29,29,
  48,48,29,29,13,48,48,48,48,13,29,29,10,29,1,70,29,29,31,
  1,22,1,1,1,1,32,31,10,31,10,32,7,70,31,99,1,13,1,227,1,
  227,38,1,1,172,1,1,38,38,38,172,1,1,10,1,1,10,6,10,1,7,
  1,1,1,10,10,7,5,11,7,6,5,11,7,12,16,16,1,13,13,13,7,7,
  20,20,20,7,7,5,5,20,20,20,1,19,84,13,31,1,84,1,1,1,1,
  32,31,10,31,31,32,7,19,10,81,1,31,1,1,227,1,38,1,172,1,
  1,1,38,38,38,172,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,
  1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,70,
  13,1,1,98,1,1,1,115,1,31,11,89,31,32,16,16,13,81,89,39,
  1,1,1,1,1,110,1,1,394,110,1,38,38,246,246,87,29,10,1,84,
  1,1,1,1,32,10,10,13,31,32,7,70,31,119,119,31,1,1,227,1,
  38,1,1,172,1,1,38,38,38,1,172,1,29,1,14,36,1,1,15,15,
  29,6,36,29,15,6,13,13,1,36,36,1,29,29,13,48,48,29,13,29,
  48,48,48,48,29,29,13,1,84,19,1,1,32,1,10,7,19,31,32,10,
  31,31,119,31,119,1,1,1,227,1,38,172,1,1,38,1,38,38,1,1,
  172,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,
  1,1,1,1,1,1,1,1,1,1,98,1,1,20,20,84,7,98,22,20,7,98,
  84,1,30,30,1,84,22,84,27,27,22,84,84,27,27,27,27,84,22,
  84,1,1,1,115,13,16,16,31,32,11,89,31,130,113,89,1,1,389,
  389,110,1,1,394,1,1,110,38,38,87,232,232,1,1,1,1,1,1,1,
  1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,
  1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,
  1,1,10,32,10,32,32,10,6,115,32,195,1,32,1,1,1,1,14,1,1,
  1,1,1,14,14,14,1,1,1,32,6,115,32,10,10,32,32,195,32,1,
  1,1,1,1,1,14,1,1,1,14,1,14,14,1,1,1,10,1,29,1,10,1,
  84,1,70,70,1,1,1,227,38,38,172,1,1,38,38,1,1,1,1,172,
  16,10,10,5,1,10,52,1,11,1,13,13,13,5,7,20,20,20,7,5,7,
  7,20,20,20,1,1,1,113,1,1,89,1,1,1,1,1,110,38,232,232,
  87,38,110,1,1,1,1,394,1,10,1,29,1,19,19,1,1,1,1,38,38,
  1,1,172,38,38,1,1,1,1,172,10,1,1,1,115,115,1,1,1,1,14,
  14,1,1,1,14,14,1,1,1,1,1,16,10,52,11,1,1,13,13,13,7,5,
  13,13,13,5,7,7,7,13,13,13,1,1,1,89,1,1,1,1,38,110,394,
  1,1,110,38,1,1,232,232,87,1,70,70,1,227,1,1,38,38,1,1,
  172,38,38,1,1,1,172,1,119,119,1,1,1,1,149,149,1,1,1,149,
  149,1,1,1,1,1,16,1,1,1,389,38,1,87,232,246,1,38,110,110,
  1,1,1,1,389,1,1,1,38,1,1,1,38,1,110,110,246,87,232,1,1,
  1,1,1,1,1,1,1,1,1,1,1,1,1,29,84,1,1,99,1,1,1,1,1,1,
  1,1,99,29,1,1,1,1,99,1,1,1,1,1,1,99,1,1,1,1,99,1,1,
  1,1,1,99,1,7,1,1,1,7,3,7,7,1,1,1,1,1,1,3,7,7,7,1,1,
  1,29,29,1,1,1,1,31,10,31,84,1,1,1,1,31,31,10,1,1,1,1,
  10,31,31,7,7,7,1,1,1,7,7,1,1,1,3,1,1,1,1,1,1,29,84,
  29,1,1,14,1,1,1,298,435,1,546,355,403,1,531,372,1,198,36,
  59,379,184,325,184,1,36,138,1,17,235,227,1,1,1,59,1,1,1,
  1,168,1,122,1,1,1,342,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,
  3,1,1,1,1,426,116,1,174,1,1,65,1,35,1,1,10,116,48,1,
  235,86,1,10,116,1,10,1,1,1,1,1,1,48,1,1,1,27,1,116,1,
  1,1,235,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,426,
  116,1,174,1,1,65,1,35,1,1,10,116,48,1,235,86,1,10,116,1,
  10,1,1,1,1,1,1,48,1,1,1,27,1,116,1,1,1,235,1,1,1,1,1,
  1,1,1,1,1,1,1,1,1,1,1,1,1,366,573,1,1,298,506,1,403,
  368,1,122,36,184,1,59,293,1,1,36,249,1,17,1,256,1,1,1,
  198,379,1,1,1,168,466,198,338,378,378,338,1,1,1,1,1,1,1,
  1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,
  1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,
  1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,
  1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,
  1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,
  1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,
  1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,
  1,1,1,368,368,1,1,676,1,676,429,777,1,600,1,1,777,1,1,
  600,1,1,1,1,712,801,882,1,1,1,1,1,958,848,1,1,1,1,1,1,
  1,1,1,1,1,1,1,1,775,522,1,1,139,1,244,104,199,104,1,1,
  402,758,415,788,1,1,1,402,1,1,499,585,1,1,1,1,808,415,1,
  1,1,505,1,1,1,967,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,
  1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,
  1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,
  1,1,1,86,1,88,1,758,402,1,415,1,1,585,1,402,788,1,499,1,
  1,1,1,1,768,415,859,1,1,505,702,1,947,840,816,1008,1,1,1,
  1,1,1,1,1,1,1,1,1,1,533,1,246,185,1,246,1,791,633,791,
  1,904,1,1,1,633,904,1,1,1,1,1,1,1,750,737,992,1,1,706,
  1009,719,978,933,951,917,1,1,1,1,1,1,1,1,1,1,1,1,522,
  778,1,1,1,246,1,877,1,807,832,1022,1055,760,1,945,892,1,
  857,753,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,
  1,1,1,1,1,1,1,787,861,368,104,1,822,365,763,363,1,912,
  590,1,365,1,1,489,589,1,1,1,1,1,363,847,1,1,519,1,1,1,
  1,1,1034,1,1,1,1,1,1,1,1,1,1,1,1,1,643,632,1,1,1002,
  803,909,1,854,1,811,1,856,871,1,1,811,1023,1,1,1,903,984,
  1,1,1,844,1,1025,1026,1019,1,1,1,1,1,1,1,1,1,1,1,1,1,
  1,1,1,835,1,1,167,1,554,867,1,486,1,167,1,1,268,502,1,1,
  1,1,1,554,1,1,1,188,596,1,1,1,1,1,1,1,1,1,1,1,1,1,1,
  1,1,1,454,454,454,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,
  1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,
  1,1,1,55,55,76,81,81,1,48,48,1,23,52,1,1,1,1,76,70,31,
  1,1,1,1,1,190,194,1,1,1,1,1,1,1,1,1,1,1,1,1,1,43,1,
  144,48,29,48,1,20,1,11,23,1,11,24,1,1,1,1,23,29,1,1,1,
  6,20,1,54,80,24,1,1,1,1,1,1,1,1,1,1,1,1,1,10,19,19,
  55,48,1,52,1,1,76,1,23,81,81,1,1,1,1,1,52,1,1,59,160,
  76,1,1,176,190,1,1,1,1,1,1,1,1,1,1,1,1,144,1,43,1,130,
  1,1,29,1,1,13,1,130,1,1,1,70,43,99,1,1,43,70,70,40,138,
  138,40,1,1,1,1,1,1,1,1,1,1,1,1,1,29,1,121,43,1,23,1,
  1,1,1,74,1,1,1,76,55,149,1,1,59,160,55,36,194,176,194,1,
  1,1,1,1,1,1,1,1,1,1,1,144,43,1,81,1,1,74,1,54,157,1,
  1,1,1,1,1,40,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,
  1,1,155,1,155,1,24,1,1,24,1,157,1,1,1,52,26,126,1,1,26,
  26,31,30,104,36,20,1,1,1,1,1,1,1,1,1,1,1,1,1,52,52,1,
  1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,
  1,1,1,1,1,1,1,1,1,1,48,1,11,20,1,1,1,1,1,13,160,1,1,
  16,61,23,1,1,61,54,1,1,1,1,1,1,1,1,1,1,1,1,19,19,10,
  1,1,43,121,1,1,1,31,70,160,1,1,31,104,76,165,104,165,36,
  1,1,1,1,1,1,1,1,1,1,1,1,1,43,144,1,1,1,1,1,1,1,1,1,
  1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,24,
  54,1,1,1,48,29,80,1,1,16,61,48,19,130,61,19,1,1,1,1,1,
  1,1,1,1,1,1,1,19,10,19,81,1,1,1,31,43,43,1,1,29,76,52,
  20,149,80,30,1,1,1,1,1,1,1,1,1,1,1,1,52,52,1,1,1,1,1,
  95,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,155,
  1,155,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,
  1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,
  1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,
  1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,31,1,48,1,36,104,107,
  1,1,1,1,1,1,1,1,1,1,1,1,1,144,43,43,1,1,43,1,55,1,1,
  70,130,1,1,1,1,1,1,1,1,1,1,1,1,1,1,29,1,1,1,1,1,1,1,
  1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,130,1,1,1,1,1,1,1,
  1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,
  1,1,1,1,1,1,1,1,1,1,1,1,1,26,1,77,104,1,1,1,1,1,1,1,
  1,1,1,1,1,1,1,17,56,56,1,1,1,70,1,1,1,1,1,1,1,1,1,1,
  1,1,1,1,192,192,107,130,1,1,1,1,1,1,1,1,1,1,1,1,1,1,
  43,144,1,1,165,19,1,1,1,1,1,1,1,1,1,1,1,1,1,174,48,
  138,194,1,1,1,1,1,1,1,1,1,1,1,1,1,1,149,1,1,1,1,1,1,
  1,1,1,1,1,1,1,104,1,1,1,1,1,1,1,1,1,1,1,1,1,1,48,
  174,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,
  1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,
  1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,
  1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,
  1,1,1,1,1,19,19,19,1,7,1,14,38,38,70,38,70,218,38,70,
  218,54,16,16,16,16,48,70,176,1,16,16,70,1,10,1,1,176,1,
  1,16,48,1,70,1,27,1,33,1,1,1,1,1,1,1,38,38,1,1,1,1,1,
  1,38,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,
  1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,
  1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,113,113,75,113,75,135,
  113,75,135,35,16,70,48,16,16,16,1,189,70,48,33,1,27,134,
  1,1,1,1,16,16,1,16,1,10,134,70,1,1,134,1,1,1,1,1,1,38,
  38,38,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,
  1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,
  1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,419,546,403,1,1,1,
  1,1,329,1,184,379,59,36,198,476,1,1,1,122,474,168,1,273,
  184,1,1,59,36,1,1,1,17,198,138,1,1,314,184,1,1,1,1,1,1,
  1,1,1,1,1,1,1,1,1,26,26,31,81,81,31,20,123,123,20,62,
  156,20,156,117,806,1,62,156,151,1,369,592,638,1,838,1,20,
  156,1,117,1,369,678,176,1,1,1,1,1,739,1,1,407,397,1,662,
  1,1,1,640,1,1,1,26,26,26,17,31,81,81,93,20,62,156,20,
  156,117,1,806,62,156,151,638,369,1,1,1,1,1,20,156,1,117,
  1,369,592,176,1,1,1,644,1,1,1,397,1,1,407,397,1,1,1,407,
  1,1,1,81,31,81,20,138,45,138,20,113,233,20,233,161,1,1,
  113,233,153,842,330,898,863,943,1,1,20,233,1,161,1,330,919,
  111,1,1,1,1,1,1,1,649,1,649,1,556,1,1,1,792,1,1,1,26,
  81,26,26,17,93,20,62,156,20,156,117,1,1,62,156,151,1,369,
  1,644,841,1,1,20,156,1,117,1,369,1,176,1,1,679,675,1,1,
  1,662,397,407,640,1,1,1,1,1,1,1,1,81,81,31,81,40,20,113,
  233,20,233,161,943,994,113,233,153,863,330,1,1,949,1,1,20,
  233,1,161,1,330,898,111,1,1,1,828,1,1,1,556,649,1,792,
  649,1,1,1,1,1,1,1,123,45,146,123,54,236,444,54,444,148,1,
  983,236,444,205,894,462,853,879,1,1,1,54,444,1,148,1,462,
  905,222,1,1,1,1,1,985,1,745,1,1,724,745,1,1,1,724,1,1,
  1,26,81,123,20,62,156,20,156,117,841,841,62,156,151,644,
  369,679,675,1,739,1,20,156,1,117,1,369,1,176,1,1,678,1,1,
  838,1,640,662,640,662,407,1,1,1,397,1,1,1,81,138,20,113,
  233,20,233,161,949,1,113,233,153,1,330,1,828,1,1,1,20,233,
  1,161,1,330,1,111,1,1,766,1,1,1,1,323,556,792,323,1,1,1,
  1,649,1,1,1,123,54,236,444,54,444,148,1,934,236,444,205,
  879,462,1,1,988,985,1,54,444,1,148,1,462,853,222,1,1,930,
  887,1,1,1,580,745,724,557,1,1,1,1,1,1,1,1,113,286,554,
  113,554,228,1,1,286,554,139,1,188,1,1,1,767,1,113,554,1,
  228,1,188,1011,133,1,1,1,1,1,767,1,486,510,521,416,510,1,
  1,1,521,1,1,1,10,10,4,10,10,79,1,7,7,10,23,10,45,45,1,
  59,1,4,7,1,7,1,10,23,10,1,1,26,26,1,59,1,1,31,31,1,31,
  1,1,1,31,1,1,1,10,10,10,10,1,1,31,43,43,70,36,1,1,151,
  198,1,7,31,1,43,1,43,70,36,1,1,1,151,1,1,1,1,126,1,126,
  89,1,1,1,89,1,1,1,10,10,10,1,181,31,43,36,89,43,45,226,
  226,1,1,7,31,1,43,1,36,89,43,1,1,151,1,1,1,1,1,1,26,
  26,11,1,1,1,70,1,1,1,10,10,1,79,10,10,6,45,6,26,26,12,
  1,1,4,10,1,10,1,6,45,6,1,1,23,23,1,1,1,31,1,1,31,31,
  1,1,1,31,1,1,1,10,181,1,43,31,43,99,36,1,1,151,1,1,7,
  43,1,31,1,43,99,36,1,1,1,52,1,1,1,26,26,1,1,70,1,1,1,
  11,1,1,1,1,1,43,31,36,116,43,226,45,226,1,1,7,43,1,31,
  1,36,116,43,1,1,52,1,1,198,1,126,1,126,1,89,1,1,1,89,1,
  1,1,1,151,226,84,226,130,139,172,503,1,1,12,151,1,226,1,
  84,226,130,1,1,422,1,1,1,1,527,1,1,295,59,1,1,1,278,1,
  1,1,128,70,204,204,194,1,1,1,1,1,12,128,1,70,1,204,204,
  194,1,1,209,215,1,1,1,1,295,527,1,323,1,1,1,59,1,1,1,
  10,43,1,43,1,151,128,1,1,10,10,1,10,1,36,1,36,1,1,70,
  70,1,198,1,89,1,126,89,1,1,1,1,126,1,1,1,43,226,43,151,
  1,128,1,1,10,10,1,10,1,36,45,36,1,1,89,89,1,1,1,70,1,
  26,11,26,1,1,1,1,1,1,1,204,10,119,119,1,122,1,10,36,1,
  36,1,10,115,6,1,1,1,1,1,122,1,1,38,38,128,1,1,1,1,128,
  1,1,1,1,1,308,1,1,1,26,1,1,45,1,115,26,194,1,1,195,308,
  1,361,1,219,414,205,186,1,1,1,1,449,1,1,1,100,100,1,37,
  1,10,36,1,36,1,6,194,10,1,1,1,1,1,17,1,144,43,10,29,29,
  1,1,1,144,1,1,1,26,226,1,1,23,1,1,52,1,104,1,93,1,1,1,
  195,1,1,1,242,1,1,205,1,1,1,1,102,1,1,1,226,620,1,23,
  52,1,1,1,104,195,93,1,1,1,308,1,1,1,414,449,1,205,186,1,
  1,1,1,1,1,1,514,1,79,70,1,70,1,1,422,1,1,1,139,172,1,
  222,1,1,278,59,1,295,1,1,1,1,1,1,1,1,59,1,1,198,1,17,
  1,122,1,1,196,1,1,1,1,149,1,1,1,1,1,1,1,149,1,1,1,1,
  1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,
  1,10,1,10,1,10,26,10,1,1,45,45,1,59,1,31,31,31,31,1,1,
  1,1,1,1,1,1,1,10,1,43,1,43,1,1,99,99,1,1,1,11,26,1,
  70,1,1,1,1,26,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,
  1,1,1,1,1,1,1,1,1,1,43,226,43,1,1,116,116,1,1,1,89,
  126,1,89,126,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,
  1,1,1,1,1,1,1,1,1,1,204,10,1,1,1,1,1,37,1,29,10,43,
  144,144,1,1,1,29,1,1,1,1,1,1,1,1,1,608,1,102,242,205,1,
  1,1,1,1,1,1,1,1,1,1,1,1,1,122,1,128,38,38,1,128,1,1,
  1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,
  1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,26,1,1,1,1,102,1,1,
  287,1,1,1,242,1,1,1,1,440,1,449,1,186,1,205,1,1,1,345,
  1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,149,149,1,
  1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,26,1,1,1,1,1,1,
  29,1,1,1,1,1,1,1,1,1,26,1,1,1,26,26,1,1,1,1,1,1,1,
  29,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,
  1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,426,116,1,174,1,1,65,1,
  35,1,1,10,116,48,1,235,86,1,10,116,1,10,1,1,1,1,1,1,48,
  1,1,1,27,1,116,1,1,1,235,1,1,1,1,1,1,1,1,1,1,1,1,1,
  1,1,14,1,113,113,75,113,75,135,113,75,135,35,16,70,16,16,
  48,16,1,1,70,16,33,134,10,1,134,1,1,1,16,48,1,16,1,27,
  1,70,189,1,1,134,1,1,1,1,1,1,1,1,38,38,38,1,1,1,1,10,
  658,1,274,1,337,1,1,337,290,64,232,1,52,1,15,1,367,1,1,
  52,1,205,15,1,110,1,1,1,1,15,1,1,1,20,1,1,110,1,1,215,
  1,1,1,32,32,1,1,1,1,1,1,32,1,1,1,1,1,274,1,68,1,1,
  274,1,58,1,1,52,1,15,1,215,1,1,52,1,1,15,1,293,110,1,1,
  232,15,1,1,1,20,1,1,1,1,1,1,1,1,1,1,1,1,1,1,32,32,32,
  1,1,1,1,43,43,31,57,57,31,13,127,127,131,131,20,131,172,
  333,528,1,131,20,333,627,34,1,688,1,1,1,131,172,1,333,1,
  388,1,455,627,1,1,1,1,764,1,1,1,614,1,1,1,1,1,975,1,
  614,1,43,43,43,26,31,57,57,52,131,131,20,131,172,333,1,
  957,131,20,333,688,34,1,1,704,1,1,131,172,1,333,1,388,1,
  455,601,1,1,1,825,1,764,1,1,1,1,614,1,1,1,1,614,1,1,57,
  16,57,13,17,43,17,205,205,10,205,95,330,1,1,205,10,330,
  371,13,1,343,343,95,312,205,95,1,330,312,243,1,107,340,1,
  1,1,337,274,274,205,205,75,70,57,330,205,107,330,57,75,70,
  43,57,43,43,26,52,131,131,20,131,172,333,704,1,131,20,333,
  1,34,1,1,625,1,1,131,172,1,333,1,388,1,455,1,1,1,601,1,
  1,1,975,1,1,614,1,1,1,1,1,1,1,614,57,57,16,57,15,205,
  205,10,205,95,330,343,740,205,10,330,343,13,1,1,371,312,95,
  205,95,1,330,410,243,1,107,409,1,1,409,68,337,274,330,205,
  70,75,75,205,205,330,205,75,70,75,127,43,45,127,383,383,
  179,383,186,326,730,1,383,179,326,774,113,1,1,1,1,1,383,
  186,1,326,1,388,1,432,665,1,1,640,1,1,1,1,1,1,1,647,1,
  1,731,1,647,1,1,43,57,127,131,131,20,131,172,333,625,1,
  131,20,333,1,34,1,601,686,1,1,131,172,1,333,1,388,1,455,
  1,1,1,627,1,825,825,975,975,614,1,1,1,1,1,1,1,614,1,57,
  17,205,205,10,205,95,330,371,804,205,10,330,1,13,1,409,340,
  410,312,205,95,1,330,410,243,1,107,1,1,1,340,337,68,337,
  107,330,75,57,70,205,330,205,205,70,75,57,127,383,383,179,
  383,186,326,1,1,383,179,326,1,113,1,640,689,977,1,383,186,
  1,326,952,388,1,432,640,1,1,665,1,1,1,664,1,1,647,1,1,1,
  1,1,1,1,647,480,480,13,480,141,358,1,1,480,13,358,1,10,1,
  495,461,540,287,480,141,1,358,575,197,1,215,529,1,1,529,
  190,428,436,536,448,214,146,146,448,448,536,448,146,214,146,
  10,10,31,29,29,70,1,31,7,10,70,10,1,13,1,1,1,31,13,1,
  31,227,84,1,84,31,1,1,1,32,32,1,1,1,38,1,38,172,1,1,
  172,38,38,1,10,29,29,31,119,119,31,10,10,31,7,1,1,13,227,
  1,31,13,1,84,1,84,1,31,70,1,1,70,32,1,32,1,172,1,38,38,
  1,1,172,1,38,1,38,10,6,10,1,52,7,5,7,11,5,1,16,16,13,
  13,7,6,1,10,13,7,1,10,11,1,1,1,6,10,10,20,20,7,7,5,20,
  20,20,20,5,7,7,29,10,1,99,10,10,31,13,7,1,1,31,1,227,
  31,29,1,29,227,22,1,31,70,1,1,70,32,1,32,172,1,1,38,38,
  1,1,172,1,38,1,38,29,36,1,29,6,29,36,6,1,36,36,29,29,
  13,14,1,13,13,14,1,13,13,1,1,13,15,15,15,29,29,48,48,48,
  29,29,13,13,48,48,48,19,1,10,7,31,19,10,1,31,1,1,1,31,
  29,1,31,1,22,1,29,31,1,1,1,32,32,1,1,1,38,1,38,1,172,
  1,172,38,38,1,119,13,16,31,16,11,1,113,89,1,1,31,36,1,1,
  389,30,1,70,89,1,1,1,32,1,115,1,1,1,110,38,87,246,232,1,
  38,1,110,99,12,119,119,52,1,130,1,1,1,81,1,1,81,1,1,1,
  1,81,1,1,1,195,195,1,1,1,149,1,149,1,1,1,1,149,149,1,
  10,29,1,10,1,70,70,1,1,29,29,1,31,1,22,1,31,1,1,1,31,
  1,32,32,1,1,38,38,1,1,172,1,172,1,38,38,10,16,5,1,1,11,
  13,13,10,6,1,10,13,7,1,7,1,1,1,11,10,10,6,20,20,7,5,7,
  20,20,20,20,7,7,5,119,10,1,19,19,1,1,84,13,1,31,227,84,
  1,31,1,1,1,31,1,32,32,1,1,38,38,1,172,1,1,172,1,38,38,
  1,1,89,1,1,389,1,36,1,31,1,30,1,70,39,1,1,89,1,32,115,
  246,87,38,110,1,1,1,1,232,1,38,110,1,11,1,13,13,10,6,1,
  7,13,7,1,10,16,1,1,16,10,6,10,13,13,5,7,7,13,13,13,13,
  7,5,7,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,
  1,1,1,1,1,1,1,1,1,16,389,1,70,13,1,1,389,98,1,31,1,1,
  1,89,115,1,32,87,232,1,38,110,1,1,394,232,110,1,38,1,1,
  70,13,1,31,1,98,1,1,89,1,1,113,115,32,1,1,1,38,1,110,
  232,87,232,394,110,38,1,29,1,29,1,227,29,22,1,1,1,1,1,1,
  1,1,1,1,1,1,1,1,1,99,1,99,1,1,1,227,13,1,1,84,84,1,
  227,1,1,1,1,1,1,1,1,1,1,1,1,99,1,1,99,1,1,1,29,1,10,
  1,84,1,10,1,1,1,13,1,32,32,1,172,38,38,1,1,172,1,1,1,
  38,38,1,29,29,14,1,29,36,1,1,36,15,15,15,13,29,48,48,48,
  29,13,29,29,48,48,48,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,
  1,1,1,1,1,1,1,1,1,84,1,10,19,1,1,19,32,1,32,172,1,1,
  38,38,172,1,1,1,38,1,38,84,1,1,1,1,1,1,1,1,1,1,99,1,
  1,1,1,99,1,1,1,1,1,1,84,98,1,1,98,20,20,20,84,22,27,
  27,27,22,84,84,84,27,27,27,1,1,1,1,1,1,1,1,1,1,1,1,1,
  1,1,1,1,1,1,1,13,1,1,1,32,32,1,1,172,38,1,38,1,172,1,
  1,38,38,1,1,1,16,1,115,32,246,246,110,38,1,1,394,1,87,1,
  110,38,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,
  1,1,1,1,1,1,1,1,1,1,1,32,115,1,232,232,110,1,38,394,1,
  1,87,38,110,1,10,10,1,1,14,14,1,1,1,1,1,1,14,14,10,1,
  1,1,14,14,1,1,1,1,14,1,14,1,1,14,1,14,1,1,1,1,14,14,
  1,29,1,1,1,31,10,31,84,1,1,1,1,1,1,10,31,31,29,1,1,1,
  7,7,1,1,1,1,7,3,7,7,1,1,1,1,7,7,3,1,1,1,1,3,7,7,29,
  29,31,1,1,1,84,31,1,1,1,10,1,1,1,1,1,1,7,7,7,1,1,1,
  1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,
  1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,
  1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,
  1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,
  1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,
  1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,
  1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,
  1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,
  1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,391,391,1,1,
  1,1,144,807,1,945,1,1,1,1,1,814,1006,890,1028,846,971,1,
  1,1,1,1,850,760,776,1,1016,922,1018,1,1041,1,1,1,1,1,1,
  1,1,787,709,1,1,1,1,1,1,1,1,128,1,1,168,1,168,763,1,
  365,1,363,1,1,1,762,365,932,976,489,813,1,1017,1,1,1,363,
  590,1,1,519,810,1,1,769,1,1,1,1,1,1,1,1,643,1,1,1,1,1,
  1,1,1,391,1,1,1,1,144,877,1,1,814,850,1,1028,922,1,1006,
  776,1,1,1,1,1039,1,1,1,1,753,890,1,1016,1,1,1,1,1046,1,
  1,1,1,1,1,709,1,1,1,1,1,1,1,1,1,1,1,128,1,17,1,1,176,
  749,206,1,1,885,749,176,1,1,162,1,1,1,1,1,1,206,472,1,1,
  227,885,819,1,698,1,990,1,1,1,1,1,446,1,446,1,1,1,1,1,
  1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,
  1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,
  1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,
  1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,
  168,822,1,365,762,363,1,976,810,1,365,1,1,489,847,1,1014,
  1,1,1,363,589,932,1,519,1,1,1,1,1,1029,1,1,1,1,1,1,1,
  1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,
  1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,
  1,1,1,1,657,1,85,1,164,1,1,1,1,85,1,1,71,897,1,1,1,1,
  1,164,381,1,1,105,1,651,1,1,1,1012,1,1,1,1,1,355,256,
  355,1,1,1,1,1,1,1,1,1,1,1,1,1,1,76,23,48,190,48,31,1,
  1,1,1,1,70,52,76,1,31,194,31,1,1,294,294,1,1,1,1,1,144,
  1,43,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,
  1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,
  1,1,1,13,1,1,1,23,11,23,54,11,1,1,1,1,1,1,29,24,48,1,
  16,80,48,1,20,160,1,1,1,1,1,1,19,19,10,1,1,1,1,1,1,1,
  1,1,1,36,48,48,76,107,23,1,1,1,1,1,1,55,52,55,1,59,130,
  76,1,1,116,1,1,1,1,1,1,1,144,43,1,1,1,1,1,1,1,1,1,1,
  55,29,55,130,29,1,1,1,1,1,1,43,26,70,1,43,99,70,1,175,
  175,95,1,1,1,1,1,1,1,29,1,1,1,1,1,1,1,1,1,1,1,1,1,1,
  1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,
  1,1,1,1,1,1,1,1,107,19,36,19,54,1,1,1,1,1,1,130,30,
  194,1,24,194,190,1,1,303,1,1,1,1,1,1,1,174,48,1,1,1,1,
  1,1,1,130,130,194,194,80,1,1,1,1,1,1,99,104,99,1,168,
  168,194,1,1,1,1,1,1,1,1,1,1,1,149,1,1,1,1,1,1,1,48,
  55,1,1,1,1,1,1,1,1,1,31,76,1,59,36,1,1,1,189,1,1,1,1,
  1,1,43,144,1,1,1,1,1,1,1,1,1,19,11,80,1,54,1,1,1,29,
  24,1,1,6,130,23,1,61,1,54,1,1,1,1,1,19,10,19,1,1,1,1,
  1,1,1,194,48,160,1,121,1,1,1,70,43,1,1,1,99,76,1,104,1,
  74,1,1,1,1,1,144,43,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,
  20,36,1,24,1,1,1,1,111,1,1,1,1,1,1,48,174,1,1,1,1,1,
  1,1,1,160,1,1,1,1,1,13,20,23,1,16,1,1,1,61,84,1,1,1,
  1,1,1,10,19,19,1,1,1,1,1,1,1,1,1,1,1,1,43,43,149,1,
  104,1,104,1,1,342,225,1,1,1,1,1,130,1,1,1,1,1,1,1,1,1,
  1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,
  1,1,1,1,1,1,1,1,95,81,74,1,77,1,81,1,1,378,1,1,1,1,1,
  1,155,1,155,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,
  1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,
  1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,
  1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,43,55,1,43,1,1,
  1,1,116,1,1,1,1,1,1,29,1,1,1,1,1,1,1,1,1,1,1,1,149,
  81,1,76,1,157,1,1,1,1,1,1,52,52,1,1,1,1,1,1,1,1,1,
  194,48,1,160,43,121,1,1,1,1,1,1,43,144,1,1,1,1,1,1,1,
  1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,168,31,
  1,80,1,77,1,1,1,1,1,56,17,56,1,1,1,1,1,1,1,1,1,1,303,
  1,1,1,1,1,1,149,1,1,1,1,1,1,1,1,1,1,160,160,1,1,1,1,
  1,1,43,1,144,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,
  1,1,1,1,1,1,175,1,1,1,1,1,1,192,192,1,1,1,1,1,1,1,1,
  294,1,1,1,1,1,1,171,1,1,1,1,1,1,1,1,1,1,1,1,1,155,1,
  155,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,
  1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,
  1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,19,19,1,1,1,
  1,1,1,1,19,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,
  1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,506,1,1,366,1,403,1,
  573,298,368,198,1,379,184,36,1,378,338,122,1,249,378,168,
  256,1,1,1,1,1,36,1,59,1,17,338,198,1,293,1,466,1,1,1,1,
  1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,
  1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,
  1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,185,185,122,116,116,
  122,26,81,81,156,198,424,156,424,54,921,1,198,424,248,908,
  479,1,911,1,1,1,156,424,906,54,1,479,908,264,1,993,1,1,1,
  1,1,1,1,580,1,703,799,513,703,1,1,1,1,185,185,185,20,122,
  116,116,144,156,198,424,156,424,54,1,946,198,424,248,911,
  479,1,1,1,1,941,156,424,993,54,1,479,1,264,937,1,1,1,1,
  1,997,1,1,1,513,580,513,1,799,1,1,1,1,116,45,116,26,91,
  38,91,169,195,356,169,356,81,1,1,195,356,269,858,493,881,1,
  950,1,1,169,356,1015,81,1,493,968,136,940,1,1,1,1,1,1,1,
  1,635,914,862,1,914,1,1,1,1,1,185,116,185,185,20,144,156,
  198,424,156,424,54,1,937,198,424,248,1,479,1,1,993,1,1,
  156,424,1,54,997,479,1,264,946,1,911,1,941,1,1,1,1,513,
  799,1,1,580,513,1,1,1,1,116,116,45,116,32,169,195,356,169,
  356,81,950,1,195,356,269,1,493,1,1,1,1,1,169,356,1,81,1,
  493,881,136,1,950,1,881,1,1,1,1,1,914,1,635,914,635,1,1,
  1,1,1,81,38,31,81,174,148,341,174,341,13,588,624,148,341,
  37,594,254,624,606,672,1,1,174,341,642,13,1,254,597,147,
  606,672,606,624,1,1,878,1,1,1,655,697,655,1,438,1,1,1,1,
  185,116,81,156,198,424,156,424,54,993,1,198,424,248,1,479,
  911,1,906,1,1,156,424,1,54,1,479,1,264,1,921,908,908,1,1,
  1,1,1,799,703,513,580,703,1,1,1,1,1,116,91,169,195,356,
  169,356,81,1,940,195,356,269,1,493,1,881,1015,1,1,169,356,
  950,81,1,493,1,136,1,1,858,968,1,1,1,1,1,1,1,914,635,
  862,914,1,1,1,1,81,174,148,341,174,341,13,672,606,148,341,
  37,606,254,606,624,642,1,1,174,341,672,13,878,254,624,147,
  624,588,594,597,1,1,1,1,1,655,438,1,1,697,655,1,1,1,1,
  252,77,440,252,440,17,1,1,77,440,185,1,142,980,980,981,1,
  700,252,440,981,17,721,142,1004,215,1,1,1,1004,700,1,721,1,
  1,343,194,419,343,419,194,1,1,1,1,43,43,31,10,10,89,1,13,
  31,21,1,21,1,89,1,1,1,31,31,89,7,127,43,89,43,102,102,
  185,185,198,1,1,1,1,89,1,126,126,1,89,1,1,1,1,43,43,43,
  6,189,189,16,21,21,144,16,104,1,104,1,1,13,36,1,10,257,
  10,144,36,1,104,1,104,257,1,122,1,1,1,38,128,38,1,128,1,
  1,1,1,10,31,10,1,89,36,43,10,99,43,102,185,185,1,84,31,
  10,102,10,1,36,181,10,99,1,89,1,17,1,1,1,1,26,26,11,1,
  70,1,1,1,1,1,43,10,1,99,43,43,10,89,36,185,185,102,1,
  127,31,10,102,10,198,36,1,10,99,1,89,1,127,1,198,1,1,1,
  126,89,1,89,126,1,1,1,1,10,99,1,21,31,21,181,13,1,89,1,
  1,1,31,43,89,7,17,43,99,31,102,185,185,102,1,1,1,1,1,70,
  1,1,26,26,11,1,1,1,1,67,67,10,7,6,67,10,39,14,39,1,59,
  7,10,14,4,59,6,67,10,14,39,14,39,59,1,1,1,1,31,1,31,1,
  31,31,1,1,1,1,189,104,185,19,185,100,330,205,284,1,275,
  102,1,150,39,1,144,39,181,330,1,284,1,217,1,564,1,1,1,
  298,99,1,172,562,1,1,1,1,1,102,116,116,1,1,284,1,1,1,
  102,99,284,14,198,189,189,89,128,330,173,284,1,1,1,1,1,
  213,1,1,287,1,65,1,1,1,1,43,21,1,10,1,104,1,1,257,43,
  43,104,6,122,16,104,36,189,189,144,144,1,1,257,1,1,38,128,
  1,1,128,38,1,1,1,1,21,185,43,89,1,89,1,1,10,31,1,10,1,
  13,102,31,1,99,181,99,1,1,17,1,1,26,11,26,70,1,1,1,1,1,
  1,116,21,160,160,160,1,1,21,10,160,6,127,21,19,10,116,19,
  116,19,1,1,127,1,1,116,116,1,116,1,116,1,1,1,1,1,1,330,
  1,1,1,185,89,330,14,392,144,185,99,173,284,128,330,1,1,1,
  1,1,65,287,1,1,1,213,1,1,1,1,1,104,1,1,162,43,36,104,6,
  37,16,100,43,189,144,144,189,179,1,179,1,1,10,29,29,144,
  144,43,1,1,1,1,185,39,1,1,89,185,185,14,198,104,330,102,
  284,205,330,150,1,1,1,1,1,213,1,1,1,287,65,1,1,1,1,185,
  1,1,1,102,39,39,1,1,150,102,1,330,1,330,1,1,198,1,1,1,
  65,287,213,1,1,1,1,1,1,1,1,89,102,185,14,275,104,284,185,
  284,150,330,205,564,1,1,1,1,99,1,298,562,1,194,1,1,1,1,
  1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,
  1,1,1,198,17,564,59,179,179,198,127,1,217,1,392,1,1,1,1,
  1,1,1,1,149,1,149,1,1,1,1,43,1,10,1,21,185,43,1,89,1,
  89,1,1,127,1,1,126,89,1,89,126,1,1,1,1,1,185,10,1,43,1,
  10,89,1,99,181,84,1,1,1,1,1,1,70,26,11,26,1,1,1,1,39,
  1,1,205,185,1,284,1,284,1,1,275,1,1,562,194,1,99,298,1,
  1,1,1,1,1,10,39,10,67,67,67,67,59,1,59,1,1,1,31,31,31,
  31,1,1,1,1,1,179,1,1,1,564,1,1,1,1,1,1,1,1,1,1,149,
  149,1,1,1,1,1,189,43,1,100,1,100,162,1,37,1,1,144,43,
  144,10,29,29,1,1,1,1,1,284,1,330,1,392,1,1,1,1,1,1,213,
  287,65,1,1,1,1,1,89,181,99,1,127,1,1,1,1,126,1,89,126,
  89,1,1,1,1,1,189,116,189,1,1,198,1,1,287,65,1,213,1,1,
  1,1,1,1,185,39,275,1,1,1,1,1,562,172,1,99,298,1,1,1,1,
  185,1,1,392,1,1,1,213,1,65,1,287,1,1,1,1,198,1,1,1,1,
  287,1,65,1,213,1,1,1,1,1,1,179,1,1,149,149,1,1,1,1,1,
  1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,149,1,149,1,1,1,
  1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,
  26,26,1,1,1,1,1,1,1,29,1,1,1,1,1,1,1,1,1,1,1,1,1,1,
  26,26,1,1,1,1,29,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,65,
  426,174,1,1,116,1,35,1,1,48,116,10,1,1,1,1,48,116,1,27,
  1,1,1,1,1,1,10,1,1,1,10,235,116,86,235,1,1,1,1,1,1,1,
  1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,244,246,246,1,1,1,
  807,945,1,1,1,877,832,892,1,1,1,753,1,1,1,1,1,1,1022,1,
  857,1,1,760,1055,1,1,1,1,1,1,787,1,1,1,1,1,1,861,1,1,
  1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,
  1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,
  1,1,1,1,368,86,368,1,104,185,104,1,1,363,763,365,1,1,1,
  822,363,1,1,519,1,589,847,1,1,1,365,1,1,1,489,1034,1,590,
  912,1,1,1,1,1,1,643,1,1,1,1,1,1,632,1,1,1,1,368,1,1,
  1,1,712,1,801,676,1,1,848,958,676,429,777,1,1,1,600,882,
  1,1,1,1,1,777,1,1,1,1,600,1,1,1,1,1,1,775,522,1,1,1,
  1,1,1,1,1,1,1,1,1,139,1,88,768,1,415,1,402,1,840,947,
  758,415,788,816,505,1,1,859,1,1,1,402,1,1,1,499,1008,1,
  585,1,1,702,1,1,1,1,533,1,1,1,1,1,1,1,1,1,1,1,199,1,
  1,903,1,984,909,856,1,1019,1026,1002,1,871,1,844,1023,811,
  1,1,1,1,803,1,854,1,1,1,1025,811,1,1,1,1,1,1,1,1,1,1,
  1,1,1,1,835,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,
  1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,
  1,1,1,1,1,1,1,1,104,808,1,415,758,402,1,967,1,1,415,1,
  1,505,1,585,1,1,1,1,402,1,788,1,499,1,1,1,1,1,1,1,1,1,
  1,1,1,1,1,1,1,1,1,1,1,1,246,750,1,737,791,633,1,933,
  978,791,1,904,951,706,1,1,992,1,1,1,633,1,904,1,1,917,
  719,1,1,1,1009,1,1,1,522,778,1,1,1,1,1,1,1,1,1,1,1,1,
  554,1,167,1,1,1,1,554,1,1,188,1,502,1,1,1,1,167,1,867,
  1,268,1,1,486,1,1,596,1,1,1,454,454,1,1,1,1,1,1,454,1,
  1,1,1,1,1,1,1,36,1,76,70,31,104,31,1,31,1,1,1,1,23,1,
  76,1,48,107,48,52,1,1,1,1,1,1,1,144,1,1,1,1,1,1,43,1,
  1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,
  1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,13,1,1,1,
  70,43,70,70,43,95,43,43,1,1,1,29,1,55,1,29,130,55,26,1,
  1,1,1,1,1,1,1,1,1,1,1,1,1,29,1,1,1,1,1,1,1,55,55,76,
  176,59,81,81,52,1,1,1,48,1,48,1,23,190,76,52,1,1,160,1,
  1,1,144,1,1,1,1,1,1,1,43,1,1,1,1,1,1,48,29,48,61,16,
  1,20,160,1,1,1,11,1,23,1,11,54,23,24,1,1,61,1,1,1,19,
  19,1,1,1,1,1,1,10,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,
  1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,
  1,1,194,138,104,138,104,1,149,1,1,1,1,80,1,194,1,130,194,
  130,104,1,1,1,1,1,1,1,1,1,1,1,1,1,1,149,1,1,1,190,40,
  165,165,77,1,20,1,1,1,1,54,1,36,1,19,19,107,30,1,1,1,1,
  1,1,1,174,1,1,1,1,1,1,48,1,1,1,55,48,1,1,1,52,31,1,1,
  1,1,1,76,1,23,1,1,81,81,1,1,1,1,1,43,1,1,1,1,1,1,1,
  144,1,1,1,1,138,43,130,1,99,1,1,1,29,1,1,1,13,40,70,1,
  130,1,70,1,1,1,1,29,1,1,1,1,1,1,1,1,1,1,165,31,121,43,
  160,1,1,1,23,1,1,1,1,36,76,1,74,1,104,1,1,1,1,43,1,1,
  1,1,1,1,144,1,1,1,1,1,80,1,1,1,1,24,1,176,1,61,1,1,
  36,1,1,70,1,1,1,104,1,1,1,1,1,1,1,1,1,1,1,1,29,1,1,
  1,1,6,1,59,1,16,1,1,26,1,1,26,1,1,1,17,56,1,1,1,1,1,
  1,56,1,1,1,81,1,1,1,1,1,1,74,1,54,1,1,157,1,1,1,1,1,
  1,155,1,1,1,1,1,1,1,155,1,1,1,43,1,1,1,24,1,1,1,24,
  30,52,1,157,1,76,1,1,1,52,52,1,1,1,1,1,1,1,1,1,1,1,1,
  1,1,1,149,1,80,1,1,126,40,1,1,1,1,1,1,1,1,1,1,1,1,1,
  130,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,
  1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,
  1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,
  1,1,1,1,1,1,1,1,1,1,1,1,48,1,11,1,1,20,1,1,20,1,1,1,
  10,19,1,1,1,1,1,1,19,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,
  1,1,1,1,1,1,1,1,1,1,1,1,1,1,194,55,43,121,1,160,1,1,
  1,144,43,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,
  1,1,1,1,1,1,1,1,1,1,1,19,48,24,54,1,61,1,1,1,19,10,1,
  1,1,1,1,1,19,1,1,1,1,20,1,1,1,1,1,1,48,174,1,1,1,1,
  1,1,1,1,1,1,31,1,1,1,1,1,1,43,144,1,1,1,1,1,1,1,1,1,
  1,81,1,26,1,1,1,1,52,1,1,1,1,1,1,52,1,1,1,1,1,1,1,1,
  155,1,1,1,1,1,1,1,155,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,
  1,1,1,1,1,1,1,1,192,1,1,1,1,1,1,192,1,1,1,1,1,1,1,1,
  1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,
  1,1,1,1,1,1,1,1,1,19,1,1,1,1,1,1,19,1,1,1,1,1,1,1,
  1,1,19,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,
  1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,
  1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,
  1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,
  1,1,1,1,1,1,391,391,1,1,1,1,144,1,1,1,814,1006,890,1,1,
  1,850,776,1046,1016,1,1,753,1,1,877,1,1039,1,1,1,1,1,922,
  1028,1,1,1,1,1,1,1,1,1,1,709,1,1,1,1,1,1,1,128,1,1,
  168,1,168,1,1,363,762,365,932,1,1,1,363,1,1,519,1,847,
  589,1,1,822,365,1014,1,1,489,1029,1,810,976,1,1,1,1,1,1,
  1,1,1,1,1,1,1,1,1,1,1,391,1,1,1,1,144,1,814,850,1,
  1006,776,1041,1,1,1,890,1,1016,1,971,760,1,1,807,945,1,1,
  1,846,1,1018,1,1,1028,922,1,1,1,1,1,1,1,1,1,709,787,1,
  1,1,1,1,1,128,1,17,1,749,206,749,176,1,698,1,1,206,1,1,
  227,1,1,472,1,1,1,176,1,1,1,162,990,819,885,1,1,885,1,1,
  1,1,1,1,1,1,446,446,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,
  1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,
  1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,
  1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,
  1,1,1,1,1,1,1,1,1,1,1,1,168,1,762,363,1,365,1,769,1,1,
  363,932,1,519,1,813,590,1,1,763,365,1017,1,1,489,1,1,1,1,
  976,810,1,1,1,1,1,1,1,1,1,1,643,1,1,1,1,1,1,1,1,1,1,
  1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,
  1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,164,1,85,1,1,1,1,
  164,1,1,105,1,897,381,1,1,657,85,1,1,1,71,1012,651,1,1,
  1,1,1,1,1,1,1,1,1,1,355,355,256,1,1,1,1,1,1,1,1,1,1,
  1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,
  1,1,1,1,1,1,1,1,1,1,1,1,1,1,48,48,76,1,1,1,55,55,189,
  59,1,1,31,1,1,76,23,1,1,1,1,1,1,130,107,1,36,1,1,1,1,
  1,1,1,1,43,1,144,1,1,1,1,55,29,55,1,1,1,43,70,116,43,
  1,43,43,1,1,70,29,95,1,1,13,1,1,99,130,1,1,1,1,1,1,1,
  1,1,1,29,1,1,1,1,1,1,48,55,1,1,1,1,76,116,59,1,1,52,
  1,1,1,1,1,1,1,23,1,76,36,1,107,130,1,1,1,1,1,1,1,1,1,
  43,144,1,1,1,1,1,61,1,1,29,1,1,6,1,80,24,1,1,23,11,54,
  1,1,11,54,23,130,19,19,130,1,1,1,1,1,1,1,1,19,19,10,1,
  1,1,1,160,1,1,70,1,43,1,1,149,1,1,1,76,48,74,1,1,23,
  121,48,99,194,36,194,1,1,1,1,1,1,1,1,1,144,43,1,1,1,1,
  1,1,175,104,175,80,1,1,76,1,1,1,20,1,1,1,61,1,160,1,1,
  1,1,1,1,1,1,1,1,1,1,192,1,192,1,1,1,1,1,1,1,1,1,1,1,
  1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,
  1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,
  1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,55,175,43,1,1,26,1,
  1,1,13,1,1,1,29,95,70,1,1,130,99,1,1,1,1,1,1,1,1,1,
  29,1,1,1,1,1,1,1,1,160,43,1,1,48,23,121,1,1,48,74,76,
  194,36,194,99,1,1,1,1,1,1,1,1,144,1,43,1,1,1,1,1,1,
  342,1,1,1,294,160,378,1,1,84,294,160,1,303,111,303,1,1,1,
  1,1,1,1,1,1,1,171,1,1,1,1,1,104,1,1,1,31,16,77,1,1,
  16,77,31,168,24,24,168,1,1,1,1,1,1,1,1,56,56,17,1,1,1,
  1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,
  1,1,1,1,1,1,1,43,1,1,31,1,1,1,1,160,225,104,1,1,1,1,
  1,1,1,1,1,1,1,1,130,1,1,1,1,1,1,1,1,52,24,81,1,1,20,
  157,81,104,30,20,149,1,1,1,1,1,1,1,1,1,52,52,1,1,1,1,
  1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,
  1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,
  1,1,1,1,1,1,1,1,1,48,294,31,1,1,190,194,1,1,1,1,1,1,
  1,1,144,43,1,1,1,1,1,1,1,1,11,1,48,1,1,54,80,1,1,1,1,
  1,1,1,1,19,10,19,1,1,1,1,1,1,1,1,81,1,1,1,1,1,1,1,1,
  1,1,1,1,155,155,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,
  1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,
  1,1,1,1,1,1,1,1,1,80,54,1,1,1,1,1,1,1,1,1,1,10,19,
  19,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,155,155,1,1,1,
  1,1,194,190,1,1,1,1,1,1,1,1,1,1,43,144,1,1,1,1,1,1,
  194,168,1,1,1,1,1,1,1,1,1,149,1,1,1,1,1,19,194,1,1,1,
  1,1,1,1,1,1,48,174,1,1,1,1,1,1,1,1,1,1,1,1,1,48,1,
  174,1,1,1,1,1,1,1,1,1,1,1,1,149,1,1,1,1,1,1,1,1,1,1,
  1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,
  1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,
  1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,
  1,1,1,1,1,19,19,1,1,1,1,19,1,1,1,1,1,1,1,1,1,1,1,1,
  1,1,3,7,7,7,5,7,7,7,38,38,54,38,54,136,1,1,38,54,136,
  379,239,538,1,1,532,1,38,54,1,136,1,239,1,54,1,379,1,1,
  1,1,1,1,1,1,1,296,296,1,1,296,1,296,1,7,7,7,5,7,7,7,
  38,38,54,38,54,136,1,1,38,54,136,379,239,538,1,1,532,1,
  38,54,1,136,1,239,1,54,1,379,1,1,1,1,1,1,1,1,1,296,296,
  1,1,296,1,296,1,7,7,5,5,5,5,31,31,31,31,31,31,126,1,31,
  31,31,1,45,1,126,1,1,1,31,31,215,31,1,45,212,11,212,1,
  126,1,1,1,1,1,1,1,1,1,1,1,518,1,518,1,1,3,7,5,7,7,54,
  54,131,54,131,75,227,1,54,131,75,347,202,1,1,1,1,1,54,
  131,319,75,1,202,258,35,258,347,227,1,1,404,1,1,1,1,1,
  500,500,1,296,500,296,500,1,7,5,7,7,54,54,131,54,131,75,
  227,1,54,131,75,347,202,1,1,1,1,1,54,131,319,75,1,202,
  258,35,258,347,227,1,1,404,1,1,1,1,1,500,500,1,296,500,
  296,500,1,5,7,7,31,31,31,31,31,31,212,1,31,31,31,212,45,
  215,215,1,1,1,31,31,1,31,1,45,1,11,1,212,212,1,1,1,1,1,
  518,518,1,518,518,518,1,518,1,518,518,7,7,20,20,20,20,20,
  26,195,1,20,20,26,1,22,194,79,1,211,1,20,20,194,26,211,
  22,1,10,1,1,195,1,1,101,1,1,81,81,1,148,148,81,148,148,
  148,148,81,3,131,131,143,131,143,59,1,1,131,143,59,1,131,
  319,271,1,1,1,131,143,258,59,404,131,251,35,251,1,1,1,1,
  1,1,1,626,626,1,1,1,626,707,1,707,1,626,131,131,143,131,
  143,59,1,1,131,143,59,1,131,319,271,1,1,1,131,143,258,59,
  404,131,251,35,251,1,1,1,1,1,1,1,626,626,1,1,1,626,707,
  1,707,1,626,38,38,31,40,40,351,1,11,31,102,191,102,158,
  284,1,377,1,31,31,283,20,509,123,260,123,128,128,1,1,1,
  313,1,1,193,393,1,586,537,1,654,741,537,654,1,38,38,38,20,
  1,1,31,40,40,128,38,283,1,1,509,1,11,31,158,102,377,102,
  128,93,191,260,1,1,1,1,1,1,393,1,1,537,654,1,537,586,586,
  1,382,40,31,40,1,1,31,107,31,333,107,107,1,1,687,1,31,31,
  107,157,687,45,354,157,333,354,1,1,1,1,1,1,870,1,1,216,
  883,1,883,864,216,707,707,38,40,1,1,38,38,20,260,38,213,
  213,1,1,1,31,40,63,40,321,38,191,11,268,1,179,1,1,1,1,1,
  1,1,1,654,1,382,586,537,741,1,356,40,1,1,40,31,40,354,26,
  1,517,1,1,1,31,107,283,31,102,107,333,93,128,370,370,1,1,
  1,1,1,1,1,1,883,707,707,216,216,864,1,1,481,1,102,31,31,
  477,102,1,172,1,1,1,20,157,26,31,572,20,477,157,120,1,38,
  1,1,381,1,1,1,467,1,784,1,559,784,747,747,872,1,1,179,
  370,38,370,270,485,302,1,1,1,26,269,49,246,920,54,246,269,
  471,1,1,1,1,1,1,1,1,1,1,820,1,1,995,996,851,1,880,1,1,
  1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,
  1,1,1,1,1,1,1,1,1,1,38,40,1,102,1,380,1,1,1,38,38,223,
  20,465,38,268,93,1,1,26,1,1,313,1,1,1,382,1,1,1,356,741,
  654,1,741,193,40,370,107,530,1,1,1,1,40,31,305,40,565,26,
  128,93,1,447,269,1,1,1,1,1,1,707,1,707,1,1,864,883,1,1,
  38,1,102,437,1,1,502,1,102,31,437,31,484,102,120,40,246,
  120,246,1,1,381,1,1,467,559,1,1,872,1,747,784,1,696,535,
  1,1,690,1,1,1,1,447,1,120,1021,198,1,546,359,725,95,1,1,
  561,1,1,1045,743,1,1,1,780,1033,1040,1,998,964,255,247,1,
  571,1,123,45,351,20,1,146,451,123,1,198,54,1,1,40,1,1,
  874,102,1,1,965,630,1,1,735,450,735,213,1,1,1,223,160,160,
  26,744,279,607,157,1,233,563,1,1,935,1,1,1,1005,1,1,1,
  963,999,929,1,1,1,1,1,1,213,145,224,224,969,376,186,194,1,
  332,1,1,1,1,1,1,1,1,1,1020,1030,1,1000,989,1007,1,722,1,
  1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,
  1,1,1,1,465,1,744,558,1,269,1,413,1021,1,682,1,1,1,1,1,
  1,1,1,1,1,728,954,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,
  1,1,1,1,1,1,1,1,1,1,1,1,1,1,38,1,40,1,102,1,123,1,
  268,191,1,1,253,1,1,382,356,1,1,741,193,1,1,1,586,393,
  160,40,1,107,1,157,447,1,221,1,1,526,1,1,707,1,1,1,1,38,
  1,707,1,864,870,224,1,1,233,1,1,607,1,1,1,641,1,1,637,
  839,1,1,999,869,1,1,1,973,925,1,102,246,157,514,514,1,1,
  1,1,1,1,559,1,1,872,696,535,1,1,696,1,515,1,1,1,1,1,
  927,1,1,1,1,1,821,1,1,1,954,821,1,1,1,1,1,1,123,358,
  324,1,1,1,1,1,1,102,630,1,965,450,735,735,1,1,1,674,1,
  725,1,1,1,1,1,1,1,961,852,1,1033,1035,732,1,1,1040,1001,
  1056,1,295,481,1,1,263,1,1,394,394,1,1,738,684,1,1,986,
  655,1,1,246,1,1,793,1,1,1,961,1,1,1033,852,1,1063,1,1035,
  732,370,1,1,1,1,1,1,1045,1,1040,1,743,1035,1024,1033,1,
  780,1,1,1,1,1,577,1,1,996,820,1,1,1,995,1,1,1,1,1,1,1,
  1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,
  1,1,1,1,886,1,1,1,1,843,1,569,1,1,1,1,1,1,1,1,1,1,1,
  1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,614,45,252,707,1,1,1,1,
  707,1,1,1,1,252,614,45,1,1,1,1,1,1,1,1,1,1,1,1,1,406,
  252,707,1,406,1,1,1,735,252,1,1,1,1,1,1,1,1,1,1,1,735,
  707,7,7,7,5,7,7,7,38,38,54,38,54,136,1,1,38,54,136,379,
  239,538,1,1,532,1,38,54,1,136,1,239,1,54,1,379,1,1,1,1,
  1,1,1,1,1,296,296,1,1,296,1,296,1,3,7,7,7,7,5,38,38,
  54,38,54,136,1,538,38,54,136,1,239,1,1,379,1,532,38,54,
  379,136,1,239,538,54,538,1,1,538,532,1,1,296,1,1,296,1,1,
  1,296,1,296,1,1,7,7,7,7,5,38,38,54,38,54,136,1,538,38,
  54,136,1,239,1,1,379,1,532,38,54,379,136,1,239,538,54,538,
  1,1,538,532,1,1,296,1,1,296,1,1,1,296,1,296,1,1,5,5,5,
  5,20,20,26,20,26,13,1,150,20,26,13,50,26,150,1,50,1,1,
  20,26,128,13,1,26,123,10,123,50,1,150,1,1,1,518,1,1,518,
  518,518,1,518,518,518,518,1,7,7,5,31,31,31,31,31,31,1,
  212,31,31,31,126,45,126,1,215,1,1,31,31,1,31,1,45,1,11,
  1,126,1,212,1,1,1,518,1,1,518,1,1,1,1,1,1,1,1,3,7,54,
  54,131,54,131,75,1,317,54,131,75,1,202,227,1,327,1,1,54,
  131,347,75,1,202,1,35,1,1,1,317,1,1,404,421,1,1,421,1,1,
  1,500,1,500,1,1,7,54,54,131,54,131,75,1,317,54,131,75,1,
  202,227,1,327,1,1,54,131,347,75,1,202,1,35,1,1,1,317,1,
  1,404,421,1,1,421,1,1,1,500,1,500,1,1,38,38,70,38,70,31,
  1,113,38,70,31,1,38,241,1,241,244,308,38,70,113,31,1,38,
  110,10,110,1,1,113,308,1,185,141,1,1,141,306,306,1,488,
  306,488,306,1,38,38,31,40,40,1,1,11,31,102,213,102,191,1,
  1,321,1,31,31,268,20,1,123,223,123,63,99,1,1,321,1,1,1,
  1,1,1,356,193,1,741,382,586,393,1,38,38,38,20,1,1,31,40,
  40,99,38,351,1,284,465,1,11,31,191,102,1,102,63,93,158,
  284,1,351,377,1,313,1,1,1,393,193,393,1,586,356,537,1,1,
  40,31,40,1,1,31,107,31,312,107,38,1,1,565,1,31,31,128,
  157,1,45,305,157,283,399,1,1,102,1,1,1,1,1,1,38,870,1,
  864,1,216,1,1,38,40,1,351,38,38,20,284,38,1,1,99,1,465,
  31,40,128,40,1,38,158,11,283,1,1,191,509,1,253,393,1,1,
  193,393,1,1,537,193,654,1,1,40,1,1,40,31,40,399,26,1,1,
  399,1,1,31,107,333,31,1,107,283,93,107,1,1,38,687,1,526,
  1,1,1,870,870,1,1,216,38,883,1,1,1,481,102,31,31,514,102,
  220,1,220,1,558,20,157,120,31,1,20,437,157,26,220,1,220,
  572,1,1,467,1,1,1,515,1,1,747,535,784,467,1,1,1,1,1,1,
  1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,
  1,1,1,1,1,1,1,1,191,38,220,220,330,1,1,1,942,1,26,221,
  471,38,1,344,344,455,49,548,1,1,920,1,1,1,1,1,1,772,955,
  1,851,567,995,953,1,38,40,1,102,1,1,213,1,321,38,38,260,
  20,1,38,283,93,213,213,1,26,1,1,253,193,1,1,356,1,1,1,
  654,393,1,382,1,40,1,107,1,1,312,1,565,40,31,354,40,1,26,
  107,93,1,517,1,221,1,1,17,870,1,1,38,1,1,1,883,870,707,
  707,1,220,102,481,1,514,558,1,102,31,477,31,1,102,26,40,1,
  172,1,38,1,1,385,1,1,1,515,1,467,1,784,515,1,559,1,1,1,
  1,1,1,910,213,517,332,172,1,246,213,1,396,581,1,504,1,1,
  827,944,1,1,1013,1,1,1,1030,1,1,722,1,330,1,1,1,1,123,
  45,451,20,1,146,351,123,344,246,1,344,570,1,570,874,1,1,
  400,400,874,1,1,674,1,102,1,1,220,1,942,179,370,370,38,1,
  270,563,49,485,302,1,22,1,1,782,955,1,1,772,1,1,1,820,
  577,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,
  1,1,1,1,1,1,1,1,1,1,910,1,380,145,1,172,1,1,539,1,1,1,
  1,302,1,1,1,1,1,1,1,1013,944,1,1,1,1007,714,1,1,509,1,
  1,502,1,269,1,464,1,1,1,1,1,1,1,708,1,1,1,1,1,1,960,
  796,1,1,1,377,102,1,484,1,1,824,1,713,639,1,1,1,1,1,1,
  1,1,1,1,708,1,1,708,1,1,1,38,1,40,1,102,213,123,1,380,
  1,179,1,1,313,356,1,1,382,1,382,1,1,1,1,356,1,370,40,1,
  107,1,157,530,1,1,269,1,1,526,38,1,1,1,1,707,1,707,1,1,
  1,1,1,1,1,233,1,1,690,1,1,1,1,1032,961,1,1,1,1045,743,
  1,1,1,1,780,1,1,102,1,157,437,1,1,1,502,1,385,515,1,1,
  535,467,559,1,1,1,872,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,
  1,1,1,1,1,1,1,1,1,344,123,255,247,1,422,571,1,353,400,1,
  1,674,874,102,1,1,400,965,630,1,1,591,1,1,1,1,1,1,637,1,
  1,1,1005,963,1,1,1,1,1,1,487,313,1,1,661,1,1,684,1,1,1,
  684,394,1,1,1,1,394,1,213,1,344,1,1,541,839,1,1,637,1,
  1005,1,1,1,1,963,1,1,220,1,1,845,714,1,1,944,1,1,1,1020,
  1,1030,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,609,1,1,
  1,1,1,1,567,772,1,1,1060,851,955,1,1,570,1,1,1,821,1,1,
  1,1,1,1,728,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,595,
  1,1,1,1,843,595,1,1,1,1,1,45,1,252,400,1,1,1,1,1,1,1,
  1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,400,1,1,1,1,252,45,1,
  1,1,1,1,614,45,1,1,707,45,1,1,1,1,1,1,1,1,614,1,1,1,
  1,1,1,707,1,1,7,3,7,7,5,7,54,54,131,54,131,75,1,1,54,
  131,75,1,202,1,227,347,1,1,54,131,327,75,404,202,317,35,
  317,1,1,1,1,1,1,500,1,1,500,1,1,1,421,1,421,1,1,7,5,5,
  5,5,31,31,31,31,31,45,1,1,31,31,45,212,146,1,1,212,128,
  128,31,31,126,45,86,146,215,8,215,212,1,1,128,86,139,26,
  13,13,26,26,26,13,13,26,13,26,13,7,7,5,7,54,54,131,54,
  131,75,1,1,54,131,75,1,202,1,227,347,1,1,54,131,327,75,
  404,202,317,35,317,1,1,1,1,1,1,500,1,1,500,1,1,1,421,1,
  421,1,1,7,5,7,20,20,20,20,20,26,1,1,20,20,26,195,22,79,
  194,194,101,1,20,20,1,26,1,22,1,10,1,195,1,1,1,211,211,
  148,148,148,148,81,81,148,1,81,1,81,148,7,3,136,136,53,
  136,53,213,1,1,136,53,213,1,106,1,1,379,215,185,136,53,
  379,213,215,106,1,17,1,1,1,1,185,215,215,157,134,134,157,
  134,134,134,157,134,157,134,134,7,31,31,38,31,38,43,1,1,
  31,38,43,113,45,92,172,106,308,1,31,38,157,43,185,45,146,
  8,146,113,1,1,1,1,244,34,148,148,34,1,1,148,24,1,24,1,
  148,136,136,53,136,53,213,1,1,136,53,213,1,106,1,1,379,
  215,185,136,53,379,213,215,106,1,17,1,1,1,1,185,215,215,
  157,134,134,157,134,134,134,157,134,157,134,134,40,40,31,
  102,102,1,1,31,20,125,316,125,322,1,1,318,1,31,20,349,20,
  485,146,1,146,145,38,1,1,1,1,1,1,383,1,1,622,460,1,551,
  553,551,498,1,40,40,40,26,1,1,31,102,102,38,26,340,1,1,
  287,1,31,20,383,125,483,125,145,17,383,298,1,1,483,1,1,1,
  1,1,1,460,498,1,551,622,551,1,383,102,11,102,1,1,20,83,
  70,426,83,63,1,1,176,402,20,11,99,81,402,16,213,81,455,
  191,1,1,81,134,134,233,233,34,34,70,356,233,151,356,24,102,
  102,40,102,1,1,40,40,26,298,26,457,457,128,1,374,31,102,
  145,102,1,26,383,36,349,1,1,1,485,1,384,1,1,1,383,498,1,
  383,551,460,551,1,1,102,1,1,102,11,102,191,16,1,186,191,
  140,140,20,83,455,70,81,83,455,24,99,1,1,1,402,365,365,34,
  34,233,233,356,102,102,24,70,151,356,233,1,1,125,70,69,509,
  125,254,235,254,1,384,20,81,304,116,473,81,343,81,304,1,1,
  1,473,384,1,478,1,478,1,1,1,433,1,568,1,1,1,1,1,1,1,1,
  1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,
  1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,
  1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,40,102,1,
  125,1,347,348,1,1,40,40,1,26,374,26,349,17,318,318,1,1,1,
  545,485,383,1,383,1,1,1,1,551,498,1,553,383,102,1,83,148,
  1,346,176,402,102,11,213,102,402,16,99,24,1,177,1,1,102,
  102,81,233,233,102,20,102,356,233,151,356,81,81,20,1,125,1,
  436,1,1,1,125,70,343,69,1,125,304,22,1,81,1,1,1,491,1,1,
  478,433,524,1,1,1,1,1,1,631,439,1,1,508,1,1,1,318,177,
  504,81,1,34,318,387,302,1,1,1,1,705,1,599,1043,471,790,1,
  1,1,616,761,1,612,729,234,458,1,162,499,146,16,324,81,499,
  20,324,146,1,34,1,1,235,95,235,330,330,26,112,112,493,330,
  208,493,146,81,146,457,254,1,1,228,213,213,146,751,534,1,
  294,497,320,1,1,1,1,893,756,1,551,552,1,1,1,1,818,1010,
  833,715,1,923,1,348,128,285,285,901,235,186,144,1,1,1,1,1,
  1,1,1,876,1,1,896,663,1,726,695,1,1,628,1,1,347,128,430,
  246,1,297,690,430,332,95,1,1,928,1,1,1,666,1,1,1,748,1,
  938,831,1,610,1,140,287,140,705,1,140,22,815,422,1,1,1,1,
  1,1,1,268,1,1,1,1,1,1,410,376,1,1,1,483,81,1,1,131,131,
  1,1,1,615,1,1,1,1,1,1,363,1,1,1,1,1,1,1,1,1,1,40,1,
  102,1,125,318,146,1,307,1,1,1,485,545,1,383,1,383,1,553,
  383,1,1,1,622,1,213,102,140,83,1,81,237,1,1,1,402,365,
  365,20,102,233,233,356,81,20,81,102,151,356,233,285,1,1,1,
  430,1,660,1,1,1,1,900,849,765,802,733,1027,1,1,1,1,1,855,
  716,1,125,1,81,338,477,1,1,543,1,477,524,433,1,439,1,631,
  439,1,1,611,1,524,131,1,1,1,823,1,1,1,1,1,420,352,1,1,
  1,459,375,1,1,1,1,1,1,146,335,1,1,1,499,489,489,112,26,
  330,330,493,81,146,146,112,208,493,330,1,617,1,1,1,785,918,
  1,765,849,665,1,734,959,1,1,1,734,755,1048,532,308,1,1,
  468,468,1,477,477,477,1,1,1,477,1,1,1,525,1,318,1,1,1,
  972,1,802,733,849,765,1,734,665,1,1057,1,959,1,1,1,834,1,
  692,471,1,1043,599,761,1,471,902,1,616,1,1,1,1,1,1,1,1,
  1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,
  102,235,1,1,420,352,1,1,1,1,1,1,1,363,134,1,1,1,1,390,
  1,1,1,1,1,1,363,1,1,1,1,1,1,1,1,1,1,363,1,34,1,1,1,
  70,70,52,112,1,1,1,1,1,1,75,31,52,102,1,1,1,34,102,1,1,
  1,1,52,75,31,112,1,1,1,1,52,70,70,1,1,1,1,57,70,45,102,
  81,70,1,1,1,146,45,1,1,1,57,1,1,1,1,1,1,81,146,102,7,
  7,7,7,5,38,38,54,38,54,136,1,538,38,54,136,1,239,1,1,
  379,1,532,38,54,379,136,1,239,538,54,538,1,1,538,532,1,1,
  296,1,1,296,1,1,1,296,1,296,1,1,7,7,5,7,54,54,131,54,
  131,75,1,1,54,131,75,1,202,1,227,347,1,1,54,131,327,75,
  404,202,317,35,317,1,1,1,1,1,1,500,1,1,500,1,1,1,421,1,
  421,1,1,3,7,7,38,38,54,38,54,136,379,1,38,54,136,1,239,
  1,538,1,1,1,38,54,1,136,1,239,1,54,1,1,379,1,1,532,1,1,
  296,296,1,1,1,296,1,1,1,1,296,7,7,38,38,54,38,54,136,
  379,1,38,54,136,1,239,1,538,1,1,1,38,54,1,136,1,239,1,
  54,1,1,379,1,1,532,1,1,296,296,1,1,1,296,1,1,1,1,296,5,
  20,20,26,20,26,13,50,123,20,26,13,1,26,1,150,128,1,1,20,
  26,50,13,1,26,150,10,150,1,50,123,1,1,1,518,518,518,518,
  1,1,518,518,1,518,1,518,38,38,70,38,70,31,1,110,38,70,31,
  1,38,1,241,113,1,308,38,70,241,31,185,38,113,10,113,1,1,
  110,308,244,1,488,306,306,488,1,1,306,141,1,141,1,306,38,
  38,31,40,40,380,1,11,31,102,1,102,1,179,1,1,1,31,31,380,
  20,313,123,179,123,26,1,213,213,377,509,1,1,356,382,1,1,1,
  1,382,1,356,1,1,38,38,38,20,213,213,31,40,40,1,38,1,1,
  260,1,1,11,31,213,102,253,102,26,93,191,1,1,283,321,1,1,
  1,382,1,654,1,1,1,356,1,193,1,393,40,31,40,1,530,31,107,
  31,1,107,1,370,370,1,1,31,31,145,157,526,45,269,157,221,1,
  517,1,102,1,1,1,1,707,707,1,1,1,1,1,38,1,1,38,40,1,283,
  38,38,20,1,38,1,1,128,1,509,31,40,99,40,253,38,191,11,
  351,1,284,158,465,1,1,654,1,1,537,1,1,393,193,1,393,1,
  193,40,517,1,40,31,40,1,26,1,1,354,1,1,31,107,312,31,17,
  107,221,93,38,1,1,107,565,1,1,707,707,1,883,1,1,1,38,1,
  870,1,870,1,437,102,31,31,1,102,1,38,1,1,502,20,157,172,
  31,385,20,1,157,38,1,172,1,484,502,1,872,1,559,1,1,1,467,
  535,1,515,1,1,213,284,1,172,1,247,1,302,690,1,1,99,399,1,
  220,845,246,220,313,548,1,581,1,639,1,1,1030,1,1,1020,1,1,
  1,944,1,714,1,1,158,107,1,1,255,1,485,1,1,1,63,283,1,26,
  541,344,344,487,49,1,396,591,713,1,1,1,963,1005,1,1,1,1,
  637,1,839,1,1,38,40,1,102,1,351,191,1,377,38,38,284,20,
  313,38,351,93,1,1,99,63,1,465,1,537,1,393,586,1,1,193,
  393,1,1,1,356,40,1,107,1,1,333,1,687,40,31,399,40,526,26,
  38,93,1,1,399,283,1,1,1,883,1,1,216,1,1,870,870,1,1,1,
  38,1,102,1,481,477,1,1,102,31,514,31,385,102,38,40,220,1,
  220,26,1,558,1,1,559,467,784,1,1,1,515,1,1,1,515,1,1,1,
  1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,
  1,1,1,1,1,1,270,1,1,571,123,45,1,20,353,146,422,123,344,
  1,246,344,1,269,1,965,630,102,1,1,1,874,674,1,400,1,400,
  1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,
  1,1,1,1,1,1,370,1,1,191,38,220,220,782,330,22,49,1,1,1,
  563,942,1,1,1,1,1,820,1,1,1,772,1,955,1,577,1,1,268,128,
  1,120,1032,451,1,1,471,1,332,233,1,1,1,1,780,743,1,1,1,
  1045,1,1,961,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,
  1,1,1,1,1,1,1,1,1,1,1,321,102,1,572,570,570,609,661,920,
  1,1,1,1,1,1,1,728,1,1,1,1,1,821,1,1,1,1,38,1,40,1,
  102,1,123,1,1,213,223,1,321,1,586,393,193,741,1,1,356,1,
  1,1,1,382,1,40,1,107,1,157,1,1,312,305,1,565,1,216,1,
  870,864,1,1,38,1,1,1,1,1,220,1,1,302,1,1,1,1,539,1,910,
  1,1007,714,944,1,1,1,1013,1,1,1,1,1,1,102,220,157,481,1,
  514,437,558,1,1,784,467,1,747,1,1,515,1,1,467,1,535,570,
  1,1,1,1,827,1,1,1,1,843,595,1,1,1,1,595,1,1,1,1,1,344,
  123,330,1,1,351,1,1,1,1,102,874,1,1,1,400,400,1,874,1,
  674,1,1,1,504,1,1,1,1,851,955,772,1,1,1,567,1,1,1,1,
  1060,455,1,1,1,1,464,1,1,394,394,1,1,1,684,1,1,684,1,1,
  1,220,344,1,942,1,995,953,955,851,1,1,772,1,1,1,1,567,1,
  1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,213,910,1,1,1,722,1,
  1030,1,1,1,1013,1,944,1,1,824,1,1,1,1,963,1,1,1,1005,1,
  1,637,1,1,1,1,1,1,708,1,1,1,1,1,1,1,1,708,1,1,1,1,
  960,1,1,1,1,1,708,1,796,1,1,1,1,1,1,1,1,1,1,1,1,707,
  1,1,1,1,614,252,1,1,1,1,1,1,1,1,45,45,1,1,1,1,707,1,
  1,1,1,1,45,1,45,1,1,1,1,1,252,1,614,1,1,1,1,1,1,1,1,
  1,1,1,1,1,400,1,1,1,1,1,1,1,1,1,1,1,1,400,1,7,5,7,7,
  54,54,131,54,131,75,227,1,54,131,75,347,202,1,1,1,1,1,54,
  131,319,75,1,202,258,35,258,347,227,1,1,404,1,1,1,1,1,
  500,500,1,296,500,296,500,1,7,3,7,54,54,131,54,131,75,347,
  258,54,131,75,227,202,1,1,319,404,1,54,131,1,75,1,202,1,
  35,1,227,347,258,1,1,1,296,500,500,296,1,1,500,1,1,1,1,
  500,7,5,31,31,31,31,31,45,212,215,31,31,45,1,146,1,1,126,
  86,128,31,31,212,45,139,146,1,8,1,1,212,215,128,128,86,13,
  26,26,13,13,13,26,26,13,26,13,26,7,54,54,131,54,131,75,
  347,258,54,131,75,227,202,1,1,319,404,1,54,131,1,75,1,202,
  1,35,1,227,347,258,1,1,1,296,500,500,296,1,1,500,1,1,1,
  1,500,26,26,20,26,20,26,1,194,26,20,26,1,17,1,1,195,86,
  34,26,20,195,26,86,17,194,8,194,1,1,194,34,86,86,81,102,
  102,81,102,102,102,81,102,81,102,102,40,40,31,102,102,347,
  1,31,20,125,348,125,1,1,1,1,1,31,20,307,20,1,146,228,146,
  107,128,457,457,318,485,1,1,1,383,1,1,383,1,553,383,622,1,
  1,40,40,40,26,457,457,31,102,102,128,26,1,1,298,374,1,31,
  20,316,125,384,125,107,17,322,1,1,340,318,1,545,1,383,1,
  498,383,1,1,622,1,460,1,1,102,11,102,1,148,20,83,70,346,
  83,1,1,1,402,176,20,11,26,81,365,16,158,81,346,191,186,1,
  17,102,102,356,233,102,102,20,233,233,356,233,70,34,34,40,
  102,1,340,40,40,26,1,26,1,1,38,1,287,31,102,38,102,485,
  26,322,36,340,1,1,322,287,1,485,498,1,1,460,1,1,1,460,
  383,498,1,383,102,186,1,102,11,102,191,16,1,1,191,140,140,
  20,83,426,70,81,83,346,24,63,1,1,63,176,402,365,102,102,
  233,356,233,34,34,70,20,356,233,233,436,1,125,70,69,1,125,
  1,1,1,1,1,20,81,54,116,1,81,536,81,1,254,235,254,419,543,
  1,1,1,433,1,524,1,478,568,439,1,478,1,457,1,1,235,1,458,
  1,1,508,1,923,128,191,95,254,837,139,254,391,1,1,652,1,
  578,1,1,663,1,1,896,1,1,876,748,666,610,1,836,322,63,254,
  254,234,1,1,1,1,1,107,346,547,1,773,423,423,1,132,1,233,
  520,667,1,1,1,1,551,1,552,756,1,1,492,685,781,1,40,102,1,
  125,1,1,316,1,318,40,40,298,26,545,26,340,17,457,457,128,
  107,1,374,384,460,1,1,622,1,1,383,498,1,1,383,1,102,1,83,
  1,1,426,402,176,102,11,191,102,365,16,63,24,1,186,191,346,
  140,140,81,356,233,34,70,34,233,233,356,233,102,102,20,254,
  125,1,1,509,384,1,125,70,509,69,477,125,1,22,254,235,254,
  1,1,384,477,1,433,478,1,1,478,1,1,524,1,433,524,1,1,1,1,
  1,1,457,186,1,235,1,139,457,1,233,652,150,1,1,670,915,748,
  1,1,1,1,1,666,663,876,1,628,866,1,1,1,499,162,146,16,252,
  81,489,20,200,146,423,139,139,423,205,95,205,493,330,26,112,
  112,330,330,493,330,112,26,112,1,1,1,1,1,1,1,1,1,1,1,1,
  1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,
  1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,
  1,307,26,163,54,692,252,563,163,547,95,1,320,1,786,1,1,1,
  471,1,790,599,1043,720,1,888,471,1,140,485,102,786,543,102,
  95,895,1,1,1,670,834,1,1,1,1,1,1,1,1,1,363,390,375,1,1,
  1,318,17,1,419,205,205,634,422,667,578,1,653,1,1,1,1,1,1,
  1,1,268,1,418,1,1,1,1,40,1,102,1,125,457,146,1,347,348,
  228,1,1,1,622,1,383,553,1,383,1,1,1,1,1,383,1,102,102,
  83,1,81,148,1,346,158,176,402,365,70,34,233,356,233,102,20,
  102,34,356,233,233,1,1,1,320,163,1,508,1,563,1,1,692,888,
  471,599,720,1043,471,790,1,1,1,1,1,1,125,254,81,1,436,1,
  536,1,1,1,1,478,1,568,478,433,524,1,1,1,1,439,205,1,1,1,
  1,915,1,1,1,1,459,420,1,1,1,379,363,1,1,1,1,1,423,146,
  234,458,1,200,162,499,489,112,26,330,493,330,26,112,112,112,
  493,330,330,1,520,1,1,1,653,834,1,1,756,552,1,551,1,492,
  1,1,833,715,979,1,391,1,1,422,1,1,1,477,477,1,477,477,
  477,1,1,1,477,1,457,254,423,1,1,773,685,781,756,1,1,551,
  552,1,1,1,1,492,1,254,923,1,837,610,1,1,748,876,1,1,896,
  836,663,1,666,457,1,1,1,1,628,1,663,666,1,1,1,866,748,1,
  876,634,895,1,833,715,1,1,492,552,551,1,979,1,756,1,140,
  205,1,1,268,418,1,1,1,1,1,1,1,1,102,1,1,1,390,363,1,1,
  1,1,1,1,375,1,1,379,1,363,1,1,1,1,459,420,1,102,1,1,1,
  70,75,38,112,1,1,1,1,1,1,70,31,70,34,1,1,1,102,34,1,1,
  1,1,70,70,31,112,1,1,1,1,38,70,75,1,1,1,1,75,31,31,34,
  102,31,1,1,1,112,31,1,1,1,75,1,1,1,1,1,1,102,112,34,7,
  7,3,131,131,143,131,143,59,1,251,131,143,59,1,131,271,319,
  258,1,1,131,143,1,59,1,131,1,35,1,1,1,251,1,1,404,707,1,
  1,707,626,626,1,1,626,1,626,1,5,7,31,31,38,31,38,43,113,
  146,31,38,43,1,45,172,92,157,1,1,31,38,106,43,244,45,1,8,
  1,1,113,146,1,308,185,24,1,1,24,148,148,1,34,148,34,148,
  1,7,31,31,45,31,45,16,86,128,31,45,16,86,31,128,128,139,
  1,1,31,45,139,16,1,31,128,8,128,86,86,128,1,1,1,518,1,1,
  518,1,1,1,518,1,518,1,1,131,131,143,131,143,59,1,251,131,
  143,59,1,131,271,319,258,1,1,131,143,1,59,1,131,1,35,1,1,
  1,251,1,1,404,707,1,1,707,626,626,1,1,626,1,626,1,107,
  107,31,75,75,292,1,13,89,70,546,70,1,292,1,1,1,31,89,292,
  10,423,59,292,59,102,102,134,134,1,423,1,1,1,1,1,826,830,
  1,470,646,547,623,1,107,107,107,10,134,134,31,75,75,102,31,
  186,1,245,1,1,13,89,435,70,598,70,102,36,546,245,1,186,1,
  1,266,1,1,1,623,830,623,1,547,826,543,1,1,75,69,75,1,255,
  89,172,13,297,172,81,200,200,1,1,89,34,134,57,1,89,573,57,
  297,1,297,1,1,570,570,783,1,1,1,495,1,1,699,991,66,1,1,
  107,75,1,186,107,107,10,245,31,134,134,102,1,1,31,75,102,
  75,579,31,546,13,292,1,292,1,1,1,366,623,1,1,830,623,1,1,
  543,830,412,1,1,75,297,1,75,69,75,1,34,1,297,1,1,1,89,
  172,281,13,269,172,297,76,134,200,200,81,1,523,1,1,1,1,1,
  1,1,1,66,495,891,783,1,274,68,70,13,8,337,70,174,19,174,
  1,1,10,57,19,8,310,10,274,57,19,174,19,174,1,310,1,542,1,
  1,1,542,1,1,395,569,221,542,1,134,292,200,19,200,292,245,
  203,324,1,1,102,1,128,65,1,102,65,1,328,1,328,1,1,1,711,
  797,1,1,1042,736,1,1,1,1,618,1,1,1,81,174,174,1,1,245,1,
  1,1,26,255,313,13,584,134,134,120,95,241,235,126,1,779,1,
  1,1,1,1,1,1,1,445,752,1047,926,1,107,75,1,70,1,292,546,
  1,1,107,107,292,10,443,31,292,36,200,200,102,26,1,443,366,
  830,1,1,826,1,1,1,412,623,1,646,1,75,200,172,255,1,297,1,
  1,75,69,573,75,1,34,134,76,1,297,1,255,1,1,40,1,1,1,495,
  1,783,1,891,1,794,794,1,174,70,68,274,337,1,1,70,13,337,
  8,350,70,19,8,65,19,65,13,1,350,246,1,1,1,542,1,542,1,
  221,542,1,569,1,1,1,324,1,1,1,200,297,328,19,648,102,200,
  288,235,328,150,313,1,603,1,1,1,1,1052,1,1,1,673,916,1,
  865,1,1,292,1,1,1,59,89,245,10,507,31,292,59,134,102,102,
  134,1,102,1,1,1,1,1,1,1,1,401,759,1,216,1,134,174,1,1,
  186,194,194,13,584,186,245,32,297,146,297,22,1,779,1,1,1,
  1,1,1,1,1,445,752,1,926,1,200,1,1,1,128,116,116,1,546,
  128,144,1,328,1,297,1,1,392,1,1,1,736,1042,797,1,1050,
  1062,987,1,1,1,1,245,128,189,19,727,292,324,189,324,150,
  328,146,1,587,1,1,1,1,1,1052,1,1,742,812,939,757,1,1,1,
  1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,
  1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,
  1,1,1,107,1,75,1,70,200,59,1,245,1,186,1,579,266,826,1,
  1,646,1,646,1,1,1,1,826,1,194,75,1,172,1,57,297,1,297,
  593,1,1,1,495,1,1,991,783,794,1,794,1,397,991,1,116,1,1,
  203,189,1,328,1,241,1,1059,549,1,1,1,916,1061,757,1,1,1,
  1,1,1,1,70,65,57,274,337,274,68,1,1,246,542,1,1,569,542,
  569,1,1,1,221,1,1,1,1,1,1,962,1,1,1,1,1,1,1,1,1,1,
  960,1,1,1,1,1,1,134,59,1,292,1,186,1,754,192,1,1,1,759,
  1,216,1,1,1,1,982,1,1,328,1,328,1,1,1,1,736,1,1,1,797,
  1037,1,1,1,746,1058,1,288,1,288,1,1,1,1,645,1,1,1,645,1,
  1,1,1,352,1,1,200,65,134,1,1,392,1,1,1,736,1,797,1,1,
  1066,1,1037,1,200,174,1,1,1044,757,1,1,1,916,1,1,1,1054,
  673,1,1,134,1,1,602,1,1,1,797,1,736,1,1,1051,1,1,1,1,
  966,1,926,1,1,1,752,1,1,1,1,445,1,1,1,1,1,1,1,1,1,1,
  1,1,1,1,1,1,570,1,1,1,954,1,1,1,1,1,821,1,1,1,1,1,1,
  899,1,1,1,1,595,886,1,1,1,1,1,1,1,252,1,1,1,1,1,1,1,
  1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,252,1,1,
  1,1,1,1,647,406,1,1,794,406,1,1,1,1,1,1,1,1,647,1,1,1,
  1,1,1,794,1,1,7,7,38,38,54,38,54,136,379,1,38,54,136,1,
  239,1,538,1,1,1,38,54,1,136,1,239,1,54,1,1,379,1,1,532,
  1,1,296,296,1,1,1,296,1,1,1,1,296,7,54,54,131,54,131,75,
  347,258,54,131,75,227,202,1,1,319,404,1,54,131,1,75,1,202,
  1,35,1,227,347,258,1,1,1,296,500,500,296,1,1,500,1,1,1,
  1,500,131,131,143,131,143,59,1,251,131,143,59,1,131,271,
  319,258,1,1,131,143,1,59,1,131,1,35,1,1,1,251,1,1,404,
  707,1,1,707,626,626,1,1,626,1,626,1,38,38,31,40,40,268,1,
  11,31,102,191,102,213,223,1,253,1,31,31,1,20,1,123,1,123,
  1,26,1,1,1,465,1,1,586,741,1,193,356,1,1,393,1,382,1,38,
  38,38,20,1,1,31,40,40,26,38,380,1,223,313,1,11,31,1,102,
  1,102,1,93,1,179,1,268,1,1,465,1,741,1,741,356,382,1,1,
  193,1,1,654,40,31,40,1,447,31,107,31,221,107,145,160,160,
  526,1,31,31,1,157,1,45,1,157,1,269,447,1,1,1,1,1,864,1,
  1,38,1,1,1,870,1,707,707,38,40,1,268,38,38,20,179,38,213,
  213,63,1,1,31,40,1,40,1,38,1,11,1,1,260,191,1,1,321,741,
  1,1,586,382,1,654,1,356,1,1,537,40,447,1,40,31,40,269,26,
  1,530,305,1,1,31,107,1,31,1,107,1,93,1,370,370,128,1,1,
  565,1,1,1,864,1,707,707,1,38,1,1,883,514,514,102,31,31,1,
  102,224,26,224,1,1,20,157,1,31,1,20,1,157,1,246,120,246,
  1,558,1,696,1,696,1,535,1,872,1,515,1,559,1,1,260,370,
  120,370,324,332,233,607,1,1,128,354,1,1,1,198,1,295,1,1,
  725,1,1,1,1,1033,1,1,1035,743,1045,1040,1,780,1,1,1024,
  191,128,246,246,358,1,1,1,793,1,128,333,1,120,1,1,1,1,1,
  471,359,725,1,1021,1,1,1035,1033,1,852,961,1,1,732,1,1,
  1063,38,40,1,102,1,283,158,1,1,38,38,1,20,1,38,1,93,1,1,
  128,128,1,509,377,586,1,654,537,1,1,537,1,382,1,393,586,
  40,370,107,517,1,283,1,1,40,31,1,40,1,26,1,93,1,1,354,
  333,1,1,102,864,1,707,216,707,1,883,1,1,1,1,216,246,102,
  1,437,437,381,1,102,31,1,31,1,102,1,40,1,38,1,120,1,502,
  484,1,696,872,747,1,559,1,1,535,1,467,784,1,1,563,1,1,1,
  1,1,1,38,1,54,1,481,1,1,95,1,1,682,927,995,1,820,1,1,1,
  996,1,1,1,577,1,376,279,1,1,1,123,45,1,20,1,146,1,123,1,
  54,198,1,1,269,1,1,1,450,735,735,630,965,1,674,1,102,1,
  213,224,1,1,284,1,1,172,1,247,1,194,1,302,690,186,1,1,
  969,1007,1,1030,1000,1,1,1020,1,722,1,1,989,160,935,1,158,
  107,1,1,1,255,1,157,1,485,1,607,1,1,744,1,1,1,999,963,
  1005,1,1,1,1,1,929,641,1,283,107,1,26,1,351,1,1,1,49,1,
  233,1,744,1,1,973,999,1,869,839,1,1,925,1,637,1,1,313,1,
  1,381,1,40,1,263,1,1,561,1,1,1,1,843,1,1,1,1,1,886,1,
  569,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,
  1,1,1,1,1,1,1,38,1,40,1,102,1,123,1,351,191,260,1,377,
  509,537,654,537,654,1,393,586,1,1,1,193,741,1,40,1,107,1,
  157,1,1,333,354,1,687,687,216,707,883,883,1,1,216,1,707,1,
  870,864,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,
  1,1,1,102,1,157,1,481,477,477,1,1,572,747,872,1,784,559,
  467,784,1,1,1,1,747,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,
  1,1,1,1,1,1,1,123,1,270,1,451,1,571,1,735,450,965,1,630,
  102,1,1,735,1,874,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,
  1,1,1,1,1,269,546,1,1,413,1,986,655,738,1,684,394,1,1,1,
  1,394,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,370,246,
  1,1,920,851,1,1,995,1,1,820,1,880,1,1,996,1,1,1,1021,1,
  998,1,1033,780,743,1,1,964,1,1045,1040,1,1,1,1040,1001,
  1035,1,732,852,1033,1,1056,1,961,1,1,1,1,1,1,1,1,1,1,1,
  1,1,1,1,1,1,1,1,954,728,1,1,1,1,1,1,1,1,1,954,1,821,
  1,1,1,1,1,821,1,1,1,1,1,252,1,1,735,1,1,1,1,1,1,614,
  406,1,707,1,1,1,1,707,1,1,1,1,1,614,406,735,1,1,1,1,1,
  252,1,1,1,1,1,1,45,252,707,1,45,1,1,1,1,252,1,1,1,1,1,
  1,1,1,1,1,1,1,707,7,54,54,131,54,131,75,1,317,54,131,75,
  1,202,227,1,327,1,1,54,131,347,75,1,202,1,35,1,1,1,317,
  1,1,404,421,1,1,421,1,1,1,500,1,500,1,1,136,136,53,136,
  53,213,1,1,136,53,213,1,106,1,1,379,215,185,136,53,379,
  213,215,106,1,17,1,1,1,1,185,215,215,157,134,134,157,134,
  134,134,157,134,157,134,134,40,40,31,102,102,307,1,31,20,
  125,1,125,348,228,1,485,1,31,20,347,20,545,146,1,146,1,1,
  318,318,483,287,1,1,622,553,1,383,1,1,383,1,1,383,1,40,
  40,40,26,318,318,31,102,102,1,26,347,1,1,545,1,31,20,348,
  125,485,125,1,17,1,1,1,349,1,1,374,1,553,1,551,1,383,1,
  1,383,383,1,498,102,11,102,1,237,20,83,70,1,83,128,213,
  213,365,402,20,11,128,81,365,16,1,81,1,1,177,1,81,140,140,
  151,356,81,81,20,233,356,233,233,20,102,102,40,102,1,349,
  40,40,26,1,26,457,457,145,1,485,31,102,128,102,384,26,1,
  36,1,1,298,383,374,1,1,551,1,1,551,383,1,498,383,1,1,1,
  460,102,177,1,102,11,102,1,16,1,148,213,102,102,20,83,346,
  70,81,83,1,24,1,1,1,99,402,176,402,81,81,356,151,233,102,
  102,20,20,233,233,356,477,338,125,70,69,1,125,285,146,285,
  1,543,20,81,246,116,477,81,1,81,1,1,81,1,1,1,1,611,1,
  631,1,439,1,1,439,524,524,433,1,318,298,1,81,1,1,1,320,
  660,1,834,38,191,95,1,692,34,1,308,1,1,1,1,615,1,823,616,
  1,1,902,471,1043,761,599,1,471,1,1,383,99,1,1,335,1,497,
  1,972,1,145,455,332,304,1,1,1,532,1,1,302,617,1,1,1,1,
  959,734,1,665,849,1,765,1,802,733,1057,40,102,1,125,1,340,
  383,1,483,40,40,1,26,1,26,1,17,1,1,38,145,1,287,483,551,
  1,498,551,1,1,460,1,383,1,1,622,102,1,83,186,1,455,365,
  402,102,11,191,102,365,16,1,24,1,1,191,455,140,140,81,151,
  356,102,24,102,233,356,233,233,34,34,70,1,125,436,1,343,
  491,1,125,70,1,69,1,125,1,22,1,1,1,304,1,1,1,1,631,1,1,
  1,433,1,524,439,1,478,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,
  1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,235,534,1,489,
  499,146,16,297,81,489,20,1,146,1,1,34,1,131,22,131,208,
  493,81,146,146,330,493,330,330,112,26,112,457,285,1,1,1,1,
  1,235,1,458,1,144,1,1,508,186,1,923,901,1,1,663,726,1,1,
  896,1,628,1,876,695,213,1,1,322,63,254,254,893,234,1,294,
  1,1,1,1,1,1,751,1010,833,1,1,1,551,1,552,715,756,1,818,
  1,1,349,99,430,304,900,324,1,430,1,1,504,1,1,705,1,1,855,
  1,1,1,802,1027,733,716,849,765,1,102,1,134,1,384,134,95,1,
  468,1,1,705,918,1,1,1,1,1,1,1,1,1,390,1,363,1,1,1,1,
  81,928,473,235,235,1,468,1,1,1,785,1,1,1,1,1,1,1,1,420,
  1,352,363,1,1,1,40,1,102,1,125,1,146,1,1,316,1,1,318,
  485,551,498,460,551,1,1,622,1,1,1,383,553,1,102,134,83,1,
  81,1,1,426,213,402,176,402,24,102,356,151,233,34,70,34,102,
  233,233,356,254,1,1,1,430,1,1,1,690,1,1,1,1,610,748,938,
  1,1,1,1,1,1,666,831,1,125,1,81,1,1,509,343,384,1,473,1,
  1,1,1,433,478,1,1,1,478,1,568,235,1,1,1,1,1,1,1,1,1,1,
  363,1,1,1,1,1,1,1,1,1,1,1,146,1,1,1,324,499,162,499,
  146,81,493,208,330,26,112,112,146,330,330,493,1,1,1,1,1,1,
  1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,387,1,1,422,1,1,525,
  1,1,477,477,1,1,1,477,477,1,1,1,1,1,1,1,1,1,1,1,1,1,
  1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,318,
  1,1,1,1,612,1,616,1,471,1,790,729,599,1043,761,1,815,1,
  734,755,959,1,1,665,734,1,1048,765,849,1,140,131,1,1,1,1,
  1,1,1,1,1,1,363,1,140,1,1,1,410,1,1,1,1,1,268,1,376,1,
  1,459,1,375,1,1,1,1,420,352,1,81,1,1,1,52,57,52,146,1,
  1,1,1,1,1,75,70,70,102,1,1,1,81,102,1,1,1,1,70,75,70,
  146,1,1,1,1,52,52,57,1,1,1,1,70,31,45,102,34,31,1,1,1,
  112,45,1,1,1,70,1,1,1,1,1,1,34,112,102,131,131,143,131,
  143,59,1,1,131,143,59,1,131,319,271,1,1,1,131,143,258,59,
  404,131,251,35,251,1,1,1,1,1,1,1,626,626,1,1,1,626,707,
  1,707,1,626,107,107,31,75,75,245,1,13,89,70,1,70,1,186,1,
  579,1,31,89,245,10,266,59,186,59,26,102,200,200,1,1,1,1,
  826,646,1,1,1,1,646,1,826,1,1,107,107,107,10,200,200,31,
  75,75,102,31,292,1,292,443,1,13,89,546,70,366,70,26,36,1,
  292,1,292,1,1,443,1,646,1,412,1,1,1,826,1,830,1,623,75,
  69,75,1,297,89,172,13,297,172,128,194,194,1,1,89,34,128,
  57,1,89,593,57,255,1,297,1,1,1,1,397,991,794,794,1,1,783,
  991,1,495,1,1,107,75,1,292,107,107,10,292,31,134,134,102,
  1,1,31,75,102,75,366,31,1,13,186,1,245,546,1,1,579,412,1,
  1,543,1,1,623,830,1,623,1,830,75,297,1,75,69,75,1,34,1,
  255,573,1,1,89,172,297,13,40,172,255,76,81,200,200,134,1,
  1,1,794,794,783,891,1,1,1,495,1,1,1,1,337,274,70,13,8,
  274,70,116,13,116,1,1,10,57,19,8,246,10,68,57,13,65,19,
  65,1,1,1,221,1,569,1,1,1,542,569,1,542,1,1,200,245,200,
  19,200,292,328,146,328,1,1,102,1,150,174,1044,102,174,1,
  241,1,328,1,1,1,962,673,1,1,1,1,1,916,1,1,757,1,1054,
  546,134,65,65,1,1,297,1,1,1,102,297,324,19,392,134,134,
  288,95,328,235,328,1,1,1,1,1037,797,1,1,1,1,736,1,1,1,
  1066,107,75,1,70,1,186,435,1,1,107,107,245,10,266,31,186,
  36,134,134,102,102,1,1,598,543,1,623,547,1,1,830,623,1,1,
  1,826,75,200,172,297,1,281,523,1,75,69,1,75,1,34,81,76,1,
  297,1,297,1,1,269,891,783,1,66,1,1,1,1,1,1,1,495,65,70,
  274,68,337,350,1,70,13,337,8,246,70,13,8,174,19,174,19,1,
  1,350,1,569,542,221,1,1,1,542,1,1,1,542,1,1,297,1,1,1,
  134,297,328,19,602,102,134,288,235,328,150,328,1,1,1,1,1,
  736,1,1,1,1,797,1,1,1,1051,546,186,1,754,1,59,89,292,10,
  192,31,186,59,134,102,102,134,1,1,1,1,982,216,1,1,1,1,
  759,1,1,1,1,134,116,1,1,292,200,200,19,392,292,297,144,
  245,203,324,128,1,1,1,987,1,797,1050,1,1,1042,736,1,1,1,
  1062,194,779,1,1,81,174,174,1,1,22,32,1,245,1,245,1,1,
  584,1,926,1,445,1,1,1,1,1,1,1,752,1059,1,292,134,189,19,
  549,245,241,189,313,128,328,203,1,1,1,1,1,757,1,1,1,1061,
  916,1,1,1,1,1,423,570,587,310,570,102,966,1,779,1,603,1,
  1,1,1,821,1,1,1,1,1,1,954,1,1,1,1,1,1,1,1,1,1,1,1,1,
  1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,107,1,75,1,70,
  134,59,1,292,546,292,1,1,423,547,623,830,470,1,1,826,1,1,
  1,1,646,200,75,570,172,1,57,255,1,297,573,1,1,1,66,1,1,
  699,1,1,495,1,1,783,1,991,174,1,1,146,189,1,324,1,324,1,
  1,727,939,757,1,742,1,1,1052,1,1,1,1,812,1,70,174,57,68,
  274,337,274,1,1,310,221,542,1,395,1,1,542,1,1,542,1,569,
  1,1,1,1,711,1,1,1,1,1,595,886,1,1,1,1,899,1,1,1,1,1,
  134,59,1,292,1,292,1,1,507,1,216,1,401,1,1,1,1,1,1,1,
  759,1,126,1,313,1,1,1,1,445,1,1,1,1,1,752,1,1,926,1,1,
  120,1,288,1,1,1,1,352,1,1,1,1,1,645,1,1,645,1,1,134,
  174,134,1,1,584,1047,926,1,445,1,1,1,1,1,1,1,752,200,65,
  1,1,1,618,1,1,1,1,1,736,1042,1,797,1,1,200,1,1,648,1,
  865,1,673,1,1,1,1052,1,1,1,916,1,1,1,746,1058,1037,1,1,
  1,797,1,1,736,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,
  1,1,1,1,1,1,1,1,1,1,1,1,960,1,1,1,1,1,1,1,1,1,794,1,
  1,1,1,647,252,1,1,1,1,1,1,1,1,406,1,1,1,1,1,794,1,1,
  1,1,1,1,1,406,1,1,1,1,1,252,1,647,1,1,1,1,1,1,1,1,1,
  1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,200,200,116,
  186,186,1,1,22,81,102,1,102,376,1,1,491,1,116,81,308,20,
  473,81,373,81,198,1,1,1,1,477,1,1,1,590,1,1,701,1,576,
  590,469,1,1,200,200,200,20,1,1,116,186,186,1,70,379,1,383,
  1,1,22,81,330,102,384,102,198,104,358,1,1,390,1,1,1,1,
  590,1,551,701,1,1,469,1,498,1,1,186,20,186,1,287,81,147,
  26,1,147,110,364,364,483,148,81,20,34,146,502,16,276,146,1,
  1,1,1,22,235,235,318,494,205,205,95,494,494,318,494,22,131,
  131,200,186,1,390,200,200,20,1,70,360,360,54,1,419,116,186,
  54,186,543,70,358,32,390,1,1,358,419,1,543,551,1,1,498,1,
  1,1,498,701,551,1,701,186,1,1,186,20,186,1,16,1,310,212,
  218,218,81,147,395,26,95,147,1,32,139,1,1,139,148,483,502,
  205,205,494,318,494,131,131,22,95,318,494,494,1,514,102,26,
  31,1,102,220,172,220,1,287,20,146,38,31,483,20,477,146,120,
  1,1,1,318,1,1,1,1,511,1,482,1,669,434,717,1,669,1,1,1,
  1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,
  1,1,1,1,1,1,1,1,1,1,1,358,139,1,1,191,1,690,1,1,1,198,
  1,1,120,670,1,1,475,150,1,1,652,1,936,1,1,948,621,1,1,1,
  1,659,693,671,970,1031,200,186,1,102,1,379,330,1,1,200,200,
  383,20,1,70,390,104,1,1,1,198,1,1,384,498,1,1,469,1,1,
  701,551,1,1,590,1,186,1,147,310,1,395,483,148,186,20,212,
  186,502,16,139,32,1,1,1,1,218,218,95,318,494,131,22,131,
  494,494,318,494,205,205,95,1,102,1,1,1,545,1,102,26,1,31,
  1,102,120,36,1,1,1,120,1,545,1,1,511,669,1,1,669,1,1,
  482,1,511,482,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,
  1,1,1,1,1,1,1,1,1,1,1,1,1,1,213,455,1,365,176,81,16,
  177,20,402,11,186,81,1,1,1,1,140,81,140,253,419,81,140,
  140,419,419,253,419,140,81,140,360,220,1,770,1,364,364,172,
  1,455,332,1,690,1,1,186,1,900,913,1,1,629,677,1,1,1003,1,
  1,1,809,817,364,900,1,376,110,220,220,913,213,186,1,1,1,1,
  332,770,1,1,1,809,1,1,1003,629,1,677,817,1,1,1,798,1,308,
  34,163,38,1,177,471,163,1,1,1,280,1,578,1,1,542,1,1,1,
  805,956,582,611,860,723,1,218,477,235,578,1,235,81,931,468,
  936,1,1,789,1,1,1,422,1,1,1,1,1,1,1,462,1,1,1,1,22,1,
  318,140,140,578,422,1,1,1,923,1,1,1,1,357,1,1,1,432,1,1,
  357,1,1,1,200,1,186,1,102,1,81,1,1,1,373,1,491,473,469,
  1,701,576,1,590,1,1,1,1,1,590,364,186,235,147,1,146,287,
  1,1,276,148,483,502,22,131,494,318,494,205,95,205,131,318,
  494,494,220,1,1,280,163,1,1,1,471,1,798,1,860,723,805,582,
  956,1,1,1,1,1,542,611,1,102,1,146,514,1,1,477,287,1,483,
  1,669,1,434,669,511,482,1,1,1,1,717,140,1,1,1,1,1,1,1,
  1,1,315,1,1,1,1,391,1,1,1,1,1,1,1,81,191,1,1,186,176,
  365,402,140,81,419,253,419,81,140,140,140,253,419,419,1,652,
  1,1,1,923,789,1,659,1,1,1,621,948,693,1,1,1,795,1,475,1,
  1,1,422,468,1,551,1,583,1,583,583,583,1,1,551,1,1,1,1,1,
  1,1,670,671,970,1,659,1,621,1,1,1031,1,948,693,1,1,1,1,
  1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,
  1,1,1,1,578,931,1,1,795,948,1,693,1,621,1,1,659,1,1,218,
  140,1,1,432,1,1,1,1,1,1,1,357,357,235,1,1,1,1,1,1,1,1,
  1,422,1,462,1,1,391,1,1,1,1,1,1,315,1,1,205,1,1,1,89,
  146,45,140,1,1,1,1,1,1,214,86,89,131,1,1,1,205,131,1,1,
  1,1,89,214,86,140,1,1,1,1,45,89,146,1,1,1,1,146,86,41,
  131,205,86,1,1,1,140,41,1,1,1,146,1,1,1,1,1,1,205,140,
  131,3,7,7,7,1,1,5,7,7,10,7,10,1,1,19,1,5,5,10,7,19,7,
  10,7,10,10,1,1,10,1,1,1,26,1,1,16,26,1,26,26,16,1,1,7,
  7,7,1,1,5,7,7,10,7,10,1,1,19,1,5,5,10,7,19,7,10,7,10,
  10,1,1,10,1,1,1,26,1,1,16,26,1,26,26,16,1,1,7,7,1,1,
  5,5,5,9,5,16,16,1,1,1,4,5,7,5,11,5,9,5,7,1,16,16,11,
  1,1,1,1,1,1,11,1,1,11,11,11,1,1,3,14,1,7,5,7,14,7,1,
  10,1,1,1,5,7,10,5,10,7,14,7,5,7,7,7,19,19,1,1,1,26,1,
  26,1,1,16,16,26,26,1,14,1,7,5,7,14,7,1,10,1,1,1,5,7,
  10,5,10,7,14,7,5,7,7,7,19,19,1,1,1,26,1,26,1,1,16,16,
  26,26,1,1,9,16,10,16,10,29,9,1,1,1,7,9,13,16,36,10,16,
  10,20,1,1,1,20,1,1,1,1,1,1,31,149,1,52,31,31,1,1,1,1,
  1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,
  1,1,1,1,1,1,1,1,1,1,5,7,1,7,1,9,1,1,1,5,5,9,4,11,5,
  9,5,5,5,5,5,1,11,1,1,1,11,1,1,1,1,11,11,1,11,1,7,16,
  7,9,1,1,10,1,5,4,9,5,10,5,7,5,1,7,16,16,16,16,1,1,11,
  16,1,16,11,1,11,11,16,16,1,1,3,14,14,1,19,1,7,5,14,5,
  19,7,10,7,7,5,7,7,1,10,1,1,29,16,1,1,29,1,29,29,1,16,
  1,1,1,29,1,1,1,16,9,29,7,36,10,16,10,16,9,10,10,1,20,
  1,1,149,31,1,1,1,1,31,52,1,31,1,14,14,1,19,1,7,5,14,5,
  19,7,10,7,7,5,7,7,1,10,1,1,29,16,1,1,29,1,29,29,1,16,
  1,16,1,1,1,9,16,16,7,20,10,29,10,9,16,10,10,1,36,1,1,
  1,31,1,1,1,1,31,31,149,52,1,1,32,1,9,7,16,16,32,10,13,
  10,1,20,1,1,29,1,1,1,38,1,1,39,70,1,70,38,39,1,1,1,1,
  1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,
  1,1,1,11,16,20,11,16,10,29,10,16,1,43,43,1,1,1,1,1,1,
  1,1,1,1,31,31,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,
  1,1,1,1,1,1,1,1,1,1,1,1,5,1,5,1,7,16,7,1,7,16,16,1,
  11,1,1,11,11,1,1,11,1,1,1,1,11,1,16,5,16,7,1,7,7,1,
  16,16,10,10,1,1,16,11,1,11,16,1,16,16,11,11,1,16,1,1,9,
  1,1,20,1,1,1,36,1,1,31,52,1,149,31,1,1,1,1,31,1,1,7,
  16,7,6,6,10,10,11,1,1,1,11,1,1,11,11,1,1,1,11,1,1,1,
  1,1,1,26,1,1,1,1,1,1,31,1,1,1,31,1,1,1,1,1,1,1,3,10,
  10,1,1,19,19,1,1,16,29,1,29,16,1,1,1,29,29,1,1,20,1,1,
  1,32,32,1,1,39,70,1,70,39,1,1,1,38,38,1,10,10,1,1,19,
  19,1,1,16,29,1,29,16,1,1,1,29,29,1,5,7,7,1,26,1,1,16,
  26,1,1,11,1,1,26,1,26,1,7,7,16,1,1,1,1,26,1,16,1,1,
  26,26,11,1,1,3,43,1,1,1,59,1,1,38,38,1,1,278,59,278,1,
  43,1,1,1,59,1,1,38,38,1,1,278,59,278,1,16,1,1,1,31,1,
  1,1,1,1,1,1,31,1,1,1,1,1,1,31,1,1,1,1,31,1,1,1,1,1,
  1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,38,1,
  11,16,1,1,1,1,16,1,1,1,1,11,38,1,1,1,1,1,1,1,1,1,1,
  1,1,1,38,11,1,1,16,11,1,1,1,1,1,1,1,1,38,1,1,1,1,1,
  1,16,1,1,7,7,7,1,1,5,7,7,10,7,10,1,1,19,1,5,5,10,7,
  19,7,10,7,10,10,1,1,10,1,1,1,26,1,1,16,26,1,26,26,16,
  1,1,3,7,1,1,7,7,7,10,5,1,1,10,1,1,5,7,10,7,10,7,10,
  5,10,1,1,10,19,1,19,1,1,1,26,26,1,1,16,16,26,1,26,7,1,
  1,7,7,7,10,5,1,1,10,1,1,5,7,10,7,10,7,10,5,10,1,1,10,
  19,1,19,1,1,1,26,26,1,1,16,16,26,1,26,10,10,5,5,5,6,5,
  16,1,16,1,1,4,5,7,5,11,5,6,5,7,16,1,16,11,1,1,1,1,1,
  1,11,1,1,11,11,11,1,1,3,16,7,7,7,16,1,1,1,1,1,5,16,
  10,7,43,7,7,16,10,1,1,1,43,1,1,1,1,1,1,38,278,1,59,38,
  59,1,278,16,7,7,7,16,1,1,1,1,1,5,16,10,7,43,7,7,16,10,
  1,1,1,43,1,1,1,1,1,1,38,278,1,59,38,59,1,278,7,7,1,5,
  1,1,9,1,1,5,5,9,5,11,5,7,4,16,16,1,7,1,1,11,1,1,1,
  11,1,1,1,11,11,1,1,11,3,7,7,14,1,14,19,1,7,5,14,7,19,
  7,5,5,1,10,1,10,1,1,10,1,26,1,16,1,26,1,26,26,1,1,16,
  7,7,14,1,14,19,1,7,5,14,7,19,7,5,5,1,10,1,10,1,1,10,
  1,26,1,16,1,26,1,26,26,1,1,16,1,1,1,1,1,1,5,7,9,5,16,
  5,5,7,16,20,1,20,1,1,26,1,26,1,26,1,1,1,11,16,1,1,26,
  9,1,1,10,1,5,4,9,5,10,5,7,5,16,7,1,16,16,1,16,1,10,1,
  16,16,10,1,10,10,16,1,16,1,16,1,1,9,16,16,10,20,10,20,
  7,29,9,1,13,1,1,36,1,1,1,31,1,1,1,31,31,149,1,52,1,1,
  1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,
  1,1,1,32,1,9,7,16,10,32,10,20,16,29,13,1,9,29,1,1,1,
  70,1,1,39,38,1,38,70,39,1,1,1,11,16,20,10,16,10,26,11,
  36,1,1,36,1,1,1,1,1,1,1,1,1,1,31,31,1,1,1,1,1,1,1,1,
  1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,5,1,7,
  1,7,5,5,1,9,1,9,1,1,11,1,11,1,11,1,11,1,1,1,1,1,11,
  16,7,16,7,1,5,9,1,1,9,10,1,10,1,16,1,11,11,16,1,16,16,
  11,1,11,1,1,1,16,16,1,29,1,29,1,1,36,1,31,1,52,149,31,
  1,1,1,1,1,31,1,3,7,7,14,14,1,14,19,1,19,1,16,1,29,29,
  16,1,1,1,29,1,29,1,1,1,1,29,1,1,1,1,1,1,31,1,1,1,31,
  1,1,1,1,1,1,7,7,14,14,1,14,19,1,19,1,16,1,29,29,16,1,
  1,1,29,1,29,1,9,1,1,1,16,1,1,1,26,1,1,11,26,1,1,1,16,
  1,26,9,9,1,1,11,1,1,1,11,1,1,11,11,1,1,1,11,1,1,16,1,
  16,1,1,20,1,52,1,31,1,31,1,1,149,1,1,31,1,16,32,1,32,
  1,1,1,38,38,1,1,39,39,70,1,70,1,1,1,1,1,1,1,1,1,1,1,
  1,1,1,1,1,20,1,1,1,52,1,1,31,31,1,1,149,31,1,1,1,16,
  1,1,1,31,1,1,1,1,1,1,1,31,1,1,1,1,1,1,1,1,1,1,1,1,
  1,1,1,1,1,31,1,1,1,1,31,1,1,1,1,1,1,1,1,1,1,1,1,1,
  1,1,1,38,1,11,16,1,1,1,1,1,1,1,1,1,1,1,1,16,1,1,1,1,
  11,1,38,1,1,1,1,38,1,11,1,16,11,1,1,1,1,1,1,1,1,38,1,
  1,1,1,1,1,1,16,1,7,3,1,14,7,7,5,14,7,7,7,7,1,19,5,7,
  5,7,19,5,14,7,10,1,10,1,10,1,1,26,1,1,1,16,1,26,26,26,
  16,1,1,7,1,1,5,5,5,9,5,1,16,16,10,10,4,5,7,5,6,5,9,
  5,7,16,16,1,6,10,10,16,16,16,16,7,16,16,7,7,7,16,16,1,
  14,7,7,5,14,7,7,7,7,1,19,5,7,5,7,19,5,14,7,10,1,10,1,
  10,1,1,26,1,1,1,16,1,26,26,26,16,1,1,1,1,1,1,1,1,1,1,
  1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,
  1,1,1,1,1,9,10,16,16,10,1,1,1,80,1,7,9,20,10,24,16,16,
  10,13,29,9,1,17,1,1,1,86,1,1,20,86,1,34,34,16,74,74,7,
  5,1,7,1,16,16,1,10,5,5,7,5,10,4,9,5,16,16,7,1,1,16,
  16,11,1,16,16,1,1,11,11,11,1,16,16,7,1,3,10,1,1,11,11,
  7,5,10,7,11,5,10,7,1,10,10,1,7,7,7,43,43,7,7,7,43,43,
  43,43,7,7,7,16,7,6,10,10,10,1,5,4,6,5,10,5,7,5,16,7,
  16,1,1,16,16,1,11,16,16,1,11,1,11,11,1,16,16,1,1,1,1,
  1,23,16,9,20,10,23,7,16,10,9,29,13,1,1,43,43,67,187,43,
  43,1,1,38,38,67,1,43,43,10,1,1,11,11,7,5,10,7,11,5,10,
  7,1,10,10,1,7,7,7,43,43,7,7,7,43,43,43,43,7,7,7,7,7,
  1,17,7,5,5,5,10,7,9,5,20,16,20,1,1,26,26,17,1,16,16,1,
  1,17,11,11,10,26,26,3,48,1,16,5,7,7,48,16,10,7,1,10,1,
  1,26,1,1,1,81,1,1,26,81,1,81,81,26,1,1,48,1,16,5,7,7,
  48,16,10,7,1,10,1,1,26,1,1,1,81,1,1,26,81,1,81,81,26,
  1,1,10,10,10,10,19,10,6,80,19,23,1,24,1,1,1,1,36,1,1,
  1,1,1,36,36,36,1,1,1,10,6,17,19,10,10,24,19,23,24,1,1,
  1,1,1,1,36,1,1,1,36,1,36,36,1,1,1,5,1,7,1,5,16,7,1,
  9,9,1,1,16,16,11,11,16,16,1,11,11,1,1,1,16,16,5,7,10,
  5,1,7,9,1,9,1,6,10,10,7,16,16,16,7,16,7,16,16,7,16,16,
  7,1,1,16,7,1,9,1,1,1,26,26,17,11,26,26,10,11,17,1,1,1,
  16,16,1,7,1,3,14,14,14,1,10,1,1,26,26,1,1,16,26,26,1,
  1,16,1,1,10,1,1,1,80,80,1,1,1,1,36,36,1,1,1,36,36,1,
  1,1,1,1,16,7,9,9,1,1,6,10,10,16,7,10,10,6,7,16,16,16,
  6,10,10,1,29,1,29,1,17,1,1,34,86,1,1,20,86,34,1,1,16,
  74,74,14,14,14,1,10,1,1,26,26,1,1,16,26,26,1,1,16,1,1,
  16,16,1,1,43,43,67,67,43,43,1,38,38,1,187,1,43,43,16,1,
  17,1,1,34,1,74,74,16,1,34,86,86,20,1,1,1,17,1,1,1,34,
  1,1,20,34,1,86,86,16,74,74,1,1,1,1,1,1,1,1,1,1,1,1,1,
  1,1,7,7,1,1,10,10,1,1,1,1,1,1,10,10,3,1,1,1,1,16,1,
  1,1,1,16,1,1,1,1,1,1,16,1,1,1,1,16,1,1,16,1,1,1,13,
  48,13,16,1,1,1,1,1,1,48,13,13,16,1,1,1,3,7,1,1,1,1,5,
  7,7,7,1,1,1,1,5,7,7,1,1,1,1,5,5,5,16,16,13,1,1,1,16,
  13,1,1,1,48,1,1,1,1,1,1,7,7,3,7,1,1,7,7,7,10,5,1,1,
  10,1,1,5,7,10,7,10,7,10,5,10,1,1,10,19,1,19,1,1,1,26,
  26,1,1,16,16,26,1,26,1,14,7,7,5,14,7,7,7,7,1,19,5,7,
  5,7,19,5,14,7,10,1,10,1,10,1,1,26,1,1,1,16,1,26,26,26,
  16,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,
  1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,10,10,16,16,9,1,1,1,
  1,1,7,10,20,10,20,16,16,9,13,1,29,29,36,1,1,1,1,1,1,
  31,1,1,31,31,52,1,149,3,7,1,7,1,1,10,1,19,7,7,10,7,19,
  5,10,5,1,1,10,10,1,1,10,26,1,1,16,1,1,26,26,26,1,1,16,
  7,1,7,1,1,10,1,19,7,7,10,7,19,5,10,5,1,1,10,10,1,1,
  10,26,1,1,16,1,1,26,26,26,1,1,16,16,5,10,10,6,1,1,5,5,
  6,5,11,5,7,4,16,1,16,7,1,1,11,1,1,1,11,1,1,1,11,11,1,
  1,11,1,1,1,1,1,32,16,10,20,10,32,7,16,9,29,1,13,29,1,
  1,29,38,1,1,39,1,1,70,70,38,1,1,39,16,16,1,1,10,5,5,7,
  5,10,4,9,5,16,1,7,16,16,1,16,10,1,1,16,16,1,10,10,10,
  16,1,16,3,7,1,1,16,7,7,7,43,16,10,5,1,1,1,10,1,1,43,
  278,1,1,38,1,1,1,38,59,278,1,59,7,1,1,16,7,7,7,43,16,
  10,5,1,1,1,10,1,1,43,278,1,1,38,1,1,1,38,59,278,1,59,
  1,1,7,5,5,5,16,7,9,5,20,1,20,16,26,1,1,1,1,1,1,26,1,
  26,16,11,26,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,
  1,1,1,1,1,1,1,1,1,1,11,10,26,10,16,16,20,11,36,1,1,36,
  1,1,1,1,1,1,1,1,1,1,31,31,1,1,1,7,1,7,1,5,16,5,1,1,
  9,9,1,1,11,11,1,1,11,1,1,11,1,1,1,1,11,7,3,1,7,1,7,
  14,1,14,14,19,1,19,16,1,1,26,26,1,16,1,1,26,1,26,7,1,
  1,16,5,1,1,1,9,1,1,16,26,1,1,16,26,1,26,1,1,1,1,11,1,
  7,1,7,14,1,14,14,19,1,19,16,1,1,26,26,1,16,1,1,26,1,
  26,16,1,1,1,1,29,1,1,1,1,31,1,1,1,1,1,31,1,1,1,1,1,
  16,5,9,1,1,9,10,1,10,16,1,1,10,10,1,16,16,16,10,1,10,
  1,29,1,29,1,36,1,1,31,1,1,1,31,1,31,1,1,52,1,149,9,1,
  9,1,11,1,1,11,1,1,1,11,1,11,1,1,11,1,1,1,16,16,1,1,
  20,52,1,1,31,1,1,31,1,149,1,1,31,1,1,1,1,1,1,1,1,1,1,
  1,1,1,1,1,1,1,16,32,1,32,1,1,1,70,70,1,1,39,39,38,1,
  38,20,1,1,52,1,1,1,31,1,31,1,149,31,1,1,1,16,1,1,1,31,
  1,1,1,1,1,1,1,31,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,
  31,1,1,1,1,31,1,1,1,1,1,1,1,38,11,16,1,1,1,1,1,1,1,
  1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,16,1,1,1,1,11,1,38,1,
  1,1,1,38,1,11,1,1,1,1,1,1,16,11,1,1,1,38,1,1,1,1,1,
  1,1,16,1,14,1,7,5,7,14,7,1,10,1,1,1,5,7,10,5,10,7,14,
  7,5,7,7,7,19,19,1,1,1,26,1,26,1,1,16,16,26,26,1,1,10,
  16,10,16,9,1,29,29,1,1,7,10,13,16,17,10,16,9,20,1,1,1,
  24,1,80,1,1,1,86,34,74,74,16,20,34,1,86,1,1,1,1,1,1,1,
  1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,
  1,1,1,1,1,7,3,1,7,1,14,14,1,1,7,7,14,5,10,7,10,5,7,
  7,7,5,1,19,19,1,1,26,26,1,1,1,16,16,1,26,26,7,16,5,1,
  1,9,10,10,5,5,9,5,6,5,7,4,1,16,16,7,10,10,6,16,16,16,
  7,16,16,16,7,7,16,16,7,1,7,1,14,14,1,1,7,7,14,5,10,7,
  10,5,7,7,7,5,1,19,19,1,1,26,26,1,1,1,16,16,1,26,26,1,
  1,29,1,1,1,16,10,29,7,17,10,16,9,16,10,10,9,1,24,80,74,
  74,34,86,1,1,1,20,16,1,34,86,1,9,1,10,10,5,5,9,4,6,5,
  9,5,5,5,5,5,10,6,10,10,10,7,16,16,10,10,6,6,16,7,16,1,
  1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,
  1,1,1,1,1,16,1,1,9,10,16,16,17,10,13,7,1,1,1,20,80,1,
  24,74,74,1,34,86,1,1,20,16,86,1,34,1,1,9,10,16,7,17,10,
  29,16,9,10,10,16,80,24,1,1,1,34,1,86,74,74,16,20,86,34,
  1,3,16,7,43,16,7,7,43,16,26,1,1,26,1,1,1,1,1,1,1,1,1,
  1,16,16,1,1,1,16,7,43,16,7,7,43,16,26,1,1,26,1,1,1,1,
  1,1,1,1,1,1,16,16,1,1,1,7,1,5,1,7,16,5,1,16,16,7,1,
  10,10,16,16,11,11,1,16,16,1,1,1,11,11,1,7,7,3,1,7,10,
  1,1,10,11,11,11,7,7,43,43,43,7,7,7,7,43,43,43,16,1,1,
  29,16,1,1,1,20,1,23,23,43,43,67,67,187,43,43,1,1,1,38,
  38,1,7,16,5,6,10,10,6,10,1,10,16,16,1,11,11,16,16,1,1,
  11,1,11,7,1,1,1,26,26,1,1,1,1,10,10,1,1,1,10,10,1,1,
  1,1,1,1,7,10,1,1,10,11,11,11,7,7,43,43,43,7,7,7,7,43,
  43,43,1,20,1,1,1,23,23,1,43,43,38,1,38,43,43,1,1,67,67,
  187,7,16,16,1,10,10,1,16,16,11,1,11,16,16,1,1,11,11,1,
  7,7,5,1,17,10,26,26,17,11,1,16,16,1,10,1,17,11,3,7,48,
  1,48,1,1,1,81,81,1,1,26,26,81,1,81,7,48,1,48,1,1,1,81,
  81,1,1,26,26,81,1,81,10,17,1,26,26,17,1,11,16,16,1,10,
  11,17,1,10,10,1,1,36,36,1,1,1,1,1,1,36,36,10,1,1,1,36,
  36,1,1,1,1,36,1,36,1,1,36,1,36,1,1,1,1,36,36,1,3,1,1,
  1,7,7,5,7,1,1,1,1,1,1,7,7,5,7,1,1,1,16,16,1,1,1,1,
  13,48,13,16,1,1,1,1,13,13,48,1,1,1,1,48,13,13,3,7,5,1,
  1,1,7,5,1,1,1,5,1,1,1,1,1,1,16,16,16,10,14,10,6,10,
  14,14,10,14,1,61,6,14,6,10,61,6,10,14,13,1,13,1,19,1,1,
  99,1,1,1,32,1,99,99,99,32,1,1,14,6,10,10,14,1,13,1,1,
  1,6,14,13,6,19,10,10,14,6,14,10,14,61,61,1,1,1,99,1,99,
  1,1,32,32,99,99,1,7,7,1,3,1,10,1,1,19,7,7,10,5,19,5,
  14,7,7,7,5,7,1,10,1,29,1,16,1,1,1,29,29,29,1,16,1,5,
  10,7,16,1,16,1,10,5,5,7,5,10,4,6,5,1,16,7,16,16,16,1,
  11,1,16,1,16,1,11,11,11,16,16,1,10,7,10,6,10,1,1,5,5,
  6,4,11,5,6,5,5,5,5,5,1,11,1,1,1,11,1,1,1,1,11,11,1,
  11,1,1,1,13,1,1,61,10,14,13,6,61,6,10,14,10,14,6,14,1,
  19,1,99,1,32,1,1,1,99,99,99,1,32,1,1,10,1,1,19,7,7,10,
  5,19,5,14,7,7,7,5,7,1,10,1,29,1,16,1,1,1,29,29,29,1,
  16,1,7,3,1,1,16,7,7,5,43,16,14,7,16,7,16,7,1,43,1,1,
  1,38,1,1,1,278,59,38,278,59,1,7,1,1,7,5,5,5,16,7,6,5,
  1,16,1,16,26,1,1,26,1,1,1,26,1,1,11,16,26,1,1,1,1,16,
  7,7,5,43,16,14,7,16,7,16,7,1,43,1,1,1,38,1,1,1,278,59,
  38,278,59,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,
  1,1,1,1,1,1,1,1,11,10,26,11,16,16,19,10,16,43,1,43,1,
  1,1,1,1,1,1,1,1,1,31,31,1,1,1,7,1,5,1,5,10,7,1,16,7,
  16,1,11,1,11,1,11,1,1,1,11,1,1,1,11,1,7,7,1,7,1,3,10,
  1,10,1,19,19,1,16,1,26,1,26,1,16,1,1,26,26,1,5,1,1,10,
  7,1,16,1,16,1,16,1,26,1,16,1,26,1,26,1,1,1,11,1,1,5,
  10,7,6,10,6,10,11,1,1,11,1,1,1,11,1,11,1,1,11,1,1,16,
  1,1,1,1,26,1,1,1,1,31,1,1,1,1,1,31,1,1,1,1,1,10,7,7,
  16,1,16,10,10,1,16,1,10,1,10,1,16,16,16,10,10,1,1,13,1,
  13,1,61,61,1,32,1,99,1,99,1,32,1,1,99,99,1,10,1,10,1,
  19,19,1,16,1,26,1,26,1,16,1,1,26,26,1,7,5,7,1,26,1,16,
  1,26,1,1,1,11,1,26,1,26,1,7,3,43,1,1,59,1,278,1,59,1,
  38,1,278,38,1,1,7,16,1,1,1,1,1,1,11,1,1,26,26,16,26,1,
  43,1,1,59,1,278,1,59,1,38,1,278,38,1,1,16,1,1,1,31,1,
  1,1,1,1,1,1,31,1,1,1,1,1,1,31,1,1,1,1,31,1,1,1,1,1,
  1,1,1,1,1,1,1,1,1,1,1,1,1,1,38,11,16,1,1,1,1,1,1,1,
  1,1,1,1,1,1,1,16,1,1,1,1,11,38,1,1,1,1,1,1,1,1,1,1,
  1,1,1,38,11,1,1,1,1,1,1,1,16,11,1,1,1,38,1,1,1,1,1,
  1,16,1,1,16,7,7,7,16,1,1,1,1,1,5,16,10,7,43,7,7,16,
  10,1,1,1,43,1,1,1,1,1,1,38,278,1,59,38,59,1,278,1,10,
  1,14,1,9,29,1,32,10,10,13,16,32,7,16,9,1,1,20,1,1,1,
  29,70,1,1,39,1,1,38,38,70,1,1,39,7,3,1,1,1,1,1,48,7,
  16,10,7,48,5,7,16,1,1,10,1,1,1,26,81,1,1,26,1,1,81,81,
  81,1,1,26,7,10,13,10,13,1,1,5,7,6,5,16,5,5,7,16,1,16,
  1,1,1,26,1,1,1,26,1,26,1,16,11,1,1,26,1,1,1,1,1,48,7,
  16,10,7,48,5,7,16,1,1,10,1,1,1,26,81,1,1,26,1,1,81,81,
  81,1,1,26,29,29,1,1,23,10,9,13,16,23,7,16,10,1,1,20,1,
  43,1,43,39,1,1,43,43,97,38,38,39,43,1,43,1,14,1,1,20,
  10,10,16,63,20,16,9,1,1,1,1,1,1,149,1,1,1,106,1,1,129,
  38,70,291,1,170,1,1,1,16,16,7,7,55,16,7,16,1,1,1,1,1,
  1,1,1,1,1,1,1,1,1,102,102,1,1,1,1,1,20,9,10,16,76,20,
  16,10,1,1,1,1,95,1,1,1,1,1,1,95,229,1,102,62,95,1,1,1,
  1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,
  1,1,16,17,40,43,26,26,43,36,76,1,1,1,1,1,1,1,1,1,1,1,
  192,1,70,81,1,1,1,10,1,7,1,5,7,10,1,1,9,1,1,1,26,11,
  1,1,26,1,26,16,1,1,1,1,26,10,1,1,10,1,14,29,1,29,1,24,
  1,80,20,1,1,86,34,74,16,74,1,34,1,86,7,1,1,7,10,1,1,1,
  1,1,1,40,17,1,1,40,26,26,40,1,1,1,1,17,1,7,3,1,14,1,
  14,1,43,1,1,38,1,1,1,38,278,59,1,1,59,1,278,26,1,1,1,
  1,149,1,1,1,1,81,1,1,1,1,192,70,1,1,1,1,1,7,10,9,1,1,
  1,10,1,17,16,1,1,17,10,10,26,26,16,10,1,17,1,14,1,14,1,
  43,1,1,38,1,1,1,38,278,59,1,1,59,1,278,29,1,29,1,20,1,
  1,31,1,1,1,31,149,52,1,1,31,1,1,1,16,1,1,1,95,122,1,1,
  95,1,318,89,1,262,1,1,95,1,1,1,1,1,1,1,1,1,1,1,1,1,1,
  1,1,1,1,63,1,1,1,1,1,1,70,291,1,170,106,38,1,129,1,1,
  1,1,1,1,1,1,1,1,1,1,1,1,1,1,43,1,1,1,70,1,1,1,1,1,
  1,1,70,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,81,1,1,1,
  1,81,1,1,1,1,1,1,60,1,26,16,1,1,1,1,1,1,1,1,1,1,1,1,
  1,1,1,1,1,1,1,1,1,1,43,1,1,1,1,16,1,110,1,1,1,1,38,
  1,16,74,74,60,1,1,1,26,26,1,1,1,1,1,1,1,1,1,1,1,43,1,
  10,1,1,14,1,29,1,1,1,10,10,29,7,20,16,16,9,10,10,16,9,
  1,36,1,1,1,31,1,1,1,1,31,31,1,52,149,7,7,10,1,1,1,17,
  1,5,7,9,5,10,5,5,7,1,20,16,20,26,26,1,1,17,16,1,16,17,
  1,11,11,26,26,10,3,1,1,14,1,1,1,7,16,14,5,43,7,7,16,7,
  16,7,16,1,43,1,1,278,38,1,1,1,1,38,59,1,59,278,1,1,14,
  1,1,1,7,16,14,5,43,7,7,16,7,16,7,16,1,43,1,1,278,38,1,
  1,1,1,38,59,1,59,278,1,29,1,80,1,10,9,29,7,24,16,16,10,
  10,9,16,10,1,17,1,1,100,20,1,1,100,1,36,36,74,16,74,1,
  1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,
  1,1,1,1,1,1,1,1,9,20,16,16,63,10,10,20,1,1,1,1,149,1,
  1,1,129,1,1,106,1,1,70,38,170,1,291,1,1,1,1,1,1,1,1,1,
  1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,26,43,95,
  26,43,26,26,29,40,1,179,149,1,1,1,1,1,1,1,1,1,1,81,81,
  1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,
  1,1,1,1,1,10,1,5,1,7,7,10,1,20,16,20,1,16,1,1,26,11,
  1,1,26,1,1,1,1,16,26,1,10,43,1,1,14,13,1,9,29,23,23,1,
  1,43,38,1,38,43,1,43,43,67,67,187,16,1,1,1,1,1,1,1,1,
  1,76,1,1,95,122,1,262,95,1,1,1,1,89,318,1,7,7,10,6,13,
  10,13,16,1,1,1,26,1,1,11,26,1,1,1,16,1,26,1,1,1,1,95,
  179,1,1,1,1,1,70,1,1,1,70,1,1,1,1,1,1,3,1,10,1,1,1,
  48,48,1,1,26,59,1,59,26,1,1,1,65,65,289,1,10,1,1,1,48,
  48,1,1,26,59,1,59,26,1,1,1,65,65,289,13,29,29,1,32,32,
  1,1,39,70,1,70,39,1,1,1,38,38,1,10,7,10,1,40,1,1,40,
  17,1,1,17,1,1,26,1,40,26,1,14,76,1,1,1,1,318,1,89,1,1,
  95,95,122,1,262,1,31,1,1,1,113,1,1,34,113,1,113,113,73,
  331,331,63,1,1,1,170,1,1,70,106,1,1,291,38,129,1,26,1,1,
  1,81,1,1,1,1,1,1,1,70,192,1,1,1,1,1,81,1,1,1,1,70,1,
  192,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,
  1,1,1,110,1,16,43,1,1,1,1,16,1,1,1,1,26,149,60,1,1,1,
  1,1,1,1,1,1,1,1,1,149,26,60,1,43,16,1,1,1,1,1,1,1,1,
  38,1,1,1,1,1,1,26,74,74,7,1,7,1,1,10,1,19,7,7,10,7,
  19,5,10,5,1,1,10,10,1,1,10,26,1,1,16,1,1,26,26,26,1,1,
  16,1,7,1,14,14,1,1,7,7,14,5,10,7,10,5,7,7,7,5,1,19,
  19,1,1,26,26,1,1,1,16,16,1,26,26,1,1,1,1,1,1,1,1,1,1,
  1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,10,
  1,1,19,7,7,10,5,19,5,14,7,7,7,5,7,1,10,1,29,1,16,1,1,
  1,29,29,29,1,16,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,
  1,1,1,1,1,1,1,1,1,1,1,1,1,1,16,1,1,10,10,16,16,36,9,
  13,7,1,1,1,20,1,1,20,149,1,1,31,1,1,1,31,52,1,1,31,1,
  1,10,10,16,7,36,9,29,16,10,10,9,16,1,20,1,1,1,31,1,1,
  1,149,52,31,1,31,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,
  1,1,1,1,1,1,1,1,1,1,1,1,10,10,29,11,16,16,20,11,43,43,
  1,16,1,1,1,1,1,1,1,1,1,1,31,31,1,1,1,3,1,7,1,7,1,7,
  1,1,10,10,1,19,19,16,1,26,26,1,1,16,1,1,1,26,26,1,7,1,
  7,1,7,1,1,10,10,1,19,19,16,1,26,26,1,1,16,1,1,1,26,26,
  16,1,1,9,16,1,1,1,20,1,32,32,39,1,38,38,1,1,39,1,1,1,
  70,70,1,5,16,5,10,10,6,6,1,1,11,11,1,1,11,1,1,11,1,1,
  1,1,11,16,1,1,1,1,26,1,1,1,1,31,1,1,1,1,1,31,1,1,1,
  1,1,16,5,16,16,1,7,1,10,10,16,1,10,10,1,1,16,16,16,1,
  10,10,1,1,1,20,1,1,36,1,31,1,31,1,1,1,31,1,1,1,52,149,
  16,16,7,1,1,11,1,11,1,11,1,1,1,11,1,1,1,11,1,3,7,7,1,
  1,43,59,1,278,59,1,1,38,1,278,1,1,38,7,7,1,1,43,59,1,
  278,59,1,1,38,1,278,1,1,38,5,1,1,16,1,1,1,11,1,1,1,26,
  26,1,26,16,1,26,1,16,1,26,1,1,1,11,1,26,1,26,1,1,1,1,
  1,1,1,1,1,1,1,1,1,1,1,16,1,1,1,31,1,1,1,1,1,1,1,31,
  1,1,31,1,1,1,1,1,1,1,31,1,1,1,1,1,1,38,11,16,1,1,1,
  1,1,1,1,1,1,1,1,1,1,16,1,1,1,1,1,1,38,11,1,1,1,1,1,
  1,11,38,1,1,1,1,1,1,1,1,1,1,1,1,1,16,11,1,1,1,38,1,
  1,1,1,1,1,1,1,16,7,7,14,1,14,19,1,7,5,14,7,19,7,5,5,
  1,10,1,10,1,1,10,1,26,1,16,1,26,1,26,26,1,1,16,1,1,1,
  1,1,48,7,16,10,7,48,5,7,16,1,1,10,1,1,1,26,81,1,1,26,
  1,1,81,81,81,1,1,26,10,1,1,11,11,7,5,10,7,11,5,10,7,1,
  10,10,1,7,7,7,43,43,7,7,7,43,43,43,43,7,7,7,1,16,1,80,
  10,16,16,10,24,9,20,7,1,29,29,13,1,1,17,86,1,1,20,1,1,
  86,34,34,74,74,16,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,
  1,1,1,1,1,1,1,1,1,1,1,1,1,23,1,10,7,16,10,23,9,20,16,
  1,13,29,29,43,43,1,1,38,43,1,43,67,187,67,38,43,43,1,10,
  19,10,24,19,10,6,17,10,1,1,24,23,1,1,1,36,1,1,1,1,1,
  36,36,36,1,1,1,19,6,80,19,10,10,10,10,1,24,1,23,1,1,1,
  1,36,1,1,1,36,1,36,36,1,1,1,7,1,3,1,7,7,7,1,14,14,14,
  1,1,10,26,26,1,16,1,26,26,1,1,1,1,16,16,7,10,5,1,5,1,
  1,9,9,10,10,6,7,16,16,7,16,16,7,16,16,16,16,7,1,1,1,
  16,16,1,29,1,29,1,1,17,86,34,74,16,74,34,86,1,1,1,1,20,
  1,7,7,7,1,14,14,14,1,1,10,26,26,1,16,1,26,26,1,1,1,1,
  16,10,1,1,1,80,80,1,1,1,1,36,36,1,1,1,36,36,1,1,1,1,
  1,5,5,1,9,1,9,10,10,6,16,7,10,6,10,7,16,16,16,10,10,6,
  1,1,1,9,1,26,26,1,11,17,16,1,16,17,11,1,1,26,26,10,1,
  9,9,1,16,16,1,11,11,16,1,16,11,11,1,1,16,16,1,1,1,1,1,
  1,1,1,1,1,1,1,1,1,1,1,1,1,1,16,16,1,1,17,34,1,74,16,
  74,1,34,86,86,1,1,20,16,1,1,17,1,34,1,20,1,34,1,86,86,
  74,74,16,43,43,1,67,67,43,1,43,38,38,1,187,43,43,1,3,7,
  1,1,1,16,1,1,1,1,1,1,1,16,7,1,1,1,16,1,1,1,1,1,1,1,
  16,1,1,10,1,10,1,1,1,1,10,10,1,16,1,1,1,13,48,13,16,1,
  1,1,1,1,1,48,13,13,16,1,1,1,7,3,1,1,1,1,7,7,5,7,1,1,
  1,1,5,5,5,1,1,1,1,7,7,5,16,16,13,1,1,1,16,13,1,1,1,
  48,1,1,1,1,1,1,3,7,7,1,1,14,1,1,1,7,16,14,5,43,7,7,
  16,7,16,7,16,1,43,1,1,278,38,1,1,1,1,38,59,1,59,278,14,
  14,1,19,1,7,5,14,5,19,7,10,7,7,5,7,7,1,10,1,1,29,16,
  1,1,29,1,29,29,1,16,1,10,10,1,1,14,10,10,6,19,14,13,6,
  14,10,14,6,1,61,61,1,1,99,99,1,1,1,32,32,1,99,99,10,61,
  1,14,6,10,10,61,14,6,6,1,13,1,13,1,1,19,1,99,1,32,1,
  99,1,99,99,1,1,32,61,1,14,6,10,6,61,14,13,10,14,6,14,
  10,1,19,1,1,99,32,1,1,99,1,99,99,1,32,1,1,10,16,19,11,
  16,10,26,11,43,1,43,16,1,1,1,1,1,1,1,1,1,1,31,31,1,1,
  1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,
  1,1,1,7,1,7,1,3,7,7,1,10,1,10,1,19,19,1,16,29,29,1,
  16,1,1,1,1,29,29,10,5,16,7,1,5,16,1,16,7,1,10,10,1,16,
  11,11,1,16,1,16,16,1,11,11,10,1,1,10,10,1,13,1,13,1,61,
  61,1,32,99,99,1,32,1,1,1,1,99,99,1,7,5,5,10,6,10,6,1,
  1,11,1,11,1,11,1,11,1,1,1,1,1,11,1,1,1,1,26,1,1,1,1,
  1,1,31,1,1,1,31,1,1,1,1,1,1,7,7,1,10,1,10,1,19,19,1,
  16,29,29,1,16,1,1,1,1,29,29,1,16,1,16,1,1,16,1,1,26,
  11,1,1,26,1,1,1,1,16,26,16,7,16,1,1,11,1,1,11,11,1,1,
  11,1,1,1,1,11,1,7,3,7,1,1,43,1,59,1,38,1,38,1,1,278,
  1,278,59,7,5,1,1,16,1,1,26,16,1,1,1,26,26,1,1,11,7,1,
  1,43,1,59,1,38,1,38,1,1,278,1,278,59,1,26,1,1,16,26,1,
  1,11,1,1,26,1,26,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,16,1,
  1,1,31,1,1,1,1,1,1,1,31,1,1,31,1,1,1,1,1,1,1,31,1,1,
  1,1,1,1,1,1,1,1,1,1,1,1,1,38,1,11,16,1,1,1,16,1,1,1,
  1,1,1,38,11,1,1,1,1,1,1,11,38,1,1,1,1,1,1,1,1,16,11,
  1,1,1,1,1,1,1,1,38,1,1,1,1,1,1,1,1,16,1,1,1,1,1,1,
  1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,
  1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,
  1,1,1,1,1,1,1,1,1,1,10,20,16,16,76,9,10,20,1,1,1,1,1,
  1,95,229,1,1,95,1,1,1,62,102,1,1,95,1,1,1,1,1,1,1,1,
  1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,
  1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,43,
  17,149,16,26,26,43,36,31,63,1,76,1,1,1,1,192,1,1,1,1,1,
  81,70,1,1,1,1,1,7,1,7,3,1,1,1,10,1,1,43,1,59,278,38,
  1,1,1,38,1,1,1,59,278,1,10,1,10,1,14,9,1,13,29,1,24,
  80,16,74,34,86,1,1,20,1,74,1,34,86,16,1,1,1,1,1,1,1,1,
  1,63,1,170,291,38,129,1,1,106,1,1,1,70,1,1,5,7,10,10,
  13,6,13,1,1,26,16,26,1,26,1,1,11,1,1,1,1,26,26,1,1,1,
  1,40,1,1,1,1,70,192,1,1,1,1,81,1,1,1,1,1,7,10,16,20,
  1,20,1,10,17,26,10,10,17,1,1,16,16,26,1,10,17,1,1,1,10,
  1,1,43,1,59,278,38,1,1,1,38,1,1,1,59,278,29,29,13,1,1,
  20,1,52,149,31,1,1,1,31,1,1,1,31,1,1,7,1,1,179,179,73,
  331,113,113,1,1,34,1,331,1,113,113,10,14,1,1,149,38,1,
  291,170,1,1,70,1,129,1,1,106,10,1,1,40,1,26,1,17,1,1,1,
  17,40,1,26,40,1,95,1,102,229,95,1,1,1,62,1,1,1,95,1,1,
  1,1,1,1,1,1,1,1,1,1,1,1,1,43,1,1,1,70,1,1,1,1,1,1,
  1,70,1,1,81,1,1,1,1,1,1,1,81,1,74,1,1,1,1,1,26,26,1,
  1,1,1,1,1,1,60,60,74,1,1,1,43,1,1,1,1,1,1,38,16,1,1,
  1,1,1,1,16,110,1,1,1,1,1,1,1,1,1,1,1,1,1,16,26,1,1,
  1,1,1,1,1,1,1,1,1,1,43,16,1,1,80,10,16,16,7,24,9,29,
  10,10,16,9,10,1,17,1,100,1,20,1,1,1,100,36,36,74,16,74,
  1,23,1,10,7,16,16,23,9,13,10,1,20,1,1,43,1,43,97,38,1,
  43,43,39,1,39,38,43,1,43,1,1,1,1,1,1,1,1,1,1,1,1,1,1,
  1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,10,19,10,24,10,10,6,80,
  19,48,1,10,48,1,1,1,17,1,1,1,1,1,17,17,17,1,1,1,19,6,
  80,10,10,10,24,19,48,10,1,48,1,1,1,1,17,1,1,1,17,1,17,
  17,1,1,1,7,1,7,1,7,1,3,1,10,10,1,1,10,1,29,29,16,1,1,
  29,29,1,1,1,16,1,16,5,10,5,1,7,16,1,7,16,10,6,10,6,10,
  7,16,16,10,6,10,10,16,7,16,16,1,1,29,1,1,20,1,1,1,17,
  1,100,36,16,74,74,36,100,1,1,1,20,1,1,5,16,7,10,6,6,10,
  16,1,16,10,10,1,16,16,10,10,1,1,16,1,16,10,1,1,1,17,17,
  1,1,1,1,17,17,1,1,1,17,17,1,1,1,1,1,16,7,16,7,1,16,
  10,6,10,10,6,6,10,10,6,10,10,10,10,6,10,1,1,1,20,1,1,
  17,1,36,100,20,1,1,100,36,1,1,74,16,74,1,10,10,1,1,10,
  1,29,29,16,1,1,29,29,1,1,1,16,1,7,7,3,1,26,1,65,65,26,
  1,1,59,59,1,289,1,26,1,5,7,26,1,26,10,1,10,26,26,1,10,
  17,17,16,1,16,7,26,1,26,1,10,1,16,16,10,1,17,17,26,10,
  26,1,26,1,65,65,26,1,1,59,59,1,289,1,26,1,7,3,1,1,16,
  1,1,1,1,1,1,1,16,1,7,1,1,1,10,10,1,1,1,1,10,1,10,1,
  1,16,1,1,1,1,1,1,1,16,1,10,1,1,1,8,27,8,10,1,1,1,1,
  1,1,27,8,8,10,1,1,1,7,7,1,1,1,1,5,5,5,3,1,1,1,1,7,
  5,7,1,1,1,1,7,5,7,10,10,8,1,1,1,10,8,1,1,1,27,1,1,1,
  1,1,1,7,3,7,7,1,1,16,7,7,7,43,16,10,5,1,1,1,10,1,1,
  43,278,1,1,38,1,1,1,38,59,278,1,59,1,1,16,7,7,5,43,16,
  14,7,16,7,16,7,1,43,1,1,1,38,1,1,1,278,59,38,278,59,1,
  1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,
  1,1,1,29,26,26,26,43,43,95,26,149,1,1,40,1,1,1,1,1,1,
  1,1,1,1,81,81,1,1,1,1,1,10,1,14,1,10,1,9,29,13,1,32,
  32,39,1,70,70,1,1,39,1,1,1,38,38,3,7,1,1,1,7,1,1,1,
  10,1,48,48,26,1,81,81,1,1,26,1,1,1,81,81,7,1,1,1,7,1,
  1,1,10,1,48,48,26,1,81,81,1,1,26,1,1,1,81,81,1,10,16,
  5,13,10,13,6,1,1,16,26,1,1,11,1,1,26,1,1,26,1,16,43,1,
  1,1,1,95,1,1,1,1,70,1,1,1,1,1,70,1,1,1,1,1,1,10,29,
  29,1,13,1,23,23,43,1,39,39,1,1,43,43,43,97,38,38,1,1,1,
  1,1,1,76,1,95,1,62,1,1,1,95,1,1,1,102,229,20,16,20,1,
  1,16,1,26,1,11,1,1,1,26,1,1,26,16,1,1,14,10,1,1,63,
  170,1,1,70,1,1,106,1,291,1,129,38,1,7,1,1,55,1,1,1,102,
  1,1,1,1,1,1,1,102,10,1,1,76,1,1,1,62,1,1,1,95,95,229,
  1,102,1,40,1,40,1,17,1,1,1,17,1,26,26,40,1,1,1,1,1,1,
  1,1,1,1,1,1,1,1,1,26,1,1,1,81,1,1,1,1,1,192,1,70,1,
  1,81,1,1,1,1,1,1,192,70,1,1,1,1,1,1,110,16,43,1,1,1,
  1,1,1,1,1,1,1,1,1,1,16,1,1,1,1,1,60,1,26,1,1,1,1,1,
  60,26,1,1,1,1,1,1,1,1,1,1,1,1,1,1,43,16,1,1,1,38,1,
  1,1,1,1,1,74,74,26,48,1,16,5,7,7,48,16,10,7,1,10,1,1,
  26,1,1,1,81,1,1,26,81,1,81,81,26,1,1,1,36,26,43,43,26,
  17,40,16,1,1,1,76,1,1,1,192,1,1,1,1,1,1,81,70,1,1,1,
  1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,
  1,1,10,1,1,1,14,10,10,1,29,1,29,1,1,20,149,52,1,31,1,
  31,1,1,1,1,1,31,7,7,26,10,1,5,1,1,1,9,17,1,10,10,26,
  1,11,17,16,1,16,26,17,1,11,3,1,1,7,7,1,14,1,14,1,1,43,
  278,59,1,59,278,38,1,1,1,1,1,38,1,1,7,7,1,14,1,14,1,1,
  43,278,59,1,59,278,38,1,1,1,1,1,38,1,1,1,1,149,1,1,1,
  1,1,192,70,1,1,1,81,1,1,1,1,1,1,10,10,1,29,1,29,80,1,
  24,74,16,1,36,100,20,1,1,74,100,1,36,1,1,1,1,1,40,1,1,
  26,40,1,1,17,17,1,1,1,40,1,26,1,9,1,1,26,1,1,26,16,1,
  1,26,11,1,1,1,26,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,
  1,1,1,1,16,1,1,63,291,1,1,38,129,1,1,106,170,1,1,70,1,
  1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,95,1,1,1,102,1,1,95,
  62,1,1,229,95,1,1,1,43,1,1,1,81,1,1,1,1,1,1,1,81,1,1,
  1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,70,1,1,1,1,70,1,1,74,
  1,1,1,60,1,60,74,1,1,1,1,1,1,1,1,26,26,1,1,1,1,1,1,
  1,1,1,1,1,1,43,1,1,1,1,16,1,38,1,1,1,1,110,1,16,1,16,
  26,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,43,1,1,36,26,43,
  16,26,17,149,43,76,1,63,55,1,1,1,1,1,1,1,1,1,192,70,81,
  1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,
  1,1,1,1,1,10,1,10,1,14,1,1,1,13,29,9,1,20,1,1,31,31,
  1,1,52,149,1,1,1,31,1,7,5,26,10,1,7,20,1,20,16,17,10,
  1,1,16,11,1,17,26,10,26,16,17,11,1,7,1,1,1,3,1,10,1,1,
  1,43,1,1,38,59,1,278,59,278,1,1,1,38,1,1,10,16,7,13,6,
  13,10,26,1,1,1,11,1,1,26,16,26,1,1,26,1,1,1,1,1,1,40,
  1,1,1,1,1,1,81,1,1,1,70,192,1,1,1,1,1,1,1,29,13,1,29,
  80,24,1,1,20,36,1,100,16,74,74,1,100,36,1,1,1,1,1,1,1,
  63,1,1,106,70,1,1,170,291,1,1,129,38,1,1,10,1,1,1,43,1,
  1,38,59,1,278,59,278,1,1,1,38,1,10,14,1,1,95,1,1,89,95,
  1,1,122,262,1,318,1,95,1,10,7,40,1,1,1,1,26,1,40,1,26,
  40,17,17,1,1,1,149,1,1,1,70,1,1,106,38,1,129,1,170,291,
  1,1,1,1,1,102,1,1,1,102,1,1,1,1,1,1,43,1,1,1,81,1,1,
  1,1,1,1,1,81,1,1,1,1,1,1,70,1,1,1,1,70,1,1,1,1,1,1,
  1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,60,
  26,16,1,1,1,1,43,1,1,1,1,16,38,1,1,1,1,1,1,1,1,1,1,
  1,1,1,110,16,1,74,26,26,1,1,1,74,60,1,1,1,1,1,1,1,1,
  1,1,43,1,1,16,7,43,16,7,7,43,16,26,1,1,26,1,1,1,1,1,
  1,1,1,1,1,16,16,1,1,1,1,1,19,1,10,1,19,1,1,20,32,1,1,
  1,31,1,1,1,1,1,31,1,1,1,1,1,1,1,3,7,1,1,1,1,43,1,1,
  1,1,16,1,1,1,1,1,16,1,1,1,1,1,43,1,1,1,43,1,1,1,63,
  1,1,1,70,1,1,1,1,1,70,1,1,1,1,1,1,10,1,19,61,1,19,61,
  1,1,1,31,1,1,1,1,1,31,1,1,1,1,1,7,1,1,1,1,43,1,1,1,
  1,16,1,1,1,1,1,16,1,1,1,1,1,26,10,17,1,1,17,1,1,1,10,
  1,1,1,1,1,10,10,10,1,1,1,1,1,1,95,1,1,1,1,81,1,1,1,
  1,1,81,1,1,1,1,1,32,1,20,1,1,1,1,31,1,1,1,1,1,31,1,
  1,1,1,1,1,43,48,1,1,1,70,1,1,1,1,1,81,1,192,1,1,1,1,
  1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,43,1,1,1,1,1,1,1,1,
  1,1,70,70,1,1,1,1,1,1,70,1,1,1,1,1,81,1,192,1,1,1,1,
  1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,
  1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,14,9,10,1,1,1,
  1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,
  1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,10,9,1,1,1,14,1,1,1,
  1,1,1,1,1,1,10,1,19,1,1,43,19,1,20,1,32,1,1,1,1,31,1,
  1,1,31,1,1,1,1,1,1,26,10,7,7,1,10,17,1,1,17,1,1,1,1,
  10,1,1,1,10,1,10,10,1,1,1,1,1,1,1,1,1,95,1,1,1,1,1,
  1,81,1,1,1,81,1,1,1,1,1,1,1,1,43,19,61,19,1,61,1,1,1,
  1,31,1,1,1,31,1,1,1,1,1,1,3,1,1,1,43,1,1,1,1,1,1,16,
  1,1,1,16,1,1,1,1,1,1,1,1,1,43,1,1,1,1,1,1,16,1,1,1,
  16,1,1,1,1,1,1,1,63,1,1,1,1,1,1,1,70,1,1,1,70,1,1,1,
  1,1,1,32,20,1,1,1,1,1,1,31,1,1,1,31,1,1,1,1,1,1,43,
  1,48,1,1,1,1,70,1,1,1,81,1,1,192,1,1,1,1,43,1,1,1,1,
  1,1,1,1,1,1,70,70,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,
  1,1,1,1,1,1,1,70,1,1,1,81,1,1,192,1,1,1,1,1,1,1,1,1,
  1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,
  1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,14,1,9,
  10,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,
  1,1,1,1,10,9,1,1,1,1,1,1,1,1,14,1,1,1,1,1,1,1,1,1,
  1,7,1,7,1,7,1,1,10,10,1,19,19,16,1,26,26,1,1,16,1,1,
  1,26,26,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,
  1,1,1,1,7,7,7,1,14,14,14,1,1,10,26,26,1,16,1,26,26,1,
  1,1,1,16,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,
  1,7,7,1,10,1,10,1,19,19,1,16,29,29,1,16,1,1,1,1,29,29,
  1,1,1,10,1,1,43,1,59,278,38,1,1,1,38,1,1,1,59,278,1,
  10,10,1,1,10,1,29,29,16,1,1,29,29,1,1,1,16,1,1,1,1,1,
  1,1,1,1,1,1,1,1,1,1,1,1,1,1,16,16,1,1,36,31,1,149,52,
  1,1,31,1,1,1,1,31,16,1,1,36,1,31,1,31,1,31,1,1,1,1,
  149,52,1,29,1,38,38,39,1,1,70,70,1,1,1,39,1,1,1,1,1,1,
  1,1,1,1,1,1,1,1,1,16,1,1,1,31,1,1,1,1,1,1,1,31,1,1,
  31,1,1,1,1,1,1,1,31,1,16,1,1,1,11,38,1,1,1,1,1,1,1,
  1,38,11,1,1,1,1,1,16,1,1,1,1,1,1,38,11,1,1,1,1,1,1,
  11,38,1,1,1,1,1,1,1,16,1,1,1,1,1,1,1,1,1,1,1,1,1,1,
  1,1,1,1,1,16,7,1,1,1,7,1,1,1,10,1,48,48,26,1,81,81,1,
  1,26,1,1,1,81,81,1,7,1,7,14,1,14,14,19,1,19,16,1,1,26,
  26,1,16,1,1,26,1,26,7,1,1,1,1,43,1,1,1,1,16,1,1,1,1,
  1,16,1,1,1,1,1,1,7,10,1,1,10,11,11,11,7,7,43,43,43,7,
  7,7,7,43,43,43,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,
  1,1,10,1,10,1,19,19,1,16,1,26,1,26,1,16,1,1,26,26,1,1,
  16,16,1,80,24,16,74,86,34,1,1,20,1,74,1,86,34,1,1,1,1,
  1,1,1,1,1,1,1,1,1,1,1,1,1,16,23,1,23,1,43,1,38,38,43,
  1,43,43,67,187,67,24,80,1,16,74,86,1,34,1,20,1,74,34,86,
  1,10,10,1,1,36,36,1,1,1,1,1,1,36,36,10,1,1,1,36,36,1,
  1,1,1,36,1,36,1,1,36,1,36,1,1,1,1,36,36,1,7,1,1,1,5,
  5,5,7,1,1,1,1,1,1,7,5,7,3,1,1,1,16,16,1,1,1,1,13,48,
  13,16,1,1,1,1,13,13,48,1,1,1,1,48,13,13,7,3,7,1,1,1,
  7,5,1,1,1,7,1,1,1,1,1,1,16,16,16,1,1,7,7,1,14,1,14,
  1,1,43,278,59,1,59,278,38,1,1,1,1,1,38,1,1,1,1,1,1,1,
  1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,
  1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,55,1,1,1,102,1,1,
  1,1,1,1,1,102,1,1,10,1,1,1,43,1,1,38,59,1,278,59,278,
  1,1,1,38,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,
  16,1,1,76,95,1,229,102,1,1,95,1,1,1,1,62,1,1,1,1,1,1,
  1,1,1,1,1,1,1,1,1,1,1,149,1,129,38,170,1,291,70,1,1,1,
  1,106,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,26,1,1,1,70,192,
  1,1,1,1,1,1,81,1,1,70,1,192,1,1,1,1,1,81,1,43,1,1,1,
  16,110,1,1,1,1,1,1,1,1,38,16,1,1,1,1,1,26,74,1,1,1,1,
  1,1,26,74,1,1,1,1,1,26,1,1,1,1,1,1,60,60,43,1,1,1,1,
  1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,16,1,1,1,1,1,1,1,1,1,
  1,1,1,1,1,1,1,1,1,1,1,1,1,7,7,14,14,1,14,19,1,19,1,
  16,1,29,29,16,1,1,1,29,1,29,1,14,1,14,1,43,1,1,38,1,1,
  1,38,278,59,1,1,59,1,278,14,14,14,1,10,1,1,26,26,1,1,
  16,26,26,1,1,16,1,1,10,10,10,1,1,19,99,99,1,32,1,99,99,
  1,1,1,1,32,10,10,61,1,61,32,1,1,99,99,1,32,1,1,99,1,
  99,10,61,1,61,1,32,1,99,99,32,1,1,1,99,1,99,19,1,1,99,
  99,1,1,32,99,99,1,1,32,1,1,1,16,1,1,1,31,1,1,1,1,1,1,
  1,31,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,31,1,1,1,1,
  31,1,1,16,1,1,1,11,38,1,1,1,1,1,1,1,1,38,11,1,1,1,1,
  1,1,1,1,1,1,1,1,1,1,16,1,1,1,1,11,1,38,1,1,1,1,38,1,
  11,16,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,16,1,1,1,
  1,43,1,1,1,1,1,1,16,1,1,1,16,1,1,1,1,1,1,1,1,1,1,1,
  1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,
  1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,
  1,1,26,1,1,1,1,81,1,1,1,1,1,81,1,1,1,1,1,1,1,1,1,1,
  81,1,1,1,81,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,
  1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,
  1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,10,1,1,1,9,14,1,1,
  1,1,1,1,1,1,14,9,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,
  1,1,1,1,1,1,1,1,1,1,1,1,10,1,1,1,1,1,1,1,1,1,1,1,1,
  1,1,1,1,1,1,1,1,1,10,1,1,1,48,48,1,1,26,59,1,59,26,1,
  1,1,65,65,289,10,10,1,1,19,19,1,1,16,29,1,29,16,1,1,1,
  29,29,1,16,1,16,1,80,24,74,16,100,36,1,20,1,1,74,1,100,
  36,1,16,23,1,23,43,1,97,38,38,1,43,43,43,39,1,39,1,1,1,
  1,1,1,1,1,1,1,1,1,1,1,1,1,24,80,1,74,16,100,1,36,20,
  1,1,74,36,100,1,10,10,1,1,17,17,1,1,1,1,1,1,17,17,10,
  1,1,1,17,17,1,1,1,1,17,1,17,1,1,17,1,17,1,1,1,1,17,
  17,1,7,1,1,1,5,7,7,3,1,1,1,1,1,1,5,5,5,7,1,1,1,10,
  10,1,1,1,1,8,27,8,10,1,1,1,1,8,8,27,1,1,1,1,27,8,8,
  7,7,5,1,1,1,3,7,1,1,1,7,1,1,1,1,1,1,10,10,10,1,1,1,
  1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,16,1,1,149,1,38,
  129,106,1,1,1,70,1,1,1,170,291,1,1,1,1,1,1,1,1,1,1,1,
  1,1,1,1,1,1,1,76,1,1,1,95,1,1,62,95,1,1,1,102,229,1,
  1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,26,1,1,1,81,1,1,1,1,
  1,1,1,70,192,1,1,1,1,1,81,1,1,1,1,70,1,192,1,1,1,1,1,
  1,1,1,1,1,1,1,43,1,1,1,16,38,1,1,1,1,1,1,1,1,110,16,
  1,1,1,1,1,1,16,1,1,1,1,26,1,60,1,1,1,1,1,1,1,1,1,1,
  1,1,1,26,60,43,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,26,
  74,74,16,16,1,1,29,1,38,38,39,1,1,70,70,1,1,1,39,1,16,
  1,36,1,1,31,1,149,1,52,1,31,1,1,31,1,1,1,36,1,1,1,31,
  1,1,31,31,1,1,1,52,149,1,1,1,1,1,1,1,1,1,1,1,1,1,1,
  1,1,16,1,1,1,31,1,1,1,1,1,1,1,31,1,1,1,1,1,1,31,1,1,
  1,1,31,1,1,1,1,1,1,1,1,1,1,1,1,1,1,16,1,1,1,11,38,1,
  1,1,1,1,1,1,1,38,11,1,1,1,1,1,1,16,1,1,1,1,11,38,1,
  1,1,1,1,1,1,1,1,1,1,1,1,38,11,1,16,1,1,1,1,1,1,1,1,
  1,1,1,1,1,1,1,1,1,16,1,1,7,7,1,1,43,59,1,278,59,1,1,
  38,1,278,1,1,38,7,1,1,43,1,59,1,38,1,38,1,1,278,1,278,
  59,1,26,1,65,65,26,1,1,59,59,1,289,1,26,1,1,1,1,1,1,1,
  1,1,1,1,1,1,1,1,43,1,1,1,81,1,1,1,1,1,1,1,81,1,1,70,
  1,1,1,1,1,1,1,70,1,26,1,1,1,26,149,1,74,1,1,1,1,1,1,
  149,26,1,74,1,1,1,43,1,1,1,1,1,1,110,16,1,1,1,1,1,1,
  16,38,1,1,1,1,1,1,1,16,1,60,1,1,1,1,60,1,1,1,1,1,1,
  1,1,1,1,1,1,43,7,48,1,48,1,1,1,81,81,1,1,26,26,81,1,
  81,43,1,1,59,1,278,1,59,1,38,1,278,38,1,1,1,26,1,1,192,
  70,1,1,1,1,1,1,1,81,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,
  192,1,70,1,1,1,1,81,1,1,1,1,1,1,1,38,16,43,1,1,1,1,1,
  1,1,1,1,1,1,1,1,74,74,1,1,1,1,60,1,60,26,1,1,1,1,26,
  1,1,1,1,1,1,1,1,26,1,1,1,1,1,1,43,16,1,1,1,110,1,1,
  1,1,1,1,1,16,1,43,1,1,1,59,1,1,38,38,1,1,278,59,278,1,
  1,26,1,1,1,81,1,1,1,1,1,1,192,70,1,1,1,1,1,1,1,1,1,
  1,1,1,1,1,1,1,1,81,1,1,1,1,70,192,1,1,1,1,1,1,1,1,1,
  1,1,1,1,1,1,38,1,16,43,1,1,1,1,1,1,1,1,1,1,1,1,16,1,
  1,1,1,26,60,1,1,1,1,1,1,60,26,1,43,16,1,1,1,1,1,1,1,
  1,110,1,1,1,1,1,1,74,26,74,43,1,1,1,70,1,1,1,1,1,1,1,
  70,1,1,1,1,1,1,81,1,1,1,1,81,1,1,1,1,1,1,1,1,1,1,1,
  1,1,1,26,1,1,1,26,1,1,74,1,1,1,1,1,1,1,26,1,74,1,1,
  1,1,43,1,1,1,1,16,110,1,1,1,1,1,1,1,1,1,1,1,1,1,38,
  16,1,16,1,60,1,1,1,1,60,1,1,1,1,1,1,1,1,1,1,43,1,1,
  7,1,1,1,16,1,1,1,1,1,1,1,16,1,1,16,1,1,1,1,1,1,1,16,
  1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,10,1,1,
  1,1,1,1,14,9,1,1,1,1,1,1,9,14,1,1,1,1,1,1,1,1,1,1,
  1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,10,1,1,1,1,16,1,
  1,1,1,16,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,
  1,1,1,1,1,1,1,1,1,1,1,10,1,1,1,1,9,1,14,1,1,1,1,14,
  1,9,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,10,1,1,1,
  1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,10,1,1,1,1,
  9,14,1,1,1,1,1,1,1,1,1,1,1,1,1,14,9,1,1,1,1,1,1,1,
  1,1,1,1,1,1,1,1,1,1,1,1,10,1,1,1,1,1,7,7,5,7,1,1,1,
  1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,7,
  5,5,1,1,1,7,7,1,1,1,7,1,1,1,1,1,1,1,1,1,1,1,1,1,1,
  1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,7,7,7,1,1,1,
  5,5,1,1,1,7,1,1,1,1,1,1,1,1,1,7,1,1,1,1,5,7,7,1,1,
  1,1,7,7,5,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,7,5,
  7,1,1,1,1,7,5,7,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,
  1,5,7,7,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,7,7,5,
  7,5,1,1,1,7,1,1,1,1,1,1,1,1,1,7,1,1,1,1,1,1,1,1,1,
  1,1,1,1,1,1,1,1,1,7 ],

sizes := List(~.sizespos,i->~.sizesset[i]),

stabilize_digitsum_base2_mod2 :=
[ 10796,10798,10816,10822,11032,11050,11056,11161,11167,11986,
  14753,14771,14777,14882,14888,15707,21612,21618,22437,25462 ],

stabilize_digitsum_base2_mod3 := 
[ 10816,10822,11986,15707 ],

stabilize_digitsum_base3_mod2 :=
[ 6764,6771,6773,6776,6778,6781,6783,6788,6791,6798,6802,6951,
  6953,6956,6958,6961,6963,6968,6971,6978,6982,7338,7341,7343,
  7346,7348,7353,7356,7363,7367,7442,7444,7447,7449,7454,7457,
  7464,7468,7588,7591,7593,7598,7601,7608,7612,7682,7684,7689,
  7692,7699,7703,7813,7818,7821,7828,7832,7899,7902,7909,7913,
  8087,8094,8098,8193,8197,8393,12811,12813,12816,12818,12821,
  12823,12828,12831,12838,12842,13198,13201,13203,13206,13208,
  13213,13216,13223,13227,13302,13304,13307,13309,13314,13317,
  13324,13328,13448,13451,13453,13458,13461,13468,13472,13542,
  13544,13549,13552,13559,13563,13673,13678,13681,13688,13692,
  13759,13762,13769,13773,13947,13954,13958,14053,14057,14253,
  18509,18512,18514,18517,18519,18524,18527,18534,18538,18613,
  18615,18618,18620,18625,18628,18635,18639,18759,18762,18764,
  18769,18772,18779,18783,18853,18855,18860,18863,18870,18874,
  18984,18989,18992,18999,19003,19070,19073,19080,19084,19258,
  19265,19269,19364,19368,19564,29022,29024,29027,29029,29034,
  29037,29044,29048,29168,29171,29173,29178,29181,29188,29192,
  29262,29264,29269,29272,29279,29283,29393,29398,29401,29408,
  29412,29479,29482,29489,29493,29667,29674,29678,29773,29777,
  29973,31668,31671,31673,31678,31681,31688,31692,31762,31764,
  31769,31772,31779,31783,31893,31898,31901,31908,31912,31979,
  31982,31989,31993,32167,32174,32178,32273,32277,32473,35147,
  35149,35154,35157,35164,35168,35278,35283,35286,35293,35297,
  35364,35367,35374,35378,35552,35559,35563,35658,35662,35858,
  37303,37308,37311,37318,37322,37389,37392,37399,37403,37577,
  37584,37588,37683,37687,37883,40099,40102,40109,40113,40287,
  40294,40298,40393,40397,40593,41887,41894,41898,41993,41997,
  42193,45328,45332,45528,47113 ],

freeproductcandidates :=
[ 14962,15870,20352,26912,26913,26916,27231,27527,28923,33592, 
  33874,33916,43904,44207,44547 ],

freeproductlikes :=
[ 421,437,1565,1585,1916,1922,2604,2631,2647,2648,2651,3796,3812,
  4104,4105,4423,8873,8888,8889,8892,10017,10018,10037,10052,
  10368,10371,14892,14908,14912,14946,14961,14965,15431,15446,
  15447,15450,15848,15869,15986,16019,16041,16164,16306,16327,
  16330,16341,16352,20334,20362,26897,27228,33582,33925,38457,
  38472,38473,38476,38791,39686,44191,44261,44262,44265,44550,
  49073,49371,49392,49457 ],

abc_torsion :=
[[30,10],[38,10],[40,6],[55,6],[56,10],[59,10],[62,10],[67,10],[109,10],
 [111,6],[115,10],[119,6],[126,10],[129,10],[130,10],[131,10],[201,6],[207,6],
 [224,20],[225,20],[233,20],[234,12],[249,12],[251,6],[252,6],[253,6],[254,6],
 [255,6],[259,6],[393,2],[450,6],[452,6],[480,12],[484,20],[490,20],[491,20],
 [495,12],[502,6],[503,6],[504,6],[506,6],[507,6],[508,6],[569,2],[1295,6],
 [1303,6],[1305,6],[1320,6],[1321,6],[1324,6],[1327,6],[1332,6],[1420,6],
 [1430,6],[1441,6],[1445,6],[1448,6],[1449,6],[1452,6],[1457,6],[1583,6],
 [1887,2],[1915,2],[1929,6],[1931,6],[1935,6],[1939,6],[1946,6],[1949,6],
 [1950,6],[1951,6],[1984,6],[2033,2],[2215,2],[2224,2],[2254,2],[2262,2],
 [2268,2],[2270,2],[2436,6],[2444,6],[2462,60],[2465,60],[2503,6],[2505,6],
 [2522,84],[2529,60],[2530,60],[2714,60],[2719,60],[3506,6],[3514,6],[3516,6],
 [3531,6],[3532,6],[3535,6],[3538,6],[3543,6],[3794,2],[3826,6],[3828,6],
 [3831,6],[3845,6],[3849,6],[3852,6],[3853,6],[3857,6],[3930,2],[4126,8],
 [4426,10],[4462,10],[4479,10],[4611,60],[4615,60],[4651,6],[4654,84],[4665,6],
 [4672,60],[4676,60],[4840,6],[4846,6],[4860,60],[4862,60],[5701,6],[5704,6],
 [5715,6],[5718,6],[5720,6],[5722,6],[5726,6],[5730,6],[6183,2],[6285,6],
 [6287,6],[6291,6],[6295,6],[6302,6],[6305,6],[6306,6],[6307,6],[6340,2],
 [6389,8],[6593,10],[6610,10],[6626,10],[6785,14],[6808,14],[6983,14],[6986,14],
 [7815,6],[7838,6],[7996,2],[8394,6],[8397,6],[8507,2],[8845,4],[9747,20],
 [9755,20],[9773,12],[9776,12],[9872,20],[9882,20],[9897,20],[9900,12],[9901,12],
 [10035,14],[10398,12],[10402,12],[10667,4],[10676,4],[10706,4],[11238,660],
 [11283,660],[11396,660],[11528,660],[11750,20],[11753,20],[11764,20],[11771,12],
 [11775,12],[12020,20],[12022,20],[12039,20],[12046,12],[12047,12],[12124,14],
 [12232,14],[12642,4],[12656,4],[13675,6],[13698,6],[13856,8],[13949,6],[15421,12],
 [15429,4],[15709,30],[15726,60],[15730,12],[15741,4],[15743,12],[15845,30],
 [15925,60],[15932,12],[16465,4],[17223,12],[17228,12],[17326,20],[17337,20],
 [17341,20],[17348,12],[17353,12],[17825,20],[17831,20],[17845,12],[17847,12],
 [17880,14],[18158,4],[18164,4],[18166,4],[19074,6],[19565,6],[19568,6],[19678,8],
 [21189,12],[21195,4],[21244,30],[21248,60],[21260,12],[21944,924],[22072,924],
 [22381,20],[22383,12],[22393,12],[22859,20],[22878,12],[22880,12],[23089,4],
 [23130,4],[23741,12],[23755,4],[24212,60],[24223,30],[24234,12],[24325,4],
 [24331,12],[24380,30],[24425,60],[24517,12],[25176,4],[25184,12],[25464,30],
 [25726,60],[25742,12],[27565,420],[27573,420],[33908,120],[33936,120],[35034,420],
 [35065,420],[37174,120],[37202,120],[38455,4],[38787,4],[39087,4],[39140,4],
 [39602,4],[40012,4],[40196,4],[41343,4],[41680,4],[41718,4],[43210,4],[44695,4],
 [45227,4],[49500,4],[49549,4],[49770,4],[49786,4],[49887,4],[50619,4]],

#############################################################################
##
#S  Data included from other files. /////////////////////////////////////////
##
#############################################################################

equalityclasses :=
  ReadAsFunction(Concatenation(PackageInfo("rcwa")[1].InstallationPath,
                               "/data/3ctsgroups6/equalityclasses.g"))(),

samegroups :=
  ReadAsFunction(Concatenation(PackageInfo("rcwa")[1].InstallationPath,
                               "/data/3ctsgroups6/samegroups.g"))(),

conjugacyclasses :=
  ReadAsFunction(Concatenation(PackageInfo("rcwa")[1].InstallationPath,
                               "/data/3ctsgroups6/conjugacyclasses.g"))(),

subgroups :=
  ReadAsFunction(Concatenation(PackageInfo("rcwa")[1].InstallationPath,
                               "/data/3ctsgroups6/subgroups.g"))(),

supergroups :=
  ReadAsFunction(Concatenation(PackageInfo("rcwa")[1].InstallationPath,
                               "/data/3ctsgroups6/supergroups.g"))(),

chains :=
  ReadAsFunction(Concatenation(PackageInfo("rcwa")[1].InstallationPath,
                               "/data/3ctsgroups6/chains.g"))(),

partitionlengths :=
  ReadAsFunction(Concatenation(PackageInfo("rcwa")[1].InstallationPath,
                               "/data/3ctsgroups6/partitionlengths.g"))(),

degrees :=
  ReadAsFunction(Concatenation(PackageInfo("rcwa")[1].InstallationPath,
                               "/data/3ctsgroups6/degrees.g"))(),

orbitlengths :=
  ReadAsFunction(Concatenation(PackageInfo("rcwa")[1].InstallationPath,
                               "/data/3ctsgroups6/orbitlengths.g"))(),

permgroupgens :=
  ReadAsFunction(Concatenation(PackageInfo("rcwa")[1].InstallationPath,
                               "/data/3ctsgroups6/permgroupgens.g"))(),

respectedpartitions :=
  ReadAsFunction(Concatenation(PackageInfo("rcwa")[1].InstallationPath,
                               "/data/3ctsgroups6/respectedpartitions.g"))(),

intransitivemodulo :=
  ReadAsFunction(Concatenation(PackageInfo("rcwa")[1].InstallationPath,
                               "/data/3ctsgroups6/intransitivemodulo.g"))(),

finiteorbits := ReadAsFunction(
                  Concatenation(PackageInfo("rcwa")[1].InstallationPath,
                                "/data/3ctsgroups6/finite-orbits.g"))(),

shortresidueclassorbitlengths := 
  ReadAsFunction(Concatenation(PackageInfo("rcwa")[1].InstallationPath,
                               "/data/3ctsgroups6/residueclassorbitlengths.g"))(),

cyclist := 
  ReadAsFunction(Concatenation(PackageInfo("rcwa")[1].InstallationPath,
                               "/data/3ctsgroups6/spheresizecycles.g"))(),

trsstatusset := 
[ "> 1 orbit (mod m)", "Mod(U DecreasingOn) exceeded <maxmod>", 
  "U DecreasingOn stable for <maxeq> steps", "exceeded memory bound", 
  "finite", "transitive up to one finite orbit", "transitive", 
  "transitivity on small points unclear", 
  "intransitive, but finitely many orbits", 
  "seemingly only finite orbits (very short)", 
  "seemingly only finite orbits (short)", 
  "seemingly only finite orbits (medium)", 
  "seemingly only finite orbits (long)", 
  "seemingly only finite orbits (very long)", 
  "there are infinite orbits which have linear sphere size growth", 
  "there are infinite orbits which have exponential sphere size growth",
  "U DecreasingOn stable and exceeded memory bound",
  "there are infinite orbits which have unclear sphere size growth",
  "seemingly only finite orbits (very long, very unclear)",
  "there are infinite orbits which have bounded sphere sizes",
  "there are infinite orbits which have quadratic sphere size growth" ],

trsstatuspos := 
  ReadAsFunction(Concatenation(PackageInfo("rcwa")[1].InstallationPath,
                               "/data/3ctsgroups6/trsstatuspos.g"))(),

trsstatus := List(~.trsstatuspos,i->~.trsstatusset[i]),

orbitgrowthtype := 
  ReadAsFunction(Concatenation(PackageInfo("rcwa")[1].InstallationPath,
                               "/data/3ctsgroups6/orbitgrowthtype.g"))()
);

#############################################################################
##
#E  database.g . . . . . . . . . . . . . . . . . . . . . . . . . .  ends here
