package vtk;

import java.io.File;

/**
 * Enum used to load native library more easily. If you don't want to set the
 * specific environment variable you can provide the path of the directory that
 * contains the VTK library through a Java Property. Like in the following
 * command line:
 *
 * > java -cp vtk.jar -Dvtk.lib.dir=.../vtk-libs vtk.sample.SimpleVTK
 *
 * The directory .../vtk-libs must contain the so/dll/dylib + the jnilib files
 *
 * @author sebastien jourdain - sebastien.jourdain@kitware.com
 */
public enum vtkNativeLibrary {

    VTKINFOVISLAYOUT("vtkInfovisLayoutJava", true),
    VTKINFOVISBOOSTGRAPHALGORITHMS("vtkInfovisBoostGraphAlgorithmsJava", true),
    VTKINFOVISPARALLEL("vtkInfovisParallelJava", false),
    VTKINFOVISCORE("vtkInfovisCoreJava", true),
    VTKCHARTSCORE("vtkChartsCoreJava", true),
    VTKIMAGINGGENERAL("vtkImagingGeneralJava", true),
    VTKIMAGINGMORPHOLOGICAL("vtkImagingMorphologicalJava", true),
    VTKIMAGINGCOLOR("vtkImagingColorJava", true),
    VTKIMAGINGMATH("vtkImagingMathJava", true),
    VTKIMAGINGSTENCIL("vtkImagingStencilJava", true),
    VTKIMAGINGSOURCES("vtkImagingSourcesJava", true),
    VTKIMAGINGFOURIER("vtkImagingFourierJava", true),
    VTKIMAGINGSTATISTICS("vtkImagingStatisticsJava", true),
    VTKIMAGINGHYBRID("vtkImagingHybridJava", true),
    VTKIMAGINGCORE("vtkImagingCoreJava", true),
    VTKWEBGLEXPORTER("vtkWebGLExporterJava", false),
    VTKWEBCORE("vtkWebCoreJava", false),
    VTKFILTERSPARALLELIMAGING("vtkFiltersParallelImagingJava", true),
    VTKFILTERSFLOWPATHS("vtkFiltersFlowPathsJava", true),
    VTKFILTERSAMR("vtkFiltersAMRJava", true),
    VTKFILTERSREEBGRAPH("vtkFiltersReebGraphJava", true),
    VTKFILTERSVERDICT("vtkFiltersVerdictJava", true),
    VTKFILTERSSELECTION("vtkFiltersSelectionJava", true),
    VTKFILTERSGENERIC("vtkFiltersGenericJava", true),
    VTKFILTERSIMAGING("vtkFiltersImagingJava", true),
    VTKFILTERSGENERAL("vtkFiltersGeneralJava", true),
    VTKFILTERSPARALLELFLOWPATHS("vtkFiltersParallelFlowPathsJava", true),
    VTKFILTERSPARALLELSTATISTICS("vtkFiltersParallelStatisticsJava", true),
    VTKFILTERSMODELING("vtkFiltersModelingJava", true),
    VTKFILTERSGEOMETRY("vtkFiltersGeometryJava", true),
    VTKFILTERSPYTHON("vtkFiltersPythonJava", true),
    VTKFILTERSPARALLEL("vtkFiltersParallelJava", true),
    VTKFILTERSSTATISTICSGNUR("vtkFiltersStatisticsGnuRJava", false),
    VTKFILTERSSOURCES("vtkFiltersSourcesJava", true),
    VTKFILTERSTEXTURE("vtkFiltersTextureJava", true),
    VTKFILTERSSTATISTICS("vtkFiltersStatisticsJava", true),
    VTKFILTERSHYPERTREE("vtkFiltersHyperTreeJava", true),
    VTKFILTERSEXTRACTION("vtkFiltersExtractionJava", true),
    VTKFILTERSPARALLELGEOMETRY("vtkFiltersParallelGeometryJava", true),
    VTKFILTERSHYBRID("vtkFiltersHybridJava", true),
    VTKFILTERSPARALLELMPI("vtkFiltersParallelMPIJava", true),
    VTKFILTERSPROGRAMMABLE("vtkFiltersProgrammableJava", true),
    VTKFILTERSCORE("vtkFiltersCoreJava", true),
    VTKFILTERSSMP("vtkFiltersSMPJava", true),
    VTKPYTHONINTERPRETER("vtkPythonInterpreterJava", true),
    VTKRENDERINGCONTEXTOPENGL2("vtkRenderingContextOpenGL2Java", false),
    VTKRENDERINGVOLUMEAMR("vtkRenderingVolumeAMRJava", true),
    VTKRENDERINGIMAGE("vtkRenderingImageJava", true),
    VTKRENDERINGVOLUMEOPENGL("vtkRenderingVolumeOpenGLJava", true),
    VTKRENDERINGEXTERNAL("vtkRenderingExternalJava", false),
    VTKRENDERINGVOLUME("vtkRenderingVolumeJava", true),
    VTKRENDERINGLABEL("vtkRenderingLabelJava", true),
    VTKRENDERINGOPENGL2("vtkRenderingOpenGL2Java", false),
    VTKRENDERINGVOLUMEOPENGL2("vtkRenderingVolumeOpenGL2Java", false),
    VTKRENDERINGOPENGL("vtkRenderingOpenGLJava", true),
    VTKRENDERINGLICOPENGL2("vtkRenderingLICOpenGL2Java", false),
    VTKRENDERINGCONTEXT2D("vtkRenderingContext2DJava", true),
    VTKRENDERINGMATPLOTLIB("vtkRenderingMatplotlibJava", true),
    VTKRENDERINGANNOTATION("vtkRenderingAnnotationJava", true),
    VTKRENDERINGPARALLEL("vtkRenderingParallelJava", true),
    VTKRENDERINGLIC("vtkRenderingLICJava", true),
    VTKRENDERINGQT("vtkRenderingQtJava", true),
    VTKRENDERINGLOD("vtkRenderingLODJava", true),
    VTKRENDERINGGL2PS("vtkRenderingGL2PSJava", true),
    VTKRENDERINGFREETYPE("vtkRenderingFreeTypeJava", true),
    VTKRENDERINGCONTEXTOPENGL("vtkRenderingContextOpenGLJava", true),
    VTKRENDERINGPARALLELLIC("vtkRenderingParallelLICJava", true),
    VTKRENDERINGCORE("vtkRenderingCoreJava", true),
    VTKIOAMR("vtkIOAMRJava", true),
    VTKIOINFOVIS("vtkIOInfovisJava", true),
    VTKIOMINC("vtkIOMINCJava", true),
    VTKIOPOSTGRESQL("vtkIOPostgreSQLJava", true),
    VTKIOIMAGE("vtkIOImageJava", true),
    VTKIONETCDF("vtkIONetCDFJava", true),
    VTKIOMYSQL("vtkIOMySQLJava", true),
    VTKIOMOVIE("vtkIOMovieJava", true),
    VTKIOPARALLELXML("vtkIOParallelXMLJava", true),
    VTKIOXDMF3("vtkIOXdmf3Java", true),
    VTKIOPLY("vtkIOPLYJava", true),
    VTKIOMPIPARALLEL("vtkIOMPIParallelJava", true),
    VTKIOADIOS("vtkIOADIOSJava", false),
    VTKIOVIDEO("vtkIOVideoJava", true),
    VTKIOXMLPARSER("vtkIOXMLParserJava", true),
    VTKIOXML("vtkIOXMLJava", true),
    VTKIOPARALLELEXODUS("vtkIOParallelExodusJava", true),
    VTKIOGEOMETRY("vtkIOGeometryJava", true),
    VTKIOMPIIMAGE("vtkIOMPIImageJava", true),
    VTKIOIMPORT("vtkIOImportJava", true),
    VTKIOODBC("vtkIOODBCJava", true),
    VTKIOPARALLEL("vtkIOParallelJava", true),
    VTKIOEXODUS("vtkIOExodusJava", true),
    VTKIOVPIC("vtkIOVPICJava", true),
    VTKIOFFMPEG("vtkIOFFMPEGJava", true),
    VTKIOGDAL("vtkIOGDALJava", true),
    VTKIOPARALLELLSDYNA("vtkIOParallelLSDynaJava", true),
    VTKIOLEGACY("vtkIOLegacyJava", true),
    VTKIOLSDYNA("vtkIOLSDynaJava", true),
    VTKIOGEOJSON("vtkIOGeoJSONJava", true),
    VTKIOSQL("vtkIOSQLJava", true),
    VTKIOENSIGHT("vtkIOEnSightJava", true),
    VTKIOPARALLELNETCDF("vtkIOParallelNetCDFJava", true),
    VTKIOEXPORT("vtkIOExportJava", true),
    VTKIOXDMF2("vtkIOXdmf2Java", true),
    VTKIOCORE("vtkIOCoreJava", true),
    VTKCOMMONCOMPUTATIONALGEOMETRY("vtkCommonComputationalGeometryJava", true),
    VTKCOMMONSYSTEM("vtkCommonSystemJava", true),
    VTKCOMMONTRANSFORMS("vtkCommonTransformsJava", true),
    VTKCOMMONCOLOR("vtkCommonColorJava", true),
    VTKCOMMONEXECUTIONMODEL("vtkCommonExecutionModelJava", true),
    VTKCOMMONDATAMODEL("vtkCommonDataModelJava", true),
    VTKCOMMONMATH("vtkCommonMathJava", true),
    VTKCOMMONMISC("vtkCommonMiscJava", true),
    VTKCOMMONCORE("vtkCommonCoreJava", true),
    VTKDOMAINSCHEMISTRY("vtkDomainsChemistryJava", true),
    VTKDOMAINSCHEMISTRYOPENGL2("vtkDomainsChemistryOpenGL2Java", false),
    VTKGEOVISCORE("vtkGeovisCoreJava", true),
    VTKPARALLELMPI("vtkParallelMPIJava", true),
    VTKPARALLELMPI4PY("vtkParallelMPI4PyJava", true),
    VTKPARALLELCORE("vtkParallelCoreJava", true),
    VTKINTERACTIONIMAGE("vtkInteractionImageJava", true),
    VTKINTERACTIONWIDGETS("vtkInteractionWidgetsJava", true),
    VTKINTERACTIONSTYLE("vtkInteractionStyleJava", true),
    VTKVIEWSINFOVIS("vtkViewsInfovisJava", true),
    VTKVIEWSCONTEXT2D("vtkViewsContext2DJava", true),
    VTKVIEWSGEOVIS("vtkViewsGeovisJava", true),
    VTKVIEWSCORE("vtkViewsCoreJava", true),
    VTKACCELERATORSPISTON("vtkAcceleratorsPistonJava", false),
    VTKACCELERATORSDAX("vtkAcceleratorsDaxJava", false),
    VTKTESTINGRENDERING("vtkTestingRenderingJava", true);


  /**
   * Try to load all library
   *
   * @return true if all library have been successfully loaded
   */
  public static boolean LoadAllNativeLibraries() {
    boolean isEveryThingLoaded = true;
    for (vtkNativeLibrary lib : values()) {
      try {
        if(lib.IsBuilt()) {
          lib.LoadLibrary();
        }
      } catch (UnsatisfiedLinkError e) {
        isEveryThingLoaded = false;
        e.printStackTrace();
      }
    }

    return isEveryThingLoaded;
  }

  /**
   * Load the set of given library and trows runtime exception if any given
   * library failed in the loading process
   *
   * @param nativeLibraries
   */
  public static void LoadNativeLibraries(vtkNativeLibrary... nativeLibraries) {
    for (vtkNativeLibrary lib : nativeLibraries) {
      lib.LoadLibrary();
    }
  }

  /**
   * Disable the pop-in vtkErrorWindow by writing the error to a log file.
   * If the provided logFile is null the default "vtkError.txt" file will be
   * used.
   *
   * @param logFile
   */
  public static void DisableOutputWindow(File logFile) {
    if(logFile == null) {
      logFile = new File("vtkError.txt");
    }
    vtkFileOutputWindow outputError = new vtkFileOutputWindow();
    outputError.SetFileName(logFile.getAbsolutePath());
    outputError.SetInstance(outputError);
  }

  private vtkNativeLibrary(String nativeLibraryName, boolean built) {
    this.nativeLibraryName = nativeLibraryName;
    this.loaded = false;
    this.built = built;
  }

  /**
   * Load the library and throws runtime exception if the library failed in
   * the loading process
   */
  public void LoadLibrary() throws UnsatisfiedLinkError {
    if (!loaded) {
      if (System.getProperty("vtk.lib.dir") != null) {
        File dir = new File(System.getProperty("vtk.lib.dir"));
        patchJavaLibraryPath(dir.getAbsolutePath());
        File libPath = new File(dir, System.mapLibraryName(nativeLibraryName));
        if (libPath.exists()) {
          try {
            Runtime.getRuntime().load(libPath.getAbsolutePath());
            loaded = true;
            return;
          } catch (UnsatisfiedLinkError e) {
            e.printStackTrace();
          }
        }
      }
      System.loadLibrary(nativeLibraryName);
    }
    loaded = true;
  }

  /**
   * @return true if the library has already been successfully loaded
   */
  public boolean IsLoaded() {
    return loaded;
  }

  /**
   * @return true if the module was enabled and therefore build
   */
  public boolean IsBuilt() {
    return built;
  }

  /**
   * @return the library name
   */
  public String GetLibraryName() {
    return nativeLibraryName;
  }

  private static void patchJavaLibraryPath(String vtkLibDir) {
    if (vtkLibDir != null) {
      String path_separator = System.getProperty("path.separator");
      String s = System.getProperty("java.library.path");
      if (!s.contains(vtkLibDir)) {
        s = s + path_separator + vtkLibDir;
        System.setProperty("java.library.path", s);
      }
    }
  }

  private String nativeLibraryName;
  private boolean loaded;
  private boolean built;
}
