/*
 * Decompiled with CFR 0.152.
 */
package com.oracle.truffle.tools;

import com.oracle.truffle.api.CompilerDirectives;
import com.oracle.truffle.api.frame.FrameDescriptor;
import com.oracle.truffle.api.frame.FrameSlot;
import com.oracle.truffle.api.frame.FrameSlotKind;
import com.oracle.truffle.api.frame.FrameSlotTypeException;
import com.oracle.truffle.api.frame.VirtualFrame;
import com.oracle.truffle.api.instrumentation.EventBinding;
import com.oracle.truffle.api.instrumentation.EventContext;
import com.oracle.truffle.api.instrumentation.ExecutionEventNode;
import com.oracle.truffle.api.instrumentation.ExecutionEventNodeFactory;
import com.oracle.truffle.api.instrumentation.Instrumenter;
import com.oracle.truffle.api.instrumentation.SourceSectionFilter;
import com.oracle.truffle.api.instrumentation.StandardTags;
import com.oracle.truffle.api.nodes.NodeCost;
import com.oracle.truffle.api.nodes.RootNode;
import com.oracle.truffle.api.profiles.ConditionProfile;
import com.oracle.truffle.api.source.Source;
import com.oracle.truffle.api.source.SourceSection;
import com.oracle.truffle.api.vm.PolyglotEngine;
import com.oracle.truffle.tools.ProfilerInstrument;
import java.io.PrintStream;
import java.util.ArrayList;
import java.util.Arrays;
import java.util.Collections;
import java.util.Comparator;
import java.util.HashMap;
import java.util.List;
import java.util.Map;

public final class Profiler {
    private static final ProfilerInstrument.Factory FACTORY = new ProfilerInstrument.Factory(){

        @Override
        public Profiler create(Instrumenter instrumenter) {
            return new Profiler(instrumenter);
        }
    };
    private final Instrumenter instrumenter;
    private boolean isCollecting;
    private boolean isTiming;
    private String[] mimeTypes = Profiler.parseMimeTypes(System.getProperty("truffle.profiling.includeMimeTypes"));
    private EventBinding binding;
    private final Map<SourceSection, Counter> counters = new HashMap<SourceSection, Counter>();
    private final SourceSectionFilter.SourcePredicate notInternal = new SourceSectionFilter.SourcePredicate(){

        @Override
        public boolean test(Source source) {
            return !source.isInternal();
        }
    };
    private Counter activeCounter;
    private boolean disposed;

    public static Profiler find(PolyglotEngine engine) {
        return Profiler.find(engine, true);
    }

    static Profiler find(PolyglotEngine engine, boolean create) {
        ProfilerInstrument profilerInstrument;
        PolyglotEngine.Instrument instrument = engine.getInstruments().get("profiler");
        if (instrument == null) {
            throw new IllegalStateException();
        }
        if (create) {
            instrument.setEnabled(true);
        }
        if ((profilerInstrument = instrument.lookup(ProfilerInstrument.class)) == null) {
            return null;
        }
        return profilerInstrument.getProfiler(create ? FACTORY : null);
    }

    Profiler(Instrumenter instrumenter) {
        this.instrumenter = instrumenter;
    }

    void dispose() {
        if (!this.disposed) {
            this.counters.clear();
            this.binding = null;
            this.disposed = true;
        }
    }

    public void setCollecting(boolean isCollecting) {
        if (this.disposed) {
            throw new IllegalStateException("disposed profiler");
        }
        if (this.isCollecting != isCollecting) {
            this.isCollecting = isCollecting;
            this.reset();
        }
    }

    public boolean isCollecting() {
        if (this.disposed) {
            throw new IllegalStateException("disposed profiler");
        }
        return this.isCollecting;
    }

    public void setTiming(boolean isTiming) {
        if (this.disposed) {
            throw new IllegalStateException("disposed profiler");
        }
        if (this.isTiming != isTiming) {
            this.isTiming = isTiming;
            this.reset();
        }
    }

    public boolean isTiming() {
        if (this.disposed) {
            throw new IllegalStateException("disposed profiler");
        }
        return this.isTiming;
    }

    public void setMimeTypes(String[] newTypes) {
        if (this.disposed) {
            throw new IllegalStateException("disposed profiler");
        }
        this.mimeTypes = newTypes != null && newTypes.length > 0 ? newTypes : null;
        this.reset();
    }

    public String[] getMimeTypes() {
        if (this.disposed) {
            throw new IllegalStateException("disposed profiler");
        }
        return this.mimeTypes == null ? null : Arrays.copyOf(this.mimeTypes, this.mimeTypes.length);
    }

    public boolean hasData() {
        if (this.disposed) {
            throw new IllegalStateException("disposed profiler");
        }
        for (Counter counter : this.counters.values()) {
            if (counter.getInvocations(Counter.TimeKind.INTERPRETED_AND_COMPILED) <= 0L) continue;
            return true;
        }
        return false;
    }

    public void clearData() {
        if (this.disposed) {
            throw new IllegalStateException("disposed profiler");
        }
        for (Counter counter : this.counters.values()) {
            counter.clear();
        }
    }

    public Map<SourceSection, Counter> getCounters() {
        if (this.disposed) {
            throw new IllegalStateException("disposed profiler");
        }
        return Collections.unmodifiableMap(this.counters);
    }

    private void reset() {
        if (this.binding != null) {
            this.binding.dispose();
            this.binding = null;
        }
        if (this.isCollecting) {
            SourceSectionFilter.Builder filterBuilder = SourceSectionFilter.newBuilder();
            if (this.mimeTypes != null) {
                filterBuilder.mimeTypeIs(this.mimeTypes);
            }
            SourceSectionFilter filter = filterBuilder.tagIs(StandardTags.RootTag.class).sourceIs(this.notInternal).build();
            this.binding = this.instrumenter.attachFactory(filter, new ExecutionEventNodeFactory(){

                @Override
                public ExecutionEventNode create(EventContext context) {
                    return Profiler.this.createCountingNode(context);
                }
            });
        }
    }

    private ExecutionEventNode createCountingNode(EventContext context) {
        SourceSection sourceSection = context.getInstrumentedSourceSection();
        Counter counter = this.counters.get(sourceSection);
        if (counter == null) {
            RootNode rootNode = context.getInstrumentedNode().getRootNode();
            counter = new Counter(sourceSection, rootNode == null ? "<unknown>>" : rootNode.getName());
            this.counters.put(sourceSection, counter);
        }
        if (this.isTiming) {
            return TimedCounterNode.create(this, counter, context);
        }
        return new CounterNode(this, counter);
    }

    public void printHistograms(PrintStream out) {
        if (this.disposed) {
            throw new IllegalStateException("disposed profiler");
        }
        ArrayList<Counter> sortedCounters = new ArrayList<Counter>(this.counters.values());
        boolean hasCompiled = false;
        for (Counter counter : sortedCounters) {
            if (counter.getInvocations(Counter.TimeKind.COMPILED) <= 0L) continue;
            hasCompiled = true;
        }
        if (hasCompiled) {
            this.printHistogram(out, sortedCounters, Counter.TimeKind.INTERPRETED_AND_COMPILED);
            this.printHistogram(out, sortedCounters, Counter.TimeKind.INTERPRETED);
            this.printHistogram(out, sortedCounters, Counter.TimeKind.COMPILED);
        } else {
            this.printHistogram(out, sortedCounters, Counter.TimeKind.INTERPRETED);
        }
    }

    private void printHistogram(PrintStream out, List<Counter> sortedCounters, final Counter.TimeKind time) {
        Collections.sort(sortedCounters, new Comparator<Counter>(){

            @Override
            public int compare(Counter o1, Counter o2) {
                if (Profiler.this.isTiming) {
                    return Long.compare(o2.getSelfTime(time), o1.getSelfTime(time));
                }
                return Long.compare(o2.getInvocations(time), o1.getInvocations(time));
            }
        });
        if (this.isTiming) {
            out.println("Truffle profiler histogram for mode " + (Object)((Object)time));
            out.println(String.format("%12s | %7s | %11s | %7s | %11s | %-15s | %s ", "Invoc", "Total", "PerInvoc", "SelfTime", "PerInvoc", "Name", "Source"));
            for (Counter counter : sortedCounters) {
                long invocations = counter.getInvocations(time);
                if (invocations <= 0L) continue;
                double totalTimems = (double)counter.getTotalTime(time) / 1000000.0;
                double selfTimems = (double)counter.getSelfTime(time) / 1000000.0;
                out.println(String.format("%12d |%6.0fms |%10.3fms |%7.0fms |%10.3fms | %-15s | %s", invocations, totalTimems, totalTimems / (double)invocations, selfTimems, selfTimems / (double)invocations, counter.getName(), Profiler.getShortDescription(counter.getSourceSection())));
            }
        } else {
            out.println("Truffle profiler histogram for mode " + (Object)((Object)time));
            out.println(String.format("%12s | %-15s | %s ", "Invoc", "Name", "Source"));
            for (Counter counter : sortedCounters) {
                long invocations = counter.getInvocations(time);
                if (invocations <= 0L) continue;
                out.println(String.format("%12d | %-15s | %s", invocations, counter.getName(), Profiler.getShortDescription(counter.getSourceSection())));
            }
        }
        out.println();
    }

    private static String getShortDescription(SourceSection sourceSection) {
        if (sourceSection.getSource() == null) {
            return "<Unknown>";
        }
        StringBuilder b = new StringBuilder();
        b.append(sourceSection.getSource().getName());
        b.append(":");
        if (sourceSection.getStartLine() == sourceSection.getEndLine()) {
            b.append(sourceSection.getStartLine());
        } else {
            b.append(sourceSection.getStartLine()).append("-").append(sourceSection.getEndLine());
        }
        return b.toString();
    }

    private static String[] parseMimeTypes(String property) {
        if (property != null) {
            return property.split(";");
        }
        return null;
    }

    public static final class Counter {
        private final SourceSection sourceSection;
        private final String name;
        private long interpretedInvocations;
        private long interpretedChildTime;
        private long interpretedTotalTime;
        private long compiledInvocations;
        private long compiledTotalTime;
        private long compiledChildTime;
        private boolean compiled;

        private Counter(SourceSection sourceSection, String name) {
            this.sourceSection = sourceSection;
            this.name = name;
        }

        private void clear() {
            this.interpretedInvocations = 0L;
            this.interpretedChildTime = 0L;
            this.interpretedTotalTime = 0L;
            this.compiledInvocations = 0L;
            this.compiledTotalTime = 0L;
            this.compiledChildTime = 0L;
        }

        public SourceSection getSourceSection() {
            return this.sourceSection;
        }

        public String getName() {
            return this.name;
        }

        public long getInvocations(TimeKind kind) {
            switch (kind) {
                case INTERPRETED_AND_COMPILED: {
                    return this.interpretedInvocations + this.compiledInvocations;
                }
                case COMPILED: {
                    return this.compiledInvocations;
                }
                case INTERPRETED: {
                    return this.interpretedInvocations;
                }
            }
            throw new AssertionError();
        }

        public long getTotalTime(TimeKind kind) {
            switch (kind) {
                case INTERPRETED_AND_COMPILED: {
                    return this.interpretedTotalTime + this.compiledTotalTime;
                }
                case COMPILED: {
                    return this.compiledTotalTime;
                }
                case INTERPRETED: {
                    return this.interpretedTotalTime;
                }
            }
            throw new AssertionError();
        }

        public long getSelfTime(TimeKind kind) {
            switch (kind) {
                case INTERPRETED_AND_COMPILED: {
                    return this.interpretedTotalTime + this.compiledTotalTime - this.compiledChildTime - this.interpretedChildTime;
                }
                case COMPILED: {
                    return this.compiledTotalTime - this.compiledChildTime;
                }
                case INTERPRETED: {
                    return this.interpretedTotalTime - this.interpretedChildTime;
                }
            }
            throw new AssertionError();
        }

        public static enum TimeKind {
            INTERPRETED_AND_COMPILED,
            INTERPRETED,
            COMPILED;

        }
    }

    private static class CounterNode
    extends ExecutionEventNode {
        protected final Profiler profiler;
        protected final Counter counter;

        CounterNode(Profiler profiler, Counter counter) {
            this.profiler = profiler;
            this.counter = counter;
        }

        @Override
        protected void onEnter(VirtualFrame frame) {
            if (CompilerDirectives.inInterpreter()) {
                this.counter.interpretedInvocations++;
            } else {
                this.counter.compiledInvocations++;
            }
        }

        @Override
        public NodeCost getCost() {
            return NodeCost.NONE;
        }
    }

    private static class TimedCounterNode
    extends CounterNode {
        private final EventContext context;
        private final FrameSlot parentCounterSlot;
        private final FrameSlot timeStartedSlot;
        private final ConditionProfile parentNotNullProfile = ConditionProfile.createBinaryProfile();
        private static final Object KEY_TIME_STARTED = new Object();
        private static final Object KEY_PARENT_COUNTER = new Object();

        TimedCounterNode(Profiler profiler, Counter counter, EventContext context) {
            super(profiler, counter);
            this.context = context;
            FrameDescriptor frameDescriptor = context.getInstrumentedNode().getRootNode().getFrameDescriptor();
            this.timeStartedSlot = frameDescriptor.findOrAddFrameSlot(KEY_TIME_STARTED, "profiler:timeStarted", FrameSlotKind.Long);
            this.parentCounterSlot = frameDescriptor.findOrAddFrameSlot(KEY_PARENT_COUNTER, "profiler:parentCounter", FrameSlotKind.Object);
        }

        @Override
        protected void onDispose(VirtualFrame frame) {
            FrameDescriptor frameDescriptor = this.context.getInstrumentedNode().getRootNode().getFrameDescriptor();
            if (frameDescriptor.getIdentifiers().contains(KEY_TIME_STARTED)) {
                frameDescriptor.removeFrameSlot(KEY_TIME_STARTED);
            }
            if (frameDescriptor.getIdentifiers().contains(KEY_PARENT_COUNTER)) {
                frameDescriptor.removeFrameSlot(KEY_PARENT_COUNTER);
            }
        }

        @Override
        protected void onEnter(VirtualFrame frame) {
            frame.setLong(this.timeStartedSlot, System.nanoTime());
            super.onEnter(frame);
            frame.setObject(this.parentCounterSlot, this.profiler.activeCounter);
            this.profiler.activeCounter = this.counter;
            if (CompilerDirectives.inInterpreter()) {
                this.counter.compiled = false;
            } else {
                this.counter.compiled = true;
            }
        }

        @Override
        protected void onReturnExceptional(VirtualFrame frame, Throwable exception) {
            this.onReturnValue(frame, null);
        }

        @Override
        protected void onReturnValue(VirtualFrame frame, Object result) {
            Counter counter;
            Counter parentCounter;
            long startTime;
            try {
                startTime = frame.getLong(this.timeStartedSlot);
                parentCounter = (Counter)frame.getObject(this.parentCounterSlot);
            }
            catch (FrameSlotTypeException e) {
                throw new AssertionError();
            }
            long timeNano = System.nanoTime() - startTime;
            if (CompilerDirectives.inInterpreter()) {
                counter = this.counter;
                counter.interpretedTotalTime = counter.interpretedTotalTime + timeNano;
            } else {
                counter = this.counter;
                counter.compiledTotalTime = counter.compiledTotalTime + timeNano;
            }
            if (this.parentNotNullProfile.profile(parentCounter != null)) {
                if (parentCounter.compiled) {
                    counter = parentCounter;
                    counter.compiledChildTime = counter.compiledChildTime + timeNano;
                } else {
                    counter = parentCounter;
                    counter.interpretedChildTime = counter.interpretedChildTime + timeNano;
                }
            }
            this.profiler.activeCounter = parentCounter;
        }

        static CounterNode create(Profiler profiler, Counter counter, EventContext context) {
            return new TimedCounterNode(profiler, counter, context);
        }
    }
}

