/*
*   Copyright (C) 2016 by Marco Martin <mart@kde.org>
*
*   This program is free software; you can redistribute it and/or modify
*   it under the terms of the GNU Library General Public License as
*   published by the Free Software Foundation; either version 2, or
*   (at your option) any later version.
*
*   This program is distributed in the hope that it will be useful,
*   but WITHOUT ANY WARRANTY; without even the implied warranty of
*   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
*   GNU Library General Public License for more details
*
*   You should have received a copy of the GNU Library General Public
*   License along with this program; if not, write to the
*   Free Software Foundation, Inc.,
*   51 Franklin Street, Fifth Floor, Boston, MA  2.010-1301, USA.
*/

import QtQuick 2.5
import QtQuick.Controls 1.3 as Controls
import org.kde.kirigami 1.0
import QtGraphicalEffects 1.0
import "private"

/**
 * An overlay sheet that covers the current Page content.
 * Its contents can be scrolled up or down, scrolling all the way up or
 * all the way down, dismisses it.
 * Use this for big, modal dialogs or information display, that can't be
 * logically done as a new separate Page, even if potentially
 * are taller than the screen space.
 */
Item {
    id: root

    z: 999

    anchors.fill: parent
    visible: false

    /**
     * contentItem: Item
     * This property holds the visual content item.
     *
     * Note: The content item is automatically resized inside the
     * padding of the control.
     */
    default property Item contentItem

    /**
     * opened: bool
     * If true the sheet is open showing the contents of the OverlaySheet
     * component.
     */
    property bool opened

    /**
     * leftPadding: int
     * default contents padding at left
     */
    property int leftPadding: Units.gridUnit

    /**
     * topPadding: int
     * default contents padding at top
     */
    property int topPadding: Units.gridUnit

    /**
     * rightPadding: int
     * default contents padding at right
     */
    property int rightPadding: Units.gridUnit

    /**
     * bottomPadding: int
     * default contents padding at bottom
     */
    property int bottomPadding: Units.gridUnit

    /**
     * background: Item
     * This property holds the background item.
     *
     * Note: If the background item has no explicit size specified,
     * it automatically follows the control's size.
     * In most cases, there is no need to specify width or
     * height for a background item.
     */
    property Item background


    function open() {
        root.visible = true;
        openAnimation.from = -root.height;
        openAnimation.to = openAnimation.topOpenPosition;
        openAnimation.running = true;
        root.opened = true;
    }

    function close() {
        if (scrollView.flickableItem.contentY < 0) {
            closeAnimation.to = -height;
        } else {
            closeAnimation.to = scrollView.flickableItem.contentHeight;
        }
        closeAnimation.running = true;
    }

    Rectangle {
        anchors.fill: parent
        color: Theme.textColor
        opacity: 0.6 * Math.min(
            (Math.min(scrollView.flickableItem.contentY + scrollView.flickableItem.height, scrollView.flickableItem.height) / scrollView.flickableItem.height),
            (2 + (scrollView.flickableItem.contentHeight - scrollView.flickableItem.contentY - scrollView.flickableItem.topMargin - scrollView.flickableItem.bottomMargin)/scrollView.flickableItem.height))
    }


    Component.onCompleted: {
        scrollView.flickableItem.interactive = true;
    }
    onBackgroundChanged: {
        background.parent = flickableContents;
        background.z = -1;
    }
    onContentItemChanged: {
        if (contentItem.hasOwnProperty("contentY") && // Check if flickable
            contentItem.hasOwnProperty("contentHeight")) {
            contentItem.parent = scrollView;
            scrollView.contentItem = contentItem;
        } else {
            contentItem.parent = contentItemParent;
            scrollView.contentItem = flickableContents;
            contentItem.anchors.left = contentItemParent.left;
            contentItem.anchors.right = contentItemParent.right;
        }
        scrollView.flickableItem.flickableDirection = Flickable.VerticalFlick;
    }
    onOpenedChanged: {
        if (opened) {
            open();
        } else {
            close();
            Qt.inputMethod.hide();
        }
    }
    onWidthChanged: {
        if (!contentItem.contentItem)
            return

        var width = Math.max(root.width/2, Math.min(root.width, root.contentItem.implicitWidth));
        contentItem.contentItem.x = (root.width - width)/2
        contentItem.contentItem.width = width;
    }
    onHeightChanged: {
        var focusItem;

        if (typeof applicationWindow !== "undefined") {
            focusItem = applicationWindow().activeFocusItem;
        //fallback: hope activeFocusItem is in context
        } else {
            focusItem = activeFocusItem;
        }

        if (!activeFocusItem) {
            return;
        }

        //NOTE: there is no function to know if an item is descended from another,
        //so we have to walk the parent hyerarchy by hand
        var isDescendent = false;
        var candidate = focusItem.parent;
        while (candidate) {
            if (candidate == root) {
                isDescendent = true;
                break;
            }
            candidate = candidate.parent;
        }
        if (!isDescendent) {
            return;
        }

        var cursorY = 0;
        if (focusItem.cursorPosition !== undefined) {
            cursorY = focusItem.positionToRectangle(focusItem.cursorPosition).y;
        }

        
        var pos = focusItem.mapToItem(flickableContents, 0, cursorY - Units.gridUnit*3);
        //focused item alreqady visible? add some margin for the space of the action buttons
        if (pos.y >= scrollView.flickableItem.contentY && pos.y <= scrollView.flickableItem.contentY + scrollView.flickableItem.height - Units.gridUnit * 8) {
            return;
        }
        scrollView.flickableItem.contentY = pos.y;
    }


    NumberAnimation {
        id: openAnimation
        property int topOpenPosition: Math.min(-root.height*0.15, scrollView.flickableItem.contentHeight - root.height + Units.gridUnit * 5)
        property int bottomOpenPosition: (scrollView.flickableItem.contentHeight - root.height) + (Units.gridUnit * 5)
        target: scrollView.flickableItem
        properties: "contentY"
        from: -root.height
        to: topOpenPosition
        duration: Units.longDuration
        easing.type: Easing.OutQuad
        onRunningChanged: {
            if (!running) {
                var width = Math.max(root.width/2, Math.min(root.width, root.contentItem.implicitWidth));
                contentItem.contentItem.x = (root.width - width)/2
                contentItem.contentItem.width = width;
            }
        }
    }

    SequentialAnimation {
        id: closeAnimation
        property int to: -root.height
        NumberAnimation {
            target: scrollView.flickableItem
            properties: "contentY"
            to: closeAnimation.to
            duration: Units.longDuration
            easing.type: Easing.InQuad
        }
        ScriptAction {
            script: {
                scrollView.flickableItem.contentY = -root.height;
                root.visible = root.opened = false;
            }
        }
    }

    MouseArea {
        anchors.fill: parent
        z: 2
        drag.filterChildren: true
        hoverEnabled: true

        onClicked: {
            var pos = mapToItem(flickableContents, mouse.x, mouse.y);
            if (!flickableContents.contains(pos)) {
                root.close();
            }
        }

        Item {
            id: flickableContents
            //anchors.horizontalCenter: parent.horizontalCenter
            x: (root.width - width) / 2
            y: scrollView.flickableItem && root.contentItem.hasOwnProperty("contentY") ? -scrollView.flickableItem.contentY : 0
            width: root.contentItem.implicitWidth <= 0 ? root.width : Math.max(root.width/2, Math.min(root.width, root.contentItem.implicitWidth))
            height: scrollView.flickableItem && root.contentItem.hasOwnProperty("contentY") ? scrollView.flickableItem.contentHeight : (root.contentItem.height + topPadding + bottomPadding + Units.iconSizes.medium + Units.gridUnit)
            Item {
                id: contentItemParent
                anchors {
                    fill: parent
                    leftMargin: leftPadding
                    topMargin: topPadding
                    rightMargin: rightPadding
                    bottomMargin: bottomPadding
                }
            }
        }
        Binding {
            when: scrollView.flickableItem != null
            target: scrollView.flickableItem
            property: "topMargin"
            value: scrollView.height
        }
        Binding {
            when: scrollView.flickableItem != null
            target: scrollView.flickableItem
            property: "bottomMargin"
            value: scrollView.height
        }

        Connections {
            target: scrollView.flickableItem
            function movementEnded() {
                //close
                if ((root.height + scrollView.flickableItem.contentY) < root.height/2) {
                    closeAnimation.to = -root.height;
                    closeAnimation.running = true;
                } else if ((root.height*0.6 + scrollView.flickableItem.contentY) > scrollView.flickableItem.contentHeight) {
                    closeAnimation.to = scrollView.flickableItem.contentHeight
                    closeAnimation.running = true;

                //reset to the default opened position
                } else if (scrollView.flickableItem.contentY < openAnimation.topOpenPosition) {
                    openAnimation.from = scrollView.flickableItem.contentY;
                    openAnimation.to = openAnimation.topOpenPosition;
                    openAnimation.running = true;
                //reset to the default "bottom" opened position
                } else if (scrollView.flickableItem.contentY > openAnimation.bottomOpenPosition) {
                    openAnimation.from = scrollView.flickableItem.contentY;
                    openAnimation.to = openAnimation.bottomOpenPosition;
                    openAnimation.running = true;
                }
            }
            onMovementEnded: movementEnded();
            onFlickEnded: movementEnded();
            onContentHeightChanged: {
                if (openAnimation.running) {
                    openAnimation.running = false;
                    open();
                }
            }
        }
        Controls.ScrollView {
            id: scrollView
            anchors.fill: parent
            horizontalScrollBarPolicy: Qt.ScrollBarAlwaysOff
        }
    }
}
