# -*- coding: utf-8 -*-
#
# Copyright © Spyder Project Contributors
# Licensed under the terms of the MIT License
#

"""
Testing utilities to be used with pytest.
"""

# Standard library imports
import shutil
import tempfile
try:
    from unittest.mock import Mock
except ImportError:
    from mock import Mock # Python 2

# Third party imports
import pytest

# Local imports
# Local imports
from spyder.widgets.editor import EditorStack
from spyder.widgets.findreplace import FindReplace
from spyder.config.user import UserConfig
from spyder.config.main import CONF_VERSION, DEFAULTS


@pytest.fixture
def tmpconfig(request):
    """
    Fixtures that returns a temporary CONF element.
    """
    SUBFOLDER = tempfile.mkdtemp()
    CONF = UserConfig('spyder-test',
                      defaults=DEFAULTS,
                      version=CONF_VERSION,
                      subfolder=SUBFOLDER,
                      raw_mode=True,
                      )

    def fin():
        """
        Fixture finalizer to delete the temporary CONF element.
        """
        shutil.rmtree(SUBFOLDER)

    request.addfinalizer(fin)
    return CONF

@pytest.fixture
def setup_editor(qtbot):
    """
    Set up EditorStack with CodeEditor containing some Python code.
    The cursor is at the empty line below the code.
    Returns tuple with EditorStack and CodeEditor.
    """
    text = ('a = 1\n'
            'print(a)\n'
            '\n'
            'x = 2')  # a newline is added at end
    editorStack = EditorStack(None, [])
    editorStack.set_introspector(Mock())
    editorStack.set_find_widget(FindReplace(editorStack))
    editorStack.set_io_actions(Mock(), Mock(), Mock(), Mock())
    finfo = editorStack.new('foo.py', 'utf-8', text)
    qtbot.addWidget(editorStack)
    return editorStack, finfo.editor
