from .utils import do_ex, trace, has_command
from .version import meta
from os.path import abspath, normcase, realpath, isfile, join
import warnings

FILES_COMMAND = 'git ls-files'
DEFAULT_DESCRIBE = 'git describe --tags --long --match *.*'


def _normalized(path):
    return normcase(abspath(realpath(path)))


class GitWorkdir(object):
    """experimental, may change at any time"""
    def __init__(self, path):
        self.path = path

    def do_ex(self, cmd):
        return do_ex(cmd, cwd=self.path)

    @classmethod
    def from_potential_worktree(cls, wd):
        real_wd, _, ret = do_ex('git rev-parse --show-toplevel', wd)
        if ret:
            return
        trace('real root', real_wd)
        if _normalized(real_wd) != _normalized(wd):
            return

        return cls(real_wd)

    def is_dirty(self):
        out, _, _ = self.do_ex("git status --porcelain --untracked-files=no")
        return bool(out)

    def is_shallow(self):
        return isfile(join(self.path, '.git/shallow'))

    def fetch_shallow(self):
        self.do_ex("git fetch --unshallow")

    def node(self):
        rev_node, _, ret = self.do_ex('git rev-parse --verify --quiet HEAD')
        if not ret:
            return rev_node[:7]

    def count_all_nodes(self):
        revs, _, _ = self.do_ex('git rev-list HEAD')
        return revs.count('\n') + 1


def warn_on_shallow(wd):
    """experimental, may change at any time"""
    if wd.is_shallow():
        warnings.warn('"%s" is shallow and may cause errors' % (wd.path,))


def fetch_on_shallow(wd):
    """experimental, may change at any time"""
    if wd.is_shallow():
        warnings.warn('"%s" was shallow, git fetch was used to rectify')
        wd.fetch_shallow()


def fail_on_shallow(wd):
    """experimental, may change at any time"""
    if wd.is_shallow():
        raise ValueError(
            '%r is shallow, please correct with '
            '"git fetch --unshallow"' % wd.path)


def parse(root, describe_command=DEFAULT_DESCRIBE, pre_parse=warn_on_shallow):
    """
    :param pre_parse: experimental pre_parse action, may change at any time
    """
    if not has_command('git'):
        return

    wd = GitWorkdir.from_potential_worktree(root)
    if wd is None:
        return
    if pre_parse:
        pre_parse(wd)
    rev_node = wd.node()
    dirty = wd.is_dirty()

    if rev_node is None:
        return meta('0.0', distance=0, dirty=dirty)

    out, err, ret = do_ex(describe_command, root)
    if ret:
        return meta(
            '0.0',
            distance=wd.count_all_nodes(),
            node=rev_node,
            dirty=dirty,
        )

    tag, number, node = out.rsplit('-', 2)
    number = int(number)
    if number:
        return meta(tag, distance=number, node=node, dirty=dirty)
    else:
        return meta(tag, node=node, dirty=dirty)
