/*
 * Copyright (c) 2013, Oracle and/or its affiliates. All rights reserved.
 * DO NOT ALTER OR REMOVE COPYRIGHT NOTICES OR THIS FILE HEADER.
 *
 * This code is free software; you can redistribute it and/or modify it
 * under the terms of the GNU General Public License version 2 only, as
 * published by the Free Software Foundation.  Oracle designates this
 * particular file as subject to the "Classpath" exception as provided
 * by Oracle in the LICENSE file that accompanied this code.
 *
 * This code is distributed in the hope that it will be useful, but WITHOUT
 * ANY WARRANTY; without even the implied warranty of MERCHANTABILITY or
 * FITNESS FOR A PARTICULAR PURPOSE.  See the GNU General Public License
 * version 2 for more details (a copy is included in the LICENSE file that
 * accompanied this code).
 *
 * You should have received a copy of the GNU General Public License version
 * 2 along with this work; if not, write to the Free Software Foundation,
 * Inc., 51 Franklin St, Fifth Floor, Boston, MA 02110-1301 USA.
 *
 * Please contact Oracle, 500 Oracle Parkway, Redwood Shores, CA 94065 USA
 * or visit www.oracle.com if you need additional information or have any
 * questions.
 */

package sun.net;

import java.security.AccessController;

/**
 * Determines the ephemeral port range in use on this system.
 * If this cannot be determined, then the default settings
 * of the OS are returned.
 */

public final class PortConfig {

    private static int defaultUpper, defaultLower;
    private final static int upper, lower;

    private PortConfig() {}

    static {
        AccessController.doPrivileged(
            new java.security.PrivilegedAction<Void>() {
                public Void run() {
                    System.loadLibrary("net");
                    String os = System.getProperty("os.name");
                    if (os.startsWith("Linux")) {
                        defaultLower = 32768;
                        defaultUpper = 61000;
                    } else if (os.startsWith("SunOS")) {
                        defaultLower = 32768;
                        defaultUpper = 65535;
                    // Ephemeral port ranges may vary by OS version on AIX,
                    // see http://publib.boulder.ibm.com/infocenter/aix/v7r1/index.jsp?topic=%2Fcom.ibm.aix.rsct315.admin%2Fbl503_ephport.htm .
                    // On tested AIX 5.3 / 6.1 / 7.1 machines we see always the settings below by
                    // /usr/sbin/no -a | fgrep ephemeral so it seems to be a pretty good default.
                    } else if (os.startsWith("AIX")) {
                        defaultLower = 32768;
                        defaultUpper = 65535;
                    } else if (os.contains("OS X")) {
                        defaultLower = 49152;
                        defaultUpper = 65535;
                    } else {
                        throw new InternalError(
                            "sun.net.PortConfig: unknown OS");
                    }
                    return null;
                }
            });

        int v = getLower0();
        if (v == -1) {
            v = defaultLower;
        }
        lower = v;

        v = getUpper0();
        if (v == -1) {
            v = defaultUpper;
        }
        upper = v;
    }

    static native int getLower0();
    static native int getUpper0();

    public static int getLower() {
        return lower;
    }

    public static int getUpper() {
        return upper;
    }
}
