#ifndef _ELM_WIDGET_EO_H_
#define _ELM_WIDGET_EO_H_

#ifndef _ELM_WIDGET_EO_CLASS_TYPE
#define _ELM_WIDGET_EO_CLASS_TYPE

typedef Eo Elm_Widget;

#endif

#ifndef _ELM_WIDGET_EO_TYPES
#define _ELM_WIDGET_EO_TYPES

/** Used in 'Virtual' function @ref elm_obj_widget_activate.
 *
 * @ingroup Elm
 */
typedef enum
{
  ELM_ACTIVATE_DEFAULT = 0,
  ELM_ACTIVATE_UP,
  ELM_ACTIVATE_DOWN,
  ELM_ACTIVATE_RIGHT,
  ELM_ACTIVATE_LEFT,
  ELM_ACTIVATE_BACK
} Elm_Activate;

typedef enum
{
  ELM_THEME_APPLY_FAILED = 0,
  ELM_THEME_APPLY_DEFAULT = 1 /* 0 */,
  ELM_THEME_APPLY_SUCCESS = 3 /* 0 */
} Elm_Theme_Apply;

typedef struct _Elm_Tooltip Elm_Tooltip;

typedef struct _Elm_Cursor Elm_Cursor;


#endif
#define ELM_WIDGET_CLASS elm_widget_class_get()

EWAPI const Eo_Class *elm_widget_class_get(void);

EOAPI void elm_obj_widget_focus_set(Eo *obj, Eina_Bool focus);

EOAPI Eina_Bool elm_obj_widget_focus_get(const Eo *obj);

EOAPI void elm_obj_widget_drag_lock_y_set(Eo *obj, Eina_Bool lock);

EOAPI Eina_Bool elm_obj_widget_drag_lock_y_get(const Eo *obj);

/**
 * @brief Control the widget focus highlight style.
 *
 * @param[in] style The name of the focus highlight style.
 *
 * @return @c true on success, @c false otherwise.
 *
 * @ingroup Elm_Widget
 */
EOAPI Eina_Bool elm_obj_widget_focus_highlight_style_set(Eo *obj, const char *style);

/**
 * @brief Control the widget focus highlight style.
 *
 * @return The name of the focus highlight style.
 *
 * @ingroup Elm_Widget
 */
EOAPI const char *elm_obj_widget_focus_highlight_style_get(const Eo *obj);

/**
 * @brief Make the elementary object and its children to be unfocusable (or
 * focusable). If the tree is set as unfocusable, newest focused object which
 * is not in this tree will get focus. This API can be helpful for an object to
 * be deleted. When an object will be deleted soon, it and its children may not
 * want to get focus (by focus reverting or by other focus controls). Then,
 * just use this API before deleting.
 *
 * @param[in] tree_unfocusable @c true for unfocusable, @c false for focusable.
 *
 * @ingroup Elm_Widget
 */
EOAPI void elm_obj_widget_tree_unfocusable_set(Eo *obj, Eina_Bool tree_unfocusable);

/**
 * @brief Make the elementary object and its children to be unfocusable (or
 * focusable). If the tree is set as unfocusable, newest focused object which
 * is not in this tree will get focus. This API can be helpful for an object to
 * be deleted. When an object will be deleted soon, it and its children may not
 * want to get focus (by focus reverting or by other focus controls). Then,
 * just use this API before deleting.
 *
 * Return @c true, if the tree is unfocusable, @c false otherwise (and on
 * errors).
 *
 * @return @c true for unfocusable, @c false for focusable.
 *
 * @ingroup Elm_Widget
 */
EOAPI Eina_Bool elm_obj_widget_tree_unfocusable_get(const Eo *obj);

/**
 * @brief Control the widget's mirrored mode.
 *
 * @param[in] mirrored
 *
 * @ingroup Elm_Widget
 */
EOAPI void elm_obj_widget_mirrored_set(Eo *obj, Eina_Bool mirrored);

/** Control the widget's mirrored mode.
 *
 * @ingroup Elm_Widget
 */
EOAPI Eina_Bool elm_obj_widget_mirrored_get(const Eo *obj);

EOAPI void elm_obj_widget_theme_set(Eo *obj, Elm_Theme *th);

EOAPI Elm_Theme *elm_obj_widget_theme_get(const Eo *obj);

EOAPI void elm_obj_widget_disabled_set(Eo *obj, Eina_Bool disabled);

EOAPI Eina_Bool elm_obj_widget_disabled_get(const Eo *obj);

EOAPI void elm_obj_widget_highlight_ignore_set(Eo *obj, Eina_Bool ignore);

EOAPI Eina_Bool elm_obj_widget_highlight_ignore_get(const Eo *obj);

/**
 * @brief Control the widget's mirrored mode setting.
 *
 * @param[in] automatic
 *
 * @ingroup Elm_Widget
 */
EOAPI void elm_obj_widget_mirrored_automatic_set(Eo *obj, Eina_Bool automatic);

/** Control the widget's mirrored mode setting.
 *
 * @ingroup Elm_Widget
 */
EOAPI Eina_Bool elm_obj_widget_mirrored_automatic_get(const Eo *obj);

EOAPI void elm_obj_widget_orientation_mode_disabled_set(Eo *obj, Eina_Bool disabled);

EOAPI Eina_Bool elm_obj_widget_orientation_mode_disabled_get(const Eo *obj);

EOAPI Elm_Theme_Apply elm_obj_widget_style_set(Eo *obj, const char *style);

EOAPI const char *elm_obj_widget_style_get(const Eo *obj);

EOAPI void elm_obj_widget_scale_set(Eo *obj, double scale);

EOAPI double elm_obj_widget_scale_get(const Eo *obj);

/**
 * @brief A custom chain of objects to pass focus.
 *
 * @note On focus cycle, only will be evaluated children of this container.
 *
 * This function overwrites any previous custom focus chain within the list of
 * objects. The previous list will be deleted and this list will be managed by
 * elementary. After it is set, don't modify it.
 *
 * @param[in] objs Chain of objects to pass focus
 *
 * @ingroup Elm_Widget
 */
EOAPI void elm_obj_widget_focus_custom_chain_set(Eo *obj, Eina_List *objs);

/**
 * @brief A custom chain of objects to pass focus.
 *
 * @note On focus cycle, only will be evaluated children of this container.
 *
 * @return Chain of objects
 *
 * @ingroup Elm_Widget
 */
EOAPI const Eina_List *elm_obj_widget_focus_custom_chain_get(const Eo *obj);

/**
 * @brief The ability for an Elementary object to be focused.
 *
 * Unfocusable objects do nothing when programmatically focused, being the
 * nearest focusable parent object the one really getting focus. Also, when
 * they receive mouse input, they will get the event, but not take away the
 * focus from where it was previously.
 *
 * @note Objects which are meant to be interacted with by input events are
 * created able to be focused, by default. All the others are not.
 *
 * @param[in] can_focus @c true if the object can be focused, @c false if not.
 *
 * @ingroup Elm_Widget
 */
EOAPI void elm_obj_widget_can_focus_set(Eo *obj, Eina_Bool can_focus);

/**
 * @brief The ability for an Elementary object to be focused.
 *
 * Unfocusable objects do nothing when programmatically focused, being the
 * nearest focusable parent object the one really getting focus. Also, when
 * they receive mouse input, they will get the event, but not take away the
 * focus from where it was previously.
 *
 * @note Objects which are meant to be interacted with by input events are
 * created able to be focused, by default. All the others are not.
 *
 * Return @c true if the object is allowed to be focused, @c false if not (and
 * on errors).
 *
 * @return @c true if the object can be focused, @c false if not.
 *
 * @ingroup Elm_Widget
 */
EOAPI Eina_Bool elm_obj_widget_can_focus_get(const Eo *obj);

EOAPI void elm_obj_widget_highlight_in_theme_set(Eo *obj, Eina_Bool highlight);

EOAPI Eina_Bool elm_obj_widget_highlight_in_theme_get(const Eo *obj);

#ifdef ELM_WIDGET_PROTECTED
/**
 * @brief The internal widget parent
 *
 * @param[in] parent
 *
 * @ingroup Elm_Widget
 */
EOAPI void elm_obj_widget_parent_set(Eo *obj, Efl_Canvas_Object *parent);
#endif

#ifdef ELM_WIDGET_PROTECTED
/** The internal widget parent
 *
 * @ingroup Elm_Widget
 */
EOAPI Efl_Canvas_Object *elm_obj_widget_parent_get(const Eo *obj);
#endif

EOAPI void elm_obj_widget_access_info_set(Eo *obj, const char *txt);

EOAPI const char *elm_obj_widget_access_info_get(const Eo *obj);

EOAPI void elm_obj_widget_drag_lock_x_set(Eo *obj, Eina_Bool lock);

EOAPI Eina_Bool elm_obj_widget_drag_lock_x_get(const Eo *obj);

EOAPI void elm_obj_widget_access_highlight_in_theme_set(Eo *obj, Eina_Bool highlight);

EOAPI Eina_Bool elm_obj_widget_access_highlight_in_theme_get(const Eo *obj);

/**
 * @brief Control the focus_region_show mode.
 *
 * @param[in] mode
 *
 * @ingroup Elm_Widget
 */
EOAPI void elm_obj_widget_focus_region_show_mode_set(Eo *obj, Elm_Focus_Region_Show_Mode mode);

/** Control the focus_region_show mode.
 *
 * @ingroup Elm_Widget
 */
EOAPI Elm_Focus_Region_Show_Mode elm_obj_widget_focus_region_show_mode_get(const Eo *obj);

EOAPI void elm_obj_widget_parent_highlight_set(Eo *obj, Eina_Bool highlighted);

EOAPI Elm_Theme_Apply elm_obj_widget_theme_object_set(Eo *obj, Efl_Canvas_Object *edj, const char *wname, const char *welement, const char *wstyle);

EOAPI void elm_obj_widget_hover_object_set(Eo *obj, Efl_Canvas_Object *sobj);

EOAPI void elm_obj_widget_display_mode_set(Eo *obj, Efl_Gfx_Size_Hint_Mode dispmode);

EOAPI void elm_obj_widget_on_show_region_hook_set(Eo *obj, region_hook_func_type func, void *data);

EOAPI void elm_obj_widget_domain_part_text_translatable_set(Eo *obj, const char *part, const char *domain, Eina_Bool translatable);

EOAPI void elm_obj_widget_orientation_set(Eo *obj, int rotation);

EOAPI void elm_obj_widget_resize_object_set(Eo *obj, Efl_Canvas_Object *sobj, Eina_Bool sub_obj);

EOAPI void elm_obj_widget_domain_translatable_part_text_set(Eo *obj, const char *part, const char *domain, const char *label);

/**
 * @brief Function to operate on a given widget's scrollabe children when
 * necessary.
 *
 * @warning free the returned list with eina_list_free().
 *
 * @ingroup Elm_Widget
 */
EOAPI Eina_List *elm_obj_widget_scrollable_children_get(const Eo *obj);

EOAPI int elm_obj_widget_scroll_hold_get(const Eo *obj);

EOAPI int elm_obj_widget_drag_child_locked_y_get(const Eo *obj);

/**
 * @brief Control item loop feature.
 *
 * @param[in] enable
 *
 * @ingroup Elm_Widget
 */
EOAPI void elm_obj_widget_item_loop_enabled_set(Eo *obj, Eina_Bool enable);

/** Control item loop feature.
 *
 * @ingroup Elm_Widget
 */
EOAPI Eina_Bool elm_obj_widget_item_loop_enabled_get(const Eo *obj);

EOAPI Eina_Bool elm_obj_widget_child_can_focus_get(const Eo *obj);

EOAPI int elm_obj_widget_scroll_freeze_get(const Eo *obj);

EOAPI Eina_Bool elm_obj_widget_focus_region_get(const Eo *obj, Evas_Coord *x, Evas_Coord *y, Evas_Coord *w, Evas_Coord *h);

EOAPI Efl_Canvas_Object *elm_obj_widget_top_get(const Eo *obj);

EOAPI unsigned int elm_obj_widget_focus_order_get(const Eo *obj);

EOAPI int elm_obj_widget_drag_child_locked_x_get(const Eo *obj);

EOAPI Eina_List *elm_obj_widget_can_focus_child_list_get(const Eo *obj);

/** Get the focused widget item.
 *
 * @ingroup Elm_Widget
 */
EOAPI Elm_Widget_Item *elm_obj_widget_focused_item_get(const Eo *obj);

EOAPI void elm_obj_widget_parents_bounce_get(const Eo *obj, Eina_Bool *horiz, Eina_Bool *vert);

EOAPI Efl_Canvas_Object *elm_obj_widget_parent_widget_get(const Eo *obj);

EOAPI Eina_Bool elm_obj_widget_highlight_get(const Eo *obj);

/**
 * @brief Current focused object in object tree.
 *
 * @return Current focused or @c null, if there is no focused object.
 *
 * @ingroup Elm_Widget
 */
EOAPI Efl_Canvas_Object *elm_obj_widget_focused_object_get(const Eo *obj);

EOAPI void elm_obj_widget_parent2_set(Eo *obj, Efl_Canvas_Object *parent);

EOAPI Efl_Canvas_Object *elm_obj_widget_parent2_get(const Eo *obj);

EOAPI void elm_obj_widget_part_text_set(Eo *obj, const char * part, const char *label);

EOAPI const char *elm_obj_widget_part_text_get(const Eo *obj, const char * part);

/**
 * @brief Sets the widget's focus move policy.
 *
 * @param[in] policy Object's focus move policy.
 *
 * @ingroup Elm_Widget
 */
EOAPI void elm_obj_widget_focus_move_policy_set(Eo *obj, Efl_Ui_Focus_Move_Policy policy);

/**
 * @brief Returns the widget's focus move policy.
 *
 * @return Object's focus move policy.
 *
 * @ingroup Elm_Widget
 */
EOAPI Efl_Ui_Focus_Move_Policy elm_obj_widget_focus_move_policy_get(const Eo *obj);

/**
 * @brief Control the widget's focus_move_policy mode setting.
 *
 * @param[in] automatic @c true to follow system focus move policy change,
 * @c false otherwise
 *
 * @since 1.18
 *
 * @ingroup Elm_Widget
 */
EOAPI void elm_obj_widget_focus_move_policy_automatic_set(Eo *obj, Eina_Bool automatic);

/**
 * @brief Control the widget's focus_move_policy mode setting.
 *
 * @return @c true to follow system focus move policy change, @c false
 * otherwise
 *
 * @since 1.18
 *
 * @ingroup Elm_Widget
 */
EOAPI Eina_Bool elm_obj_widget_focus_move_policy_automatic_get(const Eo *obj);

EOAPI Efl_Canvas_Object *elm_obj_widget_newest_focus_order_get(const Eo *obj, unsigned int *newest_focus_order, Eina_Bool can_focus_only);

EOAPI void elm_obj_widget_scroll_hold_push(Eo *obj);

EOAPI void elm_obj_widget_cursor_add(Eo *obj, Elm_Cursor *cur);

/**
 * @brief Set the next object with specific focus direction.
 *
 * @param[in] next Focus next object
 * @param[in] dir Focus direction
 *
 * @since 1.8
 *
 * @ingroup Elm_Widget
 */
EOAPI void elm_obj_widget_focus_next_object_set(Eo *obj, Efl_Canvas_Object *next, Elm_Focus_Direction dir);

/**
 * @brief Get the next object with specific focus direction.
 *
 * @param[in] dir Focus direction
 *
 * @return Focus next object
 *
 * @since 1.8
 *
 * @ingroup Elm_Widget
 */
EOAPI Efl_Canvas_Object *elm_obj_widget_focus_next_object_get(const Eo *obj, Elm_Focus_Direction dir);

/**
 * @brief Set the next object item with specific focus direction.
 *
 * @param[in] next_item Focus next object item
 * @param[in] dir Focus direction
 *
 * @since 1.16
 *
 * @ingroup Elm_Widget
 */
EOAPI void elm_obj_widget_focus_next_item_set(Eo *obj, Elm_Widget_Item *next_item, Elm_Focus_Direction dir);

/**
 * @brief Get the next object item with specific focus direction.
 *
 * @param[in] dir Focus direction
 *
 * @return Focus next object item
 *
 * @since 1.16
 *
 * @ingroup Elm_Widget
 */
EOAPI Elm_Widget_Item *elm_obj_widget_focus_next_item_get(const Eo *obj, Elm_Focus_Direction dir);

EOAPI void elm_obj_widget_focus_tree_unfocusable_handle(Eo *obj);

/**
 * @brief Prepend object to custom focus chain.
 *
 * @note If @"relative_child" equal to @c null or not in custom chain, the
 * object will be added in begin.
 *
 * @note On focus cycle, only will be evaluated children of this container.
 *
 * @param[in] child The child to be added in custom chain.
 * @param[in] relative_child The relative object to position the child.
 *
 * @ingroup Elm_Widget
 */
EOAPI void elm_obj_widget_focus_custom_chain_prepend(Eo *obj, Efl_Canvas_Object *child, Efl_Canvas_Object *relative_child);

EOAPI const char *elm_obj_widget_part_text_translate(Eo *obj, const char *part, const char *text);

/**
 * @brief Get the focus highlight geometry of widget.
 *
 * @param[out] x
 * @param[out] y
 * @param[out] w
 * @param[out] h
 *
 * @ingroup Elm_Widget
 */
EOAPI void elm_obj_widget_focus_highlight_geometry_get(const Eo *obj, Evas_Coord *x, Evas_Coord *y, Evas_Coord *w, Evas_Coord *h);

/**
 * @brief 'Virtual' function to activate widget.
 *
 * @param[in] act
 *
 * @ingroup Elm_Widget
 */
EOAPI Eina_Bool elm_obj_widget_activate(Eo *obj, Elm_Activate act);

/**
 * @brief 'Virtual' function handling sub objects being added.
 *
 * @param[in] sobj
 *
 * @ingroup Elm_Widget
 */
EOAPI Eina_Bool elm_obj_widget_sub_object_add(Eo *obj, Efl_Canvas_Object *sobj);

/** 'Virtual' function which checks if handling of passing focus to sub-objects
 * in given direction is supported by widget.
 *
 * @ingroup Elm_Widget
 */
EOAPI Eina_Bool elm_obj_widget_focus_direction_manager_is(Eo *obj);

/**
 * @brief 'Virtual' function handling input events on the widget.
 *
 * @param[in] source
 * @param[in] type
 * @param[in] event_info
 *
 * @ingroup Elm_Widget
 */
EOAPI Eina_Bool elm_obj_widget_event(Eo *obj, Efl_Canvas_Object *source, Evas_Callback_Type type, void *event_info);

EOAPI void elm_obj_widget_event_callback_add(Eo *obj, Elm_Event_Cb func, const void *data);

/**
 * @brief 'Virtual' function on the widget being set access.
 *
 * @param[in] acs
 *
 * @ingroup Elm_Widget
 */
EOAPI void elm_obj_widget_access(Eo *obj, Eina_Bool acs);

EOAPI void elm_obj_widget_cursor_del(Eo *obj, Elm_Cursor *cur);

EOAPI void *elm_obj_widget_event_callback_del(Eo *obj, Elm_Event_Cb func, const void *data);

/**
 * @brief 'Virtual' function handling focus in/out events on the widget.
 *  return true if this widget can handle focus, false otherwise
 *
 * @param[in] item
 *
 * @ingroup Elm_Widget
 */
EOAPI Eina_Bool elm_obj_widget_on_focus(Eo *obj, Elm_Widget_Item *item);

/**
 * @brief 'Virtual' function returning an inner area of a widget that should be
 * brought into the visible area of a broader viewport, may this context arise.
 *
 * @param[out] x
 * @param[out] y
 * @param[out] w
 * @param[out] h
 *
 * @ingroup Elm_Widget
 */
EOAPI Eina_Bool elm_obj_widget_on_focus_region(Eo *obj, Evas_Coord *x, Evas_Coord *y, Evas_Coord *w, Evas_Coord *h);

/**
 * @brief Give focus to next object with specific focus direction in object
 * tree.
 *
 * @param[in] dir Direction to move the focus.
 *
 * @ingroup Elm_Widget
 */
EOAPI void elm_obj_widget_focus_cycle(Eo *obj, Elm_Focus_Direction dir);

/**
 * @brief 'Virtual' function handling passing focus to sub-objects given a
 * direction, in degrees.
 *
 * @param[in] base
 * @param[in] degree
 * @param[out] direction
 * @param[out] direction_item
 * @param[out] weight
 *
 * @ingroup Elm_Widget
 */
EOAPI Eina_Bool elm_obj_widget_focus_direction(Eo *obj, const Efl_Canvas_Object *base, double degree, Efl_Canvas_Object **direction, Elm_Widget_Item **direction_item, double *weight);

EOAPI Eina_Bool elm_obj_widget_event_propagate(Eo *obj, Evas_Callback_Type type, void *event_info, Efl_Event_Flags *event_flags);

EOAPI void elm_obj_widget_signal_callback_add(Eo *obj, const char *emission, const char *source, Edje_Signal_Cb func, void *data);

/** 'Virtual' function which checks if handling of passing focus to sub-objects
 * is supported by widget.
 *
 * @ingroup Elm_Widget
 */
EOAPI Eina_Bool elm_obj_widget_focus_next_manager_is(Eo *obj);

EOAPI Efl_Canvas_Object *elm_obj_widget_name_find(const Eo *obj, const char *name, int recurse);

EOAPI Eina_Bool elm_obj_widget_focus_list_direction_get(const Eo *obj, const Efl_Canvas_Object *base, const Eina_List *items, list_data_get_func_type list_data_get, double degree, Efl_Canvas_Object **direction, Elm_Widget_Item **direction_item, double *weight);

EOAPI void elm_obj_widget_focused_object_clear(Eo *obj);

EOAPI Eina_Bool elm_obj_widget_focus_direction_go(Eo *obj, double degree);

EOAPI void elm_obj_widget_show_region_set(Eo *obj, Evas_Coord x, Evas_Coord y, Evas_Coord w, Evas_Coord h, Eina_Bool forceshow);

EOAPI void elm_obj_widget_show_region_get(const Eo *obj, Evas_Coord *x, Evas_Coord *y, Evas_Coord *w, Evas_Coord *h);

EOAPI void elm_obj_widget_scroll_freeze_pop(Eo *obj);

EOAPI void elm_obj_widget_tooltip_del(Eo *obj, Elm_Tooltip *tt);

EOAPI Eina_Bool elm_obj_widget_focus_next_get(const Eo *obj, Elm_Focus_Direction dir, Efl_Canvas_Object **next, Elm_Widget_Item **next_item);

EOAPI const char *elm_obj_widget_translatable_part_text_get(const Eo *obj, const char *part);

/**
 * @brief Restore the focus state of the sub-tree.
 *
 * This API will restore the focus state of the sub-tree to the latest state.
 * If a sub-tree is unfocused and wants to get back to the latest focus state,
 * this API will be helpful.
 *
 * @ingroup Elm_Widget
 */
EOAPI void elm_obj_widget_focus_restore(Eo *obj);

EOAPI void elm_obj_widget_scroll_hold_pop(Eo *obj);

/** 'Virtual' function handling language changes on Elementary.
 *
 * @ingroup Elm_Widget
 */
EOAPI Eina_Bool elm_obj_widget_translate(Eo *obj);

EOAPI void elm_obj_widget_scroll_freeze_push(Eo *obj);

/**
 * @brief Unset a custom focus chain on a given Elementary widget.
 *
 * Any focus chain previously set is removed entirely after this call.
 *
 * @ingroup Elm_Widget
 */
EOAPI void elm_obj_widget_focus_custom_chain_unset(Eo *obj);

EOAPI void elm_obj_widget_focus_steal(Eo *obj, Elm_Widget_Item *item);

EOAPI void elm_obj_widget_focus_hide_handle(Eo *obj);

/**
 * @brief 'Virtual' function handling passing focus to sub-objects.
 *
 * @param[in] dir
 * @param[out] next
 * @param[out] next_item
 *
 * @ingroup Elm_Widget
 */
EOAPI Eina_Bool elm_obj_widget_focus_next(Eo *obj, Elm_Focus_Direction dir, Efl_Canvas_Object **next, Elm_Widget_Item **next_item);

EOAPI Eina_Bool elm_obj_widget_focus_list_next_get(const Eo *obj, const Eina_List *items, list_data_get_func_type list_data_get, Elm_Focus_Direction dir, Efl_Canvas_Object **next, Elm_Widget_Item **next_item);

EOAPI void elm_obj_widget_focus_mouse_up_handle(Eo *obj);

/** 'Virtual' function on the widget being re-themed.
 *
 * @ingroup Elm_Widget
 */
EOAPI Elm_Theme_Apply elm_obj_widget_theme_apply(Eo *obj);

EOAPI Eina_Bool elm_obj_widget_focus_direction_get(const Eo *obj, const Efl_Canvas_Object *base, double degree, Efl_Canvas_Object **direction, Elm_Widget_Item **direction_item, double *weight);

EOAPI void *elm_obj_widget_signal_callback_del(Eo *obj, const char *emission, const char *source, Edje_Signal_Cb func);

EOAPI void elm_obj_widget_signal_emit(Eo *obj, const char *emission, const char *source);

/** 'Virtual' function on the widget being disabled.
 *
 * @ingroup Elm_Widget
 */
EOAPI Eina_Bool elm_obj_widget_disable(Eo *obj);

/**
 * @brief 'Virtual' function handling sub objects being removed.
 *
 * @param[in] sobj
 *
 * @ingroup Elm_Widget
 */
EOAPI Eina_Bool elm_obj_widget_sub_object_del(Eo *obj, Efl_Canvas_Object *sobj);

EOAPI void elm_obj_widget_tooltip_add(Eo *obj, Elm_Tooltip *tt);

EOAPI void elm_obj_widget_focus_region_show(const Eo *obj);

EOAPI void elm_obj_widget_focus_disabled_handle(Eo *obj);

/**
 * @brief Append object to custom focus chain.
 *
 * @note If @"relative_child" equal to @c null or not in custom chain, the
 * object will be added in end.
 *
 * @note On focus cycle, only will be evaluated children of this container.
 *
 * @param[in] child The child to be added in custom chain.
 * @param[in] relative_child The relative object to position the child.
 *
 * @ingroup Elm_Widget
 */
EOAPI void elm_obj_widget_focus_custom_chain_append(Eo *obj, Efl_Canvas_Object *child, Efl_Canvas_Object *relative_child);

/** No description supplied.
 *
 * @since 1.18
 *
 * @ingroup Elm_Widget
 */
EOAPI void elm_obj_widget_focus_reconfigure(Eo *obj);

EOAPI extern const Eo_Event_Description _ELM_WIDGET_EVENT_MOVED;
EOAPI extern const Eo_Event_Description _ELM_WIDGET_EVENT_FOCUSED;
EOAPI extern const Eo_Event_Description _ELM_WIDGET_EVENT_UNFOCUSED;
EOAPI extern const Eo_Event_Description _ELM_WIDGET_EVENT_LANGUAGE_CHANGED;
EOAPI extern const Eo_Event_Description _ELM_WIDGET_EVENT_ACCESS_CHANGED;

/**
 * No description
 */
#define ELM_WIDGET_EVENT_MOVED (&(_ELM_WIDGET_EVENT_MOVED))

/**
 * No description
 */
#define ELM_WIDGET_EVENT_FOCUSED (&(_ELM_WIDGET_EVENT_FOCUSED))

/**
 * No description
 */
#define ELM_WIDGET_EVENT_UNFOCUSED (&(_ELM_WIDGET_EVENT_UNFOCUSED))

/**
 * No description
 */
#define ELM_WIDGET_EVENT_LANGUAGE_CHANGED (&(_ELM_WIDGET_EVENT_LANGUAGE_CHANGED))

/**
 * No description
 */
#define ELM_WIDGET_EVENT_ACCESS_CHANGED (&(_ELM_WIDGET_EVENT_ACCESS_CHANGED))

#endif
