module Pulp.Git where

import Prelude
import Control.Monad.Eff.Exception
import Control.Monad.Error.Class
import Control.Monad.Aff
import Data.Function
import Data.Maybe
import Data.Tuple
import Data.Either
import Data.Array as Array
import Data.Foldable as Foldable
import Data.Version (Version)
import Data.Version as Version
import Data.String as String
import Node.ChildProcess as CP

import Pulp.System.FFI
import Pulp.Exec

-- | Throw an error if the git working tree is dirty.
requireCleanGitWorkingTree :: AffN Unit
requireCleanGitWorkingTree = do
  out <- execQuiet "git" ["status", "--porcelain"] Nothing
  if Foldable.all String.null (String.split (String.Pattern "\n") out)
    then pure unit
    else throwError <<< error $
      "Your git working tree is dirty. Please commit or stash your changes " <>
      "first."

-- | Get the most recently tagged version from the git tag.
getVersionFromGitTag :: AffN (Maybe (Tuple String Version))
getVersionFromGitTag =
  attempt (run "git" ["describe", "--tags", "--abbrev=0", "HEAD"] Nothing)
  # map (either (const Nothing) maxVersion)

 where
 -- Run a command, piping stderr to /dev/null
 run = execQuietWithStderr CP.Ignore

-- | Given a number of lines of text, attempt to parse each line as a version,
-- | and return the maximum.
maxVersion :: String -> Maybe (Tuple String Version)
maxVersion =
  String.split (String.Pattern "\n")
  >>> Array.mapMaybe (String.trim >>> parseMay)
  >>> Foldable.maximumBy (compare `on` snd)

  where
  parseMay str =
    str
    # dropPrefix "v"
    # Version.parseVersion
    # either (const Nothing) Just
    # map (Tuple str)

dropPrefix :: String -> String -> String
dropPrefix prefix str =
  fromMaybe str (String.stripPrefix (String.Pattern prefix) str)
