#ifndef _ELM_SLIDER_EO_H_
#define _ELM_SLIDER_EO_H_

#ifndef _ELM_SLIDER_EO_CLASS_TYPE
#define _ELM_SLIDER_EO_CLASS_TYPE

typedef Eo Elm_Slider;

#endif

#ifndef _ELM_SLIDER_EO_TYPES
#define _ELM_SLIDER_EO_TYPES


#endif
#define ELM_SLIDER_CLASS elm_slider_class_get()

EWAPI const Eo_Class *elm_slider_class_get(void);

/**
 * @brief Set the format string for the indicator label.
 *
 * The slider may display its value somewhere else then unit label, for
 * example, above the slider knob that is dragged around. This function sets
 * the format string used for this.
 *
 * If @c null, indicator label won't be visible. If not it sets the format
 * string for the label text. To the label text is provided a floating point
 * value, so the label text can display up to 1 floating point value. Note that
 * this is optional.
 *
 * Use a format string such as "%1.2f meters" for example, and it will display
 * values like: "3.14 meters" for a value equal to 3.14159.
 *
 * Default is indicator label disabled.
 *
 * @param[in] indicator The format string for the indicator display.
 *
 * @ingroup Elm_Slider
 */
EOAPI void elm_obj_slider_indicator_format_set(Eo *obj, const char *indicator);

/**
 * @brief Get the indicator label format of the slider.
 *
 * The slider may display its value somewhere else then unit label, for
 * example, above the slider knob that is dragged around. This function gets
 * the format string used for this.
 *
 * @return The format string for the indicator display.
 *
 * @ingroup Elm_Slider
 */
EOAPI const char *elm_obj_slider_indicator_format_get(const Eo *obj);

/**
 * @brief Set whether to enlarge slider indicator (augmented knob) or not.
 *
 * By default, indicator will be bigger while dragged by the user.
 *
 * @warning It won't display values set with
 * @ref elm_obj_slider_indicator_format_set if you disable indicator.
 *
 * @param[in] show @c true will make it enlarge, @c false will let the knob
 * always at default size.
 *
 * @ingroup Elm_Slider
 */
EOAPI void elm_obj_slider_indicator_show_set(Eo *obj, Eina_Bool show);

/**
 * @brief Get whether a given slider widget's enlarging indicator or not.
 *
 * @return @c true will make it enlarge, @c false will let the knob always at
 * default size.
 *
 * @ingroup Elm_Slider
 */
EOAPI Eina_Bool elm_obj_slider_indicator_show_get(const Eo *obj);

/**
 * @brief Set the minimum and maximum values for the slider.
 *
 * Define the allowed range of values to be selected by the user.
 *
 * If actual value is less than @c min, it will be updated to @c min. If it is
 * bigger then @c max, will be updated to @c max. Actual value can be get with
 * @ref efl_ui_progress_value_get
 *
 * By default, min is equal to 0.0, and max is equal to 1.0.
 *
 * @warning maximum must be greater than minimum, otherwise behavior is
 * undefined.
 *
 * @param[in] min The minimum value.
 * @param[in] max The maximum value.
 *
 * @ingroup Elm_Slider
 */
EOAPI void elm_obj_slider_min_max_set(Eo *obj, double min, double max);

/**
 * @brief Get the minimum and maximum values of the slider.
 *
 * @note If only one value is needed, the other pointer can be passed as
 * @c null.
 *
 * @param[out] min The minimum value.
 * @param[out] max The maximum value.
 *
 * @ingroup Elm_Slider
 */
EOAPI void elm_obj_slider_min_max_get(const Eo *obj, double *min, double *max);

/**
 * @brief Show the indicator of slider on focus.
 *
 * @param[in] flag
 *
 * @ingroup Elm_Slider
 */
EOAPI void elm_obj_slider_indicator_show_on_focus_set(Eo *obj, Eina_Bool flag);

/** Get whether the indicator of the slider is set or not.
 *
 * @ingroup Elm_Slider
 */
EOAPI Eina_Bool elm_obj_slider_indicator_show_on_focus_get(const Eo *obj);

/**
 * @brief Set the step by which slider indicator will move.
 *
 * This value is used when draggable object is moved automatically i.e., in
 * case of key event when up/down/left/right key is pressed or in case when
 * accessibility is set and flick event is used to inc/dec slider values. By
 * default step value is equal to 0.05.
 *
 * @param[in] step The step value.
 *
 * @since 1.8
 *
 * @ingroup Elm_Slider
 */
EOAPI void elm_obj_slider_step_set(Eo *obj, double step);

/**
 * @brief Get the step by which slider indicator moves.
 *
 * @return The step value.
 *
 * @since 1.8
 *
 * @ingroup Elm_Slider
 */
EOAPI double elm_obj_slider_step_get(const Eo *obj);

/**
 * @brief Set the visible mode of indicator.
 *
 * @param[in] indicator_visible_mode The indicator visible mode.
 *
 * @ingroup Elm_Slider
 */
EOAPI void elm_obj_slider_indicator_visible_mode_set(Eo *obj, Efl_Ui_Slider_Indicator_Visible_Mode indicator_visible_mode);

/**
 * @brief Get the visible mode of indicator.
 *
 * @return The indicator visible mode.
 *
 * @ingroup Elm_Slider
 */
EOAPI Efl_Ui_Slider_Indicator_Visible_Mode elm_obj_slider_indicator_visible_mode_get(const Eo *obj);

/**
 * @brief Set the format function pointer for the indicator label
 *
 * Set the callback function to format the indicator string.
 *
 * @param[in] func The indicator format function.
 * @param[in] free_func The freeing function for the format string.
 *
 * @ingroup Elm_Slider
 */
EOAPI void elm_obj_slider_indicator_format_function_set(Eo *obj, slider_func_type func, slider_freefunc_type free_func);

/**
 * @brief This enables two indicators in slider.
 *
 * @param[in] enable
 *
 * @since 1.18
 *
 * @ingroup Elm_Slider
 */
EOAPI void elm_obj_slider_range_enabled_set(Eo *obj, Eina_Bool enable);

/** This enables two indicators in slider.
 *
 * @since 1.18
 *
 * @ingroup Elm_Slider
 */
EOAPI Eina_Bool elm_obj_slider_range_enabled_get(const Eo *obj);

/**
 * @brief Sets up a start and end range point for the slider
 *
 * @param[in] from range minimum value
 * @param[in] to range maximum value
 *
 * @since 1.18
 *
 * @ingroup Elm_Slider
 */
EOAPI void elm_obj_slider_range_set(Eo *obj, double from, double to);

/**
 * @brief Sets up a start and end range point for the slider
 *
 * @param[out] from range minimum value
 * @param[out] to range maximum value
 *
 * @since 1.18
 *
 * @ingroup Elm_Slider
 */
EOAPI void elm_obj_slider_range_get(const Eo *obj, double *from, double *to);

EOAPI extern const Eo_Event_Description _ELM_SLIDER_EVENT_CHANGED;
EOAPI extern const Eo_Event_Description _ELM_SLIDER_EVENT_DELAY_CHANGED;
EOAPI extern const Eo_Event_Description _ELM_SLIDER_EVENT_SLIDER_DRAG_START;
EOAPI extern const Eo_Event_Description _ELM_SLIDER_EVENT_SLIDER_DRAG_STOP;

/**
 * No description
 */
#define ELM_SLIDER_EVENT_CHANGED (&(_ELM_SLIDER_EVENT_CHANGED))

/**
 * No description
 */
#define ELM_SLIDER_EVENT_DELAY_CHANGED (&(_ELM_SLIDER_EVENT_DELAY_CHANGED))

/**
 * No description
 */
#define ELM_SLIDER_EVENT_SLIDER_DRAG_START (&(_ELM_SLIDER_EVENT_SLIDER_DRAG_START))

/**
 * No description
 */
#define ELM_SLIDER_EVENT_SLIDER_DRAG_STOP (&(_ELM_SLIDER_EVENT_SLIDER_DRAG_STOP))

#endif
