# -*- coding: utf-8 -*-

# Copyright (c) 2017 Detlev Offenbach <detlev@die-offenbachs.de>
#

"""
Module implementing the navigation bar widget.
"""

from __future__ import unicode_literals

from PyQt5.QtCore import Qt
from PyQt5.QtWidgets import QWidget, QHBoxLayout, QStyle, QToolButton, \
    QSplitter, QSizePolicy, QMenu, QAction

from WebBrowser.WebBrowserWindow import WebBrowserWindow

import UI.PixmapCache


class NavigationBar(QWidget):
    """
    Class implementing the navigation bar.
    """
    def __init__(self, mainWindow, parent=None):
        """
        Constructor
        
        @param mainWindow reference to the browser main window
        @type WebBrowserWindow
        @param parent reference to the parent widget
        @type QWidget
        """
        super(NavigationBar, self).__init__(parent)
        self.setObjectName("navigationbar")
        
        self.__mw = mainWindow
        
        self.__layout = QHBoxLayout(self)
        margin = self.style().pixelMetric(QStyle.PM_ToolBarItemMargin, None,
                                          self)
        self.__layout.setContentsMargins(margin, margin, margin, margin)
        self.__layout.setSpacing(
            self.style().pixelMetric(QStyle.PM_ToolBarItemSpacing, None, self))
        self.setLayout(self.__layout)
        
        self.__backButton = QToolButton(self)
        self.__backButton.setObjectName("navigation_back_button")
        self.__backButton.setToolTip(self.tr("Move one screen backward"))
        self.__backButton.setToolButtonStyle(Qt.ToolButtonIconOnly)
        self.__backButton.setFocusPolicy(Qt.NoFocus)
        self.__backButton.setAutoRaise(True)
        self.__backButton.setIcon(
            UI.PixmapCache.getIcon("back.png"))
        self.__backButton.setEnabled(False)
        
        self.__forwardButton = QToolButton(self)
        self.__forwardButton.setObjectName("navigation_forward_button")
        self.__forwardButton.setToolTip(self.tr("Move one screen forward"))
        self.__forwardButton.setToolButtonStyle(Qt.ToolButtonIconOnly)
        self.__forwardButton.setFocusPolicy(Qt.NoFocus)
        self.__forwardButton.setAutoRaise(True)
        self.__forwardButton.setIcon(
            UI.PixmapCache.getIcon("forward.png"))
        self.__forwardButton.setEnabled(False)
        
        self.__backNextLayout = QHBoxLayout()
        self.__backNextLayout.setContentsMargins(0, 0, 0, 0)
        self.__backNextLayout.setSpacing(0)
        self.__backNextLayout.addWidget(self.__backButton)
        self.__backNextLayout.addWidget(self.__forwardButton)
        
        self.__reloadButton = QToolButton(self)
        self.__reloadButton.setObjectName("navigation_reload_button")
        self.__reloadButton.setToolTip(self.tr("Reload the current screen"))
        self.__reloadButton.setToolButtonStyle(Qt.ToolButtonIconOnly)
        self.__reloadButton.setFocusPolicy(Qt.NoFocus)
        self.__reloadButton.setAutoRaise(True)
        self.__reloadButton.setIcon(
            UI.PixmapCache.getIcon("reload.png"))
        
        self.__stopButton = QToolButton(self)
        self.__stopButton.setObjectName("navigation_stop_button")
        self.__stopButton.setToolTip(self.tr("Stop loading"))
        self.__stopButton.setToolButtonStyle(Qt.ToolButtonIconOnly)
        self.__stopButton.setFocusPolicy(Qt.NoFocus)
        self.__stopButton.setAutoRaise(True)
        self.__stopButton.setIcon(
            UI.PixmapCache.getIcon("stopLoading.png"))
        
        self.__homeButton = QToolButton(self)
        self.__homeButton.setObjectName("navigation_home_button")
        self.__homeButton.setToolTip(self.tr("Move to the initial screen"))
        self.__homeButton.setToolButtonStyle(Qt.ToolButtonIconOnly)
        self.__homeButton.setFocusPolicy(Qt.NoFocus)
        self.__homeButton.setAutoRaise(True)
        self.__homeButton.setIcon(
            UI.PixmapCache.getIcon("home.png"))
        
        self.__exitFullScreenButton = QToolButton(self)
        self.__exitFullScreenButton.setObjectName(
            "navigation_exitfullscreen_button")
        self.__exitFullScreenButton.setToolTip(self.tr("Exit Fullscreen"))
        self.__exitFullScreenButton.setToolButtonStyle(Qt.ToolButtonIconOnly)
        self.__exitFullScreenButton.setFocusPolicy(Qt.NoFocus)
        self.__exitFullScreenButton.setAutoRaise(True)
        self.__exitFullScreenButton.setIcon(
            UI.PixmapCache.getIcon("windowRestore.png"))
        self.__exitFullScreenButton.clicked.connect(self.__mw.toggleFullScreen)
        self.__exitFullScreenButton.setVisible(False)
        
        self.__navigationSplitter = QSplitter(self)
        urlBar = self.__mw.tabWidget().stackedUrlBar()
        self.__navigationSplitter.addWidget(urlBar)
        
        from WebBrowser.WebBrowserWebSearchWidget import \
            WebBrowserWebSearchWidget
        self.__searchEdit = WebBrowserWebSearchWidget(self.__mw, self)
        sizePolicy = QSizePolicy(QSizePolicy.Expanding, QSizePolicy.Preferred)
        sizePolicy.setHorizontalStretch(2)
        sizePolicy.setVerticalStretch(0)
        self.__searchEdit.setSizePolicy(sizePolicy)
        self.__searchEdit.search.connect(self.__mw.openUrl)
        self.__navigationSplitter.addWidget(self.__searchEdit)
        
        self.__navigationSplitter.setSizePolicy(
            QSizePolicy.Expanding, QSizePolicy.Maximum)
        self.__navigationSplitter.setCollapsible(0, False)
        
        self.__layout.addLayout(self.__backNextLayout)
        self.__layout.addWidget(self.__reloadButton)
        self.__layout.addWidget(self.__stopButton)
        self.__layout.addWidget(self.__homeButton)
        self.__layout.addWidget(self.__navigationSplitter)
        self.__layout.addWidget(self.__exitFullScreenButton)
        
        self.__backMenu = QMenu(self)
        self.__backMenu.aboutToShow.connect(self.__showBackMenu)
        self.__backMenu.triggered.connect(self.__navigationMenuActionTriggered)
        self.__backButton.setMenu(self.__backMenu)
        self.__backButton.setPopupMode(QToolButton.MenuButtonPopup)
        
        self.__forwardMenu = QMenu(self)
        self.__forwardMenu.aboutToShow.connect(self.__showForwardMenu)
        self.__forwardMenu.triggered.connect(
            self.__navigationMenuActionTriggered)
        self.__forwardButton.setMenu(self.__forwardMenu)
        self.__forwardButton.setPopupMode(QToolButton.MenuButtonPopup)
        
        self.__backButton.clicked.connect(self.__goBack)
        self.__forwardButton.clicked.connect(self.__goForward)
        self.__reloadButton.clicked.connect(self.__reload)
        self.__stopButton.clicked.connect(self.__stopLoad)
        self.__homeButton.clicked.connect(self.__goHome)
    
    def backButton(self):
        """
        Public method to get a reference to the back button.
        
        @return reference to the back button
        @rtype QToolButton
        """
        return self.__backButton
    
    def forwardButton(self):
        """
        Public method to get a reference to the forward button.
        
        @return reference to the forward button
        @rtype QToolButton
        """
        return self.__forwardButton
    
    def reloadButton(self):
        """
        Public method to get a reference to the reload button.
        
        @return reference to the reload button
        @rtype QToolButton
        """
        return self.__reloadButton
    
    def stopButton(self):
        """
        Public method to get a reference to the stop button.
        
        @return reference to the stop button
        @rtype QToolButton
        """
        return self.__stopButton
    
    def exitFullScreenButton(self):
        """
        Public method to get a reference to the exit full screen button.
        
        @return reference to the exit full screen button
        @rtype QToolButton
        """
        return self.__exitFullScreenButton
    
    def searchEdit(self):
        """
        Public method to get a reference to the web search edit.
        
        @return reference to the web search edit
        @rtype WebBrowserWebSearchWidget
        """
        return self.__searchEdit
    
    def __showBackMenu(self):
        """
        Private slot showing the backwards navigation menu.
        """
        self.__backMenu.clear()
        history = self.__mw.currentBrowser().history()
        historyCount = history.count()
        backItems = history.backItems(historyCount)
        
        count = 0
        for index in range(len(backItems) - 1, -1, -1):
            item = backItems[index]
            act = QAction(self)
            act.setData(-1 * (index + 1))
            icon = WebBrowserWindow.icon(item.url())
            act.setIcon(icon)
            act.setText(item.title())
            self.__backMenu.addAction(act)
            
            count += 1
            if count == 20:
                break
        
        self.__backMenu.addSeparator()
        self.__backMenu.addAction(self.tr("Clear History"),
                                  self.__clearHistory)
    
    def __showForwardMenu(self):
        """
        Private slot showing the forwards navigation menu.
        """
        self.__forwardMenu.clear()
        history = self.__mw.currentBrowser().history()
        historyCount = history.count()
        forwardItems = history.forwardItems(historyCount)
        
        count = 0
        for index in range(len(forwardItems)):
            item = forwardItems[index]
            act = QAction(self)
            act.setData(index + 1)
            icon = WebBrowserWindow.icon(item.url())
            act.setIcon(icon)
            act.setText(item.title())
            self.__forwardMenu.addAction(act)
            
            count += 1
            if count == 20:
                break
        
        self.__forwardMenu.addSeparator()
        self.__forwardMenu.addAction(self.tr("Clear History"),
                                     self.__clearHistory)
    
    def __navigationMenuActionTriggered(self, act):
        """
        Private slot to go to the selected page.
        
        @param act reference to the action selected in the navigation menu
            (QAction)
        """
        offset = act.data()
        if offset is not None:
            history = self.__mw.currentBrowser().history()
            historyCount = history.count()
            if offset < 0:
                # go back
                history.goToItem(
                    history.backItems(historyCount)[-1 * offset - 1])
            else:
                # go forward
                history.goToItem(
                    history.forwardItems(historyCount)[offset - 1])
    
    def __goBack(self):
        """
        Private slot called to handle the backward button.
        """
        self.__mw.currentBrowser().backward()
    
    def __goForward(self):
        """
        Private slot called to handle the forward button.
        """
        self.__mw.currentBrowser().forward()
    
    def __goHome(self):
        """
        Private slot called to handle the home button.
        """
        self.__mw.currentBrowser().home()
    
    def __reload(self):
        """
        Private slot called to handle the reload button.
        """
        self.__mw.currentBrowser().reloadBypassingCache()
    
    def __stopLoad(self):
        """
        Private slot called to handle loading of the current page.
        """
        self.__mw.currentBrowser().stop()
    
    def __clearHistory(self):
        """
        Private slot to clear the history of the current web browser tab.
        """
        cb = self.__mw.currentBrowser()
        if cb is not None:
            cb.history().clear()
            self.__mw.setForwardAvailable(cb.isForwardAvailable())
            self.__mw.setBackwardAvailable(cb.isBackwardAvailable())
