#!/usr/bin/python
#
# (c) 2015 Peter Sprygada, <psprygada@ansible.com>
#
# Copyright (c) 2016 Dell Inc.
#
# This file is part of Ansible
#
# Ansible is free software: you can redistribute it and/or modify
# it under the terms of the GNU General Public License as published by
# the Free Software Foundation, either version 3 of the License, or
# (at your option) any later version.
#
# Ansible is distributed in the hope that it will be useful,
# but WITHOUT ANY WARRANTY; without even the implied warranty of
# MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
# GNU General Public License for more details.
#
# You should have received a copy of the GNU General Public License
# along with Ansible.  If not, see <http://www.gnu.org/licenses/>.
#

ANSIBLE_METADATA = {'metadata_version': '1.0',
                    'status': ['preview'],
                    'supported_by': 'community'}


DOCUMENTATION = """
---
module: dellos6_facts
version_added: "2.2"
author: "Abirami N(@abirami-n)"
short_description: Collect facts from remote devices running Dell EMC Networking OS6
description:
  - Collects a base set of device facts from a remote device that
    is running OS6.  This module prepends all of the
    base network fact keys with C(ansible_net_<fact>).  The facts
    module always collects a base set of facts from the device
    and can enable or disable collection of additional facts.
extends_documentation_fragment: dellos6
options:
  gather_subset:
    description:
      - When specified, this argument restricts the facts collected
        to a given subset.  Possible values for this argument include
        all, hardware, config, and interfaces.  You can specify a list of
        values to include a larger subset.  You can also use values with an initial M(!) to specify that a specific subset should
        not be collected.
    required: false
    default: '!config'
"""

EXAMPLES = """
# Collect all facts from the device
- dellos6_facts:
    gather_subset: all

# Collect only the config and default facts
- dellos6_facts:
    gather_subset:
      - config

# Do not collect hardware facts
- dellos6_facts:
    gather_subset:
      - "!interfaces"
"""

RETURN = """
ansible_net_gather_subset:
  description: The list of fact subsets collected from the device.
  returned: Always.
  type: list

# default
ansible_net_model:
  description: The model name returned from the device.
  returned: Always.
  type: str
ansible_net_serialnum:
  description: The serial number of the remote device.
  returned: Always.
  type: str
ansible_net_version:
  description: The operating system version running on the remote device.
  returned: Always.
  type: str
ansible_net_hostname:
  description: The configured hostname of the device.
  returned: Always.
  type: string
ansible_net_image:
  description: The image file that the device is running.
  returned: Always
  type: string

# hardware
ansible_net_memfree_mb:
  description: The available free memory on the remote device in MB.
  returned: When hardware is configured.
  type: int
ansible_net_memtotal_mb:
  description: The total memory on the remote device in MB.
  returned: When hardware is configured.
  type: int

# config
ansible_net_config:
  description: The current active config from the device.
  returned: When config is configured.
  type: str

# interfaces
ansible_net_interfaces:
  description: A hash of all interfaces running on the system.
  returned: When interfaces is configured.
  type: dict
ansible_net_neighbors:
  description: The list of LLDP neighbors from the remote device.
  returned: When interfaces is configured.
  type: dict

"""
import re
import itertools

from ansible.module_utils.dellos6 import run_commands
from ansible.module_utils.dellos6 import dellos6_argument_spec, check_args
from ansible.module_utils.basic import AnsibleModule
from ansible.module_utils.six import iteritems
from ansible.module_utils.six.moves import zip


class FactsBase(object):

    COMMANDS = list()

    def __init__(self, module):
        self.module = module
        self.facts = dict()
        self.responses = None

    def populate(self):
        self.responses = run_commands(self.module, self.COMMANDS, check_rc=False)

    def run(self, cmd):
        return run_commands(self.module, cmd, check_rc=False)


class Default(FactsBase):

    COMMANDS = [
        'show version',
        'show running-config | include hostname'
    ]

    def populate(self):
        super(Default, self).populate()
        data = self.responses[0]
        self.facts['version'] = self.parse_version(data)
        self.facts['serialnum'] = self.parse_serialnum(data)
        self.facts['model'] = self.parse_model(data)
        self.facts['image'] = self.parse_image(data)
        hdata = self.responses[0]
        self.facts['hostname'] = self.parse_hostname(hdata)

    def parse_version(self, data):
        match = re.search(r'HW Version(.+)\s(\d+)', data)
        if match:
            return match.group(2)

    def parse_hostname(self, data):
        match = re.search(r'\S+\s(\S+)', data, re.M)
        if match:
            return match.group(1)

    def parse_model(self, data):
        match = re.search(r'System Model ID(.+)\s([A-Z0-9]*)\n', data, re.M)
        if match:
            return match.group(2)

    def parse_image(self, data):
        match = re.search(r'Image File(.+)\s([A-Z0-9a-z_.]*)\n', data)
        if match:
            return match.group(2)

    def parse_serialnum(self, data):
        match = re.search(r'Serial Number(.+)\s([A-Z0-9]*)\n', data)
        if match:
            return match.group(2)


class Hardware(FactsBase):

    COMMANDS = [
        'show memory cpu'
    ]

    def populate(self):
        super(Hardware, self).populate()
        data = self.responses[0]
        match = re.findall('\s(\d+)\s', data)
        if match:
            self.facts['memtotal_mb'] = int(match[0]) / 1024
            self.facts['memfree_mb'] = int(match[1]) / 1024


class Config(FactsBase):

    COMMANDS = ['show running-config']

    def populate(self):
        super(Config, self).populate()
        self.facts['config'] = self.responses[0]


class Interfaces(FactsBase):
    COMMANDS = [
        'show interfaces',
        'show interfaces status',
        'show interfaces transceiver properties',
        'show ip int',
        'show lldp',
        'show lldp remote-device all'
    ]

    def populate(self):
        vlan_info = dict()
        super(Interfaces, self).populate()
        data = self.responses[0]
        interfaces = self.parse_interfaces(data)
        desc = self.responses[1]
        properties = self.responses[2]
        vlan = self.responses[3]
        vlan_info = self.parse_vlan(vlan)
        self.facts['interfaces'] = self.populate_interfaces(interfaces, desc, properties)
        self.facts['interfaces'].update(vlan_info)
        if 'LLDP is not enabled' not in self.responses[4]:
            neighbors = self.responses[5]
            self.facts['neighbors'] = self.parse_neighbors(neighbors)

    def parse_vlan(self, vlan):
        facts = dict()
        vlan_info, vlan_info_next = vlan.split('----------   -----   --------------- --------------- -------')
        for en in vlan_info_next.splitlines():
            if en == '':
                continue
            match = re.search('^(\S+)\s+(\S+)\s+(\S+)', en)
            intf = match.group(1)
            if intf not in facts:
                facts[intf] = list()
            fact = dict()
            matc = re.search('^([\w+\s\d]*)\s+(\S+)\s+(\S+)', en)
            fact['address'] = matc.group(2)
            fact['masklen'] = matc.group(3)
            facts[intf].append(fact)
        return facts

    def populate_interfaces(self, interfaces, desc, properties):
        facts = dict()
        for key, value in interfaces.items():
            intf = dict()
            intf['description'] = self.parse_description(key, desc)
            intf['macaddress'] = self.parse_macaddress(value)
            intf['mtu'] = self.parse_mtu(value)
            intf['bandwidth'] = self.parse_bandwidth(value)
            intf['mediatype'] = self.parse_mediatype(key, properties)
            intf['duplex'] = self.parse_duplex(value)
            intf['lineprotocol'] = self.parse_lineprotocol(value)
            intf['operstatus'] = self.parse_operstatus(value)
            intf['type'] = self.parse_type(key, properties)
            facts[key] = intf
        return facts

    def parse_neighbors(self, neighbors):
        facts = dict()
        neighbor, neighbor_next = neighbors.split('--------- ------- ------------------- ----------------- -----------------')
        for en in neighbor_next.splitlines():
            if en == '':
                continue
            intf = self.parse_lldp_intf(en.split()[0])
            if intf not in facts:
                facts[intf] = list()
            fact = dict()
            fact['port'] = self.parse_lldp_port(en.split()[3])
            if (len(en.split()) > 4):
                fact['host'] = self.parse_lldp_host(en.split()[4])
            else:
                fact['host'] = "Null"
            facts[intf].append(fact)

        return facts

    def parse_interfaces(self, data):
        parsed = dict()
        for line in data.split('\n'):
            if len(line) == 0:
                continue
            else:
                match = re.match(r'Interface Name(.+)\s([A-Za-z0-9/]*)', line)
                if match:
                    key = match.group(2)
                    parsed[key] = line
                else:
                    parsed[key] += '\n%s' % line
        return parsed

    def parse_description(self, key, desc):
        desc, desc_next = desc.split('--------- --------------- ------ ------- ---- ------ ----- -- -------------------')
        if desc_next.find('Oob') > 0:
            desc_val, desc_info = desc_next.split('Oob')
        elif desc_next.find('Port') > 0:
            desc_val, desc_info = desc_next.split('Port')
        for en in desc_val.splitlines():
            if key in en:
                match = re.search('^(\S+)\s+(\S+)', en)
                if match.group(2) in ['Full', 'N/A']:
                    return "Null"
                else:
                    return match.group(2)

    def parse_macaddress(self, data):
        match = re.search(r'Burned MAC Address(.+)\s([A-Z0-9.]*)\n', data)
        if match:
            return match.group(2)

    def parse_mtu(self, data):
        match = re.search(r'MTU Size(.+)\s(\d+)\n', data)
        if match:
            return int(match.group(2))

    def parse_bandwidth(self, data):
        match = re.search(r'Port Speed\s*[:\s\.]+\s(\d+)\n', data)
        if match:
            return int(match.group(1))

    def parse_duplex(self, data):
        match = re.search(r'Port Mode\s([A-Za-z]*)(.+)\s([A-Za-z/]*)\n', data)
        if match:
            return match.group(3)

    def parse_mediatype(self, key, properties):
        mediatype, mediatype_next = properties.split('--------- ------- --------------------- --------------------- --------------')
        flag = 1
        for en in mediatype_next.splitlines():
            if key in en:
                flag = 0
                match = re.search('^(\S+)\s+(\S+)\s+(\S+)', en)
                if match:
                    strval = match.group(3)
                    return match.group(3)
        if flag == 1:
            return "null"

    def parse_type(self, key, properties):
        type_val, type_val_next = properties.split('--------- ------- --------------------- --------------------- --------------')
        flag = 1
        for en in type_val_next.splitlines():
            if key in en:
                flag = 0
                match = re.search('^(\S+)\s+(\S+)\s+(\S+)', en)
                if match:
                    strval = match.group(2)
                    return match.group(2)
        if flag == 1:
            return "null"

    def parse_lineprotocol(self, data):
        data = data.splitlines()
        for d in data:
            match = re.search(r'^Link Status\s*[:\s\.]+\s(\S+)', d)
            if match:
                return match.group(1)

    def parse_operstatus(self, data):
        data = data.splitlines()
        for d in data:
            match = re.search(r'^Link Status\s*[:\s\.]+\s(\S+)', d)
            if match:
                return match.group(1)

    def parse_lldp_intf(self, data):
        match = re.search(r'^([A-Za-z0-9/]*)', data)
        if match:
            return match.group(1)

    def parse_lldp_host(self, data):
        match = re.search(r'^([A-Za-z0-9-]*)', data)
        if match:
            return match.group(1)

    def parse_lldp_port(self, data):
        match = re.search(r'^([A-Za-z0-9/]*)', data)
        if match:
            return match.group(1)


FACT_SUBSETS = dict(
    default=Default,
    hardware=Hardware,
    interfaces=Interfaces,
    config=Config,
)

VALID_SUBSETS = frozenset(FACT_SUBSETS.keys())


def main():
    """main entry point for module execution
    """
    argument_spec = dict(
        gather_subset=dict(default=['!config'], type='list')
    )

    argument_spec.update(dellos6_argument_spec)

    module = AnsibleModule(argument_spec=argument_spec,
                           supports_check_mode=True)

    gather_subset = module.params['gather_subset']

    runable_subsets = set()
    exclude_subsets = set()

    for subset in gather_subset:
        if subset == 'all':
            runable_subsets.update(VALID_SUBSETS)
            continue

        if subset.startswith('!'):
            subset = subset[1:]
            if subset == 'all':
                exclude_subsets.update(VALID_SUBSETS)
                continue
            exclude = True
        else:
            exclude = False

        if subset not in VALID_SUBSETS:
            module.fail_json(msg='Bad subset')

        if exclude:
            exclude_subsets.add(subset)
        else:
            runable_subsets.add(subset)

    if not runable_subsets:
        runable_subsets.update(VALID_SUBSETS)

    runable_subsets.difference_update(exclude_subsets)
    runable_subsets.add('default')

    facts = dict()
    facts['gather_subset'] = list(runable_subsets)

    instances = list()
    for key in runable_subsets:
        instances.append(FACT_SUBSETS[key](module))

    for inst in instances:
        inst.populate()
        facts.update(inst.facts)

    ansible_facts = dict()
    for key, value in iteritems(facts):
        key = 'ansible_net_%s' % key
        ansible_facts[key] = value

    warnings = list()
    check_args(module, warnings)

    module.exit_json(ansible_facts=ansible_facts, warnings=warnings)

if __name__ == '__main__':
    main()
