/**
 * HTML elements and other resources for web-based applications that need to
 * interact with the browser and the DOM (Document Object Model).
 *
 * This library includes DOM element types, CSS styling, local storage,
 * media, speech, events, and more.
 * To get started,
 * check out the [Element] class, the base class for many of the HTML
 * DOM types.
 *
 * ## Other resources
 *
 * * If you've never written a web app before, try our
 * tutorials&mdash;[A Game of Darts](http://dartlang.org/docs/tutorials).
 *
 * * To see some web-based Dart apps in action and to play with the code,
 * download
 * [Dart Editor](http://www.dartlang.org/#get-started)
 * and run its built-in examples.
 *
 * * For even more examples, see
 * [Dart HTML5 Samples](https://github.com/dart-lang/dart-html5-samples)
 * on Github.
 */
library dart.dom.html;

import 'dart:async';
import 'dart:collection';
import 'dart:_internal' hide Symbol;
import 'dart:html_common';
import 'dart:indexed_db';
import 'dart:indexed_db' show indexed_dbBlinkMap;
import 'dart:isolate';
import 'dart:js' as js;
import "dart:convert";
import 'dart:math';
import 'dart:mirrors';
import 'dart:nativewrappers';
import 'dart:typed_data';
import 'dart:web_gl' as gl;
import 'dart:web_gl' show web_glBlinkMap;
import 'dart:web_sql';
import 'dart:svg' as svg;
import 'dart:svg' show svgBlinkMap;
import 'dart:svg' show Matrix;
import 'dart:svg' show SvgSvgElement;
import 'dart:web_audio' as web_audio;
import 'dart:web_audio' show web_audioBlinkMap;
import 'dart:_blink' as _blink;
import 'dart:developer';
// Copyright (c) 2012, the Dart project authors.  Please see the AUTHORS file
// for details. All rights reserved. Use of this source code is governed by a
// BSD-style license that can be found in the LICENSE file.

// DO NOT EDIT
// Auto-generated dart:html library.

// TODO(vsm): Remove this when we can do the proper checking in
// native code for custom elements.
// Not actually used, but imported since dart:html can generate these objects.

export 'dart:math' show Rectangle, Point;

// Issue 14721, order important for WrappedEvent.

Window _window;

/**
 * Top-level container for a web page, which is usually a browser tab or window.
 *
 * Each web page loaded in the browser has its own [Window], which is a
 * container for the web page.
 *
 * If the web page has any `<iframe>` elements, then each `<iframe>` has its own
 * [Window] object, which is accessible only to that `<iframe>`.
 *
 * See also:
 *
 *   * [Window](https://developer.mozilla.org/en-US/docs/Web/API/window) from MDN.
 */
Window get window {
  if (_window != null) {
    return _window;
  }
  _window = js.JsNative.toTypedObject(js.context);
  return _window;
}

HtmlDocument _document;

/**
 * Root node for all content in a web page.
 */
HtmlDocument get document {
  if (_document != null) {
    return _document;
  }
  _document = window.document;
  return _document;
}

/**
 * Spawn a DOM isolate using the given URI in the same window.
 * This isolate is not concurrent.  It runs on the browser thread
 * with full access to the DOM.
 * Note: this API is still evolving and may move to dart:isolate.
 */
@Experimental()
Future<Isolate> spawnDomUri(Uri uri, List<String> args, message) {
  // TODO(17738): Plumb arguments and return value through.
  return _Utils.spawnDomUri(uri.toString());
}

// FIXME: Can we make this private?
@Deprecated("Internal Use Only")
final htmlBlinkMap = {
  '_HistoryCrossFrame': () => _HistoryCrossFrame,
  '_LocationCrossFrame': () => _LocationCrossFrame,
  '_DOMWindowCrossFrame': () => _DOMWindowCrossFrame,
  // FIXME: Move these to better locations.
  'DateTime': () => DateTime,
  'JsObject': () => js.JsObject,
  'JsFunction': () => js.JsFunction,
  'JsArray': () => js.JsArray,
  // We have to call .instanceRuntimeType as these classes have a private
  // implementation class defined dynamically at runtime via a patch file.
  'JSObject': () => js.JSObject.instanceRuntimeType,
  'JSFunction': () => js.JSFunction.instanceRuntimeType,
  'JSArray': () => js.JSArray.instanceRuntimeType,
  'AbstractWorker': () => AbstractWorker.instanceRuntimeType,
  'Animation': () => Animation.instanceRuntimeType,
  'AnimationEffectReadOnly': () => AnimationEffectReadOnly.instanceRuntimeType,
  'AnimationEffectTiming': () => AnimationEffectTiming.instanceRuntimeType,
  'AnimationEvent': () => AnimationEvent.instanceRuntimeType,
  'AnimationPlayerEvent': () => AnimationPlayerEvent.instanceRuntimeType,
  'AnimationTimeline': () => AnimationTimeline.instanceRuntimeType,
  'AppBannerPromptResult': () => AppBannerPromptResult.instanceRuntimeType,
  'ApplicationCache': () => ApplicationCache.instanceRuntimeType,
  'ApplicationCacheErrorEvent': () =>
      ApplicationCacheErrorEvent.instanceRuntimeType,
  'Attr': () => _Attr.instanceRuntimeType,
  'AudioTrack': () => AudioTrack.instanceRuntimeType,
  'AudioTrackList': () => AudioTrackList.instanceRuntimeType,
  'AutocompleteErrorEvent': () => AutocompleteErrorEvent.instanceRuntimeType,
  'BarProp': () => BarProp.instanceRuntimeType,
  'BatteryManager': () => BatteryManager.instanceRuntimeType,
  'BeforeInstallPromptEvent': () =>
      BeforeInstallPromptEvent.instanceRuntimeType,
  'BeforeUnloadEvent': () => BeforeUnloadEvent.instanceRuntimeType,
  'Blob': () => Blob.instanceRuntimeType,
  'Bluetooth': () => Bluetooth.instanceRuntimeType,
  'BluetoothDevice': () => BluetoothDevice.instanceRuntimeType,
  'BluetoothGATTCharacteristic': () =>
      BluetoothGattCharacteristic.instanceRuntimeType,
  'BluetoothGATTRemoteServer': () =>
      BluetoothGattRemoteServer.instanceRuntimeType,
  'BluetoothGATTService': () => BluetoothGattService.instanceRuntimeType,
  'BluetoothUUID': () => BluetoothUuid.instanceRuntimeType,
  'Body': () => Body.instanceRuntimeType,
  'CDATASection': () => CDataSection.instanceRuntimeType,
  'CHROMIUMValuebuffer': () => ChromiumValuebuffer.instanceRuntimeType,
  'CSS': () => Css.instanceRuntimeType,
  'CSSCharsetRule': () => CssCharsetRule.instanceRuntimeType,
  'CSSFontFaceRule': () => CssFontFaceRule.instanceRuntimeType,
  'CSSGroupingRule': () => CssGroupingRule.instanceRuntimeType,
  'CSSImportRule': () => CssImportRule.instanceRuntimeType,
  'CSSKeyframeRule': () => CssKeyframeRule.instanceRuntimeType,
  'CSSKeyframesRule': () => CssKeyframesRule.instanceRuntimeType,
  'CSSMediaRule': () => CssMediaRule.instanceRuntimeType,
  'CSSPageRule': () => CssPageRule.instanceRuntimeType,
  'CSSRule': () => CssRule.instanceRuntimeType,
  'CSSRuleList': () => _CssRuleList.instanceRuntimeType,
  'CSSStyleDeclaration': () => CssStyleDeclaration.instanceRuntimeType,
  'CSSStyleRule': () => CssStyleRule.instanceRuntimeType,
  'CSSStyleSheet': () => CssStyleSheet.instanceRuntimeType,
  'CSSSupportsRule': () => CssSupportsRule.instanceRuntimeType,
  'CSSViewportRule': () => CssViewportRule.instanceRuntimeType,
  'Cache': () => _Cache.instanceRuntimeType,
  'CacheStorage': () => CacheStorage.instanceRuntimeType,
  'CanvasGradient': () => CanvasGradient.instanceRuntimeType,
  'CanvasPathMethods': () => _CanvasPathMethods.instanceRuntimeType,
  'CanvasPattern': () => CanvasPattern.instanceRuntimeType,
  'CanvasRenderingContext2D': () =>
      CanvasRenderingContext2D.instanceRuntimeType,
  'CharacterData': () => CharacterData.instanceRuntimeType,
  'ChildNode': () => ChildNode.instanceRuntimeType,
  'CircularGeofencingRegion': () =>
      CircularGeofencingRegion.instanceRuntimeType,
  'Client': () => Client.instanceRuntimeType,
  'ClientRect': () => _ClientRect.instanceRuntimeType,
  'ClientRectList': () => _ClientRectList.instanceRuntimeType,
  'Clients': () => Clients.instanceRuntimeType,
  'ClipboardEvent': () => ClipboardEvent.instanceRuntimeType,
  'CloseEvent': () => CloseEvent.instanceRuntimeType,
  'Comment': () => Comment.instanceRuntimeType,
  'CompositionEvent': () => CompositionEvent.instanceRuntimeType,
  'CompositorProxy': () => CompositorProxy.instanceRuntimeType,
  'CompositorWorker': () => CompositorWorker.instanceRuntimeType,
  'CompositorWorkerGlobalScope': () =>
      CompositorWorkerGlobalScope.instanceRuntimeType,
  'Console': () => Console.instanceRuntimeType,
  'ConsoleBase': () => ConsoleBase.instanceRuntimeType,
  'Coordinates': () => Coordinates.instanceRuntimeType,
  'Credential': () => Credential.instanceRuntimeType,
  'CredentialsContainer': () => CredentialsContainer.instanceRuntimeType,
  'CrossOriginConnectEvent': () => CrossOriginConnectEvent.instanceRuntimeType,
  'CrossOriginServiceWorkerClient': () =>
      CrossOriginServiceWorkerClient.instanceRuntimeType,
  'Crypto': () => Crypto.instanceRuntimeType,
  'CryptoKey': () => CryptoKey.instanceRuntimeType,
  'CustomEvent': () => CustomEvent.instanceRuntimeType,
  'DOMError': () => DomError.instanceRuntimeType,
  'DOMException': () => DomException.instanceRuntimeType,
  'DOMFileSystem': () => FileSystem.instanceRuntimeType,
  'DOMFileSystemSync': () => _DOMFileSystemSync.instanceRuntimeType,
  'DOMImplementation': () => DomImplementation.instanceRuntimeType,
  'DOMMatrix': () => DomMatrix.instanceRuntimeType,
  'DOMMatrixReadOnly': () => DomMatrixReadOnly.instanceRuntimeType,
  'DOMParser': () => DomParser.instanceRuntimeType,
  'DOMPoint': () => DomPoint.instanceRuntimeType,
  'DOMPointReadOnly': () => DomPointReadOnly.instanceRuntimeType,
  'DOMRect': () => _DomRect.instanceRuntimeType,
  'DOMRectReadOnly': () => DomRectReadOnly.instanceRuntimeType,
  'DOMSettableTokenList': () => DomSettableTokenList.instanceRuntimeType,
  'DOMStringList': () => DomStringList.instanceRuntimeType,
  'DOMStringMap': () => DomStringMap.instanceRuntimeType,
  'DOMTokenList': () => DomTokenList.instanceRuntimeType,
  'DataTransfer': () => DataTransfer.instanceRuntimeType,
  'DataTransferItem': () => DataTransferItem.instanceRuntimeType,
  'DataTransferItemList': () => DataTransferItemList.instanceRuntimeType,
  'DedicatedWorkerGlobalScope': () =>
      DedicatedWorkerGlobalScope.instanceRuntimeType,
  'DefaultSessionStartEvent': () =>
      DefaultSessionStartEvent.instanceRuntimeType,
  'DeprecatedStorageInfo': () => DeprecatedStorageInfo.instanceRuntimeType,
  'DeprecatedStorageQuota': () => DeprecatedStorageQuota.instanceRuntimeType,
  'DeviceAcceleration': () => DeviceAcceleration.instanceRuntimeType,
  'DeviceLightEvent': () => DeviceLightEvent.instanceRuntimeType,
  'DeviceMotionEvent': () => DeviceMotionEvent.instanceRuntimeType,
  'DeviceOrientationEvent': () => DeviceOrientationEvent.instanceRuntimeType,
  'DeviceRotationRate': () => DeviceRotationRate.instanceRuntimeType,
  'DirectoryEntry': () => DirectoryEntry.instanceRuntimeType,
  'DirectoryEntrySync': () => _DirectoryEntrySync.instanceRuntimeType,
  'DirectoryReader': () => DirectoryReader.instanceRuntimeType,
  'DirectoryReaderSync': () => _DirectoryReaderSync.instanceRuntimeType,
  'Document': () => Document.instanceRuntimeType,
  'DocumentFragment': () => DocumentFragment.instanceRuntimeType,
  'DocumentType': () => _DocumentType.instanceRuntimeType,
  'EffectModel': () => EffectModel.instanceRuntimeType,
  'Element': () => Element.instanceRuntimeType,
  'Entry': () => Entry.instanceRuntimeType,
  'EntrySync': () => _EntrySync.instanceRuntimeType,
  'ErrorEvent': () => ErrorEvent.instanceRuntimeType,
  'Event': () => Event.instanceRuntimeType,
  'EventSource': () => EventSource.instanceRuntimeType,
  'EventTarget': () => EventTarget.instanceRuntimeType,
  'ExtendableEvent': () => ExtendableEvent.instanceRuntimeType,
  'FederatedCredential': () => FederatedCredential.instanceRuntimeType,
  'FetchEvent': () => FetchEvent.instanceRuntimeType,
  'File': () => File.instanceRuntimeType,
  'FileEntry': () => FileEntry.instanceRuntimeType,
  'FileEntrySync': () => _FileEntrySync.instanceRuntimeType,
  'FileError': () => FileError.instanceRuntimeType,
  'FileList': () => FileList.instanceRuntimeType,
  'FileReader': () => FileReader.instanceRuntimeType,
  'FileReaderSync': () => _FileReaderSync.instanceRuntimeType,
  'FileWriter': () => FileWriter.instanceRuntimeType,
  'FileWriterSync': () => _FileWriterSync.instanceRuntimeType,
  'FocusEvent': () => FocusEvent.instanceRuntimeType,
  'FontFace': () => FontFace.instanceRuntimeType,
  'FontFaceSet': () => FontFaceSet.instanceRuntimeType,
  'FontFaceSetLoadEvent': () => FontFaceSetLoadEvent.instanceRuntimeType,
  'FormData': () => FormData.instanceRuntimeType,
  'Gamepad': () => Gamepad.instanceRuntimeType,
  'GamepadButton': () => GamepadButton.instanceRuntimeType,
  'GamepadEvent': () => GamepadEvent.instanceRuntimeType,
  'GamepadList': () => _GamepadList.instanceRuntimeType,
  'Geofencing': () => Geofencing.instanceRuntimeType,
  'GeofencingEvent': () => GeofencingEvent.instanceRuntimeType,
  'GeofencingRegion': () => GeofencingRegion.instanceRuntimeType,
  'Geolocation': () => Geolocation.instanceRuntimeType,
  'Geoposition': () => Geoposition.instanceRuntimeType,
  'GlobalEventHandlers': () => GlobalEventHandlers.instanceRuntimeType,
  'HMDVRDevice': () => HmdvrDevice.instanceRuntimeType,
  'HTMLAllCollection': () => _HTMLAllCollection.instanceRuntimeType,
  'HTMLAnchorElement': () => AnchorElement.instanceRuntimeType,
  'HTMLAppletElement': () => _HTMLAppletElement.instanceRuntimeType,
  'HTMLAreaElement': () => AreaElement.instanceRuntimeType,
  'HTMLAudioElement': () => AudioElement.instanceRuntimeType,
  'HTMLBRElement': () => BRElement.instanceRuntimeType,
  'HTMLBaseElement': () => BaseElement.instanceRuntimeType,
  'HTMLBodyElement': () => BodyElement.instanceRuntimeType,
  'HTMLButtonElement': () => ButtonElement.instanceRuntimeType,
  'HTMLCanvasElement': () => CanvasElement.instanceRuntimeType,
  'HTMLCollection': () => HtmlCollection.instanceRuntimeType,
  'HTMLContentElement': () => ContentElement.instanceRuntimeType,
  'HTMLDListElement': () => DListElement.instanceRuntimeType,
  'HTMLDataListElement': () => DataListElement.instanceRuntimeType,
  'HTMLDetailsElement': () => DetailsElement.instanceRuntimeType,
  'HTMLDialogElement': () => DialogElement.instanceRuntimeType,
  'HTMLDirectoryElement': () => _HTMLDirectoryElement.instanceRuntimeType,
  'HTMLDivElement': () => DivElement.instanceRuntimeType,
  'HTMLDocument': () => HtmlDocument.instanceRuntimeType,
  'HTMLElement': () => HtmlElement.instanceRuntimeType,
  'HTMLEmbedElement': () => EmbedElement.instanceRuntimeType,
  'HTMLFieldSetElement': () => FieldSetElement.instanceRuntimeType,
  'HTMLFontElement': () => _HTMLFontElement.instanceRuntimeType,
  'HTMLFormControlsCollection': () =>
      HtmlFormControlsCollection.instanceRuntimeType,
  'HTMLFormElement': () => FormElement.instanceRuntimeType,
  'HTMLFrameElement': () => _HTMLFrameElement.instanceRuntimeType,
  'HTMLFrameSetElement': () => _HTMLFrameSetElement.instanceRuntimeType,
  'HTMLHRElement': () => HRElement.instanceRuntimeType,
  'HTMLHeadElement': () => HeadElement.instanceRuntimeType,
  'HTMLHeadingElement': () => HeadingElement.instanceRuntimeType,
  'HTMLHtmlElement': () => HtmlHtmlElement.instanceRuntimeType,
  'HTMLIFrameElement': () => IFrameElement.instanceRuntimeType,
  'HTMLImageElement': () => ImageElement.instanceRuntimeType,
  'HTMLInputElement': () => InputElement.instanceRuntimeType,
  'HTMLKeygenElement': () => KeygenElement.instanceRuntimeType,
  'HTMLLIElement': () => LIElement.instanceRuntimeType,
  'HTMLLabelElement': () => LabelElement.instanceRuntimeType,
  'HTMLLegendElement': () => LegendElement.instanceRuntimeType,
  'HTMLLinkElement': () => LinkElement.instanceRuntimeType,
  'HTMLMapElement': () => MapElement.instanceRuntimeType,
  'HTMLMarqueeElement': () => _HTMLMarqueeElement.instanceRuntimeType,
  'HTMLMediaElement': () => MediaElement.instanceRuntimeType,
  'HTMLMenuElement': () => MenuElement.instanceRuntimeType,
  'HTMLMenuItemElement': () => MenuItemElement.instanceRuntimeType,
  'HTMLMetaElement': () => MetaElement.instanceRuntimeType,
  'HTMLMeterElement': () => MeterElement.instanceRuntimeType,
  'HTMLModElement': () => ModElement.instanceRuntimeType,
  'HTMLOListElement': () => OListElement.instanceRuntimeType,
  'HTMLObjectElement': () => ObjectElement.instanceRuntimeType,
  'HTMLOptGroupElement': () => OptGroupElement.instanceRuntimeType,
  'HTMLOptionElement': () => OptionElement.instanceRuntimeType,
  'HTMLOptionsCollection': () => HtmlOptionsCollection.instanceRuntimeType,
  'HTMLOutputElement': () => OutputElement.instanceRuntimeType,
  'HTMLParagraphElement': () => ParagraphElement.instanceRuntimeType,
  'HTMLParamElement': () => ParamElement.instanceRuntimeType,
  'HTMLPictureElement': () => PictureElement.instanceRuntimeType,
  'HTMLPreElement': () => PreElement.instanceRuntimeType,
  'HTMLProgressElement': () => ProgressElement.instanceRuntimeType,
  'HTMLQuoteElement': () => QuoteElement.instanceRuntimeType,
  'HTMLScriptElement': () => ScriptElement.instanceRuntimeType,
  'HTMLSelectElement': () => SelectElement.instanceRuntimeType,
  'HTMLShadowElement': () => ShadowElement.instanceRuntimeType,
  'HTMLSourceElement': () => SourceElement.instanceRuntimeType,
  'HTMLSpanElement': () => SpanElement.instanceRuntimeType,
  'HTMLStyleElement': () => StyleElement.instanceRuntimeType,
  'HTMLTableCaptionElement': () => TableCaptionElement.instanceRuntimeType,
  'HTMLTableCellElement': () => TableCellElement.instanceRuntimeType,
  'HTMLTableColElement': () => TableColElement.instanceRuntimeType,
  'HTMLTableElement': () => TableElement.instanceRuntimeType,
  'HTMLTableRowElement': () => TableRowElement.instanceRuntimeType,
  'HTMLTableSectionElement': () => TableSectionElement.instanceRuntimeType,
  'HTMLTemplateElement': () => TemplateElement.instanceRuntimeType,
  'HTMLTextAreaElement': () => TextAreaElement.instanceRuntimeType,
  'HTMLTitleElement': () => TitleElement.instanceRuntimeType,
  'HTMLTrackElement': () => TrackElement.instanceRuntimeType,
  'HTMLUListElement': () => UListElement.instanceRuntimeType,
  'HTMLUnknownElement': () => UnknownElement.instanceRuntimeType,
  'HTMLVideoElement': () => VideoElement.instanceRuntimeType,
  'HashChangeEvent': () => HashChangeEvent.instanceRuntimeType,
  'Headers': () => Headers.instanceRuntimeType,
  'History': () => History.instanceRuntimeType,
  'ImageBitmap': () => ImageBitmap.instanceRuntimeType,
  'ImageData': () => ImageData.instanceRuntimeType,
  'InjectedScriptHost': () => InjectedScriptHost.instanceRuntimeType,
  'InputDevice': () => InputDevice.instanceRuntimeType,
  'Iterator': () => DomIterator.instanceRuntimeType,
  'KeyboardEvent': () => KeyboardEvent.instanceRuntimeType,
  'KeyframeEffect': () => KeyframeEffect.instanceRuntimeType,
  'Location': () => Location.instanceRuntimeType,
  'MIDIAccess': () => MidiAccess.instanceRuntimeType,
  'MIDIConnectionEvent': () => MidiConnectionEvent.instanceRuntimeType,
  'MIDIInput': () => MidiInput.instanceRuntimeType,
  'MIDIInputMap': () => MidiInputMap.instanceRuntimeType,
  'MIDIMessageEvent': () => MidiMessageEvent.instanceRuntimeType,
  'MIDIOutput': () => MidiOutput.instanceRuntimeType,
  'MIDIOutputMap': () => MidiOutputMap.instanceRuntimeType,
  'MIDIPort': () => MidiPort.instanceRuntimeType,
  'MediaController': () => MediaController.instanceRuntimeType,
  'MediaDeviceInfo': () => MediaDeviceInfo.instanceRuntimeType,
  'MediaDevices': () => MediaDevices.instanceRuntimeType,
  'MediaEncryptedEvent': () => MediaEncryptedEvent.instanceRuntimeType,
  'MediaError': () => MediaError.instanceRuntimeType,
  'MediaKeyError': () => MediaKeyError.instanceRuntimeType,
  'MediaKeyEvent': () => MediaKeyEvent.instanceRuntimeType,
  'MediaKeyMessageEvent': () => MediaKeyMessageEvent.instanceRuntimeType,
  'MediaKeySession': () => MediaKeySession.instanceRuntimeType,
  'MediaKeyStatusMap': () => MediaKeyStatusMap.instanceRuntimeType,
  'MediaKeySystemAccess': () => MediaKeySystemAccess.instanceRuntimeType,
  'MediaKeys': () => MediaKeys.instanceRuntimeType,
  'MediaList': () => MediaList.instanceRuntimeType,
  'MediaQueryList': () => MediaQueryList.instanceRuntimeType,
  'MediaQueryListEvent': () => MediaQueryListEvent.instanceRuntimeType,
  'MediaSession': () => MediaSession.instanceRuntimeType,
  'MediaSource': () => MediaSource.instanceRuntimeType,
  'MediaStream': () => MediaStream.instanceRuntimeType,
  'MediaStreamEvent': () => MediaStreamEvent.instanceRuntimeType,
  'MediaStreamTrack': () => MediaStreamTrack.instanceRuntimeType,
  'MediaStreamTrackEvent': () => MediaStreamTrackEvent.instanceRuntimeType,
  'MemoryInfo': () => MemoryInfo.instanceRuntimeType,
  'MessageChannel': () => MessageChannel.instanceRuntimeType,
  'MessageEvent': () => MessageEvent.instanceRuntimeType,
  'MessagePort': () => MessagePort.instanceRuntimeType,
  'Metadata': () => Metadata.instanceRuntimeType,
  'MimeType': () => MimeType.instanceRuntimeType,
  'MimeTypeArray': () => MimeTypeArray.instanceRuntimeType,
  'MouseEvent': () => MouseEvent.instanceRuntimeType,
  'MutationObserver': () => MutationObserver.instanceRuntimeType,
  'MutationRecord': () => MutationRecord.instanceRuntimeType,
  'NamedNodeMap': () => _NamedNodeMap.instanceRuntimeType,
  'Navigator': () => Navigator.instanceRuntimeType,
  'NavigatorCPU': () => NavigatorCpu.instanceRuntimeType,
  'NavigatorID': () => NavigatorID.instanceRuntimeType,
  'NavigatorLanguage': () => NavigatorLanguage.instanceRuntimeType,
  'NavigatorOnLine': () => NavigatorOnLine.instanceRuntimeType,
  'NavigatorStorageUtils': () => NavigatorStorageUtils.instanceRuntimeType,
  'NavigatorUserMediaError': () => NavigatorUserMediaError.instanceRuntimeType,
  'NetworkInformation': () => NetworkInformation.instanceRuntimeType,
  'Node': () => Node.instanceRuntimeType,
  'NodeFilter': () => NodeFilter.instanceRuntimeType,
  'NodeIterator': () => NodeIterator.instanceRuntimeType,
  'NodeList': () => NodeList.instanceRuntimeType,
  'NonDocumentTypeChildNode': () =>
      NonDocumentTypeChildNode.instanceRuntimeType,
  'NonElementParentNode': () => NonElementParentNode.instanceRuntimeType,
  'Notification': () => Notification.instanceRuntimeType,
  'NotificationEvent': () => NotificationEvent.instanceRuntimeType,
  'PagePopupController': () => _PagePopupController.instanceRuntimeType,
  'PageTransitionEvent': () => PageTransitionEvent.instanceRuntimeType,
  'ParentNode': () => ParentNode.instanceRuntimeType,
  'PasswordCredential': () => PasswordCredential.instanceRuntimeType,
  'Path2D': () => Path2D.instanceRuntimeType,
  'Performance': () => Performance.instanceRuntimeType,
  'PerformanceCompositeTiming': () =>
      PerformanceCompositeTiming.instanceRuntimeType,
  'PerformanceEntry': () => PerformanceEntry.instanceRuntimeType,
  'PerformanceMark': () => PerformanceMark.instanceRuntimeType,
  'PerformanceMeasure': () => PerformanceMeasure.instanceRuntimeType,
  'PerformanceNavigation': () => PerformanceNavigation.instanceRuntimeType,
  'PerformanceRenderTiming': () => PerformanceRenderTiming.instanceRuntimeType,
  'PerformanceResourceTiming': () =>
      PerformanceResourceTiming.instanceRuntimeType,
  'PerformanceTiming': () => PerformanceTiming.instanceRuntimeType,
  'PeriodicSyncEvent': () => PeriodicSyncEvent.instanceRuntimeType,
  'PeriodicSyncManager': () => PeriodicSyncManager.instanceRuntimeType,
  'PeriodicSyncRegistration': () =>
      PeriodicSyncRegistration.instanceRuntimeType,
  'PermissionStatus': () => PermissionStatus.instanceRuntimeType,
  'Permissions': () => Permissions.instanceRuntimeType,
  'Plugin': () => Plugin.instanceRuntimeType,
  'PluginArray': () => PluginArray.instanceRuntimeType,
  'PluginPlaceholderElement': () =>
      PluginPlaceholderElement.instanceRuntimeType,
  'PointerEvent': () => PointerEvent.instanceRuntimeType,
  'PopStateEvent': () => PopStateEvent.instanceRuntimeType,
  'PositionError': () => PositionError.instanceRuntimeType,
  'PositionSensorVRDevice': () => PositionSensorVRDevice.instanceRuntimeType,
  'Presentation': () => Presentation.instanceRuntimeType,
  'PresentationAvailability': () =>
      PresentationAvailability.instanceRuntimeType,
  'PresentationSession': () => PresentationSession.instanceRuntimeType,
  'ProcessingInstruction': () => ProcessingInstruction.instanceRuntimeType,
  'ProgressEvent': () => ProgressEvent.instanceRuntimeType,
  'PromiseRejectionEvent': () => PromiseRejectionEvent.instanceRuntimeType,
  'PushEvent': () => PushEvent.instanceRuntimeType,
  'PushManager': () => PushManager.instanceRuntimeType,
  'PushMessageData': () => PushMessageData.instanceRuntimeType,
  'PushSubscription': () => PushSubscription.instanceRuntimeType,
  'RTCDTMFSender': () => RtcDtmfSender.instanceRuntimeType,
  'RTCDTMFToneChangeEvent': () => RtcDtmfToneChangeEvent.instanceRuntimeType,
  'RTCDataChannel': () => RtcDataChannel.instanceRuntimeType,
  'RTCDataChannelEvent': () => RtcDataChannelEvent.instanceRuntimeType,
  'RTCIceCandidate': () => RtcIceCandidate.instanceRuntimeType,
  'RTCIceCandidateEvent': () => RtcIceCandidateEvent.instanceRuntimeType,
  'RTCPeerConnection': () => RtcPeerConnection.instanceRuntimeType,
  'RTCSessionDescription': () => RtcSessionDescription.instanceRuntimeType,
  'RTCStatsReport': () => RtcStatsReport.instanceRuntimeType,
  'RTCStatsResponse': () => RtcStatsResponse.instanceRuntimeType,
  'RadioNodeList': () => _RadioNodeList.instanceRuntimeType,
  'Range': () => Range.instanceRuntimeType,
  'ReadableByteStream': () => ReadableByteStream.instanceRuntimeType,
  'ReadableByteStreamReader': () =>
      ReadableByteStreamReader.instanceRuntimeType,
  'ReadableStream': () => ReadableStream.instanceRuntimeType,
  'ReadableStreamReader': () => ReadableStreamReader.instanceRuntimeType,
  'RelatedEvent': () => RelatedEvent.instanceRuntimeType,
  'Request': () => _Request.instanceRuntimeType,
  'ResourceProgressEvent': () => _ResourceProgressEvent.instanceRuntimeType,
  'Response': () => _Response.instanceRuntimeType,
  'Screen': () => Screen.instanceRuntimeType,
  'ScreenOrientation': () => ScreenOrientation.instanceRuntimeType,
  'ScrollState': () => ScrollState.instanceRuntimeType,
  'SecurityPolicyViolationEvent': () =>
      SecurityPolicyViolationEvent.instanceRuntimeType,
  'Selection': () => Selection.instanceRuntimeType,
  'ServicePort': () => ServicePort.instanceRuntimeType,
  'ServicePortCollection': () => ServicePortCollection.instanceRuntimeType,
  'ServicePortConnectEvent': () => ServicePortConnectEvent.instanceRuntimeType,
  'ServiceWorker': () => _ServiceWorker.instanceRuntimeType,
  'ServiceWorkerContainer': () => ServiceWorkerContainer.instanceRuntimeType,
  'ServiceWorkerGlobalScope': () =>
      ServiceWorkerGlobalScope.instanceRuntimeType,
  'ServiceWorkerMessageEvent': () =>
      ServiceWorkerMessageEvent.instanceRuntimeType,
  'ServiceWorkerRegistration': () =>
      ServiceWorkerRegistration.instanceRuntimeType,
  'ShadowRoot': () => ShadowRoot.instanceRuntimeType,
  'SharedArrayBuffer': () => SharedArrayBuffer.instanceRuntimeType,
  'SharedWorker': () => SharedWorker.instanceRuntimeType,
  'SharedWorkerGlobalScope': () => SharedWorkerGlobalScope.instanceRuntimeType,
  'SourceBuffer': () => SourceBuffer.instanceRuntimeType,
  'SourceBufferList': () => SourceBufferList.instanceRuntimeType,
  'SourceInfo': () => SourceInfo.instanceRuntimeType,
  'SpeechGrammar': () => SpeechGrammar.instanceRuntimeType,
  'SpeechGrammarList': () => SpeechGrammarList.instanceRuntimeType,
  'SpeechRecognition': () => SpeechRecognition.instanceRuntimeType,
  'SpeechRecognitionAlternative': () =>
      SpeechRecognitionAlternative.instanceRuntimeType,
  'SpeechRecognitionError': () => SpeechRecognitionError.instanceRuntimeType,
  'SpeechRecognitionEvent': () => SpeechRecognitionEvent.instanceRuntimeType,
  'SpeechRecognitionResult': () => SpeechRecognitionResult.instanceRuntimeType,
  'SpeechRecognitionResultList': () =>
      _SpeechRecognitionResultList.instanceRuntimeType,
  'SpeechSynthesis': () => SpeechSynthesis.instanceRuntimeType,
  'SpeechSynthesisEvent': () => SpeechSynthesisEvent.instanceRuntimeType,
  'SpeechSynthesisUtterance': () =>
      SpeechSynthesisUtterance.instanceRuntimeType,
  'SpeechSynthesisVoice': () => SpeechSynthesisVoice.instanceRuntimeType,
  'StashedMessagePort': () => StashedMessagePort.instanceRuntimeType,
  'StashedPortCollection': () => StashedPortCollection.instanceRuntimeType,
  'Storage': () => Storage.instanceRuntimeType,
  'StorageEvent': () => StorageEvent.instanceRuntimeType,
  'StorageInfo': () => StorageInfo.instanceRuntimeType,
  'StorageQuota': () => StorageQuota.instanceRuntimeType,
  'Stream': () => FileStream.instanceRuntimeType,
  'StyleMedia': () => StyleMedia.instanceRuntimeType,
  'StyleSheet': () => StyleSheet.instanceRuntimeType,
  'StyleSheetList': () => _StyleSheetList.instanceRuntimeType,
  'SubtleCrypto': () => _SubtleCrypto.instanceRuntimeType,
  'SyncEvent': () => SyncEvent.instanceRuntimeType,
  'SyncManager': () => SyncManager.instanceRuntimeType,
  'SyncRegistration': () => SyncRegistration.instanceRuntimeType,
  'Text': () => Text.instanceRuntimeType,
  'TextEvent': () => TextEvent.instanceRuntimeType,
  'TextMetrics': () => TextMetrics.instanceRuntimeType,
  'TextTrack': () => TextTrack.instanceRuntimeType,
  'TextTrackCue': () => TextTrackCue.instanceRuntimeType,
  'TextTrackCueList': () => TextTrackCueList.instanceRuntimeType,
  'TextTrackList': () => TextTrackList.instanceRuntimeType,
  'TimeRanges': () => TimeRanges.instanceRuntimeType,
  'Touch': () => Touch.instanceRuntimeType,
  'TouchEvent': () => TouchEvent.instanceRuntimeType,
  'TouchList': () => TouchList.instanceRuntimeType,
  'TrackDefault': () => TrackDefault.instanceRuntimeType,
  'TrackDefaultList': () => TrackDefaultList.instanceRuntimeType,
  'TrackEvent': () => TrackEvent.instanceRuntimeType,
  'TransitionEvent': () => TransitionEvent.instanceRuntimeType,
  'TreeWalker': () => TreeWalker.instanceRuntimeType,
  'UIEvent': () => UIEvent.instanceRuntimeType,
  'URL': () => Url.instanceRuntimeType,
  'URLUtils': () => UrlUtils.instanceRuntimeType,
  'URLUtilsReadOnly': () => UrlUtilsReadOnly.instanceRuntimeType,
  'VRDevice': () => VRDevice.instanceRuntimeType,
  'VREyeParameters': () => VREyeParameters.instanceRuntimeType,
  'VRFieldOfView': () => VRFieldOfView.instanceRuntimeType,
  'VRPositionState': () => VRPositionState.instanceRuntimeType,
  'VTTCue': () => VttCue.instanceRuntimeType,
  'VTTRegion': () => VttRegion.instanceRuntimeType,
  'VTTRegionList': () => VttRegionList.instanceRuntimeType,
  'ValidityState': () => ValidityState.instanceRuntimeType,
  'VideoPlaybackQuality': () => VideoPlaybackQuality.instanceRuntimeType,
  'VideoTrack': () => VideoTrack.instanceRuntimeType,
  'VideoTrackList': () => VideoTrackList.instanceRuntimeType,
  'WebKitCSSMatrix': () => _WebKitCSSMatrix.instanceRuntimeType,
  'WebSocket': () => WebSocket.instanceRuntimeType,
  'WheelEvent': () => WheelEvent.instanceRuntimeType,
  'Window': () => Window.instanceRuntimeType,
  'WindowBase64': () => WindowBase64.instanceRuntimeType,
  'WindowClient': () => WindowClient.instanceRuntimeType,
  'WindowEventHandlers': () => WindowEventHandlers.instanceRuntimeType,
  'WindowTimers': () => _WindowTimers.instanceRuntimeType,
  'Worker': () => Worker.instanceRuntimeType,
  'WorkerConsole': () => WorkerConsole.instanceRuntimeType,
  'WorkerGlobalScope': () => WorkerGlobalScope.instanceRuntimeType,
  'WorkerLocation': () => _WorkerLocation.instanceRuntimeType,
  'WorkerNavigator': () => _WorkerNavigator.instanceRuntimeType,
  'WorkerPerformance': () => WorkerPerformance.instanceRuntimeType,
  'XMLDocument': () => XmlDocument.instanceRuntimeType,
  'XMLHttpRequest': () => HttpRequest.instanceRuntimeType,
  'XMLHttpRequestEventTarget': () => HttpRequestEventTarget.instanceRuntimeType,
  'XMLHttpRequestProgressEvent': () =>
      _XMLHttpRequestProgressEvent.instanceRuntimeType,
  'XMLHttpRequestUpload': () => HttpRequestUpload.instanceRuntimeType,
  'XMLSerializer': () => XmlSerializer.instanceRuntimeType,
  'XPathEvaluator': () => XPathEvaluator.instanceRuntimeType,
  'XPathExpression': () => XPathExpression.instanceRuntimeType,
  'XPathNSResolver': () => XPathNSResolver.instanceRuntimeType,
  'XPathResult': () => XPathResult.instanceRuntimeType,
  'XSLTProcessor': () => XsltProcessor.instanceRuntimeType,
};

// TODO(leafp): We may want to move this elsewhere if html becomes
// a package to avoid dartium depending on pkg:html.
@Deprecated("Internal Use Only")
getHtmlCreateType(String key) => _getType(key);

Type _getType(String key) {
  var result;

  // TODO(vsm): Add Cross Frame and JS types here as well.

  // Check the html library.
  result = _getHtmlType(key);
  if (result != null) {
    return result;
  }

  // Check the web gl library.
  result = _getWebGlType(key);
  if (result != null) {
    return result;
  }

  // Check the indexed db library.
  result = _getIndexDbType(key);
  if (result != null) {
    return result;
  }

  // Check the web audio library.
  result = _getWebAudioType(key);
  if (result != null) {
    return result;
  }

  // Check the web sql library.
  result = _getWebSqlType(key);
  if (result != null) {
    return result;
  }

  // Check the svg library.
  result = _getSvgType(key);
  if (result != null) {
    return result;
  }

  return null;
}

Type _getHtmlType(String key) {
  if (htmlBlinkMap.containsKey(key)) {
    return htmlBlinkMap[key]();
  }
  return null;
}

Type _getWebGlType(String key) {
  if (web_glBlinkMap.containsKey(key)) {
    return web_glBlinkMap[key]();
  }
  return null;
}

Type _getIndexDbType(String key) {
  if (indexed_dbBlinkMap.containsKey(key)) {
    return indexed_dbBlinkMap[key]();
  }
  return null;
}

Type _getWebAudioType(String key) {
  if (web_audioBlinkMap.containsKey(key)) {
    return web_audioBlinkMap[key]();
  }
  return null;
}

Type _getWebSqlType(String key) {
  if (web_sqlBlinkMap.containsKey(key)) {
    return web_sqlBlinkMap[key]();
  }
  return null;
}

Type _getSvgType(String key) {
  if (svgBlinkMap.containsKey(key)) {
    return svgBlinkMap[key]();
  }
  return null;
}

// TODO(jacobr): it would be nice to place these conversion methods in a consistent place for dart2js and dartium.

WindowBase _convertNativeToDart_Window(win) {
  if (win == null) return null;
  return _DOMWindowCrossFrame._createSafe(win);
}

EventTarget _convertNativeToDart_EventTarget(e) {
  if (e == null) {
    return null;
  }
  // Assume it's a Window if it contains the postMessage property.  It may be
  // from a different frame - without a patched prototype - so we cannot
  // rely on Dart type checking.
  try {
    if (js.JsNative.hasProperty(e, "postMessage")) {
      var window = _DOMWindowCrossFrame._createSafe(e);
      // If it's a native window.
      if (window is EventTarget) {
        return window;
      }
      return null;
    }
  } catch (err) {
    print("Error calling _convertNativeToDart_EventTarget... $err");
  }
  return e;
}

EventTarget _convertDartToNative_EventTarget(e) {
  // _DOMWindowCrossFrame uses an interceptor so we don't need to do anything unlike Dart2Js.
  return e;
}

_convertNativeToDart_XHR_Response(o) {
  if (o is Document) {
    return o;
  }
  return convertNativeToDart_SerializedScriptValue(o);
}

/******************************************************************************
 **********                                                          **********
 **********                    JS Interop Support                    **********
 **********                                                          **********
 ******************************************************************************/

String _getCustomElementExtends(object) {
  var entry = getCustomElementEntry(object);
  if (entry != null) {
    return entry['extends'];
  }
  return null;
}

// Return the tag name or is attribute of the custom element or data binding.
String _getCustomElementName(element) {
  var jsObject;
  var tag = "";
  var runtimeType = element.runtimeType;
  if (runtimeType == TemplateElement) {
    // Data binding with a Dart class.
    tag = element.attributes['is'];
  } else if (element is HtmlElement) {
    tag = element.attributes['is'];
    if (tag == null) {
      // It's a custom element we want the local name.
      tag = element.localName;
    }
  } else {
    throw new UnsupportedError(
        'Element is incorrect type. Got ${runtimeType}, expected HtmlElement/HtmlTemplate/JsObject.');
  }

  return tag;
}

/// An abstract class for all DOM objects we wrap in dart:html and related
///  libraries.
@Deprecated("Internal Use Only")
class DartHtmlDomObject extends js.JSObject {
  DartHtmlDomObject() : super.internal();
}

@Deprecated("Internal Use Only")
class DebugAssertException implements Exception {
  String message;
  DebugAssertException(this.message);
}

@Deprecated("Internal Use Only")
debug_or_assert(message, expression) {
  if (!expression) {
    throw new DebugAssertException("$message");
  }
}

@Deprecated("Internal Use Only")
Map<String, dynamic> convertNativeObjectToDartMap(js.JsObject jsObject) {
  var result = new Map();
  var keys = js.JsNative.callMethod(
      js.JsNative.getProperty(js.context, 'Object'), 'keys', [jsObject]);
  for (var key in keys) {
    result[key] = js.JsNative.getProperty(jsObject, key);
  }
  return result;
}

/**
 * Upgrade the JS HTMLElement to the Dart class.  Used by Dart's Polymer.
 */
_createCustomUpgrader(Type customElementClass, $this) {
  return _blink.Blink_Utils
      .setInstanceInterceptor($this, customElementClass, customElement: true);
}

// Copyright (c) 2013, the Dart project authors.  Please see the AUTHORS file
// for details. All rights reserved. Use of this source code is governed by a
// BSD-style license that can be found in the LICENSE file.

@DocsEditable()
@DomName('AbstractWorker')
abstract class AbstractWorker extends DartHtmlDomObject implements EventTarget {
  // To suppress missing implicit constructor warnings.
  factory AbstractWorker._() {
    throw new UnsupportedError("Not supported");
  }

  /**
   * Static factory designed to expose `error` events to event
   * handlers that are not necessarily instances of [AbstractWorker].
   *
   * See [EventStreamProvider] for usage information.
   */
  @DomName('AbstractWorker.errorEvent')
  @DocsEditable()
  static const EventStreamProvider<Event> errorEvent =
      const EventStreamProvider<Event>('error');

  /// Stream of `error` events handled by this [AbstractWorker].
  @DomName('AbstractWorker.onerror')
  @DocsEditable()
  Stream<Event> get onError => errorEvent.forTarget(this);
}
// Copyright (c) 2012, the Dart project authors.  Please see the AUTHORS file
// for details. All rights reserved. Use of this source code is governed by a
// BSD-style license that can be found in the LICENSE file.

// WARNING: Do not edit - generated code.

@DocsEditable()
@DomName('HTMLAnchorElement')
class AnchorElement extends HtmlElement implements UrlUtils {
  // To suppress missing implicit constructor warnings.
  factory AnchorElement._() {
    throw new UnsupportedError("Not supported");
  }

  @DomName('HTMLAnchorElement.HTMLAnchorElement')
  @DocsEditable()
  factory AnchorElement({String href}) {
    AnchorElement e = document.createElement("a");
    if (href != null) e.href = href;
    return e;
  }

  @Deprecated("Internal Use Only")
  external static Type get instanceRuntimeType;

  @Deprecated("Internal Use Only")
  AnchorElement.internal_() : super.internal_();

  /**
   * Constructor instantiated by the DOM when a custom element has been created.
   *
   * This can only be called by subclasses from their created constructor.
   */
  AnchorElement.created() : super.created();

  @DomName('HTMLAnchorElement.download')
  @DocsEditable()
  String get download =>
      _blink.BlinkHTMLAnchorElement.instance.download_Getter_(this);

  @DomName('HTMLAnchorElement.download')
  @DocsEditable()
  set download(String value) =>
      _blink.BlinkHTMLAnchorElement.instance.download_Setter_(this, value);

  @DomName('HTMLAnchorElement.hreflang')
  @DocsEditable()
  String get hreflang =>
      _blink.BlinkHTMLAnchorElement.instance.hreflang_Getter_(this);

  @DomName('HTMLAnchorElement.hreflang')
  @DocsEditable()
  set hreflang(String value) =>
      _blink.BlinkHTMLAnchorElement.instance.hreflang_Setter_(this, value);

  @DomName('HTMLAnchorElement.rel')
  @DocsEditable()
  String get rel => _blink.BlinkHTMLAnchorElement.instance.rel_Getter_(this);

  @DomName('HTMLAnchorElement.rel')
  @DocsEditable()
  set rel(String value) =>
      _blink.BlinkHTMLAnchorElement.instance.rel_Setter_(this, value);

  @DomName('HTMLAnchorElement.target')
  @DocsEditable()
  String get target =>
      _blink.BlinkHTMLAnchorElement.instance.target_Getter_(this);

  @DomName('HTMLAnchorElement.target')
  @DocsEditable()
  set target(String value) =>
      _blink.BlinkHTMLAnchorElement.instance.target_Setter_(this, value);

  @DomName('HTMLAnchorElement.type')
  @DocsEditable()
  String get type => _blink.BlinkHTMLAnchorElement.instance.type_Getter_(this);

  @DomName('HTMLAnchorElement.type')
  @DocsEditable()
  set type(String value) =>
      _blink.BlinkHTMLAnchorElement.instance.type_Setter_(this, value);

  @DomName('HTMLAnchorElement.hash')
  @DocsEditable()
  String get hash => _blink.BlinkHTMLAnchorElement.instance.hash_Getter_(this);

  @DomName('HTMLAnchorElement.hash')
  @DocsEditable()
  set hash(String value) =>
      _blink.BlinkHTMLAnchorElement.instance.hash_Setter_(this, value);

  @DomName('HTMLAnchorElement.host')
  @DocsEditable()
  String get host => _blink.BlinkHTMLAnchorElement.instance.host_Getter_(this);

  @DomName('HTMLAnchorElement.host')
  @DocsEditable()
  set host(String value) =>
      _blink.BlinkHTMLAnchorElement.instance.host_Setter_(this, value);

  @DomName('HTMLAnchorElement.hostname')
  @DocsEditable()
  String get hostname =>
      _blink.BlinkHTMLAnchorElement.instance.hostname_Getter_(this);

  @DomName('HTMLAnchorElement.hostname')
  @DocsEditable()
  set hostname(String value) =>
      _blink.BlinkHTMLAnchorElement.instance.hostname_Setter_(this, value);

  @DomName('HTMLAnchorElement.href')
  @DocsEditable()
  String get href => _blink.BlinkHTMLAnchorElement.instance.href_Getter_(this);

  @DomName('HTMLAnchorElement.href')
  @DocsEditable()
  set href(String value) =>
      _blink.BlinkHTMLAnchorElement.instance.href_Setter_(this, value);

  @DomName('HTMLAnchorElement.origin')
  @DocsEditable()
  // WebKit only
  @Experimental() // non-standard
  String get origin =>
      _blink.BlinkHTMLAnchorElement.instance.origin_Getter_(this);

  @DomName('HTMLAnchorElement.password')
  @DocsEditable()
  @Experimental() // untriaged
  String get password =>
      _blink.BlinkHTMLAnchorElement.instance.password_Getter_(this);

  @DomName('HTMLAnchorElement.password')
  @DocsEditable()
  @Experimental() // untriaged
  set password(String value) =>
      _blink.BlinkHTMLAnchorElement.instance.password_Setter_(this, value);

  @DomName('HTMLAnchorElement.pathname')
  @DocsEditable()
  String get pathname =>
      _blink.BlinkHTMLAnchorElement.instance.pathname_Getter_(this);

  @DomName('HTMLAnchorElement.pathname')
  @DocsEditable()
  set pathname(String value) =>
      _blink.BlinkHTMLAnchorElement.instance.pathname_Setter_(this, value);

  @DomName('HTMLAnchorElement.port')
  @DocsEditable()
  String get port => _blink.BlinkHTMLAnchorElement.instance.port_Getter_(this);

  @DomName('HTMLAnchorElement.port')
  @DocsEditable()
  set port(String value) =>
      _blink.BlinkHTMLAnchorElement.instance.port_Setter_(this, value);

  @DomName('HTMLAnchorElement.protocol')
  @DocsEditable()
  String get protocol =>
      _blink.BlinkHTMLAnchorElement.instance.protocol_Getter_(this);

  @DomName('HTMLAnchorElement.protocol')
  @DocsEditable()
  set protocol(String value) =>
      _blink.BlinkHTMLAnchorElement.instance.protocol_Setter_(this, value);

  @DomName('HTMLAnchorElement.search')
  @DocsEditable()
  String get search =>
      _blink.BlinkHTMLAnchorElement.instance.search_Getter_(this);

  @DomName('HTMLAnchorElement.search')
  @DocsEditable()
  set search(String value) =>
      _blink.BlinkHTMLAnchorElement.instance.search_Setter_(this, value);

  @DomName('HTMLAnchorElement.username')
  @DocsEditable()
  @Experimental() // untriaged
  String get username =>
      _blink.BlinkHTMLAnchorElement.instance.username_Getter_(this);

  @DomName('HTMLAnchorElement.username')
  @DocsEditable()
  @Experimental() // untriaged
  set username(String value) =>
      _blink.BlinkHTMLAnchorElement.instance.username_Setter_(this, value);

  @DomName('HTMLAnchorElement.toString')
  @DocsEditable()
  String toString() =>
      _blink.BlinkHTMLAnchorElement.instance.toString_Callback_0_(this);
}
// Copyright (c) 2012, the Dart project authors.  Please see the AUTHORS file
// for details. All rights reserved. Use of this source code is governed by a
// BSD-style license that can be found in the LICENSE file.

// WARNING: Do not edit - generated code.

@DocsEditable()
@DomName('Animation')
@Experimental() // untriaged
class Animation extends EventTarget {
  // To suppress missing implicit constructor warnings.
  factory Animation._() {
    throw new UnsupportedError("Not supported");
  }

  @Deprecated("Internal Use Only")
  external static Type get instanceRuntimeType;

  @Deprecated("Internal Use Only")
  Animation.internal_() : super.internal_();

  /// Checks if this type is supported on the current platform.
  static bool get supported => true;

  @DomName('Animation.currentTime')
  @DocsEditable()
  @Experimental() // untriaged
  num get currentTime =>
      _blink.BlinkAnimation.instance.currentTime_Getter_(this);

  @DomName('Animation.currentTime')
  @DocsEditable()
  @Experimental() // untriaged
  set currentTime(num value) =>
      _blink.BlinkAnimation.instance.currentTime_Setter_(this, value);

  @DomName('Animation.effect')
  @DocsEditable()
  @Experimental() // untriaged
  AnimationEffectReadOnly get effect =>
      _blink.BlinkAnimation.instance.effect_Getter_(this);

  @DomName('Animation.effect')
  @DocsEditable()
  @Experimental() // untriaged
  set effect(AnimationEffectReadOnly value) =>
      _blink.BlinkAnimation.instance.effect_Setter_(this, value);

  @DomName('Animation.endClip')
  @DocsEditable()
  @Experimental() // untriaged
  num get endClip => _blink.BlinkAnimation.instance.endClip_Getter_(this);

  @DomName('Animation.endClip')
  @DocsEditable()
  @Experimental() // untriaged
  set endClip(num value) =>
      _blink.BlinkAnimation.instance.endClip_Setter_(this, value);

  @DomName('Animation.finished')
  @DocsEditable()
  @Experimental() // untriaged
  Future get finished => convertNativePromiseToDartFuture(
      _blink.BlinkAnimation.instance.finished_Getter_(this));

  @DomName('Animation.playState')
  @DocsEditable()
  @Experimental() // untriaged
  String get playState =>
      _blink.BlinkAnimation.instance.playState_Getter_(this);

  @DomName('Animation.playbackRate')
  @DocsEditable()
  @Experimental() // untriaged
  num get playbackRate =>
      _blink.BlinkAnimation.instance.playbackRate_Getter_(this);

  @DomName('Animation.playbackRate')
  @DocsEditable()
  @Experimental() // untriaged
  set playbackRate(num value) =>
      _blink.BlinkAnimation.instance.playbackRate_Setter_(this, value);

  @DomName('Animation.ready')
  @DocsEditable()
  @Experimental() // untriaged
  Future get ready => convertNativePromiseToDartFuture(
      _blink.BlinkAnimation.instance.ready_Getter_(this));

  @DomName('Animation.startClip')
  @DocsEditable()
  @Experimental() // untriaged
  num get startClip => _blink.BlinkAnimation.instance.startClip_Getter_(this);

  @DomName('Animation.startClip')
  @DocsEditable()
  @Experimental() // untriaged
  set startClip(num value) =>
      _blink.BlinkAnimation.instance.startClip_Setter_(this, value);

  @DomName('Animation.startTime')
  @DocsEditable()
  @Experimental() // untriaged
  num get startTime => _blink.BlinkAnimation.instance.startTime_Getter_(this);

  @DomName('Animation.startTime')
  @DocsEditable()
  @Experimental() // untriaged
  set startTime(num value) =>
      _blink.BlinkAnimation.instance.startTime_Setter_(this, value);

  @DomName('Animation.cancel')
  @DocsEditable()
  @Experimental() // untriaged
  void cancel() => _blink.BlinkAnimation.instance.cancel_Callback_0_(this);

  @DomName('Animation.finish')
  @DocsEditable()
  @Experimental() // untriaged
  void finish() => _blink.BlinkAnimation.instance.finish_Callback_0_(this);

  @DomName('Animation.pause')
  @DocsEditable()
  @Experimental() // untriaged
  void pause() => _blink.BlinkAnimation.instance.pause_Callback_0_(this);

  @DomName('Animation.play')
  @DocsEditable()
  @Experimental() // untriaged
  void play() => _blink.BlinkAnimation.instance.play_Callback_0_(this);

  @DomName('Animation.reverse')
  @DocsEditable()
  @Experimental() // untriaged
  void reverse() => _blink.BlinkAnimation.instance.reverse_Callback_0_(this);
}
// Copyright (c) 2012, the Dart project authors.  Please see the AUTHORS file
// for details. All rights reserved. Use of this source code is governed by a
// BSD-style license that can be found in the LICENSE file.

// WARNING: Do not edit - generated code.

@DocsEditable()
@DomName('AnimationEffectReadOnly')
@Experimental() // untriaged
class AnimationEffectReadOnly extends DartHtmlDomObject {
  // To suppress missing implicit constructor warnings.
  factory AnimationEffectReadOnly._() {
    throw new UnsupportedError("Not supported");
  }

  @Deprecated("Internal Use Only")
  external static Type get instanceRuntimeType;

  @Deprecated("Internal Use Only")
  AnimationEffectReadOnly.internal_() {}

  @DomName('AnimationEffectReadOnly.computedTiming')
  @DocsEditable()
  @Experimental() // untriaged
  get computedTiming => convertNativeDictionaryToDartDictionary((_blink
      .BlinkAnimationEffectReadOnly.instance
      .computedTiming_Getter_(this)));

  @DomName('AnimationEffectReadOnly.timing')
  @DocsEditable()
  @Experimental() // untriaged
  AnimationEffectTiming get timing =>
      _blink.BlinkAnimationEffectReadOnly.instance.timing_Getter_(this);
}
// Copyright (c) 2012, the Dart project authors.  Please see the AUTHORS file
// for details. All rights reserved. Use of this source code is governed by a
// BSD-style license that can be found in the LICENSE file.

// WARNING: Do not edit - generated code.

@DocsEditable()
@DomName('AnimationEffectTiming')
@Experimental() // untriaged
class AnimationEffectTiming extends DartHtmlDomObject {
  // To suppress missing implicit constructor warnings.
  factory AnimationEffectTiming._() {
    throw new UnsupportedError("Not supported");
  }

  @Deprecated("Internal Use Only")
  external static Type get instanceRuntimeType;

  @Deprecated("Internal Use Only")
  AnimationEffectTiming.internal_() {}

  @DomName('AnimationEffectTiming.delay')
  @DocsEditable()
  @Experimental() // untriaged
  num get delay =>
      _blink.BlinkAnimationEffectTiming.instance.delay_Getter_(this);

  @DomName('AnimationEffectTiming.delay')
  @DocsEditable()
  @Experimental() // untriaged
  set delay(num value) =>
      _blink.BlinkAnimationEffectTiming.instance.delay_Setter_(this, value);

  @DomName('AnimationEffectTiming.direction')
  @DocsEditable()
  @Experimental() // untriaged
  String get direction =>
      _blink.BlinkAnimationEffectTiming.instance.direction_Getter_(this);

  @DomName('AnimationEffectTiming.direction')
  @DocsEditable()
  @Experimental() // untriaged
  set direction(String value) =>
      _blink.BlinkAnimationEffectTiming.instance.direction_Setter_(this, value);

  @DomName('AnimationEffectTiming.duration')
  @DocsEditable()
  @Experimental() // untriaged
  Object get duration =>
      (_blink.BlinkAnimationEffectTiming.instance.duration_Getter_(this));

  @DomName('AnimationEffectTiming.duration')
  @DocsEditable()
  @Experimental() // untriaged
  set duration(Object value) =>
      _blink.BlinkAnimationEffectTiming.instance.duration_Setter_(this, value);

  @DomName('AnimationEffectTiming.easing')
  @DocsEditable()
  @Experimental() // untriaged
  String get easing =>
      _blink.BlinkAnimationEffectTiming.instance.easing_Getter_(this);

  @DomName('AnimationEffectTiming.easing')
  @DocsEditable()
  @Experimental() // untriaged
  set easing(String value) =>
      _blink.BlinkAnimationEffectTiming.instance.easing_Setter_(this, value);

  @DomName('AnimationEffectTiming.endDelay')
  @DocsEditable()
  @Experimental() // untriaged
  num get endDelay =>
      _blink.BlinkAnimationEffectTiming.instance.endDelay_Getter_(this);

  @DomName('AnimationEffectTiming.endDelay')
  @DocsEditable()
  @Experimental() // untriaged
  set endDelay(num value) =>
      _blink.BlinkAnimationEffectTiming.instance.endDelay_Setter_(this, value);

  @DomName('AnimationEffectTiming.fill')
  @DocsEditable()
  @Experimental() // untriaged
  String get fill =>
      _blink.BlinkAnimationEffectTiming.instance.fill_Getter_(this);

  @DomName('AnimationEffectTiming.fill')
  @DocsEditable()
  @Experimental() // untriaged
  set fill(String value) =>
      _blink.BlinkAnimationEffectTiming.instance.fill_Setter_(this, value);

  @DomName('AnimationEffectTiming.iterationStart')
  @DocsEditable()
  @Experimental() // untriaged
  num get iterationStart =>
      _blink.BlinkAnimationEffectTiming.instance.iterationStart_Getter_(this);

  @DomName('AnimationEffectTiming.iterationStart')
  @DocsEditable()
  @Experimental() // untriaged
  set iterationStart(num value) => _blink.BlinkAnimationEffectTiming.instance
      .iterationStart_Setter_(this, value);

  @DomName('AnimationEffectTiming.iterations')
  @DocsEditable()
  @Experimental() // untriaged
  num get iterations =>
      _blink.BlinkAnimationEffectTiming.instance.iterations_Getter_(this);

  @DomName('AnimationEffectTiming.iterations')
  @DocsEditable()
  @Experimental() // untriaged
  set iterations(num value) => _blink.BlinkAnimationEffectTiming.instance
      .iterations_Setter_(this, value);

  @DomName('AnimationEffectTiming.playbackRate')
  @DocsEditable()
  @Experimental() // untriaged
  num get playbackRate =>
      _blink.BlinkAnimationEffectTiming.instance.playbackRate_Getter_(this);

  @DomName('AnimationEffectTiming.playbackRate')
  @DocsEditable()
  @Experimental() // untriaged
  set playbackRate(num value) => _blink.BlinkAnimationEffectTiming.instance
      .playbackRate_Setter_(this, value);
}
// Copyright (c) 2012, the Dart project authors.  Please see the AUTHORS file
// for details. All rights reserved. Use of this source code is governed by a
// BSD-style license that can be found in the LICENSE file.

// WARNING: Do not edit - generated code.

@DocsEditable()
@DomName('AnimationEvent')
@Experimental() // untriaged
class AnimationEvent extends Event {
  // To suppress missing implicit constructor warnings.
  factory AnimationEvent._() {
    throw new UnsupportedError("Not supported");
  }

  @DomName('AnimationEvent.AnimationEvent')
  @DocsEditable()
  factory AnimationEvent(String type, [Map eventInitDict]) {
    if (eventInitDict != null) {
      var eventInitDict_1 = convertDartToNative_Dictionary(eventInitDict);
      return _blink.BlinkAnimationEvent.instance
          .constructorCallback_2_(type, eventInitDict_1);
    }
    return _blink.BlinkAnimationEvent.instance.constructorCallback_1_(type);
  }

  @Deprecated("Internal Use Only")
  external static Type get instanceRuntimeType;

  @Deprecated("Internal Use Only")
  AnimationEvent.internal_() : super.internal_();

  @DomName('AnimationEvent.animationName')
  @DocsEditable()
  @Experimental() // untriaged
  String get animationName =>
      _blink.BlinkAnimationEvent.instance.animationName_Getter_(this);

  @DomName('AnimationEvent.elapsedTime')
  @DocsEditable()
  @Experimental() // untriaged
  num get elapsedTime =>
      _blink.BlinkAnimationEvent.instance.elapsedTime_Getter_(this);
}
// Copyright (c) 2012, the Dart project authors.  Please see the AUTHORS file
// for details. All rights reserved. Use of this source code is governed by a
// BSD-style license that can be found in the LICENSE file.

// WARNING: Do not edit - generated code.

@DocsEditable()
@DomName('AnimationPlayerEvent')
@Experimental() // untriaged
class AnimationPlayerEvent extends Event {
  // To suppress missing implicit constructor warnings.
  factory AnimationPlayerEvent._() {
    throw new UnsupportedError("Not supported");
  }

  @DomName('AnimationPlayerEvent.AnimationPlayerEvent')
  @DocsEditable()
  factory AnimationPlayerEvent(String type, [Map eventInitDict]) {
    if (eventInitDict != null) {
      var eventInitDict_1 = convertDartToNative_Dictionary(eventInitDict);
      return _blink.BlinkAnimationPlayerEvent.instance
          .constructorCallback_2_(type, eventInitDict_1);
    }
    return _blink.BlinkAnimationPlayerEvent.instance
        .constructorCallback_1_(type);
  }

  @Deprecated("Internal Use Only")
  external static Type get instanceRuntimeType;

  @Deprecated("Internal Use Only")
  AnimationPlayerEvent.internal_() : super.internal_();

  @DomName('AnimationPlayerEvent.currentTime')
  @DocsEditable()
  @Experimental() // untriaged
  num get currentTime =>
      _blink.BlinkAnimationPlayerEvent.instance.currentTime_Getter_(this);

  @DomName('AnimationPlayerEvent.timelineTime')
  @DocsEditable()
  @Experimental() // untriaged
  num get timelineTime =>
      _blink.BlinkAnimationPlayerEvent.instance.timelineTime_Getter_(this);
}
// Copyright (c) 2012, the Dart project authors.  Please see the AUTHORS file
// for details. All rights reserved. Use of this source code is governed by a
// BSD-style license that can be found in the LICENSE file.

// WARNING: Do not edit - generated code.

@DocsEditable()
@DomName('AnimationTimeline')
@Experimental() // untriaged
class AnimationTimeline extends DartHtmlDomObject {
  // To suppress missing implicit constructor warnings.
  factory AnimationTimeline._() {
    throw new UnsupportedError("Not supported");
  }

  @Deprecated("Internal Use Only")
  external static Type get instanceRuntimeType;

  @Deprecated("Internal Use Only")
  AnimationTimeline.internal_() {}

  @DomName('AnimationTimeline.currentTime')
  @DocsEditable()
  @Experimental() // untriaged
  num get currentTime =>
      _blink.BlinkAnimationTimeline.instance.currentTime_Getter_(this);

  @DomName('AnimationTimeline.currentTime')
  @DocsEditable()
  @Experimental() // untriaged
  set currentTime(num value) =>
      _blink.BlinkAnimationTimeline.instance.currentTime_Setter_(this, value);

  @DomName('AnimationTimeline.playbackRate')
  @DocsEditable()
  @Experimental() // untriaged
  num get playbackRate =>
      _blink.BlinkAnimationTimeline.instance.playbackRate_Getter_(this);

  @DomName('AnimationTimeline.playbackRate')
  @DocsEditable()
  @Experimental() // untriaged
  set playbackRate(num value) =>
      _blink.BlinkAnimationTimeline.instance.playbackRate_Setter_(this, value);

  @DomName('AnimationTimeline.getAnimations')
  @DocsEditable()
  @Experimental() // untriaged
  List<Animation> getAnimations() =>
      (_blink.BlinkAnimationTimeline.instance.getAnimations_Callback_0_(this));

  @DomName('AnimationTimeline.play')
  @DocsEditable()
  @Experimental() // untriaged
  Animation play(AnimationEffectReadOnly source) =>
      _blink.BlinkAnimationTimeline.instance.play_Callback_1_(this, source);
}
// Copyright (c) 2012, the Dart project authors.  Please see the AUTHORS file
// for details. All rights reserved. Use of this source code is governed by a
// BSD-style license that can be found in the LICENSE file.

// WARNING: Do not edit - generated code.

@DocsEditable()
@DomName('AppBannerPromptResult')
@Experimental() // untriaged
class AppBannerPromptResult extends DartHtmlDomObject {
  // To suppress missing implicit constructor warnings.
  factory AppBannerPromptResult._() {
    throw new UnsupportedError("Not supported");
  }

  @Deprecated("Internal Use Only")
  external static Type get instanceRuntimeType;

  @Deprecated("Internal Use Only")
  AppBannerPromptResult.internal_() {}

  @DomName('AppBannerPromptResult.outcome')
  @DocsEditable()
  @Experimental() // untriaged
  String get outcome =>
      _blink.BlinkAppBannerPromptResult.instance.outcome_Getter_(this);

  @DomName('AppBannerPromptResult.platform')
  @DocsEditable()
  @Experimental() // untriaged
  String get platform =>
      _blink.BlinkAppBannerPromptResult.instance.platform_Getter_(this);
}
// Copyright (c) 2012, the Dart project authors.  Please see the AUTHORS file
// for details. All rights reserved. Use of this source code is governed by a
// BSD-style license that can be found in the LICENSE file.

// WARNING: Do not edit - generated code.

@DocsEditable()
/**
 * ApplicationCache is accessed via [Window.applicationCache].
 */
@DomName('ApplicationCache')
@SupportedBrowser(SupportedBrowser.CHROME)
@SupportedBrowser(SupportedBrowser.FIREFOX)
@SupportedBrowser(SupportedBrowser.IE, '10')
@SupportedBrowser(SupportedBrowser.OPERA)
@SupportedBrowser(SupportedBrowser.SAFARI)
@Unstable()
class ApplicationCache extends EventTarget {
  // To suppress missing implicit constructor warnings.
  factory ApplicationCache._() {
    throw new UnsupportedError("Not supported");
  }

  /**
   * Static factory designed to expose `cached` events to event
   * handlers that are not necessarily instances of [ApplicationCache].
   *
   * See [EventStreamProvider] for usage information.
   */
  @DomName('ApplicationCache.cachedEvent')
  @DocsEditable()
  static const EventStreamProvider<Event> cachedEvent =
      const EventStreamProvider<Event>('cached');

  /**
   * Static factory designed to expose `checking` events to event
   * handlers that are not necessarily instances of [ApplicationCache].
   *
   * See [EventStreamProvider] for usage information.
   */
  @DomName('ApplicationCache.checkingEvent')
  @DocsEditable()
  static const EventStreamProvider<Event> checkingEvent =
      const EventStreamProvider<Event>('checking');

  /**
   * Static factory designed to expose `downloading` events to event
   * handlers that are not necessarily instances of [ApplicationCache].
   *
   * See [EventStreamProvider] for usage information.
   */
  @DomName('ApplicationCache.downloadingEvent')
  @DocsEditable()
  static const EventStreamProvider<Event> downloadingEvent =
      const EventStreamProvider<Event>('downloading');

  /**
   * Static factory designed to expose `error` events to event
   * handlers that are not necessarily instances of [ApplicationCache].
   *
   * See [EventStreamProvider] for usage information.
   */
  @DomName('ApplicationCache.errorEvent')
  @DocsEditable()
  static const EventStreamProvider<Event> errorEvent =
      const EventStreamProvider<Event>('error');

  /**
   * Static factory designed to expose `noupdate` events to event
   * handlers that are not necessarily instances of [ApplicationCache].
   *
   * See [EventStreamProvider] for usage information.
   */
  @DomName('ApplicationCache.noupdateEvent')
  @DocsEditable()
  static const EventStreamProvider<Event> noUpdateEvent =
      const EventStreamProvider<Event>('noupdate');

  /**
   * Static factory designed to expose `obsolete` events to event
   * handlers that are not necessarily instances of [ApplicationCache].
   *
   * See [EventStreamProvider] for usage information.
   */
  @DomName('ApplicationCache.obsoleteEvent')
  @DocsEditable()
  static const EventStreamProvider<Event> obsoleteEvent =
      const EventStreamProvider<Event>('obsolete');

  /**
   * Static factory designed to expose `progress` events to event
   * handlers that are not necessarily instances of [ApplicationCache].
   *
   * See [EventStreamProvider] for usage information.
   */
  @DomName('ApplicationCache.progressEvent')
  @DocsEditable()
  static const EventStreamProvider<ProgressEvent> progressEvent =
      const EventStreamProvider<ProgressEvent>('progress');

  /**
   * Static factory designed to expose `updateready` events to event
   * handlers that are not necessarily instances of [ApplicationCache].
   *
   * See [EventStreamProvider] for usage information.
   */
  @DomName('ApplicationCache.updatereadyEvent')
  @DocsEditable()
  static const EventStreamProvider<Event> updateReadyEvent =
      const EventStreamProvider<Event>('updateready');

  @Deprecated("Internal Use Only")
  external static Type get instanceRuntimeType;

  @Deprecated("Internal Use Only")
  ApplicationCache.internal_() : super.internal_();

  /// Checks if this type is supported on the current platform.
  static bool get supported => true;

  @DomName('ApplicationCache.CHECKING')
  @DocsEditable()
  static const int CHECKING = 2;

  @DomName('ApplicationCache.DOWNLOADING')
  @DocsEditable()
  static const int DOWNLOADING = 3;

  @DomName('ApplicationCache.IDLE')
  @DocsEditable()
  static const int IDLE = 1;

  @DomName('ApplicationCache.OBSOLETE')
  @DocsEditable()
  static const int OBSOLETE = 5;

  @DomName('ApplicationCache.UNCACHED')
  @DocsEditable()
  static const int UNCACHED = 0;

  @DomName('ApplicationCache.UPDATEREADY')
  @DocsEditable()
  static const int UPDATEREADY = 4;

  @DomName('ApplicationCache.status')
  @DocsEditable()
  int get status => _blink.BlinkApplicationCache.instance.status_Getter_(this);

  @DomName('ApplicationCache.abort')
  @DocsEditable()
  void abort() => _blink.BlinkApplicationCache.instance.abort_Callback_0_(this);

  @DomName('ApplicationCache.swapCache')
  @DocsEditable()
  void swapCache() =>
      _blink.BlinkApplicationCache.instance.swapCache_Callback_0_(this);

  @DomName('ApplicationCache.update')
  @DocsEditable()
  void update() =>
      _blink.BlinkApplicationCache.instance.update_Callback_0_(this);

  /// Stream of `cached` events handled by this [ApplicationCache].
  @DomName('ApplicationCache.oncached')
  @DocsEditable()
  Stream<Event> get onCached => cachedEvent.forTarget(this);

  /// Stream of `checking` events handled by this [ApplicationCache].
  @DomName('ApplicationCache.onchecking')
  @DocsEditable()
  Stream<Event> get onChecking => checkingEvent.forTarget(this);

  /// Stream of `downloading` events handled by this [ApplicationCache].
  @DomName('ApplicationCache.ondownloading')
  @DocsEditable()
  Stream<Event> get onDownloading => downloadingEvent.forTarget(this);

  /// Stream of `error` events handled by this [ApplicationCache].
  @DomName('ApplicationCache.onerror')
  @DocsEditable()
  Stream<Event> get onError => errorEvent.forTarget(this);

  /// Stream of `noupdate` events handled by this [ApplicationCache].
  @DomName('ApplicationCache.onnoupdate')
  @DocsEditable()
  Stream<Event> get onNoUpdate => noUpdateEvent.forTarget(this);

  /// Stream of `obsolete` events handled by this [ApplicationCache].
  @DomName('ApplicationCache.onobsolete')
  @DocsEditable()
  Stream<Event> get onObsolete => obsoleteEvent.forTarget(this);

  /// Stream of `progress` events handled by this [ApplicationCache].
  @DomName('ApplicationCache.onprogress')
  @DocsEditable()
  Stream<ProgressEvent> get onProgress => progressEvent.forTarget(this);

  /// Stream of `updateready` events handled by this [ApplicationCache].
  @DomName('ApplicationCache.onupdateready')
  @DocsEditable()
  Stream<Event> get onUpdateReady => updateReadyEvent.forTarget(this);
}
// Copyright (c) 2012, the Dart project authors.  Please see the AUTHORS file
// for details. All rights reserved. Use of this source code is governed by a
// BSD-style license that can be found in the LICENSE file.

// WARNING: Do not edit - generated code.

@DocsEditable()
@DomName('ApplicationCacheErrorEvent')
@Experimental() // untriaged
class ApplicationCacheErrorEvent extends Event {
  // To suppress missing implicit constructor warnings.
  factory ApplicationCacheErrorEvent._() {
    throw new UnsupportedError("Not supported");
  }

  @DomName('ApplicationCacheErrorEvent.ApplicationCacheErrorEvent')
  @DocsEditable()
  factory ApplicationCacheErrorEvent(String type, [Map eventInitDict]) {
    if (eventInitDict != null) {
      var eventInitDict_1 = convertDartToNative_Dictionary(eventInitDict);
      return _blink.BlinkApplicationCacheErrorEvent.instance
          .constructorCallback_2_(type, eventInitDict_1);
    }
    return _blink.BlinkApplicationCacheErrorEvent.instance
        .constructorCallback_1_(type);
  }

  @Deprecated("Internal Use Only")
  external static Type get instanceRuntimeType;

  @Deprecated("Internal Use Only")
  ApplicationCacheErrorEvent.internal_() : super.internal_();

  @DomName('ApplicationCacheErrorEvent.message')
  @DocsEditable()
  @Experimental() // untriaged
  String get message =>
      _blink.BlinkApplicationCacheErrorEvent.instance.message_Getter_(this);

  @DomName('ApplicationCacheErrorEvent.reason')
  @DocsEditable()
  @Experimental() // untriaged
  String get reason =>
      _blink.BlinkApplicationCacheErrorEvent.instance.reason_Getter_(this);

  @DomName('ApplicationCacheErrorEvent.status')
  @DocsEditable()
  @Experimental() // untriaged
  int get status =>
      _blink.BlinkApplicationCacheErrorEvent.instance.status_Getter_(this);

  @DomName('ApplicationCacheErrorEvent.url')
  @DocsEditable()
  @Experimental() // untriaged
  String get url =>
      _blink.BlinkApplicationCacheErrorEvent.instance.url_Getter_(this);
}
// Copyright (c) 2012, the Dart project authors.  Please see the AUTHORS file
// for details. All rights reserved. Use of this source code is governed by a
// BSD-style license that can be found in the LICENSE file.

// WARNING: Do not edit - generated code.

@DocsEditable()
/**
 * DOM Area Element, which links regions of an image map with a hyperlink.
 *
 * The element can also define an uninteractive region of the map.
 *
 * See also:
 *
 * * [`<area>`](https://developer.mozilla.org/en-US/docs/HTML/Element/area)
 * on MDN.
 */
@DomName('HTMLAreaElement')
class AreaElement extends HtmlElement implements UrlUtils {
  // To suppress missing implicit constructor warnings.
  factory AreaElement._() {
    throw new UnsupportedError("Not supported");
  }

  @DomName('HTMLAreaElement.HTMLAreaElement')
  @DocsEditable()
  factory AreaElement() => document.createElement("area");

  @Deprecated("Internal Use Only")
  external static Type get instanceRuntimeType;

  @Deprecated("Internal Use Only")
  AreaElement.internal_() : super.internal_();

  /**
   * Constructor instantiated by the DOM when a custom element has been created.
   *
   * This can only be called by subclasses from their created constructor.
   */
  AreaElement.created() : super.created();

  @DomName('HTMLAreaElement.alt')
  @DocsEditable()
  String get alt => _blink.BlinkHTMLAreaElement.instance.alt_Getter_(this);

  @DomName('HTMLAreaElement.alt')
  @DocsEditable()
  set alt(String value) =>
      _blink.BlinkHTMLAreaElement.instance.alt_Setter_(this, value);

  @DomName('HTMLAreaElement.coords')
  @DocsEditable()
  String get coords =>
      _blink.BlinkHTMLAreaElement.instance.coords_Getter_(this);

  @DomName('HTMLAreaElement.coords')
  @DocsEditable()
  set coords(String value) =>
      _blink.BlinkHTMLAreaElement.instance.coords_Setter_(this, value);

  @DomName('HTMLAreaElement.shape')
  @DocsEditable()
  String get shape => _blink.BlinkHTMLAreaElement.instance.shape_Getter_(this);

  @DomName('HTMLAreaElement.shape')
  @DocsEditable()
  set shape(String value) =>
      _blink.BlinkHTMLAreaElement.instance.shape_Setter_(this, value);

  @DomName('HTMLAreaElement.target')
  @DocsEditable()
  String get target =>
      _blink.BlinkHTMLAreaElement.instance.target_Getter_(this);

  @DomName('HTMLAreaElement.target')
  @DocsEditable()
  set target(String value) =>
      _blink.BlinkHTMLAreaElement.instance.target_Setter_(this, value);

  @DomName('HTMLAreaElement.hash')
  @DocsEditable()
  String get hash => _blink.BlinkHTMLAreaElement.instance.hash_Getter_(this);

  @DomName('HTMLAreaElement.hash')
  @DocsEditable()
  set hash(String value) =>
      _blink.BlinkHTMLAreaElement.instance.hash_Setter_(this, value);

  @DomName('HTMLAreaElement.host')
  @DocsEditable()
  String get host => _blink.BlinkHTMLAreaElement.instance.host_Getter_(this);

  @DomName('HTMLAreaElement.host')
  @DocsEditable()
  set host(String value) =>
      _blink.BlinkHTMLAreaElement.instance.host_Setter_(this, value);

  @DomName('HTMLAreaElement.hostname')
  @DocsEditable()
  String get hostname =>
      _blink.BlinkHTMLAreaElement.instance.hostname_Getter_(this);

  @DomName('HTMLAreaElement.hostname')
  @DocsEditable()
  set hostname(String value) =>
      _blink.BlinkHTMLAreaElement.instance.hostname_Setter_(this, value);

  @DomName('HTMLAreaElement.href')
  @DocsEditable()
  String get href => _blink.BlinkHTMLAreaElement.instance.href_Getter_(this);

  @DomName('HTMLAreaElement.href')
  @DocsEditable()
  set href(String value) =>
      _blink.BlinkHTMLAreaElement.instance.href_Setter_(this, value);

  @DomName('HTMLAreaElement.origin')
  @DocsEditable()
  @Experimental() // untriaged
  String get origin =>
      _blink.BlinkHTMLAreaElement.instance.origin_Getter_(this);

  @DomName('HTMLAreaElement.password')
  @DocsEditable()
  @Experimental() // untriaged
  String get password =>
      _blink.BlinkHTMLAreaElement.instance.password_Getter_(this);

  @DomName('HTMLAreaElement.password')
  @DocsEditable()
  @Experimental() // untriaged
  set password(String value) =>
      _blink.BlinkHTMLAreaElement.instance.password_Setter_(this, value);

  @DomName('HTMLAreaElement.pathname')
  @DocsEditable()
  String get pathname =>
      _blink.BlinkHTMLAreaElement.instance.pathname_Getter_(this);

  @DomName('HTMLAreaElement.pathname')
  @DocsEditable()
  set pathname(String value) =>
      _blink.BlinkHTMLAreaElement.instance.pathname_Setter_(this, value);

  @DomName('HTMLAreaElement.port')
  @DocsEditable()
  String get port => _blink.BlinkHTMLAreaElement.instance.port_Getter_(this);

  @DomName('HTMLAreaElement.port')
  @DocsEditable()
  set port(String value) =>
      _blink.BlinkHTMLAreaElement.instance.port_Setter_(this, value);

  @DomName('HTMLAreaElement.protocol')
  @DocsEditable()
  String get protocol =>
      _blink.BlinkHTMLAreaElement.instance.protocol_Getter_(this);

  @DomName('HTMLAreaElement.protocol')
  @DocsEditable()
  set protocol(String value) =>
      _blink.BlinkHTMLAreaElement.instance.protocol_Setter_(this, value);

  @DomName('HTMLAreaElement.search')
  @DocsEditable()
  String get search =>
      _blink.BlinkHTMLAreaElement.instance.search_Getter_(this);

  @DomName('HTMLAreaElement.search')
  @DocsEditable()
  set search(String value) =>
      _blink.BlinkHTMLAreaElement.instance.search_Setter_(this, value);

  @DomName('HTMLAreaElement.username')
  @DocsEditable()
  @Experimental() // untriaged
  String get username =>
      _blink.BlinkHTMLAreaElement.instance.username_Getter_(this);

  @DomName('HTMLAreaElement.username')
  @DocsEditable()
  @Experimental() // untriaged
  set username(String value) =>
      _blink.BlinkHTMLAreaElement.instance.username_Setter_(this, value);

  @DomName('HTMLAreaElement.toString')
  @DocsEditable()
  @Experimental() // untriaged
  String toString() =>
      _blink.BlinkHTMLAreaElement.instance.toString_Callback_0_(this);
}
// Copyright (c) 2012, the Dart project authors.  Please see the AUTHORS file
// for details. All rights reserved. Use of this source code is governed by a
// BSD-style license that can be found in the LICENSE file.

@DocsEditable()
@DomName('HTMLAudioElement')
class AudioElement extends MediaElement {
  @DomName('HTMLAudioElement.HTMLAudioElement')
  @DocsEditable()
  factory AudioElement._([String src]) {
    return _blink.BlinkHTMLAudioElement.instance.constructorCallback_1_(src);
  }

  @Deprecated("Internal Use Only")
  external static Type get instanceRuntimeType;

  @Deprecated("Internal Use Only")
  AudioElement.internal_() : super.internal_();

  /**
   * Constructor instantiated by the DOM when a custom element has been created.
   *
   * This can only be called by subclasses from their created constructor.
   */
  AudioElement.created() : super.created();

  factory AudioElement([String src]) {
    if (src == null)
      return _blink.BlinkHTMLAudioElement.instance.constructorCallback_0_();
    else
      return _blink.BlinkHTMLAudioElement.instance.constructorCallback_1_(src);
  }
}
// Copyright (c) 2012, the Dart project authors.  Please see the AUTHORS file
// for details. All rights reserved. Use of this source code is governed by a
// BSD-style license that can be found in the LICENSE file.

// WARNING: Do not edit - generated code.

@DocsEditable()
@DomName('AudioTrack')
@Experimental() // untriaged
class AudioTrack extends DartHtmlDomObject {
  // To suppress missing implicit constructor warnings.
  factory AudioTrack._() {
    throw new UnsupportedError("Not supported");
  }

  @Deprecated("Internal Use Only")
  external static Type get instanceRuntimeType;

  @Deprecated("Internal Use Only")
  AudioTrack.internal_() {}

  @DomName('AudioTrack.enabled')
  @DocsEditable()
  @Experimental() // untriaged
  bool get enabled => _blink.BlinkAudioTrack.instance.enabled_Getter_(this);

  @DomName('AudioTrack.enabled')
  @DocsEditable()
  @Experimental() // untriaged
  set enabled(bool value) =>
      _blink.BlinkAudioTrack.instance.enabled_Setter_(this, value);

  @DomName('AudioTrack.id')
  @DocsEditable()
  @Experimental() // untriaged
  String get id => _blink.BlinkAudioTrack.instance.id_Getter_(this);

  @DomName('AudioTrack.kind')
  @DocsEditable()
  @Experimental() // untriaged
  String get kind => _blink.BlinkAudioTrack.instance.kind_Getter_(this);

  @DomName('AudioTrack.label')
  @DocsEditable()
  @Experimental() // untriaged
  String get label => _blink.BlinkAudioTrack.instance.label_Getter_(this);

  @DomName('AudioTrack.language')
  @DocsEditable()
  @Experimental() // untriaged
  String get language => _blink.BlinkAudioTrack.instance.language_Getter_(this);
}
// Copyright (c) 2012, the Dart project authors.  Please see the AUTHORS file
// for details. All rights reserved. Use of this source code is governed by a
// BSD-style license that can be found in the LICENSE file.

// WARNING: Do not edit - generated code.

@DocsEditable()
@DomName('AudioTrackList')
@Experimental() // untriaged
class AudioTrackList extends EventTarget {
  // To suppress missing implicit constructor warnings.
  factory AudioTrackList._() {
    throw new UnsupportedError("Not supported");
  }

  @DomName('AudioTrackList.changeEvent')
  @DocsEditable()
  @Experimental() // untriaged
  static const EventStreamProvider<Event> changeEvent =
      const EventStreamProvider<Event>('change');

  @Deprecated("Internal Use Only")
  external static Type get instanceRuntimeType;

  @Deprecated("Internal Use Only")
  AudioTrackList.internal_() : super.internal_();

  @DomName('AudioTrackList.length')
  @DocsEditable()
  @Experimental() // untriaged
  int get length => _blink.BlinkAudioTrackList.instance.length_Getter_(this);

  @DomName('AudioTrackList.__getter__')
  @DocsEditable()
  @Experimental() // untriaged
  AudioTrack __getter__(int index) =>
      _blink.BlinkAudioTrackList.instance.$__getter___Callback_1_(this, index);

  @DomName('AudioTrackList.getTrackById')
  @DocsEditable()
  @Experimental() // untriaged
  AudioTrack getTrackById(String id) =>
      _blink.BlinkAudioTrackList.instance.getTrackById_Callback_1_(this, id);

  @DomName('AudioTrackList.onchange')
  @DocsEditable()
  @Experimental() // untriaged
  Stream<Event> get onChange => changeEvent.forTarget(this);
}
// Copyright (c) 2012, the Dart project authors.  Please see the AUTHORS file
// for details. All rights reserved. Use of this source code is governed by a
// BSD-style license that can be found in the LICENSE file.

// WARNING: Do not edit - generated code.

@DocsEditable()
@DomName('AutocompleteErrorEvent')
// http://wiki.whatwg.org/wiki/RequestAutocomplete
@Experimental()
class AutocompleteErrorEvent extends Event {
  // To suppress missing implicit constructor warnings.
  factory AutocompleteErrorEvent._() {
    throw new UnsupportedError("Not supported");
  }

  @DomName('AutocompleteErrorEvent.AutocompleteErrorEvent')
  @DocsEditable()
  factory AutocompleteErrorEvent(String type, [Map eventInitDict]) {
    if (eventInitDict != null) {
      var eventInitDict_1 = convertDartToNative_Dictionary(eventInitDict);
      return _blink.BlinkAutocompleteErrorEvent.instance
          .constructorCallback_2_(type, eventInitDict_1);
    }
    return _blink.BlinkAutocompleteErrorEvent.instance
        .constructorCallback_1_(type);
  }

  @Deprecated("Internal Use Only")
  external static Type get instanceRuntimeType;

  @Deprecated("Internal Use Only")
  AutocompleteErrorEvent.internal_() : super.internal_();

  @DomName('AutocompleteErrorEvent.reason')
  @DocsEditable()
  String get reason =>
      _blink.BlinkAutocompleteErrorEvent.instance.reason_Getter_(this);
}
// Copyright (c) 2012, the Dart project authors.  Please see the AUTHORS file
// for details. All rights reserved. Use of this source code is governed by a
// BSD-style license that can be found in the LICENSE file.

// WARNING: Do not edit - generated code.

@DocsEditable()
@DomName('HTMLBRElement')
class BRElement extends HtmlElement {
  // To suppress missing implicit constructor warnings.
  factory BRElement._() {
    throw new UnsupportedError("Not supported");
  }

  @DomName('HTMLBRElement.HTMLBRElement')
  @DocsEditable()
  factory BRElement() => document.createElement("br");

  @Deprecated("Internal Use Only")
  external static Type get instanceRuntimeType;

  @Deprecated("Internal Use Only")
  BRElement.internal_() : super.internal_();

  /**
   * Constructor instantiated by the DOM when a custom element has been created.
   *
   * This can only be called by subclasses from their created constructor.
   */
  BRElement.created() : super.created();
}
// Copyright (c) 2012, the Dart project authors.  Please see the AUTHORS file
// for details. All rights reserved. Use of this source code is governed by a
// BSD-style license that can be found in the LICENSE file.

// WARNING: Do not edit - generated code.

@DocsEditable()
@DomName('BarProp')
// http://www.whatwg.org/specs/web-apps/current-work/multipage/browsers.html#barprop
@deprecated // standard
class BarProp extends DartHtmlDomObject {
  // To suppress missing implicit constructor warnings.
  factory BarProp._() {
    throw new UnsupportedError("Not supported");
  }

  @Deprecated("Internal Use Only")
  external static Type get instanceRuntimeType;

  @Deprecated("Internal Use Only")
  BarProp.internal_() {}

  @DomName('BarProp.visible')
  @DocsEditable()
  bool get visible => _blink.BlinkBarProp.instance.visible_Getter_(this);
}
// Copyright (c) 2012, the Dart project authors.  Please see the AUTHORS file
// for details. All rights reserved. Use of this source code is governed by a
// BSD-style license that can be found in the LICENSE file.

// WARNING: Do not edit - generated code.

@DocsEditable()
@DomName('HTMLBaseElement')
class BaseElement extends HtmlElement {
  // To suppress missing implicit constructor warnings.
  factory BaseElement._() {
    throw new UnsupportedError("Not supported");
  }

  @DomName('HTMLBaseElement.HTMLBaseElement')
  @DocsEditable()
  factory BaseElement() => document.createElement("base");

  @Deprecated("Internal Use Only")
  external static Type get instanceRuntimeType;

  @Deprecated("Internal Use Only")
  BaseElement.internal_() : super.internal_();

  /**
   * Constructor instantiated by the DOM when a custom element has been created.
   *
   * This can only be called by subclasses from their created constructor.
   */
  BaseElement.created() : super.created();

  @DomName('HTMLBaseElement.href')
  @DocsEditable()
  String get href => _blink.BlinkHTMLBaseElement.instance.href_Getter_(this);

  @DomName('HTMLBaseElement.href')
  @DocsEditable()
  set href(String value) =>
      _blink.BlinkHTMLBaseElement.instance.href_Setter_(this, value);

  @DomName('HTMLBaseElement.target')
  @DocsEditable()
  String get target =>
      _blink.BlinkHTMLBaseElement.instance.target_Getter_(this);

  @DomName('HTMLBaseElement.target')
  @DocsEditable()
  set target(String value) =>
      _blink.BlinkHTMLBaseElement.instance.target_Setter_(this, value);
}
// Copyright (c) 2012, the Dart project authors.  Please see the AUTHORS file
// for details. All rights reserved. Use of this source code is governed by a
// BSD-style license that can be found in the LICENSE file.

// WARNING: Do not edit - generated code.

@DocsEditable()
@DomName('BatteryManager')
// https://dvcs.w3.org/hg/dap/raw-file/default/battery/Overview.html#batterymanager-interface
@Experimental() // stable
class BatteryManager extends EventTarget {
  // To suppress missing implicit constructor warnings.
  factory BatteryManager._() {
    throw new UnsupportedError("Not supported");
  }

  @Deprecated("Internal Use Only")
  external static Type get instanceRuntimeType;

  @Deprecated("Internal Use Only")
  BatteryManager.internal_() : super.internal_();

  @DomName('BatteryManager.charging')
  @DocsEditable()
  bool get charging =>
      _blink.BlinkBatteryManager.instance.charging_Getter_(this);

  @DomName('BatteryManager.chargingTime')
  @DocsEditable()
  num get chargingTime =>
      _blink.BlinkBatteryManager.instance.chargingTime_Getter_(this);

  @DomName('BatteryManager.dischargingTime')
  @DocsEditable()
  num get dischargingTime =>
      _blink.BlinkBatteryManager.instance.dischargingTime_Getter_(this);

  @DomName('BatteryManager.level')
  @DocsEditable()
  num get level => _blink.BlinkBatteryManager.instance.level_Getter_(this);
}
// Copyright (c) 2012, the Dart project authors.  Please see the AUTHORS file
// for details. All rights reserved. Use of this source code is governed by a
// BSD-style license that can be found in the LICENSE file.

// WARNING: Do not edit - generated code.

@DocsEditable()
@DomName('BeforeInstallPromptEvent')
@Experimental() // untriaged
class BeforeInstallPromptEvent extends Event {
  // To suppress missing implicit constructor warnings.
  factory BeforeInstallPromptEvent._() {
    throw new UnsupportedError("Not supported");
  }

  @DomName('BeforeInstallPromptEvent.BeforeInstallPromptEvent')
  @DocsEditable()
  factory BeforeInstallPromptEvent(String type, [Map eventInitDict]) {
    if (eventInitDict != null) {
      var eventInitDict_1 = convertDartToNative_Dictionary(eventInitDict);
      return _blink.BlinkBeforeInstallPromptEvent.instance
          .constructorCallback_2_(type, eventInitDict_1);
    }
    return _blink.BlinkBeforeInstallPromptEvent.instance
        .constructorCallback_1_(type);
  }

  @Deprecated("Internal Use Only")
  external static Type get instanceRuntimeType;

  @Deprecated("Internal Use Only")
  BeforeInstallPromptEvent.internal_() : super.internal_();

  @DomName('BeforeInstallPromptEvent.platforms')
  @DocsEditable()
  @Experimental() // untriaged
  List<String> get platforms =>
      _blink.BlinkBeforeInstallPromptEvent.instance.platforms_Getter_(this);

  @DomName('BeforeInstallPromptEvent.userChoice')
  @DocsEditable()
  @Experimental() // untriaged
  Future get userChoice => convertNativePromiseToDartFuture(
      _blink.BlinkBeforeInstallPromptEvent.instance.userChoice_Getter_(this));

  @DomName('BeforeInstallPromptEvent.prompt')
  @DocsEditable()
  @Experimental() // untriaged
  Future prompt() => convertNativePromiseToDartFuture(
      _blink.BlinkBeforeInstallPromptEvent.instance.prompt_Callback_0_(this));
}
// Copyright (c) 2012, the Dart project authors.  Please see the AUTHORS file
// for details. All rights reserved. Use of this source code is governed by a
// BSD-style license that can be found in the LICENSE file.

// WARNING: Do not edit - generated code.

@DocsEditable()
@DomName('BeforeUnloadEvent')
class BeforeUnloadEvent extends Event {
  // To suppress missing implicit constructor warnings.
  factory BeforeUnloadEvent._() {
    throw new UnsupportedError("Not supported");
  }

  @Deprecated("Internal Use Only")
  external static Type get instanceRuntimeType;

  @Deprecated("Internal Use Only")
  BeforeUnloadEvent.internal_() : super.internal_();

  @DomName('BeforeUnloadEvent.returnValue')
  @DocsEditable()
  String get returnValue =>
      _blink.BlinkBeforeUnloadEvent.instance.returnValue_Getter_(this);

  @DomName('BeforeUnloadEvent.returnValue')
  @DocsEditable()
  set returnValue(String value) =>
      _blink.BlinkBeforeUnloadEvent.instance.returnValue_Setter_(this, value);
}
// Copyright (c) 2012, the Dart project authors.  Please see the AUTHORS file
// for details. All rights reserved. Use of this source code is governed by a
// BSD-style license that can be found in the LICENSE file.

@DomName('Blob')
class Blob extends DartHtmlDomObject {
  // To suppress missing implicit constructor warnings.
  factory Blob._() {
    throw new UnsupportedError("Not supported");
  }

  @Deprecated("Internal Use Only")
  external static Type get instanceRuntimeType;

  @Deprecated("Internal Use Only")
  Blob.internal_() {}

  @DomName('Blob.size')
  @DocsEditable()
  int get size => _blink.BlinkBlob.instance.size_Getter_(this);

  @DomName('Blob.type')
  @DocsEditable()
  String get type => _blink.BlinkBlob.instance.type_Getter_(this);

  @DomName('Blob.close')
  @DocsEditable()
  @Experimental() // untriaged
  void close() => _blink.BlinkBlob.instance.close_Callback_0_(this);

  Blob slice([int start, int end, String contentType]) {
    if (contentType != null) {
      return _blink.BlinkBlob.instance
          .slice_Callback_3_(this, start, end, contentType);
    }
    if (end != null) {
      return _blink.BlinkBlob.instance.slice_Callback_2_(this, start, end);
    }
    if (start != null) {
      return _blink.BlinkBlob.instance.slice_Callback_1_(this, start);
    }
    return _blink.BlinkBlob.instance.slice_Callback_0_(this);
  }

  factory Blob(List blobParts, [String type, String endings]) {
    // TODO: any coercions on the elements of blobParts, e.g. coerce a typed
    // array to ArrayBuffer if it is a total view.

    var parts = convertDartToNative_List(blobParts);
    if (type == null && endings == null) {
      return _blink.BlinkBlob.instance.constructorCallback_1_(parts);
    }
    var bag = {};
    if (type != null) bag['type'] = type;
    if (endings != null) bag['endings'] = endings;
    return _blink.BlinkBlob.instance
        .constructorCallback_2_(parts, convertDartToNative_Dictionary(bag));
  }
}
// Copyright (c) 2012, the Dart project authors.  Please see the AUTHORS file
// for details. All rights reserved. Use of this source code is governed by a
// BSD-style license that can be found in the LICENSE file.

// WARNING: Do not edit - generated code.

@DocsEditable()
@DomName('Bluetooth')
@Experimental() // untriaged
class Bluetooth extends DartHtmlDomObject {
  // To suppress missing implicit constructor warnings.
  factory Bluetooth._() {
    throw new UnsupportedError("Not supported");
  }

  @Deprecated("Internal Use Only")
  external static Type get instanceRuntimeType;

  @Deprecated("Internal Use Only")
  Bluetooth.internal_() {}

  @DomName('Bluetooth.requestDevice')
  @DocsEditable()
  @Experimental() // untriaged
  Future requestDevice(Map options) => convertNativePromiseToDartFuture(
      _blink.BlinkBluetooth.instance.requestDevice_Callback_1_(
          this, convertDartToNative_Dictionary(options)));
}
// Copyright (c) 2012, the Dart project authors.  Please see the AUTHORS file
// for details. All rights reserved. Use of this source code is governed by a
// BSD-style license that can be found in the LICENSE file.

// WARNING: Do not edit - generated code.

@DocsEditable()
@DomName('BluetoothDevice')
@Experimental() // untriaged
class BluetoothDevice extends DartHtmlDomObject {
  // To suppress missing implicit constructor warnings.
  factory BluetoothDevice._() {
    throw new UnsupportedError("Not supported");
  }

  @Deprecated("Internal Use Only")
  external static Type get instanceRuntimeType;

  @Deprecated("Internal Use Only")
  BluetoothDevice.internal_() {}

  @DomName('BluetoothDevice.deviceClass')
  @DocsEditable()
  @Experimental() // untriaged
  int get deviceClass =>
      _blink.BlinkBluetoothDevice.instance.deviceClass_Getter_(this);

  @DomName('BluetoothDevice.instanceID')
  @DocsEditable()
  @Experimental() // untriaged
  String get instanceID =>
      _blink.BlinkBluetoothDevice.instance.instanceID_Getter_(this);

  @DomName('BluetoothDevice.name')
  @DocsEditable()
  @Experimental() // untriaged
  String get name => _blink.BlinkBluetoothDevice.instance.name_Getter_(this);

  @DomName('BluetoothDevice.paired')
  @DocsEditable()
  @Experimental() // untriaged
  bool get paired => _blink.BlinkBluetoothDevice.instance.paired_Getter_(this);

  @DomName('BluetoothDevice.productID')
  @DocsEditable()
  @Experimental() // untriaged
  int get productID =>
      _blink.BlinkBluetoothDevice.instance.productID_Getter_(this);

  @DomName('BluetoothDevice.productVersion')
  @DocsEditable()
  @Experimental() // untriaged
  int get productVersion =>
      _blink.BlinkBluetoothDevice.instance.productVersion_Getter_(this);

  @DomName('BluetoothDevice.vendorID')
  @DocsEditable()
  @Experimental() // untriaged
  int get vendorID =>
      _blink.BlinkBluetoothDevice.instance.vendorID_Getter_(this);

  @DomName('BluetoothDevice.vendorIDSource')
  @DocsEditable()
  @Experimental() // untriaged
  String get vendorIDSource =>
      _blink.BlinkBluetoothDevice.instance.vendorIDSource_Getter_(this);

  @DomName('BluetoothDevice.connectGATT')
  @DocsEditable()
  @Experimental() // untriaged
  Future connectGatt() => convertNativePromiseToDartFuture(
      _blink.BlinkBluetoothDevice.instance.connectGATT_Callback_0_(this));
}
// Copyright (c) 2012, the Dart project authors.  Please see the AUTHORS file
// for details. All rights reserved. Use of this source code is governed by a
// BSD-style license that can be found in the LICENSE file.

// WARNING: Do not edit - generated code.

@DocsEditable()
@DomName('BluetoothGATTCharacteristic')
@Experimental() // untriaged
class BluetoothGattCharacteristic extends DartHtmlDomObject {
  // To suppress missing implicit constructor warnings.
  factory BluetoothGattCharacteristic._() {
    throw new UnsupportedError("Not supported");
  }

  @Deprecated("Internal Use Only")
  external static Type get instanceRuntimeType;

  @Deprecated("Internal Use Only")
  BluetoothGattCharacteristic.internal_() {}

  @DomName('BluetoothGATTCharacteristic.uuid')
  @DocsEditable()
  @Experimental() // untriaged
  String get uuid =>
      _blink.BlinkBluetoothGATTCharacteristic.instance.uuid_Getter_(this);

  @DomName('BluetoothGATTCharacteristic.readValue')
  @DocsEditable()
  @Experimental() // untriaged
  Future readValue() => convertNativePromiseToDartFuture(_blink
      .BlinkBluetoothGATTCharacteristic.instance
      .readValue_Callback_0_(this));

  @DomName('BluetoothGATTCharacteristic.writeValue')
  @DocsEditable()
  @Experimental() // untriaged
  Future writeValue(/*BufferSource*/ value) => convertNativePromiseToDartFuture(
      _blink.BlinkBluetoothGATTCharacteristic.instance
          .writeValue_Callback_1_(this, value));
}
// Copyright (c) 2012, the Dart project authors.  Please see the AUTHORS file
// for details. All rights reserved. Use of this source code is governed by a
// BSD-style license that can be found in the LICENSE file.

// WARNING: Do not edit - generated code.

@DocsEditable()
@DomName('BluetoothGATTRemoteServer')
@Experimental() // untriaged
class BluetoothGattRemoteServer extends DartHtmlDomObject {
  // To suppress missing implicit constructor warnings.
  factory BluetoothGattRemoteServer._() {
    throw new UnsupportedError("Not supported");
  }

  @Deprecated("Internal Use Only")
  external static Type get instanceRuntimeType;

  @Deprecated("Internal Use Only")
  BluetoothGattRemoteServer.internal_() {}

  @DomName('BluetoothGATTRemoteServer.connected')
  @DocsEditable()
  @Experimental() // untriaged
  bool get connected =>
      _blink.BlinkBluetoothGATTRemoteServer.instance.connected_Getter_(this);

  @DomName('BluetoothGATTRemoteServer.getPrimaryService')
  @DocsEditable()
  @Experimental() // untriaged
  Future getPrimaryService(/*BluetoothServiceUUID*/ service) =>
      convertNativePromiseToDartFuture(_blink
          .BlinkBluetoothGATTRemoteServer.instance
          .getPrimaryService_Callback_1_(this, service));
}
// Copyright (c) 2012, the Dart project authors.  Please see the AUTHORS file
// for details. All rights reserved. Use of this source code is governed by a
// BSD-style license that can be found in the LICENSE file.

// WARNING: Do not edit - generated code.

@DocsEditable()
@DomName('BluetoothGATTService')
@Experimental() // untriaged
class BluetoothGattService extends DartHtmlDomObject {
  // To suppress missing implicit constructor warnings.
  factory BluetoothGattService._() {
    throw new UnsupportedError("Not supported");
  }

  @Deprecated("Internal Use Only")
  external static Type get instanceRuntimeType;

  @Deprecated("Internal Use Only")
  BluetoothGattService.internal_() {}

  @DomName('BluetoothGATTService.isPrimary')
  @DocsEditable()
  @Experimental() // untriaged
  bool get isPrimary =>
      _blink.BlinkBluetoothGATTService.instance.isPrimary_Getter_(this);

  @DomName('BluetoothGATTService.uuid')
  @DocsEditable()
  @Experimental() // untriaged
  String get uuid =>
      _blink.BlinkBluetoothGATTService.instance.uuid_Getter_(this);

  @DomName('BluetoothGATTService.getCharacteristic')
  @DocsEditable()
  @Experimental() // untriaged
  Future getCharacteristic(/*BluetoothCharacteristicUUID*/ characteristic) =>
      convertNativePromiseToDartFuture(_blink.BlinkBluetoothGATTService.instance
          .getCharacteristic_Callback_1_(this, characteristic));
}
// Copyright (c) 2012, the Dart project authors.  Please see the AUTHORS file
// for details. All rights reserved. Use of this source code is governed by a
// BSD-style license that can be found in the LICENSE file.

// WARNING: Do not edit - generated code.

@DocsEditable()
@DomName('BluetoothUUID')
@Experimental() // untriaged
class BluetoothUuid extends DartHtmlDomObject {
  // To suppress missing implicit constructor warnings.
  factory BluetoothUuid._() {
    throw new UnsupportedError("Not supported");
  }

  @Deprecated("Internal Use Only")
  external static Type get instanceRuntimeType;

  @Deprecated("Internal Use Only")
  BluetoothUuid.internal_() {}

  @DomName('BluetoothUUID.canonicalUUID')
  @DocsEditable()
  @Experimental() // untriaged
  static String canonicalUuid(int alias) =>
      _blink.BlinkBluetoothUUID.instance.canonicalUUID_Callback_1_(alias);

  @DomName('BluetoothUUID.getCharacteristic')
  @DocsEditable()
  @Experimental() // untriaged
  static String getCharacteristic(Object name) =>
      _blink.BlinkBluetoothUUID.instance.getCharacteristic_Callback_1_(name);

  @DomName('BluetoothUUID.getDescriptor')
  @DocsEditable()
  @Experimental() // untriaged
  static String getDescriptor(Object name) =>
      _blink.BlinkBluetoothUUID.instance.getDescriptor_Callback_1_(name);

  @DomName('BluetoothUUID.getService')
  @DocsEditable()
  @Experimental() // untriaged
  static String getService(Object name) =>
      _blink.BlinkBluetoothUUID.instance.getService_Callback_1_(name);
}
// Copyright (c) 2012, the Dart project authors.  Please see the AUTHORS file
// for details. All rights reserved. Use of this source code is governed by a
// BSD-style license that can be found in the LICENSE file.

// WARNING: Do not edit - generated code.

@DocsEditable()
@DomName('Body')
@Experimental() // untriaged
class Body extends DartHtmlDomObject {
  // To suppress missing implicit constructor warnings.
  factory Body._() {
    throw new UnsupportedError("Not supported");
  }

  @Deprecated("Internal Use Only")
  external static Type get instanceRuntimeType;

  @Deprecated("Internal Use Only")
  Body.internal_() {}

  @DomName('Body.bodyUsed')
  @DocsEditable()
  @Experimental() // untriaged
  bool get bodyUsed => _blink.BlinkBody.instance.bodyUsed_Getter_(this);

  @DomName('Body.arrayBuffer')
  @DocsEditable()
  @Experimental() // untriaged
  Future arrayBuffer() => convertNativePromiseToDartFuture(
      _blink.BlinkBody.instance.arrayBuffer_Callback_0_(this));

  @DomName('Body.blob')
  @DocsEditable()
  @Experimental() // untriaged
  Future blob() => convertNativePromiseToDartFuture(
      _blink.BlinkBody.instance.blob_Callback_0_(this));

  @DomName('Body.json')
  @DocsEditable()
  @Experimental() // untriaged
  Future json() => convertNativePromiseToDartFuture(
      _blink.BlinkBody.instance.json_Callback_0_(this));

  @DomName('Body.text')
  @DocsEditable()
  @Experimental() // untriaged
  Future text() => convertNativePromiseToDartFuture(
      _blink.BlinkBody.instance.text_Callback_0_(this));
}
// Copyright (c) 2012, the Dart project authors.  Please see the AUTHORS file
// for details. All rights reserved. Use of this source code is governed by a
// BSD-style license that can be found in the LICENSE file.

// WARNING: Do not edit - generated code.

@DocsEditable()
@DomName('HTMLBodyElement')
class BodyElement extends HtmlElement implements WindowEventHandlers {
  // To suppress missing implicit constructor warnings.
  factory BodyElement._() {
    throw new UnsupportedError("Not supported");
  }

  /**
   * Static factory designed to expose `blur` events to event
   * handlers that are not necessarily instances of [BodyElement].
   *
   * See [EventStreamProvider] for usage information.
   */
  @DomName('HTMLBodyElement.blurEvent')
  @DocsEditable()
  static const EventStreamProvider<Event> blurEvent =
      const EventStreamProvider<Event>('blur');

  /**
   * Static factory designed to expose `error` events to event
   * handlers that are not necessarily instances of [BodyElement].
   *
   * See [EventStreamProvider] for usage information.
   */
  @DomName('HTMLBodyElement.errorEvent')
  @DocsEditable()
  static const EventStreamProvider<Event> errorEvent =
      const EventStreamProvider<Event>('error');

  /**
   * Static factory designed to expose `focus` events to event
   * handlers that are not necessarily instances of [BodyElement].
   *
   * See [EventStreamProvider] for usage information.
   */
  @DomName('HTMLBodyElement.focusEvent')
  @DocsEditable()
  static const EventStreamProvider<Event> focusEvent =
      const EventStreamProvider<Event>('focus');

  /**
   * Static factory designed to expose `hashchange` events to event
   * handlers that are not necessarily instances of [BodyElement].
   *
   * See [EventStreamProvider] for usage information.
   */
  @DomName('HTMLBodyElement.hashchangeEvent')
  @DocsEditable()
  static const EventStreamProvider<Event> hashChangeEvent =
      const EventStreamProvider<Event>('hashchange');

  /**
   * Static factory designed to expose `load` events to event
   * handlers that are not necessarily instances of [BodyElement].
   *
   * See [EventStreamProvider] for usage information.
   */
  @DomName('HTMLBodyElement.loadEvent')
  @DocsEditable()
  static const EventStreamProvider<Event> loadEvent =
      const EventStreamProvider<Event>('load');

  /**
   * Static factory designed to expose `message` events to event
   * handlers that are not necessarily instances of [BodyElement].
   *
   * See [EventStreamProvider] for usage information.
   */
  @DomName('HTMLBodyElement.messageEvent')
  @DocsEditable()
  static const EventStreamProvider<MessageEvent> messageEvent =
      const EventStreamProvider<MessageEvent>('message');

  /**
   * Static factory designed to expose `offline` events to event
   * handlers that are not necessarily instances of [BodyElement].
   *
   * See [EventStreamProvider] for usage information.
   */
  @DomName('HTMLBodyElement.offlineEvent')
  @DocsEditable()
  static const EventStreamProvider<Event> offlineEvent =
      const EventStreamProvider<Event>('offline');

  /**
   * Static factory designed to expose `online` events to event
   * handlers that are not necessarily instances of [BodyElement].
   *
   * See [EventStreamProvider] for usage information.
   */
  @DomName('HTMLBodyElement.onlineEvent')
  @DocsEditable()
  static const EventStreamProvider<Event> onlineEvent =
      const EventStreamProvider<Event>('online');

  /**
   * Static factory designed to expose `popstate` events to event
   * handlers that are not necessarily instances of [BodyElement].
   *
   * See [EventStreamProvider] for usage information.
   */
  @DomName('HTMLBodyElement.popstateEvent')
  @DocsEditable()
  static const EventStreamProvider<PopStateEvent> popStateEvent =
      const EventStreamProvider<PopStateEvent>('popstate');

  /**
   * Static factory designed to expose `resize` events to event
   * handlers that are not necessarily instances of [BodyElement].
   *
   * See [EventStreamProvider] for usage information.
   */
  @DomName('HTMLBodyElement.resizeEvent')
  @DocsEditable()
  static const EventStreamProvider<Event> resizeEvent =
      const EventStreamProvider<Event>('resize');

  @DomName('HTMLBodyElement.scrollEvent')
  @DocsEditable()
  @Experimental() // untriaged
  static const EventStreamProvider<Event> scrollEvent =
      const EventStreamProvider<Event>('scroll');

  /**
   * Static factory designed to expose `storage` events to event
   * handlers that are not necessarily instances of [BodyElement].
   *
   * See [EventStreamProvider] for usage information.
   */
  @DomName('HTMLBodyElement.storageEvent')
  @DocsEditable()
  static const EventStreamProvider<StorageEvent> storageEvent =
      const EventStreamProvider<StorageEvent>('storage');

  /**
   * Static factory designed to expose `unload` events to event
   * handlers that are not necessarily instances of [BodyElement].
   *
   * See [EventStreamProvider] for usage information.
   */
  @DomName('HTMLBodyElement.unloadEvent')
  @DocsEditable()
  static const EventStreamProvider<Event> unloadEvent =
      const EventStreamProvider<Event>('unload');

  @DomName('HTMLBodyElement.HTMLBodyElement')
  @DocsEditable()
  factory BodyElement() => document.createElement("body");

  @Deprecated("Internal Use Only")
  external static Type get instanceRuntimeType;

  @Deprecated("Internal Use Only")
  BodyElement.internal_() : super.internal_();

  /**
   * Constructor instantiated by the DOM when a custom element has been created.
   *
   * This can only be called by subclasses from their created constructor.
   */
  BodyElement.created() : super.created();

  /// Stream of `blur` events handled by this [BodyElement].
  @DomName('HTMLBodyElement.onblur')
  @DocsEditable()
  ElementStream<Event> get onBlur => blurEvent.forElement(this);

  /// Stream of `error` events handled by this [BodyElement].
  @DomName('HTMLBodyElement.onerror')
  @DocsEditable()
  ElementStream<Event> get onError => errorEvent.forElement(this);

  /// Stream of `focus` events handled by this [BodyElement].
  @DomName('HTMLBodyElement.onfocus')
  @DocsEditable()
  ElementStream<Event> get onFocus => focusEvent.forElement(this);

  /// Stream of `hashchange` events handled by this [BodyElement].
  @DomName('HTMLBodyElement.onhashchange')
  @DocsEditable()
  ElementStream<Event> get onHashChange => hashChangeEvent.forElement(this);

  /// Stream of `load` events handled by this [BodyElement].
  @DomName('HTMLBodyElement.onload')
  @DocsEditable()
  ElementStream<Event> get onLoad => loadEvent.forElement(this);

  /// Stream of `message` events handled by this [BodyElement].
  @DomName('HTMLBodyElement.onmessage')
  @DocsEditable()
  ElementStream<MessageEvent> get onMessage => messageEvent.forElement(this);

  /// Stream of `offline` events handled by this [BodyElement].
  @DomName('HTMLBodyElement.onoffline')
  @DocsEditable()
  ElementStream<Event> get onOffline => offlineEvent.forElement(this);

  /// Stream of `online` events handled by this [BodyElement].
  @DomName('HTMLBodyElement.ononline')
  @DocsEditable()
  ElementStream<Event> get onOnline => onlineEvent.forElement(this);

  /// Stream of `popstate` events handled by this [BodyElement].
  @DomName('HTMLBodyElement.onpopstate')
  @DocsEditable()
  ElementStream<PopStateEvent> get onPopState => popStateEvent.forElement(this);

  /// Stream of `resize` events handled by this [BodyElement].
  @DomName('HTMLBodyElement.onresize')
  @DocsEditable()
  ElementStream<Event> get onResize => resizeEvent.forElement(this);

  @DomName('HTMLBodyElement.onscroll')
  @DocsEditable()
  @Experimental() // untriaged
  ElementStream<Event> get onScroll => scrollEvent.forElement(this);

  /// Stream of `storage` events handled by this [BodyElement].
  @DomName('HTMLBodyElement.onstorage')
  @DocsEditable()
  ElementStream<StorageEvent> get onStorage => storageEvent.forElement(this);

  /// Stream of `unload` events handled by this [BodyElement].
  @DomName('HTMLBodyElement.onunload')
  @DocsEditable()
  ElementStream<Event> get onUnload => unloadEvent.forElement(this);
}
// Copyright (c) 2012, the Dart project authors.  Please see the AUTHORS file
// for details. All rights reserved. Use of this source code is governed by a
// BSD-style license that can be found in the LICENSE file.

// WARNING: Do not edit - generated code.

@DocsEditable()
@DomName('HTMLButtonElement')
class ButtonElement extends HtmlElement {
  // To suppress missing implicit constructor warnings.
  factory ButtonElement._() {
    throw new UnsupportedError("Not supported");
  }

  @DomName('HTMLButtonElement.HTMLButtonElement')
  @DocsEditable()
  factory ButtonElement() => document.createElement("button");

  @Deprecated("Internal Use Only")
  external static Type get instanceRuntimeType;

  @Deprecated("Internal Use Only")
  ButtonElement.internal_() : super.internal_();

  /**
   * Constructor instantiated by the DOM when a custom element has been created.
   *
   * This can only be called by subclasses from their created constructor.
   */
  ButtonElement.created() : super.created();

  @DomName('HTMLButtonElement.autofocus')
  @DocsEditable()
  bool get autofocus =>
      _blink.BlinkHTMLButtonElement.instance.autofocus_Getter_(this);

  @DomName('HTMLButtonElement.autofocus')
  @DocsEditable()
  set autofocus(bool value) =>
      _blink.BlinkHTMLButtonElement.instance.autofocus_Setter_(this, value);

  @DomName('HTMLButtonElement.disabled')
  @DocsEditable()
  bool get disabled =>
      _blink.BlinkHTMLButtonElement.instance.disabled_Getter_(this);

  @DomName('HTMLButtonElement.disabled')
  @DocsEditable()
  set disabled(bool value) =>
      _blink.BlinkHTMLButtonElement.instance.disabled_Setter_(this, value);

  @DomName('HTMLButtonElement.form')
  @DocsEditable()
  FormElement get form =>
      _blink.BlinkHTMLButtonElement.instance.form_Getter_(this);

  @DomName('HTMLButtonElement.formAction')
  @DocsEditable()
  String get formAction =>
      _blink.BlinkHTMLButtonElement.instance.formAction_Getter_(this);

  @DomName('HTMLButtonElement.formAction')
  @DocsEditable()
  set formAction(String value) =>
      _blink.BlinkHTMLButtonElement.instance.formAction_Setter_(this, value);

  @DomName('HTMLButtonElement.formEnctype')
  @DocsEditable()
  String get formEnctype =>
      _blink.BlinkHTMLButtonElement.instance.formEnctype_Getter_(this);

  @DomName('HTMLButtonElement.formEnctype')
  @DocsEditable()
  set formEnctype(String value) =>
      _blink.BlinkHTMLButtonElement.instance.formEnctype_Setter_(this, value);

  @DomName('HTMLButtonElement.formMethod')
  @DocsEditable()
  String get formMethod =>
      _blink.BlinkHTMLButtonElement.instance.formMethod_Getter_(this);

  @DomName('HTMLButtonElement.formMethod')
  @DocsEditable()
  set formMethod(String value) =>
      _blink.BlinkHTMLButtonElement.instance.formMethod_Setter_(this, value);

  @DomName('HTMLButtonElement.formNoValidate')
  @DocsEditable()
  bool get formNoValidate =>
      _blink.BlinkHTMLButtonElement.instance.formNoValidate_Getter_(this);

  @DomName('HTMLButtonElement.formNoValidate')
  @DocsEditable()
  set formNoValidate(bool value) => _blink.BlinkHTMLButtonElement.instance
      .formNoValidate_Setter_(this, value);

  @DomName('HTMLButtonElement.formTarget')
  @DocsEditable()
  String get formTarget =>
      _blink.BlinkHTMLButtonElement.instance.formTarget_Getter_(this);

  @DomName('HTMLButtonElement.formTarget')
  @DocsEditable()
  set formTarget(String value) =>
      _blink.BlinkHTMLButtonElement.instance.formTarget_Setter_(this, value);

  @DomName('HTMLButtonElement.labels')
  @DocsEditable()
  @Unstable()
  List<Node> get labels =>
      (_blink.BlinkHTMLButtonElement.instance.labels_Getter_(this));

  @DomName('HTMLButtonElement.name')
  @DocsEditable()
  String get name => _blink.BlinkHTMLButtonElement.instance.name_Getter_(this);

  @DomName('HTMLButtonElement.name')
  @DocsEditable()
  set name(String value) =>
      _blink.BlinkHTMLButtonElement.instance.name_Setter_(this, value);

  @DomName('HTMLButtonElement.type')
  @DocsEditable()
  String get type => _blink.BlinkHTMLButtonElement.instance.type_Getter_(this);

  @DomName('HTMLButtonElement.type')
  @DocsEditable()
  set type(String value) =>
      _blink.BlinkHTMLButtonElement.instance.type_Setter_(this, value);

  @DomName('HTMLButtonElement.validationMessage')
  @DocsEditable()
  String get validationMessage =>
      _blink.BlinkHTMLButtonElement.instance.validationMessage_Getter_(this);

  @DomName('HTMLButtonElement.validity')
  @DocsEditable()
  ValidityState get validity =>
      _blink.BlinkHTMLButtonElement.instance.validity_Getter_(this);

  @DomName('HTMLButtonElement.value')
  @DocsEditable()
  String get value =>
      _blink.BlinkHTMLButtonElement.instance.value_Getter_(this);

  @DomName('HTMLButtonElement.value')
  @DocsEditable()
  set value(String value) =>
      _blink.BlinkHTMLButtonElement.instance.value_Setter_(this, value);

  @DomName('HTMLButtonElement.willValidate')
  @DocsEditable()
  bool get willValidate =>
      _blink.BlinkHTMLButtonElement.instance.willValidate_Getter_(this);

  @DomName('HTMLButtonElement.checkValidity')
  @DocsEditable()
  bool checkValidity() =>
      _blink.BlinkHTMLButtonElement.instance.checkValidity_Callback_0_(this);

  @DomName('HTMLButtonElement.reportValidity')
  @DocsEditable()
  @Experimental() // untriaged
  bool reportValidity() =>
      _blink.BlinkHTMLButtonElement.instance.reportValidity_Callback_0_(this);

  @DomName('HTMLButtonElement.setCustomValidity')
  @DocsEditable()
  void setCustomValidity(String error) => _blink.BlinkHTMLButtonElement.instance
      .setCustomValidity_Callback_1_(this, error);
}
// Copyright (c) 2012, the Dart project authors.  Please see the AUTHORS file
// for details. All rights reserved. Use of this source code is governed by a
// BSD-style license that can be found in the LICENSE file.

// WARNING: Do not edit - generated code.

@DocsEditable()
@DomName('CDATASection')
// http://dom.spec.whatwg.org/#cdatasection
@deprecated // deprecated
class CDataSection extends Text {
  // To suppress missing implicit constructor warnings.
  factory CDataSection._() {
    throw new UnsupportedError("Not supported");
  }

  @Deprecated("Internal Use Only")
  external static Type get instanceRuntimeType;

  @Deprecated("Internal Use Only")
  CDataSection.internal_() : super.internal_();
}
// Copyright (c) 2012, the Dart project authors.  Please see the AUTHORS file
// for details. All rights reserved. Use of this source code is governed by a
// BSD-style license that can be found in the LICENSE file.

// WARNING: Do not edit - generated code.

@DocsEditable()
@DomName('CacheStorage')
@Experimental() // untriaged
class CacheStorage extends DartHtmlDomObject {
  // To suppress missing implicit constructor warnings.
  factory CacheStorage._() {
    throw new UnsupportedError("Not supported");
  }

  @Deprecated("Internal Use Only")
  external static Type get instanceRuntimeType;

  @Deprecated("Internal Use Only")
  CacheStorage.internal_() {}

  @DomName('CacheStorage.delete')
  @DocsEditable()
  @Experimental() // untriaged
  Future delete(String cacheName) => convertNativePromiseToDartFuture(
      _blink.BlinkCacheStorage.instance.delete_Callback_1_(this, cacheName));

  @DomName('CacheStorage.has')
  @DocsEditable()
  @Experimental() // untriaged
  Future has(String cacheName) => convertNativePromiseToDartFuture(
      _blink.BlinkCacheStorage.instance.has_Callback_1_(this, cacheName));

  @DomName('CacheStorage.keys')
  @DocsEditable()
  @Experimental() // untriaged
  Future keys() => convertNativePromiseToDartFuture(
      _blink.BlinkCacheStorage.instance.keys_Callback_0_(this));

  Future match(/*RequestInfo*/ request, [Map options]) {
    if (options != null) {
      return _blink.BlinkCacheStorage.instance.match_Callback_2_(
          this, request, convertDartToNative_Dictionary(options));
    }
    return _blink.BlinkCacheStorage.instance.match_Callback_1_(this, request);
  }

  @DomName('CacheStorage.open')
  @DocsEditable()
  @Experimental() // untriaged
  Future open(String cacheName) => convertNativePromiseToDartFuture(
      _blink.BlinkCacheStorage.instance.open_Callback_1_(this, cacheName));
}
// Copyright (c) 2012, the Dart project authors.  Please see the AUTHORS file
// for details. All rights reserved. Use of this source code is governed by a
// BSD-style license that can be found in the LICENSE file.

@DomName('HTMLCanvasElement')
class CanvasElement extends HtmlElement implements CanvasImageSource {
  // To suppress missing implicit constructor warnings.
  factory CanvasElement._() {
    throw new UnsupportedError("Not supported");
  }

  /**
   * Static factory designed to expose `webglcontextlost` events to event
   * handlers that are not necessarily instances of [CanvasElement].
   *
   * See [EventStreamProvider] for usage information.
   */
  @DomName('HTMLCanvasElement.webglcontextlostEvent')
  @DocsEditable()
  static const EventStreamProvider<gl.ContextEvent> webGlContextLostEvent =
      const EventStreamProvider<gl.ContextEvent>('webglcontextlost');

  /**
   * Static factory designed to expose `webglcontextrestored` events to event
   * handlers that are not necessarily instances of [CanvasElement].
   *
   * See [EventStreamProvider] for usage information.
   */
  @DomName('HTMLCanvasElement.webglcontextrestoredEvent')
  @DocsEditable()
  static const EventStreamProvider<gl.ContextEvent> webGlContextRestoredEvent =
      const EventStreamProvider<gl.ContextEvent>('webglcontextrestored');

  @DomName('HTMLCanvasElement.HTMLCanvasElement')
  @DocsEditable()
  factory CanvasElement({int width, int height}) {
    CanvasElement e = document.createElement("canvas");
    if (width != null) e.width = width;
    if (height != null) e.height = height;
    return e;
  }

  @Deprecated("Internal Use Only")
  external static Type get instanceRuntimeType;

  @Deprecated("Internal Use Only")
  CanvasElement.internal_() : super.internal_();

  /**
   * Constructor instantiated by the DOM when a custom element has been created.
   *
   * This can only be called by subclasses from their created constructor.
   */
  CanvasElement.created() : super.created();

  /// The height of this canvas element in CSS pixels.
  @DomName('HTMLCanvasElement.height')
  @DocsEditable()
  int get height => _blink.BlinkHTMLCanvasElement.instance.height_Getter_(this);

  /// The height of this canvas element in CSS pixels.
  @DomName('HTMLCanvasElement.height')
  @DocsEditable()
  set height(int value) =>
      _blink.BlinkHTMLCanvasElement.instance.height_Setter_(this, value);

  /// The width of this canvas element in CSS pixels.
  @DomName('HTMLCanvasElement.width')
  @DocsEditable()
  int get width => _blink.BlinkHTMLCanvasElement.instance.width_Getter_(this);

  /// The width of this canvas element in CSS pixels.
  @DomName('HTMLCanvasElement.width')
  @DocsEditable()
  set width(int value) =>
      _blink.BlinkHTMLCanvasElement.instance.width_Setter_(this, value);

  Object getContext(String contextId, [Map attributes]) {
    if (attributes != null) {
      return _blink.BlinkHTMLCanvasElement.instance.getContext_Callback_2_(
          this, contextId, convertDartToNative_Dictionary(attributes));
    }
    return _blink.BlinkHTMLCanvasElement.instance
        .getContext_Callback_1_(this, contextId);
  }

  String _toDataUrl(String type, [arguments_OR_quality]) {
    if ((type is String || type == null) && arguments_OR_quality == null) {
      return _blink.BlinkHTMLCanvasElement.instance
          .toDataURL_Callback_1_(this, type);
    }
    if (arguments_OR_quality != null && (type is String || type == null)) {
      return _blink.BlinkHTMLCanvasElement.instance
          .toDataURL_Callback_2_(this, type, arguments_OR_quality);
    }
    if ((type is String || type == null) && arguments_OR_quality == null) {
      return _blink.BlinkHTMLCanvasElement.instance
          .toDataURL_Callback_1_(this, type);
    }
    if ((arguments_OR_quality is num || arguments_OR_quality == null) &&
        (type is String || type == null)) {
      return _blink.BlinkHTMLCanvasElement.instance
          .toDataURL_Callback_2_(this, type, arguments_OR_quality);
    }
    throw new ArgumentError("Incorrect number or type of arguments");
  }

  /// Stream of `webglcontextlost` events handled by this [CanvasElement].
  @DomName('HTMLCanvasElement.onwebglcontextlost')
  @DocsEditable()
  ElementStream<gl.ContextEvent> get onWebGlContextLost =>
      webGlContextLostEvent.forElement(this);

  /// Stream of `webglcontextrestored` events handled by this [CanvasElement].
  @DomName('HTMLCanvasElement.onwebglcontextrestored')
  @DocsEditable()
  ElementStream<gl.ContextEvent> get onWebGlContextRestored =>
      webGlContextRestoredEvent.forElement(this);

  /** An API for drawing on this canvas. */
  CanvasRenderingContext2D get context2D => getContext('2d');

  /**
   * Returns a new Web GL context for this canvas.
   *
   * ## Other resources
   *
   * * [WebGL fundamentals](http://www.html5rocks.com/en/tutorials/webgl/webgl_fundamentals/)
   *   from HTML5Rocks.
   * * [WebGL homepage](http://get.webgl.org/).
   */
  @SupportedBrowser(SupportedBrowser.CHROME)
  @SupportedBrowser(SupportedBrowser.FIREFOX)
  @Experimental()
  gl.RenderingContext getContext3d(
      {alpha: true,
      depth: true,
      stencil: false,
      antialias: true,
      premultipliedAlpha: true,
      preserveDrawingBuffer: false}) {
    var options = {
      'alpha': alpha,
      'depth': depth,
      'stencil': stencil,
      'antialias': antialias,
      'premultipliedAlpha': premultipliedAlpha,
      'preserveDrawingBuffer': preserveDrawingBuffer,
    };
    var context = getContext('webgl', options);
    if (context == null) {
      context = getContext('experimental-webgl', options);
    }
    return context;
  }

  /**
   * Returns a data URI containing a representation of the image in the
   * format specified by type (defaults to 'image/png').
   *
   * Data Uri format is as follow
   * `data:[<MIME-type>][;charset=<encoding>][;base64],<data>`
   *
   * Optional parameter [quality] in the range of 0.0 and 1.0 can be used when
   * requesting [type] 'image/jpeg' or 'image/webp'. If [quality] is not passed
   * the default value is used. Note: the default value varies by browser.
   *
   * If the height or width of this canvas element is 0, then 'data:' is
   * returned, representing no data.
   *
   * If the type requested is not 'image/png', and the returned value is
   * 'data:image/png', then the requested type is not supported.
   *
   * Example usage:
   *
   *     CanvasElement canvas = new CanvasElement();
   *     var ctx = canvas.context2D
   *     ..fillStyle = "rgb(200,0,0)"
   *     ..fillRect(10, 10, 55, 50);
   *     var dataUrl = canvas.toDataUrl("image/jpeg", 0.95);
   *     // The Data Uri would look similar to
   *     // 'data:image/png;base64,iVBORw0KGgoAAAANSUhEUgAAAAUA
   *     // AAAFCAYAAACNbyblAAAAHElEQVQI12P4//8/w38GIAXDIBKE0DHxgljNBAAO
   *     // 9TXL0Y4OHwAAAABJRU5ErkJggg=='
   *     //Create a new image element from the data URI.
   *     var img = new ImageElement();
   *     img.src = dataUrl;
   *     document.body.children.add(img);
   *
   * See also:
   *
   * * [Data URI Scheme](http://en.wikipedia.org/wiki/Data_URI_scheme) from Wikipedia.
   *
   * * [HTMLCanvasElement](https://developer.mozilla.org/en-US/docs/DOM/HTMLCanvasElement) from MDN.
   *
   * * [toDataUrl](http://dev.w3.org/html5/spec/the-canvas-element.html#dom-canvas-todataurl) from W3C.
   */
  String toDataUrl([String type = 'image/png', num quality]) =>
      _toDataUrl(type, quality);
}
// Copyright (c) 2012, the Dart project authors.  Please see the AUTHORS file
// for details. All rights reserved. Use of this source code is governed by a
// BSD-style license that can be found in the LICENSE file.

// WARNING: Do not edit - generated code.

@DocsEditable()
/**
 * An opaque canvas object representing a gradient.
 *
 * Created by calling [createLinearGradient] or [createRadialGradient] on a
 * [CanvasRenderingContext2D] object.
 *
 * Example usage:
 *
 *     var canvas = new CanvasElement(width: 600, height: 600);
 *     var ctx = canvas.context2D;
 *     ctx.clearRect(0, 0, 600, 600);
 *     ctx.save();
 *     // Create radial gradient.
 *     CanvasGradient gradient = ctx.createRadialGradient(0, 0, 0, 0, 0, 600);
 *     gradient.addColorStop(0, '#000');
 *     gradient.addColorStop(1, 'rgb(255, 255, 255)');
 *     // Assign gradients to fill.
 *     ctx.fillStyle = gradient;
 *     // Draw a rectangle with a gradient fill.
 *     ctx.fillRect(0, 0, 600, 600);
 *     ctx.save();
 *     document.body.children.add(canvas);
 *
 * See also:
 *
 * * [CanvasGradient](https://developer.mozilla.org/en-US/docs/DOM/CanvasGradient) from MDN.
 * * [CanvasGradient](https://html.spec.whatwg.org/multipage/scripting.html#canvasgradient)
 *   from WHATWG.
 * * [CanvasGradient](http://www.w3.org/TR/2010/WD-2dcontext-20100304/#canvasgradient) from W3C.
 */
@DomName('CanvasGradient')
class CanvasGradient extends DartHtmlDomObject {
  // To suppress missing implicit constructor warnings.
  factory CanvasGradient._() {
    throw new UnsupportedError("Not supported");
  }

  @Deprecated("Internal Use Only")
  external static Type get instanceRuntimeType;

  @Deprecated("Internal Use Only")
  CanvasGradient.internal_() {}

  /**
   * Adds a color stop to this gradient at the offset.
   *
   * The [offset] can range between 0.0 and 1.0.
   *
   * See also:
   *
   * * [Multiple Color Stops](https://developer.mozilla.org/en-US/docs/CSS/linear-gradient#Gradient_with_multiple_color_stops) from MDN.
   */
  @DomName('CanvasGradient.addColorStop')
  @DocsEditable()
  void addColorStop(num offset, String color) =>
      _blink.BlinkCanvasGradient.instance
          .addColorStop_Callback_2_(this, offset, color);
}
// Copyright (c) 2012, the Dart project authors.  Please see the AUTHORS file
// for details. All rights reserved. Use of this source code is governed by a
// BSD-style license that can be found in the LICENSE file.

// WARNING: Do not edit - generated code.

@DocsEditable()
/**
 * An opaque object representing a pattern of image, canvas, or video.
 *
 * Created by calling [createPattern] on a [CanvasRenderingContext2D] object.
 *
 * Example usage:
 *
 *     var canvas = new CanvasElement(width: 600, height: 600);
 *     var ctx = canvas.context2D;
 *     var img = new ImageElement();
 *     // Image src needs to be loaded before pattern is applied.
 *     img.onLoad.listen((event) {
 *       // When the image is loaded, create a pattern
 *       // from the ImageElement.
 *       CanvasPattern pattern = ctx.createPattern(img, 'repeat');
 *       ctx.rect(0, 0, canvas.width, canvas.height);
 *       ctx.fillStyle = pattern;
 *       ctx.fill();
 *     });
 *     img.src = "images/foo.jpg";
 *     document.body.children.add(canvas);
 *
 * See also:
 * * [CanvasPattern](https://developer.mozilla.org/en-US/docs/DOM/CanvasPattern) from MDN.
 * * [CanvasPattern](https://html.spec.whatwg.org/multipage/scripting.html#canvaspattern)
 *   from WHATWG.
 * * [CanvasPattern](http://www.w3.org/TR/2010/WD-2dcontext-20100304/#canvaspattern) from W3C.
 */
@DomName('CanvasPattern')
class CanvasPattern extends DartHtmlDomObject {
  // To suppress missing implicit constructor warnings.
  factory CanvasPattern._() {
    throw new UnsupportedError("Not supported");
  }

  @Deprecated("Internal Use Only")
  external static Type get instanceRuntimeType;

  @Deprecated("Internal Use Only")
  CanvasPattern.internal_() {}

  @DomName('CanvasPattern.setTransform')
  @DocsEditable()
  @Experimental() // untriaged
  void setTransform(Matrix transform) => _blink.BlinkCanvasPattern.instance
      .setTransform_Callback_1_(this, transform);
}
// Copyright (c) 2012, the Dart project authors.  Please see the AUTHORS file
// for details. All rights reserved. Use of this source code is governed by a
// BSD-style license that can be found in the LICENSE file.

abstract class CanvasRenderingContext {
  CanvasElement get canvas;
}

@DomName('CanvasRenderingContext2D')
class CanvasRenderingContext2D extends DartHtmlDomObject
    implements CanvasRenderingContext {
  // To suppress missing implicit constructor warnings.
  factory CanvasRenderingContext2D._() {
    throw new UnsupportedError("Not supported");
  }

  @Deprecated("Internal Use Only")
  external static Type get instanceRuntimeType;

  @Deprecated("Internal Use Only")
  CanvasRenderingContext2D.internal_() {}

  @DomName('CanvasRenderingContext2D.canvas')
  @DocsEditable()
  @Experimental() // untriaged
  CanvasElement get canvas =>
      _blink.BlinkCanvasRenderingContext2D.instance.canvas_Getter_(this);

  @DomName('CanvasRenderingContext2D.currentTransform')
  @DocsEditable()
  @Experimental() // untriaged
  Matrix get currentTransform => _blink.BlinkCanvasRenderingContext2D.instance
      .currentTransform_Getter_(this);

  @DomName('CanvasRenderingContext2D.currentTransform')
  @DocsEditable()
  @Experimental() // untriaged
  set currentTransform(Matrix value) =>
      _blink.BlinkCanvasRenderingContext2D.instance
          .currentTransform_Setter_(this, value);

  @DomName('CanvasRenderingContext2D.direction')
  @DocsEditable()
  @Experimental() // untriaged
  String get direction =>
      _blink.BlinkCanvasRenderingContext2D.instance.direction_Getter_(this);

  @DomName('CanvasRenderingContext2D.direction')
  @DocsEditable()
  @Experimental() // untriaged
  set direction(String value) => _blink.BlinkCanvasRenderingContext2D.instance
      .direction_Setter_(this, value);

  @DomName('CanvasRenderingContext2D.fillStyle')
  @DocsEditable()
  Object get fillStyle =>
      (_blink.BlinkCanvasRenderingContext2D.instance.fillStyle_Getter_(this));

  @DomName('CanvasRenderingContext2D.fillStyle')
  @DocsEditable()
  set fillStyle(Object value) => _blink.BlinkCanvasRenderingContext2D.instance
      .fillStyle_Setter_(this, value);

  @DomName('CanvasRenderingContext2D.filter')
  @DocsEditable()
  @Experimental() // untriaged
  String get filter =>
      _blink.BlinkCanvasRenderingContext2D.instance.filter_Getter_(this);

  @DomName('CanvasRenderingContext2D.filter')
  @DocsEditable()
  @Experimental() // untriaged
  set filter(String value) =>
      _blink.BlinkCanvasRenderingContext2D.instance.filter_Setter_(this, value);

  @DomName('CanvasRenderingContext2D.font')
  @DocsEditable()
  String get font =>
      _blink.BlinkCanvasRenderingContext2D.instance.font_Getter_(this);

  @DomName('CanvasRenderingContext2D.font')
  @DocsEditable()
  set font(String value) =>
      _blink.BlinkCanvasRenderingContext2D.instance.font_Setter_(this, value);

  @DomName('CanvasRenderingContext2D.globalAlpha')
  @DocsEditable()
  num get globalAlpha =>
      _blink.BlinkCanvasRenderingContext2D.instance.globalAlpha_Getter_(this);

  @DomName('CanvasRenderingContext2D.globalAlpha')
  @DocsEditable()
  set globalAlpha(num value) => _blink.BlinkCanvasRenderingContext2D.instance
      .globalAlpha_Setter_(this, value);

  @DomName('CanvasRenderingContext2D.globalCompositeOperation')
  @DocsEditable()
  String get globalCompositeOperation =>
      _blink.BlinkCanvasRenderingContext2D.instance
          .globalCompositeOperation_Getter_(this);

  @DomName('CanvasRenderingContext2D.globalCompositeOperation')
  @DocsEditable()
  set globalCompositeOperation(String value) =>
      _blink.BlinkCanvasRenderingContext2D.instance
          .globalCompositeOperation_Setter_(this, value);

  /**
   * Whether images and patterns on this canvas will be smoothed when this
   * canvas is scaled.
   *
   * ## Other resources
   *
   * * [Image
   *   smoothing](https://html.spec.whatwg.org/multipage/scripting.html#image-smoothing)
   *   from WHATWG.
   */
  @DomName('CanvasRenderingContext2D.imageSmoothingEnabled')
  @DocsEditable()
  @Experimental() // untriaged
  bool get imageSmoothingEnabled =>
      _blink.BlinkCanvasRenderingContext2D.instance
          .imageSmoothingEnabled_Getter_(this);

  /**
   * Whether images and patterns on this canvas will be smoothed when this
   * canvas is scaled.
   *
   * ## Other resources
   *
   * * [Image
   *   smoothing](https://html.spec.whatwg.org/multipage/scripting.html#image-smoothing)
   *   from WHATWG.
   */
  @DomName('CanvasRenderingContext2D.imageSmoothingEnabled')
  @DocsEditable()
  @Experimental() // untriaged
  set imageSmoothingEnabled(bool value) =>
      _blink.BlinkCanvasRenderingContext2D.instance
          .imageSmoothingEnabled_Setter_(this, value);

  @DomName('CanvasRenderingContext2D.lineCap')
  @DocsEditable()
  String get lineCap =>
      _blink.BlinkCanvasRenderingContext2D.instance.lineCap_Getter_(this);

  @DomName('CanvasRenderingContext2D.lineCap')
  @DocsEditable()
  set lineCap(String value) => _blink.BlinkCanvasRenderingContext2D.instance
      .lineCap_Setter_(this, value);

  @DomName('CanvasRenderingContext2D.lineDashOffset')
  @DocsEditable()
  num get lineDashOffset => _blink.BlinkCanvasRenderingContext2D.instance
      .lineDashOffset_Getter_(this);

  @DomName('CanvasRenderingContext2D.lineDashOffset')
  @DocsEditable()
  set lineDashOffset(num value) => _blink.BlinkCanvasRenderingContext2D.instance
      .lineDashOffset_Setter_(this, value);

  @DomName('CanvasRenderingContext2D.lineJoin')
  @DocsEditable()
  String get lineJoin =>
      _blink.BlinkCanvasRenderingContext2D.instance.lineJoin_Getter_(this);

  @DomName('CanvasRenderingContext2D.lineJoin')
  @DocsEditable()
  set lineJoin(String value) => _blink.BlinkCanvasRenderingContext2D.instance
      .lineJoin_Setter_(this, value);

  @DomName('CanvasRenderingContext2D.lineWidth')
  @DocsEditable()
  num get lineWidth =>
      _blink.BlinkCanvasRenderingContext2D.instance.lineWidth_Getter_(this);

  @DomName('CanvasRenderingContext2D.lineWidth')
  @DocsEditable()
  set lineWidth(num value) => _blink.BlinkCanvasRenderingContext2D.instance
      .lineWidth_Setter_(this, value);

  @DomName('CanvasRenderingContext2D.miterLimit')
  @DocsEditable()
  num get miterLimit =>
      _blink.BlinkCanvasRenderingContext2D.instance.miterLimit_Getter_(this);

  @DomName('CanvasRenderingContext2D.miterLimit')
  @DocsEditable()
  set miterLimit(num value) => _blink.BlinkCanvasRenderingContext2D.instance
      .miterLimit_Setter_(this, value);

  @DomName('CanvasRenderingContext2D.shadowBlur')
  @DocsEditable()
  num get shadowBlur =>
      _blink.BlinkCanvasRenderingContext2D.instance.shadowBlur_Getter_(this);

  @DomName('CanvasRenderingContext2D.shadowBlur')
  @DocsEditable()
  set shadowBlur(num value) => _blink.BlinkCanvasRenderingContext2D.instance
      .shadowBlur_Setter_(this, value);

  @DomName('CanvasRenderingContext2D.shadowColor')
  @DocsEditable()
  String get shadowColor =>
      _blink.BlinkCanvasRenderingContext2D.instance.shadowColor_Getter_(this);

  @DomName('CanvasRenderingContext2D.shadowColor')
  @DocsEditable()
  set shadowColor(String value) => _blink.BlinkCanvasRenderingContext2D.instance
      .shadowColor_Setter_(this, value);

  @DomName('CanvasRenderingContext2D.shadowOffsetX')
  @DocsEditable()
  num get shadowOffsetX =>
      _blink.BlinkCanvasRenderingContext2D.instance.shadowOffsetX_Getter_(this);

  @DomName('CanvasRenderingContext2D.shadowOffsetX')
  @DocsEditable()
  set shadowOffsetX(num value) => _blink.BlinkCanvasRenderingContext2D.instance
      .shadowOffsetX_Setter_(this, value);

  @DomName('CanvasRenderingContext2D.shadowOffsetY')
  @DocsEditable()
  num get shadowOffsetY =>
      _blink.BlinkCanvasRenderingContext2D.instance.shadowOffsetY_Getter_(this);

  @DomName('CanvasRenderingContext2D.shadowOffsetY')
  @DocsEditable()
  set shadowOffsetY(num value) => _blink.BlinkCanvasRenderingContext2D.instance
      .shadowOffsetY_Setter_(this, value);

  @DomName('CanvasRenderingContext2D.strokeStyle')
  @DocsEditable()
  Object get strokeStyle =>
      (_blink.BlinkCanvasRenderingContext2D.instance.strokeStyle_Getter_(this));

  @DomName('CanvasRenderingContext2D.strokeStyle')
  @DocsEditable()
  set strokeStyle(Object value) => _blink.BlinkCanvasRenderingContext2D.instance
      .strokeStyle_Setter_(this, value);

  @DomName('CanvasRenderingContext2D.textAlign')
  @DocsEditable()
  String get textAlign =>
      _blink.BlinkCanvasRenderingContext2D.instance.textAlign_Getter_(this);

  @DomName('CanvasRenderingContext2D.textAlign')
  @DocsEditable()
  set textAlign(String value) => _blink.BlinkCanvasRenderingContext2D.instance
      .textAlign_Setter_(this, value);

  @DomName('CanvasRenderingContext2D.textBaseline')
  @DocsEditable()
  String get textBaseline =>
      _blink.BlinkCanvasRenderingContext2D.instance.textBaseline_Getter_(this);

  @DomName('CanvasRenderingContext2D.textBaseline')
  @DocsEditable()
  set textBaseline(String value) =>
      _blink.BlinkCanvasRenderingContext2D.instance
          .textBaseline_Setter_(this, value);

  void addHitRegion([Map options]) {
    if (options != null) {
      _blink.BlinkCanvasRenderingContext2D.instance.addHitRegion_Callback_1_(
          this, convertDartToNative_Dictionary(options));
      return;
    }
    _blink.BlinkCanvasRenderingContext2D.instance
        .addHitRegion_Callback_0_(this);
    return;
  }

  @DomName('CanvasRenderingContext2D.beginPath')
  @DocsEditable()
  void beginPath() =>
      _blink.BlinkCanvasRenderingContext2D.instance.beginPath_Callback_0_(this);

  @DomName('CanvasRenderingContext2D.clearHitRegions')
  @DocsEditable()
  @Experimental() // untriaged
  void clearHitRegions() => _blink.BlinkCanvasRenderingContext2D.instance
      .clearHitRegions_Callback_0_(this);

  @DomName('CanvasRenderingContext2D.clearRect')
  @DocsEditable()
  void clearRect(num x, num y, num width, num height) =>
      _blink.BlinkCanvasRenderingContext2D.instance
          .clearRect_Callback_4_(this, x, y, width, height);

  void clip([path_OR_winding, String winding]) {
    if (path_OR_winding == null && winding == null) {
      _blink.BlinkCanvasRenderingContext2D.instance.clip_Callback_0_(this);
      return;
    }
    if ((path_OR_winding is String) && winding == null) {
      _blink.BlinkCanvasRenderingContext2D.instance
          .clip_Callback_1_(this, path_OR_winding);
      return;
    }
    if ((path_OR_winding is Path2D) && winding == null) {
      _blink.BlinkCanvasRenderingContext2D.instance
          .clip_Callback_1_(this, path_OR_winding);
      return;
    }
    if ((winding is String) && (path_OR_winding is Path2D)) {
      _blink.BlinkCanvasRenderingContext2D.instance
          .clip_Callback_2_(this, path_OR_winding, winding);
      return;
    }
    throw new ArgumentError("Incorrect number or type of arguments");
  }

  ImageData createImageData(imagedata_OR_sw, [num sh]) {
    if ((imagedata_OR_sw is ImageData) && sh == null) {
      return _blink.BlinkCanvasRenderingContext2D.instance
          .createImageData_Callback_1_(this, imagedata_OR_sw);
    }
    if ((sh is num) && (imagedata_OR_sw is num)) {
      return _blink.BlinkCanvasRenderingContext2D.instance
          .createImageData_Callback_2_(this, imagedata_OR_sw, sh);
    }
    throw new ArgumentError("Incorrect number or type of arguments");
  }

  @DomName('CanvasRenderingContext2D.createLinearGradient')
  @DocsEditable()
  CanvasGradient createLinearGradient(num x0, num y0, num x1, num y1) =>
      _blink.BlinkCanvasRenderingContext2D.instance
          .createLinearGradient_Callback_4_(this, x0, y0, x1, y1);

  @DomName('CanvasRenderingContext2D.createPattern')
  @DocsEditable()
  CanvasPattern createPattern(Object image, String repetitionType) =>
      _blink.BlinkCanvasRenderingContext2D.instance
          .createPattern_Callback_2_(this, image, repetitionType);

  @DomName('CanvasRenderingContext2D.createRadialGradient')
  @DocsEditable()
  CanvasGradient createRadialGradient(
          num x0, num y0, num r0, num x1, num y1, num r1) =>
      _blink.BlinkCanvasRenderingContext2D.instance
          .createRadialGradient_Callback_6_(this, x0, y0, r0, x1, y1, r1);

  void drawFocusIfNeeded(element_OR_path, [Element element]) {
    if ((element_OR_path is Element) && element == null) {
      _blink.BlinkCanvasRenderingContext2D.instance
          .drawFocusIfNeeded_Callback_1_(this, element_OR_path);
      return;
    }
    if ((element is Element) && (element_OR_path is Path2D)) {
      _blink.BlinkCanvasRenderingContext2D.instance
          .drawFocusIfNeeded_Callback_2_(this, element_OR_path, element);
      return;
    }
    throw new ArgumentError("Incorrect number or type of arguments");
  }

  void _drawImage(Object image, num sx_OR_x, num sy_OR_y,
      [num sw_OR_width, num height_OR_sh, num dx, num dy, num dw, num dh]) {
    if ((sy_OR_y is num) &&
        (sx_OR_x is num) &&
        image != null &&
        sw_OR_width == null &&
        height_OR_sh == null &&
        dx == null &&
        dy == null &&
        dw == null &&
        dh == null) {
      _blink.BlinkCanvasRenderingContext2D.instance
          .drawImage_Callback_3_(this, image, sx_OR_x, sy_OR_y);
      return;
    }
    if ((height_OR_sh is num) &&
        (sw_OR_width is num) &&
        (sy_OR_y is num) &&
        (sx_OR_x is num) &&
        image != null &&
        dx == null &&
        dy == null &&
        dw == null &&
        dh == null) {
      _blink.BlinkCanvasRenderingContext2D.instance.drawImage_Callback_5_(
          this, image, sx_OR_x, sy_OR_y, sw_OR_width, height_OR_sh);
      return;
    }
    if ((dh is num) &&
        (dw is num) &&
        (dy is num) &&
        (dx is num) &&
        (height_OR_sh is num) &&
        (sw_OR_width is num) &&
        (sy_OR_y is num) &&
        (sx_OR_x is num) &&
        image != null) {
      _blink.BlinkCanvasRenderingContext2D.instance.drawImage_Callback_9_(this,
          image, sx_OR_x, sy_OR_y, sw_OR_width, height_OR_sh, dx, dy, dw, dh);
      return;
    }
    throw new ArgumentError("Incorrect number or type of arguments");
  }

  void fill([path_OR_winding, String winding]) {
    if (path_OR_winding == null && winding == null) {
      _blink.BlinkCanvasRenderingContext2D.instance.fill_Callback_0_(this);
      return;
    }
    if ((path_OR_winding is String) && winding == null) {
      _blink.BlinkCanvasRenderingContext2D.instance
          .fill_Callback_1_(this, path_OR_winding);
      return;
    }
    if ((path_OR_winding is Path2D) && winding == null) {
      _blink.BlinkCanvasRenderingContext2D.instance
          .fill_Callback_1_(this, path_OR_winding);
      return;
    }
    if ((winding is String) && (path_OR_winding is Path2D)) {
      _blink.BlinkCanvasRenderingContext2D.instance
          .fill_Callback_2_(this, path_OR_winding, winding);
      return;
    }
    throw new ArgumentError("Incorrect number or type of arguments");
  }

  @DomName('CanvasRenderingContext2D.fillRect')
  @DocsEditable()
  void fillRect(num x, num y, num width, num height) =>
      _blink.BlinkCanvasRenderingContext2D.instance
          .fillRect_Callback_4_(this, x, y, width, height);

  void fillText(String text, num x, num y, [num maxWidth]) {
    if (maxWidth != null) {
      _blink.BlinkCanvasRenderingContext2D.instance
          .fillText_Callback_4_(this, text, x, y, maxWidth);
      return;
    }
    _blink.BlinkCanvasRenderingContext2D.instance
        .fillText_Callback_3_(this, text, x, y);
    return;
  }

  @DomName('CanvasRenderingContext2D.getContextAttributes')
  @DocsEditable()
  // http://wiki.whatwg.org/wiki/CanvasOpaque#Suggested_IDL
  @Experimental()
  getContextAttributes() => convertNativeDictionaryToDartDictionary((_blink
      .BlinkCanvasRenderingContext2D.instance
      .getContextAttributes_Callback_0_(this)));

  @DomName('CanvasRenderingContext2D.getImageData')
  @DocsEditable()
  ImageData getImageData(num sx, num sy, num sw, num sh) =>
      _blink.BlinkCanvasRenderingContext2D.instance
          .getImageData_Callback_4_(this, sx, sy, sw, sh);

  @DomName('CanvasRenderingContext2D.getLineDash')
  @DocsEditable()
  List<num> _getLineDash() => _blink.BlinkCanvasRenderingContext2D.instance
      .getLineDash_Callback_0_(this);

  @DomName('CanvasRenderingContext2D.isContextLost')
  @DocsEditable()
  @Experimental() // untriaged
  bool isContextLost() => _blink.BlinkCanvasRenderingContext2D.instance
      .isContextLost_Callback_0_(this);

  bool isPointInPath(path_OR_x, num x_OR_y, [winding_OR_y, String winding]) {
    if ((x_OR_y is num) &&
        (path_OR_x is num) &&
        winding_OR_y == null &&
        winding == null) {
      return _blink.BlinkCanvasRenderingContext2D.instance
          .isPointInPath_Callback_2_(this, path_OR_x, x_OR_y);
    }
    if ((winding_OR_y is String) &&
        (x_OR_y is num) &&
        (path_OR_x is num) &&
        winding == null) {
      return _blink.BlinkCanvasRenderingContext2D.instance
          .isPointInPath_Callback_3_(this, path_OR_x, x_OR_y, winding_OR_y);
    }
    if ((winding_OR_y is num) &&
        (x_OR_y is num) &&
        (path_OR_x is Path2D) &&
        winding == null) {
      return _blink.BlinkCanvasRenderingContext2D.instance
          .isPointInPath_Callback_3_(this, path_OR_x, x_OR_y, winding_OR_y);
    }
    if ((winding is String) &&
        (winding_OR_y is num) &&
        (x_OR_y is num) &&
        (path_OR_x is Path2D)) {
      return _blink.BlinkCanvasRenderingContext2D.instance
          .isPointInPath_Callback_4_(
              this, path_OR_x, x_OR_y, winding_OR_y, winding);
    }
    throw new ArgumentError("Incorrect number or type of arguments");
  }

  bool isPointInStroke(path_OR_x, num x_OR_y, [num y]) {
    if ((x_OR_y is num) && (path_OR_x is num) && y == null) {
      return _blink.BlinkCanvasRenderingContext2D.instance
          .isPointInStroke_Callback_2_(this, path_OR_x, x_OR_y);
    }
    if ((y is num) && (x_OR_y is num) && (path_OR_x is Path2D)) {
      return _blink.BlinkCanvasRenderingContext2D.instance
          .isPointInStroke_Callback_3_(this, path_OR_x, x_OR_y, y);
    }
    throw new ArgumentError("Incorrect number or type of arguments");
  }

  @DomName('CanvasRenderingContext2D.measureText')
  @DocsEditable()
  TextMetrics measureText(String text) =>
      _blink.BlinkCanvasRenderingContext2D.instance
          .measureText_Callback_1_(this, text);

  void putImageData(ImageData imagedata, num dx, num dy,
      [num dirtyX, num dirtyY, num dirtyWidth, num dirtyHeight]) {
    if ((dy is num) &&
        (dx is num) &&
        (imagedata is ImageData) &&
        dirtyX == null &&
        dirtyY == null &&
        dirtyWidth == null &&
        dirtyHeight == null) {
      _blink.BlinkCanvasRenderingContext2D.instance.putImageData_Callback_3_(
          this, convertDartToNative_ImageData(imagedata), dx, dy);
      return;
    }
    if ((dirtyHeight is num) &&
        (dirtyWidth is num) &&
        (dirtyY is num) &&
        (dirtyX is num) &&
        (dy is num) &&
        (dx is num) &&
        (imagedata is ImageData)) {
      _blink.BlinkCanvasRenderingContext2D.instance.putImageData_Callback_7_(
          this,
          convertDartToNative_ImageData(imagedata),
          dx,
          dy,
          dirtyX,
          dirtyY,
          dirtyWidth,
          dirtyHeight);
      return;
    }
    throw new ArgumentError("Incorrect number or type of arguments");
  }

  @DomName('CanvasRenderingContext2D.removeHitRegion')
  @DocsEditable()
  @Experimental() // untriaged
  void removeHitRegion(String id) =>
      _blink.BlinkCanvasRenderingContext2D.instance
          .removeHitRegion_Callback_1_(this, id);

  @DomName('CanvasRenderingContext2D.resetTransform')
  @DocsEditable()
  @Experimental() // untriaged
  void resetTransform() => _blink.BlinkCanvasRenderingContext2D.instance
      .resetTransform_Callback_0_(this);

  @DomName('CanvasRenderingContext2D.restore')
  @DocsEditable()
  void restore() =>
      _blink.BlinkCanvasRenderingContext2D.instance.restore_Callback_0_(this);

  @DomName('CanvasRenderingContext2D.rotate')
  @DocsEditable()
  void rotate(num angle) => _blink.BlinkCanvasRenderingContext2D.instance
      .rotate_Callback_1_(this, angle);

  @DomName('CanvasRenderingContext2D.save')
  @DocsEditable()
  void save() =>
      _blink.BlinkCanvasRenderingContext2D.instance.save_Callback_0_(this);

  @DomName('CanvasRenderingContext2D.scale')
  @DocsEditable()
  void scale(num x, num y) => _blink.BlinkCanvasRenderingContext2D.instance
      .scale_Callback_2_(this, x, y);

  void scrollPathIntoView([Path2D path]) {
    if (path != null) {
      _blink.BlinkCanvasRenderingContext2D.instance
          .scrollPathIntoView_Callback_1_(this, path);
      return;
    }
    _blink.BlinkCanvasRenderingContext2D.instance
        .scrollPathIntoView_Callback_0_(this);
    return;
  }

  @DomName('CanvasRenderingContext2D.setLineDash')
  @DocsEditable()
  void setLineDash(List<num> dash) =>
      _blink.BlinkCanvasRenderingContext2D.instance
          .setLineDash_Callback_1_(this, dash);

  @DomName('CanvasRenderingContext2D.setTransform')
  @DocsEditable()
  void setTransform(num a, num b, num c, num d, num e, num f) =>
      _blink.BlinkCanvasRenderingContext2D.instance
          .setTransform_Callback_6_(this, a, b, c, d, e, f);

  void stroke([Path2D path]) {
    if (path == null) {
      _blink.BlinkCanvasRenderingContext2D.instance.stroke_Callback_0_(this);
      return;
    }
    if ((path is Path2D)) {
      _blink.BlinkCanvasRenderingContext2D.instance
          .stroke_Callback_1_(this, path);
      return;
    }
    throw new ArgumentError("Incorrect number or type of arguments");
  }

  @DomName('CanvasRenderingContext2D.strokeRect')
  @DocsEditable()
  void strokeRect(num x, num y, num width, num height) =>
      _blink.BlinkCanvasRenderingContext2D.instance
          .strokeRect_Callback_4_(this, x, y, width, height);

  void strokeText(String text, num x, num y, [num maxWidth]) {
    if (maxWidth != null) {
      _blink.BlinkCanvasRenderingContext2D.instance
          .strokeText_Callback_4_(this, text, x, y, maxWidth);
      return;
    }
    _blink.BlinkCanvasRenderingContext2D.instance
        .strokeText_Callback_3_(this, text, x, y);
    return;
  }

  @DomName('CanvasRenderingContext2D.transform')
  @DocsEditable()
  void transform(num a, num b, num c, num d, num e, num f) =>
      _blink.BlinkCanvasRenderingContext2D.instance
          .transform_Callback_6_(this, a, b, c, d, e, f);

  @DomName('CanvasRenderingContext2D.translate')
  @DocsEditable()
  void translate(num x, num y) => _blink.BlinkCanvasRenderingContext2D.instance
      .translate_Callback_2_(this, x, y);

  @DomName('CanvasRenderingContext2D.arc')
  @DocsEditable()
  void _arc(num x, num y, num radius, num startAngle, num endAngle,
          bool anticlockwise) =>
      _blink.BlinkCanvasRenderingContext2D.instance.arc_Callback_6_(
          this, x, y, radius, startAngle, endAngle, anticlockwise);

  @DomName('CanvasRenderingContext2D.arcTo')
  @DocsEditable()
  void arcTo(num x1, num y1, num x2, num y2, num radius) =>
      _blink.BlinkCanvasRenderingContext2D.instance
          .arcTo_Callback_5_(this, x1, y1, x2, y2, radius);

  @DomName('CanvasRenderingContext2D.bezierCurveTo')
  @DocsEditable()
  void bezierCurveTo(num cp1x, num cp1y, num cp2x, num cp2y, num x, num y) =>
      _blink.BlinkCanvasRenderingContext2D.instance
          .bezierCurveTo_Callback_6_(this, cp1x, cp1y, cp2x, cp2y, x, y);

  @DomName('CanvasRenderingContext2D.closePath')
  @DocsEditable()
  void closePath() =>
      _blink.BlinkCanvasRenderingContext2D.instance.closePath_Callback_0_(this);

  @DomName('CanvasRenderingContext2D.ellipse')
  @DocsEditable()
  @Experimental() // untriaged
  void ellipse(num x, num y, num radiusX, num radiusY, num rotation,
          num startAngle, num endAngle, bool anticlockwise) =>
      _blink.BlinkCanvasRenderingContext2D.instance.ellipse_Callback_8_(this, x,
          y, radiusX, radiusY, rotation, startAngle, endAngle, anticlockwise);

  @DomName('CanvasRenderingContext2D.lineTo')
  @DocsEditable()
  void lineTo(num x, num y) => _blink.BlinkCanvasRenderingContext2D.instance
      .lineTo_Callback_2_(this, x, y);

  @DomName('CanvasRenderingContext2D.moveTo')
  @DocsEditable()
  void moveTo(num x, num y) => _blink.BlinkCanvasRenderingContext2D.instance
      .moveTo_Callback_2_(this, x, y);

  @DomName('CanvasRenderingContext2D.quadraticCurveTo')
  @DocsEditable()
  void quadraticCurveTo(num cpx, num cpy, num x, num y) =>
      _blink.BlinkCanvasRenderingContext2D.instance
          .quadraticCurveTo_Callback_4_(this, cpx, cpy, x, y);

  @DomName('CanvasRenderingContext2D.rect')
  @DocsEditable()
  void rect(num x, num y, num width, num height) =>
      _blink.BlinkCanvasRenderingContext2D.instance
          .rect_Callback_4_(this, x, y, width, height);

  @DomName('CanvasRenderingContext2D.createImageDataFromImageData')
  @DocsEditable()
  ImageData createImageDataFromImageData(ImageData imagedata) =>
      this.createImageData(imagedata);

  /**
   * Sets the color used inside shapes.
   * [r], [g], [b] are 0-255, [a] is 0-1.
   */
  void setFillColorRgb(int r, int g, int b, [num a = 1]) {
    this.fillStyle = 'rgba($r, $g, $b, $a)';
  }

  /**
   * Sets the color used inside shapes.
   * [h] is in degrees, 0-360.
   * [s], [l] are in percent, 0-100.
   * [a] is 0-1.
   */
  void setFillColorHsl(int h, num s, num l, [num a = 1]) {
    this.fillStyle = 'hsla($h, $s%, $l%, $a)';
  }

  /**
   * Sets the color used for stroking shapes.
   * [r], [g], [b] are 0-255, [a] is 0-1.
   */
  void setStrokeColorRgb(int r, int g, int b, [num a = 1]) {
    this.strokeStyle = 'rgba($r, $g, $b, $a)';
  }

  /**
   * Sets the color used for stroking shapes.
   * [h] is in degrees, 0-360.
   * [s], [l] are in percent, 0-100.
   * [a] is 0-1.
   */
  void setStrokeColorHsl(int h, num s, num l, [num a = 1]) {
    this.strokeStyle = 'hsla($h, $s%, $l%, $a)';
  }

  @DomName('CanvasRenderingContext2D.arc')
  void arc(num x, num y, num radius, num startAngle, num endAngle,
      [bool anticlockwise = false]) {
    // TODO(terry): This should not be needed: dartbug.com/20939.
    _arc(x, y, radius, startAngle, endAngle, anticlockwise);
  }

  @DomName('CanvasRenderingContext2D.createPatternFromImage')
  CanvasPattern createPatternFromImage(
          ImageElement image, String repetitionType) =>
      createPattern(image, repetitionType);

  /**
   * Draws an image from a CanvasImageSource to an area of this canvas.
   *
   * The image will be drawn to an area of this canvas defined by
   * [destRect]. [sourceRect] defines the region of the source image that is
   * drawn.
   * If [sourceRect] is not provided, then
   * the entire rectangular image from [source] will be drawn to this context.
   *
   * If the image is larger than canvas
   * will allow, the image will be clipped to fit the available space.
   *
   *     CanvasElement canvas = new CanvasElement(width: 600, height: 600);
   *     CanvasRenderingContext2D ctx = canvas.context2D;
   *     ImageElement img = document.query('img');
   *     img.width = 100;
   *     img.height = 100;
   *
   *     // Scale the image to 20x20.
   *     ctx.drawImageToRect(img, new Rectangle(50, 50, 20, 20));
   *
   *     VideoElement video = document.query('video');
   *     video.width = 100;
   *     video.height = 100;
   *     // Take the middle 20x20 pixels from the video and stretch them.
   *     ctx.drawImageToRect(video, new Rectangle(50, 50, 100, 100),
   *         sourceRect: new Rectangle(40, 40, 20, 20));
   *
   *     // Draw the top 100x20 pixels from the otherCanvas.
   *     CanvasElement otherCanvas = document.query('canvas');
   *     ctx.drawImageToRect(otherCanvas, new Rectangle(0, 0, 100, 20),
   *         sourceRect: new Rectangle(0, 0, 100, 20));
   *
   * See also:
   *
   *   * [CanvasImageSource] for more information on what data is retrieved
   * from [source].
   *   * [drawImage](http://www.whatwg.org/specs/web-apps/current-work/multipage/the-canvas-element.html#dom-context-2d-drawimage)
   * from the WHATWG.
   */
  @DomName('CanvasRenderingContext2D.drawImage')
  void drawImageToRect(CanvasImageSource source, Rectangle destRect,
      {Rectangle sourceRect}) {
    if (sourceRect == null) {
      _drawImage(
          source, destRect.left, destRect.top, destRect.width, destRect.height);
    } else {
      _drawImage(
          source,
          sourceRect.left,
          sourceRect.top,
          sourceRect.width,
          sourceRect.height,
          destRect.left,
          destRect.top,
          destRect.width,
          destRect.height);
    }
  }

  /**
   * Draws an image from a CanvasImageSource to this canvas.
   *
   * The entire image from [source] will be drawn to this context with its top
   * left corner at the point ([destX], [destY]). If the image is
   * larger than canvas will allow, the image will be clipped to fit the
   * available space.
   *
   *     CanvasElement canvas = new CanvasElement(width: 600, height: 600);
   *     CanvasRenderingContext2D ctx = canvas.context2D;
   *     ImageElement img = document.query('img');
   *
   *     ctx.drawImage(img, 100, 100);
   *
   *     VideoElement video = document.query('video');
   *     ctx.drawImage(video, 0, 0);
   *
   *     CanvasElement otherCanvas = document.query('canvas');
   *     otherCanvas.width = 100;
   *     otherCanvas.height = 100;
   *     ctx.drawImage(otherCanvas, 590, 590); // will get clipped
   *
   * See also:
   *
   *   * [CanvasImageSource] for more information on what data is retrieved
   * from [source].
   *   * [drawImage](http://www.whatwg.org/specs/web-apps/current-work/multipage/the-canvas-element.html#dom-context-2d-drawimage)
   * from the WHATWG.
   */
  @DomName('CanvasRenderingContext2D.drawImage')
  void drawImage(CanvasImageSource source, num destX, num destY) {
    _drawImage(source, destX, destY);
  }

  /**
   * Draws an image from a CanvasImageSource to an area of this canvas.
   *
   * The image will be drawn to this context with its top left corner at the
   * point ([destX], [destY]) and will be scaled to be [destWidth] wide and
   * [destHeight] tall.
   *
   * If the image is larger than canvas
   * will allow, the image will be clipped to fit the available space.
   *
   *     CanvasElement canvas = new CanvasElement(width: 600, height: 600);
   *     CanvasRenderingContext2D ctx = canvas.context2D;
   *     ImageElement img = document.query('img');
   *     img.width = 100;
   *     img.height = 100;
   *
   *     // Scale the image to 300x50 at the point (20, 20)
   *     ctx.drawImageScaled(img, 20, 20, 300, 50);
   *
   * See also:
   *
   *   * [CanvasImageSource] for more information on what data is retrieved
   * from [source].
   *   * [drawImage](http://www.whatwg.org/specs/web-apps/current-work/multipage/the-canvas-element.html#dom-context-2d-drawimage)
   * from the WHATWG.
   */
  @DomName('CanvasRenderingContext2D.drawImage')
  void drawImageScaled(CanvasImageSource source, num destX, num destY,
      num destWidth, num destHeight) {
    _drawImage(source, destX, destY, destWidth, destHeight);
  }

  /**
   * Draws an image from a CanvasImageSource to an area of this canvas.
   *
   * The image is a region of [source] that is [sourceWidth] wide and
   * [destHeight] tall with top left corner at ([sourceX], [sourceY]).
   * The image will be drawn to this context with its top left corner at the
   * point ([destX], [destY]) and will be scaled to be [destWidth] wide and
   * [destHeight] tall.
   *
   * If the image is larger than canvas
   * will allow, the image will be clipped to fit the available space.
   *
   *     VideoElement video = document.query('video');
   *     video.width = 100;
   *     video.height = 100;
   *     // Take the middle 20x20 pixels from the video and stretch them.
   *     ctx.drawImageScaledFromSource(video, 40, 40, 20, 20, 50, 50, 100, 100);
   *
   *     // Draw the top 100x20 pixels from the otherCanvas to this one.
   *     CanvasElement otherCanvas = document.query('canvas');
   *     ctx.drawImageScaledFromSource(otherCanvas, 0, 0, 100, 20, 0, 0, 100, 20);
   *
   * See also:
   *
   *   * [CanvasImageSource] for more information on what data is retrieved
   * from [source].
   *   * [drawImage](http://www.whatwg.org/specs/web-apps/current-work/multipage/the-canvas-element.html#dom-context-2d-drawimage)
   * from the WHATWG.
   */
  @DomName('CanvasRenderingContext2D.drawImage')
  void drawImageScaledFromSource(
      CanvasImageSource source,
      num sourceX,
      num sourceY,
      num sourceWidth,
      num sourceHeight,
      num destX,
      num destY,
      num destWidth,
      num destHeight) {
    _drawImage(source, sourceX, sourceY, sourceWidth, sourceHeight, destX,
        destY, destWidth, destHeight);
  }

  // TODO(amouravski): Add Dartium native methods for drawImage once we figure
  // out how to not break native bindings.

  @SupportedBrowser(SupportedBrowser.CHROME)
  @SupportedBrowser(SupportedBrowser.SAFARI)
  @SupportedBrowser(SupportedBrowser.IE, '11')
  @Unstable()
  @DomName('CanvasRenderingContext2D.getLineDash')
  List<num> getLineDash() {
    // TODO(14316): Firefox has this functionality with mozDash, but it's a bit
    // different.
    var result = _getLineDash();
    if (result == null) {
      result = [];
    }
    return result;
  }

  /** Deprecated always returns 1.0 */
  @DomName('CanvasRenderingContext2D.webkitBackingStorePixelRation')
  @Experimental()
  @deprecated
  double get backingStorePixelRatio => 1.0;
}
// Copyright (c) 2012, the Dart project authors.  Please see the AUTHORS file
// for details. All rights reserved. Use of this source code is governed by a
// BSD-style license that can be found in the LICENSE file.

// WARNING: Do not edit - generated code.

@DocsEditable()
@DomName('CharacterData')
class CharacterData extends Node
    implements NonDocumentTypeChildNode, ChildNode {
  // To suppress missing implicit constructor warnings.
  factory CharacterData._() {
    throw new UnsupportedError("Not supported");
  }

  @Deprecated("Internal Use Only")
  external static Type get instanceRuntimeType;

  @Deprecated("Internal Use Only")
  CharacterData.internal_() : super.internal_();

  @DomName('CharacterData.data')
  @DocsEditable()
  String get data => _blink.BlinkCharacterData.instance.data_Getter_(this);

  @DomName('CharacterData.data')
  @DocsEditable()
  set data(String value) =>
      _blink.BlinkCharacterData.instance.data_Setter_(this, value);

  @DomName('CharacterData.length')
  @DocsEditable()
  int get length => _blink.BlinkCharacterData.instance.length_Getter_(this);

  @DomName('CharacterData.appendData')
  @DocsEditable()
  void appendData(String data) =>
      _blink.BlinkCharacterData.instance.appendData_Callback_1_(this, data);

  @DomName('CharacterData.deleteData')
  @DocsEditable()
  void deleteData(int offset, int count) => _blink.BlinkCharacterData.instance
      .deleteData_Callback_2_(this, offset, count);

  @DomName('CharacterData.insertData')
  @DocsEditable()
  void insertData(int offset, String data) => _blink.BlinkCharacterData.instance
      .insertData_Callback_2_(this, offset, data);

  @DomName('CharacterData.replaceData')
  @DocsEditable()
  void replaceData(int offset, int count, String data) =>
      _blink.BlinkCharacterData.instance
          .replaceData_Callback_3_(this, offset, count, data);

  @DomName('CharacterData.substringData')
  @DocsEditable()
  String substringData(int offset, int count) =>
      _blink.BlinkCharacterData.instance
          .substringData_Callback_2_(this, offset, count);

  @DomName('CharacterData.after')
  @DocsEditable()
  @Experimental() // untriaged
  void after(Object nodes) =>
      _blink.BlinkCharacterData.instance.after_Callback_1_(this, nodes);

  @DomName('CharacterData.before')
  @DocsEditable()
  @Experimental() // untriaged
  void before(Object nodes) =>
      _blink.BlinkCharacterData.instance.before_Callback_1_(this, nodes);

  @DomName('CharacterData.nextElementSibling')
  @DocsEditable()
  Element get nextElementSibling =>
      _blink.BlinkCharacterData.instance.nextElementSibling_Getter_(this);

  @DomName('CharacterData.previousElementSibling')
  @DocsEditable()
  Element get previousElementSibling =>
      _blink.BlinkCharacterData.instance.previousElementSibling_Getter_(this);
}
// Copyright (c) 2012, the Dart project authors.  Please see the AUTHORS file
// for details. All rights reserved. Use of this source code is governed by a
// BSD-style license that can be found in the LICENSE file.

// WARNING: Do not edit - generated code.

@DocsEditable()
@DomName('ChildNode')
@Experimental() // untriaged
abstract class ChildNode extends DartHtmlDomObject {
  // To suppress missing implicit constructor warnings.
  factory ChildNode._() {
    throw new UnsupportedError("Not supported");
  }

  @DomName('ChildNode.after')
  @DocsEditable()
  @Experimental() // untriaged
  void after(Object nodes);

  @DomName('ChildNode.before')
  @DocsEditable()
  @Experimental() // untriaged
  void before(Object nodes);

  @DomName('ChildNode.remove')
  @DocsEditable()
  @Experimental() // untriaged
  void remove();
}
// Copyright (c) 2012, the Dart project authors.  Please see the AUTHORS file
// for details. All rights reserved. Use of this source code is governed by a
// BSD-style license that can be found in the LICENSE file.

// WARNING: Do not edit - generated code.

@DocsEditable()
@DomName('CHROMIUMValuebuffer')
@Experimental() // untriaged
class ChromiumValuebuffer extends DartHtmlDomObject {
  // To suppress missing implicit constructor warnings.
  factory ChromiumValuebuffer._() {
    throw new UnsupportedError("Not supported");
  }

  @Deprecated("Internal Use Only")
  external static Type get instanceRuntimeType;

  @Deprecated("Internal Use Only")
  ChromiumValuebuffer.internal_() {}
}
// Copyright (c) 2012, the Dart project authors.  Please see the AUTHORS file
// for details. All rights reserved. Use of this source code is governed by a
// BSD-style license that can be found in the LICENSE file.

// WARNING: Do not edit - generated code.

@DocsEditable()
@DomName('CircularGeofencingRegion')
@Experimental() // untriaged
class CircularGeofencingRegion extends GeofencingRegion {
  // To suppress missing implicit constructor warnings.
  factory CircularGeofencingRegion._() {
    throw new UnsupportedError("Not supported");
  }

  @DomName('CircularGeofencingRegion.CircularGeofencingRegion')
  @DocsEditable()
  factory CircularGeofencingRegion(Map init) {
    var init_1 = convertDartToNative_Dictionary(init);
    return _blink.BlinkCircularGeofencingRegion.instance
        .constructorCallback_1_(init_1);
  }

  @Deprecated("Internal Use Only")
  external static Type get instanceRuntimeType;

  @Deprecated("Internal Use Only")
  CircularGeofencingRegion.internal_() : super.internal_();

  @DomName('CircularGeofencingRegion.MAX_RADIUS')
  @DocsEditable()
  @Experimental() // untriaged
  static const num MAX_RADIUS = 100.0;

  @DomName('CircularGeofencingRegion.MIN_RADIUS')
  @DocsEditable()
  @Experimental() // untriaged
  static const num MIN_RADIUS = 1.0;

  @DomName('CircularGeofencingRegion.latitude')
  @DocsEditable()
  @Experimental() // untriaged
  num get latitude =>
      _blink.BlinkCircularGeofencingRegion.instance.latitude_Getter_(this);

  @DomName('CircularGeofencingRegion.longitude')
  @DocsEditable()
  @Experimental() // untriaged
  num get longitude =>
      _blink.BlinkCircularGeofencingRegion.instance.longitude_Getter_(this);

  @DomName('CircularGeofencingRegion.radius')
  @DocsEditable()
  @Experimental() // untriaged
  num get radius =>
      _blink.BlinkCircularGeofencingRegion.instance.radius_Getter_(this);
}
// Copyright (c) 2012, the Dart project authors.  Please see the AUTHORS file
// for details. All rights reserved. Use of this source code is governed by a
// BSD-style license that can be found in the LICENSE file.

// WARNING: Do not edit - generated code.

@DocsEditable()
@DomName('Client')
@Experimental() // untriaged
class Client extends DartHtmlDomObject {
  // To suppress missing implicit constructor warnings.
  factory Client._() {
    throw new UnsupportedError("Not supported");
  }

  @Deprecated("Internal Use Only")
  external static Type get instanceRuntimeType;

  @Deprecated("Internal Use Only")
  Client.internal_() {}

  @DomName('Client.frameType')
  @DocsEditable()
  @Experimental() // untriaged
  String get frameType => _blink.BlinkClient.instance.frameType_Getter_(this);

  @DomName('Client.id')
  @DocsEditable()
  @Experimental() // untriaged
  String get id => _blink.BlinkClient.instance.id_Getter_(this);

  @DomName('Client.url')
  @DocsEditable()
  @Experimental() // untriaged
  String get url => _blink.BlinkClient.instance.url_Getter_(this);

  void postMessage(/*SerializedScriptValue*/ message,
      [List<MessagePort> transfer]) {
    if (transfer != null) {
      _blink.BlinkClient.instance.postMessage_Callback_2_(
          this, convertDartToNative_SerializedScriptValue(message), transfer);
      return;
    }
    _blink.BlinkClient.instance.postMessage_Callback_1_(
        this, convertDartToNative_SerializedScriptValue(message));
    return;
  }
}
// Copyright (c) 2012, the Dart project authors.  Please see the AUTHORS file
// for details. All rights reserved. Use of this source code is governed by a
// BSD-style license that can be found in the LICENSE file.

// WARNING: Do not edit - generated code.

@DocsEditable()
@DomName('Clients')
@Experimental() // untriaged
class Clients extends DartHtmlDomObject {
  // To suppress missing implicit constructor warnings.
  factory Clients._() {
    throw new UnsupportedError("Not supported");
  }

  @Deprecated("Internal Use Only")
  external static Type get instanceRuntimeType;

  @Deprecated("Internal Use Only")
  Clients.internal_() {}

  @DomName('Clients.claim')
  @DocsEditable()
  @Experimental() // untriaged
  Future claim() => convertNativePromiseToDartFuture(
      _blink.BlinkClients.instance.claim_Callback_0_(this));

  Future matchAll([Map options]) {
    if (options != null) {
      return _blink.BlinkClients.instance
          .matchAll_Callback_1_(this, convertDartToNative_Dictionary(options));
    }
    return _blink.BlinkClients.instance.matchAll_Callback_0_(this);
  }

  @DomName('Clients.openWindow')
  @DocsEditable()
  @Experimental() // untriaged
  Future openWindow(String url) => convertNativePromiseToDartFuture(
      _blink.BlinkClients.instance.openWindow_Callback_1_(this, url));
}
// Copyright (c) 2012, the Dart project authors.  Please see the AUTHORS file
// for details. All rights reserved. Use of this source code is governed by a
// BSD-style license that can be found in the LICENSE file.

// WARNING: Do not edit - generated code.

@DocsEditable()
@DomName('ClipboardEvent')
@Experimental() // untriaged
class ClipboardEvent extends Event {
  // To suppress missing implicit constructor warnings.
  factory ClipboardEvent._() {
    throw new UnsupportedError("Not supported");
  }

  @Deprecated("Internal Use Only")
  external static Type get instanceRuntimeType;

  @Deprecated("Internal Use Only")
  ClipboardEvent.internal_() : super.internal_();

  @DomName('ClipboardEvent.clipboardData')
  @DocsEditable()
  @Experimental() // untriaged
  DataTransfer get clipboardData =>
      _blink.BlinkClipboardEvent.instance.clipboardData_Getter_(this);
}
// Copyright (c) 2012, the Dart project authors.  Please see the AUTHORS file
// for details. All rights reserved. Use of this source code is governed by a
// BSD-style license that can be found in the LICENSE file.

// WARNING: Do not edit - generated code.

@DocsEditable()
@DomName('CloseEvent')
class CloseEvent extends Event {
  // To suppress missing implicit constructor warnings.
  factory CloseEvent._() {
    throw new UnsupportedError("Not supported");
  }

  @DomName('CloseEvent.CloseEvent')
  @DocsEditable()
  factory CloseEvent(String type, [Map eventInitDict]) {
    if (eventInitDict != null) {
      var eventInitDict_1 = convertDartToNative_Dictionary(eventInitDict);
      return _blink.BlinkCloseEvent.instance
          .constructorCallback_2_(type, eventInitDict_1);
    }
    return _blink.BlinkCloseEvent.instance.constructorCallback_1_(type);
  }

  @Deprecated("Internal Use Only")
  external static Type get instanceRuntimeType;

  @Deprecated("Internal Use Only")
  CloseEvent.internal_() : super.internal_();

  @DomName('CloseEvent.code')
  @DocsEditable()
  int get code => _blink.BlinkCloseEvent.instance.code_Getter_(this);

  @DomName('CloseEvent.reason')
  @DocsEditable()
  String get reason => _blink.BlinkCloseEvent.instance.reason_Getter_(this);

  @DomName('CloseEvent.wasClean')
  @DocsEditable()
  bool get wasClean => _blink.BlinkCloseEvent.instance.wasClean_Getter_(this);
}
// Copyright (c) 2012, the Dart project authors.  Please see the AUTHORS file
// for details. All rights reserved. Use of this source code is governed by a
// BSD-style license that can be found in the LICENSE file.

@DocsEditable()
@DomName('Comment')
class Comment extends CharacterData {
  // To suppress missing implicit constructor warnings.
  factory Comment._() {
    throw new UnsupportedError("Not supported");
  }

  @DomName('Comment.Comment')
  @DocsEditable()
  factory Comment([String data]) {
    if (data != null) {
      return _blink.BlinkComment.instance.constructorCallback_1_(data);
    }
    return _blink.BlinkComment.instance.constructorCallback_0_();
  }

  @Deprecated("Internal Use Only")
  external static Type get instanceRuntimeType;

  @Deprecated("Internal Use Only")
  Comment.internal_() : super.internal_();
}
// Copyright (c) 2013, the Dart project authors.  Please see the AUTHORS file
// for details. All rights reserved. Use of this source code is governed by a
// BSD-style license that can be found in the LICENSE file.

// WARNING: Do not edit - generated code.

@DomName('CompositionEvent')
class CompositionEvent extends UIEvent {
  factory CompositionEvent(String type,
      {bool canBubble: false,
      bool cancelable: false,
      Window view,
      String data,
      String locale}) {
    if (view == null) {
      view = window;
    }
    CompositionEvent e = document._createEvent("CompositionEvent");

    e._initCompositionEvent(type, canBubble, cancelable, view, data);

    return e;
  }

  @DomName('CompositionEvent.CompositionEvent')
  @DocsEditable()
  factory CompositionEvent._(String type, [Map eventInitDict]) {
    if (eventInitDict != null) {
      var eventInitDict_1 = convertDartToNative_Dictionary(eventInitDict);
      return _blink.BlinkCompositionEvent.instance
          .constructorCallback_2_(type, eventInitDict_1);
    }
    return _blink.BlinkCompositionEvent.instance.constructorCallback_1_(type);
  }

  @Deprecated("Internal Use Only")
  external static Type get instanceRuntimeType;

  @Deprecated("Internal Use Only")
  CompositionEvent.internal_() : super.internal_();

  @DomName('CompositionEvent.data')
  @DocsEditable()
  String get data => _blink.BlinkCompositionEvent.instance.data_Getter_(this);

  @DomName('CompositionEvent.initCompositionEvent')
  @DocsEditable()
  void _initCompositionEvent(String type, bool bubbles, bool cancelable,
          Window view, String data) =>
      _blink.BlinkCompositionEvent.instance.initCompositionEvent_Callback_5_(
          this, type, bubbles, cancelable, view, data);
}
// Copyright (c) 2012, the Dart project authors.  Please see the AUTHORS file
// for details. All rights reserved. Use of this source code is governed by a
// BSD-style license that can be found in the LICENSE file.

// WARNING: Do not edit - generated code.

@DocsEditable()
@DomName('CompositorProxy')
@Experimental() // untriaged
class CompositorProxy extends DartHtmlDomObject {
  // To suppress missing implicit constructor warnings.
  factory CompositorProxy._() {
    throw new UnsupportedError("Not supported");
  }

  @DomName('CompositorProxy.CompositorProxy')
  @DocsEditable()
  factory CompositorProxy(Element element, List<String> attributeArray) {
    return _blink.BlinkCompositorProxy.instance
        .constructorCallback_2_(element, attributeArray);
  }

  @Deprecated("Internal Use Only")
  external static Type get instanceRuntimeType;

  @Deprecated("Internal Use Only")
  CompositorProxy.internal_() {}

  @DomName('CompositorProxy.opacity')
  @DocsEditable()
  @Experimental() // untriaged
  num get opacity => _blink.BlinkCompositorProxy.instance.opacity_Getter_(this);

  @DomName('CompositorProxy.opacity')
  @DocsEditable()
  @Experimental() // untriaged
  set opacity(num value) =>
      _blink.BlinkCompositorProxy.instance.opacity_Setter_(this, value);

  @DomName('CompositorProxy.scrollLeft')
  @DocsEditable()
  @Experimental() // untriaged
  num get scrollLeft =>
      _blink.BlinkCompositorProxy.instance.scrollLeft_Getter_(this);

  @DomName('CompositorProxy.scrollLeft')
  @DocsEditable()
  @Experimental() // untriaged
  set scrollLeft(num value) =>
      _blink.BlinkCompositorProxy.instance.scrollLeft_Setter_(this, value);

  @DomName('CompositorProxy.scrollTop')
  @DocsEditable()
  @Experimental() // untriaged
  num get scrollTop =>
      _blink.BlinkCompositorProxy.instance.scrollTop_Getter_(this);

  @DomName('CompositorProxy.scrollTop')
  @DocsEditable()
  @Experimental() // untriaged
  set scrollTop(num value) =>
      _blink.BlinkCompositorProxy.instance.scrollTop_Setter_(this, value);

  @DomName('CompositorProxy.transform')
  @DocsEditable()
  @Experimental() // untriaged
  DomMatrix get transform =>
      _blink.BlinkCompositorProxy.instance.transform_Getter_(this);

  @DomName('CompositorProxy.transform')
  @DocsEditable()
  @Experimental() // untriaged
  set transform(DomMatrix value) =>
      _blink.BlinkCompositorProxy.instance.transform_Setter_(this, value);

  @DomName('CompositorProxy.disconnect')
  @DocsEditable()
  @Experimental() // untriaged
  void disconnect() =>
      _blink.BlinkCompositorProxy.instance.disconnect_Callback_0_(this);

  @DomName('CompositorProxy.supports')
  @DocsEditable()
  @Experimental() // untriaged
  bool supports(String attribute) => _blink.BlinkCompositorProxy.instance
      .supports_Callback_1_(this, attribute);
}
// Copyright (c) 2012, the Dart project authors.  Please see the AUTHORS file
// for details. All rights reserved. Use of this source code is governed by a
// BSD-style license that can be found in the LICENSE file.

// WARNING: Do not edit - generated code.

@DocsEditable()
@DomName('CompositorWorker')
@Experimental() // untriaged
class CompositorWorker extends EventTarget implements AbstractWorker {
  // To suppress missing implicit constructor warnings.
  factory CompositorWorker._() {
    throw new UnsupportedError("Not supported");
  }

  @DomName('CompositorWorker.errorEvent')
  @DocsEditable()
  @Experimental() // untriaged
  static const EventStreamProvider<Event> errorEvent =
      const EventStreamProvider<Event>('error');

  @DomName('CompositorWorker.messageEvent')
  @DocsEditable()
  @Experimental() // untriaged
  static const EventStreamProvider<MessageEvent> messageEvent =
      const EventStreamProvider<MessageEvent>('message');

  @DomName('CompositorWorker.CompositorWorker')
  @DocsEditable()
  factory CompositorWorker(String scriptUrl) {
    return _blink.BlinkCompositorWorker.instance
        .constructorCallback_1_(scriptUrl);
  }

  @Deprecated("Internal Use Only")
  external static Type get instanceRuntimeType;

  @Deprecated("Internal Use Only")
  CompositorWorker.internal_() : super.internal_();

  void postMessage(/*SerializedScriptValue*/ message,
      [List<MessagePort> transfer]) {
    if (transfer != null) {
      _blink.BlinkCompositorWorker.instance.postMessage_Callback_2_(
          this, convertDartToNative_SerializedScriptValue(message), transfer);
      return;
    }
    _blink.BlinkCompositorWorker.instance.postMessage_Callback_1_(
        this, convertDartToNative_SerializedScriptValue(message));
    return;
  }

  @DomName('CompositorWorker.terminate')
  @DocsEditable()
  @Experimental() // untriaged
  void terminate() =>
      _blink.BlinkCompositorWorker.instance.terminate_Callback_0_(this);

  @DomName('CompositorWorker.onerror')
  @DocsEditable()
  @Experimental() // untriaged
  Stream<Event> get onError => errorEvent.forTarget(this);

  @DomName('CompositorWorker.onmessage')
  @DocsEditable()
  @Experimental() // untriaged
  Stream<MessageEvent> get onMessage => messageEvent.forTarget(this);
}
// Copyright (c) 2012, the Dart project authors.  Please see the AUTHORS file
// for details. All rights reserved. Use of this source code is governed by a
// BSD-style license that can be found in the LICENSE file.

// WARNING: Do not edit - generated code.

@DocsEditable()
@DomName('CompositorWorkerGlobalScope')
@Experimental() // untriaged
class CompositorWorkerGlobalScope extends WorkerGlobalScope {
  // To suppress missing implicit constructor warnings.
  factory CompositorWorkerGlobalScope._() {
    throw new UnsupportedError("Not supported");
  }

  @DomName('CompositorWorkerGlobalScope.messageEvent')
  @DocsEditable()
  @Experimental() // untriaged
  static const EventStreamProvider<MessageEvent> messageEvent =
      const EventStreamProvider<MessageEvent>('message');

  @Deprecated("Internal Use Only")
  external static Type get instanceRuntimeType;

  @Deprecated("Internal Use Only")
  CompositorWorkerGlobalScope.internal_() : super.internal_();

  @DomName('CompositorWorkerGlobalScope.cancelAnimationFrame')
  @DocsEditable()
  @Experimental() // untriaged
  void cancelAnimationFrame(int handle) =>
      _blink.BlinkCompositorWorkerGlobalScope.instance
          .cancelAnimationFrame_Callback_1_(this, handle);

  void postMessage(Object message, [List<MessagePort> transfer]) {
    if (transfer != null) {
      _blink.BlinkCompositorWorkerGlobalScope.instance.postMessage_Callback_2_(
          this, convertDartToNative_SerializedScriptValue(message), transfer);
      return;
    }
    _blink.BlinkCompositorWorkerGlobalScope.instance.postMessage_Callback_1_(
        this, convertDartToNative_SerializedScriptValue(message));
    return;
  }

  @DomName('CompositorWorkerGlobalScope.requestAnimationFrame')
  @DocsEditable()
  @Experimental() // untriaged
  int requestAnimationFrame(FrameRequestCallback callback) =>
      _blink.BlinkCompositorWorkerGlobalScope.instance
          .requestAnimationFrame_Callback_1_(this, callback);

  @DomName('CompositorWorkerGlobalScope.onmessage')
  @DocsEditable()
  @Experimental() // untriaged
  Stream<MessageEvent> get onMessage => messageEvent.forTarget(this);
}
// Copyright (c) 2012, the Dart project authors.  Please see the AUTHORS file
// for details. All rights reserved. Use of this source code is governed by a
// BSD-style license that can be found in the LICENSE file.

// WARNING: Do not edit - generated code.

@DocsEditable()
@DomName('Console')
class Console extends ConsoleBase {
  // To suppress missing implicit constructor warnings.
  factory Console._() {
    throw new UnsupportedError("Not supported");
  }

  @Deprecated("Internal Use Only")
  external static Type get instanceRuntimeType;

  @Deprecated("Internal Use Only")
  Console.internal_() : super.internal_();

  @DomName('Console.memory')
  @DocsEditable()
  @Experimental()
  MemoryInfo get memory => _blink.BlinkConsole.instance.memory_Getter_(this);

  @DomName('Console.memory')
  @DocsEditable()
  @Experimental()
  set memory(MemoryInfo value) =>
      _blink.BlinkConsole.instance.memory_Setter_(this, value);
}
// Copyright (c) 2012, the Dart project authors.  Please see the AUTHORS file
// for details. All rights reserved. Use of this source code is governed by a
// BSD-style license that can be found in the LICENSE file.

// WARNING: Do not edit - generated code.

@DocsEditable()
@DomName('ConsoleBase')
@Experimental() // untriaged
class ConsoleBase extends DartHtmlDomObject {
  // To suppress missing implicit constructor warnings.
  factory ConsoleBase._() {
    throw new UnsupportedError("Not supported");
  }

  @Deprecated("Internal Use Only")
  external static Type get instanceRuntimeType;

  @Deprecated("Internal Use Only")
  ConsoleBase.internal_() {}

  @DomName('ConsoleBase.assert')
  @DocsEditable()
  @Experimental() // untriaged
  void assertCondition(bool condition, Object arg) =>
      _blink.BlinkConsoleBase.instance.assert_Callback_2_(this, condition, arg);

  @DomName('ConsoleBase.clear')
  @DocsEditable()
  @Experimental() // untriaged
  void clear(Object arg) =>
      _blink.BlinkConsoleBase.instance.clear_Callback_1_(this, arg);

  @DomName('ConsoleBase.count')
  @DocsEditable()
  @Experimental() // untriaged
  void count(Object arg) =>
      _blink.BlinkConsoleBase.instance.count_Callback_1_(this, arg);

  @DomName('ConsoleBase.debug')
  @DocsEditable()
  @Experimental() // untriaged
  void debug(Object arg) =>
      _blink.BlinkConsoleBase.instance.debug_Callback_1_(this, arg);

  @DomName('ConsoleBase.dir')
  @DocsEditable()
  @Experimental() // untriaged
  void dir(Object arg) =>
      _blink.BlinkConsoleBase.instance.dir_Callback_1_(this, arg);

  @DomName('ConsoleBase.dirxml')
  @DocsEditable()
  @Experimental() // untriaged
  void dirxml(Object arg) =>
      _blink.BlinkConsoleBase.instance.dirxml_Callback_1_(this, arg);

  @DomName('ConsoleBase.error')
  @DocsEditable()
  @Experimental() // untriaged
  void error(Object arg) =>
      _blink.BlinkConsoleBase.instance.error_Callback_1_(this, arg);

  @DomName('ConsoleBase.group')
  @DocsEditable()
  @Experimental() // untriaged
  void group(Object arg) =>
      _blink.BlinkConsoleBase.instance.group_Callback_1_(this, arg);

  @DomName('ConsoleBase.groupCollapsed')
  @DocsEditable()
  @Experimental() // untriaged
  void groupCollapsed(Object arg) =>
      _blink.BlinkConsoleBase.instance.groupCollapsed_Callback_1_(this, arg);

  @DomName('ConsoleBase.groupEnd')
  @DocsEditable()
  @Experimental() // untriaged
  void groupEnd() =>
      _blink.BlinkConsoleBase.instance.groupEnd_Callback_0_(this);

  @DomName('ConsoleBase.info')
  @DocsEditable()
  @Experimental() // untriaged
  void info(Object arg) =>
      _blink.BlinkConsoleBase.instance.info_Callback_1_(this, arg);

  @DomName('ConsoleBase.log')
  @DocsEditable()
  @Experimental() // untriaged
  void log(Object arg) =>
      _blink.BlinkConsoleBase.instance.log_Callback_1_(this, arg);

  @DomName('ConsoleBase.markTimeline')
  @DocsEditable()
  @Experimental() // untriaged
  void markTimeline(String title) =>
      _blink.BlinkConsoleBase.instance.markTimeline_Callback_1_(this, title);

  @DomName('ConsoleBase.profile')
  @DocsEditable()
  @Experimental() // untriaged
  void profile(String title) =>
      _blink.BlinkConsoleBase.instance.profile_Callback_1_(this, title);

  @DomName('ConsoleBase.profileEnd')
  @DocsEditable()
  @Experimental() // untriaged
  void profileEnd(String title) =>
      _blink.BlinkConsoleBase.instance.profileEnd_Callback_1_(this, title);

  @DomName('ConsoleBase.table')
  @DocsEditable()
  @Experimental() // untriaged
  void table(Object arg) =>
      _blink.BlinkConsoleBase.instance.table_Callback_1_(this, arg);

  @DomName('ConsoleBase.time')
  @DocsEditable()
  @Experimental() // untriaged
  void time(String title) =>
      _blink.BlinkConsoleBase.instance.time_Callback_1_(this, title);

  @DomName('ConsoleBase.timeEnd')
  @DocsEditable()
  @Experimental() // untriaged
  void timeEnd(String title) =>
      _blink.BlinkConsoleBase.instance.timeEnd_Callback_1_(this, title);

  @DomName('ConsoleBase.timeStamp')
  @DocsEditable()
  @Experimental() // untriaged
  void timeStamp(String title) =>
      _blink.BlinkConsoleBase.instance.timeStamp_Callback_1_(this, title);

  @DomName('ConsoleBase.timeline')
  @DocsEditable()
  @Experimental() // untriaged
  void timeline(String title) =>
      _blink.BlinkConsoleBase.instance.timeline_Callback_1_(this, title);

  @DomName('ConsoleBase.timelineEnd')
  @DocsEditable()
  @Experimental() // untriaged
  void timelineEnd(String title) =>
      _blink.BlinkConsoleBase.instance.timelineEnd_Callback_1_(this, title);

  @DomName('ConsoleBase.trace')
  @DocsEditable()
  @Experimental() // untriaged
  void trace(Object arg) =>
      _blink.BlinkConsoleBase.instance.trace_Callback_1_(this, arg);

  @DomName('ConsoleBase.warn')
  @DocsEditable()
  @Experimental() // untriaged
  void warn(Object arg) =>
      _blink.BlinkConsoleBase.instance.warn_Callback_1_(this, arg);
}
// Copyright (c) 2012, the Dart project authors.  Please see the AUTHORS file
// for details. All rights reserved. Use of this source code is governed by a
// BSD-style license that can be found in the LICENSE file.

// WARNING: Do not edit - generated code.

@DocsEditable()
@DomName('HTMLContentElement')
@SupportedBrowser(SupportedBrowser.CHROME, '26')
@Experimental()
// https://dvcs.w3.org/hg/webcomponents/raw-file/tip/spec/shadow/index.html#content-element
class ContentElement extends HtmlElement {
  // To suppress missing implicit constructor warnings.
  factory ContentElement._() {
    throw new UnsupportedError("Not supported");
  }

  @DomName('HTMLContentElement.HTMLContentElement')
  @DocsEditable()
  factory ContentElement() => document.createElement("content");

  @Deprecated("Internal Use Only")
  external static Type get instanceRuntimeType;

  @Deprecated("Internal Use Only")
  ContentElement.internal_() : super.internal_();

  /**
   * Constructor instantiated by the DOM when a custom element has been created.
   *
   * This can only be called by subclasses from their created constructor.
   */
  ContentElement.created() : super.created();

  /// Checks if this type is supported on the current platform.
  static bool get supported => true;

  @DomName('HTMLContentElement.select')
  @DocsEditable()
  String get select =>
      _blink.BlinkHTMLContentElement.instance.select_Getter_(this);

  @DomName('HTMLContentElement.select')
  @DocsEditable()
  set select(String value) =>
      _blink.BlinkHTMLContentElement.instance.select_Setter_(this, value);

  @DomName('HTMLContentElement.getDistributedNodes')
  @DocsEditable()
  List<Node> getDistributedNodes() => (_blink.BlinkHTMLContentElement.instance
      .getDistributedNodes_Callback_0_(this));
}
// Copyright (c) 2012, the Dart project authors.  Please see the AUTHORS file
// for details. All rights reserved. Use of this source code is governed by a
// BSD-style license that can be found in the LICENSE file.

// WARNING: Do not edit - generated code.

@DocsEditable()
@DomName('Coordinates')
class Coordinates extends DartHtmlDomObject {
  // To suppress missing implicit constructor warnings.
  factory Coordinates._() {
    throw new UnsupportedError("Not supported");
  }

  @Deprecated("Internal Use Only")
  external static Type get instanceRuntimeType;

  @Deprecated("Internal Use Only")
  Coordinates.internal_() {}

  @DomName('Coordinates.accuracy')
  @DocsEditable()
  num get accuracy => _blink.BlinkCoordinates.instance.accuracy_Getter_(this);

  @DomName('Coordinates.altitude')
  @DocsEditable()
  num get altitude => _blink.BlinkCoordinates.instance.altitude_Getter_(this);

  @DomName('Coordinates.altitudeAccuracy')
  @DocsEditable()
  num get altitudeAccuracy =>
      _blink.BlinkCoordinates.instance.altitudeAccuracy_Getter_(this);

  @DomName('Coordinates.heading')
  @DocsEditable()
  num get heading => _blink.BlinkCoordinates.instance.heading_Getter_(this);

  @DomName('Coordinates.latitude')
  @DocsEditable()
  num get latitude => _blink.BlinkCoordinates.instance.latitude_Getter_(this);

  @DomName('Coordinates.longitude')
  @DocsEditable()
  num get longitude => _blink.BlinkCoordinates.instance.longitude_Getter_(this);

  @DomName('Coordinates.speed')
  @DocsEditable()
  num get speed => _blink.BlinkCoordinates.instance.speed_Getter_(this);
}
// Copyright (c) 2012, the Dart project authors.  Please see the AUTHORS file
// for details. All rights reserved. Use of this source code is governed by a
// BSD-style license that can be found in the LICENSE file.

// WARNING: Do not edit - generated code.

@DocsEditable()
@DomName('Credential')
@Experimental() // untriaged
class Credential extends DartHtmlDomObject {
  // To suppress missing implicit constructor warnings.
  factory Credential._() {
    throw new UnsupportedError("Not supported");
  }

  @Deprecated("Internal Use Only")
  external static Type get instanceRuntimeType;

  @Deprecated("Internal Use Only")
  Credential.internal_() {}

  @DomName('Credential.iconURL')
  @DocsEditable()
  @Experimental() // untriaged
  String get iconUrl => _blink.BlinkCredential.instance.iconURL_Getter_(this);

  @DomName('Credential.id')
  @DocsEditable()
  @Experimental() // untriaged
  String get id => _blink.BlinkCredential.instance.id_Getter_(this);

  @DomName('Credential.name')
  @DocsEditable()
  @Experimental() // untriaged
  String get name => _blink.BlinkCredential.instance.name_Getter_(this);

  @DomName('Credential.type')
  @DocsEditable()
  @Experimental() // untriaged
  String get type => _blink.BlinkCredential.instance.type_Getter_(this);
}
// Copyright (c) 2012, the Dart project authors.  Please see the AUTHORS file
// for details. All rights reserved. Use of this source code is governed by a
// BSD-style license that can be found in the LICENSE file.

// WARNING: Do not edit - generated code.

@DocsEditable()
@DomName('CredentialsContainer')
@Experimental() // untriaged
class CredentialsContainer extends DartHtmlDomObject {
  // To suppress missing implicit constructor warnings.
  factory CredentialsContainer._() {
    throw new UnsupportedError("Not supported");
  }

  @Deprecated("Internal Use Only")
  external static Type get instanceRuntimeType;

  @Deprecated("Internal Use Only")
  CredentialsContainer.internal_() {}

  @DomName('CredentialsContainer.notifySignedIn')
  @DocsEditable()
  @Experimental() // untriaged
  Future notifySignedIn(Credential credential) =>
      convertNativePromiseToDartFuture(_blink.BlinkCredentialsContainer.instance
          .notifySignedIn_Callback_1_(this, credential));

  Future request([Map options]) {
    if (options != null) {
      return _blink.BlinkCredentialsContainer.instance
          .request_Callback_1_(this, convertDartToNative_Dictionary(options));
    }
    return _blink.BlinkCredentialsContainer.instance.request_Callback_0_(this);
  }

  @DomName('CredentialsContainer.requireUserMediation')
  @DocsEditable()
  @Experimental() // untriaged
  Future requireUserMediation() =>
      convertNativePromiseToDartFuture(_blink.BlinkCredentialsContainer.instance
          .requireUserMediation_Callback_0_(this));
}
// Copyright (c) 2012, the Dart project authors.  Please see the AUTHORS file
// for details. All rights reserved. Use of this source code is governed by a
// BSD-style license that can be found in the LICENSE file.

// WARNING: Do not edit - generated code.

@DocsEditable()
@DomName('CrossOriginConnectEvent')
@Experimental() // untriaged
class CrossOriginConnectEvent extends Event {
  // To suppress missing implicit constructor warnings.
  factory CrossOriginConnectEvent._() {
    throw new UnsupportedError("Not supported");
  }

  @Deprecated("Internal Use Only")
  external static Type get instanceRuntimeType;

  @Deprecated("Internal Use Only")
  CrossOriginConnectEvent.internal_() : super.internal_();

  @DomName('CrossOriginConnectEvent.client')
  @DocsEditable()
  @Experimental() // untriaged
  CrossOriginServiceWorkerClient get client =>
      _blink.BlinkCrossOriginConnectEvent.instance.client_Getter_(this);

  @DomName('CrossOriginConnectEvent.acceptConnection')
  @DocsEditable()
  @Experimental() // untriaged
  void acceptConnection(Future shouldAccept) =>
      _blink.BlinkCrossOriginConnectEvent.instance
          .acceptConnection_Callback_1_(this, shouldAccept);
}
// Copyright (c) 2012, the Dart project authors.  Please see the AUTHORS file
// for details. All rights reserved. Use of this source code is governed by a
// BSD-style license that can be found in the LICENSE file.

// WARNING: Do not edit - generated code.

@DocsEditable()
@DomName('CrossOriginServiceWorkerClient')
@Experimental() // untriaged
class CrossOriginServiceWorkerClient extends EventTarget {
  // To suppress missing implicit constructor warnings.
  factory CrossOriginServiceWorkerClient._() {
    throw new UnsupportedError("Not supported");
  }

  @Deprecated("Internal Use Only")
  external static Type get instanceRuntimeType;

  @Deprecated("Internal Use Only")
  CrossOriginServiceWorkerClient.internal_() : super.internal_();

  @DomName('CrossOriginServiceWorkerClient.origin')
  @DocsEditable()
  @Experimental() // untriaged
  String get origin =>
      _blink.BlinkCrossOriginServiceWorkerClient.instance.origin_Getter_(this);

  @DomName('CrossOriginServiceWorkerClient.targetUrl')
  @DocsEditable()
  @Experimental() // untriaged
  String get targetUrl => _blink.BlinkCrossOriginServiceWorkerClient.instance
      .targetUrl_Getter_(this);

  void postMessage(/*SerializedScriptValue*/ message,
      [List<MessagePort> transfer]) {
    if (transfer != null) {
      _blink.BlinkCrossOriginServiceWorkerClient.instance
          .postMessage_Callback_2_(this,
              convertDartToNative_SerializedScriptValue(message), transfer);
      return;
    }
    _blink.BlinkCrossOriginServiceWorkerClient.instance.postMessage_Callback_1_(
        this, convertDartToNative_SerializedScriptValue(message));
    return;
  }
}
// Copyright (c) 2015, the Dart project authors.  Please see the AUTHORS file
// for details. All rights reserved. Use of this source code is governed by a
// BSD-style license that can be found in the LICENSE file.

@DomName('Crypto')
@SupportedBrowser(SupportedBrowser.CHROME)
@SupportedBrowser(SupportedBrowser.SAFARI)
@Experimental()
// http://www.w3.org/TR/WebCryptoAPI/
class Crypto extends DartHtmlDomObject {
  TypedData getRandomValues(TypedData array) {
    var random = _getRandomValues(array);
    // The semantics of the operation are that it modifies the argument, but we
    // have no way of making a Dart typed data created initially in Dart reference
    // externalized storage. So we copy the values back from the returned copy.
    // TODO(alanknight): Make this less ridiculously slow.
    for (var i = 0; i < random.length; i++) {
      array[i] = random[i];
    }
    return array;
  }

  // To suppress missing implicit constructor warnings.
  factory Crypto._() {
    throw new UnsupportedError("Not supported");
  }

  @Deprecated("Internal Use Only")
  external static Type get instanceRuntimeType;

  @Deprecated("Internal Use Only")
  Crypto.internal_() {}

  /// Checks if this type is supported on the current platform.
  static bool get supported => true;

  @DomName('Crypto.subtle')
  @DocsEditable()
  @Experimental() // untriaged
  _SubtleCrypto get subtle => _blink.BlinkCrypto.instance.subtle_Getter_(this);

  @DomName('Crypto.getRandomValues')
  @DocsEditable()
  TypedData _getRandomValues(TypedData array) =>
      _blink.BlinkCrypto.instance.getRandomValues_Callback_1_(this, array);
}
// Copyright (c) 2012, the Dart project authors.  Please see the AUTHORS file
// for details. All rights reserved. Use of this source code is governed by a
// BSD-style license that can be found in the LICENSE file.

// WARNING: Do not edit - generated code.

@DocsEditable()
@DomName('CryptoKey')
@Experimental() // untriaged
class CryptoKey extends DartHtmlDomObject {
  // To suppress missing implicit constructor warnings.
  factory CryptoKey._() {
    throw new UnsupportedError("Not supported");
  }

  @Deprecated("Internal Use Only")
  external static Type get instanceRuntimeType;

  @Deprecated("Internal Use Only")
  CryptoKey.internal_() {}

  @DomName('CryptoKey.algorithm')
  @DocsEditable()
  @Experimental() // untriaged
  Object get algorithm =>
      (_blink.BlinkCryptoKey.instance.algorithm_Getter_(this));

  @DomName('CryptoKey.extractable')
  @DocsEditable()
  @Experimental() // untriaged
  bool get extractable =>
      _blink.BlinkCryptoKey.instance.extractable_Getter_(this);

  @DomName('CryptoKey.type')
  @DocsEditable()
  @Experimental() // untriaged
  String get type => _blink.BlinkCryptoKey.instance.type_Getter_(this);

  @DomName('CryptoKey.usages')
  @DocsEditable()
  @Experimental() // untriaged
  List<String> get usages =>
      _blink.BlinkCryptoKey.instance.usages_Getter_(this);
}
// Copyright (c) 2012, the Dart project authors.  Please see the AUTHORS file
// for details. All rights reserved. Use of this source code is governed by a
// BSD-style license that can be found in the LICENSE file.

// WARNING: Do not edit - generated code.

@DocsEditable()
@DomName('CSS')
// http://www.w3.org/TR/css3-conditional/#the-css-interface
@Experimental() // None
class Css extends DartHtmlDomObject {
  // To suppress missing implicit constructor warnings.
  factory Css._() {
    throw new UnsupportedError("Not supported");
  }

  @Deprecated("Internal Use Only")
  external static Type get instanceRuntimeType;

  @Deprecated("Internal Use Only")
  Css.internal_() {}

  @DomName('CSS.supports')
  @DocsEditable()
  static bool supports(String property, String value) =>
      _blink.BlinkCSS.instance.supports_Callback_2_(property, value);

  @DomName('CSS.supportsCondition')
  @DocsEditable()
  static bool supportsCondition(String conditionText) =>
      _blink.BlinkCSS.instance.supports_Callback_1_(conditionText);
}
// Copyright (c) 2012, the Dart project authors.  Please see the AUTHORS file
// for details. All rights reserved. Use of this source code is governed by a
// BSD-style license that can be found in the LICENSE file.

// WARNING: Do not edit - generated code.

@DocsEditable()
@DomName('CSSCharsetRule')
// http://dev.w3.org/csswg/cssom/#the-csscharsetrule-interface
@Experimental()
class CssCharsetRule extends CssRule {
  // To suppress missing implicit constructor warnings.
  factory CssCharsetRule._() {
    throw new UnsupportedError("Not supported");
  }

  @Deprecated("Internal Use Only")
  external static Type get instanceRuntimeType;

  @Deprecated("Internal Use Only")
  CssCharsetRule.internal_() : super.internal_();

  @DomName('CSSCharsetRule.encoding')
  @DocsEditable()
  String get encoding =>
      _blink.BlinkCSSCharsetRule.instance.encoding_Getter_(this);

  @DomName('CSSCharsetRule.encoding')
  @DocsEditable()
  set encoding(String value) =>
      _blink.BlinkCSSCharsetRule.instance.encoding_Setter_(this, value);
}
// Copyright (c) 2012, the Dart project authors.  Please see the AUTHORS file
// for details. All rights reserved. Use of this source code is governed by a
// BSD-style license that can be found in the LICENSE file.

// WARNING: Do not edit - generated code.

@DocsEditable()
@DomName('CSSFontFaceRule')
class CssFontFaceRule extends CssRule {
  // To suppress missing implicit constructor warnings.
  factory CssFontFaceRule._() {
    throw new UnsupportedError("Not supported");
  }

  @Deprecated("Internal Use Only")
  external static Type get instanceRuntimeType;

  @Deprecated("Internal Use Only")
  CssFontFaceRule.internal_() : super.internal_();

  @DomName('CSSFontFaceRule.style')
  @DocsEditable()
  CssStyleDeclaration get style =>
      _blink.BlinkCSSFontFaceRule.instance.style_Getter_(this);
}
// Copyright (c) 2012, the Dart project authors.  Please see the AUTHORS file
// for details. All rights reserved. Use of this source code is governed by a
// BSD-style license that can be found in the LICENSE file.

// WARNING: Do not edit - generated code.

@DocsEditable()
@DomName('CSSGroupingRule')
@Experimental() // untriaged
class CssGroupingRule extends CssRule {
  // To suppress missing implicit constructor warnings.
  factory CssGroupingRule._() {
    throw new UnsupportedError("Not supported");
  }

  @Deprecated("Internal Use Only")
  external static Type get instanceRuntimeType;

  @Deprecated("Internal Use Only")
  CssGroupingRule.internal_() : super.internal_();

  @DomName('CSSGroupingRule.cssRules')
  @DocsEditable()
  @Experimental() // untriaged
  List<CssRule> get cssRules =>
      _blink.BlinkCSSGroupingRule.instance.cssRules_Getter_(this);

  @DomName('CSSGroupingRule.deleteRule')
  @DocsEditable()
  @Experimental() // untriaged
  void deleteRule(int index) =>
      _blink.BlinkCSSGroupingRule.instance.deleteRule_Callback_1_(this, index);

  @DomName('CSSGroupingRule.insertRule')
  @DocsEditable()
  @Experimental() // untriaged
  int insertRule(String rule, int index) => _blink.BlinkCSSGroupingRule.instance
      .insertRule_Callback_2_(this, rule, index);
}
// Copyright (c) 2012, the Dart project authors.  Please see the AUTHORS file
// for details. All rights reserved. Use of this source code is governed by a
// BSD-style license that can be found in the LICENSE file.

// WARNING: Do not edit - generated code.

@DocsEditable()
@DomName('CSSImportRule')
class CssImportRule extends CssRule {
  // To suppress missing implicit constructor warnings.
  factory CssImportRule._() {
    throw new UnsupportedError("Not supported");
  }

  @Deprecated("Internal Use Only")
  external static Type get instanceRuntimeType;

  @Deprecated("Internal Use Only")
  CssImportRule.internal_() : super.internal_();

  @DomName('CSSImportRule.href')
  @DocsEditable()
  String get href => _blink.BlinkCSSImportRule.instance.href_Getter_(this);

  @DomName('CSSImportRule.media')
  @DocsEditable()
  MediaList get media => _blink.BlinkCSSImportRule.instance.media_Getter_(this);

  @DomName('CSSImportRule.styleSheet')
  @DocsEditable()
  CssStyleSheet get styleSheet =>
      _blink.BlinkCSSImportRule.instance.styleSheet_Getter_(this);
}
// Copyright (c) 2012, the Dart project authors.  Please see the AUTHORS file
// for details. All rights reserved. Use of this source code is governed by a
// BSD-style license that can be found in the LICENSE file.

// WARNING: Do not edit - generated code.

@DocsEditable()
@DomName('CSSKeyframeRule')
@Experimental() // untriaged
class CssKeyframeRule extends CssRule {
  // To suppress missing implicit constructor warnings.
  factory CssKeyframeRule._() {
    throw new UnsupportedError("Not supported");
  }

  @Deprecated("Internal Use Only")
  external static Type get instanceRuntimeType;

  @Deprecated("Internal Use Only")
  CssKeyframeRule.internal_() : super.internal_();

  @DomName('CSSKeyframeRule.keyText')
  @DocsEditable()
  @Experimental() // untriaged
  String get keyText =>
      _blink.BlinkCSSKeyframeRule.instance.keyText_Getter_(this);

  @DomName('CSSKeyframeRule.keyText')
  @DocsEditable()
  @Experimental() // untriaged
  set keyText(String value) =>
      _blink.BlinkCSSKeyframeRule.instance.keyText_Setter_(this, value);

  @DomName('CSSKeyframeRule.style')
  @DocsEditable()
  @Experimental() // untriaged
  CssStyleDeclaration get style =>
      _blink.BlinkCSSKeyframeRule.instance.style_Getter_(this);
}
// Copyright (c) 2012, the Dart project authors.  Please see the AUTHORS file
// for details. All rights reserved. Use of this source code is governed by a
// BSD-style license that can be found in the LICENSE file.

// WARNING: Do not edit - generated code.

@DocsEditable()
@DomName('CSSKeyframesRule')
@Experimental() // untriaged
class CssKeyframesRule extends CssRule {
  // To suppress missing implicit constructor warnings.
  factory CssKeyframesRule._() {
    throw new UnsupportedError("Not supported");
  }

  @Deprecated("Internal Use Only")
  external static Type get instanceRuntimeType;

  @Deprecated("Internal Use Only")
  CssKeyframesRule.internal_() : super.internal_();

  @DomName('CSSKeyframesRule.cssRules')
  @DocsEditable()
  @Experimental() // untriaged
  List<CssRule> get cssRules =>
      _blink.BlinkCSSKeyframesRule.instance.cssRules_Getter_(this);

  @DomName('CSSKeyframesRule.name')
  @DocsEditable()
  @Experimental() // untriaged
  String get name => _blink.BlinkCSSKeyframesRule.instance.name_Getter_(this);

  @DomName('CSSKeyframesRule.name')
  @DocsEditable()
  @Experimental() // untriaged
  set name(String value) =>
      _blink.BlinkCSSKeyframesRule.instance.name_Setter_(this, value);

  @DomName('CSSKeyframesRule.__getter__')
  @DocsEditable()
  @Experimental() // untriaged
  CssKeyframeRule __getter__(int index) => _blink.BlinkCSSKeyframesRule.instance
      .$__getter___Callback_1_(this, index);

  @DomName('CSSKeyframesRule.appendRule')
  @DocsEditable()
  @Experimental() // untriaged
  void appendRule(String rule) =>
      _blink.BlinkCSSKeyframesRule.instance.appendRule_Callback_1_(this, rule);

  @DomName('CSSKeyframesRule.deleteRule')
  @DocsEditable()
  @Experimental() // untriaged
  void deleteRule(String select) => _blink.BlinkCSSKeyframesRule.instance
      .deleteRule_Callback_1_(this, select);

  @DomName('CSSKeyframesRule.findRule')
  @DocsEditable()
  @Experimental() // untriaged
  CssKeyframeRule findRule(String select) =>
      _blink.BlinkCSSKeyframesRule.instance.findRule_Callback_1_(this, select);
}
// Copyright (c) 2012, the Dart project authors.  Please see the AUTHORS file
// for details. All rights reserved. Use of this source code is governed by a
// BSD-style license that can be found in the LICENSE file.

// WARNING: Do not edit - generated code.

@DocsEditable()
@DomName('CSSMediaRule')
class CssMediaRule extends CssGroupingRule {
  // To suppress missing implicit constructor warnings.
  factory CssMediaRule._() {
    throw new UnsupportedError("Not supported");
  }

  @Deprecated("Internal Use Only")
  external static Type get instanceRuntimeType;

  @Deprecated("Internal Use Only")
  CssMediaRule.internal_() : super.internal_();

  @DomName('CSSMediaRule.media')
  @DocsEditable()
  MediaList get media => _blink.BlinkCSSMediaRule.instance.media_Getter_(this);
}
// Copyright (c) 2012, the Dart project authors.  Please see the AUTHORS file
// for details. All rights reserved. Use of this source code is governed by a
// BSD-style license that can be found in the LICENSE file.

// WARNING: Do not edit - generated code.

@DocsEditable()
@DomName('CSSPageRule')
class CssPageRule extends CssRule {
  // To suppress missing implicit constructor warnings.
  factory CssPageRule._() {
    throw new UnsupportedError("Not supported");
  }

  @Deprecated("Internal Use Only")
  external static Type get instanceRuntimeType;

  @Deprecated("Internal Use Only")
  CssPageRule.internal_() : super.internal_();

  @DomName('CSSPageRule.selectorText')
  @DocsEditable()
  String get selectorText =>
      _blink.BlinkCSSPageRule.instance.selectorText_Getter_(this);

  @DomName('CSSPageRule.selectorText')
  @DocsEditable()
  set selectorText(String value) =>
      _blink.BlinkCSSPageRule.instance.selectorText_Setter_(this, value);

  @DomName('CSSPageRule.style')
  @DocsEditable()
  CssStyleDeclaration get style =>
      _blink.BlinkCSSPageRule.instance.style_Getter_(this);
}
// Copyright (c) 2012, the Dart project authors.  Please see the AUTHORS file
// for details. All rights reserved. Use of this source code is governed by a
// BSD-style license that can be found in the LICENSE file.

// WARNING: Do not edit - generated code.

@DocsEditable()
@DomName('CSSRule')
class CssRule extends DartHtmlDomObject {
  // To suppress missing implicit constructor warnings.
  factory CssRule._() {
    throw new UnsupportedError("Not supported");
  }

  @Deprecated("Internal Use Only")
  external static Type get instanceRuntimeType;

  @Deprecated("Internal Use Only")
  CssRule.internal_() {}

  @DomName('CSSRule.CHARSET_RULE')
  @DocsEditable()
  static const int CHARSET_RULE = 2;

  @DomName('CSSRule.FONT_FACE_RULE')
  @DocsEditable()
  static const int FONT_FACE_RULE = 5;

  @DomName('CSSRule.IMPORT_RULE')
  @DocsEditable()
  static const int IMPORT_RULE = 3;

  @DomName('CSSRule.KEYFRAMES_RULE')
  @DocsEditable()
  @Experimental() // untriaged
  static const int KEYFRAMES_RULE = 7;

  @DomName('CSSRule.KEYFRAME_RULE')
  @DocsEditable()
  @Experimental() // untriaged
  static const int KEYFRAME_RULE = 8;

  @DomName('CSSRule.MEDIA_RULE')
  @DocsEditable()
  static const int MEDIA_RULE = 4;

  @DomName('CSSRule.PAGE_RULE')
  @DocsEditable()
  static const int PAGE_RULE = 6;

  @DomName('CSSRule.STYLE_RULE')
  @DocsEditable()
  static const int STYLE_RULE = 1;

  @DomName('CSSRule.SUPPORTS_RULE')
  @DocsEditable()
  static const int SUPPORTS_RULE = 12;

  @DomName('CSSRule.VIEWPORT_RULE')
  @DocsEditable()
  @Experimental() // untriaged
  static const int VIEWPORT_RULE = 15;

  @DomName('CSSRule.WEBKIT_KEYFRAMES_RULE')
  @DocsEditable()
  // http://www.w3.org/TR/css3-animations/#cssrule
  @Experimental()
  static const int WEBKIT_KEYFRAMES_RULE = 7;

  @DomName('CSSRule.WEBKIT_KEYFRAME_RULE')
  @DocsEditable()
  // http://www.w3.org/TR/css3-animations/#cssrule
  @Experimental()
  static const int WEBKIT_KEYFRAME_RULE = 8;

  @DomName('CSSRule.cssText')
  @DocsEditable()
  String get cssText => _blink.BlinkCSSRule.instance.cssText_Getter_(this);

  @DomName('CSSRule.cssText')
  @DocsEditable()
  set cssText(String value) =>
      _blink.BlinkCSSRule.instance.cssText_Setter_(this, value);

  @DomName('CSSRule.parentRule')
  @DocsEditable()
  CssRule get parentRule =>
      _blink.BlinkCSSRule.instance.parentRule_Getter_(this);

  @DomName('CSSRule.parentStyleSheet')
  @DocsEditable()
  CssStyleSheet get parentStyleSheet =>
      _blink.BlinkCSSRule.instance.parentStyleSheet_Getter_(this);

  @DomName('CSSRule.type')
  @DocsEditable()
  int get type => _blink.BlinkCSSRule.instance.type_Getter_(this);
}

// Copyright (c) 2014, the Dart project authors.  Please see the AUTHORS file
// for details. All rights reserved. Use of this source code is governed by a
// BSD-style license that can be found in the LICENSE file.

// WARNING: DO NOT EDIT THIS TEMPLATE FILE.
// The template file was generated by scripts/css_code_generator.py

// Source of CSS properties:
//   CSSPropertyNames.in

@DomName('CSSStyleDeclaration')
class CssStyleDeclaration extends DartHtmlDomObject
    with CssStyleDeclarationBase {
  factory CssStyleDeclaration() => new CssStyleDeclaration.css('');

  factory CssStyleDeclaration.css(String css) {
    final style = new DivElement().style;
    style.cssText = css;
    return style;
  }

  /// Returns the value of the property if the provided *CSS* property
  /// name is supported on this element and if the value is set. Otherwise
  /// returns an empty string.
  ///
  /// Please note the property name uses camelCase, not-hyphens.
  String getPropertyValue(String propertyName) {
    var propValue = _getPropertyValueHelper(propertyName);
    return propValue != null ? propValue : '';
  }

  String _getPropertyValueHelper(String propertyName) {
    if (_supportsProperty(_camelCase(propertyName))) {
      return _getPropertyValue(propertyName);
    } else {
      return _getPropertyValue(Device.cssPrefix + propertyName);
    }
  }

  /**
   * Returns true if the provided *CSS* property name is supported on this
   * element.
   *
   * Please note the property name camelCase, not-hyphens. This
   * method returns true if the property is accessible via an unprefixed _or_
   * prefixed property.
   */
  bool supportsProperty(String propertyName) {
    return _supportsProperty(propertyName) ||
        _supportsProperty(_camelCase(Device.cssPrefix + propertyName));
  }

  bool _supportsProperty(String propertyName) {
    // You can't just check the value of a property, because there is no way
    // to distinguish between property not being present in the browser and
    // not having a value at all. (Ultimately we'll want the native method to
    // return null if the property doesn't exist and empty string if it's
    // defined but just doesn't have a value.
    return _hasProperty(propertyName);
  }

  bool _hasProperty(String propertyName) =>
      _blink.BlinkCSSStyleDeclaration.instance
          .$__propertyQuery___Callback_1_(this, propertyName);

  @DomName('CSSStyleDeclaration.setProperty')
  void setProperty(String propertyName, String value, [String priority]) {
    return _setPropertyHelper(
        _browserPropertyName(propertyName), value, priority);
  }

  String _browserPropertyName(String propertyName) {
    String name = _readCache(propertyName);
    if (name is String) return name;
    if (_supportsProperty(_camelCase(propertyName))) {
      name = propertyName;
    } else {
      name = Device.cssPrefix + propertyName;
    }
    _writeCache(propertyName, name);
    return name;
  }

  static String _readCache(String key) => null;
  static void _writeCache(String key, value) {}

  static String _camelCase(String hyphenated) {
    // The "ms" prefix is always lowercased.
    return hyphenated.replaceFirst(new RegExp('^-ms-'), 'ms-').replaceAllMapped(
        new RegExp('-([a-z]+)', caseSensitive: false),
        (match) => match[0][1].toUpperCase() + match[0].substring(2));
  }

  void _setPropertyHelper(String propertyName, String value,
      [String priority]) {
    if (priority == null) {
      priority = '';
    }
    _setProperty(propertyName, value, priority);
  }

  /**
   * Checks to see if CSS Transitions are supported.
   */
  static bool get supportsTransitions => true;
  // To suppress missing implicit constructor warnings.
  factory CssStyleDeclaration._() {
    throw new UnsupportedError("Not supported");
  }

  @Deprecated("Internal Use Only")
  external static Type get instanceRuntimeType;

  @Deprecated("Internal Use Only")
  CssStyleDeclaration.internal_() {}

  @DomName('CSSStyleDeclaration.cssText')
  @DocsEditable()
  String get cssText =>
      _blink.BlinkCSSStyleDeclaration.instance.cssText_Getter_(this);

  @DomName('CSSStyleDeclaration.cssText')
  @DocsEditable()
  set cssText(String value) =>
      _blink.BlinkCSSStyleDeclaration.instance.cssText_Setter_(this, value);

  @DomName('CSSStyleDeclaration.length')
  @DocsEditable()
  int get length =>
      _blink.BlinkCSSStyleDeclaration.instance.length_Getter_(this);

  @DomName('CSSStyleDeclaration.parentRule')
  @DocsEditable()
  CssRule get parentRule =>
      _blink.BlinkCSSStyleDeclaration.instance.parentRule_Getter_(this);

  @DomName('CSSStyleDeclaration.__propertyQuery__')
  @DocsEditable()
  @Experimental() // untriaged
  bool __propertyQuery__(String name) =>
      _blink.BlinkCSSStyleDeclaration.instance
          .$__propertyQuery___Callback_1_(this, name);

  @DomName('CSSStyleDeclaration.getPropertyPriority')
  @DocsEditable()
  String getPropertyPriority(String property) =>
      _blink.BlinkCSSStyleDeclaration.instance
          .getPropertyPriority_Callback_1_(this, property);

  @DomName('CSSStyleDeclaration.getPropertyValue')
  @DocsEditable()
  String _getPropertyValue(String property) =>
      _blink.BlinkCSSStyleDeclaration.instance
          .getPropertyValue_Callback_1_(this, property);

  @DomName('CSSStyleDeclaration.item')
  @DocsEditable()
  String item(int index) =>
      _blink.BlinkCSSStyleDeclaration.instance.item_Callback_1_(this, index);

  @DomName('CSSStyleDeclaration.removeProperty')
  @DocsEditable()
  String removeProperty(String property) =>
      _blink.BlinkCSSStyleDeclaration.instance
          .removeProperty_Callback_1_(this, property);

  @DomName('CSSStyleDeclaration.setProperty')
  @DocsEditable()
  void _setProperty(String property, String value, String priority) =>
      _blink.BlinkCSSStyleDeclaration.instance
          .setProperty_Callback_3_(this, property, value, priority);
}

class _CssStyleDeclarationSet extends Object with CssStyleDeclarationBase {
  final Iterable<Element> _elementIterable;
  Iterable<CssStyleDeclaration> _elementCssStyleDeclarationSetIterable;

  _CssStyleDeclarationSet(this._elementIterable) {
    _elementCssStyleDeclarationSetIterable =
        new List.from(_elementIterable).map((e) => e.style);
  }

  String getPropertyValue(String propertyName) =>
      _elementCssStyleDeclarationSetIterable.first
          .getPropertyValue(propertyName);

  void setProperty(String propertyName, String value, [String priority]) {
    _elementCssStyleDeclarationSetIterable
        .forEach((e) => e.setProperty(propertyName, value, priority));
  }

  // Important note: CssStyleDeclarationSet does NOT implement every method
  // available in CssStyleDeclaration. Some of the methods don't make so much
  // sense in terms of having a resonable value to return when you're
  // considering a list of Elements. You will need to manually add any of the
  // items in the MEMBERS set if you want that functionality.
}

class CssStyleDeclarationBase {
  String getPropertyValue(String propertyName) =>
      throw new StateError('getProperty not overridden in dart:html');
  void setProperty(String propertyName, String value, [String priority]) =>
      throw new StateError('setProperty not overridden in dart:html');

  /** Gets the value of "align-content" */
  String get alignContent => getPropertyValue('align-content');

  /** Sets the value of "align-content" */
  set alignContent(String value) {
    setProperty('align-content', value, '');
  }

  /** Gets the value of "align-items" */
  String get alignItems => getPropertyValue('align-items');

  /** Sets the value of "align-items" */
  set alignItems(String value) {
    setProperty('align-items', value, '');
  }

  /** Gets the value of "align-self" */
  String get alignSelf => getPropertyValue('align-self');

  /** Sets the value of "align-self" */
  set alignSelf(String value) {
    setProperty('align-self', value, '');
  }

  /** Gets the value of "animation" */
  String get animation => getPropertyValue('animation');

  /** Sets the value of "animation" */
  set animation(String value) {
    setProperty('animation', value, '');
  }

  /** Gets the value of "animation-delay" */
  String get animationDelay => getPropertyValue('animation-delay');

  /** Sets the value of "animation-delay" */
  set animationDelay(String value) {
    setProperty('animation-delay', value, '');
  }

  /** Gets the value of "animation-direction" */
  String get animationDirection => getPropertyValue('animation-direction');

  /** Sets the value of "animation-direction" */
  set animationDirection(String value) {
    setProperty('animation-direction', value, '');
  }

  /** Gets the value of "animation-duration" */
  String get animationDuration => getPropertyValue('animation-duration');

  /** Sets the value of "animation-duration" */
  set animationDuration(String value) {
    setProperty('animation-duration', value, '');
  }

  /** Gets the value of "animation-fill-mode" */
  String get animationFillMode => getPropertyValue('animation-fill-mode');

  /** Sets the value of "animation-fill-mode" */
  set animationFillMode(String value) {
    setProperty('animation-fill-mode', value, '');
  }

  /** Gets the value of "animation-iteration-count" */
  String get animationIterationCount =>
      getPropertyValue('animation-iteration-count');

  /** Sets the value of "animation-iteration-count" */
  set animationIterationCount(String value) {
    setProperty('animation-iteration-count', value, '');
  }

  /** Gets the value of "animation-name" */
  String get animationName => getPropertyValue('animation-name');

  /** Sets the value of "animation-name" */
  set animationName(String value) {
    setProperty('animation-name', value, '');
  }

  /** Gets the value of "animation-play-state" */
  String get animationPlayState => getPropertyValue('animation-play-state');

  /** Sets the value of "animation-play-state" */
  set animationPlayState(String value) {
    setProperty('animation-play-state', value, '');
  }

  /** Gets the value of "animation-timing-function" */
  String get animationTimingFunction =>
      getPropertyValue('animation-timing-function');

  /** Sets the value of "animation-timing-function" */
  set animationTimingFunction(String value) {
    setProperty('animation-timing-function', value, '');
  }

  /** Gets the value of "app-region" */
  String get appRegion => getPropertyValue('app-region');

  /** Sets the value of "app-region" */
  set appRegion(String value) {
    setProperty('app-region', value, '');
  }

  /** Gets the value of "appearance" */
  String get appearance => getPropertyValue('appearance');

  /** Sets the value of "appearance" */
  set appearance(String value) {
    setProperty('appearance', value, '');
  }

  /** Gets the value of "aspect-ratio" */
  String get aspectRatio => getPropertyValue('aspect-ratio');

  /** Sets the value of "aspect-ratio" */
  set aspectRatio(String value) {
    setProperty('aspect-ratio', value, '');
  }

  /** Gets the value of "backface-visibility" */
  String get backfaceVisibility => getPropertyValue('backface-visibility');

  /** Sets the value of "backface-visibility" */
  set backfaceVisibility(String value) {
    setProperty('backface-visibility', value, '');
  }

  /** Gets the value of "background" */
  String get background => getPropertyValue('background');

  /** Sets the value of "background" */
  set background(String value) {
    setProperty('background', value, '');
  }

  /** Gets the value of "background-attachment" */
  String get backgroundAttachment => getPropertyValue('background-attachment');

  /** Sets the value of "background-attachment" */
  set backgroundAttachment(String value) {
    setProperty('background-attachment', value, '');
  }

  /** Gets the value of "background-blend-mode" */
  String get backgroundBlendMode => getPropertyValue('background-blend-mode');

  /** Sets the value of "background-blend-mode" */
  set backgroundBlendMode(String value) {
    setProperty('background-blend-mode', value, '');
  }

  /** Gets the value of "background-clip" */
  String get backgroundClip => getPropertyValue('background-clip');

  /** Sets the value of "background-clip" */
  set backgroundClip(String value) {
    setProperty('background-clip', value, '');
  }

  /** Gets the value of "background-color" */
  String get backgroundColor => getPropertyValue('background-color');

  /** Sets the value of "background-color" */
  set backgroundColor(String value) {
    setProperty('background-color', value, '');
  }

  /** Gets the value of "background-composite" */
  String get backgroundComposite => getPropertyValue('background-composite');

  /** Sets the value of "background-composite" */
  set backgroundComposite(String value) {
    setProperty('background-composite', value, '');
  }

  /** Gets the value of "background-image" */
  String get backgroundImage => getPropertyValue('background-image');

  /** Sets the value of "background-image" */
  set backgroundImage(String value) {
    setProperty('background-image', value, '');
  }

  /** Gets the value of "background-origin" */
  String get backgroundOrigin => getPropertyValue('background-origin');

  /** Sets the value of "background-origin" */
  set backgroundOrigin(String value) {
    setProperty('background-origin', value, '');
  }

  /** Gets the value of "background-position" */
  String get backgroundPosition => getPropertyValue('background-position');

  /** Sets the value of "background-position" */
  set backgroundPosition(String value) {
    setProperty('background-position', value, '');
  }

  /** Gets the value of "background-position-x" */
  String get backgroundPositionX => getPropertyValue('background-position-x');

  /** Sets the value of "background-position-x" */
  set backgroundPositionX(String value) {
    setProperty('background-position-x', value, '');
  }

  /** Gets the value of "background-position-y" */
  String get backgroundPositionY => getPropertyValue('background-position-y');

  /** Sets the value of "background-position-y" */
  set backgroundPositionY(String value) {
    setProperty('background-position-y', value, '');
  }

  /** Gets the value of "background-repeat" */
  String get backgroundRepeat => getPropertyValue('background-repeat');

  /** Sets the value of "background-repeat" */
  set backgroundRepeat(String value) {
    setProperty('background-repeat', value, '');
  }

  /** Gets the value of "background-repeat-x" */
  String get backgroundRepeatX => getPropertyValue('background-repeat-x');

  /** Sets the value of "background-repeat-x" */
  set backgroundRepeatX(String value) {
    setProperty('background-repeat-x', value, '');
  }

  /** Gets the value of "background-repeat-y" */
  String get backgroundRepeatY => getPropertyValue('background-repeat-y');

  /** Sets the value of "background-repeat-y" */
  set backgroundRepeatY(String value) {
    setProperty('background-repeat-y', value, '');
  }

  /** Gets the value of "background-size" */
  String get backgroundSize => getPropertyValue('background-size');

  /** Sets the value of "background-size" */
  set backgroundSize(String value) {
    setProperty('background-size', value, '');
  }

  /** Gets the value of "border" */
  String get border => getPropertyValue('border');

  /** Sets the value of "border" */
  set border(String value) {
    setProperty('border', value, '');
  }

  /** Gets the value of "border-after" */
  String get borderAfter => getPropertyValue('border-after');

  /** Sets the value of "border-after" */
  set borderAfter(String value) {
    setProperty('border-after', value, '');
  }

  /** Gets the value of "border-after-color" */
  String get borderAfterColor => getPropertyValue('border-after-color');

  /** Sets the value of "border-after-color" */
  set borderAfterColor(String value) {
    setProperty('border-after-color', value, '');
  }

  /** Gets the value of "border-after-style" */
  String get borderAfterStyle => getPropertyValue('border-after-style');

  /** Sets the value of "border-after-style" */
  set borderAfterStyle(String value) {
    setProperty('border-after-style', value, '');
  }

  /** Gets the value of "border-after-width" */
  String get borderAfterWidth => getPropertyValue('border-after-width');

  /** Sets the value of "border-after-width" */
  set borderAfterWidth(String value) {
    setProperty('border-after-width', value, '');
  }

  /** Gets the value of "border-before" */
  String get borderBefore => getPropertyValue('border-before');

  /** Sets the value of "border-before" */
  set borderBefore(String value) {
    setProperty('border-before', value, '');
  }

  /** Gets the value of "border-before-color" */
  String get borderBeforeColor => getPropertyValue('border-before-color');

  /** Sets the value of "border-before-color" */
  set borderBeforeColor(String value) {
    setProperty('border-before-color', value, '');
  }

  /** Gets the value of "border-before-style" */
  String get borderBeforeStyle => getPropertyValue('border-before-style');

  /** Sets the value of "border-before-style" */
  set borderBeforeStyle(String value) {
    setProperty('border-before-style', value, '');
  }

  /** Gets the value of "border-before-width" */
  String get borderBeforeWidth => getPropertyValue('border-before-width');

  /** Sets the value of "border-before-width" */
  set borderBeforeWidth(String value) {
    setProperty('border-before-width', value, '');
  }

  /** Gets the value of "border-bottom" */
  String get borderBottom => getPropertyValue('border-bottom');

  /** Sets the value of "border-bottom" */
  set borderBottom(String value) {
    setProperty('border-bottom', value, '');
  }

  /** Gets the value of "border-bottom-color" */
  String get borderBottomColor => getPropertyValue('border-bottom-color');

  /** Sets the value of "border-bottom-color" */
  set borderBottomColor(String value) {
    setProperty('border-bottom-color', value, '');
  }

  /** Gets the value of "border-bottom-left-radius" */
  String get borderBottomLeftRadius =>
      getPropertyValue('border-bottom-left-radius');

  /** Sets the value of "border-bottom-left-radius" */
  set borderBottomLeftRadius(String value) {
    setProperty('border-bottom-left-radius', value, '');
  }

  /** Gets the value of "border-bottom-right-radius" */
  String get borderBottomRightRadius =>
      getPropertyValue('border-bottom-right-radius');

  /** Sets the value of "border-bottom-right-radius" */
  set borderBottomRightRadius(String value) {
    setProperty('border-bottom-right-radius', value, '');
  }

  /** Gets the value of "border-bottom-style" */
  String get borderBottomStyle => getPropertyValue('border-bottom-style');

  /** Sets the value of "border-bottom-style" */
  set borderBottomStyle(String value) {
    setProperty('border-bottom-style', value, '');
  }

  /** Gets the value of "border-bottom-width" */
  String get borderBottomWidth => getPropertyValue('border-bottom-width');

  /** Sets the value of "border-bottom-width" */
  set borderBottomWidth(String value) {
    setProperty('border-bottom-width', value, '');
  }

  /** Gets the value of "border-collapse" */
  String get borderCollapse => getPropertyValue('border-collapse');

  /** Sets the value of "border-collapse" */
  set borderCollapse(String value) {
    setProperty('border-collapse', value, '');
  }

  /** Gets the value of "border-color" */
  String get borderColor => getPropertyValue('border-color');

  /** Sets the value of "border-color" */
  set borderColor(String value) {
    setProperty('border-color', value, '');
  }

  /** Gets the value of "border-end" */
  String get borderEnd => getPropertyValue('border-end');

  /** Sets the value of "border-end" */
  set borderEnd(String value) {
    setProperty('border-end', value, '');
  }

  /** Gets the value of "border-end-color" */
  String get borderEndColor => getPropertyValue('border-end-color');

  /** Sets the value of "border-end-color" */
  set borderEndColor(String value) {
    setProperty('border-end-color', value, '');
  }

  /** Gets the value of "border-end-style" */
  String get borderEndStyle => getPropertyValue('border-end-style');

  /** Sets the value of "border-end-style" */
  set borderEndStyle(String value) {
    setProperty('border-end-style', value, '');
  }

  /** Gets the value of "border-end-width" */
  String get borderEndWidth => getPropertyValue('border-end-width');

  /** Sets the value of "border-end-width" */
  set borderEndWidth(String value) {
    setProperty('border-end-width', value, '');
  }

  /** Gets the value of "border-fit" */
  String get borderFit => getPropertyValue('border-fit');

  /** Sets the value of "border-fit" */
  set borderFit(String value) {
    setProperty('border-fit', value, '');
  }

  /** Gets the value of "border-horizontal-spacing" */
  String get borderHorizontalSpacing =>
      getPropertyValue('border-horizontal-spacing');

  /** Sets the value of "border-horizontal-spacing" */
  set borderHorizontalSpacing(String value) {
    setProperty('border-horizontal-spacing', value, '');
  }

  /** Gets the value of "border-image" */
  String get borderImage => getPropertyValue('border-image');

  /** Sets the value of "border-image" */
  set borderImage(String value) {
    setProperty('border-image', value, '');
  }

  /** Gets the value of "border-image-outset" */
  String get borderImageOutset => getPropertyValue('border-image-outset');

  /** Sets the value of "border-image-outset" */
  set borderImageOutset(String value) {
    setProperty('border-image-outset', value, '');
  }

  /** Gets the value of "border-image-repeat" */
  String get borderImageRepeat => getPropertyValue('border-image-repeat');

  /** Sets the value of "border-image-repeat" */
  set borderImageRepeat(String value) {
    setProperty('border-image-repeat', value, '');
  }

  /** Gets the value of "border-image-slice" */
  String get borderImageSlice => getPropertyValue('border-image-slice');

  /** Sets the value of "border-image-slice" */
  set borderImageSlice(String value) {
    setProperty('border-image-slice', value, '');
  }

  /** Gets the value of "border-image-source" */
  String get borderImageSource => getPropertyValue('border-image-source');

  /** Sets the value of "border-image-source" */
  set borderImageSource(String value) {
    setProperty('border-image-source', value, '');
  }

  /** Gets the value of "border-image-width" */
  String get borderImageWidth => getPropertyValue('border-image-width');

  /** Sets the value of "border-image-width" */
  set borderImageWidth(String value) {
    setProperty('border-image-width', value, '');
  }

  /** Gets the value of "border-left" */
  String get borderLeft => getPropertyValue('border-left');

  /** Sets the value of "border-left" */
  set borderLeft(String value) {
    setProperty('border-left', value, '');
  }

  /** Gets the value of "border-left-color" */
  String get borderLeftColor => getPropertyValue('border-left-color');

  /** Sets the value of "border-left-color" */
  set borderLeftColor(String value) {
    setProperty('border-left-color', value, '');
  }

  /** Gets the value of "border-left-style" */
  String get borderLeftStyle => getPropertyValue('border-left-style');

  /** Sets the value of "border-left-style" */
  set borderLeftStyle(String value) {
    setProperty('border-left-style', value, '');
  }

  /** Gets the value of "border-left-width" */
  String get borderLeftWidth => getPropertyValue('border-left-width');

  /** Sets the value of "border-left-width" */
  set borderLeftWidth(String value) {
    setProperty('border-left-width', value, '');
  }

  /** Gets the value of "border-radius" */
  String get borderRadius => getPropertyValue('border-radius');

  /** Sets the value of "border-radius" */
  set borderRadius(String value) {
    setProperty('border-radius', value, '');
  }

  /** Gets the value of "border-right" */
  String get borderRight => getPropertyValue('border-right');

  /** Sets the value of "border-right" */
  set borderRight(String value) {
    setProperty('border-right', value, '');
  }

  /** Gets the value of "border-right-color" */
  String get borderRightColor => getPropertyValue('border-right-color');

  /** Sets the value of "border-right-color" */
  set borderRightColor(String value) {
    setProperty('border-right-color', value, '');
  }

  /** Gets the value of "border-right-style" */
  String get borderRightStyle => getPropertyValue('border-right-style');

  /** Sets the value of "border-right-style" */
  set borderRightStyle(String value) {
    setProperty('border-right-style', value, '');
  }

  /** Gets the value of "border-right-width" */
  String get borderRightWidth => getPropertyValue('border-right-width');

  /** Sets the value of "border-right-width" */
  set borderRightWidth(String value) {
    setProperty('border-right-width', value, '');
  }

  /** Gets the value of "border-spacing" */
  String get borderSpacing => getPropertyValue('border-spacing');

  /** Sets the value of "border-spacing" */
  set borderSpacing(String value) {
    setProperty('border-spacing', value, '');
  }

  /** Gets the value of "border-start" */
  String get borderStart => getPropertyValue('border-start');

  /** Sets the value of "border-start" */
  set borderStart(String value) {
    setProperty('border-start', value, '');
  }

  /** Gets the value of "border-start-color" */
  String get borderStartColor => getPropertyValue('border-start-color');

  /** Sets the value of "border-start-color" */
  set borderStartColor(String value) {
    setProperty('border-start-color', value, '');
  }

  /** Gets the value of "border-start-style" */
  String get borderStartStyle => getPropertyValue('border-start-style');

  /** Sets the value of "border-start-style" */
  set borderStartStyle(String value) {
    setProperty('border-start-style', value, '');
  }

  /** Gets the value of "border-start-width" */
  String get borderStartWidth => getPropertyValue('border-start-width');

  /** Sets the value of "border-start-width" */
  set borderStartWidth(String value) {
    setProperty('border-start-width', value, '');
  }

  /** Gets the value of "border-style" */
  String get borderStyle => getPropertyValue('border-style');

  /** Sets the value of "border-style" */
  set borderStyle(String value) {
    setProperty('border-style', value, '');
  }

  /** Gets the value of "border-top" */
  String get borderTop => getPropertyValue('border-top');

  /** Sets the value of "border-top" */
  set borderTop(String value) {
    setProperty('border-top', value, '');
  }

  /** Gets the value of "border-top-color" */
  String get borderTopColor => getPropertyValue('border-top-color');

  /** Sets the value of "border-top-color" */
  set borderTopColor(String value) {
    setProperty('border-top-color', value, '');
  }

  /** Gets the value of "border-top-left-radius" */
  String get borderTopLeftRadius => getPropertyValue('border-top-left-radius');

  /** Sets the value of "border-top-left-radius" */
  set borderTopLeftRadius(String value) {
    setProperty('border-top-left-radius', value, '');
  }

  /** Gets the value of "border-top-right-radius" */
  String get borderTopRightRadius =>
      getPropertyValue('border-top-right-radius');

  /** Sets the value of "border-top-right-radius" */
  set borderTopRightRadius(String value) {
    setProperty('border-top-right-radius', value, '');
  }

  /** Gets the value of "border-top-style" */
  String get borderTopStyle => getPropertyValue('border-top-style');

  /** Sets the value of "border-top-style" */
  set borderTopStyle(String value) {
    setProperty('border-top-style', value, '');
  }

  /** Gets the value of "border-top-width" */
  String get borderTopWidth => getPropertyValue('border-top-width');

  /** Sets the value of "border-top-width" */
  set borderTopWidth(String value) {
    setProperty('border-top-width', value, '');
  }

  /** Gets the value of "border-vertical-spacing" */
  String get borderVerticalSpacing =>
      getPropertyValue('border-vertical-spacing');

  /** Sets the value of "border-vertical-spacing" */
  set borderVerticalSpacing(String value) {
    setProperty('border-vertical-spacing', value, '');
  }

  /** Gets the value of "border-width" */
  String get borderWidth => getPropertyValue('border-width');

  /** Sets the value of "border-width" */
  set borderWidth(String value) {
    setProperty('border-width', value, '');
  }

  /** Gets the value of "bottom" */
  String get bottom => getPropertyValue('bottom');

  /** Sets the value of "bottom" */
  set bottom(String value) {
    setProperty('bottom', value, '');
  }

  /** Gets the value of "box-align" */
  String get boxAlign => getPropertyValue('box-align');

  /** Sets the value of "box-align" */
  set boxAlign(String value) {
    setProperty('box-align', value, '');
  }

  /** Gets the value of "box-decoration-break" */
  String get boxDecorationBreak => getPropertyValue('box-decoration-break');

  /** Sets the value of "box-decoration-break" */
  set boxDecorationBreak(String value) {
    setProperty('box-decoration-break', value, '');
  }

  /** Gets the value of "box-direction" */
  String get boxDirection => getPropertyValue('box-direction');

  /** Sets the value of "box-direction" */
  set boxDirection(String value) {
    setProperty('box-direction', value, '');
  }

  /** Gets the value of "box-flex" */
  String get boxFlex => getPropertyValue('box-flex');

  /** Sets the value of "box-flex" */
  set boxFlex(String value) {
    setProperty('box-flex', value, '');
  }

  /** Gets the value of "box-flex-group" */
  String get boxFlexGroup => getPropertyValue('box-flex-group');

  /** Sets the value of "box-flex-group" */
  set boxFlexGroup(String value) {
    setProperty('box-flex-group', value, '');
  }

  /** Gets the value of "box-lines" */
  String get boxLines => getPropertyValue('box-lines');

  /** Sets the value of "box-lines" */
  set boxLines(String value) {
    setProperty('box-lines', value, '');
  }

  /** Gets the value of "box-ordinal-group" */
  String get boxOrdinalGroup => getPropertyValue('box-ordinal-group');

  /** Sets the value of "box-ordinal-group" */
  set boxOrdinalGroup(String value) {
    setProperty('box-ordinal-group', value, '');
  }

  /** Gets the value of "box-orient" */
  String get boxOrient => getPropertyValue('box-orient');

  /** Sets the value of "box-orient" */
  set boxOrient(String value) {
    setProperty('box-orient', value, '');
  }

  /** Gets the value of "box-pack" */
  String get boxPack => getPropertyValue('box-pack');

  /** Sets the value of "box-pack" */
  set boxPack(String value) {
    setProperty('box-pack', value, '');
  }

  /** Gets the value of "box-reflect" */
  String get boxReflect => getPropertyValue('box-reflect');

  /** Sets the value of "box-reflect" */
  set boxReflect(String value) {
    setProperty('box-reflect', value, '');
  }

  /** Gets the value of "box-shadow" */
  String get boxShadow => getPropertyValue('box-shadow');

  /** Sets the value of "box-shadow" */
  set boxShadow(String value) {
    setProperty('box-shadow', value, '');
  }

  /** Gets the value of "box-sizing" */
  String get boxSizing => getPropertyValue('box-sizing');

  /** Sets the value of "box-sizing" */
  set boxSizing(String value) {
    setProperty('box-sizing', value, '');
  }

  /** Gets the value of "caption-side" */
  String get captionSide => getPropertyValue('caption-side');

  /** Sets the value of "caption-side" */
  set captionSide(String value) {
    setProperty('caption-side', value, '');
  }

  /** Gets the value of "clear" */
  String get clear => getPropertyValue('clear');

  /** Sets the value of "clear" */
  set clear(String value) {
    setProperty('clear', value, '');
  }

  /** Gets the value of "clip" */
  String get clip => getPropertyValue('clip');

  /** Sets the value of "clip" */
  set clip(String value) {
    setProperty('clip', value, '');
  }

  /** Gets the value of "clip-path" */
  String get clipPath => getPropertyValue('clip-path');

  /** Sets the value of "clip-path" */
  set clipPath(String value) {
    setProperty('clip-path', value, '');
  }

  /** Gets the value of "color" */
  String get color => getPropertyValue('color');

  /** Sets the value of "color" */
  set color(String value) {
    setProperty('color', value, '');
  }

  /** Gets the value of "column-break-after" */
  String get columnBreakAfter => getPropertyValue('column-break-after');

  /** Sets the value of "column-break-after" */
  set columnBreakAfter(String value) {
    setProperty('column-break-after', value, '');
  }

  /** Gets the value of "column-break-before" */
  String get columnBreakBefore => getPropertyValue('column-break-before');

  /** Sets the value of "column-break-before" */
  set columnBreakBefore(String value) {
    setProperty('column-break-before', value, '');
  }

  /** Gets the value of "column-break-inside" */
  String get columnBreakInside => getPropertyValue('column-break-inside');

  /** Sets the value of "column-break-inside" */
  set columnBreakInside(String value) {
    setProperty('column-break-inside', value, '');
  }

  /** Gets the value of "column-count" */
  String get columnCount => getPropertyValue('column-count');

  /** Sets the value of "column-count" */
  set columnCount(String value) {
    setProperty('column-count', value, '');
  }

  /** Gets the value of "column-fill" */
  String get columnFill => getPropertyValue('column-fill');

  /** Sets the value of "column-fill" */
  set columnFill(String value) {
    setProperty('column-fill', value, '');
  }

  /** Gets the value of "column-gap" */
  String get columnGap => getPropertyValue('column-gap');

  /** Sets the value of "column-gap" */
  set columnGap(String value) {
    setProperty('column-gap', value, '');
  }

  /** Gets the value of "column-rule" */
  String get columnRule => getPropertyValue('column-rule');

  /** Sets the value of "column-rule" */
  set columnRule(String value) {
    setProperty('column-rule', value, '');
  }

  /** Gets the value of "column-rule-color" */
  String get columnRuleColor => getPropertyValue('column-rule-color');

  /** Sets the value of "column-rule-color" */
  set columnRuleColor(String value) {
    setProperty('column-rule-color', value, '');
  }

  /** Gets the value of "column-rule-style" */
  String get columnRuleStyle => getPropertyValue('column-rule-style');

  /** Sets the value of "column-rule-style" */
  set columnRuleStyle(String value) {
    setProperty('column-rule-style', value, '');
  }

  /** Gets the value of "column-rule-width" */
  String get columnRuleWidth => getPropertyValue('column-rule-width');

  /** Sets the value of "column-rule-width" */
  set columnRuleWidth(String value) {
    setProperty('column-rule-width', value, '');
  }

  /** Gets the value of "column-span" */
  String get columnSpan => getPropertyValue('column-span');

  /** Sets the value of "column-span" */
  set columnSpan(String value) {
    setProperty('column-span', value, '');
  }

  /** Gets the value of "column-width" */
  String get columnWidth => getPropertyValue('column-width');

  /** Sets the value of "column-width" */
  set columnWidth(String value) {
    setProperty('column-width', value, '');
  }

  /** Gets the value of "columns" */
  String get columns => getPropertyValue('columns');

  /** Sets the value of "columns" */
  set columns(String value) {
    setProperty('columns', value, '');
  }

  /** Gets the value of "content" */
  String get content => getPropertyValue('content');

  /** Sets the value of "content" */
  set content(String value) {
    setProperty('content', value, '');
  }

  /** Gets the value of "counter-increment" */
  String get counterIncrement => getPropertyValue('counter-increment');

  /** Sets the value of "counter-increment" */
  set counterIncrement(String value) {
    setProperty('counter-increment', value, '');
  }

  /** Gets the value of "counter-reset" */
  String get counterReset => getPropertyValue('counter-reset');

  /** Sets the value of "counter-reset" */
  set counterReset(String value) {
    setProperty('counter-reset', value, '');
  }

  /** Gets the value of "cursor" */
  String get cursor => getPropertyValue('cursor');

  /** Sets the value of "cursor" */
  set cursor(String value) {
    setProperty('cursor', value, '');
  }

  /** Gets the value of "direction" */
  String get direction => getPropertyValue('direction');

  /** Sets the value of "direction" */
  set direction(String value) {
    setProperty('direction', value, '');
  }

  /** Gets the value of "display" */
  String get display => getPropertyValue('display');

  /** Sets the value of "display" */
  set display(String value) {
    setProperty('display', value, '');
  }

  /** Gets the value of "empty-cells" */
  String get emptyCells => getPropertyValue('empty-cells');

  /** Sets the value of "empty-cells" */
  set emptyCells(String value) {
    setProperty('empty-cells', value, '');
  }

  /** Gets the value of "filter" */
  String get filter => getPropertyValue('filter');

  /** Sets the value of "filter" */
  set filter(String value) {
    setProperty('filter', value, '');
  }

  /** Gets the value of "flex" */
  String get flex => getPropertyValue('flex');

  /** Sets the value of "flex" */
  set flex(String value) {
    setProperty('flex', value, '');
  }

  /** Gets the value of "flex-basis" */
  String get flexBasis => getPropertyValue('flex-basis');

  /** Sets the value of "flex-basis" */
  set flexBasis(String value) {
    setProperty('flex-basis', value, '');
  }

  /** Gets the value of "flex-direction" */
  String get flexDirection => getPropertyValue('flex-direction');

  /** Sets the value of "flex-direction" */
  set flexDirection(String value) {
    setProperty('flex-direction', value, '');
  }

  /** Gets the value of "flex-flow" */
  String get flexFlow => getPropertyValue('flex-flow');

  /** Sets the value of "flex-flow" */
  set flexFlow(String value) {
    setProperty('flex-flow', value, '');
  }

  /** Gets the value of "flex-grow" */
  String get flexGrow => getPropertyValue('flex-grow');

  /** Sets the value of "flex-grow" */
  set flexGrow(String value) {
    setProperty('flex-grow', value, '');
  }

  /** Gets the value of "flex-shrink" */
  String get flexShrink => getPropertyValue('flex-shrink');

  /** Sets the value of "flex-shrink" */
  set flexShrink(String value) {
    setProperty('flex-shrink', value, '');
  }

  /** Gets the value of "flex-wrap" */
  String get flexWrap => getPropertyValue('flex-wrap');

  /** Sets the value of "flex-wrap" */
  set flexWrap(String value) {
    setProperty('flex-wrap', value, '');
  }

  /** Gets the value of "float" */
  String get float => getPropertyValue('float');

  /** Sets the value of "float" */
  set float(String value) {
    setProperty('float', value, '');
  }

  /** Gets the value of "font" */
  String get font => getPropertyValue('font');

  /** Sets the value of "font" */
  set font(String value) {
    setProperty('font', value, '');
  }

  /** Gets the value of "font-family" */
  String get fontFamily => getPropertyValue('font-family');

  /** Sets the value of "font-family" */
  set fontFamily(String value) {
    setProperty('font-family', value, '');
  }

  /** Gets the value of "font-feature-settings" */
  String get fontFeatureSettings => getPropertyValue('font-feature-settings');

  /** Sets the value of "font-feature-settings" */
  set fontFeatureSettings(String value) {
    setProperty('font-feature-settings', value, '');
  }

  /** Gets the value of "font-kerning" */
  String get fontKerning => getPropertyValue('font-kerning');

  /** Sets the value of "font-kerning" */
  set fontKerning(String value) {
    setProperty('font-kerning', value, '');
  }

  /** Gets the value of "font-size" */
  String get fontSize => getPropertyValue('font-size');

  /** Sets the value of "font-size" */
  set fontSize(String value) {
    setProperty('font-size', value, '');
  }

  /** Gets the value of "font-size-delta" */
  String get fontSizeDelta => getPropertyValue('font-size-delta');

  /** Sets the value of "font-size-delta" */
  set fontSizeDelta(String value) {
    setProperty('font-size-delta', value, '');
  }

  /** Gets the value of "font-smoothing" */
  String get fontSmoothing => getPropertyValue('font-smoothing');

  /** Sets the value of "font-smoothing" */
  set fontSmoothing(String value) {
    setProperty('font-smoothing', value, '');
  }

  /** Gets the value of "font-stretch" */
  String get fontStretch => getPropertyValue('font-stretch');

  /** Sets the value of "font-stretch" */
  set fontStretch(String value) {
    setProperty('font-stretch', value, '');
  }

  /** Gets the value of "font-style" */
  String get fontStyle => getPropertyValue('font-style');

  /** Sets the value of "font-style" */
  set fontStyle(String value) {
    setProperty('font-style', value, '');
  }

  /** Gets the value of "font-variant" */
  String get fontVariant => getPropertyValue('font-variant');

  /** Sets the value of "font-variant" */
  set fontVariant(String value) {
    setProperty('font-variant', value, '');
  }

  /** Gets the value of "font-variant-ligatures" */
  String get fontVariantLigatures => getPropertyValue('font-variant-ligatures');

  /** Sets the value of "font-variant-ligatures" */
  set fontVariantLigatures(String value) {
    setProperty('font-variant-ligatures', value, '');
  }

  /** Gets the value of "font-weight" */
  String get fontWeight => getPropertyValue('font-weight');

  /** Sets the value of "font-weight" */
  set fontWeight(String value) {
    setProperty('font-weight', value, '');
  }

  /** Gets the value of "grid" */
  String get grid => getPropertyValue('grid');

  /** Sets the value of "grid" */
  set grid(String value) {
    setProperty('grid', value, '');
  }

  /** Gets the value of "grid-area" */
  String get gridArea => getPropertyValue('grid-area');

  /** Sets the value of "grid-area" */
  set gridArea(String value) {
    setProperty('grid-area', value, '');
  }

  /** Gets the value of "grid-auto-columns" */
  String get gridAutoColumns => getPropertyValue('grid-auto-columns');

  /** Sets the value of "grid-auto-columns" */
  set gridAutoColumns(String value) {
    setProperty('grid-auto-columns', value, '');
  }

  /** Gets the value of "grid-auto-flow" */
  String get gridAutoFlow => getPropertyValue('grid-auto-flow');

  /** Sets the value of "grid-auto-flow" */
  set gridAutoFlow(String value) {
    setProperty('grid-auto-flow', value, '');
  }

  /** Gets the value of "grid-auto-rows" */
  String get gridAutoRows => getPropertyValue('grid-auto-rows');

  /** Sets the value of "grid-auto-rows" */
  set gridAutoRows(String value) {
    setProperty('grid-auto-rows', value, '');
  }

  /** Gets the value of "grid-column" */
  String get gridColumn => getPropertyValue('grid-column');

  /** Sets the value of "grid-column" */
  set gridColumn(String value) {
    setProperty('grid-column', value, '');
  }

  /** Gets the value of "grid-column-end" */
  String get gridColumnEnd => getPropertyValue('grid-column-end');

  /** Sets the value of "grid-column-end" */
  set gridColumnEnd(String value) {
    setProperty('grid-column-end', value, '');
  }

  /** Gets the value of "grid-column-start" */
  String get gridColumnStart => getPropertyValue('grid-column-start');

  /** Sets the value of "grid-column-start" */
  set gridColumnStart(String value) {
    setProperty('grid-column-start', value, '');
  }

  /** Gets the value of "grid-row" */
  String get gridRow => getPropertyValue('grid-row');

  /** Sets the value of "grid-row" */
  set gridRow(String value) {
    setProperty('grid-row', value, '');
  }

  /** Gets the value of "grid-row-end" */
  String get gridRowEnd => getPropertyValue('grid-row-end');

  /** Sets the value of "grid-row-end" */
  set gridRowEnd(String value) {
    setProperty('grid-row-end', value, '');
  }

  /** Gets the value of "grid-row-start" */
  String get gridRowStart => getPropertyValue('grid-row-start');

  /** Sets the value of "grid-row-start" */
  set gridRowStart(String value) {
    setProperty('grid-row-start', value, '');
  }

  /** Gets the value of "grid-template" */
  String get gridTemplate => getPropertyValue('grid-template');

  /** Sets the value of "grid-template" */
  set gridTemplate(String value) {
    setProperty('grid-template', value, '');
  }

  /** Gets the value of "grid-template-areas" */
  String get gridTemplateAreas => getPropertyValue('grid-template-areas');

  /** Sets the value of "grid-template-areas" */
  set gridTemplateAreas(String value) {
    setProperty('grid-template-areas', value, '');
  }

  /** Gets the value of "grid-template-columns" */
  String get gridTemplateColumns => getPropertyValue('grid-template-columns');

  /** Sets the value of "grid-template-columns" */
  set gridTemplateColumns(String value) {
    setProperty('grid-template-columns', value, '');
  }

  /** Gets the value of "grid-template-rows" */
  String get gridTemplateRows => getPropertyValue('grid-template-rows');

  /** Sets the value of "grid-template-rows" */
  set gridTemplateRows(String value) {
    setProperty('grid-template-rows', value, '');
  }

  /** Gets the value of "height" */
  String get height => getPropertyValue('height');

  /** Sets the value of "height" */
  set height(String value) {
    setProperty('height', value, '');
  }

  /** Gets the value of "highlight" */
  String get highlight => getPropertyValue('highlight');

  /** Sets the value of "highlight" */
  set highlight(String value) {
    setProperty('highlight', value, '');
  }

  /** Gets the value of "hyphenate-character" */
  String get hyphenateCharacter => getPropertyValue('hyphenate-character');

  /** Sets the value of "hyphenate-character" */
  set hyphenateCharacter(String value) {
    setProperty('hyphenate-character', value, '');
  }

  /** Gets the value of "image-rendering" */
  String get imageRendering => getPropertyValue('image-rendering');

  /** Sets the value of "image-rendering" */
  set imageRendering(String value) {
    setProperty('image-rendering', value, '');
  }

  /** Gets the value of "isolation" */
  String get isolation => getPropertyValue('isolation');

  /** Sets the value of "isolation" */
  set isolation(String value) {
    setProperty('isolation', value, '');
  }

  /** Gets the value of "justify-content" */
  String get justifyContent => getPropertyValue('justify-content');

  /** Sets the value of "justify-content" */
  set justifyContent(String value) {
    setProperty('justify-content', value, '');
  }

  /** Gets the value of "justify-self" */
  String get justifySelf => getPropertyValue('justify-self');

  /** Sets the value of "justify-self" */
  set justifySelf(String value) {
    setProperty('justify-self', value, '');
  }

  /** Gets the value of "left" */
  String get left => getPropertyValue('left');

  /** Sets the value of "left" */
  set left(String value) {
    setProperty('left', value, '');
  }

  /** Gets the value of "letter-spacing" */
  String get letterSpacing => getPropertyValue('letter-spacing');

  /** Sets the value of "letter-spacing" */
  set letterSpacing(String value) {
    setProperty('letter-spacing', value, '');
  }

  /** Gets the value of "line-box-contain" */
  String get lineBoxContain => getPropertyValue('line-box-contain');

  /** Sets the value of "line-box-contain" */
  set lineBoxContain(String value) {
    setProperty('line-box-contain', value, '');
  }

  /** Gets the value of "line-break" */
  String get lineBreak => getPropertyValue('line-break');

  /** Sets the value of "line-break" */
  set lineBreak(String value) {
    setProperty('line-break', value, '');
  }

  /** Gets the value of "line-clamp" */
  String get lineClamp => getPropertyValue('line-clamp');

  /** Sets the value of "line-clamp" */
  set lineClamp(String value) {
    setProperty('line-clamp', value, '');
  }

  /** Gets the value of "line-height" */
  String get lineHeight => getPropertyValue('line-height');

  /** Sets the value of "line-height" */
  set lineHeight(String value) {
    setProperty('line-height', value, '');
  }

  /** Gets the value of "list-style" */
  String get listStyle => getPropertyValue('list-style');

  /** Sets the value of "list-style" */
  set listStyle(String value) {
    setProperty('list-style', value, '');
  }

  /** Gets the value of "list-style-image" */
  String get listStyleImage => getPropertyValue('list-style-image');

  /** Sets the value of "list-style-image" */
  set listStyleImage(String value) {
    setProperty('list-style-image', value, '');
  }

  /** Gets the value of "list-style-position" */
  String get listStylePosition => getPropertyValue('list-style-position');

  /** Sets the value of "list-style-position" */
  set listStylePosition(String value) {
    setProperty('list-style-position', value, '');
  }

  /** Gets the value of "list-style-type" */
  String get listStyleType => getPropertyValue('list-style-type');

  /** Sets the value of "list-style-type" */
  set listStyleType(String value) {
    setProperty('list-style-type', value, '');
  }

  /** Gets the value of "locale" */
  String get locale => getPropertyValue('locale');

  /** Sets the value of "locale" */
  set locale(String value) {
    setProperty('locale', value, '');
  }

  /** Gets the value of "logical-height" */
  String get logicalHeight => getPropertyValue('logical-height');

  /** Sets the value of "logical-height" */
  set logicalHeight(String value) {
    setProperty('logical-height', value, '');
  }

  /** Gets the value of "logical-width" */
  String get logicalWidth => getPropertyValue('logical-width');

  /** Sets the value of "logical-width" */
  set logicalWidth(String value) {
    setProperty('logical-width', value, '');
  }

  /** Gets the value of "margin" */
  String get margin => getPropertyValue('margin');

  /** Sets the value of "margin" */
  set margin(String value) {
    setProperty('margin', value, '');
  }

  /** Gets the value of "margin-after" */
  String get marginAfter => getPropertyValue('margin-after');

  /** Sets the value of "margin-after" */
  set marginAfter(String value) {
    setProperty('margin-after', value, '');
  }

  /** Gets the value of "margin-after-collapse" */
  String get marginAfterCollapse => getPropertyValue('margin-after-collapse');

  /** Sets the value of "margin-after-collapse" */
  set marginAfterCollapse(String value) {
    setProperty('margin-after-collapse', value, '');
  }

  /** Gets the value of "margin-before" */
  String get marginBefore => getPropertyValue('margin-before');

  /** Sets the value of "margin-before" */
  set marginBefore(String value) {
    setProperty('margin-before', value, '');
  }

  /** Gets the value of "margin-before-collapse" */
  String get marginBeforeCollapse => getPropertyValue('margin-before-collapse');

  /** Sets the value of "margin-before-collapse" */
  set marginBeforeCollapse(String value) {
    setProperty('margin-before-collapse', value, '');
  }

  /** Gets the value of "margin-bottom" */
  String get marginBottom => getPropertyValue('margin-bottom');

  /** Sets the value of "margin-bottom" */
  set marginBottom(String value) {
    setProperty('margin-bottom', value, '');
  }

  /** Gets the value of "margin-bottom-collapse" */
  String get marginBottomCollapse => getPropertyValue('margin-bottom-collapse');

  /** Sets the value of "margin-bottom-collapse" */
  set marginBottomCollapse(String value) {
    setProperty('margin-bottom-collapse', value, '');
  }

  /** Gets the value of "margin-collapse" */
  String get marginCollapse => getPropertyValue('margin-collapse');

  /** Sets the value of "margin-collapse" */
  set marginCollapse(String value) {
    setProperty('margin-collapse', value, '');
  }

  /** Gets the value of "margin-end" */
  String get marginEnd => getPropertyValue('margin-end');

  /** Sets the value of "margin-end" */
  set marginEnd(String value) {
    setProperty('margin-end', value, '');
  }

  /** Gets the value of "margin-left" */
  String get marginLeft => getPropertyValue('margin-left');

  /** Sets the value of "margin-left" */
  set marginLeft(String value) {
    setProperty('margin-left', value, '');
  }

  /** Gets the value of "margin-right" */
  String get marginRight => getPropertyValue('margin-right');

  /** Sets the value of "margin-right" */
  set marginRight(String value) {
    setProperty('margin-right', value, '');
  }

  /** Gets the value of "margin-start" */
  String get marginStart => getPropertyValue('margin-start');

  /** Sets the value of "margin-start" */
  set marginStart(String value) {
    setProperty('margin-start', value, '');
  }

  /** Gets the value of "margin-top" */
  String get marginTop => getPropertyValue('margin-top');

  /** Sets the value of "margin-top" */
  set marginTop(String value) {
    setProperty('margin-top', value, '');
  }

  /** Gets the value of "margin-top-collapse" */
  String get marginTopCollapse => getPropertyValue('margin-top-collapse');

  /** Sets the value of "margin-top-collapse" */
  set marginTopCollapse(String value) {
    setProperty('margin-top-collapse', value, '');
  }

  /** Gets the value of "mask" */
  String get mask => getPropertyValue('mask');

  /** Sets the value of "mask" */
  set mask(String value) {
    setProperty('mask', value, '');
  }

  /** Gets the value of "mask-box-image" */
  String get maskBoxImage => getPropertyValue('mask-box-image');

  /** Sets the value of "mask-box-image" */
  set maskBoxImage(String value) {
    setProperty('mask-box-image', value, '');
  }

  /** Gets the value of "mask-box-image-outset" */
  String get maskBoxImageOutset => getPropertyValue('mask-box-image-outset');

  /** Sets the value of "mask-box-image-outset" */
  set maskBoxImageOutset(String value) {
    setProperty('mask-box-image-outset', value, '');
  }

  /** Gets the value of "mask-box-image-repeat" */
  String get maskBoxImageRepeat => getPropertyValue('mask-box-image-repeat');

  /** Sets the value of "mask-box-image-repeat" */
  set maskBoxImageRepeat(String value) {
    setProperty('mask-box-image-repeat', value, '');
  }

  /** Gets the value of "mask-box-image-slice" */
  String get maskBoxImageSlice => getPropertyValue('mask-box-image-slice');

  /** Sets the value of "mask-box-image-slice" */
  set maskBoxImageSlice(String value) {
    setProperty('mask-box-image-slice', value, '');
  }

  /** Gets the value of "mask-box-image-source" */
  String get maskBoxImageSource => getPropertyValue('mask-box-image-source');

  /** Sets the value of "mask-box-image-source" */
  set maskBoxImageSource(String value) {
    setProperty('mask-box-image-source', value, '');
  }

  /** Gets the value of "mask-box-image-width" */
  String get maskBoxImageWidth => getPropertyValue('mask-box-image-width');

  /** Sets the value of "mask-box-image-width" */
  set maskBoxImageWidth(String value) {
    setProperty('mask-box-image-width', value, '');
  }

  /** Gets the value of "mask-clip" */
  String get maskClip => getPropertyValue('mask-clip');

  /** Sets the value of "mask-clip" */
  set maskClip(String value) {
    setProperty('mask-clip', value, '');
  }

  /** Gets the value of "mask-composite" */
  String get maskComposite => getPropertyValue('mask-composite');

  /** Sets the value of "mask-composite" */
  set maskComposite(String value) {
    setProperty('mask-composite', value, '');
  }

  /** Gets the value of "mask-image" */
  String get maskImage => getPropertyValue('mask-image');

  /** Sets the value of "mask-image" */
  set maskImage(String value) {
    setProperty('mask-image', value, '');
  }

  /** Gets the value of "mask-origin" */
  String get maskOrigin => getPropertyValue('mask-origin');

  /** Sets the value of "mask-origin" */
  set maskOrigin(String value) {
    setProperty('mask-origin', value, '');
  }

  /** Gets the value of "mask-position" */
  String get maskPosition => getPropertyValue('mask-position');

  /** Sets the value of "mask-position" */
  set maskPosition(String value) {
    setProperty('mask-position', value, '');
  }

  /** Gets the value of "mask-position-x" */
  String get maskPositionX => getPropertyValue('mask-position-x');

  /** Sets the value of "mask-position-x" */
  set maskPositionX(String value) {
    setProperty('mask-position-x', value, '');
  }

  /** Gets the value of "mask-position-y" */
  String get maskPositionY => getPropertyValue('mask-position-y');

  /** Sets the value of "mask-position-y" */
  set maskPositionY(String value) {
    setProperty('mask-position-y', value, '');
  }

  /** Gets the value of "mask-repeat" */
  String get maskRepeat => getPropertyValue('mask-repeat');

  /** Sets the value of "mask-repeat" */
  set maskRepeat(String value) {
    setProperty('mask-repeat', value, '');
  }

  /** Gets the value of "mask-repeat-x" */
  String get maskRepeatX => getPropertyValue('mask-repeat-x');

  /** Sets the value of "mask-repeat-x" */
  set maskRepeatX(String value) {
    setProperty('mask-repeat-x', value, '');
  }

  /** Gets the value of "mask-repeat-y" */
  String get maskRepeatY => getPropertyValue('mask-repeat-y');

  /** Sets the value of "mask-repeat-y" */
  set maskRepeatY(String value) {
    setProperty('mask-repeat-y', value, '');
  }

  /** Gets the value of "mask-size" */
  String get maskSize => getPropertyValue('mask-size');

  /** Sets the value of "mask-size" */
  set maskSize(String value) {
    setProperty('mask-size', value, '');
  }

  /** Gets the value of "mask-source-type" */
  String get maskSourceType => getPropertyValue('mask-source-type');

  /** Sets the value of "mask-source-type" */
  set maskSourceType(String value) {
    setProperty('mask-source-type', value, '');
  }

  /** Gets the value of "max-height" */
  String get maxHeight => getPropertyValue('max-height');

  /** Sets the value of "max-height" */
  set maxHeight(String value) {
    setProperty('max-height', value, '');
  }

  /** Gets the value of "max-logical-height" */
  String get maxLogicalHeight => getPropertyValue('max-logical-height');

  /** Sets the value of "max-logical-height" */
  set maxLogicalHeight(String value) {
    setProperty('max-logical-height', value, '');
  }

  /** Gets the value of "max-logical-width" */
  String get maxLogicalWidth => getPropertyValue('max-logical-width');

  /** Sets the value of "max-logical-width" */
  set maxLogicalWidth(String value) {
    setProperty('max-logical-width', value, '');
  }

  /** Gets the value of "max-width" */
  String get maxWidth => getPropertyValue('max-width');

  /** Sets the value of "max-width" */
  set maxWidth(String value) {
    setProperty('max-width', value, '');
  }

  /** Gets the value of "max-zoom" */
  String get maxZoom => getPropertyValue('max-zoom');

  /** Sets the value of "max-zoom" */
  set maxZoom(String value) {
    setProperty('max-zoom', value, '');
  }

  /** Gets the value of "min-height" */
  String get minHeight => getPropertyValue('min-height');

  /** Sets the value of "min-height" */
  set minHeight(String value) {
    setProperty('min-height', value, '');
  }

  /** Gets the value of "min-logical-height" */
  String get minLogicalHeight => getPropertyValue('min-logical-height');

  /** Sets the value of "min-logical-height" */
  set minLogicalHeight(String value) {
    setProperty('min-logical-height', value, '');
  }

  /** Gets the value of "min-logical-width" */
  String get minLogicalWidth => getPropertyValue('min-logical-width');

  /** Sets the value of "min-logical-width" */
  set minLogicalWidth(String value) {
    setProperty('min-logical-width', value, '');
  }

  /** Gets the value of "min-width" */
  String get minWidth => getPropertyValue('min-width');

  /** Sets the value of "min-width" */
  set minWidth(String value) {
    setProperty('min-width', value, '');
  }

  /** Gets the value of "min-zoom" */
  String get minZoom => getPropertyValue('min-zoom');

  /** Sets the value of "min-zoom" */
  set minZoom(String value) {
    setProperty('min-zoom', value, '');
  }

  /** Gets the value of "mix-blend-mode" */
  String get mixBlendMode => getPropertyValue('mix-blend-mode');

  /** Sets the value of "mix-blend-mode" */
  set mixBlendMode(String value) {
    setProperty('mix-blend-mode', value, '');
  }

  /** Gets the value of "object-fit" */
  String get objectFit => getPropertyValue('object-fit');

  /** Sets the value of "object-fit" */
  set objectFit(String value) {
    setProperty('object-fit', value, '');
  }

  /** Gets the value of "object-position" */
  String get objectPosition => getPropertyValue('object-position');

  /** Sets the value of "object-position" */
  set objectPosition(String value) {
    setProperty('object-position', value, '');
  }

  /** Gets the value of "opacity" */
  String get opacity => getPropertyValue('opacity');

  /** Sets the value of "opacity" */
  set opacity(String value) {
    setProperty('opacity', value, '');
  }

  /** Gets the value of "order" */
  String get order => getPropertyValue('order');

  /** Sets the value of "order" */
  set order(String value) {
    setProperty('order', value, '');
  }

  /** Gets the value of "orientation" */
  String get orientation => getPropertyValue('orientation');

  /** Sets the value of "orientation" */
  set orientation(String value) {
    setProperty('orientation', value, '');
  }

  /** Gets the value of "orphans" */
  String get orphans => getPropertyValue('orphans');

  /** Sets the value of "orphans" */
  set orphans(String value) {
    setProperty('orphans', value, '');
  }

  /** Gets the value of "outline" */
  String get outline => getPropertyValue('outline');

  /** Sets the value of "outline" */
  set outline(String value) {
    setProperty('outline', value, '');
  }

  /** Gets the value of "outline-color" */
  String get outlineColor => getPropertyValue('outline-color');

  /** Sets the value of "outline-color" */
  set outlineColor(String value) {
    setProperty('outline-color', value, '');
  }

  /** Gets the value of "outline-offset" */
  String get outlineOffset => getPropertyValue('outline-offset');

  /** Sets the value of "outline-offset" */
  set outlineOffset(String value) {
    setProperty('outline-offset', value, '');
  }

  /** Gets the value of "outline-style" */
  String get outlineStyle => getPropertyValue('outline-style');

  /** Sets the value of "outline-style" */
  set outlineStyle(String value) {
    setProperty('outline-style', value, '');
  }

  /** Gets the value of "outline-width" */
  String get outlineWidth => getPropertyValue('outline-width');

  /** Sets the value of "outline-width" */
  set outlineWidth(String value) {
    setProperty('outline-width', value, '');
  }

  /** Gets the value of "overflow" */
  String get overflow => getPropertyValue('overflow');

  /** Sets the value of "overflow" */
  set overflow(String value) {
    setProperty('overflow', value, '');
  }

  /** Gets the value of "overflow-wrap" */
  String get overflowWrap => getPropertyValue('overflow-wrap');

  /** Sets the value of "overflow-wrap" */
  set overflowWrap(String value) {
    setProperty('overflow-wrap', value, '');
  }

  /** Gets the value of "overflow-x" */
  String get overflowX => getPropertyValue('overflow-x');

  /** Sets the value of "overflow-x" */
  set overflowX(String value) {
    setProperty('overflow-x', value, '');
  }

  /** Gets the value of "overflow-y" */
  String get overflowY => getPropertyValue('overflow-y');

  /** Sets the value of "overflow-y" */
  set overflowY(String value) {
    setProperty('overflow-y', value, '');
  }

  /** Gets the value of "padding" */
  String get padding => getPropertyValue('padding');

  /** Sets the value of "padding" */
  set padding(String value) {
    setProperty('padding', value, '');
  }

  /** Gets the value of "padding-after" */
  String get paddingAfter => getPropertyValue('padding-after');

  /** Sets the value of "padding-after" */
  set paddingAfter(String value) {
    setProperty('padding-after', value, '');
  }

  /** Gets the value of "padding-before" */
  String get paddingBefore => getPropertyValue('padding-before');

  /** Sets the value of "padding-before" */
  set paddingBefore(String value) {
    setProperty('padding-before', value, '');
  }

  /** Gets the value of "padding-bottom" */
  String get paddingBottom => getPropertyValue('padding-bottom');

  /** Sets the value of "padding-bottom" */
  set paddingBottom(String value) {
    setProperty('padding-bottom', value, '');
  }

  /** Gets the value of "padding-end" */
  String get paddingEnd => getPropertyValue('padding-end');

  /** Sets the value of "padding-end" */
  set paddingEnd(String value) {
    setProperty('padding-end', value, '');
  }

  /** Gets the value of "padding-left" */
  String get paddingLeft => getPropertyValue('padding-left');

  /** Sets the value of "padding-left" */
  set paddingLeft(String value) {
    setProperty('padding-left', value, '');
  }

  /** Gets the value of "padding-right" */
  String get paddingRight => getPropertyValue('padding-right');

  /** Sets the value of "padding-right" */
  set paddingRight(String value) {
    setProperty('padding-right', value, '');
  }

  /** Gets the value of "padding-start" */
  String get paddingStart => getPropertyValue('padding-start');

  /** Sets the value of "padding-start" */
  set paddingStart(String value) {
    setProperty('padding-start', value, '');
  }

  /** Gets the value of "padding-top" */
  String get paddingTop => getPropertyValue('padding-top');

  /** Sets the value of "padding-top" */
  set paddingTop(String value) {
    setProperty('padding-top', value, '');
  }

  /** Gets the value of "page" */
  String get page => getPropertyValue('page');

  /** Sets the value of "page" */
  set page(String value) {
    setProperty('page', value, '');
  }

  /** Gets the value of "page-break-after" */
  String get pageBreakAfter => getPropertyValue('page-break-after');

  /** Sets the value of "page-break-after" */
  set pageBreakAfter(String value) {
    setProperty('page-break-after', value, '');
  }

  /** Gets the value of "page-break-before" */
  String get pageBreakBefore => getPropertyValue('page-break-before');

  /** Sets the value of "page-break-before" */
  set pageBreakBefore(String value) {
    setProperty('page-break-before', value, '');
  }

  /** Gets the value of "page-break-inside" */
  String get pageBreakInside => getPropertyValue('page-break-inside');

  /** Sets the value of "page-break-inside" */
  set pageBreakInside(String value) {
    setProperty('page-break-inside', value, '');
  }

  /** Gets the value of "perspective" */
  String get perspective => getPropertyValue('perspective');

  /** Sets the value of "perspective" */
  set perspective(String value) {
    setProperty('perspective', value, '');
  }

  /** Gets the value of "perspective-origin" */
  String get perspectiveOrigin => getPropertyValue('perspective-origin');

  /** Sets the value of "perspective-origin" */
  set perspectiveOrigin(String value) {
    setProperty('perspective-origin', value, '');
  }

  /** Gets the value of "perspective-origin-x" */
  String get perspectiveOriginX => getPropertyValue('perspective-origin-x');

  /** Sets the value of "perspective-origin-x" */
  set perspectiveOriginX(String value) {
    setProperty('perspective-origin-x', value, '');
  }

  /** Gets the value of "perspective-origin-y" */
  String get perspectiveOriginY => getPropertyValue('perspective-origin-y');

  /** Sets the value of "perspective-origin-y" */
  set perspectiveOriginY(String value) {
    setProperty('perspective-origin-y', value, '');
  }

  /** Gets the value of "pointer-events" */
  String get pointerEvents => getPropertyValue('pointer-events');

  /** Sets the value of "pointer-events" */
  set pointerEvents(String value) {
    setProperty('pointer-events', value, '');
  }

  /** Gets the value of "position" */
  String get position => getPropertyValue('position');

  /** Sets the value of "position" */
  set position(String value) {
    setProperty('position', value, '');
  }

  /** Gets the value of "print-color-adjust" */
  String get printColorAdjust => getPropertyValue('print-color-adjust');

  /** Sets the value of "print-color-adjust" */
  set printColorAdjust(String value) {
    setProperty('print-color-adjust', value, '');
  }

  /** Gets the value of "quotes" */
  String get quotes => getPropertyValue('quotes');

  /** Sets the value of "quotes" */
  set quotes(String value) {
    setProperty('quotes', value, '');
  }

  /** Gets the value of "resize" */
  String get resize => getPropertyValue('resize');

  /** Sets the value of "resize" */
  set resize(String value) {
    setProperty('resize', value, '');
  }

  /** Gets the value of "right" */
  String get right => getPropertyValue('right');

  /** Sets the value of "right" */
  set right(String value) {
    setProperty('right', value, '');
  }

  /** Gets the value of "rtl-ordering" */
  String get rtlOrdering => getPropertyValue('rtl-ordering');

  /** Sets the value of "rtl-ordering" */
  set rtlOrdering(String value) {
    setProperty('rtl-ordering', value, '');
  }

  /** Gets the value of "ruby-position" */
  String get rubyPosition => getPropertyValue('ruby-position');

  /** Sets the value of "ruby-position" */
  set rubyPosition(String value) {
    setProperty('ruby-position', value, '');
  }

  /** Gets the value of "scroll-behavior" */
  String get scrollBehavior => getPropertyValue('scroll-behavior');

  /** Sets the value of "scroll-behavior" */
  set scrollBehavior(String value) {
    setProperty('scroll-behavior', value, '');
  }

  /** Gets the value of "shape-image-threshold" */
  String get shapeImageThreshold => getPropertyValue('shape-image-threshold');

  /** Sets the value of "shape-image-threshold" */
  set shapeImageThreshold(String value) {
    setProperty('shape-image-threshold', value, '');
  }

  /** Gets the value of "shape-margin" */
  String get shapeMargin => getPropertyValue('shape-margin');

  /** Sets the value of "shape-margin" */
  set shapeMargin(String value) {
    setProperty('shape-margin', value, '');
  }

  /** Gets the value of "shape-outside" */
  String get shapeOutside => getPropertyValue('shape-outside');

  /** Sets the value of "shape-outside" */
  set shapeOutside(String value) {
    setProperty('shape-outside', value, '');
  }

  /** Gets the value of "size" */
  String get size => getPropertyValue('size');

  /** Sets the value of "size" */
  set size(String value) {
    setProperty('size', value, '');
  }

  /** Gets the value of "speak" */
  String get speak => getPropertyValue('speak');

  /** Sets the value of "speak" */
  set speak(String value) {
    setProperty('speak', value, '');
  }

  /** Gets the value of "src" */
  String get src => getPropertyValue('src');

  /** Sets the value of "src" */
  set src(String value) {
    setProperty('src', value, '');
  }

  /** Gets the value of "tab-size" */
  String get tabSize => getPropertyValue('tab-size');

  /** Sets the value of "tab-size" */
  set tabSize(String value) {
    setProperty('tab-size', value, '');
  }

  /** Gets the value of "table-layout" */
  String get tableLayout => getPropertyValue('table-layout');

  /** Sets the value of "table-layout" */
  set tableLayout(String value) {
    setProperty('table-layout', value, '');
  }

  /** Gets the value of "tap-highlight-color" */
  String get tapHighlightColor => getPropertyValue('tap-highlight-color');

  /** Sets the value of "tap-highlight-color" */
  set tapHighlightColor(String value) {
    setProperty('tap-highlight-color', value, '');
  }

  /** Gets the value of "text-align" */
  String get textAlign => getPropertyValue('text-align');

  /** Sets the value of "text-align" */
  set textAlign(String value) {
    setProperty('text-align', value, '');
  }

  /** Gets the value of "text-align-last" */
  String get textAlignLast => getPropertyValue('text-align-last');

  /** Sets the value of "text-align-last" */
  set textAlignLast(String value) {
    setProperty('text-align-last', value, '');
  }

  /** Gets the value of "text-combine" */
  String get textCombine => getPropertyValue('text-combine');

  /** Sets the value of "text-combine" */
  set textCombine(String value) {
    setProperty('text-combine', value, '');
  }

  /** Gets the value of "text-decoration" */
  String get textDecoration => getPropertyValue('text-decoration');

  /** Sets the value of "text-decoration" */
  set textDecoration(String value) {
    setProperty('text-decoration', value, '');
  }

  /** Gets the value of "text-decoration-color" */
  String get textDecorationColor => getPropertyValue('text-decoration-color');

  /** Sets the value of "text-decoration-color" */
  set textDecorationColor(String value) {
    setProperty('text-decoration-color', value, '');
  }

  /** Gets the value of "text-decoration-line" */
  String get textDecorationLine => getPropertyValue('text-decoration-line');

  /** Sets the value of "text-decoration-line" */
  set textDecorationLine(String value) {
    setProperty('text-decoration-line', value, '');
  }

  /** Gets the value of "text-decoration-style" */
  String get textDecorationStyle => getPropertyValue('text-decoration-style');

  /** Sets the value of "text-decoration-style" */
  set textDecorationStyle(String value) {
    setProperty('text-decoration-style', value, '');
  }

  /** Gets the value of "text-decorations-in-effect" */
  String get textDecorationsInEffect =>
      getPropertyValue('text-decorations-in-effect');

  /** Sets the value of "text-decorations-in-effect" */
  set textDecorationsInEffect(String value) {
    setProperty('text-decorations-in-effect', value, '');
  }

  /** Gets the value of "text-emphasis" */
  String get textEmphasis => getPropertyValue('text-emphasis');

  /** Sets the value of "text-emphasis" */
  set textEmphasis(String value) {
    setProperty('text-emphasis', value, '');
  }

  /** Gets the value of "text-emphasis-color" */
  String get textEmphasisColor => getPropertyValue('text-emphasis-color');

  /** Sets the value of "text-emphasis-color" */
  set textEmphasisColor(String value) {
    setProperty('text-emphasis-color', value, '');
  }

  /** Gets the value of "text-emphasis-position" */
  String get textEmphasisPosition => getPropertyValue('text-emphasis-position');

  /** Sets the value of "text-emphasis-position" */
  set textEmphasisPosition(String value) {
    setProperty('text-emphasis-position', value, '');
  }

  /** Gets the value of "text-emphasis-style" */
  String get textEmphasisStyle => getPropertyValue('text-emphasis-style');

  /** Sets the value of "text-emphasis-style" */
  set textEmphasisStyle(String value) {
    setProperty('text-emphasis-style', value, '');
  }

  /** Gets the value of "text-fill-color" */
  String get textFillColor => getPropertyValue('text-fill-color');

  /** Sets the value of "text-fill-color" */
  set textFillColor(String value) {
    setProperty('text-fill-color', value, '');
  }

  /** Gets the value of "text-indent" */
  String get textIndent => getPropertyValue('text-indent');

  /** Sets the value of "text-indent" */
  set textIndent(String value) {
    setProperty('text-indent', value, '');
  }

  /** Gets the value of "text-justify" */
  String get textJustify => getPropertyValue('text-justify');

  /** Sets the value of "text-justify" */
  set textJustify(String value) {
    setProperty('text-justify', value, '');
  }

  /** Gets the value of "text-line-through-color" */
  String get textLineThroughColor =>
      getPropertyValue('text-line-through-color');

  /** Sets the value of "text-line-through-color" */
  set textLineThroughColor(String value) {
    setProperty('text-line-through-color', value, '');
  }

  /** Gets the value of "text-line-through-mode" */
  String get textLineThroughMode => getPropertyValue('text-line-through-mode');

  /** Sets the value of "text-line-through-mode" */
  set textLineThroughMode(String value) {
    setProperty('text-line-through-mode', value, '');
  }

  /** Gets the value of "text-line-through-style" */
  String get textLineThroughStyle =>
      getPropertyValue('text-line-through-style');

  /** Sets the value of "text-line-through-style" */
  set textLineThroughStyle(String value) {
    setProperty('text-line-through-style', value, '');
  }

  /** Gets the value of "text-line-through-width" */
  String get textLineThroughWidth =>
      getPropertyValue('text-line-through-width');

  /** Sets the value of "text-line-through-width" */
  set textLineThroughWidth(String value) {
    setProperty('text-line-through-width', value, '');
  }

  /** Gets the value of "text-orientation" */
  String get textOrientation => getPropertyValue('text-orientation');

  /** Sets the value of "text-orientation" */
  set textOrientation(String value) {
    setProperty('text-orientation', value, '');
  }

  /** Gets the value of "text-overflow" */
  String get textOverflow => getPropertyValue('text-overflow');

  /** Sets the value of "text-overflow" */
  set textOverflow(String value) {
    setProperty('text-overflow', value, '');
  }

  /** Gets the value of "text-overline-color" */
  String get textOverlineColor => getPropertyValue('text-overline-color');

  /** Sets the value of "text-overline-color" */
  set textOverlineColor(String value) {
    setProperty('text-overline-color', value, '');
  }

  /** Gets the value of "text-overline-mode" */
  String get textOverlineMode => getPropertyValue('text-overline-mode');

  /** Sets the value of "text-overline-mode" */
  set textOverlineMode(String value) {
    setProperty('text-overline-mode', value, '');
  }

  /** Gets the value of "text-overline-style" */
  String get textOverlineStyle => getPropertyValue('text-overline-style');

  /** Sets the value of "text-overline-style" */
  set textOverlineStyle(String value) {
    setProperty('text-overline-style', value, '');
  }

  /** Gets the value of "text-overline-width" */
  String get textOverlineWidth => getPropertyValue('text-overline-width');

  /** Sets the value of "text-overline-width" */
  set textOverlineWidth(String value) {
    setProperty('text-overline-width', value, '');
  }

  /** Gets the value of "text-rendering" */
  String get textRendering => getPropertyValue('text-rendering');

  /** Sets the value of "text-rendering" */
  set textRendering(String value) {
    setProperty('text-rendering', value, '');
  }

  /** Gets the value of "text-security" */
  String get textSecurity => getPropertyValue('text-security');

  /** Sets the value of "text-security" */
  set textSecurity(String value) {
    setProperty('text-security', value, '');
  }

  /** Gets the value of "text-shadow" */
  String get textShadow => getPropertyValue('text-shadow');

  /** Sets the value of "text-shadow" */
  set textShadow(String value) {
    setProperty('text-shadow', value, '');
  }

  /** Gets the value of "text-stroke" */
  String get textStroke => getPropertyValue('text-stroke');

  /** Sets the value of "text-stroke" */
  set textStroke(String value) {
    setProperty('text-stroke', value, '');
  }

  /** Gets the value of "text-stroke-color" */
  String get textStrokeColor => getPropertyValue('text-stroke-color');

  /** Sets the value of "text-stroke-color" */
  set textStrokeColor(String value) {
    setProperty('text-stroke-color', value, '');
  }

  /** Gets the value of "text-stroke-width" */
  String get textStrokeWidth => getPropertyValue('text-stroke-width');

  /** Sets the value of "text-stroke-width" */
  set textStrokeWidth(String value) {
    setProperty('text-stroke-width', value, '');
  }

  /** Gets the value of "text-transform" */
  String get textTransform => getPropertyValue('text-transform');

  /** Sets the value of "text-transform" */
  set textTransform(String value) {
    setProperty('text-transform', value, '');
  }

  /** Gets the value of "text-underline-color" */
  String get textUnderlineColor => getPropertyValue('text-underline-color');

  /** Sets the value of "text-underline-color" */
  set textUnderlineColor(String value) {
    setProperty('text-underline-color', value, '');
  }

  /** Gets the value of "text-underline-mode" */
  String get textUnderlineMode => getPropertyValue('text-underline-mode');

  /** Sets the value of "text-underline-mode" */
  set textUnderlineMode(String value) {
    setProperty('text-underline-mode', value, '');
  }

  /** Gets the value of "text-underline-position" */
  String get textUnderlinePosition =>
      getPropertyValue('text-underline-position');

  /** Sets the value of "text-underline-position" */
  set textUnderlinePosition(String value) {
    setProperty('text-underline-position', value, '');
  }

  /** Gets the value of "text-underline-style" */
  String get textUnderlineStyle => getPropertyValue('text-underline-style');

  /** Sets the value of "text-underline-style" */
  set textUnderlineStyle(String value) {
    setProperty('text-underline-style', value, '');
  }

  /** Gets the value of "text-underline-width" */
  String get textUnderlineWidth => getPropertyValue('text-underline-width');

  /** Sets the value of "text-underline-width" */
  set textUnderlineWidth(String value) {
    setProperty('text-underline-width', value, '');
  }

  /** Gets the value of "top" */
  String get top => getPropertyValue('top');

  /** Sets the value of "top" */
  set top(String value) {
    setProperty('top', value, '');
  }

  /** Gets the value of "touch-action" */
  String get touchAction => getPropertyValue('touch-action');

  /** Sets the value of "touch-action" */
  set touchAction(String value) {
    setProperty('touch-action', value, '');
  }

  /** Gets the value of "touch-action-delay" */
  String get touchActionDelay => getPropertyValue('touch-action-delay');

  /** Sets the value of "touch-action-delay" */
  set touchActionDelay(String value) {
    setProperty('touch-action-delay', value, '');
  }

  /** Gets the value of "transform" */
  String get transform => getPropertyValue('transform');

  /** Sets the value of "transform" */
  set transform(String value) {
    setProperty('transform', value, '');
  }

  /** Gets the value of "transform-origin" */
  String get transformOrigin => getPropertyValue('transform-origin');

  /** Sets the value of "transform-origin" */
  set transformOrigin(String value) {
    setProperty('transform-origin', value, '');
  }

  /** Gets the value of "transform-origin-x" */
  String get transformOriginX => getPropertyValue('transform-origin-x');

  /** Sets the value of "transform-origin-x" */
  set transformOriginX(String value) {
    setProperty('transform-origin-x', value, '');
  }

  /** Gets the value of "transform-origin-y" */
  String get transformOriginY => getPropertyValue('transform-origin-y');

  /** Sets the value of "transform-origin-y" */
  set transformOriginY(String value) {
    setProperty('transform-origin-y', value, '');
  }

  /** Gets the value of "transform-origin-z" */
  String get transformOriginZ => getPropertyValue('transform-origin-z');

  /** Sets the value of "transform-origin-z" */
  set transformOriginZ(String value) {
    setProperty('transform-origin-z', value, '');
  }

  /** Gets the value of "transform-style" */
  String get transformStyle => getPropertyValue('transform-style');

  /** Sets the value of "transform-style" */
  set transformStyle(String value) {
    setProperty('transform-style', value, '');
  }

  /** Gets the value of "transition" */ @SupportedBrowser(
      SupportedBrowser.CHROME)
  @SupportedBrowser(SupportedBrowser.FIREFOX)
  @SupportedBrowser(SupportedBrowser.IE, '10')
  @SupportedBrowser(SupportedBrowser.SAFARI)
  String get transition => getPropertyValue('transition');

  /** Sets the value of "transition" */ @SupportedBrowser(
      SupportedBrowser.CHROME)
  @SupportedBrowser(SupportedBrowser.FIREFOX)
  @SupportedBrowser(SupportedBrowser.IE, '10')
  @SupportedBrowser(SupportedBrowser.SAFARI)
  set transition(String value) {
    setProperty('transition', value, '');
  }

  /** Gets the value of "transition-delay" */
  String get transitionDelay => getPropertyValue('transition-delay');

  /** Sets the value of "transition-delay" */
  set transitionDelay(String value) {
    setProperty('transition-delay', value, '');
  }

  /** Gets the value of "transition-duration" */
  String get transitionDuration => getPropertyValue('transition-duration');

  /** Sets the value of "transition-duration" */
  set transitionDuration(String value) {
    setProperty('transition-duration', value, '');
  }

  /** Gets the value of "transition-property" */
  String get transitionProperty => getPropertyValue('transition-property');

  /** Sets the value of "transition-property" */
  set transitionProperty(String value) {
    setProperty('transition-property', value, '');
  }

  /** Gets the value of "transition-timing-function" */
  String get transitionTimingFunction =>
      getPropertyValue('transition-timing-function');

  /** Sets the value of "transition-timing-function" */
  set transitionTimingFunction(String value) {
    setProperty('transition-timing-function', value, '');
  }

  /** Gets the value of "unicode-bidi" */
  String get unicodeBidi => getPropertyValue('unicode-bidi');

  /** Sets the value of "unicode-bidi" */
  set unicodeBidi(String value) {
    setProperty('unicode-bidi', value, '');
  }

  /** Gets the value of "unicode-range" */
  String get unicodeRange => getPropertyValue('unicode-range');

  /** Sets the value of "unicode-range" */
  set unicodeRange(String value) {
    setProperty('unicode-range', value, '');
  }

  /** Gets the value of "user-drag" */
  String get userDrag => getPropertyValue('user-drag');

  /** Sets the value of "user-drag" */
  set userDrag(String value) {
    setProperty('user-drag', value, '');
  }

  /** Gets the value of "user-modify" */
  String get userModify => getPropertyValue('user-modify');

  /** Sets the value of "user-modify" */
  set userModify(String value) {
    setProperty('user-modify', value, '');
  }

  /** Gets the value of "user-select" */
  String get userSelect => getPropertyValue('user-select');

  /** Sets the value of "user-select" */
  set userSelect(String value) {
    setProperty('user-select', value, '');
  }

  /** Gets the value of "user-zoom" */
  String get userZoom => getPropertyValue('user-zoom');

  /** Sets the value of "user-zoom" */
  set userZoom(String value) {
    setProperty('user-zoom', value, '');
  }

  /** Gets the value of "vertical-align" */
  String get verticalAlign => getPropertyValue('vertical-align');

  /** Sets the value of "vertical-align" */
  set verticalAlign(String value) {
    setProperty('vertical-align', value, '');
  }

  /** Gets the value of "visibility" */
  String get visibility => getPropertyValue('visibility');

  /** Sets the value of "visibility" */
  set visibility(String value) {
    setProperty('visibility', value, '');
  }

  /** Gets the value of "white-space" */
  String get whiteSpace => getPropertyValue('white-space');

  /** Sets the value of "white-space" */
  set whiteSpace(String value) {
    setProperty('white-space', value, '');
  }

  /** Gets the value of "widows" */
  String get widows => getPropertyValue('widows');

  /** Sets the value of "widows" */
  set widows(String value) {
    setProperty('widows', value, '');
  }

  /** Gets the value of "width" */
  String get width => getPropertyValue('width');

  /** Sets the value of "width" */
  set width(String value) {
    setProperty('width', value, '');
  }

  /** Gets the value of "will-change" */
  String get willChange => getPropertyValue('will-change');

  /** Sets the value of "will-change" */
  set willChange(String value) {
    setProperty('will-change', value, '');
  }

  /** Gets the value of "word-break" */
  String get wordBreak => getPropertyValue('word-break');

  /** Sets the value of "word-break" */
  set wordBreak(String value) {
    setProperty('word-break', value, '');
  }

  /** Gets the value of "word-spacing" */
  String get wordSpacing => getPropertyValue('word-spacing');

  /** Sets the value of "word-spacing" */
  set wordSpacing(String value) {
    setProperty('word-spacing', value, '');
  }

  /** Gets the value of "word-wrap" */
  String get wordWrap => getPropertyValue('word-wrap');

  /** Sets the value of "word-wrap" */
  set wordWrap(String value) {
    setProperty('word-wrap', value, '');
  }

  /** Gets the value of "wrap-flow" */
  String get wrapFlow => getPropertyValue('wrap-flow');

  /** Sets the value of "wrap-flow" */
  set wrapFlow(String value) {
    setProperty('wrap-flow', value, '');
  }

  /** Gets the value of "wrap-through" */
  String get wrapThrough => getPropertyValue('wrap-through');

  /** Sets the value of "wrap-through" */
  set wrapThrough(String value) {
    setProperty('wrap-through', value, '');
  }

  /** Gets the value of "writing-mode" */
  String get writingMode => getPropertyValue('writing-mode');

  /** Sets the value of "writing-mode" */
  set writingMode(String value) {
    setProperty('writing-mode', value, '');
  }

  /** Gets the value of "z-index" */
  String get zIndex => getPropertyValue('z-index');

  /** Sets the value of "z-index" */
  set zIndex(String value) {
    setProperty('z-index', value, '');
  }

  /** Gets the value of "zoom" */
  String get zoom => getPropertyValue('zoom');

  /** Sets the value of "zoom" */
  set zoom(String value) {
    setProperty('zoom', value, '');
  }
}
// Copyright (c) 2012, the Dart project authors.  Please see the AUTHORS file
// for details. All rights reserved. Use of this source code is governed by a
// BSD-style license that can be found in the LICENSE file.

// WARNING: Do not edit - generated code.

@DocsEditable()
@DomName('CSSStyleRule')
class CssStyleRule extends CssRule {
  // To suppress missing implicit constructor warnings.
  factory CssStyleRule._() {
    throw new UnsupportedError("Not supported");
  }

  @Deprecated("Internal Use Only")
  external static Type get instanceRuntimeType;

  @Deprecated("Internal Use Only")
  CssStyleRule.internal_() : super.internal_();

  @DomName('CSSStyleRule.selectorText')
  @DocsEditable()
  String get selectorText =>
      _blink.BlinkCSSStyleRule.instance.selectorText_Getter_(this);

  @DomName('CSSStyleRule.selectorText')
  @DocsEditable()
  set selectorText(String value) =>
      _blink.BlinkCSSStyleRule.instance.selectorText_Setter_(this, value);

  @DomName('CSSStyleRule.style')
  @DocsEditable()
  CssStyleDeclaration get style =>
      _blink.BlinkCSSStyleRule.instance.style_Getter_(this);
}
// Copyright (c) 2012, the Dart project authors.  Please see the AUTHORS file
// for details. All rights reserved. Use of this source code is governed by a
// BSD-style license that can be found in the LICENSE file.

// WARNING: Do not edit - generated code.

@DocsEditable()
@DomName('CSSStyleSheet')
class CssStyleSheet extends StyleSheet {
  // To suppress missing implicit constructor warnings.
  factory CssStyleSheet._() {
    throw new UnsupportedError("Not supported");
  }

  @Deprecated("Internal Use Only")
  external static Type get instanceRuntimeType;

  @Deprecated("Internal Use Only")
  CssStyleSheet.internal_() : super.internal_();

  @DomName('CSSStyleSheet.cssRules')
  @DocsEditable()
  List<CssRule> get cssRules =>
      _blink.BlinkCSSStyleSheet.instance.cssRules_Getter_(this);

  @DomName('CSSStyleSheet.ownerRule')
  @DocsEditable()
  CssRule get ownerRule =>
      _blink.BlinkCSSStyleSheet.instance.ownerRule_Getter_(this);

  @DomName('CSSStyleSheet.rules')
  @DocsEditable()
  @Experimental() // non-standard
  List<CssRule> get rules =>
      _blink.BlinkCSSStyleSheet.instance.rules_Getter_(this);

  int addRule(String selector, String style, [int index]) {
    if (index != null) {
      return _blink.BlinkCSSStyleSheet.instance
          .addRule_Callback_3_(this, selector, style, index);
    }
    return _blink.BlinkCSSStyleSheet.instance
        .addRule_Callback_2_(this, selector, style);
  }

  @DomName('CSSStyleSheet.deleteRule')
  @DocsEditable()
  void deleteRule(int index) =>
      _blink.BlinkCSSStyleSheet.instance.deleteRule_Callback_1_(this, index);

  int insertRule(String rule, [int index]) {
    if (index != null) {
      return _blink.BlinkCSSStyleSheet.instance
          .insertRule_Callback_2_(this, rule, index);
    }
    return _blink.BlinkCSSStyleSheet.instance
        .insertRule_Callback_1_(this, rule);
  }

  @DomName('CSSStyleSheet.removeRule')
  @DocsEditable()
  @Experimental() // non-standard
  void removeRule(int index) =>
      _blink.BlinkCSSStyleSheet.instance.removeRule_Callback_1_(this, index);
}
// Copyright (c) 2012, the Dart project authors.  Please see the AUTHORS file
// for details. All rights reserved. Use of this source code is governed by a
// BSD-style license that can be found in the LICENSE file.

// WARNING: Do not edit - generated code.

@DocsEditable()
@DomName('CSSSupportsRule')
class CssSupportsRule extends CssRule {
  // To suppress missing implicit constructor warnings.
  factory CssSupportsRule._() {
    throw new UnsupportedError("Not supported");
  }

  @Deprecated("Internal Use Only")
  external static Type get instanceRuntimeType;

  @Deprecated("Internal Use Only")
  CssSupportsRule.internal_() : super.internal_();

  @DomName('CSSSupportsRule.conditionText')
  @DocsEditable()
  String get conditionText =>
      _blink.BlinkCSSSupportsRule.instance.conditionText_Getter_(this);

  @DomName('CSSSupportsRule.cssRules')
  @DocsEditable()
  List<CssRule> get cssRules =>
      _blink.BlinkCSSSupportsRule.instance.cssRules_Getter_(this);

  @DomName('CSSSupportsRule.deleteRule')
  @DocsEditable()
  void deleteRule(int index) =>
      _blink.BlinkCSSSupportsRule.instance.deleteRule_Callback_1_(this, index);

  @DomName('CSSSupportsRule.insertRule')
  @DocsEditable()
  int insertRule(String rule, int index) => _blink.BlinkCSSSupportsRule.instance
      .insertRule_Callback_2_(this, rule, index);
}
// Copyright (c) 2012, the Dart project authors.  Please see the AUTHORS file
// for details. All rights reserved. Use of this source code is governed by a
// BSD-style license that can be found in the LICENSE file.

// WARNING: Do not edit - generated code.

@DocsEditable()
@DomName('CSSViewportRule')
@Experimental() // untriaged
class CssViewportRule extends CssRule {
  // To suppress missing implicit constructor warnings.
  factory CssViewportRule._() {
    throw new UnsupportedError("Not supported");
  }

  @Deprecated("Internal Use Only")
  external static Type get instanceRuntimeType;

  @Deprecated("Internal Use Only")
  CssViewportRule.internal_() : super.internal_();

  @DomName('CSSViewportRule.style')
  @DocsEditable()
  @Experimental() // untriaged
  CssStyleDeclaration get style =>
      _blink.BlinkCSSViewportRule.instance.style_Getter_(this);
}
// Copyright (c) 2012, the Dart project authors.  Please see the AUTHORS file
// for details. All rights reserved. Use of this source code is governed by a
// BSD-style license that can be found in the LICENSE file.

// WARNING: Do not edit - generated code.

@DomName('CustomEvent')
class CustomEvent extends Event {
  var _dartDetail;

  factory CustomEvent(String type,
      {bool canBubble: true, bool cancelable: true, Object detail}) {
    final CustomEvent e = document._createEvent('CustomEvent');

    e._dartDetail = detail;

    // Only try setting the detail if it's one of these types to avoid
    // first-chance exceptions. Can expand this list in the future as needed.
    if (detail is List || detail is Map || detail is String || detail is num) {
      try {
        detail = convertDartToNative_SerializedScriptValue(detail);
        e._initCustomEvent(type, canBubble, cancelable, detail);
      } catch (_) {
        e._initCustomEvent(type, canBubble, cancelable, null);
      }
    } else {
      e._initCustomEvent(type, canBubble, cancelable, null);
    }

    return e;
  }

  @DomName('CustomEvent.detail')
  get detail {
    if (_dartDetail != null) {
      return _dartDetail;
    }
    return _detail;
  }

  @DomName('CustomEvent.CustomEvent')
  @DocsEditable()
  factory CustomEvent._(String type, [Map eventInitDict]) {
    if (eventInitDict != null) {
      var eventInitDict_1 = convertDartToNative_Dictionary(eventInitDict);
      return _blink.BlinkCustomEvent.instance
          .constructorCallback_2_(type, eventInitDict_1);
    }
    return _blink.BlinkCustomEvent.instance.constructorCallback_1_(type);
  }

  @Deprecated("Internal Use Only")
  external static Type get instanceRuntimeType;

  @Deprecated("Internal Use Only")
  CustomEvent.internal_() : super.internal_();

  @DomName('CustomEvent.detail')
  @DocsEditable()
  Object get _detail => convertNativeToDart_SerializedScriptValue(
      _blink.BlinkCustomEvent.instance.detail_Getter_(this));

  @DomName('CustomEvent.initCustomEvent')
  @DocsEditable()
  void _initCustomEvent(
          String type, bool bubbles, bool cancelable, Object detail) =>
      _blink.BlinkCustomEvent.instance
          .initCustomEvent_Callback_4_(this, type, bubbles, cancelable, detail);
}
// Copyright (c) 2012, the Dart project authors.  Please see the AUTHORS file
// for details. All rights reserved. Use of this source code is governed by a
// BSD-style license that can be found in the LICENSE file.

// WARNING: Do not edit - generated code.

@DocsEditable()
@DomName('HTMLDListElement')
class DListElement extends HtmlElement {
  // To suppress missing implicit constructor warnings.
  factory DListElement._() {
    throw new UnsupportedError("Not supported");
  }

  @DomName('HTMLDListElement.HTMLDListElement')
  @DocsEditable()
  factory DListElement() => document.createElement("dl");

  @Deprecated("Internal Use Only")
  external static Type get instanceRuntimeType;

  @Deprecated("Internal Use Only")
  DListElement.internal_() : super.internal_();

  /**
   * Constructor instantiated by the DOM when a custom element has been created.
   *
   * This can only be called by subclasses from their created constructor.
   */
  DListElement.created() : super.created();
}
// Copyright (c) 2012, the Dart project authors.  Please see the AUTHORS file
// for details. All rights reserved. Use of this source code is governed by a
// BSD-style license that can be found in the LICENSE file.

// WARNING: Do not edit - generated code.

@DocsEditable()
@DomName('HTMLDataListElement')
@SupportedBrowser(SupportedBrowser.CHROME)
@SupportedBrowser(SupportedBrowser.FIREFOX)
@SupportedBrowser(SupportedBrowser.IE, '10')
@SupportedBrowser(SupportedBrowser.SAFARI)
class DataListElement extends HtmlElement {
  // To suppress missing implicit constructor warnings.
  factory DataListElement._() {
    throw new UnsupportedError("Not supported");
  }

  @DomName('HTMLDataListElement.HTMLDataListElement')
  @DocsEditable()
  factory DataListElement() => document.createElement("datalist");

  @Deprecated("Internal Use Only")
  external static Type get instanceRuntimeType;

  @Deprecated("Internal Use Only")
  DataListElement.internal_() : super.internal_();

  /**
   * Constructor instantiated by the DOM when a custom element has been created.
   *
   * This can only be called by subclasses from their created constructor.
   */
  DataListElement.created() : super.created();

  /// Checks if this type is supported on the current platform.
  static bool get supported => true;

  @DomName('HTMLDataListElement.options')
  @DocsEditable()
  List<Node> get options =>
      (_blink.BlinkHTMLDataListElement.instance.options_Getter_(this));
}
// Copyright (c) 2012, the Dart project authors.  Please see the AUTHORS file
// for details. All rights reserved. Use of this source code is governed by a
// BSD-style license that can be found in the LICENSE file.

// WARNING: Do not edit - generated code.

@DocsEditable()
@DomName('DataTransfer')
@Experimental() // untriaged
class DataTransfer extends DartHtmlDomObject {
  // To suppress missing implicit constructor warnings.
  factory DataTransfer._() {
    throw new UnsupportedError("Not supported");
  }

  @Deprecated("Internal Use Only")
  external static Type get instanceRuntimeType;

  @Deprecated("Internal Use Only")
  DataTransfer.internal_() {}

  @DomName('DataTransfer.dropEffect')
  @DocsEditable()
  @Experimental() // untriaged
  String get dropEffect =>
      _blink.BlinkDataTransfer.instance.dropEffect_Getter_(this);

  @DomName('DataTransfer.dropEffect')
  @DocsEditable()
  @Experimental() // untriaged
  set dropEffect(String value) =>
      _blink.BlinkDataTransfer.instance.dropEffect_Setter_(this, value);

  @DomName('DataTransfer.effectAllowed')
  @DocsEditable()
  @Experimental() // untriaged
  String get effectAllowed =>
      _blink.BlinkDataTransfer.instance.effectAllowed_Getter_(this);

  @DomName('DataTransfer.effectAllowed')
  @DocsEditable()
  @Experimental() // untriaged
  set effectAllowed(String value) =>
      _blink.BlinkDataTransfer.instance.effectAllowed_Setter_(this, value);

  @DomName('DataTransfer.files')
  @DocsEditable()
  @Experimental() // untriaged
  List<File> get files =>
      (_blink.BlinkDataTransfer.instance.files_Getter_(this));

  @DomName('DataTransfer.items')
  @DocsEditable()
  @Experimental() // untriaged
  DataTransferItemList get items =>
      _blink.BlinkDataTransfer.instance.items_Getter_(this);

  @DomName('DataTransfer.types')
  @DocsEditable()
  @Experimental() // untriaged
  List<String> get types =>
      _blink.BlinkDataTransfer.instance.types_Getter_(this);

  void clearData([String format]) {
    if (format != null) {
      _blink.BlinkDataTransfer.instance.clearData_Callback_1_(this, format);
      return;
    }
    _blink.BlinkDataTransfer.instance.clearData_Callback_0_(this);
    return;
  }

  @DomName('DataTransfer.getData')
  @DocsEditable()
  @Experimental() // untriaged
  String getData(String format) =>
      _blink.BlinkDataTransfer.instance.getData_Callback_1_(this, format);

  @DomName('DataTransfer.setData')
  @DocsEditable()
  @Experimental() // untriaged
  void setData(String format, String data) =>
      _blink.BlinkDataTransfer.instance.setData_Callback_2_(this, format, data);

  @DomName('DataTransfer.setDragImage')
  @DocsEditable()
  @Experimental() // untriaged
  void setDragImage(Element image, int x, int y) =>
      _blink.BlinkDataTransfer.instance
          .setDragImage_Callback_3_(this, image, x, y);
}
// Copyright (c) 2012, the Dart project authors.  Please see the AUTHORS file
// for details. All rights reserved. Use of this source code is governed by a
// BSD-style license that can be found in the LICENSE file.

// WARNING: Do not edit - generated code.

@DocsEditable()
@DomName('DataTransferItem')
// http://www.w3.org/TR/2011/WD-html5-20110113/dnd.html#the-datatransferitem-interface
@Experimental()
class DataTransferItem extends DartHtmlDomObject {
  // To suppress missing implicit constructor warnings.
  factory DataTransferItem._() {
    throw new UnsupportedError("Not supported");
  }

  @Deprecated("Internal Use Only")
  external static Type get instanceRuntimeType;

  @Deprecated("Internal Use Only")
  DataTransferItem.internal_() {}

  @DomName('DataTransferItem.kind')
  @DocsEditable()
  String get kind => _blink.BlinkDataTransferItem.instance.kind_Getter_(this);

  @DomName('DataTransferItem.type')
  @DocsEditable()
  String get type => _blink.BlinkDataTransferItem.instance.type_Getter_(this);

  @DomName('DataTransferItem.getAsFile')
  @DocsEditable()
  Blob getAsFile() =>
      _blink.BlinkDataTransferItem.instance.getAsFile_Callback_0_(this);

  @DomName('DataTransferItem.getAsString')
  @DocsEditable()
  void _getAsString(_StringCallback callback) =>
      _blink.BlinkDataTransferItem.instance
          .getAsString_Callback_1_(this, callback);

  Future<String> getAsString() {
    var completer = new Completer<String>();
    _getAsString((value) {
      completer.complete(value);
    });
    return completer.future;
  }

  @DomName('DataTransferItem.webkitGetAsEntry')
  @DocsEditable()
  @SupportedBrowser(SupportedBrowser.CHROME)
  @SupportedBrowser(SupportedBrowser.SAFARI)
  @Experimental()
  Entry getAsEntry() =>
      _blink.BlinkDataTransferItem.instance.webkitGetAsEntry_Callback_0_(this);
}
// Copyright (c) 2013, the Dart project authors.  Please see the AUTHORS file
// for details. All rights reserved. Use of this source code is governed by a
// BSD-style license that can be found in the LICENSE file.

@DocsEditable()
@DomName('DataTransferItemList')
// http://www.whatwg.org/specs/web-apps/current-work/multipage/dnd.html#the-datatransferitemlist-interface
@Experimental()
class DataTransferItemList extends DartHtmlDomObject {
  // To suppress missing implicit constructor warnings.
  factory DataTransferItemList._() {
    throw new UnsupportedError("Not supported");
  }

  @Deprecated("Internal Use Only")
  external static Type get instanceRuntimeType;

  @Deprecated("Internal Use Only")
  DataTransferItemList.internal_() {}

  @DomName('DataTransferItemList.length')
  @DocsEditable()
  int get length =>
      _blink.BlinkDataTransferItemList.instance.length_Getter_(this);

  DataTransferItem add(data_OR_file, [String type]) {
    if ((type is String) && (data_OR_file is String)) {
      return _blink.BlinkDataTransferItemList.instance
          .add_Callback_2_(this, data_OR_file, type);
    }
    if ((data_OR_file is File || data_OR_file == null) && type == null) {
      return _blink.BlinkDataTransferItemList.instance
          .add_Callback_1_(this, data_OR_file);
    }
    throw new ArgumentError("Incorrect number or type of arguments");
  }

  @DomName('DataTransferItemList.addData')
  @DocsEditable()
  DataTransferItem addData(String data, String type) =>
      _blink.BlinkDataTransferItemList.instance
          .add_Callback_2_(this, data, type);

  @DomName('DataTransferItemList.addFile')
  @DocsEditable()
  DataTransferItem addFile(File file) =>
      _blink.BlinkDataTransferItemList.instance.add_Callback_1_(this, file);

  @DomName('DataTransferItemList.clear')
  @DocsEditable()
  void clear() =>
      _blink.BlinkDataTransferItemList.instance.clear_Callback_0_(this);

  @DomName('DataTransferItemList.item')
  @DocsEditable()
  DataTransferItem item(int index) =>
      _blink.BlinkDataTransferItemList.instance.item_Callback_1_(this, index);

  @DomName('DataTransferItemList.remove')
  @DocsEditable()
  @Experimental() // untriaged
  void remove(int index) =>
      _blink.BlinkDataTransferItemList.instance.remove_Callback_1_(this, index);

  DataTransferItem operator [](int index) {
    // TODO(alanknight): I think that all the __getter__ generators should just
    // do property access, but that's major surgery. This one is a problem, so
    // just hard-code it for now.
    return _blink.Blink_JsNative_DomException
        .getProperty(this, index.toString());
  }
}
// Copyright (c) 2012, the Dart project authors.  Please see the AUTHORS file
// for details. All rights reserved. Use of this source code is governed by a
// BSD-style license that can be found in the LICENSE file.

// WARNING: Do not edit - generated code.

@DomName('DatabaseCallback')
// http://www.w3.org/TR/webdatabase/#databasecallback
@Experimental() // deprecated
typedef void DatabaseCallback(SqlDatabase database);
// Copyright (c) 2012, the Dart project authors.  Please see the AUTHORS file
// for details. All rights reserved. Use of this source code is governed by a
// BSD-style license that can be found in the LICENSE file.

// WARNING: Do not edit - generated code.

@DocsEditable()
@DomName('DedicatedWorkerGlobalScope')
@Experimental() // untriaged
class DedicatedWorkerGlobalScope extends WorkerGlobalScope {
  // To suppress missing implicit constructor warnings.
  factory DedicatedWorkerGlobalScope._() {
    throw new UnsupportedError("Not supported");
  }

  /**
   * Static factory designed to expose `message` events to event
   * handlers that are not necessarily instances of [DedicatedWorkerGlobalScope].
   *
   * See [EventStreamProvider] for usage information.
   */
  @DomName('DedicatedWorkerGlobalScope.messageEvent')
  @DocsEditable()
  @Experimental() // untriaged
  static const EventStreamProvider<MessageEvent> messageEvent =
      const EventStreamProvider<MessageEvent>('message');

  @Deprecated("Internal Use Only")
  external static Type get instanceRuntimeType;

  @Deprecated("Internal Use Only")
  DedicatedWorkerGlobalScope.internal_() : super.internal_();

  void postMessage(Object message, [List<MessagePort> transfer]) {
    if (transfer != null) {
      _blink.BlinkDedicatedWorkerGlobalScope.instance.postMessage_Callback_2_(
          this, convertDartToNative_SerializedScriptValue(message), transfer);
      return;
    }
    _blink.BlinkDedicatedWorkerGlobalScope.instance.postMessage_Callback_1_(
        this, convertDartToNative_SerializedScriptValue(message));
    return;
  }

  /// Stream of `message` events handled by this [DedicatedWorkerGlobalScope].
  @DomName('DedicatedWorkerGlobalScope.onmessage')
  @DocsEditable()
  @Experimental() // untriaged
  Stream<MessageEvent> get onMessage => messageEvent.forTarget(this);
}
// Copyright (c) 2012, the Dart project authors.  Please see the AUTHORS file
// for details. All rights reserved. Use of this source code is governed by a
// BSD-style license that can be found in the LICENSE file.

// WARNING: Do not edit - generated code.

@DocsEditable()
@DomName('DefaultSessionStartEvent')
@Experimental() // untriaged
class DefaultSessionStartEvent extends Event {
  // To suppress missing implicit constructor warnings.
  factory DefaultSessionStartEvent._() {
    throw new UnsupportedError("Not supported");
  }

  @DomName('DefaultSessionStartEvent.DefaultSessionStartEvent')
  @DocsEditable()
  factory DefaultSessionStartEvent(String type, [Map eventInitDict]) {
    if (eventInitDict != null) {
      var eventInitDict_1 = convertDartToNative_Dictionary(eventInitDict);
      return _blink.BlinkDefaultSessionStartEvent.instance
          .constructorCallback_2_(type, eventInitDict_1);
    }
    return _blink.BlinkDefaultSessionStartEvent.instance
        .constructorCallback_1_(type);
  }

  @Deprecated("Internal Use Only")
  external static Type get instanceRuntimeType;

  @Deprecated("Internal Use Only")
  DefaultSessionStartEvent.internal_() : super.internal_();

  @DomName('DefaultSessionStartEvent.session')
  @DocsEditable()
  @Experimental() // untriaged
  PresentationSession get session =>
      _blink.BlinkDefaultSessionStartEvent.instance.session_Getter_(this);
}
// Copyright (c) 2012, the Dart project authors.  Please see the AUTHORS file
// for details. All rights reserved. Use of this source code is governed by a
// BSD-style license that can be found in the LICENSE file.

// WARNING: Do not edit - generated code.

@DocsEditable()
@DomName('DeprecatedStorageInfo')
@Experimental() // untriaged
class DeprecatedStorageInfo extends DartHtmlDomObject {
  // To suppress missing implicit constructor warnings.
  factory DeprecatedStorageInfo._() {
    throw new UnsupportedError("Not supported");
  }

  @Deprecated("Internal Use Only")
  external static Type get instanceRuntimeType;

  @Deprecated("Internal Use Only")
  DeprecatedStorageInfo.internal_() {}

  @DomName('DeprecatedStorageInfo.PERSISTENT')
  @DocsEditable()
  @Experimental() // untriaged
  static const int PERSISTENT = 1;

  @DomName('DeprecatedStorageInfo.TEMPORARY')
  @DocsEditable()
  @Experimental() // untriaged
  static const int TEMPORARY = 0;

  void queryUsageAndQuota(int storageType,
      [StorageUsageCallback usageCallback,
      StorageErrorCallback errorCallback]) {
    if (errorCallback != null) {
      _blink.BlinkDeprecatedStorageInfo.instance.queryUsageAndQuota_Callback_3_(
          this, storageType, usageCallback, errorCallback);
      return;
    }
    if (usageCallback != null) {
      _blink.BlinkDeprecatedStorageInfo.instance
          .queryUsageAndQuota_Callback_2_(this, storageType, usageCallback);
      return;
    }
    _blink.BlinkDeprecatedStorageInfo.instance
        .queryUsageAndQuota_Callback_1_(this, storageType);
    return;
  }

  void requestQuota(int storageType, int newQuotaInBytes,
      [StorageQuotaCallback quotaCallback,
      StorageErrorCallback errorCallback]) {
    if (errorCallback != null) {
      _blink.BlinkDeprecatedStorageInfo.instance.requestQuota_Callback_4_(
          this, storageType, newQuotaInBytes, quotaCallback, errorCallback);
      return;
    }
    if (quotaCallback != null) {
      _blink.BlinkDeprecatedStorageInfo.instance.requestQuota_Callback_3_(
          this, storageType, newQuotaInBytes, quotaCallback);
      return;
    }
    _blink.BlinkDeprecatedStorageInfo.instance
        .requestQuota_Callback_2_(this, storageType, newQuotaInBytes);
    return;
  }
}
// Copyright (c) 2012, the Dart project authors.  Please see the AUTHORS file
// for details. All rights reserved. Use of this source code is governed by a
// BSD-style license that can be found in the LICENSE file.

// WARNING: Do not edit - generated code.

@DocsEditable()
@DomName('DeprecatedStorageQuota')
@Experimental() // untriaged
class DeprecatedStorageQuota extends DartHtmlDomObject {
  // To suppress missing implicit constructor warnings.
  factory DeprecatedStorageQuota._() {
    throw new UnsupportedError("Not supported");
  }

  @Deprecated("Internal Use Only")
  external static Type get instanceRuntimeType;

  @Deprecated("Internal Use Only")
  DeprecatedStorageQuota.internal_() {}

  void queryUsageAndQuota(StorageUsageCallback usageCallback,
      [StorageErrorCallback errorCallback]) {
    if (errorCallback != null) {
      _blink.BlinkDeprecatedStorageQuota.instance
          .queryUsageAndQuota_Callback_2_(this, usageCallback, errorCallback);
      return;
    }
    _blink.BlinkDeprecatedStorageQuota.instance
        .queryUsageAndQuota_Callback_1_(this, usageCallback);
    return;
  }

  void requestQuota(int newQuotaInBytes,
      [StorageQuotaCallback quotaCallback,
      StorageErrorCallback errorCallback]) {
    if (errorCallback != null) {
      _blink.BlinkDeprecatedStorageQuota.instance.requestQuota_Callback_3_(
          this, newQuotaInBytes, quotaCallback, errorCallback);
      return;
    }
    if (quotaCallback != null) {
      _blink.BlinkDeprecatedStorageQuota.instance
          .requestQuota_Callback_2_(this, newQuotaInBytes, quotaCallback);
      return;
    }
    _blink.BlinkDeprecatedStorageQuota.instance
        .requestQuota_Callback_1_(this, newQuotaInBytes);
    return;
  }
}
// Copyright (c) 2012, the Dart project authors.  Please see the AUTHORS file
// for details. All rights reserved. Use of this source code is governed by a
// BSD-style license that can be found in the LICENSE file.

// WARNING: Do not edit - generated code.

@DocsEditable()
@DomName('HTMLDetailsElement')
@SupportedBrowser(SupportedBrowser.CHROME)
@SupportedBrowser(SupportedBrowser.SAFARI)
@Experimental()
class DetailsElement extends HtmlElement {
  // To suppress missing implicit constructor warnings.
  factory DetailsElement._() {
    throw new UnsupportedError("Not supported");
  }

  @DomName('HTMLDetailsElement.HTMLDetailsElement')
  @DocsEditable()
  factory DetailsElement() => document.createElement("details");

  @Deprecated("Internal Use Only")
  external static Type get instanceRuntimeType;

  @Deprecated("Internal Use Only")
  DetailsElement.internal_() : super.internal_();

  /**
   * Constructor instantiated by the DOM when a custom element has been created.
   *
   * This can only be called by subclasses from their created constructor.
   */
  DetailsElement.created() : super.created();

  /// Checks if this type is supported on the current platform.
  static bool get supported => true;

  @DomName('HTMLDetailsElement.open')
  @DocsEditable()
  bool get open => _blink.BlinkHTMLDetailsElement.instance.open_Getter_(this);

  @DomName('HTMLDetailsElement.open')
  @DocsEditable()
  set open(bool value) =>
      _blink.BlinkHTMLDetailsElement.instance.open_Setter_(this, value);
}
// Copyright (c) 2012, the Dart project authors.  Please see the AUTHORS file
// for details. All rights reserved. Use of this source code is governed by a
// BSD-style license that can be found in the LICENSE file.

// WARNING: Do not edit - generated code.

@DocsEditable()
@DomName('DeviceAcceleration')
// http://dev.w3.org/geo/api/spec-source-orientation.html#devicemotion
@Experimental()
class DeviceAcceleration extends DartHtmlDomObject {
  // To suppress missing implicit constructor warnings.
  factory DeviceAcceleration._() {
    throw new UnsupportedError("Not supported");
  }

  @Deprecated("Internal Use Only")
  external static Type get instanceRuntimeType;

  @Deprecated("Internal Use Only")
  DeviceAcceleration.internal_() {}

  @DomName('DeviceAcceleration.x')
  @DocsEditable()
  num get x => _blink.BlinkDeviceAcceleration.instance.x_Getter_(this);

  @DomName('DeviceAcceleration.y')
  @DocsEditable()
  num get y => _blink.BlinkDeviceAcceleration.instance.y_Getter_(this);

  @DomName('DeviceAcceleration.z')
  @DocsEditable()
  num get z => _blink.BlinkDeviceAcceleration.instance.z_Getter_(this);
}
// Copyright (c) 2012, the Dart project authors.  Please see the AUTHORS file
// for details. All rights reserved. Use of this source code is governed by a
// BSD-style license that can be found in the LICENSE file.

// WARNING: Do not edit - generated code.

@DocsEditable()
@DomName('DeviceLightEvent')
@Experimental() // untriaged
class DeviceLightEvent extends Event {
  // To suppress missing implicit constructor warnings.
  factory DeviceLightEvent._() {
    throw new UnsupportedError("Not supported");
  }

  @DomName('DeviceLightEvent.DeviceLightEvent')
  @DocsEditable()
  factory DeviceLightEvent(String type, [Map eventInitDict]) {
    if (eventInitDict != null) {
      var eventInitDict_1 = convertDartToNative_Dictionary(eventInitDict);
      return _blink.BlinkDeviceLightEvent.instance
          .constructorCallback_2_(type, eventInitDict_1);
    }
    return _blink.BlinkDeviceLightEvent.instance.constructorCallback_1_(type);
  }

  @Deprecated("Internal Use Only")
  external static Type get instanceRuntimeType;

  @Deprecated("Internal Use Only")
  DeviceLightEvent.internal_() : super.internal_();

  @DomName('DeviceLightEvent.value')
  @DocsEditable()
  @Experimental() // untriaged
  num get value => _blink.BlinkDeviceLightEvent.instance.value_Getter_(this);
}
// Copyright (c) 2012, the Dart project authors.  Please see the AUTHORS file
// for details. All rights reserved. Use of this source code is governed by a
// BSD-style license that can be found in the LICENSE file.

// WARNING: Do not edit - generated code.

@DocsEditable()
@DomName('DeviceMotionEvent')
// http://dev.w3.org/geo/api/spec-source-orientation.html#devicemotion
@Experimental()
class DeviceMotionEvent extends Event {
  // To suppress missing implicit constructor warnings.
  factory DeviceMotionEvent._() {
    throw new UnsupportedError("Not supported");
  }

  @Deprecated("Internal Use Only")
  external static Type get instanceRuntimeType;

  @Deprecated("Internal Use Only")
  DeviceMotionEvent.internal_() : super.internal_();

  @DomName('DeviceMotionEvent.acceleration')
  @DocsEditable()
  DeviceAcceleration get acceleration =>
      _blink.BlinkDeviceMotionEvent.instance.acceleration_Getter_(this);

  @DomName('DeviceMotionEvent.accelerationIncludingGravity')
  @DocsEditable()
  DeviceAcceleration get accelerationIncludingGravity =>
      _blink.BlinkDeviceMotionEvent.instance
          .accelerationIncludingGravity_Getter_(this);

  @DomName('DeviceMotionEvent.interval')
  @DocsEditable()
  num get interval =>
      _blink.BlinkDeviceMotionEvent.instance.interval_Getter_(this);

  @DomName('DeviceMotionEvent.rotationRate')
  @DocsEditable()
  DeviceRotationRate get rotationRate =>
      _blink.BlinkDeviceMotionEvent.instance.rotationRate_Getter_(this);

  @DomName('DeviceMotionEvent.initDeviceMotionEvent')
  @DocsEditable()
  @Experimental() // untriaged
  void initDeviceMotionEvent(
          String type,
          bool bubbles,
          bool cancelable,
          DeviceAcceleration acceleration,
          DeviceAcceleration accelerationIncludingGravity,
          DeviceRotationRate rotationRate,
          num interval) =>
      _blink.BlinkDeviceMotionEvent.instance.initDeviceMotionEvent_Callback_7_(
          this,
          type,
          bubbles,
          cancelable,
          acceleration,
          accelerationIncludingGravity,
          rotationRate,
          interval);
}
// Copyright (c) 2013, the Dart project authors.  Please see the AUTHORS file
// for details. All rights reserved. Use of this source code is governed by a
// BSD-style license that can be found in the LICENSE file.

// WARNING: Do not edit - generated code.

@DomName('DeviceOrientationEvent')
// http://dev.w3.org/geo/api/spec-source-orientation.html#devicemotion
@Experimental()
class DeviceOrientationEvent extends Event {
  factory DeviceOrientationEvent(String type,
      {bool canBubble: true,
      bool cancelable: true,
      num alpha: 0,
      num beta: 0,
      num gamma: 0,
      bool absolute: false}) {
    DeviceOrientationEvent e = document._createEvent("DeviceOrientationEvent");
    e._initDeviceOrientationEvent(
        type, canBubble, cancelable, alpha, beta, gamma, absolute);
    return e;
  }
  // To suppress missing implicit constructor warnings.
  factory DeviceOrientationEvent._() {
    throw new UnsupportedError("Not supported");
  }

  @Deprecated("Internal Use Only")
  external static Type get instanceRuntimeType;

  @Deprecated("Internal Use Only")
  DeviceOrientationEvent.internal_() : super.internal_();

  @DomName('DeviceOrientationEvent.absolute')
  @DocsEditable()
  bool get absolute =>
      _blink.BlinkDeviceOrientationEvent.instance.absolute_Getter_(this);

  @DomName('DeviceOrientationEvent.alpha')
  @DocsEditable()
  num get alpha =>
      _blink.BlinkDeviceOrientationEvent.instance.alpha_Getter_(this);

  @DomName('DeviceOrientationEvent.beta')
  @DocsEditable()
  num get beta =>
      _blink.BlinkDeviceOrientationEvent.instance.beta_Getter_(this);

  @DomName('DeviceOrientationEvent.gamma')
  @DocsEditable()
  num get gamma =>
      _blink.BlinkDeviceOrientationEvent.instance.gamma_Getter_(this);

  @DomName('DeviceOrientationEvent.initDeviceOrientationEvent')
  @DocsEditable()
  void _initDeviceOrientationEvent(String type, bool bubbles, bool cancelable,
          num alpha, num beta, num gamma, bool absolute) =>
      _blink.BlinkDeviceOrientationEvent.instance
          .initDeviceOrientationEvent_Callback_7_(
              this, type, bubbles, cancelable, alpha, beta, gamma, absolute);
}
// Copyright (c) 2012, the Dart project authors.  Please see the AUTHORS file
// for details. All rights reserved. Use of this source code is governed by a
// BSD-style license that can be found in the LICENSE file.

// WARNING: Do not edit - generated code.

@DocsEditable()
@DomName('DeviceRotationRate')
// http://dev.w3.org/geo/api/spec-source-orientation.html#devicemotion
@Experimental()
class DeviceRotationRate extends DartHtmlDomObject {
  // To suppress missing implicit constructor warnings.
  factory DeviceRotationRate._() {
    throw new UnsupportedError("Not supported");
  }

  @Deprecated("Internal Use Only")
  external static Type get instanceRuntimeType;

  @Deprecated("Internal Use Only")
  DeviceRotationRate.internal_() {}

  @DomName('DeviceRotationRate.alpha')
  @DocsEditable()
  num get alpha => _blink.BlinkDeviceRotationRate.instance.alpha_Getter_(this);

  @DomName('DeviceRotationRate.beta')
  @DocsEditable()
  num get beta => _blink.BlinkDeviceRotationRate.instance.beta_Getter_(this);

  @DomName('DeviceRotationRate.gamma')
  @DocsEditable()
  num get gamma => _blink.BlinkDeviceRotationRate.instance.gamma_Getter_(this);
}
// Copyright (c) 2012, the Dart project authors.  Please see the AUTHORS file
// for details. All rights reserved. Use of this source code is governed by a
// BSD-style license that can be found in the LICENSE file.

// WARNING: Do not edit - generated code.

@DocsEditable()
@DomName('HTMLDialogElement')
@Unstable()
class DialogElement extends HtmlElement {
  // To suppress missing implicit constructor warnings.
  factory DialogElement._() {
    throw new UnsupportedError("Not supported");
  }

  @Deprecated("Internal Use Only")
  external static Type get instanceRuntimeType;

  @Deprecated("Internal Use Only")
  DialogElement.internal_() : super.internal_();

  /**
   * Constructor instantiated by the DOM when a custom element has been created.
   *
   * This can only be called by subclasses from their created constructor.
   */
  DialogElement.created() : super.created();

  @DomName('HTMLDialogElement.open')
  @DocsEditable()
  bool get open => _blink.BlinkHTMLDialogElement.instance.open_Getter_(this);

  @DomName('HTMLDialogElement.open')
  @DocsEditable()
  set open(bool value) =>
      _blink.BlinkHTMLDialogElement.instance.open_Setter_(this, value);

  @DomName('HTMLDialogElement.returnValue')
  @DocsEditable()
  @Experimental() // untriaged
  String get returnValue =>
      _blink.BlinkHTMLDialogElement.instance.returnValue_Getter_(this);

  @DomName('HTMLDialogElement.returnValue')
  @DocsEditable()
  @Experimental() // untriaged
  set returnValue(String value) =>
      _blink.BlinkHTMLDialogElement.instance.returnValue_Setter_(this, value);

  @DomName('HTMLDialogElement.close')
  @DocsEditable()
  void close(String returnValue) => _blink.BlinkHTMLDialogElement.instance
      .close_Callback_1_(this, returnValue);

  @DomName('HTMLDialogElement.show')
  @DocsEditable()
  void show() => _blink.BlinkHTMLDialogElement.instance.show_Callback_0_(this);

  @DomName('HTMLDialogElement.showModal')
  @DocsEditable()
  void showModal() =>
      _blink.BlinkHTMLDialogElement.instance.showModal_Callback_0_(this);
}
// Copyright (c) 2013, the Dart project authors.  Please see the AUTHORS file
// for details. All rights reserved. Use of this source code is governed by a
// BSD-style license that can be found in the LICENSE file.

@DomName('DirectoryEntry')
// http://www.w3.org/TR/file-system-api/#the-directoryentry-interface
@Experimental()
class DirectoryEntry extends Entry {
  /**
   * Create a new directory with the specified `path`. If `exclusive` is true,
   * the returned Future will complete with an error if a directory already
   * exists with the specified `path`.
   */
  Future<Entry> createDirectory(String path, {bool exclusive: false}) {
    return _getDirectory(path,
        options: {'create': true, 'exclusive': exclusive});
  }

  /**
   * Retrieve an already existing directory entry. The returned future will
   * result in an error if a directory at `path` does not exist or if the item
   * at `path` is not a directory.
   */
  Future<Entry> getDirectory(String path) {
    return _getDirectory(path);
  }

  /**
   * Create a new file with the specified `path`. If `exclusive` is true,
   * the returned Future will complete with an error if a file already
   * exists at the specified `path`.
   */
  Future<Entry> createFile(String path, {bool exclusive: false}) {
    return _getFile(path, options: {'create': true, 'exclusive': exclusive});
  }

  /**
   * Retrieve an already existing file entry. The returned future will
   * result in an error if a file at `path` does not exist or if the item at
   * `path` is not a file.
   */
  Future<Entry> getFile(String path) {
    return _getFile(path);
  }

  // To suppress missing implicit constructor warnings.
  factory DirectoryEntry._() {
    throw new UnsupportedError("Not supported");
  }

  @Deprecated("Internal Use Only")
  external static Type get instanceRuntimeType;

  @Deprecated("Internal Use Only")
  DirectoryEntry.internal_() : super.internal_();

  @DomName('DirectoryEntry.createReader')
  @DocsEditable()
  DirectoryReader createReader() =>
      _blink.BlinkDirectoryEntry.instance.createReader_Callback_0_(this);

  void __getDirectory(String path,
      {Map options,
      _EntryCallback successCallback,
      _ErrorCallback errorCallback}) {
    if (errorCallback != null) {
      _blink.BlinkDirectoryEntry.instance.getDirectory_Callback_4_(
          this,
          path,
          convertDartToNative_Dictionary(options),
          successCallback,
          errorCallback);
      return;
    }
    if (successCallback != null) {
      _blink.BlinkDirectoryEntry.instance.getDirectory_Callback_3_(
          this, path, convertDartToNative_Dictionary(options), successCallback);
      return;
    }
    if (options != null) {
      _blink.BlinkDirectoryEntry.instance.getDirectory_Callback_2_(
          this, path, convertDartToNative_Dictionary(options));
      return;
    }
    _blink.BlinkDirectoryEntry.instance.getDirectory_Callback_1_(this, path);
    return;
  }

  Future<Entry> _getDirectory(String path, {Map options}) {
    var completer = new Completer<Entry>();
    __getDirectory(path, options: options, successCallback: (value) {
      completer.complete(value);
    }, errorCallback: (error) {
      completer.completeError(error);
    });
    return completer.future;
  }

  void __getFile(String path,
      {Map options,
      _EntryCallback successCallback,
      _ErrorCallback errorCallback}) {
    if (errorCallback != null) {
      _blink.BlinkDirectoryEntry.instance.getFile_Callback_4_(
          this,
          path,
          convertDartToNative_Dictionary(options),
          successCallback,
          errorCallback);
      return;
    }
    if (successCallback != null) {
      _blink.BlinkDirectoryEntry.instance.getFile_Callback_3_(
          this, path, convertDartToNative_Dictionary(options), successCallback);
      return;
    }
    if (options != null) {
      _blink.BlinkDirectoryEntry.instance.getFile_Callback_2_(
          this, path, convertDartToNative_Dictionary(options));
      return;
    }
    _blink.BlinkDirectoryEntry.instance.getFile_Callback_1_(this, path);
    return;
  }

  Future<Entry> _getFile(String path, {Map options}) {
    var completer = new Completer<Entry>();
    __getFile(path, options: options, successCallback: (value) {
      completer.complete(value);
    }, errorCallback: (error) {
      completer.completeError(error);
    });
    return completer.future;
  }

  void _removeRecursively(VoidCallback successCallback,
      [_ErrorCallback errorCallback]) {
    if (errorCallback != null) {
      _blink.BlinkDirectoryEntry.instance
          .removeRecursively_Callback_2_(this, successCallback, errorCallback);
      return;
    }
    _blink.BlinkDirectoryEntry.instance
        .removeRecursively_Callback_1_(this, successCallback);
    return;
  }

  Future removeRecursively() {
    var completer = new Completer();
    _removeRecursively(() {
      completer.complete();
    }, (error) {
      completer.completeError(error);
    });
    return completer.future;
  }
}
// Copyright (c) 2012, the Dart project authors.  Please see the AUTHORS file
// for details. All rights reserved. Use of this source code is governed by a
// BSD-style license that can be found in the LICENSE file.

// WARNING: Do not edit - generated code.

@DocsEditable()
@DomName('DirectoryReader')
// http://www.w3.org/TR/file-system-api/#the-directoryreader-interface
@Experimental()
class DirectoryReader extends DartHtmlDomObject {
  // To suppress missing implicit constructor warnings.
  factory DirectoryReader._() {
    throw new UnsupportedError("Not supported");
  }

  @Deprecated("Internal Use Only")
  external static Type get instanceRuntimeType;

  @Deprecated("Internal Use Only")
  DirectoryReader.internal_() {}

  void _readEntries(_EntriesCallback successCallback,
      [_ErrorCallback errorCallback]) {
    if (errorCallback != null) {
      _blink.BlinkDirectoryReader.instance
          .readEntries_Callback_2_(this, successCallback, errorCallback);
      return;
    }
    _blink.BlinkDirectoryReader.instance
        .readEntries_Callback_1_(this, successCallback);
    return;
  }

  Future<List<Entry>> readEntries() {
    var completer = new Completer<List<Entry>>();
    _readEntries((value) {
      completer.complete(value);
    }, (error) {
      completer.completeError(error);
    });
    return completer.future;
  }
}
// Copyright (c) 2012, the Dart project authors.  Please see the AUTHORS file
// for details. All rights reserved. Use of this source code is governed by a
// BSD-style license that can be found in the LICENSE file.

// WARNING: Do not edit - generated code.

@DocsEditable()
/**
 * A generic container for content on an HTML page;
 * corresponds to the &lt;div&gt; tag.
 *
 * The [DivElement] is a generic container and does not have any semantic
 * significance. It is functionally similar to [SpanElement].
 *
 * The [DivElement] is a block-level element, as opposed to [SpanElement],
 * which is an inline-level element.
 *
 * Example usage:
 *
 *     DivElement div = new DivElement();
 *     div.text = 'Here's my new DivElem
 *     document.body.elements.add(elem);
 *
 * See also:
 *
 * * [HTML `<div>` element](http://www.w3.org/TR/html-markup/div.html) from W3C.
 * * [Block-level element](http://www.w3.org/TR/CSS2/visuren.html#block-boxes) from W3C.
 * * [Inline-level element](http://www.w3.org/TR/CSS2/visuren.html#inline-boxes) from W3C.
 */
@DomName('HTMLDivElement')
class DivElement extends HtmlElement {
  // To suppress missing implicit constructor warnings.
  factory DivElement._() {
    throw new UnsupportedError("Not supported");
  }

  @DomName('HTMLDivElement.HTMLDivElement')
  @DocsEditable()
  factory DivElement() => document.createElement("div");

  @Deprecated("Internal Use Only")
  external static Type get instanceRuntimeType;

  @Deprecated("Internal Use Only")
  DivElement.internal_() : super.internal_();

  /**
   * Constructor instantiated by the DOM when a custom element has been created.
   *
   * This can only be called by subclasses from their created constructor.
   */
  DivElement.created() : super.created();
}
// Copyright (c) 2012, the Dart project authors.  Please see the AUTHORS file
// for details. All rights reserved. Use of this source code is governed by a
// BSD-style license that can be found in the LICENSE file.

@DocsEditable()
/**
 * The base class for all documents.
 *
 * Each web page loaded in the browser has its own [Document] object, which is
 * typically an [HtmlDocument].
 *
 * If you aren't comfortable with DOM concepts, see the Dart tutorial
 * [Target 2: Connect Dart & HTML](http://www.dartlang.org/docs/tutorials/connect-dart-html/).
 */
@DomName('Document')
class Document extends Node {
  // To suppress missing implicit constructor warnings.
  factory Document._() {
    throw new UnsupportedError("Not supported");
  }

  @DomName('Document.pointerlockchangeEvent')
  @DocsEditable()
  @Experimental() // untriaged
  static const EventStreamProvider<Event> pointerLockChangeEvent =
      const EventStreamProvider<Event>('pointerlockchange');

  @DomName('Document.pointerlockerrorEvent')
  @DocsEditable()
  @Experimental() // untriaged
  static const EventStreamProvider<Event> pointerLockErrorEvent =
      const EventStreamProvider<Event>('pointerlockerror');

  /**
   * Static factory designed to expose `readystatechange` events to event
   * handlers that are not necessarily instances of [Document].
   *
   * See [EventStreamProvider] for usage information.
   */
  @DomName('Document.readystatechangeEvent')
  @DocsEditable()
  static const EventStreamProvider<Event> readyStateChangeEvent =
      const EventStreamProvider<Event>('readystatechange');

  /**
   * Static factory designed to expose `securitypolicyviolation` events to event
   * handlers that are not necessarily instances of [Document].
   *
   * See [EventStreamProvider] for usage information.
   */
  @DomName('Document.securitypolicyviolationEvent')
  @DocsEditable()
  // https://dvcs.w3.org/hg/content-security-policy/raw-file/tip/csp-specification.dev.html#widl-Document-onsecuritypolicyviolation
  @Experimental()
  static const EventStreamProvider<SecurityPolicyViolationEvent>
      securityPolicyViolationEvent =
      const EventStreamProvider<SecurityPolicyViolationEvent>(
          'securitypolicyviolation');

  /**
   * Static factory designed to expose `selectionchange` events to event
   * handlers that are not necessarily instances of [Document].
   *
   * See [EventStreamProvider] for usage information.
   */
  @DomName('Document.selectionchangeEvent')
  @DocsEditable()
  static const EventStreamProvider<Event> selectionChangeEvent =
      const EventStreamProvider<Event>('selectionchange');

  @Deprecated("Internal Use Only")
  external static Type get instanceRuntimeType;

  @Deprecated("Internal Use Only")
  Document.internal_() : super.internal_();

  @DomName('Document.activeElement')
  @DocsEditable()
  @Experimental() // untriaged
  Element get activeElement =>
      _blink.BlinkDocument.instance.activeElement_Getter_(this);

  @DomName('Document.body')
  @DocsEditable()
  HtmlElement get _body => _blink.BlinkDocument.instance.body_Getter_(this);

  @DomName('Document.body')
  @DocsEditable()
  set _body(HtmlElement value) =>
      _blink.BlinkDocument.instance.body_Setter_(this, value);

  @DomName('Document.contentType')
  @DocsEditable()
  @Experimental() // untriaged
  String get contentType =>
      _blink.BlinkDocument.instance.contentType_Getter_(this);

  @DomName('Document.cookie')
  @DocsEditable()
  String get cookie => _blink.BlinkDocument.instance.cookie_Getter_(this);

  @DomName('Document.cookie')
  @DocsEditable()
  set cookie(String value) =>
      _blink.BlinkDocument.instance.cookie_Setter_(this, value);

  @DomName('Document.currentScript')
  @DocsEditable()
  @Experimental() // untriaged
  ScriptElement get currentScript =>
      _blink.BlinkDocument.instance.currentScript_Getter_(this);

  @DomName('Document.defaultView')
  @DocsEditable()
  WindowBase get window => _convertNativeToDart_Window(
      _blink.BlinkDocument.instance.defaultView_Getter_(this));

  @DomName('Document.documentElement')
  @DocsEditable()
  Element get documentElement =>
      _blink.BlinkDocument.instance.documentElement_Getter_(this);

  @DomName('Document.domain')
  @DocsEditable()
  String get domain => _blink.BlinkDocument.instance.domain_Getter_(this);

  @DomName('Document.fonts')
  @DocsEditable()
  @Experimental() // untriaged
  FontFaceSet get fonts => _blink.BlinkDocument.instance.fonts_Getter_(this);

  @DomName('Document.fullscreenElement')
  @DocsEditable()
  @Experimental() // untriaged
  Element get fullscreenElement =>
      _blink.BlinkDocument.instance.fullscreenElement_Getter_(this);

  @DomName('Document.fullscreenEnabled')
  @DocsEditable()
  @Experimental() // untriaged
  bool get fullscreenEnabled =>
      _blink.BlinkDocument.instance.fullscreenEnabled_Getter_(this);

  @DomName('Document.head')
  @DocsEditable()
  HeadElement get _head => _blink.BlinkDocument.instance.head_Getter_(this);

  @DomName('Document.hidden')
  @DocsEditable()
  @Experimental() // untriaged
  bool get hidden => _blink.BlinkDocument.instance.hidden_Getter_(this);

  @DomName('Document.implementation')
  @DocsEditable()
  DomImplementation get implementation =>
      _blink.BlinkDocument.instance.implementation_Getter_(this);

  @DomName('Document.lastModified')
  @DocsEditable()
  String get _lastModified =>
      _blink.BlinkDocument.instance.lastModified_Getter_(this);

  @DomName('Document.origin')
  @DocsEditable()
  @Experimental() // untriaged
  String get origin => _blink.BlinkDocument.instance.origin_Getter_(this);

  @DomName('Document.pointerLockElement')
  @DocsEditable()
  @Experimental() // untriaged
  Element get pointerLockElement =>
      _blink.BlinkDocument.instance.pointerLockElement_Getter_(this);

  @DomName('Document.preferredStylesheetSet')
  @DocsEditable()
  String get _preferredStylesheetSet =>
      _blink.BlinkDocument.instance.preferredStylesheetSet_Getter_(this);

  @DomName('Document.readyState')
  @DocsEditable()
  String get readyState =>
      _blink.BlinkDocument.instance.readyState_Getter_(this);

  @DomName('Document.referrer')
  @DocsEditable()
  String get _referrer => _blink.BlinkDocument.instance.referrer_Getter_(this);

  @DomName('Document.rootElement')
  @DocsEditable()
  @Experimental() // untriaged
  SvgSvgElement get rootElement =>
      _blink.BlinkDocument.instance.rootElement_Getter_(this);

  @DomName('Document.scrollingElement')
  @DocsEditable()
  @Experimental() // untriaged
  Element get scrollingElement =>
      _blink.BlinkDocument.instance.scrollingElement_Getter_(this);

  @DomName('Document.selectedStylesheetSet')
  @DocsEditable()
  String get _selectedStylesheetSet =>
      _blink.BlinkDocument.instance.selectedStylesheetSet_Getter_(this);

  @DomName('Document.selectedStylesheetSet')
  @DocsEditable()
  set _selectedStylesheetSet(String value) =>
      _blink.BlinkDocument.instance.selectedStylesheetSet_Setter_(this, value);

  @DomName('Document.styleSheets')
  @DocsEditable()
  List<StyleSheet> get _styleSheets =>
      (_blink.BlinkDocument.instance.styleSheets_Getter_(this));

  @DomName('Document.timeline')
  @DocsEditable()
  @Experimental() // untriaged
  AnimationTimeline get timeline =>
      _blink.BlinkDocument.instance.timeline_Getter_(this);

  @DomName('Document.title')
  @DocsEditable()
  String get _title => _blink.BlinkDocument.instance.title_Getter_(this);

  @DomName('Document.title')
  @DocsEditable()
  set _title(String value) =>
      _blink.BlinkDocument.instance.title_Setter_(this, value);

  @DomName('Document.visibilityState')
  @DocsEditable()
  @Experimental() // untriaged
  String get _visibilityState =>
      _blink.BlinkDocument.instance.visibilityState_Getter_(this);

  @DomName('Document.webkitFullscreenElement')
  @DocsEditable()
  @SupportedBrowser(SupportedBrowser.CHROME)
  @SupportedBrowser(SupportedBrowser.SAFARI)
  @Experimental()
  // https://dvcs.w3.org/hg/fullscreen/raw-file/tip/Overview.html#dom-document-fullscreenelement
  Element get _webkitFullscreenElement =>
      _blink.BlinkDocument.instance.webkitFullscreenElement_Getter_(this);

  @DomName('Document.webkitFullscreenEnabled')
  @DocsEditable()
  @SupportedBrowser(SupportedBrowser.CHROME)
  @SupportedBrowser(SupportedBrowser.SAFARI)
  @Experimental()
  // https://dvcs.w3.org/hg/fullscreen/raw-file/tip/Overview.html#dom-document-fullscreenenabled
  bool get _webkitFullscreenEnabled =>
      _blink.BlinkDocument.instance.webkitFullscreenEnabled_Getter_(this);

  @DomName('Document.webkitHidden')
  @DocsEditable()
  @SupportedBrowser(SupportedBrowser.CHROME)
  @SupportedBrowser(SupportedBrowser.SAFARI)
  @Experimental()
  // https://dvcs.w3.org/hg/webperf/raw-file/tip/specs/PageVisibility/Overview.html#document
  bool get _webkitHidden =>
      _blink.BlinkDocument.instance.webkitHidden_Getter_(this);

  @DomName('Document.webkitVisibilityState')
  @DocsEditable()
  @SupportedBrowser(SupportedBrowser.CHROME)
  @SupportedBrowser(SupportedBrowser.SAFARI)
  @Experimental()
  // https://dvcs.w3.org/hg/webperf/raw-file/tip/specs/PageVisibility/Overview.html#dom-document-visibilitystate
  String get _webkitVisibilityState =>
      _blink.BlinkDocument.instance.webkitVisibilityState_Getter_(this);

  @DomName('Document.adoptNode')
  @DocsEditable()
  Node adoptNode(Node node) =>
      _blink.BlinkDocument.instance.adoptNode_Callback_1_(this, node);

  @DomName('Document.caretRangeFromPoint')
  @DocsEditable()
  // http://www.w3.org/TR/2009/WD-cssom-view-20090804/#dom-documentview-caretrangefrompoint
  @Experimental()
  Range _caretRangeFromPoint(int x, int y) =>
      _blink.BlinkDocument.instance.caretRangeFromPoint_Callback_2_(this, x, y);

  @DomName('Document.createDocumentFragment')
  @DocsEditable()
  DocumentFragment createDocumentFragment() =>
      _blink.BlinkDocument.instance.createDocumentFragment_Callback_0_(this);

  @DomName('Document.createElement')
  @DocsEditable()
  Element _createElement(String localName_OR_tagName, [String typeExtension]) =>
      _blink.BlinkDocument.instance
          .createElement_Callback_2_(this, localName_OR_tagName, typeExtension);

  @DomName('Document.createElementNS')
  @DocsEditable()
  Element _createElementNS(String namespaceURI, String qualifiedName,
          [String typeExtension]) =>
      _blink.BlinkDocument.instance.createElementNS_Callback_3_(
          this, namespaceURI, qualifiedName, typeExtension);

  @DomName('Document.createEvent')
  @DocsEditable()
  Event _createEvent(String eventType) =>
      _blink.BlinkDocument.instance.createEvent_Callback_1_(this, eventType);

  NodeIterator _createNodeIterator(Node root,
      [int whatToShow, NodeFilter filter]) {
    if (whatToShow != null) {
      return _blink.BlinkDocument.instance
          .createNodeIterator_Callback_3_(this, root, whatToShow, filter);
    }
    return _blink.BlinkDocument.instance
        .createNodeIterator_Callback_1_(this, root);
  }

  @DomName('Document.createRange')
  @DocsEditable()
  Range createRange() =>
      _blink.BlinkDocument.instance.createRange_Callback_0_(this);

  @DomName('Document.createTextNode')
  @DocsEditable()
  Text _createTextNode(String data) =>
      _blink.BlinkDocument.instance.createTextNode_Callback_1_(this, data);

  @DomName('Document.createTouch')
  @DocsEditable()
  // http://www.w3.org/TR/touch-events/, http://www.chromestatus.com/features
  @Experimental()
  Touch _createTouch(
          Window window,
          EventTarget target,
          int identifier,
          num pageX,
          num pageY,
          num screenX,
          num screenY,
          num radiusX,
          num radiusY,
          num rotationAngle,
          num force) =>
      _blink.BlinkDocument.instance.createTouch_Callback_11_(
          this,
          window,
          _convertDartToNative_EventTarget(target),
          identifier,
          pageX,
          pageY,
          screenX,
          screenY,
          radiusX,
          radiusY,
          rotationAngle,
          force);

  @DomName('Document.createTouchList')
  @DocsEditable()
  // http://www.w3.org/TR/touch-events/, http://www.chromestatus.com/features
  @Experimental()
  TouchList _createTouchList(Touch touches) =>
      _blink.BlinkDocument.instance.createTouchList_Callback_1_(this, touches);

  TreeWalker _createTreeWalker(Node root, [int whatToShow, NodeFilter filter]) {
    if (whatToShow != null) {
      return _blink.BlinkDocument.instance
          .createTreeWalker_Callback_3_(this, root, whatToShow, filter);
    }
    return _blink.BlinkDocument.instance
        .createTreeWalker_Callback_1_(this, root);
  }

  @DomName('Document.elementFromPoint')
  @DocsEditable()
  Element _elementFromPoint(int x, int y) =>
      _blink.BlinkDocument.instance.elementFromPoint_Callback_2_(this, x, y);

  @DomName('Document.elementsFromPoint')
  @DocsEditable()
  @Experimental() // untriaged
  List<Element> elementsFromPoint(int x, int y) =>
      (_blink.BlinkDocument.instance.elementsFromPoint_Callback_2_(this, x, y));

  bool execCommand(String commandId, [bool showUI, String value]) {
    if (value != null) {
      return _blink.BlinkDocument.instance
          .execCommand_Callback_3_(this, commandId, showUI, value);
    }
    if (showUI != null) {
      return _blink.BlinkDocument.instance
          .execCommand_Callback_2_(this, commandId, showUI);
    }
    return _blink.BlinkDocument.instance
        .execCommand_Callback_1_(this, commandId);
  }

  @DomName('Document.exitFullscreen')
  @DocsEditable()
  @Experimental() // untriaged
  void exitFullscreen() =>
      _blink.BlinkDocument.instance.exitFullscreen_Callback_0_(this);

  @DomName('Document.exitPointerLock')
  @DocsEditable()
  @Experimental() // untriaged
  void exitPointerLock() =>
      _blink.BlinkDocument.instance.exitPointerLock_Callback_0_(this);

  @DomName('Document.getCSSCanvasContext')
  @DocsEditable()
  // https://developer.apple.com/library/safari/#documentation/AppleApplications/Reference/SafariCSSRef/Articles/Functions.html
  @Experimental() // non-standard
  Object _getCssCanvasContext(
          String contextId, String name, int width, int height) =>
      (_blink.BlinkDocument.instance.getCSSCanvasContext_Callback_4_(
          this, contextId, name, width, height));

  @DomName('Document.getElementsByClassName')
  @DocsEditable()
  List<Node> getElementsByClassName(String classNames) =>
      (_blink.BlinkDocument.instance
          .getElementsByClassName_Callback_1_(this, classNames));

  @DomName('Document.getElementsByName')
  @DocsEditable()
  List<Node> getElementsByName(String elementName) =>
      (_blink.BlinkDocument.instance
          .getElementsByName_Callback_1_(this, elementName));

  @DomName('Document.getElementsByTagName')
  @DocsEditable()
  List<Node> getElementsByTagName(String localName) =>
      (_blink.BlinkDocument.instance
          .getElementsByTagName_Callback_1_(this, localName));

  Node importNode(Node node, [bool deep]) {
    if (deep != null) {
      return _blink.BlinkDocument.instance
          .importNode_Callback_2_(this, node, deep);
    }
    return _blink.BlinkDocument.instance.importNode_Callback_1_(this, node);
  }

  @DomName('Document.queryCommandEnabled')
  @DocsEditable()
  bool queryCommandEnabled(String commandId) => _blink.BlinkDocument.instance
      .queryCommandEnabled_Callback_1_(this, commandId);

  @DomName('Document.queryCommandIndeterm')
  @DocsEditable()
  bool queryCommandIndeterm(String commandId) => _blink.BlinkDocument.instance
      .queryCommandIndeterm_Callback_1_(this, commandId);

  @DomName('Document.queryCommandState')
  @DocsEditable()
  bool queryCommandState(String commandId) => _blink.BlinkDocument.instance
      .queryCommandState_Callback_1_(this, commandId);

  @DomName('Document.queryCommandSupported')
  @DocsEditable()
  bool queryCommandSupported(String commandId) => _blink.BlinkDocument.instance
      .queryCommandSupported_Callback_1_(this, commandId);

  @DomName('Document.queryCommandValue')
  @DocsEditable()
  String queryCommandValue(String commandId) => _blink.BlinkDocument.instance
      .queryCommandValue_Callback_1_(this, commandId);

  @DomName('Document.transformDocumentToTreeView')
  @DocsEditable()
  @Experimental() // untriaged
  void transformDocumentToTreeView(String noStyleMessage) =>
      _blink.BlinkDocument.instance
          .transformDocumentToTreeView_Callback_1_(this, noStyleMessage);

  @DomName('Document.webkitExitFullscreen')
  @DocsEditable()
  @SupportedBrowser(SupportedBrowser.CHROME)
  @SupportedBrowser(SupportedBrowser.SAFARI)
  @Experimental()
  // https://dvcs.w3.org/hg/fullscreen/raw-file/tip/Overview.html#dom-document-exitfullscreen
  void _webkitExitFullscreen() =>
      _blink.BlinkDocument.instance.webkitExitFullscreen_Callback_0_(this);

  @DomName('Document.getElementById')
  @DocsEditable()
  Element getElementById(String elementId) =>
      _blink.BlinkDocument.instance.getElementById_Callback_1_(this, elementId);

  @DomName('Document.childElementCount')
  @DocsEditable()
  int get _childElementCount =>
      _blink.BlinkDocument.instance.childElementCount_Getter_(this);

  @DomName('Document.children')
  @DocsEditable()
  List<Node> get _children =>
      (_blink.BlinkDocument.instance.children_Getter_(this));

  @DomName('Document.firstElementChild')
  @DocsEditable()
  Element get _firstElementChild =>
      _blink.BlinkDocument.instance.firstElementChild_Getter_(this);

  @DomName('Document.lastElementChild')
  @DocsEditable()
  Element get _lastElementChild =>
      _blink.BlinkDocument.instance.lastElementChild_Getter_(this);

  /**
   * Finds the first descendant element of this document that matches the
   * specified group of selectors.
   *
   * Unless your webpage contains multiple documents, the top-level
   * [querySelector]
   * method behaves the same as this method, so you should use it instead to
   * save typing a few characters.
   *
   * [selectors] should be a string using CSS selector syntax.
   *
   *     var element1 = document.querySelector('.className');
   *     var element2 = document.querySelector('#id');
   *
   * For details about CSS selector syntax, see the
   * [CSS selector specification](http://www.w3.org/TR/css3-selectors/).
   */
  @DomName('Document.querySelector')
  @DocsEditable()
  Element querySelector(String selectors) =>
      _blink.BlinkDocument.instance.querySelector_Callback_1_(this, selectors);

  @DomName('Document.querySelectorAll')
  @DocsEditable()
  List<Node> _querySelectorAll(String selectors) =>
      (_blink.BlinkDocument.instance
          .querySelectorAll_Callback_1_(this, selectors));

  /// Stream of `abort` events handled by this [Document].
  @DomName('Document.onabort')
  @DocsEditable()
  Stream<Event> get onAbort => Element.abortEvent.forTarget(this);

  /// Stream of `beforecopy` events handled by this [Document].
  @DomName('Document.onbeforecopy')
  @DocsEditable()
  Stream<Event> get onBeforeCopy => Element.beforeCopyEvent.forTarget(this);

  /// Stream of `beforecut` events handled by this [Document].
  @DomName('Document.onbeforecut')
  @DocsEditable()
  Stream<Event> get onBeforeCut => Element.beforeCutEvent.forTarget(this);

  /// Stream of `beforepaste` events handled by this [Document].
  @DomName('Document.onbeforepaste')
  @DocsEditable()
  Stream<Event> get onBeforePaste => Element.beforePasteEvent.forTarget(this);

  /// Stream of `blur` events handled by this [Document].
  @DomName('Document.onblur')
  @DocsEditable()
  Stream<Event> get onBlur => Element.blurEvent.forTarget(this);

  @DomName('Document.oncanplay')
  @DocsEditable()
  @Experimental() // untriaged
  Stream<Event> get onCanPlay => Element.canPlayEvent.forTarget(this);

  @DomName('Document.oncanplaythrough')
  @DocsEditable()
  @Experimental() // untriaged
  Stream<Event> get onCanPlayThrough =>
      Element.canPlayThroughEvent.forTarget(this);

  /// Stream of `change` events handled by this [Document].
  @DomName('Document.onchange')
  @DocsEditable()
  Stream<Event> get onChange => Element.changeEvent.forTarget(this);

  /// Stream of `click` events handled by this [Document].
  @DomName('Document.onclick')
  @DocsEditable()
  Stream<MouseEvent> get onClick => Element.clickEvent.forTarget(this);

  /// Stream of `contextmenu` events handled by this [Document].
  @DomName('Document.oncontextmenu')
  @DocsEditable()
  Stream<MouseEvent> get onContextMenu =>
      Element.contextMenuEvent.forTarget(this);

  /// Stream of `copy` events handled by this [Document].
  @DomName('Document.oncopy')
  @DocsEditable()
  Stream<ClipboardEvent> get onCopy => Element.copyEvent.forTarget(this);

  /// Stream of `cut` events handled by this [Document].
  @DomName('Document.oncut')
  @DocsEditable()
  Stream<ClipboardEvent> get onCut => Element.cutEvent.forTarget(this);

  /// Stream of `doubleclick` events handled by this [Document].
  @DomName('Document.ondblclick')
  @DocsEditable()
  Stream<Event> get onDoubleClick => Element.doubleClickEvent.forTarget(this);

  /// Stream of `drag` events handled by this [Document].
  @DomName('Document.ondrag')
  @DocsEditable()
  Stream<MouseEvent> get onDrag => Element.dragEvent.forTarget(this);

  /// Stream of `dragend` events handled by this [Document].
  @DomName('Document.ondragend')
  @DocsEditable()
  Stream<MouseEvent> get onDragEnd => Element.dragEndEvent.forTarget(this);

  /// Stream of `dragenter` events handled by this [Document].
  @DomName('Document.ondragenter')
  @DocsEditable()
  Stream<MouseEvent> get onDragEnter => Element.dragEnterEvent.forTarget(this);

  /// Stream of `dragleave` events handled by this [Document].
  @DomName('Document.ondragleave')
  @DocsEditable()
  Stream<MouseEvent> get onDragLeave => Element.dragLeaveEvent.forTarget(this);

  /// Stream of `dragover` events handled by this [Document].
  @DomName('Document.ondragover')
  @DocsEditable()
  Stream<MouseEvent> get onDragOver => Element.dragOverEvent.forTarget(this);

  /// Stream of `dragstart` events handled by this [Document].
  @DomName('Document.ondragstart')
  @DocsEditable()
  Stream<MouseEvent> get onDragStart => Element.dragStartEvent.forTarget(this);

  /// Stream of `drop` events handled by this [Document].
  @DomName('Document.ondrop')
  @DocsEditable()
  Stream<MouseEvent> get onDrop => Element.dropEvent.forTarget(this);

  @DomName('Document.ondurationchange')
  @DocsEditable()
  @Experimental() // untriaged
  Stream<Event> get onDurationChange =>
      Element.durationChangeEvent.forTarget(this);

  @DomName('Document.onemptied')
  @DocsEditable()
  @Experimental() // untriaged
  Stream<Event> get onEmptied => Element.emptiedEvent.forTarget(this);

  @DomName('Document.onended')
  @DocsEditable()
  @Experimental() // untriaged
  Stream<Event> get onEnded => Element.endedEvent.forTarget(this);

  /// Stream of `error` events handled by this [Document].
  @DomName('Document.onerror')
  @DocsEditable()
  Stream<Event> get onError => Element.errorEvent.forTarget(this);

  /// Stream of `focus` events handled by this [Document].
  @DomName('Document.onfocus')
  @DocsEditable()
  Stream<Event> get onFocus => Element.focusEvent.forTarget(this);

  /// Stream of `input` events handled by this [Document].
  @DomName('Document.oninput')
  @DocsEditable()
  Stream<Event> get onInput => Element.inputEvent.forTarget(this);

  /// Stream of `invalid` events handled by this [Document].
  @DomName('Document.oninvalid')
  @DocsEditable()
  Stream<Event> get onInvalid => Element.invalidEvent.forTarget(this);

  /// Stream of `keydown` events handled by this [Document].
  @DomName('Document.onkeydown')
  @DocsEditable()
  Stream<KeyboardEvent> get onKeyDown => Element.keyDownEvent.forTarget(this);

  /// Stream of `keypress` events handled by this [Document].
  @DomName('Document.onkeypress')
  @DocsEditable()
  Stream<KeyboardEvent> get onKeyPress => Element.keyPressEvent.forTarget(this);

  /// Stream of `keyup` events handled by this [Document].
  @DomName('Document.onkeyup')
  @DocsEditable()
  Stream<KeyboardEvent> get onKeyUp => Element.keyUpEvent.forTarget(this);

  /// Stream of `load` events handled by this [Document].
  @DomName('Document.onload')
  @DocsEditable()
  Stream<Event> get onLoad => Element.loadEvent.forTarget(this);

  @DomName('Document.onloadeddata')
  @DocsEditable()
  @Experimental() // untriaged
  Stream<Event> get onLoadedData => Element.loadedDataEvent.forTarget(this);

  @DomName('Document.onloadedmetadata')
  @DocsEditable()
  @Experimental() // untriaged
  Stream<Event> get onLoadedMetadata =>
      Element.loadedMetadataEvent.forTarget(this);

  /// Stream of `mousedown` events handled by this [Document].
  @DomName('Document.onmousedown')
  @DocsEditable()
  Stream<MouseEvent> get onMouseDown => Element.mouseDownEvent.forTarget(this);

  /// Stream of `mouseenter` events handled by this [Document].
  @DomName('Document.onmouseenter')
  @DocsEditable()
  @Experimental() // untriaged
  Stream<MouseEvent> get onMouseEnter =>
      Element.mouseEnterEvent.forTarget(this);

  /// Stream of `mouseleave` events handled by this [Document].
  @DomName('Document.onmouseleave')
  @DocsEditable()
  @Experimental() // untriaged
  Stream<MouseEvent> get onMouseLeave =>
      Element.mouseLeaveEvent.forTarget(this);

  /// Stream of `mousemove` events handled by this [Document].
  @DomName('Document.onmousemove')
  @DocsEditable()
  Stream<MouseEvent> get onMouseMove => Element.mouseMoveEvent.forTarget(this);

  /// Stream of `mouseout` events handled by this [Document].
  @DomName('Document.onmouseout')
  @DocsEditable()
  Stream<MouseEvent> get onMouseOut => Element.mouseOutEvent.forTarget(this);

  /// Stream of `mouseover` events handled by this [Document].
  @DomName('Document.onmouseover')
  @DocsEditable()
  Stream<MouseEvent> get onMouseOver => Element.mouseOverEvent.forTarget(this);

  /// Stream of `mouseup` events handled by this [Document].
  @DomName('Document.onmouseup')
  @DocsEditable()
  Stream<MouseEvent> get onMouseUp => Element.mouseUpEvent.forTarget(this);

  /// Stream of `mousewheel` events handled by this [Document].
  @DomName('Document.onmousewheel')
  @DocsEditable()
  Stream<WheelEvent> get onMouseWheel =>
      Element.mouseWheelEvent.forTarget(this);

  /// Stream of `paste` events handled by this [Document].
  @DomName('Document.onpaste')
  @DocsEditable()
  Stream<ClipboardEvent> get onPaste => Element.pasteEvent.forTarget(this);

  @DomName('Document.onpause')
  @DocsEditable()
  @Experimental() // untriaged
  Stream<Event> get onPause => Element.pauseEvent.forTarget(this);

  @DomName('Document.onplay')
  @DocsEditable()
  @Experimental() // untriaged
  Stream<Event> get onPlay => Element.playEvent.forTarget(this);

  @DomName('Document.onplaying')
  @DocsEditable()
  @Experimental() // untriaged
  Stream<Event> get onPlaying => Element.playingEvent.forTarget(this);

  @DomName('Document.onpointerlockchange')
  @DocsEditable()
  @Experimental() // untriaged
  Stream<Event> get onPointerLockChange =>
      pointerLockChangeEvent.forTarget(this);

  @DomName('Document.onpointerlockerror')
  @DocsEditable()
  @Experimental() // untriaged
  Stream<Event> get onPointerLockError => pointerLockErrorEvent.forTarget(this);

  @DomName('Document.onratechange')
  @DocsEditable()
  @Experimental() // untriaged
  Stream<Event> get onRateChange => Element.rateChangeEvent.forTarget(this);

  /// Stream of `readystatechange` events handled by this [Document].
  @DomName('Document.onreadystatechange')
  @DocsEditable()
  Stream<Event> get onReadyStateChange => readyStateChangeEvent.forTarget(this);

  /// Stream of `reset` events handled by this [Document].
  @DomName('Document.onreset')
  @DocsEditable()
  Stream<Event> get onReset => Element.resetEvent.forTarget(this);

  @DomName('Document.onresize')
  @DocsEditable()
  @Experimental() // untriaged
  Stream<Event> get onResize => Element.resizeEvent.forTarget(this);

  /// Stream of `scroll` events handled by this [Document].
  @DomName('Document.onscroll')
  @DocsEditable()
  Stream<Event> get onScroll => Element.scrollEvent.forTarget(this);

  /// Stream of `search` events handled by this [Document].
  @DomName('Document.onsearch')
  @DocsEditable()
  // http://www.w3.org/TR/html-markup/input.search.html
  @Experimental()
  Stream<Event> get onSearch => Element.searchEvent.forTarget(this);

  /// Stream of `securitypolicyviolation` events handled by this [Document].
  @DomName('Document.onsecuritypolicyviolation')
  @DocsEditable()
  // https://dvcs.w3.org/hg/content-security-policy/raw-file/tip/csp-specification.dev.html#widl-Document-onsecuritypolicyviolation
  @Experimental()
  Stream<SecurityPolicyViolationEvent> get onSecurityPolicyViolation =>
      securityPolicyViolationEvent.forTarget(this);

  @DomName('Document.onseeked')
  @DocsEditable()
  @Experimental() // untriaged
  Stream<Event> get onSeeked => Element.seekedEvent.forTarget(this);

  @DomName('Document.onseeking')
  @DocsEditable()
  @Experimental() // untriaged
  Stream<Event> get onSeeking => Element.seekingEvent.forTarget(this);

  /// Stream of `select` events handled by this [Document].
  @DomName('Document.onselect')
  @DocsEditable()
  Stream<Event> get onSelect => Element.selectEvent.forTarget(this);

  /// Stream of `selectionchange` events handled by this [Document].
  @DomName('Document.onselectionchange')
  @DocsEditable()
  Stream<Event> get onSelectionChange => selectionChangeEvent.forTarget(this);

  /// Stream of `selectstart` events handled by this [Document].
  @DomName('Document.onselectstart')
  @DocsEditable()
  Stream<Event> get onSelectStart => Element.selectStartEvent.forTarget(this);

  @DomName('Document.onstalled')
  @DocsEditable()
  @Experimental() // untriaged
  Stream<Event> get onStalled => Element.stalledEvent.forTarget(this);

  /// Stream of `submit` events handled by this [Document].
  @DomName('Document.onsubmit')
  @DocsEditable()
  Stream<Event> get onSubmit => Element.submitEvent.forTarget(this);

  @DomName('Document.onsuspend')
  @DocsEditable()
  @Experimental() // untriaged
  Stream<Event> get onSuspend => Element.suspendEvent.forTarget(this);

  @DomName('Document.ontimeupdate')
  @DocsEditable()
  @Experimental() // untriaged
  Stream<Event> get onTimeUpdate => Element.timeUpdateEvent.forTarget(this);

  /// Stream of `touchcancel` events handled by this [Document].
  @DomName('Document.ontouchcancel')
  @DocsEditable()
  // http://www.w3.org/TR/touch-events/, http://www.chromestatus.com/features
  @Experimental()
  Stream<TouchEvent> get onTouchCancel =>
      Element.touchCancelEvent.forTarget(this);

  /// Stream of `touchend` events handled by this [Document].
  @DomName('Document.ontouchend')
  @DocsEditable()
  // http://www.w3.org/TR/touch-events/, http://www.chromestatus.com/features
  @Experimental()
  Stream<TouchEvent> get onTouchEnd => Element.touchEndEvent.forTarget(this);

  /// Stream of `touchmove` events handled by this [Document].
  @DomName('Document.ontouchmove')
  @DocsEditable()
  // http://www.w3.org/TR/touch-events/, http://www.chromestatus.com/features
  @Experimental()
  Stream<TouchEvent> get onTouchMove => Element.touchMoveEvent.forTarget(this);

  /// Stream of `touchstart` events handled by this [Document].
  @DomName('Document.ontouchstart')
  @DocsEditable()
  // http://www.w3.org/TR/touch-events/, http://www.chromestatus.com/features
  @Experimental()
  Stream<TouchEvent> get onTouchStart =>
      Element.touchStartEvent.forTarget(this);

  @DomName('Document.onvolumechange')
  @DocsEditable()
  @Experimental() // untriaged
  Stream<Event> get onVolumeChange => Element.volumeChangeEvent.forTarget(this);

  @DomName('Document.onwaiting')
  @DocsEditable()
  @Experimental() // untriaged
  Stream<Event> get onWaiting => Element.waitingEvent.forTarget(this);

  /// Stream of `fullscreenchange` events handled by this [Document].
  @DomName('Document.onwebkitfullscreenchange')
  @DocsEditable()
  // https://dvcs.w3.org/hg/fullscreen/raw-file/tip/Overview.html
  @Experimental()
  Stream<Event> get onFullscreenChange =>
      Element.fullscreenChangeEvent.forTarget(this);

  /// Stream of `fullscreenerror` events handled by this [Document].
  @DomName('Document.onwebkitfullscreenerror')
  @DocsEditable()
  // https://dvcs.w3.org/hg/fullscreen/raw-file/tip/Overview.html
  @Experimental()
  Stream<Event> get onFullscreenError =>
      Element.fullscreenErrorEvent.forTarget(this);

  /**
   * Finds all descendant elements of this document that match the specified
   * group of selectors.
   *
   * Unless your webpage contains multiple documents, the top-level
   * [querySelectorAll]
   * method behaves the same as this method, so you should use it instead to
   * save typing a few characters.
   *
   * [selectors] should be a string using CSS selector syntax.
   *
   *     var items = document.querySelectorAll('.itemClassName');
   *
   * For details about CSS selector syntax, see the
   * [CSS selector specification](http://www.w3.org/TR/css3-selectors/).
   */
  ElementList<Element/*=T*/ > querySelectorAll/*<T extends Element>*/(
          String selectors) =>
      new _FrozenElementList/*<T>*/ ._wrap(_querySelectorAll(selectors));

  /**
   * Alias for [querySelector]. Note this function is deprecated because its
   * semantics will be changing in the future.
   */
  @deprecated
  @Experimental()
  @DomName('Document.querySelector')
  Element query(String relativeSelectors) => querySelector(relativeSelectors);

  /**
   * Alias for [querySelectorAll]. Note this function is deprecated because its
   * semantics will be changing in the future.
   */
  @deprecated
  @Experimental()
  @DomName('Document.querySelectorAll')
  ElementList<Element/*=T*/ > queryAll/*<T extends Element>*/(
          String relativeSelectors) =>
      querySelectorAll(relativeSelectors);

  /// Checks if [registerElement] is supported on the current platform.
  bool get supportsRegisterElement {
    return true;
  }

  /// *Deprecated*: use [supportsRegisterElement] instead.
  @deprecated
  bool get supportsRegister => supportsRegisterElement;

  @DomName('Document.createElement')
  Element createElement(String tagName, [String typeExtension]) {
    return (typeExtension == null)
        ? _blink.BlinkDocument.instance.createElement_Callback_1_(this, tagName)
        : _blink.BlinkDocument.instance
            .createElement_Callback_2_(this, tagName, typeExtension);
  }

  @DomName('Document.createElementNS')
  @DocsEditable()
  Element createElementNS(String namespaceURI, String qualifiedName,
      [String typeExtension]) {
    return (typeExtension == null)
        ? _blink.BlinkDocument.instance
            .createElementNS_Callback_2_(this, namespaceURI, qualifiedName)
        : _blink.BlinkDocument.instance.createElementNS_Callback_3_(
            this, namespaceURI, qualifiedName, typeExtension);
  }

  @DomName('Document.visibilityState')
  @SupportedBrowser(SupportedBrowser.CHROME)
  @SupportedBrowser(SupportedBrowser.FIREFOX)
  @SupportedBrowser(SupportedBrowser.IE, '10')
  @Experimental()
  String get visibilityState => _visibilityState;
}
// Copyright (c) 2011, the Dart project authors.  Please see the AUTHORS file
// for details. All rights reserved. Use of this source code is governed by a
// BSD-style license that can be found in the LICENSE file.

@DomName('DocumentFragment')
class DocumentFragment extends Node
    implements NonElementParentNode, ParentNode {
  factory DocumentFragment() => document.createDocumentFragment();

  factory DocumentFragment.html(String html,
      {NodeValidator validator, NodeTreeSanitizer treeSanitizer}) {
    return document.body.createFragment(html,
        validator: validator, treeSanitizer: treeSanitizer);
  }

  factory DocumentFragment.svg(String svgContent,
      {NodeValidator validator, NodeTreeSanitizer treeSanitizer}) {
    return new svg.SvgSvgElement().createFragment(svgContent,
        validator: validator, treeSanitizer: treeSanitizer);
  }

  HtmlCollection get _children =>
      throw new UnimplementedError('Use _docChildren instead');

  List<Element> _docChildren;

  List<Element> get children {
    if (_docChildren == null) {
      _docChildren = new FilteredElementList(this);
    }
    return _docChildren;
  }

  set children(List<Element> value) {
    // Copy list first since we don't want liveness during iteration.
    var copy = value.toList();
    var children = this.children;
    children.clear();
    children.addAll(copy);
  }

  /**
   * Finds all descendant elements of this document fragment that match the
   * specified group of selectors.
   *
   * [selectors] should be a string using CSS selector syntax.
   *
   *     var items = document.querySelectorAll('.itemClassName');
   *
   * For details about CSS selector syntax, see the
   * [CSS selector specification](http://www.w3.org/TR/css3-selectors/).
   */
  ElementList<Element/*=T*/ > querySelectorAll/*<T extends Element>*/(
          String selectors) =>
      new _FrozenElementList/*<T>*/ ._wrap(_querySelectorAll(selectors));

  String get innerHtml {
    final e = new DivElement();
    e.append(this.clone(true));
    return e.innerHtml;
  }

  set innerHtml(String value) {
    this.setInnerHtml(value);
  }

  void setInnerHtml(String html,
      {NodeValidator validator, NodeTreeSanitizer treeSanitizer}) {
    this.nodes.clear();
    append(document.body.createFragment(html,
        validator: validator, treeSanitizer: treeSanitizer));
  }

  /**
   * Adds the specified text as a text node after the last child of this
   * document fragment.
   */
  void appendText(String text) {
    this.append(new Text(text));
  }

  /**
   * Parses the specified text as HTML and adds the resulting node after the
   * last child of this document fragment.
   */
  void appendHtml(String text,
      {NodeValidator validator, NodeTreeSanitizer, treeSanitizer}) {
    this.append(new DocumentFragment.html(text,
        validator: validator, treeSanitizer: treeSanitizer));
  }

  /** 
   * Alias for [querySelector]. Note this function is deprecated because its
   * semantics will be changing in the future.
   */
  @deprecated
  @Experimental()
  @DomName('DocumentFragment.querySelector')
  Element query(String relativeSelectors) {
    return querySelector(relativeSelectors);
  }

  /** 
   * Alias for [querySelectorAll]. Note this function is deprecated because its
   * semantics will be changing in the future.
   */
  @deprecated
  @Experimental()
  @DomName('DocumentFragment.querySelectorAll')
  ElementList<Element/*=T*/ > queryAll/*<T extends Element>*/(
          String relativeSelectors) =>
      querySelectorAll(relativeSelectors);
  // To suppress missing implicit constructor warnings.
  factory DocumentFragment._() {
    throw new UnsupportedError("Not supported");
  }

  @Deprecated("Internal Use Only")
  external static Type get instanceRuntimeType;

  @Deprecated("Internal Use Only")
  DocumentFragment.internal_() : super.internal_();

  @DomName('DocumentFragment.getElementById')
  @DocsEditable()
  @Experimental() // untriaged
  Element getElementById(String elementId) =>
      _blink.BlinkDocumentFragment.instance
          .getElementById_Callback_1_(this, elementId);

  @DomName('DocumentFragment.childElementCount')
  @DocsEditable()
  int get _childElementCount =>
      _blink.BlinkDocumentFragment.instance.childElementCount_Getter_(this);

  @DomName('DocumentFragment.firstElementChild')
  @DocsEditable()
  Element get _firstElementChild =>
      _blink.BlinkDocumentFragment.instance.firstElementChild_Getter_(this);

  @DomName('DocumentFragment.lastElementChild')
  @DocsEditable()
  Element get _lastElementChild =>
      _blink.BlinkDocumentFragment.instance.lastElementChild_Getter_(this);

  /**
   * Finds the first descendant element of this document fragment that matches
   * the specified group of selectors.
   *
   * [selectors] should be a string using CSS selector syntax.
   *
   *     var element1 = fragment.querySelector('.className');
   *     var element2 = fragment.querySelector('#id');
   *
   * For details about CSS selector syntax, see the
   * [CSS selector specification](http://www.w3.org/TR/css3-selectors/).
   */
  @DomName('DocumentFragment.querySelector')
  @DocsEditable()
  Element querySelector(String selectors) =>
      _blink.BlinkDocumentFragment.instance
          .querySelector_Callback_1_(this, selectors);

  @DomName('DocumentFragment.querySelectorAll')
  @DocsEditable()
  List<Node> _querySelectorAll(String selectors) =>
      (_blink.BlinkDocumentFragment.instance
          .querySelectorAll_Callback_1_(this, selectors));
}
// Copyright (c) 2012, the Dart project authors.  Please see the AUTHORS file
// for details. All rights reserved. Use of this source code is governed by a
// BSD-style license that can be found in the LICENSE file.

// WARNING: Do not edit - generated code.

@DocsEditable()
@DomName('DOMError')
class DomError extends DartHtmlDomObject {
  // To suppress missing implicit constructor warnings.
  factory DomError._() {
    throw new UnsupportedError("Not supported");
  }

  @DomName('DOMError.DOMError')
  @DocsEditable()
  factory DomError(String name, [String message]) {
    return _blink.BlinkDOMError.instance.constructorCallback_2_(name, message);
  }

  @Deprecated("Internal Use Only")
  external static Type get instanceRuntimeType;

  @Deprecated("Internal Use Only")
  DomError.internal_() {}

  @DomName('DOMError.message')
  @DocsEditable()
  @Experimental() // untriaged
  String get message => _blink.BlinkDOMError.instance.message_Getter_(this);

  @DomName('DOMError.name')
  @DocsEditable()
  String get name => _blink.BlinkDOMError.instance.name_Getter_(this);
}
// Copyright (c) 2013, the Dart project authors.  Please see the AUTHORS file
// for details. All rights reserved. Use of this source code is governed by a
// BSD-style license that can be found in the LICENSE file.

@DomName('DOMException')
@Unstable()
class DomException extends DartHtmlDomObject {
  static const String INDEX_SIZE = 'IndexSizeError';
  static const String HIERARCHY_REQUEST = 'HierarchyRequestError';
  static const String WRONG_DOCUMENT = 'WrongDocumentError';
  static const String INVALID_CHARACTER = 'InvalidCharacterError';
  static const String NO_MODIFICATION_ALLOWED = 'NoModificationAllowedError';
  static const String NOT_FOUND = 'NotFoundError';
  static const String NOT_SUPPORTED = 'NotSupportedError';
  static const String INVALID_STATE = 'InvalidStateError';
  static const String SYNTAX = 'SyntaxError';
  static const String INVALID_MODIFICATION = 'InvalidModificationError';
  static const String NAMESPACE = 'NamespaceError';
  static const String INVALID_ACCESS = 'InvalidAccessError';
  static const String TYPE_MISMATCH = 'TypeMismatchError';
  static const String SECURITY = 'SecurityError';
  static const String NETWORK = 'NetworkError';
  static const String ABORT = 'AbortError';
  static const String URL_MISMATCH = 'URLMismatchError';
  static const String QUOTA_EXCEEDED = 'QuotaExceededError';
  static const String TIMEOUT = 'TimeoutError';
  static const String INVALID_NODE_TYPE = 'InvalidNodeTypeError';
  static const String DATA_CLONE = 'DataCloneError';

  String _name;
  String _message;

  // To suppress missing implicit constructor warnings.
  factory DomException._() {
    throw new UnsupportedError("Not supported");
  }

  @Deprecated("Internal Use Only")
  static DomException internalCreateDomException() {
    return new DomException._internalWrap();
  }

  external factory DomException._internalWrap();

  @Deprecated("Internal Use Only")
  DomException.internal_() {}

  @Deprecated("Internal Use Only")
  DomException.jsInterop(String m) {
    var name_index = m.indexOf(': ');
    if (name_index < 0) {
      _name = "";
      _message = m;
    } else {
      _name = m.substring(0, name_index);
      _message = m.substring(name_index + 1).trim();
    }
  }

  @DomName('DOMException.message')
  @DocsEditable()
  String get message => _message;

  @DomName('DOMException.name')
  @DocsEditable()
  String get name => _name;

  @DomName('DOMException.toString')
  @DocsEditable()
  String toString() => "${_name}: $_message";
}
// Copyright (c) 2012, the Dart project authors.  Please see the AUTHORS file
// for details. All rights reserved. Use of this source code is governed by a
// BSD-style license that can be found in the LICENSE file.

// WARNING: Do not edit - generated code.

@DocsEditable()
@DomName('DOMImplementation')
class DomImplementation extends DartHtmlDomObject {
  // To suppress missing implicit constructor warnings.
  factory DomImplementation._() {
    throw new UnsupportedError("Not supported");
  }

  @Deprecated("Internal Use Only")
  external static Type get instanceRuntimeType;

  @Deprecated("Internal Use Only")
  DomImplementation.internal_() {}

  @DomName('DOMImplementation.createDocument')
  @DocsEditable()
  XmlDocument createDocument(
          String namespaceURI, String qualifiedName, _DocumentType doctype) =>
      _blink.BlinkDOMImplementation.instance.createDocument_Callback_3_(
          this, namespaceURI, qualifiedName, doctype);

  @DomName('DOMImplementation.createDocumentType')
  @DocsEditable()
  _DocumentType createDocumentType(
          String qualifiedName, String publicId, String systemId) =>
      _blink.BlinkDOMImplementation.instance.createDocumentType_Callback_3_(
          this, qualifiedName, publicId, systemId);

  @DomName('DOMImplementation.createHTMLDocument')
  @DocsEditable()
  HtmlDocument createHtmlDocument(String title) =>
      _blink.BlinkDOMImplementation.instance
          .createHTMLDocument_Callback_1_(this, title);

  @DomName('DOMImplementation.hasFeature')
  @DocsEditable()
  bool hasFeature() =>
      _blink.BlinkDOMImplementation.instance.hasFeature_Callback_0_(this);
}
// Copyright (c) 2012, the Dart project authors.  Please see the AUTHORS file
// for details. All rights reserved. Use of this source code is governed by a
// BSD-style license that can be found in the LICENSE file.

// WARNING: Do not edit - generated code.

@DocsEditable()
@DomName('Iterator')
@Experimental() // untriaged
class DomIterator extends DartHtmlDomObject {
  // To suppress missing implicit constructor warnings.
  factory DomIterator._() {
    throw new UnsupportedError("Not supported");
  }

  @Deprecated("Internal Use Only")
  external static Type get instanceRuntimeType;

  @Deprecated("Internal Use Only")
  DomIterator.internal_() {}

  Object next([Object value]) {
    if (value != null) {
      return _blink.BlinkIterator.instance.next_Callback_1_(this, value);
    }
    return _blink.BlinkIterator.instance.next_Callback_0_(this);
  }
}
// Copyright (c) 2012, the Dart project authors.  Please see the AUTHORS file
// for details. All rights reserved. Use of this source code is governed by a
// BSD-style license that can be found in the LICENSE file.

// WARNING: Do not edit - generated code.

@DocsEditable()
@DomName('DOMMatrix')
@Experimental() // untriaged
class DomMatrix extends DomMatrixReadOnly {
  // To suppress missing implicit constructor warnings.
  factory DomMatrix._() {
    throw new UnsupportedError("Not supported");
  }

  @DomName('DOMMatrix.DOMMatrix')
  @DocsEditable()
  factory DomMatrix([DomMatrixReadOnly other]) {
    if (other == null) {
      return _blink.BlinkDOMMatrix.instance.constructorCallback_0_();
    }
    if ((other is DomMatrixReadOnly || other == null)) {
      return _blink.BlinkDOMMatrix.instance.constructorCallback_1_(other);
    }
    throw new ArgumentError("Incorrect number or type of arguments");
  }

  @Deprecated("Internal Use Only")
  external static Type get instanceRuntimeType;

  @Deprecated("Internal Use Only")
  DomMatrix.internal_() : super.internal_();

  @DomName('DOMMatrix.a')
  @DocsEditable()
  @Experimental() // untriaged
  num get a => _blink.BlinkDOMMatrix.instance.a_Getter_(this);

  @DomName('DOMMatrix.a')
  @DocsEditable()
  @Experimental() // untriaged
  set a(num value) => _blink.BlinkDOMMatrix.instance.a_Setter_(this, value);

  @DomName('DOMMatrix.b')
  @DocsEditable()
  @Experimental() // untriaged
  num get b => _blink.BlinkDOMMatrix.instance.b_Getter_(this);

  @DomName('DOMMatrix.b')
  @DocsEditable()
  @Experimental() // untriaged
  set b(num value) => _blink.BlinkDOMMatrix.instance.b_Setter_(this, value);

  @DomName('DOMMatrix.c')
  @DocsEditable()
  @Experimental() // untriaged
  num get c => _blink.BlinkDOMMatrix.instance.c_Getter_(this);

  @DomName('DOMMatrix.c')
  @DocsEditable()
  @Experimental() // untriaged
  set c(num value) => _blink.BlinkDOMMatrix.instance.c_Setter_(this, value);

  @DomName('DOMMatrix.d')
  @DocsEditable()
  @Experimental() // untriaged
  num get d => _blink.BlinkDOMMatrix.instance.d_Getter_(this);

  @DomName('DOMMatrix.d')
  @DocsEditable()
  @Experimental() // untriaged
  set d(num value) => _blink.BlinkDOMMatrix.instance.d_Setter_(this, value);

  @DomName('DOMMatrix.e')
  @DocsEditable()
  @Experimental() // untriaged
  num get e => _blink.BlinkDOMMatrix.instance.e_Getter_(this);

  @DomName('DOMMatrix.e')
  @DocsEditable()
  @Experimental() // untriaged
  set e(num value) => _blink.BlinkDOMMatrix.instance.e_Setter_(this, value);

  @DomName('DOMMatrix.f')
  @DocsEditable()
  @Experimental() // untriaged
  num get f => _blink.BlinkDOMMatrix.instance.f_Getter_(this);

  @DomName('DOMMatrix.f')
  @DocsEditable()
  @Experimental() // untriaged
  set f(num value) => _blink.BlinkDOMMatrix.instance.f_Setter_(this, value);

  @DomName('DOMMatrix.m11')
  @DocsEditable()
  @Experimental() // untriaged
  num get m11 => _blink.BlinkDOMMatrix.instance.m11_Getter_(this);

  @DomName('DOMMatrix.m11')
  @DocsEditable()
  @Experimental() // untriaged
  set m11(num value) => _blink.BlinkDOMMatrix.instance.m11_Setter_(this, value);

  @DomName('DOMMatrix.m12')
  @DocsEditable()
  @Experimental() // untriaged
  num get m12 => _blink.BlinkDOMMatrix.instance.m12_Getter_(this);

  @DomName('DOMMatrix.m12')
  @DocsEditable()
  @Experimental() // untriaged
  set m12(num value) => _blink.BlinkDOMMatrix.instance.m12_Setter_(this, value);

  @DomName('DOMMatrix.m13')
  @DocsEditable()
  @Experimental() // untriaged
  num get m13 => _blink.BlinkDOMMatrix.instance.m13_Getter_(this);

  @DomName('DOMMatrix.m13')
  @DocsEditable()
  @Experimental() // untriaged
  set m13(num value) => _blink.BlinkDOMMatrix.instance.m13_Setter_(this, value);

  @DomName('DOMMatrix.m14')
  @DocsEditable()
  @Experimental() // untriaged
  num get m14 => _blink.BlinkDOMMatrix.instance.m14_Getter_(this);

  @DomName('DOMMatrix.m14')
  @DocsEditable()
  @Experimental() // untriaged
  set m14(num value) => _blink.BlinkDOMMatrix.instance.m14_Setter_(this, value);

  @DomName('DOMMatrix.m21')
  @DocsEditable()
  @Experimental() // untriaged
  num get m21 => _blink.BlinkDOMMatrix.instance.m21_Getter_(this);

  @DomName('DOMMatrix.m21')
  @DocsEditable()
  @Experimental() // untriaged
  set m21(num value) => _blink.BlinkDOMMatrix.instance.m21_Setter_(this, value);

  @DomName('DOMMatrix.m22')
  @DocsEditable()
  @Experimental() // untriaged
  num get m22 => _blink.BlinkDOMMatrix.instance.m22_Getter_(this);

  @DomName('DOMMatrix.m22')
  @DocsEditable()
  @Experimental() // untriaged
  set m22(num value) => _blink.BlinkDOMMatrix.instance.m22_Setter_(this, value);

  @DomName('DOMMatrix.m23')
  @DocsEditable()
  @Experimental() // untriaged
  num get m23 => _blink.BlinkDOMMatrix.instance.m23_Getter_(this);

  @DomName('DOMMatrix.m23')
  @DocsEditable()
  @Experimental() // untriaged
  set m23(num value) => _blink.BlinkDOMMatrix.instance.m23_Setter_(this, value);

  @DomName('DOMMatrix.m24')
  @DocsEditable()
  @Experimental() // untriaged
  num get m24 => _blink.BlinkDOMMatrix.instance.m24_Getter_(this);

  @DomName('DOMMatrix.m24')
  @DocsEditable()
  @Experimental() // untriaged
  set m24(num value) => _blink.BlinkDOMMatrix.instance.m24_Setter_(this, value);

  @DomName('DOMMatrix.m31')
  @DocsEditable()
  @Experimental() // untriaged
  num get m31 => _blink.BlinkDOMMatrix.instance.m31_Getter_(this);

  @DomName('DOMMatrix.m31')
  @DocsEditable()
  @Experimental() // untriaged
  set m31(num value) => _blink.BlinkDOMMatrix.instance.m31_Setter_(this, value);

  @DomName('DOMMatrix.m32')
  @DocsEditable()
  @Experimental() // untriaged
  num get m32 => _blink.BlinkDOMMatrix.instance.m32_Getter_(this);

  @DomName('DOMMatrix.m32')
  @DocsEditable()
  @Experimental() // untriaged
  set m32(num value) => _blink.BlinkDOMMatrix.instance.m32_Setter_(this, value);

  @DomName('DOMMatrix.m33')
  @DocsEditable()
  @Experimental() // untriaged
  num get m33 => _blink.BlinkDOMMatrix.instance.m33_Getter_(this);

  @DomName('DOMMatrix.m33')
  @DocsEditable()
  @Experimental() // untriaged
  set m33(num value) => _blink.BlinkDOMMatrix.instance.m33_Setter_(this, value);

  @DomName('DOMMatrix.m34')
  @DocsEditable()
  @Experimental() // untriaged
  num get m34 => _blink.BlinkDOMMatrix.instance.m34_Getter_(this);

  @DomName('DOMMatrix.m34')
  @DocsEditable()
  @Experimental() // untriaged
  set m34(num value) => _blink.BlinkDOMMatrix.instance.m34_Setter_(this, value);

  @DomName('DOMMatrix.m41')
  @DocsEditable()
  @Experimental() // untriaged
  num get m41 => _blink.BlinkDOMMatrix.instance.m41_Getter_(this);

  @DomName('DOMMatrix.m41')
  @DocsEditable()
  @Experimental() // untriaged
  set m41(num value) => _blink.BlinkDOMMatrix.instance.m41_Setter_(this, value);

  @DomName('DOMMatrix.m42')
  @DocsEditable()
  @Experimental() // untriaged
  num get m42 => _blink.BlinkDOMMatrix.instance.m42_Getter_(this);

  @DomName('DOMMatrix.m42')
  @DocsEditable()
  @Experimental() // untriaged
  set m42(num value) => _blink.BlinkDOMMatrix.instance.m42_Setter_(this, value);

  @DomName('DOMMatrix.m43')
  @DocsEditable()
  @Experimental() // untriaged
  num get m43 => _blink.BlinkDOMMatrix.instance.m43_Getter_(this);

  @DomName('DOMMatrix.m43')
  @DocsEditable()
  @Experimental() // untriaged
  set m43(num value) => _blink.BlinkDOMMatrix.instance.m43_Setter_(this, value);

  @DomName('DOMMatrix.m44')
  @DocsEditable()
  @Experimental() // untriaged
  num get m44 => _blink.BlinkDOMMatrix.instance.m44_Getter_(this);

  @DomName('DOMMatrix.m44')
  @DocsEditable()
  @Experimental() // untriaged
  set m44(num value) => _blink.BlinkDOMMatrix.instance.m44_Setter_(this, value);

  @DomName('DOMMatrix.multiplySelf')
  @DocsEditable()
  @Experimental() // untriaged
  DomMatrix multiplySelf(DomMatrix other) =>
      _blink.BlinkDOMMatrix.instance.multiplySelf_Callback_1_(this, other);

  @DomName('DOMMatrix.preMultiplySelf')
  @DocsEditable()
  @Experimental() // untriaged
  DomMatrix preMultiplySelf(DomMatrix other) =>
      _blink.BlinkDOMMatrix.instance.preMultiplySelf_Callback_1_(this, other);

  DomMatrix scale3dSelf(num scale, [num originX, num originY, num originZ]) {
    if (originZ != null) {
      return _blink.BlinkDOMMatrix.instance
          .scale3dSelf_Callback_4_(this, scale, originX, originY, originZ);
    }
    if (originY != null) {
      return _blink.BlinkDOMMatrix.instance
          .scale3dSelf_Callback_3_(this, scale, originX, originY);
    }
    if (originX != null) {
      return _blink.BlinkDOMMatrix.instance
          .scale3dSelf_Callback_2_(this, scale, originX);
    }
    return _blink.BlinkDOMMatrix.instance.scale3dSelf_Callback_1_(this, scale);
  }

  DomMatrix scaleNonUniformSelf(num scaleX,
      [num scaleY, num scaleZ, num originX, num originY, num originZ]) {
    if (originZ != null) {
      return _blink.BlinkDOMMatrix.instance.scaleNonUniformSelf_Callback_6_(
          this, scaleX, scaleY, scaleZ, originX, originY, originZ);
    }
    if (originY != null) {
      return _blink.BlinkDOMMatrix.instance.scaleNonUniformSelf_Callback_5_(
          this, scaleX, scaleY, scaleZ, originX, originY);
    }
    if (originX != null) {
      return _blink.BlinkDOMMatrix.instance.scaleNonUniformSelf_Callback_4_(
          this, scaleX, scaleY, scaleZ, originX);
    }
    if (scaleZ != null) {
      return _blink.BlinkDOMMatrix.instance
          .scaleNonUniformSelf_Callback_3_(this, scaleX, scaleY, scaleZ);
    }
    if (scaleY != null) {
      return _blink.BlinkDOMMatrix.instance
          .scaleNonUniformSelf_Callback_2_(this, scaleX, scaleY);
    }
    return _blink.BlinkDOMMatrix.instance
        .scaleNonUniformSelf_Callback_1_(this, scaleX);
  }

  DomMatrix scaleSelf(num scale, [num originX, num originY]) {
    if (originY != null) {
      return _blink.BlinkDOMMatrix.instance
          .scaleSelf_Callback_3_(this, scale, originX, originY);
    }
    if (originX != null) {
      return _blink.BlinkDOMMatrix.instance
          .scaleSelf_Callback_2_(this, scale, originX);
    }
    return _blink.BlinkDOMMatrix.instance.scaleSelf_Callback_1_(this, scale);
  }

  DomMatrix translateSelf(num tx, num ty, [num tz]) {
    if (tz != null) {
      return _blink.BlinkDOMMatrix.instance
          .translateSelf_Callback_3_(this, tx, ty, tz);
    }
    return _blink.BlinkDOMMatrix.instance
        .translateSelf_Callback_2_(this, tx, ty);
  }
}
// Copyright (c) 2012, the Dart project authors.  Please see the AUTHORS file
// for details. All rights reserved. Use of this source code is governed by a
// BSD-style license that can be found in the LICENSE file.

// WARNING: Do not edit - generated code.

@DocsEditable()
@DomName('DOMMatrixReadOnly')
@Experimental() // untriaged
class DomMatrixReadOnly extends DartHtmlDomObject {
  // To suppress missing implicit constructor warnings.
  factory DomMatrixReadOnly._() {
    throw new UnsupportedError("Not supported");
  }

  @Deprecated("Internal Use Only")
  external static Type get instanceRuntimeType;

  @Deprecated("Internal Use Only")
  DomMatrixReadOnly.internal_() {}

  @DomName('DOMMatrixReadOnly.a')
  @DocsEditable()
  @Experimental() // untriaged
  num get a => _blink.BlinkDOMMatrixReadOnly.instance.a_Getter_(this);

  @DomName('DOMMatrixReadOnly.b')
  @DocsEditable()
  @Experimental() // untriaged
  num get b => _blink.BlinkDOMMatrixReadOnly.instance.b_Getter_(this);

  @DomName('DOMMatrixReadOnly.c')
  @DocsEditable()
  @Experimental() // untriaged
  num get c => _blink.BlinkDOMMatrixReadOnly.instance.c_Getter_(this);

  @DomName('DOMMatrixReadOnly.d')
  @DocsEditable()
  @Experimental() // untriaged
  num get d => _blink.BlinkDOMMatrixReadOnly.instance.d_Getter_(this);

  @DomName('DOMMatrixReadOnly.e')
  @DocsEditable()
  @Experimental() // untriaged
  num get e => _blink.BlinkDOMMatrixReadOnly.instance.e_Getter_(this);

  @DomName('DOMMatrixReadOnly.f')
  @DocsEditable()
  @Experimental() // untriaged
  num get f => _blink.BlinkDOMMatrixReadOnly.instance.f_Getter_(this);

  @DomName('DOMMatrixReadOnly.is2D')
  @DocsEditable()
  @Experimental() // untriaged
  bool get is2D => _blink.BlinkDOMMatrixReadOnly.instance.is2D_Getter_(this);

  @DomName('DOMMatrixReadOnly.isIdentity')
  @DocsEditable()
  @Experimental() // untriaged
  bool get isIdentity =>
      _blink.BlinkDOMMatrixReadOnly.instance.isIdentity_Getter_(this);

  @DomName('DOMMatrixReadOnly.m11')
  @DocsEditable()
  @Experimental() // untriaged
  num get m11 => _blink.BlinkDOMMatrixReadOnly.instance.m11_Getter_(this);

  @DomName('DOMMatrixReadOnly.m12')
  @DocsEditable()
  @Experimental() // untriaged
  num get m12 => _blink.BlinkDOMMatrixReadOnly.instance.m12_Getter_(this);

  @DomName('DOMMatrixReadOnly.m13')
  @DocsEditable()
  @Experimental() // untriaged
  num get m13 => _blink.BlinkDOMMatrixReadOnly.instance.m13_Getter_(this);

  @DomName('DOMMatrixReadOnly.m14')
  @DocsEditable()
  @Experimental() // untriaged
  num get m14 => _blink.BlinkDOMMatrixReadOnly.instance.m14_Getter_(this);

  @DomName('DOMMatrixReadOnly.m21')
  @DocsEditable()
  @Experimental() // untriaged
  num get m21 => _blink.BlinkDOMMatrixReadOnly.instance.m21_Getter_(this);

  @DomName('DOMMatrixReadOnly.m22')
  @DocsEditable()
  @Experimental() // untriaged
  num get m22 => _blink.BlinkDOMMatrixReadOnly.instance.m22_Getter_(this);

  @DomName('DOMMatrixReadOnly.m23')
  @DocsEditable()
  @Experimental() // untriaged
  num get m23 => _blink.BlinkDOMMatrixReadOnly.instance.m23_Getter_(this);

  @DomName('DOMMatrixReadOnly.m24')
  @DocsEditable()
  @Experimental() // untriaged
  num get m24 => _blink.BlinkDOMMatrixReadOnly.instance.m24_Getter_(this);

  @DomName('DOMMatrixReadOnly.m31')
  @DocsEditable()
  @Experimental() // untriaged
  num get m31 => _blink.BlinkDOMMatrixReadOnly.instance.m31_Getter_(this);

  @DomName('DOMMatrixReadOnly.m32')
  @DocsEditable()
  @Experimental() // untriaged
  num get m32 => _blink.BlinkDOMMatrixReadOnly.instance.m32_Getter_(this);

  @DomName('DOMMatrixReadOnly.m33')
  @DocsEditable()
  @Experimental() // untriaged
  num get m33 => _blink.BlinkDOMMatrixReadOnly.instance.m33_Getter_(this);

  @DomName('DOMMatrixReadOnly.m34')
  @DocsEditable()
  @Experimental() // untriaged
  num get m34 => _blink.BlinkDOMMatrixReadOnly.instance.m34_Getter_(this);

  @DomName('DOMMatrixReadOnly.m41')
  @DocsEditable()
  @Experimental() // untriaged
  num get m41 => _blink.BlinkDOMMatrixReadOnly.instance.m41_Getter_(this);

  @DomName('DOMMatrixReadOnly.m42')
  @DocsEditable()
  @Experimental() // untriaged
  num get m42 => _blink.BlinkDOMMatrixReadOnly.instance.m42_Getter_(this);

  @DomName('DOMMatrixReadOnly.m43')
  @DocsEditable()
  @Experimental() // untriaged
  num get m43 => _blink.BlinkDOMMatrixReadOnly.instance.m43_Getter_(this);

  @DomName('DOMMatrixReadOnly.m44')
  @DocsEditable()
  @Experimental() // untriaged
  num get m44 => _blink.BlinkDOMMatrixReadOnly.instance.m44_Getter_(this);

  @DomName('DOMMatrixReadOnly.multiply')
  @DocsEditable()
  @Experimental() // untriaged
  DomMatrix multiply(DomMatrix other) =>
      _blink.BlinkDOMMatrixReadOnly.instance.multiply_Callback_1_(this, other);

  DomMatrix scale(num scale, [num originX, num originY]) {
    if (originY != null) {
      return _blink.BlinkDOMMatrixReadOnly.instance
          .scale_Callback_3_(this, scale, originX, originY);
    }
    if (originX != null) {
      return _blink.BlinkDOMMatrixReadOnly.instance
          .scale_Callback_2_(this, scale, originX);
    }
    return _blink.BlinkDOMMatrixReadOnly.instance
        .scale_Callback_1_(this, scale);
  }

  DomMatrix scale3d(num scale, [num originX, num originY, num originZ]) {
    if (originZ != null) {
      return _blink.BlinkDOMMatrixReadOnly.instance
          .scale3d_Callback_4_(this, scale, originX, originY, originZ);
    }
    if (originY != null) {
      return _blink.BlinkDOMMatrixReadOnly.instance
          .scale3d_Callback_3_(this, scale, originX, originY);
    }
    if (originX != null) {
      return _blink.BlinkDOMMatrixReadOnly.instance
          .scale3d_Callback_2_(this, scale, originX);
    }
    return _blink.BlinkDOMMatrixReadOnly.instance
        .scale3d_Callback_1_(this, scale);
  }

  DomMatrix scaleNonUniform(num scaleX,
      [num scaleY, num scaleZn, num originX, num originY, num originZ]) {
    if (originZ != null) {
      return _blink.BlinkDOMMatrixReadOnly.instance.scaleNonUniform_Callback_6_(
          this, scaleX, scaleY, scaleZn, originX, originY, originZ);
    }
    if (originY != null) {
      return _blink.BlinkDOMMatrixReadOnly.instance.scaleNonUniform_Callback_5_(
          this, scaleX, scaleY, scaleZn, originX, originY);
    }
    if (originX != null) {
      return _blink.BlinkDOMMatrixReadOnly.instance
          .scaleNonUniform_Callback_4_(this, scaleX, scaleY, scaleZn, originX);
    }
    if (scaleZn != null) {
      return _blink.BlinkDOMMatrixReadOnly.instance
          .scaleNonUniform_Callback_3_(this, scaleX, scaleY, scaleZn);
    }
    if (scaleY != null) {
      return _blink.BlinkDOMMatrixReadOnly.instance
          .scaleNonUniform_Callback_2_(this, scaleX, scaleY);
    }
    return _blink.BlinkDOMMatrixReadOnly.instance
        .scaleNonUniform_Callback_1_(this, scaleX);
  }

  @DomName('DOMMatrixReadOnly.toFloat32Array')
  @DocsEditable()
  @Experimental() // untriaged
  Float32List toFloat32Array() =>
      _blink.BlinkDOMMatrixReadOnly.instance.toFloat32Array_Callback_0_(this);

  @DomName('DOMMatrixReadOnly.toFloat64Array')
  @DocsEditable()
  @Experimental() // untriaged
  Float64List toFloat64Array() =>
      _blink.BlinkDOMMatrixReadOnly.instance.toFloat64Array_Callback_0_(this);

  DomMatrix translate(num tx, num ty, [num tz]) {
    if (tz != null) {
      return _blink.BlinkDOMMatrixReadOnly.instance
          .translate_Callback_3_(this, tx, ty, tz);
    }
    return _blink.BlinkDOMMatrixReadOnly.instance
        .translate_Callback_2_(this, tx, ty);
  }
}
// Copyright (c) 2012, the Dart project authors.  Please see the AUTHORS file
// for details. All rights reserved. Use of this source code is governed by a
// BSD-style license that can be found in the LICENSE file.

// WARNING: Do not edit - generated code.

@DocsEditable()
@DomName('DOMParser')
class DomParser extends DartHtmlDomObject {
  // To suppress missing implicit constructor warnings.
  factory DomParser._() {
    throw new UnsupportedError("Not supported");
  }

  @DomName('DOMParser.DOMParser')
  @DocsEditable()
  factory DomParser() {
    return _blink.BlinkDOMParser.instance.constructorCallback_0_();
  }

  @Deprecated("Internal Use Only")
  external static Type get instanceRuntimeType;

  @Deprecated("Internal Use Only")
  DomParser.internal_() {}

  @DomName('DOMParser.parseFromString')
  @DocsEditable()
  Document parseFromString(String str, String type) =>
      _blink.BlinkDOMParser.instance
          .parseFromString_Callback_2_(this, str, type);
}
// Copyright (c) 2012, the Dart project authors.  Please see the AUTHORS file
// for details. All rights reserved. Use of this source code is governed by a
// BSD-style license that can be found in the LICENSE file.

// WARNING: Do not edit - generated code.

@DocsEditable()
@DomName('DOMPoint')
@Experimental() // untriaged
class DomPoint extends DomPointReadOnly {
  // To suppress missing implicit constructor warnings.
  factory DomPoint._() {
    throw new UnsupportedError("Not supported");
  }

  @DomName('DOMPoint.DOMPoint')
  @DocsEditable()
  factory DomPoint([point_OR_x, num y, num z, num w]) {
    if ((point_OR_x is Map || point_OR_x == null) &&
        y == null &&
        z == null &&
        w == null) {
      var point_1 = convertDartToNative_Dictionary(point_OR_x);
      return _blink.BlinkDOMPoint.instance.constructorCallback_1_(point_1);
    }
    if (point_OR_x == null && y == null && z == null && w == null) {
      return _blink.BlinkDOMPoint.instance.constructorCallback_0_();
    }
    if ((point_OR_x is num || point_OR_x == null) &&
        y == null &&
        z == null &&
        w == null) {
      return _blink.BlinkDOMPoint.instance.constructorCallback_1_(point_OR_x);
    }
    if ((y is num || y == null) &&
        (point_OR_x is num || point_OR_x == null) &&
        z == null &&
        w == null) {
      return _blink.BlinkDOMPoint.instance
          .constructorCallback_2_(point_OR_x, y);
    }
    if ((z is num || z == null) &&
        (y is num || y == null) &&
        (point_OR_x is num || point_OR_x == null) &&
        w == null) {
      return _blink.BlinkDOMPoint.instance
          .constructorCallback_3_(point_OR_x, y, z);
    }
    if ((w is num || w == null) &&
        (z is num || z == null) &&
        (y is num || y == null) &&
        (point_OR_x is num || point_OR_x == null)) {
      return _blink.BlinkDOMPoint.instance
          .constructorCallback_4_(point_OR_x, y, z, w);
    }
    throw new ArgumentError("Incorrect number or type of arguments");
  }

  @Deprecated("Internal Use Only")
  external static Type get instanceRuntimeType;

  @Deprecated("Internal Use Only")
  DomPoint.internal_() : super.internal_();

  /// Checks if this type is supported on the current platform.
  static bool get supported => true;

  @DomName('DOMPoint.w')
  @DocsEditable()
  @Experimental() // untriaged
  num get w => _blink.BlinkDOMPoint.instance.w_Getter_(this);

  @DomName('DOMPoint.w')
  @DocsEditable()
  @Experimental() // untriaged
  set w(num value) => _blink.BlinkDOMPoint.instance.w_Setter_(this, value);

  @DomName('DOMPoint.x')
  @DocsEditable()
  @Experimental() // untriaged
  num get x => _blink.BlinkDOMPoint.instance.x_Getter_(this);

  @DomName('DOMPoint.x')
  @DocsEditable()
  @Experimental() // untriaged
  set x(num value) => _blink.BlinkDOMPoint.instance.x_Setter_(this, value);

  @DomName('DOMPoint.y')
  @DocsEditable()
  @Experimental() // untriaged
  num get y => _blink.BlinkDOMPoint.instance.y_Getter_(this);

  @DomName('DOMPoint.y')
  @DocsEditable()
  @Experimental() // untriaged
  set y(num value) => _blink.BlinkDOMPoint.instance.y_Setter_(this, value);

  @DomName('DOMPoint.z')
  @DocsEditable()
  @Experimental() // untriaged
  num get z => _blink.BlinkDOMPoint.instance.z_Getter_(this);

  @DomName('DOMPoint.z')
  @DocsEditable()
  @Experimental() // untriaged
  set z(num value) => _blink.BlinkDOMPoint.instance.z_Setter_(this, value);
}
// Copyright (c) 2012, the Dart project authors.  Please see the AUTHORS file
// for details. All rights reserved. Use of this source code is governed by a
// BSD-style license that can be found in the LICENSE file.

// WARNING: Do not edit - generated code.

@DocsEditable()
@DomName('DOMPointReadOnly')
@Experimental() // untriaged
class DomPointReadOnly extends DartHtmlDomObject {
  // To suppress missing implicit constructor warnings.
  factory DomPointReadOnly._() {
    throw new UnsupportedError("Not supported");
  }

  @DomName('DOMPointReadOnly.DOMPointReadOnly')
  @DocsEditable()
  factory DomPointReadOnly(num x, num y, num z, num w) {
    return _blink.BlinkDOMPointReadOnly.instance
        .constructorCallback_4_(x, y, z, w);
  }

  @Deprecated("Internal Use Only")
  external static Type get instanceRuntimeType;

  @Deprecated("Internal Use Only")
  DomPointReadOnly.internal_() {}

  @DomName('DOMPointReadOnly.w')
  @DocsEditable()
  @Experimental() // untriaged
  num get w => _blink.BlinkDOMPointReadOnly.instance.w_Getter_(this);

  @DomName('DOMPointReadOnly.x')
  @DocsEditable()
  @Experimental() // untriaged
  num get x => _blink.BlinkDOMPointReadOnly.instance.x_Getter_(this);

  @DomName('DOMPointReadOnly.y')
  @DocsEditable()
  @Experimental() // untriaged
  num get y => _blink.BlinkDOMPointReadOnly.instance.y_Getter_(this);

  @DomName('DOMPointReadOnly.z')
  @DocsEditable()
  @Experimental() // untriaged
  num get z => _blink.BlinkDOMPointReadOnly.instance.z_Getter_(this);
}
// Copyright (c) 2013, the Dart project authors.  Please see the AUTHORS file
// for details. All rights reserved. Use of this source code is governed by a
// BSD-style license that can be found in the LICENSE file.

@DocsEditable()
@DomName('DOMRectReadOnly')
@Experimental() // untriaged
class DomRectReadOnly extends DartHtmlDomObject implements Rectangle {
  // NOTE! All code below should be common with RectangleBase.
  String toString() {
    return 'Rectangle ($left, $top) $width x $height';
  }

  bool operator ==(other) {
    if (other is! Rectangle) return false;
    return left == other.left &&
        top == other.top &&
        width == other.width &&
        height == other.height;
  }

  int get hashCode => _JenkinsSmiHash.hash4(
      left.hashCode, top.hashCode, width.hashCode, height.hashCode);

  /**
   * Computes the intersection of `this` and [other].
   *
   * The intersection of two axis-aligned rectangles, if any, is always another
   * axis-aligned rectangle.
   *
   * Returns the intersection of this and `other`, or null if they don't
   * intersect.
   */
  Rectangle intersection(Rectangle other) {
    var x0 = max(left, other.left);
    var x1 = min(left + width, other.left + other.width);

    if (x0 <= x1) {
      var y0 = max(top, other.top);
      var y1 = min(top + height, other.top + other.height);

      if (y0 <= y1) {
        return new Rectangle(x0, y0, x1 - x0, y1 - y0);
      }
    }
    return null;
  }

  /**
   * Returns true if `this` intersects [other].
   */
  bool intersects(Rectangle<num> other) {
    return (left <= other.left + other.width &&
        other.left <= left + width &&
        top <= other.top + other.height &&
        other.top <= top + height);
  }

  /**
   * Returns a new rectangle which completely contains `this` and [other].
   */
  Rectangle boundingBox(Rectangle other) {
    var right = max(this.left + this.width, other.left + other.width);
    var bottom = max(this.top + this.height, other.top + other.height);

    var left = min(this.left, other.left);
    var top = min(this.top, other.top);

    return new Rectangle(left, top, right - left, bottom - top);
  }

  /**
   * Tests whether `this` entirely contains [another].
   */
  bool containsRectangle(Rectangle<num> another) {
    return left <= another.left &&
        left + width >= another.left + another.width &&
        top <= another.top &&
        top + height >= another.top + another.height;
  }

  /**
   * Tests whether [another] is inside or along the edges of `this`.
   */
  bool containsPoint(Point<num> another) {
    return another.x >= left &&
        another.x <= left + width &&
        another.y >= top &&
        another.y <= top + height;
  }

  Point get topLeft => new Point/*<num>*/(this.left, this.top);
  Point get topRight => new Point/*<num>*/(this.left + this.width, this.top);
  Point get bottomRight =>
      new Point/*<num>*/(this.left + this.width, this.top + this.height);
  Point get bottomLeft => new Point/*<num>*/(this.left, this.top + this.height);

  // To suppress missing implicit constructor warnings.
  factory DomRectReadOnly._() {
    throw new UnsupportedError("Not supported");
  }

  @DomName('DOMRectReadOnly.DOMRectReadOnly')
  @DocsEditable()
  factory DomRectReadOnly(num x, num y, num width, num height) {
    return _blink.BlinkDOMRectReadOnly.instance
        .constructorCallback_4_(x, y, width, height);
  }

  @Deprecated("Internal Use Only")
  external static Type get instanceRuntimeType;

  @Deprecated("Internal Use Only")
  DomRectReadOnly.internal_() {}

  @DomName('DOMRectReadOnly.bottom')
  @DocsEditable()
  @Experimental() // untriaged
  num get bottom => _blink.BlinkDOMRectReadOnly.instance.bottom_Getter_(this);

  @DomName('DOMRectReadOnly.height')
  @DocsEditable()
  @Experimental() // untriaged
  num get height => _blink.BlinkDOMRectReadOnly.instance.height_Getter_(this);

  @DomName('DOMRectReadOnly.left')
  @DocsEditable()
  @Experimental() // untriaged
  num get left => _blink.BlinkDOMRectReadOnly.instance.left_Getter_(this);

  @DomName('DOMRectReadOnly.right')
  @DocsEditable()
  @Experimental() // untriaged
  num get right => _blink.BlinkDOMRectReadOnly.instance.right_Getter_(this);

  @DomName('DOMRectReadOnly.top')
  @DocsEditable()
  @Experimental() // untriaged
  num get top => _blink.BlinkDOMRectReadOnly.instance.top_Getter_(this);

  @DomName('DOMRectReadOnly.width')
  @DocsEditable()
  @Experimental() // untriaged
  num get width => _blink.BlinkDOMRectReadOnly.instance.width_Getter_(this);

  @DomName('DOMRectReadOnly.x')
  @DocsEditable()
  @Experimental() // untriaged
  num get x => _blink.BlinkDOMRectReadOnly.instance.x_Getter_(this);

  @DomName('DOMRectReadOnly.y')
  @DocsEditable()
  @Experimental() // untriaged
  num get y => _blink.BlinkDOMRectReadOnly.instance.y_Getter_(this);
}

// Copyright (c) 2012, the Dart project authors.  Please see the AUTHORS file
// for details. All rights reserved. Use of this source code is governed by a
// BSD-style license that can be found in the LICENSE file.

// WARNING: Do not edit - generated code.

@DocsEditable()
@DomName('DOMSettableTokenList')
class DomSettableTokenList extends DomTokenList {
  // To suppress missing implicit constructor warnings.
  factory DomSettableTokenList._() {
    throw new UnsupportedError("Not supported");
  }

  @Deprecated("Internal Use Only")
  external static Type get instanceRuntimeType;

  @Deprecated("Internal Use Only")
  DomSettableTokenList.internal_() : super.internal_();

  @DomName('DOMSettableTokenList.value')
  @DocsEditable()
  String get value =>
      _blink.BlinkDOMSettableTokenList.instance.value_Getter_(this);

  @DomName('DOMSettableTokenList.value')
  @DocsEditable()
  set value(String value) =>
      _blink.BlinkDOMSettableTokenList.instance.value_Setter_(this, value);

  @DomName('DOMSettableTokenList.item')
  @DocsEditable()
  @Experimental() // untriaged
  String item(int index) =>
      _blink.BlinkDOMSettableTokenList.instance.item_Callback_1_(this, index);
}
// Copyright (c) 2012, the Dart project authors.  Please see the AUTHORS file
// for details. All rights reserved. Use of this source code is governed by a
// BSD-style license that can be found in the LICENSE file.

// WARNING: Do not edit - generated code.

@DocsEditable()
@DomName('DOMStringList')
class DomStringList extends DartHtmlDomObject
    with ListMixin<String>, ImmutableListMixin<String>
    implements List<String> {
  // To suppress missing implicit constructor warnings.
  factory DomStringList._() {
    throw new UnsupportedError("Not supported");
  }

  @Deprecated("Internal Use Only")
  external static Type get instanceRuntimeType;

  @Deprecated("Internal Use Only")
  DomStringList.internal_() {}

  @DomName('DOMStringList.length')
  @DocsEditable()
  int get length => _blink.BlinkDOMStringList.instance.length_Getter_(this);

  String operator [](int index) {
    if (index < 0 || index >= length) throw new RangeError.index(index, this);
    return _nativeIndexedGetter(index);
  }

  String _nativeIndexedGetter(int index) =>
      (_blink.BlinkDOMStringList.instance.item_Callback_1_(this, index));

  void operator []=(int index, String value) {
    throw new UnsupportedError("Cannot assign element of immutable List.");
  }
  // -- start List<String> mixins.
  // String is the element type.

  set length(int value) {
    throw new UnsupportedError("Cannot resize immutable List.");
  }

  String get first {
    if (this.length > 0) {
      return _nativeIndexedGetter(0);
    }
    throw new StateError("No elements");
  }

  String get last {
    int len = this.length;
    if (len > 0) {
      return _nativeIndexedGetter(len - 1);
    }
    throw new StateError("No elements");
  }

  String get single {
    int len = this.length;
    if (len == 1) {
      return _nativeIndexedGetter(0);
    }
    if (len == 0) throw new StateError("No elements");
    throw new StateError("More than one element");
  }

  String elementAt(int index) => this[index];
  // -- end List<String> mixins.

  @DomName('DOMStringList.__getter__')
  @DocsEditable()
  @Experimental() // untriaged
  String __getter__(int index) =>
      _blink.BlinkDOMStringList.instance.$__getter___Callback_1_(this, index);

  @DomName('DOMStringList.contains')
  @DocsEditable()
  bool contains(String string) =>
      _blink.BlinkDOMStringList.instance.contains_Callback_1_(this, string);

  @DomName('DOMStringList.item')
  @DocsEditable()
  String item(int index) =>
      _blink.BlinkDOMStringList.instance.item_Callback_1_(this, index);
}
// Copyright (c) 2012, the Dart project authors.  Please see the AUTHORS file
// for details. All rights reserved. Use of this source code is governed by a
// BSD-style license that can be found in the LICENSE file.

// WARNING: Do not edit - generated code.

@DocsEditable()
@DomName('DOMStringMap')
class DomStringMap extends DartHtmlDomObject {
  // To suppress missing implicit constructor warnings.
  factory DomStringMap._() {
    throw new UnsupportedError("Not supported");
  }

  @Deprecated("Internal Use Only")
  external static Type get instanceRuntimeType;

  @Deprecated("Internal Use Only")
  DomStringMap.internal_() {}

  void __delete__(index_OR_name) {
    if ((index_OR_name is String || index_OR_name == null)) {
      _blink.BlinkDOMStringMap.instance
          .$__delete___Callback_1_(this, index_OR_name);
      return;
    }
    if ((index_OR_name is int || index_OR_name == null)) {
      _blink.BlinkDOMStringMap.instance
          .$__delete___Callback_1_(this, index_OR_name);
      return;
    }
    throw new ArgumentError("Incorrect number or type of arguments");
  }

  @DomName('DOMStringMap.__getter__')
  @DocsEditable()
  String __getter__(int index) =>
      _blink.BlinkDOMStringMap.instance.$__getter___Callback_1_(this, index);

  void __setter__(index_OR_name, String value) {
    if ((value is String || value == null) &&
        (index_OR_name is String || index_OR_name == null)) {
      _blink.BlinkDOMStringMap.instance
          .$__setter___Callback_2_(this, index_OR_name, value);
      return;
    }
    if ((value is String || value == null) &&
        (index_OR_name is int || index_OR_name == null)) {
      _blink.BlinkDOMStringMap.instance
          .$__setter___Callback_2_(this, index_OR_name, value);
      return;
    }
    throw new ArgumentError("Incorrect number or type of arguments");
  }

  @DomName('DOMStringMap.item')
  @DocsEditable()
  @Experimental() // untriaged
  String item(String name) =>
      _blink.BlinkDOMStringMap.instance.item_Callback_1_(this, name);
}
// Copyright (c) 2012, the Dart project authors.  Please see the AUTHORS file
// for details. All rights reserved. Use of this source code is governed by a
// BSD-style license that can be found in the LICENSE file.

// WARNING: Do not edit - generated code.

@DocsEditable()
@DomName('DOMTokenList')
class DomTokenList extends DartHtmlDomObject {
  // To suppress missing implicit constructor warnings.
  factory DomTokenList._() {
    throw new UnsupportedError("Not supported");
  }

  @Deprecated("Internal Use Only")
  external static Type get instanceRuntimeType;

  @Deprecated("Internal Use Only")
  DomTokenList.internal_() {}

  @DomName('DOMTokenList.length')
  @DocsEditable()
  int get length => _blink.BlinkDOMTokenList.instance.length_Getter_(this);

  @DomName('DOMTokenList.add')
  @DocsEditable()
  @Experimental() // untriaged
  void add(String tokens) =>
      _blink.BlinkDOMTokenList.instance.add_Callback_1_(this, tokens);

  @DomName('DOMTokenList.contains')
  @DocsEditable()
  bool contains(String token) =>
      _blink.BlinkDOMTokenList.instance.contains_Callback_1_(this, token);

  @DomName('DOMTokenList.item')
  @DocsEditable()
  String item(int index) =>
      _blink.BlinkDOMTokenList.instance.item_Callback_1_(this, index);

  @DomName('DOMTokenList.remove')
  @DocsEditable()
  @Experimental() // untriaged
  void remove(String tokens) =>
      _blink.BlinkDOMTokenList.instance.remove_Callback_1_(this, tokens);

  bool toggle(String token, [bool force]) {
    if (force != null) {
      return _blink.BlinkDOMTokenList.instance
          .toggle_Callback_2_(this, token, force);
    }
    return _blink.BlinkDOMTokenList.instance.toggle_Callback_1_(this, token);
  }
}
// Copyright (c) 2012, the Dart project authors.  Please see the AUTHORS file
// for details. All rights reserved. Use of this source code is governed by a
// BSD-style license that can be found in the LICENSE file.

// WARNING: Do not edit - generated code.

@DocsEditable()
@DomName('EffectModel')
@Experimental() // untriaged
class EffectModel extends DartHtmlDomObject {
  // To suppress missing implicit constructor warnings.
  factory EffectModel._() {
    throw new UnsupportedError("Not supported");
  }

  @Deprecated("Internal Use Only")
  external static Type get instanceRuntimeType;

  @Deprecated("Internal Use Only")
  EffectModel.internal_() {}
}
// Copyright (c) 2012, the Dart project authors.  Please see the AUTHORS file
// for details. All rights reserved. Use of this source code is governed by a
// BSD-style license that can be found in the LICENSE file.

class _ChildrenElementList extends ListBase<Element>
    implements NodeListWrapper {
  // Raw Element.
  final Element _element;
  final HtmlCollection _childElements;

  _ChildrenElementList._wrap(Element element)
      : _childElements = element._children,
        _element = element;

  bool contains(Object element) => _childElements.contains(element);

  bool get isEmpty {
    return _element._firstElementChild == null;
  }

  int get length {
    return _childElements.length;
  }

  Element operator [](int index) {
    return _childElements[index];
  }

  void operator []=(int index, Element value) {
    _element._replaceChild(value, _childElements[index]);
  }

  set length(int newLength) {
    // TODO(jacobr): remove children when length is reduced.
    throw new UnsupportedError('Cannot resize element lists');
  }

  Element add(Element value) {
    _element.append(value);
    return value;
  }

  Iterator<Element> get iterator => toList().iterator;

  void addAll(Iterable<Element> iterable) {
    if (iterable is _ChildNodeListLazy) {
      iterable = new List.from(iterable);
    }

    for (Element element in iterable) {
      _element.append(element);
    }
  }

  void sort([int compare(Element a, Element b)]) {
    throw new UnsupportedError('Cannot sort element lists');
  }

  void shuffle([Random random]) {
    throw new UnsupportedError('Cannot shuffle element lists');
  }

  void removeWhere(bool test(Element element)) {
    _filter(test, false);
  }

  void retainWhere(bool test(Element element)) {
    _filter(test, true);
  }

  void _filter(bool test(Element element), bool retainMatching) {
    var removed;
    if (retainMatching) {
      removed = _element.children.where((e) => !test(e));
    } else {
      removed = _element.children.where(test);
    }
    for (var e in removed) e.remove();
  }

  void setRange(int start, int end, Iterable<Element> iterable,
      [int skipCount = 0]) {
    throw new UnimplementedError();
  }

  void replaceRange(int start, int end, Iterable<Element> iterable) {
    throw new UnimplementedError();
  }

  void fillRange(int start, int end, [Element fillValue]) {
    throw new UnimplementedError();
  }

  bool remove(Object object) {
    if (object is Element) {
      Element element = object;
      // We aren't preserving identity of nodes in JSINTEROP mode
      if (element.parentNode == _element) {
        _element._removeChild(element);
        return true;
      }
    }
    return false;
  }

  void insert(int index, Element element) {
    if (index < 0 || index > length) {
      throw new RangeError.range(index, 0, length);
    }
    if (index == length) {
      _element.append(element);
    } else {
      _element.insertBefore(element, this[index]);
    }
  }

  void setAll(int index, Iterable<Element> iterable) {
    throw new UnimplementedError();
  }

  void clear() {
    _element._clearChildren();
  }

  Element removeAt(int index) {
    final result = this[index];
    if (result != null) {
      _element._removeChild(result);
    }
    return result;
  }

  Element removeLast() {
    final result = this.last;
    if (result != null) {
      _element._removeChild(result);
    }
    return result;
  }

  Element get first {
    Element result = _element._firstElementChild;
    if (result == null) throw new StateError("No elements");
    return result;
  }

  Element get last {
    Element result = _element._lastElementChild;
    if (result == null) throw new StateError("No elements");
    return result;
  }

  Element get single {
    if (length > 1) throw new StateError("More than one element");
    return first;
  }

  List<Node> get rawList => _childElements;
}

/**
 * An immutable list containing HTML elements. This list contains some
 * additional methods when compared to regular lists for ease of CSS
 * manipulation on a group of elements.
 */
abstract class ElementList<T extends Element> extends ListBase<T> {
  /**
   * The union of all CSS classes applied to the elements in this list.
   *
   * This set makes it easy to add, remove or toggle (add if not present, remove
   * if present) the classes applied to a collection of elements.
   *
   *     htmlList.classes.add('selected');
   *     htmlList.classes.toggle('isOnline');
   *     htmlList.classes.remove('selected');
   */
  CssClassSet get classes;

  /** Replace the classes with `value` for every element in this list. */
  set classes(Iterable<String> value);

  /**
   * Access the union of all [CssStyleDeclaration]s that are associated with an
   * [ElementList].
   *
   * Grouping the style objects all together provides easy editing of specific
   * properties of a collection of elements. Setting a specific property value
   * will set that property in all [Element]s in the [ElementList]. Getting a
   * specific property value will return the value of the property of the first
   * element in the [ElementList].
   */
  CssStyleDeclarationBase get style;

  /**
   * Access dimensions and position of the Elements in this list.
   *
   * Setting the height or width properties will set the height or width
   * property for all elements in the list. This returns a rectangle with the
   * dimenions actually available for content
   * in this element, in pixels, regardless of this element's box-sizing
   * property. Getting the height or width returns the height or width of the
   * first Element in this list.
   *
   * Unlike [getBoundingClientRect], the dimensions of this rectangle
   * will return the same numerical height if the element is hidden or not.
   */
  @Experimental()
  CssRect get contentEdge;

  /**
   * Access dimensions and position of the first Element's content + padding box
   * in this list.
   *
   * This returns a rectangle with the dimenions actually available for content
   * in this element, in pixels, regardless of this element's box-sizing
   * property. Unlike [getBoundingClientRect], the dimensions of this rectangle
   * will return the same numerical height if the element is hidden or not. This
   * can be used to retrieve jQuery's `innerHeight` value for an element. This
   * is also a rectangle equalling the dimensions of clientHeight and
   * clientWidth.
   */
  @Experimental()
  CssRect get paddingEdge;

  /**
   * Access dimensions and position of the first Element's content + padding +
   * border box in this list.
   *
   * This returns a rectangle with the dimenions actually available for content
   * in this element, in pixels, regardless of this element's box-sizing
   * property. Unlike [getBoundingClientRect], the dimensions of this rectangle
   * will return the same numerical height if the element is hidden or not. This
   * can be used to retrieve jQuery's `outerHeight` value for an element.
   */
  @Experimental()
  CssRect get borderEdge;

  /**
   * Access dimensions and position of the first Element's content + padding +
   * border + margin box in this list.
   *
   * This returns a rectangle with the dimenions actually available for content
   * in this element, in pixels, regardless of this element's box-sizing
   * property. Unlike [getBoundingClientRect], the dimensions of this rectangle
   * will return the same numerical height if the element is hidden or not. This
   * can be used to retrieve jQuery's `outerHeight` value for an element.
   */
  @Experimental()
  CssRect get marginEdge;

  /// Stream of `abort` events handled by this [Element].
  @DomName('Element.onabort')
  @DocsEditable()
  ElementStream<Event> get onAbort;

  /// Stream of `beforecopy` events handled by this [Element].
  @DomName('Element.onbeforecopy')
  @DocsEditable()
  ElementStream<Event> get onBeforeCopy;

  /// Stream of `beforecut` events handled by this [Element].
  @DomName('Element.onbeforecut')
  @DocsEditable()
  ElementStream<Event> get onBeforeCut;

  /// Stream of `beforepaste` events handled by this [Element].
  @DomName('Element.onbeforepaste')
  @DocsEditable()
  ElementStream<Event> get onBeforePaste;

  /// Stream of `blur` events handled by this [Element].
  @DomName('Element.onblur')
  @DocsEditable()
  ElementStream<Event> get onBlur;

  @DomName('Element.oncanplay')
  @DocsEditable()
  @Experimental() // untriaged
  ElementStream<Event> get onCanPlay;

  @DomName('Element.oncanplaythrough')
  @DocsEditable()
  @Experimental() // untriaged
  ElementStream<Event> get onCanPlayThrough;

  /// Stream of `change` events handled by this [Element].
  @DomName('Element.onchange')
  @DocsEditable()
  ElementStream<Event> get onChange;

  /// Stream of `click` events handled by this [Element].
  @DomName('Element.onclick')
  @DocsEditable()
  ElementStream<MouseEvent> get onClick;

  /// Stream of `contextmenu` events handled by this [Element].
  @DomName('Element.oncontextmenu')
  @DocsEditable()
  ElementStream<MouseEvent> get onContextMenu;

  /// Stream of `copy` events handled by this [Element].
  @DomName('Element.oncopy')
  @DocsEditable()
  ElementStream<ClipboardEvent> get onCopy;

  /// Stream of `cut` events handled by this [Element].
  @DomName('Element.oncut')
  @DocsEditable()
  ElementStream<ClipboardEvent> get onCut;

  /// Stream of `doubleclick` events handled by this [Element].
  @DomName('Element.ondblclick')
  @DocsEditable()
  ElementStream<Event> get onDoubleClick;

  /**
   * A stream of `drag` events fired when this element currently being dragged.
   *
   * A `drag` event is added to this stream as soon as the drag begins.
   * A `drag` event is also added to this stream at intervals while the drag
   * operation is still ongoing.
   *
   * ## Other resources
   *
   * * [Drag and drop
   *   sample](https://github.com/dart-lang/dart-samples/tree/master/html5/web/dnd/basics)
   *   based on [the tutorial](http://www.html5rocks.com/en/tutorials/dnd/basics/)
   *   from HTML5Rocks.
   * * [Drag and drop
   *   specification](https://html.spec.whatwg.org/multipage/interaction.html#dnd)
   *   from WHATWG.
   */
  @DomName('Element.ondrag')
  @DocsEditable()
  ElementStream<MouseEvent> get onDrag;

  /**
   * A stream of `dragend` events fired when this element completes a drag
   * operation.
   *
   * ## Other resources
   *
   * * [Drag and drop
   *   sample](https://github.com/dart-lang/dart-samples/tree/master/html5/web/dnd/basics)
   *   based on [the tutorial](http://www.html5rocks.com/en/tutorials/dnd/basics/)
   *   from HTML5Rocks.
   * * [Drag and drop
   *   specification](https://html.spec.whatwg.org/multipage/interaction.html#dnd)
   *   from WHATWG.
   */
  @DomName('Element.ondragend')
  @DocsEditable()
  ElementStream<MouseEvent> get onDragEnd;

  /**
   * A stream of `dragenter` events fired when a dragged object is first dragged
   * over this element.
   *
   * ## Other resources
   *
   * * [Drag and drop
   *   sample](https://github.com/dart-lang/dart-samples/tree/master/html5/web/dnd/basics)
   *   based on [the tutorial](http://www.html5rocks.com/en/tutorials/dnd/basics/)
   *   from HTML5Rocks.
   * * [Drag and drop
   *   specification](https://html.spec.whatwg.org/multipage/interaction.html#dnd)
   *   from WHATWG.
   */
  @DomName('Element.ondragenter')
  @DocsEditable()
  ElementStream<MouseEvent> get onDragEnter;

  /**
   * A stream of `dragleave` events fired when an object being dragged over this
   * element leaves this element's target area.
   *
   * ## Other resources
   *
   * * [Drag and drop
   *   sample](https://github.com/dart-lang/dart-samples/tree/master/html5/web/dnd/basics)
   *   based on [the tutorial](http://www.html5rocks.com/en/tutorials/dnd/basics/)
   *   from HTML5Rocks.
   * * [Drag and drop
   *   specification](https://html.spec.whatwg.org/multipage/interaction.html#dnd)
   *   from WHATWG.
   */
  @DomName('Element.ondragleave')
  @DocsEditable()
  ElementStream<MouseEvent> get onDragLeave;

  /**
   * A stream of `dragover` events fired when a dragged object is currently
   * being dragged over this element.
   *
   * ## Other resources
   *
   * * [Drag and drop
   *   sample](https://github.com/dart-lang/dart-samples/tree/master/html5/web/dnd/basics)
   *   based on [the tutorial](http://www.html5rocks.com/en/tutorials/dnd/basics/)
   *   from HTML5Rocks.
   * * [Drag and drop
   *   specification](https://html.spec.whatwg.org/multipage/interaction.html#dnd)
   *   from WHATWG.
   */
  @DomName('Element.ondragover')
  @DocsEditable()
  ElementStream<MouseEvent> get onDragOver;

  /**
   * A stream of `dragstart` events fired when this element starts being
   * dragged.
   *
   * ## Other resources
   *
   * * [Drag and drop
   *   sample](https://github.com/dart-lang/dart-samples/tree/master/html5/web/dnd/basics)
   *   based on [the tutorial](http://www.html5rocks.com/en/tutorials/dnd/basics/)
   *   from HTML5Rocks.
   * * [Drag and drop
   *   specification](https://html.spec.whatwg.org/multipage/interaction.html#dnd)
   *   from WHATWG.
   */
  @DomName('Element.ondragstart')
  @DocsEditable()
  ElementStream<MouseEvent> get onDragStart;

  /**
   * A stream of `drop` events fired when a dragged object is dropped on this
   * element.
   *
   * ## Other resources
   *
   * * [Drag and drop
   *   sample](https://github.com/dart-lang/dart-samples/tree/master/html5/web/dnd/basics)
   *   based on [the tutorial](http://www.html5rocks.com/en/tutorials/dnd/basics/)
   *   from HTML5Rocks.
   * * [Drag and drop
   *   specification](https://html.spec.whatwg.org/multipage/interaction.html#dnd)
   *   from WHATWG.
   */
  @DomName('Element.ondrop')
  @DocsEditable()
  ElementStream<MouseEvent> get onDrop;

  @DomName('Element.ondurationchange')
  @DocsEditable()
  @Experimental() // untriaged
  ElementStream<Event> get onDurationChange;

  @DomName('Element.onemptied')
  @DocsEditable()
  @Experimental() // untriaged
  ElementStream<Event> get onEmptied;

  @DomName('Element.onended')
  @DocsEditable()
  @Experimental() // untriaged
  ElementStream<Event> get onEnded;

  /// Stream of `error` events handled by this [Element].
  @DomName('Element.onerror')
  @DocsEditable()
  ElementStream<Event> get onError;

  /// Stream of `focus` events handled by this [Element].
  @DomName('Element.onfocus')
  @DocsEditable()
  ElementStream<Event> get onFocus;

  /// Stream of `input` events handled by this [Element].
  @DomName('Element.oninput')
  @DocsEditable()
  ElementStream<Event> get onInput;

  /// Stream of `invalid` events handled by this [Element].
  @DomName('Element.oninvalid')
  @DocsEditable()
  ElementStream<Event> get onInvalid;

  /// Stream of `keydown` events handled by this [Element].
  @DomName('Element.onkeydown')
  @DocsEditable()
  ElementStream<KeyboardEvent> get onKeyDown;

  /// Stream of `keypress` events handled by this [Element].
  @DomName('Element.onkeypress')
  @DocsEditable()
  ElementStream<KeyboardEvent> get onKeyPress;

  /// Stream of `keyup` events handled by this [Element].
  @DomName('Element.onkeyup')
  @DocsEditable()
  ElementStream<KeyboardEvent> get onKeyUp;

  /// Stream of `load` events handled by this [Element].
  @DomName('Element.onload')
  @DocsEditable()
  ElementStream<Event> get onLoad;

  @DomName('Element.onloadeddata')
  @DocsEditable()
  @Experimental() // untriaged
  ElementStream<Event> get onLoadedData;

  @DomName('Element.onloadedmetadata')
  @DocsEditable()
  @Experimental() // untriaged
  ElementStream<Event> get onLoadedMetadata;

  /// Stream of `mousedown` events handled by this [Element].
  @DomName('Element.onmousedown')
  @DocsEditable()
  ElementStream<MouseEvent> get onMouseDown;

  /// Stream of `mouseenter` events handled by this [Element].
  @DomName('Element.onmouseenter')
  @DocsEditable()
  @Experimental() // untriaged
  ElementStream<MouseEvent> get onMouseEnter;

  /// Stream of `mouseleave` events handled by this [Element].
  @DomName('Element.onmouseleave')
  @DocsEditable()
  @Experimental() // untriaged
  ElementStream<MouseEvent> get onMouseLeave;

  /// Stream of `mousemove` events handled by this [Element].
  @DomName('Element.onmousemove')
  @DocsEditable()
  ElementStream<MouseEvent> get onMouseMove;

  /// Stream of `mouseout` events handled by this [Element].
  @DomName('Element.onmouseout')
  @DocsEditable()
  ElementStream<MouseEvent> get onMouseOut;

  /// Stream of `mouseover` events handled by this [Element].
  @DomName('Element.onmouseover')
  @DocsEditable()
  ElementStream<MouseEvent> get onMouseOver;

  /// Stream of `mouseup` events handled by this [Element].
  @DomName('Element.onmouseup')
  @DocsEditable()
  ElementStream<MouseEvent> get onMouseUp;

  /// Stream of `mousewheel` events handled by this [Element].
  @DomName('Element.onmousewheel')
  @DocsEditable()
  // http://www.w3.org/TR/DOM-Level-3-Events/#events-wheelevents
  @Experimental() // non-standard
  ElementStream<WheelEvent> get onMouseWheel;

  /// Stream of `paste` events handled by this [Element].
  @DomName('Element.onpaste')
  @DocsEditable()
  ElementStream<ClipboardEvent> get onPaste;

  @DomName('Element.onpause')
  @DocsEditable()
  @Experimental() // untriaged
  ElementStream<Event> get onPause;

  @DomName('Element.onplay')
  @DocsEditable()
  @Experimental() // untriaged
  ElementStream<Event> get onPlay;

  @DomName('Element.onplaying')
  @DocsEditable()
  @Experimental() // untriaged
  ElementStream<Event> get onPlaying;

  @DomName('Element.onratechange')
  @DocsEditable()
  @Experimental() // untriaged
  ElementStream<Event> get onRateChange;

  /// Stream of `reset` events handled by this [Element].
  @DomName('Element.onreset')
  @DocsEditable()
  ElementStream<Event> get onReset;

  @DomName('Element.onresize')
  @DocsEditable()
  @Experimental() // untriaged
  ElementStream<Event> get onResize;

  /// Stream of `scroll` events handled by this [Element].
  @DomName('Element.onscroll')
  @DocsEditable()
  ElementStream<Event> get onScroll;

  /// Stream of `search` events handled by this [Element].
  @DomName('Element.onsearch')
  @DocsEditable()
  // http://www.w3.org/TR/html-markup/input.search.html
  @Experimental()
  ElementStream<Event> get onSearch;

  @DomName('Element.onseeked')
  @DocsEditable()
  @Experimental() // untriaged
  ElementStream<Event> get onSeeked;

  @DomName('Element.onseeking')
  @DocsEditable()
  @Experimental() // untriaged
  ElementStream<Event> get onSeeking;

  /// Stream of `select` events handled by this [Element].
  @DomName('Element.onselect')
  @DocsEditable()
  ElementStream<Event> get onSelect;

  /// Stream of `selectstart` events handled by this [Element].
  @DomName('Element.onselectstart')
  @DocsEditable()
  @Experimental() // nonstandard
  ElementStream<Event> get onSelectStart;

  @DomName('Element.onstalled')
  @DocsEditable()
  @Experimental() // untriaged
  ElementStream<Event> get onStalled;

  /// Stream of `submit` events handled by this [Element].
  @DomName('Element.onsubmit')
  @DocsEditable()
  ElementStream<Event> get onSubmit;

  @DomName('Element.onsuspend')
  @DocsEditable()
  @Experimental() // untriaged
  ElementStream<Event> get onSuspend;

  @DomName('Element.ontimeupdate')
  @DocsEditable()
  @Experimental() // untriaged
  ElementStream<Event> get onTimeUpdate;

  /// Stream of `touchcancel` events handled by this [Element].
  @DomName('Element.ontouchcancel')
  @DocsEditable()
  // http://www.w3.org/TR/touch-events/, http://www.chromestatus.com/features
  @Experimental()
  ElementStream<TouchEvent> get onTouchCancel;

  /// Stream of `touchend` events handled by this [Element].
  @DomName('Element.ontouchend')
  @DocsEditable()
  // http://www.w3.org/TR/touch-events/, http://www.chromestatus.com/features
  @Experimental()
  ElementStream<TouchEvent> get onTouchEnd;

  /// Stream of `touchenter` events handled by this [Element].
  @DomName('Element.ontouchenter')
  @DocsEditable()
  // http://www.w3.org/TR/touch-events/, http://www.chromestatus.com/features
  @Experimental()
  ElementStream<TouchEvent> get onTouchEnter;

  /// Stream of `touchleave` events handled by this [Element].
  @DomName('Element.ontouchleave')
  @DocsEditable()
  // http://www.w3.org/TR/touch-events/, http://www.chromestatus.com/features
  @Experimental()
  ElementStream<TouchEvent> get onTouchLeave;

  /// Stream of `touchmove` events handled by this [Element].
  @DomName('Element.ontouchmove')
  @DocsEditable()
  // http://www.w3.org/TR/touch-events/, http://www.chromestatus.com/features
  @Experimental()
  ElementStream<TouchEvent> get onTouchMove;

  /// Stream of `touchstart` events handled by this [Element].
  @DomName('Element.ontouchstart')
  @DocsEditable()
  // http://www.w3.org/TR/touch-events/, http://www.chromestatus.com/features
  @Experimental()
  ElementStream<TouchEvent> get onTouchStart;

  /// Stream of `transitionend` events handled by this [Element].
  @DomName('Element.ontransitionend')
  @DocsEditable()
  @SupportedBrowser(SupportedBrowser.CHROME)
  @SupportedBrowser(SupportedBrowser.FIREFOX)
  @SupportedBrowser(SupportedBrowser.IE, '10')
  @SupportedBrowser(SupportedBrowser.SAFARI)
  ElementStream<TransitionEvent> get onTransitionEnd;

  @DomName('Element.onvolumechange')
  @DocsEditable()
  @Experimental() // untriaged
  ElementStream<Event> get onVolumeChange;

  @DomName('Element.onwaiting')
  @DocsEditable()
  @Experimental() // untriaged
  ElementStream<Event> get onWaiting;

  /// Stream of `fullscreenchange` events handled by this [Element].
  @DomName('Element.onwebkitfullscreenchange')
  @DocsEditable()
  // https://dvcs.w3.org/hg/fullscreen/raw-file/tip/Overview.html
  @Experimental()
  ElementStream<Event> get onFullscreenChange;

  /// Stream of `fullscreenerror` events handled by this [Element].
  @DomName('Element.onwebkitfullscreenerror')
  @DocsEditable()
  // https://dvcs.w3.org/hg/fullscreen/raw-file/tip/Overview.html
  @Experimental()
  ElementStream<Event> get onFullscreenError;
}

// Wrapper over an immutable NodeList to make it implement ElementList.
//
// Clients are {`Document`, `DocumentFragment`}.`querySelectorAll` which are
// declared to return `ElementList`.  This provides all the static analysis
// benefit so there is no need for this class have a constrained type parameter.
//
class _FrozenElementList<E extends Element> extends ListBase<E>
    implements ElementList<E>, NodeListWrapper {
  final List<Node> _nodeList;

  var dartClass_instance;

  _FrozenElementList._wrap(this._nodeList) {
    this.dartClass_instance = this._nodeList;
  }

  int get length => _nodeList.length;

  E operator [](int index) => _downcast/*<Node, E>*/(_nodeList[index]);

  void operator []=(int index, E value) {
    throw new UnsupportedError('Cannot modify list');
  }

  set length(int newLength) {
    throw new UnsupportedError('Cannot modify list');
  }

  void sort([Comparator<E> compare]) {
    throw new UnsupportedError('Cannot sort list');
  }

  void shuffle([Random random]) {
    throw new UnsupportedError('Cannot shuffle list');
  }

  E get first => _downcast/*<Node, E>*/(_nodeList.first);

  E get last => _downcast/*<Node, E>*/(_nodeList.last);

  E get single => _downcast/*<Node, E>*/(_nodeList.single);

  CssClassSet get classes => new _MultiElementCssClassSet(this);

  CssStyleDeclarationBase get style => new _CssStyleDeclarationSet(this);

  set classes(Iterable<String> value) {
    // TODO(sra): This might be faster for Sets:
    //
    //     new _MultiElementCssClassSet(this).writeClasses(value)
    //
    // as the code below converts the Iterable[value] to a string multiple
    // times.  Maybe compute the string and set className here.
    forEach((e) => e.classes = value);
  }

  CssRect get contentEdge => new _ContentCssListRect(this);

  CssRect get paddingEdge => this.first.paddingEdge;

  CssRect get borderEdge => this.first.borderEdge;

  CssRect get marginEdge => this.first.marginEdge;

  List<Node> get rawList => _nodeList;

  /// Stream of `abort` events handled by this [Element].
  @DomName('Element.onabort')
  @DocsEditable()
  ElementStream<Event> get onAbort => Element.abortEvent._forElementList(this);

  /// Stream of `beforecopy` events handled by this [Element].
  @DomName('Element.onbeforecopy')
  @DocsEditable()
  ElementStream<Event> get onBeforeCopy =>
      Element.beforeCopyEvent._forElementList(this);

  /// Stream of `beforecut` events handled by this [Element].
  @DomName('Element.onbeforecut')
  @DocsEditable()
  ElementStream<Event> get onBeforeCut =>
      Element.beforeCutEvent._forElementList(this);

  /// Stream of `beforepaste` events handled by this [Element].
  @DomName('Element.onbeforepaste')
  @DocsEditable()
  ElementStream<Event> get onBeforePaste =>
      Element.beforePasteEvent._forElementList(this);

  /// Stream of `blur` events handled by this [Element].
  @DomName('Element.onblur')
  @DocsEditable()
  ElementStream<Event> get onBlur => Element.blurEvent._forElementList(this);

  @DomName('Element.oncanplay')
  @DocsEditable()
  @Experimental() // untriaged
  ElementStream<Event> get onCanPlay =>
      Element.canPlayEvent._forElementList(this);

  @DomName('Element.oncanplaythrough')
  @DocsEditable()
  @Experimental() // untriaged
  ElementStream<Event> get onCanPlayThrough =>
      Element.canPlayThroughEvent._forElementList(this);

  /// Stream of `change` events handled by this [Element].
  @DomName('Element.onchange')
  @DocsEditable()
  ElementStream<Event> get onChange =>
      Element.changeEvent._forElementList(this);

  /// Stream of `click` events handled by this [Element].
  @DomName('Element.onclick')
  @DocsEditable()
  ElementStream<MouseEvent> get onClick =>
      Element.clickEvent._forElementList(this);

  /// Stream of `contextmenu` events handled by this [Element].
  @DomName('Element.oncontextmenu')
  @DocsEditable()
  ElementStream<MouseEvent> get onContextMenu =>
      Element.contextMenuEvent._forElementList(this);

  /// Stream of `copy` events handled by this [Element].
  @DomName('Element.oncopy')
  @DocsEditable()
  ElementStream<ClipboardEvent> get onCopy =>
      Element.copyEvent._forElementList(this);

  /// Stream of `cut` events handled by this [Element].
  @DomName('Element.oncut')
  @DocsEditable()
  ElementStream<ClipboardEvent> get onCut =>
      Element.cutEvent._forElementList(this);

  /// Stream of `doubleclick` events handled by this [Element].
  @DomName('Element.ondblclick')
  @DocsEditable()
  ElementStream<Event> get onDoubleClick =>
      Element.doubleClickEvent._forElementList(this);

  /**
   * A stream of `drag` events fired when this element currently being dragged.
   *
   * A `drag` event is added to this stream as soon as the drag begins.
   * A `drag` event is also added to this stream at intervals while the drag
   * operation is still ongoing.
   *
   * ## Other resources
   *
   * * [Drag and drop
   *   sample](https://github.com/dart-lang/dart-samples/tree/master/html5/web/dnd/basics)
   *   based on [the tutorial](http://www.html5rocks.com/en/tutorials/dnd/basics/)
   *   from HTML5Rocks.
   * * [Drag and drop
   *   specification](https://html.spec.whatwg.org/multipage/interaction.html#dnd)
   *   from WHATWG.
   */
  @DomName('Element.ondrag')
  @DocsEditable()
  ElementStream<MouseEvent> get onDrag =>
      Element.dragEvent._forElementList(this);

  /**
   * A stream of `dragend` events fired when this element completes a drag
   * operation.
   *
   * ## Other resources
   *
   * * [Drag and drop
   *   sample](https://github.com/dart-lang/dart-samples/tree/master/html5/web/dnd/basics)
   *   based on [the tutorial](http://www.html5rocks.com/en/tutorials/dnd/basics/)
   *   from HTML5Rocks.
   * * [Drag and drop
   *   specification](https://html.spec.whatwg.org/multipage/interaction.html#dnd)
   *   from WHATWG.
   */
  @DomName('Element.ondragend')
  @DocsEditable()
  ElementStream<MouseEvent> get onDragEnd =>
      Element.dragEndEvent._forElementList(this);

  /**
   * A stream of `dragenter` events fired when a dragged object is first dragged
   * over this element.
   *
   * ## Other resources
   *
   * * [Drag and drop
   *   sample](https://github.com/dart-lang/dart-samples/tree/master/html5/web/dnd/basics)
   *   based on [the tutorial](http://www.html5rocks.com/en/tutorials/dnd/basics/)
   *   from HTML5Rocks.
   * * [Drag and drop
   *   specification](https://html.spec.whatwg.org/multipage/interaction.html#dnd)
   *   from WHATWG.
   */
  @DomName('Element.ondragenter')
  @DocsEditable()
  ElementStream<MouseEvent> get onDragEnter =>
      Element.dragEnterEvent._forElementList(this);

  /**
   * A stream of `dragleave` events fired when an object being dragged over this
   * element leaves this element's target area.
   *
   * ## Other resources
   *
   * * [Drag and drop
   *   sample](https://github.com/dart-lang/dart-samples/tree/master/html5/web/dnd/basics)
   *   based on [the tutorial](http://www.html5rocks.com/en/tutorials/dnd/basics/)
   *   from HTML5Rocks.
   * * [Drag and drop
   *   specification](https://html.spec.whatwg.org/multipage/interaction.html#dnd)
   *   from WHATWG.
   */
  @DomName('Element.ondragleave')
  @DocsEditable()
  ElementStream<MouseEvent> get onDragLeave =>
      Element.dragLeaveEvent._forElementList(this);

  /**
   * A stream of `dragover` events fired when a dragged object is currently
   * being dragged over this element.
   *
   * ## Other resources
   *
   * * [Drag and drop
   *   sample](https://github.com/dart-lang/dart-samples/tree/master/html5/web/dnd/basics)
   *   based on [the tutorial](http://www.html5rocks.com/en/tutorials/dnd/basics/)
   *   from HTML5Rocks.
   * * [Drag and drop
   *   specification](https://html.spec.whatwg.org/multipage/interaction.html#dnd)
   *   from WHATWG.
   */
  @DomName('Element.ondragover')
  @DocsEditable()
  ElementStream<MouseEvent> get onDragOver =>
      Element.dragOverEvent._forElementList(this);

  /**
   * A stream of `dragstart` events fired when this element starts being
   * dragged.
   *
   * ## Other resources
   *
   * * [Drag and drop
   *   sample](https://github.com/dart-lang/dart-samples/tree/master/html5/web/dnd/basics)
   *   based on [the tutorial](http://www.html5rocks.com/en/tutorials/dnd/basics/)
   *   from HTML5Rocks.
   * * [Drag and drop
   *   specification](https://html.spec.whatwg.org/multipage/interaction.html#dnd)
   *   from WHATWG.
   */
  @DomName('Element.ondragstart')
  @DocsEditable()
  ElementStream<MouseEvent> get onDragStart =>
      Element.dragStartEvent._forElementList(this);

  /**
   * A stream of `drop` events fired when a dragged object is dropped on this
   * element.
   *
   * ## Other resources
   *
   * * [Drag and drop
   *   sample](https://github.com/dart-lang/dart-samples/tree/master/html5/web/dnd/basics)
   *   based on [the tutorial](http://www.html5rocks.com/en/tutorials/dnd/basics/)
   *   from HTML5Rocks.
   * * [Drag and drop
   *   specification](https://html.spec.whatwg.org/multipage/interaction.html#dnd)
   *   from WHATWG.
   */
  @DomName('Element.ondrop')
  @DocsEditable()
  ElementStream<MouseEvent> get onDrop =>
      Element.dropEvent._forElementList(this);

  @DomName('Element.ondurationchange')
  @DocsEditable()
  @Experimental() // untriaged
  ElementStream<Event> get onDurationChange =>
      Element.durationChangeEvent._forElementList(this);

  @DomName('Element.onemptied')
  @DocsEditable()
  @Experimental() // untriaged
  ElementStream<Event> get onEmptied =>
      Element.emptiedEvent._forElementList(this);

  @DomName('Element.onended')
  @DocsEditable()
  @Experimental() // untriaged
  ElementStream<Event> get onEnded => Element.endedEvent._forElementList(this);

  /// Stream of `error` events handled by this [Element].
  @DomName('Element.onerror')
  @DocsEditable()
  ElementStream<Event> get onError => Element.errorEvent._forElementList(this);

  /// Stream of `focus` events handled by this [Element].
  @DomName('Element.onfocus')
  @DocsEditable()
  ElementStream<Event> get onFocus => Element.focusEvent._forElementList(this);

  /// Stream of `input` events handled by this [Element].
  @DomName('Element.oninput')
  @DocsEditable()
  ElementStream<Event> get onInput => Element.inputEvent._forElementList(this);

  /// Stream of `invalid` events handled by this [Element].
  @DomName('Element.oninvalid')
  @DocsEditable()
  ElementStream<Event> get onInvalid =>
      Element.invalidEvent._forElementList(this);

  /// Stream of `keydown` events handled by this [Element].
  @DomName('Element.onkeydown')
  @DocsEditable()
  ElementStream<KeyboardEvent> get onKeyDown =>
      Element.keyDownEvent._forElementList(this);

  /// Stream of `keypress` events handled by this [Element].
  @DomName('Element.onkeypress')
  @DocsEditable()
  ElementStream<KeyboardEvent> get onKeyPress =>
      Element.keyPressEvent._forElementList(this);

  /// Stream of `keyup` events handled by this [Element].
  @DomName('Element.onkeyup')
  @DocsEditable()
  ElementStream<KeyboardEvent> get onKeyUp =>
      Element.keyUpEvent._forElementList(this);

  /// Stream of `load` events handled by this [Element].
  @DomName('Element.onload')
  @DocsEditable()
  ElementStream<Event> get onLoad => Element.loadEvent._forElementList(this);

  @DomName('Element.onloadeddata')
  @DocsEditable()
  @Experimental() // untriaged
  ElementStream<Event> get onLoadedData =>
      Element.loadedDataEvent._forElementList(this);

  @DomName('Element.onloadedmetadata')
  @DocsEditable()
  @Experimental() // untriaged
  ElementStream<Event> get onLoadedMetadata =>
      Element.loadedMetadataEvent._forElementList(this);

  /// Stream of `mousedown` events handled by this [Element].
  @DomName('Element.onmousedown')
  @DocsEditable()
  ElementStream<MouseEvent> get onMouseDown =>
      Element.mouseDownEvent._forElementList(this);

  /// Stream of `mouseenter` events handled by this [Element].
  @DomName('Element.onmouseenter')
  @DocsEditable()
  @Experimental() // untriaged
  ElementStream<MouseEvent> get onMouseEnter =>
      Element.mouseEnterEvent._forElementList(this);

  /// Stream of `mouseleave` events handled by this [Element].
  @DomName('Element.onmouseleave')
  @DocsEditable()
  @Experimental() // untriaged
  ElementStream<MouseEvent> get onMouseLeave =>
      Element.mouseLeaveEvent._forElementList(this);

  /// Stream of `mousemove` events handled by this [Element].
  @DomName('Element.onmousemove')
  @DocsEditable()
  ElementStream<MouseEvent> get onMouseMove =>
      Element.mouseMoveEvent._forElementList(this);

  /// Stream of `mouseout` events handled by this [Element].
  @DomName('Element.onmouseout')
  @DocsEditable()
  ElementStream<MouseEvent> get onMouseOut =>
      Element.mouseOutEvent._forElementList(this);

  /// Stream of `mouseover` events handled by this [Element].
  @DomName('Element.onmouseover')
  @DocsEditable()
  ElementStream<MouseEvent> get onMouseOver =>
      Element.mouseOverEvent._forElementList(this);

  /// Stream of `mouseup` events handled by this [Element].
  @DomName('Element.onmouseup')
  @DocsEditable()
  ElementStream<MouseEvent> get onMouseUp =>
      Element.mouseUpEvent._forElementList(this);

  /// Stream of `mousewheel` events handled by this [Element].
  @DomName('Element.onmousewheel')
  @DocsEditable()
  // http://www.w3.org/TR/DOM-Level-3-Events/#events-wheelevents
  @Experimental() // non-standard
  ElementStream<WheelEvent> get onMouseWheel =>
      Element.mouseWheelEvent._forElementList(this);

  /// Stream of `paste` events handled by this [Element].
  @DomName('Element.onpaste')
  @DocsEditable()
  ElementStream<ClipboardEvent> get onPaste =>
      Element.pasteEvent._forElementList(this);

  @DomName('Element.onpause')
  @DocsEditable()
  @Experimental() // untriaged
  ElementStream<Event> get onPause => Element.pauseEvent._forElementList(this);

  @DomName('Element.onplay')
  @DocsEditable()
  @Experimental() // untriaged
  ElementStream<Event> get onPlay => Element.playEvent._forElementList(this);

  @DomName('Element.onplaying')
  @DocsEditable()
  @Experimental() // untriaged
  ElementStream<Event> get onPlaying =>
      Element.playingEvent._forElementList(this);

  @DomName('Element.onratechange')
  @DocsEditable()
  @Experimental() // untriaged
  ElementStream<Event> get onRateChange =>
      Element.rateChangeEvent._forElementList(this);

  /// Stream of `reset` events handled by this [Element].
  @DomName('Element.onreset')
  @DocsEditable()
  ElementStream<Event> get onReset => Element.resetEvent._forElementList(this);

  @DomName('Element.onresize')
  @DocsEditable()
  @Experimental() // untriaged
  ElementStream<Event> get onResize =>
      Element.resizeEvent._forElementList(this);

  /// Stream of `scroll` events handled by this [Element].
  @DomName('Element.onscroll')
  @DocsEditable()
  ElementStream<Event> get onScroll =>
      Element.scrollEvent._forElementList(this);

  /// Stream of `search` events handled by this [Element].
  @DomName('Element.onsearch')
  @DocsEditable()
  // http://www.w3.org/TR/html-markup/input.search.html
  @Experimental()
  ElementStream<Event> get onSearch =>
      Element.searchEvent._forElementList(this);

  @DomName('Element.onseeked')
  @DocsEditable()
  @Experimental() // untriaged
  ElementStream<Event> get onSeeked =>
      Element.seekedEvent._forElementList(this);

  @DomName('Element.onseeking')
  @DocsEditable()
  @Experimental() // untriaged
  ElementStream<Event> get onSeeking =>
      Element.seekingEvent._forElementList(this);

  /// Stream of `select` events handled by this [Element].
  @DomName('Element.onselect')
  @DocsEditable()
  ElementStream<Event> get onSelect =>
      Element.selectEvent._forElementList(this);

  /// Stream of `selectstart` events handled by this [Element].
  @DomName('Element.onselectstart')
  @DocsEditable()
  @Experimental() // nonstandard
  ElementStream<Event> get onSelectStart =>
      Element.selectStartEvent._forElementList(this);

  @DomName('Element.onstalled')
  @DocsEditable()
  @Experimental() // untriaged
  ElementStream<Event> get onStalled =>
      Element.stalledEvent._forElementList(this);

  /// Stream of `submit` events handled by this [Element].
  @DomName('Element.onsubmit')
  @DocsEditable()
  ElementStream<Event> get onSubmit =>
      Element.submitEvent._forElementList(this);

  @DomName('Element.onsuspend')
  @DocsEditable()
  @Experimental() // untriaged
  ElementStream<Event> get onSuspend =>
      Element.suspendEvent._forElementList(this);

  @DomName('Element.ontimeupdate')
  @DocsEditable()
  @Experimental() // untriaged
  ElementStream<Event> get onTimeUpdate =>
      Element.timeUpdateEvent._forElementList(this);

  /// Stream of `touchcancel` events handled by this [Element].
  @DomName('Element.ontouchcancel')
  @DocsEditable()
  // http://www.w3.org/TR/touch-events/, http://www.chromestatus.com/features
  @Experimental()
  ElementStream<TouchEvent> get onTouchCancel =>
      Element.touchCancelEvent._forElementList(this);

  /// Stream of `touchend` events handled by this [Element].
  @DomName('Element.ontouchend')
  @DocsEditable()
  // http://www.w3.org/TR/touch-events/, http://www.chromestatus.com/features
  @Experimental()
  ElementStream<TouchEvent> get onTouchEnd =>
      Element.touchEndEvent._forElementList(this);

  /// Stream of `touchenter` events handled by this [Element].
  @DomName('Element.ontouchenter')
  @DocsEditable()
  // http://www.w3.org/TR/touch-events/, http://www.chromestatus.com/features
  @Experimental()
  ElementStream<TouchEvent> get onTouchEnter =>
      Element.touchEnterEvent._forElementList(this);

  /// Stream of `touchleave` events handled by this [Element].
  @DomName('Element.ontouchleave')
  @DocsEditable()
  // http://www.w3.org/TR/touch-events/, http://www.chromestatus.com/features
  @Experimental()
  ElementStream<TouchEvent> get onTouchLeave =>
      Element.touchLeaveEvent._forElementList(this);

  /// Stream of `touchmove` events handled by this [Element].
  @DomName('Element.ontouchmove')
  @DocsEditable()
  // http://www.w3.org/TR/touch-events/, http://www.chromestatus.com/features
  @Experimental()
  ElementStream<TouchEvent> get onTouchMove =>
      Element.touchMoveEvent._forElementList(this);

  /// Stream of `touchstart` events handled by this [Element].
  @DomName('Element.ontouchstart')
  @DocsEditable()
  // http://www.w3.org/TR/touch-events/, http://www.chromestatus.com/features
  @Experimental()
  ElementStream<TouchEvent> get onTouchStart =>
      Element.touchStartEvent._forElementList(this);

  /// Stream of `transitionend` events handled by this [Element].
  @DomName('Element.ontransitionend')
  @DocsEditable()
  @SupportedBrowser(SupportedBrowser.CHROME)
  @SupportedBrowser(SupportedBrowser.FIREFOX)
  @SupportedBrowser(SupportedBrowser.IE, '10')
  @SupportedBrowser(SupportedBrowser.SAFARI)
  ElementStream<TransitionEvent> get onTransitionEnd =>
      Element.transitionEndEvent._forElementList(this);

  @DomName('Element.onvolumechange')
  @DocsEditable()
  @Experimental() // untriaged
  ElementStream<Event> get onVolumeChange =>
      Element.volumeChangeEvent._forElementList(this);

  @DomName('Element.onwaiting')
  @DocsEditable()
  @Experimental() // untriaged
  ElementStream<Event> get onWaiting =>
      Element.waitingEvent._forElementList(this);

  /// Stream of `fullscreenchange` events handled by this [Element].
  @DomName('Element.onwebkitfullscreenchange')
  @DocsEditable()
  // https://dvcs.w3.org/hg/fullscreen/raw-file/tip/Overview.html
  @Experimental()
  ElementStream<Event> get onFullscreenChange =>
      Element.fullscreenChangeEvent._forElementList(this);

  /// Stream of `fullscreenerror` events handled by this [Element].
  @DomName('Element.onwebkitfullscreenerror')
  @DocsEditable()
  // https://dvcs.w3.org/hg/fullscreen/raw-file/tip/Overview.html
  @Experimental()
  ElementStream<Event> get onFullscreenError =>
      Element.fullscreenErrorEvent._forElementList(this);
}

@DocsEditable()
/**
 * An abstract class, which all HTML elements extend.
 */
@DomName('Element')
class Element extends Node
    implements
        NonDocumentTypeChildNode,
        GlobalEventHandlers,
        ParentNode,
        ChildNode {
  /**
   * Creates an HTML element from a valid fragment of HTML.
   *
   *     var element = new Element.html('<div class="foo">content</div>');
   *
   * The HTML fragment should contain only one single root element, any
   * leading or trailing text nodes will be removed.
   *
   * The HTML fragment is parsed as if it occurred within the context of a
   * `<body>` tag, this means that special elements such as `<caption>` which
   * must be parsed within the scope of a `<table>` element will be dropped. Use
   * [createFragment] to parse contextual HTML fragments.
   *
   * Unless a validator is provided this will perform the default validation
   * and remove all scriptable elements and attributes.
   *
   * See also:
   *
   * * [NodeValidator]
   *
   */
  factory Element.html(String html,
      {NodeValidator validator, NodeTreeSanitizer treeSanitizer}) {
    var fragment = document.body.createFragment(html,
        validator: validator, treeSanitizer: treeSanitizer);

    return fragment.nodes.where((e) => e is Element).single;
  }

  /**
   * Custom element creation constructor.
   *
   * This constructor is used by the DOM when a custom element has been
   * created. It can only be invoked by subclasses of Element from
   * that classes created constructor.
   *
   *     class CustomElement extends Element {
   *       factory CustomElement() => new Element.tag('x-custom');
   *
   *       CustomElement.created() : super.created() {
   *          // Perform any element initialization.
   *       }
   *     }
   *     document.registerElement('x-custom', CustomElement);
   */
  Element.created() : super._created() {
    // Validate that this is a custom element & possibly perform additional
    // initialization.
    _blink.Blink_Utils.initializeCustomElement(this);
  }

  /**
   * Creates the HTML element specified by the tag name.
   *
   * This is similar to [Document.createElement].
   * [tag] should be a valid HTML tag name. If [tag] is an unknown tag then
   * this will create an [UnknownElement].
   *
   *     var divElement = new Element.tag('div');
   *     print(divElement is DivElement); // 'true'
   *     var myElement = new Element.tag('unknownTag');
   *     print(myElement is UnknownElement); // 'true'
   *
   * For standard elements it is better to use the element type constructors:
   *
   *     var element = new DivElement();
   *
   * It is better to use e.g `new CanvasElement()` because the type of the
   * expression is `CanvasElement`, whereas the type of `Element.tag` is the
   * less specific `Element`.
   *
   * See also:
   *
   * * [isTagSupported]
   */
  factory Element.tag(String tag, [String typeExtention]) =>
      _ElementFactoryProvider.createElement_tag(tag, typeExtention);

  /// Creates a new `<a>` element.
  ///
  /// This is equivalent to calling `new Element.tag('a')`.
  factory Element.a() => new AnchorElement();

  /// Creates a new `<article>` element.
  ///
  /// This is equivalent to calling `new Element.tag('article')`.
  factory Element.article() => new Element.tag('article');

  /// Creates a new `<aside>` element.
  ///
  /// This is equivalent to calling `new Element.tag('aside')`.
  factory Element.aside() => new Element.tag('aside');

  /// Creates a new `<audio>` element.
  ///
  /// This is equivalent to calling `new Element.tag('audio')`.
  factory Element.audio() => new Element.tag('audio');

  /// Creates a new `<br>` element.
  ///
  /// This is equivalent to calling `new Element.tag('br')`.
  factory Element.br() => new BRElement();

  /// Creates a new `<canvas>` element.
  ///
  /// This is equivalent to calling `new Element.tag('canvas')`.
  factory Element.canvas() => new CanvasElement();

  /// Creates a new `<div>` element.
  ///
  /// This is equivalent to calling `new Element.tag('div')`.
  factory Element.div() => new DivElement();

  /// Creates a new `<footer>` element.
  ///
  /// This is equivalent to calling `new Element.tag('footer')`.
  factory Element.footer() => new Element.tag('footer');

  /// Creates a new `<header>` element.
  ///
  /// This is equivalent to calling `new Element.tag('header')`.
  factory Element.header() => new Element.tag('header');

  /// Creates a new `<hr>` element.
  ///
  /// This is equivalent to calling `new Element.tag('hr')`.
  factory Element.hr() => new Element.tag('hr');

  /// Creates a new `<iframe>` element.
  ///
  /// This is equivalent to calling `new Element.tag('iframe')`.
  factory Element.iframe() => new Element.tag('iframe');

  /// Creates a new `<img>` element.
  ///
  /// This is equivalent to calling `new Element.tag('img')`.
  factory Element.img() => new Element.tag('img');

  /// Creates a new `<li>` element.
  ///
  /// This is equivalent to calling `new Element.tag('li')`.
  factory Element.li() => new Element.tag('li');

  /// Creates a new `<nav>` element.
  ///
  /// This is equivalent to calling `new Element.tag('nav')`.
  factory Element.nav() => new Element.tag('nav');

  /// Creates a new `<ol>` element.
  ///
  /// This is equivalent to calling `new Element.tag('ol')`.
  factory Element.ol() => new Element.tag('ol');

  /// Creates a new `<option>` element.
  ///
  /// This is equivalent to calling `new Element.tag('option')`.
  factory Element.option() => new Element.tag('option');

  /// Creates a new `<p>` element.
  ///
  /// This is equivalent to calling `new Element.tag('p')`.
  factory Element.p() => new Element.tag('p');

  /// Creates a new `<pre>` element.
  ///
  /// This is equivalent to calling `new Element.tag('pre')`.
  factory Element.pre() => new Element.tag('pre');

  /// Creates a new `<section>` element.
  ///
  /// This is equivalent to calling `new Element.tag('section')`.
  factory Element.section() => new Element.tag('section');

  /// Creates a new `<select>` element.
  ///
  /// This is equivalent to calling `new Element.tag('select')`.
  factory Element.select() => new Element.tag('select');

  /// Creates a new `<span>` element.
  ///
  /// This is equivalent to calling `new Element.tag('span')`.
  factory Element.span() => new Element.tag('span');

  /// Creates a new `<svg>` element.
  ///
  /// This is equivalent to calling `new Element.tag('svg')`.
  factory Element.svg() => new Element.tag('svg');

  /// Creates a new `<table>` element.
  ///
  /// This is equivalent to calling `new Element.tag('table')`.
  factory Element.table() => new Element.tag('table');

  /// Creates a new `<td>` element.
  ///
  /// This is equivalent to calling `new Element.tag('td')`.
  factory Element.td() => new Element.tag('td');

  /// Creates a new `<textarea>` element.
  ///
  /// This is equivalent to calling `new Element.tag('textarea')`.
  factory Element.textarea() => new Element.tag('textarea');

  /// Creates a new `<th>` element.
  ///
  /// This is equivalent to calling `new Element.tag('th')`.
  factory Element.th() => new Element.tag('th');

  /// Creates a new `<tr>` element.
  ///
  /// This is equivalent to calling `new Element.tag('tr')`.
  factory Element.tr() => new Element.tag('tr');

  /// Creates a new `<ul>` element.
  ///
  /// This is equivalent to calling `new Element.tag('ul')`.
  factory Element.ul() => new Element.tag('ul');

  /// Creates a new `<video>` element.
  ///
  /// This is equivalent to calling `new Element.tag('video')`.
  factory Element.video() => new Element.tag('video');

  /**
   * All attributes on this element.
   *
   * Any modifications to the attribute map will automatically be applied to
   * this element.
   *
   * This only includes attributes which are not in a namespace
   * (such as 'xlink:href'), additional attributes can be accessed via
   * [getNamespacedAttributes].
   */
  Map<String, String> get attributes => new _ElementAttributeMap(this);

  set attributes(Map<String, String> value) {
    Map<String, String> attributes = this.attributes;
    attributes.clear();
    for (String key in value.keys) {
      attributes[key] = value[key];
    }
  }

  /**
   * List of the direct children of this element.
   *
   * This collection can be used to add and remove elements from the document.
   *
   *     var item = new DivElement();
   *     item.text = 'Something';
   *     document.body.children.add(item) // Item is now displayed on the page.
   *     for (var element in document.body.children) {
   *       element.style.background = 'red'; // Turns every child of body red.
   *     }
   */
  List<Element> get children => new _ChildrenElementList._wrap(this);

  set children(List<Element> value) {
    // Copy list first since we don't want liveness during iteration.
    var copy = value.toList();
    var children = this.children;
    children.clear();
    children.addAll(copy);
  }

  /**
   * Finds all descendent elements of this element that match the specified
   * group of selectors.
   *
   * [selectors] should be a string using CSS selector syntax.
   *
   *     var items = element.querySelectorAll('.itemClassName');
   *
   * For details about CSS selector syntax, see the
   * [CSS selector specification](http://www.w3.org/TR/css3-selectors/).
   */
  @DomName('Element.querySelectorAll')
  ElementList<Element/*=T*/ > querySelectorAll/*<T extends Element>*/(
          String selectors) =>
      new _FrozenElementList/*<T>*/ ._wrap(_querySelectorAll(selectors));

  /**
   * Alias for [querySelector]. Note this function is deprecated because its
   * semantics will be changing in the future.
   */
  @deprecated
  @DomName('Element.querySelector')
  @Experimental()
  Element query(String relativeSelectors) => querySelector(relativeSelectors);

  /**
   * Alias for [querySelectorAll]. Note this function is deprecated because its
   * semantics will be changing in the future.
   */
  @deprecated
  @DomName('Element.querySelectorAll')
  @Experimental()
  ElementList<Element/*=T*/ > queryAll/*<T extends Element>*/(
          String relativeSelectors) =>
      querySelectorAll(relativeSelectors);

  /**
   * The set of CSS classes applied to this element.
   *
   * This set makes it easy to add, remove or toggle the classes applied to
   * this element.
   *
   *     element.classes.add('selected');
   *     element.classes.toggle('isOnline');
   *     element.classes.remove('selected');
   */
  CssClassSet get classes => new _ElementCssClassSet(this);

  set classes(Iterable<String> value) {
    // TODO(sra): Do this without reading the classes in clear() and addAll(),
    // or writing the classes in clear().
    CssClassSet classSet = classes;
    classSet.clear();
    classSet.addAll(value);
  }

  /**
   * Allows access to all custom data attributes (data-*) set on this element.
   *
   * The keys for the map must follow these rules:
   *
   * * The name must not begin with 'xml'.
   * * The name cannot contain a semi-colon (';').
   * * The name cannot contain any capital letters.
   *
   * Any keys from markup will be converted to camel-cased keys in the map.
   *
   * For example, HTML specified as:
   *
   *     <div data-my-random-value='value'></div>
   *
   * Would be accessed in Dart as:
   *
   *     var value = element.dataset['myRandomValue'];
   *
   * See also:
   *
   * * [Custom data
   *   attributes](http://dev.w3.org/html5/spec-preview/global-attributes.html#custom-data-attribute)
   */
  Map<String, String> get dataset => new _DataAttributeMap(attributes);

  set dataset(Map<String, String> value) {
    final data = this.dataset;
    data.clear();
    for (String key in value.keys) {
      data[key] = value[key];
    }
  }

  /**
   * Gets a map for manipulating the attributes of a particular namespace.
   *
   * This is primarily useful for SVG attributes such as xref:link.
   */
  Map<String, String> getNamespacedAttributes(String namespace) {
    return new _NamespacedAttributeMap(this, namespace);
  }

  /**
   * The set of all CSS values applied to this element, including inherited
   * and default values.
   *
   * The computedStyle contains values that are inherited from other
   * sources, such as parent elements or stylesheets. This differs from the
   * [style] property, which contains only the values specified directly on this
   * element.
   *
   * PseudoElement can be values such as `::after`, `::before`, `::marker`,
   * `::line-marker`.
   *
   * See also:
   *
   * * [CSS Inheritance and Cascade](http://docs.webplatform.org/wiki/tutorials/inheritance_and_cascade)
   * * [Pseudo-elements](http://docs.webplatform.org/wiki/css/selectors/pseudo-elements)
   */
  CssStyleDeclaration getComputedStyle([String pseudoElement]) {
    if (pseudoElement == null) {
      pseudoElement = '';
    }
    // TODO(jacobr): last param should be null, see b/5045788
    return window._getComputedStyle(this, pseudoElement);
  }

  /**
   * Gets the position of this element relative to the client area of the page.
   */
  Rectangle get client =>
      new Rectangle(clientLeft, clientTop, clientWidth, clientHeight);

  /**
   * Gets the offset of this element relative to its offsetParent.
   */
  Rectangle get offset =>
      new Rectangle(offsetLeft, offsetTop, offsetWidth, offsetHeight);

  /**
   * Adds the specified text after the last child of this element.
   */
  void appendText(String text) {
    this.append(new Text(text));
  }

  /**
   * Parses the specified text as HTML and adds the resulting node after the
   * last child of this element.
   */
  void appendHtml(String text,
      {NodeValidator validator, NodeTreeSanitizer treeSanitizer}) {
    this.insertAdjacentHtml('beforeend', text,
        validator: validator, treeSanitizer: treeSanitizer);
  }

  /**
   * Checks to see if the tag name is supported by the current platform.
   *
   * The tag should be a valid HTML tag name.
   */
  static bool isTagSupported(String tag) {
    var e = _ElementFactoryProvider.createElement_tag(tag, null);
    return e is Element && !(e is UnknownElement);
  }

  /**
   * Called by the DOM when this element has been inserted into the live
   * document.
   *
   * More information can be found in the
   * [Custom Elements](http://w3c.github.io/webcomponents/spec/custom/#dfn-attached-callback)
   * draft specification.
   */
  @Experimental()
  void attached() {
    // For the deprecation period, call the old callback.
    enteredView();
  }

  /**
   * Called by the DOM when this element has been removed from the live
   * document.
   *
   * More information can be found in the
   * [Custom Elements](http://w3c.github.io/webcomponents/spec/custom/#dfn-detached-callback)
   * draft specification.
   */
  @Experimental()
  void detached() {
    // For the deprecation period, call the old callback.
    leftView();
  }

  /** *Deprecated*: override [attached] instead. */
  @Experimental()
  @deprecated
  void enteredView() {}

  /** *Deprecated*: override [detached] instead. */
  @Experimental()
  @deprecated
  void leftView() {}

  /**
   * Creates a new AnimationEffect object whose target element is the object
   * on which the method is called, and calls the play() method of the
   * AnimationTimeline object of the document timeline of the node document
   * of the element, passing the newly created AnimationEffect as the argument
   * to the method. Returns an Animation for the effect.
   *
   * Examples
   *
   *     var animation = elem.animate([{"opacity": 75}, {"opacity": 0}], 200);
   *
   *     var animation = elem.animate([
   *       {"transform": "translate(100px, -100%)"},
   *       {"transform" : "translate(400px, 500px)"}
   *     ], 1500);
   *
   * The [frames] parameter is an Iterable<Map>, where the
   * map entries specify CSS animation effects. The
   * [timing] paramter can be a double, representing the number of milliseconds
   * for the transition, or a Map with fields corresponding to those
   * of the [Timing] object.
  **/
  @Experimental()
  @SupportedBrowser(SupportedBrowser.CHROME, '36')
  Animation animate(Iterable<Map<String, dynamic>> frames, [timing]) {
    if (frames is! Iterable || !(frames.every((x) => x is Map))) {
      throw new ArgumentError("The frames parameter should be a List of Maps "
          "with frame information");
    }
    var convertedFrames;
    if (frames is Iterable) {
      convertedFrames = convertDartToNative_List(
          frames.map(convertDartToNative_Dictionary).toList());
    } else {
      convertedFrames = frames;
    }
    var convertedTiming =
        timing is Map ? convertDartToNative_Dictionary(timing) : timing;
    return convertedTiming == null
        ? _animate(convertedFrames)
        : _animate(convertedFrames, convertedTiming);
  }

  /**
   * Called by the DOM whenever an attribute on this has been changed.
   */
  void attributeChanged(String name, String oldValue, String newValue) {}

  // Hooks to support custom WebComponents.

  Element _xtag;

  /**
   * Experimental support for [web components][wc]. This field stores a
   * reference to the component implementation. It was inspired by Mozilla's
   * [x-tags][] project. Please note: in the future it may be possible to
   * `extend Element` from your class, in which case this field will be
   * deprecated.
   *
   * If xtag has not been set, it will simply return `this` [Element].
   *
   * [wc]: http://dvcs.w3.org/hg/webcomponents/raw-file/tip/explainer/index.html
   * [x-tags]: http://x-tags.org/
   */
  // Note: return type is `dynamic` for convenience to suppress warnings when
  // members of the component are used. The actual type is a subtype of Element.
  get xtag => _xtag != null ? _xtag : this;

  set xtag(Element value) {
    _xtag = value;
  }

  @DomName('Element.localName')
  @DocsEditable()
  String get localName => _localName;

  /**
   * A URI that identifies the XML namespace of this element.
   *
   * `null` if no namespace URI is specified.
   *
   * ## Other resources
   *
   * * [Node.namespaceURI](http://www.w3.org/TR/DOM-Level-3-Core/core.html#ID-NodeNSname)
   *   from W3C.
   */
  @DomName('Element.namespaceUri')
  String get namespaceUri => _namespaceUri;

  /**
   * The string representation of this element.
   *
   * This is equivalent to reading the [localName] property.
   */
  String toString() => localName;

  /**
   * Scrolls this element into view.
   *
   * Only one of of the alignment options may be specified at a time.
   *
   * If no options are specified then this will attempt to scroll the minimum
   * amount needed to bring the element into view.
   *
   * Note that alignCenter is currently only supported on WebKit platforms. If
   * alignCenter is specified but not supported then this will fall back to
   * alignTop.
   *
   * See also:
   *
   * * [scrollIntoView](http://docs.webplatform.org/wiki/dom/methods/scrollIntoView)
   * * [scrollIntoViewIfNeeded](http://docs.webplatform.org/wiki/dom/methods/scrollIntoViewIfNeeded)
   */
  void scrollIntoView([ScrollAlignment alignment]) {
    var hasScrollIntoViewIfNeeded = true;
    if (alignment == ScrollAlignment.TOP) {
      this._scrollIntoView(true);
    } else if (alignment == ScrollAlignment.BOTTOM) {
      this._scrollIntoView(false);
    } else if (hasScrollIntoViewIfNeeded) {
      if (alignment == ScrollAlignment.CENTER) {
        this._scrollIntoViewIfNeeded(true);
      } else {
        this._scrollIntoViewIfNeeded();
      }
    } else {
      this._scrollIntoView();
    }
  }

  /**
   * Parses text as an HTML fragment and inserts it into the DOM at the
   * specified location.
   *
   * The [where] parameter indicates where to insert the HTML fragment:
   *
   * * 'beforeBegin': Immediately before this element.
   * * 'afterBegin': As the first child of this element.
   * * 'beforeEnd': As the last child of this element.
   * * 'afterEnd': Immediately after this element.
   *
   *     var html = '<div class="something">content</div>';
   *     // Inserts as the first child
   *     document.body.insertAdjacentHtml('afterBegin', html);
   *     var createdElement = document.body.children[0];
   *     print(createdElement.classes[0]); // Prints 'something'
   *
   * See also:
   *
   * * [insertAdjacentText]
   * * [insertAdjacentElement]
   */
  void insertAdjacentHtml(String where, String html,
      {NodeValidator validator, NodeTreeSanitizer treeSanitizer}) {
    if (treeSanitizer is _TrustedHtmlTreeSanitizer) {
      _insertAdjacentHtml(where, html);
    } else {
      _insertAdjacentNode(
          where,
          createFragment(html,
              validator: validator, treeSanitizer: treeSanitizer));
    }
  }

  void _insertAdjacentNode(String where, Node node) {
    switch (where.toLowerCase()) {
      case 'beforebegin':
        this.parentNode.insertBefore(node, this);
        break;
      case 'afterbegin':
        var first = this.nodes.length > 0 ? this.nodes[0] : null;
        this.insertBefore(node, first);
        break;
      case 'beforeend':
        this.append(node);
        break;
      case 'afterend':
        this.parentNode.insertBefore(node, this.nextNode);
        break;
      default:
        throw new ArgumentError("Invalid position ${where}");
    }
  }

  /** Checks if this element or any of its parents match the CSS selectors. */
  @Experimental()
  bool matchesWithAncestors(String selectors) {
    var elem = this;
    do {
      if (elem.matches(selectors)) return true;
      elem = elem.parent;
    } while (elem != null);
    return false;
  }

  /**
   * Access this element's content position.
   *
   * This returns a rectangle with the dimenions actually available for content
   * in this element, in pixels, regardless of this element's box-sizing
   * property. Unlike [getBoundingClientRect], the dimensions of this rectangle
   * will return the same numerical height if the element is hidden or not.
   *
   * _Important_ _note_: use of this method _will_ perform CSS calculations that
   * can trigger a browser reflow. Therefore, use of this property _during_ an
   * animation frame is discouraged. See also:
   * [Browser Reflow](https://developers.google.com/speed/articles/reflow)
   */
  @Experimental()
  CssRect get contentEdge => new _ContentCssRect(this);

  /**
   * Access the dimensions and position of this element's content + padding box.
   *
   * This returns a rectangle with the dimenions actually available for content
   * in this element, in pixels, regardless of this element's box-sizing
   * property. Unlike [getBoundingClientRect], the dimensions of this rectangle
   * will return the same numerical height if the element is hidden or not. This
   * can be used to retrieve jQuery's
   * [innerHeight](http://api.jquery.com/innerHeight/) value for an element.
   * This is also a rectangle equalling the dimensions of clientHeight and
   * clientWidth.
   *
   * _Important_ _note_: use of this method _will_ perform CSS calculations that
   * can trigger a browser reflow. Therefore, use of this property _during_ an
   * animation frame is discouraged. See also:
   * [Browser Reflow](https://developers.google.com/speed/articles/reflow)
   */
  @Experimental()
  CssRect get paddingEdge => new _PaddingCssRect(this);

  /**
   * Access the dimensions and position of this element's content + padding +
   * border box.
   *
   * This returns a rectangle with the dimenions actually available for content
   * in this element, in pixels, regardless of this element's box-sizing
   * property. Unlike [getBoundingClientRect], the dimensions of this rectangle
   * will return the same numerical height if the element is hidden or not. This
   * can be used to retrieve jQuery's
   * [outerHeight](http://api.jquery.com/outerHeight/) value for an element.
   *
   * _Important_ _note_: use of this method _will_ perform CSS calculations that
   * can trigger a browser reflow. Therefore, use of this property _during_ an
   * animation frame is discouraged. See also:
   * [Browser Reflow](https://developers.google.com/speed/articles/reflow)
   */
  @Experimental()
  CssRect get borderEdge => new _BorderCssRect(this);

  /**
   * Access the dimensions and position of this element's content + padding +
   * border + margin box.
   *
   * This returns a rectangle with the dimenions actually available for content
   * in this element, in pixels, regardless of this element's box-sizing
   * property. Unlike [getBoundingClientRect], the dimensions of this rectangle
   * will return the same numerical height if the element is hidden or not. This
   * can be used to retrieve jQuery's
   * [outerHeight](http://api.jquery.com/outerHeight/) value for an element.
   *
   * _Important_ _note_: use of this method will perform CSS calculations that
   * can trigger a browser reflow. Therefore, use of this property _during_ an
   * animation frame is discouraged. See also:
   * [Browser Reflow](https://developers.google.com/speed/articles/reflow)
   */
  @Experimental()
  CssRect get marginEdge => new _MarginCssRect(this);

  /**
   * Provides the coordinates of the element relative to the top of the
   * document.
   *
   * This method is the Dart equivalent to jQuery's
   * [offset](http://api.jquery.com/offset/) method.
   */
  @Experimental()
  Point get documentOffset => offsetTo(document.documentElement);

  /**
   * Provides the offset of this element's [borderEdge] relative to the
   * specified [parent].
   *
   * This is the Dart equivalent of jQuery's
   * [position](http://api.jquery.com/position/) method. Unlike jQuery's
   * position, however, [parent] can be any parent element of `this`,
   * rather than only `this`'s immediate [offsetParent]. If the specified
   * element is _not_ an offset parent or transitive offset parent to this
   * element, an [ArgumentError] is thrown.
   */
  @Experimental()
  Point offsetTo(Element parent) {
    return Element._offsetToHelper(this, parent);
  }

  static Point _offsetToHelper(Element current, Element parent) {
    // We're hopping from _offsetParent_ to offsetParent (not just parent), so
    // offsetParent, "tops out" at BODY. But people could conceivably pass in
    // the document.documentElement and I want it to return an absolute offset,
    // so we have the special case checking for HTML.
    bool sameAsParent = current == parent;
    bool foundAsParent = sameAsParent || parent.tagName == 'HTML';
    if (current == null || sameAsParent) {
      if (foundAsParent) return new Point/*<num>*/(0, 0);
      throw new ArgumentError("Specified element is not a transitive offset "
          "parent of this element.");
    }
    Element parentOffset = current.offsetParent;
    Point p = Element._offsetToHelper(parentOffset, parent);
    return new Point/*<num>*/(
        p.x + current.offsetLeft, p.y + current.offsetTop);
  }

  static HtmlDocument _parseDocument;
  static Range _parseRange;
  static NodeValidatorBuilder _defaultValidator;
  static _ValidatingTreeSanitizer _defaultSanitizer;

  /**
   * Create a DocumentFragment from the HTML fragment and ensure that it follows
   * the sanitization rules specified by the validator or treeSanitizer.
   *
   * If the default validation behavior is too restrictive then a new
   * NodeValidator should be created, either extending or wrapping a default
   * validator and overriding the validation APIs.
   *
   * The treeSanitizer is used to walk the generated node tree and sanitize it.
   * A custom treeSanitizer can also be provided to perform special validation
   * rules but since the API is more complex to implement this is discouraged.
   *
   * The returned tree is guaranteed to only contain nodes and attributes which
   * are allowed by the provided validator.
   *
   * See also:
   *
   * * [NodeValidator]
   * * [NodeTreeSanitizer]
   */
  DocumentFragment createFragment(String html,
      {NodeValidator validator, NodeTreeSanitizer treeSanitizer}) {
    if (treeSanitizer == null) {
      if (validator == null) {
        if (_defaultValidator == null) {
          _defaultValidator = new NodeValidatorBuilder.common();
        }
        validator = _defaultValidator;
      }
      if (_defaultSanitizer == null) {
        _defaultSanitizer = new _ValidatingTreeSanitizer(validator);
      } else {
        _defaultSanitizer.validator = validator;
      }
      treeSanitizer = _defaultSanitizer;
    } else if (validator != null) {
      throw new ArgumentError(
          'validator can only be passed if treeSanitizer is null');
    }

    if (_parseDocument == null) {
      _parseDocument = document.implementation.createHtmlDocument('');
      _parseRange = _parseDocument.createRange();

      // Workaround for Safari bug. Was also previously Chrome bug 229142
      // - URIs are not resolved in new doc.
      BaseElement base = _parseDocument.createElement('base');
      base.href = document.baseUri;
      _parseDocument.head.append(base);
    }
    var contextElement;
    if (this is BodyElement) {
      contextElement = _parseDocument.body;
    } else {
      contextElement = _parseDocument.createElement(tagName);
      _parseDocument.body.append(contextElement);
    }
    var fragment;
    if (Range.supportsCreateContextualFragment &&
        _canBeUsedToCreateContextualFragment) {
      _parseRange.selectNodeContents(contextElement);
      fragment = _parseRange.createContextualFragment(html);
    } else {
      contextElement._innerHtml = html;

      fragment = _parseDocument.createDocumentFragment();
      while (contextElement.firstChild != null) {
        fragment.append(contextElement.firstChild);
      }
    }
    if (contextElement != _parseDocument.body) {
      contextElement.remove();
    }

    treeSanitizer.sanitizeTree(fragment);
    // Copy the fragment over to the main document (fix for 14184)
    document.adoptNode(fragment);

    return fragment;
  }

  /** Test if createContextualFragment is supported for this element type */
  bool get _canBeUsedToCreateContextualFragment =>
      !_cannotBeUsedToCreateContextualFragment;

  /** Test if createContextualFragment is NOT supported for this element type */
  bool get _cannotBeUsedToCreateContextualFragment =>
      _tagsForWhichCreateContextualFragmentIsNotSupported.contains(tagName);

  /**
   * A hard-coded list of the tag names for which createContextualFragment
   * isn't supported.
   */
  static const _tagsForWhichCreateContextualFragmentIsNotSupported = const [
    'HEAD',
    'AREA',
    'BASE',
    'BASEFONT',
    'BR',
    'COL',
    'COLGROUP',
    'EMBED',
    'FRAME',
    'FRAMESET',
    'HR',
    'IMAGE',
    'IMG',
    'INPUT',
    'ISINDEX',
    'LINK',
    'META',
    'PARAM',
    'SOURCE',
    'STYLE',
    'TITLE',
    'WBR'
  ];

  /**
   * Parses the HTML fragment and sets it as the contents of this element.
   *
   * This uses the default sanitization behavior to sanitize the HTML fragment,
   * use [setInnerHtml] to override the default behavior.
   */
  set innerHtml(String html) {
    this.setInnerHtml(html);
  }

  /**
   * Parses the HTML fragment and sets it as the contents of this element.
   * This ensures that the generated content follows the sanitization rules
   * specified by the validator or treeSanitizer.
   *
   * If the default validation behavior is too restrictive then a new
   * NodeValidator should be created, either extending or wrapping a default
   * validator and overriding the validation APIs.
   *
   * The treeSanitizer is used to walk the generated node tree and sanitize it.
   * A custom treeSanitizer can also be provided to perform special validation
   * rules but since the API is more complex to implement this is discouraged.
   *
   * The resulting tree is guaranteed to only contain nodes and attributes which
   * are allowed by the provided validator.
   *
   * See also:
   *
   * * [NodeValidator]
   * * [NodeTreeSanitizer]
   */
  void setInnerHtml(String html,
      {NodeValidator validator, NodeTreeSanitizer treeSanitizer}) {
    text = null;
    if (treeSanitizer is _TrustedHtmlTreeSanitizer) {
      _innerHtml = html;
    } else {
      append(createFragment(html,
          validator: validator, treeSanitizer: treeSanitizer));
    }
  }

  String get innerHtml => _innerHtml;

  /**
   * This is an ease-of-use accessor for event streams which should only be
   * used when an explicit accessor is not available.
   */
  ElementEvents get on => new ElementEvents(this);

  /**
   * Verify if any of the attributes that we use in the sanitizer look unexpected,
   * possibly indicating DOM clobbering attacks.
   *
   * Those attributes are: attributes, lastChild, children, previousNode and tagName.
   */

  static var _namedNodeMap = js.context["NamedNodeMap"];
  static var _htmlCollection = js.context["HTMLCollection"];
  static var _nodeList = js.context["NodeList"];

  static const _evilAttributeNames = const [
    'attributes',
    'lastChild',
    'children',
    'childNodes'
  ];

  static bool _hasCorruptedAttributes(Element element) {
    // We have trusted access to children and to attributes of objects,
    // so we can inspect directly for attempts at DOM clobbering.
    var child = element.firstChild;
    while (child != null) {
      if (child is Element) {
        for (var attributeName in ["id", "name"]) {
          var childAttribute = child.getAttribute(attributeName);
          if (_evilAttributeNames.contains(childAttribute)) return true;
        }
      }
      child = child.nextNode;
    }
    return false;
  }

  /// A secondary check for corruption, needed on IE
  static bool _hasCorruptedAttributesAdditionalCheck(Element element) => false;

  static String _safeTagName(element) {
    try {
      // Safe as we plumb directly to a C++ native method.
      return element.tagName;
    } catch (e) {}
    return 'element tag unavailable';
  }

  // Need to explicitly delegate because Element is no longer abstract for Dartium.
  bool get isContentEditable =>
      _blink.BlinkHTMLElement.instance.isContentEditable_Getter_(this);
  void click() => _blink.BlinkHTMLElement.instance.click_Callback_0_(this);

  @DomName('Element.offsetParent')
  @DocsEditable()
  Element get offsetParent =>
      _blink.BlinkElement.instance.offsetParent_Getter_(this);

  @DomName('Element.offsetHeight')
  @DocsEditable()
  int get offsetHeight =>
      _blink.BlinkElement.instance.offsetHeight_Getter_(this);

  @DomName('Element.offsetLeft')
  @DocsEditable()
  int get offsetLeft => _blink.BlinkElement.instance.offsetLeft_Getter_(this);

  @DomName('Element.offsetTop')
  @DocsEditable()
  int get offsetTop => _blink.BlinkElement.instance.offsetTop_Getter_(this);

  @DomName('Element.offsetWidth')
  @DocsEditable()
  int get offsetWidth => _blink.BlinkElement.instance.offsetWidth_Getter_(this);

  @DomName('Element.scrollHeight')
  @DocsEditable()
  int get scrollHeight =>
      _blink.BlinkElement.instance.scrollHeight_Getter_(this).round();

  @DomName('Element.scrollLeft')
  @DocsEditable()
  int get scrollLeft =>
      _blink.BlinkElement.instance.scrollLeft_Getter_(this).round();

  @DomName('Element.scrollLeft')
  @DocsEditable()
  set scrollLeft(int value) =>
      _blink.BlinkElement.instance.scrollLeft_Setter_(this, value.round());

  @DomName('Element.scrollTop')
  @DocsEditable()
  int get scrollTop =>
      _blink.BlinkElement.instance.scrollTop_Getter_(this).round();

  @DomName('Element.scrollTop')
  @DocsEditable()
  set scrollTop(int value) =>
      _blink.BlinkElement.instance.scrollTop_Setter_(this, value.round());

  @DomName('Element.scrollWidth')
  @DocsEditable()
  int get scrollWidth =>
      _blink.BlinkElement.instance.scrollWidth_Getter_(this).round();

  // To suppress missing implicit constructor warnings.
  factory Element._() {
    throw new UnsupportedError("Not supported");
  }

  /**
   * Static factory designed to expose `abort` events to event
   * handlers that are not necessarily instances of [Element].
   *
   * See [EventStreamProvider] for usage information.
   */
  @DomName('Element.abortEvent')
  @DocsEditable()
  static const EventStreamProvider<Event> abortEvent =
      const EventStreamProvider<Event>('abort');

  /**
   * Static factory designed to expose `beforecopy` events to event
   * handlers that are not necessarily instances of [Element].
   *
   * See [EventStreamProvider] for usage information.
   */
  @DomName('Element.beforecopyEvent')
  @DocsEditable()
  static const EventStreamProvider<Event> beforeCopyEvent =
      const EventStreamProvider<Event>('beforecopy');

  /**
   * Static factory designed to expose `beforecut` events to event
   * handlers that are not necessarily instances of [Element].
   *
   * See [EventStreamProvider] for usage information.
   */
  @DomName('Element.beforecutEvent')
  @DocsEditable()
  static const EventStreamProvider<Event> beforeCutEvent =
      const EventStreamProvider<Event>('beforecut');

  /**
   * Static factory designed to expose `beforepaste` events to event
   * handlers that are not necessarily instances of [Element].
   *
   * See [EventStreamProvider] for usage information.
   */
  @DomName('Element.beforepasteEvent')
  @DocsEditable()
  static const EventStreamProvider<Event> beforePasteEvent =
      const EventStreamProvider<Event>('beforepaste');

  /**
   * Static factory designed to expose `blur` events to event
   * handlers that are not necessarily instances of [Element].
   *
   * See [EventStreamProvider] for usage information.
   */
  @DomName('Element.blurEvent')
  @DocsEditable()
  static const EventStreamProvider<Event> blurEvent =
      const EventStreamProvider<Event>('blur');

  @DomName('Element.canplayEvent')
  @DocsEditable()
  @Experimental() // untriaged
  static const EventStreamProvider<Event> canPlayEvent =
      const EventStreamProvider<Event>('canplay');

  @DomName('Element.canplaythroughEvent')
  @DocsEditable()
  @Experimental() // untriaged
  static const EventStreamProvider<Event> canPlayThroughEvent =
      const EventStreamProvider<Event>('canplaythrough');

  /**
   * Static factory designed to expose `change` events to event
   * handlers that are not necessarily instances of [Element].
   *
   * See [EventStreamProvider] for usage information.
   */
  @DomName('Element.changeEvent')
  @DocsEditable()
  static const EventStreamProvider<Event> changeEvent =
      const EventStreamProvider<Event>('change');

  /**
   * Static factory designed to expose `click` events to event
   * handlers that are not necessarily instances of [Element].
   *
   * See [EventStreamProvider] for usage information.
   */
  @DomName('Element.clickEvent')
  @DocsEditable()
  static const EventStreamProvider<MouseEvent> clickEvent =
      const EventStreamProvider<MouseEvent>('click');

  /**
   * Static factory designed to expose `contextmenu` events to event
   * handlers that are not necessarily instances of [Element].
   *
   * See [EventStreamProvider] for usage information.
   */
  @DomName('Element.contextmenuEvent')
  @DocsEditable()
  static const EventStreamProvider<MouseEvent> contextMenuEvent =
      const EventStreamProvider<MouseEvent>('contextmenu');

  /**
   * Static factory designed to expose `copy` events to event
   * handlers that are not necessarily instances of [Element].
   *
   * See [EventStreamProvider] for usage information.
   */
  @DomName('Element.copyEvent')
  @DocsEditable()
  static const EventStreamProvider<ClipboardEvent> copyEvent =
      const EventStreamProvider<ClipboardEvent>('copy');

  /**
   * Static factory designed to expose `cut` events to event
   * handlers that are not necessarily instances of [Element].
   *
   * See [EventStreamProvider] for usage information.
   */
  @DomName('Element.cutEvent')
  @DocsEditable()
  static const EventStreamProvider<ClipboardEvent> cutEvent =
      const EventStreamProvider<ClipboardEvent>('cut');

  /**
   * Static factory designed to expose `doubleclick` events to event
   * handlers that are not necessarily instances of [Element].
   *
   * See [EventStreamProvider] for usage information.
   */
  @DomName('Element.dblclickEvent')
  @DocsEditable()
  static const EventStreamProvider<Event> doubleClickEvent =
      const EventStreamProvider<Event>('dblclick');

  /**
   * A stream of `drag` events fired when an element is currently being dragged.
   *
   * A `drag` event is added to this stream as soon as the drag begins.
   * A `drag` event is also added to this stream at intervals while the drag
   * operation is still ongoing.
   *
   * ## Other resources
   *
   * * [Drag and drop
   *   sample](https://github.com/dart-lang/dart-samples/tree/master/html5/web/dnd/basics)
   *   based on [the tutorial](http://www.html5rocks.com/en/tutorials/dnd/basics/)
   *   from HTML5Rocks.
   * * [Drag and drop
   *   specification](https://html.spec.whatwg.org/multipage/interaction.html#dnd)
   *   from WHATWG.
   */
  @DomName('Element.dragEvent')
  @DocsEditable()
  static const EventStreamProvider<MouseEvent> dragEvent =
      const EventStreamProvider<MouseEvent>('drag');

  /**
   * A stream of `dragend` events fired when an element completes a drag
   * operation.
   *
   * ## Other resources
   *
   * * [Drag and drop
   *   sample](https://github.com/dart-lang/dart-samples/tree/master/html5/web/dnd/basics)
   *   based on [the tutorial](http://www.html5rocks.com/en/tutorials/dnd/basics/)
   *   from HTML5Rocks.
   * * [Drag and drop
   *   specification](https://html.spec.whatwg.org/multipage/interaction.html#dnd)
   *   from WHATWG.
   */
  @DomName('Element.dragendEvent')
  @DocsEditable()
  static const EventStreamProvider<MouseEvent> dragEndEvent =
      const EventStreamProvider<MouseEvent>('dragend');

  /**
   * A stream of `dragenter` events fired when a dragged object is first dragged
   * over an element.
   *
   * ## Other resources
   *
   * * [Drag and drop
   *   sample](https://github.com/dart-lang/dart-samples/tree/master/html5/web/dnd/basics)
   *   based on [the tutorial](http://www.html5rocks.com/en/tutorials/dnd/basics/)
   *   from HTML5Rocks.
   * * [Drag and drop
   *   specification](https://html.spec.whatwg.org/multipage/interaction.html#dnd)
   *   from WHATWG.
   */
  @DomName('Element.dragenterEvent')
  @DocsEditable()
  static const EventStreamProvider<MouseEvent> dragEnterEvent =
      const EventStreamProvider<MouseEvent>('dragenter');

  /**
   * A stream of `dragleave` events fired when an object being dragged over an
   * element leaves the element's target area.
   *
   * ## Other resources
   *
   * * [Drag and drop
   *   sample](https://github.com/dart-lang/dart-samples/tree/master/html5/web/dnd/basics)
   *   based on [the tutorial](http://www.html5rocks.com/en/tutorials/dnd/basics/)
   *   from HTML5Rocks.
   * * [Drag and drop
   *   specification](https://html.spec.whatwg.org/multipage/interaction.html#dnd)
   *   from WHATWG.
   */
  @DomName('Element.dragleaveEvent')
  @DocsEditable()
  static const EventStreamProvider<MouseEvent> dragLeaveEvent =
      const EventStreamProvider<MouseEvent>('dragleave');

  /**
   * A stream of `dragover` events fired when a dragged object is currently
   * being dragged over an element.
   *
   * ## Other resources
   *
   * * [Drag and drop
   *   sample](https://github.com/dart-lang/dart-samples/tree/master/html5/web/dnd/basics)
   *   based on [the tutorial](http://www.html5rocks.com/en/tutorials/dnd/basics/)
   *   from HTML5Rocks.
   * * [Drag and drop
   *   specification](https://html.spec.whatwg.org/multipage/interaction.html#dnd)
   *   from WHATWG.
   */
  @DomName('Element.dragoverEvent')
  @DocsEditable()
  static const EventStreamProvider<MouseEvent> dragOverEvent =
      const EventStreamProvider<MouseEvent>('dragover');

  /**
   * A stream of `dragstart` events for a dragged element whose drag has begun.
   *
   * ## Other resources
   *
   * * [Drag and drop
   *   sample](https://github.com/dart-lang/dart-samples/tree/master/html5/web/dnd/basics)
   *   based on [the tutorial](http://www.html5rocks.com/en/tutorials/dnd/basics/)
   *   from HTML5Rocks.
   * * [Drag and drop
   *   specification](https://html.spec.whatwg.org/multipage/interaction.html#dnd)
   *   from WHATWG.
   */
  @DomName('Element.dragstartEvent')
  @DocsEditable()
  static const EventStreamProvider<MouseEvent> dragStartEvent =
      const EventStreamProvider<MouseEvent>('dragstart');

  /**
   * A stream of `drop` events fired when a dragged object is dropped on an
   * element.
   *
   * ## Other resources
   *
   * * [Drag and drop
   *   sample](https://github.com/dart-lang/dart-samples/tree/master/html5/web/dnd/basics)
   *   based on [the tutorial](http://www.html5rocks.com/en/tutorials/dnd/basics/)
   *   from HTML5Rocks.
   * * [Drag and drop
   *   specification](https://html.spec.whatwg.org/multipage/interaction.html#dnd)
   *   from WHATWG.
   */
  @DomName('Element.dropEvent')
  @DocsEditable()
  static const EventStreamProvider<MouseEvent> dropEvent =
      const EventStreamProvider<MouseEvent>('drop');

  @DomName('Element.durationchangeEvent')
  @DocsEditable()
  @Experimental() // untriaged
  static const EventStreamProvider<Event> durationChangeEvent =
      const EventStreamProvider<Event>('durationchange');

  @DomName('Element.emptiedEvent')
  @DocsEditable()
  @Experimental() // untriaged
  static const EventStreamProvider<Event> emptiedEvent =
      const EventStreamProvider<Event>('emptied');

  @DomName('Element.endedEvent')
  @DocsEditable()
  @Experimental() // untriaged
  static const EventStreamProvider<Event> endedEvent =
      const EventStreamProvider<Event>('ended');

  /**
   * Static factory designed to expose `error` events to event
   * handlers that are not necessarily instances of [Element].
   *
   * See [EventStreamProvider] for usage information.
   */
  @DomName('Element.errorEvent')
  @DocsEditable()
  static const EventStreamProvider<Event> errorEvent =
      const EventStreamProvider<Event>('error');

  /**
   * Static factory designed to expose `focus` events to event
   * handlers that are not necessarily instances of [Element].
   *
   * See [EventStreamProvider] for usage information.
   */
  @DomName('Element.focusEvent')
  @DocsEditable()
  static const EventStreamProvider<Event> focusEvent =
      const EventStreamProvider<Event>('focus');

  /**
   * Static factory designed to expose `input` events to event
   * handlers that are not necessarily instances of [Element].
   *
   * See [EventStreamProvider] for usage information.
   */
  @DomName('Element.inputEvent')
  @DocsEditable()
  static const EventStreamProvider<Event> inputEvent =
      const EventStreamProvider<Event>('input');

  /**
   * Static factory designed to expose `invalid` events to event
   * handlers that are not necessarily instances of [Element].
   *
   * See [EventStreamProvider] for usage information.
   */
  @DomName('Element.invalidEvent')
  @DocsEditable()
  static const EventStreamProvider<Event> invalidEvent =
      const EventStreamProvider<Event>('invalid');

  /**
   * Static factory designed to expose `keydown` events to event
   * handlers that are not necessarily instances of [Element].
   *
   * See [EventStreamProvider] for usage information.
   */
  @DomName('Element.keydownEvent')
  @DocsEditable()
  static const EventStreamProvider<KeyboardEvent> keyDownEvent =
      const EventStreamProvider<KeyboardEvent>('keydown');

  /**
   * Static factory designed to expose `keypress` events to event
   * handlers that are not necessarily instances of [Element].
   *
   * See [EventStreamProvider] for usage information.
   */
  @DomName('Element.keypressEvent')
  @DocsEditable()
  static const EventStreamProvider<KeyboardEvent> keyPressEvent =
      const EventStreamProvider<KeyboardEvent>('keypress');

  /**
   * Static factory designed to expose `keyup` events to event
   * handlers that are not necessarily instances of [Element].
   *
   * See [EventStreamProvider] for usage information.
   */
  @DomName('Element.keyupEvent')
  @DocsEditable()
  static const EventStreamProvider<KeyboardEvent> keyUpEvent =
      const EventStreamProvider<KeyboardEvent>('keyup');

  /**
   * Static factory designed to expose `load` events to event
   * handlers that are not necessarily instances of [Element].
   *
   * See [EventStreamProvider] for usage information.
   */
  @DomName('Element.loadEvent')
  @DocsEditable()
  static const EventStreamProvider<Event> loadEvent =
      const EventStreamProvider<Event>('load');

  @DomName('Element.loadeddataEvent')
  @DocsEditable()
  @Experimental() // untriaged
  static const EventStreamProvider<Event> loadedDataEvent =
      const EventStreamProvider<Event>('loadeddata');

  @DomName('Element.loadedmetadataEvent')
  @DocsEditable()
  @Experimental() // untriaged
  static const EventStreamProvider<Event> loadedMetadataEvent =
      const EventStreamProvider<Event>('loadedmetadata');

  /**
   * Static factory designed to expose `mousedown` events to event
   * handlers that are not necessarily instances of [Element].
   *
   * See [EventStreamProvider] for usage information.
   */
  @DomName('Element.mousedownEvent')
  @DocsEditable()
  static const EventStreamProvider<MouseEvent> mouseDownEvent =
      const EventStreamProvider<MouseEvent>('mousedown');

  /**
   * Static factory designed to expose `mouseenter` events to event
   * handlers that are not necessarily instances of [Element].
   *
   * See [EventStreamProvider] for usage information.
   */
  @DomName('Element.mouseenterEvent')
  @DocsEditable()
  @Experimental() // untriaged
  static const EventStreamProvider<MouseEvent> mouseEnterEvent =
      const EventStreamProvider<MouseEvent>('mouseenter');

  /**
   * Static factory designed to expose `mouseleave` events to event
   * handlers that are not necessarily instances of [Element].
   *
   * See [EventStreamProvider] for usage information.
   */
  @DomName('Element.mouseleaveEvent')
  @DocsEditable()
  @Experimental() // untriaged
  static const EventStreamProvider<MouseEvent> mouseLeaveEvent =
      const EventStreamProvider<MouseEvent>('mouseleave');

  /**
   * Static factory designed to expose `mousemove` events to event
   * handlers that are not necessarily instances of [Element].
   *
   * See [EventStreamProvider] for usage information.
   */
  @DomName('Element.mousemoveEvent')
  @DocsEditable()
  static const EventStreamProvider<MouseEvent> mouseMoveEvent =
      const EventStreamProvider<MouseEvent>('mousemove');

  /**
   * Static factory designed to expose `mouseout` events to event
   * handlers that are not necessarily instances of [Element].
   *
   * See [EventStreamProvider] for usage information.
   */
  @DomName('Element.mouseoutEvent')
  @DocsEditable()
  static const EventStreamProvider<MouseEvent> mouseOutEvent =
      const EventStreamProvider<MouseEvent>('mouseout');

  /**
   * Static factory designed to expose `mouseover` events to event
   * handlers that are not necessarily instances of [Element].
   *
   * See [EventStreamProvider] for usage information.
   */
  @DomName('Element.mouseoverEvent')
  @DocsEditable()
  static const EventStreamProvider<MouseEvent> mouseOverEvent =
      const EventStreamProvider<MouseEvent>('mouseover');

  /**
   * Static factory designed to expose `mouseup` events to event
   * handlers that are not necessarily instances of [Element].
   *
   * See [EventStreamProvider] for usage information.
   */
  @DomName('Element.mouseupEvent')
  @DocsEditable()
  static const EventStreamProvider<MouseEvent> mouseUpEvent =
      const EventStreamProvider<MouseEvent>('mouseup');

  @DomName('Element.mousewheelEvent')
  @DocsEditable()
  // http://www.w3.org/TR/DOM-Level-3-Events/#events-wheelevents
  @Experimental() // non-standard
  static const EventStreamProvider<WheelEvent> mouseWheelEvent =
      const EventStreamProvider<WheelEvent>('mousewheel');

  /**
   * Static factory designed to expose `paste` events to event
   * handlers that are not necessarily instances of [Element].
   *
   * See [EventStreamProvider] for usage information.
   */
  @DomName('Element.pasteEvent')
  @DocsEditable()
  static const EventStreamProvider<ClipboardEvent> pasteEvent =
      const EventStreamProvider<ClipboardEvent>('paste');

  @DomName('Element.pauseEvent')
  @DocsEditable()
  @Experimental() // untriaged
  static const EventStreamProvider<Event> pauseEvent =
      const EventStreamProvider<Event>('pause');

  @DomName('Element.playEvent')
  @DocsEditable()
  @Experimental() // untriaged
  static const EventStreamProvider<Event> playEvent =
      const EventStreamProvider<Event>('play');

  @DomName('Element.playingEvent')
  @DocsEditable()
  @Experimental() // untriaged
  static const EventStreamProvider<Event> playingEvent =
      const EventStreamProvider<Event>('playing');

  @DomName('Element.ratechangeEvent')
  @DocsEditable()
  @Experimental() // untriaged
  static const EventStreamProvider<Event> rateChangeEvent =
      const EventStreamProvider<Event>('ratechange');

  /**
   * Static factory designed to expose `reset` events to event
   * handlers that are not necessarily instances of [Element].
   *
   * See [EventStreamProvider] for usage information.
   */
  @DomName('Element.resetEvent')
  @DocsEditable()
  static const EventStreamProvider<Event> resetEvent =
      const EventStreamProvider<Event>('reset');

  @DomName('Element.resizeEvent')
  @DocsEditable()
  @Experimental() // untriaged
  static const EventStreamProvider<Event> resizeEvent =
      const EventStreamProvider<Event>('resize');

  /**
   * Static factory designed to expose `scroll` events to event
   * handlers that are not necessarily instances of [Element].
   *
   * See [EventStreamProvider] for usage information.
   */
  @DomName('Element.scrollEvent')
  @DocsEditable()
  static const EventStreamProvider<Event> scrollEvent =
      const EventStreamProvider<Event>('scroll');

  /**
   * Static factory designed to expose `search` events to event
   * handlers that are not necessarily instances of [Element].
   *
   * See [EventStreamProvider] for usage information.
   */
  @DomName('Element.searchEvent')
  @DocsEditable()
  // http://www.w3.org/TR/html-markup/input.search.html
  @Experimental()
  static const EventStreamProvider<Event> searchEvent =
      const EventStreamProvider<Event>('search');

  @DomName('Element.seekedEvent')
  @DocsEditable()
  @Experimental() // untriaged
  static const EventStreamProvider<Event> seekedEvent =
      const EventStreamProvider<Event>('seeked');

  @DomName('Element.seekingEvent')
  @DocsEditable()
  @Experimental() // untriaged
  static const EventStreamProvider<Event> seekingEvent =
      const EventStreamProvider<Event>('seeking');

  /**
   * Static factory designed to expose `select` events to event
   * handlers that are not necessarily instances of [Element].
   *
   * See [EventStreamProvider] for usage information.
   */
  @DomName('Element.selectEvent')
  @DocsEditable()
  static const EventStreamProvider<Event> selectEvent =
      const EventStreamProvider<Event>('select');

  /**
   * Static factory designed to expose `selectstart` events to event
   * handlers that are not necessarily instances of [Element].
   *
   * See [EventStreamProvider] for usage information.
   */
  @DomName('Element.selectstartEvent')
  @DocsEditable()
  @Experimental() // nonstandard
  static const EventStreamProvider<Event> selectStartEvent =
      const EventStreamProvider<Event>('selectstart');

  @DomName('Element.stalledEvent')
  @DocsEditable()
  @Experimental() // untriaged
  static const EventStreamProvider<Event> stalledEvent =
      const EventStreamProvider<Event>('stalled');

  /**
   * Static factory designed to expose `submit` events to event
   * handlers that are not necessarily instances of [Element].
   *
   * See [EventStreamProvider] for usage information.
   */
  @DomName('Element.submitEvent')
  @DocsEditable()
  static const EventStreamProvider<Event> submitEvent =
      const EventStreamProvider<Event>('submit');

  @DomName('Element.suspendEvent')
  @DocsEditable()
  @Experimental() // untriaged
  static const EventStreamProvider<Event> suspendEvent =
      const EventStreamProvider<Event>('suspend');

  @DomName('Element.timeupdateEvent')
  @DocsEditable()
  @Experimental() // untriaged
  static const EventStreamProvider<Event> timeUpdateEvent =
      const EventStreamProvider<Event>('timeupdate');

  /**
   * Static factory designed to expose `touchcancel` events to event
   * handlers that are not necessarily instances of [Element].
   *
   * See [EventStreamProvider] for usage information.
   */
  @DomName('Element.touchcancelEvent')
  @DocsEditable()
  // http://www.w3.org/TR/touch-events/, http://www.chromestatus.com/features
  @Experimental()
  static const EventStreamProvider<TouchEvent> touchCancelEvent =
      const EventStreamProvider<TouchEvent>('touchcancel');

  /**
   * Static factory designed to expose `touchend` events to event
   * handlers that are not necessarily instances of [Element].
   *
   * See [EventStreamProvider] for usage information.
   */
  @DomName('Element.touchendEvent')
  @DocsEditable()
  // http://www.w3.org/TR/touch-events/, http://www.chromestatus.com/features
  @Experimental()
  static const EventStreamProvider<TouchEvent> touchEndEvent =
      const EventStreamProvider<TouchEvent>('touchend');

  /**
   * Static factory designed to expose `touchenter` events to event
   * handlers that are not necessarily instances of [Element].
   *
   * See [EventStreamProvider] for usage information.
   */
  @DomName('Element.touchenterEvent')
  @DocsEditable()
  // http://www.w3.org/TR/touch-events/, http://www.chromestatus.com/features
  @Experimental()
  static const EventStreamProvider<TouchEvent> touchEnterEvent =
      const EventStreamProvider<TouchEvent>('touchenter');

  /**
   * Static factory designed to expose `touchleave` events to event
   * handlers that are not necessarily instances of [Element].
   *
   * See [EventStreamProvider] for usage information.
   */
  @DomName('Element.touchleaveEvent')
  @DocsEditable()
  // http://www.w3.org/TR/touch-events/, http://www.chromestatus.com/features
  @Experimental()
  static const EventStreamProvider<TouchEvent> touchLeaveEvent =
      const EventStreamProvider<TouchEvent>('touchleave');

  /**
   * Static factory designed to expose `touchmove` events to event
   * handlers that are not necessarily instances of [Element].
   *
   * See [EventStreamProvider] for usage information.
   */
  @DomName('Element.touchmoveEvent')
  @DocsEditable()
  // http://www.w3.org/TR/touch-events/, http://www.chromestatus.com/features
  @Experimental()
  static const EventStreamProvider<TouchEvent> touchMoveEvent =
      const EventStreamProvider<TouchEvent>('touchmove');

  /**
   * Static factory designed to expose `touchstart` events to event
   * handlers that are not necessarily instances of [Element].
   *
   * See [EventStreamProvider] for usage information.
   */
  @DomName('Element.touchstartEvent')
  @DocsEditable()
  // http://www.w3.org/TR/touch-events/, http://www.chromestatus.com/features
  @Experimental()
  static const EventStreamProvider<TouchEvent> touchStartEvent =
      const EventStreamProvider<TouchEvent>('touchstart');

  @DomName('Element.transitionendEvent')
  @DocsEditable()
  static const EventStreamProvider<TransitionEvent> transitionEndEvent =
      const EventStreamProvider<TransitionEvent>('transitionend');

  @DomName('Element.volumechangeEvent')
  @DocsEditable()
  @Experimental() // untriaged
  static const EventStreamProvider<Event> volumeChangeEvent =
      const EventStreamProvider<Event>('volumechange');

  @DomName('Element.waitingEvent')
  @DocsEditable()
  @Experimental() // untriaged
  static const EventStreamProvider<Event> waitingEvent =
      const EventStreamProvider<Event>('waiting');

  /**
   * Static factory designed to expose `fullscreenchange` events to event
   * handlers that are not necessarily instances of [Element].
   *
   * See [EventStreamProvider] for usage information.
   */
  @DomName('Element.webkitfullscreenchangeEvent')
  @DocsEditable()
  @SupportedBrowser(SupportedBrowser.CHROME)
  @SupportedBrowser(SupportedBrowser.SAFARI)
  @Experimental()
  // https://dvcs.w3.org/hg/fullscreen/raw-file/tip/Overview.html
  static const EventStreamProvider<Event> fullscreenChangeEvent =
      const EventStreamProvider<Event>('webkitfullscreenchange');

  /**
   * Static factory designed to expose `fullscreenerror` events to event
   * handlers that are not necessarily instances of [Element].
   *
   * See [EventStreamProvider] for usage information.
   */
  @DomName('Element.webkitfullscreenerrorEvent')
  @DocsEditable()
  @SupportedBrowser(SupportedBrowser.CHROME)
  @SupportedBrowser(SupportedBrowser.SAFARI)
  @Experimental()
  // https://dvcs.w3.org/hg/fullscreen/raw-file/tip/Overview.html
  static const EventStreamProvider<Event> fullscreenErrorEvent =
      const EventStreamProvider<Event>('webkitfullscreenerror');

  @Deprecated("Internal Use Only")
  external static Type get instanceRuntimeType;

  @Deprecated("Internal Use Only")
  Element.internal_() : super.internal_();

  String contentEditable;

  MenuElement contextMenu;

  String dir;

  bool draggable;

  bool hidden;

  String lang;

  bool spellcheck;

  CssStyleDeclaration get style;

  int tabIndex;

  String title;

  bool translate;

  String dropzone;

  void blur();

  void focus();

  @DomName('Element.attributes')
  @DocsEditable()
  _NamedNodeMap get _attributes =>
      _blink.BlinkElement.instance.attributes_Getter_(this);

  @DomName('Element.className')
  @DocsEditable()
  String get className => _blink.BlinkElement.instance.className_Getter_(this);

  @DomName('Element.className')
  @DocsEditable()
  set className(String value) =>
      _blink.BlinkElement.instance.className_Setter_(this, value);

  @DomName('Element.clientHeight')
  @DocsEditable()
  int get clientHeight =>
      _blink.BlinkElement.instance.clientHeight_Getter_(this);

  @DomName('Element.clientLeft')
  @DocsEditable()
  int get clientLeft => _blink.BlinkElement.instance.clientLeft_Getter_(this);

  @DomName('Element.clientTop')
  @DocsEditable()
  int get clientTop => _blink.BlinkElement.instance.clientTop_Getter_(this);

  @DomName('Element.clientWidth')
  @DocsEditable()
  int get clientWidth => _blink.BlinkElement.instance.clientWidth_Getter_(this);

  @DomName('Element.computedName')
  @DocsEditable()
  @Experimental() // untriaged
  String get computedName =>
      _blink.BlinkElement.instance.computedName_Getter_(this);

  @DomName('Element.computedRole')
  @DocsEditable()
  @Experimental() // untriaged
  String get computedRole =>
      _blink.BlinkElement.instance.computedRole_Getter_(this);

  @DomName('Element.id')
  @DocsEditable()
  String get id => _blink.BlinkElement.instance.id_Getter_(this);

  @DomName('Element.id')
  @DocsEditable()
  set id(String value) => _blink.BlinkElement.instance.id_Setter_(this, value);

  @DomName('Element.innerHTML')
  @DocsEditable()
  String get _innerHtml => _blink.BlinkElement.instance.innerHTML_Getter_(this);

  @DomName('Element.innerHTML')
  @DocsEditable()
  set _innerHtml(String value) =>
      _blink.BlinkElement.instance.innerHTML_Setter_(this, value);

  @DomName('Element.localName')
  @DocsEditable()
  @Experimental() // untriaged
  String get _localName => _blink.BlinkElement.instance.localName_Getter_(this);

  @DomName('Element.namespaceURI')
  @DocsEditable()
  @Experimental() // untriaged
  String get _namespaceUri =>
      _blink.BlinkElement.instance.namespaceURI_Getter_(this);

  @DomName('Element.outerHTML')
  @DocsEditable()
  String get outerHtml => _blink.BlinkElement.instance.outerHTML_Getter_(this);

  @DomName('Element.scrollHeight')
  @DocsEditable()
  int get _scrollHeight =>
      _blink.BlinkElement.instance.scrollHeight_Getter_(this);

  @DomName('Element.scrollLeft')
  @DocsEditable()
  num get _scrollLeft => _blink.BlinkElement.instance.scrollLeft_Getter_(this);

  @DomName('Element.scrollLeft')
  @DocsEditable()
  set _scrollLeft(num value) =>
      _blink.BlinkElement.instance.scrollLeft_Setter_(this, value);

  @DomName('Element.scrollTop')
  @DocsEditable()
  num get _scrollTop => _blink.BlinkElement.instance.scrollTop_Getter_(this);

  @DomName('Element.scrollTop')
  @DocsEditable()
  set _scrollTop(num value) =>
      _blink.BlinkElement.instance.scrollTop_Setter_(this, value);

  @DomName('Element.scrollWidth')
  @DocsEditable()
  int get _scrollWidth =>
      _blink.BlinkElement.instance.scrollWidth_Getter_(this);

  @DomName('Element.shadowRoot')
  @DocsEditable()
  // https://dvcs.w3.org/hg/webcomponents/raw-file/tip/spec/shadow/index.html#api-shadow-aware-create-shadow-root
  @Experimental()
  ShadowRoot get shadowRoot =>
      _blink.BlinkElement.instance.shadowRoot_Getter_(this);

  @DomName('Element.tagName')
  @DocsEditable()
  String get tagName => _blink.BlinkElement.instance.tagName_Getter_(this);

  Animation _animate(Object effect, [timing]) {
    if (effect != null && timing == null) {
      return _blink.BlinkElement.instance.animate_Callback_1_(this, effect);
    }
    if ((timing is num) && effect != null) {
      return _blink.BlinkElement.instance
          .animate_Callback_2_(this, effect, timing);
    }
    if (timing != null && effect != null) {
      return _blink.BlinkElement.instance
          .animate_Callback_2_(this, effect, timing);
    }
    throw new ArgumentError("Incorrect number or type of arguments");
  }

  @DomName('Element.closest')
  @DocsEditable()
  @Experimental() // untriaged
  Element closest(String selectors) =>
      _blink.BlinkElement.instance.closest_Callback_1_(this, selectors);

  ShadowRoot createShadowRoot([Map shadowRootInitDict]) {
    if (shadowRootInitDict == null) {
      return _blink.BlinkElement.instance.createShadowRoot_Callback_0_(this);
    }
    if ((shadowRootInitDict is Map)) {
      return _blink.BlinkElement.instance.createShadowRoot_Callback_1_(
          this, convertDartToNative_Dictionary(shadowRootInitDict));
    }
    throw new ArgumentError("Incorrect number or type of arguments");
  }

  @DomName('Element.getAnimations')
  @DocsEditable()
  @Experimental() // untriaged
  List<Animation> getAnimations() =>
      (_blink.BlinkElement.instance.getAnimations_Callback_0_(this));

  @DomName('Element.getAttribute')
  @DocsEditable()
  @Experimental() // untriaged
  String getAttribute(String name) =>
      _blink.BlinkElement.instance.getAttribute_Callback_1_(this, name);

  @DomName('Element.getAttributeNS')
  @DocsEditable()
  @Experimental() // untriaged
  String getAttributeNS(String namespaceURI, String localName) =>
      _blink.BlinkElement.instance
          .getAttributeNS_Callback_2_(this, namespaceURI, localName);

  /**
   * Returns the smallest bounding rectangle that encompasses this element's
   * padding, scrollbar, and border.
   *
   * ## Other resources
   *
   * * [Element.getBoundingClientRect](https://developer.mozilla.org/en-US/docs/Web/API/Element.getBoundingClientRect)
   *   from MDN.
   * * [The getBoundingClientRect()
   *   method](http://www.w3.org/TR/cssom-view/#the-getclientrects()-and-getboundingclientrect()-methods)
   *   from W3C.
   */
  @DomName('Element.getBoundingClientRect')
  @DocsEditable()
  Rectangle getBoundingClientRect() => make_dart_rectangle(
      _blink.BlinkElement.instance.getBoundingClientRect_Callback_0_(this));

  /**
   * Returns a list of bounding rectangles for each box associated with this
   * element.
   *
   * ## Other resources
   *
   * * [Element.getClientRects](https://developer.mozilla.org/en-US/docs/Web/API/Element.getClientRects)
   *   from MDN.
   * * [The getClientRects()
   *   method](http://www.w3.org/TR/cssom-view/#the-getclientrects()-and-getboundingclientrect()-methods)
   *   from W3C.
   */
  @DomName('Element.getClientRects')
  @DocsEditable()
  List<Rectangle> getClientRects() =>
      _blink.BlinkElement.instance.getClientRects_Callback_0_(this);

  /**
   * Returns a list of shadow DOM insertion points to which this element is
   * distributed.
   *
   * ## Other resources
   *
   * * [Shadow DOM
   *   specification](https://dvcs.w3.org/hg/webcomponents/raw-file/tip/spec/shadow/index.html)
   *   from W3C.
   */
  @DomName('Element.getDestinationInsertionPoints')
  @DocsEditable()
  @Experimental() // untriaged
  List<Node> getDestinationInsertionPoints() => (_blink.BlinkElement.instance
      .getDestinationInsertionPoints_Callback_0_(this));

  /**
   * Returns a list of nodes with the given class name inside this element.
   *
   * ## Other resources
   *
   * * [getElementsByClassName](https://developer.mozilla.org/en-US/docs/Web/API/document.getElementsByClassName)
   *   from MDN.
   * * [DOM specification](http://www.w3.org/TR/domcore/) from W3C.
   */
  @DomName('Element.getElementsByClassName')
  @DocsEditable()
  List<Node> getElementsByClassName(String classNames) =>
      (_blink.BlinkElement.instance
          .getElementsByClassName_Callback_1_(this, classNames));

  @DomName('Element.getElementsByTagName')
  @DocsEditable()
  List<Node> _getElementsByTagName(String localName) =>
      (_blink.BlinkElement.instance
          .getElementsByTagName_Callback_1_(this, localName));

  @DomName('Element.hasAttribute')
  @DocsEditable()
  bool _hasAttribute(String name) =>
      _blink.BlinkElement.instance.hasAttribute_Callback_1_(this, name);

  @DomName('Element.hasAttributeNS')
  @DocsEditable()
  bool _hasAttributeNS(String namespaceURI, String localName) =>
      _blink.BlinkElement.instance
          .hasAttributeNS_Callback_2_(this, namespaceURI, localName);

  @DomName('Element.insertAdjacentElement')
  @DocsEditable()
  @Experimental() // untriaged
  Element insertAdjacentElement(String where, Element element) =>
      _blink.BlinkElement.instance
          .insertAdjacentElement_Callback_2_(this, where, element);

  @DomName('Element.insertAdjacentHTML')
  @DocsEditable()
  @Experimental() // untriaged
  void _insertAdjacentHtml(String position, String text) =>
      _blink.BlinkElement.instance
          .insertAdjacentHTML_Callback_2_(this, position, text);

  @DomName('Element.insertAdjacentText')
  @DocsEditable()
  @Experimental() // untriaged
  void insertAdjacentText(String where, String text) =>
      _blink.BlinkElement.instance
          .insertAdjacentText_Callback_2_(this, where, text);

  @DomName('Element.matches')
  @DocsEditable()
  @Experimental() // untriaged
  bool matches(String selectors) =>
      _blink.BlinkElement.instance.matches_Callback_1_(this, selectors);

  @DomName('Element.removeAttribute')
  @DocsEditable()
  void _removeAttribute(String name) =>
      _blink.BlinkElement.instance.removeAttribute_Callback_1_(this, name);

  @DomName('Element.removeAttributeNS')
  @DocsEditable()
  void _removeAttributeNS(String namespaceURI, String localName) =>
      _blink.BlinkElement.instance
          .removeAttributeNS_Callback_2_(this, namespaceURI, localName);

  @DomName('Element.requestFullscreen')
  @DocsEditable()
  @Experimental() // untriaged
  void requestFullscreen() =>
      _blink.BlinkElement.instance.requestFullscreen_Callback_0_(this);

  @DomName('Element.requestPointerLock')
  @DocsEditable()
  @Experimental() // untriaged
  void requestPointerLock() =>
      _blink.BlinkElement.instance.requestPointerLock_Callback_0_(this);

  void scroll([options_OR_x, num y]) {
    if (options_OR_x == null && y == null) {
      _blink.BlinkElement.instance.scroll_Callback_0_(this);
      return;
    }
    if ((options_OR_x is Map) && y == null) {
      _blink.BlinkElement.instance.scroll_Callback_1_(this, options_OR_x);
      return;
    }
    if ((y is num) && (options_OR_x is num)) {
      _blink.BlinkElement.instance.scroll_Callback_2_(this, options_OR_x, y);
      return;
    }
    throw new ArgumentError("Incorrect number or type of arguments");
  }

  void scrollBy([options_OR_x, num y]) {
    if (options_OR_x == null && y == null) {
      _blink.BlinkElement.instance.scrollBy_Callback_0_(this);
      return;
    }
    if ((options_OR_x is Map) && y == null) {
      _blink.BlinkElement.instance.scrollBy_Callback_1_(this, options_OR_x);
      return;
    }
    if ((y is num) && (options_OR_x is num)) {
      _blink.BlinkElement.instance.scrollBy_Callback_2_(this, options_OR_x, y);
      return;
    }
    throw new ArgumentError("Incorrect number or type of arguments");
  }

  void _scrollIntoView([bool alignWithTop]) {
    if (alignWithTop != null) {
      _blink.BlinkElement.instance
          .scrollIntoView_Callback_1_(this, alignWithTop);
      return;
    }
    _blink.BlinkElement.instance.scrollIntoView_Callback_0_(this);
    return;
  }

  void _scrollIntoViewIfNeeded([bool centerIfNeeded]) {
    if (centerIfNeeded != null) {
      _blink.BlinkElement.instance
          .scrollIntoViewIfNeeded_Callback_1_(this, centerIfNeeded);
      return;
    }
    _blink.BlinkElement.instance.scrollIntoViewIfNeeded_Callback_0_(this);
    return;
  }

  void scrollTo([options_OR_x, num y]) {
    if (options_OR_x == null && y == null) {
      _blink.BlinkElement.instance.scrollTo_Callback_0_(this);
      return;
    }
    if ((options_OR_x is Map) && y == null) {
      _blink.BlinkElement.instance.scrollTo_Callback_1_(this, options_OR_x);
      return;
    }
    if ((y is num) && (options_OR_x is num)) {
      _blink.BlinkElement.instance.scrollTo_Callback_2_(this, options_OR_x, y);
      return;
    }
    throw new ArgumentError("Incorrect number or type of arguments");
  }

  @DomName('Element.setAttribute')
  @DocsEditable()
  void setAttribute(String name, String value) =>
      _blink.BlinkElement.instance.setAttribute_Callback_2_(this, name, value);

  @DomName('Element.setAttributeNS')
  @DocsEditable()
  void setAttributeNS(String namespaceURI, String name, String value) =>
      _blink.BlinkElement.instance
          .setAttributeNS_Callback_3_(this, namespaceURI, name, value);

  @DomName('Element.after')
  @DocsEditable()
  @Experimental() // untriaged
  void after(Object nodes) =>
      _blink.BlinkElement.instance.after_Callback_1_(this, nodes);

  @DomName('Element.before')
  @DocsEditable()
  @Experimental() // untriaged
  void before(Object nodes) =>
      _blink.BlinkElement.instance.before_Callback_1_(this, nodes);

  @DomName('Element.remove')
  @DocsEditable()
  void remove() => _blink.BlinkElement.instance.remove_Callback_0_(this);

  @DomName('Element.nextElementSibling')
  @DocsEditable()
  Element get nextElementSibling =>
      _blink.BlinkElement.instance.nextElementSibling_Getter_(this);

  @DomName('Element.previousElementSibling')
  @DocsEditable()
  Element get previousElementSibling =>
      _blink.BlinkElement.instance.previousElementSibling_Getter_(this);

  @DomName('Element.childElementCount')
  @DocsEditable()
  int get _childElementCount =>
      _blink.BlinkElement.instance.childElementCount_Getter_(this);

  @DomName('Element.children')
  @DocsEditable()
  List<Node> get _children =>
      (_blink.BlinkElement.instance.children_Getter_(this));

  @DomName('Element.firstElementChild')
  @DocsEditable()
  Element get _firstElementChild =>
      _blink.BlinkElement.instance.firstElementChild_Getter_(this);

  @DomName('Element.lastElementChild')
  @DocsEditable()
  Element get _lastElementChild =>
      _blink.BlinkElement.instance.lastElementChild_Getter_(this);

  /**
   * Finds the first descendant element of this element that matches the
   * specified group of selectors.
   *
   * [selectors] should be a string using CSS selector syntax.
   *
   *     // Gets the first descendant with the class 'classname'
   *     var element = element.querySelector('.className');
   *     // Gets the element with id 'id'
   *     var element = element.querySelector('#id');
   *     // Gets the first descendant [ImageElement]
   *     var img = element.querySelector('img');
   *
   * For details about CSS selector syntax, see the
   * [CSS selector specification](http://www.w3.org/TR/css3-selectors/).
   */
  @DomName('Element.querySelector')
  @DocsEditable()
  Element querySelector(String selectors) =>
      _blink.BlinkElement.instance.querySelector_Callback_1_(this, selectors);

  @DomName('Element.querySelectorAll')
  @DocsEditable()
  List<Node> _querySelectorAll(String selectors) =>
      (_blink.BlinkElement.instance
          .querySelectorAll_Callback_1_(this, selectors));

  /// Stream of `abort` events handled by this [Element].
  @DomName('Element.onabort')
  @DocsEditable()
  ElementStream<Event> get onAbort => abortEvent.forElement(this);

  /// Stream of `beforecopy` events handled by this [Element].
  @DomName('Element.onbeforecopy')
  @DocsEditable()
  ElementStream<Event> get onBeforeCopy => beforeCopyEvent.forElement(this);

  /// Stream of `beforecut` events handled by this [Element].
  @DomName('Element.onbeforecut')
  @DocsEditable()
  ElementStream<Event> get onBeforeCut => beforeCutEvent.forElement(this);

  /// Stream of `beforepaste` events handled by this [Element].
  @DomName('Element.onbeforepaste')
  @DocsEditable()
  ElementStream<Event> get onBeforePaste => beforePasteEvent.forElement(this);

  /// Stream of `blur` events handled by this [Element].
  @DomName('Element.onblur')
  @DocsEditable()
  ElementStream<Event> get onBlur => blurEvent.forElement(this);

  @DomName('Element.oncanplay')
  @DocsEditable()
  @Experimental() // untriaged
  ElementStream<Event> get onCanPlay => canPlayEvent.forElement(this);

  @DomName('Element.oncanplaythrough')
  @DocsEditable()
  @Experimental() // untriaged
  ElementStream<Event> get onCanPlayThrough =>
      canPlayThroughEvent.forElement(this);

  /// Stream of `change` events handled by this [Element].
  @DomName('Element.onchange')
  @DocsEditable()
  ElementStream<Event> get onChange => changeEvent.forElement(this);

  /// Stream of `click` events handled by this [Element].
  @DomName('Element.onclick')
  @DocsEditable()
  ElementStream<MouseEvent> get onClick => clickEvent.forElement(this);

  /// Stream of `contextmenu` events handled by this [Element].
  @DomName('Element.oncontextmenu')
  @DocsEditable()
  ElementStream<MouseEvent> get onContextMenu =>
      contextMenuEvent.forElement(this);

  /// Stream of `copy` events handled by this [Element].
  @DomName('Element.oncopy')
  @DocsEditable()
  ElementStream<ClipboardEvent> get onCopy => copyEvent.forElement(this);

  /// Stream of `cut` events handled by this [Element].
  @DomName('Element.oncut')
  @DocsEditable()
  ElementStream<ClipboardEvent> get onCut => cutEvent.forElement(this);

  /// Stream of `doubleclick` events handled by this [Element].
  @DomName('Element.ondblclick')
  @DocsEditable()
  ElementStream<Event> get onDoubleClick => doubleClickEvent.forElement(this);

  /**
   * A stream of `drag` events fired when this element currently being dragged.
   *
   * A `drag` event is added to this stream as soon as the drag begins.
   * A `drag` event is also added to this stream at intervals while the drag
   * operation is still ongoing.
   *
   * ## Other resources
   *
   * * [Drag and drop
   *   sample](https://github.com/dart-lang/dart-samples/tree/master/html5/web/dnd/basics)
   *   based on [the tutorial](http://www.html5rocks.com/en/tutorials/dnd/basics/)
   *   from HTML5Rocks.
   * * [Drag and drop
   *   specification](https://html.spec.whatwg.org/multipage/interaction.html#dnd)
   *   from WHATWG.
   */
  @DomName('Element.ondrag')
  @DocsEditable()
  ElementStream<MouseEvent> get onDrag => dragEvent.forElement(this);

  /**
   * A stream of `dragend` events fired when this element completes a drag
   * operation.
   *
   * ## Other resources
   *
   * * [Drag and drop
   *   sample](https://github.com/dart-lang/dart-samples/tree/master/html5/web/dnd/basics)
   *   based on [the tutorial](http://www.html5rocks.com/en/tutorials/dnd/basics/)
   *   from HTML5Rocks.
   * * [Drag and drop
   *   specification](https://html.spec.whatwg.org/multipage/interaction.html#dnd)
   *   from WHATWG.
   */
  @DomName('Element.ondragend')
  @DocsEditable()
  ElementStream<MouseEvent> get onDragEnd => dragEndEvent.forElement(this);

  /**
   * A stream of `dragenter` events fired when a dragged object is first dragged
   * over this element.
   *
   * ## Other resources
   *
   * * [Drag and drop
   *   sample](https://github.com/dart-lang/dart-samples/tree/master/html5/web/dnd/basics)
   *   based on [the tutorial](http://www.html5rocks.com/en/tutorials/dnd/basics/)
   *   from HTML5Rocks.
   * * [Drag and drop
   *   specification](https://html.spec.whatwg.org/multipage/interaction.html#dnd)
   *   from WHATWG.
   */
  @DomName('Element.ondragenter')
  @DocsEditable()
  ElementStream<MouseEvent> get onDragEnter => dragEnterEvent.forElement(this);

  /**
   * A stream of `dragleave` events fired when an object being dragged over this
   * element leaves this element's target area.
   *
   * ## Other resources
   *
   * * [Drag and drop
   *   sample](https://github.com/dart-lang/dart-samples/tree/master/html5/web/dnd/basics)
   *   based on [the tutorial](http://www.html5rocks.com/en/tutorials/dnd/basics/)
   *   from HTML5Rocks.
   * * [Drag and drop
   *   specification](https://html.spec.whatwg.org/multipage/interaction.html#dnd)
   *   from WHATWG.
   */
  @DomName('Element.ondragleave')
  @DocsEditable()
  ElementStream<MouseEvent> get onDragLeave => dragLeaveEvent.forElement(this);

  /**
   * A stream of `dragover` events fired when a dragged object is currently
   * being dragged over this element.
   *
   * ## Other resources
   *
   * * [Drag and drop
   *   sample](https://github.com/dart-lang/dart-samples/tree/master/html5/web/dnd/basics)
   *   based on [the tutorial](http://www.html5rocks.com/en/tutorials/dnd/basics/)
   *   from HTML5Rocks.
   * * [Drag and drop
   *   specification](https://html.spec.whatwg.org/multipage/interaction.html#dnd)
   *   from WHATWG.
   */
  @DomName('Element.ondragover')
  @DocsEditable()
  ElementStream<MouseEvent> get onDragOver => dragOverEvent.forElement(this);

  /**
   * A stream of `dragstart` events fired when this element starts being
   * dragged.
   *
   * ## Other resources
   *
   * * [Drag and drop
   *   sample](https://github.com/dart-lang/dart-samples/tree/master/html5/web/dnd/basics)
   *   based on [the tutorial](http://www.html5rocks.com/en/tutorials/dnd/basics/)
   *   from HTML5Rocks.
   * * [Drag and drop
   *   specification](https://html.spec.whatwg.org/multipage/interaction.html#dnd)
   *   from WHATWG.
   */
  @DomName('Element.ondragstart')
  @DocsEditable()
  ElementStream<MouseEvent> get onDragStart => dragStartEvent.forElement(this);

  /**
   * A stream of `drop` events fired when a dragged object is dropped on this
   * element.
   *
   * ## Other resources
   *
   * * [Drag and drop
   *   sample](https://github.com/dart-lang/dart-samples/tree/master/html5/web/dnd/basics)
   *   based on [the tutorial](http://www.html5rocks.com/en/tutorials/dnd/basics/)
   *   from HTML5Rocks.
   * * [Drag and drop
   *   specification](https://html.spec.whatwg.org/multipage/interaction.html#dnd)
   *   from WHATWG.
   */
  @DomName('Element.ondrop')
  @DocsEditable()
  ElementStream<MouseEvent> get onDrop => dropEvent.forElement(this);

  @DomName('Element.ondurationchange')
  @DocsEditable()
  @Experimental() // untriaged
  ElementStream<Event> get onDurationChange =>
      durationChangeEvent.forElement(this);

  @DomName('Element.onemptied')
  @DocsEditable()
  @Experimental() // untriaged
  ElementStream<Event> get onEmptied => emptiedEvent.forElement(this);

  @DomName('Element.onended')
  @DocsEditable()
  @Experimental() // untriaged
  ElementStream<Event> get onEnded => endedEvent.forElement(this);

  /// Stream of `error` events handled by this [Element].
  @DomName('Element.onerror')
  @DocsEditable()
  ElementStream<Event> get onError => errorEvent.forElement(this);

  /// Stream of `focus` events handled by this [Element].
  @DomName('Element.onfocus')
  @DocsEditable()
  ElementStream<Event> get onFocus => focusEvent.forElement(this);

  /// Stream of `input` events handled by this [Element].
  @DomName('Element.oninput')
  @DocsEditable()
  ElementStream<Event> get onInput => inputEvent.forElement(this);

  /// Stream of `invalid` events handled by this [Element].
  @DomName('Element.oninvalid')
  @DocsEditable()
  ElementStream<Event> get onInvalid => invalidEvent.forElement(this);

  /// Stream of `keydown` events handled by this [Element].
  @DomName('Element.onkeydown')
  @DocsEditable()
  ElementStream<KeyboardEvent> get onKeyDown => keyDownEvent.forElement(this);

  /// Stream of `keypress` events handled by this [Element].
  @DomName('Element.onkeypress')
  @DocsEditable()
  ElementStream<KeyboardEvent> get onKeyPress => keyPressEvent.forElement(this);

  /// Stream of `keyup` events handled by this [Element].
  @DomName('Element.onkeyup')
  @DocsEditable()
  ElementStream<KeyboardEvent> get onKeyUp => keyUpEvent.forElement(this);

  /// Stream of `load` events handled by this [Element].
  @DomName('Element.onload')
  @DocsEditable()
  ElementStream<Event> get onLoad => loadEvent.forElement(this);

  @DomName('Element.onloadeddata')
  @DocsEditable()
  @Experimental() // untriaged
  ElementStream<Event> get onLoadedData => loadedDataEvent.forElement(this);

  @DomName('Element.onloadedmetadata')
  @DocsEditable()
  @Experimental() // untriaged
  ElementStream<Event> get onLoadedMetadata =>
      loadedMetadataEvent.forElement(this);

  /// Stream of `mousedown` events handled by this [Element].
  @DomName('Element.onmousedown')
  @DocsEditable()
  ElementStream<MouseEvent> get onMouseDown => mouseDownEvent.forElement(this);

  /// Stream of `mouseenter` events handled by this [Element].
  @DomName('Element.onmouseenter')
  @DocsEditable()
  @Experimental() // untriaged
  ElementStream<MouseEvent> get onMouseEnter =>
      mouseEnterEvent.forElement(this);

  /// Stream of `mouseleave` events handled by this [Element].
  @DomName('Element.onmouseleave')
  @DocsEditable()
  @Experimental() // untriaged
  ElementStream<MouseEvent> get onMouseLeave =>
      mouseLeaveEvent.forElement(this);

  /// Stream of `mousemove` events handled by this [Element].
  @DomName('Element.onmousemove')
  @DocsEditable()
  ElementStream<MouseEvent> get onMouseMove => mouseMoveEvent.forElement(this);

  /// Stream of `mouseout` events handled by this [Element].
  @DomName('Element.onmouseout')
  @DocsEditable()
  ElementStream<MouseEvent> get onMouseOut => mouseOutEvent.forElement(this);

  /// Stream of `mouseover` events handled by this [Element].
  @DomName('Element.onmouseover')
  @DocsEditable()
  ElementStream<MouseEvent> get onMouseOver => mouseOverEvent.forElement(this);

  /// Stream of `mouseup` events handled by this [Element].
  @DomName('Element.onmouseup')
  @DocsEditable()
  ElementStream<MouseEvent> get onMouseUp => mouseUpEvent.forElement(this);

  /// Stream of `mousewheel` events handled by this [Element].
  @DomName('Element.onmousewheel')
  @DocsEditable()
  // http://www.w3.org/TR/DOM-Level-3-Events/#events-wheelevents
  @Experimental() // non-standard
  ElementStream<WheelEvent> get onMouseWheel =>
      mouseWheelEvent.forElement(this);

  /// Stream of `paste` events handled by this [Element].
  @DomName('Element.onpaste')
  @DocsEditable()
  ElementStream<ClipboardEvent> get onPaste => pasteEvent.forElement(this);

  @DomName('Element.onpause')
  @DocsEditable()
  @Experimental() // untriaged
  ElementStream<Event> get onPause => pauseEvent.forElement(this);

  @DomName('Element.onplay')
  @DocsEditable()
  @Experimental() // untriaged
  ElementStream<Event> get onPlay => playEvent.forElement(this);

  @DomName('Element.onplaying')
  @DocsEditable()
  @Experimental() // untriaged
  ElementStream<Event> get onPlaying => playingEvent.forElement(this);

  @DomName('Element.onratechange')
  @DocsEditable()
  @Experimental() // untriaged
  ElementStream<Event> get onRateChange => rateChangeEvent.forElement(this);

  /// Stream of `reset` events handled by this [Element].
  @DomName('Element.onreset')
  @DocsEditable()
  ElementStream<Event> get onReset => resetEvent.forElement(this);

  @DomName('Element.onresize')
  @DocsEditable()
  @Experimental() // untriaged
  ElementStream<Event> get onResize => resizeEvent.forElement(this);

  /// Stream of `scroll` events handled by this [Element].
  @DomName('Element.onscroll')
  @DocsEditable()
  ElementStream<Event> get onScroll => scrollEvent.forElement(this);

  /// Stream of `search` events handled by this [Element].
  @DomName('Element.onsearch')
  @DocsEditable()
  // http://www.w3.org/TR/html-markup/input.search.html
  @Experimental()
  ElementStream<Event> get onSearch => searchEvent.forElement(this);

  @DomName('Element.onseeked')
  @DocsEditable()
  @Experimental() // untriaged
  ElementStream<Event> get onSeeked => seekedEvent.forElement(this);

  @DomName('Element.onseeking')
  @DocsEditable()
  @Experimental() // untriaged
  ElementStream<Event> get onSeeking => seekingEvent.forElement(this);

  /// Stream of `select` events handled by this [Element].
  @DomName('Element.onselect')
  @DocsEditable()
  ElementStream<Event> get onSelect => selectEvent.forElement(this);

  /// Stream of `selectstart` events handled by this [Element].
  @DomName('Element.onselectstart')
  @DocsEditable()
  @Experimental() // nonstandard
  ElementStream<Event> get onSelectStart => selectStartEvent.forElement(this);

  @DomName('Element.onstalled')
  @DocsEditable()
  @Experimental() // untriaged
  ElementStream<Event> get onStalled => stalledEvent.forElement(this);

  /// Stream of `submit` events handled by this [Element].
  @DomName('Element.onsubmit')
  @DocsEditable()
  ElementStream<Event> get onSubmit => submitEvent.forElement(this);

  @DomName('Element.onsuspend')
  @DocsEditable()
  @Experimental() // untriaged
  ElementStream<Event> get onSuspend => suspendEvent.forElement(this);

  @DomName('Element.ontimeupdate')
  @DocsEditable()
  @Experimental() // untriaged
  ElementStream<Event> get onTimeUpdate => timeUpdateEvent.forElement(this);

  /// Stream of `touchcancel` events handled by this [Element].
  @DomName('Element.ontouchcancel')
  @DocsEditable()
  // http://www.w3.org/TR/touch-events/, http://www.chromestatus.com/features
  @Experimental()
  ElementStream<TouchEvent> get onTouchCancel =>
      touchCancelEvent.forElement(this);

  /// Stream of `touchend` events handled by this [Element].
  @DomName('Element.ontouchend')
  @DocsEditable()
  // http://www.w3.org/TR/touch-events/, http://www.chromestatus.com/features
  @Experimental()
  ElementStream<TouchEvent> get onTouchEnd => touchEndEvent.forElement(this);

  /// Stream of `touchenter` events handled by this [Element].
  @DomName('Element.ontouchenter')
  @DocsEditable()
  // http://www.w3.org/TR/touch-events/, http://www.chromestatus.com/features
  @Experimental()
  ElementStream<TouchEvent> get onTouchEnter =>
      touchEnterEvent.forElement(this);

  /// Stream of `touchleave` events handled by this [Element].
  @DomName('Element.ontouchleave')
  @DocsEditable()
  // http://www.w3.org/TR/touch-events/, http://www.chromestatus.com/features
  @Experimental()
  ElementStream<TouchEvent> get onTouchLeave =>
      touchLeaveEvent.forElement(this);

  /// Stream of `touchmove` events handled by this [Element].
  @DomName('Element.ontouchmove')
  @DocsEditable()
  // http://www.w3.org/TR/touch-events/, http://www.chromestatus.com/features
  @Experimental()
  ElementStream<TouchEvent> get onTouchMove => touchMoveEvent.forElement(this);

  /// Stream of `touchstart` events handled by this [Element].
  @DomName('Element.ontouchstart')
  @DocsEditable()
  // http://www.w3.org/TR/touch-events/, http://www.chromestatus.com/features
  @Experimental()
  ElementStream<TouchEvent> get onTouchStart =>
      touchStartEvent.forElement(this);

  /// Stream of `transitionend` events handled by this [Element].
  @DomName('Element.ontransitionend')
  @DocsEditable()
  @SupportedBrowser(SupportedBrowser.CHROME)
  @SupportedBrowser(SupportedBrowser.FIREFOX)
  @SupportedBrowser(SupportedBrowser.IE, '10')
  @SupportedBrowser(SupportedBrowser.SAFARI)
  ElementStream<TransitionEvent> get onTransitionEnd =>
      transitionEndEvent.forElement(this);

  @DomName('Element.onvolumechange')
  @DocsEditable()
  @Experimental() // untriaged
  ElementStream<Event> get onVolumeChange => volumeChangeEvent.forElement(this);

  @DomName('Element.onwaiting')
  @DocsEditable()
  @Experimental() // untriaged
  ElementStream<Event> get onWaiting => waitingEvent.forElement(this);

  /// Stream of `fullscreenchange` events handled by this [Element].
  @DomName('Element.onwebkitfullscreenchange')
  @DocsEditable()
  // https://dvcs.w3.org/hg/fullscreen/raw-file/tip/Overview.html
  @Experimental()
  ElementStream<Event> get onFullscreenChange =>
      fullscreenChangeEvent.forElement(this);

  /// Stream of `fullscreenerror` events handled by this [Element].
  @DomName('Element.onwebkitfullscreenerror')
  @DocsEditable()
  // https://dvcs.w3.org/hg/fullscreen/raw-file/tip/Overview.html
  @Experimental()
  ElementStream<Event> get onFullscreenError =>
      fullscreenErrorEvent.forElement(this);
}

class _ElementFactoryProvider {
  @DomName('Document.createElement')
  static Element createElement_tag(String tag, String typeExtension) =>
      document.createElement(tag, typeExtension);
}

/**
 * Options for Element.scrollIntoView.
 */
class ScrollAlignment {
  final _value;
  const ScrollAlignment._internal(this._value);
  toString() => 'ScrollAlignment.$_value';

  /// Attempt to align the element to the top of the scrollable area.
  static const TOP = const ScrollAlignment._internal('TOP');

  /// Attempt to center the element in the scrollable area.
  static const CENTER = const ScrollAlignment._internal('CENTER');

  /// Attempt to align the element to the bottom of the scrollable area.
  static const BOTTOM = const ScrollAlignment._internal('BOTTOM');
}
// Copyright (c) 2012, the Dart project authors.  Please see the AUTHORS file
// for details. All rights reserved. Use of this source code is governed by a
// BSD-style license that can be found in the LICENSE file.

// WARNING: Do not edit - generated code.

@DocsEditable()
@DomName('HTMLEmbedElement')
@SupportedBrowser(SupportedBrowser.CHROME)
@SupportedBrowser(SupportedBrowser.IE)
@SupportedBrowser(SupportedBrowser.SAFARI)
@Unstable()
class EmbedElement extends HtmlElement {
  // To suppress missing implicit constructor warnings.
  factory EmbedElement._() {
    throw new UnsupportedError("Not supported");
  }

  @DomName('HTMLEmbedElement.HTMLEmbedElement')
  @DocsEditable()
  factory EmbedElement() => document.createElement("embed");

  @Deprecated("Internal Use Only")
  external static Type get instanceRuntimeType;

  @Deprecated("Internal Use Only")
  EmbedElement.internal_() : super.internal_();

  /**
   * Constructor instantiated by the DOM when a custom element has been created.
   *
   * This can only be called by subclasses from their created constructor.
   */
  EmbedElement.created() : super.created();

  /// Checks if this type is supported on the current platform.
  static bool get supported => true;

  @DomName('HTMLEmbedElement.height')
  @DocsEditable()
  String get height =>
      _blink.BlinkHTMLEmbedElement.instance.height_Getter_(this);

  @DomName('HTMLEmbedElement.height')
  @DocsEditable()
  set height(String value) =>
      _blink.BlinkHTMLEmbedElement.instance.height_Setter_(this, value);

  @DomName('HTMLEmbedElement.name')
  @DocsEditable()
  String get name => _blink.BlinkHTMLEmbedElement.instance.name_Getter_(this);

  @DomName('HTMLEmbedElement.name')
  @DocsEditable()
  set name(String value) =>
      _blink.BlinkHTMLEmbedElement.instance.name_Setter_(this, value);

  @DomName('HTMLEmbedElement.src')
  @DocsEditable()
  String get src => _blink.BlinkHTMLEmbedElement.instance.src_Getter_(this);

  @DomName('HTMLEmbedElement.src')
  @DocsEditable()
  set src(String value) =>
      _blink.BlinkHTMLEmbedElement.instance.src_Setter_(this, value);

  @DomName('HTMLEmbedElement.type')
  @DocsEditable()
  String get type => _blink.BlinkHTMLEmbedElement.instance.type_Getter_(this);

  @DomName('HTMLEmbedElement.type')
  @DocsEditable()
  set type(String value) =>
      _blink.BlinkHTMLEmbedElement.instance.type_Setter_(this, value);

  @DomName('HTMLEmbedElement.width')
  @DocsEditable()
  String get width => _blink.BlinkHTMLEmbedElement.instance.width_Getter_(this);

  @DomName('HTMLEmbedElement.width')
  @DocsEditable()
  set width(String value) =>
      _blink.BlinkHTMLEmbedElement.instance.width_Setter_(this, value);

  @DomName('HTMLEmbedElement.__getter__')
  @DocsEditable()
  bool __getter__(index_OR_name) => _blink.BlinkHTMLEmbedElement.instance
      .$__getter___Callback_1_(this, index_OR_name);

  @DomName('HTMLEmbedElement.__setter__')
  @DocsEditable()
  void __setter__(index_OR_name, Node value) =>
      _blink.BlinkHTMLEmbedElement.instance
          .$__setter___Callback_2_(this, index_OR_name, value);
}
// Copyright (c) 2012, the Dart project authors.  Please see the AUTHORS file
// for details. All rights reserved. Use of this source code is governed by a
// BSD-style license that can be found in the LICENSE file.

// WARNING: Do not edit - generated code.

@DomName('EntriesCallback')
// http://www.w3.org/TR/file-system-api/#the-entriescallback-interface
@Experimental()
typedef void _EntriesCallback(List<Entry> entries);
// Copyright (c) 2012, the Dart project authors.  Please see the AUTHORS file
// for details. All rights reserved. Use of this source code is governed by a
// BSD-style license that can be found in the LICENSE file.

// WARNING: Do not edit - generated code.

@DocsEditable()
@DomName('Entry')
// http://www.w3.org/TR/file-system-api/#the-entry-interface
@Experimental()
class Entry extends DartHtmlDomObject {
  // To suppress missing implicit constructor warnings.
  factory Entry._() {
    throw new UnsupportedError("Not supported");
  }

  @Deprecated("Internal Use Only")
  external static Type get instanceRuntimeType;

  @Deprecated("Internal Use Only")
  Entry.internal_() {}

  @DomName('Entry.filesystem')
  @DocsEditable()
  FileSystem get filesystem =>
      _blink.BlinkEntry.instance.filesystem_Getter_(this);

  @DomName('Entry.fullPath')
  @DocsEditable()
  String get fullPath => _blink.BlinkEntry.instance.fullPath_Getter_(this);

  @DomName('Entry.isDirectory')
  @DocsEditable()
  bool get isDirectory => _blink.BlinkEntry.instance.isDirectory_Getter_(this);

  @DomName('Entry.isFile')
  @DocsEditable()
  bool get isFile => _blink.BlinkEntry.instance.isFile_Getter_(this);

  @DomName('Entry.name')
  @DocsEditable()
  String get name => _blink.BlinkEntry.instance.name_Getter_(this);

  void _copyTo(DirectoryEntry parent,
      {String name,
      _EntryCallback successCallback,
      _ErrorCallback errorCallback}) {
    if (errorCallback != null) {
      _blink.BlinkEntry.instance.copyTo_Callback_4_(
          this, parent, name, successCallback, errorCallback);
      return;
    }
    if (successCallback != null) {
      _blink.BlinkEntry.instance
          .copyTo_Callback_3_(this, parent, name, successCallback);
      return;
    }
    if (name != null) {
      _blink.BlinkEntry.instance.copyTo_Callback_2_(this, parent, name);
      return;
    }
    _blink.BlinkEntry.instance.copyTo_Callback_1_(this, parent);
    return;
  }

  Future<Entry> copyTo(DirectoryEntry parent, {String name}) {
    var completer = new Completer<Entry>();
    _copyTo(parent, name: name, successCallback: (value) {
      completer.complete(value);
    }, errorCallback: (error) {
      completer.completeError(error);
    });
    return completer.future;
  }

  void _getMetadata(MetadataCallback successCallback,
      [_ErrorCallback errorCallback]) {
    if (errorCallback != null) {
      _blink.BlinkEntry.instance
          .getMetadata_Callback_2_(this, successCallback, errorCallback);
      return;
    }
    _blink.BlinkEntry.instance.getMetadata_Callback_1_(this, successCallback);
    return;
  }

  Future<Metadata> getMetadata() {
    var completer = new Completer<Metadata>();
    _getMetadata((value) {
      completer.complete(value);
    }, (error) {
      completer.completeError(error);
    });
    return completer.future;
  }

  void _getParent(
      [_EntryCallback successCallback, _ErrorCallback errorCallback]) {
    if (errorCallback != null) {
      _blink.BlinkEntry.instance
          .getParent_Callback_2_(this, successCallback, errorCallback);
      return;
    }
    if (successCallback != null) {
      _blink.BlinkEntry.instance.getParent_Callback_1_(this, successCallback);
      return;
    }
    _blink.BlinkEntry.instance.getParent_Callback_0_(this);
    return;
  }

  Future<Entry> getParent() {
    var completer = new Completer<Entry>();
    _getParent((value) {
      completer.complete(value);
    }, (error) {
      completer.completeError(error);
    });
    return completer.future;
  }

  void _moveTo(DirectoryEntry parent,
      {String name,
      _EntryCallback successCallback,
      _ErrorCallback errorCallback}) {
    if (errorCallback != null) {
      _blink.BlinkEntry.instance.moveTo_Callback_4_(
          this, parent, name, successCallback, errorCallback);
      return;
    }
    if (successCallback != null) {
      _blink.BlinkEntry.instance
          .moveTo_Callback_3_(this, parent, name, successCallback);
      return;
    }
    if (name != null) {
      _blink.BlinkEntry.instance.moveTo_Callback_2_(this, parent, name);
      return;
    }
    _blink.BlinkEntry.instance.moveTo_Callback_1_(this, parent);
    return;
  }

  Future<Entry> moveTo(DirectoryEntry parent, {String name}) {
    var completer = new Completer<Entry>();
    _moveTo(parent, name: name, successCallback: (value) {
      completer.complete(value);
    }, errorCallback: (error) {
      completer.completeError(error);
    });
    return completer.future;
  }

  void _remove(VoidCallback successCallback, [_ErrorCallback errorCallback]) {
    if (errorCallback != null) {
      _blink.BlinkEntry.instance
          .remove_Callback_2_(this, successCallback, errorCallback);
      return;
    }
    _blink.BlinkEntry.instance.remove_Callback_1_(this, successCallback);
    return;
  }

  Future remove() {
    var completer = new Completer();
    _remove(() {
      completer.complete();
    }, (error) {
      completer.completeError(error);
    });
    return completer.future;
  }

  @DomName('Entry.toURL')
  @DocsEditable()
  String toUrl() => _blink.BlinkEntry.instance.toURL_Callback_0_(this);
}
// Copyright (c) 2012, the Dart project authors.  Please see the AUTHORS file
// for details. All rights reserved. Use of this source code is governed by a
// BSD-style license that can be found in the LICENSE file.

// WARNING: Do not edit - generated code.

@DomName('EntryCallback')
// http://www.w3.org/TR/file-system-api/#the-entrycallback-interface
@Experimental()
typedef void _EntryCallback(Entry entry);
// Copyright (c) 2012, the Dart project authors.  Please see the AUTHORS file
// for details. All rights reserved. Use of this source code is governed by a
// BSD-style license that can be found in the LICENSE file.

// WARNING: Do not edit - generated code.

@DomName('ErrorCallback')
// http://www.w3.org/TR/file-system-api/#the-errorcallback-interface
@Experimental()
typedef void _ErrorCallback(FileError error);
// Copyright (c) 2012, the Dart project authors.  Please see the AUTHORS file
// for details. All rights reserved. Use of this source code is governed by a
// BSD-style license that can be found in the LICENSE file.

// WARNING: Do not edit - generated code.

@DocsEditable()
@DomName('ErrorEvent')
@Unstable()
class ErrorEvent extends Event {
  // To suppress missing implicit constructor warnings.
  factory ErrorEvent._() {
    throw new UnsupportedError("Not supported");
  }

  @DomName('ErrorEvent.ErrorEvent')
  @DocsEditable()
  factory ErrorEvent(String type, [Map eventInitDict]) {
    if (eventInitDict != null) {
      var eventInitDict_1 = convertDartToNative_Dictionary(eventInitDict);
      return _blink.BlinkErrorEvent.instance
          .constructorCallback_2_(type, eventInitDict_1);
    }
    return _blink.BlinkErrorEvent.instance.constructorCallback_1_(type);
  }

  @Deprecated("Internal Use Only")
  external static Type get instanceRuntimeType;

  @Deprecated("Internal Use Only")
  ErrorEvent.internal_() : super.internal_();

  @DomName('ErrorEvent.colno')
  @DocsEditable()
  @Experimental() // untriaged
  int get colno => _blink.BlinkErrorEvent.instance.colno_Getter_(this);

  @DomName('ErrorEvent.error')
  @DocsEditable()
  @Experimental() // untriaged
  Object get error => (_blink.BlinkErrorEvent.instance.error_Getter_(this));

  @DomName('ErrorEvent.filename')
  @DocsEditable()
  String get filename => _blink.BlinkErrorEvent.instance.filename_Getter_(this);

  @DomName('ErrorEvent.lineno')
  @DocsEditable()
  int get lineno => _blink.BlinkErrorEvent.instance.lineno_Getter_(this);

  @DomName('ErrorEvent.message')
  @DocsEditable()
  String get message => _blink.BlinkErrorEvent.instance.message_Getter_(this);
}
// Copyright (c) 2012, the Dart project authors.  Please see the AUTHORS file
// for details. All rights reserved. Use of this source code is governed by a
// BSD-style license that can be found in the LICENSE file.

// WARNING: Do not edit - generated code.

@DomName('Event')
class Event extends DartHtmlDomObject {
  // In JS, canBubble and cancelable are technically required parameters to
  // init*Event. In practice, though, if they aren't provided they simply
  // default to false (since that's Boolean(undefined)).
  //
  // Contrary to JS, we default canBubble and cancelable to true, since that's
  // what people want most of the time anyway.
  factory Event(String type, {bool canBubble: true, bool cancelable: true}) {
    return new Event.eventType('Event', type,
        canBubble: canBubble, cancelable: cancelable);
  }

  /**
   * Creates a new Event object of the specified type.
   *
   * This is analogous to document.createEvent.
   * Normally events should be created via their constructors, if available.
   *
   *     var e = new Event.type('MouseEvent', 'mousedown', true, true);
   */
  factory Event.eventType(String type, String name,
      {bool canBubble: true, bool cancelable: true}) {
    final Event e = document._createEvent(type);
    e._initEvent(name, canBubble, cancelable);
    return e;
  }

  /** The CSS selector involved with event delegation. */
  String _selector;

  /**
   * A pointer to the element whose CSS selector matched within which an event
   * was fired. If this Event was not associated with any Event delegation,
   * accessing this value will throw an [UnsupportedError].
   */
  Element get matchingTarget {
    if (_selector == null) {
      throw new UnsupportedError('Cannot call matchingTarget if this Event did'
          ' not arise as a result of event delegation.');
    }
    Element currentTarget = this.currentTarget;
    Element target = this.target;
    var matchedTarget;
    do {
      if (target.matches(_selector)) return target;
      target = target.parent;
    } while (target != null && target != currentTarget.parent);
    throw new StateError('No selector matched for populating matchedTarget.');
  }

  @DomName('Event.Event')
  @DocsEditable()
  factory Event._(String type, [Map eventInitDict]) {
    if (eventInitDict != null) {
      var eventInitDict_1 = convertDartToNative_Dictionary(eventInitDict);
      return _blink.BlinkEvent.instance
          .constructorCallback_2_(type, eventInitDict_1);
    }
    return _blink.BlinkEvent.instance.constructorCallback_1_(type);
  }

  @Deprecated("Internal Use Only")
  external static Type get instanceRuntimeType;

  @Deprecated("Internal Use Only")
  Event.internal_() {}

  /**
   * This event is being handled by the event target.
   *
   * ## Other resources
   *
   * * [Target phase](http://www.w3.org/TR/DOM-Level-3-Events/#target-phase)
   *   from W3C.
   */
  @DomName('Event.AT_TARGET')
  @DocsEditable()
  static const int AT_TARGET = 2;

  /**
   * This event is bubbling up through the target's ancestors.
   *
   * ## Other resources
   *
   * * [Bubble phase](http://www.w3.org/TR/DOM-Level-3-Events/#bubble-phase)
   *   from W3C.
   */
  @DomName('Event.BUBBLING_PHASE')
  @DocsEditable()
  static const int BUBBLING_PHASE = 3;

  /**
   * This event is propagating through the target's ancestors, starting from the
   * document.
   *
   * ## Other resources
   *
   * * [Bubble phase](http://www.w3.org/TR/DOM-Level-3-Events/#bubble-phase)
   *   from W3C.
   */
  @DomName('Event.CAPTURING_PHASE')
  @DocsEditable()
  static const int CAPTURING_PHASE = 1;

  @DomName('Event.bubbles')
  @DocsEditable()
  bool get bubbles => _blink.BlinkEvent.instance.bubbles_Getter_(this);

  @DomName('Event.cancelable')
  @DocsEditable()
  bool get cancelable => _blink.BlinkEvent.instance.cancelable_Getter_(this);

  @DomName('Event.currentTarget')
  @DocsEditable()
  EventTarget get currentTarget => _convertNativeToDart_EventTarget(
      _blink.BlinkEvent.instance.currentTarget_Getter_(this));

  @DomName('Event.defaultPrevented')
  @DocsEditable()
  bool get defaultPrevented =>
      _blink.BlinkEvent.instance.defaultPrevented_Getter_(this);

  @DomName('Event.eventPhase')
  @DocsEditable()
  int get eventPhase => _blink.BlinkEvent.instance.eventPhase_Getter_(this);

  /**
   * This event's path, taking into account shadow DOM.
   *
   * ## Other resources
   *
   * * [Shadow DOM extensions to
   *   Event](http://w3c.github.io/webcomponents/spec/shadow/#extensions-to-event)
   *   from W3C.
   */
  @DomName('Event.path')
  @DocsEditable()
  // https://dvcs.w3.org/hg/webcomponents/raw-file/tip/spec/shadow/index.html#extensions-to-event
  @Experimental()
  List<EventTarget> get path => (_blink.BlinkEvent.instance.path_Getter_(this));

  @DomName('Event.target')
  @DocsEditable()
  EventTarget get target => _convertNativeToDart_EventTarget(
      _blink.BlinkEvent.instance.target_Getter_(this));

  @DomName('Event.timeStamp')
  @DocsEditable()
  int get timeStamp => _blink.BlinkEvent.instance.timeStamp_Getter_(this);

  @DomName('Event.type')
  @DocsEditable()
  String get type => _blink.BlinkEvent.instance.type_Getter_(this);

  @DomName('Event.initEvent')
  @DocsEditable()
  void _initEvent(String type, bool bubbles, bool cancelable) =>
      _blink.BlinkEvent.instance
          .initEvent_Callback_3_(this, type, bubbles, cancelable);

  @DomName('Event.preventDefault')
  @DocsEditable()
  void preventDefault() =>
      _blink.BlinkEvent.instance.preventDefault_Callback_0_(this);

  @DomName('Event.stopImmediatePropagation')
  @DocsEditable()
  void stopImmediatePropagation() =>
      _blink.BlinkEvent.instance.stopImmediatePropagation_Callback_0_(this);

  @DomName('Event.stopPropagation')
  @DocsEditable()
  void stopPropagation() =>
      _blink.BlinkEvent.instance.stopPropagation_Callback_0_(this);
}
// Copyright (c) 2013, the Dart project authors.  Please see the AUTHORS file
// for details. All rights reserved. Use of this source code is governed by a
// BSD-style license that can be found in the LICENSE file.

@DomName('EventSource')
// http://www.w3.org/TR/eventsource/#the-eventsource-interface
@Experimental() // stable
class EventSource extends EventTarget {
  factory EventSource(String url, {withCredentials: false}) {
    var parsedOptions = {
      'withCredentials': withCredentials,
    };
    return EventSource._factoryEventSource(url, parsedOptions);
  }
  // To suppress missing implicit constructor warnings.
  factory EventSource._() {
    throw new UnsupportedError("Not supported");
  }

  /**
   * Static factory designed to expose `error` events to event
   * handlers that are not necessarily instances of [EventSource].
   *
   * See [EventStreamProvider] for usage information.
   */
  @DomName('EventSource.errorEvent')
  @DocsEditable()
  static const EventStreamProvider<Event> errorEvent =
      const EventStreamProvider<Event>('error');

  /**
   * Static factory designed to expose `message` events to event
   * handlers that are not necessarily instances of [EventSource].
   *
   * See [EventStreamProvider] for usage information.
   */
  @DomName('EventSource.messageEvent')
  @DocsEditable()
  static const EventStreamProvider<MessageEvent> messageEvent =
      const EventStreamProvider<MessageEvent>('message');

  /**
   * Static factory designed to expose `open` events to event
   * handlers that are not necessarily instances of [EventSource].
   *
   * See [EventStreamProvider] for usage information.
   */
  @DomName('EventSource.openEvent')
  @DocsEditable()
  static const EventStreamProvider<Event> openEvent =
      const EventStreamProvider<Event>('open');

  @DomName('EventSource.EventSource')
  @DocsEditable()
  static EventSource _factoryEventSource(String url,
      [Map eventSourceInitDict]) {
    if (eventSourceInitDict != null) {
      var eventSourceInitDict_1 =
          convertDartToNative_Dictionary(eventSourceInitDict);
      return _blink.BlinkEventSource.instance
          .constructorCallback_2_(url, eventSourceInitDict_1);
    }
    return _blink.BlinkEventSource.instance.constructorCallback_1_(url);
  }

  @Deprecated("Internal Use Only")
  external static Type get instanceRuntimeType;

  @Deprecated("Internal Use Only")
  EventSource.internal_() : super.internal_();

  @DomName('EventSource.CLOSED')
  @DocsEditable()
  static const int CLOSED = 2;

  @DomName('EventSource.CONNECTING')
  @DocsEditable()
  static const int CONNECTING = 0;

  @DomName('EventSource.OPEN')
  @DocsEditable()
  static const int OPEN = 1;

  @DomName('EventSource.readyState')
  @DocsEditable()
  int get readyState =>
      _blink.BlinkEventSource.instance.readyState_Getter_(this);

  @DomName('EventSource.url')
  @DocsEditable()
  String get url => _blink.BlinkEventSource.instance.url_Getter_(this);

  @DomName('EventSource.withCredentials')
  @DocsEditable()
  bool get withCredentials =>
      _blink.BlinkEventSource.instance.withCredentials_Getter_(this);

  @DomName('EventSource.close')
  @DocsEditable()
  void close() => _blink.BlinkEventSource.instance.close_Callback_0_(this);

  /// Stream of `error` events handled by this [EventSource].
  @DomName('EventSource.onerror')
  @DocsEditable()
  Stream<Event> get onError => errorEvent.forTarget(this);

  /// Stream of `message` events handled by this [EventSource].
  @DomName('EventSource.onmessage')
  @DocsEditable()
  Stream<MessageEvent> get onMessage => messageEvent.forTarget(this);

  /// Stream of `open` events handled by this [EventSource].
  @DomName('EventSource.onopen')
  @DocsEditable()
  Stream<Event> get onOpen => openEvent.forTarget(this);
}
// Copyright (c) 2012, the Dart project authors.  Please see the AUTHORS file
// for details. All rights reserved. Use of this source code is governed by a
// BSD-style license that can be found in the LICENSE file.

/**
 * Base class that supports listening for and dispatching browser events.
 *
 * Normally events are accessed via the Stream getter:
 *
 *     element.onMouseOver.listen((e) => print('Mouse over!'));
 *
 * To access bubbling events which are declared on one element, but may bubble
 * up to another element type (common for MediaElement events):
 *
 *     MediaElement.pauseEvent.forTarget(document.body).listen(...);
 *
 * To useCapture on events:
 *
 *     Element.keyDownEvent.forTarget(element, useCapture: true).listen(...);
 *
 * Custom events can be declared as:
 *
 *     class DataGenerator {
 *       static EventStreamProvider<Event> dataEvent =
 *           new EventStreamProvider('data');
 *     }
 *
 * Then listeners should access the event with:
 *
 *     DataGenerator.dataEvent.forTarget(element).listen(...);
 *
 * Custom events can also be accessed as:
 *
 *     element.on['some_event'].listen(...);
 *
 * This approach is generally discouraged as it loses the event typing and
 * some DOM events may have multiple platform-dependent event names under the
 * covers. By using the standard Stream getters you will get the platform
 * specific event name automatically.
 */
class Events {
  /* Raw event target. */
  final EventTarget _ptr;

  Events(this._ptr);

  Stream operator [](String type) {
    return new _EventStream(_ptr, type, false);
  }
}

class ElementEvents extends Events {
  static final webkitEvents = {
    'animationend': 'webkitAnimationEnd',
    'animationiteration': 'webkitAnimationIteration',
    'animationstart': 'webkitAnimationStart',
    'fullscreenchange': 'webkitfullscreenchange',
    'fullscreenerror': 'webkitfullscreenerror',
    'keyadded': 'webkitkeyadded',
    'keyerror': 'webkitkeyerror',
    'keymessage': 'webkitkeymessage',
    'needkey': 'webkitneedkey',
    'pointerlockchange': 'webkitpointerlockchange',
    'pointerlockerror': 'webkitpointerlockerror',
    'resourcetimingbufferfull': 'webkitresourcetimingbufferfull',
    'transitionend': 'webkitTransitionEnd',
    'speechchange': 'webkitSpeechChange'
  };

  ElementEvents(Element ptr) : super(ptr);

  Stream operator [](String type) {
    if (webkitEvents.keys.contains(type.toLowerCase())) {
      if (Device.isWebKit) {
        return new _ElementEventStreamImpl(
            _ptr, webkitEvents[type.toLowerCase()], false);
      }
    }
    return new _ElementEventStreamImpl(_ptr, type, false);
  }
}

/**
 * Base class for all browser objects that support events.
 *
 * Use the [on] property to add, and remove events
 * for compile-time type checks and a more concise API.
 */
@DomName('EventTarget')
class EventTarget extends DartHtmlDomObject {
  // Default constructor to allow other classes e.g. GlobalEventHandlers to be
  // constructed using _internalWrap when mapping Blink object to Dart class.
  EventTarget();

  // Custom element created callback.
  EventTarget._created();

  /**
   * This is an ease-of-use accessor for event streams which should only be
   * used when an explicit accessor is not available.
   */
  Events get on => new Events(this);

  void addEventListener(String type, EventListener listener,
      [bool useCapture]) {
    // TODO(leafp): This check is avoid a bug in our dispatch code when
    // listener is null.  The browser treats this call as a no-op in this
    // case, so it's fine to short-circuit it, but we should not have to.
    if (listener != null) {
      _addEventListener(type, listener, useCapture);
    }
  }

  void removeEventListener(String type, EventListener listener,
      [bool useCapture]) {
    // TODO(leafp): This check is avoid a bug in our dispatch code when
    // listener is null.  The browser treats this call as a no-op in this
    // case, so it's fine to short-circuit it, but we should not have to.
    if (listener != null) {
      _removeEventListener(type, listener, useCapture);
    }
  }

  // To suppress missing implicit constructor warnings.
  factory EventTarget._() {
    throw new UnsupportedError("Not supported");
  }

  @Deprecated("Internal Use Only")
  external static Type get instanceRuntimeType;

  @Deprecated("Internal Use Only")
  EventTarget.internal_() {}

  @DomName('EventTarget.addEventListener')
  @DocsEditable()
  void _addEventListener(String type, EventListener listener, [bool capture]) =>
      _blink.BlinkEventTarget.instance
          .addEventListener_Callback_3_(this, type, listener, capture);

  @DomName('EventTarget.dispatchEvent')
  @DocsEditable()
  bool dispatchEvent(Event event) =>
      _blink.BlinkEventTarget.instance.dispatchEvent_Callback_1_(this, event);

  @DomName('EventTarget.removeEventListener')
  @DocsEditable()
  void _removeEventListener(String type, EventListener listener,
          [bool capture]) =>
      _blink.BlinkEventTarget.instance
          .removeEventListener_Callback_3_(this, type, listener, capture);
}
// Copyright (c) 2012, the Dart project authors.  Please see the AUTHORS file
// for details. All rights reserved. Use of this source code is governed by a
// BSD-style license that can be found in the LICENSE file.

// WARNING: Do not edit - generated code.

@DocsEditable()
@DomName('ExtendableEvent')
@Experimental() // untriaged
class ExtendableEvent extends Event {
  // To suppress missing implicit constructor warnings.
  factory ExtendableEvent._() {
    throw new UnsupportedError("Not supported");
  }

  @DomName('ExtendableEvent.ExtendableEvent')
  @DocsEditable()
  factory ExtendableEvent(String type, [Map eventInitDict]) {
    if (eventInitDict != null) {
      var eventInitDict_1 = convertDartToNative_Dictionary(eventInitDict);
      return _blink.BlinkExtendableEvent.instance
          .constructorCallback_2_(type, eventInitDict_1);
    }
    return _blink.BlinkExtendableEvent.instance.constructorCallback_1_(type);
  }

  @Deprecated("Internal Use Only")
  external static Type get instanceRuntimeType;

  @Deprecated("Internal Use Only")
  ExtendableEvent.internal_() : super.internal_();

  @DomName('ExtendableEvent.waitUntil')
  @DocsEditable()
  @Experimental() // untriaged
  void waitUntil(Object value) =>
      _blink.BlinkExtendableEvent.instance.waitUntil_Callback_1_(this, value);
}
// Copyright (c) 2012, the Dart project authors.  Please see the AUTHORS file
// for details. All rights reserved. Use of this source code is governed by a
// BSD-style license that can be found in the LICENSE file.

// WARNING: Do not edit - generated code.

@DocsEditable()
@DomName('FederatedCredential')
@Experimental() // untriaged
class FederatedCredential extends Credential {
  // To suppress missing implicit constructor warnings.
  factory FederatedCredential._() {
    throw new UnsupportedError("Not supported");
  }

  @DomName('FederatedCredential.FederatedCredential')
  @DocsEditable()
  factory FederatedCredential(Map data) {
    var data_1 = convertDartToNative_Dictionary(data);
    return _blink.BlinkFederatedCredential.instance
        .constructorCallback_1_(data_1);
  }

  @Deprecated("Internal Use Only")
  external static Type get instanceRuntimeType;

  @Deprecated("Internal Use Only")
  FederatedCredential.internal_() : super.internal_();

  @DomName('FederatedCredential.protocol')
  @DocsEditable()
  @Experimental() // untriaged
  String get protocol =>
      _blink.BlinkFederatedCredential.instance.protocol_Getter_(this);

  @DomName('FederatedCredential.provider')
  @DocsEditable()
  @Experimental() // untriaged
  String get provider =>
      _blink.BlinkFederatedCredential.instance.provider_Getter_(this);
}
// Copyright (c) 2012, the Dart project authors.  Please see the AUTHORS file
// for details. All rights reserved. Use of this source code is governed by a
// BSD-style license that can be found in the LICENSE file.

// WARNING: Do not edit - generated code.

@DocsEditable()
@DomName('FetchEvent')
@Experimental() // untriaged
class FetchEvent extends ExtendableEvent {
  // To suppress missing implicit constructor warnings.
  factory FetchEvent._() {
    throw new UnsupportedError("Not supported");
  }

  @DomName('FetchEvent.FetchEvent')
  @DocsEditable()
  factory FetchEvent(String type, [Map eventInitDict]) {
    if (eventInitDict != null) {
      var eventInitDict_1 = convertDartToNative_Dictionary(eventInitDict);
      return _blink.BlinkFetchEvent.instance
          .constructorCallback_2_(type, eventInitDict_1);
    }
    return _blink.BlinkFetchEvent.instance.constructorCallback_1_(type);
  }

  @Deprecated("Internal Use Only")
  external static Type get instanceRuntimeType;

  @Deprecated("Internal Use Only")
  FetchEvent.internal_() : super.internal_();

  @DomName('FetchEvent.isReload')
  @DocsEditable()
  @Experimental() // untriaged
  bool get isReload => _blink.BlinkFetchEvent.instance.isReload_Getter_(this);

  @DomName('FetchEvent.request')
  @DocsEditable()
  @Experimental() // untriaged
  _Request get request => _blink.BlinkFetchEvent.instance.request_Getter_(this);

  @DomName('FetchEvent.respondWith')
  @DocsEditable()
  @Experimental() // untriaged
  void respondWith(Object value) =>
      _blink.BlinkFetchEvent.instance.respondWith_Callback_1_(this, value);
}
// Copyright (c) 2012, the Dart project authors.  Please see the AUTHORS file
// for details. All rights reserved. Use of this source code is governed by a
// BSD-style license that can be found in the LICENSE file.

// WARNING: Do not edit - generated code.

@DocsEditable()
@DomName('HTMLFieldSetElement')
@Unstable()
class FieldSetElement extends HtmlElement {
  // To suppress missing implicit constructor warnings.
  factory FieldSetElement._() {
    throw new UnsupportedError("Not supported");
  }

  @DomName('HTMLFieldSetElement.HTMLFieldSetElement')
  @DocsEditable()
  factory FieldSetElement() => document.createElement("fieldset");

  @Deprecated("Internal Use Only")
  external static Type get instanceRuntimeType;

  @Deprecated("Internal Use Only")
  FieldSetElement.internal_() : super.internal_();

  /**
   * Constructor instantiated by the DOM when a custom element has been created.
   *
   * This can only be called by subclasses from their created constructor.
   */
  FieldSetElement.created() : super.created();

  @DomName('HTMLFieldSetElement.disabled')
  @DocsEditable()
  bool get disabled =>
      _blink.BlinkHTMLFieldSetElement.instance.disabled_Getter_(this);

  @DomName('HTMLFieldSetElement.disabled')
  @DocsEditable()
  set disabled(bool value) =>
      _blink.BlinkHTMLFieldSetElement.instance.disabled_Setter_(this, value);

  @DomName('HTMLFieldSetElement.elements')
  @DocsEditable()
  HtmlFormControlsCollection get elements =>
      _blink.BlinkHTMLFieldSetElement.instance.elements_Getter_(this);

  @DomName('HTMLFieldSetElement.form')
  @DocsEditable()
  FormElement get form =>
      _blink.BlinkHTMLFieldSetElement.instance.form_Getter_(this);

  @DomName('HTMLFieldSetElement.name')
  @DocsEditable()
  String get name =>
      _blink.BlinkHTMLFieldSetElement.instance.name_Getter_(this);

  @DomName('HTMLFieldSetElement.name')
  @DocsEditable()
  set name(String value) =>
      _blink.BlinkHTMLFieldSetElement.instance.name_Setter_(this, value);

  @DomName('HTMLFieldSetElement.type')
  @DocsEditable()
  String get type =>
      _blink.BlinkHTMLFieldSetElement.instance.type_Getter_(this);

  @DomName('HTMLFieldSetElement.validationMessage')
  @DocsEditable()
  String get validationMessage =>
      _blink.BlinkHTMLFieldSetElement.instance.validationMessage_Getter_(this);

  @DomName('HTMLFieldSetElement.validity')
  @DocsEditable()
  ValidityState get validity =>
      _blink.BlinkHTMLFieldSetElement.instance.validity_Getter_(this);

  @DomName('HTMLFieldSetElement.willValidate')
  @DocsEditable()
  bool get willValidate =>
      _blink.BlinkHTMLFieldSetElement.instance.willValidate_Getter_(this);

  @DomName('HTMLFieldSetElement.checkValidity')
  @DocsEditable()
  bool checkValidity() =>
      _blink.BlinkHTMLFieldSetElement.instance.checkValidity_Callback_0_(this);

  @DomName('HTMLFieldSetElement.reportValidity')
  @DocsEditable()
  @Experimental() // untriaged
  bool reportValidity() =>
      _blink.BlinkHTMLFieldSetElement.instance.reportValidity_Callback_0_(this);

  @DomName('HTMLFieldSetElement.setCustomValidity')
  @DocsEditable()
  void setCustomValidity(String error) =>
      _blink.BlinkHTMLFieldSetElement.instance
          .setCustomValidity_Callback_1_(this, error);
}
// Copyright (c) 2012, the Dart project authors.  Please see the AUTHORS file
// for details. All rights reserved. Use of this source code is governed by a
// BSD-style license that can be found in the LICENSE file.

// WARNING: Do not edit - generated code.

@DocsEditable()
@DomName('File')
class File extends Blob {
  // To suppress missing implicit constructor warnings.
  factory File._() {
    throw new UnsupportedError("Not supported");
  }

  @DomName('File.File')
  @DocsEditable()
  factory File(List<Object> fileBits, String fileName, [Map options]) {
    if (options != null) {
      var options_1 = convertDartToNative_Dictionary(options);
      return _blink.BlinkFile.instance
          .constructorCallback_3_(fileBits, fileName, options_1);
    }
    return _blink.BlinkFile.instance.constructorCallback_2_(fileBits, fileName);
  }

  @Deprecated("Internal Use Only")
  external static Type get instanceRuntimeType;

  @Deprecated("Internal Use Only")
  File.internal_() : super.internal_();

  @DomName('File.lastModified')
  @DocsEditable()
  @Experimental() // untriaged
  int get lastModified => _blink.BlinkFile.instance.lastModified_Getter_(this);

  @DomName('File.lastModifiedDate')
  @DocsEditable()
  DateTime get lastModifiedDate =>
      _blink.BlinkFile.instance.lastModifiedDate_Getter_(this);

  @DomName('File.name')
  @DocsEditable()
  String get name => _blink.BlinkFile.instance.name_Getter_(this);

  @DomName('File.webkitRelativePath')
  @DocsEditable()
  @SupportedBrowser(SupportedBrowser.CHROME)
  @SupportedBrowser(SupportedBrowser.SAFARI)
  @Experimental()
  // https://plus.sandbox.google.com/+AddyOsmani/posts/Dk5UhZ6zfF3
  String get relativePath =>
      _blink.BlinkFile.instance.webkitRelativePath_Getter_(this);
}
// Copyright (c) 2012, the Dart project authors.  Please see the AUTHORS file
// for details. All rights reserved. Use of this source code is governed by a
// BSD-style license that can be found in the LICENSE file.

// WARNING: Do not edit - generated code.

@DomName('FileCallback')
// http://www.w3.org/TR/file-system-api/#the-filecallback-interface
@Experimental()
typedef void _FileCallback(File file);
// Copyright (c) 2012, the Dart project authors.  Please see the AUTHORS file
// for details. All rights reserved. Use of this source code is governed by a
// BSD-style license that can be found in the LICENSE file.

// WARNING: Do not edit - generated code.

@DocsEditable()
@DomName('FileEntry')
// http://www.w3.org/TR/file-system-api/#the-fileentry-interface
@Experimental()
class FileEntry extends Entry {
  // To suppress missing implicit constructor warnings.
  factory FileEntry._() {
    throw new UnsupportedError("Not supported");
  }

  @Deprecated("Internal Use Only")
  external static Type get instanceRuntimeType;

  @Deprecated("Internal Use Only")
  FileEntry.internal_() : super.internal_();

  void _createWriter(_FileWriterCallback successCallback,
      [_ErrorCallback errorCallback]) {
    if (errorCallback != null) {
      _blink.BlinkFileEntry.instance
          .createWriter_Callback_2_(this, successCallback, errorCallback);
      return;
    }
    _blink.BlinkFileEntry.instance
        .createWriter_Callback_1_(this, successCallback);
    return;
  }

  Future<FileWriter> createWriter() {
    var completer = new Completer<FileWriter>();
    _createWriter((value) {
      completer.complete(value);
    }, (error) {
      completer.completeError(error);
    });
    return completer.future;
  }

  void _file(_FileCallback successCallback, [_ErrorCallback errorCallback]) {
    if (errorCallback != null) {
      _blink.BlinkFileEntry.instance
          .file_Callback_2_(this, successCallback, errorCallback);
      return;
    }
    _blink.BlinkFileEntry.instance.file_Callback_1_(this, successCallback);
    return;
  }

  Future<File> file() {
    var completer = new Completer<File>();
    _file((value) {
      completer.complete(value);
    }, (error) {
      completer.completeError(error);
    });
    return completer.future;
  }
}
// Copyright (c) 2012, the Dart project authors.  Please see the AUTHORS file
// for details. All rights reserved. Use of this source code is governed by a
// BSD-style license that can be found in the LICENSE file.

// WARNING: Do not edit - generated code.

@DocsEditable()
@DomName('FileError')
// http://dev.w3.org/2009/dap/file-system/pub/FileSystem/
@Experimental()
class FileError extends DomError {
  // To suppress missing implicit constructor warnings.
  factory FileError._() {
    throw new UnsupportedError("Not supported");
  }

  @Deprecated("Internal Use Only")
  external static Type get instanceRuntimeType;

  @Deprecated("Internal Use Only")
  FileError.internal_() : super.internal_();

  @DomName('FileError.ABORT_ERR')
  @DocsEditable()
  static const int ABORT_ERR = 3;

  @DomName('FileError.ENCODING_ERR')
  @DocsEditable()
  static const int ENCODING_ERR = 5;

  @DomName('FileError.INVALID_MODIFICATION_ERR')
  @DocsEditable()
  static const int INVALID_MODIFICATION_ERR = 9;

  @DomName('FileError.INVALID_STATE_ERR')
  @DocsEditable()
  static const int INVALID_STATE_ERR = 7;

  @DomName('FileError.NOT_FOUND_ERR')
  @DocsEditable()
  static const int NOT_FOUND_ERR = 1;

  @DomName('FileError.NOT_READABLE_ERR')
  @DocsEditable()
  static const int NOT_READABLE_ERR = 4;

  @DomName('FileError.NO_MODIFICATION_ALLOWED_ERR')
  @DocsEditable()
  static const int NO_MODIFICATION_ALLOWED_ERR = 6;

  @DomName('FileError.PATH_EXISTS_ERR')
  @DocsEditable()
  static const int PATH_EXISTS_ERR = 12;

  @DomName('FileError.QUOTA_EXCEEDED_ERR')
  @DocsEditable()
  static const int QUOTA_EXCEEDED_ERR = 10;

  @DomName('FileError.SECURITY_ERR')
  @DocsEditable()
  static const int SECURITY_ERR = 2;

  @DomName('FileError.SYNTAX_ERR')
  @DocsEditable()
  static const int SYNTAX_ERR = 8;

  @DomName('FileError.TYPE_MISMATCH_ERR')
  @DocsEditable()
  static const int TYPE_MISMATCH_ERR = 11;

  @DomName('FileError.code')
  @DocsEditable()
  int get code => _blink.BlinkFileError.instance.code_Getter_(this);
}
// Copyright (c) 2012, the Dart project authors.  Please see the AUTHORS file
// for details. All rights reserved. Use of this source code is governed by a
// BSD-style license that can be found in the LICENSE file.

// WARNING: Do not edit - generated code.

@DocsEditable()
@DomName('FileList')
class FileList extends DartHtmlDomObject
    with ListMixin<File>, ImmutableListMixin<File>
    implements List<File> {
  // To suppress missing implicit constructor warnings.
  factory FileList._() {
    throw new UnsupportedError("Not supported");
  }

  @Deprecated("Internal Use Only")
  external static Type get instanceRuntimeType;

  @Deprecated("Internal Use Only")
  FileList.internal_() {}

  @DomName('FileList.length')
  @DocsEditable()
  int get length => _blink.BlinkFileList.instance.length_Getter_(this);

  File operator [](int index) {
    if (index < 0 || index >= length) throw new RangeError.index(index, this);
    return _nativeIndexedGetter(index);
  }

  File _nativeIndexedGetter(int index) =>
      (_blink.BlinkFileList.instance.item_Callback_1_(this, index));

  void operator []=(int index, File value) {
    throw new UnsupportedError("Cannot assign element of immutable List.");
  }
  // -- start List<File> mixins.
  // File is the element type.

  set length(int value) {
    throw new UnsupportedError("Cannot resize immutable List.");
  }

  File get first {
    if (this.length > 0) {
      return _nativeIndexedGetter(0);
    }
    throw new StateError("No elements");
  }

  File get last {
    int len = this.length;
    if (len > 0) {
      return _nativeIndexedGetter(len - 1);
    }
    throw new StateError("No elements");
  }

  File get single {
    int len = this.length;
    if (len == 1) {
      return _nativeIndexedGetter(0);
    }
    if (len == 0) throw new StateError("No elements");
    throw new StateError("More than one element");
  }

  File elementAt(int index) => this[index];
  // -- end List<File> mixins.

  @DomName('FileList.item')
  @DocsEditable()
  File item(int index) =>
      _blink.BlinkFileList.instance.item_Callback_1_(this, index);
}
// Copyright (c) 2014, the Dart project authors.  Please see the AUTHORS file
// for details. All rights reserved. Use of this source code is governed by a
// BSD-style license that can be found in the LICENSE file.

@DocsEditable()
@DomName('FileReader')
class FileReader extends EventTarget {
  @DomName('FileReader.result')
  @DocsEditable()
  Object get result {
    var res = _blink.BlinkFileReader.instance.result_Getter_(this);
    if (res is ByteBuffer) {
      return new Uint8List.view(res);
    }
    return res;
  }

  // To suppress missing implicit constructor warnings.
  factory FileReader._() {
    throw new UnsupportedError("Not supported");
  }

  /**
   * Static factory designed to expose `abort` events to event
   * handlers that are not necessarily instances of [FileReader].
   *
   * See [EventStreamProvider] for usage information.
   */
  @DomName('FileReader.abortEvent')
  @DocsEditable()
  static const EventStreamProvider<ProgressEvent> abortEvent =
      const EventStreamProvider<ProgressEvent>('abort');

  /**
   * Static factory designed to expose `error` events to event
   * handlers that are not necessarily instances of [FileReader].
   *
   * See [EventStreamProvider] for usage information.
   */
  @DomName('FileReader.errorEvent')
  @DocsEditable()
  static const EventStreamProvider<Event> errorEvent =
      const EventStreamProvider<Event>('error');

  /**
   * Static factory designed to expose `load` events to event
   * handlers that are not necessarily instances of [FileReader].
   *
   * See [EventStreamProvider] for usage information.
   */
  @DomName('FileReader.loadEvent')
  @DocsEditable()
  static const EventStreamProvider<ProgressEvent> loadEvent =
      const EventStreamProvider<ProgressEvent>('load');

  /**
   * Static factory designed to expose `loadend` events to event
   * handlers that are not necessarily instances of [FileReader].
   *
   * See [EventStreamProvider] for usage information.
   */
  @DomName('FileReader.loadendEvent')
  @DocsEditable()
  static const EventStreamProvider<ProgressEvent> loadEndEvent =
      const EventStreamProvider<ProgressEvent>('loadend');

  /**
   * Static factory designed to expose `loadstart` events to event
   * handlers that are not necessarily instances of [FileReader].
   *
   * See [EventStreamProvider] for usage information.
   */
  @DomName('FileReader.loadstartEvent')
  @DocsEditable()
  static const EventStreamProvider<ProgressEvent> loadStartEvent =
      const EventStreamProvider<ProgressEvent>('loadstart');

  /**
   * Static factory designed to expose `progress` events to event
   * handlers that are not necessarily instances of [FileReader].
   *
   * See [EventStreamProvider] for usage information.
   */
  @DomName('FileReader.progressEvent')
  @DocsEditable()
  static const EventStreamProvider<ProgressEvent> progressEvent =
      const EventStreamProvider<ProgressEvent>('progress');

  @DomName('FileReader.FileReader')
  @DocsEditable()
  factory FileReader() {
    return _blink.BlinkFileReader.instance.constructorCallback_0_();
  }

  @Deprecated("Internal Use Only")
  external static Type get instanceRuntimeType;

  @Deprecated("Internal Use Only")
  FileReader.internal_() : super.internal_();

  @DomName('FileReader.DONE')
  @DocsEditable()
  static const int DONE = 2;

  @DomName('FileReader.EMPTY')
  @DocsEditable()
  static const int EMPTY = 0;

  @DomName('FileReader.LOADING')
  @DocsEditable()
  static const int LOADING = 1;

  @DomName('FileReader.error')
  @DocsEditable()
  FileError get error => _blink.BlinkFileReader.instance.error_Getter_(this);

  @DomName('FileReader.readyState')
  @DocsEditable()
  int get readyState =>
      _blink.BlinkFileReader.instance.readyState_Getter_(this);

  @DomName('FileReader.result')
  @DocsEditable()
  Object get _result => (_blink.BlinkFileReader.instance.result_Getter_(this));

  @DomName('FileReader.abort')
  @DocsEditable()
  void abort() => _blink.BlinkFileReader.instance.abort_Callback_0_(this);

  @DomName('FileReader.readAsArrayBuffer')
  @DocsEditable()
  void readAsArrayBuffer(Blob blob) =>
      _blink.BlinkFileReader.instance.readAsArrayBuffer_Callback_1_(this, blob);

  @DomName('FileReader.readAsDataURL')
  @DocsEditable()
  void readAsDataUrl(Blob blob) =>
      _blink.BlinkFileReader.instance.readAsDataURL_Callback_1_(this, blob);

  void readAsText(Blob blob, [String label]) {
    if (label != null) {
      _blink.BlinkFileReader.instance.readAsText_Callback_2_(this, blob, label);
      return;
    }
    _blink.BlinkFileReader.instance.readAsText_Callback_1_(this, blob);
    return;
  }

  /// Stream of `abort` events handled by this [FileReader].
  @DomName('FileReader.onabort')
  @DocsEditable()
  Stream<ProgressEvent> get onAbort => abortEvent.forTarget(this);

  /// Stream of `error` events handled by this [FileReader].
  @DomName('FileReader.onerror')
  @DocsEditable()
  Stream<Event> get onError => errorEvent.forTarget(this);

  /// Stream of `load` events handled by this [FileReader].
  @DomName('FileReader.onload')
  @DocsEditable()
  Stream<ProgressEvent> get onLoad => loadEvent.forTarget(this);

  /// Stream of `loadend` events handled by this [FileReader].
  @DomName('FileReader.onloadend')
  @DocsEditable()
  Stream<ProgressEvent> get onLoadEnd => loadEndEvent.forTarget(this);

  /// Stream of `loadstart` events handled by this [FileReader].
  @DomName('FileReader.onloadstart')
  @DocsEditable()
  Stream<ProgressEvent> get onLoadStart => loadStartEvent.forTarget(this);

  /// Stream of `progress` events handled by this [FileReader].
  @DomName('FileReader.onprogress')
  @DocsEditable()
  Stream<ProgressEvent> get onProgress => progressEvent.forTarget(this);
}
// Copyright (c) 2012, the Dart project authors.  Please see the AUTHORS file
// for details. All rights reserved. Use of this source code is governed by a
// BSD-style license that can be found in the LICENSE file.

// WARNING: Do not edit - generated code.

@DocsEditable()
@DomName('Stream')
@Experimental() // untriaged
class FileStream extends DartHtmlDomObject {
  // To suppress missing implicit constructor warnings.
  factory FileStream._() {
    throw new UnsupportedError("Not supported");
  }

  @Deprecated("Internal Use Only")
  external static Type get instanceRuntimeType;

  @Deprecated("Internal Use Only")
  FileStream.internal_() {}

  @DomName('Stream.type')
  @DocsEditable()
  @Experimental() // untriaged
  String get type => _blink.BlinkStream.instance.type_Getter_(this);
}
// Copyright (c) 2012, the Dart project authors.  Please see the AUTHORS file
// for details. All rights reserved. Use of this source code is governed by a
// BSD-style license that can be found in the LICENSE file.

// WARNING: Do not edit - generated code.

@DocsEditable()
@DomName('DOMFileSystem')
@SupportedBrowser(SupportedBrowser.CHROME)
@Experimental()
// http://www.w3.org/TR/file-system-api/
class FileSystem extends DartHtmlDomObject {
  // To suppress missing implicit constructor warnings.
  factory FileSystem._() {
    throw new UnsupportedError("Not supported");
  }

  @Deprecated("Internal Use Only")
  external static Type get instanceRuntimeType;

  @Deprecated("Internal Use Only")
  FileSystem.internal_() {}

  /// Checks if this type is supported on the current platform.
  static bool get supported => true;

  @DomName('DOMFileSystem.name')
  @DocsEditable()
  String get name => _blink.BlinkDOMFileSystem.instance.name_Getter_(this);

  @DomName('DOMFileSystem.root')
  @DocsEditable()
  DirectoryEntry get root =>
      _blink.BlinkDOMFileSystem.instance.root_Getter_(this);
}
// Copyright (c) 2012, the Dart project authors.  Please see the AUTHORS file
// for details. All rights reserved. Use of this source code is governed by a
// BSD-style license that can be found in the LICENSE file.

// WARNING: Do not edit - generated code.

@DomName('FileSystemCallback')
// http://www.w3.org/TR/file-system-api/#the-filesystemcallback-interface
@Experimental()
typedef void _FileSystemCallback(FileSystem fileSystem);
// Copyright (c) 2012, the Dart project authors.  Please see the AUTHORS file
// for details. All rights reserved. Use of this source code is governed by a
// BSD-style license that can be found in the LICENSE file.

// WARNING: Do not edit - generated code.

@DocsEditable()
@DomName('FileWriter')
// http://www.w3.org/TR/file-writer-api/#the-filewriter-interface
@Experimental()
class FileWriter extends EventTarget {
  // To suppress missing implicit constructor warnings.
  factory FileWriter._() {
    throw new UnsupportedError("Not supported");
  }

  /**
   * Static factory designed to expose `abort` events to event
   * handlers that are not necessarily instances of [FileWriter].
   *
   * See [EventStreamProvider] for usage information.
   */
  @DomName('FileWriter.abortEvent')
  @DocsEditable()
  static const EventStreamProvider<ProgressEvent> abortEvent =
      const EventStreamProvider<ProgressEvent>('abort');

  /**
   * Static factory designed to expose `error` events to event
   * handlers that are not necessarily instances of [FileWriter].
   *
   * See [EventStreamProvider] for usage information.
   */
  @DomName('FileWriter.errorEvent')
  @DocsEditable()
  static const EventStreamProvider<Event> errorEvent =
      const EventStreamProvider<Event>('error');

  /**
   * Static factory designed to expose `progress` events to event
   * handlers that are not necessarily instances of [FileWriter].
   *
   * See [EventStreamProvider] for usage information.
   */
  @DomName('FileWriter.progressEvent')
  @DocsEditable()
  static const EventStreamProvider<ProgressEvent> progressEvent =
      const EventStreamProvider<ProgressEvent>('progress');

  /**
   * Static factory designed to expose `write` events to event
   * handlers that are not necessarily instances of [FileWriter].
   *
   * See [EventStreamProvider] for usage information.
   */
  @DomName('FileWriter.writeEvent')
  @DocsEditable()
  static const EventStreamProvider<ProgressEvent> writeEvent =
      const EventStreamProvider<ProgressEvent>('write');

  /**
   * Static factory designed to expose `writeend` events to event
   * handlers that are not necessarily instances of [FileWriter].
   *
   * See [EventStreamProvider] for usage information.
   */
  @DomName('FileWriter.writeendEvent')
  @DocsEditable()
  static const EventStreamProvider<ProgressEvent> writeEndEvent =
      const EventStreamProvider<ProgressEvent>('writeend');

  /**
   * Static factory designed to expose `writestart` events to event
   * handlers that are not necessarily instances of [FileWriter].
   *
   * See [EventStreamProvider] for usage information.
   */
  @DomName('FileWriter.writestartEvent')
  @DocsEditable()
  static const EventStreamProvider<ProgressEvent> writeStartEvent =
      const EventStreamProvider<ProgressEvent>('writestart');

  @Deprecated("Internal Use Only")
  external static Type get instanceRuntimeType;

  @Deprecated("Internal Use Only")
  FileWriter.internal_() : super.internal_();

  @DomName('FileWriter.DONE')
  @DocsEditable()
  static const int DONE = 2;

  @DomName('FileWriter.INIT')
  @DocsEditable()
  static const int INIT = 0;

  @DomName('FileWriter.WRITING')
  @DocsEditable()
  static const int WRITING = 1;

  @DomName('FileWriter.error')
  @DocsEditable()
  FileError get error => _blink.BlinkFileWriter.instance.error_Getter_(this);

  @DomName('FileWriter.length')
  @DocsEditable()
  int get length => _blink.BlinkFileWriter.instance.length_Getter_(this);

  @DomName('FileWriter.position')
  @DocsEditable()
  int get position => _blink.BlinkFileWriter.instance.position_Getter_(this);

  @DomName('FileWriter.readyState')
  @DocsEditable()
  int get readyState =>
      _blink.BlinkFileWriter.instance.readyState_Getter_(this);

  @DomName('FileWriter.abort')
  @DocsEditable()
  void abort() => _blink.BlinkFileWriter.instance.abort_Callback_0_(this);

  @DomName('FileWriter.seek')
  @DocsEditable()
  void seek(int position) =>
      _blink.BlinkFileWriter.instance.seek_Callback_1_(this, position);

  @DomName('FileWriter.truncate')
  @DocsEditable()
  void truncate(int size) =>
      _blink.BlinkFileWriter.instance.truncate_Callback_1_(this, size);

  @DomName('FileWriter.write')
  @DocsEditable()
  void write(Blob data) =>
      _blink.BlinkFileWriter.instance.write_Callback_1_(this, data);

  /// Stream of `abort` events handled by this [FileWriter].
  @DomName('FileWriter.onabort')
  @DocsEditable()
  Stream<ProgressEvent> get onAbort => abortEvent.forTarget(this);

  /// Stream of `error` events handled by this [FileWriter].
  @DomName('FileWriter.onerror')
  @DocsEditable()
  Stream<Event> get onError => errorEvent.forTarget(this);

  /// Stream of `progress` events handled by this [FileWriter].
  @DomName('FileWriter.onprogress')
  @DocsEditable()
  Stream<ProgressEvent> get onProgress => progressEvent.forTarget(this);

  /// Stream of `write` events handled by this [FileWriter].
  @DomName('FileWriter.onwrite')
  @DocsEditable()
  Stream<ProgressEvent> get onWrite => writeEvent.forTarget(this);

  /// Stream of `writeend` events handled by this [FileWriter].
  @DomName('FileWriter.onwriteend')
  @DocsEditable()
  Stream<ProgressEvent> get onWriteEnd => writeEndEvent.forTarget(this);

  /// Stream of `writestart` events handled by this [FileWriter].
  @DomName('FileWriter.onwritestart')
  @DocsEditable()
  Stream<ProgressEvent> get onWriteStart => writeStartEvent.forTarget(this);
}
// Copyright (c) 2012, the Dart project authors.  Please see the AUTHORS file
// for details. All rights reserved. Use of this source code is governed by a
// BSD-style license that can be found in the LICENSE file.

// WARNING: Do not edit - generated code.

@DomName('FileWriterCallback')
// http://www.w3.org/TR/file-writer-api/#idl-def-FileWriter
@Experimental()
typedef void _FileWriterCallback(FileWriter fileWriter);
// Copyright (c) 2012, the Dart project authors.  Please see the AUTHORS file
// for details. All rights reserved. Use of this source code is governed by a
// BSD-style license that can be found in the LICENSE file.

// WARNING: Do not edit - generated code.

@DocsEditable()
@DomName('FocusEvent')
class FocusEvent extends UIEvent {
  // To suppress missing implicit constructor warnings.
  factory FocusEvent._() {
    throw new UnsupportedError("Not supported");
  }

  @DomName('FocusEvent.FocusEvent')
  @DocsEditable()
  factory FocusEvent(String type, [Map eventInitDict]) {
    if (eventInitDict != null) {
      var eventInitDict_1 = convertDartToNative_Dictionary(eventInitDict);
      return _blink.BlinkFocusEvent.instance
          .constructorCallback_2_(type, eventInitDict_1);
    }
    return _blink.BlinkFocusEvent.instance.constructorCallback_1_(type);
  }

  @Deprecated("Internal Use Only")
  external static Type get instanceRuntimeType;

  @Deprecated("Internal Use Only")
  FocusEvent.internal_() : super.internal_();

  @DomName('FocusEvent.relatedTarget')
  @DocsEditable()
  EventTarget get relatedTarget => _convertNativeToDart_EventTarget(
      _blink.BlinkFocusEvent.instance.relatedTarget_Getter_(this));
}
// Copyright (c) 2012, the Dart project authors.  Please see the AUTHORS file
// for details. All rights reserved. Use of this source code is governed by a
// BSD-style license that can be found in the LICENSE file.

// WARNING: Do not edit - generated code.

@DocsEditable()
@DomName('FontFace')
@Experimental() // untriaged
class FontFace extends DartHtmlDomObject {
  // To suppress missing implicit constructor warnings.
  factory FontFace._() {
    throw new UnsupportedError("Not supported");
  }

  @DomName('FontFace.FontFace')
  @DocsEditable()
  factory FontFace(String family, Object source, [Map descriptors]) {
    if (descriptors != null) {
      var descriptors_1 = convertDartToNative_Dictionary(descriptors);
      return _blink.BlinkFontFace.instance
          .constructorCallback_3_(family, source, descriptors_1);
    }
    return _blink.BlinkFontFace.instance.constructorCallback_2_(family, source);
  }

  @Deprecated("Internal Use Only")
  external static Type get instanceRuntimeType;

  @Deprecated("Internal Use Only")
  FontFace.internal_() {}

  @DomName('FontFace.family')
  @DocsEditable()
  @Experimental() // untriaged
  String get family => _blink.BlinkFontFace.instance.family_Getter_(this);

  @DomName('FontFace.family')
  @DocsEditable()
  @Experimental() // untriaged
  set family(String value) =>
      _blink.BlinkFontFace.instance.family_Setter_(this, value);

  @DomName('FontFace.featureSettings')
  @DocsEditable()
  @Experimental() // untriaged
  String get featureSettings =>
      _blink.BlinkFontFace.instance.featureSettings_Getter_(this);

  @DomName('FontFace.featureSettings')
  @DocsEditable()
  @Experimental() // untriaged
  set featureSettings(String value) =>
      _blink.BlinkFontFace.instance.featureSettings_Setter_(this, value);

  @DomName('FontFace.loaded')
  @DocsEditable()
  @Experimental() // untriaged
  Future get loaded => convertNativePromiseToDartFuture(
      _blink.BlinkFontFace.instance.loaded_Getter_(this));

  @DomName('FontFace.status')
  @DocsEditable()
  @Experimental() // untriaged
  String get status => _blink.BlinkFontFace.instance.status_Getter_(this);

  @DomName('FontFace.stretch')
  @DocsEditable()
  @Experimental() // untriaged
  String get stretch => _blink.BlinkFontFace.instance.stretch_Getter_(this);

  @DomName('FontFace.stretch')
  @DocsEditable()
  @Experimental() // untriaged
  set stretch(String value) =>
      _blink.BlinkFontFace.instance.stretch_Setter_(this, value);

  @DomName('FontFace.style')
  @DocsEditable()
  @Experimental() // untriaged
  String get style => _blink.BlinkFontFace.instance.style_Getter_(this);

  @DomName('FontFace.style')
  @DocsEditable()
  @Experimental() // untriaged
  set style(String value) =>
      _blink.BlinkFontFace.instance.style_Setter_(this, value);

  @DomName('FontFace.unicodeRange')
  @DocsEditable()
  @Experimental() // untriaged
  String get unicodeRange =>
      _blink.BlinkFontFace.instance.unicodeRange_Getter_(this);

  @DomName('FontFace.unicodeRange')
  @DocsEditable()
  @Experimental() // untriaged
  set unicodeRange(String value) =>
      _blink.BlinkFontFace.instance.unicodeRange_Setter_(this, value);

  @DomName('FontFace.variant')
  @DocsEditable()
  @Experimental() // untriaged
  String get variant => _blink.BlinkFontFace.instance.variant_Getter_(this);

  @DomName('FontFace.variant')
  @DocsEditable()
  @Experimental() // untriaged
  set variant(String value) =>
      _blink.BlinkFontFace.instance.variant_Setter_(this, value);

  @DomName('FontFace.weight')
  @DocsEditable()
  @Experimental() // untriaged
  String get weight => _blink.BlinkFontFace.instance.weight_Getter_(this);

  @DomName('FontFace.weight')
  @DocsEditable()
  @Experimental() // untriaged
  set weight(String value) =>
      _blink.BlinkFontFace.instance.weight_Setter_(this, value);

  @DomName('FontFace.load')
  @DocsEditable()
  @Experimental() // untriaged
  Future load() => convertNativePromiseToDartFuture(
      _blink.BlinkFontFace.instance.load_Callback_0_(this));
}
// Copyright (c) 2012, the Dart project authors.  Please see the AUTHORS file
// for details. All rights reserved. Use of this source code is governed by a
// BSD-style license that can be found in the LICENSE file.

// WARNING: Do not edit - generated code.

@DocsEditable()
@DomName('FontFaceSet')
@Experimental() // untriaged
class FontFaceSet extends EventTarget {
  // To suppress missing implicit constructor warnings.
  factory FontFaceSet._() {
    throw new UnsupportedError("Not supported");
  }

  @Deprecated("Internal Use Only")
  external static Type get instanceRuntimeType;

  @Deprecated("Internal Use Only")
  FontFaceSet.internal_() : super.internal_();

  @DomName('FontFaceSet.size')
  @DocsEditable()
  @Experimental() // untriaged
  int get size => _blink.BlinkFontFaceSet.instance.size_Getter_(this);

  @DomName('FontFaceSet.status')
  @DocsEditable()
  @Experimental() // untriaged
  String get status => _blink.BlinkFontFaceSet.instance.status_Getter_(this);

  @DomName('FontFaceSet.add')
  @DocsEditable()
  @Experimental() // untriaged
  void add(FontFace fontFace) =>
      _blink.BlinkFontFaceSet.instance.add_Callback_1_(this, fontFace);

  bool check(String font, [String text]) {
    if (text != null) {
      return _blink.BlinkFontFaceSet.instance
          .check_Callback_2_(this, font, text);
    }
    return _blink.BlinkFontFaceSet.instance.check_Callback_1_(this, font);
  }

  @DomName('FontFaceSet.clear')
  @DocsEditable()
  @Experimental() // untriaged
  void clear() => _blink.BlinkFontFaceSet.instance.clear_Callback_0_(this);

  @DomName('FontFaceSet.delete')
  @DocsEditable()
  @Experimental() // untriaged
  bool delete(FontFace fontFace) =>
      _blink.BlinkFontFaceSet.instance.delete_Callback_1_(this, fontFace);

  void forEach(FontFaceSetForEachCallback callback, [Object thisArg]) {
    if (thisArg != null) {
      _blink.BlinkFontFaceSet.instance
          .forEach_Callback_2_(this, callback, thisArg);
      return;
    }
    _blink.BlinkFontFaceSet.instance.forEach_Callback_1_(this, callback);
    return;
  }

  @DomName('FontFaceSet.has')
  @DocsEditable()
  @Experimental() // untriaged
  bool has(FontFace fontFace) =>
      _blink.BlinkFontFaceSet.instance.has_Callback_1_(this, fontFace);
}
// Copyright (c) 2012, the Dart project authors.  Please see the AUTHORS file
// for details. All rights reserved. Use of this source code is governed by a
// BSD-style license that can be found in the LICENSE file.

// WARNING: Do not edit - generated code.

@DomName('FontFaceSetForEachCallback')
@Experimental() // untriaged
typedef void FontFaceSetForEachCallback(
    FontFace fontFace, FontFace fontFaceAgain, FontFaceSet set);
// Copyright (c) 2012, the Dart project authors.  Please see the AUTHORS file
// for details. All rights reserved. Use of this source code is governed by a
// BSD-style license that can be found in the LICENSE file.

// WARNING: Do not edit - generated code.

@DocsEditable()
@DomName('FontFaceSetLoadEvent')
@Experimental() // untriaged
class FontFaceSetLoadEvent extends Event {
  // To suppress missing implicit constructor warnings.
  factory FontFaceSetLoadEvent._() {
    throw new UnsupportedError("Not supported");
  }

  @Deprecated("Internal Use Only")
  external static Type get instanceRuntimeType;

  @Deprecated("Internal Use Only")
  FontFaceSetLoadEvent.internal_() : super.internal_();

  @DomName('FontFaceSetLoadEvent.fontfaces')
  @DocsEditable()
  @Experimental() // untriaged
  List<FontFace> get fontfaces =>
      (_blink.BlinkFontFaceSetLoadEvent.instance.fontfaces_Getter_(this));
}
// Copyright (c) 2012, the Dart project authors.  Please see the AUTHORS file
// for details. All rights reserved. Use of this source code is governed by a
// BSD-style license that can be found in the LICENSE file.

// WARNING: Do not edit - generated code.

@DocsEditable()
@DomName('FormData')
@SupportedBrowser(SupportedBrowser.CHROME)
@SupportedBrowser(SupportedBrowser.FIREFOX)
@SupportedBrowser(SupportedBrowser.IE, '10')
@SupportedBrowser(SupportedBrowser.SAFARI)
class FormData extends DartHtmlDomObject {
  // To suppress missing implicit constructor warnings.
  factory FormData._() {
    throw new UnsupportedError("Not supported");
  }

  @DomName('FormData.FormData')
  @DocsEditable()
  factory FormData([FormElement form]) => _create(form);

  @DocsEditable()
  static FormData _create(form) =>
      _blink.BlinkFormData.instance.constructorCallback_1_(form);

  @Deprecated("Internal Use Only")
  external static Type get instanceRuntimeType;

  @Deprecated("Internal Use Only")
  FormData.internal_() {}

  /// Checks if this type is supported on the current platform.
  static bool get supported => true;

  @DomName('FormData.append')
  @DocsEditable()
  void append(String name, String value) =>
      _blink.BlinkFormData.instance.append_Callback_2_(this, name, value);

  @DomName('FormData.appendBlob')
  @DocsEditable()
  void appendBlob(String name, Blob value, [String filename]) =>
      _blink.BlinkFormData.instance
          .append_Callback_3_(this, name, value, filename);

  @DomName('FormData.delete')
  @DocsEditable()
  @Experimental() // untriaged
  void delete(String name) =>
      _blink.BlinkFormData.instance.delete_Callback_1_(this, name);

  @DomName('FormData.get')
  @DocsEditable()
  @Experimental() // untriaged
  Object get(String name) =>
      (_blink.BlinkFormData.instance.get_Callback_1_(this, name));

  @DomName('FormData.getAll')
  @DocsEditable()
  @Experimental() // untriaged
  List<Object> getAll(String name) =>
      _blink.BlinkFormData.instance.getAll_Callback_1_(this, name);

  @DomName('FormData.has')
  @DocsEditable()
  @Experimental() // untriaged
  bool has(String name) =>
      _blink.BlinkFormData.instance.has_Callback_1_(this, name);

  void set(String name, value, [String filename]) {
    if ((value is Blob || value == null) &&
        (name is String || name == null) &&
        filename == null) {
      _blink.BlinkFormData.instance.set_Callback_2_(this, name, value);
      return;
    }
    if ((filename is String || filename == null) &&
        (value is Blob || value == null) &&
        (name is String || name == null)) {
      _blink.BlinkFormData.instance
          .set_Callback_3_(this, name, value, filename);
      return;
    }
    if ((value is String || value == null) &&
        (name is String || name == null) &&
        filename == null) {
      _blink.BlinkFormData.instance.set_Callback_2_(this, name, value);
      return;
    }
    throw new ArgumentError("Incorrect number or type of arguments");
  }
}
// Copyright (c) 2012, the Dart project authors.  Please see the AUTHORS file
// for details. All rights reserved. Use of this source code is governed by a
// BSD-style license that can be found in the LICENSE file.

// WARNING: Do not edit - generated code.

@DocsEditable()
@DomName('HTMLFormElement')
class FormElement extends HtmlElement {
  // To suppress missing implicit constructor warnings.
  factory FormElement._() {
    throw new UnsupportedError("Not supported");
  }

  @DomName('HTMLFormElement.HTMLFormElement')
  @DocsEditable()
  factory FormElement() => document.createElement("form");

  @Deprecated("Internal Use Only")
  external static Type get instanceRuntimeType;

  @Deprecated("Internal Use Only")
  FormElement.internal_() : super.internal_();

  /**
   * Constructor instantiated by the DOM when a custom element has been created.
   *
   * This can only be called by subclasses from their created constructor.
   */
  FormElement.created() : super.created();

  @DomName('HTMLFormElement.acceptCharset')
  @DocsEditable()
  String get acceptCharset =>
      _blink.BlinkHTMLFormElement.instance.acceptCharset_Getter_(this);

  @DomName('HTMLFormElement.acceptCharset')
  @DocsEditable()
  set acceptCharset(String value) =>
      _blink.BlinkHTMLFormElement.instance.acceptCharset_Setter_(this, value);

  @DomName('HTMLFormElement.action')
  @DocsEditable()
  String get action =>
      _blink.BlinkHTMLFormElement.instance.action_Getter_(this);

  @DomName('HTMLFormElement.action')
  @DocsEditable()
  set action(String value) =>
      _blink.BlinkHTMLFormElement.instance.action_Setter_(this, value);

  @DomName('HTMLFormElement.autocomplete')
  @DocsEditable()
  // http://www.whatwg.org/specs/web-apps/current-work/multipage/association-of-controls-and-forms.html#autofilling-form-controls:-the-autocomplete-attribute
  @Experimental()
  String get autocomplete =>
      _blink.BlinkHTMLFormElement.instance.autocomplete_Getter_(this);

  @DomName('HTMLFormElement.autocomplete')
  @DocsEditable()
  // http://www.whatwg.org/specs/web-apps/current-work/multipage/association-of-controls-and-forms.html#autofilling-form-controls:-the-autocomplete-attribute
  @Experimental()
  set autocomplete(String value) =>
      _blink.BlinkHTMLFormElement.instance.autocomplete_Setter_(this, value);

  @DomName('HTMLFormElement.encoding')
  @DocsEditable()
  String get encoding =>
      _blink.BlinkHTMLFormElement.instance.encoding_Getter_(this);

  @DomName('HTMLFormElement.encoding')
  @DocsEditable()
  set encoding(String value) =>
      _blink.BlinkHTMLFormElement.instance.encoding_Setter_(this, value);

  @DomName('HTMLFormElement.enctype')
  @DocsEditable()
  String get enctype =>
      _blink.BlinkHTMLFormElement.instance.enctype_Getter_(this);

  @DomName('HTMLFormElement.enctype')
  @DocsEditable()
  set enctype(String value) =>
      _blink.BlinkHTMLFormElement.instance.enctype_Setter_(this, value);

  @DomName('HTMLFormElement.length')
  @DocsEditable()
  int get length => _blink.BlinkHTMLFormElement.instance.length_Getter_(this);

  @DomName('HTMLFormElement.method')
  @DocsEditable()
  String get method =>
      _blink.BlinkHTMLFormElement.instance.method_Getter_(this);

  @DomName('HTMLFormElement.method')
  @DocsEditable()
  set method(String value) =>
      _blink.BlinkHTMLFormElement.instance.method_Setter_(this, value);

  @DomName('HTMLFormElement.name')
  @DocsEditable()
  String get name => _blink.BlinkHTMLFormElement.instance.name_Getter_(this);

  @DomName('HTMLFormElement.name')
  @DocsEditable()
  set name(String value) =>
      _blink.BlinkHTMLFormElement.instance.name_Setter_(this, value);

  @DomName('HTMLFormElement.noValidate')
  @DocsEditable()
  bool get noValidate =>
      _blink.BlinkHTMLFormElement.instance.noValidate_Getter_(this);

  @DomName('HTMLFormElement.noValidate')
  @DocsEditable()
  set noValidate(bool value) =>
      _blink.BlinkHTMLFormElement.instance.noValidate_Setter_(this, value);

  @DomName('HTMLFormElement.target')
  @DocsEditable()
  String get target =>
      _blink.BlinkHTMLFormElement.instance.target_Getter_(this);

  @DomName('HTMLFormElement.target')
  @DocsEditable()
  set target(String value) =>
      _blink.BlinkHTMLFormElement.instance.target_Setter_(this, value);

  @DomName('HTMLFormElement.__getter__')
  @DocsEditable()
  Object __getter__(String name) => (_blink.BlinkHTMLFormElement.instance
      .$__getter___Callback_1_(this, name));

  @DomName('HTMLFormElement.checkValidity')
  @DocsEditable()
  bool checkValidity() =>
      _blink.BlinkHTMLFormElement.instance.checkValidity_Callback_0_(this);

  @DomName('HTMLFormElement.item')
  @DocsEditable()
  @Experimental() // untriaged
  Element item(int index) =>
      _blink.BlinkHTMLFormElement.instance.item_Callback_1_(this, index);

  @DomName('HTMLFormElement.reportValidity')
  @DocsEditable()
  @Experimental() // untriaged
  bool reportValidity() =>
      _blink.BlinkHTMLFormElement.instance.reportValidity_Callback_0_(this);

  @DomName('HTMLFormElement.requestAutocomplete')
  @DocsEditable()
  // http://lists.whatwg.org/htdig.cgi/whatwg-whatwg.org/2012-October/037711.html
  @Experimental()
  void requestAutocomplete(Map details) =>
      _blink.BlinkHTMLFormElement.instance.requestAutocomplete_Callback_1_(
          this, convertDartToNative_Dictionary(details));

  @DomName('HTMLFormElement.reset')
  @DocsEditable()
  void reset() => _blink.BlinkHTMLFormElement.instance.reset_Callback_0_(this);

  @DomName('HTMLFormElement.submit')
  @DocsEditable()
  void submit() =>
      _blink.BlinkHTMLFormElement.instance.submit_Callback_0_(this);
}
// Copyright (c) 2012, the Dart project authors.  Please see the AUTHORS file
// for details. All rights reserved. Use of this source code is governed by a
// BSD-style license that can be found in the LICENSE file.

// WARNING: Do not edit - generated code.

@DomName('FrameRequestCallback')
@Experimental() // untriaged
typedef void FrameRequestCallback(num highResTime);
// Copyright (c) 2012, the Dart project authors.  Please see the AUTHORS file
// for details. All rights reserved. Use of this source code is governed by a
// BSD-style license that can be found in the LICENSE file.

// WARNING: Do not edit - generated code.

@DocsEditable()
@DomName('Gamepad')
// https://dvcs.w3.org/hg/gamepad/raw-file/default/gamepad.html#gamepad-interface
@Experimental()
class Gamepad extends DartHtmlDomObject {
  // To suppress missing implicit constructor warnings.
  factory Gamepad._() {
    throw new UnsupportedError("Not supported");
  }

  @Deprecated("Internal Use Only")
  external static Type get instanceRuntimeType;

  @Deprecated("Internal Use Only")
  Gamepad.internal_() {}

  @DomName('Gamepad.axes')
  @DocsEditable()
  List<num> get axes => _blink.BlinkGamepad.instance.axes_Getter_(this);

  @DomName('Gamepad.buttons')
  @DocsEditable()
  List<GamepadButton> get buttons =>
      (_blink.BlinkGamepad.instance.buttons_Getter_(this));

  @DomName('Gamepad.connected')
  @DocsEditable()
  @Experimental() // untriaged
  bool get connected => _blink.BlinkGamepad.instance.connected_Getter_(this);

  @DomName('Gamepad.id')
  @DocsEditable()
  String get id => _blink.BlinkGamepad.instance.id_Getter_(this);

  @DomName('Gamepad.index')
  @DocsEditable()
  int get index => _blink.BlinkGamepad.instance.index_Getter_(this);

  @DomName('Gamepad.mapping')
  @DocsEditable()
  @Experimental() // untriaged
  String get mapping => _blink.BlinkGamepad.instance.mapping_Getter_(this);

  @DomName('Gamepad.timestamp')
  @DocsEditable()
  int get timestamp => _blink.BlinkGamepad.instance.timestamp_Getter_(this);
}
// Copyright (c) 2012, the Dart project authors.  Please see the AUTHORS file
// for details. All rights reserved. Use of this source code is governed by a
// BSD-style license that can be found in the LICENSE file.

// WARNING: Do not edit - generated code.

@DocsEditable()
@DomName('GamepadButton')
@Experimental() // untriaged
class GamepadButton extends DartHtmlDomObject {
  // To suppress missing implicit constructor warnings.
  factory GamepadButton._() {
    throw new UnsupportedError("Not supported");
  }

  @Deprecated("Internal Use Only")
  external static Type get instanceRuntimeType;

  @Deprecated("Internal Use Only")
  GamepadButton.internal_() {}

  @DomName('GamepadButton.pressed')
  @DocsEditable()
  @Experimental() // untriaged
  bool get pressed => _blink.BlinkGamepadButton.instance.pressed_Getter_(this);

  @DomName('GamepadButton.value')
  @DocsEditable()
  @Experimental() // untriaged
  num get value => _blink.BlinkGamepadButton.instance.value_Getter_(this);
}
// Copyright (c) 2012, the Dart project authors.  Please see the AUTHORS file
// for details. All rights reserved. Use of this source code is governed by a
// BSD-style license that can be found in the LICENSE file.

// WARNING: Do not edit - generated code.

@DocsEditable()
@DomName('GamepadEvent')
@Experimental() // untriaged
class GamepadEvent extends Event {
  // To suppress missing implicit constructor warnings.
  factory GamepadEvent._() {
    throw new UnsupportedError("Not supported");
  }

  @DomName('GamepadEvent.GamepadEvent')
  @DocsEditable()
  factory GamepadEvent(String type, [Map eventInitDict]) {
    if (eventInitDict != null) {
      var eventInitDict_1 = convertDartToNative_Dictionary(eventInitDict);
      return _blink.BlinkGamepadEvent.instance
          .constructorCallback_2_(type, eventInitDict_1);
    }
    return _blink.BlinkGamepadEvent.instance.constructorCallback_1_(type);
  }

  @Deprecated("Internal Use Only")
  external static Type get instanceRuntimeType;

  @Deprecated("Internal Use Only")
  GamepadEvent.internal_() : super.internal_();

  @DomName('GamepadEvent.gamepad')
  @DocsEditable()
  @Experimental() // untriaged
  Gamepad get gamepad =>
      _blink.BlinkGamepadEvent.instance.gamepad_Getter_(this);
}
// Copyright (c) 2012, the Dart project authors.  Please see the AUTHORS file
// for details. All rights reserved. Use of this source code is governed by a
// BSD-style license that can be found in the LICENSE file.

// WARNING: Do not edit - generated code.

@DocsEditable()
@DomName('Geofencing')
@Experimental() // untriaged
class Geofencing extends DartHtmlDomObject {
  // To suppress missing implicit constructor warnings.
  factory Geofencing._() {
    throw new UnsupportedError("Not supported");
  }

  @Deprecated("Internal Use Only")
  external static Type get instanceRuntimeType;

  @Deprecated("Internal Use Only")
  Geofencing.internal_() {}

  @DomName('Geofencing.getRegisteredRegions')
  @DocsEditable()
  @Experimental() // untriaged
  Future getRegisteredRegions() => convertNativePromiseToDartFuture(
      _blink.BlinkGeofencing.instance.getRegisteredRegions_Callback_0_(this));

  @DomName('Geofencing.registerRegion')
  @DocsEditable()
  @Experimental() // untriaged
  Future registerRegion(GeofencingRegion region) =>
      convertNativePromiseToDartFuture(_blink.BlinkGeofencing.instance
          .registerRegion_Callback_1_(this, region));

  @DomName('Geofencing.unregisterRegion')
  @DocsEditable()
  @Experimental() // untriaged
  Future unregisterRegion(String regionId) =>
      convertNativePromiseToDartFuture(_blink.BlinkGeofencing.instance
          .unregisterRegion_Callback_1_(this, regionId));
}
// Copyright (c) 2012, the Dart project authors.  Please see the AUTHORS file
// for details. All rights reserved. Use of this source code is governed by a
// BSD-style license that can be found in the LICENSE file.

// WARNING: Do not edit - generated code.

@DocsEditable()
@DomName('GeofencingEvent')
@Experimental() // untriaged
class GeofencingEvent extends Event {
  // To suppress missing implicit constructor warnings.
  factory GeofencingEvent._() {
    throw new UnsupportedError("Not supported");
  }

  @Deprecated("Internal Use Only")
  external static Type get instanceRuntimeType;

  @Deprecated("Internal Use Only")
  GeofencingEvent.internal_() : super.internal_();

  @DomName('GeofencingEvent.id')
  @DocsEditable()
  @Experimental() // untriaged
  String get id => _blink.BlinkGeofencingEvent.instance.id_Getter_(this);

  @DomName('GeofencingEvent.region')
  @DocsEditable()
  @Experimental() // untriaged
  GeofencingRegion get region =>
      _blink.BlinkGeofencingEvent.instance.region_Getter_(this);
}
// Copyright (c) 2012, the Dart project authors.  Please see the AUTHORS file
// for details. All rights reserved. Use of this source code is governed by a
// BSD-style license that can be found in the LICENSE file.

// WARNING: Do not edit - generated code.

@DocsEditable()
@DomName('GeofencingRegion')
@Experimental() // untriaged
class GeofencingRegion extends DartHtmlDomObject {
  // To suppress missing implicit constructor warnings.
  factory GeofencingRegion._() {
    throw new UnsupportedError("Not supported");
  }

  @Deprecated("Internal Use Only")
  external static Type get instanceRuntimeType;

  @Deprecated("Internal Use Only")
  GeofencingRegion.internal_() {}

  @DomName('GeofencingRegion.id')
  @DocsEditable()
  @Experimental() // untriaged
  String get id => _blink.BlinkGeofencingRegion.instance.id_Getter_(this);
}
// Copyright (c) 2012, the Dart project authors.  Please see the AUTHORS file
// for details. All rights reserved. Use of this source code is governed by a
// BSD-style license that can be found in the LICENSE file.

@DocsEditable()
@DomName('Geolocation')
@Unstable()
class Geolocation extends DartHtmlDomObject {
  @DomName('Geolocation.getCurrentPosition')
  Future<Geoposition> getCurrentPosition(
      {bool enableHighAccuracy, Duration timeout, Duration maximumAge}) {
    var options = {};
    if (enableHighAccuracy != null) {
      options['enableHighAccuracy'] = enableHighAccuracy;
    }
    if (timeout != null) {
      options['timeout'] = timeout.inMilliseconds;
    }
    if (maximumAge != null) {
      options['maximumAge'] = maximumAge.inMilliseconds;
    }
    var completer = new Completer<Geoposition>();
    try {
      _getCurrentPosition((position) {
        completer.complete(_ensurePosition(position));
      }, (error) {
        completer.completeError(error);
      }, options);
    } catch (e, stacktrace) {
      completer.completeError(e, stacktrace);
    }
    return completer.future;
  }

  @DomName('Geolocation.watchPosition')
  Stream<Geoposition> watchPosition(
      {bool enableHighAccuracy, Duration timeout, Duration maximumAge}) {
    var options = {};
    if (enableHighAccuracy != null) {
      options['enableHighAccuracy'] = enableHighAccuracy;
    }
    if (timeout != null) {
      options['timeout'] = timeout.inMilliseconds;
    }
    if (maximumAge != null) {
      options['maximumAge'] = maximumAge.inMilliseconds;
    }

    int watchId;
    // TODO(jacobr): it seems like a bug that we have to specifiy the static
    // type here for controller.stream to have the right type.
    // dartbug.com/26278
    StreamController<Geoposition> controller;
    controller = new StreamController<Geoposition>(
        sync: true,
        onListen: () {
          assert(watchId == null);
          watchId = _watchPosition((position) {
            controller.add(_ensurePosition(position));
          }, (error) {
            controller.addError(error);
          }, options);
        },
        onCancel: () {
          assert(watchId != null);
          _clearWatch(watchId);
        });

    return controller.stream;
  }

  Geoposition _ensurePosition(domPosition) {
    return domPosition;
  }

  // To suppress missing implicit constructor warnings.
  factory Geolocation._() {
    throw new UnsupportedError("Not supported");
  }

  @Deprecated("Internal Use Only")
  external static Type get instanceRuntimeType;

  @Deprecated("Internal Use Only")
  Geolocation.internal_() {}

  @DomName('Geolocation.clearWatch')
  @DocsEditable()
  void _clearWatch(int watchID) =>
      _blink.BlinkGeolocation.instance.clearWatch_Callback_1_(this, watchID);

  void _getCurrentPosition(_PositionCallback successCallback,
      [_PositionErrorCallback errorCallback, Map options]) {
    if (options != null) {
      _blink.BlinkGeolocation.instance.getCurrentPosition_Callback_3_(
          this,
          successCallback,
          errorCallback,
          convertDartToNative_Dictionary(options));
      return;
    }
    if (errorCallback != null) {
      _blink.BlinkGeolocation.instance
          .getCurrentPosition_Callback_2_(this, successCallback, errorCallback);
      return;
    }
    _blink.BlinkGeolocation.instance
        .getCurrentPosition_Callback_1_(this, successCallback);
    return;
  }

  int _watchPosition(_PositionCallback successCallback,
      [_PositionErrorCallback errorCallback, Map options]) {
    if (options != null) {
      return _blink.BlinkGeolocation.instance.watchPosition_Callback_3_(
          this,
          successCallback,
          errorCallback,
          convertDartToNative_Dictionary(options));
    }
    if (errorCallback != null) {
      return _blink.BlinkGeolocation.instance
          .watchPosition_Callback_2_(this, successCallback, errorCallback);
    }
    return _blink.BlinkGeolocation.instance
        .watchPosition_Callback_1_(this, successCallback);
  }
}

// Copyright (c) 2012, the Dart project authors.  Please see the AUTHORS file
// for details. All rights reserved. Use of this source code is governed by a
// BSD-style license that can be found in the LICENSE file.

// WARNING: Do not edit - generated code.

@DocsEditable()
@DomName('Geoposition')
@Unstable()
class Geoposition extends DartHtmlDomObject {
  // To suppress missing implicit constructor warnings.
  factory Geoposition._() {
    throw new UnsupportedError("Not supported");
  }

  @Deprecated("Internal Use Only")
  external static Type get instanceRuntimeType;

  @Deprecated("Internal Use Only")
  Geoposition.internal_() {}

  @DomName('Geoposition.coords')
  @DocsEditable()
  Coordinates get coords =>
      _blink.BlinkGeoposition.instance.coords_Getter_(this);

  @DomName('Geoposition.timestamp')
  @DocsEditable()
  int get timestamp => _blink.BlinkGeoposition.instance.timestamp_Getter_(this);
}
// Copyright (c) 2012, the Dart project authors.  Please see the AUTHORS file
// for details. All rights reserved. Use of this source code is governed by a
// BSD-style license that can be found in the LICENSE file.

// We implement EventTarget and have stubs for its methods because it's tricky to
// convince the scripts to make our instance methods abstract, and the bodies that
// get generated require `this` to be an EventTarget.
@DocsEditable()
@DomName('GlobalEventHandlers')
@Experimental() // untriaged
abstract class GlobalEventHandlers implements EventTarget {
  void addEventListener(String type, dynamic listener(Event event),
      [bool useCapture]);
  bool dispatchEvent(Event event);
  void removeEventListener(String type, dynamic listener(Event event),
      [bool useCapture]);
  Events get on;

  // To suppress missing implicit constructor warnings.
  factory GlobalEventHandlers._() {
    throw new UnsupportedError("Not supported");
  }

  @DomName('GlobalEventHandlers.abortEvent')
  @DocsEditable()
  @Experimental() // untriaged
  static const EventStreamProvider<Event> abortEvent =
      const EventStreamProvider<Event>('abort');

  @DomName('GlobalEventHandlers.blurEvent')
  @DocsEditable()
  @Experimental() // untriaged
  static const EventStreamProvider<Event> blurEvent =
      const EventStreamProvider<Event>('blur');

  @DomName('GlobalEventHandlers.canplayEvent')
  @DocsEditable()
  @Experimental() // untriaged
  static const EventStreamProvider<Event> canPlayEvent =
      const EventStreamProvider<Event>('canplay');

  @DomName('GlobalEventHandlers.canplaythroughEvent')
  @DocsEditable()
  @Experimental() // untriaged
  static const EventStreamProvider<Event> canPlayThroughEvent =
      const EventStreamProvider<Event>('canplaythrough');

  @DomName('GlobalEventHandlers.changeEvent')
  @DocsEditable()
  @Experimental() // untriaged
  static const EventStreamProvider<Event> changeEvent =
      const EventStreamProvider<Event>('change');

  @DomName('GlobalEventHandlers.clickEvent')
  @DocsEditable()
  @Experimental() // untriaged
  static const EventStreamProvider<MouseEvent> clickEvent =
      const EventStreamProvider<MouseEvent>('click');

  @DomName('GlobalEventHandlers.contextmenuEvent')
  @DocsEditable()
  @Experimental() // untriaged
  static const EventStreamProvider<MouseEvent> contextMenuEvent =
      const EventStreamProvider<MouseEvent>('contextmenu');

  @DomName('GlobalEventHandlers.dblclickEvent')
  @DocsEditable()
  @Experimental() // untriaged
  static const EventStreamProvider<Event> doubleClickEvent =
      const EventStreamProvider<Event>('dblclick');

  @DomName('GlobalEventHandlers.dragEvent')
  @DocsEditable()
  @Experimental() // untriaged
  static const EventStreamProvider<MouseEvent> dragEvent =
      const EventStreamProvider<MouseEvent>('drag');

  @DomName('GlobalEventHandlers.dragendEvent')
  @DocsEditable()
  @Experimental() // untriaged
  static const EventStreamProvider<MouseEvent> dragEndEvent =
      const EventStreamProvider<MouseEvent>('dragend');

  @DomName('GlobalEventHandlers.dragenterEvent')
  @DocsEditable()
  @Experimental() // untriaged
  static const EventStreamProvider<MouseEvent> dragEnterEvent =
      const EventStreamProvider<MouseEvent>('dragenter');

  @DomName('GlobalEventHandlers.dragleaveEvent')
  @DocsEditable()
  @Experimental() // untriaged
  static const EventStreamProvider<MouseEvent> dragLeaveEvent =
      const EventStreamProvider<MouseEvent>('dragleave');

  @DomName('GlobalEventHandlers.dragoverEvent')
  @DocsEditable()
  @Experimental() // untriaged
  static const EventStreamProvider<MouseEvent> dragOverEvent =
      const EventStreamProvider<MouseEvent>('dragover');

  @DomName('GlobalEventHandlers.dragstartEvent')
  @DocsEditable()
  @Experimental() // untriaged
  static const EventStreamProvider<MouseEvent> dragStartEvent =
      const EventStreamProvider<MouseEvent>('dragstart');

  @DomName('GlobalEventHandlers.dropEvent')
  @DocsEditable()
  @Experimental() // untriaged
  static const EventStreamProvider<MouseEvent> dropEvent =
      const EventStreamProvider<MouseEvent>('drop');

  @DomName('GlobalEventHandlers.durationchangeEvent')
  @DocsEditable()
  @Experimental() // untriaged
  static const EventStreamProvider<Event> durationChangeEvent =
      const EventStreamProvider<Event>('durationchange');

  @DomName('GlobalEventHandlers.emptiedEvent')
  @DocsEditable()
  @Experimental() // untriaged
  static const EventStreamProvider<Event> emptiedEvent =
      const EventStreamProvider<Event>('emptied');

  @DomName('GlobalEventHandlers.endedEvent')
  @DocsEditable()
  @Experimental() // untriaged
  static const EventStreamProvider<Event> endedEvent =
      const EventStreamProvider<Event>('ended');

  @DomName('GlobalEventHandlers.errorEvent')
  @DocsEditable()
  @Experimental() // untriaged
  static const EventStreamProvider<Event> errorEvent =
      const EventStreamProvider<Event>('error');

  @DomName('GlobalEventHandlers.focusEvent')
  @DocsEditable()
  @Experimental() // untriaged
  static const EventStreamProvider<Event> focusEvent =
      const EventStreamProvider<Event>('focus');

  @DomName('GlobalEventHandlers.inputEvent')
  @DocsEditable()
  @Experimental() // untriaged
  static const EventStreamProvider<Event> inputEvent =
      const EventStreamProvider<Event>('input');

  @DomName('GlobalEventHandlers.invalidEvent')
  @DocsEditable()
  @Experimental() // untriaged
  static const EventStreamProvider<Event> invalidEvent =
      const EventStreamProvider<Event>('invalid');

  @DomName('GlobalEventHandlers.keydownEvent')
  @DocsEditable()
  @Experimental() // untriaged
  static const EventStreamProvider<KeyboardEvent> keyDownEvent =
      const EventStreamProvider<KeyboardEvent>('keydown');

  @DomName('GlobalEventHandlers.keypressEvent')
  @DocsEditable()
  @Experimental() // untriaged
  static const EventStreamProvider<KeyboardEvent> keyPressEvent =
      const EventStreamProvider<KeyboardEvent>('keypress');

  @DomName('GlobalEventHandlers.keyupEvent')
  @DocsEditable()
  @Experimental() // untriaged
  static const EventStreamProvider<KeyboardEvent> keyUpEvent =
      const EventStreamProvider<KeyboardEvent>('keyup');

  @DomName('GlobalEventHandlers.loadEvent')
  @DocsEditable()
  @Experimental() // untriaged
  static const EventStreamProvider<Event> loadEvent =
      const EventStreamProvider<Event>('load');

  @DomName('GlobalEventHandlers.loadeddataEvent')
  @DocsEditable()
  @Experimental() // untriaged
  static const EventStreamProvider<Event> loadedDataEvent =
      const EventStreamProvider<Event>('loadeddata');

  @DomName('GlobalEventHandlers.loadedmetadataEvent')
  @DocsEditable()
  @Experimental() // untriaged
  static const EventStreamProvider<Event> loadedMetadataEvent =
      const EventStreamProvider<Event>('loadedmetadata');

  @DomName('GlobalEventHandlers.mousedownEvent')
  @DocsEditable()
  @Experimental() // untriaged
  static const EventStreamProvider<MouseEvent> mouseDownEvent =
      const EventStreamProvider<MouseEvent>('mousedown');

  @DomName('GlobalEventHandlers.mouseenterEvent')
  @DocsEditable()
  @Experimental() // untriaged
  static const EventStreamProvider<MouseEvent> mouseEnterEvent =
      const EventStreamProvider<MouseEvent>('mouseenter');

  @DomName('GlobalEventHandlers.mouseleaveEvent')
  @DocsEditable()
  @Experimental() // untriaged
  static const EventStreamProvider<MouseEvent> mouseLeaveEvent =
      const EventStreamProvider<MouseEvent>('mouseleave');

  @DomName('GlobalEventHandlers.mousemoveEvent')
  @DocsEditable()
  @Experimental() // untriaged
  static const EventStreamProvider<MouseEvent> mouseMoveEvent =
      const EventStreamProvider<MouseEvent>('mousemove');

  @DomName('GlobalEventHandlers.mouseoutEvent')
  @DocsEditable()
  @Experimental() // untriaged
  static const EventStreamProvider<MouseEvent> mouseOutEvent =
      const EventStreamProvider<MouseEvent>('mouseout');

  @DomName('GlobalEventHandlers.mouseoverEvent')
  @DocsEditable()
  @Experimental() // untriaged
  static const EventStreamProvider<MouseEvent> mouseOverEvent =
      const EventStreamProvider<MouseEvent>('mouseover');

  @DomName('GlobalEventHandlers.mouseupEvent')
  @DocsEditable()
  @Experimental() // untriaged
  static const EventStreamProvider<MouseEvent> mouseUpEvent =
      const EventStreamProvider<MouseEvent>('mouseup');

  @DomName('GlobalEventHandlers.mousewheelEvent')
  @DocsEditable()
  @Experimental() // untriaged
  static const EventStreamProvider<WheelEvent> mouseWheelEvent =
      const EventStreamProvider<WheelEvent>('mousewheel');

  @DomName('GlobalEventHandlers.pauseEvent')
  @DocsEditable()
  @Experimental() // untriaged
  static const EventStreamProvider<Event> pauseEvent =
      const EventStreamProvider<Event>('pause');

  @DomName('GlobalEventHandlers.playEvent')
  @DocsEditable()
  @Experimental() // untriaged
  static const EventStreamProvider<Event> playEvent =
      const EventStreamProvider<Event>('play');

  @DomName('GlobalEventHandlers.playingEvent')
  @DocsEditable()
  @Experimental() // untriaged
  static const EventStreamProvider<Event> playingEvent =
      const EventStreamProvider<Event>('playing');

  @DomName('GlobalEventHandlers.ratechangeEvent')
  @DocsEditable()
  @Experimental() // untriaged
  static const EventStreamProvider<Event> rateChangeEvent =
      const EventStreamProvider<Event>('ratechange');

  @DomName('GlobalEventHandlers.resetEvent')
  @DocsEditable()
  @Experimental() // untriaged
  static const EventStreamProvider<Event> resetEvent =
      const EventStreamProvider<Event>('reset');

  @DomName('GlobalEventHandlers.resizeEvent')
  @DocsEditable()
  @Experimental() // untriaged
  static const EventStreamProvider<Event> resizeEvent =
      const EventStreamProvider<Event>('resize');

  @DomName('GlobalEventHandlers.scrollEvent')
  @DocsEditable()
  @Experimental() // untriaged
  static const EventStreamProvider<Event> scrollEvent =
      const EventStreamProvider<Event>('scroll');

  @DomName('GlobalEventHandlers.seekedEvent')
  @DocsEditable()
  @Experimental() // untriaged
  static const EventStreamProvider<Event> seekedEvent =
      const EventStreamProvider<Event>('seeked');

  @DomName('GlobalEventHandlers.seekingEvent')
  @DocsEditable()
  @Experimental() // untriaged
  static const EventStreamProvider<Event> seekingEvent =
      const EventStreamProvider<Event>('seeking');

  @DomName('GlobalEventHandlers.selectEvent')
  @DocsEditable()
  @Experimental() // untriaged
  static const EventStreamProvider<Event> selectEvent =
      const EventStreamProvider<Event>('select');

  @DomName('GlobalEventHandlers.stalledEvent')
  @DocsEditable()
  @Experimental() // untriaged
  static const EventStreamProvider<Event> stalledEvent =
      const EventStreamProvider<Event>('stalled');

  @DomName('GlobalEventHandlers.submitEvent')
  @DocsEditable()
  @Experimental() // untriaged
  static const EventStreamProvider<Event> submitEvent =
      const EventStreamProvider<Event>('submit');

  @DomName('GlobalEventHandlers.suspendEvent')
  @DocsEditable()
  @Experimental() // untriaged
  static const EventStreamProvider<Event> suspendEvent =
      const EventStreamProvider<Event>('suspend');

  @DomName('GlobalEventHandlers.timeupdateEvent')
  @DocsEditable()
  @Experimental() // untriaged
  static const EventStreamProvider<Event> timeUpdateEvent =
      const EventStreamProvider<Event>('timeupdate');

  @DomName('GlobalEventHandlers.volumechangeEvent')
  @DocsEditable()
  @Experimental() // untriaged
  static const EventStreamProvider<Event> volumeChangeEvent =
      const EventStreamProvider<Event>('volumechange');

  @DomName('GlobalEventHandlers.waitingEvent')
  @DocsEditable()
  @Experimental() // untriaged
  static const EventStreamProvider<Event> waitingEvent =
      const EventStreamProvider<Event>('waiting');

  @DomName('GlobalEventHandlers.onabort')
  @DocsEditable()
  @Experimental() // untriaged
  Stream<Event> get onAbort => abortEvent.forTarget(this);

  @DomName('GlobalEventHandlers.onblur')
  @DocsEditable()
  @Experimental() // untriaged
  Stream<Event> get onBlur => blurEvent.forTarget(this);

  @DomName('GlobalEventHandlers.oncanplay')
  @DocsEditable()
  @Experimental() // untriaged
  Stream<Event> get onCanPlay => canPlayEvent.forTarget(this);

  @DomName('GlobalEventHandlers.oncanplaythrough')
  @DocsEditable()
  @Experimental() // untriaged
  Stream<Event> get onCanPlayThrough => canPlayThroughEvent.forTarget(this);

  @DomName('GlobalEventHandlers.onchange')
  @DocsEditable()
  @Experimental() // untriaged
  Stream<Event> get onChange => changeEvent.forTarget(this);

  @DomName('GlobalEventHandlers.onclick')
  @DocsEditable()
  @Experimental() // untriaged
  Stream<MouseEvent> get onClick => clickEvent.forTarget(this);

  @DomName('GlobalEventHandlers.oncontextmenu')
  @DocsEditable()
  @Experimental() // untriaged
  Stream<MouseEvent> get onContextMenu => contextMenuEvent.forTarget(this);

  @DomName('GlobalEventHandlers.ondblclick')
  @DocsEditable()
  @Experimental() // untriaged
  Stream<Event> get onDoubleClick => doubleClickEvent.forTarget(this);

  @DomName('GlobalEventHandlers.ondrag')
  @DocsEditable()
  @Experimental() // untriaged
  Stream<MouseEvent> get onDrag => dragEvent.forTarget(this);

  @DomName('GlobalEventHandlers.ondragend')
  @DocsEditable()
  @Experimental() // untriaged
  Stream<MouseEvent> get onDragEnd => dragEndEvent.forTarget(this);

  @DomName('GlobalEventHandlers.ondragenter')
  @DocsEditable()
  @Experimental() // untriaged
  Stream<MouseEvent> get onDragEnter => dragEnterEvent.forTarget(this);

  @DomName('GlobalEventHandlers.ondragleave')
  @DocsEditable()
  @Experimental() // untriaged
  Stream<MouseEvent> get onDragLeave => dragLeaveEvent.forTarget(this);

  @DomName('GlobalEventHandlers.ondragover')
  @DocsEditable()
  @Experimental() // untriaged
  Stream<MouseEvent> get onDragOver => dragOverEvent.forTarget(this);

  @DomName('GlobalEventHandlers.ondragstart')
  @DocsEditable()
  @Experimental() // untriaged
  Stream<MouseEvent> get onDragStart => dragStartEvent.forTarget(this);

  @DomName('GlobalEventHandlers.ondrop')
  @DocsEditable()
  @Experimental() // untriaged
  Stream<MouseEvent> get onDrop => dropEvent.forTarget(this);

  @DomName('GlobalEventHandlers.ondurationchange')
  @DocsEditable()
  @Experimental() // untriaged
  Stream<Event> get onDurationChange => durationChangeEvent.forTarget(this);

  @DomName('GlobalEventHandlers.onemptied')
  @DocsEditable()
  @Experimental() // untriaged
  Stream<Event> get onEmptied => emptiedEvent.forTarget(this);

  @DomName('GlobalEventHandlers.onended')
  @DocsEditable()
  @Experimental() // untriaged
  Stream<Event> get onEnded => endedEvent.forTarget(this);

  @DomName('GlobalEventHandlers.onerror')
  @DocsEditable()
  @Experimental() // untriaged
  Stream<Event> get onError => errorEvent.forTarget(this);

  @DomName('GlobalEventHandlers.onfocus')
  @DocsEditable()
  @Experimental() // untriaged
  Stream<Event> get onFocus => focusEvent.forTarget(this);

  @DomName('GlobalEventHandlers.oninput')
  @DocsEditable()
  @Experimental() // untriaged
  Stream<Event> get onInput => inputEvent.forTarget(this);

  @DomName('GlobalEventHandlers.oninvalid')
  @DocsEditable()
  @Experimental() // untriaged
  Stream<Event> get onInvalid => invalidEvent.forTarget(this);

  @DomName('GlobalEventHandlers.onkeydown')
  @DocsEditable()
  @Experimental() // untriaged
  Stream<KeyboardEvent> get onKeyDown => keyDownEvent.forTarget(this);

  @DomName('GlobalEventHandlers.onkeypress')
  @DocsEditable()
  @Experimental() // untriaged
  Stream<KeyboardEvent> get onKeyPress => keyPressEvent.forTarget(this);

  @DomName('GlobalEventHandlers.onkeyup')
  @DocsEditable()
  @Experimental() // untriaged
  Stream<KeyboardEvent> get onKeyUp => keyUpEvent.forTarget(this);

  @DomName('GlobalEventHandlers.onload')
  @DocsEditable()
  @Experimental() // untriaged
  Stream<Event> get onLoad => loadEvent.forTarget(this);

  @DomName('GlobalEventHandlers.onloadeddata')
  @DocsEditable()
  @Experimental() // untriaged
  Stream<Event> get onLoadedData => loadedDataEvent.forTarget(this);

  @DomName('GlobalEventHandlers.onloadedmetadata')
  @DocsEditable()
  @Experimental() // untriaged
  Stream<Event> get onLoadedMetadata => loadedMetadataEvent.forTarget(this);

  @DomName('GlobalEventHandlers.onmousedown')
  @DocsEditable()
  @Experimental() // untriaged
  Stream<MouseEvent> get onMouseDown => mouseDownEvent.forTarget(this);

  @DomName('GlobalEventHandlers.onmouseenter')
  @DocsEditable()
  @Experimental() // untriaged
  Stream<MouseEvent> get onMouseEnter => mouseEnterEvent.forTarget(this);

  @DomName('GlobalEventHandlers.onmouseleave')
  @DocsEditable()
  @Experimental() // untriaged
  Stream<MouseEvent> get onMouseLeave => mouseLeaveEvent.forTarget(this);

  @DomName('GlobalEventHandlers.onmousemove')
  @DocsEditable()
  @Experimental() // untriaged
  Stream<MouseEvent> get onMouseMove => mouseMoveEvent.forTarget(this);

  @DomName('GlobalEventHandlers.onmouseout')
  @DocsEditable()
  @Experimental() // untriaged
  Stream<MouseEvent> get onMouseOut => mouseOutEvent.forTarget(this);

  @DomName('GlobalEventHandlers.onmouseover')
  @DocsEditable()
  @Experimental() // untriaged
  Stream<MouseEvent> get onMouseOver => mouseOverEvent.forTarget(this);

  @DomName('GlobalEventHandlers.onmouseup')
  @DocsEditable()
  @Experimental() // untriaged
  Stream<MouseEvent> get onMouseUp => mouseUpEvent.forTarget(this);

  @DomName('GlobalEventHandlers.onmousewheel')
  @DocsEditable()
  @Experimental() // untriaged
  Stream<WheelEvent> get onMouseWheel => mouseWheelEvent.forTarget(this);

  @DomName('GlobalEventHandlers.onpause')
  @DocsEditable()
  @Experimental() // untriaged
  Stream<Event> get onPause => pauseEvent.forTarget(this);

  @DomName('GlobalEventHandlers.onplay')
  @DocsEditable()
  @Experimental() // untriaged
  Stream<Event> get onPlay => playEvent.forTarget(this);

  @DomName('GlobalEventHandlers.onplaying')
  @DocsEditable()
  @Experimental() // untriaged
  Stream<Event> get onPlaying => playingEvent.forTarget(this);

  @DomName('GlobalEventHandlers.onratechange')
  @DocsEditable()
  @Experimental() // untriaged
  Stream<Event> get onRateChange => rateChangeEvent.forTarget(this);

  @DomName('GlobalEventHandlers.onreset')
  @DocsEditable()
  @Experimental() // untriaged
  Stream<Event> get onReset => resetEvent.forTarget(this);

  @DomName('GlobalEventHandlers.onresize')
  @DocsEditable()
  @Experimental() // untriaged
  Stream<Event> get onResize => resizeEvent.forTarget(this);

  @DomName('GlobalEventHandlers.onscroll')
  @DocsEditable()
  @Experimental() // untriaged
  Stream<Event> get onScroll => scrollEvent.forTarget(this);

  @DomName('GlobalEventHandlers.onseeked')
  @DocsEditable()
  @Experimental() // untriaged
  Stream<Event> get onSeeked => seekedEvent.forTarget(this);

  @DomName('GlobalEventHandlers.onseeking')
  @DocsEditable()
  @Experimental() // untriaged
  Stream<Event> get onSeeking => seekingEvent.forTarget(this);

  @DomName('GlobalEventHandlers.onselect')
  @DocsEditable()
  @Experimental() // untriaged
  Stream<Event> get onSelect => selectEvent.forTarget(this);

  @DomName('GlobalEventHandlers.onstalled')
  @DocsEditable()
  @Experimental() // untriaged
  Stream<Event> get onStalled => stalledEvent.forTarget(this);

  @DomName('GlobalEventHandlers.onsubmit')
  @DocsEditable()
  @Experimental() // untriaged
  Stream<Event> get onSubmit => submitEvent.forTarget(this);

  @DomName('GlobalEventHandlers.onsuspend')
  @DocsEditable()
  @Experimental() // untriaged
  Stream<Event> get onSuspend => suspendEvent.forTarget(this);

  @DomName('GlobalEventHandlers.ontimeupdate')
  @DocsEditable()
  @Experimental() // untriaged
  Stream<Event> get onTimeUpdate => timeUpdateEvent.forTarget(this);

  @DomName('GlobalEventHandlers.onvolumechange')
  @DocsEditable()
  @Experimental() // untriaged
  Stream<Event> get onVolumeChange => volumeChangeEvent.forTarget(this);

  @DomName('GlobalEventHandlers.onwaiting')
  @DocsEditable()
  @Experimental() // untriaged
  Stream<Event> get onWaiting => waitingEvent.forTarget(this);
}
// Copyright (c) 2012, the Dart project authors.  Please see the AUTHORS file
// for details. All rights reserved. Use of this source code is governed by a
// BSD-style license that can be found in the LICENSE file.

// WARNING: Do not edit - generated code.

@DocsEditable()
/**
 * An `<hr>` tag.
 */
@DomName('HTMLHRElement')
class HRElement extends HtmlElement {
  // To suppress missing implicit constructor warnings.
  factory HRElement._() {
    throw new UnsupportedError("Not supported");
  }

  @DomName('HTMLHRElement.HTMLHRElement')
  @DocsEditable()
  factory HRElement() => document.createElement("hr");

  @Deprecated("Internal Use Only")
  external static Type get instanceRuntimeType;

  @Deprecated("Internal Use Only")
  HRElement.internal_() : super.internal_();

  /**
   * Constructor instantiated by the DOM when a custom element has been created.
   *
   * This can only be called by subclasses from their created constructor.
   */
  HRElement.created() : super.created();

  @DomName('HTMLHRElement.color')
  @DocsEditable()
  @Experimental() // untriaged
  String get color => _blink.BlinkHTMLHRElement.instance.color_Getter_(this);

  @DomName('HTMLHRElement.color')
  @DocsEditable()
  @Experimental() // untriaged
  set color(String value) =>
      _blink.BlinkHTMLHRElement.instance.color_Setter_(this, value);
}
// Copyright (c) 2013, the Dart project authors.  Please see the AUTHORS file
// for details. All rights reserved. Use of this source code is governed by a
// BSD-style license that can be found in the LICENSE file.

// WARNING: Do not edit - generated code.

@DomName('HashChangeEvent')
@SupportedBrowser(SupportedBrowser.CHROME)
@SupportedBrowser(SupportedBrowser.FIREFOX)
@SupportedBrowser(SupportedBrowser.SAFARI)
@Unstable()
class HashChangeEvent extends Event {
  factory HashChangeEvent(String type,
      {bool canBubble: true,
      bool cancelable: true,
      String oldUrl,
      String newUrl}) {
    // TODO(alanknight): This is required while we're on Dartium 39, but will need
    // to look like dart2js with later versions when initHashChange is removed.
    var event = document._createEvent("HashChangeEvent");
    event._initHashChangeEvent(type, canBubble, cancelable, oldUrl, newUrl);
    return event;
  }

  @DomName('HashChangeEvent.HashChangeEvent')
  @DocsEditable()
  factory HashChangeEvent._(String type, [Map eventInitDict]) {
    if (eventInitDict != null) {
      var eventInitDict_1 = convertDartToNative_Dictionary(eventInitDict);
      return _blink.BlinkHashChangeEvent.instance
          .constructorCallback_2_(type, eventInitDict_1);
    }
    return _blink.BlinkHashChangeEvent.instance.constructorCallback_1_(type);
  }

  @Deprecated("Internal Use Only")
  external static Type get instanceRuntimeType;

  @Deprecated("Internal Use Only")
  HashChangeEvent.internal_() : super.internal_();

  /// Checks if this type is supported on the current platform.
  static bool get supported => true;

  @DomName('HashChangeEvent.newURL')
  @DocsEditable()
  String get newUrl =>
      _blink.BlinkHashChangeEvent.instance.newURL_Getter_(this);

  @DomName('HashChangeEvent.oldURL')
  @DocsEditable()
  String get oldUrl =>
      _blink.BlinkHashChangeEvent.instance.oldURL_Getter_(this);

  @DomName('HashChangeEvent.initHashChangeEvent')
  @DocsEditable()
  void _initHashChangeEvent(String type, bool canBubble, bool cancelable,
          String oldURL, String newURL) =>
      _blink.BlinkHashChangeEvent.instance.initHashChangeEvent_Callback_5_(
          this, type, canBubble, cancelable, oldURL, newURL);
}
// Copyright (c) 2012, the Dart project authors.  Please see the AUTHORS file
// for details. All rights reserved. Use of this source code is governed by a
// BSD-style license that can be found in the LICENSE file.

// WARNING: Do not edit - generated code.

@DocsEditable()
@DomName('HTMLHeadElement')
class HeadElement extends HtmlElement {
  // To suppress missing implicit constructor warnings.
  factory HeadElement._() {
    throw new UnsupportedError("Not supported");
  }

  @DomName('HTMLHeadElement.HTMLHeadElement')
  @DocsEditable()
  factory HeadElement() => document.createElement("head");

  @Deprecated("Internal Use Only")
  external static Type get instanceRuntimeType;

  @Deprecated("Internal Use Only")
  HeadElement.internal_() : super.internal_();

  /**
   * Constructor instantiated by the DOM when a custom element has been created.
   *
   * This can only be called by subclasses from their created constructor.
   */
  HeadElement.created() : super.created();
}
// Copyright (c) 2012, the Dart project authors.  Please see the AUTHORS file
// for details. All rights reserved. Use of this source code is governed by a
// BSD-style license that can be found in the LICENSE file.

// WARNING: Do not edit - generated code.

@DocsEditable()
@DomName('Headers')
@Experimental() // untriaged
class Headers extends DartHtmlDomObject {
  // To suppress missing implicit constructor warnings.
  factory Headers._() {
    throw new UnsupportedError("Not supported");
  }

  @DomName('Headers.Headers')
  @DocsEditable()
  factory Headers([input]) {
    if (input == null) {
      return _blink.BlinkHeaders.instance.constructorCallback_0_();
    }
    if ((input is Headers)) {
      return _blink.BlinkHeaders.instance.constructorCallback_1_(input);
    }
    if ((input is Map)) {
      var input_1 = convertDartToNative_Dictionary(input);
      return _blink.BlinkHeaders.instance.constructorCallback_1_(input_1);
    }
    if ((input is List<Object>)) {
      return _blink.BlinkHeaders.instance.constructorCallback_1_(input);
    }
    throw new ArgumentError("Incorrect number or type of arguments");
  }

  @Deprecated("Internal Use Only")
  external static Type get instanceRuntimeType;

  @Deprecated("Internal Use Only")
  Headers.internal_() {}
}
// Copyright (c) 2012, the Dart project authors.  Please see the AUTHORS file
// for details. All rights reserved. Use of this source code is governed by a
// BSD-style license that can be found in the LICENSE file.

// WARNING: Do not edit - generated code.

@DocsEditable()
@DomName('HTMLHeadingElement')
class HeadingElement extends HtmlElement {
  // To suppress missing implicit constructor warnings.
  factory HeadingElement._() {
    throw new UnsupportedError("Not supported");
  }

  @DomName('HTMLHeadingElement.HTMLHeadingElement')
  @DocsEditable()
  factory HeadingElement.h1() => document.createElement("h1");

  @DomName('HTMLHeadingElement.HTMLHeadingElement')
  @DocsEditable()
  factory HeadingElement.h2() => document.createElement("h2");

  @DomName('HTMLHeadingElement.HTMLHeadingElement')
  @DocsEditable()
  factory HeadingElement.h3() => document.createElement("h3");

  @DomName('HTMLHeadingElement.HTMLHeadingElement')
  @DocsEditable()
  factory HeadingElement.h4() => document.createElement("h4");

  @DomName('HTMLHeadingElement.HTMLHeadingElement')
  @DocsEditable()
  factory HeadingElement.h5() => document.createElement("h5");

  @DomName('HTMLHeadingElement.HTMLHeadingElement')
  @DocsEditable()
  factory HeadingElement.h6() => document.createElement("h6");

  @Deprecated("Internal Use Only")
  external static Type get instanceRuntimeType;

  @Deprecated("Internal Use Only")
  HeadingElement.internal_() : super.internal_();

  /**
   * Constructor instantiated by the DOM when a custom element has been created.
   *
   * This can only be called by subclasses from their created constructor.
   */
  HeadingElement.created() : super.created();
}
// Copyright (c) 2012, the Dart project authors.  Please see the AUTHORS file
// for details. All rights reserved. Use of this source code is governed by a
// BSD-style license that can be found in the LICENSE file.

@DomName('History')
class History extends DartHtmlDomObject implements HistoryBase {
  /**
   * Checks if the State APIs are supported on the current platform.
   *
   * See also:
   *
   * * [pushState]
   * * [replaceState]
   * * [state]
   */
  static bool get supportsState => true;
  // To suppress missing implicit constructor warnings.
  factory History._() {
    throw new UnsupportedError("Not supported");
  }

  @Deprecated("Internal Use Only")
  external static Type get instanceRuntimeType;

  @Deprecated("Internal Use Only")
  History.internal_() {}

  @DomName('History.length')
  @DocsEditable()
  int get length => _blink.BlinkHistory.instance.length_Getter_(this);

  @DomName('History.options')
  @DocsEditable()
  @Experimental() // untriaged
  get options => convertNativeDictionaryToDartDictionary(
      (_blink.BlinkHistory.instance.options_Getter_(this)));

  @DomName('History.state')
  @DocsEditable()
  dynamic get state => convertNativeToDart_SerializedScriptValue(
      _blink.BlinkHistory.instance.state_Getter_(this));

  @DomName('History.back')
  @DocsEditable()
  void back() => _blink.BlinkHistory.instance.back_Callback_0_(this);

  @DomName('History.forward')
  @DocsEditable()
  void forward() => _blink.BlinkHistory.instance.forward_Callback_0_(this);

  void go([int delta]) {
    if (delta != null) {
      _blink.BlinkHistory.instance.go_Callback_1_(this, delta);
      return;
    }
    _blink.BlinkHistory.instance.go_Callback_0_(this);
    return;
  }

  void pushState(/*SerializedScriptValue*/ data, String title, String url,
      [Map options]) {
    if (options != null) {
      _blink.BlinkHistory.instance.pushState_Callback_4_(
          this,
          convertDartToNative_SerializedScriptValue(data),
          title,
          url,
          convertDartToNative_Dictionary(options));
      return;
    }
    _blink.BlinkHistory.instance.pushState_Callback_3_(
        this, convertDartToNative_SerializedScriptValue(data), title, url);
    return;
  }

  void replaceState(/*SerializedScriptValue*/ data, String title, String url,
      [Map options]) {
    if (options != null) {
      _blink.BlinkHistory.instance.replaceState_Callback_4_(
          this,
          convertDartToNative_SerializedScriptValue(data),
          title,
          url,
          convertDartToNative_Dictionary(options));
      return;
    }
    _blink.BlinkHistory.instance.replaceState_Callback_3_(
        this, convertDartToNative_SerializedScriptValue(data), title, url);
    return;
  }
}
// Copyright (c) 2012, the Dart project authors.  Please see the AUTHORS file
// for details. All rights reserved. Use of this source code is governed by a
// BSD-style license that can be found in the LICENSE file.

// WARNING: Do not edit - generated code.

@DocsEditable()
@DomName('HMDVRDevice')
@Experimental() // untriaged
class HmdvrDevice extends VRDevice {
  // To suppress missing implicit constructor warnings.
  factory HmdvrDevice._() {
    throw new UnsupportedError("Not supported");
  }

  @Deprecated("Internal Use Only")
  external static Type get instanceRuntimeType;

  @Deprecated("Internal Use Only")
  HmdvrDevice.internal_() : super.internal_();

  @DomName('HMDVRDevice.getEyeParameters')
  @DocsEditable()
  @Experimental() // untriaged
  VREyeParameters getEyeParameters(String whichEye) =>
      _blink.BlinkHMDVRDevice.instance
          .getEyeParameters_Callback_1_(this, whichEye);

  void setFieldOfView([VRFieldOfView leftFov, VRFieldOfView rightFov]) {
    if (rightFov != null) {
      _blink.BlinkHMDVRDevice.instance
          .setFieldOfView_Callback_2_(this, leftFov, rightFov);
      return;
    }
    if (leftFov != null) {
      _blink.BlinkHMDVRDevice.instance
          .setFieldOfView_Callback_1_(this, leftFov);
      return;
    }
    _blink.BlinkHMDVRDevice.instance.setFieldOfView_Callback_0_(this);
    return;
  }
}
// Copyright (c) 2012, the Dart project authors.  Please see the AUTHORS file
// for details. All rights reserved. Use of this source code is governed by a
// BSD-style license that can be found in the LICENSE file.

// WARNING: Do not edit - generated code.

@DocsEditable()
@DomName('HTMLCollection')
class HtmlCollection extends DartHtmlDomObject
    with ListMixin<Node>, ImmutableListMixin<Node>
    implements List<Node> {
  // To suppress missing implicit constructor warnings.
  factory HtmlCollection._() {
    throw new UnsupportedError("Not supported");
  }

  @Deprecated("Internal Use Only")
  external static Type get instanceRuntimeType;

  @Deprecated("Internal Use Only")
  HtmlCollection.internal_() {}

  @DomName('HTMLCollection.length')
  @DocsEditable()
  int get length => _blink.BlinkHTMLCollection.instance.length_Getter_(this);

  Node operator [](int index) {
    if (index < 0 || index >= length) throw new RangeError.index(index, this);
    return _nativeIndexedGetter(index);
  }

  Node _nativeIndexedGetter(int index) =>
      (_blink.BlinkHTMLCollection.instance.item_Callback_1_(this, index));

  void operator []=(int index, Node value) {
    throw new UnsupportedError("Cannot assign element of immutable List.");
  }
  // -- start List<Node> mixins.
  // Node is the element type.

  set length(int value) {
    throw new UnsupportedError("Cannot resize immutable List.");
  }

  Node get first {
    if (this.length > 0) {
      return _nativeIndexedGetter(0);
    }
    throw new StateError("No elements");
  }

  Node get last {
    int len = this.length;
    if (len > 0) {
      return _nativeIndexedGetter(len - 1);
    }
    throw new StateError("No elements");
  }

  Node get single {
    int len = this.length;
    if (len == 1) {
      return _nativeIndexedGetter(0);
    }
    if (len == 0) throw new StateError("No elements");
    throw new StateError("More than one element");
  }

  Node elementAt(int index) => this[index];
  // -- end List<Node> mixins.

  @DomName('HTMLCollection.item')
  @DocsEditable()
  Node item(int index) =>
      _blink.BlinkHTMLCollection.instance.item_Callback_1_(this, index);

  @DomName('HTMLCollection.namedItem')
  @DocsEditable()
  Object namedItem(String name) =>
      (_blink.BlinkHTMLCollection.instance.namedItem_Callback_1_(this, name));
}
// Copyright (c) 2012, the Dart project authors.  Please see the AUTHORS file
// for details. All rights reserved. Use of this source code is governed by a
// BSD-style license that can be found in the LICENSE file.

// WARNING: Do not edit - generated code.

@DomName('HTMLDocument')
class HtmlDocument extends Document {
  // To suppress missing implicit constructor warnings.
  factory HtmlDocument._() {
    throw new UnsupportedError("Not supported");
  }

  @Deprecated("Internal Use Only")
  external static Type get instanceRuntimeType;

  @Deprecated("Internal Use Only")
  HtmlDocument.internal_() : super.internal_();

  @DomName('Document.body')
  BodyElement get body => _body;

  @DomName('Document.body')
  set body(BodyElement value) {
    _body = value;
  }

  /// UNSTABLE: Chrome-only - create a Range from the given point.
  @DomName('Document.caretRangeFromPoint')
  @Unstable()
  Range caretRangeFromPoint(int x, int y) {
    return _caretRangeFromPoint(x, y);
  }

  @DomName('Document.elementFromPoint')
  Element elementFromPoint(int x, int y) {
    return _elementFromPoint(x, y);
  }

  /**
   * Checks if the getCssCanvasContext API is supported on the current platform.
   *
   * See also:
   *
   * * [getCssCanvasContext]
   */
  static bool get supportsCssCanvasContext => true;

  /**
   * Gets a CanvasRenderingContext which can be used as the CSS background of an
   * element.
   *
   * CSS:
   *
   *     background: -webkit-canvas(backgroundCanvas)
   *
   * Generate the canvas:
   *
   *     var context = document.getCssCanvasContext('2d', 'backgroundCanvas',
   *         100, 100);
   *     context.fillStyle = 'red';
   *     context.fillRect(0, 0, 100, 100);
   *
   * See also:
   *
   * * [supportsCssCanvasContext]
   * * [CanvasElement.getContext]
   */
  @SupportedBrowser(SupportedBrowser.CHROME)
  @SupportedBrowser(SupportedBrowser.SAFARI)
  @Experimental()
  @DomName('Document.getCSSCanvasContext')
  CanvasRenderingContext getCssCanvasContext(
      String contextId, String name, int width, int height) {
    return _getCssCanvasContext(contextId, name, width, height);
  }

  @DomName('Document.head')
  HeadElement get head => _head;

  @DomName('Document.lastModified')
  String get lastModified => _lastModified;

  @DomName('Document.preferredStylesheetSet')
  String get preferredStylesheetSet => _preferredStylesheetSet;

  @DomName('Document.referrer')
  String get referrer => _referrer;

  @DomName('Document.selectedStylesheetSet')
  String get selectedStylesheetSet => _selectedStylesheetSet;
  set selectedStylesheetSet(String value) {
    _selectedStylesheetSet = value;
  }

  @DomName('Document.styleSheets')
  List<StyleSheet> get styleSheets => _styleSheets;

  @DomName('Document.title')
  String get title => _title;

  @DomName('Document.title')
  set title(String value) {
    _title = value;
  }

  /**
   * Returns page to standard layout.
   *
   * Has no effect if the page is not in fullscreen mode.
   *
   * ## Other resources
   *
   * * [Using the fullscreen
   *   API](http://docs.webplatform.org/wiki/tutorials/using_the_full-screen_api)
   *   from WebPlatform.org.
   * * [Fullscreen specification](http://www.w3.org/TR/fullscreen/) from W3C.
   */
  @DomName('Document.webkitExitFullscreen')
  @SupportedBrowser(SupportedBrowser.CHROME)
  @SupportedBrowser(SupportedBrowser.SAFARI)
  @Experimental()
  void exitFullscreen() {
    _webkitExitFullscreen();
  }

  /**
   * Internal routine to find the DOM JS class name being extended for custom
   * elements.
   */
  String _getJSClassName(ClassMirror classMirror) {
    var jsClassName = null;
    var isElement = false;

    while (classMirror.superclass != null) {
      var fullName = classMirror.superclass.qualifiedName;
      isElement = isElement ||
          (fullName == #dart.dom.html.Element ||
              fullName == #dart.dom.svg.Element);

      var domLibrary = MirrorSystem.getName(fullName).startsWith('dart.dom.');
      if (jsClassName == null && domLibrary) {
        // Lookup JS class name (if not found).
        var metadatas = classMirror.metadata;
        for (var metadata in metadatas) {
          var metaDataMirror = metadata.reflectee;
          var metaType = reflectClass(metaDataMirror.runtimeType);
          if (MirrorSystem.getName(metaType.simpleName) == 'DomName' &&
              (metaDataMirror.name.startsWith('HTML') ||
                  metaDataMirror.name.startsWith('SVG'))) {
            jsClassName = metadata.reflectee.name;
          }
        }
      }

      classMirror = classMirror.superclass;
    }

    // If we're an element then everything is okay.
    return isElement ? jsClassName : null;
  }

  // Get the first class that's a super of a dart.dom library.
  ClassMirror _getDartHtmlClassName(ClassMirror classMirror) {
    while (classMirror.superclass != null) {
      var fullName = classMirror.superclass.qualifiedName;
      var domLibrary = MirrorSystem.getName(fullName).startsWith('dart.dom.');
      if (domLibrary) {
        return classMirror.superclass;
      }

      classMirror = classMirror.superclass;
    }

    return null;
  }

  /**
   * Get the class that immediately derived from a class in dart:html or
   * dart:svg (has an attribute DomName of either HTML* or SVG*).
   */
  ClassMirror _getDomSuperClass(ClassMirror classMirror) {
    var isElement = false;
    var foundSuperElement = null;

    while (classMirror.superclass != null) {
      var fullName = classMirror.superclass.qualifiedName;
      isElement = isElement ||
          (fullName == #dart.dom.html.Element ||
              fullName == #dart.dom.svg.Element);

      var domLibrary = MirrorSystem.getName(fullName).startsWith('dart.dom.');
      if (domLibrary) {
        if (foundSuperElement == null) {
          foundSuperElement = classMirror.superclass;
        }
        // Lookup JS class (if not found).
        var metadatas = classMirror.metadata;
        for (var metadata in metadatas) {
          var metaDataMirror = metadata.reflectee;
          var metaType = reflectClass(metaDataMirror.runtimeType);
          if (MirrorSystem.getName(metaType.simpleName) == 'DomName' &&
              (metaDataMirror.name.startsWith('HTML') ||
                  metaDataMirror.name.startsWith('SVG'))) {
            if (isElement) return foundSuperElement;
          }
        }
      }

      classMirror = classMirror.superclass;
    }

    return null;
  }

  /**
   * Does this CustomElement class have:
   *
   *   - a created constructor with no arguments?
   *   - a created constructor with a super.created() initializer?
   *
   * e.g.,    MyCustomClass.created() : super.created();
   */
  bool _hasCreatedConstructor(ClassMirror classToRegister) {
    var htmlClassMirror = _getDomSuperClass(classToRegister);

    var classMirror = classToRegister;
    while (classMirror != null && classMirror != htmlClassMirror) {
      var createdParametersValid = false;
      var superCreatedCalled = false;
      var className = MirrorSystem.getName(classMirror.simpleName);
      var methodMirror =
          classMirror.declarations[new Symbol("$className.created")];
      if (methodMirror != null && methodMirror.isConstructor) {
        createdParametersValid = true; // Assume no parameters.
        if (methodMirror.parameters.length != 0) {
          // If any parameters each one must be optional.
          methodMirror.parameters.forEach((parameter) {
            createdParametersValid =
                createdParametersValid && parameter.isOptional;
          });
        }
      }

      if (!createdParametersValid) {
        throw new DomException.jsInterop(
            'created constructor must have no parameters');
      }

      classMirror = classMirror.superclass;
      while (classMirror != classMirror.mixin) {
        // Skip the mixins.
        classMirror = classMirror.superclass;
      }
    }

    return true;
  }

  @Experimental()
  /**
   * Register a custom subclass of Element to be instantiatable by the DOM.
   *
   * This is necessary to allow the construction of any custom elements.
   *
   * The class being registered must either subclass HtmlElement or SvgElement.
   * If they subclass these directly then they can be used as:
   *
   *     class FooElement extends HtmlElement{
   *        void created() {
   *          print('FooElement created!');
   *        }
   *     }
   *
   *     main() {
   *       document.registerElement('x-foo', FooElement);
   *       var myFoo = new Element.tag('x-foo');
   *       // prints 'FooElement created!' to the console.
   *     }
   *
   * The custom element can also be instantiated via HTML using the syntax
   * `<x-foo></x-foo>`
   *
   * Other elements can be subclassed as well:
   *
   *     class BarElement extends InputElement{
   *        void created() {
   *          print('BarElement created!');
   *        }
   *     }
   *
   *     main() {
   *       document.registerElement('x-bar', BarElement);
   *       var myBar = new Element.tag('input', 'x-bar');
   *       // prints 'BarElement created!' to the console.
   *     }
   *
   * This custom element can also be instantiated via HTML using the syntax
   * `<input is="x-bar"></input>`
   *
   */
  void registerElement(String tag, Type customElementClass,
      {String extendsTag}) {
    // Hack to setup an interceptor for HTMLElement so it isn't changed when a custom element is created.
    var jsHTMLElementPrototype = js.JsNative.getProperty(
        js.JsNative.getProperty(js.context, 'HTMLElement'), 'prototype');
    _blink.Blink_Utils.defineInterceptor(
        jsHTMLElementPrototype, HtmlElement.instanceRuntimeType);

    // Figure out which DOM class is being extended from the user's Dart class.
    var classMirror = reflectClass(customElementClass);

    var locationUri = classMirror.location.sourceUri.toString();
    if (locationUri == 'dart:html' || locationUri == 'dart:svg') {
      throw new DomException.jsInterop(
          "HierarchyRequestError: Cannot register an existing dart:html or dart:svg type.");
    }

    if (classMirror.isAbstract) {
      throw new DomException.jsInterop(
          "HierarchyRequestError: Cannot register an abstract class.");
    }

    var jsClassName = _getJSClassName(classMirror);
    if (jsClassName == null) {
      // Only components derived from HTML* can be extended.
      throw new DomException.jsInterop(
          "HierarchyRequestError: Only HTML elements can be customized.");
    }

    var customClassType = _getDartHtmlClassName(classMirror);

    if (extendsTag != null) {
      var nativeElement = document.createElement(extendsTag);

      // Trying to extend a native element is it the Dart class consistent with the
      // extendsTag?
      if (nativeElement.runtimeType != customClassType.reflectedType) {
        var nativeElementClassMirror = reflectClass(nativeElement.runtimeType);
        var customClassNativeElement =
            MirrorSystem.getName(customClassType.simpleName);
        var extendsNativeElement =
            MirrorSystem.getName(nativeElementClassMirror.simpleName);
        throw new DomException.jsInterop(
            "HierarchyRequestError: Custom class type ($customClassNativeElement) and extendsTag class ($extendsNativeElement) don't match .");
      }
    } else if (customClassType.reflectedType != HtmlElement &&
        customClassType.reflectedType != svg.SvgElement) {
      var customClassName = MirrorSystem.getName(classMirror.simpleName);
      var customClassElement = MirrorSystem.getName(customClassType.simpleName);
      throw new DomException.jsInterop(
          "HierarchyRequestError: Custom element $customClassName is a native $customClassElement should be derived from HtmlElement or SvgElement.");
    }

    if (_hasCreatedConstructor(classMirror)) {
      // Start the hookup the JS way create an <x-foo> element that extends the
      // <x-base> custom element. Inherit its prototype and signal what tag is
      // inherited:
      //
      //     var myProto = Object.create(HTMLElement.prototype);
      //     var myElement = document.registerElement('x-foo', {prototype: myProto});
      var baseElement = js.JsNative.getProperty(js.context, jsClassName);
      if (baseElement == null) {
        // Couldn't find the HTML element so use a generic one.
        baseElement = js.JsNative.getProperty(js.context, 'HTMLElement');
      }
      var elemProto = js.JsNative.callMethod(
          js.JsNative.getProperty(js.context, 'Object'),
          "create",
          [js.JsNative.getProperty(baseElement, 'prototype')]);

      // TODO(terry): Hack to stop recursion re-creating custom element when the
      //              created() constructor of the custom element does e.g.,
      //
      //                  MyElement.created() : super.created() {
      //                    this.innerHtml = "<b>I'm an x-foo-with-markup!</b>";
      //                  }
      //
      //              sanitizing causes custom element to created recursively
      //              until stack overflow.
      //
      //              See https://github.com/dart-lang/sdk/issues/23666
      int creating =
          0; // TODO(jacobr): I think I broke thise case. Will fix monday.

      // If any JS code is hooked we want to call it too.
      var oldCreatedCallback =
          js.JsNative.getProperty(elemProto, 'createdCallback');
      var oldAttributeChangedCallback =
          js.JsNative.getProperty(elemProto, 'attributeChangedCallback');
      var oldAttachedCallback =
          js.JsNative.getProperty(elemProto, 'attachedCallback');
      var oldDetachedCallback =
          js.JsNative.getProperty(elemProto, 'detachedCallback');

      js.JsNative.setProperty(elemProto, 'createdCallback',
          js.allowInteropCaptureThis(($this) {
        // The created callback has already been called by the very act of passing a JS
        // custom element from JS to Dart.

        //  Make element's interceptor a CustomElementClass.
        _blink.Blink_Utils.setInstanceInterceptorCustomUpgrade($this);

        if (oldCreatedCallback != null)
          oldCreatedCallback.apply([], thisArg: $this);
      }));
      js.JsNative.setProperty(elemProto, 'attributeChangedCallback',
          js.allowInteropCaptureThis(($this, attrName, oldVal, newVal) {
        $this.attributeChanged(attrName, oldVal, newVal);

        if (oldAttributeChangedCallback != null)
          oldAttributeChangedCallback.apply([], thisArg: $this);
      }));
      js.JsNative.setProperty(elemProto, 'attachedCallback',
          js.allowInteropCaptureThis(($this) {
        $this.attached();

        if (oldAttachedCallback != null)
          oldAttachedCallback.apply([], thisArg: $this);
      }));
      js.JsNative.setProperty(elemProto, 'detachedCallback',
          js.allowInteropCaptureThis(($this) {
        $this.detached();

        if (oldDetachedCallback != null)
          oldDetachedCallback.apply([], thisArg: $this);
      }));
      // document.registerElement('x-foo', {prototype: elemProto, extends: extendsTag});
      var jsMap = new js.JsObject.jsify(
          {'prototype': elemProto, 'extends': extendsTag});
      _blink.Blink_Utils
          .defineInterceptorCustomElement(elemProto, customElementClass);
      js.JsNative.callMethod(document, 'registerElement', [tag, jsMap]);
    }
  }

  /** *Deprecated*: use [registerElement] instead. */
  @deprecated
  @Experimental()
  void register(String tag, Type customElementClass, {String extendsTag}) {
    return registerElement(tag, customElementClass, extendsTag: extendsTag);
  }

  /**
   * Static factory designed to expose `visibilitychange` events to event
   * handlers that are not necessarily instances of [Document].
   *
   * See [EventStreamProvider] for usage information.
   */
  @DomName('Document.visibilityChange')
  @SupportedBrowser(SupportedBrowser.CHROME)
  @SupportedBrowser(SupportedBrowser.FIREFOX)
  @SupportedBrowser(SupportedBrowser.IE, '10')
  @Experimental()
  static const EventStreamProvider<Event> visibilityChangeEvent =
      const _CustomEventStreamProvider<Event>(
          _determineVisibilityChangeEventType);

  static String _determineVisibilityChangeEventType(EventTarget e) {
    return 'webkitvisibilitychange';
  }

  @SupportedBrowser(SupportedBrowser.CHROME)
  @SupportedBrowser(SupportedBrowser.FIREFOX)
  @SupportedBrowser(SupportedBrowser.IE, '10')
  @Experimental()
  Stream<Event> get onVisibilityChange => visibilityChangeEvent.forTarget(this);

  /// Creates an element upgrader which can be used to change the Dart wrapper
  /// type for elements.
  ///
  /// The type specified must be a subclass of HtmlElement, when an element is
  /// upgraded then the created constructor will be invoked on that element.
  ///
  /// If the type is not a direct subclass of HtmlElement then the extendsTag
  /// parameter must be provided.
  @Experimental()
  ElementUpgrader createElementUpgrader(Type type, {String extendsTag}) {
    return new _VMElementUpgrader(this, type, extendsTag);
  }
}
// Copyright (c) 2012, the Dart project authors.  Please see the AUTHORS file
// for details. All rights reserved. Use of this source code is governed by a
// BSD-style license that can be found in the LICENSE file.

@DocsEditable()
@DomName('HTMLElement')
class HtmlElement extends Element implements GlobalEventHandlers {
  // To suppress missing implicit constructor warnings.
  factory HtmlElement._() {
    throw new UnsupportedError("Not supported");
  }

  @DomName('HTMLElement.abortEvent')
  @DocsEditable()
  @Experimental() // untriaged
  static const EventStreamProvider<Event> abortEvent =
      const EventStreamProvider<Event>('abort');

  @DomName('HTMLElement.blurEvent')
  @DocsEditable()
  @Experimental() // untriaged
  static const EventStreamProvider<Event> blurEvent =
      const EventStreamProvider<Event>('blur');

  @DomName('HTMLElement.canplayEvent')
  @DocsEditable()
  @Experimental() // untriaged
  static const EventStreamProvider<Event> canPlayEvent =
      const EventStreamProvider<Event>('canplay');

  @DomName('HTMLElement.canplaythroughEvent')
  @DocsEditable()
  @Experimental() // untriaged
  static const EventStreamProvider<Event> canPlayThroughEvent =
      const EventStreamProvider<Event>('canplaythrough');

  @DomName('HTMLElement.changeEvent')
  @DocsEditable()
  @Experimental() // untriaged
  static const EventStreamProvider<Event> changeEvent =
      const EventStreamProvider<Event>('change');

  @DomName('HTMLElement.clickEvent')
  @DocsEditable()
  @Experimental() // untriaged
  static const EventStreamProvider<MouseEvent> clickEvent =
      const EventStreamProvider<MouseEvent>('click');

  @DomName('HTMLElement.contextmenuEvent')
  @DocsEditable()
  @Experimental() // untriaged
  static const EventStreamProvider<MouseEvent> contextMenuEvent =
      const EventStreamProvider<MouseEvent>('contextmenu');

  @DomName('HTMLElement.dblclickEvent')
  @DocsEditable()
  @Experimental() // untriaged
  static const EventStreamProvider<Event> doubleClickEvent =
      const EventStreamProvider<Event>('dblclick');

  @DomName('HTMLElement.dragEvent')
  @DocsEditable()
  @Experimental() // untriaged
  static const EventStreamProvider<MouseEvent> dragEvent =
      const EventStreamProvider<MouseEvent>('drag');

  @DomName('HTMLElement.dragendEvent')
  @DocsEditable()
  @Experimental() // untriaged
  static const EventStreamProvider<MouseEvent> dragEndEvent =
      const EventStreamProvider<MouseEvent>('dragend');

  @DomName('HTMLElement.dragenterEvent')
  @DocsEditable()
  @Experimental() // untriaged
  static const EventStreamProvider<MouseEvent> dragEnterEvent =
      const EventStreamProvider<MouseEvent>('dragenter');

  @DomName('HTMLElement.dragleaveEvent')
  @DocsEditable()
  @Experimental() // untriaged
  static const EventStreamProvider<MouseEvent> dragLeaveEvent =
      const EventStreamProvider<MouseEvent>('dragleave');

  @DomName('HTMLElement.dragoverEvent')
  @DocsEditable()
  @Experimental() // untriaged
  static const EventStreamProvider<MouseEvent> dragOverEvent =
      const EventStreamProvider<MouseEvent>('dragover');

  @DomName('HTMLElement.dragstartEvent')
  @DocsEditable()
  @Experimental() // untriaged
  static const EventStreamProvider<MouseEvent> dragStartEvent =
      const EventStreamProvider<MouseEvent>('dragstart');

  @DomName('HTMLElement.dropEvent')
  @DocsEditable()
  @Experimental() // untriaged
  static const EventStreamProvider<MouseEvent> dropEvent =
      const EventStreamProvider<MouseEvent>('drop');

  @DomName('HTMLElement.durationchangeEvent')
  @DocsEditable()
  @Experimental() // untriaged
  static const EventStreamProvider<Event> durationChangeEvent =
      const EventStreamProvider<Event>('durationchange');

  @DomName('HTMLElement.emptiedEvent')
  @DocsEditable()
  @Experimental() // untriaged
  static const EventStreamProvider<Event> emptiedEvent =
      const EventStreamProvider<Event>('emptied');

  @DomName('HTMLElement.endedEvent')
  @DocsEditable()
  @Experimental() // untriaged
  static const EventStreamProvider<Event> endedEvent =
      const EventStreamProvider<Event>('ended');

  @DomName('HTMLElement.errorEvent')
  @DocsEditable()
  @Experimental() // untriaged
  static const EventStreamProvider<Event> errorEvent =
      const EventStreamProvider<Event>('error');

  @DomName('HTMLElement.focusEvent')
  @DocsEditable()
  @Experimental() // untriaged
  static const EventStreamProvider<Event> focusEvent =
      const EventStreamProvider<Event>('focus');

  @DomName('HTMLElement.inputEvent')
  @DocsEditable()
  @Experimental() // untriaged
  static const EventStreamProvider<Event> inputEvent =
      const EventStreamProvider<Event>('input');

  @DomName('HTMLElement.invalidEvent')
  @DocsEditable()
  @Experimental() // untriaged
  static const EventStreamProvider<Event> invalidEvent =
      const EventStreamProvider<Event>('invalid');

  @DomName('HTMLElement.keydownEvent')
  @DocsEditable()
  @Experimental() // untriaged
  static const EventStreamProvider<KeyboardEvent> keyDownEvent =
      const EventStreamProvider<KeyboardEvent>('keydown');

  @DomName('HTMLElement.keypressEvent')
  @DocsEditable()
  @Experimental() // untriaged
  static const EventStreamProvider<KeyboardEvent> keyPressEvent =
      const EventStreamProvider<KeyboardEvent>('keypress');

  @DomName('HTMLElement.keyupEvent')
  @DocsEditable()
  @Experimental() // untriaged
  static const EventStreamProvider<KeyboardEvent> keyUpEvent =
      const EventStreamProvider<KeyboardEvent>('keyup');

  @DomName('HTMLElement.loadEvent')
  @DocsEditable()
  @Experimental() // untriaged
  static const EventStreamProvider<Event> loadEvent =
      const EventStreamProvider<Event>('load');

  @DomName('HTMLElement.loadeddataEvent')
  @DocsEditable()
  @Experimental() // untriaged
  static const EventStreamProvider<Event> loadedDataEvent =
      const EventStreamProvider<Event>('loadeddata');

  @DomName('HTMLElement.loadedmetadataEvent')
  @DocsEditable()
  @Experimental() // untriaged
  static const EventStreamProvider<Event> loadedMetadataEvent =
      const EventStreamProvider<Event>('loadedmetadata');

  @DomName('HTMLElement.mousedownEvent')
  @DocsEditable()
  @Experimental() // untriaged
  static const EventStreamProvider<MouseEvent> mouseDownEvent =
      const EventStreamProvider<MouseEvent>('mousedown');

  @DomName('HTMLElement.mouseenterEvent')
  @DocsEditable()
  @Experimental() // untriaged
  static const EventStreamProvider<MouseEvent> mouseEnterEvent =
      const EventStreamProvider<MouseEvent>('mouseenter');

  @DomName('HTMLElement.mouseleaveEvent')
  @DocsEditable()
  @Experimental() // untriaged
  static const EventStreamProvider<MouseEvent> mouseLeaveEvent =
      const EventStreamProvider<MouseEvent>('mouseleave');

  @DomName('HTMLElement.mousemoveEvent')
  @DocsEditable()
  @Experimental() // untriaged
  static const EventStreamProvider<MouseEvent> mouseMoveEvent =
      const EventStreamProvider<MouseEvent>('mousemove');

  @DomName('HTMLElement.mouseoutEvent')
  @DocsEditable()
  @Experimental() // untriaged
  static const EventStreamProvider<MouseEvent> mouseOutEvent =
      const EventStreamProvider<MouseEvent>('mouseout');

  @DomName('HTMLElement.mouseoverEvent')
  @DocsEditable()
  @Experimental() // untriaged
  static const EventStreamProvider<MouseEvent> mouseOverEvent =
      const EventStreamProvider<MouseEvent>('mouseover');

  @DomName('HTMLElement.mouseupEvent')
  @DocsEditable()
  @Experimental() // untriaged
  static const EventStreamProvider<MouseEvent> mouseUpEvent =
      const EventStreamProvider<MouseEvent>('mouseup');

  @DomName('HTMLElement.mousewheelEvent')
  @DocsEditable()
  @Experimental() // untriaged
  static const EventStreamProvider<WheelEvent> mouseWheelEvent =
      const EventStreamProvider<WheelEvent>('mousewheel');

  @DomName('HTMLElement.pauseEvent')
  @DocsEditable()
  @Experimental() // untriaged
  static const EventStreamProvider<Event> pauseEvent =
      const EventStreamProvider<Event>('pause');

  @DomName('HTMLElement.playEvent')
  @DocsEditable()
  @Experimental() // untriaged
  static const EventStreamProvider<Event> playEvent =
      const EventStreamProvider<Event>('play');

  @DomName('HTMLElement.playingEvent')
  @DocsEditable()
  @Experimental() // untriaged
  static const EventStreamProvider<Event> playingEvent =
      const EventStreamProvider<Event>('playing');

  @DomName('HTMLElement.ratechangeEvent')
  @DocsEditable()
  @Experimental() // untriaged
  static const EventStreamProvider<Event> rateChangeEvent =
      const EventStreamProvider<Event>('ratechange');

  @DomName('HTMLElement.resetEvent')
  @DocsEditable()
  @Experimental() // untriaged
  static const EventStreamProvider<Event> resetEvent =
      const EventStreamProvider<Event>('reset');

  @DomName('HTMLElement.resizeEvent')
  @DocsEditable()
  @Experimental() // untriaged
  static const EventStreamProvider<Event> resizeEvent =
      const EventStreamProvider<Event>('resize');

  @DomName('HTMLElement.scrollEvent')
  @DocsEditable()
  @Experimental() // untriaged
  static const EventStreamProvider<Event> scrollEvent =
      const EventStreamProvider<Event>('scroll');

  @DomName('HTMLElement.seekedEvent')
  @DocsEditable()
  @Experimental() // untriaged
  static const EventStreamProvider<Event> seekedEvent =
      const EventStreamProvider<Event>('seeked');

  @DomName('HTMLElement.seekingEvent')
  @DocsEditable()
  @Experimental() // untriaged
  static const EventStreamProvider<Event> seekingEvent =
      const EventStreamProvider<Event>('seeking');

  @DomName('HTMLElement.selectEvent')
  @DocsEditable()
  @Experimental() // untriaged
  static const EventStreamProvider<Event> selectEvent =
      const EventStreamProvider<Event>('select');

  @DomName('HTMLElement.stalledEvent')
  @DocsEditable()
  @Experimental() // untriaged
  static const EventStreamProvider<Event> stalledEvent =
      const EventStreamProvider<Event>('stalled');

  @DomName('HTMLElement.submitEvent')
  @DocsEditable()
  @Experimental() // untriaged
  static const EventStreamProvider<Event> submitEvent =
      const EventStreamProvider<Event>('submit');

  @DomName('HTMLElement.suspendEvent')
  @DocsEditable()
  @Experimental() // untriaged
  static const EventStreamProvider<Event> suspendEvent =
      const EventStreamProvider<Event>('suspend');

  @DomName('HTMLElement.timeupdateEvent')
  @DocsEditable()
  @Experimental() // untriaged
  static const EventStreamProvider<Event> timeUpdateEvent =
      const EventStreamProvider<Event>('timeupdate');

  @DomName('HTMLElement.volumechangeEvent')
  @DocsEditable()
  @Experimental() // untriaged
  static const EventStreamProvider<Event> volumeChangeEvent =
      const EventStreamProvider<Event>('volumechange');

  @DomName('HTMLElement.waitingEvent')
  @DocsEditable()
  @Experimental() // untriaged
  static const EventStreamProvider<Event> waitingEvent =
      const EventStreamProvider<Event>('waiting');

  @Deprecated("Internal Use Only")
  external static Type get instanceRuntimeType;

  @Deprecated("Internal Use Only")
  HtmlElement.internal_() : super.internal_();

  /**
   * Constructor instantiated by the DOM when a custom element has been created.
   *
   * This can only be called by subclasses from their created constructor.
   */
  HtmlElement.created() : super.created();

  @DomName('HTMLElement.contentEditable')
  @DocsEditable()
  String get contentEditable =>
      _blink.BlinkHTMLElement.instance.contentEditable_Getter_(this);

  @DomName('HTMLElement.contentEditable')
  @DocsEditable()
  set contentEditable(String value) =>
      _blink.BlinkHTMLElement.instance.contentEditable_Setter_(this, value);

  @DomName('HTMLElement.contextMenu')
  @DocsEditable()
  @Experimental() // untriaged
  MenuElement get contextMenu =>
      _blink.BlinkHTMLElement.instance.contextMenu_Getter_(this);

  @DomName('HTMLElement.contextMenu')
  @DocsEditable()
  @Experimental() // untriaged
  set contextMenu(MenuElement value) =>
      _blink.BlinkHTMLElement.instance.contextMenu_Setter_(this, value);

  @DomName('HTMLElement.dir')
  @DocsEditable()
  String get dir => _blink.BlinkHTMLElement.instance.dir_Getter_(this);

  @DomName('HTMLElement.dir')
  @DocsEditable()
  set dir(String value) =>
      _blink.BlinkHTMLElement.instance.dir_Setter_(this, value);

  @DomName('HTMLElement.draggable')
  @DocsEditable()
  bool get draggable =>
      _blink.BlinkHTMLElement.instance.draggable_Getter_(this);

  @DomName('HTMLElement.draggable')
  @DocsEditable()
  set draggable(bool value) =>
      _blink.BlinkHTMLElement.instance.draggable_Setter_(this, value);

  @DomName('HTMLElement.hidden')
  @DocsEditable()
  bool get hidden => _blink.BlinkHTMLElement.instance.hidden_Getter_(this);

  @DomName('HTMLElement.hidden')
  @DocsEditable()
  set hidden(bool value) =>
      _blink.BlinkHTMLElement.instance.hidden_Setter_(this, value);

  @DomName('HTMLElement.isContentEditable')
  @DocsEditable()
  bool get isContentEditable =>
      _blink.BlinkHTMLElement.instance.isContentEditable_Getter_(this);

  @DomName('HTMLElement.lang')
  @DocsEditable()
  String get lang => _blink.BlinkHTMLElement.instance.lang_Getter_(this);

  @DomName('HTMLElement.lang')
  @DocsEditable()
  set lang(String value) =>
      _blink.BlinkHTMLElement.instance.lang_Setter_(this, value);

  @DomName('HTMLElement.spellcheck')
  @DocsEditable()
  // http://blog.whatwg.org/the-road-to-html-5-spellchecking
  @Experimental() // nonstandard
  bool get spellcheck =>
      _blink.BlinkHTMLElement.instance.spellcheck_Getter_(this);

  @DomName('HTMLElement.spellcheck')
  @DocsEditable()
  // http://blog.whatwg.org/the-road-to-html-5-spellchecking
  @Experimental() // nonstandard
  set spellcheck(bool value) =>
      _blink.BlinkHTMLElement.instance.spellcheck_Setter_(this, value);

  @DomName('HTMLElement.style')
  @DocsEditable()
  @Experimental() // untriaged
  CssStyleDeclaration get style =>
      _blink.BlinkHTMLElement.instance.style_Getter_(this);

  @DomName('HTMLElement.tabIndex')
  @DocsEditable()
  int get tabIndex => _blink.BlinkHTMLElement.instance.tabIndex_Getter_(this);

  @DomName('HTMLElement.tabIndex')
  @DocsEditable()
  set tabIndex(int value) =>
      _blink.BlinkHTMLElement.instance.tabIndex_Setter_(this, value);

  @DomName('HTMLElement.title')
  @DocsEditable()
  String get title => _blink.BlinkHTMLElement.instance.title_Getter_(this);

  @DomName('HTMLElement.title')
  @DocsEditable()
  set title(String value) =>
      _blink.BlinkHTMLElement.instance.title_Setter_(this, value);

  @DomName('HTMLElement.translate')
  @DocsEditable()
  bool get translate =>
      _blink.BlinkHTMLElement.instance.translate_Getter_(this);

  @DomName('HTMLElement.translate')
  @DocsEditable()
  set translate(bool value) =>
      _blink.BlinkHTMLElement.instance.translate_Setter_(this, value);

  @DomName('HTMLElement.webkitdropzone')
  @DocsEditable()
  @SupportedBrowser(SupportedBrowser.CHROME)
  @SupportedBrowser(SupportedBrowser.SAFARI)
  @Experimental()
  // http://www.whatwg.org/specs/web-apps/current-work/multipage/dnd.html#the-dropzone-attribute
  String get dropzone =>
      _blink.BlinkHTMLElement.instance.webkitdropzone_Getter_(this);

  @DomName('HTMLElement.webkitdropzone')
  @DocsEditable()
  @SupportedBrowser(SupportedBrowser.CHROME)
  @SupportedBrowser(SupportedBrowser.SAFARI)
  @Experimental()
  // http://www.whatwg.org/specs/web-apps/current-work/multipage/dnd.html#the-dropzone-attribute
  set dropzone(String value) =>
      _blink.BlinkHTMLElement.instance.webkitdropzone_Setter_(this, value);

  @DomName('HTMLElement.blur')
  @DocsEditable()
  @Experimental() // untriaged
  void blur() => _blink.BlinkHTMLElement.instance.blur_Callback_0_(this);

  @DomName('HTMLElement.click')
  @DocsEditable()
  void click() => _blink.BlinkHTMLElement.instance.click_Callback_0_(this);

  @DomName('HTMLElement.focus')
  @DocsEditable()
  @Experimental() // untriaged
  void focus() => _blink.BlinkHTMLElement.instance.focus_Callback_0_(this);

  @DomName('HTMLElement.onabort')
  @DocsEditable()
  @Experimental() // untriaged
  ElementStream<Event> get onAbort => abortEvent.forElement(this);

  @DomName('HTMLElement.onblur')
  @DocsEditable()
  @Experimental() // untriaged
  ElementStream<Event> get onBlur => blurEvent.forElement(this);

  @DomName('HTMLElement.oncanplay')
  @DocsEditable()
  @Experimental() // untriaged
  ElementStream<Event> get onCanPlay => canPlayEvent.forElement(this);

  @DomName('HTMLElement.oncanplaythrough')
  @DocsEditable()
  @Experimental() // untriaged
  ElementStream<Event> get onCanPlayThrough =>
      canPlayThroughEvent.forElement(this);

  @DomName('HTMLElement.onchange')
  @DocsEditable()
  @Experimental() // untriaged
  ElementStream<Event> get onChange => changeEvent.forElement(this);

  @DomName('HTMLElement.onclick')
  @DocsEditable()
  @Experimental() // untriaged
  ElementStream<MouseEvent> get onClick => clickEvent.forElement(this);

  @DomName('HTMLElement.oncontextmenu')
  @DocsEditable()
  @Experimental() // untriaged
  ElementStream<MouseEvent> get onContextMenu =>
      contextMenuEvent.forElement(this);

  @DomName('HTMLElement.ondblclick')
  @DocsEditable()
  @Experimental() // untriaged
  ElementStream<Event> get onDoubleClick => doubleClickEvent.forElement(this);

  @DomName('HTMLElement.ondrag')
  @DocsEditable()
  @Experimental() // untriaged
  ElementStream<MouseEvent> get onDrag => dragEvent.forElement(this);

  @DomName('HTMLElement.ondragend')
  @DocsEditable()
  @Experimental() // untriaged
  ElementStream<MouseEvent> get onDragEnd => dragEndEvent.forElement(this);

  @DomName('HTMLElement.ondragenter')
  @DocsEditable()
  @Experimental() // untriaged
  ElementStream<MouseEvent> get onDragEnter => dragEnterEvent.forElement(this);

  @DomName('HTMLElement.ondragleave')
  @DocsEditable()
  @Experimental() // untriaged
  ElementStream<MouseEvent> get onDragLeave => dragLeaveEvent.forElement(this);

  @DomName('HTMLElement.ondragover')
  @DocsEditable()
  @Experimental() // untriaged
  ElementStream<MouseEvent> get onDragOver => dragOverEvent.forElement(this);

  @DomName('HTMLElement.ondragstart')
  @DocsEditable()
  @Experimental() // untriaged
  ElementStream<MouseEvent> get onDragStart => dragStartEvent.forElement(this);

  @DomName('HTMLElement.ondrop')
  @DocsEditable()
  @Experimental() // untriaged
  ElementStream<MouseEvent> get onDrop => dropEvent.forElement(this);

  @DomName('HTMLElement.ondurationchange')
  @DocsEditable()
  @Experimental() // untriaged
  ElementStream<Event> get onDurationChange =>
      durationChangeEvent.forElement(this);

  @DomName('HTMLElement.onemptied')
  @DocsEditable()
  @Experimental() // untriaged
  ElementStream<Event> get onEmptied => emptiedEvent.forElement(this);

  @DomName('HTMLElement.onended')
  @DocsEditable()
  @Experimental() // untriaged
  ElementStream<Event> get onEnded => endedEvent.forElement(this);

  @DomName('HTMLElement.onerror')
  @DocsEditable()
  @Experimental() // untriaged
  ElementStream<Event> get onError => errorEvent.forElement(this);

  @DomName('HTMLElement.onfocus')
  @DocsEditable()
  @Experimental() // untriaged
  ElementStream<Event> get onFocus => focusEvent.forElement(this);

  @DomName('HTMLElement.oninput')
  @DocsEditable()
  @Experimental() // untriaged
  ElementStream<Event> get onInput => inputEvent.forElement(this);

  @DomName('HTMLElement.oninvalid')
  @DocsEditable()
  @Experimental() // untriaged
  ElementStream<Event> get onInvalid => invalidEvent.forElement(this);

  @DomName('HTMLElement.onkeydown')
  @DocsEditable()
  @Experimental() // untriaged
  ElementStream<KeyboardEvent> get onKeyDown => keyDownEvent.forElement(this);

  @DomName('HTMLElement.onkeypress')
  @DocsEditable()
  @Experimental() // untriaged
  ElementStream<KeyboardEvent> get onKeyPress => keyPressEvent.forElement(this);

  @DomName('HTMLElement.onkeyup')
  @DocsEditable()
  @Experimental() // untriaged
  ElementStream<KeyboardEvent> get onKeyUp => keyUpEvent.forElement(this);

  @DomName('HTMLElement.onload')
  @DocsEditable()
  @Experimental() // untriaged
  ElementStream<Event> get onLoad => loadEvent.forElement(this);

  @DomName('HTMLElement.onloadeddata')
  @DocsEditable()
  @Experimental() // untriaged
  ElementStream<Event> get onLoadedData => loadedDataEvent.forElement(this);

  @DomName('HTMLElement.onloadedmetadata')
  @DocsEditable()
  @Experimental() // untriaged
  ElementStream<Event> get onLoadedMetadata =>
      loadedMetadataEvent.forElement(this);

  @DomName('HTMLElement.onmousedown')
  @DocsEditable()
  @Experimental() // untriaged
  ElementStream<MouseEvent> get onMouseDown => mouseDownEvent.forElement(this);

  @DomName('HTMLElement.onmouseenter')
  @DocsEditable()
  @Experimental() // untriaged
  ElementStream<MouseEvent> get onMouseEnter =>
      mouseEnterEvent.forElement(this);

  @DomName('HTMLElement.onmouseleave')
  @DocsEditable()
  @Experimental() // untriaged
  ElementStream<MouseEvent> get onMouseLeave =>
      mouseLeaveEvent.forElement(this);

  @DomName('HTMLElement.onmousemove')
  @DocsEditable()
  @Experimental() // untriaged
  ElementStream<MouseEvent> get onMouseMove => mouseMoveEvent.forElement(this);

  @DomName('HTMLElement.onmouseout')
  @DocsEditable()
  @Experimental() // untriaged
  ElementStream<MouseEvent> get onMouseOut => mouseOutEvent.forElement(this);

  @DomName('HTMLElement.onmouseover')
  @DocsEditable()
  @Experimental() // untriaged
  ElementStream<MouseEvent> get onMouseOver => mouseOverEvent.forElement(this);

  @DomName('HTMLElement.onmouseup')
  @DocsEditable()
  @Experimental() // untriaged
  ElementStream<MouseEvent> get onMouseUp => mouseUpEvent.forElement(this);

  @DomName('HTMLElement.onmousewheel')
  @DocsEditable()
  @Experimental() // untriaged
  ElementStream<WheelEvent> get onMouseWheel =>
      mouseWheelEvent.forElement(this);

  @DomName('HTMLElement.onpause')
  @DocsEditable()
  @Experimental() // untriaged
  ElementStream<Event> get onPause => pauseEvent.forElement(this);

  @DomName('HTMLElement.onplay')
  @DocsEditable()
  @Experimental() // untriaged
  ElementStream<Event> get onPlay => playEvent.forElement(this);

  @DomName('HTMLElement.onplaying')
  @DocsEditable()
  @Experimental() // untriaged
  ElementStream<Event> get onPlaying => playingEvent.forElement(this);

  @DomName('HTMLElement.onratechange')
  @DocsEditable()
  @Experimental() // untriaged
  ElementStream<Event> get onRateChange => rateChangeEvent.forElement(this);

  @DomName('HTMLElement.onreset')
  @DocsEditable()
  @Experimental() // untriaged
  ElementStream<Event> get onReset => resetEvent.forElement(this);

  @DomName('HTMLElement.onresize')
  @DocsEditable()
  @Experimental() // untriaged
  ElementStream<Event> get onResize => resizeEvent.forElement(this);

  @DomName('HTMLElement.onscroll')
  @DocsEditable()
  @Experimental() // untriaged
  ElementStream<Event> get onScroll => scrollEvent.forElement(this);

  @DomName('HTMLElement.onseeked')
  @DocsEditable()
  @Experimental() // untriaged
  ElementStream<Event> get onSeeked => seekedEvent.forElement(this);

  @DomName('HTMLElement.onseeking')
  @DocsEditable()
  @Experimental() // untriaged
  ElementStream<Event> get onSeeking => seekingEvent.forElement(this);

  @DomName('HTMLElement.onselect')
  @DocsEditable()
  @Experimental() // untriaged
  ElementStream<Event> get onSelect => selectEvent.forElement(this);

  @DomName('HTMLElement.onstalled')
  @DocsEditable()
  @Experimental() // untriaged
  ElementStream<Event> get onStalled => stalledEvent.forElement(this);

  @DomName('HTMLElement.onsubmit')
  @DocsEditable()
  @Experimental() // untriaged
  ElementStream<Event> get onSubmit => submitEvent.forElement(this);

  @DomName('HTMLElement.onsuspend')
  @DocsEditable()
  @Experimental() // untriaged
  ElementStream<Event> get onSuspend => suspendEvent.forElement(this);

  @DomName('HTMLElement.ontimeupdate')
  @DocsEditable()
  @Experimental() // untriaged
  ElementStream<Event> get onTimeUpdate => timeUpdateEvent.forElement(this);

  @DomName('HTMLElement.onvolumechange')
  @DocsEditable()
  @Experimental() // untriaged
  ElementStream<Event> get onVolumeChange => volumeChangeEvent.forElement(this);

  @DomName('HTMLElement.onwaiting')
  @DocsEditable()
  @Experimental() // untriaged
  ElementStream<Event> get onWaiting => waitingEvent.forElement(this);
}
// Copyright (c) 2012, the Dart project authors.  Please see the AUTHORS file
// for details. All rights reserved. Use of this source code is governed by a
// BSD-style license that can be found in the LICENSE file.

// WARNING: Do not edit - generated code.

@DocsEditable()
@DomName('HTMLFormControlsCollection')
class HtmlFormControlsCollection extends HtmlCollection {
  // To suppress missing implicit constructor warnings.
  factory HtmlFormControlsCollection._() {
    throw new UnsupportedError("Not supported");
  }

  @Deprecated("Internal Use Only")
  external static Type get instanceRuntimeType;

  @Deprecated("Internal Use Only")
  HtmlFormControlsCollection.internal_() : super.internal_();

  @DomName('HTMLFormControlsCollection.item')
  @DocsEditable()
  @Experimental() // untriaged
  Node item(int index) => _blink.BlinkHTMLFormControlsCollection.instance
      .item_Callback_1_(this, index);

  @DomName('HTMLFormControlsCollection.namedItem')
  @DocsEditable()
  Object namedItem(String name) =>
      (_blink.BlinkHTMLFormControlsCollection.instance
          .namedItem_Callback_1_(this, name));
}
// Copyright (c) 2012, the Dart project authors.  Please see the AUTHORS file
// for details. All rights reserved. Use of this source code is governed by a
// BSD-style license that can be found in the LICENSE file.

// WARNING: Do not edit - generated code.

@DocsEditable()
@DomName('HTMLHtmlElement')
class HtmlHtmlElement extends HtmlElement {
  // To suppress missing implicit constructor warnings.
  factory HtmlHtmlElement._() {
    throw new UnsupportedError("Not supported");
  }

  @DomName('HTMLHtmlElement.HTMLHtmlElement')
  @DocsEditable()
  factory HtmlHtmlElement() => document.createElement("html");

  @Deprecated("Internal Use Only")
  external static Type get instanceRuntimeType;

  @Deprecated("Internal Use Only")
  HtmlHtmlElement.internal_() : super.internal_();

  /**
   * Constructor instantiated by the DOM when a custom element has been created.
   *
   * This can only be called by subclasses from their created constructor.
   */
  HtmlHtmlElement.created() : super.created();
}
// Copyright (c) 2012, the Dart project authors.  Please see the AUTHORS file
// for details. All rights reserved. Use of this source code is governed by a
// BSD-style license that can be found in the LICENSE file.

// WARNING: Do not edit - generated code.

@DocsEditable()
@DomName('HTMLOptionsCollection')
class HtmlOptionsCollection extends HtmlCollection {
  // To suppress missing implicit constructor warnings.
  factory HtmlOptionsCollection._() {
    throw new UnsupportedError("Not supported");
  }

  @Deprecated("Internal Use Only")
  external static Type get instanceRuntimeType;

  @Deprecated("Internal Use Only")
  HtmlOptionsCollection.internal_() : super.internal_();

  @DomName('HTMLOptionsCollection.item')
  @DocsEditable()
  @Experimental() // untriaged
  Node _item(int index) =>
      _blink.BlinkHTMLOptionsCollection.instance.item_Callback_1_(this, index);
}
// Copyright (c) 2012, the Dart project authors.  Please see the AUTHORS file
// for details. All rights reserved. Use of this source code is governed by a
// BSD-style license that can be found in the LICENSE file.

/**
  * A client-side XHR request for getting data from a URL,
  * formally known as XMLHttpRequest.
  *
  * HttpRequest can be used to obtain data from HTTP and FTP protocols,
  * and is useful for AJAX-style page updates.
  *
  * The simplest way to get the contents of a text file, such as a
  * JSON-formatted file, is with [getString].
  * For example, the following code gets the contents of a JSON file
  * and prints its length:
  *
  *     var path = 'myData.json';
  *     HttpRequest.getString(path)
  *         .then((String fileContents) {
  *           print(fileContents.length);
  *         })
  *         .catchError((Error error) {
  *           print(error.toString());
  *         });
  *
  * ## Fetching data from other servers
  *
  * For security reasons, browsers impose restrictions on requests
  * made by embedded apps.
  * With the default behavior of this class,
  * the code making the request must be served from the same origin
  * (domain name, port, and application layer protocol)
  * as the requested resource.
  * In the example above, the myData.json file must be co-located with the
  * app that uses it.
  * You might be able to
  * [get around this restriction](http://www.dartlang.org/articles/json-web-service/#a-note-on-cors-and-httprequest)
  * by using CORS headers or JSONP.
  *
  * ## Other resources
  *
  * * [Fetch Data Dynamically](https://www.dartlang.org/docs/tutorials/fetchdata/),
  * a tutorial from _A Game of Darts_,
  * shows two different ways to use HttpRequest to get a JSON file.
  * * [Get Input from a Form](https://www.dartlang.org/docs/tutorials/forms/),
  * another tutorial from _A Game of Darts_,
  * shows using HttpRequest with a custom server.
  * * [Dart article on using HttpRequests](http://www.dartlang.org/articles/json-web-service/#getting-data)
  * * [JS XMLHttpRequest](https://developer.mozilla.org/en-US/docs/DOM/XMLHttpRequest)
  * * [Using XMLHttpRequest](https://developer.mozilla.org/en-US/docs/DOM/XMLHttpRequest/Using_XMLHttpRequest)
 */
@DomName('XMLHttpRequest')
class HttpRequest extends HttpRequestEventTarget {
  /**
   * Creates a GET request for the specified [url].
   *
   * The server response must be a `text/` mime type for this request to
   * succeed.
   *
   * This is similar to [request] but specialized for HTTP GET requests which
   * return text content.
   *
   * To add query parameters, append them to the [url] following a `?`,
   * joining each key to its value with `=` and separating key-value pairs with
   * `&`.
   *
   *     var name = Uri.encodeQueryComponent('John');
   *     var id = Uri.encodeQueryComponent('42');
   *     HttpRequest.getString('users.json?name=$name&id=$id')
   *       .then((HttpRequest resp) {
   *         // Do something with the response.
   *     });
   *
   * See also:
   *
   * * [request]
   */
  static Future<String> getString(String url,
      {bool withCredentials, void onProgress(ProgressEvent e)}) {
    return request(url,
            withCredentials: withCredentials, onProgress: onProgress)
        .then((HttpRequest xhr) => xhr.responseText);
  }

  /**
   * Makes a server POST request with the specified data encoded as form data.
   *
   * This is roughly the POST equivalent of getString. This method is similar
   * to sending a FormData object with broader browser support but limited to
   * String values.
   *
   * If [data] is supplied, the key/value pairs are URI encoded with
   * [Uri.encodeQueryComponent] and converted into an HTTP query string.
   *
   * Unless otherwise specified, this method appends the following header:
   *
   *     Content-Type: application/x-www-form-urlencoded; charset=UTF-8
   *
   * Here's an example of using this method:
   *
   *     var data = { 'firstName' : 'John', 'lastName' : 'Doe' };
   *     HttpRequest.postFormData('/send', data).then((HttpRequest resp) {
   *       // Do something with the response.
   *     });
   *
   * See also:
   *
   * * [request]
   */
  static Future<HttpRequest> postFormData(String url, Map<String, String> data,
      {bool withCredentials,
      String responseType,
      Map<String, String> requestHeaders,
      void onProgress(ProgressEvent e)}) {
    var parts = [];
    data.forEach((key, value) {
      parts.add('${Uri.encodeQueryComponent(key)}='
          '${Uri.encodeQueryComponent(value)}');
    });
    var formData = parts.join('&');

    if (requestHeaders == null) {
      requestHeaders = <String, String>{};
    }
    requestHeaders.putIfAbsent('Content-Type',
        () => 'application/x-www-form-urlencoded; charset=UTF-8');

    return request(url,
        method: 'POST',
        withCredentials: withCredentials,
        responseType: responseType,
        requestHeaders: requestHeaders,
        sendData: formData,
        onProgress: onProgress);
  }

  /**
   * Creates and sends a URL request for the specified [url].
   *
   * By default `request` will perform an HTTP GET request, but a different
   * method (`POST`, `PUT`, `DELETE`, etc) can be used by specifying the
   * [method] parameter. (See also [HttpRequest.postFormData] for `POST` 
   * requests only.
   *
   * The Future is completed when the response is available.
   *
   * If specified, `sendData` will send data in the form of a [ByteBuffer],
   * [Blob], [Document], [String], or [FormData] along with the HttpRequest.
   *
   * If specified, [responseType] sets the desired response format for the
   * request. By default it is [String], but can also be 'arraybuffer', 'blob', 
   * 'document', 'json', or 'text'. See also [HttpRequest.responseType] 
   * for more information.
   *
   * The [withCredentials] parameter specified that credentials such as a cookie
   * (already) set in the header or
   * [authorization headers](http://tools.ietf.org/html/rfc1945#section-10.2)
   * should be specified for the request. Details to keep in mind when using
   * credentials:
   *
   * * Using credentials is only useful for cross-origin requests.
   * * The `Access-Control-Allow-Origin` header of `url` cannot contain a wildcard (*).
   * * The `Access-Control-Allow-Credentials` header of `url` must be set to true.
   * * If `Access-Control-Expose-Headers` has not been set to true, only a subset of all the response headers will be returned when calling [getAllRequestHeaders].
   *
   * The following is equivalent to the [getString] sample above:
   *
   *     var name = Uri.encodeQueryComponent('John');
   *     var id = Uri.encodeQueryComponent('42');
   *     HttpRequest.request('users.json?name=$name&id=$id')
   *       .then((HttpRequest resp) {
   *         // Do something with the response.
   *     });
   *
   * Here's an example of submitting an entire form with [FormData].
   *
   *     var myForm = querySelector('form#myForm');
   *     var data = new FormData(myForm);
   *     HttpRequest.request('/submit', method: 'POST', sendData: data)
   *       .then((HttpRequest resp) {
   *         // Do something with the response.
   *     });
   *
   * Note that requests for file:// URIs are only supported by Chrome extensions
   * with appropriate permissions in their manifest. Requests to file:// URIs
   * will also never fail- the Future will always complete successfully, even
   * when the file cannot be found.
   *
   * See also: [authorization headers](http://en.wikipedia.org/wiki/Basic_access_authentication).
   */
  static Future<HttpRequest> request(String url,
      {String method,
      bool withCredentials,
      String responseType,
      String mimeType,
      Map<String, String> requestHeaders,
      sendData,
      void onProgress(ProgressEvent e)}) {
    var completer = new Completer<HttpRequest>();

    var xhr = new HttpRequest();
    if (method == null) {
      method = 'GET';
    }
    xhr.open(method, url, async: true);

    if (withCredentials != null) {
      xhr.withCredentials = withCredentials;
    }

    if (responseType != null) {
      xhr.responseType = responseType;
    }

    if (mimeType != null) {
      xhr.overrideMimeType(mimeType);
    }

    if (requestHeaders != null) {
      requestHeaders.forEach((header, value) {
        xhr.setRequestHeader(header, value);
      });
    }

    if (onProgress != null) {
      xhr.onProgress.listen(onProgress);
    }

    xhr.onLoad.listen((e) {
      var accepted = xhr.status >= 200 && xhr.status < 300;
      var fileUri = xhr.status == 0; // file:// URIs have status of 0.
      var notModified = xhr.status == 304;
      // Redirect status is specified up to 307, but others have been used in
      // practice. Notably Google Drive uses 308 Resume Incomplete for
      // resumable uploads, and it's also been used as a redirect. The
      // redirect case will be handled by the browser before it gets to us,
      // so if we see it we should pass it through to the user.
      var unknownRedirect = xhr.status > 307 && xhr.status < 400;

      if (accepted || fileUri || notModified || unknownRedirect) {
        completer.complete(xhr);
      } else {
        completer.completeError(e);
      }
    });

    xhr.onError.listen(completer.completeError);

    if (sendData != null) {
      xhr.send(sendData);
    } else {
      xhr.send();
    }

    return completer.future;
  }

  /**
   * Checks to see if the Progress event is supported on the current platform.
   */
  static bool get supportsProgressEvent {
    return true;
  }

  /**
   * Checks to see if the current platform supports making cross origin
   * requests.
   *
   * Note that even if cross origin requests are supported, they still may fail
   * if the destination server does not support CORS requests.
   */
  static bool get supportsCrossOrigin {
    return true;
  }

  /**
   * Checks to see if the LoadEnd event is supported on the current platform.
   */
  static bool get supportsLoadEndEvent {
    return true;
  }

  /**
   * Checks to see if the overrideMimeType method is supported on the current
   * platform.
   */
  static bool get supportsOverrideMimeType {
    return true;
  }

  /**
   * Makes a cross-origin request to the specified URL.
   *
   * This API provides a subset of [request] which works on IE9. If IE9
   * cross-origin support is not required then [request] should be used instead.
   */
  @Experimental()
  static Future<String> requestCrossOrigin(String url,
      {String method, String sendData}) {
    if (supportsCrossOrigin) {
      return request(url, method: method, sendData: sendData).then((xhr) {
        return xhr.responseText;
      });
    }
  }

  /**
   * Returns all response headers as a key-value map.
   *
   * Multiple values for the same header key can be combined into one,
   * separated by a comma and a space.
   *
   * See: http://www.w3.org/TR/XMLHttpRequest/#the-getresponseheader()-method
   */
  Map<String, String> get responseHeaders {
    // from Closure's goog.net.Xhrio.getResponseHeaders.
    var headers = <String, String>{};
    var headersString = this.getAllResponseHeaders();
    if (headersString == null) {
      return headers;
    }
    var headersList = headersString.split('\r\n');
    for (var header in headersList) {
      if (header.isEmpty) {
        continue;
      }

      var splitIdx = header.indexOf(': ');
      if (splitIdx == -1) {
        continue;
      }
      var key = header.substring(0, splitIdx).toLowerCase();
      var value = header.substring(splitIdx + 2);
      if (headers.containsKey(key)) {
        headers[key] = '${headers[key]}, $value';
      } else {
        headers[key] = value;
      }
    }
    return headers;
  }

  /**
   * Specify the desired `url`, and `method` to use in making the request.
   *
   * By default the request is done asyncronously, with no user or password
   * authentication information. If `async` is false, the request will be send
   * synchronously.
   *
   * Calling `open` again on a currently active request is equivalent to
   * calling `abort`.
   *
   * Note: Most simple HTTP requests can be accomplished using the [getString],
   * [request], [requestCrossOrigin], or [postFormData] methods. Use of this
   * `open` method is intended only for more complext HTTP requests where
   * finer-grained control is needed.
   */
  @DomName('XMLHttpRequest.open')
  @DocsEditable()
  void open(String method, String url,
      {bool async, String user, String password}) {
    if (async == null && user == null && password == null) {
      _blink.BlinkXMLHttpRequest.instance.open_Callback_2_(this, method, url);
    } else {
      _blink.BlinkXMLHttpRequest.instance
          .open_Callback_5_(this, method, url, async, user, password);
    }
  }

  // To suppress missing implicit constructor warnings.
  factory HttpRequest._() {
    throw new UnsupportedError("Not supported");
  }

  /**
   * Static factory designed to expose `readystatechange` events to event
   * handlers that are not necessarily instances of [HttpRequest].
   *
   * See [EventStreamProvider] for usage information.
   */
  @DomName('XMLHttpRequest.readystatechangeEvent')
  @DocsEditable()
  static const EventStreamProvider<ProgressEvent> readyStateChangeEvent =
      const EventStreamProvider<ProgressEvent>('readystatechange');

  /**
   * General constructor for any type of request (GET, POST, etc).
   *
   * This call is used in conjunction with [open]:
   *
   *     var request = new HttpRequest();
   *     request.open('GET', 'http://dartlang.org');
   *     request.onLoad.listen((event) => print(
   *         'Request complete ${event.target.reponseText}'));
   *     request.send();
   *
   * is the (more verbose) equivalent of
   *
   *     HttpRequest.getString('http://dartlang.org').then(
   *         (result) => print('Request complete: $result'));
   */
  @DomName('XMLHttpRequest.XMLHttpRequest')
  @DocsEditable()
  factory HttpRequest() {
    return _blink.BlinkXMLHttpRequest.instance.constructorCallback_0_();
  }

  @Deprecated("Internal Use Only")
  external static Type get instanceRuntimeType;

  @Deprecated("Internal Use Only")
  HttpRequest.internal_() : super.internal_();

  @DomName('XMLHttpRequest.DONE')
  @DocsEditable()
  static const int DONE = 4;

  @DomName('XMLHttpRequest.HEADERS_RECEIVED')
  @DocsEditable()
  static const int HEADERS_RECEIVED = 2;

  @DomName('XMLHttpRequest.LOADING')
  @DocsEditable()
  static const int LOADING = 3;

  @DomName('XMLHttpRequest.OPENED')
  @DocsEditable()
  static const int OPENED = 1;

  @DomName('XMLHttpRequest.UNSENT')
  @DocsEditable()
  static const int UNSENT = 0;

  /**
   * Indicator of the current state of the request:
   *
   * <table>
   *   <tr>
   *     <td>Value</td>
   *     <td>State</td>
   *     <td>Meaning</td>
   *   </tr>
   *   <tr>
   *     <td>0</td>
   *     <td>unsent</td>
   *     <td><code>open()</code> has not yet been called</td>
   *   </tr>
   *   <tr>
   *     <td>1</td>
   *     <td>opened</td>
   *     <td><code>send()</code> has not yet been called</td>
   *   </tr>
   *   <tr>
   *     <td>2</td>
   *     <td>headers received</td>
   *     <td><code>sent()</code> has been called; response headers and <code>status</code> are available</td>
   *   </tr>
   *   <tr>
   *     <td>3</td> <td>loading</td> <td><code>responseText</code> holds some data</td>
   *   </tr>
   *   <tr>
   *     <td>4</td> <td>done</td> <td>request is complete</td>
   *   </tr>
   * </table>
   */
  @DomName('XMLHttpRequest.readyState')
  @DocsEditable()
  int get readyState =>
      _blink.BlinkXMLHttpRequest.instance.readyState_Getter_(this);

  /**
   * The data received as a reponse from the request.
   *
   * The data could be in the
   * form of a [String], [ByteBuffer], [Document], [Blob], or json (also a
   * [String]). `null` indicates request failure.
   */
  @DomName('XMLHttpRequest.response')
  @DocsEditable()
  @SupportedBrowser(SupportedBrowser.CHROME)
  @SupportedBrowser(SupportedBrowser.FIREFOX)
  @SupportedBrowser(SupportedBrowser.IE, '10')
  @SupportedBrowser(SupportedBrowser.SAFARI)
  Object get response => _convertNativeToDart_XHR_Response(
      _blink.BlinkXMLHttpRequest.instance.response_Getter_(this));

  /**
   * The response in String form or empty String on failure.
   */
  @DomName('XMLHttpRequest.responseText')
  @DocsEditable()
  String get responseText =>
      _blink.BlinkXMLHttpRequest.instance.responseText_Getter_(this);

  /**
   * [String] telling the server the desired response format.
   *
   * Default is `String`.
   * Other options are one of 'arraybuffer', 'blob', 'document', 'json',
   * 'text'. Some newer browsers will throw NS_ERROR_DOM_INVALID_ACCESS_ERR if
   * `responseType` is set while performing a synchronous request.
   *
   * See also: [MDN
   * responseType](https://developer.mozilla.org/en-US/docs/Web/API/XMLHttpRequest#xmlhttprequest-responsetype)
   */
  @DomName('XMLHttpRequest.responseType')
  @DocsEditable()
  String get responseType =>
      _blink.BlinkXMLHttpRequest.instance.responseType_Getter_(this);

  /**
   * [String] telling the server the desired response format.
   *
   * Default is `String`.
   * Other options are one of 'arraybuffer', 'blob', 'document', 'json',
   * 'text'. Some newer browsers will throw NS_ERROR_DOM_INVALID_ACCESS_ERR if
   * `responseType` is set while performing a synchronous request.
   *
   * See also: [MDN
   * responseType](https://developer.mozilla.org/en-US/docs/Web/API/XMLHttpRequest#xmlhttprequest-responsetype)
   */
  @DomName('XMLHttpRequest.responseType')
  @DocsEditable()
  set responseType(String value) =>
      _blink.BlinkXMLHttpRequest.instance.responseType_Setter_(this, value);

  @DomName('XMLHttpRequest.responseURL')
  @DocsEditable()
  @Experimental() // untriaged
  String get responseUrl =>
      _blink.BlinkXMLHttpRequest.instance.responseURL_Getter_(this);

  /**
   * The request response, or null on failure.
   *
   * The response is processed as
   * `text/xml` stream, unless responseType = 'document' and the request is
   * synchronous.
   */
  @DomName('XMLHttpRequest.responseXML')
  @DocsEditable()
  Document get responseXml =>
      _blink.BlinkXMLHttpRequest.instance.responseXML_Getter_(this);

  /**
   * The HTTP result code from the request (200, 404, etc).
   * See also: [HTTP Status Codes](http://en.wikipedia.org/wiki/List_of_HTTP_status_codes)
   */
  @DomName('XMLHttpRequest.status')
  @DocsEditable()
  int get status => _blink.BlinkXMLHttpRequest.instance.status_Getter_(this);

  /**
   * The request response string (such as \"200 OK\").
   * See also: [HTTP Status Codes](http://en.wikipedia.org/wiki/List_of_HTTP_status_codes)
   */
  @DomName('XMLHttpRequest.statusText')
  @DocsEditable()
  String get statusText =>
      _blink.BlinkXMLHttpRequest.instance.statusText_Getter_(this);

  /**
   * Length of time before a request is automatically terminated.
   *
   * When the time has passed, a [TimeoutEvent] is dispatched.
   *
   * If [timeout] is set to 0, then the request will not time out.
   *
   * ## Other resources
   *
   * * [XMLHttpRequest.timeout](https://developer.mozilla.org/en-US/docs/Web/API/XMLHttpRequest#xmlhttprequest-timeout)
   *   from MDN.
   * * [The timeout attribute](http://www.w3.org/TR/XMLHttpRequest/#the-timeout-attribute)
   *   from W3C.
   */
  @DomName('XMLHttpRequest.timeout')
  @DocsEditable()
  @Experimental() // untriaged
  int get timeout => _blink.BlinkXMLHttpRequest.instance.timeout_Getter_(this);

  /**
   * Length of time before a request is automatically terminated.
   *
   * When the time has passed, a [TimeoutEvent] is dispatched.
   *
   * If [timeout] is set to 0, then the request will not time out.
   *
   * ## Other resources
   *
   * * [XMLHttpRequest.timeout](https://developer.mozilla.org/en-US/docs/Web/API/XMLHttpRequest#xmlhttprequest-timeout)
   *   from MDN.
   * * [The timeout attribute](http://www.w3.org/TR/XMLHttpRequest/#the-timeout-attribute)
   *   from W3C.
   */
  @DomName('XMLHttpRequest.timeout')
  @DocsEditable()
  @Experimental() // untriaged
  set timeout(int value) =>
      _blink.BlinkXMLHttpRequest.instance.timeout_Setter_(this, value);

  /**
   * [EventTarget] that can hold listeners to track the progress of the request.
   * The events fired will be members of [HttpRequestUploadEvents].
   */
  @DomName('XMLHttpRequest.upload')
  @DocsEditable()
  @Unstable()
  HttpRequestUpload get upload =>
      _blink.BlinkXMLHttpRequest.instance.upload_Getter_(this);

  /**
   * True if cross-site requests should use credentials such as cookies
   * or authorization headers; false otherwise.
   *
   * This value is ignored for same-site requests.
   */
  @DomName('XMLHttpRequest.withCredentials')
  @DocsEditable()
  bool get withCredentials =>
      _blink.BlinkXMLHttpRequest.instance.withCredentials_Getter_(this);

  /**
   * True if cross-site requests should use credentials such as cookies
   * or authorization headers; false otherwise.
   *
   * This value is ignored for same-site requests.
   */
  @DomName('XMLHttpRequest.withCredentials')
  @DocsEditable()
  set withCredentials(bool value) =>
      _blink.BlinkXMLHttpRequest.instance.withCredentials_Setter_(this, value);

  /**
   * Stop the current request.
   *
   * The request can only be stopped if readyState is `HEADERS_RECIEVED` or
   * `LOADING`. If this method is not in the process of being sent, the method
   * has no effect.
   */
  @DomName('XMLHttpRequest.abort')
  @DocsEditable()
  void abort() => _blink.BlinkXMLHttpRequest.instance.abort_Callback_0_(this);

  /**
   * Retrieve all the response headers from a request.
   *
   * `null` if no headers have been received. For multipart requests,
   * `getAllResponseHeaders` will return the response headers for the current
   * part of the request.
   *
   * See also [HTTP response
   * headers](https://en.wikipedia.org/wiki/List_of_HTTP_header_fields#Response_fields)
   * for a list of common response headers.
   */
  @DomName('XMLHttpRequest.getAllResponseHeaders')
  @DocsEditable()
  @Unstable()
  String getAllResponseHeaders() => _blink.BlinkXMLHttpRequest.instance
      .getAllResponseHeaders_Callback_0_(this);

  /**
   * Return the response header named `header`, or null if not found.
   *
   * See also [HTTP response
   * headers](https://en.wikipedia.org/wiki/List_of_HTTP_header_fields#Response_fields)
   * for a list of common response headers.
   */
  @DomName('XMLHttpRequest.getResponseHeader')
  @DocsEditable()
  @Unstable()
  String getResponseHeader(String name) => _blink.BlinkXMLHttpRequest.instance
      .getResponseHeader_Callback_1_(this, name);

  /**
   * Specify a particular MIME type (such as `text/xml`) desired for the
   * response.
   *
   * This value must be set before the request has been sent. See also the list
   * of [IANA Official MIME types](https://www.iana.org/assignments/media-types/media-types.xhtml)
   */
  @DomName('XMLHttpRequest.overrideMimeType')
  @DocsEditable()
  @SupportedBrowser(SupportedBrowser.CHROME)
  @SupportedBrowser(SupportedBrowser.FIREFOX)
  @SupportedBrowser(SupportedBrowser.SAFARI)
  void overrideMimeType(String mime) => _blink.BlinkXMLHttpRequest.instance
      .overrideMimeType_Callback_1_(this, mime);

  void send([body_OR_data]) {
    if (body_OR_data != null) {
      _blink.BlinkXMLHttpRequest.instance.send_Callback_1_(this, body_OR_data);
      return;
    }
    if ((body_OR_data is TypedData || body_OR_data == null)) {
      _blink.BlinkXMLHttpRequest.instance.send_Callback_1_(this, body_OR_data);
      return;
    }
    if ((body_OR_data is ByteBuffer || body_OR_data == null)) {
      _blink.BlinkXMLHttpRequest.instance.send_Callback_1_(this, body_OR_data);
      return;
    }
    if ((body_OR_data is Document || body_OR_data == null)) {
      _blink.BlinkXMLHttpRequest.instance.send_Callback_1_(this, body_OR_data);
      return;
    }
    if ((body_OR_data is String || body_OR_data == null)) {
      _blink.BlinkXMLHttpRequest.instance.send_Callback_1_(this, body_OR_data);
      return;
    }
    if ((body_OR_data is FormData || body_OR_data == null)) {
      _blink.BlinkXMLHttpRequest.instance.send_Callback_1_(this, body_OR_data);
      return;
    }
    if ((body_OR_data is Blob || body_OR_data == null)) {
      _blink.BlinkXMLHttpRequest.instance.send_Callback_1_(this, body_OR_data);
      return;
    }
    if (body_OR_data == null) {
      _blink.BlinkXMLHttpRequest.instance.send_Callback_0_(this);
      return;
    }
    throw new ArgumentError("Incorrect number or type of arguments");
  }

  /**
   * Sets the value of an HTTP requst header.
   *
   * This method should be called after the request is opened, but before
   * the request is sent.
   *
   * Multiple calls with the same header will combine all their values into a
   * single header.
   *
   * ## Other resources
   *
   * * [XMLHttpRequest.setRequestHeader](https://developer.mozilla.org/en-US/docs/Web/API/XMLHttpRequest#setRequestHeader())
   *   from MDN.
   * * [The setRequestHeader()
   *   method](http://www.w3.org/TR/XMLHttpRequest/#the-setrequestheader()-method)
   *   from W3C.
   */
  @DomName('XMLHttpRequest.setRequestHeader')
  @DocsEditable()
  void setRequestHeader(String name, String value) =>
      _blink.BlinkXMLHttpRequest.instance
          .setRequestHeader_Callback_2_(this, name, value);

  /// Stream of `readystatechange` events handled by this [HttpRequest].
/**
   * Event listeners to be notified every time the [HttpRequest]
   * object's `readyState` changes values.
   */
  @DomName('XMLHttpRequest.onreadystatechange')
  @DocsEditable()
  Stream<ProgressEvent> get onReadyStateChange =>
      readyStateChangeEvent.forTarget(this);
}
// Copyright (c) 2012, the Dart project authors.  Please see the AUTHORS file
// for details. All rights reserved. Use of this source code is governed by a
// BSD-style license that can be found in the LICENSE file.

// WARNING: Do not edit - generated code.

@DocsEditable()
@DomName('XMLHttpRequestEventTarget')
@Experimental() // untriaged
class HttpRequestEventTarget extends EventTarget {
  // To suppress missing implicit constructor warnings.
  factory HttpRequestEventTarget._() {
    throw new UnsupportedError("Not supported");
  }

  /**
   * Static factory designed to expose `abort` events to event
   * handlers that are not necessarily instances of [HttpRequestEventTarget].
   *
   * See [EventStreamProvider] for usage information.
   */
  @DomName('XMLHttpRequestEventTarget.abortEvent')
  @DocsEditable()
  @Experimental() // untriaged
  static const EventStreamProvider<ProgressEvent> abortEvent =
      const EventStreamProvider<ProgressEvent>('abort');

  /**
   * Static factory designed to expose `error` events to event
   * handlers that are not necessarily instances of [HttpRequestEventTarget].
   *
   * See [EventStreamProvider] for usage information.
   */
  @DomName('XMLHttpRequestEventTarget.errorEvent')
  @DocsEditable()
  @Experimental() // untriaged
  static const EventStreamProvider<ProgressEvent> errorEvent =
      const EventStreamProvider<ProgressEvent>('error');

  /**
   * Static factory designed to expose `load` events to event
   * handlers that are not necessarily instances of [HttpRequestEventTarget].
   *
   * See [EventStreamProvider] for usage information.
   */
  @DomName('XMLHttpRequestEventTarget.loadEvent')
  @DocsEditable()
  @Experimental() // untriaged
  static const EventStreamProvider<ProgressEvent> loadEvent =
      const EventStreamProvider<ProgressEvent>('load');

  /**
   * Static factory designed to expose `loadend` events to event
   * handlers that are not necessarily instances of [HttpRequestEventTarget].
   *
   * See [EventStreamProvider] for usage information.
   */
  @DomName('XMLHttpRequestEventTarget.loadendEvent')
  @DocsEditable()
  @Experimental() // untriaged
  static const EventStreamProvider<ProgressEvent> loadEndEvent =
      const EventStreamProvider<ProgressEvent>('loadend');

  /**
   * Static factory designed to expose `loadstart` events to event
   * handlers that are not necessarily instances of [HttpRequestEventTarget].
   *
   * See [EventStreamProvider] for usage information.
   */
  @DomName('XMLHttpRequestEventTarget.loadstartEvent')
  @DocsEditable()
  @Experimental() // untriaged
  static const EventStreamProvider<ProgressEvent> loadStartEvent =
      const EventStreamProvider<ProgressEvent>('loadstart');

  /**
   * Static factory designed to expose `progress` events to event
   * handlers that are not necessarily instances of [HttpRequestEventTarget].
   *
   * See [EventStreamProvider] for usage information.
   */
  @DomName('XMLHttpRequestEventTarget.progressEvent')
  @DocsEditable()
  @Experimental() // untriaged
  static const EventStreamProvider<ProgressEvent> progressEvent =
      const EventStreamProvider<ProgressEvent>('progress');

  /**
   * Static factory designed to expose `timeout` events to event
   * handlers that are not necessarily instances of [HttpRequestEventTarget].
   *
   * See [EventStreamProvider] for usage information.
   */
  @DomName('XMLHttpRequestEventTarget.timeoutEvent')
  @DocsEditable()
  @Experimental() // untriaged
  static const EventStreamProvider<ProgressEvent> timeoutEvent =
      const EventStreamProvider<ProgressEvent>('timeout');

  @Deprecated("Internal Use Only")
  external static Type get instanceRuntimeType;

  @Deprecated("Internal Use Only")
  HttpRequestEventTarget.internal_() : super.internal_();

  /// Stream of `abort` events handled by this [HttpRequestEventTarget].
  @DomName('XMLHttpRequestEventTarget.onabort')
  @DocsEditable()
  @Experimental() // untriaged
  Stream<ProgressEvent> get onAbort => abortEvent.forTarget(this);

  /// Stream of `error` events handled by this [HttpRequestEventTarget].
  @DomName('XMLHttpRequestEventTarget.onerror')
  @DocsEditable()
  @Experimental() // untriaged
  Stream<ProgressEvent> get onError => errorEvent.forTarget(this);

  /// Stream of `load` events handled by this [HttpRequestEventTarget].
  @DomName('XMLHttpRequestEventTarget.onload')
  @DocsEditable()
  @Experimental() // untriaged
  Stream<ProgressEvent> get onLoad => loadEvent.forTarget(this);

  /// Stream of `loadend` events handled by this [HttpRequestEventTarget].
  @DomName('XMLHttpRequestEventTarget.onloadend')
  @DocsEditable()
  @SupportedBrowser(SupportedBrowser.CHROME)
  @SupportedBrowser(SupportedBrowser.FIREFOX)
  @SupportedBrowser(SupportedBrowser.IE, '10')
  @SupportedBrowser(SupportedBrowser.SAFARI)
  @Experimental() // untriaged
  Stream<ProgressEvent> get onLoadEnd => loadEndEvent.forTarget(this);

  /// Stream of `loadstart` events handled by this [HttpRequestEventTarget].
  @DomName('XMLHttpRequestEventTarget.onloadstart')
  @DocsEditable()
  @Experimental() // untriaged
  Stream<ProgressEvent> get onLoadStart => loadStartEvent.forTarget(this);

  /// Stream of `progress` events handled by this [HttpRequestEventTarget].
  @DomName('XMLHttpRequestEventTarget.onprogress')
  @DocsEditable()
  @SupportedBrowser(SupportedBrowser.CHROME)
  @SupportedBrowser(SupportedBrowser.FIREFOX)
  @SupportedBrowser(SupportedBrowser.IE, '10')
  @SupportedBrowser(SupportedBrowser.SAFARI)
  @Experimental() // untriaged
  Stream<ProgressEvent> get onProgress => progressEvent.forTarget(this);

  /// Stream of `timeout` events handled by this [HttpRequestEventTarget].
  @DomName('XMLHttpRequestEventTarget.ontimeout')
  @DocsEditable()
  @Experimental() // untriaged
  Stream<ProgressEvent> get onTimeout => timeoutEvent.forTarget(this);
}
// Copyright (c) 2012, the Dart project authors.  Please see the AUTHORS file
// for details. All rights reserved. Use of this source code is governed by a
// BSD-style license that can be found in the LICENSE file.

// WARNING: Do not edit - generated code.

@DocsEditable()
@DomName('XMLHttpRequestUpload')
// http://xhr.spec.whatwg.org/#xmlhttprequestupload
@Experimental()
class HttpRequestUpload extends HttpRequestEventTarget {
  // To suppress missing implicit constructor warnings.
  factory HttpRequestUpload._() {
    throw new UnsupportedError("Not supported");
  }

  @Deprecated("Internal Use Only")
  external static Type get instanceRuntimeType;

  @Deprecated("Internal Use Only")
  HttpRequestUpload.internal_() : super.internal_();
}
// Copyright (c) 2012, the Dart project authors.  Please see the AUTHORS file
// for details. All rights reserved. Use of this source code is governed by a
// BSD-style license that can be found in the LICENSE file.

// WARNING: Do not edit - generated code.

@DocsEditable()
@DomName('HTMLIFrameElement')
class IFrameElement extends HtmlElement {
  // To suppress missing implicit constructor warnings.
  factory IFrameElement._() {
    throw new UnsupportedError("Not supported");
  }

  @DomName('HTMLIFrameElement.HTMLIFrameElement')
  @DocsEditable()
  factory IFrameElement() => document.createElement("iframe");

  @Deprecated("Internal Use Only")
  external static Type get instanceRuntimeType;

  @Deprecated("Internal Use Only")
  IFrameElement.internal_() : super.internal_();

  /**
   * Constructor instantiated by the DOM when a custom element has been created.
   *
   * This can only be called by subclasses from their created constructor.
   */
  IFrameElement.created() : super.created();

  @DomName('HTMLIFrameElement.allowFullscreen')
  @DocsEditable()
  @Experimental() // untriaged
  bool get allowFullscreen =>
      _blink.BlinkHTMLIFrameElement.instance.allowFullscreen_Getter_(this);

  @DomName('HTMLIFrameElement.allowFullscreen')
  @DocsEditable()
  @Experimental() // untriaged
  set allowFullscreen(bool value) => _blink.BlinkHTMLIFrameElement.instance
      .allowFullscreen_Setter_(this, value);

  @DomName('HTMLIFrameElement.contentWindow')
  @DocsEditable()
  WindowBase get contentWindow => _convertNativeToDart_Window(
      _blink.BlinkHTMLIFrameElement.instance.contentWindow_Getter_(this));

  @DomName('HTMLIFrameElement.height')
  @DocsEditable()
  String get height =>
      _blink.BlinkHTMLIFrameElement.instance.height_Getter_(this);

  @DomName('HTMLIFrameElement.height')
  @DocsEditable()
  set height(String value) =>
      _blink.BlinkHTMLIFrameElement.instance.height_Setter_(this, value);

  @DomName('HTMLIFrameElement.name')
  @DocsEditable()
  String get name => _blink.BlinkHTMLIFrameElement.instance.name_Getter_(this);

  @DomName('HTMLIFrameElement.name')
  @DocsEditable()
  set name(String value) =>
      _blink.BlinkHTMLIFrameElement.instance.name_Setter_(this, value);

  @DomName('HTMLIFrameElement.sandbox')
  @DocsEditable()
  DomSettableTokenList get sandbox =>
      _blink.BlinkHTMLIFrameElement.instance.sandbox_Getter_(this);

  @DomName('HTMLIFrameElement.src')
  @DocsEditable()
  String get src => _blink.BlinkHTMLIFrameElement.instance.src_Getter_(this);

  @DomName('HTMLIFrameElement.src')
  @DocsEditable()
  set src(String value) =>
      _blink.BlinkHTMLIFrameElement.instance.src_Setter_(this, value);

  @DomName('HTMLIFrameElement.srcdoc')
  @DocsEditable()
  String get srcdoc =>
      _blink.BlinkHTMLIFrameElement.instance.srcdoc_Getter_(this);

  @DomName('HTMLIFrameElement.srcdoc')
  @DocsEditable()
  set srcdoc(String value) =>
      _blink.BlinkHTMLIFrameElement.instance.srcdoc_Setter_(this, value);

  @DomName('HTMLIFrameElement.width')
  @DocsEditable()
  String get width =>
      _blink.BlinkHTMLIFrameElement.instance.width_Getter_(this);

  @DomName('HTMLIFrameElement.width')
  @DocsEditable()
  set width(String value) =>
      _blink.BlinkHTMLIFrameElement.instance.width_Setter_(this, value);
}
// Copyright (c) 2012, the Dart project authors.  Please see the AUTHORS file
// for details. All rights reserved. Use of this source code is governed by a
// BSD-style license that can be found in the LICENSE file.

// WARNING: Do not edit - generated code.

@DocsEditable()
@DomName('ImageBitmap')
@Experimental() // untriaged
class ImageBitmap extends DartHtmlDomObject {
  // To suppress missing implicit constructor warnings.
  factory ImageBitmap._() {
    throw new UnsupportedError("Not supported");
  }

  @Deprecated("Internal Use Only")
  external static Type get instanceRuntimeType;

  @Deprecated("Internal Use Only")
  ImageBitmap.internal_() {}

  @DomName('ImageBitmap.height')
  @DocsEditable()
  @Experimental() // untriaged
  int get height => _blink.BlinkImageBitmap.instance.height_Getter_(this);

  @DomName('ImageBitmap.width')
  @DocsEditable()
  @Experimental() // untriaged
  int get width => _blink.BlinkImageBitmap.instance.width_Getter_(this);
}
// Copyright (c) 2013, the Dart project authors.  Please see the AUTHORS file
// for details. All rights reserved. Use of this source code is governed by a
// BSD-style license that can be found in the LICENSE file.

@DomName('ImageData')
class ImageData extends DartHtmlDomObject {
  List<int> __data;

  List<int> get data {
    if (__data == null) {
      __data = _data;
    }
    return __data;
  }

  // To suppress missing implicit constructor warnings.
  factory ImageData._() {
    throw new UnsupportedError("Not supported");
  }

  @DomName('ImageData.ImageData')
  @DocsEditable()
  factory ImageData(data_OR_sw, int sh_OR_sw, [int sh]) {
    if ((sh_OR_sw is int) && (data_OR_sw is int) && sh == null) {
      return _blink.BlinkImageData.instance
          .constructorCallback_2_(data_OR_sw, sh_OR_sw);
    }
    if ((sh_OR_sw is int) && (data_OR_sw is Uint8ClampedList) && sh == null) {
      return _blink.BlinkImageData.instance
          .constructorCallback_2_(data_OR_sw, sh_OR_sw);
    }
    if ((sh is int) && (sh_OR_sw is int) && (data_OR_sw is Uint8ClampedList)) {
      return _blink.BlinkImageData.instance
          .constructorCallback_3_(data_OR_sw, sh_OR_sw, sh);
    }
    throw new ArgumentError("Incorrect number or type of arguments");
  }

  @Deprecated("Internal Use Only")
  external static Type get instanceRuntimeType;

  @Deprecated("Internal Use Only")
  ImageData.internal_() {}

  @DomName('ImageData.data')
  @DocsEditable()
  Uint8ClampedList get _data =>
      _blink.BlinkImageData.instance.data_Getter_(this);

  @DomName('ImageData.height')
  @DocsEditable()
  int get height => _blink.BlinkImageData.instance.height_Getter_(this);

  @DomName('ImageData.width')
  @DocsEditable()
  int get width => _blink.BlinkImageData.instance.width_Getter_(this);
}
// Copyright (c) 2012, the Dart project authors.  Please see the AUTHORS file
// for details. All rights reserved. Use of this source code is governed by a
// BSD-style license that can be found in the LICENSE file.

@DomName('HTMLImageElement')
class ImageElement extends HtmlElement implements CanvasImageSource {
  // To suppress missing implicit constructor warnings.
  factory ImageElement._() {
    throw new UnsupportedError("Not supported");
  }

  @DomName('HTMLImageElement.HTMLImageElement')
  @DocsEditable()
  factory ImageElement({String src, int width, int height}) {
    ImageElement e = document.createElement("img");
    if (src != null) e.src = src;
    if (width != null) e.width = width;
    if (height != null) e.height = height;
    return e;
  }

  @Deprecated("Internal Use Only")
  external static Type get instanceRuntimeType;

  @Deprecated("Internal Use Only")
  ImageElement.internal_() : super.internal_();

  /**
   * Constructor instantiated by the DOM when a custom element has been created.
   *
   * This can only be called by subclasses from their created constructor.
   */
  ImageElement.created() : super.created();

  @DomName('HTMLImageElement.alt')
  @DocsEditable()
  String get alt => _blink.BlinkHTMLImageElement.instance.alt_Getter_(this);

  @DomName('HTMLImageElement.alt')
  @DocsEditable()
  set alt(String value) =>
      _blink.BlinkHTMLImageElement.instance.alt_Setter_(this, value);

  @DomName('HTMLImageElement.complete')
  @DocsEditable()
  bool get complete =>
      _blink.BlinkHTMLImageElement.instance.complete_Getter_(this);

  @DomName('HTMLImageElement.crossOrigin')
  @DocsEditable()
  String get crossOrigin =>
      _blink.BlinkHTMLImageElement.instance.crossOrigin_Getter_(this);

  @DomName('HTMLImageElement.crossOrigin')
  @DocsEditable()
  set crossOrigin(String value) =>
      _blink.BlinkHTMLImageElement.instance.crossOrigin_Setter_(this, value);

  @DomName('HTMLImageElement.currentSrc')
  @DocsEditable()
  @Experimental() // untriaged
  String get currentSrc =>
      _blink.BlinkHTMLImageElement.instance.currentSrc_Getter_(this);

  @DomName('HTMLImageElement.height')
  @DocsEditable()
  int get height => _blink.BlinkHTMLImageElement.instance.height_Getter_(this);

  @DomName('HTMLImageElement.height')
  @DocsEditable()
  set height(int value) =>
      _blink.BlinkHTMLImageElement.instance.height_Setter_(this, value);

  @DomName('HTMLImageElement.isMap')
  @DocsEditable()
  bool get isMap => _blink.BlinkHTMLImageElement.instance.isMap_Getter_(this);

  @DomName('HTMLImageElement.isMap')
  @DocsEditable()
  set isMap(bool value) =>
      _blink.BlinkHTMLImageElement.instance.isMap_Setter_(this, value);

  @DomName('HTMLImageElement.naturalHeight')
  @DocsEditable()
  int get naturalHeight =>
      _blink.BlinkHTMLImageElement.instance.naturalHeight_Getter_(this);

  @DomName('HTMLImageElement.naturalWidth')
  @DocsEditable()
  int get naturalWidth =>
      _blink.BlinkHTMLImageElement.instance.naturalWidth_Getter_(this);

  @DomName('HTMLImageElement.sizes')
  @DocsEditable()
  @Experimental() // untriaged
  String get sizes => _blink.BlinkHTMLImageElement.instance.sizes_Getter_(this);

  @DomName('HTMLImageElement.sizes')
  @DocsEditable()
  @Experimental() // untriaged
  set sizes(String value) =>
      _blink.BlinkHTMLImageElement.instance.sizes_Setter_(this, value);

  @DomName('HTMLImageElement.src')
  @DocsEditable()
  String get src => _blink.BlinkHTMLImageElement.instance.src_Getter_(this);

  @DomName('HTMLImageElement.src')
  @DocsEditable()
  set src(String value) =>
      _blink.BlinkHTMLImageElement.instance.src_Setter_(this, value);

  @DomName('HTMLImageElement.srcset')
  @DocsEditable()
  @Experimental() // untriaged
  String get srcset =>
      _blink.BlinkHTMLImageElement.instance.srcset_Getter_(this);

  @DomName('HTMLImageElement.srcset')
  @DocsEditable()
  @Experimental() // untriaged
  set srcset(String value) =>
      _blink.BlinkHTMLImageElement.instance.srcset_Setter_(this, value);

  @DomName('HTMLImageElement.useMap')
  @DocsEditable()
  String get useMap =>
      _blink.BlinkHTMLImageElement.instance.useMap_Getter_(this);

  @DomName('HTMLImageElement.useMap')
  @DocsEditable()
  set useMap(String value) =>
      _blink.BlinkHTMLImageElement.instance.useMap_Setter_(this, value);

  @DomName('HTMLImageElement.width')
  @DocsEditable()
  int get width => _blink.BlinkHTMLImageElement.instance.width_Getter_(this);

  @DomName('HTMLImageElement.width')
  @DocsEditable()
  set width(int value) =>
      _blink.BlinkHTMLImageElement.instance.width_Setter_(this, value);
}
// Copyright (c) 2012, the Dart project authors.  Please see the AUTHORS file
// for details. All rights reserved. Use of this source code is governed by a
// BSD-style license that can be found in the LICENSE file.

// WARNING: Do not edit - generated code.

@DocsEditable()
@DomName('InjectedScriptHost')
@Experimental() // untriaged
class InjectedScriptHost extends DartHtmlDomObject {
  // To suppress missing implicit constructor warnings.
  factory InjectedScriptHost._() {
    throw new UnsupportedError("Not supported");
  }

  @Deprecated("Internal Use Only")
  external static Type get instanceRuntimeType;

  @Deprecated("Internal Use Only")
  InjectedScriptHost.internal_() {}

  @DomName('InjectedScriptHost.inspect')
  @DocsEditable()
  @Experimental() // untriaged
  void inspect(Object objectId, Object hints) =>
      _blink.BlinkInjectedScriptHost.instance
          .inspect_Callback_2_(this, objectId, hints);
}
// Copyright (c) 2012, the Dart project authors.  Please see the AUTHORS file
// for details. All rights reserved. Use of this source code is governed by a
// BSD-style license that can be found in the LICENSE file.

// WARNING: Do not edit - generated code.

@DocsEditable()
@DomName('InputDevice')
@Experimental() // untriaged
class InputDevice extends DartHtmlDomObject {
  // To suppress missing implicit constructor warnings.
  factory InputDevice._() {
    throw new UnsupportedError("Not supported");
  }

  @DomName('InputDevice.InputDevice')
  @DocsEditable()
  factory InputDevice([Map deviceInitDict]) {
    if (deviceInitDict != null) {
      var deviceInitDict_1 = convertDartToNative_Dictionary(deviceInitDict);
      return _blink.BlinkInputDevice.instance
          .constructorCallback_1_(deviceInitDict_1);
    }
    return _blink.BlinkInputDevice.instance.constructorCallback_0_();
  }

  @Deprecated("Internal Use Only")
  external static Type get instanceRuntimeType;

  @Deprecated("Internal Use Only")
  InputDevice.internal_() {}

  @DomName('InputDevice.firesTouchEvents')
  @DocsEditable()
  @Experimental() // untriaged
  bool get firesTouchEvents =>
      _blink.BlinkInputDevice.instance.firesTouchEvents_Getter_(this);
}
// Copyright (c) 2012, the Dart project authors.  Please see the AUTHORS file
// for details. All rights reserved. Use of this source code is governed by a
// BSD-style license that can be found in the LICENSE file.

@DomName('HTMLInputElement')
class InputElement extends HtmlElement
    implements
        HiddenInputElement,
        SearchInputElement,
        TextInputElement,
        UrlInputElement,
        TelephoneInputElement,
        EmailInputElement,
        PasswordInputElement,
        DateInputElement,
        MonthInputElement,
        WeekInputElement,
        TimeInputElement,
        LocalDateTimeInputElement,
        NumberInputElement,
        RangeInputElement,
        CheckboxInputElement,
        RadioButtonInputElement,
        FileUploadInputElement,
        SubmitButtonInputElement,
        ImageButtonInputElement,
        ResetButtonInputElement,
        ButtonInputElement {
  factory InputElement({String type}) {
    InputElement e = document.createElement("input");
    if (type != null) {
      try {
        // IE throws an exception for unknown types.
        e.type = type;
      } catch (_) {}
    }
    return e;
  }

  // To suppress missing implicit constructor warnings.
  factory InputElement._() {
    throw new UnsupportedError("Not supported");
  }

  @Deprecated("Internal Use Only")
  external static Type get instanceRuntimeType;

  @Deprecated("Internal Use Only")
  InputElement.internal_() : super.internal_();

  /**
   * Constructor instantiated by the DOM when a custom element has been created.
   *
   * This can only be called by subclasses from their created constructor.
   */
  InputElement.created() : super.created();

  @DomName('HTMLInputElement.accept')
  @DocsEditable()
  String get accept =>
      _blink.BlinkHTMLInputElement.instance.accept_Getter_(this);

  @DomName('HTMLInputElement.accept')
  @DocsEditable()
  set accept(String value) =>
      _blink.BlinkHTMLInputElement.instance.accept_Setter_(this, value);

  @DomName('HTMLInputElement.alt')
  @DocsEditable()
  String get alt => _blink.BlinkHTMLInputElement.instance.alt_Getter_(this);

  @DomName('HTMLInputElement.alt')
  @DocsEditable()
  set alt(String value) =>
      _blink.BlinkHTMLInputElement.instance.alt_Setter_(this, value);

  @DomName('HTMLInputElement.autocapitalize')
  @DocsEditable()
  @Experimental() // untriaged
  String get autocapitalize =>
      _blink.BlinkHTMLInputElement.instance.autocapitalize_Getter_(this);

  @DomName('HTMLInputElement.autocapitalize')
  @DocsEditable()
  @Experimental() // untriaged
  set autocapitalize(String value) =>
      _blink.BlinkHTMLInputElement.instance.autocapitalize_Setter_(this, value);

  @DomName('HTMLInputElement.autocomplete')
  @DocsEditable()
  String get autocomplete =>
      _blink.BlinkHTMLInputElement.instance.autocomplete_Getter_(this);

  @DomName('HTMLInputElement.autocomplete')
  @DocsEditable()
  set autocomplete(String value) =>
      _blink.BlinkHTMLInputElement.instance.autocomplete_Setter_(this, value);

  @DomName('HTMLInputElement.autofocus')
  @DocsEditable()
  bool get autofocus =>
      _blink.BlinkHTMLInputElement.instance.autofocus_Getter_(this);

  @DomName('HTMLInputElement.autofocus')
  @DocsEditable()
  set autofocus(bool value) =>
      _blink.BlinkHTMLInputElement.instance.autofocus_Setter_(this, value);

  @DomName('HTMLInputElement.capture')
  @DocsEditable()
  @Experimental() // untriaged
  bool get capture =>
      _blink.BlinkHTMLInputElement.instance.capture_Getter_(this);

  @DomName('HTMLInputElement.capture')
  @DocsEditable()
  @Experimental() // untriaged
  set capture(bool value) =>
      _blink.BlinkHTMLInputElement.instance.capture_Setter_(this, value);

  @DomName('HTMLInputElement.checked')
  @DocsEditable()
  bool get checked =>
      _blink.BlinkHTMLInputElement.instance.checked_Getter_(this);

  @DomName('HTMLInputElement.checked')
  @DocsEditable()
  set checked(bool value) =>
      _blink.BlinkHTMLInputElement.instance.checked_Setter_(this, value);

  @DomName('HTMLInputElement.defaultChecked')
  @DocsEditable()
  bool get defaultChecked =>
      _blink.BlinkHTMLInputElement.instance.defaultChecked_Getter_(this);

  @DomName('HTMLInputElement.defaultChecked')
  @DocsEditable()
  set defaultChecked(bool value) =>
      _blink.BlinkHTMLInputElement.instance.defaultChecked_Setter_(this, value);

  @DomName('HTMLInputElement.defaultValue')
  @DocsEditable()
  String get defaultValue =>
      _blink.BlinkHTMLInputElement.instance.defaultValue_Getter_(this);

  @DomName('HTMLInputElement.defaultValue')
  @DocsEditable()
  set defaultValue(String value) =>
      _blink.BlinkHTMLInputElement.instance.defaultValue_Setter_(this, value);

  @DomName('HTMLInputElement.dirName')
  @DocsEditable()
  String get dirName =>
      _blink.BlinkHTMLInputElement.instance.dirName_Getter_(this);

  @DomName('HTMLInputElement.dirName')
  @DocsEditable()
  set dirName(String value) =>
      _blink.BlinkHTMLInputElement.instance.dirName_Setter_(this, value);

  @DomName('HTMLInputElement.disabled')
  @DocsEditable()
  bool get disabled =>
      _blink.BlinkHTMLInputElement.instance.disabled_Getter_(this);

  @DomName('HTMLInputElement.disabled')
  @DocsEditable()
  set disabled(bool value) =>
      _blink.BlinkHTMLInputElement.instance.disabled_Setter_(this, value);

  @DomName('HTMLInputElement.files')
  @DocsEditable()
  List<File> get files =>
      (_blink.BlinkHTMLInputElement.instance.files_Getter_(this));

  @DomName('HTMLInputElement.files')
  @DocsEditable()
  set files(List<File> value) =>
      _blink.BlinkHTMLInputElement.instance.files_Setter_(this, value);

  @DomName('HTMLInputElement.form')
  @DocsEditable()
  FormElement get form =>
      _blink.BlinkHTMLInputElement.instance.form_Getter_(this);

  @DomName('HTMLInputElement.formAction')
  @DocsEditable()
  String get formAction =>
      _blink.BlinkHTMLInputElement.instance.formAction_Getter_(this);

  @DomName('HTMLInputElement.formAction')
  @DocsEditable()
  set formAction(String value) =>
      _blink.BlinkHTMLInputElement.instance.formAction_Setter_(this, value);

  @DomName('HTMLInputElement.formEnctype')
  @DocsEditable()
  String get formEnctype =>
      _blink.BlinkHTMLInputElement.instance.formEnctype_Getter_(this);

  @DomName('HTMLInputElement.formEnctype')
  @DocsEditable()
  set formEnctype(String value) =>
      _blink.BlinkHTMLInputElement.instance.formEnctype_Setter_(this, value);

  @DomName('HTMLInputElement.formMethod')
  @DocsEditable()
  String get formMethod =>
      _blink.BlinkHTMLInputElement.instance.formMethod_Getter_(this);

  @DomName('HTMLInputElement.formMethod')
  @DocsEditable()
  set formMethod(String value) =>
      _blink.BlinkHTMLInputElement.instance.formMethod_Setter_(this, value);

  @DomName('HTMLInputElement.formNoValidate')
  @DocsEditable()
  bool get formNoValidate =>
      _blink.BlinkHTMLInputElement.instance.formNoValidate_Getter_(this);

  @DomName('HTMLInputElement.formNoValidate')
  @DocsEditable()
  set formNoValidate(bool value) =>
      _blink.BlinkHTMLInputElement.instance.formNoValidate_Setter_(this, value);

  @DomName('HTMLInputElement.formTarget')
  @DocsEditable()
  String get formTarget =>
      _blink.BlinkHTMLInputElement.instance.formTarget_Getter_(this);

  @DomName('HTMLInputElement.formTarget')
  @DocsEditable()
  set formTarget(String value) =>
      _blink.BlinkHTMLInputElement.instance.formTarget_Setter_(this, value);

  @DomName('HTMLInputElement.height')
  @DocsEditable()
  int get height => _blink.BlinkHTMLInputElement.instance.height_Getter_(this);

  @DomName('HTMLInputElement.height')
  @DocsEditable()
  set height(int value) =>
      _blink.BlinkHTMLInputElement.instance.height_Setter_(this, value);

  @DomName('HTMLInputElement.incremental')
  @DocsEditable()
  // http://www.w3.org/TR/html-markup/input.search.html
  @Experimental()
  bool get incremental =>
      _blink.BlinkHTMLInputElement.instance.incremental_Getter_(this);

  @DomName('HTMLInputElement.incremental')
  @DocsEditable()
  // http://www.w3.org/TR/html-markup/input.search.html
  @Experimental()
  set incremental(bool value) =>
      _blink.BlinkHTMLInputElement.instance.incremental_Setter_(this, value);

  @DomName('HTMLInputElement.indeterminate')
  @DocsEditable()
  bool get indeterminate =>
      _blink.BlinkHTMLInputElement.instance.indeterminate_Getter_(this);

  @DomName('HTMLInputElement.indeterminate')
  @DocsEditable()
  set indeterminate(bool value) =>
      _blink.BlinkHTMLInputElement.instance.indeterminate_Setter_(this, value);

  @DomName('HTMLInputElement.inputMode')
  @DocsEditable()
  @Experimental() // untriaged
  String get inputMode =>
      _blink.BlinkHTMLInputElement.instance.inputMode_Getter_(this);

  @DomName('HTMLInputElement.inputMode')
  @DocsEditable()
  @Experimental() // untriaged
  set inputMode(String value) =>
      _blink.BlinkHTMLInputElement.instance.inputMode_Setter_(this, value);

  @DomName('HTMLInputElement.labels')
  @DocsEditable()
  List<Node> get labels =>
      (_blink.BlinkHTMLInputElement.instance.labels_Getter_(this));

  @DomName('HTMLInputElement.list')
  @DocsEditable()
  HtmlElement get list =>
      _blink.BlinkHTMLInputElement.instance.list_Getter_(this);

  @DomName('HTMLInputElement.max')
  @DocsEditable()
  String get max => _blink.BlinkHTMLInputElement.instance.max_Getter_(this);

  @DomName('HTMLInputElement.max')
  @DocsEditable()
  set max(String value) =>
      _blink.BlinkHTMLInputElement.instance.max_Setter_(this, value);

  @DomName('HTMLInputElement.maxLength')
  @DocsEditable()
  int get maxLength =>
      _blink.BlinkHTMLInputElement.instance.maxLength_Getter_(this);

  @DomName('HTMLInputElement.maxLength')
  @DocsEditable()
  set maxLength(int value) =>
      _blink.BlinkHTMLInputElement.instance.maxLength_Setter_(this, value);

  @DomName('HTMLInputElement.min')
  @DocsEditable()
  String get min => _blink.BlinkHTMLInputElement.instance.min_Getter_(this);

  @DomName('HTMLInputElement.min')
  @DocsEditable()
  set min(String value) =>
      _blink.BlinkHTMLInputElement.instance.min_Setter_(this, value);

  @DomName('HTMLInputElement.minLength')
  @DocsEditable()
  @Experimental() // untriaged
  int get minLength =>
      _blink.BlinkHTMLInputElement.instance.minLength_Getter_(this);

  @DomName('HTMLInputElement.minLength')
  @DocsEditable()
  @Experimental() // untriaged
  set minLength(int value) =>
      _blink.BlinkHTMLInputElement.instance.minLength_Setter_(this, value);

  @DomName('HTMLInputElement.multiple')
  @DocsEditable()
  bool get multiple =>
      _blink.BlinkHTMLInputElement.instance.multiple_Getter_(this);

  @DomName('HTMLInputElement.multiple')
  @DocsEditable()
  set multiple(bool value) =>
      _blink.BlinkHTMLInputElement.instance.multiple_Setter_(this, value);

  @DomName('HTMLInputElement.name')
  @DocsEditable()
  String get name => _blink.BlinkHTMLInputElement.instance.name_Getter_(this);

  @DomName('HTMLInputElement.name')
  @DocsEditable()
  set name(String value) =>
      _blink.BlinkHTMLInputElement.instance.name_Setter_(this, value);

  @DomName('HTMLInputElement.pattern')
  @DocsEditable()
  String get pattern =>
      _blink.BlinkHTMLInputElement.instance.pattern_Getter_(this);

  @DomName('HTMLInputElement.pattern')
  @DocsEditable()
  set pattern(String value) =>
      _blink.BlinkHTMLInputElement.instance.pattern_Setter_(this, value);

  @DomName('HTMLInputElement.placeholder')
  @DocsEditable()
  String get placeholder =>
      _blink.BlinkHTMLInputElement.instance.placeholder_Getter_(this);

  @DomName('HTMLInputElement.placeholder')
  @DocsEditable()
  set placeholder(String value) =>
      _blink.BlinkHTMLInputElement.instance.placeholder_Setter_(this, value);

  @DomName('HTMLInputElement.readOnly')
  @DocsEditable()
  bool get readOnly =>
      _blink.BlinkHTMLInputElement.instance.readOnly_Getter_(this);

  @DomName('HTMLInputElement.readOnly')
  @DocsEditable()
  set readOnly(bool value) =>
      _blink.BlinkHTMLInputElement.instance.readOnly_Setter_(this, value);

  @DomName('HTMLInputElement.required')
  @DocsEditable()
  bool get required =>
      _blink.BlinkHTMLInputElement.instance.required_Getter_(this);

  @DomName('HTMLInputElement.required')
  @DocsEditable()
  set required(bool value) =>
      _blink.BlinkHTMLInputElement.instance.required_Setter_(this, value);

  @DomName('HTMLInputElement.selectionDirection')
  @DocsEditable()
  String get selectionDirection =>
      _blink.BlinkHTMLInputElement.instance.selectionDirection_Getter_(this);

  @DomName('HTMLInputElement.selectionDirection')
  @DocsEditable()
  set selectionDirection(String value) => _blink.BlinkHTMLInputElement.instance
      .selectionDirection_Setter_(this, value);

  @DomName('HTMLInputElement.selectionEnd')
  @DocsEditable()
  int get selectionEnd =>
      _blink.BlinkHTMLInputElement.instance.selectionEnd_Getter_(this);

  @DomName('HTMLInputElement.selectionEnd')
  @DocsEditable()
  set selectionEnd(int value) =>
      _blink.BlinkHTMLInputElement.instance.selectionEnd_Setter_(this, value);

  @DomName('HTMLInputElement.selectionStart')
  @DocsEditable()
  int get selectionStart =>
      _blink.BlinkHTMLInputElement.instance.selectionStart_Getter_(this);

  @DomName('HTMLInputElement.selectionStart')
  @DocsEditable()
  set selectionStart(int value) =>
      _blink.BlinkHTMLInputElement.instance.selectionStart_Setter_(this, value);

  @DomName('HTMLInputElement.size')
  @DocsEditable()
  int get size => _blink.BlinkHTMLInputElement.instance.size_Getter_(this);

  @DomName('HTMLInputElement.size')
  @DocsEditable()
  set size(int value) =>
      _blink.BlinkHTMLInputElement.instance.size_Setter_(this, value);

  @DomName('HTMLInputElement.src')
  @DocsEditable()
  String get src => _blink.BlinkHTMLInputElement.instance.src_Getter_(this);

  @DomName('HTMLInputElement.src')
  @DocsEditable()
  set src(String value) =>
      _blink.BlinkHTMLInputElement.instance.src_Setter_(this, value);

  @DomName('HTMLInputElement.step')
  @DocsEditable()
  String get step => _blink.BlinkHTMLInputElement.instance.step_Getter_(this);

  @DomName('HTMLInputElement.step')
  @DocsEditable()
  set step(String value) =>
      _blink.BlinkHTMLInputElement.instance.step_Setter_(this, value);

  @DomName('HTMLInputElement.type')
  @DocsEditable()
  String get type => _blink.BlinkHTMLInputElement.instance.type_Getter_(this);

  @DomName('HTMLInputElement.type')
  @DocsEditable()
  set type(String value) =>
      _blink.BlinkHTMLInputElement.instance.type_Setter_(this, value);

  @DomName('HTMLInputElement.validationMessage')
  @DocsEditable()
  String get validationMessage =>
      _blink.BlinkHTMLInputElement.instance.validationMessage_Getter_(this);

  @DomName('HTMLInputElement.validity')
  @DocsEditable()
  ValidityState get validity =>
      _blink.BlinkHTMLInputElement.instance.validity_Getter_(this);

  @DomName('HTMLInputElement.value')
  @DocsEditable()
  String get value => _blink.BlinkHTMLInputElement.instance.value_Getter_(this);

  @DomName('HTMLInputElement.value')
  @DocsEditable()
  set value(String value) =>
      _blink.BlinkHTMLInputElement.instance.value_Setter_(this, value);

  @DomName('HTMLInputElement.valueAsDate')
  @DocsEditable()
  DateTime get valueAsDate =>
      _blink.BlinkHTMLInputElement.instance.valueAsDate_Getter_(this);

  @DomName('HTMLInputElement.valueAsDate')
  @DocsEditable()
  set valueAsDate(DateTime value) =>
      _blink.BlinkHTMLInputElement.instance.valueAsDate_Setter_(this, value);

  @DomName('HTMLInputElement.valueAsNumber')
  @DocsEditable()
  num get valueAsNumber =>
      _blink.BlinkHTMLInputElement.instance.valueAsNumber_Getter_(this);

  @DomName('HTMLInputElement.valueAsNumber')
  @DocsEditable()
  set valueAsNumber(num value) =>
      _blink.BlinkHTMLInputElement.instance.valueAsNumber_Setter_(this, value);

  @DomName('HTMLInputElement.webkitEntries')
  @DocsEditable()
  @SupportedBrowser(SupportedBrowser.CHROME)
  @SupportedBrowser(SupportedBrowser.SAFARI)
  @Experimental()
  // http://www.whatwg.org/specs/web-apps/current-work/multipage/states-of-the-type-attribute.html#concept-input-type-file-selected
  List<Entry> get entries =>
      (_blink.BlinkHTMLInputElement.instance.webkitEntries_Getter_(this));

  @DomName('HTMLInputElement.webkitdirectory')
  @DocsEditable()
  @SupportedBrowser(SupportedBrowser.CHROME)
  @SupportedBrowser(SupportedBrowser.SAFARI)
  @Experimental()
  // https://plus.sandbox.google.com/+AddyOsmani/posts/Dk5UhZ6zfF3
  bool get directory =>
      _blink.BlinkHTMLInputElement.instance.webkitdirectory_Getter_(this);

  @DomName('HTMLInputElement.webkitdirectory')
  @DocsEditable()
  @SupportedBrowser(SupportedBrowser.CHROME)
  @SupportedBrowser(SupportedBrowser.SAFARI)
  @Experimental()
  // https://plus.sandbox.google.com/+AddyOsmani/posts/Dk5UhZ6zfF3
  set directory(bool value) => _blink.BlinkHTMLInputElement.instance
      .webkitdirectory_Setter_(this, value);

  @DomName('HTMLInputElement.width')
  @DocsEditable()
  int get width => _blink.BlinkHTMLInputElement.instance.width_Getter_(this);

  @DomName('HTMLInputElement.width')
  @DocsEditable()
  set width(int value) =>
      _blink.BlinkHTMLInputElement.instance.width_Setter_(this, value);

  @DomName('HTMLInputElement.willValidate')
  @DocsEditable()
  bool get willValidate =>
      _blink.BlinkHTMLInputElement.instance.willValidate_Getter_(this);

  @DomName('HTMLInputElement.checkValidity')
  @DocsEditable()
  bool checkValidity() =>
      _blink.BlinkHTMLInputElement.instance.checkValidity_Callback_0_(this);

  @DomName('HTMLInputElement.reportValidity')
  @DocsEditable()
  @Experimental() // untriaged
  bool reportValidity() =>
      _blink.BlinkHTMLInputElement.instance.reportValidity_Callback_0_(this);

  @DomName('HTMLInputElement.select')
  @DocsEditable()
  void select() =>
      _blink.BlinkHTMLInputElement.instance.select_Callback_0_(this);

  @DomName('HTMLInputElement.setCustomValidity')
  @DocsEditable()
  void setCustomValidity(String error) => _blink.BlinkHTMLInputElement.instance
      .setCustomValidity_Callback_1_(this, error);

  void setRangeText(String replacement,
      {int start, int end, String selectionMode}) {
    if ((replacement is String || replacement == null) &&
        start == null &&
        end == null &&
        selectionMode == null) {
      _blink.BlinkHTMLInputElement.instance
          .setRangeText_Callback_1_(this, replacement);
      return;
    }
    if ((end is int || end == null) &&
        (start is int || start == null) &&
        (replacement is String || replacement == null) &&
        selectionMode == null) {
      _blink.BlinkHTMLInputElement.instance
          .setRangeText_Callback_3_(this, replacement, start, end);
      return;
    }
    if ((selectionMode is String || selectionMode == null) &&
        (end is int || end == null) &&
        (start is int || start == null) &&
        (replacement is String || replacement == null)) {
      _blink.BlinkHTMLInputElement.instance.setRangeText_Callback_4_(
          this, replacement, start, end, selectionMode);
      return;
    }
    throw new ArgumentError("Incorrect number or type of arguments");
  }

  void setSelectionRange(int start, int end, [String direction]) {
    if (direction != null) {
      _blink.BlinkHTMLInputElement.instance
          .setSelectionRange_Callback_3_(this, start, end, direction);
      return;
    }
    _blink.BlinkHTMLInputElement.instance
        .setSelectionRange_Callback_2_(this, start, end);
    return;
  }

  void stepDown([int n]) {
    if (n != null) {
      _blink.BlinkHTMLInputElement.instance.stepDown_Callback_1_(this, n);
      return;
    }
    _blink.BlinkHTMLInputElement.instance.stepDown_Callback_0_(this);
    return;
  }

  void stepUp([int n]) {
    if (n != null) {
      _blink.BlinkHTMLInputElement.instance.stepUp_Callback_1_(this, n);
      return;
    }
    _blink.BlinkHTMLInputElement.instance.stepUp_Callback_0_(this);
    return;
  }
}

// Interfaces representing the InputElement APIs which are supported
// for the various types of InputElement. From:
// http://www.w3.org/html/wg/drafts/html/master/forms.html#the-input-element.

/**
 * Exposes the functionality common between all InputElement types.
 */
abstract class InputElementBase implements Element {
  @DomName('HTMLInputElement.autofocus')
  bool autofocus;

  @DomName('HTMLInputElement.disabled')
  bool disabled;

  @DomName('HTMLInputElement.incremental')
  bool incremental;

  @DomName('HTMLInputElement.indeterminate')
  bool indeterminate;

  @DomName('HTMLInputElement.labels')
  List<Node> get labels;

  @DomName('HTMLInputElement.name')
  String name;

  @DomName('HTMLInputElement.validationMessage')
  String get validationMessage;

  @DomName('HTMLInputElement.validity')
  ValidityState get validity;

  @DomName('HTMLInputElement.value')
  String value;

  @DomName('HTMLInputElement.willValidate')
  bool get willValidate;

  @DomName('HTMLInputElement.checkValidity')
  bool checkValidity();

  @DomName('HTMLInputElement.setCustomValidity')
  void setCustomValidity(String error);
}

/**
 * Hidden input which is not intended to be seen or edited by the user.
 */
abstract class HiddenInputElement implements InputElementBase {
  factory HiddenInputElement() => new InputElement(type: 'hidden');
}

/**
 * Base interface for all inputs which involve text editing.
 */
abstract class TextInputElementBase implements InputElementBase {
  @DomName('HTMLInputElement.autocomplete')
  String autocomplete;

  @DomName('HTMLInputElement.maxLength')
  int maxLength;

  @DomName('HTMLInputElement.pattern')
  String pattern;

  @DomName('HTMLInputElement.placeholder')
  String placeholder;

  @DomName('HTMLInputElement.readOnly')
  bool readOnly;

  @DomName('HTMLInputElement.required')
  bool required;

  @DomName('HTMLInputElement.size')
  int size;

  @DomName('HTMLInputElement.select')
  void select();

  @DomName('HTMLInputElement.selectionDirection')
  String selectionDirection;

  @DomName('HTMLInputElement.selectionEnd')
  int selectionEnd;

  @DomName('HTMLInputElement.selectionStart')
  int selectionStart;

  @DomName('HTMLInputElement.setSelectionRange')
  void setSelectionRange(int start, int end, [String direction]);
}

/**
 * Similar to [TextInputElement], but on platforms where search is styled
 * differently this will get the search style.
 *
 * Use [supported] to check if this is supported on the current platform.
 */
@SupportedBrowser(SupportedBrowser.CHROME)
@SupportedBrowser(SupportedBrowser.FIREFOX)
@SupportedBrowser(SupportedBrowser.IE, '10')
@SupportedBrowser(SupportedBrowser.SAFARI)
abstract class SearchInputElement implements TextInputElementBase {
  factory SearchInputElement() => new InputElement(type: 'search');

  @DomName('HTMLInputElement.dirName')
  String dirName;

  @DomName('HTMLInputElement.list')
  Element get list;

  /// Returns true if this input type is supported on the current platform.
  static bool get supported {
    return (new InputElement(type: 'search')).type == 'search';
  }
}

/**
 * A basic text input editor control.
 */
abstract class TextInputElement implements TextInputElementBase {
  factory TextInputElement() => new InputElement(type: 'text');

  @DomName('HTMLInputElement.dirName')
  String dirName;

  @DomName('HTMLInputElement.list')
  Element get list;
}

/**
 * A control for editing an absolute URL.
 *
 * Use [supported] to check if this is supported on the current platform.
 */
@SupportedBrowser(SupportedBrowser.CHROME)
@SupportedBrowser(SupportedBrowser.FIREFOX)
@SupportedBrowser(SupportedBrowser.IE, '10')
@SupportedBrowser(SupportedBrowser.SAFARI)
abstract class UrlInputElement implements TextInputElementBase {
  factory UrlInputElement() => new InputElement(type: 'url');

  @DomName('HTMLInputElement.list')
  Element get list;

  /// Returns true if this input type is supported on the current platform.
  static bool get supported {
    return (new InputElement(type: 'url')).type == 'url';
  }
}

/**
 * Represents a control for editing a telephone number.
 *
 * This provides a single line of text with minimal formatting help since
 * there is a wide variety of telephone numbers.
 *
 * Use [supported] to check if this is supported on the current platform.
 */
@SupportedBrowser(SupportedBrowser.CHROME)
@SupportedBrowser(SupportedBrowser.FIREFOX)
@SupportedBrowser(SupportedBrowser.IE, '10')
@SupportedBrowser(SupportedBrowser.SAFARI)
abstract class TelephoneInputElement implements TextInputElementBase {
  factory TelephoneInputElement() => new InputElement(type: 'tel');

  @DomName('HTMLInputElement.list')
  Element get list;

  /// Returns true if this input type is supported on the current platform.
  static bool get supported {
    return (new InputElement(type: 'tel')).type == 'tel';
  }
}

/**
 * An e-mail address or list of e-mail addresses.
 *
 * Use [supported] to check if this is supported on the current platform.
 */
@SupportedBrowser(SupportedBrowser.CHROME)
@SupportedBrowser(SupportedBrowser.FIREFOX)
@SupportedBrowser(SupportedBrowser.IE, '10')
@SupportedBrowser(SupportedBrowser.SAFARI)
abstract class EmailInputElement implements TextInputElementBase {
  factory EmailInputElement() => new InputElement(type: 'email');

  @DomName('HTMLInputElement.autocomplete')
  String autocomplete;

  @DomName('HTMLInputElement.autofocus')
  bool autofocus;

  @DomName('HTMLInputElement.list')
  Element get list;

  @DomName('HTMLInputElement.maxLength')
  int maxLength;

  @DomName('HTMLInputElement.multiple')
  bool multiple;

  @DomName('HTMLInputElement.pattern')
  String pattern;

  @DomName('HTMLInputElement.placeholder')
  String placeholder;

  @DomName('HTMLInputElement.readOnly')
  bool readOnly;

  @DomName('HTMLInputElement.required')
  bool required;

  @DomName('HTMLInputElement.size')
  int size;

  /// Returns true if this input type is supported on the current platform.
  static bool get supported {
    return (new InputElement(type: 'email')).type == 'email';
  }
}

/**
 * Text with no line breaks (sensitive information).
 */
abstract class PasswordInputElement implements TextInputElementBase {
  factory PasswordInputElement() => new InputElement(type: 'password');
}

/**
 * Base interface for all input element types which involve ranges.
 */
abstract class RangeInputElementBase implements InputElementBase {
  @DomName('HTMLInputElement.list')
  Element get list;

  @DomName('HTMLInputElement.max')
  String max;

  @DomName('HTMLInputElement.min')
  String min;

  @DomName('HTMLInputElement.step')
  String step;

  @DomName('HTMLInputElement.valueAsNumber')
  num valueAsNumber;

  @DomName('HTMLInputElement.stepDown')
  void stepDown([int n]);

  @DomName('HTMLInputElement.stepUp')
  void stepUp([int n]);
}

/**
 * A date (year, month, day) with no time zone.
 *
 * Use [supported] to check if this is supported on the current platform.
 */
@SupportedBrowser(SupportedBrowser.CHROME, '25')
@Experimental()
abstract class DateInputElement implements RangeInputElementBase {
  factory DateInputElement() => new InputElement(type: 'date');

  @DomName('HTMLInputElement.valueAsDate')
  DateTime valueAsDate;

  @DomName('HTMLInputElement.readOnly')
  bool readOnly;

  @DomName('HTMLInputElement.required')
  bool required;

  /// Returns true if this input type is supported on the current platform.
  static bool get supported {
    return (new InputElement(type: 'date')).type == 'date';
  }
}

/**
 * A date consisting of a year and a month with no time zone.
 *
 * Use [supported] to check if this is supported on the current platform.
 */
@SupportedBrowser(SupportedBrowser.CHROME, '25')
@Experimental()
abstract class MonthInputElement implements RangeInputElementBase {
  factory MonthInputElement() => new InputElement(type: 'month');

  @DomName('HTMLInputElement.valueAsDate')
  DateTime valueAsDate;

  @DomName('HTMLInputElement.readOnly')
  bool readOnly;

  @DomName('HTMLInputElement.required')
  bool required;

  /// Returns true if this input type is supported on the current platform.
  static bool get supported {
    return (new InputElement(type: 'month')).type == 'month';
  }
}

/**
 * A date consisting of a week-year number and a week number with no time zone.
 *
 * Use [supported] to check if this is supported on the current platform.
 */
@SupportedBrowser(SupportedBrowser.CHROME, '25')
@Experimental()
abstract class WeekInputElement implements RangeInputElementBase {
  factory WeekInputElement() => new InputElement(type: 'week');

  @DomName('HTMLInputElement.valueAsDate')
  DateTime valueAsDate;

  @DomName('HTMLInputElement.readOnly')
  bool readOnly;

  @DomName('HTMLInputElement.required')
  bool required;

  /// Returns true if this input type is supported on the current platform.
  static bool get supported {
    return (new InputElement(type: 'week')).type == 'week';
  }
}

/**
 * A time (hour, minute, seconds, fractional seconds) with no time zone.
 *
 * Use [supported] to check if this is supported on the current platform.
 */
@SupportedBrowser(SupportedBrowser.CHROME)
@Experimental()
abstract class TimeInputElement implements RangeInputElementBase {
  factory TimeInputElement() => new InputElement(type: 'time');

  @DomName('HTMLInputElement.valueAsDate')
  DateTime valueAsDate;

  @DomName('HTMLInputElement.readOnly')
  bool readOnly;

  @DomName('HTMLInputElement.required')
  bool required;

  /// Returns true if this input type is supported on the current platform.
  static bool get supported {
    return (new InputElement(type: 'time')).type == 'time';
  }
}

/**
 * A date and time (year, month, day, hour, minute, second, fraction of a
 * second) with no time zone.
 *
 * Use [supported] to check if this is supported on the current platform.
 */
@SupportedBrowser(SupportedBrowser.CHROME, '25')
@Experimental()
abstract class LocalDateTimeInputElement implements RangeInputElementBase {
  factory LocalDateTimeInputElement() =>
      new InputElement(type: 'datetime-local');

  @DomName('HTMLInputElement.readOnly')
  bool readOnly;

  @DomName('HTMLInputElement.required')
  bool required;

  /// Returns true if this input type is supported on the current platform.
  static bool get supported {
    return (new InputElement(type: 'datetime-local')).type == 'datetime-local';
  }
}

/**
 * A numeric editor control.
 */
@SupportedBrowser(SupportedBrowser.CHROME)
@SupportedBrowser(SupportedBrowser.IE)
@SupportedBrowser(SupportedBrowser.SAFARI)
@Experimental()
abstract class NumberInputElement implements RangeInputElementBase {
  factory NumberInputElement() => new InputElement(type: 'number');

  @DomName('HTMLInputElement.placeholder')
  String placeholder;

  @DomName('HTMLInputElement.readOnly')
  bool readOnly;

  @DomName('HTMLInputElement.required')
  bool required;

  /// Returns true if this input type is supported on the current platform.
  static bool get supported {
    return (new InputElement(type: 'number')).type == 'number';
  }
}

/**
 * Similar to [NumberInputElement] but the browser may provide more optimal
 * styling (such as a slider control).
 *
 * Use [supported] to check if this is supported on the current platform.
 */
@SupportedBrowser(SupportedBrowser.CHROME)
@SupportedBrowser(SupportedBrowser.IE, '10')
@Experimental()
abstract class RangeInputElement implements RangeInputElementBase {
  factory RangeInputElement() => new InputElement(type: 'range');

  /// Returns true if this input type is supported on the current platform.
  static bool get supported {
    return (new InputElement(type: 'range')).type == 'range';
  }
}

/**
 * A boolean editor control.
 *
 * Note that if [indeterminate] is set then this control is in a third
 * indeterminate state.
 */
abstract class CheckboxInputElement implements InputElementBase {
  factory CheckboxInputElement() => new InputElement(type: 'checkbox');

  @DomName('HTMLInputElement.checked')
  bool checked;

  @DomName('HTMLInputElement.required')
  bool required;
}

/**
 * A control that when used with other [ReadioButtonInputElement] controls
 * forms a radio button group in which only one control can be checked at a
 * time.
 *
 * Radio buttons are considered to be in the same radio button group if:
 *
 * * They are all of type 'radio'.
 * * They all have either the same [FormElement] owner, or no owner.
 * * Their name attributes contain the same name.
 */
abstract class RadioButtonInputElement implements InputElementBase {
  factory RadioButtonInputElement() => new InputElement(type: 'radio');

  @DomName('HTMLInputElement.checked')
  bool checked;

  @DomName('HTMLInputElement.required')
  bool required;
}

/**
 * A control for picking files from the user's computer.
 */
abstract class FileUploadInputElement implements InputElementBase {
  factory FileUploadInputElement() => new InputElement(type: 'file');

  @DomName('HTMLInputElement.accept')
  String accept;

  @DomName('HTMLInputElement.multiple')
  bool multiple;

  @DomName('HTMLInputElement.required')
  bool required;

  @DomName('HTMLInputElement.files')
  List<File> files;
}

/**
 * A button, which when clicked, submits the form.
 */
abstract class SubmitButtonInputElement implements InputElementBase {
  factory SubmitButtonInputElement() => new InputElement(type: 'submit');

  @DomName('HTMLInputElement.formAction')
  String formAction;

  @DomName('HTMLInputElement.formEnctype')
  String formEnctype;

  @DomName('HTMLInputElement.formMethod')
  String formMethod;

  @DomName('HTMLInputElement.formNoValidate')
  bool formNoValidate;

  @DomName('HTMLInputElement.formTarget')
  String formTarget;
}

/**
 * Either an image which the user can select a coordinate to or a form
 * submit button.
 */
abstract class ImageButtonInputElement implements InputElementBase {
  factory ImageButtonInputElement() => new InputElement(type: 'image');

  @DomName('HTMLInputElement.alt')
  String alt;

  @DomName('HTMLInputElement.formAction')
  String formAction;

  @DomName('HTMLInputElement.formEnctype')
  String formEnctype;

  @DomName('HTMLInputElement.formMethod')
  String formMethod;

  @DomName('HTMLInputElement.formNoValidate')
  bool formNoValidate;

  @DomName('HTMLInputElement.formTarget')
  String formTarget;

  @DomName('HTMLInputElement.height')
  int height;

  @DomName('HTMLInputElement.src')
  String src;

  @DomName('HTMLInputElement.width')
  int width;
}

/**
 * A button, which when clicked, resets the form.
 */
abstract class ResetButtonInputElement implements InputElementBase {
  factory ResetButtonInputElement() => new InputElement(type: 'reset');
}

/**
 * A button, with no default behavior.
 */
abstract class ButtonInputElement implements InputElementBase {
  factory ButtonInputElement() => new InputElement(type: 'button');
}
// Copyright (c) 2012, the Dart project authors.  Please see the AUTHORS file
// for details. All rights reserved. Use of this source code is governed by a
// BSD-style license that can be found in the LICENSE file.

@DomName('KeyboardEvent')
class KeyboardEvent extends UIEvent {
  factory KeyboardEvent(String type,
      {Window view,
      bool canBubble: true,
      bool cancelable: true,
      int keyLocation: 1,
      bool ctrlKey: false,
      bool altKey: false,
      bool shiftKey: false,
      bool metaKey: false}) {
    if (view == null) {
      view = window;
    }
    final e = document._createEvent("KeyboardEvent");
    e._initKeyboardEvent(type, canBubble, cancelable, view, "", keyLocation,
        ctrlKey, altKey, shiftKey, metaKey);
    return e;
  }

  @DomName('KeyboardEvent.keyCode')
  int get keyCode => _keyCode;

  @DomName('KeyboardEvent.charCode')
  int get charCode => _charCode;

  @DomName('KeyboardEvent.which')
  int get which => _which;

  @DomName('KeyboardEvent.KeyboardEvent')
  @DocsEditable()
  factory KeyboardEvent._(String type, [Map eventInitDict]) {
    if (eventInitDict != null) {
      var eventInitDict_1 = convertDartToNative_Dictionary(eventInitDict);
      return _blink.BlinkKeyboardEvent.instance
          .constructorCallback_2_(type, eventInitDict_1);
    }
    return _blink.BlinkKeyboardEvent.instance.constructorCallback_1_(type);
  }

  @Deprecated("Internal Use Only")
  external static Type get instanceRuntimeType;

  @Deprecated("Internal Use Only")
  KeyboardEvent.internal_() : super.internal_();

  @DomName('KeyboardEvent.DOM_KEY_LOCATION_LEFT')
  @DocsEditable()
  @Experimental() // untriaged
  static const int DOM_KEY_LOCATION_LEFT = 0x01;

  @DomName('KeyboardEvent.DOM_KEY_LOCATION_NUMPAD')
  @DocsEditable()
  @Experimental() // untriaged
  static const int DOM_KEY_LOCATION_NUMPAD = 0x03;

  @DomName('KeyboardEvent.DOM_KEY_LOCATION_RIGHT')
  @DocsEditable()
  @Experimental() // untriaged
  static const int DOM_KEY_LOCATION_RIGHT = 0x02;

  @DomName('KeyboardEvent.DOM_KEY_LOCATION_STANDARD')
  @DocsEditable()
  @Experimental() // untriaged
  static const int DOM_KEY_LOCATION_STANDARD = 0x00;

  @DomName('KeyboardEvent.altKey')
  @DocsEditable()
  bool get altKey => _blink.BlinkKeyboardEvent.instance.altKey_Getter_(this);

  @DomName('KeyboardEvent.code')
  @DocsEditable()
  @Experimental() // untriaged
  String get code => _blink.BlinkKeyboardEvent.instance.code_Getter_(this);

  @DomName('KeyboardEvent.ctrlKey')
  @DocsEditable()
  bool get ctrlKey => _blink.BlinkKeyboardEvent.instance.ctrlKey_Getter_(this);

  @DomName('KeyboardEvent.key')
  @DocsEditable()
  @Experimental() // untriaged
  String get key => _blink.BlinkKeyboardEvent.instance.key_Getter_(this);

  @DomName('KeyboardEvent.keyIdentifier')
  @DocsEditable()
  @Experimental() // nonstandard
  String get _keyIdentifier =>
      _blink.BlinkKeyboardEvent.instance.keyIdentifier_Getter_(this);

  @DomName('KeyboardEvent.keyLocation')
  @DocsEditable()
  @Experimental() // nonstandard
  int get keyLocation =>
      _blink.BlinkKeyboardEvent.instance.keyLocation_Getter_(this);

  @DomName('KeyboardEvent.location')
  @DocsEditable()
  @Experimental() // untriaged
  int get location => _blink.BlinkKeyboardEvent.instance.location_Getter_(this);

  @DomName('KeyboardEvent.metaKey')
  @DocsEditable()
  bool get metaKey => _blink.BlinkKeyboardEvent.instance.metaKey_Getter_(this);

  @DomName('KeyboardEvent.repeat')
  @DocsEditable()
  @Experimental() // untriaged
  bool get repeat => _blink.BlinkKeyboardEvent.instance.repeat_Getter_(this);

  @DomName('KeyboardEvent.shiftKey')
  @DocsEditable()
  bool get shiftKey =>
      _blink.BlinkKeyboardEvent.instance.shiftKey_Getter_(this);

  @DomName('KeyboardEvent.getModifierState')
  @DocsEditable()
  @Experimental() // untriaged
  bool getModifierState(String keyArg) => _blink.BlinkKeyboardEvent.instance
      .getModifierState_Callback_1_(this, keyArg);

  @DomName('KeyboardEvent.initKeyboardEvent')
  @DocsEditable()
  void _initKeyboardEvent(
          String type,
          bool bubbles,
          bool cancelable,
          Window view,
          String keyIdentifier,
          int location,
          bool ctrlKey,
          bool altKey,
          bool shiftKey,
          bool metaKey) =>
      _blink.BlinkKeyboardEvent.instance.initKeyboardEvent_Callback_10_(
          this,
          type,
          bubbles,
          cancelable,
          view,
          keyIdentifier,
          location,
          ctrlKey,
          altKey,
          shiftKey,
          metaKey);
}
// Copyright (c) 2012, the Dart project authors.  Please see the AUTHORS file
// for details. All rights reserved. Use of this source code is governed by a
// BSD-style license that can be found in the LICENSE file.

// WARNING: Do not edit - generated code.

@DocsEditable()
@DomName('KeyframeEffect')
@Experimental() // untriaged
class KeyframeEffect extends AnimationEffectReadOnly {
  // To suppress missing implicit constructor warnings.
  factory KeyframeEffect._() {
    throw new UnsupportedError("Not supported");
  }

  @DomName('KeyframeEffect.KeyframeEffect')
  @DocsEditable()
  factory KeyframeEffect(Element target, List<Map> keyframes, [timing]) {
    if ((keyframes is List<Map>) &&
        (target is Element || target == null) &&
        timing == null) {
      return _blink.BlinkKeyframeEffect.instance
          .constructorCallback_2_(target, keyframes);
    }
    if ((timing is num) &&
        (keyframes is List<Map>) &&
        (target is Element || target == null)) {
      return _blink.BlinkKeyframeEffect.instance
          .constructorCallback_3_(target, keyframes, timing);
    }
    if ((timing is Map) &&
        (keyframes is List<Map>) &&
        (target is Element || target == null)) {
      var timing_1 = convertDartToNative_Dictionary(timing);
      return _blink.BlinkKeyframeEffect.instance
          .constructorCallback_3_(target, keyframes, timing_1);
    }
    throw new ArgumentError("Incorrect number or type of arguments");
  }

  @Deprecated("Internal Use Only")
  external static Type get instanceRuntimeType;

  @Deprecated("Internal Use Only")
  KeyframeEffect.internal_() : super.internal_();
}
// Copyright (c) 2012, the Dart project authors.  Please see the AUTHORS file
// for details. All rights reserved. Use of this source code is governed by a
// BSD-style license that can be found in the LICENSE file.

// WARNING: Do not edit - generated code.

@DocsEditable()
@DomName('HTMLKeygenElement')
@SupportedBrowser(SupportedBrowser.CHROME)
@SupportedBrowser(SupportedBrowser.SAFARI)
@Experimental()
// http://www.whatwg.org/specs/web-apps/current-work/multipage/the-button-element.html#the-keygen-element
class KeygenElement extends HtmlElement {
  // To suppress missing implicit constructor warnings.
  factory KeygenElement._() {
    throw new UnsupportedError("Not supported");
  }

  @DomName('HTMLKeygenElement.HTMLKeygenElement')
  @DocsEditable()
  factory KeygenElement() => document.createElement("keygen");

  @Deprecated("Internal Use Only")
  external static Type get instanceRuntimeType;

  @Deprecated("Internal Use Only")
  KeygenElement.internal_() : super.internal_();

  /**
   * Constructor instantiated by the DOM when a custom element has been created.
   *
   * This can only be called by subclasses from their created constructor.
   */
  KeygenElement.created() : super.created();

  /// Checks if this type is supported on the current platform.
  static bool get supported => true;

  @DomName('HTMLKeygenElement.autofocus')
  @DocsEditable()
  bool get autofocus =>
      _blink.BlinkHTMLKeygenElement.instance.autofocus_Getter_(this);

  @DomName('HTMLKeygenElement.autofocus')
  @DocsEditable()
  set autofocus(bool value) =>
      _blink.BlinkHTMLKeygenElement.instance.autofocus_Setter_(this, value);

  @DomName('HTMLKeygenElement.challenge')
  @DocsEditable()
  String get challenge =>
      _blink.BlinkHTMLKeygenElement.instance.challenge_Getter_(this);

  @DomName('HTMLKeygenElement.challenge')
  @DocsEditable()
  set challenge(String value) =>
      _blink.BlinkHTMLKeygenElement.instance.challenge_Setter_(this, value);

  @DomName('HTMLKeygenElement.disabled')
  @DocsEditable()
  bool get disabled =>
      _blink.BlinkHTMLKeygenElement.instance.disabled_Getter_(this);

  @DomName('HTMLKeygenElement.disabled')
  @DocsEditable()
  set disabled(bool value) =>
      _blink.BlinkHTMLKeygenElement.instance.disabled_Setter_(this, value);

  @DomName('HTMLKeygenElement.form')
  @DocsEditable()
  FormElement get form =>
      _blink.BlinkHTMLKeygenElement.instance.form_Getter_(this);

  @DomName('HTMLKeygenElement.keytype')
  @DocsEditable()
  String get keytype =>
      _blink.BlinkHTMLKeygenElement.instance.keytype_Getter_(this);

  @DomName('HTMLKeygenElement.keytype')
  @DocsEditable()
  set keytype(String value) =>
      _blink.BlinkHTMLKeygenElement.instance.keytype_Setter_(this, value);

  @DomName('HTMLKeygenElement.labels')
  @DocsEditable()
  @Unstable()
  List<Node> get labels =>
      (_blink.BlinkHTMLKeygenElement.instance.labels_Getter_(this));

  @DomName('HTMLKeygenElement.name')
  @DocsEditable()
  String get name => _blink.BlinkHTMLKeygenElement.instance.name_Getter_(this);

  @DomName('HTMLKeygenElement.name')
  @DocsEditable()
  set name(String value) =>
      _blink.BlinkHTMLKeygenElement.instance.name_Setter_(this, value);

  @DomName('HTMLKeygenElement.type')
  @DocsEditable()
  String get type => _blink.BlinkHTMLKeygenElement.instance.type_Getter_(this);

  @DomName('HTMLKeygenElement.validationMessage')
  @DocsEditable()
  String get validationMessage =>
      _blink.BlinkHTMLKeygenElement.instance.validationMessage_Getter_(this);

  @DomName('HTMLKeygenElement.validity')
  @DocsEditable()
  ValidityState get validity =>
      _blink.BlinkHTMLKeygenElement.instance.validity_Getter_(this);

  @DomName('HTMLKeygenElement.willValidate')
  @DocsEditable()
  bool get willValidate =>
      _blink.BlinkHTMLKeygenElement.instance.willValidate_Getter_(this);

  @DomName('HTMLKeygenElement.checkValidity')
  @DocsEditable()
  bool checkValidity() =>
      _blink.BlinkHTMLKeygenElement.instance.checkValidity_Callback_0_(this);

  @DomName('HTMLKeygenElement.reportValidity')
  @DocsEditable()
  @Experimental() // untriaged
  bool reportValidity() =>
      _blink.BlinkHTMLKeygenElement.instance.reportValidity_Callback_0_(this);

  @DomName('HTMLKeygenElement.setCustomValidity')
  @DocsEditable()
  void setCustomValidity(String error) => _blink.BlinkHTMLKeygenElement.instance
      .setCustomValidity_Callback_1_(this, error);
}
// Copyright (c) 2012, the Dart project authors.  Please see the AUTHORS file
// for details. All rights reserved. Use of this source code is governed by a
// BSD-style license that can be found in the LICENSE file.

// WARNING: Do not edit - generated code.

@DocsEditable()
@DomName('HTMLLIElement')
class LIElement extends HtmlElement {
  // To suppress missing implicit constructor warnings.
  factory LIElement._() {
    throw new UnsupportedError("Not supported");
  }

  @DomName('HTMLLIElement.HTMLLIElement')
  @DocsEditable()
  factory LIElement() => document.createElement("li");

  @Deprecated("Internal Use Only")
  external static Type get instanceRuntimeType;

  @Deprecated("Internal Use Only")
  LIElement.internal_() : super.internal_();

  /**
   * Constructor instantiated by the DOM when a custom element has been created.
   *
   * This can only be called by subclasses from their created constructor.
   */
  LIElement.created() : super.created();

  @DomName('HTMLLIElement.value')
  @DocsEditable()
  int get value => _blink.BlinkHTMLLIElement.instance.value_Getter_(this);

  @DomName('HTMLLIElement.value')
  @DocsEditable()
  set value(int value) =>
      _blink.BlinkHTMLLIElement.instance.value_Setter_(this, value);
}
// Copyright (c) 2012, the Dart project authors.  Please see the AUTHORS file
// for details. All rights reserved. Use of this source code is governed by a
// BSD-style license that can be found in the LICENSE file.

// WARNING: Do not edit - generated code.

@DocsEditable()
@DomName('HTMLLabelElement')
class LabelElement extends HtmlElement {
  // To suppress missing implicit constructor warnings.
  factory LabelElement._() {
    throw new UnsupportedError("Not supported");
  }

  @DomName('HTMLLabelElement.HTMLLabelElement')
  @DocsEditable()
  factory LabelElement() => document.createElement("label");

  @Deprecated("Internal Use Only")
  external static Type get instanceRuntimeType;

  @Deprecated("Internal Use Only")
  LabelElement.internal_() : super.internal_();

  /**
   * Constructor instantiated by the DOM when a custom element has been created.
   *
   * This can only be called by subclasses from their created constructor.
   */
  LabelElement.created() : super.created();

  @DomName('HTMLLabelElement.control')
  @DocsEditable()
  HtmlElement get control =>
      _blink.BlinkHTMLLabelElement.instance.control_Getter_(this);

  @DomName('HTMLLabelElement.form')
  @DocsEditable()
  FormElement get form =>
      _blink.BlinkHTMLLabelElement.instance.form_Getter_(this);

  @DomName('HTMLLabelElement.htmlFor')
  @DocsEditable()
  String get htmlFor =>
      _blink.BlinkHTMLLabelElement.instance.htmlFor_Getter_(this);

  @DomName('HTMLLabelElement.htmlFor')
  @DocsEditable()
  set htmlFor(String value) =>
      _blink.BlinkHTMLLabelElement.instance.htmlFor_Setter_(this, value);
}
// Copyright (c) 2012, the Dart project authors.  Please see the AUTHORS file
// for details. All rights reserved. Use of this source code is governed by a
// BSD-style license that can be found in the LICENSE file.

// WARNING: Do not edit - generated code.

@DocsEditable()
@DomName('HTMLLegendElement')
class LegendElement extends HtmlElement {
  // To suppress missing implicit constructor warnings.
  factory LegendElement._() {
    throw new UnsupportedError("Not supported");
  }

  @DomName('HTMLLegendElement.HTMLLegendElement')
  @DocsEditable()
  factory LegendElement() => document.createElement("legend");

  @Deprecated("Internal Use Only")
  external static Type get instanceRuntimeType;

  @Deprecated("Internal Use Only")
  LegendElement.internal_() : super.internal_();

  /**
   * Constructor instantiated by the DOM when a custom element has been created.
   *
   * This can only be called by subclasses from their created constructor.
   */
  LegendElement.created() : super.created();

  @DomName('HTMLLegendElement.form')
  @DocsEditable()
  FormElement get form =>
      _blink.BlinkHTMLLegendElement.instance.form_Getter_(this);
}
// Copyright (c) 2014, the Dart project authors.  Please see the AUTHORS file
// for details. All rights reserved. Use of this source code is governed by a
// BSD-style license that can be found in the LICENSE file.

@DocsEditable()
@DomName('HTMLLinkElement')
class LinkElement extends HtmlElement {
  // To suppress missing implicit constructor warnings.
  factory LinkElement._() {
    throw new UnsupportedError("Not supported");
  }

  @DomName('HTMLLinkElement.HTMLLinkElement')
  @DocsEditable()
  factory LinkElement() => document.createElement("link");

  @Deprecated("Internal Use Only")
  external static Type get instanceRuntimeType;

  @Deprecated("Internal Use Only")
  LinkElement.internal_() : super.internal_();

  /**
   * Constructor instantiated by the DOM when a custom element has been created.
   *
   * This can only be called by subclasses from their created constructor.
   */
  LinkElement.created() : super.created();

  @DomName('HTMLLinkElement.crossOrigin')
  @DocsEditable()
  @Experimental() // untriaged
  String get crossOrigin =>
      _blink.BlinkHTMLLinkElement.instance.crossOrigin_Getter_(this);

  @DomName('HTMLLinkElement.crossOrigin')
  @DocsEditable()
  @Experimental() // untriaged
  set crossOrigin(String value) =>
      _blink.BlinkHTMLLinkElement.instance.crossOrigin_Setter_(this, value);

  @DomName('HTMLLinkElement.disabled')
  @DocsEditable()
  bool get disabled =>
      _blink.BlinkHTMLLinkElement.instance.disabled_Getter_(this);

  @DomName('HTMLLinkElement.disabled')
  @DocsEditable()
  set disabled(bool value) =>
      _blink.BlinkHTMLLinkElement.instance.disabled_Setter_(this, value);

  @DomName('HTMLLinkElement.href')
  @DocsEditable()
  String get href => _blink.BlinkHTMLLinkElement.instance.href_Getter_(this);

  @DomName('HTMLLinkElement.href')
  @DocsEditable()
  set href(String value) =>
      _blink.BlinkHTMLLinkElement.instance.href_Setter_(this, value);

  @DomName('HTMLLinkElement.hreflang')
  @DocsEditable()
  String get hreflang =>
      _blink.BlinkHTMLLinkElement.instance.hreflang_Getter_(this);

  @DomName('HTMLLinkElement.hreflang')
  @DocsEditable()
  set hreflang(String value) =>
      _blink.BlinkHTMLLinkElement.instance.hreflang_Setter_(this, value);

  @DomName('HTMLLinkElement.import')
  @DocsEditable()
  // https://dvcs.w3.org/hg/webcomponents/raw-file/tip/spec/imports/index.html#interface-import
  @Experimental()
  Document get import =>
      _blink.BlinkHTMLLinkElement.instance.import_Getter_(this);

  @DomName('HTMLLinkElement.integrity')
  @DocsEditable()
  @Experimental() // untriaged
  String get integrity =>
      _blink.BlinkHTMLLinkElement.instance.integrity_Getter_(this);

  @DomName('HTMLLinkElement.integrity')
  @DocsEditable()
  @Experimental() // untriaged
  set integrity(String value) =>
      _blink.BlinkHTMLLinkElement.instance.integrity_Setter_(this, value);

  @DomName('HTMLLinkElement.media')
  @DocsEditable()
  String get media => _blink.BlinkHTMLLinkElement.instance.media_Getter_(this);

  @DomName('HTMLLinkElement.media')
  @DocsEditable()
  set media(String value) =>
      _blink.BlinkHTMLLinkElement.instance.media_Setter_(this, value);

  @DomName('HTMLLinkElement.rel')
  @DocsEditable()
  String get rel => _blink.BlinkHTMLLinkElement.instance.rel_Getter_(this);

  @DomName('HTMLLinkElement.rel')
  @DocsEditable()
  set rel(String value) =>
      _blink.BlinkHTMLLinkElement.instance.rel_Setter_(this, value);

  @DomName('HTMLLinkElement.sheet')
  @DocsEditable()
  StyleSheet get sheet =>
      _blink.BlinkHTMLLinkElement.instance.sheet_Getter_(this);

  @DomName('HTMLLinkElement.sizes')
  @DocsEditable()
  DomSettableTokenList get sizes =>
      _blink.BlinkHTMLLinkElement.instance.sizes_Getter_(this);

  @DomName('HTMLLinkElement.type')
  @DocsEditable()
  String get type => _blink.BlinkHTMLLinkElement.instance.type_Getter_(this);

  @DomName('HTMLLinkElement.type')
  @DocsEditable()
  set type(String value) =>
      _blink.BlinkHTMLLinkElement.instance.type_Setter_(this, value);

  /// Checks if HTML imports are supported on the current platform.
  bool get supportsImport {
    return true;
  }
}
// Copyright (c) 2013, the Dart project authors.  Please see the AUTHORS file
// for details. All rights reserved. Use of this source code is governed by a
// BSD-style license that can be found in the LICENSE file.

@DocsEditable()
@DomName('Location')
class Location extends DartHtmlDomObject implements LocationBase {
  // To suppress missing implicit constructor warnings.
  factory Location._() {
    throw new UnsupportedError("Not supported");
  }

  @Deprecated("Internal Use Only")
  external static Type get instanceRuntimeType;

  @Deprecated("Internal Use Only")
  Location.internal_() {}

  @DomName('Location.ancestorOrigins')
  @DocsEditable()
  @Experimental() // nonstandard
  List<String> get ancestorOrigins =>
      _blink.BlinkLocation.instance.ancestorOrigins_Getter_(this);

  @DomName('Location.hash')
  @DocsEditable()
  String get hash => _blink.BlinkLocation.instance.hash_Getter_(this);

  @DomName('Location.hash')
  @DocsEditable()
  set hash(String value) =>
      _blink.BlinkLocation.instance.hash_Setter_(this, value);

  @DomName('Location.host')
  @DocsEditable()
  String get host => _blink.BlinkLocation.instance.host_Getter_(this);

  @DomName('Location.host')
  @DocsEditable()
  set host(String value) =>
      _blink.BlinkLocation.instance.host_Setter_(this, value);

  @DomName('Location.hostname')
  @DocsEditable()
  String get hostname => _blink.BlinkLocation.instance.hostname_Getter_(this);

  @DomName('Location.hostname')
  @DocsEditable()
  set hostname(String value) =>
      _blink.BlinkLocation.instance.hostname_Setter_(this, value);

  @DomName('Location.href')
  @DocsEditable()
  String get href => _blink.BlinkLocation.instance.href_Getter_(this);

  @DomName('Location.href')
  @DocsEditable()
  set href(String value) =>
      _blink.BlinkLocation.instance.href_Setter_(this, value);

  @DomName('Location.origin')
  @DocsEditable()
  // http://url.spec.whatwg.org/#urlutils Webkit Only
  @Experimental() // non-standard
  String get origin => _blink.BlinkLocation.instance.origin_Getter_(this);

  @DomName('Location.pathname')
  @DocsEditable()
  String get pathname => _blink.BlinkLocation.instance.pathname_Getter_(this);

  @DomName('Location.pathname')
  @DocsEditable()
  set pathname(String value) =>
      _blink.BlinkLocation.instance.pathname_Setter_(this, value);

  @DomName('Location.port')
  @DocsEditable()
  String get port => _blink.BlinkLocation.instance.port_Getter_(this);

  @DomName('Location.port')
  @DocsEditable()
  set port(String value) =>
      _blink.BlinkLocation.instance.port_Setter_(this, value);

  @DomName('Location.protocol')
  @DocsEditable()
  String get protocol => _blink.BlinkLocation.instance.protocol_Getter_(this);

  @DomName('Location.protocol')
  @DocsEditable()
  set protocol(String value) =>
      _blink.BlinkLocation.instance.protocol_Setter_(this, value);

  @DomName('Location.search')
  @DocsEditable()
  String get search => _blink.BlinkLocation.instance.search_Getter_(this);

  @DomName('Location.search')
  @DocsEditable()
  set search(String value) =>
      _blink.BlinkLocation.instance.search_Setter_(this, value);

  @DomName('Location.assign')
  @DocsEditable()
  void assign([String url]) =>
      _blink.BlinkLocation.instance.assign_Callback_1_(this, url);

  @DomName('Location.reload')
  @DocsEditable()
  void reload() => _blink.BlinkLocation.instance.reload_Callback_0_(this);

  @DomName('Location.replace')
  @DocsEditable()
  void replace(String url) =>
      _blink.BlinkLocation.instance.replace_Callback_1_(this, url);

  @DomName('Location.toString')
  @DocsEditable()
  String toString() => _blink.BlinkLocation.instance.toString_Callback_0_(this);
}
// Copyright (c) 2012, the Dart project authors.  Please see the AUTHORS file
// for details. All rights reserved. Use of this source code is governed by a
// BSD-style license that can be found in the LICENSE file.

// WARNING: Do not edit - generated code.

@DocsEditable()
@DomName('HTMLMapElement')
class MapElement extends HtmlElement {
  // To suppress missing implicit constructor warnings.
  factory MapElement._() {
    throw new UnsupportedError("Not supported");
  }

  @DomName('HTMLMapElement.HTMLMapElement')
  @DocsEditable()
  factory MapElement() => document.createElement("map");

  @Deprecated("Internal Use Only")
  external static Type get instanceRuntimeType;

  @Deprecated("Internal Use Only")
  MapElement.internal_() : super.internal_();

  /**
   * Constructor instantiated by the DOM when a custom element has been created.
   *
   * This can only be called by subclasses from their created constructor.
   */
  MapElement.created() : super.created();

  @DomName('HTMLMapElement.areas')
  @DocsEditable()
  List<Node> get areas =>
      (_blink.BlinkHTMLMapElement.instance.areas_Getter_(this));

  @DomName('HTMLMapElement.name')
  @DocsEditable()
  String get name => _blink.BlinkHTMLMapElement.instance.name_Getter_(this);

  @DomName('HTMLMapElement.name')
  @DocsEditable()
  set name(String value) =>
      _blink.BlinkHTMLMapElement.instance.name_Setter_(this, value);
}
// Copyright (c) 2012, the Dart project authors.  Please see the AUTHORS file
// for details. All rights reserved. Use of this source code is governed by a
// BSD-style license that can be found in the LICENSE file.

// WARNING: Do not edit - generated code.

@DocsEditable()
@DomName('MediaController')
// http://www.whatwg.org/specs/web-apps/current-work/multipage/the-video-element.html#mediacontroller
@Experimental()
class MediaController extends EventTarget {
  // To suppress missing implicit constructor warnings.
  factory MediaController._() {
    throw new UnsupportedError("Not supported");
  }

  @DomName('MediaController.MediaController')
  @DocsEditable()
  factory MediaController() {
    return _blink.BlinkMediaController.instance.constructorCallback_0_();
  }

  @Deprecated("Internal Use Only")
  external static Type get instanceRuntimeType;

  @Deprecated("Internal Use Only")
  MediaController.internal_() : super.internal_();

  @DomName('MediaController.buffered')
  @DocsEditable()
  TimeRanges get buffered =>
      _blink.BlinkMediaController.instance.buffered_Getter_(this);

  @DomName('MediaController.currentTime')
  @DocsEditable()
  num get currentTime =>
      _blink.BlinkMediaController.instance.currentTime_Getter_(this);

  @DomName('MediaController.currentTime')
  @DocsEditable()
  set currentTime(num value) =>
      _blink.BlinkMediaController.instance.currentTime_Setter_(this, value);

  @DomName('MediaController.defaultPlaybackRate')
  @DocsEditable()
  num get defaultPlaybackRate =>
      _blink.BlinkMediaController.instance.defaultPlaybackRate_Getter_(this);

  @DomName('MediaController.defaultPlaybackRate')
  @DocsEditable()
  set defaultPlaybackRate(num value) => _blink.BlinkMediaController.instance
      .defaultPlaybackRate_Setter_(this, value);

  @DomName('MediaController.duration')
  @DocsEditable()
  num get duration =>
      _blink.BlinkMediaController.instance.duration_Getter_(this);

  @DomName('MediaController.muted')
  @DocsEditable()
  bool get muted => _blink.BlinkMediaController.instance.muted_Getter_(this);

  @DomName('MediaController.muted')
  @DocsEditable()
  set muted(bool value) =>
      _blink.BlinkMediaController.instance.muted_Setter_(this, value);

  @DomName('MediaController.paused')
  @DocsEditable()
  bool get paused => _blink.BlinkMediaController.instance.paused_Getter_(this);

  @DomName('MediaController.playbackRate')
  @DocsEditable()
  num get playbackRate =>
      _blink.BlinkMediaController.instance.playbackRate_Getter_(this);

  @DomName('MediaController.playbackRate')
  @DocsEditable()
  set playbackRate(num value) =>
      _blink.BlinkMediaController.instance.playbackRate_Setter_(this, value);

  @DomName('MediaController.playbackState')
  @DocsEditable()
  String get playbackState =>
      _blink.BlinkMediaController.instance.playbackState_Getter_(this);

  @DomName('MediaController.played')
  @DocsEditable()
  TimeRanges get played =>
      _blink.BlinkMediaController.instance.played_Getter_(this);

  @DomName('MediaController.seekable')
  @DocsEditable()
  TimeRanges get seekable =>
      _blink.BlinkMediaController.instance.seekable_Getter_(this);

  @DomName('MediaController.volume')
  @DocsEditable()
  num get volume => _blink.BlinkMediaController.instance.volume_Getter_(this);

  @DomName('MediaController.volume')
  @DocsEditable()
  set volume(num value) =>
      _blink.BlinkMediaController.instance.volume_Setter_(this, value);

  @DomName('MediaController.pause')
  @DocsEditable()
  void pause() => _blink.BlinkMediaController.instance.pause_Callback_0_(this);

  @DomName('MediaController.play')
  @DocsEditable()
  void play() => _blink.BlinkMediaController.instance.play_Callback_0_(this);

  @DomName('MediaController.unpause')
  @DocsEditable()
  void unpause() =>
      _blink.BlinkMediaController.instance.unpause_Callback_0_(this);
}
// Copyright (c) 2012, the Dart project authors.  Please see the AUTHORS file
// for details. All rights reserved. Use of this source code is governed by a
// BSD-style license that can be found in the LICENSE file.

// WARNING: Do not edit - generated code.

@DocsEditable()
@DomName('MediaDeviceInfo')
@Experimental() // untriaged
class MediaDeviceInfo extends DartHtmlDomObject {
  // To suppress missing implicit constructor warnings.
  factory MediaDeviceInfo._() {
    throw new UnsupportedError("Not supported");
  }

  @Deprecated("Internal Use Only")
  external static Type get instanceRuntimeType;

  @Deprecated("Internal Use Only")
  MediaDeviceInfo.internal_() {}

  @DomName('MediaDeviceInfo.deviceId')
  @DocsEditable()
  @Experimental() // untriaged
  String get deviceId =>
      _blink.BlinkMediaDeviceInfo.instance.deviceId_Getter_(this);

  @DomName('MediaDeviceInfo.groupId')
  @DocsEditable()
  @Experimental() // untriaged
  String get groupId =>
      _blink.BlinkMediaDeviceInfo.instance.groupId_Getter_(this);

  @DomName('MediaDeviceInfo.kind')
  @DocsEditable()
  @Experimental() // untriaged
  String get kind => _blink.BlinkMediaDeviceInfo.instance.kind_Getter_(this);

  @DomName('MediaDeviceInfo.label')
  @DocsEditable()
  @Experimental() // untriaged
  String get label => _blink.BlinkMediaDeviceInfo.instance.label_Getter_(this);
}
// Copyright (c) 2012, the Dart project authors.  Please see the AUTHORS file
// for details. All rights reserved. Use of this source code is governed by a
// BSD-style license that can be found in the LICENSE file.

// WARNING: Do not edit - generated code.

@DocsEditable()
@DomName('MediaDevices')
@Experimental() // untriaged
class MediaDevices extends DartHtmlDomObject {
  // To suppress missing implicit constructor warnings.
  factory MediaDevices._() {
    throw new UnsupportedError("Not supported");
  }

  @Deprecated("Internal Use Only")
  external static Type get instanceRuntimeType;

  @Deprecated("Internal Use Only")
  MediaDevices.internal_() {}

  @DomName('MediaDevices.enumerateDevices')
  @DocsEditable()
  @Experimental() // untriaged
  Future enumerateDevices() => convertNativePromiseToDartFuture(
      _blink.BlinkMediaDevices.instance.enumerateDevices_Callback_0_(this));

  @DomName('MediaDevices.getUserMedia')
  @DocsEditable()
  @Experimental() // untriaged
  Future getUserMedia(Map options) => convertNativePromiseToDartFuture(_blink
      .BlinkMediaDevices.instance
      .getUserMedia_Callback_1_(this, convertDartToNative_Dictionary(options)));
}
// Copyright (c) 2012, the Dart project authors.  Please see the AUTHORS file
// for details. All rights reserved. Use of this source code is governed by a
// BSD-style license that can be found in the LICENSE file.

// WARNING: Do not edit - generated code.

@DocsEditable()
@DomName('HTMLMediaElement')
@Unstable()
class MediaElement extends HtmlElement {
  // To suppress missing implicit constructor warnings.
  factory MediaElement._() {
    throw new UnsupportedError("Not supported");
  }

  /**
   * Static factory designed to expose `keyadded` events to event
   * handlers that are not necessarily instances of [MediaElement].
   *
   * See [EventStreamProvider] for usage information.
   */
  @DomName('HTMLMediaElement.webkitkeyaddedEvent')
  @DocsEditable()
  @SupportedBrowser(SupportedBrowser.CHROME)
  @SupportedBrowser(SupportedBrowser.SAFARI)
  @Experimental()
  // https://dvcs.w3.org/hg/html-media/raw-file/eme-v0.1/encrypted-media/encrypted-media.html#dom-keyadded
  static const EventStreamProvider<MediaKeyEvent> keyAddedEvent =
      const EventStreamProvider<MediaKeyEvent>('webkitkeyadded');

  /**
   * Static factory designed to expose `keyerror` events to event
   * handlers that are not necessarily instances of [MediaElement].
   *
   * See [EventStreamProvider] for usage information.
   */
  @DomName('HTMLMediaElement.webkitkeyerrorEvent')
  @DocsEditable()
  @SupportedBrowser(SupportedBrowser.CHROME)
  @SupportedBrowser(SupportedBrowser.SAFARI)
  @Experimental()
  // https://dvcs.w3.org/hg/html-media/raw-file/eme-v0.1/encrypted-media/encrypted-media.html#dom-keyadded
  static const EventStreamProvider<MediaKeyEvent> keyErrorEvent =
      const EventStreamProvider<MediaKeyEvent>('webkitkeyerror');

  /**
   * Static factory designed to expose `keymessage` events to event
   * handlers that are not necessarily instances of [MediaElement].
   *
   * See [EventStreamProvider] for usage information.
   */
  @DomName('HTMLMediaElement.webkitkeymessageEvent')
  @DocsEditable()
  @SupportedBrowser(SupportedBrowser.CHROME)
  @SupportedBrowser(SupportedBrowser.SAFARI)
  @Experimental()
  // https://dvcs.w3.org/hg/html-media/raw-file/eme-v0.1/encrypted-media/encrypted-media.html#dom-keyadded
  static const EventStreamProvider<MediaKeyEvent> keyMessageEvent =
      const EventStreamProvider<MediaKeyEvent>('webkitkeymessage');

  /**
   * Static factory designed to expose `needkey` events to event
   * handlers that are not necessarily instances of [MediaElement].
   *
   * See [EventStreamProvider] for usage information.
   */
  @DomName('HTMLMediaElement.webkitneedkeyEvent')
  @DocsEditable()
  @SupportedBrowser(SupportedBrowser.CHROME)
  @SupportedBrowser(SupportedBrowser.SAFARI)
  @Experimental()
  // https://dvcs.w3.org/hg/html-media/raw-file/eme-v0.1/encrypted-media/encrypted-media.html#dom-keyadded
  static const EventStreamProvider<MediaKeyEvent> needKeyEvent =
      const EventStreamProvider<MediaKeyEvent>('webkitneedkey');

  @Deprecated("Internal Use Only")
  external static Type get instanceRuntimeType;

  @Deprecated("Internal Use Only")
  MediaElement.internal_() : super.internal_();

  /**
   * Constructor instantiated by the DOM when a custom element has been created.
   *
   * This can only be called by subclasses from their created constructor.
   */
  MediaElement.created() : super.created();

  @DomName('HTMLMediaElement.HAVE_CURRENT_DATA')
  @DocsEditable()
  static const int HAVE_CURRENT_DATA = 2;

  @DomName('HTMLMediaElement.HAVE_ENOUGH_DATA')
  @DocsEditable()
  static const int HAVE_ENOUGH_DATA = 4;

  @DomName('HTMLMediaElement.HAVE_FUTURE_DATA')
  @DocsEditable()
  static const int HAVE_FUTURE_DATA = 3;

  @DomName('HTMLMediaElement.HAVE_METADATA')
  @DocsEditable()
  static const int HAVE_METADATA = 1;

  @DomName('HTMLMediaElement.HAVE_NOTHING')
  @DocsEditable()
  static const int HAVE_NOTHING = 0;

  @DomName('HTMLMediaElement.NETWORK_EMPTY')
  @DocsEditable()
  static const int NETWORK_EMPTY = 0;

  @DomName('HTMLMediaElement.NETWORK_IDLE')
  @DocsEditable()
  static const int NETWORK_IDLE = 1;

  @DomName('HTMLMediaElement.NETWORK_LOADING')
  @DocsEditable()
  static const int NETWORK_LOADING = 2;

  @DomName('HTMLMediaElement.NETWORK_NO_SOURCE')
  @DocsEditable()
  static const int NETWORK_NO_SOURCE = 3;

  @DomName('HTMLMediaElement.audioTracks')
  @DocsEditable()
  @Experimental() // untriaged
  AudioTrackList get audioTracks =>
      _blink.BlinkHTMLMediaElement.instance.audioTracks_Getter_(this);

  @DomName('HTMLMediaElement.autoplay')
  @DocsEditable()
  bool get autoplay =>
      _blink.BlinkHTMLMediaElement.instance.autoplay_Getter_(this);

  @DomName('HTMLMediaElement.autoplay')
  @DocsEditable()
  set autoplay(bool value) =>
      _blink.BlinkHTMLMediaElement.instance.autoplay_Setter_(this, value);

  @DomName('HTMLMediaElement.buffered')
  @DocsEditable()
  TimeRanges get buffered =>
      _blink.BlinkHTMLMediaElement.instance.buffered_Getter_(this);

  @DomName('HTMLMediaElement.controller')
  @DocsEditable()
  MediaController get controller =>
      _blink.BlinkHTMLMediaElement.instance.controller_Getter_(this);

  @DomName('HTMLMediaElement.controller')
  @DocsEditable()
  set controller(MediaController value) =>
      _blink.BlinkHTMLMediaElement.instance.controller_Setter_(this, value);

  @DomName('HTMLMediaElement.controls')
  @DocsEditable()
  bool get controls =>
      _blink.BlinkHTMLMediaElement.instance.controls_Getter_(this);

  @DomName('HTMLMediaElement.controls')
  @DocsEditable()
  set controls(bool value) =>
      _blink.BlinkHTMLMediaElement.instance.controls_Setter_(this, value);

  @DomName('HTMLMediaElement.crossOrigin')
  @DocsEditable()
  @Experimental() // untriaged
  String get crossOrigin =>
      _blink.BlinkHTMLMediaElement.instance.crossOrigin_Getter_(this);

  @DomName('HTMLMediaElement.crossOrigin')
  @DocsEditable()
  @Experimental() // untriaged
  set crossOrigin(String value) =>
      _blink.BlinkHTMLMediaElement.instance.crossOrigin_Setter_(this, value);

  @DomName('HTMLMediaElement.currentSrc')
  @DocsEditable()
  String get currentSrc =>
      _blink.BlinkHTMLMediaElement.instance.currentSrc_Getter_(this);

  @DomName('HTMLMediaElement.currentTime')
  @DocsEditable()
  num get currentTime =>
      _blink.BlinkHTMLMediaElement.instance.currentTime_Getter_(this);

  @DomName('HTMLMediaElement.currentTime')
  @DocsEditable()
  set currentTime(num value) =>
      _blink.BlinkHTMLMediaElement.instance.currentTime_Setter_(this, value);

  @DomName('HTMLMediaElement.defaultMuted')
  @DocsEditable()
  bool get defaultMuted =>
      _blink.BlinkHTMLMediaElement.instance.defaultMuted_Getter_(this);

  @DomName('HTMLMediaElement.defaultMuted')
  @DocsEditable()
  set defaultMuted(bool value) =>
      _blink.BlinkHTMLMediaElement.instance.defaultMuted_Setter_(this, value);

  @DomName('HTMLMediaElement.defaultPlaybackRate')
  @DocsEditable()
  num get defaultPlaybackRate =>
      _blink.BlinkHTMLMediaElement.instance.defaultPlaybackRate_Getter_(this);

  @DomName('HTMLMediaElement.defaultPlaybackRate')
  @DocsEditable()
  set defaultPlaybackRate(num value) => _blink.BlinkHTMLMediaElement.instance
      .defaultPlaybackRate_Setter_(this, value);

  @DomName('HTMLMediaElement.duration')
  @DocsEditable()
  num get duration =>
      _blink.BlinkHTMLMediaElement.instance.duration_Getter_(this);

  @DomName('HTMLMediaElement.ended')
  @DocsEditable()
  bool get ended => _blink.BlinkHTMLMediaElement.instance.ended_Getter_(this);

  @DomName('HTMLMediaElement.error')
  @DocsEditable()
  MediaError get error =>
      _blink.BlinkHTMLMediaElement.instance.error_Getter_(this);

  @DomName('HTMLMediaElement.loop')
  @DocsEditable()
  bool get loop => _blink.BlinkHTMLMediaElement.instance.loop_Getter_(this);

  @DomName('HTMLMediaElement.loop')
  @DocsEditable()
  set loop(bool value) =>
      _blink.BlinkHTMLMediaElement.instance.loop_Setter_(this, value);

  @DomName('HTMLMediaElement.mediaGroup')
  @DocsEditable()
  String get mediaGroup =>
      _blink.BlinkHTMLMediaElement.instance.mediaGroup_Getter_(this);

  @DomName('HTMLMediaElement.mediaGroup')
  @DocsEditable()
  set mediaGroup(String value) =>
      _blink.BlinkHTMLMediaElement.instance.mediaGroup_Setter_(this, value);

  @DomName('HTMLMediaElement.mediaKeys')
  @DocsEditable()
  // https://dvcs.w3.org/hg/html-media/raw-file/eme-v0.1/encrypted-media/encrypted-media.html
  @Experimental()
  MediaKeys get mediaKeys =>
      _blink.BlinkHTMLMediaElement.instance.mediaKeys_Getter_(this);

  @DomName('HTMLMediaElement.muted')
  @DocsEditable()
  bool get muted => _blink.BlinkHTMLMediaElement.instance.muted_Getter_(this);

  @DomName('HTMLMediaElement.muted')
  @DocsEditable()
  set muted(bool value) =>
      _blink.BlinkHTMLMediaElement.instance.muted_Setter_(this, value);

  @DomName('HTMLMediaElement.networkState')
  @DocsEditable()
  int get networkState =>
      _blink.BlinkHTMLMediaElement.instance.networkState_Getter_(this);

  @DomName('HTMLMediaElement.paused')
  @DocsEditable()
  bool get paused => _blink.BlinkHTMLMediaElement.instance.paused_Getter_(this);

  @DomName('HTMLMediaElement.playbackRate')
  @DocsEditable()
  num get playbackRate =>
      _blink.BlinkHTMLMediaElement.instance.playbackRate_Getter_(this);

  @DomName('HTMLMediaElement.playbackRate')
  @DocsEditable()
  set playbackRate(num value) =>
      _blink.BlinkHTMLMediaElement.instance.playbackRate_Setter_(this, value);

  @DomName('HTMLMediaElement.played')
  @DocsEditable()
  TimeRanges get played =>
      _blink.BlinkHTMLMediaElement.instance.played_Getter_(this);

  @DomName('HTMLMediaElement.preload')
  @DocsEditable()
  String get preload =>
      _blink.BlinkHTMLMediaElement.instance.preload_Getter_(this);

  @DomName('HTMLMediaElement.preload')
  @DocsEditable()
  set preload(String value) =>
      _blink.BlinkHTMLMediaElement.instance.preload_Setter_(this, value);

  @DomName('HTMLMediaElement.readyState')
  @DocsEditable()
  int get readyState =>
      _blink.BlinkHTMLMediaElement.instance.readyState_Getter_(this);

  @DomName('HTMLMediaElement.seekable')
  @DocsEditable()
  TimeRanges get seekable =>
      _blink.BlinkHTMLMediaElement.instance.seekable_Getter_(this);

  @DomName('HTMLMediaElement.seeking')
  @DocsEditable()
  bool get seeking =>
      _blink.BlinkHTMLMediaElement.instance.seeking_Getter_(this);

  @DomName('HTMLMediaElement.session')
  @DocsEditable()
  @Experimental() // untriaged
  MediaSession get session =>
      _blink.BlinkHTMLMediaElement.instance.session_Getter_(this);

  @DomName('HTMLMediaElement.session')
  @DocsEditable()
  @Experimental() // untriaged
  set session(MediaSession value) =>
      _blink.BlinkHTMLMediaElement.instance.session_Setter_(this, value);

  @DomName('HTMLMediaElement.sinkId')
  @DocsEditable()
  @Experimental() // untriaged
  String get sinkId =>
      _blink.BlinkHTMLMediaElement.instance.sinkId_Getter_(this);

  @DomName('HTMLMediaElement.src')
  @DocsEditable()
  String get src => _blink.BlinkHTMLMediaElement.instance.src_Getter_(this);

  @DomName('HTMLMediaElement.src')
  @DocsEditable()
  set src(String value) =>
      _blink.BlinkHTMLMediaElement.instance.src_Setter_(this, value);

  @DomName('HTMLMediaElement.textTracks')
  @DocsEditable()
  // http://www.whatwg.org/specs/web-apps/current-work/multipage/the-video-element.html#dom-media-texttracks
  @Experimental()
  TextTrackList get textTracks =>
      _blink.BlinkHTMLMediaElement.instance.textTracks_Getter_(this);

  @DomName('HTMLMediaElement.videoTracks')
  @DocsEditable()
  @Experimental() // untriaged
  VideoTrackList get videoTracks =>
      _blink.BlinkHTMLMediaElement.instance.videoTracks_Getter_(this);

  @DomName('HTMLMediaElement.volume')
  @DocsEditable()
  num get volume => _blink.BlinkHTMLMediaElement.instance.volume_Getter_(this);

  @DomName('HTMLMediaElement.volume')
  @DocsEditable()
  set volume(num value) =>
      _blink.BlinkHTMLMediaElement.instance.volume_Setter_(this, value);

  @DomName('HTMLMediaElement.webkitAudioDecodedByteCount')
  @DocsEditable()
  @SupportedBrowser(SupportedBrowser.CHROME)
  @SupportedBrowser(SupportedBrowser.SAFARI)
  @Experimental()
  @Experimental() // nonstandard
  int get audioDecodedByteCount => _blink.BlinkHTMLMediaElement.instance
      .webkitAudioDecodedByteCount_Getter_(this);

  @DomName('HTMLMediaElement.webkitVideoDecodedByteCount')
  @DocsEditable()
  @SupportedBrowser(SupportedBrowser.CHROME)
  @SupportedBrowser(SupportedBrowser.SAFARI)
  @Experimental()
  @Experimental() // nonstandard
  int get videoDecodedByteCount => _blink.BlinkHTMLMediaElement.instance
      .webkitVideoDecodedByteCount_Getter_(this);

  TextTrack addTextTrack(String kind, [String label, String language]) {
    if (language != null) {
      return _blink.BlinkHTMLMediaElement.instance
          .addTextTrack_Callback_3_(this, kind, label, language);
    }
    if (label != null) {
      return _blink.BlinkHTMLMediaElement.instance
          .addTextTrack_Callback_2_(this, kind, label);
    }
    return _blink.BlinkHTMLMediaElement.instance
        .addTextTrack_Callback_1_(this, kind);
  }

  String canPlayType(String type, [String keySystem]) {
    if ((type is String) && keySystem == null) {
      return _blink.BlinkHTMLMediaElement.instance
          .canPlayType_Callback_1_(this, type);
    }
    if ((type is String || type == null) && keySystem == null) {
      return _blink.BlinkHTMLMediaElement.instance
          .canPlayType_Callback_1_(this, type);
    }
    if ((keySystem is String || keySystem == null) &&
        (type is String || type == null)) {
      return _blink.BlinkHTMLMediaElement.instance
          .canPlayType_Callback_2_(this, type, keySystem);
    }
    throw new ArgumentError("Incorrect number or type of arguments");
  }

  @DomName('HTMLMediaElement.load')
  @DocsEditable()
  void load() => _blink.BlinkHTMLMediaElement.instance.load_Callback_0_(this);

  @DomName('HTMLMediaElement.pause')
  @DocsEditable()
  void pause() => _blink.BlinkHTMLMediaElement.instance.pause_Callback_0_(this);

  @DomName('HTMLMediaElement.play')
  @DocsEditable()
  void play() => _blink.BlinkHTMLMediaElement.instance.play_Callback_0_(this);

  @DomName('HTMLMediaElement.setMediaKeys')
  @DocsEditable()
  @Experimental() // untriaged
  Future setMediaKeys(MediaKeys mediaKeys) =>
      convertNativePromiseToDartFuture(_blink.BlinkHTMLMediaElement.instance
          .setMediaKeys_Callback_1_(this, mediaKeys));

  @DomName('HTMLMediaElement.setSinkId')
  @DocsEditable()
  @Experimental() // untriaged
  Future setSinkId(String sinkId) =>
      convertNativePromiseToDartFuture(_blink.BlinkHTMLMediaElement.instance
          .setSinkId_Callback_1_(this, sinkId));

  void addKey(String keySystem, Uint8List key,
      [Uint8List initData, String sessionId]) {
    if (initData != null) {
      _blink.BlinkHTMLMediaElement.instance
          .webkitAddKey_Callback_4_(this, keySystem, key, initData, sessionId);
      return;
    }
    _blink.BlinkHTMLMediaElement.instance
        .webkitAddKey_Callback_2_(this, keySystem, key);
    return;
  }

  @DomName('HTMLMediaElement.webkitCancelKeyRequest')
  @DocsEditable()
  @SupportedBrowser(SupportedBrowser.CHROME)
  @SupportedBrowser(SupportedBrowser.SAFARI)
  @Experimental()
  // https://dvcs.w3.org/hg/html-media/raw-file/eme-v0.1/encrypted-media/encrypted-media.html#extensions
  void cancelKeyRequest(String keySystem, String sessionId) =>
      _blink.BlinkHTMLMediaElement.instance
          .webkitCancelKeyRequest_Callback_2_(this, keySystem, sessionId);

  void generateKeyRequest(String keySystem, [Uint8List initData]) {
    if (initData != null) {
      _blink.BlinkHTMLMediaElement.instance
          .webkitGenerateKeyRequest_Callback_2_(this, keySystem, initData);
      return;
    }
    _blink.BlinkHTMLMediaElement.instance
        .webkitGenerateKeyRequest_Callback_1_(this, keySystem);
    return;
  }

  /// Stream of `keyadded` events handled by this [MediaElement].
  @DomName('HTMLMediaElement.onwebkitkeyadded')
  @DocsEditable()
  // https://dvcs.w3.org/hg/html-media/raw-file/eme-v0.1/encrypted-media/encrypted-media.html#dom-keyadded
  @Experimental()
  ElementStream<MediaKeyEvent> get onKeyAdded => keyAddedEvent.forElement(this);

  /// Stream of `keyerror` events handled by this [MediaElement].
  @DomName('HTMLMediaElement.onwebkitkeyerror')
  @DocsEditable()
  // https://dvcs.w3.org/hg/html-media/raw-file/eme-v0.1/encrypted-media/encrypted-media.html#dom-keyadded
  @Experimental()
  ElementStream<MediaKeyEvent> get onKeyError => keyErrorEvent.forElement(this);

  /// Stream of `keymessage` events handled by this [MediaElement].
  @DomName('HTMLMediaElement.onwebkitkeymessage')
  @DocsEditable()
  // https://dvcs.w3.org/hg/html-media/raw-file/eme-v0.1/encrypted-media/encrypted-media.html#dom-keyadded
  @Experimental()
  ElementStream<MediaKeyEvent> get onKeyMessage =>
      keyMessageEvent.forElement(this);

  /// Stream of `needkey` events handled by this [MediaElement].
  @DomName('HTMLMediaElement.onwebkitneedkey')
  @DocsEditable()
  // https://dvcs.w3.org/hg/html-media/raw-file/eme-v0.1/encrypted-media/encrypted-media.html#dom-keyadded
  @Experimental()
  ElementStream<MediaKeyEvent> get onNeedKey => needKeyEvent.forElement(this);
}
// Copyright (c) 2012, the Dart project authors.  Please see the AUTHORS file
// for details. All rights reserved. Use of this source code is governed by a
// BSD-style license that can be found in the LICENSE file.

// WARNING: Do not edit - generated code.

@DocsEditable()
@DomName('MediaEncryptedEvent')
@Experimental() // untriaged
class MediaEncryptedEvent extends Event {
  // To suppress missing implicit constructor warnings.
  factory MediaEncryptedEvent._() {
    throw new UnsupportedError("Not supported");
  }

  @DomName('MediaEncryptedEvent.MediaEncryptedEvent')
  @DocsEditable()
  factory MediaEncryptedEvent(String type, [Map eventInitDict]) {
    if (eventInitDict != null) {
      var eventInitDict_1 = convertDartToNative_Dictionary(eventInitDict);
      return _blink.BlinkMediaEncryptedEvent.instance
          .constructorCallback_2_(type, eventInitDict_1);
    }
    return _blink.BlinkMediaEncryptedEvent.instance
        .constructorCallback_1_(type);
  }

  @Deprecated("Internal Use Only")
  external static Type get instanceRuntimeType;

  @Deprecated("Internal Use Only")
  MediaEncryptedEvent.internal_() : super.internal_();

  @DomName('MediaEncryptedEvent.initData')
  @DocsEditable()
  @Experimental() // untriaged
  ByteBuffer get initData =>
      _blink.BlinkMediaEncryptedEvent.instance.initData_Getter_(this);

  @DomName('MediaEncryptedEvent.initDataType')
  @DocsEditable()
  @Experimental() // untriaged
  String get initDataType =>
      _blink.BlinkMediaEncryptedEvent.instance.initDataType_Getter_(this);
}
// Copyright (c) 2012, the Dart project authors.  Please see the AUTHORS file
// for details. All rights reserved. Use of this source code is governed by a
// BSD-style license that can be found in the LICENSE file.

// WARNING: Do not edit - generated code.

@DocsEditable()
@DomName('MediaError')
@Unstable()
class MediaError extends DartHtmlDomObject {
  // To suppress missing implicit constructor warnings.
  factory MediaError._() {
    throw new UnsupportedError("Not supported");
  }

  @Deprecated("Internal Use Only")
  external static Type get instanceRuntimeType;

  @Deprecated("Internal Use Only")
  MediaError.internal_() {}

  @DomName('MediaError.MEDIA_ERR_ABORTED')
  @DocsEditable()
  static const int MEDIA_ERR_ABORTED = 1;

  @DomName('MediaError.MEDIA_ERR_DECODE')
  @DocsEditable()
  static const int MEDIA_ERR_DECODE = 3;

  @DomName('MediaError.MEDIA_ERR_NETWORK')
  @DocsEditable()
  static const int MEDIA_ERR_NETWORK = 2;

  @DomName('MediaError.MEDIA_ERR_SRC_NOT_SUPPORTED')
  @DocsEditable()
  static const int MEDIA_ERR_SRC_NOT_SUPPORTED = 4;

  @DomName('MediaError.code')
  @DocsEditable()
  int get code => _blink.BlinkMediaError.instance.code_Getter_(this);
}
// Copyright (c) 2012, the Dart project authors.  Please see the AUTHORS file
// for details. All rights reserved. Use of this source code is governed by a
// BSD-style license that can be found in the LICENSE file.

// WARNING: Do not edit - generated code.

@DocsEditable()
@DomName('MediaKeyError')
// https://dvcs.w3.org/hg/html-media/raw-file/eme-v0.1/encrypted-media/encrypted-media.html#error-codes
@Experimental()
class MediaKeyError extends DartHtmlDomObject {
  // To suppress missing implicit constructor warnings.
  factory MediaKeyError._() {
    throw new UnsupportedError("Not supported");
  }

  @Deprecated("Internal Use Only")
  external static Type get instanceRuntimeType;

  @Deprecated("Internal Use Only")
  MediaKeyError.internal_() {}

  @DomName('MediaKeyError.MEDIA_KEYERR_CLIENT')
  @DocsEditable()
  static const int MEDIA_KEYERR_CLIENT = 2;

  @DomName('MediaKeyError.MEDIA_KEYERR_DOMAIN')
  @DocsEditable()
  static const int MEDIA_KEYERR_DOMAIN = 6;

  @DomName('MediaKeyError.MEDIA_KEYERR_HARDWARECHANGE')
  @DocsEditable()
  static const int MEDIA_KEYERR_HARDWARECHANGE = 5;

  @DomName('MediaKeyError.MEDIA_KEYERR_OUTPUT')
  @DocsEditable()
  static const int MEDIA_KEYERR_OUTPUT = 4;

  @DomName('MediaKeyError.MEDIA_KEYERR_SERVICE')
  @DocsEditable()
  static const int MEDIA_KEYERR_SERVICE = 3;

  @DomName('MediaKeyError.MEDIA_KEYERR_UNKNOWN')
  @DocsEditable()
  static const int MEDIA_KEYERR_UNKNOWN = 1;

  @DomName('MediaKeyError.code')
  @DocsEditable()
  int get code => _blink.BlinkMediaKeyError.instance.code_Getter_(this);

  @DomName('MediaKeyError.systemCode')
  @DocsEditable()
  @Experimental() // non-standard
  int get systemCode =>
      _blink.BlinkMediaKeyError.instance.systemCode_Getter_(this);
}
// Copyright (c) 2012, the Dart project authors.  Please see the AUTHORS file
// for details. All rights reserved. Use of this source code is governed by a
// BSD-style license that can be found in the LICENSE file.

// WARNING: Do not edit - generated code.

@DocsEditable()
@DomName('MediaKeyEvent')
// https://dvcs.w3.org/hg/html-media/raw-file/eme-v0.1/encrypted-media/encrypted-media.html#event-definitions
@Experimental()
class MediaKeyEvent extends Event {
  // To suppress missing implicit constructor warnings.
  factory MediaKeyEvent._() {
    throw new UnsupportedError("Not supported");
  }

  @DomName('MediaKeyEvent.MediaKeyEvent')
  @DocsEditable()
  factory MediaKeyEvent(String type, [Map eventInitDict]) {
    if (eventInitDict != null) {
      var eventInitDict_1 = convertDartToNative_Dictionary(eventInitDict);
      return _blink.BlinkMediaKeyEvent.instance
          .constructorCallback_2_(type, eventInitDict_1);
    }
    return _blink.BlinkMediaKeyEvent.instance.constructorCallback_1_(type);
  }

  @Deprecated("Internal Use Only")
  external static Type get instanceRuntimeType;

  @Deprecated("Internal Use Only")
  MediaKeyEvent.internal_() : super.internal_();

  @DomName('MediaKeyEvent.defaultURL')
  @DocsEditable()
  String get defaultUrl =>
      _blink.BlinkMediaKeyEvent.instance.defaultURL_Getter_(this);

  @DomName('MediaKeyEvent.errorCode')
  @DocsEditable()
  MediaKeyError get errorCode =>
      _blink.BlinkMediaKeyEvent.instance.errorCode_Getter_(this);

  @DomName('MediaKeyEvent.initData')
  @DocsEditable()
  Uint8List get initData =>
      _blink.BlinkMediaKeyEvent.instance.initData_Getter_(this);

  @DomName('MediaKeyEvent.keySystem')
  @DocsEditable()
  String get keySystem =>
      _blink.BlinkMediaKeyEvent.instance.keySystem_Getter_(this);

  @DomName('MediaKeyEvent.message')
  @DocsEditable()
  Uint8List get message =>
      _blink.BlinkMediaKeyEvent.instance.message_Getter_(this);

  @DomName('MediaKeyEvent.sessionId')
  @DocsEditable()
  String get sessionId =>
      _blink.BlinkMediaKeyEvent.instance.sessionId_Getter_(this);

  @DomName('MediaKeyEvent.systemCode')
  @DocsEditable()
  int get systemCode =>
      _blink.BlinkMediaKeyEvent.instance.systemCode_Getter_(this);
}
// Copyright (c) 2012, the Dart project authors.  Please see the AUTHORS file
// for details. All rights reserved. Use of this source code is governed by a
// BSD-style license that can be found in the LICENSE file.

// WARNING: Do not edit - generated code.

@DocsEditable()
@DomName('MediaKeyMessageEvent')
// https://dvcs.w3.org/hg/html-media/raw-file/eme-v0.1/encrypted-media/encrypted-media.html#dom-mediakeymessageevent
@Experimental()
class MediaKeyMessageEvent extends Event {
  // To suppress missing implicit constructor warnings.
  factory MediaKeyMessageEvent._() {
    throw new UnsupportedError("Not supported");
  }

  @DomName('MediaKeyMessageEvent.MediaKeyMessageEvent')
  @DocsEditable()
  factory MediaKeyMessageEvent(String type, [Map eventInitDict]) {
    if (eventInitDict != null) {
      var eventInitDict_1 = convertDartToNative_Dictionary(eventInitDict);
      return _blink.BlinkMediaKeyMessageEvent.instance
          .constructorCallback_2_(type, eventInitDict_1);
    }
    return _blink.BlinkMediaKeyMessageEvent.instance
        .constructorCallback_1_(type);
  }

  @Deprecated("Internal Use Only")
  external static Type get instanceRuntimeType;

  @Deprecated("Internal Use Only")
  MediaKeyMessageEvent.internal_() : super.internal_();

  @DomName('MediaKeyMessageEvent.message')
  @DocsEditable()
  ByteBuffer get message =>
      _blink.BlinkMediaKeyMessageEvent.instance.message_Getter_(this);

  @DomName('MediaKeyMessageEvent.messageType')
  @DocsEditable()
  @Experimental() // untriaged
  String get messageType =>
      _blink.BlinkMediaKeyMessageEvent.instance.messageType_Getter_(this);
}
// Copyright (c) 2012, the Dart project authors.  Please see the AUTHORS file
// for details. All rights reserved. Use of this source code is governed by a
// BSD-style license that can be found in the LICENSE file.

// WARNING: Do not edit - generated code.

@DocsEditable()
@DomName('MediaKeySession')
// https://dvcs.w3.org/hg/html-media/raw-file/eme-v0.1/encrypted-media/encrypted-media.html#dom-mediakeysession
@Experimental()
class MediaKeySession extends EventTarget {
  // To suppress missing implicit constructor warnings.
  factory MediaKeySession._() {
    throw new UnsupportedError("Not supported");
  }

  @Deprecated("Internal Use Only")
  external static Type get instanceRuntimeType;

  @Deprecated("Internal Use Only")
  MediaKeySession.internal_() : super.internal_();

  @DomName('MediaKeySession.closed')
  @DocsEditable()
  @Experimental() // untriaged
  Future get closed => convertNativePromiseToDartFuture(
      _blink.BlinkMediaKeySession.instance.closed_Getter_(this));

  @DomName('MediaKeySession.expiration')
  @DocsEditable()
  @Experimental() // untriaged
  num get expiration =>
      _blink.BlinkMediaKeySession.instance.expiration_Getter_(this);

  @DomName('MediaKeySession.keyStatuses')
  @DocsEditable()
  @Experimental() // untriaged
  MediaKeyStatusMap get keyStatuses =>
      _blink.BlinkMediaKeySession.instance.keyStatuses_Getter_(this);

  @DomName('MediaKeySession.sessionId')
  @DocsEditable()
  String get sessionId =>
      _blink.BlinkMediaKeySession.instance.sessionId_Getter_(this);

  @DomName('MediaKeySession.close')
  @DocsEditable()
  Future close() => convertNativePromiseToDartFuture(
      _blink.BlinkMediaKeySession.instance.close_Callback_0_(this));

  @DomName('MediaKeySession.generateRequest')
  @DocsEditable()
  @Experimental() // untriaged
  Future generateRequest(String initDataType, /*BufferSource*/ initData) =>
      convertNativePromiseToDartFuture(_blink.BlinkMediaKeySession.instance
          .generateRequest_Callback_2_(this, initDataType, initData));

  @DomName('MediaKeySession.load')
  @DocsEditable()
  @Experimental() // untriaged
  Future load(String sessionId) => convertNativePromiseToDartFuture(
      _blink.BlinkMediaKeySession.instance.load_Callback_1_(this, sessionId));

  @DomName('MediaKeySession.remove')
  @DocsEditable()
  @Experimental() // untriaged
  Future remove() => convertNativePromiseToDartFuture(
      _blink.BlinkMediaKeySession.instance.remove_Callback_0_(this));

  @DomName('MediaKeySession.update')
  @DocsEditable()
  Future _update(/*BufferSource*/ response) => convertNativePromiseToDartFuture(
      _blink.BlinkMediaKeySession.instance.update_Callback_1_(this, response));
}
// Copyright (c) 2012, the Dart project authors.  Please see the AUTHORS file
// for details. All rights reserved. Use of this source code is governed by a
// BSD-style license that can be found in the LICENSE file.

// WARNING: Do not edit - generated code.

@DocsEditable()
@DomName('MediaKeyStatusMap')
@Experimental() // untriaged
class MediaKeyStatusMap extends DartHtmlDomObject {
  // To suppress missing implicit constructor warnings.
  factory MediaKeyStatusMap._() {
    throw new UnsupportedError("Not supported");
  }

  @Deprecated("Internal Use Only")
  external static Type get instanceRuntimeType;

  @Deprecated("Internal Use Only")
  MediaKeyStatusMap.internal_() {}

  @DomName('MediaKeyStatusMap.size')
  @DocsEditable()
  @Experimental() // untriaged
  int get size => _blink.BlinkMediaKeyStatusMap.instance.size_Getter_(this);
}
// Copyright (c) 2012, the Dart project authors.  Please see the AUTHORS file
// for details. All rights reserved. Use of this source code is governed by a
// BSD-style license that can be found in the LICENSE file.

// WARNING: Do not edit - generated code.

@DocsEditable()
@DomName('MediaKeySystemAccess')
@Experimental() // untriaged
class MediaKeySystemAccess extends DartHtmlDomObject {
  // To suppress missing implicit constructor warnings.
  factory MediaKeySystemAccess._() {
    throw new UnsupportedError("Not supported");
  }

  @Deprecated("Internal Use Only")
  external static Type get instanceRuntimeType;

  @Deprecated("Internal Use Only")
  MediaKeySystemAccess.internal_() {}

  @DomName('MediaKeySystemAccess.keySystem')
  @DocsEditable()
  @Experimental() // untriaged
  String get keySystem =>
      _blink.BlinkMediaKeySystemAccess.instance.keySystem_Getter_(this);

  @DomName('MediaKeySystemAccess.createMediaKeys')
  @DocsEditable()
  @Experimental() // untriaged
  Future createMediaKeys() =>
      convertNativePromiseToDartFuture(_blink.BlinkMediaKeySystemAccess.instance
          .createMediaKeys_Callback_0_(this));

  @DomName('MediaKeySystemAccess.getConfiguration')
  @DocsEditable()
  @Experimental() // untriaged
  getConfiguration() => convertNativeDictionaryToDartDictionary((_blink
      .BlinkMediaKeySystemAccess.instance
      .getConfiguration_Callback_0_(this)));
}
// Copyright (c) 2012, the Dart project authors.  Please see the AUTHORS file
// for details. All rights reserved. Use of this source code is governed by a
// BSD-style license that can be found in the LICENSE file.

// WARNING: Do not edit - generated code.

@DocsEditable()
@DomName('MediaKeys')
// https://dvcs.w3.org/hg/html-media/raw-file/eme-v0.1/encrypted-media/encrypted-media.html
@Experimental()
class MediaKeys extends DartHtmlDomObject {
  // To suppress missing implicit constructor warnings.
  factory MediaKeys._() {
    throw new UnsupportedError("Not supported");
  }

  @Deprecated("Internal Use Only")
  external static Type get instanceRuntimeType;

  @Deprecated("Internal Use Only")
  MediaKeys.internal_() {}

  MediaKeySession _createSession([String sessionType]) {
    if (sessionType != null) {
      return _blink.BlinkMediaKeys.instance
          .createSession_Callback_1_(this, sessionType);
    }
    return _blink.BlinkMediaKeys.instance.createSession_Callback_0_(this);
  }

  @DomName('MediaKeys.setServerCertificate')
  @DocsEditable()
  @Experimental() // untriaged
  Future setServerCertificate(/*BufferSource*/ serverCertificate) =>
      convertNativePromiseToDartFuture(_blink.BlinkMediaKeys.instance
          .setServerCertificate_Callback_1_(this, serverCertificate));
}
// Copyright (c) 2012, the Dart project authors.  Please see the AUTHORS file
// for details. All rights reserved. Use of this source code is governed by a
// BSD-style license that can be found in the LICENSE file.

// WARNING: Do not edit - generated code.

@DocsEditable()
@DomName('MediaList')
@Unstable()
class MediaList extends DartHtmlDomObject {
  // To suppress missing implicit constructor warnings.
  factory MediaList._() {
    throw new UnsupportedError("Not supported");
  }

  @Deprecated("Internal Use Only")
  external static Type get instanceRuntimeType;

  @Deprecated("Internal Use Only")
  MediaList.internal_() {}

  @DomName('MediaList.length')
  @DocsEditable()
  int get length => _blink.BlinkMediaList.instance.length_Getter_(this);

  @DomName('MediaList.mediaText')
  @DocsEditable()
  String get mediaText =>
      _blink.BlinkMediaList.instance.mediaText_Getter_(this);

  @DomName('MediaList.mediaText')
  @DocsEditable()
  set mediaText(String value) =>
      _blink.BlinkMediaList.instance.mediaText_Setter_(this, value);

  @DomName('MediaList.appendMedium')
  @DocsEditable()
  void appendMedium(String medium) =>
      _blink.BlinkMediaList.instance.appendMedium_Callback_1_(this, medium);

  @DomName('MediaList.deleteMedium')
  @DocsEditable()
  void deleteMedium(String medium) =>
      _blink.BlinkMediaList.instance.deleteMedium_Callback_1_(this, medium);

  @DomName('MediaList.item')
  @DocsEditable()
  String item(int index) =>
      _blink.BlinkMediaList.instance.item_Callback_1_(this, index);
}
// Copyright (c) 2012, the Dart project authors.  Please see the AUTHORS file
// for details. All rights reserved. Use of this source code is governed by a
// BSD-style license that can be found in the LICENSE file.

// WARNING: Do not edit - generated code.

@DocsEditable()
@DomName('MediaQueryList')
@Unstable()
class MediaQueryList extends EventTarget {
  // To suppress missing implicit constructor warnings.
  factory MediaQueryList._() {
    throw new UnsupportedError("Not supported");
  }

  @DomName('MediaQueryList.changeEvent')
  @DocsEditable()
  @Experimental() // untriaged
  static const EventStreamProvider<Event> changeEvent =
      const EventStreamProvider<Event>('change');

  @Deprecated("Internal Use Only")
  external static Type get instanceRuntimeType;

  @Deprecated("Internal Use Only")
  MediaQueryList.internal_() : super.internal_();

  @DomName('MediaQueryList.matches')
  @DocsEditable()
  bool get matches => _blink.BlinkMediaQueryList.instance.matches_Getter_(this);

  @DomName('MediaQueryList.media')
  @DocsEditable()
  String get media => _blink.BlinkMediaQueryList.instance.media_Getter_(this);

  @DomName('MediaQueryList.addListener')
  @DocsEditable()
  void addListener(EventListener listener) =>
      _blink.BlinkMediaQueryList.instance
          .addListener_Callback_1_(this, listener);

  @DomName('MediaQueryList.removeListener')
  @DocsEditable()
  void removeListener(EventListener listener) =>
      _blink.BlinkMediaQueryList.instance
          .removeListener_Callback_1_(this, listener);

  @DomName('MediaQueryList.onchange')
  @DocsEditable()
  @Experimental() // untriaged
  Stream<Event> get onChange => changeEvent.forTarget(this);
}
// Copyright (c) 2012, the Dart project authors.  Please see the AUTHORS file
// for details. All rights reserved. Use of this source code is governed by a
// BSD-style license that can be found in the LICENSE file.

// WARNING: Do not edit - generated code.

@DocsEditable()
@DomName('MediaQueryListEvent')
@Experimental() // untriaged
class MediaQueryListEvent extends Event {
  // To suppress missing implicit constructor warnings.
  factory MediaQueryListEvent._() {
    throw new UnsupportedError("Not supported");
  }

  @DomName('MediaQueryListEvent.MediaQueryListEvent')
  @DocsEditable()
  factory MediaQueryListEvent(String type, [Map eventInitDict]) {
    if (eventInitDict != null) {
      var eventInitDict_1 = convertDartToNative_Dictionary(eventInitDict);
      return _blink.BlinkMediaQueryListEvent.instance
          .constructorCallback_2_(type, eventInitDict_1);
    }
    return _blink.BlinkMediaQueryListEvent.instance
        .constructorCallback_1_(type);
  }

  @Deprecated("Internal Use Only")
  external static Type get instanceRuntimeType;

  @Deprecated("Internal Use Only")
  MediaQueryListEvent.internal_() : super.internal_();

  @DomName('MediaQueryListEvent.matches')
  @DocsEditable()
  @Experimental() // untriaged
  bool get matches =>
      _blink.BlinkMediaQueryListEvent.instance.matches_Getter_(this);

  @DomName('MediaQueryListEvent.media')
  @DocsEditable()
  @Experimental() // untriaged
  String get media =>
      _blink.BlinkMediaQueryListEvent.instance.media_Getter_(this);
}
// Copyright (c) 2012, the Dart project authors.  Please see the AUTHORS file
// for details. All rights reserved. Use of this source code is governed by a
// BSD-style license that can be found in the LICENSE file.

// WARNING: Do not edit - generated code.

@DocsEditable()
@DomName('MediaSession')
@Experimental() // untriaged
class MediaSession extends DartHtmlDomObject {
  // To suppress missing implicit constructor warnings.
  factory MediaSession._() {
    throw new UnsupportedError("Not supported");
  }

  @DomName('MediaSession.MediaSession')
  @DocsEditable()
  factory MediaSession() {
    return _blink.BlinkMediaSession.instance.constructorCallback_0_();
  }

  @Deprecated("Internal Use Only")
  external static Type get instanceRuntimeType;

  @Deprecated("Internal Use Only")
  MediaSession.internal_() {}

  @DomName('MediaSession.activate')
  @DocsEditable()
  @Experimental() // untriaged
  void activate() =>
      _blink.BlinkMediaSession.instance.activate_Callback_0_(this);

  @DomName('MediaSession.deactivate')
  @DocsEditable()
  @Experimental() // untriaged
  void deactivate() =>
      _blink.BlinkMediaSession.instance.deactivate_Callback_0_(this);
}
// Copyright (c) 2012, the Dart project authors.  Please see the AUTHORS file
// for details. All rights reserved. Use of this source code is governed by a
// BSD-style license that can be found in the LICENSE file.

// WARNING: Do not edit - generated code.

@DocsEditable()
@DomName('MediaSource')
@SupportedBrowser(SupportedBrowser.CHROME)
@SupportedBrowser(SupportedBrowser.IE, '11')
// https://dvcs.w3.org/hg/html-media/raw-file/tip/media-source/media-source.html#mediasource
@Experimental()
class MediaSource extends EventTarget {
  // To suppress missing implicit constructor warnings.
  factory MediaSource._() {
    throw new UnsupportedError("Not supported");
  }

  @DomName('MediaSource.MediaSource')
  @DocsEditable()
  factory MediaSource() {
    return _blink.BlinkMediaSource.instance.constructorCallback_0_();
  }

  @Deprecated("Internal Use Only")
  external static Type get instanceRuntimeType;

  @Deprecated("Internal Use Only")
  MediaSource.internal_() : super.internal_();

  /// Checks if this type is supported on the current platform.
  static bool get supported => true;

  @DomName('MediaSource.activeSourceBuffers')
  @DocsEditable()
  SourceBufferList get activeSourceBuffers =>
      _blink.BlinkMediaSource.instance.activeSourceBuffers_Getter_(this);

  @DomName('MediaSource.duration')
  @DocsEditable()
  num get duration => _blink.BlinkMediaSource.instance.duration_Getter_(this);

  @DomName('MediaSource.duration')
  @DocsEditable()
  set duration(num value) =>
      _blink.BlinkMediaSource.instance.duration_Setter_(this, value);

  @DomName('MediaSource.readyState')
  @DocsEditable()
  String get readyState =>
      _blink.BlinkMediaSource.instance.readyState_Getter_(this);

  @DomName('MediaSource.sourceBuffers')
  @DocsEditable()
  SourceBufferList get sourceBuffers =>
      _blink.BlinkMediaSource.instance.sourceBuffers_Getter_(this);

  @DomName('MediaSource.addSourceBuffer')
  @DocsEditable()
  SourceBuffer addSourceBuffer(String type) =>
      _blink.BlinkMediaSource.instance.addSourceBuffer_Callback_1_(this, type);

  void endOfStream([String error]) {
    if (error != null) {
      _blink.BlinkMediaSource.instance.endOfStream_Callback_1_(this, error);
      return;
    }
    _blink.BlinkMediaSource.instance.endOfStream_Callback_0_(this);
    return;
  }

  @DomName('MediaSource.isTypeSupported')
  @DocsEditable()
  static bool isTypeSupported(String type) =>
      _blink.BlinkMediaSource.instance.isTypeSupported_Callback_1_(type);

  @DomName('MediaSource.removeSourceBuffer')
  @DocsEditable()
  void removeSourceBuffer(SourceBuffer buffer) =>
      _blink.BlinkMediaSource.instance
          .removeSourceBuffer_Callback_1_(this, buffer);
}
// Copyright (c) 2013, the Dart project authors.  Please see the AUTHORS file
// for details. All rights reserved. Use of this source code is governed by a
// BSD-style license that can be found in the LICENSE file.

@DomName('MediaStream')
@SupportedBrowser(SupportedBrowser.CHROME)
@Experimental()
// http://dev.w3.org/2011/webrtc/editor/getusermedia.html#mediastream
class MediaStream extends EventTarget {
  // To suppress missing implicit constructor warnings.
  factory MediaStream._() {
    throw new UnsupportedError("Not supported");
  }

  /**
   * Static factory designed to expose `addtrack` events to event
   * handlers that are not necessarily instances of [MediaStream].
   *
   * See [EventStreamProvider] for usage information.
   */
  @DomName('MediaStream.addtrackEvent')
  @DocsEditable()
  static const EventStreamProvider<Event> addTrackEvent =
      const EventStreamProvider<Event>('addtrack');

  /**
   * Static factory designed to expose `ended` events to event
   * handlers that are not necessarily instances of [MediaStream].
   *
   * See [EventStreamProvider] for usage information.
   */
  @DomName('MediaStream.endedEvent')
  @DocsEditable()
  static const EventStreamProvider<Event> endedEvent =
      const EventStreamProvider<Event>('ended');

  /**
   * Static factory designed to expose `removetrack` events to event
   * handlers that are not necessarily instances of [MediaStream].
   *
   * See [EventStreamProvider] for usage information.
   */
  @DomName('MediaStream.removetrackEvent')
  @DocsEditable()
  static const EventStreamProvider<Event> removeTrackEvent =
      const EventStreamProvider<Event>('removetrack');

  @DomName('MediaStream.MediaStream')
  @DocsEditable()
  factory MediaStream([stream_OR_tracks]) {
    if (stream_OR_tracks == null) {
      return _blink.BlinkMediaStream.instance.constructorCallback_0_();
    }
    if ((stream_OR_tracks is MediaStream || stream_OR_tracks == null)) {
      return _blink.BlinkMediaStream.instance
          .constructorCallback_1_(stream_OR_tracks);
    }
    if ((stream_OR_tracks is List<MediaStreamTrack> ||
        stream_OR_tracks == null)) {
      return _blink.BlinkMediaStream.instance
          .constructorCallback_1_(stream_OR_tracks);
    }
    throw new ArgumentError("Incorrect number or type of arguments");
  }

  @Deprecated("Internal Use Only")
  external static Type get instanceRuntimeType;

  @Deprecated("Internal Use Only")
  MediaStream.internal_() : super.internal_();

  @DomName('MediaStream.active')
  @DocsEditable()
  @Experimental() // untriaged
  bool get active => _blink.BlinkMediaStream.instance.active_Getter_(this);

  @DomName('MediaStream.ended')
  @DocsEditable()
  bool get ended => _blink.BlinkMediaStream.instance.ended_Getter_(this);

  @DomName('MediaStream.id')
  @DocsEditable()
  String get id => _blink.BlinkMediaStream.instance.id_Getter_(this);

  @DomName('MediaStream.label')
  @DocsEditable()
  @Experimental() // non-standard
  String get label => _blink.BlinkMediaStream.instance.label_Getter_(this);

  @DomName('MediaStream.addTrack')
  @DocsEditable()
  void addTrack(MediaStreamTrack track) =>
      _blink.BlinkMediaStream.instance.addTrack_Callback_1_(this, track);

  @DomName('MediaStream.clone')
  @DocsEditable()
  @Experimental() // untriaged
  MediaStream clone() =>
      _blink.BlinkMediaStream.instance.clone_Callback_0_(this);

  @DomName('MediaStream.getAudioTracks')
  @DocsEditable()
  List<MediaStreamTrack> getAudioTracks() =>
      (_blink.BlinkMediaStream.instance.getAudioTracks_Callback_0_(this));

  @DomName('MediaStream.getTrackById')
  @DocsEditable()
  MediaStreamTrack getTrackById(String trackId) =>
      _blink.BlinkMediaStream.instance.getTrackById_Callback_1_(this, trackId);

  @DomName('MediaStream.getTracks')
  @DocsEditable()
  @Experimental() // untriaged
  List<MediaStreamTrack> getTracks() =>
      (_blink.BlinkMediaStream.instance.getTracks_Callback_0_(this));

  @DomName('MediaStream.getVideoTracks')
  @DocsEditable()
  List<MediaStreamTrack> getVideoTracks() =>
      (_blink.BlinkMediaStream.instance.getVideoTracks_Callback_0_(this));

  @DomName('MediaStream.removeTrack')
  @DocsEditable()
  void removeTrack(MediaStreamTrack track) =>
      _blink.BlinkMediaStream.instance.removeTrack_Callback_1_(this, track);

  @DomName('MediaStream.stop')
  @DocsEditable()
  void stop() => _blink.BlinkMediaStream.instance.stop_Callback_0_(this);

  /// Stream of `addtrack` events handled by this [MediaStream].
  @DomName('MediaStream.onaddtrack')
  @DocsEditable()
  Stream<Event> get onAddTrack => addTrackEvent.forTarget(this);

  /// Stream of `ended` events handled by this [MediaStream].
  @DomName('MediaStream.onended')
  @DocsEditable()
  Stream<Event> get onEnded => endedEvent.forTarget(this);

  /// Stream of `removetrack` events handled by this [MediaStream].
  @DomName('MediaStream.onremovetrack')
  @DocsEditable()
  Stream<Event> get onRemoveTrack => removeTrackEvent.forTarget(this);

  /**
   * Checks if the MediaStream APIs are supported on the current platform.
   *
   * See also:
   *
   * * [Navigator.getUserMedia]
   */
  static bool get supported => true;
}
// Copyright (c) 2012, the Dart project authors.  Please see the AUTHORS file
// for details. All rights reserved. Use of this source code is governed by a
// BSD-style license that can be found in the LICENSE file.

// WARNING: Do not edit - generated code.

@DocsEditable()
@DomName('MediaStreamEvent')
@SupportedBrowser(SupportedBrowser.CHROME)
@Experimental()
// http://dev.w3.org/2011/webrtc/editor/getusermedia.html
class MediaStreamEvent extends Event {
  // To suppress missing implicit constructor warnings.
  factory MediaStreamEvent._() {
    throw new UnsupportedError("Not supported");
  }

  @DomName('MediaStreamEvent.MediaStreamEvent')
  @DocsEditable()
  factory MediaStreamEvent(String type, [Map eventInitDict]) {
    if (eventInitDict != null) {
      var eventInitDict_1 = convertDartToNative_Dictionary(eventInitDict);
      return _blink.BlinkMediaStreamEvent.instance
          .constructorCallback_2_(type, eventInitDict_1);
    }
    return _blink.BlinkMediaStreamEvent.instance.constructorCallback_1_(type);
  }

  @Deprecated("Internal Use Only")
  external static Type get instanceRuntimeType;

  @Deprecated("Internal Use Only")
  MediaStreamEvent.internal_() : super.internal_();

  /// Checks if this type is supported on the current platform.
  static bool get supported => true;

  @DomName('MediaStreamEvent.stream')
  @DocsEditable()
  MediaStream get stream =>
      _blink.BlinkMediaStreamEvent.instance.stream_Getter_(this);
}
// Copyright (c) 2012, the Dart project authors.  Please see the AUTHORS file
// for details. All rights reserved. Use of this source code is governed by a
// BSD-style license that can be found in the LICENSE file.

// WARNING: Do not edit - generated code.

@DocsEditable()
@DomName('MediaStreamTrack')
@SupportedBrowser(SupportedBrowser.CHROME)
@Experimental()
// http://dev.w3.org/2011/webrtc/editor/getusermedia.html#mediastreamtrack
class MediaStreamTrack extends EventTarget {
  // To suppress missing implicit constructor warnings.
  factory MediaStreamTrack._() {
    throw new UnsupportedError("Not supported");
  }

  /**
   * Static factory designed to expose `ended` events to event
   * handlers that are not necessarily instances of [MediaStreamTrack].
   *
   * See [EventStreamProvider] for usage information.
   */
  @DomName('MediaStreamTrack.endedEvent')
  @DocsEditable()
  static const EventStreamProvider<Event> endedEvent =
      const EventStreamProvider<Event>('ended');

  /**
   * Static factory designed to expose `mute` events to event
   * handlers that are not necessarily instances of [MediaStreamTrack].
   *
   * See [EventStreamProvider] for usage information.
   */
  @DomName('MediaStreamTrack.muteEvent')
  @DocsEditable()
  static const EventStreamProvider<Event> muteEvent =
      const EventStreamProvider<Event>('mute');

  /**
   * Static factory designed to expose `unmute` events to event
   * handlers that are not necessarily instances of [MediaStreamTrack].
   *
   * See [EventStreamProvider] for usage information.
   */
  @DomName('MediaStreamTrack.unmuteEvent')
  @DocsEditable()
  static const EventStreamProvider<Event> unmuteEvent =
      const EventStreamProvider<Event>('unmute');

  @Deprecated("Internal Use Only")
  external static Type get instanceRuntimeType;

  @Deprecated("Internal Use Only")
  MediaStreamTrack.internal_() : super.internal_();

  @DomName('MediaStreamTrack.enabled')
  @DocsEditable()
  bool get enabled =>
      _blink.BlinkMediaStreamTrack.instance.enabled_Getter_(this);

  @DomName('MediaStreamTrack.enabled')
  @DocsEditable()
  set enabled(bool value) =>
      _blink.BlinkMediaStreamTrack.instance.enabled_Setter_(this, value);

  @DomName('MediaStreamTrack.id')
  @DocsEditable()
  String get id => _blink.BlinkMediaStreamTrack.instance.id_Getter_(this);

  @DomName('MediaStreamTrack.kind')
  @DocsEditable()
  String get kind => _blink.BlinkMediaStreamTrack.instance.kind_Getter_(this);

  @DomName('MediaStreamTrack.label')
  @DocsEditable()
  String get label => _blink.BlinkMediaStreamTrack.instance.label_Getter_(this);

  @DomName('MediaStreamTrack.muted')
  @DocsEditable()
  @Experimental() // untriaged
  bool get muted => _blink.BlinkMediaStreamTrack.instance.muted_Getter_(this);

  @DomName('MediaStreamTrack.readyState')
  @DocsEditable()
  String get readyState =>
      _blink.BlinkMediaStreamTrack.instance.readyState_Getter_(this);

  @DomName('MediaStreamTrack.clone')
  @DocsEditable()
  @Experimental() // untriaged
  MediaStreamTrack clone() =>
      _blink.BlinkMediaStreamTrack.instance.clone_Callback_0_(this);

  @DomName('MediaStreamTrack.getSources')
  @DocsEditable()
  @Experimental() // untriaged
  static void _getSources(MediaStreamTrackSourcesCallback callback) =>
      _blink.BlinkMediaStreamTrack.instance.getSources_Callback_1_(callback);

  static Future<List<SourceInfo>> getSources() {
    var completer = new Completer<List<SourceInfo>>();
    _getSources((value) {
      completer.complete(value);
    });
    return completer.future;
  }

  @DomName('MediaStreamTrack.stop')
  @DocsEditable()
  @Experimental() // untriaged
  void stop() => _blink.BlinkMediaStreamTrack.instance.stop_Callback_0_(this);

  /// Stream of `ended` events handled by this [MediaStreamTrack].
  @DomName('MediaStreamTrack.onended')
  @DocsEditable()
  Stream<Event> get onEnded => endedEvent.forTarget(this);

  /// Stream of `mute` events handled by this [MediaStreamTrack].
  @DomName('MediaStreamTrack.onmute')
  @DocsEditable()
  Stream<Event> get onMute => muteEvent.forTarget(this);

  /// Stream of `unmute` events handled by this [MediaStreamTrack].
  @DomName('MediaStreamTrack.onunmute')
  @DocsEditable()
  Stream<Event> get onUnmute => unmuteEvent.forTarget(this);
}
// Copyright (c) 2012, the Dart project authors.  Please see the AUTHORS file
// for details. All rights reserved. Use of this source code is governed by a
// BSD-style license that can be found in the LICENSE file.

// WARNING: Do not edit - generated code.

@DocsEditable()
@DomName('MediaStreamTrackEvent')
@SupportedBrowser(SupportedBrowser.CHROME)
@Experimental()
// http://dev.w3.org/2011/webrtc/editor/getusermedia.html
class MediaStreamTrackEvent extends Event {
  // To suppress missing implicit constructor warnings.
  factory MediaStreamTrackEvent._() {
    throw new UnsupportedError("Not supported");
  }

  @Deprecated("Internal Use Only")
  external static Type get instanceRuntimeType;

  @Deprecated("Internal Use Only")
  MediaStreamTrackEvent.internal_() : super.internal_();

  /// Checks if this type is supported on the current platform.
  static bool get supported => true;

  @DomName('MediaStreamTrackEvent.track')
  @DocsEditable()
  MediaStreamTrack get track =>
      _blink.BlinkMediaStreamTrackEvent.instance.track_Getter_(this);
}
// Copyright (c) 2012, the Dart project authors.  Please see the AUTHORS file
// for details. All rights reserved. Use of this source code is governed by a
// BSD-style license that can be found in the LICENSE file.

// WARNING: Do not edit - generated code.

@DomName('MediaStreamTrackSourcesCallback')
@Experimental() // untriaged
typedef void MediaStreamTrackSourcesCallback(List<SourceInfo> sources);
// Copyright (c) 2012, the Dart project authors.  Please see the AUTHORS file
// for details. All rights reserved. Use of this source code is governed by a
// BSD-style license that can be found in the LICENSE file.

// WARNING: Do not edit - generated code.

@DocsEditable()
@DomName('MemoryInfo')
@Experimental() // nonstandard
class MemoryInfo extends DartHtmlDomObject {
  // To suppress missing implicit constructor warnings.
  factory MemoryInfo._() {
    throw new UnsupportedError("Not supported");
  }

  @Deprecated("Internal Use Only")
  external static Type get instanceRuntimeType;

  @Deprecated("Internal Use Only")
  MemoryInfo.internal_() {}

  @DomName('MemoryInfo.jsHeapSizeLimit')
  @DocsEditable()
  int get jsHeapSizeLimit =>
      _blink.BlinkMemoryInfo.instance.jsHeapSizeLimit_Getter_(this);

  @DomName('MemoryInfo.totalJSHeapSize')
  @DocsEditable()
  int get totalJSHeapSize =>
      _blink.BlinkMemoryInfo.instance.totalJSHeapSize_Getter_(this);

  @DomName('MemoryInfo.usedJSHeapSize')
  @DocsEditable()
  int get usedJSHeapSize =>
      _blink.BlinkMemoryInfo.instance.usedJSHeapSize_Getter_(this);
}
// Copyright (c) 2012, the Dart project authors.  Please see the AUTHORS file
// for details. All rights reserved. Use of this source code is governed by a
// BSD-style license that can be found in the LICENSE file.

// WARNING: Do not edit - generated code.

@DocsEditable()
/**
 * An HTML <menu> element.
 *
 * A <menu> element represents an unordered list of menu commands.
 *
 * See also:
 *
 *  * [Menu Element](https://developer.mozilla.org/en-US/docs/HTML/Element/menu) from MDN.
 *  * [Menu Element](http://www.w3.org/TR/html5/the-menu-element.html#the-menu-element) from the W3C.
 */
@DomName('HTMLMenuElement')
class MenuElement extends HtmlElement {
  // To suppress missing implicit constructor warnings.
  factory MenuElement._() {
    throw new UnsupportedError("Not supported");
  }

  @DomName('HTMLMenuElement.HTMLMenuElement')
  @DocsEditable()
  factory MenuElement() => document.createElement("menu");

  @Deprecated("Internal Use Only")
  external static Type get instanceRuntimeType;

  @Deprecated("Internal Use Only")
  MenuElement.internal_() : super.internal_();

  /**
   * Constructor instantiated by the DOM when a custom element has been created.
   *
   * This can only be called by subclasses from their created constructor.
   */
  MenuElement.created() : super.created();

  @DomName('HTMLMenuElement.label')
  @DocsEditable()
  @Experimental() // untriaged
  String get label => _blink.BlinkHTMLMenuElement.instance.label_Getter_(this);

  @DomName('HTMLMenuElement.label')
  @DocsEditable()
  @Experimental() // untriaged
  set label(String value) =>
      _blink.BlinkHTMLMenuElement.instance.label_Setter_(this, value);

  @DomName('HTMLMenuElement.type')
  @DocsEditable()
  @Experimental() // untriaged
  String get type => _blink.BlinkHTMLMenuElement.instance.type_Getter_(this);

  @DomName('HTMLMenuElement.type')
  @DocsEditable()
  @Experimental() // untriaged
  set type(String value) =>
      _blink.BlinkHTMLMenuElement.instance.type_Setter_(this, value);
}
// Copyright (c) 2012, the Dart project authors.  Please see the AUTHORS file
// for details. All rights reserved. Use of this source code is governed by a
// BSD-style license that can be found in the LICENSE file.

// WARNING: Do not edit - generated code.

@DocsEditable()
@DomName('HTMLMenuItemElement')
@Experimental() // untriaged
class MenuItemElement extends HtmlElement {
  // To suppress missing implicit constructor warnings.
  factory MenuItemElement._() {
    throw new UnsupportedError("Not supported");
  }

  @Deprecated("Internal Use Only")
  external static Type get instanceRuntimeType;

  @Deprecated("Internal Use Only")
  MenuItemElement.internal_() : super.internal_();

  /**
   * Constructor instantiated by the DOM when a custom element has been created.
   *
   * This can only be called by subclasses from their created constructor.
   */
  MenuItemElement.created() : super.created();

  @DomName('HTMLMenuItemElement.checked')
  @DocsEditable()
  @Experimental() // untriaged
  bool get checked =>
      _blink.BlinkHTMLMenuItemElement.instance.checked_Getter_(this);

  @DomName('HTMLMenuItemElement.checked')
  @DocsEditable()
  @Experimental() // untriaged
  set checked(bool value) =>
      _blink.BlinkHTMLMenuItemElement.instance.checked_Setter_(this, value);

  @DomName('HTMLMenuItemElement.default')
  @DocsEditable()
  @Experimental() // untriaged
  bool get defaultValue =>
      _blink.BlinkHTMLMenuItemElement.instance.default_Getter_(this);

  @DomName('HTMLMenuItemElement.default')
  @DocsEditable()
  @Experimental() // untriaged
  set defaultValue(bool value) =>
      _blink.BlinkHTMLMenuItemElement.instance.default_Setter_(this, value);

  @DomName('HTMLMenuItemElement.disabled')
  @DocsEditable()
  @Experimental() // untriaged
  bool get disabled =>
      _blink.BlinkHTMLMenuItemElement.instance.disabled_Getter_(this);

  @DomName('HTMLMenuItemElement.disabled')
  @DocsEditable()
  @Experimental() // untriaged
  set disabled(bool value) =>
      _blink.BlinkHTMLMenuItemElement.instance.disabled_Setter_(this, value);

  @DomName('HTMLMenuItemElement.icon')
  @DocsEditable()
  @Experimental() // untriaged
  String get icon =>
      _blink.BlinkHTMLMenuItemElement.instance.icon_Getter_(this);

  @DomName('HTMLMenuItemElement.icon')
  @DocsEditable()
  @Experimental() // untriaged
  set icon(String value) =>
      _blink.BlinkHTMLMenuItemElement.instance.icon_Setter_(this, value);

  @DomName('HTMLMenuItemElement.label')
  @DocsEditable()
  @Experimental() // untriaged
  String get label =>
      _blink.BlinkHTMLMenuItemElement.instance.label_Getter_(this);

  @DomName('HTMLMenuItemElement.label')
  @DocsEditable()
  @Experimental() // untriaged
  set label(String value) =>
      _blink.BlinkHTMLMenuItemElement.instance.label_Setter_(this, value);

  @DomName('HTMLMenuItemElement.radiogroup')
  @DocsEditable()
  @Experimental() // untriaged
  String get radiogroup =>
      _blink.BlinkHTMLMenuItemElement.instance.radiogroup_Getter_(this);

  @DomName('HTMLMenuItemElement.radiogroup')
  @DocsEditable()
  @Experimental() // untriaged
  set radiogroup(String value) =>
      _blink.BlinkHTMLMenuItemElement.instance.radiogroup_Setter_(this, value);

  @DomName('HTMLMenuItemElement.type')
  @DocsEditable()
  @Experimental() // untriaged
  String get type =>
      _blink.BlinkHTMLMenuItemElement.instance.type_Getter_(this);

  @DomName('HTMLMenuItemElement.type')
  @DocsEditable()
  @Experimental() // untriaged
  set type(String value) =>
      _blink.BlinkHTMLMenuItemElement.instance.type_Setter_(this, value);
}
// Copyright (c) 2012, the Dart project authors.  Please see the AUTHORS file
// for details. All rights reserved. Use of this source code is governed by a
// BSD-style license that can be found in the LICENSE file.

// WARNING: Do not edit - generated code.

@DocsEditable()
@DomName('MessageChannel')
@Unstable()
class MessageChannel extends DartHtmlDomObject {
  // To suppress missing implicit constructor warnings.
  factory MessageChannel._() {
    throw new UnsupportedError("Not supported");
  }

  @Deprecated("Internal Use Only")
  external static Type get instanceRuntimeType;

  @Deprecated("Internal Use Only")
  MessageChannel.internal_() {}

  @DomName('MessageChannel.port1')
  @DocsEditable()
  MessagePort get port1 =>
      _blink.BlinkMessageChannel.instance.port1_Getter_(this);

  @DomName('MessageChannel.port2')
  @DocsEditable()
  MessagePort get port2 =>
      _blink.BlinkMessageChannel.instance.port2_Getter_(this);
}
// Copyright (c) 2013, the Dart project authors.  Please see the AUTHORS file
// for details. All rights reserved. Use of this source code is governed by a
// BSD-style license that can be found in the LICENSE file.

// WARNING: Do not edit - generated code.

@DomName('MessageEvent')
class MessageEvent extends Event {
  factory MessageEvent(String type,
      {bool canBubble: false,
      bool cancelable: false,
      Object data,
      String origin,
      String lastEventId,
      Window source,
      List<MessagePort> messagePorts}) {
    if (source == null) {
      source = window;
    }
    MessageEvent event = document._createEvent("MessageEvent");
    event._initMessageEvent(type, canBubble, cancelable, data, origin,
        lastEventId, source, messagePorts);
    return event;
  }

  // TODO(alanknight): This really should be generated by the
  // _OutputConversion in the systemnative.py script, but that doesn't
  // use those conversions right now, so do this as a one-off.
  @DomName('MessageEvent.data')
  @DocsEditable()
  dynamic get data => convertNativeToDart_SerializedScriptValue(
      _blink.BlinkMessageEvent.instance.data_Getter_(this));

  @DomName('MessageEvent.MessageEvent')
  @DocsEditable()
  factory MessageEvent._(String type, [Map eventInitDict]) {
    if (eventInitDict != null) {
      var eventInitDict_1 = convertDartToNative_Dictionary(eventInitDict);
      return _blink.BlinkMessageEvent.instance
          .constructorCallback_2_(type, eventInitDict_1);
    }
    return _blink.BlinkMessageEvent.instance.constructorCallback_1_(type);
  }

  @Deprecated("Internal Use Only")
  external static Type get instanceRuntimeType;

  @Deprecated("Internal Use Only")
  MessageEvent.internal_() : super.internal_();

  @DomName('MessageEvent.lastEventId')
  @DocsEditable()
  @Unstable()
  String get lastEventId =>
      _blink.BlinkMessageEvent.instance.lastEventId_Getter_(this);

  @DomName('MessageEvent.origin')
  @DocsEditable()
  String get origin => _blink.BlinkMessageEvent.instance.origin_Getter_(this);

  @DomName('MessageEvent.source')
  @DocsEditable()
  EventTarget get source => _convertNativeToDart_EventTarget(
      _blink.BlinkMessageEvent.instance.source_Getter_(this));

  @DomName('MessageEvent.initMessageEvent')
  @DocsEditable()
  void _initMessageEvent(
          String typeArg,
          bool canBubbleArg,
          bool cancelableArg,
          Object dataArg,
          String originArg,
          String lastEventIdArg,
          Window sourceArg,
          List<MessagePort> portsArg) =>
      _blink.BlinkMessageEvent.instance.initMessageEvent_Callback_8_(
          this,
          typeArg,
          canBubbleArg,
          cancelableArg,
          dataArg,
          originArg,
          lastEventIdArg,
          sourceArg,
          portsArg);
}
// Copyright (c) 2012, the Dart project authors.  Please see the AUTHORS file
// for details. All rights reserved. Use of this source code is governed by a
// BSD-style license that can be found in the LICENSE file.

// WARNING: Do not edit - generated code.

@DocsEditable()
@DomName('MessagePort')
@Unstable()
class MessagePort extends EventTarget {
  // To suppress missing implicit constructor warnings.
  factory MessagePort._() {
    throw new UnsupportedError("Not supported");
  }

  /**
   * Static factory designed to expose `message` events to event
   * handlers that are not necessarily instances of [MessagePort].
   *
   * See [EventStreamProvider] for usage information.
   */
  @DomName('MessagePort.messageEvent')
  @DocsEditable()
  static const EventStreamProvider<MessageEvent> messageEvent =
      const EventStreamProvider<MessageEvent>('message');

  @Deprecated("Internal Use Only")
  external static Type get instanceRuntimeType;

  @Deprecated("Internal Use Only")
  MessagePort.internal_() : super.internal_();

  @DomName('MessagePort.close')
  @DocsEditable()
  void close() => _blink.BlinkMessagePort.instance.close_Callback_0_(this);

  void postMessage(Object message, [List<MessagePort> transfer]) {
    if (transfer != null) {
      _blink.BlinkMessagePort.instance.postMessage_Callback_2_(
          this, convertDartToNative_SerializedScriptValue(message), transfer);
      return;
    }
    _blink.BlinkMessagePort.instance.postMessage_Callback_1_(
        this, convertDartToNative_SerializedScriptValue(message));
    return;
  }

  @DomName('MessagePort.start')
  @DocsEditable()
  void start() => _blink.BlinkMessagePort.instance.start_Callback_0_(this);

  /// Stream of `message` events handled by this [MessagePort].
  @DomName('MessagePort.onmessage')
  @DocsEditable()
  Stream<MessageEvent> get onMessage => messageEvent.forTarget(this);
}
// Copyright (c) 2012, the Dart project authors.  Please see the AUTHORS file
// for details. All rights reserved. Use of this source code is governed by a
// BSD-style license that can be found in the LICENSE file.

// WARNING: Do not edit - generated code.

@DocsEditable()
@DomName('HTMLMetaElement')
class MetaElement extends HtmlElement {
  // To suppress missing implicit constructor warnings.
  factory MetaElement._() {
    throw new UnsupportedError("Not supported");
  }

  @DomName('HTMLMetaElement.HTMLMetaElement')
  @DocsEditable()
  factory MetaElement() => document.createElement("meta");

  @Deprecated("Internal Use Only")
  external static Type get instanceRuntimeType;

  @Deprecated("Internal Use Only")
  MetaElement.internal_() : super.internal_();

  /**
   * Constructor instantiated by the DOM when a custom element has been created.
   *
   * This can only be called by subclasses from their created constructor.
   */
  MetaElement.created() : super.created();

  @DomName('HTMLMetaElement.content')
  @DocsEditable()
  String get content =>
      _blink.BlinkHTMLMetaElement.instance.content_Getter_(this);

  @DomName('HTMLMetaElement.content')
  @DocsEditable()
  set content(String value) =>
      _blink.BlinkHTMLMetaElement.instance.content_Setter_(this, value);

  @DomName('HTMLMetaElement.httpEquiv')
  @DocsEditable()
  String get httpEquiv =>
      _blink.BlinkHTMLMetaElement.instance.httpEquiv_Getter_(this);

  @DomName('HTMLMetaElement.httpEquiv')
  @DocsEditable()
  set httpEquiv(String value) =>
      _blink.BlinkHTMLMetaElement.instance.httpEquiv_Setter_(this, value);

  @DomName('HTMLMetaElement.name')
  @DocsEditable()
  String get name => _blink.BlinkHTMLMetaElement.instance.name_Getter_(this);

  @DomName('HTMLMetaElement.name')
  @DocsEditable()
  set name(String value) =>
      _blink.BlinkHTMLMetaElement.instance.name_Setter_(this, value);
}
// Copyright (c) 2012, the Dart project authors.  Please see the AUTHORS file
// for details. All rights reserved. Use of this source code is governed by a
// BSD-style license that can be found in the LICENSE file.

// WARNING: Do not edit - generated code.

@DocsEditable()
@DomName('Metadata')
// http://www.w3.org/TR/file-system-api/#the-metadata-interface
@Experimental()
class Metadata extends DartHtmlDomObject {
  // To suppress missing implicit constructor warnings.
  factory Metadata._() {
    throw new UnsupportedError("Not supported");
  }

  @Deprecated("Internal Use Only")
  external static Type get instanceRuntimeType;

  @Deprecated("Internal Use Only")
  Metadata.internal_() {}

  @DomName('Metadata.modificationTime')
  @DocsEditable()
  DateTime get modificationTime =>
      _blink.BlinkMetadata.instance.modificationTime_Getter_(this);

  @DomName('Metadata.size')
  @DocsEditable()
  int get size => _blink.BlinkMetadata.instance.size_Getter_(this);
}
// Copyright (c) 2012, the Dart project authors.  Please see the AUTHORS file
// for details. All rights reserved. Use of this source code is governed by a
// BSD-style license that can be found in the LICENSE file.

// WARNING: Do not edit - generated code.

@DomName('MetadataCallback')
// http://www.w3.org/TR/file-system-api/#idl-def-MetadataCallback
@Experimental()
typedef void MetadataCallback(Metadata metadata);
// Copyright (c) 2012, the Dart project authors.  Please see the AUTHORS file
// for details. All rights reserved. Use of this source code is governed by a
// BSD-style license that can be found in the LICENSE file.

// WARNING: Do not edit - generated code.

@DocsEditable()
@DomName('HTMLMeterElement')
@SupportedBrowser(SupportedBrowser.CHROME)
@SupportedBrowser(SupportedBrowser.FIREFOX)
@SupportedBrowser(SupportedBrowser.SAFARI)
@Unstable()
class MeterElement extends HtmlElement {
  // To suppress missing implicit constructor warnings.
  factory MeterElement._() {
    throw new UnsupportedError("Not supported");
  }

  @DomName('HTMLMeterElement.HTMLMeterElement')
  @DocsEditable()
  factory MeterElement() => document.createElement("meter");

  @Deprecated("Internal Use Only")
  external static Type get instanceRuntimeType;

  @Deprecated("Internal Use Only")
  MeterElement.internal_() : super.internal_();

  /**
   * Constructor instantiated by the DOM when a custom element has been created.
   *
   * This can only be called by subclasses from their created constructor.
   */
  MeterElement.created() : super.created();

  /// Checks if this type is supported on the current platform.
  static bool get supported => true;

  @DomName('HTMLMeterElement.high')
  @DocsEditable()
  num get high => _blink.BlinkHTMLMeterElement.instance.high_Getter_(this);

  @DomName('HTMLMeterElement.high')
  @DocsEditable()
  set high(num value) =>
      _blink.BlinkHTMLMeterElement.instance.high_Setter_(this, value);

  @DomName('HTMLMeterElement.labels')
  @DocsEditable()
  @Unstable()
  List<Node> get labels =>
      (_blink.BlinkHTMLMeterElement.instance.labels_Getter_(this));

  @DomName('HTMLMeterElement.low')
  @DocsEditable()
  num get low => _blink.BlinkHTMLMeterElement.instance.low_Getter_(this);

  @DomName('HTMLMeterElement.low')
  @DocsEditable()
  set low(num value) =>
      _blink.BlinkHTMLMeterElement.instance.low_Setter_(this, value);

  @DomName('HTMLMeterElement.max')
  @DocsEditable()
  num get max => _blink.BlinkHTMLMeterElement.instance.max_Getter_(this);

  @DomName('HTMLMeterElement.max')
  @DocsEditable()
  set max(num value) =>
      _blink.BlinkHTMLMeterElement.instance.max_Setter_(this, value);

  @DomName('HTMLMeterElement.min')
  @DocsEditable()
  num get min => _blink.BlinkHTMLMeterElement.instance.min_Getter_(this);

  @DomName('HTMLMeterElement.min')
  @DocsEditable()
  set min(num value) =>
      _blink.BlinkHTMLMeterElement.instance.min_Setter_(this, value);

  @DomName('HTMLMeterElement.optimum')
  @DocsEditable()
  num get optimum =>
      _blink.BlinkHTMLMeterElement.instance.optimum_Getter_(this);

  @DomName('HTMLMeterElement.optimum')
  @DocsEditable()
  set optimum(num value) =>
      _blink.BlinkHTMLMeterElement.instance.optimum_Setter_(this, value);

  @DomName('HTMLMeterElement.value')
  @DocsEditable()
  num get value => _blink.BlinkHTMLMeterElement.instance.value_Getter_(this);

  @DomName('HTMLMeterElement.value')
  @DocsEditable()
  set value(num value) =>
      _blink.BlinkHTMLMeterElement.instance.value_Setter_(this, value);
}
// Copyright (c) 2012, the Dart project authors.  Please see the AUTHORS file
// for details. All rights reserved. Use of this source code is governed by a
// BSD-style license that can be found in the LICENSE file.

// WARNING: Do not edit - generated code.

@DocsEditable()
@DomName('MIDIAccess')
// http://webaudio.github.io/web-midi-api/#midiaccess-interface
@Experimental()
class MidiAccess extends EventTarget {
  // To suppress missing implicit constructor warnings.
  factory MidiAccess._() {
    throw new UnsupportedError("Not supported");
  }

  @Deprecated("Internal Use Only")
  external static Type get instanceRuntimeType;

  @Deprecated("Internal Use Only")
  MidiAccess.internal_() : super.internal_();

  @DomName('MIDIAccess.inputs')
  @DocsEditable()
  MidiInputMap get inputs =>
      _blink.BlinkMIDIAccess.instance.inputs_Getter_(this);

  @DomName('MIDIAccess.outputs')
  @DocsEditable()
  MidiOutputMap get outputs =>
      _blink.BlinkMIDIAccess.instance.outputs_Getter_(this);

  @DomName('MIDIAccess.sysexEnabled')
  @DocsEditable()
  @Experimental() // untriaged
  bool get sysexEnabled =>
      _blink.BlinkMIDIAccess.instance.sysexEnabled_Getter_(this);
}
// Copyright (c) 2012, the Dart project authors.  Please see the AUTHORS file
// for details. All rights reserved. Use of this source code is governed by a
// BSD-style license that can be found in the LICENSE file.

// WARNING: Do not edit - generated code.

@DocsEditable()
@DomName('MIDIConnectionEvent')
// http://webaudio.github.io/web-midi-api/#midiconnectionevent-interface
@Experimental()
class MidiConnectionEvent extends Event {
  // To suppress missing implicit constructor warnings.
  factory MidiConnectionEvent._() {
    throw new UnsupportedError("Not supported");
  }

  @DomName('MIDIConnectionEvent.MIDIConnectionEvent')
  @DocsEditable()
  factory MidiConnectionEvent(String type, [Map eventInitDict]) {
    if (eventInitDict != null) {
      var eventInitDict_1 = convertDartToNative_Dictionary(eventInitDict);
      return _blink.BlinkMIDIConnectionEvent.instance
          .constructorCallback_2_(type, eventInitDict_1);
    }
    return _blink.BlinkMIDIConnectionEvent.instance
        .constructorCallback_1_(type);
  }

  @Deprecated("Internal Use Only")
  external static Type get instanceRuntimeType;

  @Deprecated("Internal Use Only")
  MidiConnectionEvent.internal_() : super.internal_();

  @DomName('MIDIConnectionEvent.port')
  @DocsEditable()
  MidiPort get port =>
      _blink.BlinkMIDIConnectionEvent.instance.port_Getter_(this);
}
// Copyright (c) 2012, the Dart project authors.  Please see the AUTHORS file
// for details. All rights reserved. Use of this source code is governed by a
// BSD-style license that can be found in the LICENSE file.

// WARNING: Do not edit - generated code.

@DocsEditable()
@DomName('MIDIInput')
// http://webaudio.github.io/web-midi-api/#idl-def-MIDIInput
@Experimental()
class MidiInput extends MidiPort {
  // To suppress missing implicit constructor warnings.
  factory MidiInput._() {
    throw new UnsupportedError("Not supported");
  }

  /**
   * Static factory designed to expose `midimessage` events to event
   * handlers that are not necessarily instances of [MidiInput].
   *
   * See [EventStreamProvider] for usage information.
   */
  @DomName('MIDIInput.midimessageEvent')
  @DocsEditable()
  static const EventStreamProvider<MidiMessageEvent> midiMessageEvent =
      const EventStreamProvider<MidiMessageEvent>('midimessage');

  @Deprecated("Internal Use Only")
  external static Type get instanceRuntimeType;

  @Deprecated("Internal Use Only")
  MidiInput.internal_() : super.internal_();

  /// Stream of `midimessage` events handled by this [MidiInput].
  @DomName('MIDIInput.onmidimessage')
  @DocsEditable()
  Stream<MidiMessageEvent> get onMidiMessage =>
      midiMessageEvent.forTarget(this);
}
// Copyright (c) 2012, the Dart project authors.  Please see the AUTHORS file
// for details. All rights reserved. Use of this source code is governed by a
// BSD-style license that can be found in the LICENSE file.

// WARNING: Do not edit - generated code.

@DocsEditable()
@DomName('MIDIInputMap')
@Experimental() // untriaged
class MidiInputMap extends DartHtmlDomObject {
  // To suppress missing implicit constructor warnings.
  factory MidiInputMap._() {
    throw new UnsupportedError("Not supported");
  }

  @Deprecated("Internal Use Only")
  external static Type get instanceRuntimeType;

  @Deprecated("Internal Use Only")
  MidiInputMap.internal_() {}

  @DomName('MIDIInputMap.size')
  @DocsEditable()
  @Experimental() // untriaged
  int get size => _blink.BlinkMIDIInputMap.instance.size_Getter_(this);
}
// Copyright (c) 2012, the Dart project authors.  Please see the AUTHORS file
// for details. All rights reserved. Use of this source code is governed by a
// BSD-style license that can be found in the LICENSE file.

// WARNING: Do not edit - generated code.

@DocsEditable()
@DomName('MIDIMessageEvent')
// http://webaudio.github.io/web-midi-api/#midimessageevent-interface
@Experimental()
class MidiMessageEvent extends Event {
  // To suppress missing implicit constructor warnings.
  factory MidiMessageEvent._() {
    throw new UnsupportedError("Not supported");
  }

  @DomName('MIDIMessageEvent.MIDIMessageEvent')
  @DocsEditable()
  factory MidiMessageEvent(String type, [Map eventInitDict]) {
    if (eventInitDict != null) {
      var eventInitDict_1 = convertDartToNative_Dictionary(eventInitDict);
      return _blink.BlinkMIDIMessageEvent.instance
          .constructorCallback_2_(type, eventInitDict_1);
    }
    return _blink.BlinkMIDIMessageEvent.instance.constructorCallback_1_(type);
  }

  @Deprecated("Internal Use Only")
  external static Type get instanceRuntimeType;

  @Deprecated("Internal Use Only")
  MidiMessageEvent.internal_() : super.internal_();

  @DomName('MIDIMessageEvent.data')
  @DocsEditable()
  Uint8List get data =>
      _blink.BlinkMIDIMessageEvent.instance.data_Getter_(this);

  @DomName('MIDIMessageEvent.receivedTime')
  @DocsEditable()
  num get receivedTime =>
      _blink.BlinkMIDIMessageEvent.instance.receivedTime_Getter_(this);
}
// Copyright (c) 2012, the Dart project authors.  Please see the AUTHORS file
// for details. All rights reserved. Use of this source code is governed by a
// BSD-style license that can be found in the LICENSE file.

// WARNING: Do not edit - generated code.

@DocsEditable()
@DomName('MIDIOutput')
// http://webaudio.github.io/web-midi-api/#midioutput-interface
@Experimental()
class MidiOutput extends MidiPort {
  // To suppress missing implicit constructor warnings.
  factory MidiOutput._() {
    throw new UnsupportedError("Not supported");
  }

  @Deprecated("Internal Use Only")
  external static Type get instanceRuntimeType;

  @Deprecated("Internal Use Only")
  MidiOutput.internal_() : super.internal_();

  void send(Uint8List data, [num timestamp]) {
    if (timestamp != null) {
      _blink.BlinkMIDIOutput.instance.send_Callback_2_(this, data, timestamp);
      return;
    }
    _blink.BlinkMIDIOutput.instance.send_Callback_1_(this, data);
    return;
  }
}
// Copyright (c) 2012, the Dart project authors.  Please see the AUTHORS file
// for details. All rights reserved. Use of this source code is governed by a
// BSD-style license that can be found in the LICENSE file.

// WARNING: Do not edit - generated code.

@DocsEditable()
@DomName('MIDIOutputMap')
@Experimental() // untriaged
class MidiOutputMap extends DartHtmlDomObject {
  // To suppress missing implicit constructor warnings.
  factory MidiOutputMap._() {
    throw new UnsupportedError("Not supported");
  }

  @Deprecated("Internal Use Only")
  external static Type get instanceRuntimeType;

  @Deprecated("Internal Use Only")
  MidiOutputMap.internal_() {}

  @DomName('MIDIOutputMap.size')
  @DocsEditable()
  @Experimental() // untriaged
  int get size => _blink.BlinkMIDIOutputMap.instance.size_Getter_(this);
}
// Copyright (c) 2012, the Dart project authors.  Please see the AUTHORS file
// for details. All rights reserved. Use of this source code is governed by a
// BSD-style license that can be found in the LICENSE file.

// WARNING: Do not edit - generated code.

@DocsEditable()
@DomName('MIDIPort')
// http://webaudio.github.io/web-midi-api/#idl-def-MIDIPort
@Experimental()
class MidiPort extends EventTarget {
  // To suppress missing implicit constructor warnings.
  factory MidiPort._() {
    throw new UnsupportedError("Not supported");
  }

  @Deprecated("Internal Use Only")
  external static Type get instanceRuntimeType;

  @Deprecated("Internal Use Only")
  MidiPort.internal_() : super.internal_();

  @DomName('MIDIPort.connection')
  @DocsEditable()
  @Experimental() // untriaged
  String get connection =>
      _blink.BlinkMIDIPort.instance.connection_Getter_(this);

  @DomName('MIDIPort.id')
  @DocsEditable()
  String get id => _blink.BlinkMIDIPort.instance.id_Getter_(this);

  @DomName('MIDIPort.manufacturer')
  @DocsEditable()
  String get manufacturer =>
      _blink.BlinkMIDIPort.instance.manufacturer_Getter_(this);

  @DomName('MIDIPort.name')
  @DocsEditable()
  String get name => _blink.BlinkMIDIPort.instance.name_Getter_(this);

  @DomName('MIDIPort.state')
  @DocsEditable()
  @Experimental() // untriaged
  String get state => _blink.BlinkMIDIPort.instance.state_Getter_(this);

  @DomName('MIDIPort.type')
  @DocsEditable()
  String get type => _blink.BlinkMIDIPort.instance.type_Getter_(this);

  @DomName('MIDIPort.version')
  @DocsEditable()
  String get version => _blink.BlinkMIDIPort.instance.version_Getter_(this);

  @DomName('MIDIPort.close')
  @DocsEditable()
  @Experimental() // untriaged
  Future close() => convertNativePromiseToDartFuture(
      _blink.BlinkMIDIPort.instance.close_Callback_0_(this));

  @DomName('MIDIPort.open')
  @DocsEditable()
  @Experimental() // untriaged
  Future open() => convertNativePromiseToDartFuture(
      _blink.BlinkMIDIPort.instance.open_Callback_0_(this));
}
// Copyright (c) 2012, the Dart project authors.  Please see the AUTHORS file
// for details. All rights reserved. Use of this source code is governed by a
// BSD-style license that can be found in the LICENSE file.

// WARNING: Do not edit - generated code.

@DocsEditable()
@DomName('MimeType')
@Experimental() // non-standard
class MimeType extends DartHtmlDomObject {
  // To suppress missing implicit constructor warnings.
  factory MimeType._() {
    throw new UnsupportedError("Not supported");
  }

  @Deprecated("Internal Use Only")
  external static Type get instanceRuntimeType;

  @Deprecated("Internal Use Only")
  MimeType.internal_() {}

  @DomName('MimeType.description')
  @DocsEditable()
  String get description =>
      _blink.BlinkMimeType.instance.description_Getter_(this);

  @DomName('MimeType.enabledPlugin')
  @DocsEditable()
  Plugin get enabledPlugin =>
      _blink.BlinkMimeType.instance.enabledPlugin_Getter_(this);

  @DomName('MimeType.suffixes')
  @DocsEditable()
  String get suffixes => _blink.BlinkMimeType.instance.suffixes_Getter_(this);

  @DomName('MimeType.type')
  @DocsEditable()
  String get type => _blink.BlinkMimeType.instance.type_Getter_(this);
}
// Copyright (c) 2012, the Dart project authors.  Please see the AUTHORS file
// for details. All rights reserved. Use of this source code is governed by a
// BSD-style license that can be found in the LICENSE file.

// WARNING: Do not edit - generated code.

@DocsEditable()
@DomName('MimeTypeArray')
@Experimental() // non-standard
class MimeTypeArray extends DartHtmlDomObject
    with ListMixin<MimeType>, ImmutableListMixin<MimeType>
    implements List<MimeType> {
  // To suppress missing implicit constructor warnings.
  factory MimeTypeArray._() {
    throw new UnsupportedError("Not supported");
  }

  @Deprecated("Internal Use Only")
  external static Type get instanceRuntimeType;

  @Deprecated("Internal Use Only")
  MimeTypeArray.internal_() {}

  @DomName('MimeTypeArray.length')
  @DocsEditable()
  int get length => _blink.BlinkMimeTypeArray.instance.length_Getter_(this);

  MimeType operator [](int index) {
    if (index < 0 || index >= length) throw new RangeError.index(index, this);
    return _nativeIndexedGetter(index);
  }

  MimeType _nativeIndexedGetter(int index) =>
      (_blink.BlinkMimeTypeArray.instance.item_Callback_1_(this, index));

  void operator []=(int index, MimeType value) {
    throw new UnsupportedError("Cannot assign element of immutable List.");
  }
  // -- start List<MimeType> mixins.
  // MimeType is the element type.

  set length(int value) {
    throw new UnsupportedError("Cannot resize immutable List.");
  }

  MimeType get first {
    if (this.length > 0) {
      return _nativeIndexedGetter(0);
    }
    throw new StateError("No elements");
  }

  MimeType get last {
    int len = this.length;
    if (len > 0) {
      return _nativeIndexedGetter(len - 1);
    }
    throw new StateError("No elements");
  }

  MimeType get single {
    int len = this.length;
    if (len == 1) {
      return _nativeIndexedGetter(0);
    }
    if (len == 0) throw new StateError("No elements");
    throw new StateError("More than one element");
  }

  MimeType elementAt(int index) => this[index];
  // -- end List<MimeType> mixins.

  @DomName('MimeTypeArray.item')
  @DocsEditable()
  MimeType item(int index) =>
      _blink.BlinkMimeTypeArray.instance.item_Callback_1_(this, index);

  MimeType namedItem(String name) {
    if ((name is String || name == null)) {
      return _blink.BlinkMimeTypeArray.instance
          .namedItem_Callback_1_(this, name);
    }
    if ((name is String || name == null)) {
      return _blink.BlinkMimeTypeArray.instance
          .namedItem_Callback_1_(this, name);
    }
    throw new ArgumentError("Incorrect number or type of arguments");
  }
}
// Copyright (c) 2012, the Dart project authors.  Please see the AUTHORS file
// for details. All rights reserved. Use of this source code is governed by a
// BSD-style license that can be found in the LICENSE file.

// WARNING: Do not edit - generated code.

@DocsEditable()
@DomName('HTMLModElement')
@Unstable()
class ModElement extends HtmlElement {
  // To suppress missing implicit constructor warnings.
  factory ModElement._() {
    throw new UnsupportedError("Not supported");
  }

  @Deprecated("Internal Use Only")
  external static Type get instanceRuntimeType;

  @Deprecated("Internal Use Only")
  ModElement.internal_() : super.internal_();

  /**
   * Constructor instantiated by the DOM when a custom element has been created.
   *
   * This can only be called by subclasses from their created constructor.
   */
  ModElement.created() : super.created();

  @DomName('HTMLModElement.cite')
  @DocsEditable()
  String get cite => _blink.BlinkHTMLModElement.instance.cite_Getter_(this);

  @DomName('HTMLModElement.cite')
  @DocsEditable()
  set cite(String value) =>
      _blink.BlinkHTMLModElement.instance.cite_Setter_(this, value);

  @DomName('HTMLModElement.dateTime')
  @DocsEditable()
  String get dateTime =>
      _blink.BlinkHTMLModElement.instance.dateTime_Getter_(this);

  @DomName('HTMLModElement.dateTime')
  @DocsEditable()
  set dateTime(String value) =>
      _blink.BlinkHTMLModElement.instance.dateTime_Setter_(this, value);
}
// Copyright (c) 2012, the Dart project authors.  Please see the AUTHORS file
// for details. All rights reserved. Use of this source code is governed by a
// BSD-style license that can be found in the LICENSE file.

// WARNING: Do not edit - generated code.

@DomName('MouseEvent')
class MouseEvent extends UIEvent {
  factory MouseEvent(String type,
      {Window view,
      int detail: 0,
      int screenX: 0,
      int screenY: 0,
      int clientX: 0,
      int clientY: 0,
      int button: 0,
      bool canBubble: true,
      bool cancelable: true,
      bool ctrlKey: false,
      bool altKey: false,
      bool shiftKey: false,
      bool metaKey: false,
      EventTarget relatedTarget}) {
    if (view == null) {
      view = window;
    }
    var event = document._createEvent('MouseEvent');
    event._initMouseEvent(
        type,
        canBubble,
        cancelable,
        view,
        detail,
        screenX,
        screenY,
        clientX,
        clientY,
        ctrlKey,
        altKey,
        shiftKey,
        metaKey,
        button,
        relatedTarget);
    return event;
  }

  @DomName('MouseEvent.MouseEvent')
  @DocsEditable()
  factory MouseEvent._(String type, [Map eventInitDict]) {
    if (eventInitDict != null) {
      var eventInitDict_1 = convertDartToNative_Dictionary(eventInitDict);
      return _blink.BlinkMouseEvent.instance
          .constructorCallback_2_(type, eventInitDict_1);
    }
    return _blink.BlinkMouseEvent.instance.constructorCallback_1_(type);
  }

  @Deprecated("Internal Use Only")
  external static Type get instanceRuntimeType;

  @Deprecated("Internal Use Only")
  MouseEvent.internal_() : super.internal_();

  @DomName('MouseEvent.altKey')
  @DocsEditable()
  bool get altKey => _blink.BlinkMouseEvent.instance.altKey_Getter_(this);

  @DomName('MouseEvent.button')
  @DocsEditable()
  int get button => _blink.BlinkMouseEvent.instance.button_Getter_(this);

  @DomName('MouseEvent.buttons')
  @DocsEditable()
  @Experimental() // untriaged
  int get buttons => _blink.BlinkMouseEvent.instance.buttons_Getter_(this);

  @DomName('MouseEvent.clientX')
  @DocsEditable()
  int get _clientX => _blink.BlinkMouseEvent.instance.clientX_Getter_(this);

  @DomName('MouseEvent.clientY')
  @DocsEditable()
  int get _clientY => _blink.BlinkMouseEvent.instance.clientY_Getter_(this);

  @DomName('MouseEvent.ctrlKey')
  @DocsEditable()
  bool get ctrlKey => _blink.BlinkMouseEvent.instance.ctrlKey_Getter_(this);

  @DomName('MouseEvent.dataTransfer')
  @DocsEditable()
  @Unstable()
  DataTransfer get dataTransfer =>
      _blink.BlinkMouseEvent.instance.dataTransfer_Getter_(this);

  /**
   * The nonstandard way to access the element that the mouse comes
   * from in the case of a `mouseover` event.
   *
   * This member is deprecated and not cross-browser compatible; use
   * relatedTarget to get the same information in the standard way.
   */
  @DomName('MouseEvent.fromElement')
  @DocsEditable()
  @deprecated
  Node get fromElement =>
      _blink.BlinkMouseEvent.instance.fromElement_Getter_(this);

  @DomName('MouseEvent.layerX')
  @DocsEditable()
  @Experimental() // untriaged
  int get _layerX => _blink.BlinkMouseEvent.instance.layerX_Getter_(this);

  @DomName('MouseEvent.layerY')
  @DocsEditable()
  @Experimental() // untriaged
  int get _layerY => _blink.BlinkMouseEvent.instance.layerY_Getter_(this);

  @DomName('MouseEvent.metaKey')
  @DocsEditable()
  bool get metaKey => _blink.BlinkMouseEvent.instance.metaKey_Getter_(this);

  @DomName('MouseEvent.movementX')
  @DocsEditable()
  @Experimental() // untriaged
  int get _movementX => _blink.BlinkMouseEvent.instance.movementX_Getter_(this);

  @DomName('MouseEvent.movementY')
  @DocsEditable()
  @Experimental() // untriaged
  int get _movementY => _blink.BlinkMouseEvent.instance.movementY_Getter_(this);

  @DomName('MouseEvent.offsetX')
  @DocsEditable()
  @Unstable()
  int get _offsetX => _blink.BlinkMouseEvent.instance.offsetX_Getter_(this);

  @DomName('MouseEvent.offsetY')
  @DocsEditable()
  @Unstable()
  int get _offsetY => _blink.BlinkMouseEvent.instance.offsetY_Getter_(this);

  @DomName('MouseEvent.pageX')
  @DocsEditable()
  @Experimental() // untriaged
  int get _pageX => _blink.BlinkMouseEvent.instance.pageX_Getter_(this);

  @DomName('MouseEvent.pageY')
  @DocsEditable()
  @Experimental() // untriaged
  int get _pageY => _blink.BlinkMouseEvent.instance.pageY_Getter_(this);

  @DomName('MouseEvent.region')
  @DocsEditable()
  @Experimental() // untriaged
  String get region => _blink.BlinkMouseEvent.instance.region_Getter_(this);

  @DomName('MouseEvent.relatedTarget')
  @DocsEditable()
  EventTarget get relatedTarget => _convertNativeToDart_EventTarget(
      _blink.BlinkMouseEvent.instance.relatedTarget_Getter_(this));

  @DomName('MouseEvent.screenX')
  @DocsEditable()
  int get _screenX => _blink.BlinkMouseEvent.instance.screenX_Getter_(this);

  @DomName('MouseEvent.screenY')
  @DocsEditable()
  int get _screenY => _blink.BlinkMouseEvent.instance.screenY_Getter_(this);

  @DomName('MouseEvent.shiftKey')
  @DocsEditable()
  bool get shiftKey => _blink.BlinkMouseEvent.instance.shiftKey_Getter_(this);

  /**
   * The nonstandard way to access the element that the mouse goes
   * to in the case of a `mouseout` event.
   *
   * This member is deprecated and not cross-browser compatible; use
   * relatedTarget to get the same information in the standard way.
   */
  @DomName('MouseEvent.toElement')
  @DocsEditable()
  @deprecated
  Node get toElement => _blink.BlinkMouseEvent.instance.toElement_Getter_(this);

  @DomName('MouseEvent.which')
  @DocsEditable()
  @Experimental() // untriaged
  int get _which => _blink.BlinkMouseEvent.instance.which_Getter_(this);

  @DomName('MouseEvent.initMouseEvent')
  @DocsEditable()
  void _initMouseEvent(
          String type,
          bool bubbles,
          bool cancelable,
          Window view,
          int detail,
          int screenX,
          int screenY,
          int clientX,
          int clientY,
          bool ctrlKey,
          bool altKey,
          bool shiftKey,
          bool metaKey,
          int button,
          EventTarget relatedTarget) =>
      _blink.BlinkMouseEvent.instance.initMouseEvent_Callback_15_(
          this,
          type,
          bubbles,
          cancelable,
          view,
          detail,
          screenX,
          screenY,
          clientX,
          clientY,
          ctrlKey,
          altKey,
          shiftKey,
          metaKey,
          button,
          _convertDartToNative_EventTarget(relatedTarget));

  @deprecated
  int get clientX => client.x;
  @deprecated
  int get clientY => client.y;
  @deprecated
  int get offsetX => offset.x;
  @deprecated
  int get offsetY => offset.y;
  @deprecated
  int get movementX => movement.x;
  @deprecated
  int get movementY => movement.y;
  @deprecated
  int get screenX => screen.x;
  @deprecated
  int get screenY => screen.y;

  @DomName('MouseEvent.clientX')
  @DomName('MouseEvent.clientY')
  Point get client => new Point/*<num>*/(_clientX, _clientY);

  @DomName('MouseEvent.movementX')
  @DomName('MouseEvent.movementY')
  @SupportedBrowser(SupportedBrowser.CHROME)
  @SupportedBrowser(SupportedBrowser.FIREFOX)
  @Experimental()
  Point get movement => new Point/*<num>*/(_movementX, _movementY);

  /**
   * The coordinates of the mouse pointer in target node coordinates.
   *
   * This value may vary between platforms if the target node moves
   * after the event has fired or if the element has CSS transforms affecting
   * it.
   */
  Point get offset => new Point/*<num>*/(_offsetX, _offsetY);

  @DomName('MouseEvent.screenX')
  @DomName('MouseEvent.screenY')
  Point get screen => new Point/*<num>*/(_screenX, _screenY);

  @DomName('MouseEvent.layerX')
  @DomName('MouseEvent.layerY')
  Point get layer => new Point/*<num>*/(_layerX, _layerY);

  @DomName('MouseEvent.pageX')
  @DomName('MouseEvent.pageY')
  Point get page => new Point/*<num>*/(_pageX, _pageY);
}
// Copyright (c) 2012, the Dart project authors.  Please see the AUTHORS file
// for details. All rights reserved. Use of this source code is governed by a
// BSD-style license that can be found in the LICENSE file.

// WARNING: Do not edit - generated code.

@DomName('MutationCallback')
typedef void MutationCallback(
    List<MutationRecord> mutations, MutationObserver observer);
// Copyright (c) 2012, the Dart project authors.  Please see the AUTHORS file
// for details. All rights reserved. Use of this source code is governed by a
// BSD-style license that can be found in the LICENSE file.

@DomName('MutationObserver')
@SupportedBrowser(SupportedBrowser.CHROME)
@SupportedBrowser(SupportedBrowser.FIREFOX)
@SupportedBrowser(SupportedBrowser.SAFARI)
@Experimental()
class MutationObserver extends DartHtmlDomObject {
  @DomName('MutationObserver.MutationObserver')
  @DocsEditable()
  factory MutationObserver._(MutationCallback callback) => _create(callback);

  @Deprecated("Internal Use Only")
  external static Type get instanceRuntimeType;

  @Deprecated("Internal Use Only")
  MutationObserver.internal_() {}

  @DomName('MutationObserver.disconnect')
  @DocsEditable()
  void disconnect() =>
      _blink.BlinkMutationObserver.instance.disconnect_Callback_0_(this);

  @DomName('MutationObserver.observe')
  @DocsEditable()
  void _observe(Node target, Map options) =>
      _blink.BlinkMutationObserver.instance.observe_Callback_2_(
          this, target, convertDartToNative_Dictionary(options));

  @DomName('MutationObserver.takeRecords')
  @DocsEditable()
  List<MutationRecord> takeRecords() =>
      (_blink.BlinkMutationObserver.instance.takeRecords_Callback_0_(this));

  /**
   * Checks to see if the mutation observer API is supported on the current
   * platform.
   */
  static bool get supported {
    return true;
  }

  @DocsEditable()
  static MutationObserver _create(callback) =>
      _blink.BlinkMutationObserver.instance.constructorCallback_1_(callback);

  /**
   * Observes the target for the specified changes.
   *
   * Some requirements for the optional parameters:
   *
   * * Either childList, attributes or characterData must be true.
   * * If attributeOldValue is true then attributes must also be true.
   * * If attributeFilter is specified then attributes must be true.
   * * If characterDataOldValue is true then characterData must be true.
   */
  void observe(Node target,
      {bool childList,
      bool attributes,
      bool characterData,
      bool subtree,
      bool attributeOldValue,
      bool characterDataOldValue,
      List<String> attributeFilter}) {
    // Parse options into map of known type.
    var parsedOptions = _createDict();

    // Override options passed in the map with named optional arguments.
    override(key, value) {
      if (value != null) _add(parsedOptions, key, value);
    }

    override('childList', childList);
    override('attributes', attributes);
    override('characterData', characterData);
    override('subtree', subtree);
    override('attributeOldValue', attributeOldValue);
    override('characterDataOldValue', characterDataOldValue);
    if (attributeFilter != null) {
      override('attributeFilter', _fixupList(attributeFilter));
    }

    _call(target, parsedOptions);
  }

  // TODO: Change to a set when const Sets are available.
  static final _boolKeys = const {
    'childList': true,
    'attributes': true,
    'characterData': true,
    'subtree': true,
    'attributeOldValue': true,
    'characterDataOldValue': true
  };

  static _createDict() => {};
  static _add(m, String key, value) {
    m[key] = value;
  }

  static _fixupList(list) => list;

  void _call(Node target, options) {
    _observe(target, options);
  }

  factory MutationObserver(MutationCallback callback) =>
      new MutationObserver._(_wrapBinaryZone(callback));
}
// Copyright (c) 2012, the Dart project authors.  Please see the AUTHORS file
// for details. All rights reserved. Use of this source code is governed by a
// BSD-style license that can be found in the LICENSE file.

// WARNING: Do not edit - generated code.

@DocsEditable()
@DomName('MutationRecord')
class MutationRecord extends DartHtmlDomObject {
  // To suppress missing implicit constructor warnings.
  factory MutationRecord._() {
    throw new UnsupportedError("Not supported");
  }

  @Deprecated("Internal Use Only")
  external static Type get instanceRuntimeType;

  @Deprecated("Internal Use Only")
  MutationRecord.internal_() {}

  @DomName('MutationRecord.addedNodes')
  @DocsEditable()
  List<Node> get addedNodes =>
      (_blink.BlinkMutationRecord.instance.addedNodes_Getter_(this));

  @DomName('MutationRecord.attributeName')
  @DocsEditable()
  String get attributeName =>
      _blink.BlinkMutationRecord.instance.attributeName_Getter_(this);

  @DomName('MutationRecord.attributeNamespace')
  @DocsEditable()
  String get attributeNamespace =>
      _blink.BlinkMutationRecord.instance.attributeNamespace_Getter_(this);

  @DomName('MutationRecord.nextSibling')
  @DocsEditable()
  Node get nextSibling =>
      _blink.BlinkMutationRecord.instance.nextSibling_Getter_(this);

  @DomName('MutationRecord.oldValue')
  @DocsEditable()
  String get oldValue =>
      _blink.BlinkMutationRecord.instance.oldValue_Getter_(this);

  @DomName('MutationRecord.previousSibling')
  @DocsEditable()
  Node get previousSibling =>
      _blink.BlinkMutationRecord.instance.previousSibling_Getter_(this);

  @DomName('MutationRecord.removedNodes')
  @DocsEditable()
  List<Node> get removedNodes =>
      (_blink.BlinkMutationRecord.instance.removedNodes_Getter_(this));

  @DomName('MutationRecord.target')
  @DocsEditable()
  Node get target => _blink.BlinkMutationRecord.instance.target_Getter_(this);

  @DomName('MutationRecord.type')
  @DocsEditable()
  String get type => _blink.BlinkMutationRecord.instance.type_Getter_(this);
}
// Copyright (c) 2012, the Dart project authors.  Please see the AUTHORS file
// for details. All rights reserved. Use of this source code is governed by a
// BSD-style license that can be found in the LICENSE file.

@DomName('Navigator')
class Navigator extends DartHtmlDomObject
    implements
        NavigatorStorageUtils,
        NavigatorCpu,
        NavigatorLanguage,
        NavigatorOnLine,
        NavigatorID {
  /**
   * Gets a stream (video and or audio) from the local computer.
   *
   * Use [MediaStream.supported] to check if this is supported by the current
   * platform. The arguments `audio` and `video` default to `false` (stream does
   * not use audio or video, respectively).
   *
   * Simple example usage:
   *
   *     window.navigator.getUserMedia(audio: true, video: true).then((stream) {
   *       var video = new VideoElement()
   *         ..autoplay = true
   *         ..src = Url.createObjectUrlFromStream(stream);
   *       document.body.append(video);
   *     });
   *
   * The user can also pass in Maps to the audio or video parameters to specify
   * mandatory and optional constraints for the media stream. Not passing in a
   * map, but passing in `true` will provide a MediaStream with audio or
   * video capabilities, but without any additional constraints. The particular
   * constraint names for audio and video are still in flux, but as of this
   * writing, here is an example providing more constraints.
   *
   *     window.navigator.getUserMedia(
   *         audio: true,
   *         video: {'mandatory':
   *                    { 'minAspectRatio': 1.333, 'maxAspectRatio': 1.334 },
   *                 'optional':
   *                    [{ 'minFrameRate': 60 },
   *                     { 'maxWidth': 640 }]
   *     });
   *
   * See also:
   * * [MediaStream.supported]
   */
  @DomName('Navigator.webkitGetUserMedia')
  @SupportedBrowser(SupportedBrowser.CHROME)
  @Experimental()
  Future<MediaStream> getUserMedia({audio: false, video: false}) {
    var completer = new Completer<MediaStream>();
    var options = {'audio': audio, 'video': video};
    this._getUserMedia(options, (stream) {
      completer.complete(stream);
    }, (error) {
      completer.completeError(error);
    });
    return completer.future;
  }

  // To suppress missing implicit constructor warnings.
  factory Navigator._() {
    throw new UnsupportedError("Not supported");
  }

  @Deprecated("Internal Use Only")
  external static Type get instanceRuntimeType;

  @Deprecated("Internal Use Only")
  Navigator.internal_() {}

  @DomName('Navigator.bluetooth')
  @DocsEditable()
  @Experimental() // untriaged
  Bluetooth get bluetooth =>
      _blink.BlinkNavigator.instance.bluetooth_Getter_(this);

  @DomName('Navigator.connection')
  @DocsEditable()
  @Experimental() // untriaged
  NetworkInformation get connection =>
      _blink.BlinkNavigator.instance.connection_Getter_(this);

  @DomName('Navigator.credentials')
  @DocsEditable()
  @Experimental() // untriaged
  CredentialsContainer get credentials =>
      _blink.BlinkNavigator.instance.credentials_Getter_(this);

  @DomName('Navigator.doNotTrack')
  @DocsEditable()
  // http://www.w3.org/2011/tracking-protection/drafts/tracking-dnt.html#js-dom
  @Experimental() // experimental
  String get doNotTrack =>
      _blink.BlinkNavigator.instance.doNotTrack_Getter_(this);

  @DomName('Navigator.geolocation')
  @DocsEditable()
  @Unstable()
  Geolocation get geolocation =>
      _blink.BlinkNavigator.instance.geolocation_Getter_(this);

  @DomName('Navigator.maxTouchPoints')
  @DocsEditable()
  @Experimental() // untriaged
  int get maxTouchPoints =>
      _blink.BlinkNavigator.instance.maxTouchPoints_Getter_(this);

  @DomName('Navigator.mediaDevices')
  @DocsEditable()
  @Experimental() // untriaged
  MediaDevices get mediaDevices =>
      _blink.BlinkNavigator.instance.mediaDevices_Getter_(this);

  @DomName('Navigator.mimeTypes')
  @DocsEditable()
  @Experimental() // nonstandard
  MimeTypeArray get mimeTypes =>
      _blink.BlinkNavigator.instance.mimeTypes_Getter_(this);

  @DomName('Navigator.permissions')
  @DocsEditable()
  @Experimental() // untriaged
  Permissions get permissions =>
      _blink.BlinkNavigator.instance.permissions_Getter_(this);

  @DomName('Navigator.presentation')
  @DocsEditable()
  @Experimental() // untriaged
  Presentation get presentation =>
      _blink.BlinkNavigator.instance.presentation_Getter_(this);

  @DomName('Navigator.productSub')
  @DocsEditable()
  @Unstable()
  String get productSub =>
      _blink.BlinkNavigator.instance.productSub_Getter_(this);

  @DomName('Navigator.serviceWorker')
  @DocsEditable()
  @Experimental() // untriaged
  ServiceWorkerContainer get serviceWorker =>
      _blink.BlinkNavigator.instance.serviceWorker_Getter_(this);

  @DomName('Navigator.services')
  @DocsEditable()
  @Experimental() // untriaged
  ServicePortCollection get services =>
      _blink.BlinkNavigator.instance.services_Getter_(this);

  @DomName('Navigator.storageQuota')
  @DocsEditable()
  @Experimental() // untriaged
  StorageQuota get storageQuota =>
      _blink.BlinkNavigator.instance.storageQuota_Getter_(this);

  @DomName('Navigator.vendor')
  @DocsEditable()
  @Unstable()
  String get vendor => _blink.BlinkNavigator.instance.vendor_Getter_(this);

  @DomName('Navigator.vendorSub')
  @DocsEditable()
  @Unstable()
  String get vendorSub =>
      _blink.BlinkNavigator.instance.vendorSub_Getter_(this);

  @DomName('Navigator.webkitPersistentStorage')
  @DocsEditable()
  @SupportedBrowser(SupportedBrowser.CHROME)
  @SupportedBrowser(SupportedBrowser.SAFARI)
  @Experimental()
  // http://www.w3.org/TR/quota-api/#accessing-storagequota
  DeprecatedStorageQuota get persistentStorage =>
      _blink.BlinkNavigator.instance.webkitPersistentStorage_Getter_(this);

  @DomName('Navigator.webkitTemporaryStorage')
  @DocsEditable()
  @SupportedBrowser(SupportedBrowser.CHROME)
  @SupportedBrowser(SupportedBrowser.SAFARI)
  @Experimental()
  // http://www.w3.org/TR/quota-api/#accessing-storagequota
  DeprecatedStorageQuota get temporaryStorage =>
      _blink.BlinkNavigator.instance.webkitTemporaryStorage_Getter_(this);

  @DomName('Navigator.getBattery')
  @DocsEditable()
  @Experimental() // untriaged
  Future getBattery() => convertNativePromiseToDartFuture(
      _blink.BlinkNavigator.instance.getBattery_Callback_0_(this));

  @DomName('Navigator.getGamepads')
  @DocsEditable()
  @Experimental() // untriaged
  List<Gamepad> getGamepads() =>
      (_blink.BlinkNavigator.instance.getGamepads_Callback_0_(this));

  @DomName('Navigator.getVRDevices')
  @DocsEditable()
  @Experimental() // untriaged
  Future getVRDevices() => convertNativePromiseToDartFuture(
      _blink.BlinkNavigator.instance.getVRDevices_Callback_0_(this));

  @DomName('Navigator.registerProtocolHandler')
  @DocsEditable()
  @Unstable()
  void registerProtocolHandler(String scheme, String url, String title) =>
      _blink.BlinkNavigator.instance
          .registerProtocolHandler_Callback_3_(this, scheme, url, title);

  Future requestMidiAccess([Map options]) {
    if (options != null) {
      return _blink.BlinkNavigator.instance.requestMIDIAccess_Callback_1_(
          this, convertDartToNative_Dictionary(options));
    }
    return _blink.BlinkNavigator.instance.requestMIDIAccess_Callback_0_(this);
  }

  @DomName('Navigator.requestMediaKeySystemAccess')
  @DocsEditable()
  @Experimental() // untriaged
  Future requestMediaKeySystemAccess(
          String keySystem, List<Map> supportedConfigurations) =>
      convertNativePromiseToDartFuture(_blink.BlinkNavigator.instance
          .requestMediaKeySystemAccess_Callback_2_(
              this, keySystem, supportedConfigurations));

  @DomName('Navigator.sendBeacon')
  @DocsEditable()
  @Experimental() // untriaged
  bool sendBeacon(String url, Object data) =>
      _blink.BlinkNavigator.instance.sendBeacon_Callback_2_(this, url, data);

  @DomName('Navigator.webkitGetUserMedia')
  @DocsEditable()
  // http://dev.w3.org/2011/webrtc/editor/getusermedia.html#navigatorusermedia
  @Experimental()
  void _getUserMedia(
          Map options,
          _NavigatorUserMediaSuccessCallback successCallback,
          _NavigatorUserMediaErrorCallback errorCallback) =>
      _blink.BlinkNavigator.instance.webkitGetUserMedia_Callback_3_(
          this,
          convertDartToNative_Dictionary(options),
          successCallback,
          errorCallback);

  @DomName('Navigator.hardwareConcurrency')
  @DocsEditable()
  @Experimental() // untriaged
  int get hardwareConcurrency =>
      _blink.BlinkNavigator.instance.hardwareConcurrency_Getter_(this);

  @DomName('Navigator.appCodeName')
  @DocsEditable()
  @Experimental() // non-standard
  String get appCodeName =>
      _blink.BlinkNavigator.instance.appCodeName_Getter_(this);

  @DomName('Navigator.appName')
  @DocsEditable()
  String get appName => _blink.BlinkNavigator.instance.appName_Getter_(this);

  @DomName('Navigator.appVersion')
  @DocsEditable()
  String get appVersion =>
      _blink.BlinkNavigator.instance.appVersion_Getter_(this);

  @DomName('Navigator.dartEnabled')
  @DocsEditable()
  @Experimental() // untriaged
  bool get dartEnabled =>
      _blink.BlinkNavigator.instance.dartEnabled_Getter_(this);

  @DomName('Navigator.platform')
  @DocsEditable()
  String get platform => _blink.BlinkNavigator.instance.platform_Getter_(this);

  @DomName('Navigator.product')
  @DocsEditable()
  @Unstable()
  String get product => _blink.BlinkNavigator.instance.product_Getter_(this);

  @DomName('Navigator.userAgent')
  @DocsEditable()
  String get userAgent =>
      _blink.BlinkNavigator.instance.userAgent_Getter_(this);

  @DomName('Navigator.language')
  @DocsEditable()
  String get language => _blink.BlinkNavigator.instance.language_Getter_(this);

  @DomName('Navigator.languages')
  @DocsEditable()
  @Experimental() // untriaged
  List<String> get languages =>
      _blink.BlinkNavigator.instance.languages_Getter_(this);

  @DomName('Navigator.onLine')
  @DocsEditable()
  @Unstable()
  bool get onLine => _blink.BlinkNavigator.instance.onLine_Getter_(this);

  @DomName('Navigator.cookieEnabled')
  @DocsEditable()
  @Unstable()
  bool get cookieEnabled =>
      _blink.BlinkNavigator.instance.cookieEnabled_Getter_(this);

  @DomName('Navigator.getStorageUpdates')
  @DocsEditable()
  // http://www.whatwg.org/specs/web-apps/current-work/multipage/timers.html#navigatorstorageutils
  @Experimental()
  void getStorageUpdates() =>
      _blink.BlinkNavigator.instance.getStorageUpdates_Callback_0_(this);
}
// Copyright (c) 2012, the Dart project authors.  Please see the AUTHORS file
// for details. All rights reserved. Use of this source code is governed by a
// BSD-style license that can be found in the LICENSE file.

// WARNING: Do not edit - generated code.

@DocsEditable()
@DomName('NavigatorCPU')
@Experimental() // untriaged
abstract class NavigatorCpu extends DartHtmlDomObject {
  // To suppress missing implicit constructor warnings.
  factory NavigatorCpu._() {
    throw new UnsupportedError("Not supported");
  }

  @DomName('NavigatorCPU.hardwareConcurrency')
  @DocsEditable()
  @Experimental() // untriaged
  int get hardwareConcurrency;
}
// Copyright (c) 2012, the Dart project authors.  Please see the AUTHORS file
// for details. All rights reserved. Use of this source code is governed by a
// BSD-style license that can be found in the LICENSE file.

// WARNING: Do not edit - generated code.

@DocsEditable()
@DomName('NavigatorID')
@Experimental() // untriaged
abstract class NavigatorID extends DartHtmlDomObject {
  // To suppress missing implicit constructor warnings.
  factory NavigatorID._() {
    throw new UnsupportedError("Not supported");
  }

  @DomName('NavigatorID.appCodeName')
  @DocsEditable()
  @Experimental() // untriaged
  String get appCodeName;

  @DomName('NavigatorID.appName')
  @DocsEditable()
  @Experimental() // untriaged
  String get appName;

  @DomName('NavigatorID.appVersion')
  @DocsEditable()
  @Experimental() // untriaged
  String get appVersion;

  @DomName('NavigatorID.dartEnabled')
  @DocsEditable()
  @Experimental() // untriaged
  bool get dartEnabled;

  @DomName('NavigatorID.platform')
  @DocsEditable()
  @Experimental() // untriaged
  String get platform;

  @DomName('NavigatorID.product')
  @DocsEditable()
  @Experimental() // untriaged
  String get product;

  @DomName('NavigatorID.userAgent')
  @DocsEditable()
  @Experimental() // untriaged
  String get userAgent;
}
// Copyright (c) 2012, the Dart project authors.  Please see the AUTHORS file
// for details. All rights reserved. Use of this source code is governed by a
// BSD-style license that can be found in the LICENSE file.

// WARNING: Do not edit - generated code.

@DocsEditable()
@DomName('NavigatorLanguage')
@Experimental() // untriaged
abstract class NavigatorLanguage extends DartHtmlDomObject {
  // To suppress missing implicit constructor warnings.
  factory NavigatorLanguage._() {
    throw new UnsupportedError("Not supported");
  }

  @DomName('NavigatorLanguage.language')
  @DocsEditable()
  @Experimental() // untriaged
  String get language;

  @DomName('NavigatorLanguage.languages')
  @DocsEditable()
  @Experimental() // untriaged
  List<String> get languages;
}
// Copyright (c) 2012, the Dart project authors.  Please see the AUTHORS file
// for details. All rights reserved. Use of this source code is governed by a
// BSD-style license that can be found in the LICENSE file.

// WARNING: Do not edit - generated code.

@DocsEditable()
@DomName('NavigatorOnLine')
@Experimental() // untriaged
abstract class NavigatorOnLine extends DartHtmlDomObject {
  // To suppress missing implicit constructor warnings.
  factory NavigatorOnLine._() {
    throw new UnsupportedError("Not supported");
  }

  @DomName('NavigatorOnLine.onLine')
  @DocsEditable()
  @Experimental() // untriaged
  bool get onLine;
}
// Copyright (c) 2012, the Dart project authors.  Please see the AUTHORS file
// for details. All rights reserved. Use of this source code is governed by a
// BSD-style license that can be found in the LICENSE file.

// WARNING: Do not edit - generated code.

@DocsEditable()
@DomName('NavigatorStorageUtils')
@Experimental() // untriaged
class NavigatorStorageUtils extends DartHtmlDomObject {
  // To suppress missing implicit constructor warnings.
  factory NavigatorStorageUtils._() {
    throw new UnsupportedError("Not supported");
  }

  @Deprecated("Internal Use Only")
  external static Type get instanceRuntimeType;

  @Deprecated("Internal Use Only")
  NavigatorStorageUtils.internal_() {}

  @DomName('NavigatorStorageUtils.cookieEnabled')
  @DocsEditable()
  @Experimental() // untriaged
  bool get cookieEnabled =>
      _blink.BlinkNavigatorStorageUtils.instance.cookieEnabled_Getter_(this);

  @DomName('NavigatorStorageUtils.getStorageUpdates')
  @DocsEditable()
  @Experimental() // untriaged
  void getStorageUpdates() => _blink.BlinkNavigatorStorageUtils.instance
      .getStorageUpdates_Callback_0_(this);
}
// Copyright (c) 2012, the Dart project authors.  Please see the AUTHORS file
// for details. All rights reserved. Use of this source code is governed by a
// BSD-style license that can be found in the LICENSE file.

// WARNING: Do not edit - generated code.

@DocsEditable()
@DomName('NavigatorUserMediaError')
// http://dev.w3.org/2011/webrtc/editor/getusermedia.html#idl-def-NavigatorUserMediaError
@Experimental()
class NavigatorUserMediaError extends DartHtmlDomObject {
  // To suppress missing implicit constructor warnings.
  factory NavigatorUserMediaError._() {
    throw new UnsupportedError("Not supported");
  }

  @Deprecated("Internal Use Only")
  external static Type get instanceRuntimeType;

  @Deprecated("Internal Use Only")
  NavigatorUserMediaError.internal_() {}

  @DomName('NavigatorUserMediaError.constraintName')
  @DocsEditable()
  String get constraintName =>
      _blink.BlinkNavigatorUserMediaError.instance.constraintName_Getter_(this);

  @DomName('NavigatorUserMediaError.message')
  @DocsEditable()
  String get message =>
      _blink.BlinkNavigatorUserMediaError.instance.message_Getter_(this);

  @DomName('NavigatorUserMediaError.name')
  @DocsEditable()
  String get name =>
      _blink.BlinkNavigatorUserMediaError.instance.name_Getter_(this);
}
// Copyright (c) 2012, the Dart project authors.  Please see the AUTHORS file
// for details. All rights reserved. Use of this source code is governed by a
// BSD-style license that can be found in the LICENSE file.

// WARNING: Do not edit - generated code.

@DomName('NavigatorUserMediaErrorCallback')
// http://dev.w3.org/2011/webrtc/editor/getusermedia.html#idl-def-NavigatorUserMediaErrorCallback
@Experimental()
typedef void _NavigatorUserMediaErrorCallback(NavigatorUserMediaError error);
// Copyright (c) 2012, the Dart project authors.  Please see the AUTHORS file
// for details. All rights reserved. Use of this source code is governed by a
// BSD-style license that can be found in the LICENSE file.

// WARNING: Do not edit - generated code.

@DomName('NavigatorUserMediaSuccessCallback')
// http://dev.w3.org/2011/webrtc/editor/getusermedia.html#idl-def-NavigatorUserMediaSuccessCallback
@Experimental()
typedef void _NavigatorUserMediaSuccessCallback(MediaStream stream);
// Copyright (c) 2012, the Dart project authors.  Please see the AUTHORS file
// for details. All rights reserved. Use of this source code is governed by a
// BSD-style license that can be found in the LICENSE file.

// WARNING: Do not edit - generated code.

@DocsEditable()
@DomName('NetworkInformation')
@Experimental() // untriaged
class NetworkInformation extends EventTarget {
  // To suppress missing implicit constructor warnings.
  factory NetworkInformation._() {
    throw new UnsupportedError("Not supported");
  }

  @Deprecated("Internal Use Only")
  external static Type get instanceRuntimeType;

  @Deprecated("Internal Use Only")
  NetworkInformation.internal_() : super.internal_();

  @DomName('NetworkInformation.type')
  @DocsEditable()
  @Experimental() // untriaged
  String get type => _blink.BlinkNetworkInformation.instance.type_Getter_(this);
}
// Copyright (c) 2012, the Dart project authors.  Please see the AUTHORS file
// for details. All rights reserved. Use of this source code is governed by a
// BSD-style license that can be found in the LICENSE file.

/**
 * Lazy implementation of the child nodes of an element that does not request
 * the actual child nodes of an element until strictly necessary greatly
 * improving performance for the typical cases where it is not required.
 */
class _ChildNodeListLazy extends ListBase<Node> implements NodeListWrapper {
  final Node _this;

  _ChildNodeListLazy(this._this);

  Node get first {
    Node result = _this.firstChild;
    if (result == null) throw new StateError("No elements");
    return result;
  }

  Node get last {
    Node result = _this.lastChild;
    if (result == null) throw new StateError("No elements");
    return result;
  }

  Node get single {
    int l = this.length;
    if (l == 0) throw new StateError("No elements");
    if (l > 1) throw new StateError("More than one element");
    return _this.firstChild;
  }

  void add(Node value) {
    _this.append(value);
  }

  void addAll(Iterable<Node> iterable) {
    if (iterable is _ChildNodeListLazy) {
      _ChildNodeListLazy otherList = iterable;
      if (!identical(otherList._this, _this)) {
        // Optimized route for copying between nodes.
        for (var i = 0, len = otherList.length; i < len; ++i) {
          _this.append(otherList._this.firstChild);
        }
      }
      return;
    }
    for (Node node in iterable) {
      _this.append(node);
    }
  }

  void insert(int index, Node node) {
    if (index < 0 || index > length) {
      throw new RangeError.range(index, 0, length);
    }
    if (index == length) {
      _this.append(node);
    } else {
      _this.insertBefore(node, this[index]);
    }
  }

  void insertAll(int index, Iterable<Node> iterable) {
    if (index == length) {
      addAll(iterable);
    } else {
      var item = this[index];
      _this.insertAllBefore(iterable, item);
    }
  }

  void setAll(int index, Iterable<Node> iterable) {
    throw new UnsupportedError("Cannot setAll on Node list");
  }

  Node removeLast() {
    final result = last;
    if (result != null) {
      _this._removeChild(result);
    }
    return result;
  }

  Node removeAt(int index) {
    var result = this[index];
    if (result != null) {
      _this._removeChild(result);
    }
    return result;
  }

  bool remove(Object object) {
    if (object is! Node) return false;
    Node node = object;
    // We aren't preserving identity of nodes in JSINTEROP mode
    if (_this != node.parentNode) return false;
    _this._removeChild(node);
    return true;
  }

  void _filter(bool test(Node node), bool removeMatching) {
    // This implementation of removeWhere/retainWhere is more efficient
    // than the default in ListBase. Child nodes can be removed in constant
    // time.
    Node child = _this.firstChild;
    while (child != null) {
      Node nextChild = child.nextNode;
      if (test(child) == removeMatching) {
        _this._removeChild(child);
      }
      child = nextChild;
    }
  }

  void removeWhere(bool test(Node node)) {
    _filter(test, true);
  }

  void retainWhere(bool test(Node node)) {
    _filter(test, false);
  }

  void clear() {
    _this._clearChildren();
  }

  void operator []=(int index, Node value) {
    _this._replaceChild(value, this[index]);
  }

  Iterator<Node> get iterator => _this.childNodes.iterator;

  // From List<Node>:

  // TODO(jacobr): this could be implemented for child node lists.
  // The exception we throw here is misleading.
  void sort([Comparator<Node> compare]) {
    throw new UnsupportedError("Cannot sort Node list");
  }

  void shuffle([Random random]) {
    throw new UnsupportedError("Cannot shuffle Node list");
  }

  // FIXME: implement these.
  void setRange(int start, int end, Iterable<Node> iterable,
      [int skipCount = 0]) {
    throw new UnsupportedError("Cannot setRange on Node list");
  }

  void fillRange(int start, int end, [Node fill]) {
    throw new UnsupportedError("Cannot fillRange on Node list");
  }
  // -- end List<Node> mixins.

  // TODO(jacobr): benchmark whether this is more efficient or whether caching
  // a local copy of childNodes is more efficient.
  int get length => _this.childNodes.length;

  set length(int value) {
    throw new UnsupportedError("Cannot set length on immutable List.");
  }

  Node operator [](int index) => _this.childNodes[index];

  List<Node> get rawList => _this.childNodes;
}

@DomName('Node')
class Node extends EventTarget {
  // Custom element created callback.
  Node._created() : super._created();

  /**
   * A modifiable list of this node's children.
   */
  List<Node> get nodes {
    return new _ChildNodeListLazy(this);
  }

  set nodes(Iterable<Node> value) {
    // Copy list first since we don't want liveness during iteration.
    // TODO(jacobr): there is a better way to do this.
    var copy = value.toList();
    text = '';
    for (Node node in copy) {
      append(node);
    }
  }

  /**
   * Removes this node from the DOM.
   */
  @DomName('Node.removeChild')
  void remove() {
    // TODO(jacobr): should we throw an exception if parent is already null?
    // TODO(vsm): Use the native remove when available.
    if (this.parentNode != null) {
      final Node parent = this.parentNode;
      parentNode._removeChild(this);
    }
  }

  /**
   * Replaces this node with another node.
   */
  @DomName('Node.replaceChild')
  Node replaceWith(Node otherNode) {
    try {
      final Node parent = this.parentNode;
      parent._replaceChild(otherNode, this);
    } catch (e) {}
    ;
    return this;
  }

  /**
   * Inserts all of the nodes into this node directly before refChild.
   *
   * See also:
   *
   * * [insertBefore]
   */
  Node insertAllBefore(Iterable<Node> newNodes, Node refChild) {
    if (newNodes is _ChildNodeListLazy) {
      _ChildNodeListLazy otherList = newNodes;
      if (identical(otherList._this, this)) {
        throw new ArgumentError(newNodes);
      }

      // Optimized route for copying between nodes.
      for (var i = 0, len = otherList.length; i < len; ++i) {
        this.insertBefore(otherList._this.firstChild, refChild);
      }
    } else {
      for (var node in newNodes) {
        this.insertBefore(node, refChild);
      }
    }
  }

  void _clearChildren() {
    while (firstChild != null) {
      _removeChild(firstChild);
    }
  }

  /**
   * Print out a String representation of this Node.
   */
  String toString() {
    String value = nodeValue; // Fetch DOM Node property once.
    return value == null ? super.toString() : value;
  }

  /**
   * A list of this node's children.
   *
   * ## Other resources
   *
   * * [Node.childNodes](https://developer.mozilla.org/en-US/docs/Web/API/Node.childNodes)
   *   from MDN.
   */
  @DomName('Node.childNodes')
  @DocsEditable()
  List<Node> get childNodes =>
      _blink.BlinkNode.instance.childNodes_Getter_(this);
  // To suppress missing implicit constructor warnings.
  factory Node._() {
    throw new UnsupportedError("Not supported");
  }

  @Deprecated("Internal Use Only")
  external static Type get instanceRuntimeType;

  @Deprecated("Internal Use Only")
  Node.internal_() : super.internal_();

  @DomName('Node.ATTRIBUTE_NODE')
  @DocsEditable()
  static const int ATTRIBUTE_NODE = 2;

  @DomName('Node.CDATA_SECTION_NODE')
  @DocsEditable()
  static const int CDATA_SECTION_NODE = 4;

  @DomName('Node.COMMENT_NODE')
  @DocsEditable()
  static const int COMMENT_NODE = 8;

  @DomName('Node.DOCUMENT_FRAGMENT_NODE')
  @DocsEditable()
  static const int DOCUMENT_FRAGMENT_NODE = 11;

  @DomName('Node.DOCUMENT_NODE')
  @DocsEditable()
  static const int DOCUMENT_NODE = 9;

  @DomName('Node.DOCUMENT_TYPE_NODE')
  @DocsEditable()
  static const int DOCUMENT_TYPE_NODE = 10;

  @DomName('Node.ELEMENT_NODE')
  @DocsEditable()
  static const int ELEMENT_NODE = 1;

  @DomName('Node.ENTITY_NODE')
  @DocsEditable()
  static const int ENTITY_NODE = 6;

  @DomName('Node.ENTITY_REFERENCE_NODE')
  @DocsEditable()
  static const int ENTITY_REFERENCE_NODE = 5;

  @DomName('Node.NOTATION_NODE')
  @DocsEditable()
  static const int NOTATION_NODE = 12;

  @DomName('Node.PROCESSING_INSTRUCTION_NODE')
  @DocsEditable()
  static const int PROCESSING_INSTRUCTION_NODE = 7;

  @DomName('Node.TEXT_NODE')
  @DocsEditable()
  static const int TEXT_NODE = 3;

  @DomName('Node.baseURI')
  @DocsEditable()
  String get baseUri => _blink.BlinkNode.instance.baseURI_Getter_(this);

  /**
   * The first child of this node.
   *
   * ## Other resources
   *
   * * [Node.firstChild](https://developer.mozilla.org/en-US/docs/Web/API/Node.firstChild)
   *   from MDN.
   */
  @DomName('Node.firstChild')
  @DocsEditable()
  Node get firstChild => _blink.BlinkNode.instance.firstChild_Getter_(this);

  /**
   * The last child of this node.
   *
   * ## Other resources
   *
   * * [Node.lastChild](https://developer.mozilla.org/en-US/docs/Web/API/Node.lastChild)
   *   from MDN.
   */
  @DomName('Node.lastChild')
  @DocsEditable()
  Node get lastChild => _blink.BlinkNode.instance.lastChild_Getter_(this);

  @DomName('Node.localName')
  @DocsEditable()
  String get _localName => _blink.BlinkNode.instance.localName_Getter_(this);

  @DomName('Node.namespaceURI')
  @DocsEditable()
  String get _namespaceUri =>
      _blink.BlinkNode.instance.namespaceURI_Getter_(this);

  /**
   * The next sibling node.
   *
   * ## Other resources
   *
   * * [Node.nextSibling](https://developer.mozilla.org/en-US/docs/Web/API/Node.nextSibling)
   *   from MDN.
   */
  @DomName('Node.nextSibling')
  @DocsEditable()
  Node get nextNode => _blink.BlinkNode.instance.nextSibling_Getter_(this);

  /**
   * The name of this node.
   *
   * This varies by this node's [nodeType].
   *
   * ## Other resources
   *
   * * [Node.nodeName](https://developer.mozilla.org/en-US/docs/Web/API/Node.nodeName)
   *   from MDN. This page contains a table of [nodeName] values for each
   *   [nodeType].
   */
  @DomName('Node.nodeName')
  @DocsEditable()
  String get nodeName => _blink.BlinkNode.instance.nodeName_Getter_(this);

  /**
   * The type of node.
   *
   * This value is one of:
   *
   * * [ATTRIBUTE_NODE] if this node is an attribute.
   * * [CDATA_SECTION_NODE] if this node is a [CDataSection].
   * * [COMMENT_NODE] if this node is a [Comment].
   * * [DOCUMENT_FRAGMENT_NODE] if this node is a [DocumentFragment].
   * * [DOCUMENT_NODE] if this node is a [Document].
   * * [DOCUMENT_TYPE_NODE] if this node is a [DocumentType] node.
   * * [ELEMENT_NODE] if this node is an [Element].
   * * [ENTITY_NODE] if this node is an entity.
   * * [ENTITY_REFERENCE_NODE] if this node is an entity reference.
   * * [NOTATION_NODE] if this node is a notation.
   * * [PROCESSING_INSTRUCTION_NODE] if this node is a [ProcessingInstruction].
   * * [TEXT_NODE] if this node is a [Text] node.
   *
   * ## Other resources
   *
   * * [Node.nodeType](https://developer.mozilla.org/en-US/docs/Web/API/Node.nodeType)
   *   from MDN.
   */
  @DomName('Node.nodeType')
  @DocsEditable()
  int get nodeType => _blink.BlinkNode.instance.nodeType_Getter_(this);

  /**
   * The value of this node.
   *
   * This varies by this type's [nodeType].
   *
   * ## Other resources
   *
   * * [Node.nodeValue](https://developer.mozilla.org/en-US/docs/Web/API/Node.nodeValue)
   *   from MDN. This page contains a table of [nodeValue] values for each
   *   [nodeType].
   */
  @DomName('Node.nodeValue')
  @DocsEditable()
  String get nodeValue => _blink.BlinkNode.instance.nodeValue_Getter_(this);

  /**
   * The document this node belongs to.
   *
   * Returns null if this node does not belong to any document.
   *
   * ## Other resources
   *
   * * [Node.ownerDocument](https://developer.mozilla.org/en-US/docs/Web/API/Node.ownerDocument)
   *   from MDN.
   */
  @DomName('Node.ownerDocument')
  @DocsEditable()
  Document get ownerDocument =>
      _blink.BlinkNode.instance.ownerDocument_Getter_(this);

  /**
   * The parent element of this node.
   *
   * Returns null if this node either does not have a parent or its parent is
   * not an element.
   *
   * ## Other resources
   *
   * * [Node.parentElement](https://developer.mozilla.org/en-US/docs/Web/API/Node.parentElement)
   *   from W3C.
   */
  @DomName('Node.parentElement')
  @DocsEditable()
  Element get parent => _blink.BlinkNode.instance.parentElement_Getter_(this);

  /**
   * The parent node of this node.
   *
   * ## Other resources
   *
   * * [Node.parentNode](https://developer.mozilla.org/en-US/docs/Web/API/Node.parentNode)
   *   from MDN.
   */
  @DomName('Node.parentNode')
  @DocsEditable()
  Node get parentNode => _blink.BlinkNode.instance.parentNode_Getter_(this);

  /**
   * The previous sibling node.
   *
   * ## Other resources
   *
   * * [Node.previousSibling](https://developer.mozilla.org/en-US/docs/Web/API/Node.previousSibling)
   *   from MDN.
   */
  @DomName('Node.previousSibling')
  @DocsEditable()
  Node get previousNode =>
      _blink.BlinkNode.instance.previousSibling_Getter_(this);

  /**
   * All text within this node and its decendents.
   *
   * ## Other resources
   *
   * * [Node.textContent](https://developer.mozilla.org/en-US/docs/Web/API/Node.textContent)
   *   from MDN.
   */
  @DomName('Node.textContent')
  @DocsEditable()
  String get text => _blink.BlinkNode.instance.textContent_Getter_(this);

  /**
   * All text within this node and its decendents.
   *
   * ## Other resources
   *
   * * [Node.textContent](https://developer.mozilla.org/en-US/docs/Web/API/Node.textContent)
   *   from MDN.
   */
  @DomName('Node.textContent')
  @DocsEditable()
  set text(String value) =>
      _blink.BlinkNode.instance.textContent_Setter_(this, value);

  /**
   * Adds a node to the end of the child [nodes] list of this node.
   *
   * If the node already exists in this document, it will be removed from its
   * current parent node, then added to this node.
   *
   * This method is more efficient than `nodes.add`, and is the preferred
   * way of appending a child node.
   */
  @DomName('Node.appendChild')
  @DocsEditable()
  Node append(Node node) =>
      _blink.BlinkNode.instance.appendChild_Callback_1_(this, node);

  /**
   * Returns a copy of this node.
   *
   * If [deep] is `true`, then all of this node's children and decendents are
   * copied as well. If [deep] is `false`, then only this node is copied.
   *
   * ## Other resources
   *
   * * [Node.cloneNode](https://developer.mozilla.org/en-US/docs/Web/API/Node.cloneNode)
   *   from MDN.
   */
  @DomName('Node.cloneNode')
  @DocsEditable()
  Node clone(bool deep) =>
      _blink.BlinkNode.instance.cloneNode_Callback_1_(this, deep);

  /**
   * Returns true if this node contains the specified node.
   *
   * ## Other resources
   *
   * * [Node.contains](https://developer.mozilla.org/en-US/docs/Web/API/Node.contains)
   *   from MDN.
   */
  @DomName('Node.contains')
  @DocsEditable()
  bool contains(Node other) =>
      _blink.BlinkNode.instance.contains_Callback_1_(this, other);

  /**
   * Returns true if this node has any children.
   *
   * ## Other resources
   *
   * * [Node.hasChildNodes](https://developer.mozilla.org/en-US/docs/Web/API/Node.hasChildNodes)
   *   from MDN.
   */
  @DomName('Node.hasChildNodes')
  @DocsEditable()
  bool hasChildNodes() =>
      _blink.BlinkNode.instance.hasChildNodes_Callback_0_(this);

  /**
   * Inserts all of the nodes into this node directly before refChild.
   *
   * ## Other resources
   *
   * * [Node.insertBefore](https://developer.mozilla.org/en-US/docs/Web/API/Node.insertBefore)
   *   from MDN.
   */
  @DomName('Node.insertBefore')
  @DocsEditable()
  Node insertBefore(Node node, Node child) =>
      _blink.BlinkNode.instance.insertBefore_Callback_2_(this, node, child);

  @DomName('Node.removeChild')
  @DocsEditable()
  Node _removeChild(Node child) =>
      _blink.BlinkNode.instance.removeChild_Callback_1_(this, child);

  @DomName('Node.replaceChild')
  @DocsEditable()
  Node _replaceChild(Node node, Node child) =>
      _blink.BlinkNode.instance.replaceChild_Callback_2_(this, node, child);
}
// Copyright (c) 2012, the Dart project authors.  Please see the AUTHORS file
// for details. All rights reserved. Use of this source code is governed by a
// BSD-style license that can be found in the LICENSE file.

// WARNING: Do not edit - generated code.

@DocsEditable()
@DomName('NodeFilter')
@Unstable()
class NodeFilter extends DartHtmlDomObject {
  // To suppress missing implicit constructor warnings.
  factory NodeFilter._() {
    throw new UnsupportedError("Not supported");
  }

  @Deprecated("Internal Use Only")
  external static Type get instanceRuntimeType;

  @Deprecated("Internal Use Only")
  NodeFilter.internal_() {}

  @DomName('NodeFilter.FILTER_ACCEPT')
  @DocsEditable()
  static const int FILTER_ACCEPT = 1;

  @DomName('NodeFilter.FILTER_REJECT')
  @DocsEditable()
  static const int FILTER_REJECT = 2;

  @DomName('NodeFilter.FILTER_SKIP')
  @DocsEditable()
  static const int FILTER_SKIP = 3;

  @DomName('NodeFilter.SHOW_ALL')
  @DocsEditable()
  static const int SHOW_ALL = 0xFFFFFFFF;

  @DomName('NodeFilter.SHOW_COMMENT')
  @DocsEditable()
  static const int SHOW_COMMENT = 0x80;

  @DomName('NodeFilter.SHOW_DOCUMENT')
  @DocsEditable()
  static const int SHOW_DOCUMENT = 0x100;

  @DomName('NodeFilter.SHOW_DOCUMENT_FRAGMENT')
  @DocsEditable()
  static const int SHOW_DOCUMENT_FRAGMENT = 0x400;

  @DomName('NodeFilter.SHOW_DOCUMENT_TYPE')
  @DocsEditable()
  static const int SHOW_DOCUMENT_TYPE = 0x200;

  @DomName('NodeFilter.SHOW_ELEMENT')
  @DocsEditable()
  static const int SHOW_ELEMENT = 0x1;

  @DomName('NodeFilter.SHOW_PROCESSING_INSTRUCTION')
  @DocsEditable()
  static const int SHOW_PROCESSING_INSTRUCTION = 0x40;

  @DomName('NodeFilter.SHOW_TEXT')
  @DocsEditable()
  static const int SHOW_TEXT = 0x4;
}
// Copyright (c) 2013, the Dart project authors.  Please see the AUTHORS file
// for details. All rights reserved. Use of this source code is governed by a
// BSD-style license that can be found in the LICENSE file.

@DomName('NodeIterator')
@Unstable()
class NodeIterator extends DartHtmlDomObject {
  factory NodeIterator(Node root, int whatToShow) {
    return document._createNodeIterator(root, whatToShow, null);
  }
  // To suppress missing implicit constructor warnings.
  factory NodeIterator._() {
    throw new UnsupportedError("Not supported");
  }

  @Deprecated("Internal Use Only")
  external static Type get instanceRuntimeType;

  @Deprecated("Internal Use Only")
  NodeIterator.internal_() {}

  @DomName('NodeIterator.pointerBeforeReferenceNode')
  @DocsEditable()
  bool get pointerBeforeReferenceNode => _blink.BlinkNodeIterator.instance
      .pointerBeforeReferenceNode_Getter_(this);

  @DomName('NodeIterator.referenceNode')
  @DocsEditable()
  Node get referenceNode =>
      _blink.BlinkNodeIterator.instance.referenceNode_Getter_(this);

  @DomName('NodeIterator.root')
  @DocsEditable()
  Node get root => _blink.BlinkNodeIterator.instance.root_Getter_(this);

  @DomName('NodeIterator.whatToShow')
  @DocsEditable()
  int get whatToShow =>
      _blink.BlinkNodeIterator.instance.whatToShow_Getter_(this);

  @DomName('NodeIterator.detach')
  @DocsEditable()
  void detach() => _blink.BlinkNodeIterator.instance.detach_Callback_0_(this);

  @DomName('NodeIterator.nextNode')
  @DocsEditable()
  Node nextNode() =>
      _blink.BlinkNodeIterator.instance.nextNode_Callback_0_(this);

  @DomName('NodeIterator.previousNode')
  @DocsEditable()
  Node previousNode() =>
      _blink.BlinkNodeIterator.instance.previousNode_Callback_0_(this);
}
// Copyright (c) 2012, the Dart project authors.  Please see the AUTHORS file
// for details. All rights reserved. Use of this source code is governed by a
// BSD-style license that can be found in the LICENSE file.

// WARNING: Do not edit - generated code.

@DocsEditable()
@DomName('NodeList')
class NodeList extends DartHtmlDomObject
    with ListMixin<Node>, ImmutableListMixin<Node>
    implements List<Node> {
  // To suppress missing implicit constructor warnings.
  factory NodeList._() {
    throw new UnsupportedError("Not supported");
  }

  @Deprecated("Internal Use Only")
  external static Type get instanceRuntimeType;

  @Deprecated("Internal Use Only")
  NodeList.internal_() {}

  @DomName('NodeList.length')
  @DocsEditable()
  int get length => _blink.BlinkNodeList.instance.length_Getter_(this);

  Node operator [](int index) {
    if (index < 0 || index >= length) throw new RangeError.index(index, this);
    return _nativeIndexedGetter(index);
  }

  Node _nativeIndexedGetter(int index) =>
      (_blink.BlinkNodeList.instance.item_Callback_1_(this, index));

  void operator []=(int index, Node value) {
    throw new UnsupportedError("Cannot assign element of immutable List.");
  }
  // -- start List<Node> mixins.
  // Node is the element type.

  set length(int value) {
    throw new UnsupportedError("Cannot resize immutable List.");
  }

  Node get first {
    if (this.length > 0) {
      return _nativeIndexedGetter(0);
    }
    throw new StateError("No elements");
  }

  Node get last {
    int len = this.length;
    if (len > 0) {
      return _nativeIndexedGetter(len - 1);
    }
    throw new StateError("No elements");
  }

  Node get single {
    int len = this.length;
    if (len == 1) {
      return _nativeIndexedGetter(0);
    }
    if (len == 0) throw new StateError("No elements");
    throw new StateError("More than one element");
  }

  Node elementAt(int index) => this[index];
  // -- end List<Node> mixins.

  @DomName('NodeList.item')
  @DocsEditable()
  Node _item(int index) =>
      _blink.BlinkNodeList.instance.item_Callback_1_(this, index);
}
// Copyright (c) 2012, the Dart project authors.  Please see the AUTHORS file
// for details. All rights reserved. Use of this source code is governed by a
// BSD-style license that can be found in the LICENSE file.

// WARNING: Do not edit - generated code.

@DocsEditable()
@DomName('NonDocumentTypeChildNode')
@Experimental() // untriaged
class NonDocumentTypeChildNode extends DartHtmlDomObject {
  // To suppress missing implicit constructor warnings.
  factory NonDocumentTypeChildNode._() {
    throw new UnsupportedError("Not supported");
  }

  @Deprecated("Internal Use Only")
  external static Type get instanceRuntimeType;

  @Deprecated("Internal Use Only")
  NonDocumentTypeChildNode.internal_() {}

  @DomName('NonDocumentTypeChildNode.nextElementSibling')
  @DocsEditable()
  @Experimental() // untriaged
  Element get nextElementSibling =>
      _blink.BlinkNonDocumentTypeChildNode.instance
          .nextElementSibling_Getter_(this);

  @DomName('NonDocumentTypeChildNode.previousElementSibling')
  @DocsEditable()
  @Experimental() // untriaged
  Element get previousElementSibling =>
      _blink.BlinkNonDocumentTypeChildNode.instance
          .previousElementSibling_Getter_(this);
}
// Copyright (c) 2012, the Dart project authors.  Please see the AUTHORS file
// for details. All rights reserved. Use of this source code is governed by a
// BSD-style license that can be found in the LICENSE file.

// WARNING: Do not edit - generated code.

@DocsEditable()
@DomName('NonElementParentNode')
@Experimental() // untriaged
class NonElementParentNode extends DartHtmlDomObject {
  // To suppress missing implicit constructor warnings.
  factory NonElementParentNode._() {
    throw new UnsupportedError("Not supported");
  }

  @Deprecated("Internal Use Only")
  external static Type get instanceRuntimeType;

  @Deprecated("Internal Use Only")
  NonElementParentNode.internal_() {}

  @DomName('NonElementParentNode.getElementById')
  @DocsEditable()
  @Experimental() // untriaged
  Element getElementById(String elementId) =>
      _blink.BlinkNonElementParentNode.instance
          .getElementById_Callback_1_(this, elementId);
}
// Copyright (c) 2013, the Dart project authors.  Please see the AUTHORS file
// for details. All rights reserved. Use of this source code is governed by a
// BSD-style license that can be found in the LICENSE file.

@DomName('Notification')
// http://www.w3.org/TR/notifications/#notification
@Experimental() // experimental
class Notification extends EventTarget {
  factory Notification(String title,
      {String dir: null,
      String body: null,
      String lang: null,
      String tag: null,
      String icon: null}) {
    var parsedOptions = {};
    if (dir != null) parsedOptions['dir'] = dir;
    if (body != null) parsedOptions['body'] = body;
    if (lang != null) parsedOptions['lang'] = lang;
    if (tag != null) parsedOptions['tag'] = tag;
    if (icon != null) parsedOptions['icon'] = icon;
    return Notification._factoryNotification(title, parsedOptions);
  }
  // To suppress missing implicit constructor warnings.
  factory Notification._() {
    throw new UnsupportedError("Not supported");
  }

  /**
   * Static factory designed to expose `click` events to event
   * handlers that are not necessarily instances of [Notification].
   *
   * See [EventStreamProvider] for usage information.
   */
  @DomName('Notification.clickEvent')
  @DocsEditable()
  static const EventStreamProvider<Event> clickEvent =
      const EventStreamProvider<Event>('click');

  /**
   * Static factory designed to expose `close` events to event
   * handlers that are not necessarily instances of [Notification].
   *
   * See [EventStreamProvider] for usage information.
   */
  @DomName('Notification.closeEvent')
  @DocsEditable()
  static const EventStreamProvider<Event> closeEvent =
      const EventStreamProvider<Event>('close');

  /**
   * Static factory designed to expose `error` events to event
   * handlers that are not necessarily instances of [Notification].
   *
   * See [EventStreamProvider] for usage information.
   */
  @DomName('Notification.errorEvent')
  @DocsEditable()
  static const EventStreamProvider<Event> errorEvent =
      const EventStreamProvider<Event>('error');

  /**
   * Static factory designed to expose `show` events to event
   * handlers that are not necessarily instances of [Notification].
   *
   * See [EventStreamProvider] for usage information.
   */
  @DomName('Notification.showEvent')
  @DocsEditable()
  static const EventStreamProvider<Event> showEvent =
      const EventStreamProvider<Event>('show');

  @DomName('Notification.Notification')
  @DocsEditable()
  static Notification _factoryNotification(String title, [Map options]) {
    if (options != null) {
      var options_1 = convertDartToNative_Dictionary(options);
      return _blink.BlinkNotification.instance
          .constructorCallback_2_(title, options_1);
    }
    return _blink.BlinkNotification.instance.constructorCallback_1_(title);
  }

  @Deprecated("Internal Use Only")
  external static Type get instanceRuntimeType;

  @Deprecated("Internal Use Only")
  Notification.internal_() : super.internal_();

  /// Checks if this type is supported on the current platform.
  static bool get supported => true;

  @DomName('Notification.body')
  @DocsEditable()
  @Experimental() // untriaged
  String get body => _blink.BlinkNotification.instance.body_Getter_(this);

  @DomName('Notification.data')
  @DocsEditable()
  @Experimental() // untriaged
  Object get data => (_blink.BlinkNotification.instance.data_Getter_(this));

  @DomName('Notification.dir')
  @DocsEditable()
  @Experimental() // nonstandard
  String get dir => _blink.BlinkNotification.instance.dir_Getter_(this);

  @DomName('Notification.icon')
  @DocsEditable()
  @Experimental() // untriaged
  String get icon => _blink.BlinkNotification.instance.icon_Getter_(this);

  @DomName('Notification.lang')
  @DocsEditable()
  @Experimental() // untriaged
  String get lang => _blink.BlinkNotification.instance.lang_Getter_(this);

  @DomName('Notification.permission')
  @DocsEditable()
  String get permission =>
      _blink.BlinkNotification.instance.permission_Getter_();

  @DomName('Notification.silent')
  @DocsEditable()
  @Experimental() // untriaged
  bool get silent => _blink.BlinkNotification.instance.silent_Getter_(this);

  @DomName('Notification.tag')
  @DocsEditable()
  @Experimental() // nonstandard
  String get tag => _blink.BlinkNotification.instance.tag_Getter_(this);

  @DomName('Notification.title')
  @DocsEditable()
  @Experimental() // untriaged
  String get title => _blink.BlinkNotification.instance.title_Getter_(this);

  @DomName('Notification.vibrate')
  @DocsEditable()
  @Experimental() // untriaged
  List<int> get vibrate =>
      _blink.BlinkNotification.instance.vibrate_Getter_(this);

  @DomName('Notification.close')
  @DocsEditable()
  void close() => _blink.BlinkNotification.instance.close_Callback_0_(this);

  static void _requestPermission([_NotificationPermissionCallback callback]) {
    if (callback != null) {
      _blink.BlinkNotification.instance.requestPermission_Callback_1_(callback);
      return;
    }
    _blink.BlinkNotification.instance.requestPermission_Callback_0_();
    return;
  }

  static Future<String> requestPermission() {
    var completer = new Completer<String>();
    _requestPermission((value) {
      completer.complete(value);
    });
    return completer.future;
  }

  /// Stream of `click` events handled by this [Notification].
  @DomName('Notification.onclick')
  @DocsEditable()
  Stream<Event> get onClick => clickEvent.forTarget(this);

  /// Stream of `close` events handled by this [Notification].
  @DomName('Notification.onclose')
  @DocsEditable()
  Stream<Event> get onClose => closeEvent.forTarget(this);

  /// Stream of `error` events handled by this [Notification].
  @DomName('Notification.onerror')
  @DocsEditable()
  Stream<Event> get onError => errorEvent.forTarget(this);

  /// Stream of `show` events handled by this [Notification].
  @DomName('Notification.onshow')
  @DocsEditable()
  Stream<Event> get onShow => showEvent.forTarget(this);
}
// Copyright (c) 2012, the Dart project authors.  Please see the AUTHORS file
// for details. All rights reserved. Use of this source code is governed by a
// BSD-style license that can be found in the LICENSE file.

// WARNING: Do not edit - generated code.

@DocsEditable()
@DomName('NotificationEvent')
@Experimental() // untriaged
class NotificationEvent extends ExtendableEvent {
  // To suppress missing implicit constructor warnings.
  factory NotificationEvent._() {
    throw new UnsupportedError("Not supported");
  }

  @DomName('NotificationEvent.NotificationEvent')
  @DocsEditable()
  factory NotificationEvent(String type, [Map eventInitDict]) {
    if (eventInitDict != null) {
      var eventInitDict_1 = convertDartToNative_Dictionary(eventInitDict);
      return _blink.BlinkNotificationEvent.instance
          .constructorCallback_2_(type, eventInitDict_1);
    }
    return _blink.BlinkNotificationEvent.instance.constructorCallback_1_(type);
  }

  @Deprecated("Internal Use Only")
  external static Type get instanceRuntimeType;

  @Deprecated("Internal Use Only")
  NotificationEvent.internal_() : super.internal_();

  @DomName('NotificationEvent.notification')
  @DocsEditable()
  @Experimental() // untriaged
  Notification get notification =>
      _blink.BlinkNotificationEvent.instance.notification_Getter_(this);
}
// Copyright (c) 2012, the Dart project authors.  Please see the AUTHORS file
// for details. All rights reserved. Use of this source code is governed by a
// BSD-style license that can be found in the LICENSE file.

// WARNING: Do not edit - generated code.

@DomName('NotificationPermissionCallback')
// http://www.w3.org/TR/notifications/#notificationpermissioncallback
@Experimental()
typedef void _NotificationPermissionCallback(String permission);
// Copyright (c) 2012, the Dart project authors.  Please see the AUTHORS file
// for details. All rights reserved. Use of this source code is governed by a
// BSD-style license that can be found in the LICENSE file.

// WARNING: Do not edit - generated code.

@DocsEditable()
@DomName('HTMLOListElement')
class OListElement extends HtmlElement {
  // To suppress missing implicit constructor warnings.
  factory OListElement._() {
    throw new UnsupportedError("Not supported");
  }

  @DomName('HTMLOListElement.HTMLOListElement')
  @DocsEditable()
  factory OListElement() => document.createElement("ol");

  @Deprecated("Internal Use Only")
  external static Type get instanceRuntimeType;

  @Deprecated("Internal Use Only")
  OListElement.internal_() : super.internal_();

  /**
   * Constructor instantiated by the DOM when a custom element has been created.
   *
   * This can only be called by subclasses from their created constructor.
   */
  OListElement.created() : super.created();

  @DomName('HTMLOListElement.reversed')
  @DocsEditable()
  bool get reversed =>
      _blink.BlinkHTMLOListElement.instance.reversed_Getter_(this);

  @DomName('HTMLOListElement.reversed')
  @DocsEditable()
  set reversed(bool value) =>
      _blink.BlinkHTMLOListElement.instance.reversed_Setter_(this, value);

  @DomName('HTMLOListElement.start')
  @DocsEditable()
  int get start => _blink.BlinkHTMLOListElement.instance.start_Getter_(this);

  @DomName('HTMLOListElement.start')
  @DocsEditable()
  set start(int value) =>
      _blink.BlinkHTMLOListElement.instance.start_Setter_(this, value);

  @DomName('HTMLOListElement.type')
  @DocsEditable()
  String get type => _blink.BlinkHTMLOListElement.instance.type_Getter_(this);

  @DomName('HTMLOListElement.type')
  @DocsEditable()
  set type(String value) =>
      _blink.BlinkHTMLOListElement.instance.type_Setter_(this, value);
}
// Copyri