# -*- coding: utf-8 -*-
# Copyright 2006 Joe Wreschnig
#
# This program is free software; you can redistribute it and/or modify
# it under the terms of the GNU General Public License version 2 as
# published by the Free Software Foundation

"""Various library classes.

Libraries are (roughly) dicts of items, and in the case of Quod Libet,
these items are songs. Libraries are in charge of alerting the rest of
the program when songs have been added, changed, or removed. They can
also be queried in various ways.
"""

import time

from quodlibet import print_d

from quodlibet.library.libraries import SongFileLibrary, SongLibrary
from quodlibet.library.librarians import SongLibrarian
from quodlibet.util.path import mtime


def init(cache_fn=None):
    """Set up the library and return the main one.

    Return a main library, and set a librarian for
    all future SongLibraries.
    """

    SongFileLibrary.librarian = SongLibrary.librarian = SongLibrarian()
    library = SongFileLibrary("main")
    if cache_fn:
        library.load(cache_fn)
    return library


def save(save_period=None):
    """Save all registered libraries that have a filename and are marked dirty.

    If `save_period` (seconds) is given the library will only be saved if
    it hasn't been in the last `save_period` seconds.
    """

    print_d("Saving all libraries...")

    librarian = SongFileLibrary.librarian
    for lib in librarian.libraries.values():
        filename = lib.filename
        if not filename or not lib.dirty:
            continue

        if not save_period or abs(time.time() - mtime(filename)) > save_period:
            lib.save()
