
'use strict';

var commander = require('commander');

var debug     = require('debug')('pm2:cli');
var PM2       = require('../..');
var Log       = require('../../lib/API/Log');
var cst       = require('../../constants.js');
var pkg       = require('../../package.json');
var path      = require('path');
var pm2;

//process.env.PM2_SILENT = 'true';

commander.version(pkg.version)
  .option('--raw', 'raw log output')
  .option('--json', 'output logs in json format')
  .option('--log-date-format <momentjs format>', 'add custom prefix timestamp to logs')
  .option('--delay <seconds>', 'delay start of configuration file by <seconds>', 0)
  .option('--web [port]', 'launch process web api on [port] default to 9615')
  .option('--format', 'output logs formated like key=val')
  .option('--only <application-name>', 'only act on one application of configuration')
  .option('--secret [key]', 'keymetrics secret key')
  .option('--secret [key]', 'keymetrics secret key')
  .option('--public [key]', 'keymetrics public key')
  .option('--machine-name [name]', 'keymetrics machine name')
  .option('--auto-exit', 'exit if all processes are errored/stopped or 0 apps launched')
  .option('--env [name]', 'select env_[name] env variables in process config file')
  .option('--watch', 'Watch and Restart')
  .option('-i --instances <number>', 'launch [number] instances with load-balancer')
  .usage('start <app>');

function start(cmd, opts) {
  pm2 = new PM2.custom({
    secret_key : process.env.KEYMETRICS_SECRET || commander.secret,
    public_key : process.env.KEYMETRICS_PUBLIC || commander.public,
    machine_name : process.env.INSTANCE_NAME || commander.machineName,
    daemon_mode : true
  });

  if (commander.autoExit)
    autoExit();

  pm2.connect(function() {

    if (opts.web) {
      var port = opts.web === true ? cst.WEB_PORT : opts.web;
      pm2.web(port);
    }

    run(cmd, opts);
  });
}

commander.command('*')
  .action(function(cmd){
    start(cmd, commander);
  });

// @todo need to allow passing same option than pm2 start
commander.command('start <file|json_file>')
  .description('start json_file or application')
  .action(function(cmd) {
    start(cmd, commander);
  });

if (process.argv.length == 2) {
  commander.outputHelp();
  process.exit(1);
}

commander.parse(process.argv);

process.on('SIGINT', function() {
  exitPM2();
});

process.on('SIGTERM', function() {
  exitPM2();
});

function run(cmd, opts) {
  var needRaw = commander.raw;

  var timestamp = commander.logDateFormat;
  if (timestamp)
    timestamp = typeof cmd.timestamp === 'string' ? cmd.timestamp : 'YYYY-MM-DD-HH:mm:ss';

  function exec() {
    pm2.start(cmd, opts, function(err, obj) {

      if (process.env.PM2_RUNTIME_DEBUG)
        return pm2.disconnect(function() {});

      if (err)
        throw new Error(err.message);
      if (commander.format)
        Log.formatStream(pm2.Client, 'all', needRaw, timestamp, false);
      else if (commander.json)
        Log.jsonStream(pm2.Client);
      else
        Log.stream(pm2.Client, 'all', needRaw, timestamp, false);
    });
  }
  setTimeout(exec.bind(this), opts.delay * 1000);
}

function exitPM2() {
  console.log('Exiting PM2');
  pm2.kill(function() {
    process.exit(0);
  });
}

/**
 * Exit current PM2 instance if 0 app is online
 * function activated via --auto-exit
 */
function autoExit() {
  var interval = 3000;
  var aliveInterval = interval * 1.5;

  setTimeout(function () {
    var alive = false
    var aliveTimer = setTimeout(function () {
      if (!alive) {
        console.error('PM2 Daemon is dead');
        process.exit(1);
      }
    }, aliveInterval);

    pm2.list(function (err, apps) {
      if (err) {
        console.log('pm2.list got error')
        console.error(err);
        exitPM2();
      }

      clearTimeout(aliveTimer);
      alive = true;

      var appOnline = 0;

      apps.forEach(function (app) {
        if (app.pm2_env.status === cst.ONLINE_STATUS ||
              app.pm2_env.status === cst.LAUNCHING_STATUS) {
          appOnline++;
        }
      });

      if (appOnline === 0) {
        console.log('0 application online, exiting');
        exitPM2();
      }
      autoExit();
    });
  }, interval);
}
