/*
 * Decompiled with CFR 0.152.
 */
package org.elasticsearch.index.engine;

import java.io.Closeable;
import java.io.FileNotFoundException;
import java.io.IOException;
import java.nio.file.NoSuchFileException;
import java.util.Arrays;
import java.util.Base64;
import java.util.Comparator;
import java.util.HashMap;
import java.util.List;
import java.util.Locale;
import java.util.Map;
import java.util.Objects;
import java.util.concurrent.TimeUnit;
import java.util.concurrent.atomic.AtomicBoolean;
import java.util.concurrent.locks.Condition;
import java.util.concurrent.locks.Lock;
import java.util.concurrent.locks.ReentrantLock;
import java.util.concurrent.locks.ReentrantReadWriteLock;
import java.util.function.Function;
import org.apache.logging.log4j.Logger;
import org.apache.logging.log4j.message.ParameterizedMessage;
import org.apache.lucene.index.DirectoryReader;
import org.apache.lucene.index.FilterLeafReader;
import org.apache.lucene.index.IndexCommit;
import org.apache.lucene.index.IndexFileNames;
import org.apache.lucene.index.IndexReader;
import org.apache.lucene.index.LeafReader;
import org.apache.lucene.index.LeafReaderContext;
import org.apache.lucene.index.SegmentCommitInfo;
import org.apache.lucene.index.SegmentInfos;
import org.apache.lucene.index.SegmentReader;
import org.apache.lucene.index.SnapshotDeletionPolicy;
import org.apache.lucene.index.Term;
import org.apache.lucene.search.IndexSearcher;
import org.apache.lucene.search.SearcherManager;
import org.apache.lucene.store.AlreadyClosedException;
import org.apache.lucene.store.Directory;
import org.apache.lucene.store.IOContext;
import org.apache.lucene.util.Accountable;
import org.apache.lucene.util.Accountables;
import org.apache.lucene.util.SetOnce;
import org.elasticsearch.ExceptionsHelper;
import org.elasticsearch.common.Nullable;
import org.elasticsearch.common.bytes.BytesReference;
import org.elasticsearch.common.collect.ImmutableOpenMap;
import org.elasticsearch.common.io.stream.StreamInput;
import org.elasticsearch.common.io.stream.StreamOutput;
import org.elasticsearch.common.io.stream.Writeable;
import org.elasticsearch.common.lease.Releasable;
import org.elasticsearch.common.lease.Releasables;
import org.elasticsearch.common.logging.Loggers;
import org.elasticsearch.common.lucene.Lucene;
import org.elasticsearch.common.lucene.uid.VersionsResolver;
import org.elasticsearch.common.metrics.CounterMetric;
import org.elasticsearch.common.unit.TimeValue;
import org.elasticsearch.common.util.concurrent.ReleasableLock;
import org.elasticsearch.index.VersionType;
import org.elasticsearch.index.engine.CommitStats;
import org.elasticsearch.index.engine.EngineConfig;
import org.elasticsearch.index.engine.EngineException;
import org.elasticsearch.index.engine.EngineSearcher;
import org.elasticsearch.index.engine.Segment;
import org.elasticsearch.index.engine.SegmentsStats;
import org.elasticsearch.index.engine.VersionConflictEngineException;
import org.elasticsearch.index.mapper.ParseContext;
import org.elasticsearch.index.mapper.ParsedDocument;
import org.elasticsearch.index.mapper.Uid;
import org.elasticsearch.index.merge.MergeStats;
import org.elasticsearch.index.shard.ShardId;
import org.elasticsearch.index.store.Store;
import org.elasticsearch.index.translog.Translog;

public abstract class Engine
implements Closeable {
    public static final String SYNC_COMMIT_ID = "sync_id";
    protected final ShardId shardId;
    protected final Logger logger;
    protected final EngineConfig engineConfig;
    protected final Store store;
    protected final AtomicBoolean isClosed = new AtomicBoolean(false);
    protected final EventListener eventListener;
    protected final SnapshotDeletionPolicy deletionPolicy;
    protected final ReentrantLock failEngineLock = new ReentrantLock();
    protected final ReentrantReadWriteLock rwl = new ReentrantReadWriteLock();
    protected final ReleasableLock readLock = new ReleasableLock(this.rwl.readLock());
    protected final ReleasableLock writeLock = new ReleasableLock(this.rwl.writeLock());
    protected final SetOnce<Exception> failedEngine = new SetOnce();
    protected volatile long lastWriteNanos = System.nanoTime();

    protected Engine(EngineConfig engineConfig) {
        Objects.requireNonNull(engineConfig.getStore(), "Store must be provided to the engine");
        Objects.requireNonNull(engineConfig.getDeletionPolicy(), "Snapshot deletion policy must be provided to the engine");
        this.engineConfig = engineConfig;
        this.shardId = engineConfig.getShardId();
        this.store = engineConfig.getStore();
        this.logger = Loggers.getLogger(Engine.class, engineConfig.getIndexSettings().getSettings(), engineConfig.getShardId(), new String[0]);
        this.eventListener = engineConfig.getEventListener();
        this.deletionPolicy = engineConfig.getDeletionPolicy();
    }

    protected static long guardedRamBytesUsed(Accountable a) {
        if (a == null) {
            return 0L;
        }
        return a.ramBytesUsed();
    }

    protected static SegmentReader segmentReader(LeafReader reader) {
        if (reader instanceof SegmentReader) {
            return (SegmentReader)reader;
        }
        if (reader instanceof FilterLeafReader) {
            FilterLeafReader fReader = (FilterLeafReader)reader;
            return Engine.segmentReader(FilterLeafReader.unwrap((LeafReader)fReader));
        }
        throw new IllegalStateException("Can not extract segment reader from given index reader [" + reader + "]");
    }

    protected static boolean isMergedSegment(LeafReader reader) {
        Map diagnostics = Engine.segmentReader((LeafReader)reader).getSegmentInfo().info.getDiagnostics();
        String source = (String)diagnostics.get("source");
        assert (Arrays.asList("addIndexes(CodecReader...)", "flush", "merge").contains(source)) : "Unknown source " + source;
        return "merge".equals(source);
    }

    protected Searcher newSearcher(String source, IndexSearcher searcher, SearcherManager manager) {
        return new EngineSearcher(source, searcher, manager, this.store, this.logger);
    }

    public final EngineConfig config() {
        return this.engineConfig;
    }

    protected abstract SegmentInfos getLastCommittedSegmentInfos();

    public MergeStats getMergeStats() {
        return new MergeStats();
    }

    public abstract long getIndexThrottleTimeInMillis();

    public abstract boolean isThrottled();

    public abstract IndexResult index(Index var1) throws IOException;

    public abstract DeleteResult delete(Delete var1) throws IOException;

    public abstract SyncedFlushResult syncFlush(String var1, CommitId var2) throws EngineException;

    protected final GetResult getFromSearcher(Get get, Function<String, Searcher> searcherFactory) throws EngineException {
        VersionsResolver.DocIdAndVersion docIdAndVersion;
        Searcher searcher = searcherFactory.apply("get");
        try {
            docIdAndVersion = VersionsResolver.loadDocIdAndVersion(searcher.reader(), get.uid());
        }
        catch (Exception e) {
            Releasables.closeWhileHandlingException(searcher);
            throw new EngineException(this.shardId, "Couldn't resolve version", e, new Object[0]);
        }
        if (docIdAndVersion != null && get.versionType().isVersionConflictForReads(docIdAndVersion.version, get.version())) {
            Releasables.close(searcher);
            Uid uid = Uid.createUid(get.uid().text());
            throw new VersionConflictEngineException(this.shardId, uid.type(), uid.id(), get.versionType().explainConflictForReads(docIdAndVersion.version, get.version()));
        }
        if (docIdAndVersion != null) {
            return new GetResult(searcher, docIdAndVersion);
        }
        Releasables.close(searcher);
        return GetResult.NOT_EXISTS;
    }

    public final GetResult get(Get get) throws EngineException {
        return this.get(get, this::acquireSearcher);
    }

    public abstract GetResult get(Get var1, Function<String, Searcher> var2) throws EngineException;

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    public final Searcher acquireSearcher(String source) throws EngineException {
        boolean success = false;
        this.store.incRef();
        try {
            Searcher searcher;
            block11: {
                SearcherManager manager = this.getSearcherManager();
                IndexSearcher searcher2 = (IndexSearcher)manager.acquire();
                try {
                    Searcher retVal = this.newSearcher(source, searcher2, manager);
                    success = true;
                    searcher = retVal;
                    if (success) break block11;
                }
                catch (Throwable throwable) {
                    try {
                        if (!success) {
                            manager.release((Object)searcher2);
                        }
                        throw throwable;
                    }
                    catch (AlreadyClosedException ex) {
                        throw ex;
                    }
                    catch (Exception ex) {
                        this.ensureOpen();
                        this.logger.error(() -> new ParameterizedMessage("failed to acquire searcher, source {}", (Object)source), (Throwable)ex);
                        throw new EngineException(this.shardId, "failed to acquire searcher, source " + source, ex, new Object[0]);
                    }
                }
                manager.release((Object)searcher2);
            }
            return searcher;
        }
        finally {
            if (!success) {
                this.store.decRef();
            }
        }
    }

    public abstract Translog getTranslog();

    protected void ensureOpen() {
        if (this.isClosed.get()) {
            throw new AlreadyClosedException(this.shardId + " engine is closed", (Throwable)this.failedEngine.get());
        }
    }

    public CommitStats commitStats() {
        return new CommitStats(this.getLastCommittedSegmentInfos());
    }

    protected static SegmentInfos readLastCommittedSegmentInfos(SearcherManager sm, Store store) throws IOException {
        IndexSearcher searcher = (IndexSearcher)sm.acquire();
        try {
            IndexCommit latestCommit = ((DirectoryReader)searcher.getIndexReader()).getIndexCommit();
            SegmentInfos segmentInfos = Lucene.readSegmentInfos(latestCommit);
            return segmentInfos;
        }
        catch (IOException e) {
            try {
                SegmentInfos segmentInfos = store.readLastCommittedSegmentsInfo();
                return segmentInfos;
            }
            catch (IOException e2) {
                e2.addSuppressed(e);
                throw e2;
            }
        }
        finally {
            sm.release((Object)searcher);
        }
    }

    public final SegmentsStats segmentsStats(boolean includeSegmentFileSizes) {
        this.ensureOpen();
        try (Searcher searcher = this.acquireSearcher("segments_stats");){
            SegmentsStats stats = new SegmentsStats();
            for (LeafReaderContext reader : searcher.reader().leaves()) {
                SegmentReader segmentReader = Engine.segmentReader(reader.reader());
                stats.add(1L, segmentReader.ramBytesUsed());
                stats.addTermsMemoryInBytes(Engine.guardedRamBytesUsed((Accountable)segmentReader.getPostingsReader()));
                stats.addStoredFieldsMemoryInBytes(Engine.guardedRamBytesUsed((Accountable)segmentReader.getFieldsReader()));
                stats.addTermVectorsMemoryInBytes(Engine.guardedRamBytesUsed((Accountable)segmentReader.getTermVectorsReader()));
                stats.addNormsMemoryInBytes(Engine.guardedRamBytesUsed((Accountable)segmentReader.getNormsReader()));
                stats.addPointsMemoryInBytes(Engine.guardedRamBytesUsed((Accountable)segmentReader.getPointsReader()));
                stats.addDocValuesMemoryInBytes(Engine.guardedRamBytesUsed((Accountable)segmentReader.getDocValuesReader()));
                if (!includeSegmentFileSizes) continue;
                stats.addFileSizes(this.getSegmentFileSizes(segmentReader));
            }
            this.writerSegmentStats(stats);
            SegmentsStats segmentsStats = stats;
            return segmentsStats;
        }
    }

    private ImmutableOpenMap<String, Long> getSegmentFileSizes(SegmentReader segmentReader) {
        String[] files;
        Directory directory = null;
        SegmentCommitInfo segmentCommitInfo = segmentReader.getSegmentInfo();
        boolean useCompoundFile = segmentCommitInfo.info.getUseCompoundFile();
        if (useCompoundFile) {
            try {
                directory = this.engineConfig.getCodec().compoundFormat().getCompoundReader(segmentReader.directory(), segmentCommitInfo.info, IOContext.READ);
            }
            catch (IOException e) {
                this.logger.warn(() -> new ParameterizedMessage("Error when opening compound reader for Directory [{}] and SegmentCommitInfo [{}]", (Object)segmentReader.directory(), (Object)segmentCommitInfo), (Throwable)e);
                return ImmutableOpenMap.of();
            }
        } else {
            directory = segmentReader.directory();
        }
        assert (directory != null);
        if (useCompoundFile) {
            try {
                files = directory.listAll();
            }
            catch (IOException e) {
                Directory finalDirectory = directory;
                this.logger.warn(() -> new ParameterizedMessage("Couldn't list Compound Reader Directory [{}]", (Object)finalDirectory), (Throwable)e);
                return ImmutableOpenMap.of();
            }
        }
        try {
            files = segmentReader.getSegmentInfo().files().toArray(new String[0]);
        }
        catch (IOException e) {
            this.logger.warn(() -> new ParameterizedMessage("Couldn't list Directory from SegmentReader [{}] and SegmentInfo [{}]", (Object)segmentReader, (Object)segmentReader.getSegmentInfo()), (Throwable)e);
            return ImmutableOpenMap.of();
        }
        ImmutableOpenMap.Builder<String, Long> map = ImmutableOpenMap.builder();
        for (String file : files) {
            Directory finalDirectory;
            String extension = IndexFileNames.getExtension((String)file);
            long length = 0L;
            try {
                length = directory.fileLength(file);
            }
            catch (FileNotFoundException | NoSuchFileException e) {
                finalDirectory = directory;
                this.logger.warn(() -> new ParameterizedMessage("Tried to query fileLength but file is gone [{}] [{}]", (Object)finalDirectory, (Object)file), (Throwable)e);
            }
            catch (IOException e) {
                finalDirectory = directory;
                this.logger.warn(() -> new ParameterizedMessage("Error when trying to query fileLength [{}] [{}]", (Object)finalDirectory, (Object)file), (Throwable)e);
            }
            if (length == 0L) continue;
            map.put(extension, length);
        }
        if (useCompoundFile && directory != null) {
            try {
                directory.close();
            }
            catch (IOException e) {
                Directory finalDirectory = directory;
                this.logger.warn(() -> new ParameterizedMessage("Error when closing compound reader on Directory [{}]", (Object)finalDirectory), (Throwable)e);
            }
        }
        return map.build();
    }

    protected void writerSegmentStats(SegmentsStats stats) {
        stats.addVersionMapMemoryInBytes(0L);
        stats.addIndexWriterMemoryInBytes(0L);
    }

    public abstract long getIndexBufferRAMBytesUsed();

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    protected Segment[] getSegmentInfo(SegmentInfos lastCommittedSegmentInfos, boolean verbose) {
        Segment segment;
        this.ensureOpen();
        HashMap<String, Segment> segments = new HashMap<String, Segment>();
        try (Searcher searcher = this.acquireSearcher("segments");){
            for (LeafReaderContext reader : searcher.reader().leaves()) {
                SegmentCommitInfo info = Engine.segmentReader(reader.reader()).getSegmentInfo();
                assert (!segments.containsKey(info.info.name));
                segment = new Segment(info.info.name);
                segment.search = true;
                segment.docCount = reader.reader().numDocs();
                segment.delDocCount = reader.reader().numDeletedDocs();
                segment.version = info.info.getVersion();
                segment.compound = info.info.getUseCompoundFile();
                try {
                    segment.sizeInBytes = info.sizeInBytes();
                }
                catch (IOException e) {
                    this.logger.trace(() -> new ParameterizedMessage("failed to get size for [{}]", (Object)info.info.name), (Throwable)e);
                }
                SegmentReader segmentReader = Engine.segmentReader(reader.reader());
                segment.memoryInBytes = segmentReader.ramBytesUsed();
                if (verbose) {
                    segment.ramTree = Accountables.namedAccountable((String)"root", (Accountable)segmentReader);
                }
                segments.put(info.info.name, segment);
            }
        }
        if (lastCommittedSegmentInfos != null) {
            SegmentInfos infos = lastCommittedSegmentInfos;
            for (SegmentCommitInfo info : infos) {
                segment = (Segment)segments.get(info.info.name);
                if (segment == null) {
                    segment = new Segment(info.info.name);
                    segment.search = false;
                    segment.committed = true;
                    segment.docCount = info.info.maxDoc();
                    segment.delDocCount = info.getDelCount();
                    segment.version = info.info.getVersion();
                    segment.compound = info.info.getUseCompoundFile();
                    try {
                        segment.sizeInBytes = info.sizeInBytes();
                    }
                    catch (IOException e) {
                        this.logger.trace(() -> new ParameterizedMessage("failed to get size for [{}]", (Object)info.info.name), (Throwable)e);
                    }
                    segments.put(info.info.name, segment);
                    continue;
                }
                segment.committed = true;
            }
        }
        Segment[] segmentsArr = segments.values().toArray(new Segment[segments.values().size()]);
        Arrays.sort(segmentsArr, new Comparator<Segment>(){

            @Override
            public int compare(Segment o1, Segment o2) {
                return (int)(o1.getGeneration() - o2.getGeneration());
            }
        });
        return segmentsArr;
    }

    public abstract List<Segment> segments(boolean var1);

    public final boolean refreshNeeded() {
        if (this.store.tryIncRef()) {
            try {
                boolean bl = !this.getSearcherManager().isSearcherCurrent();
                return bl;
            }
            catch (IOException e) {
                this.logger.error("failed to access searcher manager", (Throwable)e);
                this.failEngine("failed to access searcher manager", e);
                throw new EngineException(this.shardId, "failed to access searcher manager", e, new Object[0]);
            }
            finally {
                this.store.decRef();
            }
        }
        return false;
    }

    @Nullable
    public abstract void refresh(String var1) throws EngineException;

    public abstract void writeIndexingBuffer() throws EngineException;

    public abstract CommitId flush(boolean var1, boolean var2) throws EngineException;

    public abstract CommitId flush() throws EngineException;

    public void forceMerge(boolean flush) throws IOException {
        this.forceMerge(flush, 1, false, false, false);
    }

    public abstract void forceMerge(boolean var1, int var2, boolean var3, boolean var4, boolean var5) throws EngineException, IOException;

    public abstract IndexCommit acquireIndexCommit(boolean var1) throws EngineException;

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    public void failEngine(String reason, @Nullable Exception failure) {
        if (this.failEngineLock.tryLock()) {
            this.store.incRef();
            try {
                if (this.failedEngine.get() != null) {
                    this.logger.warn(() -> new ParameterizedMessage("tried to fail engine but engine is already failed. ignoring. [{}]", (Object)reason), (Throwable)failure);
                    return;
                }
                this.failedEngine.set((Object)(failure != null ? failure : new IllegalStateException(reason)));
                try {
                    this.closeNoLock("engine failed on: [" + reason + "]");
                }
                finally {
                    this.logger.warn(() -> new ParameterizedMessage("failed engine [{}]", (Object)reason), (Throwable)failure);
                    if (Lucene.isCorruptionException(failure)) {
                        try {
                            this.store.markStoreCorrupted(new IOException("failed engine (reason: [" + reason + "])", ExceptionsHelper.unwrapCorruption(failure)));
                        }
                        catch (IOException e) {
                            this.logger.warn("Couldn't mark store corrupted", (Throwable)e);
                        }
                    }
                    this.eventListener.onFailedEngine(reason, failure);
                }
            }
            catch (Exception inner) {
                if (failure != null) {
                    inner.addSuppressed(failure);
                }
                this.logger.warn("failEngine threw exception", (Throwable)inner);
            }
            finally {
                this.store.decRef();
            }
        } else {
            this.logger.debug(() -> new ParameterizedMessage("tried to fail engine but could not acquire lock - engine should be failed by now [{}]", (Object)reason), (Throwable)failure);
        }
    }

    protected boolean maybeFailEngine(String source, Exception e) {
        if (Lucene.isCorruptionException(e)) {
            this.failEngine("corrupt file (source: [" + source + "])", e);
            return true;
        }
        return false;
    }

    protected abstract SearcherManager getSearcherManager();

    protected abstract void closeNoLock(String var1);

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    public void flushAndClose() throws IOException {
        if (!this.isClosed.get()) {
            this.logger.trace("flushAndClose now acquire writeLock");
            try (ReleasableLock lock = this.writeLock.acquire();){
                this.logger.trace("flushAndClose now acquired writeLock");
                try {
                    this.logger.debug("flushing shard on close - this might take some time to sync files to disk");
                    try {
                        this.flush();
                    }
                    catch (AlreadyClosedException ex) {
                        this.logger.debug("engine already closed - skipping flushAndClose");
                    }
                }
                finally {
                    this.close();
                }
            }
        }
    }

    @Override
    public void close() throws IOException {
        if (!this.isClosed.get()) {
            this.logger.debug("close now acquiring writeLock");
            try (ReleasableLock lock = this.writeLock.acquire();){
                this.logger.debug("close acquired writeLock");
                this.closeNoLock("api");
            }
        }
    }

    public void onSettingsChanged() {
    }

    public long getLastWriteNanos() {
        return this.lastWriteNanos;
    }

    public abstract void activateThrottling();

    public abstract void deactivateThrottling();

    public abstract Engine recoverFromTranslog() throws IOException;

    public boolean isRecovering() {
        return false;
    }

    public static interface Warmer {
        public void warm(Searcher var1);
    }

    public static class CommitId
    implements Writeable {
        private final byte[] id;

        public CommitId(byte[] id) {
            assert (id != null);
            this.id = Arrays.copyOf(id, id.length);
        }

        public CommitId(StreamInput in) throws IOException {
            assert (in != null);
            this.id = in.readByteArray();
        }

        @Override
        public void writeTo(StreamOutput out) throws IOException {
            out.writeByteArray(this.id);
        }

        public String toString() {
            return Base64.getEncoder().encodeToString(this.id);
        }

        public boolean idsEqual(byte[] id) {
            return Arrays.equals(id, this.id);
        }

        public boolean equals(Object o) {
            if (this == o) {
                return true;
            }
            if (o == null || this.getClass() != o.getClass()) {
                return false;
            }
            CommitId commitId = (CommitId)o;
            return Arrays.equals(this.id, commitId.id);
        }

        public int hashCode() {
            return Arrays.hashCode(this.id);
        }
    }

    public static class GetResult
    implements Releasable {
        private final boolean exists;
        private final long version;
        private final VersionsResolver.DocIdAndVersion docIdAndVersion;
        private final Searcher searcher;
        public static final GetResult NOT_EXISTS = new GetResult(false, -1L, null, null);

        private GetResult(boolean exists, long version, VersionsResolver.DocIdAndVersion docIdAndVersion, Searcher searcher) {
            this.exists = exists;
            this.version = version;
            this.docIdAndVersion = docIdAndVersion;
            this.searcher = searcher;
        }

        public GetResult(Searcher searcher, VersionsResolver.DocIdAndVersion docIdAndVersion) {
            this(true, docIdAndVersion.version, docIdAndVersion, searcher);
        }

        public boolean exists() {
            return this.exists;
        }

        public long version() {
            return this.version;
        }

        public Searcher searcher() {
            return this.searcher;
        }

        public VersionsResolver.DocIdAndVersion docIdAndVersion() {
            return this.docIdAndVersion;
        }

        @Override
        public void close() {
            this.release();
        }

        public void release() {
            Releasables.close(this.searcher);
        }
    }

    public static class Get {
        private final boolean realtime;
        private final Term uid;
        private long version = -3L;
        private VersionType versionType = VersionType.INTERNAL;

        public Get(boolean realtime, Term uid) {
            this.realtime = realtime;
            this.uid = uid;
        }

        public boolean realtime() {
            return this.realtime;
        }

        public Term uid() {
            return this.uid;
        }

        public long version() {
            return this.version;
        }

        public Get version(long version) {
            this.version = version;
            return this;
        }

        public VersionType versionType() {
            return this.versionType;
        }

        public Get versionType(VersionType versionType) {
            this.versionType = versionType;
            return this;
        }
    }

    public static class Delete
    extends Operation {
        private final String type;
        private final String id;

        public Delete(String type, String id, Term uid, long version, VersionType versionType, Operation.Origin origin, long startTime) {
            super(uid, version, versionType, origin, startTime);
            this.type = type;
            this.id = id;
        }

        public Delete(String type, String id, Term uid) {
            this(type, id, uid, -3L, VersionType.INTERNAL, Operation.Origin.PRIMARY, System.nanoTime());
        }

        public Delete(Delete template, VersionType versionType) {
            this(template.type(), template.id(), template.uid(), template.version(), versionType, template.origin(), template.startTime());
        }

        @Override
        public String type() {
            return this.type;
        }

        @Override
        public String id() {
            return this.id;
        }

        @Override
        Operation.TYPE operationType() {
            return Operation.TYPE.DELETE;
        }

        @Override
        public int estimatedSizeInBytes() {
            return (this.uid().field().length() + this.uid().text().length()) * 2 + 20;
        }
    }

    public static class Index
    extends Operation {
        private final ParsedDocument doc;
        private final long autoGeneratedIdTimestamp;
        private final boolean isRetry;

        public Index(Term uid, ParsedDocument doc, long version, VersionType versionType, Operation.Origin origin, long startTime, long autoGeneratedIdTimestamp, boolean isRetry) {
            super(uid, version, versionType, origin, startTime);
            assert (uid.bytes().equals((Object)doc.uid())) : "term uid " + uid + " doesn't match doc uid " + doc.uid();
            this.doc = doc;
            this.isRetry = isRetry;
            this.autoGeneratedIdTimestamp = autoGeneratedIdTimestamp;
        }

        public Index(Term uid, ParsedDocument doc) {
            this(uid, doc, -3L);
        }

        Index(Term uid, ParsedDocument doc, long version) {
            this(uid, doc, version, VersionType.INTERNAL, Operation.Origin.PRIMARY, System.nanoTime(), -1L, false);
        }

        public ParsedDocument parsedDoc() {
            return this.doc;
        }

        @Override
        public String type() {
            return this.doc.type();
        }

        @Override
        public String id() {
            return this.doc.id();
        }

        @Override
        Operation.TYPE operationType() {
            return Operation.TYPE.INDEX;
        }

        public String routing() {
            return this.doc.routing();
        }

        public long timestamp() {
            return this.doc.timestamp();
        }

        public long ttl() {
            return this.doc.ttl();
        }

        public String parent() {
            return this.doc.parent();
        }

        public List<ParseContext.Document> docs() {
            return this.doc.docs();
        }

        public BytesReference source() {
            return this.doc.source();
        }

        @Override
        public int estimatedSizeInBytes() {
            return (this.id().length() + this.type().length()) * 2 + this.source().length() + 12;
        }

        public long getAutoGeneratedIdTimestamp() {
            return this.autoGeneratedIdTimestamp;
        }

        public boolean isRetry() {
            return this.isRetry;
        }
    }

    public static abstract class Operation {
        private final Term uid;
        private final long version;
        private final VersionType versionType;
        private final Origin origin;
        private final long startTime;

        public Operation(Term uid, long version, VersionType versionType, Origin origin, long startTime) {
            this.uid = uid;
            this.version = version;
            this.versionType = versionType;
            this.origin = origin;
            this.startTime = startTime;
        }

        public Origin origin() {
            return this.origin;
        }

        public Term uid() {
            return this.uid;
        }

        public long version() {
            return this.version;
        }

        public abstract int estimatedSizeInBytes();

        public VersionType versionType() {
            return this.versionType;
        }

        public long startTime() {
            return this.startTime;
        }

        public abstract String type();

        abstract String id();

        abstract TYPE operationType();

        public static enum Origin {
            PRIMARY,
            REPLICA,
            PEER_RECOVERY,
            LOCAL_TRANSLOG_RECOVERY;


            public boolean isRecovery() {
                return this == PEER_RECOVERY || this == LOCAL_TRANSLOG_RECOVERY;
            }
        }

        public static enum TYPE {
            INDEX,
            DELETE;

            private final String lowercase = this.toString().toLowerCase(Locale.ROOT);

            public String getLowercase() {
                return this.lowercase;
            }
        }
    }

    public static class Searcher
    implements Releasable {
        private final String source;
        private final IndexSearcher searcher;

        public Searcher(String source, IndexSearcher searcher) {
            this.source = source;
            this.searcher = searcher;
        }

        public String source() {
            return this.source;
        }

        public IndexReader reader() {
            return this.searcher.getIndexReader();
        }

        public DirectoryReader getDirectoryReader() {
            if (this.reader() instanceof DirectoryReader) {
                return (DirectoryReader)this.reader();
            }
            throw new IllegalStateException("Can't use " + this.reader().getClass() + " as a directory reader");
        }

        public IndexSearcher searcher() {
            return this.searcher;
        }

        @Override
        public void close() {
        }
    }

    public static interface EventListener {
        default public void onFailedEngine(String reason, @Nullable Exception e) {
        }
    }

    public static enum SyncedFlushResult {
        SUCCESS,
        COMMIT_MISMATCH,
        PENDING_OPERATIONS;

    }

    public static class DeleteResult
    extends Result {
        private final boolean found;

        public DeleteResult(long version, boolean found) {
            super(Operation.TYPE.DELETE, version);
            this.found = found;
        }

        public DeleteResult(Exception failure, long version, boolean found) {
            super(Operation.TYPE.DELETE, failure, version);
            this.found = found;
        }

        public boolean isFound() {
            return this.found;
        }
    }

    public static class IndexResult
    extends Result {
        private final boolean created;

        public IndexResult(long version, boolean created) {
            super(Operation.TYPE.INDEX, version);
            this.created = created;
        }

        public IndexResult(Exception failure, long version) {
            super(Operation.TYPE.INDEX, failure, version);
            this.created = false;
        }

        public boolean isCreated() {
            return this.created;
        }
    }

    public static abstract class Result {
        private final Operation.TYPE operationType;
        private final long version;
        private final Exception failure;
        private final SetOnce<Boolean> freeze = new SetOnce();
        private Translog.Location translogLocation;
        private long took;

        protected Result(Operation.TYPE operationType, Exception failure, long version) {
            this.operationType = operationType;
            this.failure = failure;
            this.version = version;
        }

        protected Result(Operation.TYPE operationType, long version) {
            this(operationType, null, version);
        }

        public boolean hasFailure() {
            return this.failure != null;
        }

        public long getVersion() {
            return this.version;
        }

        public Translog.Location getTranslogLocation() {
            return this.translogLocation;
        }

        public Exception getFailure() {
            return this.failure;
        }

        public long getTook() {
            return this.took;
        }

        public Operation.TYPE getOperationType() {
            return this.operationType;
        }

        void setTranslogLocation(Translog.Location translogLocation) {
            if (this.freeze.get() == null) {
                assert (this.failure == null) : "failure has to be null to set translog location";
            } else {
                throw new IllegalStateException("result is already frozen");
            }
            this.translogLocation = translogLocation;
        }

        void setTook(long took) {
            if (this.freeze.get() != null) {
                throw new IllegalStateException("result is already frozen");
            }
            this.took = took;
        }

        void freeze() {
            this.freeze.set((Object)true);
        }
    }

    protected static final class NoOpLock
    implements Lock {
        protected NoOpLock() {
        }

        @Override
        public void lock() {
        }

        @Override
        public void lockInterruptibly() throws InterruptedException {
        }

        @Override
        public boolean tryLock() {
            return true;
        }

        @Override
        public boolean tryLock(long time, TimeUnit unit) throws InterruptedException {
            return true;
        }

        @Override
        public void unlock() {
        }

        @Override
        public Condition newCondition() {
            throw new UnsupportedOperationException("NoOpLock can't provide a condition");
        }
    }

    protected static final class IndexThrottle {
        private final CounterMetric throttleTimeMillisMetric = new CounterMetric();
        private volatile long startOfThrottleNS;
        private static final ReleasableLock NOOP_LOCK = new ReleasableLock(new NoOpLock());
        private final ReleasableLock lockReference = new ReleasableLock(new ReentrantLock());
        private volatile ReleasableLock lock = NOOP_LOCK;

        protected IndexThrottle() {
        }

        public Releasable acquireThrottle() {
            return this.lock.acquire();
        }

        public void activate() {
            assert (this.lock == NOOP_LOCK) : "throttling activated while already active";
            this.startOfThrottleNS = System.nanoTime();
            this.lock = this.lockReference;
        }

        public void deactivate() {
            assert (this.lock != NOOP_LOCK) : "throttling deactivated but not active";
            this.lock = NOOP_LOCK;
            assert (this.startOfThrottleNS > 0L) : "Bad state of startOfThrottleNS";
            long throttleTimeNS = System.nanoTime() - this.startOfThrottleNS;
            if (throttleTimeNS >= 0L) {
                this.throttleTimeMillisMetric.inc(TimeValue.nsecToMSec(throttleTimeNS));
            }
        }

        long getThrottleTimeInMillis() {
            long currentThrottleNS = 0L;
            if (this.isThrottled() && this.startOfThrottleNS != 0L && (currentThrottleNS += System.nanoTime() - this.startOfThrottleNS) < 0L) {
                currentThrottleNS = 0L;
            }
            return this.throttleTimeMillisMetric.count() + TimeValue.nsecToMSec(currentThrottleNS);
        }

        boolean isThrottled() {
            return this.lock != NOOP_LOCK;
        }
    }
}

