package vtk;

import java.io.File;

/**
 * Enum used to load native library more easily. If you don't want to set the
 * specific environment variable you can provide the path of the directory that
 * contains the VTK library through a Java Property. Like in the following
 * command line:
 *
 * > java -cp vtk.jar -Dvtk.lib.dir=.../vtk-libs vtk.sample.SimpleVTK
 *
 * The directory .../vtk-libs must contain the so/dll/dylib + the jnilib files
 *
 * @author sebastien jourdain - sebastien.jourdain@kitware.com
 */
public enum vtkNativeLibrary {

    VTKINTERACTIONWIDGETS("vtkInteractionWidgetsJava", true),
    VTKINTERACTIONIMAGE("vtkInteractionImageJava", true),
    VTKINTERACTIONSTYLE("vtkInteractionStyleJava", true),
    VTKTESTINGRENDERING("vtkTestingRenderingJava", true),
    VTKVIEWSCONTEXT2D("vtkViewsContext2DJava", true),
    VTKVIEWSINFOVIS("vtkViewsInfovisJava", true),
    VTKVIEWSGEOVIS("vtkViewsGeovisJava", true),
    VTKVIEWSCORE("vtkViewsCoreJava", true),
    VTKINFOVISPARALLEL("vtkInfovisParallelJava", false),
    VTKINFOVISLAYOUT("vtkInfovisLayoutJava", true),
    VTKINFOVISBOOSTGRAPHALGORITHMS("vtkInfovisBoostGraphAlgorithmsJava", true),
    VTKINFOVISCORE("vtkInfovisCoreJava", true),
    VTKIMAGINGHYBRID("vtkImagingHybridJava", true),
    VTKIMAGINGSTATISTICS("vtkImagingStatisticsJava", true),
    VTKIMAGINGMATH("vtkImagingMathJava", true),
    VTKIMAGINGMORPHOLOGICAL("vtkImagingMorphologicalJava", true),
    VTKIMAGINGCOLOR("vtkImagingColorJava", true),
    VTKIMAGINGSOURCES("vtkImagingSourcesJava", true),
    VTKIMAGINGGENERAL("vtkImagingGeneralJava", true),
    VTKIMAGINGFOURIER("vtkImagingFourierJava", true),
    VTKIMAGINGSTENCIL("vtkImagingStencilJava", true),
    VTKIMAGINGCORE("vtkImagingCoreJava", true),
    VTKDOMAINSCHEMISTRY("vtkDomainsChemistryJava", true),
    VTKDOMAINSCHEMISTRYOPENGL2("vtkDomainsChemistryOpenGL2Java", false),
    VTKWEBGLEXPORTER("vtkWebGLExporterJava", false),
    VTKWEBCORE("vtkWebCoreJava", false),
    VTKRENDERINGCONTEXT2D("vtkRenderingContext2DJava", true),
    VTKRENDERINGVOLUMEAMR("vtkRenderingVolumeAMRJava", true),
    VTKRENDERINGQT("vtkRenderingQtJava", true),
    VTKRENDERINGFREETYPE("vtkRenderingFreeTypeJava", true),
    VTKRENDERINGCONTEXTOPENGL2("vtkRenderingContextOpenGL2Java", false),
    VTKRENDERINGLIC("vtkRenderingLICJava", true),
    VTKRENDERINGLOD("vtkRenderingLODJava", true),
    VTKRENDERINGCONTEXTOPENGL("vtkRenderingContextOpenGLJava", true),
    VTKRENDERINGIMAGE("vtkRenderingImageJava", true),
    VTKRENDERINGEXTERNAL("vtkRenderingExternalJava", false),
    VTKRENDERINGPARALLEL("vtkRenderingParallelJava", true),
    VTKRENDERINGOPENGL2("vtkRenderingOpenGL2Java", false),
    VTKRENDERINGVOLUME("vtkRenderingVolumeJava", true),
    VTKRENDERINGOPENGL("vtkRenderingOpenGLJava", true),
    VTKRENDERINGVOLUMEOPENGL("vtkRenderingVolumeOpenGLJava", true),
    VTKRENDERINGANNOTATION("vtkRenderingAnnotationJava", true),
    VTKRENDERINGGL2PS("vtkRenderingGL2PSJava", true),
    VTKRENDERINGMATPLOTLIB("vtkRenderingMatplotlibJava", true),
    VTKRENDERINGPARALLELLIC("vtkRenderingParallelLICJava", true),
    VTKRENDERINGVOLUMEOPENGL2("vtkRenderingVolumeOpenGL2Java", false),
    VTKRENDERINGCORE("vtkRenderingCoreJava", true),
    VTKRENDERINGLICOPENGL2("vtkRenderingLICOpenGL2Java", false),
    VTKRENDERINGLABEL("vtkRenderingLabelJava", true),
    VTKIOENSIGHT("vtkIOEnSightJava", true),
    VTKIOPARALLELEXODUS("vtkIOParallelExodusJava", true),
    VTKIOVPIC("vtkIOVPICJava", true),
    VTKIOPARALLELNETCDF("vtkIOParallelNetCDFJava", true),
    VTKIOINFOVIS("vtkIOInfovisJava", true),
    VTKIOLSDYNA("vtkIOLSDynaJava", true),
    VTKIOPARALLELXML("vtkIOParallelXMLJava", true),
    VTKIOGDAL("vtkIOGDALJava", true),
    VTKIOXML("vtkIOXMLJava", true),
    VTKIOXDMF2("vtkIOXdmf2Java", true),
    VTKIOMPIPARALLEL("vtkIOMPIParallelJava", true),
    VTKIOGEOJSON("vtkIOGeoJSONJava", true),
    VTKIOLEGACY("vtkIOLegacyJava", true),
    VTKIOMPIIMAGE("vtkIOMPIImageJava", true),
    VTKIOIMAGE("vtkIOImageJava", true),
    VTKIOMINC("vtkIOMINCJava", true),
    VTKIOPARALLEL("vtkIOParallelJava", true),
    VTKIOGEOMETRY("vtkIOGeometryJava", true),
    VTKIOXDMF3("vtkIOXdmf3Java", true),
    VTKIOVIDEO("vtkIOVideoJava", true),
    VTKIOFFMPEG("vtkIOFFMPEGJava", true),
    VTKIOODBC("vtkIOODBCJava", true),
    VTKIOXMLPARSER("vtkIOXMLParserJava", true),
    VTKIOIMPORT("vtkIOImportJava", true),
    VTKIOAMR("vtkIOAMRJava", true),
    VTKIOEXODUS("vtkIOExodusJava", true),
    VTKIOSQL("vtkIOSQLJava", true),
    VTKIOPARALLELLSDYNA("vtkIOParallelLSDynaJava", true),
    VTKIOADIOS("vtkIOADIOSJava", false),
    VTKIOPOSTGRESQL("vtkIOPostgreSQLJava", true),
    VTKIOMOVIE("vtkIOMovieJava", true),
    VTKIONETCDF("vtkIONetCDFJava", true),
    VTKIOMYSQL("vtkIOMySQLJava", true),
    VTKIOPLY("vtkIOPLYJava", true),
    VTKIOEXPORT("vtkIOExportJava", true),
    VTKIOCORE("vtkIOCoreJava", true),
    VTKPARALLELMPI4PY("vtkParallelMPI4PyJava", true),
    VTKPARALLELMPI("vtkParallelMPIJava", true),
    VTKPARALLELCORE("vtkParallelCoreJava", true),
    VTKCOMMONEXECUTIONMODEL("vtkCommonExecutionModelJava", true),
    VTKCOMMONMATH("vtkCommonMathJava", true),
    VTKCOMMONCOLOR("vtkCommonColorJava", true),
    VTKCOMMONCOMPUTATIONALGEOMETRY("vtkCommonComputationalGeometryJava", true),
    VTKCOMMONSYSTEM("vtkCommonSystemJava", true),
    VTKCOMMONMISC("vtkCommonMiscJava", true),
    VTKCOMMONTRANSFORMS("vtkCommonTransformsJava", true),
    VTKCOMMONDATAMODEL("vtkCommonDataModelJava", true),
    VTKCOMMONCORE("vtkCommonCoreJava", true),
    VTKCHARTSCORE("vtkChartsCoreJava", true),
    VTKPYTHONINTERPRETER("vtkPythonInterpreterJava", true),
    VTKGEOVISCORE("vtkGeovisCoreJava", true),
    VTKFILTERSPARALLELFLOWPATHS("vtkFiltersParallelFlowPathsJava", true),
    VTKFILTERSPARALLELGEOMETRY("vtkFiltersParallelGeometryJava", true),
    VTKFILTERSHYBRID("vtkFiltersHybridJava", true),
    VTKFILTERSSELECTION("vtkFiltersSelectionJava", true),
    VTKFILTERSVERDICT("vtkFiltersVerdictJava", true),
    VTKFILTERSSTATISTICS("vtkFiltersStatisticsJava", true),
    VTKFILTERSIMAGING("vtkFiltersImagingJava", true),
    VTKFILTERSPARALLELIMAGING("vtkFiltersParallelImagingJava", true),
    VTKFILTERSSOURCES("vtkFiltersSourcesJava", true),
    VTKFILTERSPROGRAMMABLE("vtkFiltersProgrammableJava", true),
    VTKFILTERSPARALLEL("vtkFiltersParallelJava", true),
    VTKFILTERSMODELING("vtkFiltersModelingJava", true),
    VTKFILTERSGENERAL("vtkFiltersGeneralJava", true),
    VTKFILTERSHYPERTREE("vtkFiltersHyperTreeJava", true),
    VTKFILTERSGEOMETRY("vtkFiltersGeometryJava", true),
    VTKFILTERSSTATISTICSGNUR("vtkFiltersStatisticsGnuRJava", false),
    VTKFILTERSTEXTURE("vtkFiltersTextureJava", true),
    VTKFILTERSREEBGRAPH("vtkFiltersReebGraphJava", true),
    VTKFILTERSPARALLELMPI("vtkFiltersParallelMPIJava", true),
    VTKFILTERSAMR("vtkFiltersAMRJava", true),
    VTKFILTERSSMP("vtkFiltersSMPJava", true),
    VTKFILTERSPYTHON("vtkFiltersPythonJava", true),
    VTKFILTERSGENERIC("vtkFiltersGenericJava", true),
    VTKFILTERSEXTRACTION("vtkFiltersExtractionJava", true),
    VTKFILTERSFLOWPATHS("vtkFiltersFlowPathsJava", true),
    VTKFILTERSPARALLELSTATISTICS("vtkFiltersParallelStatisticsJava", true),
    VTKFILTERSCORE("vtkFiltersCoreJava", true),
    VTKACCELERATORSDAX("vtkAcceleratorsDaxJava", false),
    VTKACCELERATORSPISTON("vtkAcceleratorsPistonJava", false);


  /**
   * Try to load all library
   *
   * @return true if all library have been successfully loaded
   */
  public static boolean LoadAllNativeLibraries() {
    boolean isEveryThingLoaded = true;
    for (vtkNativeLibrary lib : values()) {
      try {
        if(lib.IsBuilt()) {
          lib.LoadLibrary();
        }
      } catch (UnsatisfiedLinkError e) {
        isEveryThingLoaded = false;
        e.printStackTrace();
      }
    }

    return isEveryThingLoaded;
  }

  /**
   * Load the set of given library and trows runtime exception if any given
   * library failed in the loading process
   *
   * @param nativeLibraries
   */
  public static void LoadNativeLibraries(vtkNativeLibrary... nativeLibraries) {
    for (vtkNativeLibrary lib : nativeLibraries) {
      lib.LoadLibrary();
    }
  }

  /**
   * Disable the pop-in vtkErrorWindow by writing the error to a log file.
   * If the provided logFile is null the default "vtkError.txt" file will be
   * used.
   *
   * @param logFile
   */
  public static void DisableOutputWindow(File logFile) {
    if(logFile == null) {
      logFile = new File("vtkError.txt");
    }
    vtkFileOutputWindow outputError = new vtkFileOutputWindow();
    outputError.SetFileName(logFile.getAbsolutePath());
    outputError.SetInstance(outputError);
  }

  private vtkNativeLibrary(String nativeLibraryName, boolean built) {
    this.nativeLibraryName = nativeLibraryName;
    this.loaded = false;
    this.built = built;
  }

  /**
   * Load the library and throws runtime exception if the library failed in
   * the loading process
   */
  public void LoadLibrary() throws UnsatisfiedLinkError {
    if (!loaded) {
      if (System.getProperty("vtk.lib.dir") != null) {
        File dir = new File(System.getProperty("vtk.lib.dir"));
        patchJavaLibraryPath(dir.getAbsolutePath());
        File libPath = new File(dir, System.mapLibraryName(nativeLibraryName));
        if (libPath.exists()) {
          try {
            Runtime.getRuntime().load(libPath.getAbsolutePath());
            loaded = true;
            return;
          } catch (UnsatisfiedLinkError e) {
            e.printStackTrace();
          }
        }
      }
      System.loadLibrary(nativeLibraryName);
    }
    loaded = true;
  }

  /**
   * @return true if the library has already been successfully loaded
   */
  public boolean IsLoaded() {
    return loaded;
  }

  /**
   * @return true if the module was enabled and therefore build
   */
  public boolean IsBuilt() {
    return built;
  }

  /**
   * @return the library name
   */
  public String GetLibraryName() {
    return nativeLibraryName;
  }

  private static void patchJavaLibraryPath(String vtkLibDir) {
    if (vtkLibDir != null) {
      String path_separator = System.getProperty("path.separator");
      String s = System.getProperty("java.library.path");
      if (!s.contains(vtkLibDir)) {
        s = s + path_separator + vtkLibDir;
        System.setProperty("java.library.path", s);
      }
    }
  }

  private String nativeLibraryName;
  private boolean loaded;
  private boolean built;
}
