/*
 * Copyright 2017 the original author or authors.
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */

package org.gradle.api.internal.changedetection.state;

import com.google.common.base.Objects;
import com.google.common.hash.HashCode;
import org.gradle.api.UncheckedIOException;
import org.gradle.caching.internal.BuildCacheHasher;
import org.gradle.internal.io.ClassLoaderObjectInputStream;

import java.io.ByteArrayInputStream;
import java.util.Arrays;

/**
 * An immutable snapshot of the state of some value.
 */
public class SerializedValueSnapshot implements ValueSnapshot {
    private final HashCode implementationHash;
    private final byte[] serializedValue;

    public SerializedValueSnapshot(HashCode implementationHash, byte[] serializedValue) {
        this.implementationHash = implementationHash;
        this.serializedValue = serializedValue;
    }

    public HashCode getImplementationHash() {
        return implementationHash;
    }

    public byte[] getValue() {
        return serializedValue;
    }

    @Override
    public ValueSnapshot snapshot(Object value, ValueSnapshotter snapshotter) {
        ValueSnapshot snapshot = snapshotter.snapshot(value);

        if (snapshot instanceof SerializedValueSnapshot) {
            SerializedValueSnapshot newSnapshot = (SerializedValueSnapshot) snapshot;
            if (!Objects.equal(implementationHash, newSnapshot.implementationHash)) {
                // Different implementation - assume value has changed
                return newSnapshot;
            }
            if (Arrays.equals(serializedValue, newSnapshot.serializedValue)) {
                // Same serialized content - value has not changed
                return this;
            }

            // Deserialize the old value and use the equals() implementation. This will be removed at some point
            Object oldValue;
            try {
                oldValue = new ClassLoaderObjectInputStream(new ByteArrayInputStream(serializedValue), value.getClass().getClassLoader()).readObject();
            } catch (Exception e) {
                throw new UncheckedIOException(e);
            }
            if (oldValue.equals(value)) {
                // Same value
                return this;
            }
        }

        return snapshot;
    }

    @Override
    public void appendToHasher(BuildCacheHasher hasher) {
        if (implementationHash == null) {
            hasher.putNull();
        } else {
            hasher.putBytes(implementationHash.asBytes());
        }
        hasher.putBytes(serializedValue);
    }

    @Override
    public boolean equals(Object obj) {
        if (obj == this) {
            return true;
        }
        if (obj == null || obj.getClass() != getClass()) {
            return false;
        }
        SerializedValueSnapshot other = (SerializedValueSnapshot) obj;
        return Objects.equal(implementationHash, other.implementationHash) && Arrays.equals(serializedValue, other.serializedValue);
    }

    @Override
    public int hashCode() {
        return Arrays.hashCode(serializedValue);
    }
}
