/* 
   Unix SMB/CIFS implementation.
   SMB parameters and setup, plus a whole lot more.
   
   Copyright (C) Andrew Tridgell              2001
   
   This program is free software; you can redistribute it and/or modify
   it under the terms of the GNU General Public License as published by
   the Free Software Foundation; either version 3 of the License, or
   (at your option) any later version.
   
   This program is distributed in the hope that it will be useful,
   but WITHOUT ANY WARRANTY; without even the implied warranty of
   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
   GNU General Public License for more details.
   
   You should have received a copy of the GNU General Public License
   along with this program.  If not, see <http://www.gnu.org/licenses/>.
*/

#ifndef _WERROR_H_
#define _WERROR_H_

#include <stdint.h>

/* the following rather strange looking definitions of NTSTATUS and WERROR
   and there in order to catch common coding errors where different error types
   are mixed up. This is especially important as we slowly convert Samba
   from using bool for internal functions 
*/

#if defined(HAVE_IMMEDIATE_STRUCTURES)
typedef struct {uint32_t w;} WERROR;
#define W_ERROR(x) ((WERROR) { x })
#define W_ERROR_V(x) ((x).w)
#else
typedef uint32_t WERROR;
#define W_ERROR(x) (x)
#define W_ERROR_V(x) (x)
#endif

#define W_ERROR_IS_OK(x) (W_ERROR_V(x) == 0)
#define W_ERROR_EQUAL(x,y) (W_ERROR_V(x) == W_ERROR_V(y))

#define W_ERROR_HAVE_NO_MEMORY(x) do { \
	if (!(x)) {\
		return WERR_NOT_ENOUGH_MEMORY;\
	}\
} while (0)

#define W_ERROR_HAVE_NO_MEMORY_AND_FREE(x, ctx) do { \
	if (!(x)) {\
		talloc_free(ctx); \
		return WERR_NOT_ENOUGH_MEMORY;\
	}\
} while (0)

#define W_ERROR_IS_OK_RETURN(x) do { \
	if (W_ERROR_IS_OK(x)) {\
		return x;\
	}\
} while (0)

#define W_ERROR_NOT_OK_RETURN(x) do { \
	if (!W_ERROR_IS_OK(x)) {\
		return x;\
	}\
} while (0)

#define W_ERROR_NOT_OK_GOTO_DONE(x) do { \
	if (!W_ERROR_IS_OK(x)) {\
		goto done;\
	}\
} while (0)

#define W_ERROR_NOT_OK_GOTO(x, y) do {\
	if (!W_ERROR_IS_OK(x)) {\
		goto y;\
	}\
} while(0)

/* these are win32 error codes. There are only a few places where
   these matter for Samba, primarily in the NT printing code */
#define WERR_OK W_ERROR(0x00000000)
#define WERR_STATUS_MORE_ENTRIES W_ERROR(0x00000105)

/* Configuration Manager Errors */
/* Basically Win32 errors meanings are specific to the \ntsvcs pipe */
#define WERR_CM_INVALID_POINTER W_ERROR(3)
#define WERR_CM_BUFFER_SMALL W_ERROR(26)
#define WERR_CM_NO_MORE_HW_PROFILES W_ERROR(35)
#define WERR_CM_NO_SUCH_VALUE W_ERROR(37)

/* DFS errors */

#ifndef NERR_BASE
#define NERR_BASE (2100)
#endif

#ifndef MAX_NERR
#define MAX_NERR (NERR_BASE+899)
#endif

/* Generic error code aliases */
#define WERR_FOOBAR WERR_GEN_FAILURE

/*****************************************************************************
 Auto-generated Win32 error from:
 http://msdn.microsoft.com/en-us/library/cc231199%28PROT.10%29.aspx
 *****************************************************************************/
/* BEGIN GENERATED-WIN32-ERROR-CODES */
#define WERR_NERR_SUCCESS	W_ERROR(0x00000000)
#define WERR_INVALID_FUNCTION	W_ERROR(0x00000001)
#define WERR_FILE_NOT_FOUND	W_ERROR(0x00000002)
#define WERR_PATH_NOT_FOUND	W_ERROR(0x00000003)
#define WERR_TOO_MANY_OPEN_FILES	W_ERROR(0x00000004)
#define WERR_ACCESS_DENIED	W_ERROR(0x00000005)
#define WERR_INVALID_HANDLE	W_ERROR(0x00000006)
#define WERR_ARENA_TRASHED	W_ERROR(0x00000007)
#define WERR_NOT_ENOUGH_MEMORY	W_ERROR(0x00000008)
#define WERR_INVALID_BLOCK	W_ERROR(0x00000009)
#define WERR_BAD_ENVIRONMENT	W_ERROR(0x0000000A)
#define WERR_BAD_FORMAT	W_ERROR(0x0000000B)
#define WERR_INVALID_ACCESS	W_ERROR(0x0000000C)
#define WERR_INVALID_DATA	W_ERROR(0x0000000D)
#define WERR_OUTOFMEMORY	W_ERROR(0x0000000E)
#define WERR_INVALID_DRIVE	W_ERROR(0x0000000F)
#define WERR_CURRENT_DIRECTORY	W_ERROR(0x00000010)
#define WERR_NOT_SAME_DEVICE	W_ERROR(0x00000011)
#define WERR_NO_MORE_FILES	W_ERROR(0x00000012)
#define WERR_WRITE_PROTECT	W_ERROR(0x00000013)
#define WERR_BAD_UNIT	W_ERROR(0x00000014)
#define WERR_NOT_READY	W_ERROR(0x00000015)
#define WERR_BAD_COMMAND	W_ERROR(0x00000016)
#define WERR_CRC	W_ERROR(0x00000017)
#define WERR_BAD_LENGTH	W_ERROR(0x00000018)
#define WERR_SEEK	W_ERROR(0x00000019)
#define WERR_NOT_DOS_DISK	W_ERROR(0x0000001A)
#define WERR_SECTOR_NOT_FOUND	W_ERROR(0x0000001B)
#define WERR_OUT_OF_PAPER	W_ERROR(0x0000001C)
#define WERR_WRITE_FAULT	W_ERROR(0x0000001D)
#define WERR_READ_FAULT	W_ERROR(0x0000001E)
#define WERR_GEN_FAILURE	W_ERROR(0x0000001F)
#define WERR_SHARING_VIOLATION	W_ERROR(0x00000020)
#define WERR_LOCK_VIOLATION	W_ERROR(0x00000021)
#define WERR_WRONG_DISK	W_ERROR(0x00000022)
#define WERR_SHARING_BUFFER_EXCEEDED	W_ERROR(0x00000024)
#define WERR_HANDLE_EOF	W_ERROR(0x00000026)
#define WERR_HANDLE_DISK_FULL	W_ERROR(0x00000027)
#define WERR_NOT_SUPPORTED	W_ERROR(0x00000032)
#define WERR_REM_NOT_LIST	W_ERROR(0x00000033)
#define WERR_DUP_NAME	W_ERROR(0x00000034)
#define WERR_BAD_NETPATH	W_ERROR(0x00000035)
#define WERR_NETWORK_BUSY	W_ERROR(0x00000036)
#define WERR_DEV_NOT_EXIST	W_ERROR(0x00000037)
#define WERR_TOO_MANY_CMDS	W_ERROR(0x00000038)
#define WERR_ADAP_HDW_ERR	W_ERROR(0x00000039)
#define WERR_BAD_NET_RESP	W_ERROR(0x0000003A)
#define WERR_UNEXP_NET_ERR	W_ERROR(0x0000003B)
#define WERR_BAD_REM_ADAP	W_ERROR(0x0000003C)
#define WERR_PRINTQ_FULL	W_ERROR(0x0000003D)
#define WERR_NO_SPOOL_SPACE	W_ERROR(0x0000003E)
#define WERR_PRINT_CANCELLED	W_ERROR(0x0000003F)
#define WERR_NETNAME_DELETED	W_ERROR(0x00000040)
#define WERR_NETWORK_ACCESS_DENIED	W_ERROR(0x00000041)
#define WERR_BAD_DEV_TYPE	W_ERROR(0x00000042)
#define WERR_BAD_NET_NAME	W_ERROR(0x00000043)
#define WERR_TOO_MANY_NAMES	W_ERROR(0x00000044)
#define WERR_TOO_MANY_SESS	W_ERROR(0x00000045)
#define WERR_SHARING_PAUSED	W_ERROR(0x00000046)
#define WERR_REQ_NOT_ACCEP	W_ERROR(0x00000047)
#define WERR_REDIR_PAUSED	W_ERROR(0x00000048)
#define WERR_FILE_EXISTS	W_ERROR(0x00000050)
#define WERR_CANNOT_MAKE	W_ERROR(0x00000052)
#define WERR_FAIL_I24	W_ERROR(0x00000053)
#define WERR_OUT_OF_STRUCTURES	W_ERROR(0x00000054)
#define WERR_ALREADY_ASSIGNED	W_ERROR(0x00000055)
#define WERR_INVALID_PASSWORD	W_ERROR(0x00000056)
#define WERR_INVALID_PARAMETER	W_ERROR(0x00000057)
#define WERR_NET_WRITE_FAULT	W_ERROR(0x00000058)
#define WERR_NO_PROC_SLOTS	W_ERROR(0x00000059)
#define WERR_TOO_MANY_SEMAPHORES	W_ERROR(0x00000064)
#define WERR_EXCL_SEM_ALREADY_OWNED	W_ERROR(0x00000065)
#define WERR_SEM_IS_SET	W_ERROR(0x00000066)
#define WERR_TOO_MANY_SEM_REQUESTS	W_ERROR(0x00000067)
#define WERR_INVALID_AT_INTERRUPT_TIME	W_ERROR(0x00000068)
#define WERR_SEM_OWNER_DIED	W_ERROR(0x00000069)
#define WERR_SEM_USER_LIMIT	W_ERROR(0x0000006A)
#define WERR_DISK_CHANGE	W_ERROR(0x0000006B)
#define WERR_DRIVE_LOCKED	W_ERROR(0x0000006C)
#define WERR_BROKEN_PIPE	W_ERROR(0x0000006D)
#define WERR_OPEN_FAILED	W_ERROR(0x0000006E)
#define WERR_BUFFER_OVERFLOW	W_ERROR(0x0000006F)
#define WERR_DISK_FULL	W_ERROR(0x00000070)
#define WERR_NO_MORE_SEARCH_HANDLES	W_ERROR(0x00000071)
#define WERR_INVALID_TARGET_HANDLE	W_ERROR(0x00000072)
#define WERR_INVALID_CATEGORY	W_ERROR(0x00000075)
#define WERR_INVALID_VERIFY_SWITCH	W_ERROR(0x00000076)
#define WERR_BAD_DRIVER_LEVEL	W_ERROR(0x00000077)
#define WERR_CALL_NOT_IMPLEMENTED	W_ERROR(0x00000078)
#define WERR_SEM_TIMEOUT	W_ERROR(0x00000079)
#define WERR_INSUFFICIENT_BUFFER	W_ERROR(0x0000007A)
#define WERR_INVALID_NAME	W_ERROR(0x0000007B)
#define WERR_INVALID_LEVEL	W_ERROR(0x0000007C)
#define WERR_NO_VOLUME_LABEL	W_ERROR(0x0000007D)
#define WERR_MOD_NOT_FOUND	W_ERROR(0x0000007E)
#define WERR_PROC_NOT_FOUND	W_ERROR(0x0000007F)
#define WERR_WAIT_NO_CHILDREN	W_ERROR(0x00000080)
#define WERR_CHILD_NOT_COMPLETE	W_ERROR(0x00000081)
#define WERR_DIRECT_ACCESS_HANDLE	W_ERROR(0x00000082)
#define WERR_NEGATIVE_SEEK	W_ERROR(0x00000083)
#define WERR_SEEK_ON_DEVICE	W_ERROR(0x00000084)
#define WERR_NOT_SUBSTED	W_ERROR(0x00000089)
#define WERR_JOIN_TO_JOIN	W_ERROR(0x0000008A)
#define WERR_SUBST_TO_SUBST	W_ERROR(0x0000008B)
#define WERR_JOIN_TO_SUBST	W_ERROR(0x0000008C)
#define WERR_SAME_DRIVE	W_ERROR(0x0000008F)
#define WERR_DIR_NOT_ROOT	W_ERROR(0x00000090)
#define WERR_DIR_NOT_EMPTY	W_ERROR(0x00000091)
#define WERR_IS_SUBST_PATH	W_ERROR(0x00000092)
#define WERR_IS_JOIN_PATH	W_ERROR(0x00000093)
#define WERR_PATH_BUSY	W_ERROR(0x00000094)
#define WERR_IS_SUBST_TARGET	W_ERROR(0x00000095)
#define WERR_SYSTEM_TRACE	W_ERROR(0x00000096)
#define WERR_INVALID_EVENT_COUNT	W_ERROR(0x00000097)
#define WERR_TOO_MANY_MUXWAITERS	W_ERROR(0x00000098)
#define WERR_INVALID_LIST_FORMAT	W_ERROR(0x00000099)
#define WERR_LABEL_TOO_LONG	W_ERROR(0x0000009A)
#define WERR_TOO_MANY_TCBS	W_ERROR(0x0000009B)
#define WERR_SIGNAL_REFUSED	W_ERROR(0x0000009C)
#define WERR_DISCARDED	W_ERROR(0x0000009D)
#define WERR_NOT_LOCKED	W_ERROR(0x0000009E)
#define WERR_BAD_THREADID_ADDR	W_ERROR(0x0000009F)
#define WERR_BAD_ARGUMENTS	W_ERROR(0x000000A0)
#define WERR_BAD_PATHNAME	W_ERROR(0x000000A1)
#define WERR_SIGNAL_PENDING	W_ERROR(0x000000A2)
#define WERR_MAX_THRDS_REACHED	W_ERROR(0x000000A4)
#define WERR_LOCK_FAILED	W_ERROR(0x000000A7)
#define WERR_BUSY	W_ERROR(0x000000AA)
#define WERR_CANCEL_VIOLATION	W_ERROR(0x000000AD)
#define WERR_ATOMIC_LOCKS_NOT_SUPPORTED	W_ERROR(0x000000AE)
#define WERR_INVALID_SEGMENT_NUMBER	W_ERROR(0x000000B4)
#define WERR_INVALID_ORDINAL	W_ERROR(0x000000B6)
#define WERR_ALREADY_EXISTS	W_ERROR(0x000000B7)
#define WERR_INVALID_FLAG_NUMBER	W_ERROR(0x000000BA)
#define WERR_SEM_NOT_FOUND	W_ERROR(0x000000BB)
#define WERR_INVALID_STARTING_CODESEG	W_ERROR(0x000000BC)
#define WERR_INVALID_STACKSEG	W_ERROR(0x000000BD)
#define WERR_INVALID_MODULETYPE	W_ERROR(0x000000BE)
#define WERR_INVALID_EXE_SIGNATURE	W_ERROR(0x000000BF)
#define WERR_EXE_MARKED_INVALID	W_ERROR(0x000000C0)
#define WERR_BAD_EXE_FORMAT	W_ERROR(0x000000C1)
#define WERR_ITERATED_DATA_EXCEEDS_64K	W_ERROR(0x000000C2)
#define WERR_INVALID_MINALLOCSIZE	W_ERROR(0x000000C3)
#define WERR_DYNLINK_FROM_INVALID_RING	W_ERROR(0x000000C4)
#define WERR_IOPL_NOT_ENABLED	W_ERROR(0x000000C5)
#define WERR_INVALID_SEGDPL	W_ERROR(0x000000C6)
#define WERR_AUTODATASEG_EXCEEDS_64K	W_ERROR(0x000000C7)
#define WERR_RING2SEG_MUST_BE_MOVABLE	W_ERROR(0x000000C8)
#define WERR_RELOC_CHAIN_XEEDS_SEGLIM	W_ERROR(0x000000C9)
#define WERR_INFLOOP_IN_RELOC_CHAIN	W_ERROR(0x000000CA)
#define WERR_ENVVAR_NOT_FOUND	W_ERROR(0x000000CB)
#define WERR_NO_SIGNAL_SENT	W_ERROR(0x000000CD)
#define WERR_FILENAME_EXCED_RANGE	W_ERROR(0x000000CE)
#define WERR_RING2_STACK_IN_USE	W_ERROR(0x000000CF)
#define WERR_META_EXPANSION_TOO_LONG	W_ERROR(0x000000D0)
#define WERR_INVALID_SIGNAL_NUMBER	W_ERROR(0x000000D1)
#define WERR_THREAD_1_INACTIVE	W_ERROR(0x000000D2)
#define WERR_LOCKED	W_ERROR(0x000000D4)
#define WERR_TOO_MANY_MODULES	W_ERROR(0x000000D6)
#define WERR_NESTING_NOT_ALLOWED	W_ERROR(0x000000D7)
#define WERR_EXE_MACHINE_TYPE_MISMATCH	W_ERROR(0x000000D8)
#define WERR_EXE_CANNOT_MODIFY_SIGNED_BINARY	W_ERROR(0x000000D9)
#define WERR_EXE_CANNOT_MODIFY_STRONG_SIGNED_BINARY	W_ERROR(0x000000DA)
#define WERR_FILE_CHECKED_OUT	W_ERROR(0x000000DC)
#define WERR_CHECKOUT_REQUIRED	W_ERROR(0x000000DD)
#define WERR_BAD_FILE_TYPE	W_ERROR(0x000000DE)
#define WERR_FILE_TOO_LARGE	W_ERROR(0x000000DF)
#define WERR_FORMS_AUTH_REQUIRED	W_ERROR(0x000000E0)
#define WERR_VIRUS_INFECTED	W_ERROR(0x000000E1)
#define WERR_VIRUS_DELETED	W_ERROR(0x000000E2)
#define WERR_PIPE_LOCAL	W_ERROR(0x000000E5)
#define WERR_BAD_PIPE	W_ERROR(0x000000E6)
#define WERR_PIPE_BUSY	W_ERROR(0x000000E7)
#define WERR_NO_DATA	W_ERROR(0x000000E8)
#define WERR_PIPE_NOT_CONNECTED	W_ERROR(0x000000E9)
#define WERR_MORE_DATA	W_ERROR(0x000000EA)
#define WERR_VC_DISCONNECTED	W_ERROR(0x000000F0)
#define WERR_INVALID_EA_NAME	W_ERROR(0x000000FE)
#define WERR_EA_LIST_INCONSISTENT	W_ERROR(0x000000FF)
#define WERR_WAIT_TIMEOUT	W_ERROR(0x00000102)
#define WERR_NO_MORE_ITEMS	W_ERROR(0x00000103)
#define WERR_CANNOT_COPY	W_ERROR(0x0000010A)
#define WERR_DIRECTORY	W_ERROR(0x0000010B)
#define WERR_EAS_DIDNT_FIT	W_ERROR(0x00000113)
#define WERR_EA_FILE_CORRUPT	W_ERROR(0x00000114)
#define WERR_EA_TABLE_FULL	W_ERROR(0x00000115)
#define WERR_INVALID_EA_HANDLE	W_ERROR(0x00000116)
#define WERR_EAS_NOT_SUPPORTED	W_ERROR(0x0000011A)
#define WERR_NOT_OWNER	W_ERROR(0x00000120)
#define WERR_TOO_MANY_POSTS	W_ERROR(0x0000012A)
#define WERR_PARTIAL_COPY	W_ERROR(0x0000012B)
#define WERR_OPLOCK_NOT_GRANTED	W_ERROR(0x0000012C)
#define WERR_INVALID_OPLOCK_PROTOCOL	W_ERROR(0x0000012D)
#define WERR_DISK_TOO_FRAGMENTED	W_ERROR(0x0000012E)
#define WERR_DELETE_PENDING	W_ERROR(0x0000012F)
#define WERR_MR_MID_NOT_FOUND	W_ERROR(0x0000013D)
#define WERR_SCOPE_NOT_FOUND	W_ERROR(0x0000013E)
#define WERR_FAIL_NOACTION_REBOOT	W_ERROR(0x0000015E)
#define WERR_FAIL_SHUTDOWN	W_ERROR(0x0000015F)
#define WERR_FAIL_RESTART	W_ERROR(0x00000160)
#define WERR_MAX_SESSIONS_REACHED	W_ERROR(0x00000161)
#define WERR_THREAD_MODE_ALREADY_BACKGROUND	W_ERROR(0x00000190)
#define WERR_THREAD_MODE_NOT_BACKGROUND	W_ERROR(0x00000191)
#define WERR_PROCESS_MODE_ALREADY_BACKGROUND	W_ERROR(0x00000192)
#define WERR_PROCESS_MODE_NOT_BACKGROUND	W_ERROR(0x00000193)
#define WERR_INVALID_ADDRESS	W_ERROR(0x000001E7)
#define WERR_USER_PROFILE_LOAD	W_ERROR(0x000001F4)
#define WERR_ARITHMETIC_OVERFLOW	W_ERROR(0x00000216)
#define WERR_PIPE_CONNECTED	W_ERROR(0x00000217)
#define WERR_PIPE_LISTENING	W_ERROR(0x00000218)
#define WERR_VERIFIER_STOP	W_ERROR(0x00000219)
#define WERR_ABIOS_ERROR	W_ERROR(0x0000021A)
#define WERR_WX86_WARNING	W_ERROR(0x0000021B)
#define WERR_WX86_ERROR	W_ERROR(0x0000021C)
#define WERR_TIMER_NOT_CANCELED	W_ERROR(0x0000021D)
#define WERR_UNWIND	W_ERROR(0x0000021E)
#define WERR_BAD_STACK	W_ERROR(0x0000021F)
#define WERR_INVALID_UNWIND_TARGET	W_ERROR(0x00000220)
#define WERR_INVALID_PORT_ATTRIBUTES	W_ERROR(0x00000221)
#define WERR_PORT_MESSAGE_TOO_LONG	W_ERROR(0x00000222)
#define WERR_INVALID_QUOTA_LOWER	W_ERROR(0x00000223)
#define WERR_DEVICE_ALREADY_ATTACHED	W_ERROR(0x00000224)
#define WERR_INSTRUCTION_MISALIGNMENT	W_ERROR(0x00000225)
#define WERR_PROFILING_NOT_STARTED	W_ERROR(0x00000226)
#define WERR_PROFILING_NOT_STOPPED	W_ERROR(0x00000227)
#define WERR_COULD_NOT_INTERPRET	W_ERROR(0x00000228)
#define WERR_PROFILING_AT_LIMIT	W_ERROR(0x00000229)
#define WERR_CANT_WAIT	W_ERROR(0x0000022A)
#define WERR_CANT_TERMINATE_SELF	W_ERROR(0x0000022B)
#define WERR_UNEXPECTED_MM_CREATE_ERR	W_ERROR(0x0000022C)
#define WERR_UNEXPECTED_MM_MAP_ERROR	W_ERROR(0x0000022D)
#define WERR_UNEXPECTED_MM_EXTEND_ERR	W_ERROR(0x0000022E)
#define WERR_BAD_FUNCTION_TABLE	W_ERROR(0x0000022F)
#define WERR_NO_GUID_TRANSLATION	W_ERROR(0x00000230)
#define WERR_INVALID_LDT_SIZE	W_ERROR(0x00000231)
#define WERR_INVALID_LDT_OFFSET	W_ERROR(0x00000233)
#define WERR_INVALID_LDT_DESCRIPTOR	W_ERROR(0x00000234)
#define WERR_TOO_MANY_THREADS	W_ERROR(0x00000235)
#define WERR_THREAD_NOT_IN_PROCESS	W_ERROR(0x00000236)
#define WERR_PAGEFILE_QUOTA_EXCEEDED	W_ERROR(0x00000237)
#define WERR_LOGON_SERVER_CONFLICT	W_ERROR(0x00000238)
#define WERR_SYNCHRONIZATION_REQUIRED	W_ERROR(0x00000239)
#define WERR_NET_OPEN_FAILED	W_ERROR(0x0000023A)
#define WERR_IO_PRIVILEGE_FAILED	W_ERROR(0x0000023B)
#define WERR_CONTROL_C_EXIT	W_ERROR(0x0000023C)
#define WERR_MISSING_SYSTEMFILE	W_ERROR(0x0000023D)
#define WERR_UNHANDLED_EXCEPTION	W_ERROR(0x0000023E)
#define WERR_APP_INIT_FAILURE	W_ERROR(0x0000023F)
#define WERR_PAGEFILE_CREATE_FAILED	W_ERROR(0x00000240)
#define WERR_INVALID_IMAGE_HASH	W_ERROR(0x00000241)
#define WERR_NO_PAGEFILE	W_ERROR(0x00000242)
#define WERR_ILLEGAL_FLOAT_CONTEXT	W_ERROR(0x00000243)
#define WERR_NO_EVENT_PAIR	W_ERROR(0x00000244)
#define WERR_DOMAIN_CTRLR_CONFIG_ERROR	W_ERROR(0x00000245)
#define WERR_ILLEGAL_CHARACTER	W_ERROR(0x00000246)
#define WERR_UNDEFINED_CHARACTER	W_ERROR(0x00000247)
#define WERR_FLOPPY_VOLUME	W_ERROR(0x00000248)
#define WERR_BIOS_FAILED_TO_CONNECT_INTERRUPT	W_ERROR(0x00000249)
#define WERR_BACKUP_CONTROLLER	W_ERROR(0x0000024A)
#define WERR_MUTANT_LIMIT_EXCEEDED	W_ERROR(0x0000024B)
#define WERR_FS_DRIVER_REQUIRED	W_ERROR(0x0000024C)
#define WERR_CANNOT_LOAD_REGISTRY_FILE	W_ERROR(0x0000024D)
#define WERR_DEBUG_ATTACH_FAILED	W_ERROR(0x0000024E)
#define WERR_SYSTEM_PROCESS_TERMINATED	W_ERROR(0x0000024F)
#define WERR_DATA_NOT_ACCEPTED	W_ERROR(0x00000250)
#define WERR_VDM_HARD_ERROR	W_ERROR(0x00000251)
#define WERR_DRIVER_CANCEL_TIMEOUT	W_ERROR(0x00000252)
#define WERR_REPLY_MESSAGE_MISMATCH	W_ERROR(0x00000253)
#define WERR_LOST_WRITEBEHIND_DATA	W_ERROR(0x00000254)
#define WERR_CLIENT_SERVER_PARAMETERS_INVALID	W_ERROR(0x00000255)
#define WERR_NOT_TINY_STREAM	W_ERROR(0x00000256)
#define WERR_STACK_OVERFLOW_READ	W_ERROR(0x00000257)
#define WERR_CONVERT_TO_LARGE	W_ERROR(0x00000258)
#define WERR_FOUND_OUT_OF_SCOPE	W_ERROR(0x00000259)
#define WERR_ALLOCATE_BUCKET	W_ERROR(0x0000025A)
#define WERR_MARSHALL_OVERFLOW	W_ERROR(0x0000025B)
#define WERR_INVALID_VARIANT	W_ERROR(0x0000025C)
#define WERR_BAD_COMPRESSION_BUFFER	W_ERROR(0x0000025D)
#define WERR_AUDIT_FAILED	W_ERROR(0x0000025E)
#define WERR_TIMER_RESOLUTION_NOT_SET	W_ERROR(0x0000025F)
#define WERR_INSUFFICIENT_LOGON_INFO	W_ERROR(0x00000260)
#define WERR_BAD_DLL_ENTRYPOINT	W_ERROR(0x00000261)
#define WERR_BAD_SERVICE_ENTRYPOINT	W_ERROR(0x00000262)
#define WERR_IP_ADDRESS_CONFLICT1	W_ERROR(0x00000263)
#define WERR_IP_ADDRESS_CONFLICT2	W_ERROR(0x00000264)
#define WERR_REGISTRY_QUOTA_LIMIT	W_ERROR(0x00000265)
#define WERR_NO_CALLBACK_ACTIVE	W_ERROR(0x00000266)
#define WERR_PWD_TOO_SHORT	W_ERROR(0x00000267)
#define WERR_PWD_TOO_RECENT	W_ERROR(0x00000268)
#define WERR_PWD_HISTORY_CONFLICT	W_ERROR(0x00000269)
#define WERR_UNSUPPORTED_COMPRESSION	W_ERROR(0x0000026A)
#define WERR_INVALID_HW_PROFILE	W_ERROR(0x0000026B)
#define WERR_INVALID_PLUGPLAY_DEVICE_PATH	W_ERROR(0x0000026C)
#define WERR_QUOTA_LIST_INCONSISTENT	W_ERROR(0x0000026D)
#define WERR_EVALUATION_EXPIRATION	W_ERROR(0x0000026E)
#define WERR_ILLEGAL_DLL_RELOCATION	W_ERROR(0x0000026F)
#define WERR_DLL_INIT_FAILED_LOGOFF	W_ERROR(0x00000270)
#define WERR_VALIDATE_CONTINUE	W_ERROR(0x00000271)
#define WERR_NO_MORE_MATCHES	W_ERROR(0x00000272)
#define WERR_RANGE_LIST_CONFLICT	W_ERROR(0x00000273)
#define WERR_SERVER_SID_MISMATCH	W_ERROR(0x00000274)
#define WERR_CANT_ENABLE_DENY_ONLY	W_ERROR(0x00000275)
#define WERR_FLOAT_MULTIPLE_FAULTS	W_ERROR(0x00000276)
#define WERR_FLOAT_MULTIPLE_TRAPS	W_ERROR(0x00000277)
#define WERR_NOINTERFACE	W_ERROR(0x00000278)
#define WERR_DRIVER_FAILED_SLEEP	W_ERROR(0x00000279)
#define WERR_CORRUPT_SYSTEM_FILE	W_ERROR(0x0000027A)
#define WERR_COMMITMENT_MINIMUM	W_ERROR(0x0000027B)
#define WERR_PNP_RESTART_ENUMERATION	W_ERROR(0x0000027C)
#define WERR_SYSTEM_IMAGE_BAD_SIGNATURE	W_ERROR(0x0000027D)
#define WERR_PNP_REBOOT_REQUIRED	W_ERROR(0x0000027E)
#define WERR_INSUFFICIENT_POWER	W_ERROR(0x0000027F)
#define WERR_MULTIPLE_FAULT_VIOLATION	W_ERROR(0x00000280)
#define WERR_SYSTEM_SHUTDOWN	W_ERROR(0x00000281)
#define WERR_PORT_NOT_SET	W_ERROR(0x00000282)
#define WERR_DS_VERSION_CHECK_FAILURE	W_ERROR(0x00000283)
#define WERR_RANGE_NOT_FOUND	W_ERROR(0x00000284)
#define WERR_NOT_SAFE_MODE_DRIVER	W_ERROR(0x00000286)
#define WERR_FAILED_DRIVER_ENTRY	W_ERROR(0x00000287)
#define WERR_DEVICE_ENUMERATION_ERROR	W_ERROR(0x00000288)
#define WERR_MOUNT_POINT_NOT_RESOLVED	W_ERROR(0x00000289)
#define WERR_INVALID_DEVICE_OBJECT_PARAMETER	W_ERROR(0x0000028A)
#define WERR_MCA_OCCURED	W_ERROR(0x0000028B)
#define WERR_DRIVER_DATABASE_ERROR	W_ERROR(0x0000028C)
#define WERR_SYSTEM_HIVE_TOO_LARGE	W_ERROR(0x0000028D)
#define WERR_DRIVER_FAILED_PRIOR_UNLOAD	W_ERROR(0x0000028E)
#define WERR_VOLSNAP_PREPARE_HIBERNATE	W_ERROR(0x0000028F)
#define WERR_HIBERNATION_FAILURE	W_ERROR(0x00000290)
#define WERR_FILE_SYSTEM_LIMITATION	W_ERROR(0x00000299)
#define WERR_ASSERTION_FAILURE	W_ERROR(0x0000029C)
#define WERR_ACPI_ERROR	W_ERROR(0x0000029D)
#define WERR_WOW_ASSERTION	W_ERROR(0x0000029E)
#define WERR_PNP_BAD_MPS_TABLE	W_ERROR(0x0000029F)
#define WERR_PNP_TRANSLATION_FAILED	W_ERROR(0x000002A0)
#define WERR_PNP_IRQ_TRANSLATION_FAILED	W_ERROR(0x000002A1)
#define WERR_PNP_INVALID_ID	W_ERROR(0x000002A2)
#define WERR_WAKE_SYSTEM_DEBUGGER	W_ERROR(0x000002A3)
#define WERR_HANDLES_CLOSED	W_ERROR(0x000002A4)
#define WERR_EXTRANEOUS_INFORMATION	W_ERROR(0x000002A5)
#define WERR_RXACT_COMMIT_NECESSARY	W_ERROR(0x000002A6)
#define WERR_MEDIA_CHECK	W_ERROR(0x000002A7)
#define WERR_GUID_SUBSTITUTION_MADE	W_ERROR(0x000002A8)
#define WERR_STOPPED_ON_SYMLINK	W_ERROR(0x000002A9)
#define WERR_LONGJUMP	W_ERROR(0x000002AA)
#define WERR_PLUGPLAY_QUERY_VETOED	W_ERROR(0x000002AB)
#define WERR_UNWIND_CONSOLIDATE	W_ERROR(0x000002AC)
#define WERR_REGISTRY_HIVE_RECOVERED	W_ERROR(0x000002AD)
#define WERR_DLL_MIGHT_BE_INSECURE	W_ERROR(0x000002AE)
#define WERR_DLL_MIGHT_BE_INCOMPATIBLE	W_ERROR(0x000002AF)
#define WERR_DBG_EXCEPTION_NOT_HANDLED	W_ERROR(0x000002B0)
#define WERR_DBG_REPLY_LATER	W_ERROR(0x000002B1)
#define WERR_DBG_UNABLE_TO_PROVIDE_HANDLE	W_ERROR(0x000002B2)
#define WERR_DBG_TERMINATE_THREAD	W_ERROR(0x000002B3)
#define WERR_DBG_TERMINATE_PROCESS	W_ERROR(0x000002B4)
#define WERR_DBG_CONTROL_C	W_ERROR(0x000002B5)
#define WERR_DBG_PRINTEXCEPTION_C	W_ERROR(0x000002B6)
#define WERR_DBG_RIPEXCEPTION	W_ERROR(0x000002B7)
#define WERR_DBG_CONTROL_BREAK	W_ERROR(0x000002B8)
#define WERR_DBG_COMMAND_EXCEPTION	W_ERROR(0x000002B9)
#define WERR_OBJECT_NAME_EXISTS	W_ERROR(0x000002BA)
#define WERR_THREAD_WAS_SUSPENDED	W_ERROR(0x000002BB)
#define WERR_IMAGE_NOT_AT_BASE	W_ERROR(0x000002BC)
#define WERR_RXACT_STATE_CREATED	W_ERROR(0x000002BD)
#define WERR_SEGMENT_NOTIFICATION	W_ERROR(0x000002BE)
#define WERR_BAD_CURRENT_DIRECTORY	W_ERROR(0x000002BF)
#define WERR_FT_READ_RECOVERY_FROM_BACKUP	W_ERROR(0x000002C0)
#define WERR_FT_WRITE_RECOVERY	W_ERROR(0x000002C1)
#define WERR_IMAGE_MACHINE_TYPE_MISMATCH	W_ERROR(0x000002C2)
#define WERR_RECEIVE_PARTIAL	W_ERROR(0x000002C3)
#define WERR_RECEIVE_EXPEDITED	W_ERROR(0x000002C4)
#define WERR_RECEIVE_PARTIAL_EXPEDITED	W_ERROR(0x000002C5)
#define WERR_EVENT_DONE	W_ERROR(0x000002C6)
#define WERR_EVENT_PENDING	W_ERROR(0x000002C7)
#define WERR_CHECKING_FILE_SYSTEM	W_ERROR(0x000002C8)
#define WERR_FATAL_APP_EXIT	W_ERROR(0x000002C9)
#define WERR_PREDEFINED_HANDLE	W_ERROR(0x000002CA)
#define WERR_WAS_UNLOCKED	W_ERROR(0x000002CB)
#define WERR_SERVICE_NOTIFICATION	W_ERROR(0x000002CC)
#define WERR_WAS_LOCKED	W_ERROR(0x000002CD)
#define WERR_LOG_HARD_ERROR	W_ERROR(0x000002CE)
#define WERR_ALREADY_WIN32	W_ERROR(0x000002CF)
#define WERR_IMAGE_MACHINE_TYPE_MISMATCH_EXE	W_ERROR(0x000002D0)
#define WERR_NO_YIELD_PERFORMED	W_ERROR(0x000002D1)
#define WERR_TIMER_RESUME_IGNORED	W_ERROR(0x000002D2)
#define WERR_ARBITRATION_UNHANDLED	W_ERROR(0x000002D3)
#define WERR_CARDBUS_NOT_SUPPORTED	W_ERROR(0x000002D4)
#define WERR_MP_PROCESSOR_MISMATCH	W_ERROR(0x000002D5)
#define WERR_HIBERNATED	W_ERROR(0x000002D6)
#define WERR_RESUME_HIBERNATION	W_ERROR(0x000002D7)
#define WERR_FIRMWARE_UPDATED	W_ERROR(0x000002D8)
#define WERR_DRIVERS_LEAKING_LOCKED_PAGES	W_ERROR(0x000002D9)
#define WERR_WAKE_SYSTEM	W_ERROR(0x000002DA)
#define WERR_WAIT_1	W_ERROR(0x000002DB)
#define WERR_WAIT_2	W_ERROR(0x000002DC)
#define WERR_WAIT_3	W_ERROR(0x000002DD)
#define WERR_WAIT_63	W_ERROR(0x000002DE)
#define WERR_ABANDONED_WAIT_0	W_ERROR(0x000002DF)
#define WERR_ABANDONED_WAIT_63	W_ERROR(0x000002E0)
#define WERR_USER_APC	W_ERROR(0x000002E1)
#define WERR_KERNEL_APC	W_ERROR(0x000002E2)
#define WERR_ALERTED	W_ERROR(0x000002E3)
#define WERR_ELEVATION_REQUIRED	W_ERROR(0x000002E4)
#define WERR_REPARSE	W_ERROR(0x000002E5)
#define WERR_OPLOCK_BREAK_IN_PROGRESS	W_ERROR(0x000002E6)
#define WERR_VOLUME_MOUNTED	W_ERROR(0x000002E7)
#define WERR_RXACT_COMMITTED	W_ERROR(0x000002E8)
#define WERR_NOTIFY_CLEANUP	W_ERROR(0x000002E9)
#define WERR_PRIMARY_TRANSPORT_CONNECT_FAILED	W_ERROR(0x000002EA)
#define WERR_PAGE_FAULT_TRANSITION	W_ERROR(0x000002EB)
#define WERR_PAGE_FAULT_DEMAND_ZERO	W_ERROR(0x000002EC)
#define WERR_PAGE_FAULT_COPY_ON_WRITE	W_ERROR(0x000002ED)
#define WERR_PAGE_FAULT_GUARD_PAGE	W_ERROR(0x000002EE)
#define WERR_PAGE_FAULT_PAGING_FILE	W_ERROR(0x000002EF)
#define WERR_CACHE_PAGE_LOCKED	W_ERROR(0x000002F0)
#define WERR_CRASH_DUMP	W_ERROR(0x000002F1)
#define WERR_BUFFER_ALL_ZEROS	W_ERROR(0x000002F2)
#define WERR_REPARSE_OBJECT	W_ERROR(0x000002F3)
#define WERR_RESOURCE_REQUIREMENTS_CHANGED	W_ERROR(0x000002F4)
#define WERR_TRANSLATION_COMPLETE	W_ERROR(0x000002F5)
#define WERR_NOTHING_TO_TERMINATE	W_ERROR(0x000002F6)
#define WERR_PROCESS_NOT_IN_JOB	W_ERROR(0x000002F7)
#define WERR_PROCESS_IN_JOB	W_ERROR(0x000002F8)
#define WERR_VOLSNAP_HIBERNATE_READY	W_ERROR(0x000002F9)
#define WERR_FSFILTER_OP_COMPLETED_SUCCESSFULLY	W_ERROR(0x000002FA)
#define WERR_INTERRUPT_VECTOR_ALREADY_CONNECTED	W_ERROR(0x000002FB)
#define WERR_INTERRUPT_STILL_CONNECTED	W_ERROR(0x000002FC)
#define WERR_WAIT_FOR_OPLOCK	W_ERROR(0x000002FD)
#define WERR_DBG_EXCEPTION_HANDLED	W_ERROR(0x000002FE)
#define WERR_DBG_CONTINUE	W_ERROR(0x000002FF)
#define WERR_CALLBACK_POP_STACK	W_ERROR(0x00000300)
#define WERR_COMPRESSION_DISABLED	W_ERROR(0x00000301)
#define WERR_CANTFETCHBACKWARDS	W_ERROR(0x00000302)
#define WERR_CANTSCROLLBACKWARDS	W_ERROR(0x00000303)
#define WERR_ROWSNOTRELEASED	W_ERROR(0x00000304)
#define WERR_BAD_ACCESSOR_FLAGS	W_ERROR(0x00000305)
#define WERR_ERRORS_ENCOUNTERED	W_ERROR(0x00000306)
#define WERR_NOT_CAPABLE	W_ERROR(0x00000307)
#define WERR_REQUEST_OUT_OF_SEQUENCE	W_ERROR(0x00000308)
#define WERR_VERSION_PARSE_ERROR	W_ERROR(0x00000309)
#define WERR_BADSTARTPOSITION	W_ERROR(0x0000030A)
#define WERR_MEMORY_HARDWARE	W_ERROR(0x0000030B)
#define WERR_DISK_REPAIR_DISABLED	W_ERROR(0x0000030C)
#define WERR_INSUFFICIENT_RESOURCE_FOR_SPECIFIED_SHARED_SECTION_SIZE	W_ERROR(0x0000030D)
#define WERR_SYSTEM_POWERSTATE_TRANSITION	W_ERROR(0x0000030E)
#define WERR_SYSTEM_POWERSTATE_COMPLEX_TRANSITION	W_ERROR(0x0000030F)
#define WERR_MCA_EXCEPTION	W_ERROR(0x00000310)
#define WERR_ACCESS_AUDIT_BY_POLICY	W_ERROR(0x00000311)
#define WERR_ACCESS_DISABLED_NO_SAFER_UI_BY_POLICY	W_ERROR(0x00000312)
#define WERR_ABANDON_HIBERFILE	W_ERROR(0x00000313)
#define WERR_LOST_WRITEBEHIND_DATA_NETWORK_DISCONNECTED	W_ERROR(0x00000314)
#define WERR_LOST_WRITEBEHIND_DATA_NETWORK_SERVER_ERROR	W_ERROR(0x00000315)
#define WERR_LOST_WRITEBEHIND_DATA_LOCAL_DISK_ERROR	W_ERROR(0x00000316)
#define WERR_EA_ACCESS_DENIED	W_ERROR(0x000003E2)
#define WERR_OPERATION_ABORTED	W_ERROR(0x000003E3)
#define WERR_IO_INCOMPLETE	W_ERROR(0x000003E4)
#define WERR_IO_PENDING	W_ERROR(0x000003E5)
#define WERR_NOACCESS	W_ERROR(0x000003E6)
#define WERR_SWAPERROR	W_ERROR(0x000003E7)
#define WERR_STACK_OVERFLOW	W_ERROR(0x000003E9)
#define WERR_INVALID_MESSAGE	W_ERROR(0x000003EA)
#define WERR_CAN_NOT_COMPLETE	W_ERROR(0x000003EB)
#define WERR_INVALID_FLAGS	W_ERROR(0x000003EC)
#define WERR_UNRECOGNIZED_VOLUME	W_ERROR(0x000003ED)
#define WERR_FILE_INVALID	W_ERROR(0x000003EE)
#define WERR_FULLSCREEN_MODE	W_ERROR(0x000003EF)
#define WERR_NO_TOKEN	W_ERROR(0x000003F0)
#define WERR_BADDB	W_ERROR(0x000003F1)
#define WERR_BADKEY	W_ERROR(0x000003F2)
#define WERR_CANTOPEN	W_ERROR(0x000003F3)
#define WERR_CANTREAD	W_ERROR(0x000003F4)
#define WERR_CANTWRITE	W_ERROR(0x000003F5)
#define WERR_REGISTRY_RECOVERED	W_ERROR(0x000003F6)
#define WERR_REGISTRY_CORRUPT	W_ERROR(0x000003F7)
#define WERR_REGISTRY_IO_FAILED	W_ERROR(0x000003F8)
#define WERR_NOT_REGISTRY_FILE	W_ERROR(0x000003F9)
#define WERR_KEY_DELETED	W_ERROR(0x000003FA)
#define WERR_NO_LOG_SPACE	W_ERROR(0x000003FB)
#define WERR_KEY_HAS_CHILDREN	W_ERROR(0x000003FC)
#define WERR_CHILD_MUST_BE_VOLATILE	W_ERROR(0x000003FD)
#define WERR_NOTIFY_ENUM_DIR	W_ERROR(0x000003FE)
#define WERR_DEPENDENT_SERVICES_RUNNING	W_ERROR(0x0000041B)
#define WERR_INVALID_SERVICE_CONTROL	W_ERROR(0x0000041C)
#define WERR_SERVICE_REQUEST_TIMEOUT	W_ERROR(0x0000041D)
#define WERR_SERVICE_NO_THREAD	W_ERROR(0x0000041E)
#define WERR_SERVICE_DATABASE_LOCKED	W_ERROR(0x0000041F)
#define WERR_SERVICE_ALREADY_RUNNING	W_ERROR(0x00000420)
#define WERR_INVALID_SERVICE_ACCOUNT	W_ERROR(0x00000421)
#define WERR_SERVICE_DISABLED	W_ERROR(0x00000422)
#define WERR_CIRCULAR_DEPENDENCY	W_ERROR(0x00000423)
#define WERR_SERVICE_DOES_NOT_EXIST	W_ERROR(0x00000424)
#define WERR_SERVICE_CANNOT_ACCEPT_CTRL	W_ERROR(0x00000425)
#define WERR_SERVICE_NOT_ACTIVE	W_ERROR(0x00000426)
#define WERR_FAILED_SERVICE_CONTROLLER_CONNECT	W_ERROR(0x00000427)
#define WERR_EXCEPTION_IN_SERVICE	W_ERROR(0x00000428)
#define WERR_DATABASE_DOES_NOT_EXIST	W_ERROR(0x00000429)
#define WERR_SERVICE_SPECIFIC_ERROR	W_ERROR(0x0000042A)
#define WERR_PROCESS_ABORTED	W_ERROR(0x0000042B)
#define WERR_SERVICE_DEPENDENCY_FAIL	W_ERROR(0x0000042C)
#define WERR_SERVICE_LOGON_FAILED	W_ERROR(0x0000042D)
#define WERR_SERVICE_START_HANG	W_ERROR(0x0000042E)
#define WERR_INVALID_SERVICE_LOCK	W_ERROR(0x0000042F)
#define WERR_SERVICE_MARKED_FOR_DELETE	W_ERROR(0x00000430)
#define WERR_SERVICE_EXISTS	W_ERROR(0x00000431)
#define WERR_ALREADY_RUNNING_LKG	W_ERROR(0x00000432)
#define WERR_SERVICE_DEPENDENCY_DELETED	W_ERROR(0x00000433)
#define WERR_BOOT_ALREADY_ACCEPTED	W_ERROR(0x00000434)
#define WERR_SERVICE_NEVER_STARTED	W_ERROR(0x00000435)
#define WERR_DUPLICATE_SERVICE_NAME	W_ERROR(0x00000436)
#define WERR_DIFFERENT_SERVICE_ACCOUNT	W_ERROR(0x00000437)
#define WERR_CANNOT_DETECT_DRIVER_FAILURE	W_ERROR(0x00000438)
#define WERR_CANNOT_DETECT_PROCESS_ABORT	W_ERROR(0x00000439)
#define WERR_NO_RECOVERY_PROGRAM	W_ERROR(0x0000043A)
#define WERR_SERVICE_NOT_IN_EXE	W_ERROR(0x0000043B)
#define WERR_NOT_SAFEBOOT_SERVICE	W_ERROR(0x0000043C)
#define WERR_END_OF_MEDIA	W_ERROR(0x0000044C)
#define WERR_FILEMARK_DETECTED	W_ERROR(0x0000044D)
#define WERR_BEGINNING_OF_MEDIA	W_ERROR(0x0000044E)
#define WERR_SETMARK_DETECTED	W_ERROR(0x0000044F)
#define WERR_NO_DATA_DETECTED	W_ERROR(0x00000450)
#define WERR_PARTITION_FAILURE	W_ERROR(0x00000451)
#define WERR_INVALID_BLOCK_LENGTH	W_ERROR(0x00000452)
#define WERR_DEVICE_NOT_PARTITIONED	W_ERROR(0x00000453)
#define WERR_UNABLE_TO_LOCK_MEDIA	W_ERROR(0x00000454)
#define WERR_UNABLE_TO_UNLOAD_MEDIA	W_ERROR(0x00000455)
#define WERR_MEDIA_CHANGED	W_ERROR(0x00000456)
#define WERR_BUS_RESET	W_ERROR(0x00000457)
#define WERR_NO_MEDIA_IN_DRIVE	W_ERROR(0x00000458)
#define WERR_NO_UNICODE_TRANSLATION	W_ERROR(0x00000459)
#define WERR_DLL_INIT_FAILED	W_ERROR(0x0000045A)
#define WERR_SHUTDOWN_IN_PROGRESS	W_ERROR(0x0000045B)
#define WERR_NO_SHUTDOWN_IN_PROGRESS	W_ERROR(0x0000045C)
#define WERR_IO_DEVICE	W_ERROR(0x0000045D)
#define WERR_SERIAL_NO_DEVICE	W_ERROR(0x0000045E)
#define WERR_IRQ_BUSY	W_ERROR(0x0000045F)
#define WERR_MORE_WRITES	W_ERROR(0x00000460)
#define WERR_COUNTER_TIMEOUT	W_ERROR(0x00000461)
#define WERR_FLOPPY_ID_MARK_NOT_FOUND	W_ERROR(0x00000462)
#define WERR_FLOPPY_WRONG_CYLINDER	W_ERROR(0x00000463)
#define WERR_FLOPPY_UNKNOWN_ERROR	W_ERROR(0x00000464)
#define WERR_FLOPPY_BAD_REGISTERS	W_ERROR(0x00000465)
#define WERR_DISK_RECALIBRATE_FAILED	W_ERROR(0x00000466)
#define WERR_DISK_OPERATION_FAILED	W_ERROR(0x00000467)
#define WERR_DISK_RESET_FAILED	W_ERROR(0x00000468)
#define WERR_EOM_OVERFLOW	W_ERROR(0x00000469)
#define WERR_NOT_ENOUGH_SERVER_MEMORY	W_ERROR(0x0000046A)
#define WERR_POSSIBLE_DEADLOCK	W_ERROR(0x0000046B)
#define WERR_MAPPED_ALIGNMENT	W_ERROR(0x0000046C)
#define WERR_SET_POWER_STATE_VETOED	W_ERROR(0x00000474)
#define WERR_SET_POWER_STATE_FAILED	W_ERROR(0x00000475)
#define WERR_TOO_MANY_LINKS	W_ERROR(0x00000476)
#define WERR_OLD_WIN_VERSION	W_ERROR(0x0000047E)
#define WERR_APP_WRONG_OS	W_ERROR(0x0000047F)
#define WERR_SINGLE_INSTANCE_APP	W_ERROR(0x00000480)
#define WERR_RMODE_APP	W_ERROR(0x00000481)
#define WERR_INVALID_DLL	W_ERROR(0x00000482)
#define WERR_NO_ASSOCIATION	W_ERROR(0x00000483)
#define WERR_DDE_FAIL	W_ERROR(0x00000484)
#define WERR_DLL_NOT_FOUND	W_ERROR(0x00000485)
#define WERR_NO_MORE_USER_HANDLES	W_ERROR(0x00000486)
#define WERR_MESSAGE_SYNC_ONLY	W_ERROR(0x00000487)
#define WERR_SOURCE_ELEMENT_EMPTY	W_ERROR(0x00000488)
#define WERR_DESTINATION_ELEMENT_FULL	W_ERROR(0x00000489)
#define WERR_ILLEGAL_ELEMENT_ADDRESS	W_ERROR(0x0000048A)
#define WERR_MAGAZINE_NOT_PRESENT	W_ERROR(0x0000048B)
#define WERR_DEVICE_REINITIALIZATION_NEEDED	W_ERROR(0x0000048C)
#define WERR_DEVICE_REQUIRES_CLEANING	W_ERROR(0x0000048D)
#define WERR_DEVICE_DOOR_OPEN	W_ERROR(0x0000048E)
#define WERR_DEVICE_NOT_CONNECTED	W_ERROR(0x0000048F)
#define WERR_NOT_FOUND	W_ERROR(0x00000490)
#define WERR_NO_MATCH	W_ERROR(0x00000491)
#define WERR_SET_NOT_FOUND	W_ERROR(0x00000492)
#define WERR_POINT_NOT_FOUND	W_ERROR(0x00000493)
#define WERR_NO_TRACKING_SERVICE	W_ERROR(0x00000494)
#define WERR_NO_VOLUME_ID	W_ERROR(0x00000495)
#define WERR_UNABLE_TO_REMOVE_REPLACED	W_ERROR(0x00000497)
#define WERR_UNABLE_TO_MOVE_REPLACEMENT	W_ERROR(0x00000498)
#define WERR_UNABLE_TO_MOVE_REPLACEMENT_2	W_ERROR(0x00000499)
#define WERR_JOURNAL_DELETE_IN_PROGRESS	W_ERROR(0x0000049A)
#define WERR_JOURNAL_NOT_ACTIVE	W_ERROR(0x0000049B)
#define WERR_POTENTIAL_FILE_FOUND	W_ERROR(0x0000049C)
#define WERR_JOURNAL_ENTRY_DELETED	W_ERROR(0x0000049D)
#define WERR_SHUTDOWN_IS_SCHEDULED	W_ERROR(0x000004A6)
#define WERR_SHUTDOWN_USERS_LOGGED_ON	W_ERROR(0x000004A7)
#define WERR_BAD_DEVICE	W_ERROR(0x000004B0)
#define WERR_CONNECTION_UNAVAIL	W_ERROR(0x000004B1)
#define WERR_DEVICE_ALREADY_REMEMBERED	W_ERROR(0x000004B2)
#define WERR_NO_NET_OR_BAD_PATH	W_ERROR(0x000004B3)
#define WERR_BAD_PROVIDER	W_ERROR(0x000004B4)
#define WERR_CANNOT_OPEN_PROFILE	W_ERROR(0x000004B5)
#define WERR_BAD_PROFILE	W_ERROR(0x000004B6)
#define WERR_NOT_CONTAINER	W_ERROR(0x000004B7)
#define WERR_EXTENDED_ERROR	W_ERROR(0x000004B8)
#define WERR_INVALID_GROUPNAME	W_ERROR(0x000004B9)
#define WERR_INVALID_COMPUTERNAME	W_ERROR(0x000004BA)
#define WERR_INVALID_EVENTNAME	W_ERROR(0x000004BB)
#define WERR_INVALID_DOMAINNAME	W_ERROR(0x000004BC)
#define WERR_INVALID_SERVICENAME	W_ERROR(0x000004BD)
#define WERR_INVALID_NETNAME	W_ERROR(0x000004BE)
#define WERR_INVALID_SHARENAME	W_ERROR(0x000004BF)
#define WERR_INVALID_PASSWORDNAME	W_ERROR(0x000004C0)
#define WERR_INVALID_MESSAGENAME	W_ERROR(0x000004C1)
#define WERR_INVALID_MESSAGEDEST	W_ERROR(0x000004C2)
#define WERR_SESSION_CREDENTIAL_CONFLICT	W_ERROR(0x000004C3)
#define WERR_REMOTE_SESSION_LIMIT_EXCEEDED	W_ERROR(0x000004C4)
#define WERR_DUP_DOMAINNAME	W_ERROR(0x000004C5)
#define WERR_NO_NETWORK	W_ERROR(0x000004C6)
#define WERR_CANCELLED	W_ERROR(0x000004C7)
#define WERR_USER_MAPPED_FILE	W_ERROR(0x000004C8)
#define WERR_CONNECTION_REFUSED	W_ERROR(0x000004C9)
#define WERR_GRACEFUL_DISCONNECT	W_ERROR(0x000004CA)
#define WERR_ADDRESS_ALREADY_ASSOCIATED	W_ERROR(0x000004CB)
#define WERR_ADDRESS_NOT_ASSOCIATED	W_ERROR(0x000004CC)
#define WERR_CONNECTION_INVALID	W_ERROR(0x000004CD)
#define WERR_CONNECTION_ACTIVE	W_ERROR(0x000004CE)
#define WERR_NETWORK_UNREACHABLE	W_ERROR(0x000004CF)
#define WERR_HOST_UNREACHABLE	W_ERROR(0x000004D0)
#define WERR_PROTOCOL_UNREACHABLE	W_ERROR(0x000004D1)
#define WERR_PORT_UNREACHABLE	W_ERROR(0x000004D2)
#define WERR_REQUEST_ABORTED	W_ERROR(0x000004D3)
#define WERR_CONNECTION_ABORTED	W_ERROR(0x000004D4)
#define WERR_RETRY	W_ERROR(0x000004D5)
#define WERR_CONNECTION_COUNT_LIMIT	W_ERROR(0x000004D6)
#define WERR_LOGIN_TIME_RESTRICTION	W_ERROR(0x000004D7)
#define WERR_LOGIN_WKSTA_RESTRICTION	W_ERROR(0x000004D8)
#define WERR_INCORRECT_ADDRESS	W_ERROR(0x000004D9)
#define WERR_ALREADY_REGISTERED	W_ERROR(0x000004DA)
#define WERR_SERVICE_NOT_FOUND	W_ERROR(0x000004DB)
#define WERR_NOT_AUTHENTICATED	W_ERROR(0x000004DC)
#define WERR_NOT_LOGGED_ON	W_ERROR(0x000004DD)
#define WERR_CONTINUE	W_ERROR(0x000004DE)
#define WERR_ALREADY_INITIALIZED	W_ERROR(0x000004DF)
#define WERR_NO_MORE_DEVICES	W_ERROR(0x000004E0)
#define WERR_NO_SUCH_SITE	W_ERROR(0x000004E1)
#define WERR_DOMAIN_CONTROLLER_EXISTS	W_ERROR(0x000004E2)
#define WERR_ONLY_IF_CONNECTED	W_ERROR(0x000004E3)
#define WERR_OVERRIDE_NOCHANGES	W_ERROR(0x000004E4)
#define WERR_BAD_USER_PROFILE	W_ERROR(0x000004E5)
#define WERR_NOT_SUPPORTED_ON_SBS	W_ERROR(0x000004E6)
#define WERR_SERVER_SHUTDOWN_IN_PROGRESS	W_ERROR(0x000004E7)
#define WERR_HOST_DOWN	W_ERROR(0x000004E8)
#define WERR_NON_ACCOUNT_SID	W_ERROR(0x000004E9)
#define WERR_NON_DOMAIN_SID	W_ERROR(0x000004EA)
#define WERR_APPHELP_BLOCK	W_ERROR(0x000004EB)
#define WERR_ACCESS_DISABLED_BY_POLICY	W_ERROR(0x000004EC)
#define WERR_REG_NAT_CONSUMPTION	W_ERROR(0x000004ED)
#define WERR_CSCSHARE_OFFLINE	W_ERROR(0x000004EE)
#define WERR_PKINIT_FAILURE	W_ERROR(0x000004EF)
#define WERR_SMARTCARD_SUBSYSTEM_FAILURE	W_ERROR(0x000004F0)
#define WERR_DOWNGRADE_DETECTED	W_ERROR(0x000004F1)
#define WERR_MACHINE_LOCKED	W_ERROR(0x000004F7)
#define WERR_CALLBACK_SUPPLIED_INVALID_DATA	W_ERROR(0x000004F9)
#define WERR_SYNC_FOREGROUND_REFRESH_REQUIRED	W_ERROR(0x000004FA)
#define WERR_DRIVER_BLOCKED	W_ERROR(0x000004FB)
#define WERR_INVALID_IMPORT_OF_NON_DLL	W_ERROR(0x000004FC)
#define WERR_ACCESS_DISABLED_WEBBLADE	W_ERROR(0x000004FD)
#define WERR_ACCESS_DISABLED_WEBBLADE_TAMPER	W_ERROR(0x000004FE)
#define WERR_RECOVERY_FAILURE	W_ERROR(0x000004FF)
#define WERR_ALREADY_FIBER	W_ERROR(0x00000500)
#define WERR_ALREADY_THREAD	W_ERROR(0x00000501)
#define WERR_STACK_BUFFER_OVERRUN	W_ERROR(0x00000502)
#define WERR_PARAMETER_QUOTA_EXCEEDED	W_ERROR(0x00000503)
#define WERR_DEBUGGER_INACTIVE	W_ERROR(0x00000504)
#define WERR_DELAY_LOAD_FAILED	W_ERROR(0x00000505)
#define WERR_VDM_DISALLOWED	W_ERROR(0x00000506)
#define WERR_UNIDENTIFIED_ERROR	W_ERROR(0x00000507)
#define WERR_BEYOND_VDL	W_ERROR(0x00000509)
#define WERR_INCOMPATIBLE_SERVICE_SID_TYPE	W_ERROR(0x0000050A)
#define WERR_DRIVER_PROCESS_TERMINATED	W_ERROR(0x0000050B)
#define WERR_IMPLEMENTATION_LIMIT	W_ERROR(0x0000050C)
#define WERR_PROCESS_IS_PROTECTED	W_ERROR(0x0000050D)
#define WERR_SERVICE_NOTIFY_CLIENT_LAGGING	W_ERROR(0x0000050E)
#define WERR_DISK_QUOTA_EXCEEDED	W_ERROR(0x0000050F)
#define WERR_CONTENT_BLOCKED	W_ERROR(0x00000510)
#define WERR_INCOMPATIBLE_SERVICE_PRIVILEGE	W_ERROR(0x00000511)
#define WERR_INVALID_LABEL	W_ERROR(0x00000513)
#define WERR_NOT_ALL_ASSIGNED	W_ERROR(0x00000514)
#define WERR_SOME_NOT_MAPPED	W_ERROR(0x00000515)
#define WERR_NO_QUOTAS_FOR_ACCOUNT	W_ERROR(0x00000516)
#define WERR_LOCAL_USER_SESSION_KEY	W_ERROR(0x00000517)
#define WERR_NULL_LM_PASSWORD	W_ERROR(0x00000518)
#define WERR_UNKNOWN_REVISION	W_ERROR(0x00000519)
#define WERR_REVISION_MISMATCH	W_ERROR(0x0000051A)
#define WERR_INVALID_OWNER	W_ERROR(0x0000051B)
#define WERR_INVALID_PRIMARY_GROUP	W_ERROR(0x0000051C)
#define WERR_NO_IMPERSONATION_TOKEN	W_ERROR(0x0000051D)
#define WERR_CANT_DISABLE_MANDATORY	W_ERROR(0x0000051E)
#define WERR_NO_LOGON_SERVERS	W_ERROR(0x0000051F)
#define WERR_NO_SUCH_LOGON_SESSION	W_ERROR(0x00000520)
#define WERR_NO_SUCH_PRIVILEGE	W_ERROR(0x00000521)
#define WERR_PRIVILEGE_NOT_HELD	W_ERROR(0x00000522)
#define WERR_INVALID_ACCOUNT_NAME	W_ERROR(0x00000523)
#define WERR_USER_EXISTS	W_ERROR(0x00000524)
#define WERR_NO_SUCH_USER	W_ERROR(0x00000525)
#define WERR_GROUP_EXISTS	W_ERROR(0x00000526)
#define WERR_NO_SUCH_GROUP	W_ERROR(0x00000527)
#define WERR_MEMBER_IN_GROUP	W_ERROR(0x00000528)
#define WERR_MEMBER_NOT_IN_GROUP	W_ERROR(0x00000529)
#define WERR_LAST_ADMIN	W_ERROR(0x0000052A)
#define WERR_WRONG_PASSWORD	W_ERROR(0x0000052B)
#define WERR_ILL_FORMED_PASSWORD	W_ERROR(0x0000052C)
#define WERR_PASSWORD_RESTRICTION	W_ERROR(0x0000052D)
#define WERR_LOGON_FAILURE	W_ERROR(0x0000052E)
#define WERR_ACCOUNT_RESTRICTION	W_ERROR(0x0000052F)
#define WERR_INVALID_LOGON_HOURS	W_ERROR(0x00000530)
#define WERR_INVALID_WORKSTATION	W_ERROR(0x00000531)
#define WERR_PASSWORD_EXPIRED	W_ERROR(0x00000532)
#define WERR_ACCOUNT_DISABLED	W_ERROR(0x00000533)
#define WERR_NONE_MAPPED	W_ERROR(0x00000534)
#define WERR_TOO_MANY_LUIDS_REQUESTED	W_ERROR(0x00000535)
#define WERR_LUIDS_EXHAUSTED	W_ERROR(0x00000536)
#define WERR_INVALID_SUB_AUTHORITY	W_ERROR(0x00000537)
#define WERR_INVALID_ACL	W_ERROR(0x00000538)
#define WERR_INVALID_SID	W_ERROR(0x00000539)
#define WERR_INVALID_SECURITY_DESCR	W_ERROR(0x0000053A)
#define WERR_BAD_INHERITANCE_ACL	W_ERROR(0x0000053C)
#define WERR_SERVER_DISABLED	W_ERROR(0x0000053D)
#define WERR_SERVER_NOT_DISABLED	W_ERROR(0x0000053E)
#define WERR_INVALID_ID_AUTHORITY	W_ERROR(0x0000053F)
#define WERR_ALLOTTED_SPACE_EXCEEDED	W_ERROR(0x00000540)
#define WERR_INVALID_GROUP_ATTRIBUTES	W_ERROR(0x00000541)
#define WERR_BAD_IMPERSONATION_LEVEL	W_ERROR(0x00000542)
#define WERR_CANT_OPEN_ANONYMOUS	W_ERROR(0x00000543)
#define WERR_BAD_VALIDATION_CLASS	W_ERROR(0x00000544)
#define WERR_BAD_TOKEN_TYPE	W_ERROR(0x00000545)
#define WERR_NO_SECURITY_ON_OBJECT	W_ERROR(0x00000546)
#define WERR_CANT_ACCESS_DOMAIN_INFO	W_ERROR(0x00000547)
#define WERR_INVALID_SERVER_STATE	W_ERROR(0x00000548)
#define WERR_INVALID_DOMAIN_STATE	W_ERROR(0x00000549)
#define WERR_INVALID_DOMAIN_ROLE	W_ERROR(0x0000054A)
#define WERR_NO_SUCH_DOMAIN	W_ERROR(0x0000054B)
#define WERR_DOMAIN_EXISTS	W_ERROR(0x0000054C)
#define WERR_DOMAIN_LIMIT_EXCEEDED	W_ERROR(0x0000054D)
#define WERR_INTERNAL_DB_CORRUPTION	W_ERROR(0x0000054E)
#define WERR_INTERNAL_ERROR	W_ERROR(0x0000054F)
#define WERR_GENERIC_NOT_MAPPED	W_ERROR(0x00000550)
#define WERR_BAD_DESCRIPTOR_FORMAT	W_ERROR(0x00000551)
#define WERR_NOT_LOGON_PROCESS	W_ERROR(0x00000552)
#define WERR_LOGON_SESSION_EXISTS	W_ERROR(0x00000553)
#define WERR_NO_SUCH_PACKAGE	W_ERROR(0x00000554)
#define WERR_BAD_LOGON_SESSION_STATE	W_ERROR(0x00000555)
#define WERR_LOGON_SESSION_COLLISION	W_ERROR(0x00000556)
#define WERR_INVALID_LOGON_TYPE	W_ERROR(0x00000557)
#define WERR_CANNOT_IMPERSONATE	W_ERROR(0x00000558)
#define WERR_RXACT_INVALID_STATE	W_ERROR(0x00000559)
#define WERR_RXACT_COMMIT_FAILURE	W_ERROR(0x0000055A)
#define WERR_SPECIAL_ACCOUNT	W_ERROR(0x0000055B)
#define WERR_SPECIAL_GROUP	W_ERROR(0x0000055C)
#define WERR_SPECIAL_USER	W_ERROR(0x0000055D)
#define WERR_MEMBERS_PRIMARY_GROUP	W_ERROR(0x0000055E)
#define WERR_TOKEN_ALREADY_IN_USE	W_ERROR(0x0000055F)
#define WERR_NO_SUCH_ALIAS	W_ERROR(0x00000560)
#define WERR_MEMBER_NOT_IN_ALIAS	W_ERROR(0x00000561)
#define WERR_MEMBER_IN_ALIAS	W_ERROR(0x00000562)
#define WERR_ALIAS_EXISTS	W_ERROR(0x00000563)
#define WERR_LOGON_NOT_GRANTED	W_ERROR(0x00000564)
#define WERR_TOO_MANY_SECRETS	W_ERROR(0x00000565)
#define WERR_SECRET_TOO_LONG	W_ERROR(0x00000566)
#define WERR_INTERNAL_DB_ERROR	W_ERROR(0x00000567)
#define WERR_TOO_MANY_CONTEXT_IDS	W_ERROR(0x00000568)
#define WERR_LOGON_TYPE_NOT_GRANTED	W_ERROR(0x00000569)
#define WERR_NT_CROSS_ENCRYPTION_REQUIRED	W_ERROR(0x0000056A)
#define WERR_NO_SUCH_MEMBER	W_ERROR(0x0000056B)
#define WERR_INVALID_MEMBER	W_ERROR(0x0000056C)
#define WERR_TOO_MANY_SIDS	W_ERROR(0x0000056D)
#define WERR_LM_CROSS_ENCRYPTION_REQUIRED	W_ERROR(0x0000056E)
#define WERR_NO_INHERITANCE	W_ERROR(0x0000056F)
#define WERR_FILE_CORRUPT	W_ERROR(0x00000570)
#define WERR_DISK_CORRUPT	W_ERROR(0x00000571)
#define WERR_NO_USER_SESSION_KEY	W_ERROR(0x00000572)
#define WERR_LICENSE_QUOTA_EXCEEDED	W_ERROR(0x00000573)
#define WERR_WRONG_TARGET_NAME	W_ERROR(0x00000574)
#define WERR_MUTUAL_AUTH_FAILED	W_ERROR(0x00000575)
#define WERR_TIME_SKEW	W_ERROR(0x00000576)
#define WERR_CURRENT_DOMAIN_NOT_ALLOWED	W_ERROR(0x00000577)
#define WERR_INVALID_WINDOW_HANDLE	W_ERROR(0x00000578)
#define WERR_INVALID_MENU_HANDLE	W_ERROR(0x00000579)
#define WERR_INVALID_CURSOR_HANDLE	W_ERROR(0x0000057A)
#define WERR_INVALID_ACCEL_HANDLE	W_ERROR(0x0000057B)
#define WERR_INVALID_HOOK_HANDLE	W_ERROR(0x0000057C)
#define WERR_INVALID_DWP_HANDLE	W_ERROR(0x0000057D)
#define WERR_TLW_WITH_WSCHILD	W_ERROR(0x0000057E)
#define WERR_CANNOT_FIND_WND_CLASS	W_ERROR(0x0000057F)
#define WERR_WINDOW_OF_OTHER_THREAD	W_ERROR(0x00000580)
#define WERR_HOTKEY_ALREADY_REGISTERED	W_ERROR(0x00000581)
#define WERR_CLASS_ALREADY_EXISTS	W_ERROR(0x00000582)
#define WERR_CLASS_DOES_NOT_EXIST	W_ERROR(0x00000583)
#define WERR_CLASS_HAS_WINDOWS	W_ERROR(0x00000584)
#define WERR_INVALID_INDEX	W_ERROR(0x00000585)
#define WERR_INVALID_ICON_HANDLE	W_ERROR(0x00000586)
#define WERR_PRIVATE_DIALOG_INDEX	W_ERROR(0x00000587)
#define WERR_LISTBOX_ID_NOT_FOUND	W_ERROR(0x00000588)
#define WERR_NO_WILDCARD_CHARACTERS	W_ERROR(0x00000589)
#define WERR_CLIPBOARD_NOT_OPEN	W_ERROR(0x0000058A)
#define WERR_HOTKEY_NOT_REGISTERED	W_ERROR(0x0000058B)
#define WERR_WINDOW_NOT_DIALOG	W_ERROR(0x0000058C)
#define WERR_CONTROL_ID_NOT_FOUND	W_ERROR(0x0000058D)
#define WERR_INVALID_COMBOBOX_MESSAGE	W_ERROR(0x0000058E)
#define WERR_WINDOW_NOT_COMBOBOX	W_ERROR(0x0000058F)
#define WERR_INVALID_EDIT_HEIGHT	W_ERROR(0x00000590)
#define WERR_DC_NOT_FOUND	W_ERROR(0x00000591)
#define WERR_INVALID_HOOK_FILTER	W_ERROR(0x00000592)
#define WERR_INVALID_FILTER_PROC	W_ERROR(0x00000593)
#define WERR_HOOK_NEEDS_HMOD	W_ERROR(0x00000594)
#define WERR_GLOBAL_ONLY_HOOK	W_ERROR(0x00000595)
#define WERR_JOURNAL_HOOK_SET	W_ERROR(0x00000596)
#define WERR_HOOK_NOT_INSTALLED	W_ERROR(0x00000597)
#define WERR_INVALID_LB_MESSAGE	W_ERROR(0x00000598)
#define WERR_SETCOUNT_ON_BAD_LB	W_ERROR(0x00000599)
#define WERR_LB_WITHOUT_TABSTOPS	W_ERROR(0x0000059A)
#define WERR_DESTROY_OBJECT_OF_OTHER_THREAD	W_ERROR(0x0000059B)
#define WERR_CHILD_WINDOW_MENU	W_ERROR(0x0000059C)
#define WERR_NO_SYSTEM_MENU	W_ERROR(0x0000059D)
#define WERR_INVALID_MSGBOX_STYLE	W_ERROR(0x0000059E)
#define WERR_INVALID_SPI_VALUE	W_ERROR(0x0000059F)
#define WERR_SCREEN_ALREADY_LOCKED	W_ERROR(0x000005A0)
#define WERR_HWNDS_HAVE_DIFF_PARENT	W_ERROR(0x000005A1)
#define WERR_NOT_CHILD_WINDOW	W_ERROR(0x000005A2)
#define WERR_INVALID_GW_COMMAND	W_ERROR(0x000005A3)
#define WERR_INVALID_THREAD_ID	W_ERROR(0x000005A4)
#define WERR_NON_MDICHILD_WINDOW	W_ERROR(0x000005A5)
#define WERR_POPUP_ALREADY_ACTIVE	W_ERROR(0x000005A6)
#define WERR_NO_SCROLLBARS	W_ERROR(0x000005A7)
#define WERR_INVALID_SCROLLBAR_RANGE	W_ERROR(0x000005A8)
#define WERR_INVALID_SHOWWIN_COMMAND	W_ERROR(0x000005A9)
#define WERR_NO_SYSTEM_RESOURCES	W_ERROR(0x000005AA)
#define WERR_NONPAGED_SYSTEM_RESOURCES	W_ERROR(0x000005AB)
#define WERR_PAGED_SYSTEM_RESOURCES	W_ERROR(0x000005AC)
#define WERR_WORKING_SET_QUOTA	W_ERROR(0x000005AD)
#define WERR_PAGEFILE_QUOTA	W_ERROR(0x000005AE)
#define WERR_COMMITMENT_LIMIT	W_ERROR(0x000005AF)
#define WERR_MENU_ITEM_NOT_FOUND	W_ERROR(0x000005B0)
#define WERR_INVALID_KEYBOARD_HANDLE	W_ERROR(0x000005B1)
#define WERR_HOOK_TYPE_NOT_ALLOWED	W_ERROR(0x000005B2)
#define WERR_REQUIRES_INTERACTIVE_WINDOWSTATION	W_ERROR(0x000005B3)
#define WERR_TIMEOUT	W_ERROR(0x000005B4)
#define WERR_INVALID_MONITOR_HANDLE	W_ERROR(0x000005B5)
#define WERR_INCORRECT_SIZE	W_ERROR(0x000005B6)
#define WERR_SYMLINK_CLASS_DISABLED	W_ERROR(0x000005B7)
#define WERR_SYMLINK_NOT_SUPPORTED	W_ERROR(0x000005B8)
#define WERR_EVENTLOG_FILE_CORRUPT	W_ERROR(0x000005DC)
#define WERR_EVENTLOG_CANT_START	W_ERROR(0x000005DD)
#define WERR_LOG_FILE_FULL	W_ERROR(0x000005DE)
#define WERR_EVENTLOG_FILE_CHANGED	W_ERROR(0x000005DF)
#define WERR_INVALID_TASK_NAME	W_ERROR(0x0000060E)
#define WERR_INVALID_TASK_INDEX	W_ERROR(0x0000060F)
#define WERR_THREAD_ALREADY_IN_TASK	W_ERROR(0x00000610)
#define WERR_INSTALL_SERVICE_FAILURE	W_ERROR(0x00000641)
#define WERR_INSTALL_USEREXIT	W_ERROR(0x00000642)
#define WERR_INSTALL_FAILURE	W_ERROR(0x00000643)
#define WERR_INSTALL_SUSPEND	W_ERROR(0x00000644)
#define WERR_UNKNOWN_PRODUCT	W_ERROR(0x00000645)
#define WERR_UNKNOWN_FEATURE	W_ERROR(0x00000646)
#define WERR_UNKNOWN_COMPONENT	W_ERROR(0x00000647)
#define WERR_UNKNOWN_PROPERTY	W_ERROR(0x00000648)
#define WERR_INVALID_HANDLE_STATE	W_ERROR(0x00000649)
#define WERR_BAD_CONFIGURATION	W_ERROR(0x0000064A)
#define WERR_INDEX_ABSENT	W_ERROR(0x0000064B)
#define WERR_INSTALL_SOURCE_ABSENT	W_ERROR(0x0000064C)
#define WERR_INSTALL_PACKAGE_VERSION	W_ERROR(0x0000064D)
#define WERR_PRODUCT_UNINSTALLED	W_ERROR(0x0000064E)
#define WERR_BAD_QUERY_SYNTAX	W_ERROR(0x0000064F)
#define WERR_INVALID_FIELD	W_ERROR(0x00000650)
#define WERR_DEVICE_REMOVED	W_ERROR(0x00000651)
#define WERR_INSTALL_ALREADY_RUNNING	W_ERROR(0x00000652)
#define WERR_INSTALL_PACKAGE_OPEN_FAILED	W_ERROR(0x00000653)
#define WERR_INSTALL_PACKAGE_INVALID	W_ERROR(0x00000654)
#define WERR_INSTALL_UI_FAILURE	W_ERROR(0x00000655)
#define WERR_INSTALL_LOG_FAILURE	W_ERROR(0x00000656)
#define WERR_INSTALL_LANGUAGE_UNSUPPORTED	W_ERROR(0x00000657)
#define WERR_INSTALL_TRANSFORM_FAILURE	W_ERROR(0x00000658)
#define WERR_INSTALL_PACKAGE_REJECTED	W_ERROR(0x00000659)
#define WERR_FUNCTION_NOT_CALLED	W_ERROR(0x0000065A)
#define WERR_FUNCTION_FAILED	W_ERROR(0x0000065B)
#define WERR_INVALID_TABLE	W_ERROR(0x0000065C)
#define WERR_DATATYPE_MISMATCH	W_ERROR(0x0000065D)
#define WERR_UNSUPPORTED_TYPE	W_ERROR(0x0000065E)
#define WERR_CREATE_FAILED	W_ERROR(0x0000065F)
#define WERR_INSTALL_TEMP_UNWRITABLE	W_ERROR(0x00000660)
#define WERR_INSTALL_PLATFORM_UNSUPPORTED	W_ERROR(0x00000661)
#define WERR_INSTALL_NOTUSED	W_ERROR(0x00000662)
#define WERR_PATCH_PACKAGE_OPEN_FAILED	W_ERROR(0x00000663)
#define WERR_PATCH_PACKAGE_INVALID	W_ERROR(0x00000664)
#define WERR_PATCH_PACKAGE_UNSUPPORTED	W_ERROR(0x00000665)
#define WERR_PRODUCT_VERSION	W_ERROR(0x00000666)
#define WERR_INVALID_COMMAND_LINE	W_ERROR(0x00000667)
#define WERR_INSTALL_REMOTE_DISALLOWED	W_ERROR(0x00000668)
#define WERR_SUCCESS_REBOOT_INITIATED	W_ERROR(0x00000669)
#define WERR_PATCH_TARGET_NOT_FOUND	W_ERROR(0x0000066A)
#define WERR_PATCH_PACKAGE_REJECTED	W_ERROR(0x0000066B)
#define WERR_INSTALL_TRANSFORM_REJECTED	W_ERROR(0x0000066C)
#define WERR_INSTALL_REMOTE_PROHIBITED	W_ERROR(0x0000066D)
#define WERR_PATCH_REMOVAL_UNSUPPORTED	W_ERROR(0x0000066E)
#define WERR_UNKNOWN_PATCH	W_ERROR(0x0000066F)
#define WERR_PATCH_NO_SEQUENCE	W_ERROR(0x00000670)
#define WERR_PATCH_REMOVAL_DISALLOWED	W_ERROR(0x00000671)
#define WERR_INVALID_PATCH_XML	W_ERROR(0x00000672)
#define WERR_PATCH_MANAGED_ADVERTISED_PRODUCT	W_ERROR(0x00000673)
#define WERR_INSTALL_SERVICE_SAFEBOOT	W_ERROR(0x00000674)
#define WERR_RPC_S_INVALID_STRING_BINDING	W_ERROR(0x000006A4)
#define WERR_RPC_S_WRONG_KIND_OF_BINDING	W_ERROR(0x000006A5)
#define WERR_RPC_S_INVALID_BINDING	W_ERROR(0x000006A6)
#define WERR_RPC_S_PROTSEQ_NOT_SUPPORTED	W_ERROR(0x000006A7)
#define WERR_RPC_S_INVALID_RPC_PROTSEQ	W_ERROR(0x000006A8)
#define WERR_RPC_S_INVALID_STRING_UUID	W_ERROR(0x000006A9)
#define WERR_RPC_S_INVALID_ENDPOINT_FORMAT	W_ERROR(0x000006AA)
#define WERR_RPC_S_INVALID_NET_ADDR	W_ERROR(0x000006AB)
#define WERR_RPC_S_NO_ENDPOINT_FOUND	W_ERROR(0x000006AC)
#define WERR_RPC_S_INVALID_TIMEOUT	W_ERROR(0x000006AD)
#define WERR_RPC_S_OBJECT_NOT_FOUND	W_ERROR(0x000006AE)
#define WERR_RPC_S_ALREADY_REGISTERED	W_ERROR(0x000006AF)
#define WERR_RPC_S_TYPE_ALREADY_REGISTERED	W_ERROR(0x000006B0)
#define WERR_RPC_S_ALREADY_LISTENING	W_ERROR(0x000006B1)
#define WERR_RPC_S_NO_PROTSEQS_REGISTERED	W_ERROR(0x000006B2)
#define WERR_RPC_S_NOT_LISTENING	W_ERROR(0x000006B3)
#define WERR_RPC_S_UNKNOWN_MGR_TYPE	W_ERROR(0x000006B4)
#define WERR_RPC_S_UNKNOWN_IF	W_ERROR(0x000006B5)
#define WERR_RPC_S_NO_BINDINGS	W_ERROR(0x000006B6)
#define WERR_RPC_S_NO_PROTSEQS	W_ERROR(0x000006B7)
#define WERR_RPC_S_CANT_CREATE_ENDPOINT	W_ERROR(0x000006B8)
#define WERR_RPC_S_OUT_OF_RESOURCES	W_ERROR(0x000006B9)
#define WERR_RPC_S_SERVER_UNAVAILABLE	W_ERROR(0x000006BA)
#define WERR_RPC_S_SERVER_TOO_BUSY	W_ERROR(0x000006BB)
#define WERR_RPC_S_INVALID_NETWORK_OPTIONS	W_ERROR(0x000006BC)
#define WERR_RPC_S_NO_CALL_ACTIVE	W_ERROR(0x000006BD)
#define WERR_RPC_S_CALL_FAILED	W_ERROR(0x000006BE)
#define WERR_RPC_S_CALL_FAILED_DNE	W_ERROR(0x000006BF)
#define WERR_RPC_S_PROTOCOL_ERROR	W_ERROR(0x000006C0)
#define WERR_RPC_S_PROXY_ACCESS_DENIED	W_ERROR(0x000006C1)
#define WERR_RPC_S_UNSUPPORTED_TRANS_SYN	W_ERROR(0x000006C2)
#define WERR_RPC_S_UNSUPPORTED_TYPE	W_ERROR(0x000006C4)
#define WERR_RPC_S_INVALID_TAG	W_ERROR(0x000006C5)
#define WERR_RPC_S_INVALID_BOUND	W_ERROR(0x000006C6)
#define WERR_RPC_S_NO_ENTRY_NAME	W_ERROR(0x000006C7)
#define WERR_RPC_S_INVALID_NAME_SYNTAX	W_ERROR(0x000006C8)
#define WERR_RPC_S_UNSUPPORTED_NAME_SYNTAX	W_ERROR(0x000006C9)
#define WERR_RPC_S_UUID_NO_ADDRESS	W_ERROR(0x000006CB)
#define WERR_RPC_S_DUPLICATE_ENDPOINT	W_ERROR(0x000006CC)
#define WERR_RPC_S_UNKNOWN_AUTHN_TYPE	W_ERROR(0x000006CD)
#define WERR_RPC_S_MAX_CALLS_TOO_SMALL	W_ERROR(0x000006CE)
#define WERR_RPC_S_STRING_TOO_LONG	W_ERROR(0x000006CF)
#define WERR_RPC_S_PROTSEQ_NOT_FOUND	W_ERROR(0x000006D0)
#define WERR_RPC_S_PROCNUM_OUT_OF_RANGE	W_ERROR(0x000006D1)
#define WERR_RPC_S_BINDING_HAS_NO_AUTH	W_ERROR(0x000006D2)
#define WERR_RPC_S_UNKNOWN_AUTHN_SERVICE	W_ERROR(0x000006D3)
#define WERR_RPC_S_UNKNOWN_AUTHN_LEVEL	W_ERROR(0x000006D4)
#define WERR_RPC_S_INVALID_AUTH_IDENTITY	W_ERROR(0x000006D5)
#define WERR_RPC_S_UNKNOWN_AUTHZ_SERVICE	W_ERROR(0x000006D6)
#define WERR_EPT_S_INVALID_ENTRY	W_ERROR(0x000006D7)
#define WERR_EPT_S_CANT_PERFORM_OP	W_ERROR(0x000006D8)
#define WERR_EPT_S_NOT_REGISTERED	W_ERROR(0x000006D9)
#define WERR_RPC_S_NOTHING_TO_EXPORT	W_ERROR(0x000006DA)
#define WERR_RPC_S_INCOMPLETE_NAME	W_ERROR(0x000006DB)
#define WERR_RPC_S_INVALID_VERS_OPTION	W_ERROR(0x000006DC)
#define WERR_RPC_S_NO_MORE_MEMBERS	W_ERROR(0x000006DD)
#define WERR_RPC_S_NOT_ALL_OBJS_UNEXPORTED	W_ERROR(0x000006DE)
#define WERR_RPC_S_INTERFACE_NOT_FOUND	W_ERROR(0x000006DF)
#define WERR_RPC_S_ENTRY_ALREADY_EXISTS	W_ERROR(0x000006E0)
#define WERR_RPC_S_ENTRY_NOT_FOUND	W_ERROR(0x000006E1)
#define WERR_RPC_S_NAME_SERVICE_UNAVAILABLE	W_ERROR(0x000006E2)
#define WERR_RPC_S_INVALID_NAF_ID	W_ERROR(0x000006E3)
#define WERR_RPC_S_CANNOT_SUPPORT	W_ERROR(0x000006E4)
#define WERR_RPC_S_NO_CONTEXT_AVAILABLE	W_ERROR(0x000006E5)
#define WERR_RPC_S_INTERNAL_ERROR	W_ERROR(0x000006E6)
#define WERR_RPC_S_ZERO_DIVIDE	W_ERROR(0x000006E7)
#define WERR_RPC_S_ADDRESS_ERROR	W_ERROR(0x000006E8)
#define WERR_RPC_S_FP_DIV_ZERO	W_ERROR(0x000006E9)
#define WERR_RPC_S_FP_UNDERFLOW	W_ERROR(0x000006EA)
#define WERR_RPC_S_FP_OVERFLOW	W_ERROR(0x000006EB)
#define WERR_RPC_X_NO_MORE_ENTRIES	W_ERROR(0x000006EC)
#define WERR_RPC_X_SS_CHAR_TRANS_OPEN_FAIL	W_ERROR(0x000006ED)
#define WERR_RPC_X_SS_CHAR_TRANS_SHORT_FILE	W_ERROR(0x000006EE)
#define WERR_RPC_X_SS_IN_NULL_CONTEXT	W_ERROR(0x000006EF)
#define WERR_RPC_X_SS_CONTEXT_DAMAGED	W_ERROR(0x000006F1)
#define WERR_RPC_X_SS_HANDLES_MISMATCH	W_ERROR(0x000006F2)
#define WERR_RPC_X_SS_CANNOT_GET_CALL_HANDLE	W_ERROR(0x000006F3)
#define WERR_RPC_X_NULL_REF_POINTER	W_ERROR(0x000006F4)
#define WERR_RPC_X_ENUM_VALUE_OUT_OF_RANGE	W_ERROR(0x000006F5)
#define WERR_RPC_X_BYTE_COUNT_TOO_SMALL	W_ERROR(0x000006F6)
#define WERR_RPC_X_BAD_STUB_DATA	W_ERROR(0x000006F7)
#define WERR_INVALID_USER_BUFFER	W_ERROR(0x000006F8)
#define WERR_UNRECOGNIZED_MEDIA	W_ERROR(0x000006F9)
#define WERR_NO_TRUST_LSA_SECRET	W_ERROR(0x000006FA)
#define WERR_NO_TRUST_SAM_ACCOUNT	W_ERROR(0x000006FB)
#define WERR_TRUSTED_DOMAIN_FAILURE	W_ERROR(0x000006FC)
#define WERR_TRUSTED_RELATIONSHIP_FAILURE	W_ERROR(0x000006FD)
#define WERR_TRUST_FAILURE	W_ERROR(0x000006FE)
#define WERR_RPC_S_CALL_IN_PROGRESS	W_ERROR(0x000006FF)
#define WERR_NETLOGON_NOT_STARTED	W_ERROR(0x00000700)
#define WERR_ACCOUNT_EXPIRED	W_ERROR(0x00000701)
#define WERR_REDIRECTOR_HAS_OPEN_HANDLES	W_ERROR(0x00000702)
#define WERR_PRINTER_DRIVER_ALREADY_INSTALLED	W_ERROR(0x00000703)
#define WERR_UNKNOWN_PORT	W_ERROR(0x00000704)
#define WERR_UNKNOWN_PRINTER_DRIVER	W_ERROR(0x00000705)
#define WERR_UNKNOWN_PRINTPROCESSOR	W_ERROR(0x00000706)
#define WERR_INVALID_SEPARATOR_FILE	W_ERROR(0x00000707)
#define WERR_INVALID_PRIORITY	W_ERROR(0x00000708)
#define WERR_INVALID_PRINTER_NAME	W_ERROR(0x00000709)
#define WERR_PRINTER_ALREADY_EXISTS	W_ERROR(0x0000070A)
#define WERR_INVALID_PRINTER_COMMAND	W_ERROR(0x0000070B)
#define WERR_INVALID_DATATYPE	W_ERROR(0x0000070C)
#define WERR_INVALID_ENVIRONMENT	W_ERROR(0x0000070D)
#define WERR_RPC_S_NO_MORE_BINDINGS	W_ERROR(0x0000070E)
#define WERR_NOLOGON_INTERDOMAIN_TRUST_ACCOUNT	W_ERROR(0x0000070F)
#define WERR_NOLOGON_WORKSTATION_TRUST_ACCOUNT	W_ERROR(0x00000710)
#define WERR_NOLOGON_SERVER_TRUST_ACCOUNT	W_ERROR(0x00000711)
#define WERR_DOMAIN_TRUST_INCONSISTENT	W_ERROR(0x00000712)
#define WERR_SERVER_HAS_OPEN_HANDLES	W_ERROR(0x00000713)
#define WERR_RESOURCE_DATA_NOT_FOUND	W_ERROR(0x00000714)
#define WERR_RESOURCE_TYPE_NOT_FOUND	W_ERROR(0x00000715)
#define WERR_RESOURCE_NAME_NOT_FOUND	W_ERROR(0x00000716)
#define WERR_RESOURCE_LANG_NOT_FOUND	W_ERROR(0x00000717)
#define WERR_NOT_ENOUGH_QUOTA	W_ERROR(0x00000718)
#define WERR_RPC_S_NO_INTERFACES	W_ERROR(0x00000719)
#define WERR_RPC_S_CALL_CANCELLED	W_ERROR(0x0000071A)
#define WERR_RPC_S_BINDING_INCOMPLETE	W_ERROR(0x0000071B)
#define WERR_RPC_S_COMM_FAILURE	W_ERROR(0x0000071C)
#define WERR_RPC_S_UNSUPPORTED_AUTHN_LEVEL	W_ERROR(0x0000071D)
#define WERR_RPC_S_NO_PRINC_NAME	W_ERROR(0x0000071E)
#define WERR_RPC_S_NOT_RPC_ERROR	W_ERROR(0x0000071F)
#define WERR_RPC_S_UUID_LOCAL_ONLY	W_ERROR(0x00000720)
#define WERR_RPC_S_SEC_PKG_ERROR	W_ERROR(0x00000721)
#define WERR_RPC_S_NOT_CANCELLED	W_ERROR(0x00000722)
#define WERR_RPC_X_INVALID_ES_ACTION	W_ERROR(0x00000723)
#define WERR_RPC_X_WRONG_ES_VERSION	W_ERROR(0x00000724)
#define WERR_RPC_X_WRONG_STUB_VERSION	W_ERROR(0x00000725)
#define WERR_RPC_X_INVALID_PIPE_OBJECT	W_ERROR(0x00000726)
#define WERR_RPC_X_WRONG_PIPE_ORDER	W_ERROR(0x00000727)
#define WERR_RPC_X_WRONG_PIPE_VERSION	W_ERROR(0x00000728)
#define WERR_RPC_S_GROUP_MEMBER_NOT_FOUND	W_ERROR(0x0000076A)
#define WERR_EPT_S_CANT_CREATE	W_ERROR(0x0000076B)
#define WERR_RPC_S_INVALID_OBJECT	W_ERROR(0x0000076C)
#define WERR_INVALID_TIME	W_ERROR(0x0000076D)
#define WERR_INVALID_FORM_NAME	W_ERROR(0x0000076E)
#define WERR_INVALID_FORM_SIZE	W_ERROR(0x0000076F)
#define WERR_ALREADY_WAITING	W_ERROR(0x00000770)
#define WERR_PRINTER_DELETED	W_ERROR(0x00000771)
#define WERR_INVALID_PRINTER_STATE	W_ERROR(0x00000772)
#define WERR_PASSWORD_MUST_CHANGE	W_ERROR(0x00000773)
#define WERR_DOMAIN_CONTROLLER_NOT_FOUND	W_ERROR(0x00000774)
#define WERR_ACCOUNT_LOCKED_OUT	W_ERROR(0x00000775)
#define WERR_OR_INVALID_OXID	W_ERROR(0x00000776)
#define WERR_OR_INVALID_OID	W_ERROR(0x00000777)
#define WERR_OR_INVALID_SET	W_ERROR(0x00000778)
#define WERR_RPC_S_SEND_INCOMPLETE	W_ERROR(0x00000779)
#define WERR_RPC_S_INVALID_ASYNC_HANDLE	W_ERROR(0x0000077A)
#define WERR_RPC_S_INVALID_ASYNC_CALL	W_ERROR(0x0000077B)
#define WERR_RPC_X_PIPE_CLOSED	W_ERROR(0x0000077C)
#define WERR_RPC_X_PIPE_DISCIPLINE_ERROR	W_ERROR(0x0000077D)
#define WERR_RPC_X_PIPE_EMPTY	W_ERROR(0x0000077E)
#define WERR_NO_SITENAME	W_ERROR(0x0000077F)
#define WERR_CANT_ACCESS_FILE	W_ERROR(0x00000780)
#define WERR_CANT_RESOLVE_FILENAME	W_ERROR(0x00000781)
#define WERR_RPC_S_ENTRY_TYPE_MISMATCH	W_ERROR(0x00000782)
#define WERR_RPC_S_NOT_ALL_OBJS_EXPORTED	W_ERROR(0x00000783)
#define WERR_RPC_S_INTERFACE_NOT_EXPORTED	W_ERROR(0x00000784)
#define WERR_RPC_S_PROFILE_NOT_ADDED	W_ERROR(0x00000785)
#define WERR_RPC_S_PRF_ELT_NOT_ADDED	W_ERROR(0x00000786)
#define WERR_RPC_S_PRF_ELT_NOT_REMOVED	W_ERROR(0x00000787)
#define WERR_RPC_S_GRP_ELT_NOT_ADDED	W_ERROR(0x00000788)
#define WERR_RPC_S_GRP_ELT_NOT_REMOVED	W_ERROR(0x00000789)
#define WERR_KM_DRIVER_BLOCKED	W_ERROR(0x0000078A)
#define WERR_CONTEXT_EXPIRED	W_ERROR(0x0000078B)
#define WERR_PER_USER_TRUST_QUOTA_EXCEEDED	W_ERROR(0x0000078C)
#define WERR_ALL_USER_TRUST_QUOTA_EXCEEDED	W_ERROR(0x0000078D)
#define WERR_USER_DELETE_TRUST_QUOTA_EXCEEDED	W_ERROR(0x0000078E)
#define WERR_AUTHENTICATION_FIREWALL_FAILED	W_ERROR(0x0000078F)
#define WERR_REMOTE_PRINT_CONNECTIONS_BLOCKED	W_ERROR(0x00000790)
#define WERR_INVALID_PIXEL_FORMAT	W_ERROR(0x000007D0)
#define WERR_BAD_DRIVER	W_ERROR(0x000007D1)
#define WERR_INVALID_WINDOW_STYLE	W_ERROR(0x000007D2)
#define WERR_METAFILE_NOT_SUPPORTED	W_ERROR(0x000007D3)
#define WERR_TRANSFORM_NOT_SUPPORTED	W_ERROR(0x000007D4)
#define WERR_CLIPPING_NOT_SUPPORTED	W_ERROR(0x000007D5)
#define WERR_INVALID_CMM	W_ERROR(0x000007DA)
#define WERR_INVALID_PROFILE	W_ERROR(0x000007DB)
#define WERR_TAG_NOT_FOUND	W_ERROR(0x000007DC)
#define WERR_TAG_NOT_PRESENT	W_ERROR(0x000007DD)
#define WERR_DUPLICATE_TAG	W_ERROR(0x000007DE)
#define WERR_PROFILE_NOT_ASSOCIATED_WITH_DEVICE	W_ERROR(0x000007DF)
#define WERR_PROFILE_NOT_FOUND	W_ERROR(0x000007E0)
#define WERR_INVALID_COLORSPACE	W_ERROR(0x000007E1)
#define WERR_ICM_NOT_ENABLED	W_ERROR(0x000007E2)
#define WERR_DELETING_ICM_XFORM	W_ERROR(0x000007E3)
#define WERR_INVALID_TRANSFORM	W_ERROR(0x000007E4)
#define WERR_COLORSPACE_MISMATCH	W_ERROR(0x000007E5)
#define WERR_INVALID_COLORINDEX	W_ERROR(0x000007E6)
#define WERR_PROFILE_DOES_NOT_MATCH_DEVICE	W_ERROR(0x000007E7)
#define WERR_NERR_NETNOTSTARTED	W_ERROR(0x00000836)
#define WERR_NERR_UNKNOWNSERVER	W_ERROR(0x00000837)
#define WERR_NERR_SHAREMEM	W_ERROR(0x00000838)
#define WERR_NERR_NONETWORKRESOURCE	W_ERROR(0x00000839)
#define WERR_NERR_REMOTEONLY	W_ERROR(0x0000083A)
#define WERR_NERR_DEVNOTREDIRECTED	W_ERROR(0x0000083B)
#define WERR_CONNECTED_OTHER_PASSWORD	W_ERROR(0x0000083C)
#define WERR_CONNECTED_OTHER_PASSWORD_DEFAULT	W_ERROR(0x0000083D)
#define WERR_NERR_SERVERNOTSTARTED	W_ERROR(0x00000842)
#define WERR_NERR_ITEMNOTFOUND	W_ERROR(0x00000843)
#define WERR_NERR_UNKNOWNDEVDIR	W_ERROR(0x00000844)
#define WERR_NERR_REDIRECTEDPATH	W_ERROR(0x00000845)
#define WERR_NERR_DUPLICATESHARE	W_ERROR(0x00000846)
#define WERR_NERR_NOROOM	W_ERROR(0x00000847)
#define WERR_NERR_TOOMANYITEMS	W_ERROR(0x00000849)
#define WERR_NERR_INVALIDMAXUSERS	W_ERROR(0x0000084A)
#define WERR_NERR_BUFTOOSMALL	W_ERROR(0x0000084B)
#define WERR_NERR_REMOTEERR	W_ERROR(0x0000084F)
#define WERR_NERR_LANMANINIERROR	W_ERROR(0x00000853)
#define WERR_NERR_NETWORKERROR	W_ERROR(0x00000858)
#define WERR_NERR_WKSTAINCONSISTENTSTATE	W_ERROR(0x00000859)
#define WERR_NERR_WKSTANOTSTARTED	W_ERROR(0x0000085A)
#define WERR_NERR_BROWSERNOTSTARTED	W_ERROR(0x0000085B)
#define WERR_NERR_INTERNALERROR	W_ERROR(0x0000085C)
#define WERR_NERR_BADTRANSACTCONFIG	W_ERROR(0x0000085D)
#define WERR_NERR_INVALIDAPI	W_ERROR(0x0000085E)
#define WERR_NERR_BADEVENTNAME	W_ERROR(0x0000085F)
#define WERR_NERR_DUPNAMEREBOOT	W_ERROR(0x00000860)
#define WERR_NERR_CFGCOMPNOTFOUND	W_ERROR(0x00000862)
#define WERR_NERR_CFGPARAMNOTFOUND	W_ERROR(0x00000863)
#define WERR_NERR_LINETOOLONG	W_ERROR(0x00000865)
#define WERR_NERR_QNOTFOUND	W_ERROR(0x00000866)
#define WERR_NERR_JOBNOTFOUND	W_ERROR(0x00000867)
#define WERR_NERR_DESTNOTFOUND	W_ERROR(0x00000868)
#define WERR_NERR_DESTEXISTS	W_ERROR(0x00000869)
#define WERR_NERR_QEXISTS	W_ERROR(0x0000086A)
#define WERR_NERR_QNOROOM	W_ERROR(0x0000086B)
#define WERR_NERR_JOBNOROOM	W_ERROR(0x0000086C)
#define WERR_NERR_DESTNOROOM	W_ERROR(0x0000086D)
#define WERR_NERR_DESTIDLE	W_ERROR(0x0000086E)
#define WERR_NERR_DESTINVALIDOP	W_ERROR(0x0000086F)
#define WERR_NERR_PROCNORESPOND	W_ERROR(0x00000870)
#define WERR_NERR_SPOOLERNOTLOADED	W_ERROR(0x00000871)
#define WERR_NERR_DESTINVALIDSTATE	W_ERROR(0x00000872)
#define WERR_NERR_QINVALIDSTATE	W_ERROR(0x00000873)
#define WERR_NERR_JOBINVALIDSTATE	W_ERROR(0x00000874)
#define WERR_NERR_SPOOLNOMEMORY	W_ERROR(0x00000875)
#define WERR_NERR_DRIVERNOTFOUND	W_ERROR(0x00000876)
#define WERR_NERR_DATATYPEINVALID	W_ERROR(0x00000877)
#define WERR_NERR_PROCNOTFOUND	W_ERROR(0x00000878)
#define WERR_NERR_SERVICETABLELOCKED	W_ERROR(0x00000884)
#define WERR_NERR_SERVICETABLEFULL	W_ERROR(0x00000885)
#define WERR_NERR_SERVICEINSTALLED	W_ERROR(0x00000886)
#define WERR_NERR_SERVICEENTRYLOCKED	W_ERROR(0x00000887)
#define WERR_NERR_SERVICENOTINSTALLED	W_ERROR(0x00000888)
#define WERR_NERR_BADSERVICENAME	W_ERROR(0x00000889)
#define WERR_NERR_SERVICECTLTIMEOUT	W_ERROR(0x0000088A)
#define WERR_NERR_SERVICECTLBUSY	W_ERROR(0x0000088B)
#define WERR_NERR_BADSERVICEPROGNAME	W_ERROR(0x0000088C)
#define WERR_NERR_SERVICENOTCTRL	W_ERROR(0x0000088D)
#define WERR_NERR_SERVICEKILLPROC	W_ERROR(0x0000088E)
#define WERR_NERR_SERVICECTLNOTVALID	W_ERROR(0x0000088F)
#define WERR_NERR_NOTINDISPATCHTBL	W_ERROR(0x00000890)
#define WERR_NERR_BADCONTROLRECV	W_ERROR(0x00000891)
#define WERR_NERR_SERVICENOTSTARTING	W_ERROR(0x00000892)
#define WERR_NERR_ALREADYLOGGEDON	W_ERROR(0x00000898)
#define WERR_NERR_NOTLOGGEDON	W_ERROR(0x00000899)
#define WERR_NERR_BADUSERNAME	W_ERROR(0x0000089A)
#define WERR_NERR_BADPASSWORD	W_ERROR(0x0000089B)
#define WERR_NERR_UNABLETOADDNAME_W	W_ERROR(0x0000089C)
#define WERR_NERR_UNABLETOADDNAME_F	W_ERROR(0x0000089D)
#define WERR_NERR_UNABLETODELNAME_W	W_ERROR(0x0000089E)
#define WERR_NERR_UNABLETODELNAME_F	W_ERROR(0x0000089F)
#define WERR_NERR_LOGONSPAUSED	W_ERROR(0x000008A1)
#define WERR_NERR_LOGONSERVERCONFLICT	W_ERROR(0x000008A2)
#define WERR_NERR_LOGONNOUSERPATH	W_ERROR(0x000008A3)
#define WERR_NERR_LOGONSCRIPTERROR	W_ERROR(0x000008A4)
#define WERR_NERR_STANDALONELOGON	W_ERROR(0x000008A6)
#define WERR_NERR_LOGONSERVERNOTFOUND	W_ERROR(0x000008A7)
#define WERR_NERR_LOGONDOMAINEXISTS	W_ERROR(0x000008A8)
#define WERR_NERR_NONVALIDATEDLOGON	W_ERROR(0x000008A9)
#define WERR_NERR_ACFNOTFOUND	W_ERROR(0x000008AB)
#define WERR_NERR_GROUPNOTFOUND	W_ERROR(0x000008AC)
#define WERR_NERR_USERNOTFOUND	W_ERROR(0x000008AD)
#define WERR_NERR_RESOURCENOTFOUND	W_ERROR(0x000008AE)
#define WERR_NERR_GROUPEXISTS	W_ERROR(0x000008AF)
#define WERR_NERR_USEREXISTS	W_ERROR(0x000008B0)
#define WERR_NERR_RESOURCEEXISTS	W_ERROR(0x000008B1)
#define WERR_NERR_NOTPRIMARY	W_ERROR(0x000008B2)
#define WERR_NERR_ACFNOTLOADED	W_ERROR(0x000008B3)
#define WERR_NERR_ACFNOROOM	W_ERROR(0x000008B4)
#define WERR_NERR_ACFFILEIOFAIL	W_ERROR(0x000008B5)
#define WERR_NERR_ACFTOOMANYLISTS	W_ERROR(0x000008B6)
#define WERR_NERR_USERLOGON	W_ERROR(0x000008B7)
#define WERR_NERR_ACFNOPARENT	W_ERROR(0x000008B8)
#define WERR_NERR_CANNOTGROWSEGMENT	W_ERROR(0x000008B9)
#define WERR_NERR_SPEGROUPOP	W_ERROR(0x000008BA)
#define WERR_NERR_NOTINCACHE	W_ERROR(0x000008BB)
#define WERR_NERR_USERINGROUP	W_ERROR(0x000008BC)
#define WERR_NERR_USERNOTINGROUP	W_ERROR(0x000008BD)
#define WERR_NERR_ACCOUNTUNDEFINED	W_ERROR(0x000008BE)
#define WERR_NERR_ACCOUNTEXPIRED	W_ERROR(0x000008BF)
#define WERR_NERR_INVALIDWORKSTATION	W_ERROR(0x000008C0)
#define WERR_NERR_INVALIDLOGONHOURS	W_ERROR(0x000008C1)
#define WERR_NERR_PASSWORDEXPIRED	W_ERROR(0x000008C2)
#define WERR_NERR_PASSWORDCANTCHANGE	W_ERROR(0x000008C3)
#define WERR_NERR_PASSWORDHISTCONFLICT	W_ERROR(0x000008C4)
#define WERR_NERR_PASSWORDTOOSHORT	W_ERROR(0x000008C5)
#define WERR_NERR_PASSWORDTOORECENT	W_ERROR(0x000008C6)
#define WERR_NERR_INVALIDDATABASE	W_ERROR(0x000008C7)
#define WERR_NERR_DATABASEUPTODATE	W_ERROR(0x000008C8)
#define WERR_NERR_SYNCREQUIRED	W_ERROR(0x000008C9)
#define WERR_NERR_USENOTFOUND	W_ERROR(0x000008CA)
#define WERR_NERR_BADASGTYPE	W_ERROR(0x000008CB)
#define WERR_NERR_DEVICEISSHARED	W_ERROR(0x000008CC)
#define WERR_NERR_NOCOMPUTERNAME	W_ERROR(0x000008DE)
#define WERR_NERR_MSGALREADYSTARTED	W_ERROR(0x000008DF)
#define WERR_NERR_MSGINITFAILED	W_ERROR(0x000008E0)
#define WERR_NERR_NAMENOTFOUND	W_ERROR(0x000008E1)
#define WERR_NERR_ALREADYFORWARDED	W_ERROR(0x000008E2)
#define WERR_NERR_ADDFORWARDED	W_ERROR(0x000008E3)
#define WERR_NERR_ALREADYEXISTS	W_ERROR(0x000008E4)
#define WERR_NERR_TOOMANYNAMES	W_ERROR(0x000008E5)
#define WERR_NERR_DELCOMPUTERNAME	W_ERROR(0x000008E6)
#define WERR_NERR_LOCALFORWARD	W_ERROR(0x000008E7)
#define WERR_NERR_GRPMSGPROCESSOR	W_ERROR(0x000008E8)
#define WERR_NERR_PAUSEDREMOTE	W_ERROR(0x000008E9)
#define WERR_NERR_BADRECEIVE	W_ERROR(0x000008EA)
#define WERR_NERR_NAMEINUSE	W_ERROR(0x000008EB)
#define WERR_NERR_MSGNOTSTARTED	W_ERROR(0x000008EC)
#define WERR_NERR_NOTLOCALNAME	W_ERROR(0x000008ED)
#define WERR_NERR_NOFORWARDNAME	W_ERROR(0x000008EE)
#define WERR_NERR_REMOTEFULL	W_ERROR(0x000008EF)
#define WERR_NERR_NAMENOTFORWARDED	W_ERROR(0x000008F0)
#define WERR_NERR_TRUNCATEDBROADCAST	W_ERROR(0x000008F1)
#define WERR_NERR_INVALIDDEVICE	W_ERROR(0x000008F6)
#define WERR_NERR_WRITEFAULT	W_ERROR(0x000008F7)
#define WERR_NERR_DUPLICATENAME	W_ERROR(0x000008F9)
#define WERR_NERR_DELETELATER	W_ERROR(0x000008FA)
#define WERR_NERR_INCOMPLETEDEL	W_ERROR(0x000008FB)
#define WERR_NERR_MULTIPLENETS	W_ERROR(0x000008FC)
#define WERR_NERR_NETNAMENOTFOUND	W_ERROR(0x00000906)
#define WERR_NERR_DEVICENOTSHARED	W_ERROR(0x00000907)
#define WERR_NERR_CLIENTNAMENOTFOUND	W_ERROR(0x00000908)
#define WERR_NERR_FILEIDNOTFOUND	W_ERROR(0x0000090A)
#define WERR_NERR_EXECFAILURE	W_ERROR(0x0000090B)
#define WERR_NERR_TMPFILE	W_ERROR(0x0000090C)
#define WERR_NERR_TOOMUCHDATA	W_ERROR(0x0000090D)
#define WERR_NERR_DEVICESHARECONFLICT	W_ERROR(0x0000090E)
#define WERR_NERR_BROWSERTABLEINCOMPLETE	W_ERROR(0x0000090F)
#define WERR_NERR_NOTLOCALDOMAIN	W_ERROR(0x00000910)
#define WERR_NERR_ISDFSSHARE	W_ERROR(0x00000911)
#define WERR_NERR_DEVINVALIDOPCODE	W_ERROR(0x0000091B)
#define WERR_NERR_DEVNOTFOUND	W_ERROR(0x0000091C)
#define WERR_NERR_DEVNOTOPEN	W_ERROR(0x0000091D)
#define WERR_NERR_BADQUEUEDEVSTRING	W_ERROR(0x0000091E)
#define WERR_NERR_BADQUEUEPRIORITY	W_ERROR(0x0000091F)
#define WERR_NERR_NOCOMMDEVS	W_ERROR(0x00000921)
#define WERR_NERR_QUEUENOTFOUND	W_ERROR(0x00000922)
#define WERR_NERR_BADDEVSTRING	W_ERROR(0x00000924)
#define WERR_NERR_BADDEV	W_ERROR(0x00000925)
#define WERR_NERR_INUSEBYSPOOLER	W_ERROR(0x00000926)
#define WERR_NERR_COMMDEVINUSE	W_ERROR(0x00000927)
#define WERR_NERR_INVALIDCOMPUTER	W_ERROR(0x0000092F)
#define WERR_NERR_MAXLENEXCEEDED	W_ERROR(0x00000932)
#define WERR_NERR_BADCOMPONENT	W_ERROR(0x00000934)
#define WERR_NERR_CANTTYPE	W_ERROR(0x00000935)
#define WERR_NERR_TOOMANYENTRIES	W_ERROR(0x0000093A)
#define WERR_NERR_PROFILEFILETOOBIG	W_ERROR(0x00000942)
#define WERR_NERR_PROFILEOFFSET	W_ERROR(0x00000943)
#define WERR_NERR_PROFILECLEANUP	W_ERROR(0x00000944)
#define WERR_NERR_PROFILEUNKNOWNCMD	W_ERROR(0x00000945)
#define WERR_NERR_PROFILELOADERR	W_ERROR(0x00000946)
#define WERR_NERR_PROFILESAVEERR	W_ERROR(0x00000947)
#define WERR_NERR_LOGOVERFLOW	W_ERROR(0x00000949)
#define WERR_NERR_LOGFILECHANGED	W_ERROR(0x0000094A)
#define WERR_NERR_LOGFILECORRUPT	W_ERROR(0x0000094B)
#define WERR_NERR_SOURCEISDIR	W_ERROR(0x0000094C)
#define WERR_NERR_BADSOURCE	W_ERROR(0x0000094D)
#define WERR_NERR_BADDEST	W_ERROR(0x0000094E)
#define WERR_NERR_DIFFERENTSERVERS	W_ERROR(0x0000094F)
#define WERR_NERR_RUNSRVPAUSED	W_ERROR(0x00000951)
#define WERR_NERR_ERRCOMMRUNSRV	W_ERROR(0x00000955)
#define WERR_NERR_ERROREXECINGGHOST	W_ERROR(0x00000957)
#define WERR_NERR_SHARENOTFOUND	W_ERROR(0x00000958)
#define WERR_NERR_INVALIDLANA	W_ERROR(0x00000960)
#define WERR_NERR_OPENFILES	W_ERROR(0x00000961)
#define WERR_NERR_ACTIVECONNS	W_ERROR(0x00000962)
#define WERR_NERR_BADPASSWORDCORE	W_ERROR(0x00000963)
#define WERR_NERR_DEVINUSE	W_ERROR(0x00000964)
#define WERR_NERR_LOCALDRIVE	W_ERROR(0x00000965)
#define WERR_NERR_ALERTEXISTS	W_ERROR(0x0000097E)
#define WERR_NERR_TOOMANYALERTS	W_ERROR(0x0000097F)
#define WERR_NERR_NOSUCHALERT	W_ERROR(0x00000980)
#define WERR_NERR_BADRECIPIENT	W_ERROR(0x00000981)
#define WERR_NERR_ACCTLIMITEXCEEDED	W_ERROR(0x00000982)
#define WERR_NERR_INVALIDLOGSEEK	W_ERROR(0x00000988)
#define WERR_NERR_BADUASCONFIG	W_ERROR(0x00000992)
#define WERR_NERR_INVALIDUASOP	W_ERROR(0x00000993)
#define WERR_NERR_LASTADMIN	W_ERROR(0x00000994)
#define WERR_NERR_DCNOTFOUND	W_ERROR(0x00000995)
#define WERR_NERR_LOGONTRACKINGERROR	W_ERROR(0x00000996)
#define WERR_NERR_NETLOGONNOTSTARTED	W_ERROR(0x00000997)
#define WERR_NERR_CANNOTGROWUASFILE	W_ERROR(0x00000998)
#define WERR_NERR_TIMEDIFFATDC	W_ERROR(0x00000999)
#define WERR_NERR_PASSWORDMISMATCH	W_ERROR(0x0000099A)
#define WERR_NERR_NOSUCHSERVER	W_ERROR(0x0000099C)
#define WERR_NERR_NOSUCHSESSION	W_ERROR(0x0000099D)
#define WERR_NERR_NOSUCHCONNECTION	W_ERROR(0x0000099E)
#define WERR_NERR_TOOMANYSERVERS	W_ERROR(0x0000099F)
#define WERR_NERR_TOOMANYSESSIONS	W_ERROR(0x000009A0)
#define WERR_NERR_TOOMANYCONNECTIONS	W_ERROR(0x000009A1)
#define WERR_NERR_TOOMANYFILES	W_ERROR(0x000009A2)
#define WERR_NERR_NOALTERNATESERVERS	W_ERROR(0x000009A3)
#define WERR_NERR_TRYDOWNLEVEL	W_ERROR(0x000009A6)
#define WERR_NERR_UPSDRIVERNOTSTARTED	W_ERROR(0x000009B0)
#define WERR_NERR_UPSINVALIDCONFIG	W_ERROR(0x000009B1)
#define WERR_NERR_UPSINVALIDCOMMPORT	W_ERROR(0x000009B2)
#define WERR_NERR_UPSSIGNALASSERTED	W_ERROR(0x000009B3)
#define WERR_NERR_UPSSHUTDOWNFAILED	W_ERROR(0x000009B4)
#define WERR_NERR_BADDOSRETCODE	W_ERROR(0x000009C4)
#define WERR_NERR_PROGNEEDSEXTRAMEM	W_ERROR(0x000009C5)
#define WERR_NERR_BADDOSFUNCTION	W_ERROR(0x000009C6)
#define WERR_NERR_REMOTEBOOTFAILED	W_ERROR(0x000009C7)
#define WERR_NERR_BADFILECHECKSUM	W_ERROR(0x000009C8)
#define WERR_NERR_NORPLBOOTSYSTEM	W_ERROR(0x000009C9)
#define WERR_NERR_RPLLOADRNETBIOSERR	W_ERROR(0x000009CA)
#define WERR_NERR_RPLLOADRDISKERR	W_ERROR(0x000009CB)
#define WERR_NERR_IMAGEPARAMERR	W_ERROR(0x000009CC)
#define WERR_NERR_TOOMANYIMAGEPARAMS	W_ERROR(0x000009CD)
#define WERR_NERR_NONDOSFLOPPYUSED	W_ERROR(0x000009CE)
#define WERR_NERR_RPLBOOTRESTART	W_ERROR(0x000009CF)
#define WERR_NERR_RPLSRVRCALLFAILED	W_ERROR(0x000009D0)
#define WERR_NERR_CANTCONNECTRPLSRVR	W_ERROR(0x000009D1)
#define WERR_NERR_CANTOPENIMAGEFILE	W_ERROR(0x000009D2)
#define WERR_NERR_CALLINGRPLSRVR	W_ERROR(0x000009D3)
#define WERR_NERR_STARTINGRPLBOOT	W_ERROR(0x000009D4)
#define WERR_NERR_RPLBOOTSERVICETERM	W_ERROR(0x000009D5)
#define WERR_NERR_RPLBOOTSTARTFAILED	W_ERROR(0x000009D6)
#define WERR_NERR_RPL_CONNECTED	W_ERROR(0x000009D7)
#define WERR_NERR_BROWSERCONFIGUREDTONOTRUN	W_ERROR(0x000009F6)
#define WERR_NERR_RPLNOADAPTERSSTARTED	W_ERROR(0x00000A32)
#define WERR_NERR_RPLBADREGISTRY	W_ERROR(0x00000A33)
#define WERR_NERR_RPLBADDATABASE	W_ERROR(0x00000A34)
#define WERR_NERR_RPLRPLFILESSHARE	W_ERROR(0x00000A35)
#define WERR_NERR_RPLNOTRPLSERVER	W_ERROR(0x00000A36)
#define WERR_NERR_RPLCANNOTENUM	W_ERROR(0x00000A37)
#define WERR_NERR_RPLWKSTAINFOCORRUPTED	W_ERROR(0x00000A38)
#define WERR_NERR_RPLWKSTANOTFOUND	W_ERROR(0x00000A39)
#define WERR_NERR_RPLWKSTANAMEUNAVAILABLE	W_ERROR(0x00000A3A)
#define WERR_NERR_RPLPROFILEINFOCORRUPTED	W_ERROR(0x00000A3B)
#define WERR_NERR_RPLPROFILENOTFOUND	W_ERROR(0x00000A3C)
#define WERR_NERR_RPLPROFILENAMEUNAVAILABLE	W_ERROR(0x00000A3D)
#define WERR_NERR_RPLPROFILENOTEMPTY	W_ERROR(0x00000A3E)
#define WERR_NERR_RPLCONFIGINFOCORRUPTED	W_ERROR(0x00000A3F)
#define WERR_NERR_RPLCONFIGNOTFOUND	W_ERROR(0x00000A40)
#define WERR_NERR_RPLADAPTERINFOCORRUPTED	W_ERROR(0x00000A41)
#define WERR_NERR_RPLINTERNAL	W_ERROR(0x00000A42)
#define WERR_NERR_RPLVENDORINFOCORRUPTED	W_ERROR(0x00000A43)
#define WERR_NERR_RPLBOOTINFOCORRUPTED	W_ERROR(0x00000A44)
#define WERR_NERR_RPLWKSTANEEDSUSERACCT	W_ERROR(0x00000A45)
#define WERR_NERR_RPLNEEDSRPLUSERACCT	W_ERROR(0x00000A46)
#define WERR_NERR_RPLBOOTNOTFOUND	W_ERROR(0x00000A47)
#define WERR_NERR_RPLINCOMPATIBLEPROFILE	W_ERROR(0x00000A48)
#define WERR_NERR_RPLADAPTERNAMEUNAVAILABLE	W_ERROR(0x00000A49)
#define WERR_NERR_RPLCONFIGNOTEMPTY	W_ERROR(0x00000A4A)
#define WERR_NERR_RPLBOOTINUSE	W_ERROR(0x00000A4B)
#define WERR_NERR_RPLBACKUPDATABASE	W_ERROR(0x00000A4C)
#define WERR_NERR_RPLADAPTERNOTFOUND	W_ERROR(0x00000A4D)
#define WERR_NERR_RPLVENDORNOTFOUND	W_ERROR(0x00000A4E)
#define WERR_NERR_RPLVENDORNAMEUNAVAILABLE	W_ERROR(0x00000A4F)
#define WERR_NERR_RPLBOOTNAMEUNAVAILABLE	W_ERROR(0x00000A50)
#define WERR_NERR_RPLCONFIGNAMEUNAVAILABLE	W_ERROR(0x00000A51)
#define WERR_NERR_DFSINTERNALCORRUPTION	W_ERROR(0x00000A64)
#define WERR_NERR_DFSVOLUMEDATACORRUPT	W_ERROR(0x00000A65)
#define WERR_NERR_DFSNOSUCHVOLUME	W_ERROR(0x00000A66)
#define WERR_NERR_DFSVOLUMEALREADYEXISTS	W_ERROR(0x00000A67)
#define WERR_NERR_DFSALREADYSHARED	W_ERROR(0x00000A68)
#define WERR_NERR_DFSNOSUCHSHARE	W_ERROR(0x00000A69)
#define WERR_NERR_DFSNOTALEAFVOLUME	W_ERROR(0x00000A6A)
#define WERR_NERR_DFSLEAFVOLUME	W_ERROR(0x00000A6B)
#define WERR_NERR_DFSVOLUMEHASMULTIPLESERVERS	W_ERROR(0x00000A6C)
#define WERR_NERR_DFSCANTCREATEJUNCTIONPOINT	W_ERROR(0x00000A6D)
#define WERR_NERR_DFSSERVERNOTDFSAWARE	W_ERROR(0x00000A6E)
#define WERR_NERR_DFSBADRENAMEPATH	W_ERROR(0x00000A6F)
#define WERR_NERR_DFSVOLUMEISOFFLINE	W_ERROR(0x00000A70)
#define WERR_NERR_DFSNOSUCHSERVER	W_ERROR(0x00000A71)
#define WERR_NERR_DFSCYCLICALNAME	W_ERROR(0x00000A72)
#define WERR_NERR_DFSNOTSUPPORTEDINSERVERDFS	W_ERROR(0x00000A73)
#define WERR_NERR_DFSDUPLICATESERVICE	W_ERROR(0x00000A74)
#define WERR_NERR_DFSCANTREMOVELASTSERVERSHARE	W_ERROR(0x00000A75)
#define WERR_NERR_DFSVOLUMEISINTERDFS	W_ERROR(0x00000A76)
#define WERR_NERR_DFSINCONSISTENT	W_ERROR(0x00000A77)
#define WERR_NERR_DFSSERVERUPGRADED	W_ERROR(0x00000A78)
#define WERR_NERR_DFSDATAISIDENTICAL	W_ERROR(0x00000A79)
#define WERR_NERR_DFSCANTREMOVEDFSROOT	W_ERROR(0x00000A7A)
#define WERR_NERR_DFSCHILDORPARENTINDFS	W_ERROR(0x00000A7B)
#define WERR_NERR_DFSINTERNALERROR	W_ERROR(0x00000A82)
#define WERR_NERR_SETUPALREADYJOINED	W_ERROR(0x00000A83)
#define WERR_NERR_SETUPNOTJOINED	W_ERROR(0x00000A84)
#define WERR_NERR_SETUPDOMAINCONTROLLER	W_ERROR(0x00000A85)
#define WERR_NERR_DEFAULTJOINREQUIRED	W_ERROR(0x00000A86)
#define WERR_NERR_INVALIDWORKGROUPNAME	W_ERROR(0x00000A87)
#define WERR_NERR_NAMEUSESINCOMPATIBLECODEPAGE	W_ERROR(0x00000A88)
#define WERR_NERR_COMPUTERACCOUNTNOTFOUND	W_ERROR(0x00000A89)
#define WERR_NERR_PERSONALSKU	W_ERROR(0x00000A8A)
#define WERR_NERR_PASSWORDMUSTCHANGE	W_ERROR(0x00000A8D)
#define WERR_NERR_ACCOUNTLOCKEDOUT	W_ERROR(0x00000A8E)
#define WERR_NERR_PASSWORDTOOLONG	W_ERROR(0x00000A8F)
#define WERR_NERR_PASSWORDNOTCOMPLEXENOUGH	W_ERROR(0x00000A90)
#define WERR_NERR_PASSWORDFILTERERROR	W_ERROR(0x00000A91)
#define WERR_UNKNOWN_PRINT_MONITOR	W_ERROR(0x00000BB8)
#define WERR_PRINTER_DRIVER_IN_USE	W_ERROR(0x00000BB9)
#define WERR_SPOOL_FILE_NOT_FOUND	W_ERROR(0x00000BBA)
#define WERR_SPL_NO_STARTDOC	W_ERROR(0x00000BBB)
#define WERR_SPL_NO_ADDJOB	W_ERROR(0x00000BBC)
#define WERR_PRINT_PROCESSOR_ALREADY_INSTALLED	W_ERROR(0x00000BBD)
#define WERR_PRINT_MONITOR_ALREADY_INSTALLED	W_ERROR(0x00000BBE)
#define WERR_INVALID_PRINT_MONITOR	W_ERROR(0x00000BBF)
#define WERR_PRINT_MONITOR_IN_USE	W_ERROR(0x00000BC0)
#define WERR_PRINTER_HAS_JOBS_QUEUED	W_ERROR(0x00000BC1)
#define WERR_SUCCESS_REBOOT_REQUIRED	W_ERROR(0x00000BC2)
#define WERR_SUCCESS_RESTART_REQUIRED	W_ERROR(0x00000BC3)
#define WERR_PRINTER_NOT_FOUND	W_ERROR(0x00000BC4)
#define WERR_PRINTER_DRIVER_WARNED	W_ERROR(0x00000BC5)
#define WERR_PRINTER_DRIVER_BLOCKED	W_ERROR(0x00000BC6)
#define WERR_PRINTER_DRIVER_PACKAGE_IN_USE	W_ERROR(0x00000BC7)
#define WERR_CORE_DRIVER_PACKAGE_NOT_FOUND	W_ERROR(0x00000BC8)
#define WERR_FAIL_REBOOT_REQUIRED	W_ERROR(0x00000BC9)
#define WERR_FAIL_REBOOT_INITIATED	W_ERROR(0x00000BCA)
#define WERR_IO_REISSUE_AS_CACHED	W_ERROR(0x00000F6E)
#define WERR_WINS_INTERNAL	W_ERROR(0x00000FA0)
#define WERR_CAN_NOT_DEL_LOCAL_WINS	W_ERROR(0x00000FA1)
#define WERR_STATIC_INIT	W_ERROR(0x00000FA2)
#define WERR_INC_BACKUP	W_ERROR(0x00000FA3)
#define WERR_FULL_BACKUP	W_ERROR(0x00000FA4)
#define WERR_REC_NON_EXISTENT	W_ERROR(0x00000FA5)
#define WERR_RPL_NOT_ALLOWED	W_ERROR(0x00000FA6)
#define WERR_DHCP_ADDRESS_CONFLICT	W_ERROR(0x00001004)
#define WERR_WMI_GUID_NOT_FOUND	W_ERROR(0x00001068)
#define WERR_WMI_INSTANCE_NOT_FOUND	W_ERROR(0x00001069)
#define WERR_WMI_ITEMID_NOT_FOUND	W_ERROR(0x0000106A)
#define WERR_WMI_TRY_AGAIN	W_ERROR(0x0000106B)
#define WERR_WMI_DP_NOT_FOUND	W_ERROR(0x0000106C)
#define WERR_WMI_UNRESOLVED_INSTANCE_REF	W_ERROR(0x0000106D)
#define WERR_WMI_ALREADY_ENABLED	W_ERROR(0x0000106E)
#define WERR_WMI_GUID_DISCONNECTED	W_ERROR(0x0000106F)
#define WERR_WMI_SERVER_UNAVAILABLE	W_ERROR(0x00001070)
#define WERR_WMI_DP_FAILED	W_ERROR(0x00001071)
#define WERR_WMI_INVALID_MOF	W_ERROR(0x00001072)
#define WERR_WMI_INVALID_REGINFO	W_ERROR(0x00001073)
#define WERR_WMI_ALREADY_DISABLED	W_ERROR(0x00001074)
#define WERR_WMI_READ_ONLY	W_ERROR(0x00001075)
#define WERR_WMI_SET_FAILURE	W_ERROR(0x00001076)
#define WERR_INVALID_MEDIA	W_ERROR(0x000010CC)
#define WERR_INVALID_LIBRARY	W_ERROR(0x000010CD)
#define WERR_INVALID_MEDIA_POOL	W_ERROR(0x000010CE)
#define WERR_DRIVE_MEDIA_MISMATCH	W_ERROR(0x000010CF)
#define WERR_MEDIA_OFFLINE	W_ERROR(0x000010D0)
#define WERR_LIBRARY_OFFLINE	W_ERROR(0x000010D1)
#define WERR_EMPTY	W_ERROR(0x000010D2)
#define WERR_NOT_EMPTY	W_ERROR(0x000010D3)
#define WERR_MEDIA_UNAVAILABLE	W_ERROR(0x000010D4)
#define WERR_RESOURCE_DISABLED	W_ERROR(0x000010D5)
#define WERR_INVALID_CLEANER	W_ERROR(0x000010D6)
#define WERR_UNABLE_TO_CLEAN	W_ERROR(0x000010D7)
#define WERR_OBJECT_NOT_FOUND	W_ERROR(0x000010D8)
#define WERR_DATABASE_FAILURE	W_ERROR(0x000010D9)
#define WERR_DATABASE_FULL	W_ERROR(0x000010DA)
#define WERR_MEDIA_INCOMPATIBLE	W_ERROR(0x000010DB)
#define WERR_RESOURCE_NOT_PRESENT	W_ERROR(0x000010DC)
#define WERR_INVALID_OPERATION	W_ERROR(0x000010DD)
#define WERR_MEDIA_NOT_AVAILABLE	W_ERROR(0x000010DE)
#define WERR_DEVICE_NOT_AVAILABLE	W_ERROR(0x000010DF)
#define WERR_REQUEST_REFUSED	W_ERROR(0x000010E0)
#define WERR_INVALID_DRIVE_OBJECT	W_ERROR(0x000010E1)
#define WERR_LIBRARY_FULL	W_ERROR(0x000010E2)
#define WERR_MEDIUM_NOT_ACCESSIBLE	W_ERROR(0x000010E3)
#define WERR_UNABLE_TO_LOAD_MEDIUM	W_ERROR(0x000010E4)
#define WERR_UNABLE_TO_INVENTORY_DRIVE	W_ERROR(0x000010E5)
#define WERR_UNABLE_TO_INVENTORY_SLOT	W_ERROR(0x000010E6)
#define WERR_UNABLE_TO_INVENTORY_TRANSPORT	W_ERROR(0x000010E7)
#define WERR_TRANSPORT_FULL	W_ERROR(0x000010E8)
#define WERR_CONTROLLING_IEPORT	W_ERROR(0x000010E9)
#define WERR_UNABLE_TO_EJECT_MOUNTED_MEDIA	W_ERROR(0x000010EA)
#define WERR_CLEANER_SLOT_SET	W_ERROR(0x000010EB)
#define WERR_CLEANER_SLOT_NOT_SET	W_ERROR(0x000010EC)
#define WERR_CLEANER_CARTRIDGE_SPENT	W_ERROR(0x000010ED)
#define WERR_UNEXPECTED_OMID	W_ERROR(0x000010EE)
#define WERR_CANT_DELETE_LAST_ITEM	W_ERROR(0x000010EF)
#define WERR_MESSAGE_EXCEEDS_MAX_SIZE	W_ERROR(0x000010F0)
#define WERR_VOLUME_CONTAINS_SYS_FILES	W_ERROR(0x000010F1)
#define WERR_INDIGENOUS_TYPE	W_ERROR(0x000010F2)
#define WERR_NO_SUPPORTING_DRIVES	W_ERROR(0x000010F3)
#define WERR_CLEANER_CARTRIDGE_INSTALLED	W_ERROR(0x000010F4)
#define WERR_IEPORT_FULL	W_ERROR(0x000010F5)
#define WERR_FILE_OFFLINE	W_ERROR(0x000010FE)
#define WERR_REMOTE_STORAGE_NOT_ACTIVE	W_ERROR(0x000010FF)
#define WERR_REMOTE_STORAGE_MEDIA_ERROR	W_ERROR(0x00001100)
#define WERR_NOT_A_REPARSE_POINT	W_ERROR(0x00001126)
#define WERR_REPARSE_ATTRIBUTE_CONFLICT	W_ERROR(0x00001127)
#define WERR_INVALID_REPARSE_DATA	W_ERROR(0x00001128)
#define WERR_REPARSE_TAG_INVALID	W_ERROR(0x00001129)
#define WERR_REPARSE_TAG_MISMATCH	W_ERROR(0x0000112A)
#define WERR_VOLUME_NOT_SIS_ENABLED	W_ERROR(0x00001194)
#define WERR_DEPENDENT_RESOURCE_EXISTS	W_ERROR(0x00001389)
#define WERR_DEPENDENCY_NOT_FOUND	W_ERROR(0x0000138A)
#define WERR_DEPENDENCY_ALREADY_EXISTS	W_ERROR(0x0000138B)
#define WERR_RESOURCE_NOT_ONLINE	W_ERROR(0x0000138C)
#define WERR_HOST_NODE_NOT_AVAILABLE	W_ERROR(0x0000138D)
#define WERR_RESOURCE_NOT_AVAILABLE	W_ERROR(0x0000138E)
#define WERR_RESOURCE_NOT_FOUND	W_ERROR(0x0000138F)
#define WERR_SHUTDOWN_CLUSTER	W_ERROR(0x00001390)
#define WERR_CANT_EVICT_ACTIVE_NODE	W_ERROR(0x00001391)
#define WERR_OBJECT_ALREADY_EXISTS	W_ERROR(0x00001392)
#define WERR_OBJECT_IN_LIST	W_ERROR(0x00001393)
#define WERR_GROUP_NOT_AVAILABLE	W_ERROR(0x00001394)
#define WERR_GROUP_NOT_FOUND	W_ERROR(0x00001395)
#define WERR_GROUP_NOT_ONLINE	W_ERROR(0x00001396)
#define WERR_HOST_NODE_NOT_RESOURCE_OWNER	W_ERROR(0x00001397)
#define WERR_HOST_NODE_NOT_GROUP_OWNER	W_ERROR(0x00001398)
#define WERR_RESMON_CREATE_FAILED	W_ERROR(0x00001399)
#define WERR_RESMON_ONLINE_FAILED	W_ERROR(0x0000139A)
#define WERR_RESOURCE_ONLINE	W_ERROR(0x0000139B)
#define WERR_QUORUM_RESOURCE	W_ERROR(0x0000139C)
#define WERR_NOT_QUORUM_CAPABLE	W_ERROR(0x0000139D)
#define WERR_CLUSTER_SHUTTING_DOWN	W_ERROR(0x0000139E)
#define WERR_INVALID_STATE	W_ERROR(0x0000139F)
#define WERR_RESOURCE_PROPERTIES_STORED	W_ERROR(0x000013A0)
#define WERR_NOT_QUORUM_CLASS	W_ERROR(0x000013A1)
#define WERR_CORE_RESOURCE	W_ERROR(0x000013A2)
#define WERR_QUORUM_RESOURCE_ONLINE_FAILED	W_ERROR(0x000013A3)
#define WERR_QUORUMLOG_OPEN_FAILED	W_ERROR(0x000013A4)
#define WERR_CLUSTERLOG_CORRUPT	W_ERROR(0x000013A5)
#define WERR_CLUSTERLOG_RECORD_EXCEEDS_MAXSIZE	W_ERROR(0x000013A6)
#define WERR_CLUSTERLOG_EXCEEDS_MAXSIZE	W_ERROR(0x000013A7)
#define WERR_CLUSTERLOG_CHKPOINT_NOT_FOUND	W_ERROR(0x000013A8)
#define WERR_CLUSTERLOG_NOT_ENOUGH_SPACE	W_ERROR(0x000013A9)
#define WERR_QUORUM_OWNER_ALIVE	W_ERROR(0x000013AA)
#define WERR_NETWORK_NOT_AVAILABLE	W_ERROR(0x000013AB)
#define WERR_NODE_NOT_AVAILABLE	W_ERROR(0x000013AC)
#define WERR_ALL_NODES_NOT_AVAILABLE	W_ERROR(0x000013AD)
#define WERR_RESOURCE_FAILED	W_ERROR(0x000013AE)
#define WERR_CLUSTER_INVALID_NODE	W_ERROR(0x000013AF)
#define WERR_CLUSTER_NODE_EXISTS	W_ERROR(0x000013B0)
#define WERR_CLUSTER_JOIN_IN_PROGRESS	W_ERROR(0x000013B1)
#define WERR_CLUSTER_NODE_NOT_FOUND	W_ERROR(0x000013B2)
#define WERR_CLUSTER_LOCAL_NODE_NOT_FOUND	W_ERROR(0x000013B3)
#define WERR_CLUSTER_NETWORK_EXISTS	W_ERROR(0x000013B4)
#define WERR_CLUSTER_NETWORK_NOT_FOUND	W_ERROR(0x000013B5)
#define WERR_CLUSTER_NETINTERFACE_EXISTS	W_ERROR(0x000013B6)
#define WERR_CLUSTER_NETINTERFACE_NOT_FOUND	W_ERROR(0x000013B7)
#define WERR_CLUSTER_INVALID_REQUEST	W_ERROR(0x000013B8)
#define WERR_CLUSTER_INVALID_NETWORK_PROVIDER	W_ERROR(0x000013B9)
#define WERR_CLUSTER_NODE_DOWN	W_ERROR(0x000013BA)
#define WERR_CLUSTER_NODE_UNREACHABLE	W_ERROR(0x000013BB)
#define WERR_CLUSTER_NODE_NOT_MEMBER	W_ERROR(0x000013BC)
#define WERR_CLUSTER_JOIN_NOT_IN_PROGRESS	W_ERROR(0x000013BD)
#define WERR_CLUSTER_INVALID_NETWORK	W_ERROR(0x000013BE)
#define WERR_CLUSTER_NODE_UP	W_ERROR(0x000013C0)
#define WERR_CLUSTER_IPADDR_IN_USE	W_ERROR(0x000013C1)
#define WERR_CLUSTER_NODE_NOT_PAUSED	W_ERROR(0x000013C2)
#define WERR_CLUSTER_NO_SECURITY_CONTEXT	W_ERROR(0x000013C3)
#define WERR_CLUSTER_NETWORK_NOT_INTERNAL	W_ERROR(0x000013C4)
#define WERR_CLUSTER_NODE_ALREADY_UP	W_ERROR(0x000013C5)
#define WERR_CLUSTER_NODE_ALREADY_DOWN	W_ERROR(0x000013C6)
#define WERR_CLUSTER_NETWORK_ALREADY_ONLINE	W_ERROR(0x000013C7)
#define WERR_CLUSTER_NETWORK_ALREADY_OFFLINE	W_ERROR(0x000013C8)
#define WERR_CLUSTER_NODE_ALREADY_MEMBER	W_ERROR(0x000013C9)
#define WERR_CLUSTER_LAST_INTERNAL_NETWORK	W_ERROR(0x000013CA)
#define WERR_CLUSTER_NETWORK_HAS_DEPENDENTS	W_ERROR(0x000013CB)
#define WERR_INVALID_OPERATION_ON_QUORUM	W_ERROR(0x000013CC)
#define WERR_DEPENDENCY_NOT_ALLOWED	W_ERROR(0x000013CD)
#define WERR_CLUSTER_NODE_PAUSED	W_ERROR(0x000013CE)
#define WERR_NODE_CANT_HOST_RESOURCE	W_ERROR(0x000013CF)
#define WERR_CLUSTER_NODE_NOT_READY	W_ERROR(0x000013D0)
#define WERR_CLUSTER_NODE_SHUTTING_DOWN	W_ERROR(0x000013D1)
#define WERR_CLUSTER_JOIN_ABORTED	W_ERROR(0x000013D2)
#define WERR_CLUSTER_INCOMPATIBLE_VERSIONS	W_ERROR(0x000013D3)
#define WERR_CLUSTER_MAXNUM_OF_RESOURCES_EXCEEDED	W_ERROR(0x000013D4)
#define WERR_CLUSTER_SYSTEM_CONFIG_CHANGED	W_ERROR(0x000013D5)
#define WERR_CLUSTER_RESOURCE_TYPE_NOT_FOUND	W_ERROR(0x000013D6)
#define WERR_CLUSTER_RESTYPE_NOT_SUPPORTED	W_ERROR(0x000013D7)
#define WERR_CLUSTER_RESNAME_NOT_FOUND	W_ERROR(0x000013D8)
#define WERR_CLUSTER_NO_RPC_PACKAGES_REGISTERED	W_ERROR(0x000013D9)
#define WERR_CLUSTER_OWNER_NOT_IN_PREFLIST	W_ERROR(0x000013DA)
#define WERR_CLUSTER_DATABASE_SEQMISMATCH	W_ERROR(0x000013DB)
#define WERR_RESMON_INVALID_STATE	W_ERROR(0x000013DC)
#define WERR_CLUSTER_GUM_NOT_LOCKER	W_ERROR(0x000013DD)
#define WERR_QUORUM_DISK_NOT_FOUND	W_ERROR(0x000013DE)
#define WERR_DATABASE_BACKUP_CORRUPT	W_ERROR(0x000013DF)
#define WERR_CLUSTER_NODE_ALREADY_HAS_DFS_ROOT	W_ERROR(0x000013E0)
#define WERR_RESOURCE_PROPERTY_UNCHANGEABLE	W_ERROR(0x000013E1)
#define WERR_CLUSTER_MEMBERSHIP_INVALID_STATE	W_ERROR(0x00001702)
#define WERR_CLUSTER_QUORUMLOG_NOT_FOUND	W_ERROR(0x00001703)
#define WERR_CLUSTER_MEMBERSHIP_HALT	W_ERROR(0x00001704)
#define WERR_CLUSTER_INSTANCE_ID_MISMATCH	W_ERROR(0x00001705)
#define WERR_CLUSTER_NETWORK_NOT_FOUND_FOR_IP	W_ERROR(0x00001706)
#define WERR_CLUSTER_PROPERTY_DATA_TYPE_MISMATCH	W_ERROR(0x00001707)
#define WERR_CLUSTER_EVICT_WITHOUT_CLEANUP	W_ERROR(0x00001708)
#define WERR_CLUSTER_PARAMETER_MISMATCH	W_ERROR(0x00001709)
#define WERR_NODE_CANNOT_BE_CLUSTERED	W_ERROR(0x0000170A)
#define WERR_CLUSTER_WRONG_OS_VERSION	W_ERROR(0x0000170B)
#define WERR_CLUSTER_CANT_CREATE_DUP_CLUSTER_NAME	W_ERROR(0x0000170C)
#define WERR_CLUSCFG_ALREADY_COMMITTED	W_ERROR(0x0000170D)
#define WERR_CLUSCFG_ROLLBACK_FAILED	W_ERROR(0x0000170E)
#define WERR_CLUSCFG_SYSTEM_DISK_DRIVE_LETTER_CONFLICT	W_ERROR(0x0000170F)
#define WERR_CLUSTER_OLD_VERSION	W_ERROR(0x00001710)
#define WERR_CLUSTER_MISMATCHED_COMPUTER_ACCT_NAME	W_ERROR(0x00001711)
#define WERR_CLUSTER_NO_NET_ADAPTERS	W_ERROR(0x00001712)
#define WERR_CLUSTER_POISONED	W_ERROR(0x00001713)
#define WERR_CLUSTER_GROUP_MOVING	W_ERROR(0x00001714)
#define WERR_CLUSTER_RESOURCE_TYPE_BUSY	W_ERROR(0x00001715)
#define WERR_RESOURCE_CALL_TIMED_OUT	W_ERROR(0x00001716)
#define WERR_INVALID_CLUSTER_IPV6_ADDRESS	W_ERROR(0x00001717)
#define WERR_CLUSTER_INTERNAL_INVALID_FUNCTION	W_ERROR(0x00001718)
#define WERR_CLUSTER_PARAMETER_OUT_OF_BOUNDS	W_ERROR(0x00001719)
#define WERR_CLUSTER_PARTIAL_SEND	W_ERROR(0x0000171A)
#define WERR_CLUSTER_REGISTRY_INVALID_FUNCTION	W_ERROR(0x0000171B)
#define WERR_CLUSTER_INVALID_STRING_TERMINATION	W_ERROR(0x0000171C)
#define WERR_CLUSTER_INVALID_STRING_FORMAT	W_ERROR(0x0000171D)
#define WERR_CLUSTER_DATABASE_TRANSACTION_IN_PROGRESS	W_ERROR(0x0000171E)
#define WERR_CLUSTER_DATABASE_TRANSACTION_NOT_IN_PROGRESS	W_ERROR(0x0000171F)
#define WERR_CLUSTER_NULL_DATA	W_ERROR(0x00001720)
#define WERR_CLUSTER_PARTIAL_READ	W_ERROR(0x00001721)
#define WERR_CLUSTER_PARTIAL_WRITE	W_ERROR(0x00001722)
#define WERR_CLUSTER_CANT_DESERIALIZE_DATA	W_ERROR(0x00001723)
#define WERR_DEPENDENT_RESOURCE_PROPERTY_CONFLICT	W_ERROR(0x00001724)
#define WERR_CLUSTER_NO_QUORUM	W_ERROR(0x00001725)
#define WERR_CLUSTER_INVALID_IPV6_NETWORK	W_ERROR(0x00001726)
#define WERR_CLUSTER_INVALID_IPV6_TUNNEL_NETWORK	W_ERROR(0x00001727)
#define WERR_QUORUM_NOT_ALLOWED_IN_THIS_GROUP	W_ERROR(0x00001728)
#define WERR_ENCRYPTION_FAILED	W_ERROR(0x00001770)
#define WERR_DECRYPTION_FAILED	W_ERROR(0x00001771)
#define WERR_FILE_ENCRYPTED	W_ERROR(0x00001772)
#define WERR_NO_RECOVERY_POLICY	W_ERROR(0x00001773)
#define WERR_NO_EFS	W_ERROR(0x00001774)
#define WERR_WRONG_EFS	W_ERROR(0x00001775)
#define WERR_NO_USER_KEYS	W_ERROR(0x00001776)
#define WERR_FILE_NOT_ENCRYPTED	W_ERROR(0x00001777)
#define WERR_NOT_EXPORT_FORMAT	W_ERROR(0x00001778)
#define WERR_FILE_READ_ONLY	W_ERROR(0x00001779)
#define WERR_DIR_EFS_DISALLOWED	W_ERROR(0x0000177A)
#define WERR_EFS_SERVER_NOT_TRUSTED	W_ERROR(0x0000177B)
#define WERR_BAD_RECOVERY_POLICY	W_ERROR(0x0000177C)
#define WERR_EFS_ALG_BLOB_TOO_BIG	W_ERROR(0x0000177D)
#define WERR_VOLUME_NOT_SUPPORT_EFS	W_ERROR(0x0000177E)
#define WERR_EFS_DISABLED	W_ERROR(0x0000177F)
#define WERR_EFS_VERSION_NOT_SUPPORT	W_ERROR(0x00001780)
#define WERR_CS_ENCRYPTION_INVALID_SERVER_RESPONSE	W_ERROR(0x00001781)
#define WERR_CS_ENCRYPTION_UNSUPPORTED_SERVER	W_ERROR(0x00001782)
#define WERR_CS_ENCRYPTION_EXISTING_ENCRYPTED_FILE	W_ERROR(0x00001783)
#define WERR_CS_ENCRYPTION_NEW_ENCRYPTED_FILE	W_ERROR(0x00001784)
#define WERR_CS_ENCRYPTION_FILE_NOT_CSE	W_ERROR(0x00001785)
#define WERR_NO_BROWSER_SERVERS_FOUND	W_ERROR(0x000017E6)
#define WERR_LOG_SECTOR_INVALID	W_ERROR(0x000019C8)
#define WERR_LOG_SECTOR_PARITY_INVALID	W_ERROR(0x000019C9)
#define WERR_LOG_SECTOR_REMAPPED	W_ERROR(0x000019CA)
#define WERR_LOG_BLOCK_INCOMPLETE	W_ERROR(0x000019CB)
#define WERR_LOG_INVALID_RANGE	W_ERROR(0x000019CC)
#define WERR_LOG_BLOCKS_EXHAUSTED	W_ERROR(0x000019CD)
#define WERR_LOG_READ_CONTEXT_INVALID	W_ERROR(0x000019CE)
#define WERR_LOG_RESTART_INVALID	W_ERROR(0x000019CF)
#define WERR_LOG_BLOCK_VERSION	W_ERROR(0x000019D0)
#define WERR_LOG_BLOCK_INVALID	W_ERROR(0x000019D1)
#define WERR_LOG_READ_MODE_INVALID	W_ERROR(0x000019D2)
#define WERR_LOG_NO_RESTART	W_ERROR(0x000019D3)
#define WERR_LOG_METADATA_CORRUPT	W_ERROR(0x000019D4)
#define WERR_LOG_METADATA_INVALID	W_ERROR(0x000019D5)
#define WERR_LOG_METADATA_INCONSISTENT	W_ERROR(0x000019D6)
#define WERR_LOG_RESERVATION_INVALID	W_ERROR(0x000019D7)
#define WERR_LOG_CANT_DELETE	W_ERROR(0x000019D8)
#define WERR_LOG_CONTAINER_LIMIT_EXCEEDED	W_ERROR(0x000019D9)
#define WERR_LOG_START_OF_LOG	W_ERROR(0x000019DA)
#define WERR_LOG_POLICY_ALREADY_INSTALLED	W_ERROR(0x000019DB)
#define WERR_LOG_POLICY_NOT_INSTALLED	W_ERROR(0x000019DC)
#define WERR_LOG_POLICY_INVALID	W_ERROR(0x000019DD)
#define WERR_LOG_POLICY_CONFLICT	W_ERROR(0x000019DE)
#define WERR_LOG_PINNED_ARCHIVE_TAIL	W_ERROR(0x000019DF)
#define WERR_LOG_RECORD_NONEXISTENT	W_ERROR(0x000019E0)
#define WERR_LOG_RECORDS_RESERVED_INVALID	W_ERROR(0x000019E1)
#define WERR_LOG_SPACE_RESERVED_INVALID	W_ERROR(0x000019E2)
#define WERR_LOG_TAIL_INVALID	W_ERROR(0x000019E3)
#define WERR_LOG_FULL	W_ERROR(0x000019E4)
#define WERR_COULD_NOT_RESIZE_LOG	W_ERROR(0x000019E5)
#define WERR_LOG_MULTIPLEXED	W_ERROR(0x000019E6)
#define WERR_LOG_DEDICATED	W_ERROR(0x000019E7)
#define WERR_LOG_ARCHIVE_NOT_IN_PROGRESS	W_ERROR(0x000019E8)
#define WERR_LOG_ARCHIVE_IN_PROGRESS	W_ERROR(0x000019E9)
#define WERR_LOG_EPHEMERAL	W_ERROR(0x000019EA)
#define WERR_LOG_NOT_ENOUGH_CONTAINERS	W_ERROR(0x000019EB)
#define WERR_LOG_CLIENT_ALREADY_REGISTERED	W_ERROR(0x000019EC)
#define WERR_LOG_CLIENT_NOT_REGISTERED	W_ERROR(0x000019ED)
#define WERR_LOG_FULL_HANDLER_IN_PROGRESS	W_ERROR(0x000019EE)
#define WERR_LOG_CONTAINER_READ_FAILED	W_ERROR(0x000019EF)
#define WERR_LOG_CONTAINER_WRITE_FAILED	W_ERROR(0x000019F0)
#define WERR_LOG_CONTAINER_OPEN_FAILED	W_ERROR(0x000019F1)
#define WERR_LOG_CONTAINER_STATE_INVALID	W_ERROR(0x000019F2)
#define WERR_LOG_STATE_INVALID	W_ERROR(0x000019F3)
#define WERR_LOG_PINNED	W_ERROR(0x000019F4)
#define WERR_LOG_METADATA_FLUSH_FAILED	W_ERROR(0x000019F5)
#define WERR_LOG_INCONSISTENT_SECURITY	W_ERROR(0x000019F6)
#define WERR_LOG_APPENDED_FLUSH_FAILED	W_ERROR(0x000019F7)
#define WERR_LOG_PINNED_RESERVATION	W_ERROR(0x000019F8)
#define WERR_INVALID_TRANSACTION	W_ERROR(0x00001A2C)
#define WERR_TRANSACTION_NOT_ACTIVE	W_ERROR(0x00001A2D)
#define WERR_TRANSACTION_REQUEST_NOT_VALID	W_ERROR(0x00001A2E)
#define WERR_TRANSACTION_NOT_REQUESTED	W_ERROR(0x00001A2F)
#define WERR_TRANSACTION_ALREADY_ABORTED	W_ERROR(0x00001A30)
#define WERR_TRANSACTION_ALREADY_COMMITTED	W_ERROR(0x00001A31)
#define WERR_TM_INITIALIZATION_FAILED	W_ERROR(0x00001A32)
#define WERR_RESOURCEMANAGER_READ_ONLY	W_ERROR(0x00001A33)
#define WERR_TRANSACTION_NOT_JOINED	W_ERROR(0x00001A34)
#define WERR_TRANSACTION_SUPERIOR_EXISTS	W_ERROR(0x00001A35)
#define WERR_CRM_PROTOCOL_ALREADY_EXISTS	W_ERROR(0x00001A36)
#define WERR_TRANSACTION_PROPAGATION_FAILED	W_ERROR(0x00001A37)
#define WERR_CRM_PROTOCOL_NOT_FOUND	W_ERROR(0x00001A38)
#define WERR_TRANSACTION_INVALID_MARSHALL_BUFFER	W_ERROR(0x00001A39)
#define WERR_CURRENT_TRANSACTION_NOT_VALID	W_ERROR(0x00001A3A)
#define WERR_TRANSACTION_NOT_FOUND	W_ERROR(0x00001A3B)
#define WERR_RESOURCEMANAGER_NOT_FOUND	W_ERROR(0x00001A3C)
#define WERR_ENLISTMENT_NOT_FOUND	W_ERROR(0x00001A3D)
#define WERR_TRANSACTIONMANAGER_NOT_FOUND	W_ERROR(0x00001A3E)
#define WERR_TRANSACTIONMANAGER_NOT_ONLINE	W_ERROR(0x00001A3F)
#define WERR_TRANSACTIONMANAGER_RECOVERY_NAME_COLLISION	W_ERROR(0x00001A40)
#define WERR_TRANSACTIONAL_CONFLICT	W_ERROR(0x00001A90)
#define WERR_RM_NOT_ACTIVE	W_ERROR(0x00001A91)
#define WERR_RM_METADATA_CORRUPT	W_ERROR(0x00001A92)
#define WERR_DIRECTORY_NOT_RM	W_ERROR(0x00001A93)
#define WERR_TRANSACTIONS_UNSUPPORTED_REMOTE	W_ERROR(0x00001A95)
#define WERR_LOG_RESIZE_INVALID_SIZE	W_ERROR(0x00001A96)
#define WERR_OBJECT_NO_LONGER_EXISTS	W_ERROR(0x00001A97)
#define WERR_STREAM_MINIVERSION_NOT_FOUND	W_ERROR(0x00001A98)
#define WERR_STREAM_MINIVERSION_NOT_VALID	W_ERROR(0x00001A99)
#define WERR_MINIVERSION_INACCESSIBLE_FROM_SPECIFIED_TRANSACTION	W_ERROR(0x00001A9A)
#define WERR_CANT_OPEN_MINIVERSION_WITH_MODIFY_INTENT	W_ERROR(0x00001A9B)
#define WERR_CANT_CREATE_MORE_STREAM_MINIVERSIONS	W_ERROR(0x00001A9C)
#define WERR_REMOTE_FILE_VERSION_MISMATCH	W_ERROR(0x00001A9E)
#define WERR_HANDLE_NO_LONGER_VALID	W_ERROR(0x00001A9F)
#define WERR_NO_TXF_METADATA	W_ERROR(0x00001AA0)
#define WERR_LOG_CORRUPTION_DETECTED	W_ERROR(0x00001AA1)
#define WERR_CANT_RECOVER_WITH_HANDLE_OPEN	W_ERROR(0x00001AA2)
#define WERR_RM_DISCONNECTED	W_ERROR(0x00001AA3)
#define WERR_ENLISTMENT_NOT_SUPERIOR	W_ERROR(0x00001AA4)
#define WERR_RECOVERY_NOT_NEEDED	W_ERROR(0x00001AA5)
#define WERR_RM_ALREADY_STARTED	W_ERROR(0x00001AA6)
#define WERR_FILE_IDENTITY_NOT_PERSISTENT	W_ERROR(0x00001AA7)
#define WERR_CANT_BREAK_TRANSACTIONAL_DEPENDENCY	W_ERROR(0x00001AA8)
#define WERR_CANT_CROSS_RM_BOUNDARY	W_ERROR(0x00001AA9)
#define WERR_TXF_DIR_NOT_EMPTY	W_ERROR(0x00001AAA)
#define WERR_INDOUBT_TRANSACTIONS_EXIST	W_ERROR(0x00001AAB)
#define WERR_TM_VOLATILE	W_ERROR(0x00001AAC)
#define WERR_ROLLBACK_TIMER_EXPIRED	W_ERROR(0x00001AAD)
#define WERR_TXF_ATTRIBUTE_CORRUPT	W_ERROR(0x00001AAE)
#define WERR_EFS_NOT_ALLOWED_IN_TRANSACTION	W_ERROR(0x00001AAF)
#define WERR_TRANSACTIONAL_OPEN_NOT_ALLOWED	W_ERROR(0x00001AB0)
#define WERR_LOG_GROWTH_FAILED	W_ERROR(0x00001AB1)
#define WERR_TRANSACTED_MAPPING_UNSUPPORTED_REMOTE	W_ERROR(0x00001AB2)
#define WERR_TXF_METADATA_ALREADY_PRESENT	W_ERROR(0x00001AB3)
#define WERR_TRANSACTION_SCOPE_CALLBACKS_NOT_SET	W_ERROR(0x00001AB4)
#define WERR_TRANSACTION_REQUIRED_PROMOTION	W_ERROR(0x00001AB5)
#define WERR_CANNOT_EXECUTE_FILE_IN_TRANSACTION	W_ERROR(0x00001AB6)
#define WERR_TRANSACTIONS_NOT_FROZEN	W_ERROR(0x00001AB7)
#define WERR_TRANSACTION_FREEZE_IN_PROGRESS	W_ERROR(0x00001AB8)
#define WERR_NOT_SNAPSHOT_VOLUME	W_ERROR(0x00001AB9)
#define WERR_NO_SAVEPOINT_WITH_OPEN_FILES	W_ERROR(0x00001ABA)
#define WERR_DATA_LOST_REPAIR	W_ERROR(0x00001ABB)
#define WERR_SPARSE_NOT_ALLOWED_IN_TRANSACTION	W_ERROR(0x00001ABC)
#define WERR_TM_IDENTITY_MISMATCH	W_ERROR(0x00001ABD)
#define WERR_FLOATED_SECTION	W_ERROR(0x00001ABE)
#define WERR_CANNOT_ACCEPT_TRANSACTED_WORK	W_ERROR(0x00001ABF)
#define WERR_CANNOT_ABORT_TRANSACTIONS	W_ERROR(0x00001AC0)
#define WERR_CTX_WINSTATION_NAME_INVALID	W_ERROR(0x00001B59)
#define WERR_CTX_INVALID_PD	W_ERROR(0x00001B5A)
#define WERR_CTX_PD_NOT_FOUND	W_ERROR(0x00001B5B)
#define WERR_CTX_WD_NOT_FOUND	W_ERROR(0x00001B5C)
#define WERR_CTX_CANNOT_MAKE_EVENTLOG_ENTRY	W_ERROR(0x00001B5D)
#define WERR_CTX_SERVICE_NAME_COLLISION	W_ERROR(0x00001B5E)
#define WERR_CTX_CLOSE_PENDING	W_ERROR(0x00001B5F)
#define WERR_CTX_NO_OUTBUF	W_ERROR(0x00001B60)
#define WERR_CTX_MODEM_INF_NOT_FOUND	W_ERROR(0x00001B61)
#define WERR_CTX_INVALID_MODEMNAME	W_ERROR(0x00001B62)
#define WERR_CTX_MODEM_RESPONSE_ERROR	W_ERROR(0x00001B63)
#define WERR_CTX_MODEM_RESPONSE_TIMEOUT	W_ERROR(0x00001B64)
#define WERR_CTX_MODEM_RESPONSE_NO_CARRIER	W_ERROR(0x00001B65)
#define WERR_CTX_MODEM_RESPONSE_NO_DIALTONE	W_ERROR(0x00001B66)
#define WERR_CTX_MODEM_RESPONSE_BUSY	W_ERROR(0x00001B67)
#define WERR_CTX_MODEM_RESPONSE_VOICE	W_ERROR(0x00001B68)
#define WERR_CTX_TD_ERROR	W_ERROR(0x00001B69)
#define WERR_CTX_WINSTATION_NOT_FOUND	W_ERROR(0x00001B6E)
#define WERR_CTX_WINSTATION_ALREADY_EXISTS	W_ERROR(0x00001B6F)
#define WERR_CTX_WINSTATION_BUSY	W_ERROR(0x00001B70)
#define WERR_CTX_BAD_VIDEO_MODE	W_ERROR(0x00001B71)
#define WERR_CTX_GRAPHICS_INVALID	W_ERROR(0x00001B7B)
#define WERR_CTX_LOGON_DISABLED	W_ERROR(0x00001B7D)
#define WERR_CTX_NOT_CONSOLE	W_ERROR(0x00001B7E)
#define WERR_CTX_CLIENT_QUERY_TIMEOUT	W_ERROR(0x00001B80)
#define WERR_CTX_CONSOLE_DISCONNECT	W_ERROR(0x00001B81)
#define WERR_CTX_CONSOLE_CONNECT	W_ERROR(0x00001B82)
#define WERR_CTX_SHADOW_DENIED	W_ERROR(0x00001B84)
#define WERR_CTX_WINSTATION_ACCESS_DENIED	W_ERROR(0x00001B85)
#define WERR_CTX_INVALID_WD	W_ERROR(0x00001B89)
#define WERR_CTX_SHADOW_INVALID	W_ERROR(0x00001B8A)
#define WERR_CTX_SHADOW_DISABLED	W_ERROR(0x00001B8B)
#define WERR_CTX_CLIENT_LICENSE_IN_USE	W_ERROR(0x00001B8C)
#define WERR_CTX_CLIENT_LICENSE_NOT_SET	W_ERROR(0x00001B8D)
#define WERR_CTX_LICENSE_NOT_AVAILABLE	W_ERROR(0x00001B8E)
#define WERR_CTX_LICENSE_CLIENT_INVALID	W_ERROR(0x00001B8F)
#define WERR_CTX_LICENSE_EXPIRED	W_ERROR(0x00001B90)
#define WERR_CTX_SHADOW_NOT_RUNNING	W_ERROR(0x00001B91)
#define WERR_CTX_SHADOW_ENDED_BY_MODE_CHANGE	W_ERROR(0x00001B92)
#define WERR_ACTIVATION_COUNT_EXCEEDED	W_ERROR(0x00001B93)
#define WERR_CTX_WINSTATIONS_DISABLED	W_ERROR(0x00001B94)
#define WERR_CTX_ENCRYPTION_LEVEL_REQUIRED	W_ERROR(0x00001B95)
#define WERR_CTX_SESSION_IN_USE	W_ERROR(0x00001B96)
#define WERR_CTX_NO_FORCE_LOGOFF	W_ERROR(0x00001B97)
#define WERR_CTX_ACCOUNT_RESTRICTION	W_ERROR(0x00001B98)
#define WERR_RDP_PROTOCOL_ERROR	W_ERROR(0x00001B99)
#define WERR_CTX_CDM_CONNECT	W_ERROR(0x00001B9A)
#define WERR_CTX_CDM_DISCONNECT	W_ERROR(0x00001B9B)
#define WERR_CTX_SECURITY_LAYER_ERROR	W_ERROR(0x00001B9C)
#define WERR_TS_INCOMPATIBLE_SESSIONS	W_ERROR(0x00001B9D)
#define WERR_FRS_ERR_INVALID_API_SEQUENCE	W_ERROR(0x00001F41)
#define WERR_FRS_ERR_STARTING_SERVICE	W_ERROR(0x00001F42)
#define WERR_FRS_ERR_STOPPING_SERVICE	W_ERROR(0x00001F43)
#define WERR_FRS_ERR_INTERNAL_API	W_ERROR(0x00001F44)
#define WERR_FRS_ERR_INTERNAL	W_ERROR(0x00001F45)
#define WERR_FRS_ERR_SERVICE_COMM	W_ERROR(0x00001F46)
#define WERR_FRS_ERR_INSUFFICIENT_PRIV	W_ERROR(0x00001F47)
#define WERR_FRS_ERR_AUTHENTICATION	W_ERROR(0x00001F48)
#define WERR_FRS_ERR_PARENT_INSUFFICIENT_PRIV	W_ERROR(0x00001F49)
#define WERR_FRS_ERR_PARENT_AUTHENTICATION	W_ERROR(0x00001F4A)
#define WERR_FRS_ERR_CHILD_TO_PARENT_COMM	W_ERROR(0x00001F4B)
#define WERR_FRS_ERR_PARENT_TO_CHILD_COMM	W_ERROR(0x00001F4C)
#define WERR_FRS_ERR_SYSVOL_POPULATE	W_ERROR(0x00001F4D)
#define WERR_FRS_ERR_SYSVOL_POPULATE_TIMEOUT	W_ERROR(0x00001F4E)
#define WERR_FRS_ERR_SYSVOL_IS_BUSY	W_ERROR(0x00001F4F)
#define WERR_FRS_ERR_SYSVOL_DEMOTE	W_ERROR(0x00001F50)
#define WERR_FRS_ERR_INVALID_SERVICE_PARAMETER	W_ERROR(0x00001F51)
#define WERR_DS_NOT_INSTALLED	W_ERROR(0x00002008)
#define WERR_DS_MEMBERSHIP_EVALUATED_LOCALLY	W_ERROR(0x00002009)
#define WERR_DS_NO_ATTRIBUTE_OR_VALUE	W_ERROR(0x0000200A)
#define WERR_DS_INVALID_ATTRIBUTE_SYNTAX	W_ERROR(0x0000200B)
#define WERR_DS_ATTRIBUTE_TYPE_UNDEFINED	W_ERROR(0x0000200C)
#define WERR_DS_ATTRIBUTE_OR_VALUE_EXISTS	W_ERROR(0x0000200D)
#define WERR_DS_BUSY	W_ERROR(0x0000200E)
#define WERR_DS_UNAVAILABLE	W_ERROR(0x0000200F)
#define WERR_DS_NO_RIDS_ALLOCATED	W_ERROR(0x00002010)
#define WERR_DS_NO_MORE_RIDS	W_ERROR(0x00002011)
#define WERR_DS_INCORRECT_ROLE_OWNER	W_ERROR(0x00002012)
#define WERR_DS_RIDMGR_INIT_ERROR	W_ERROR(0x00002013)
#define WERR_DS_OBJ_CLASS_VIOLATION	W_ERROR(0x00002014)
#define WERR_DS_CANT_ON_NON_LEAF	W_ERROR(0x00002015)
#define WERR_DS_CANT_ON_RDN	 W_ERROR(0x00002016)
#define WERR_DS_CANT_MOD_OBJ_CLASS	W_ERROR(0x00002017)
#define WERR_DS_CROSS_DOM_MOVE_ERROR	W_ERROR(0x00002018)
#define WERR_DS_GC_NOT_AVAILABLE	W_ERROR(0x00002019)
#define WERR_SHARED_POLICY	W_ERROR(0x0000201A)
#define WERR_POLICY_OBJECT_NOT_FOUND	W_ERROR(0x0000201B)
#define WERR_POLICY_ONLY_IN_DS	W_ERROR(0x0000201C)
#define WERR_PROMOTION_ACTIVE	W_ERROR(0x0000201D)
#define WERR_NO_PROMOTION_ACTIVE	W_ERROR(0x0000201E)
#define WERR_DS_OPERATIONS_ERROR	W_ERROR(0x00002020)
#define WERR_DS_PROTOCOL_ERROR 	W_ERROR(0x00002021)
#define WERR_DS_TIMELIMIT_EXCEEDED	W_ERROR(0x00002022)
#define WERR_DS_SIZELIMIT_EXCEEDED	W_ERROR(0x00002023)
#define WERR_DS_ADMIN_LIMIT_EXCEEDED	W_ERROR(0x00002024)
#define WERR_DS_COMPARE_FALSE	W_ERROR(0x00002025)
#define WERR_DS_COMPARE_TRUE	W_ERROR(0x00002026)
#define WERR_DS_AUTH_METHOD_NOT_SUPPORTED	W_ERROR(0x00002027)
#define WERR_DS_STRONG_AUTH_REQUIRED	W_ERROR(0x00002028)
#define WERR_DS_INAPPROPRIATE_AUTH	W_ERROR(0x00002029)
#define WERR_DS_AUTH_UNKNOWN	W_ERROR(0x0000202A)
#define WERR_DS_REFERRAL	W_ERROR(0x0000202B)
#define WERR_DS_UNAVAILABLE_CRIT_EXTENSION	W_ERROR(0x0000202C)
#define WERR_DS_CONFIDENTIALITY_REQUIRED	W_ERROR(0x0000202D)
#define WERR_DS_INAPPROPRIATE_MATCHING	W_ERROR(0x0000202E)
#define WERR_DS_CONSTRAINT_VIOLATION	W_ERROR(0x0000202F)
#define WERR_DS_NO_SUCH_OBJECT	W_ERROR(0x00002030)
#define WERR_DS_ALIAS_PROBLEM	W_ERROR(0x00002031)
#define WERR_DS_INVALID_DN_SYNTAX	W_ERROR(0x00002032)
#define WERR_DS_IS_LEAF	W_ERROR(0x00002033)
#define WERR_DS_ALIAS_DEREF_PROBLEM	W_ERROR(0x00002034)
#define WERR_DS_UNWILLING_TO_PERFORM	W_ERROR(0x00002035)
#define WERR_DS_LOOP_DETECT	W_ERROR(0x00002036)
#define WERR_DS_NAMING_VIOLATION	W_ERROR(0x00002037)
#define WERR_DS_OBJECT_RESULTS_TOO_LARGE	W_ERROR(0x00002038)
#define WERR_DS_AFFECTS_MULTIPLE_DSAS	W_ERROR(0x00002039)
#define WERR_DS_SERVER_DOWN	W_ERROR(0x0000203A)
#define WERR_DS_LOCAL_ERROR	W_ERROR(0x0000203B)
#define WERR_DS_ENCODING_ERROR	W_ERROR(0x0000203C)
#define WERR_DS_DECODING_ERROR	W_ERROR(0x0000203D)
#define WERR_DS_FILTER_UNKNOWN	W_ERROR(0x0000203E)
#define WERR_DS_PARAM_ERROR	W_ERROR(0x0000203F)
#define WERR_DS_NOT_SUPPORTED	W_ERROR(0x00002040)
#define WERR_DS_NO_RESULTS_RETURNED	W_ERROR(0x00002041)
#define WERR_DS_CONTROL_NOT_FOUND	W_ERROR(0x00002042)
#define WERR_DS_CLIENT_LOOP	W_ERROR(0x00002043)
#define WERR_DS_REFERRAL_LIMIT_EXCEEDED	W_ERROR(0x00002044)
#define WERR_DS_SORT_CONTROL_MISSING	W_ERROR(0x00002045)
#define WERR_DS_OFFSET_RANGE_ERROR	W_ERROR(0x00002046)
#define WERR_DS_ROOT_MUST_BE_NC	W_ERROR(0x0000206D)
#define WERR_DS_ADD_REPLICA_INHIBITED	W_ERROR(0x0000206E)
#define WERR_DS_ATT_NOT_DEF_IN_SCHEMA	W_ERROR(0x0000206F)
#define WERR_DS_MAX_OBJ_SIZE_EXCEEDED	W_ERROR(0x00002070)
#define WERR_DS_OBJ_STRING_NAME_EXISTS	W_ERROR(0x00002071)
#define WERR_DS_NO_RDN_DEFINED_IN_SCHEMA	W_ERROR(0x00002072)
#define WERR_DS_RDN_DOESNT_MATCH_SCHEMA	W_ERROR(0x00002073)
#define WERR_DS_NO_REQUESTED_ATTS_FOUND	W_ERROR(0x00002074)
#define WERR_DS_USER_BUFFER_TO_SMALL	W_ERROR(0x00002075)
#define WERR_DS_ATT_IS_NOT_ON_OBJ	W_ERROR(0x00002076)
#define WERR_DS_ILLEGAL_MOD_OPERATION	W_ERROR(0x00002077)
#define WERR_DS_OBJ_TOO_LARGE	W_ERROR(0x00002078)
#define WERR_DS_BAD_INSTANCE_TYPE	W_ERROR(0x00002079)
#define WERR_DS_MASTERDSA_REQUIRED	W_ERROR(0x0000207A)
#define WERR_DS_OBJECT_CLASS_REQUIRED	W_ERROR(0x0000207B)
#define WERR_DS_MISSING_REQUIRED_ATT	W_ERROR(0x0000207C)
#define WERR_DS_ATT_NOT_DEF_FOR_CLASS	W_ERROR(0x0000207D)
#define WERR_DS_ATT_ALREADY_EXISTS	W_ERROR(0x0000207E)
#define WERR_DS_CANT_ADD_ATT_VALUES	W_ERROR(0x00002080)
#define WERR_DS_SINGLE_VALUE_CONSTRAINT	W_ERROR(0x00002081)
#define WERR_DS_RANGE_CONSTRAINT	W_ERROR(0x00002082)
#define WERR_DS_ATT_VAL_ALREADY_EXISTS	W_ERROR(0x00002083)
#define WERR_DS_CANT_REM_MISSING_ATT	W_ERROR(0x00002084)
#define WERR_DS_CANT_REM_MISSING_ATT_VAL	W_ERROR(0x00002085)
#define WERR_DS_ROOT_CANT_BE_SUBREF	W_ERROR(0x00002086)
#define WERR_DS_NO_CHAINING	W_ERROR(0x00002087)
#define WERR_DS_NO_CHAINED_EVAL	W_ERROR(0x00002088)
#define WERR_DS_NO_PARENT_OBJECT	W_ERROR(0x00002089)
#define WERR_DS_PARENT_IS_AN_ALIAS	W_ERROR(0x0000208A)
#define WERR_DS_CANT_MIX_MASTER_AND_REPS	W_ERROR(0x0000208B)
#define WERR_DS_CHILDREN_EXIST	W_ERROR(0x0000208C)
#define WERR_DS_OBJ_NOT_FOUND	W_ERROR(0x0000208D)
#define WERR_DS_ALIASED_OBJ_MISSING	W_ERROR(0x0000208E)
#define WERR_DS_BAD_NAME_SYNTAX	W_ERROR(0x0000208F)
#define WERR_DS_ALIAS_POINTS_TO_ALIAS	W_ERROR(0x00002090)
#define WERR_DS_CANT_DEREF_ALIAS	W_ERROR(0x00002091)
#define WERR_DS_OUT_OF_SCOPE	W_ERROR(0x00002092)
#define WERR_DS_OBJECT_BEING_REMOVED	W_ERROR(0x00002093)
#define WERR_DS_CANT_DELETE_DSA_OBJ	W_ERROR(0x00002094)
#define WERR_DS_GENERIC_ERROR	W_ERROR(0x00002095)
#define WERR_DS_DSA_MUST_BE_INT_MASTER	W_ERROR(0x00002096)
#define WERR_DS_CLASS_NOT_DSA	W_ERROR(0x00002097)
#define WERR_DS_INSUFF_ACCESS_RIGHTS	W_ERROR(0x00002098)
#define WERR_DS_ILLEGAL_SUPERIOR	W_ERROR(0x00002099)
#define WERR_DS_ATTRIBUTE_OWNED_BY_SAM	W_ERROR(0x0000209A)
#define WERR_DS_NAME_TOO_MANY_PARTS	W_ERROR(0x0000209B)
#define WERR_DS_NAME_TOO_LONG	W_ERROR(0x0000209C)
#define WERR_DS_NAME_VALUE_TOO_LONG	W_ERROR(0x0000209D)
#define WERR_DS_NAME_UNPARSEABLE	W_ERROR(0x0000209E)
#define WERR_DS_NAME_TYPE_UNKNOWN	W_ERROR(0x0000209F)
#define WERR_DS_NOT_AN_OBJECT	W_ERROR(0x000020A0)
#define WERR_DS_SEC_DESC_TOO_SHORT	W_ERROR(0x000020A1)
#define WERR_DS_SEC_DESC_INVALID	W_ERROR(0x000020A2)
#define WERR_DS_NO_DELETED_NAME	W_ERROR(0x000020A3)
#define WERR_DS_SUBREF_MUST_HAVE_PARENT	W_ERROR(0x000020A4)
#define WERR_DS_NCNAME_MUST_BE_NC	W_ERROR(0x000020A5)
#define WERR_DS_CANT_ADD_SYSTEM_ONLY	W_ERROR(0x000020A6)
#define WERR_DS_CLASS_MUST_BE_CONCRETE	W_ERROR(0x000020A7)
#define WERR_DS_INVALID_DMD	W_ERROR(0x000020A8)
#define WERR_DS_OBJ_GUID_EXISTS	W_ERROR(0x000020A9)
#define WERR_DS_NOT_ON_BACKLINK	W_ERROR(0x000020AA)
#define WERR_DS_NO_CROSSREF_FOR_NC	W_ERROR(0x000020AB)
#define WERR_DS_SHUTTING_DOWN	W_ERROR(0x000020AC)
#define WERR_DS_UNKNOWN_OPERATION	W_ERROR(0x000020AD)
#define WERR_DS_INVALID_ROLE_OWNER	W_ERROR(0x000020AE)
#define WERR_DS_COULDNT_CONTACT_FSMO	W_ERROR(0x000020AF)
#define WERR_DS_CROSS_NC_DN_RENAME	W_ERROR(0x000020B0)
#define WERR_DS_CANT_MOD_SYSTEM_ONLY	W_ERROR(0x000020B1)
#define WERR_DS_REPLICATOR_ONLY	W_ERROR(0x000020B2)
#define WERR_DS_OBJ_CLASS_NOT_DEFINED	W_ERROR(0x000020B3)
#define WERR_DS_OBJ_CLASS_NOT_SUBCLASS	W_ERROR(0x000020B4)
#define WERR_DS_NAME_REFERENCE_INVALID	W_ERROR(0x000020B5)
#define WERR_DS_CROSS_REF_EXISTS	W_ERROR(0x000020B6)
#define WERR_DS_CANT_DEL_MASTER_CROSSREF	W_ERROR(0x000020B7)
#define WERR_DS_SUBTREE_NOTIFY_NOT_NC_HEAD	W_ERROR(0x000020B8)
#define WERR_DS_NOTIFY_FILTER_TOO_COMPLEX	W_ERROR(0x000020B9)
#define WERR_DS_DUP_RDN	W_ERROR(0x000020BA)
#define WERR_DS_DUP_OID	W_ERROR(0x000020BB)
#define WERR_DS_DUP_MAPI_ID	W_ERROR(0x000020BC)
#define WERR_DS_DUP_SCHEMA_ID_GUID	W_ERROR(0x000020BD)
#define WERR_DS_DUP_LDAP_DISPLAY_NAME	W_ERROR(0x000020BE)
#define WERR_DS_SEMANTIC_ATT_TEST	W_ERROR(0x000020BF)
#define WERR_DS_SYNTAX_MISMATCH	W_ERROR(0x000020C0)
#define WERR_DS_EXISTS_IN_MUST_HAVE	W_ERROR(0x000020C1)
#define WERR_DS_EXISTS_IN_MAY_HAVE	W_ERROR(0x000020C2)
#define WERR_DS_NONEXISTENT_MAY_HAVE	W_ERROR(0x000020C3)
#define WERR_DS_NONEXISTENT_MUST_HAVE	W_ERROR(0x000020C4)
#define WERR_DS_AUX_CLS_TEST_FAIL	W_ERROR(0x000020C5)
#define WERR_DS_NONEXISTENT_POSS_SUP	W_ERROR(0x000020C6)
#define WERR_DS_SUB_CLS_TEST_FAIL	W_ERROR(0x000020C7)
#define WERR_DS_BAD_RDN_ATT_ID_SYNTAX	W_ERROR(0x000020C8)
#define WERR_DS_EXISTS_IN_AUX_CLS	W_ERROR(0x000020C9)
#define WERR_DS_EXISTS_IN_SUB_CLS	W_ERROR(0x000020CA)
#define WERR_DS_EXISTS_IN_POSS_SUP	W_ERROR(0x000020CB)
#define WERR_DS_RECALCSCHEMA_FAILED	W_ERROR(0x000020CC)
#define WERR_DS_TREE_DELETE_NOT_FINISHED	W_ERROR(0x000020CD)
#define WERR_DS_CANT_DELETE	W_ERROR(0x000020CE)
#define WERR_DS_ATT_SCHEMA_REQ_ID	W_ERROR(0x000020CF)
#define WERR_DS_BAD_ATT_SCHEMA_SYNTAX	W_ERROR(0x000020D0)
#define WERR_DS_CANT_CACHE_ATT	W_ERROR(0x000020D1)
#define WERR_DS_CANT_CACHE_CLASS	W_ERROR(0x000020D2)
#define WERR_DS_CANT_REMOVE_ATT_CACHE	W_ERROR(0x000020D3)
#define WERR_DS_CANT_REMOVE_CLASS_CACHE	W_ERROR(0x000020D4)
#define WERR_DS_CANT_RETRIEVE_DN	W_ERROR(0x000020D5)
#define WERR_DS_MISSING_SUPREF	W_ERROR(0x000020D6)
#define WERR_DS_CANT_RETRIEVE_INSTANCE	W_ERROR(0x000020D7)
#define WERR_DS_CODE_INCONSISTENCY	W_ERROR(0x000020D8)
#define WERR_DS_DATABASE_ERROR	W_ERROR(0x000020D9)
#define WERR_DS_MISSING_EXPECTED_ATT	W_ERROR(0x000020DB)
#define WERR_DS_NCNAME_MISSING_CR_REF	W_ERROR(0x000020DC)
#define WERR_DS_SECURITY_CHECKING_ERROR	W_ERROR(0x000020DD)
#define WERR_DS_SCHEMA_NOT_LOADED	W_ERROR(0x20DE)
#define WERR_DS_SCHEMA_ALLOC_FAILED	W_ERROR(0x20DF)
#define WERR_DS_ATT_SCHEMA_REQ_SYNTAX	W_ERROR(0x000020E0)
#define WERR_DS_GCVERIFY_ERROR	W_ERROR(0x000020E1)
#define WERR_DS_DRA_SCHEMA_MISMATCH	W_ERROR(0x000020E2)
#define WERR_DS_CANT_FIND_DSA_OBJ	W_ERROR(0x000020E3)
#define WERR_DS_CANT_FIND_EXPECTED_NC	W_ERROR(0x000020E4)
#define WERR_DS_CANT_FIND_NC_IN_CACHE	W_ERROR(0x000020E5)
#define WERR_DS_CANT_RETRIEVE_CHILD	W_ERROR(0x000020E6)
#define WERR_DS_SECURITY_ILLEGAL_MODIFY	W_ERROR(0x000020E7)
#define WERR_DS_CANT_REPLACE_HIDDEN_REC	W_ERROR(0x000020E8)
#define WERR_DS_BAD_HIERARCHY_FILE	W_ERROR(0x000020E9)
#define WERR_DS_BUILD_HIERARCHY_TABLE_FAILED	W_ERROR(0x000020EA)
#define WERR_DS_CONFIG_PARAM_MISSING	W_ERROR(0x000020EB)
#define WERR_DS_COUNTING_AB_INDICES_FAILED	W_ERROR(0x000020EC)
#define WERR_DS_HIERARCHY_TABLE_MALLOC_FAILED	W_ERROR(0x000020ED)
#define WERR_DS_INTERNAL_FAILURE	W_ERROR(0x000020EE)
#define WERR_DS_UNKNOWN_ERROR	W_ERROR(0x000020EF)
#define WERR_DS_ROOT_REQUIRES_CLASS_TOP	W_ERROR(0x000020F0)
#define WERR_DS_REFUSING_FSMO_ROLES	W_ERROR(0x000020F1)
#define WERR_DS_MISSING_FSMO_SETTINGS	W_ERROR(0x000020F2)
#define WERR_DS_UNABLE_TO_SURRENDER_ROLES	W_ERROR(0x000020F3)
#define WERR_DS_DRA_GENERIC	W_ERROR(0x000020F4)
#define WERR_DS_DRA_INVALID_PARAMETER	W_ERROR(0x000020F5)
#define WERR_DS_DRA_BUSY	W_ERROR(0x000020F6)
#define WERR_DS_DRA_BAD_DN	W_ERROR(0x000020F7)
#define WERR_DS_DRA_BAD_NC	W_ERROR(0x000020F8)
#define WERR_DS_DRA_DN_EXISTS	W_ERROR(0x000020F9)
#define WERR_DS_DRA_INTERNAL_ERROR	W_ERROR(0x000020FA)
#define WERR_DS_DRA_INCONSISTENT_DIT	W_ERROR(0x000020FB)
#define WERR_DS_DRA_CONNECTION_FAILED	W_ERROR(0x000020FC)
#define WERR_DS_DRA_BAD_INSTANCE_TYPE	W_ERROR(0x000020FD)
#define WERR_DS_DRA_OUT_OF_MEM	W_ERROR(0x000020FE)
#define WERR_DS_DRA_MAIL_PROBLEM	W_ERROR(0x000020FF)
#define WERR_DS_DRA_REF_ALREADY_EXISTS	W_ERROR(0x00002100)
#define WERR_DS_DRA_REF_NOT_FOUND	W_ERROR(0x00002101)
#define WERR_DS_DRA_OBJ_IS_REP_SOURCE	W_ERROR(0x00002102)
#define WERR_DS_DRA_DB_ERROR	W_ERROR(0x00002103)
#define WERR_DS_DRA_NO_REPLICA	W_ERROR(0x00002104)
#define WERR_DS_DRA_ACCESS_DENIED	W_ERROR(0x00002105)
#define WERR_DS_DRA_NOT_SUPPORTED	W_ERROR(0x00002106)
#define WERR_DS_DRA_RPC_CANCELLED	W_ERROR(0x00002107)
#define WERR_DS_DRA_SOURCE_DISABLED	W_ERROR(0x00002108)
#define WERR_DS_DRA_SINK_DISABLED	W_ERROR(0x00002109)
#define WERR_DS_DRA_NAME_COLLISION	W_ERROR(0x0000210A)
#define WERR_DS_DRA_SOURCE_REINSTALLED	W_ERROR(0x0000210B)
#define WERR_DS_DRA_MISSING_PARENT	W_ERROR(0x0000210C)
#define WERR_DS_DRA_PREEMPTED	W_ERROR(0x0000210D)
#define WERR_DS_DRA_ABANDON_SYNC	W_ERROR(0x0000210E)
#define WERR_DS_DRA_SHUTDOWN	W_ERROR(0x0000210F)
#define WERR_DS_DRA_INCOMPATIBLE_PARTIAL_SET	W_ERROR(0x00002110)
#define WERR_DS_DRA_SOURCE_IS_PARTIAL_REPLICA	W_ERROR(0x00002111)
#define WERR_DS_DRA_EXTN_CONNECTION_FAILED	W_ERROR(0x00002112)
#define WERR_DS_INSTALL_SCHEMA_MISMATCH	W_ERROR(0x00002113)
#define WERR_DS_DUP_LINK_ID	W_ERROR(0x00002114)
#define WERR_DS_NAME_ERROR_RESOLVING	W_ERROR(0x00002115)
#define WERR_DS_NAME_ERROR_NOT_FOUND	W_ERROR(0x00002116)
#define WERR_DS_NAME_ERROR_NOT_UNIQUE	W_ERROR(0x00002117)
#define WERR_DS_NAME_ERROR_NO_MAPPING	W_ERROR(0x00002118)
#define WERR_DS_NAME_ERROR_DOMAIN_ONLY	W_ERROR(0x00002119)
#define WERR_DS_NAME_ERROR_NO_SYNTACTICAL_MAPPING	W_ERROR(0x0000211A)
#define WERR_DS_CONSTRUCTED_ATT_MOD	W_ERROR(0x0000211B)
#define WERR_DS_WRONG_OM_OBJ_CLASS	W_ERROR(0x0000211C)
#define WERR_DS_DRA_REPL_PENDING	W_ERROR(0x0000211D)
#define WERR_DS_DS_REQUIRED	W_ERROR(0x0000211E)
#define WERR_DS_INVALID_LDAP_DISPLAY_NAME	W_ERROR(0x0000211F)
#define WERR_DS_NON_BASE_SEARCH	W_ERROR(0x00002120)
#define WERR_DS_CANT_RETRIEVE_ATTS	W_ERROR(0x00002121)
#define WERR_DS_BACKLINK_WITHOUT_LINK	W_ERROR(0x00002122)
#define WERR_DS_EPOCH_MISMATCH	W_ERROR(0x00002123)
#define WERR_DS_SRC_NAME_MISMATCH	W_ERROR(0x00002124)
#define WERR_DS_SRC_AND_DST_NC_IDENTICAL	W_ERROR(0x00002125)
#define WERR_DS_DST_NC_MISMATCH	W_ERROR(0x00002126)
#define WERR_DS_NOT_AUTHORITIVE_FOR_DST_NC	W_ERROR(0x00002127)
#define WERR_DS_SRC_GUID_MISMATCH	W_ERROR(0x00002128)
#define WERR_DS_CANT_MOVE_DELETED_OBJECT	W_ERROR(0x00002129)
#define WERR_DS_PDC_OPERATION_IN_PROGRESS	W_ERROR(0x0000212A)
#define WERR_DS_CROSS_DOMAIN_CLEANUP_REQD	W_ERROR(0x0000212B)
#define WERR_DS_ILLEGAL_XDOM_MOVE_OPERATION	W_ERROR(0x0000212C)
#define WERR_DS_CANT_WITH_ACCT_GROUP_MEMBERSHPS	W_ERROR(0x0000212D)
#define WERR_DS_NC_MUST_HAVE_NC_PARENT	W_ERROR(0x0000212E)
#define WERR_DS_CR_IMPOSSIBLE_TO_VALIDATE	W_ERROR(0x0000212F)
#define WERR_DS_DST_DOMAIN_NOT_NATIVE	W_ERROR(0x00002130)
#define WERR_DS_MISSING_INFRASTRUCTURE_CONTAINER	W_ERROR(0x00002131)
#define WERR_DS_CANT_MOVE_ACCOUNT_GROUP	W_ERROR(0x00002132)
#define WERR_DS_CANT_MOVE_RESOURCE_GROUP	W_ERROR(0x00002133)
#define WERR_DS_INVALID_SEARCH_FLAG	W_ERROR(0x00002134)
#define WERR_DS_NO_TREE_DELETE_ABOVE_NC	W_ERROR(0x00002135)
#define WERR_DS_COULDNT_LOCK_TREE_FOR_DELETE	W_ERROR(0x00002136)
#define WERR_DS_COULDNT_IDENTIFY_OBJECTS_FOR_TREE_DELETE	W_ERROR(0x00002137)
#define WERR_DS_SAM_INIT_FAILURE	W_ERROR(0x00002138)
#define WERR_DS_SENSITIVE_GROUP_VIOLATION	W_ERROR(0x00002139)
#define WERR_DS_CANT_MOD_PRIMARYGROUPID	W_ERROR(0x0000213A)
#define WERR_DS_ILLEGAL_BASE_SCHEMA_MOD	W_ERROR(0x0000213B)
#define WERR_DS_NONSAFE_SCHEMA_CHANGE	W_ERROR(0x0000213C)
#define WERR_DS_SCHEMA_UPDATE_DISALLOWED	W_ERROR(0x0000213D)
#define WERR_DS_CANT_CREATE_UNDER_SCHEMA	W_ERROR(0x0000213E)
#define WERR_DS_INVALID_GROUP_TYPE	W_ERROR(0x00002141)
#define WERR_DS_NO_NEST_GLOBALGROUP_IN_MIXEDDOMAIN	W_ERROR(0x00002142)
#define WERR_DS_NO_NEST_LOCALGROUP_IN_MIXEDDOMAIN	W_ERROR(0x00002143)
#define WERR_DS_GLOBAL_CANT_HAVE_LOCAL_MEMBER	W_ERROR(0x00002144)
#define WERR_DS_GLOBAL_CANT_HAVE_UNIVERSAL_MEMBER	W_ERROR(0x00002145)
#define WERR_DS_UNIVERSAL_CANT_HAVE_LOCAL_MEMBER	W_ERROR(0x00002146)
#define WERR_DS_GLOBAL_CANT_HAVE_CROSSDOMAIN_MEMBER	W_ERROR(0x00002147)
#define WERR_DS_LOCAL_CANT_HAVE_CROSSDOMAIN_LOCAL_MEMBER	W_ERROR(0x00002148)
#define WERR_DS_HAVE_PRIMARY_MEMBERS	W_ERROR(0x00002149)
#define WERR_DS_STRING_SD_CONVERSION_FAILED	W_ERROR(0x0000214A)
#define WERR_DS_NAMING_MASTER_GC	W_ERROR(0x0000214B)
#define WERR_DS_DNS_LOOKUP_FAILURE	W_ERROR(0x0000214C)
#define WERR_DS_COULDNT_UPDATE_SPNS	W_ERROR(0x0000214D)
#define WERR_DS_CANT_RETRIEVE_SD	W_ERROR(0x0000214E)
#define WERR_DS_KEY_NOT_UNIQUE	W_ERROR(0x0000214F)
#define WERR_DS_WRONG_LINKED_ATT_SYNTAX	W_ERROR(0x00002150)
#define WERR_DS_SAM_NEED_BOOTKEY_PASSWORD	W_ERROR(0x00002151)
#define WERR_DS_SAM_NEED_BOOTKEY_FLOPPY	W_ERROR(0x00002152)
#define WERR_DS_CANT_START	W_ERROR(0x00002153)
#define WERR_DS_INIT_FAILURE	W_ERROR(0x00002154)
#define WERR_DS_NO_PKT_PRIVACY_ON_CONNECTION	W_ERROR(0x00002155)
#define WERR_DS_SOURCE_DOMAIN_IN_FOREST	W_ERROR(0x00002156)
#define WERR_DS_DESTINATION_DOMAIN_NOT_IN_FOREST	W_ERROR(0x00002157)
#define WERR_DS_DESTINATION_AUDITING_NOT_ENABLED	W_ERROR(0x00002158)
#define WERR_DS_CANT_FIND_DC_FOR_SRC_DOMAIN	W_ERROR(0x00002159)
#define WERR_DS_SRC_OBJ_NOT_GROUP_OR_USER	W_ERROR(0x0000215A)
#define WERR_DS_SRC_SID_EXISTS_IN_FOREST	W_ERROR(0x0000215B)
#define WERR_DS_SRC_AND_DST_OBJECT_CLASS_MISMATCH	W_ERROR(0x0000215C)
#define WERR_SAM_INIT_FAILURE	W_ERROR(0x0000215D)
#define WERR_DS_DRA_SCHEMA_INFO_SHIP	W_ERROR(0x0000215E)
#define WERR_DS_DRA_SCHEMA_CONFLICT	W_ERROR(0x0000215F)
#define WERR_DS_DRA_EARLIER_SCHEMA_CONFLICT	W_ERROR(0x00002160)
#define WERR_DS_DRA_OBJ_NC_MISMATCH	W_ERROR(0x00002161)
#define WERR_DS_NC_STILL_HAS_DSAS	W_ERROR(0x00002162)
#define WERR_DS_GC_REQUIRED	W_ERROR(0x00002163)
#define WERR_DS_LOCAL_MEMBER_OF_LOCAL_ONLY	W_ERROR(0x00002164)
#define WERR_DS_NO_FPO_IN_UNIVERSAL_GROUPS	W_ERROR(0x00002165)
#define WERR_DS_CANT_ADD_TO_GC	W_ERROR(0x00002166)
#define WERR_DS_NO_CHECKPOINT_WITH_PDC	W_ERROR(0x00002167)
#define WERR_DS_SOURCE_AUDITING_NOT_ENABLED	W_ERROR(0x00002168)
#define WERR_DS_CANT_CREATE_IN_NONDOMAIN_NC	W_ERROR(0x00002169)
#define WERR_DS_INVALID_NAME_FOR_SPN	W_ERROR(0x0000216A)
#define WERR_DS_FILTER_USES_CONTRUCTED_ATTRS	W_ERROR(0x0000216B)
#define WERR_DS_MACHINE_ACCOUNT_QUOTA_EXCEEDED	W_ERROR(0x0000216D)
#define WERR_DS_MUST_BE_RUN_ON_DST_DC	W_ERROR(0x0000216E)
#define WERR_DS_SRC_DC_MUST_BE_SP4_OR_GREATER	W_ERROR(0x0000216F)
#define WERR_DS_CANT_TREE_DELETE_CRITICAL_OBJ	W_ERROR(0x00002170)
#define WERR_DS_INIT_FAILURE_CONSOLE	W_ERROR(0x00002171)
#define WERR_DS_SAM_INIT_FAILURE_CONSOLE	W_ERROR(0x00002172)
#define WERR_DS_FOREST_VERSION_TOO_HIGH	W_ERROR(0x00002173)
#define WERR_DS_DOMAIN_VERSION_TOO_HIGH	W_ERROR(0x00002174)
#define WERR_DS_FOREST_VERSION_TOO_LOW	W_ERROR(0x00002175)
#define WERR_DS_DOMAIN_VERSION_TOO_LOW	W_ERROR(0x00002176)
#define WERR_DS_INCOMPATIBLE_VERSION	W_ERROR(0x00002177)
#define WERR_DS_LOW_DSA_VERSION	W_ERROR(0x00002178)
#define WERR_DS_NO_BEHAVIOR_VERSION_IN_MIXEDDOMAIN	W_ERROR(0x00002179)
#define WERR_DS_NOT_SUPPORTED_SORT_ORDER	W_ERROR(0x0000217A)
#define WERR_DS_NAME_NOT_UNIQUE	W_ERROR(0x0000217B)
#define WERR_DS_MACHINE_ACCOUNT_CREATED_PRENT4	W_ERROR(0x0000217C)
#define WERR_DS_OUT_OF_VERSION_STORE	W_ERROR(0x0000217D)
#define WERR_DS_INCOMPATIBLE_CONTROLS_USED	W_ERROR(0x0000217E)
#define WERR_DS_NO_REF_DOMAIN	W_ERROR(0x0000217F)
#define WERR_DS_RESERVED_LINK_ID	W_ERROR(0x00002180)
#define WERR_DS_LINK_ID_NOT_AVAILABLE	W_ERROR(0x00002181)
#define WERR_DS_AG_CANT_HAVE_UNIVERSAL_MEMBER	W_ERROR(0x00002182)
#define WERR_DS_MODIFYDN_DISALLOWED_BY_INSTANCE_TYPE	W_ERROR(0x00002183)
#define WERR_DS_NO_OBJECT_MOVE_IN_SCHEMA_NC	W_ERROR(0x00002184)
#define WERR_DS_MODIFYDN_DISALLOWED_BY_FLAG	W_ERROR(0x00002185)
#define WERR_DS_MODIFYDN_WRONG_GRANDPARENT	W_ERROR(0x00002186)
#define WERR_DS_NAME_ERROR_TRUST_REFERRAL	W_ERROR(0x00002187)
#define WERR_NOT_SUPPORTED_ON_STANDARD_SERVER	W_ERROR(0x00002188)
#define WERR_DS_CANT_ACCESS_REMOTE_PART_OF_AD	W_ERROR(0x00002189)
#define WERR_DS_CR_IMPOSSIBLE_TO_VALIDATE_V2	W_ERROR(0x0000218A)
#define WERR_DS_THREAD_LIMIT_EXCEEDED	W_ERROR(0x0000218B)
#define WERR_DS_NOT_CLOSEST	W_ERROR(0x0000218C)
#define WERR_DS_SINGLE_USER_MODE_FAILED	W_ERROR(0x0000218E)
#define WERR_DS_NTDSCRIPT_SYNTAX_ERROR	W_ERROR(0x0000218F)
#define WERR_DS_NTDSCRIPT_PROCESS_ERROR	W_ERROR(0x00002190)
#define WERR_DS_DIFFERENT_REPL_EPOCHS	W_ERROR(0x00002191)
#define WERR_DS_DRS_EXTENSIONS_CHANGED	W_ERROR(0x00002192)
#define WERR_DS_REPLICA_SET_CHANGE_NOT_ALLOWED_ON_DISABLED_CR	W_ERROR(0x00002193)
#define WERR_DS_NO_MSDS_INTID	W_ERROR(0x00002194)
#define WERR_DS_DUP_MSDS_INTID	W_ERROR(0x00002195)
#define WERR_DS_EXISTS_IN_RDNATTID	W_ERROR(0x00002196)
#define WERR_DS_AUTHORIZATION_FAILED	W_ERROR(0x00002197)
#define WERR_DS_INVALID_SCRIPT	W_ERROR(0x00002198)
#define WERR_DS_REMOTE_CROSSREF_OP_FAILED	W_ERROR(0x00002199)
#define WERR_DS_CROSS_REF_BUSY	W_ERROR(0x0000219A)
#define WERR_DS_CANT_DERIVE_SPN_FOR_DELETED_DOMAIN	W_ERROR(0x0000219B)
#define WERR_DS_CANT_DEMOTE_WITH_WRITEABLE_NC	W_ERROR(0x0000219C)
#define WERR_DS_DUPLICATE_ID_FOUND	W_ERROR(0x0000219D)
#define WERR_DS_INSUFFICIENT_ATTR_TO_CREATE_OBJECT	W_ERROR(0x0000219E)
#define WERR_DS_GROUP_CONVERSION_ERROR	W_ERROR(0x0000219F)
#define WERR_DS_CANT_MOVE_APP_BASIC_GROUP	W_ERROR(0x000021A0)
#define WERR_DS_CANT_MOVE_APP_QUERY_GROUP	W_ERROR(0x000021A1)
#define WERR_DS_ROLE_NOT_VERIFIED	W_ERROR(0x000021A2)
#define WERR_DS_WKO_CONTAINER_CANNOT_BE_SPECIAL	W_ERROR(0x000021A3)
#define WERR_DS_DOMAIN_RENAME_IN_PROGRESS	W_ERROR(0x000021A4)
#define WERR_DS_EXISTING_AD_CHILD_NC	W_ERROR(0x000021A5)
#define WERR_DS_REPL_LIFETIME_EXCEEDED	W_ERROR(0x000021A6)
#define WERR_DS_DISALLOWED_IN_SYSTEM_CONTAINER	W_ERROR(0x000021A7)
#define WERR_DS_LDAP_SEND_QUEUE_FULL	W_ERROR(0x000021A8)
#define WERR_DS_DRA_OUT_SCHEDULE_WINDOW	W_ERROR(0x000021A9)
#define WERR_DS_POLICY_NOT_KNOWN	W_ERROR(0x000021AA)
#define WERR_NO_SITE_SETTINGS_OBJECT	W_ERROR(0x000021AB)
#define WERR_NO_SECRETS	W_ERROR(0x000021AC)
#define WERR_NO_WRITABLE_DC_FOUND	W_ERROR(0x000021AD)
#define WERR_DS_NO_SERVER_OBJECT	W_ERROR(0x000021AE)
#define WERR_DS_NO_NTDSA_OBJECT	W_ERROR(0x000021AF)
#define WERR_DS_NON_ASQ_SEARCH	W_ERROR(0x000021B0)
#define WERR_DS_AUDIT_FAILURE	W_ERROR(0x000021B1)
#define WERR_DS_INVALID_SEARCH_FLAG_SUBTREE	W_ERROR(0x000021B2)
#define WERR_DS_INVALID_SEARCH_FLAG_TUPLE	W_ERROR(0x000021B3)
#define WERR_DS_HIGH_DSA_VERSION	W_ERROR(0x000021C2)
#define WERR_DS_SPN_VALUE_NOT_UNIQUE_IN_FOREST	W_ERROR(0x000021C7)
#define WERR_DS_UPN_VALUE_NOT_UNIQUE_IN_FOREST	W_ERROR(0x000021C8)
#define WERR_DNS_ERROR_RCODE_FORMAT_ERROR	W_ERROR(0x00002329)
#define WERR_DNS_ERROR_RCODE_SERVER_FAILURE	W_ERROR(0x0000232A)
#define WERR_DNS_ERROR_RCODE_NAME_ERROR	W_ERROR(0x0000232B)
#define WERR_DNS_ERROR_RCODE_NOT_IMPLEMENTED	W_ERROR(0x0000232C)
#define WERR_DNS_ERROR_RCODE_REFUSED	W_ERROR(0x0000232D)
#define WERR_DNS_ERROR_RCODE_YXDOMAIN	W_ERROR(0x0000232E)
#define WERR_DNS_ERROR_RCODE_YXRRSET	W_ERROR(0x0000232F)
#define WERR_DNS_ERROR_RCODE_NXRRSET	W_ERROR(0x00002330)
#define WERR_DNS_ERROR_RCODE_NOTAUTH	W_ERROR(0x00002331)
#define WERR_DNS_ERROR_RCODE_NOTZONE	W_ERROR(0x00002332)
#define WERR_DNS_ERROR_RCODE_BADSIG	W_ERROR(0x00002338)
#define WERR_DNS_ERROR_RCODE_BADKEY	W_ERROR(0x00002339)
#define WERR_DNS_ERROR_RCODE_BADTIME	W_ERROR(0x0000233A)
#define WERR_DNS_ERROR_KEYMASTER_REQUIRED		W_ERROR(0x0000238D)
#define WERR_DNS_ERROR_NOT_ALLOWED_ON_SIGNED_ZONE	W_ERROR(0x0000238E)
#define WERR_DNS_ERROR_INVALID_NSEC3_PARAMETERS		W_ERROR(0x0000238F)
#define WERR_DNS_ERROR_NOT_ENOUGH_SIGNING_KEY_DESCRIPTORS	W_ERROR(0x00002390)
#define WERR_DNS_ERROR_UNSUPPORTED_ALGORITHM		W_ERROR(0x00002391)
#define WERR_DNS_ERROR_INVALID_KEY_SIZE			W_ERROR(0x00002392)
#define WERR_DNS_ERROR_SIGNING_KEY_NOT_ACCESSIBLE	W_ERROR(0x00002393)
#define WERR_DNS_ERROR_KSP_DOES_NOT_SUPPORT_PROTECTION	W_ERROR(0x00002394)
#define WERR_DNS_ERROR_UNEXPECTED_DATA_PROTECTION_ERROR	W_ERROR(0x00002395)
#define WERR_DNS_ERROR_UNEXPECTED_CNG_ERROR		W_ERROR(0x00002396)
#define WERR_DNS_ERROR_UNKNOWN_SIGNING_PARAMETER_VERSION	W_ERROR(0x00002397)
#define WERR_DNS_ERROR_KSP_NOT_ACCESSIBLE		W_ERROR(0x00002398)
#define WERR_DNS_ERROR_TOO_MANY_SKDS			W_ERROR(0x00002399)
#define WERR_DNS_ERROR_INVALID_ROLLOVER_PERIOD		W_ERROR(0x0000239A)
#define WERR_DNS_ERROR_INVALID_INITIAL_ROLLOVER_OFFSET	W_ERROR(0x0000239B)
#define WERR_DNS_ERROR_ROLLOVER_IN_PROGRESS		W_ERROR(0x0000239C)
#define WERR_DNS_ERROR_STANDBY_KEY_NOT_PRESENT		W_ERROR(0x0000239D)
#define WERR_DNS_ERROR_NOT_ALLOWED_ON_ZSK		W_ERROR(0x0000239E)
#define WERR_DNS_ERROR_NOT_ALLOWED_ON_ACTIVE_SKD	W_ERROR(0x0000239F)
#define WERR_DNS_ERROR_ROLLOVER_ALREADY_QUEUED		W_ERROR(0x000023A0)
#define WERR_DNS_ERROR_NOT_ALLOWED_ON_UNSIGNED_ZONE	W_ERROR(0x000023A1)
#define WERR_DNS_ERROR_BAD_KEYMASTER			W_ERROR(0x000023A2)
#define WERR_DNS_ERROR_INVALID_SIGNATURE_VALIDITY_PERIOD	W_ERROR(0x000023A3)
#define WERR_DNS_ERROR_INVALID_NSEC3_ITERATION_COUNT	W_ERROR(0x000023A4)
#define WERR_DNS_ERROR_DNSSEC_IS_DISABLED		W_ERROR(0x000023A5)
#define WERR_DNS_ERROR_INVALID_XML			W_ERROR(0x000023A6)
#define WERR_DNS_ERROR_NO_VALID_TRUST_ANCHORS		W_ERROR(0x000023A7)
#define WERR_DNS_ERROR_ROLLOVER_NOT_POKEABLE		W_ERROR(0x000023A8)
#define WERR_DNS_ERROR_NSEC3_NAME_COLLISION		W_ERROR(0x000023A9)
#define WERR_DNS_INFO_NO_RECORDS	W_ERROR(0x0000251D)
#define WERR_DNS_ERROR_BAD_PACKET	W_ERROR(0x0000251E)
#define WERR_DNS_ERROR_NO_PACKET	W_ERROR(0x0000251F)
#define WERR_DNS_ERROR_RCODE	W_ERROR(0x00002520)
#define WERR_DNS_ERROR_UNSECURE_PACKET	W_ERROR(0x00002521)
#define WERR_DNS_REQUEST_PENDING	W_ERROR(0x00002522)
#define WERR_DNS_ERROR_INVALID_TYPE	W_ERROR(0x0000254F)
#define WERR_DNS_ERROR_INVALID_IP_ADDRESS	W_ERROR(0x00002550)
#define WERR_DNS_ERROR_INVALID_PROPERTY	W_ERROR(0x00002551)
#define WERR_DNS_ERROR_TRY_AGAIN_LATER	W_ERROR(0x00002552)
#define WERR_DNS_ERROR_NOT_UNIQUE	W_ERROR(0x00002553)
#define WERR_DNS_ERROR_NON_RFC_NAME	W_ERROR(0x00002554)
#define WERR_DNS_STATUS_FQDN	W_ERROR(0x00002555)
#define WERR_DNS_STATUS_DOTTED_NAME	W_ERROR(0x00002556)
#define WERR_DNS_STATUS_SINGLE_PART_NAME	W_ERROR(0x00002557)
#define WERR_DNS_ERROR_INVALID_NAME_CHAR	W_ERROR(0x00002558)
#define WERR_DNS_ERROR_NUMERIC_NAME	W_ERROR(0x00002559)
#define WERR_DNS_ERROR_NOT_ALLOWED_ON_ROOT_SERVER	W_ERROR(0x0000255A)
#define WERR_DNS_ERROR_NOT_ALLOWED_UNDER_DELEGATION	W_ERROR(0x0000255B)
#define WERR_DNS_ERROR_CANNOT_FIND_ROOT_HINTS	W_ERROR(0x0000255C)
#define WERR_DNS_ERROR_INCONSISTENT_ROOT_HINTS	W_ERROR(0x0000255D)
#define WERR_DNS_ERROR_DWORD_VALUE_TOO_SMALL	W_ERROR(0x0000255E)
#define WERR_DNS_ERROR_DWORD_VALUE_TOO_LARGE	W_ERROR(0x0000255F)
#define WERR_DNS_ERROR_BACKGROUND_LOADING	W_ERROR(0x00002560)
#define WERR_DNS_ERROR_NOT_ALLOWED_ON_RODC	W_ERROR(0x00002561)
#define WERR_DNS_ERROR_NOT_ALLOWED_UNDER_DNAME	W_ERROR(0x00002562)
#define WERR_DNS_ERROR_DELEGATION_REQUIRED	W_ERROR(0x00002563)
#define WERR_DNS_ERROR_INVALID_POLICY_TABLE	W_ERROR(0x00002564)
#define WERR_DNS_ERROR_ZONE_DOES_NOT_EXIST	W_ERROR(0x00002581)
#define WERR_DNS_ERROR_NO_ZONE_INFO	W_ERROR(0x00002582)
#define WERR_DNS_ERROR_INVALID_ZONE_OPERATION	W_ERROR(0x00002583)
#define WERR_DNS_ERROR_ZONE_CONFIGURATION_ERROR	W_ERROR(0x00002584)
#define WERR_DNS_ERROR_ZONE_HAS_NO_SOA_RECORD	W_ERROR(0x00002585)
#define WERR_DNS_ERROR_ZONE_HAS_NO_NS_RECORDS	W_ERROR(0x00002586)
#define WERR_DNS_ERROR_ZONE_LOCKED	W_ERROR(0x00002587)
#define WERR_DNS_ERROR_ZONE_CREATION_FAILED	W_ERROR(0x00002588)
#define WERR_DNS_ERROR_ZONE_ALREADY_EXISTS	W_ERROR(0x00002589)
#define WERR_DNS_ERROR_AUTOZONE_ALREADY_EXISTS	W_ERROR(0x0000258A)
#define WERR_DNS_ERROR_INVALID_ZONE_TYPE	W_ERROR(0x0000258B)
#define WERR_DNS_ERROR_SECONDARY_REQUIRES_MASTER_IP	W_ERROR(0x0000258C)
#define WERR_DNS_ERROR_ZONE_NOT_SECONDARY	W_ERROR(0x0000258D)
#define WERR_DNS_ERROR_NEED_SECONDARY_ADDRESSES	W_ERROR(0x0000258E)
#define WERR_DNS_ERROR_WINS_INIT_FAILED	W_ERROR(0x0000258F)
#define WERR_DNS_ERROR_NEED_WINS_SERVERS	W_ERROR(0x00002590)
#define WERR_DNS_ERROR_NBSTAT_INIT_FAILED	W_ERROR(0x00002591)
#define WERR_DNS_ERROR_SOA_DELETE_INVALID	W_ERROR(0x00002592)
#define WERR_DNS_ERROR_FORWARDER_ALREADY_EXISTS	W_ERROR(0x00002593)
#define WERR_DNS_ERROR_ZONE_REQUIRES_MASTER_IP	W_ERROR(0x00002594)
#define WERR_DNS_ERROR_ZONE_IS_SHUTDOWN	W_ERROR(0x00002595)
#define WERR_DNS_ERROR_PRIMARY_REQUIRES_DATAFILE	W_ERROR(0x000025B3)
#define WERR_DNS_ERROR_INVALID_DATAFILE_NAME	W_ERROR(0x000025B4)
#define WERR_DNS_ERROR_DATAFILE_OPEN_FAILURE	W_ERROR(0x000025B5)
#define WERR_DNS_ERROR_FILE_WRITEBACK_FAILED	W_ERROR(0x000025B6)
#define WERR_DNS_ERROR_DATAFILE_PARSING	W_ERROR(0x000025B7)
#define WERR_DNS_ERROR_RECORD_DOES_NOT_EXIST	W_ERROR(0x000025E5)
#define WERR_DNS_ERROR_RECORD_FORMAT	W_ERROR(0x000025E6)
#define WERR_DNS_ERROR_NODE_CREATION_FAILED	W_ERROR(0x000025E7)
#define WERR_DNS_ERROR_UNKNOWN_RECORD_TYPE	W_ERROR(0x000025E8)
#define WERR_DNS_ERROR_RECORD_TIMED_OUT	W_ERROR(0x000025E9)
#define WERR_DNS_ERROR_NAME_NOT_IN_ZONE	W_ERROR(0x000025EA)
#define WERR_DNS_ERROR_CNAME_LOOP	W_ERROR(0x000025EB)
#define WERR_DNS_ERROR_NODE_IS_CNAME	W_ERROR(0x000025EC)
#define WERR_DNS_ERROR_CNAME_COLLISION	W_ERROR(0x000025ED)
#define WERR_DNS_ERROR_RECORD_ONLY_AT_ZONE_ROOT	W_ERROR(0x000025EE)
#define WERR_DNS_ERROR_RECORD_ALREADY_EXISTS	W_ERROR(0x000025EF)
#define WERR_DNS_ERROR_SECONDARY_DATA	W_ERROR(0x000025F0)
#define WERR_DNS_ERROR_NO_CREATE_CACHE_DATA	W_ERROR(0x000025F1)
#define WERR_DNS_ERROR_NAME_DOES_NOT_EXIST	W_ERROR(0x000025F2)
#define WERR_DNS_WARNING_PTR_CREATE_FAILED	W_ERROR(0x000025F3)
#define WERR_DNS_WARNING_DOMAIN_UNDELETED	W_ERROR(0x000025F4)
#define WERR_DNS_ERROR_DS_UNAVAILABLE	W_ERROR(0x000025F5)
#define WERR_DNS_ERROR_DS_ZONE_ALREADY_EXISTS	W_ERROR(0x000025F6)
#define WERR_DNS_ERROR_NO_BOOTFILE_IF_DS_ZONE	W_ERROR(0x000025F7)
#define WERR_DNS_ERROR_NODE_IS_DNMAE	W_ERROR(0x000025F8)
#define WERR_DNS_ERROR_DNAME_COLLISION	W_ERROR(0x000025F9)
#define WERR_DNS_ERROR_ALIAS_LOOP	W_ERROR(0x000025FA)
#define WERR_DNS_INFO_AXFR_COMPLETE	W_ERROR(0x00002617)
#define WERR_DNS_ERROR_AXFR	W_ERROR(0x00002618)
#define WERR_DNS_INFO_ADDED_LOCAL_WINS	W_ERROR(0x00002619)
#define WERR_DNS_STATUS_CONTINUE_NEEDED	W_ERROR(0x00002649)
#define WERR_DNS_ERROR_NO_TCPIP	W_ERROR(0x0000267B)
#define WERR_DNS_ERROR_NO_DNS_SERVERS	W_ERROR(0x0000267C)
#define WERR_DNS_ERROR_DP_DOES_NOT_EXIST	W_ERROR(0x000026AD)
#define WERR_DNS_ERROR_DP_ALREADY_EXISTS	W_ERROR(0x000026AE)
#define WERR_DNS_ERROR_DP_NOT_ENLISTED	W_ERROR(0x000026AF)
#define WERR_DNS_ERROR_DP_ALREADY_ENLISTED	W_ERROR(0x000026B0)
#define WERR_DNS_ERROR_DP_NOT_AVAILABLE	W_ERROR(0x000026B1)
#define WERR_DNS_ERROR_DP_FSMO_ERROR	W_ERROR(0x000026B2)
#define WERR_IPSEC_QM_POLICY_EXISTS	W_ERROR(0x000032C8)
#define WERR_IPSEC_QM_POLICY_NOT_FOUND	W_ERROR(0x000032C9)
#define WERR_IPSEC_QM_POLICY_IN_USE	W_ERROR(0x000032CA)
#define WERR_IPSEC_MM_POLICY_EXISTS	W_ERROR(0x000032CB)
#define WERR_IPSEC_MM_POLICY_NOT_FOUND	W_ERROR(0x000032CC)
#define WERR_IPSEC_MM_POLICY_IN_USE	W_ERROR(0x000032CD)
#define WERR_IPSEC_MM_FILTER_EXISTS	W_ERROR(0x000032CE)
#define WERR_IPSEC_MM_FILTER_NOT_FOUND	W_ERROR(0x000032CF)
#define WERR_IPSEC_TRANSPORT_FILTER_EXISTS	W_ERROR(0x000032D0)
#define WERR_IPSEC_TRANSPORT_FILTER_NOT_FOUND	W_ERROR(0x000032D1)
#define WERR_IPSEC_MM_AUTH_EXISTS	W_ERROR(0x000032D2)
#define WERR_IPSEC_MM_AUTH_NOT_FOUND	W_ERROR(0x000032D3)
#define WERR_IPSEC_MM_AUTH_IN_USE	W_ERROR(0x000032D4)
#define WERR_IPSEC_DEFAULT_MM_POLICY_NOT_FOUND	W_ERROR(0x000032D5)
#define WERR_IPSEC_DEFAULT_MM_AUTH_NOT_FOUND	W_ERROR(0x000032D6)
#define WERR_IPSEC_DEFAULT_QM_POLICY_NOT_FOUND	W_ERROR(0x000032D7)
#define WERR_IPSEC_TUNNEL_FILTER_EXISTS	W_ERROR(0x000032D8)
#define WERR_IPSEC_TUNNEL_FILTER_NOT_FOUND	W_ERROR(0x000032D9)
#define WERR_IPSEC_MM_FILTER_PENDING_DELETION	W_ERROR(0x000032DA)
#define WERR_IPSEC_TRANSPORT_FILTER_ENDING_DELETION	W_ERROR(0x000032DB)
#define WERR_IPSEC_TUNNEL_FILTER_PENDING_DELETION	W_ERROR(0x000032DC)
#define WERR_IPSEC_MM_POLICY_PENDING_ELETION	W_ERROR(0x000032DD)
#define WERR_IPSEC_MM_AUTH_PENDING_DELETION	W_ERROR(0x000032DE)
#define WERR_IPSEC_QM_POLICY_PENDING_DELETION	W_ERROR(0x000032DF)
#define WERR_IPSEC_IKE_NEG_STATUS_BEGIN	W_ERROR(0x000035E8)
#define WERR_IPSEC_IKE_AUTH_FAIL	W_ERROR(0x000035E9)
#define WERR_IPSEC_IKE_ATTRIB_FAIL	W_ERROR(0x000035EA)
#define WERR_IPSEC_IKE_NEGOTIATION_PENDING	W_ERROR(0x000035EB)
#define WERR_IPSEC_IKE_GENERAL_PROCESSING_ERROR	W_ERROR(0x000035EC)
#define WERR_IPSEC_IKE_TIMED_OUT	W_ERROR(0x000035ED)
#define WERR_IPSEC_IKE_NO_CERT	W_ERROR(0x000035EE)
#define WERR_IPSEC_IKE_SA_DELETED	W_ERROR(0x000035EF)
#define WERR_IPSEC_IKE_SA_REAPED	W_ERROR(0x000035F0)
#define WERR_IPSEC_IKE_MM_ACQUIRE_DROP	W_ERROR(0x000035F1)
#define WERR_IPSEC_IKE_QM_ACQUIRE_DROP	W_ERROR(0x000035F2)
#define WERR_IPSEC_IKE_QUEUE_DROP_MM	W_ERROR(0x000035F3)
#define WERR_IPSEC_IKE_QUEUE_DROP_NO_MM	W_ERROR(0x000035F4)
#define WERR_IPSEC_IKE_DROP_NO_RESPONSE	W_ERROR(0x000035F5)
#define WERR_IPSEC_IKE_MM_DELAY_DROP	W_ERROR(0x000035F6)
#define WERR_IPSEC_IKE_QM_DELAY_DROP	W_ERROR(0x000035F7)
#define WERR_IPSEC_IKE_ERROR	W_ERROR(0x000035F8)
#define WERR_IPSEC_IKE_CRL_FAILED	W_ERROR(0x000035F9)
#define WERR_IPSEC_IKE_INVALID_KEY_USAGE	W_ERROR(0x000035FA)
#define WERR_IPSEC_IKE_INVALID_CERT_TYPE	W_ERROR(0x000035FB)
#define WERR_IPSEC_IKE_NO_PRIVATE_KEY	W_ERROR(0x000035FC)
#define WERR_IPSEC_IKE_DH_FAIL	W_ERROR(0x000035FE)
#define WERR_IPSEC_IKE_INVALID_HEADER	W_ERROR(0x00003600)
#define WERR_IPSEC_IKE_NO_POLICY	W_ERROR(0x00003601)
#define WERR_IPSEC_IKE_INVALID_SIGNATURE	W_ERROR(0x00003602)
#define WERR_IPSEC_IKE_KERBEROS_ERROR	W_ERROR(0x00003603)
#define WERR_IPSEC_IKE_NO_PUBLIC_KEY	W_ERROR(0x00003604)
#define WERR_IPSEC_IKE_PROCESS_ERR	W_ERROR(0x00003605)
#define WERR_IPSEC_IKE_PROCESS_ERR_SA	W_ERROR(0x00003606)
#define WERR_IPSEC_IKE_PROCESS_ERR_PROP	W_ERROR(0x00003607)
#define WERR_IPSEC_IKE_PROCESS_ERR_TRANS	W_ERROR(0x00003608)
#define WERR_IPSEC_IKE_PROCESS_ERR_KE	W_ERROR(0x00003609)
#define WERR_IPSEC_IKE_PROCESS_ERR_ID	W_ERROR(0x0000360A)
#define WERR_IPSEC_IKE_PROCESS_ERR_CERT	W_ERROR(0x0000360B)
#define WERR_IPSEC_IKE_PROCESS_ERR_CERT_REQ	W_ERROR(0x0000360C)
#define WERR_IPSEC_IKE_PROCESS_ERR_HASH	W_ERROR(0x0000360D)
#define WERR_IPSEC_IKE_PROCESS_ERR_SIG	W_ERROR(0x0000360E)
#define WERR_IPSEC_IKE_PROCESS_ERR_NONCE	W_ERROR(0x0000360F)
#define WERR_IPSEC_IKE_PROCESS_ERR_NOTIFY	W_ERROR(0x00003610)
#define WERR_IPSEC_IKE_PROCESS_ERR_DELETE	W_ERROR(0x00003611)
#define WERR_IPSEC_IKE_PROCESS_ERR_VENDOR	W_ERROR(0x00003612)
#define WERR_IPSEC_IKE_INVALID_PAYLOAD	W_ERROR(0x00003613)
#define WERR_IPSEC_IKE_LOAD_SOFT_SA	W_ERROR(0x00003614)
#define WERR_IPSEC_IKE_SOFT_SA_TORN_DOWN	W_ERROR(0x00003615)
#define WERR_IPSEC_IKE_INVALID_COOKIE	W_ERROR(0x00003616)
#define WERR_IPSEC_IKE_NO_PEER_CERT	W_ERROR(0x00003617)
#define WERR_IPSEC_IKE_PEER_CRL_FAILED	W_ERROR(0x00003618)
#define WERR_IPSEC_IKE_POLICY_CHANGE	W_ERROR(0x00003619)
#define WERR_IPSEC_IKE_NO_MM_POLICY	W_ERROR(0x0000361A)
#define WERR_IPSEC_IKE_NOTCBPRIV	W_ERROR(0x0000361B)
#define WERR_IPSEC_IKE_SECLOADFAIL	W_ERROR(0x0000361C)
#define WERR_IPSEC_IKE_FAILSSPINIT	W_ERROR(0x0000361D)
#define WERR_IPSEC_IKE_FAILQUERYSSP	W_ERROR(0x0000361E)
#define WERR_IPSEC_IKE_SRVACQFAIL	W_ERROR(0x0000361F)
#define WERR_IPSEC_IKE_SRVQUERYCRED	W_ERROR(0x00003620)
#define WERR_IPSEC_IKE_GETSPIFAIL	W_ERROR(0x00003621)
#define WERR_IPSEC_IKE_INVALID_FILTER	W_ERROR(0x00003622)
#define WERR_IPSEC_IKE_OUT_OF_MEMORY	W_ERROR(0x00003623)
#define WERR_IPSEC_IKE_ADD_UPDATE_KEY_FAILED	W_ERROR(0x00003624)
#define WERR_IPSEC_IKE_INVALID_POLICY	W_ERROR(0x00003625)
#define WERR_IPSEC_IKE_UNKNOWN_DOI	W_ERROR(0x00003626)
#define WERR_IPSEC_IKE_INVALID_SITUATION	W_ERROR(0x00003627)
#define WERR_IPSEC_IKE_DH_FAILURE	W_ERROR(0x00003628)
#define WERR_IPSEC_IKE_INVALID_GROUP	W_ERROR(0x00003629)
#define WERR_IPSEC_IKE_ENCRYPT	W_ERROR(0x0000362A)
#define WERR_IPSEC_IKE_DECRYPT	W_ERROR(0x0000362B)
#define WERR_IPSEC_IKE_POLICY_MATCH	W_ERROR(0x0000362C)
#define WERR_IPSEC_IKE_UNSUPPORTED_ID	W_ERROR(0x0000362D)
#define WERR_IPSEC_IKE_INVALID_HASH	W_ERROR(0x0000362E)
#define WERR_IPSEC_IKE_INVALID_HASH_ALG	W_ERROR(0x0000362F)
#define WERR_IPSEC_IKE_INVALID_HASH_SIZE	W_ERROR(0x00003630)
#define WERR_IPSEC_IKE_INVALID_ENCRYPT_ALG	W_ERROR(0x00003631)
#define WERR_IPSEC_IKE_INVALID_AUTH_ALG	W_ERROR(0x00003632)
#define WERR_IPSEC_IKE_INVALID_SIG	W_ERROR(0x00003633)
#define WERR_IPSEC_IKE_LOAD_FAILED	W_ERROR(0x00003634)
#define WERR_IPSEC_IKE_RPC_DELETE	W_ERROR(0x00003635)
#define WERR_IPSEC_IKE_BENIGN_REINIT	W_ERROR(0x00003636)
#define WERR_IPSEC_IKE_INVALID_RESPONDER_LIFETIME_NOTIFY	W_ERROR(0x00003637)
#define WERR_IPSEC_IKE_INVALID_CERT_KEYLEN	W_ERROR(0x00003639)
#define WERR_IPSEC_IKE_MM_LIMIT	W_ERROR(0x0000363A)
#define WERR_IPSEC_IKE_NEGOTIATION_DISABLED	W_ERROR(0x0000363B)
#define WERR_IPSEC_IKE_QM_LIMIT	W_ERROR(0x0000363C)
#define WERR_IPSEC_IKE_MM_EXPIRED	W_ERROR(0x0000363D)
#define WERR_IPSEC_IKE_PEER_MM_ASSUMED_INVALID	W_ERROR(0x0000363E)
#define WERR_IPSEC_IKE_CERT_CHAIN_POLICY_MISMATCH	W_ERROR(0x0000363F)
#define WERR_IPSEC_IKE_UNEXPECTED_MESSAGE_ID	W_ERROR(0x00003640)
#define WERR_IPSEC_IKE_INVALID_UMATTS	W_ERROR(0x00003641)
#define WERR_IPSEC_IKE_DOS_COOKIE_SENT	W_ERROR(0x00003642)
#define WERR_IPSEC_IKE_SHUTTING_DOWN	W_ERROR(0x00003643)
#define WERR_IPSEC_IKE_CGA_AUTH_FAILED	W_ERROR(0x00003644)
#define WERR_IPSEC_IKE_PROCESS_ERR_NATOA	W_ERROR(0x00003645)
#define WERR_IPSEC_IKE_INVALID_MM_FOR_QM	W_ERROR(0x00003646)
#define WERR_IPSEC_IKE_QM_EXPIRED	W_ERROR(0x00003647)
#define WERR_IPSEC_IKE_TOO_MANY_FILTERS	W_ERROR(0x00003648)
#define WERR_IPSEC_IKE_NEG_STATUS_END	W_ERROR(0x00003649)
#define WERR_SXS_SECTION_NOT_FOUND	W_ERROR(0x000036B0)
#define WERR_SXS_CANT_GEN_ACTCTX	W_ERROR(0x000036B1)
#define WERR_SXS_INVALID_ACTCTXDATA_FORMAT	W_ERROR(0x000036B2)
#define WERR_SXS_ASSEMBLY_NOT_FOUND	W_ERROR(0x000036B3)
#define WERR_SXS_MANIFEST_FORMAT_ERROR	W_ERROR(0x000036B4)
#define WERR_SXS_MANIFEST_PARSE_ERROR	W_ERROR(0x000036B5)
#define WERR_SXS_ACTIVATION_CONTEXT_DISABLED	W_ERROR(0x000036B6)
#define WERR_SXS_KEY_NOT_FOUND	W_ERROR(0x000036B7)
#define WERR_SXS_VERSION_CONFLICT	W_ERROR(0x000036B8)
#define WERR_SXS_WRONG_SECTION_TYPE	W_ERROR(0x000036B9)
#define WERR_SXS_THREAD_QUERIES_DISABLED	W_ERROR(0x000036BA)
#define WERR_SXS_PROCESS_DEFAULT_ALREADY_SET	W_ERROR(0x000036BB)
#define WERR_SXS_UNKNOWN_ENCODING_GROUP	W_ERROR(0x000036BC)
#define WERR_SXS_UNKNOWN_ENCODING	W_ERROR(0x000036BD)
#define WERR_SXS_INVALID_XML_NAMESPACE_URI	W_ERROR(0x000036BE)
#define WERR_SXS_ROOT_MANIFEST_DEPENDENCY_OT_INSTALLED	W_ERROR(0x000036BF)
#define WERR_SXS_LEAF_MANIFEST_DEPENDENCY_NOT_INSTALLED	W_ERROR(0x000036C0)
#define WERR_SXS_INVALID_ASSEMBLY_IDENTITY_ATTRIBUTE	W_ERROR(0x000036C1)
#define WERR_SXS_MANIFEST_MISSING_REQUIRED_DEFAULT_NAMESPACE	W_ERROR(0x000036C2)
#define WERR_SXS_MANIFEST_INVALID_REQUIRED_DEFAULT_NAMESPACE	W_ERROR(0x000036C3)
#define WERR_SXS_PRIVATE_MANIFEST_CROSS_PATH_WITH_REPARSE_POINT	W_ERROR(0x000036C4)
#define WERR_SXS_DUPLICATE_DLL_NAME	W_ERROR(0x000036C5)
#define WERR_SXS_DUPLICATE_WINDOWCLASS_NAME	W_ERROR(0x000036C6)
#define WERR_SXS_DUPLICATE_CLSID	W_ERROR(0x000036C7)
#define WERR_SXS_DUPLICATE_IID	W_ERROR(0x000036C8)
#define WERR_SXS_DUPLICATE_TLBID	W_ERROR(0x000036C9)
#define WERR_SXS_DUPLICATE_PROGID	W_ERROR(0x000036CA)
#define WERR_SXS_DUPLICATE_ASSEMBLY_NAME	W_ERROR(0x000036CB)
#define WERR_SXS_FILE_HASH_MISMATCH	W_ERROR(0x000036CC)
#define WERR_SXS_POLICY_PARSE_ERROR	W_ERROR(0x000036CD)
#define WERR_SXS_XML_E_MISSINGQUOTE	W_ERROR(0x000036CE)
#define WERR_SXS_XML_E_COMMENTSYNTAX	W_ERROR(0x000036CF)
#define WERR_SXS_XML_E_BADSTARTNAMECHAR	W_ERROR(0x000036D0)
#define WERR_SXS_XML_E_BADNAMECHAR	W_ERROR(0x000036D1)
#define WERR_SXS_XML_E_BADCHARINSTRING	W_ERROR(0x000036D2)
#define WERR_SXS_XML_E_XMLDECLSYNTAX	W_ERROR(0x000036D3)
#define WERR_SXS_XML_E_BADCHARDATA	W_ERROR(0x000036D4)
#define WERR_SXS_XML_E_MISSINGWHITESPACE	W_ERROR(0x000036D5)
#define WERR_SXS_XML_E_EXPECTINGTAGEND	W_ERROR(0x000036D6)
#define WERR_SXS_XML_E_MISSINGSEMICOLON	W_ERROR(0x000036D7)
#define WERR_SXS_XML_E_UNBALANCEDPAREN	W_ERROR(0x000036D8)
#define WERR_SXS_XML_E_INTERNALERROR	W_ERROR(0x000036D9)
#define WERR_SXS_XML_E_UNEXPECTED_WHITESPACE	W_ERROR(0x000036DA)
#define WERR_SXS_XML_E_INCOMPLETE_ENCODING	W_ERROR(0x000036DB)
#define WERR_SXS_XML_E_MISSING_PAREN	W_ERROR(0x000036DC)
#define WERR_SXS_XML_E_EXPECTINGCLOSEQUOTE	W_ERROR(0x000036DD)
#define WERR_SXS_XML_E_MULTIPLE_COLONS	W_ERROR(0x000036DE)
#define WERR_SXS_XML_E_INVALID_DECIMAL	W_ERROR(0x000036DF)
#define WERR_SXS_XML_E_INVALID_HEXIDECIMAL	W_ERROR(0x000036E0)
#define WERR_SXS_XML_E_INVALID_UNICODE	W_ERROR(0x000036E1)
#define WERR_SXS_XML_E_WHITESPACEORQUESTIONMARK	W_ERROR(0x000036E2)
#define WERR_SXS_XML_E_UNEXPECTEDENDTAG	W_ERROR(0x000036E3)
#define WERR_SXS_XML_E_UNCLOSEDTAG	W_ERROR(0x000036E4)
#define WERR_SXS_XML_E_DUPLICATEATTRIBUTE	W_ERROR(0x000036E5)
#define WERR_SXS_XML_E_MULTIPLEROOTS	W_ERROR(0x000036E6)
#define WERR_SXS_XML_E_INVALIDATROOTLEVEL	W_ERROR(0x000036E7)
#define WERR_SXS_XML_E_BADXMLDECL	W_ERROR(0x000036E8)
#define WERR_SXS_XML_E_MISSINGROOT	W_ERROR(0x000036E9)
#define WERR_SXS_XML_E_UNEXPECTEDEOF	W_ERROR(0x000036EA)
#define WERR_SXS_XML_E_BADPEREFINSUBSET	W_ERROR(0x000036EB)
#define WERR_SXS_XML_E_UNCLOSEDSTARTTAG	W_ERROR(0x000036EC)
#define WERR_SXS_XML_E_UNCLOSEDENDTAG	W_ERROR(0x000036ED)
#define WERR_SXS_XML_E_UNCLOSEDSTRING	W_ERROR(0x000036EE)
#define WERR_SXS_XML_E_UNCLOSEDCOMMENT	W_ERROR(0x000036EF)
#define WERR_SXS_XML_E_UNCLOSEDDECL	W_ERROR(0x000036F0)
#define WERR_SXS_XML_E_UNCLOSEDCDATA	W_ERROR(0x000036F1)
#define WERR_SXS_XML_E_RESERVEDNAMESPACE	W_ERROR(0x000036F2)
#define WERR_SXS_XML_E_INVALIDENCODING	W_ERROR(0x000036F3)
#define WERR_SXS_XML_E_INVALIDSWITCH	W_ERROR(0x000036F4)
#define WERR_SXS_XML_E_BADXMLCASE	W_ERROR(0x000036F5)
#define WERR_SXS_XML_E_INVALID_STANDALONE	W_ERROR(0x000036F6)
#define WERR_SXS_XML_E_UNEXPECTED_STANDALONE	W_ERROR(0x000036F7)
#define WERR_SXS_XML_E_INVALID_VERSION	W_ERROR(0x000036F8)
#define WERR_SXS_XML_E_MISSINGEQUALS	W_ERROR(0x000036F9)
#define WERR_SXS_PROTECTION_RECOVERY_FAILED	W_ERROR(0x000036FA)
#define WERR_SXS_PROTECTION_PUBLIC_KEY_OO_SHORT	W_ERROR(0x000036FB)
#define WERR_SXS_PROTECTION_CATALOG_NOT_VALID	W_ERROR(0x000036FC)
#define WERR_SXS_UNTRANSLATABLE_HRESULT	W_ERROR(0x000036FD)
#define WERR_SXS_PROTECTION_CATALOG_FILE_MISSING	W_ERROR(0x000036FE)
#define WERR_SXS_MISSING_ASSEMBLY_IDENTITY_ATTRIBUTE	W_ERROR(0x000036FF)
#define WERR_SXS_INVALID_ASSEMBLY_IDENTITY_ATTRIBUTE_NAME	W_ERROR(0x00003700)
#define WERR_SXS_ASSEMBLY_MISSING	W_ERROR(0x00003701)
#define WERR_SXS_CORRUPT_ACTIVATION_STACK	W_ERROR(0x00003702)
#define WERR_SXS_CORRUPTION	W_ERROR(0x00003703)
#define WERR_SXS_EARLY_DEACTIVATION	W_ERROR(0x00003704)
#define WERR_SXS_INVALID_DEACTIVATION	W_ERROR(0x00003705)
#define WERR_SXS_MULTIPLE_DEACTIVATION	W_ERROR(0x00003706)
#define WERR_SXS_PROCESS_TERMINATION_REQUESTED	W_ERROR(0x00003707)
#define WERR_SXS_RELEASE_ACTIVATION_ONTEXT	W_ERROR(0x00003708)
#define WERR_SXS_SYSTEM_DEFAULT_ACTIVATION_CONTEXT_EMPTY	W_ERROR(0x00003709)
#define WERR_SXS_INVALID_IDENTITY_ATTRIBUTE_VALUE	W_ERROR(0x0000370A)
#define WERR_SXS_INVALID_IDENTITY_ATTRIBUTE_NAME	W_ERROR(0x0000370B)
#define WERR_SXS_IDENTITY_DUPLICATE_ATTRIBUTE	W_ERROR(0x0000370C)
#define WERR_SXS_IDENTITY_PARSE_ERROR	W_ERROR(0x0000370D)
#define WERR_MALFORMED_SUBSTITUTION_STRING	W_ERROR(0x0000370E)
#define WERR_SXS_INCORRECT_PUBLIC_KEY_OKEN	W_ERROR(0x0000370F)
#define WERR_UNMAPPED_SUBSTITUTION_STRING	W_ERROR(0x00003710)
#define WERR_SXS_ASSEMBLY_NOT_LOCKED	W_ERROR(0x00003711)
#define WERR_SXS_COMPONENT_STORE_CORRUPT	W_ERROR(0x00003712)
#define WERR_ADVANCED_INSTALLER_FAILED	W_ERROR(0x00003713)
#define WERR_XML_ENCODING_MISMATCH	W_ERROR(0x00003714)
#define WERR_SXS_MANIFEST_IDENTITY_SAME_BUT_CONTENTS_DIFFERENT	W_ERROR(0x00003715)
#define WERR_SXS_IDENTITIES_DIFFERENT	W_ERROR(0x00003716)
#define WERR_SXS_ASSEMBLY_IS_NOT_A_DEPLOYMENT	W_ERROR(0x00003717)
#define WERR_SXS_FILE_NOT_PART_OF_ASSEMBLY	W_ERROR(0x00003718)
#define WERR_SXS_MANIFEST_TOO_BIG	W_ERROR(0x00003719)
#define WERR_SXS_SETTING_NOT_REGISTERED	W_ERROR(0x0000371A)
#define WERR_SXS_TRANSACTION_CLOSURE_INCOMPLETE	W_ERROR(0x0000371B)
#define WERR_EVT_INVALID_CHANNEL_PATH	W_ERROR(0x00003A98)
#define WERR_EVT_INVALID_QUERY	W_ERROR(0x00003A99)
#define WERR_EVT_PUBLISHER_METADATA_NOT_FOUND	W_ERROR(0x00003A9A)
#define WERR_EVT_EVENT_TEMPLATE_NOT_FOUND	W_ERROR(0x00003A9B)
#define WERR_EVT_INVALID_PUBLISHER_NAME	W_ERROR(0x00003A9C)
#define WERR_EVT_INVALID_EVENT_DATA	W_ERROR(0x00003A9D)
#define WERR_EVT_CHANNEL_NOT_FOUND	W_ERROR(0x00003A9F)
#define WERR_EVT_MALFORMED_XML_TEXT	W_ERROR(0x00003AA0)
#define WERR_EVT_SUBSCRIPTION_TO_DIRECT_CHANNEL	W_ERROR(0x00003AA1)
#define WERR_EVT_CONFIGURATION_ERROR	W_ERROR(0x00003AA2)
#define WERR_EVT_QUERY_RESULT_STALE	W_ERROR(0x00003AA3)
#define WERR_EVT_QUERY_RESULT_INVALID_POSITION	W_ERROR(0x00003AA4)
#define WERR_EVT_NON_VALIDATING_MSXML	W_ERROR(0x00003AA5)
#define WERR_EVT_FILTER_ALREADYSCOPED	W_ERROR(0x00003AA6)
#define WERR_EVT_FILTER_NOTELTSET	W_ERROR(0x00003AA7)
#define WERR_EVT_FILTER_INVARG	W_ERROR(0x00003AA8)
#define WERR_EVT_FILTER_INVTEST	W_ERROR(0x00003AA9)
#define WERR_EVT_FILTER_INVTYPE	W_ERROR(0x00003AAA)
#define WERR_EVT_FILTER_PARSEERR	W_ERROR(0x00003AAB)
#define WERR_EVT_FILTER_UNSUPPORTEDOP	W_ERROR(0x00003AAC)
#define WERR_EVT_FILTER_UNEXPECTEDTOKEN	W_ERROR(0x00003AAD)
#define WERR_EVT_INVALID_OPERATION_OVER_ENABLED_DIRECT_CHANNEL	W_ERROR(0x00003AAE)
#define WERR_EVT_INVALID_CHANNEL_PROPERTY_VALUE	W_ERROR(0x00003AAF)
#define WERR_EVT_INVALID_PUBLISHER_PROPERTY_VALUE	W_ERROR(0x00003AB0)
#define WERR_EVT_CHANNEL_CANNOT_ACTIVATE	W_ERROR(0x00003AB1)
#define WERR_EVT_FILTER_TOO_COMPLEX	W_ERROR(0x00003AB2)
#define WERR_EVT_MESSAGE_NOT_FOUND	W_ERROR(0x00003AB3)
#define WERR_EVT_MESSAGE_ID_NOT_FOUND	W_ERROR(0x00003AB4)
#define WERR_EVT_UNRESOLVED_VALUE_INSERT	W_ERROR(0x00003AB5)
#define WERR_EVT_UNRESOLVED_PARAMETER_INSERT	W_ERROR(0x00003AB6)
#define WERR_EVT_MAX_INSERTS_REACHED	W_ERROR(0x00003AB7)
#define WERR_EVT_EVENT_DEFINITION_NOT_OUND	W_ERROR(0x00003AB8)
#define WERR_EVT_MESSAGE_LOCALE_NOT_FOUND	W_ERROR(0x00003AB9)
#define WERR_EVT_VERSION_TOO_OLD	W_ERROR(0x00003ABA)
#define WERR_EVT_VERSION_TOO_NEW	W_ERROR(0x00003ABB)
#define WERR_EVT_CANNOT_OPEN_CHANNEL_OF_QUERY	W_ERROR(0x00003ABC)
#define WERR_EVT_PUBLISHER_DISABLED	W_ERROR(0x00003ABD)
#define WERR_EC_SUBSCRIPTION_CANNOT_ACTIVATE	W_ERROR(0x00003AE8)
#define WERR_EC_LOG_DISABLED	W_ERROR(0x00003AE9)
#define WERR_MUI_FILE_NOT_FOUND	W_ERROR(0x00003AFC)
#define WERR_MUI_INVALID_FILE	W_ERROR(0x00003AFD)
#define WERR_MUI_INVALID_RC_CONFIG	W_ERROR(0x00003AFE)
#define WERR_MUI_INVALID_LOCALE_NAME	W_ERROR(0x00003AFF)
#define WERR_MUI_INVALID_ULTIMATEFALLBACK_NAME	W_ERROR(0x00003B00)
#define WERR_MUI_FILE_NOT_LOADED	W_ERROR(0x00003B01)
#define WERR_RESOURCE_ENUM_USER_STOP	W_ERROR(0x00003B02)
#define WERR_MUI_INTLSETTINGS_UILANG_NOT_INSTALLED	W_ERROR(0x00003B03)
#define WERR_MUI_INTLSETTINGS_INVALID_LOCALE_NAME	W_ERROR(0x00003B04)
#define WERR_MCA_INVALID_CAPABILITIES_STRING	W_ERROR(0x00003B60)
#define WERR_MCA_INVALID_VCP_VERSION	W_ERROR(0x00003B61)
#define WERR_MCA_MONITOR_VIOLATES_MCCS_SPECIFICATION	W_ERROR(0x00003B62)
#define WERR_MCA_MCCS_VERSION_MISMATCH	W_ERROR(0x00003B63)
#define WERR_MCA_UNSUPPORTED_MCCS_VERSION	W_ERROR(0x00003B64)
#define WERR_MCA_INTERNAL_ERROR	W_ERROR(0x00003B65)
#define WERR_MCA_INVALID_TECHNOLOGY_TYPE_RETURNED	W_ERROR(0x00003B66)
#define WERR_MCA_UNSUPPORTED_COLOR_TEMPERATURE	W_ERROR(0x00003B67)
#define WERR_AMBIGUOUS_SYSTEM_DEVICE	W_ERROR(0x00003B92)
#define WERR_SYSTEM_DEVICE_NOT_FOUND	W_ERROR(0x00003BC3)
/* END GENERATED-WIN32-ERROR-CODES */

/*****************************************************************************
 returns a windows error message.  not amazingly helpful, but better than a number.
 *****************************************************************************/
const char *win_errstr(WERROR werror);

const char *get_friendly_werror_msg(WERROR werror);


#endif
