#    Licensed under the Apache License, Version 2.0 (the "License"); you may
#    not use this file except in compliance with the License. You may obtain
#    a copy of the License at
#
#         http://www.apache.org/licenses/LICENSE-2.0
#
#    Unless required by applicable law or agreed to in writing, software
#    distributed under the License is distributed on an "AS IS" BASIS, WITHOUT
#    WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied. See the
#    License for the specific language governing permissions and limitations
#    under the License.

import time
import uuid

from openstackclient.tests.functional.volume.v1 import common


class VolumeTypeTests(common.BaseVolumeTests):
    """Functional tests for volume type. """

    NAME = uuid.uuid4().hex
    HEADERS = ['"Name"']
    FIELDS = ['name']

    @classmethod
    def setUpClass(cls):
        super(VolumeTypeTests, cls).setUpClass()
        opts = cls.get_opts(cls.FIELDS)
        raw_output = cls.openstack('volume type create ' + cls.NAME + opts)
        expected = cls.NAME + '\n'
        cls.assertOutput(expected, raw_output)

    @classmethod
    def tearDownClass(cls):
        raw_output = cls.openstack('volume type delete ' + cls.NAME)
        cls.assertOutput('', raw_output)

    def test_volume_type_list(self):
        opts = self.get_opts(self.HEADERS)
        raw_output = self.openstack('volume type list' + opts)
        self.assertIn(self.NAME, raw_output)

    def test_volume_type_show(self):
        opts = self.get_opts(self.FIELDS)
        raw_output = self.openstack('volume type show ' + self.NAME + opts)
        self.assertEqual(self.NAME + "\n", raw_output)

    def test_volume_type_set_unset_properties(self):
        raw_output = self.openstack(
            'volume type set --property a=b --property c=d ' + self.NAME)
        self.assertEqual("", raw_output)

        opts = self.get_opts(["properties"])
        raw_output = self.openstack('volume type show ' + self.NAME + opts)
        self.assertEqual("a='b', c='d'\n", raw_output)

        raw_output = self.openstack('volume type unset --property a '
                                    + self.NAME)
        self.assertEqual("", raw_output)
        raw_output = self.openstack('volume type show ' + self.NAME + opts)
        self.assertEqual("c='d'\n", raw_output)

    def test_volume_type_set_unset_multiple_properties(self):
        raw_output = self.openstack(
            'volume type set --property a=b --property c=d ' + self.NAME)
        self.assertEqual("", raw_output)

        opts = self.get_opts(["properties"])
        raw_output = self.openstack('volume type show ' + self.NAME + opts)
        self.assertEqual("a='b', c='d'\n", raw_output)

        raw_output = self.openstack(
            'volume type unset --property a --property c ' + self.NAME)
        self.assertEqual("", raw_output)
        raw_output = self.openstack('volume type show ' + self.NAME + opts)
        self.assertEqual("\n", raw_output)

    def test_multi_delete(self):
        vol_type1 = uuid.uuid4().hex
        vol_type2 = uuid.uuid4().hex
        self.openstack('volume type create ' + vol_type1)
        time.sleep(5)
        self.openstack('volume type create ' + vol_type2)
        time.sleep(5)
        cmd = 'volume type delete %s %s' % (vol_type1, vol_type2)
        time.sleep(5)
        raw_output = self.openstack(cmd)
        self.assertOutput('', raw_output)

    # NOTE: Add some basic funtional tests with the old format to
    #       make sure the command works properly, need to change
    #       these to new test format when beef up all tests for
    #       volume tye commands.
    def test_encryption_type(self):
        encryption_type = uuid.uuid4().hex
        # test create new encryption type
        opts = self.get_opts(['encryption'])
        raw_output = self.openstack(
            'volume type create '
            '--encryption-provider LuksEncryptor '
            '--encryption-cipher aes-xts-plain64 '
            '--encryption-key-size 128 '
            '--encryption-control-location front-end ' +
            encryption_type + opts)
        expected = ["provider='LuksEncryptor'",
                    "cipher='aes-xts-plain64'",
                    "key_size='128'",
                    "control_location='front-end'"]
        for attr in expected:
            self.assertIn(attr, raw_output)
        # test show encryption type
        opts = self.get_opts(['encryption'])
        raw_output = self.openstack(
            'volume type show --encryption-type ' + encryption_type + opts)
        expected = ["provider='LuksEncryptor'",
                    "cipher='aes-xts-plain64'",
                    "key_size='128'",
                    "control_location='front-end'"]
        for attr in expected:
            self.assertIn(attr, raw_output)
        # test list encryption type
        opts = self.get_opts(['Encryption'])
        raw_output = self.openstack(
            'volume type list --encryption-type ' + opts)
        expected = ["provider='LuksEncryptor'",
                    "cipher='aes-xts-plain64'",
                    "key_size='128'",
                    "control_location='front-end'"]
        for attr in expected:
            self.assertIn(attr, raw_output)
        # test set new encryption type
        raw_output = self.openstack(
            'volume type set '
            '--encryption-provider LuksEncryptor '
            '--encryption-cipher aes-xts-plain64 '
            '--encryption-key-size 128 '
            '--encryption-control-location front-end ' +
            self.NAME)
        self.assertEqual('', raw_output)
        opts = self.get_opts(['encryption'])
        raw_output = self.openstack(
            'volume type show --encryption-type ' + self.NAME + opts)
        expected = ["provider='LuksEncryptor'",
                    "cipher='aes-xts-plain64'",
                    "key_size='128'",
                    "control_location='front-end'"]
        for attr in expected:
            self.assertIn(attr, raw_output)
        # test unset encryption type
        raw_output = self.openstack(
            'volume type unset --encryption-type ' + self.NAME)
        self.assertEqual('', raw_output)
        opts = self.get_opts(['encryption'])
        raw_output = self.openstack(
            'volume type show --encryption-type ' + self.NAME + opts)
        self.assertEqual('\n', raw_output)
        # test delete encryption type
        raw_output = self.openstack('volume type delete ' + encryption_type)
        self.assertEqual('', raw_output)
