// Copyright 2012 The Go Authors. All rights reserved.
// Use of this source code is governed by a BSD-style
// license that can be found in the LICENSE file.

package main

import (
	"bytes"
	"fmt"
	"os"
	"sort"
)

/*
 * Helpers for building cmd/go and cmd/cgo.
 */

// mkzdefaultcc writes zdefaultcc.go:
//
//	package main
//	const defaultCC = <defaultcc>
//	const defaultCXX = <defaultcxx>
//	const defaultPkgConfig = <defaultpkgconfig>
//
// It is invoked to write cmd/go/zdefaultcc.go
// but we also write cmd/cgo/zdefaultcc.go
func mkzdefaultcc(dir, file string) {
	out := fmt.Sprintf(
		"// auto generated by go tool dist\n"+
			"\n"+
			"package main\n"+
			"\n"+
			"const defaultCC = `%s`\n"+
			"const defaultCXX = `%s`\n"+
			"const defaultPkgConfig = `%s`\n",
		defaultcctarget, defaultcxxtarget, defaultpkgconfigtarget)

	writefile(out, file, writeSkipSame)

	// Convert file name to replace: turn go into cgo.
	i := len(file) - len("go/zdefaultcc.go")
	file = file[:i] + "c" + file[i:]
	writefile(out, file, writeSkipSame)
}

// mkzcgo writes zosarch.go for cmd/go.
func mkzosarch(dir, file string) {
	// sort for deterministic zosarch.go file
	var list []string
	for plat := range cgoEnabled {
		list = append(list, plat)
	}
	sort.Strings(list)

	var buf bytes.Buffer
	buf.WriteString("// auto generated by go tool dist\n\n")
	buf.WriteString("package main\n\n")
	fmt.Fprintf(&buf, "var osArchSupportsCgo = map[string]bool{\n")
	for _, plat := range list {
		fmt.Fprintf(&buf, "\t%q: %v,\n", plat, cgoEnabled[plat])
	}
	fmt.Fprintf(&buf, "}\n")
	writefile(buf.String(), file, writeSkipSame)
}

// mkzcgo writes zcgo.go for the go/build package:
//
//	package build
//  var cgoEnabled = map[string]bool{}
//
// It is invoked to write go/build/zcgo.go.
func mkzcgo(dir, file string) {
	// sort for deterministic zcgo.go file
	var list []string
	for plat, hasCgo := range cgoEnabled {
		if hasCgo {
			list = append(list, plat)
		}
	}
	sort.Strings(list)

	var buf bytes.Buffer

	fmt.Fprintf(&buf,
		"// auto generated by go tool dist\n"+
			"\n"+
			"package build\n"+
			"\n"+
			"const defaultCGO_ENABLED = %q\n\n"+
			"var cgoEnabled = map[string]bool{\n", os.Getenv("CGO_ENABLED"))
	for _, plat := range list {
		fmt.Fprintf(&buf, "\t%q: true,\n", plat)
	}
	fmt.Fprintf(&buf, "}")

	writefile(buf.String(), file, writeSkipSame)
}
