#ifndef _EFL_CANVAS_TEXT_CURSOR_EO_H_
#define _EFL_CANVAS_TEXT_CURSOR_EO_H_

#ifndef _EFL_CANVAS_TEXT_CURSOR_EO_CLASS_TYPE
#define _EFL_CANVAS_TEXT_CURSOR_EO_CLASS_TYPE

typedef Eo Efl_Canvas_Text_Cursor;

#endif

#ifndef _EFL_CANVAS_TEXT_CURSOR_EO_TYPES
#define _EFL_CANVAS_TEXT_CURSOR_EO_TYPES

typedef enum
{
  EFL_CANVAS_TEXT_CURSOR_TYPE_BEFORE = 0,
  EFL_CANVAS_TEXT_CURSOR_TYPE_UNDER
} Efl_Canvas_Text_Cursor_Type;


#endif
#define EFL_CANVAS_TEXT_CURSOR_CLASS efl_canvas_text_cursor_class_get()

EWAPI const Eo_Class *efl_canvas_text_cursor_class_get(void);

/**
 * @brief The text object the cursor is associated with.
 *
 * @param[in] tb The text object
 *
 * @ingroup Efl_Canvas_Text_Cursor
 */
EOAPI void efl_canvas_text_cursor_text_object_set(Eo *obj, const Efl_Canvas_Object *tb);

/**
 * @brief The text object the cursor is associated with.
 *
 * @return The text object
 *
 * @ingroup Efl_Canvas_Text_Cursor
 */
EOAPI const Efl_Canvas_Object *efl_canvas_text_cursor_text_object_get(const Eo *obj);

/**
 * @brief Sets position of cursor to given pos.
 *
 * @param[in] position Position in the text to move the cursor.
 *
 * @ingroup Efl_Canvas_Text_Cursor
 */
EOAPI void efl_canvas_text_cursor_position_set(Eo *obj, int position);

/**
 * @brief Sets position of cursor to given pos.
 *
 * @return Position in the text to move the cursor.
 *
 * @ingroup Efl_Canvas_Text_Cursor
 */
EOAPI int efl_canvas_text_cursor_position_get(const Eo *obj);

/**
 * @brief Returns the geometry of two cursors ("split cursor"), if logical
 * cursor is between LTR/RTL text, also considering paragraph direction. Upper
 * cursor is shown for the text of the same direction as paragraph, lower
 * cursor - for opposite.
 *
 * Split cursor geometry is valid only  in '|' cursor mode. In this case
 * @c true is returned and @c cx2, @c cy2, @c cw2, @c ch2 are set.
 *
 * @param[in] ctype the type of the cursor.
 * @param[out] cx the x of the cursor (or upper cursor)
 * @param[out] cy the y of the cursor (or upper cursor)
 * @param[out] cw the width of the cursor (or upper cursor)
 * @param[out] ch the height of the cursor (or upper cursor)
 * @param[out] cx2 the x of the lower cursor
 * @param[out] cy2 the y of the lower cursor
 * @param[out] cw2 the width of the lower cursor
 * @param[out] ch2 the height of the lower cursor
 *
 * @return @c true if split cursor, @c false otherwise.
 *
 * @ingroup Efl_Canvas_Text_Cursor
 */
EOAPI Eina_Bool efl_canvas_text_cursor_geometry_get(const Eo *obj, Efl_Canvas_Text_Cursor_Type ctype, Evas_Coord *cx, Evas_Coord *cy, Evas_Coord *cw, Evas_Coord *ch, Evas_Coord *cx2, Evas_Coord *cy2, Evas_Coord *cw2, Evas_Coord *ch2);

/**
 * @brief The content of the cursor (the character under the cursor)
 *
 * @return The unicode codepoint of the character
 *
 * @ingroup Efl_Canvas_Text_Cursor
 */
EOAPI Eina_Unicode efl_canvas_text_cursor_content_get(const Eo *obj);

/** The object-item annotation at the cursor's position.
 *
 * @ingroup Efl_Canvas_Text_Cursor
 */
EOAPI Efl_Canvas_Text_Annotation *efl_canvas_text_cursor_object_item_annotation_get(const Eo *obj);

/**
 * @brief Copies source cursor.
 *
 * @param[in] src Source cursor.
 *
 * @ingroup Efl_Canvas_Text_Cursor
 */
EOAPI void efl_canvas_text_cursor_copy(Eo *obj, const Efl_Canvas_Text_Cursor *src);

/**
 * @brief Adds text to the current cursor position and set the cursor to
 * *after* the start of the text just added.
 *
 * @param[in] text Text to append (UTF-8 format).
 *
 * @return Length of the appended text.
 *
 * @ingroup Efl_Canvas_Text_Cursor
 */
EOAPI int efl_canvas_text_cursor_text_insert(Eo *obj, const char *text);

/** Advances the cursor one char backwards.
 *
 * @ingroup Efl_Canvas_Text_Cursor
 */
EOAPI void efl_canvas_text_cursor_char_prev(Eo *obj);

/** Advances the cursor one char forward.
 *
 * @ingroup Efl_Canvas_Text_Cursor
 */
EOAPI void efl_canvas_text_cursor_char_next(Eo *obj);

/** Deletes a single character from position pointed by given cursor.
 *
 * @ingroup Efl_Canvas_Text_Cursor
 */
EOAPI void efl_canvas_text_cursor_char_delete(Eo *obj);

/** Sets the cursor to the start of the first text node
 *
 * @ingroup Efl_Canvas_Text_Cursor
 */
EOAPI void efl_canvas_text_cursor_paragraph_first(Eo *obj);

/** Sets the cursor to the end of the last text node
 *
 * @ingroup Efl_Canvas_Text_Cursor
 */
EOAPI void efl_canvas_text_cursor_paragraph_last(Eo *obj);

/** Advances to the start of the next text node
 *
 * @ingroup Efl_Canvas_Text_Cursor
 */
EOAPI void efl_canvas_text_cursor_paragraph_next(Eo *obj);

/** Advances to the end of the previous text node
 *
 * @ingroup Efl_Canvas_Text_Cursor
 */
EOAPI void efl_canvas_text_cursor_paragraph_prev(Eo *obj);

/** Go to the first char in the node the cursor is pointing on.
 *
 * @ingroup Efl_Canvas_Text_Cursor
 */
EOAPI void efl_canvas_text_cursor_paragraph_char_first(Eo *obj);

/** Go to the last char in a text node.
 *
 * @ingroup Efl_Canvas_Text_Cursor
 */
EOAPI void efl_canvas_text_cursor_paragraph_char_last(Eo *obj);

/** Moves the cursor to the start of the word under the cursor
 *
 * @ingroup Efl_Canvas_Text_Cursor
 */
EOAPI void efl_canvas_text_cursor_word_start(Eo *obj);

/** Moves the cursor to the end of the word under the cursor
 *
 * @ingroup Efl_Canvas_Text_Cursor
 */
EOAPI void efl_canvas_text_cursor_word_end(Eo *obj);

/** Go to the start of the current line
 *
 * @ingroup Efl_Canvas_Text_Cursor
 */
EOAPI void efl_canvas_text_cursor_line_char_first(Eo *obj);

/** Go to the end of the current line
 *
 * @ingroup Efl_Canvas_Text_Cursor
 */
EOAPI void efl_canvas_text_cursor_line_char_last(Eo *obj);

/**
 * @brief Move the cursor @c by lines up if negative, down if positive.
 *
 * If jumping by @c by means jumping outside of the textblock, move the cursor
 * before the first char of the textblock if jumping up, and after the last if
 * jumping down.
 *
 * @param[in] by How many lines to jump by. Negative values are also supported.
 *
 * @ingroup Efl_Canvas_Text_Cursor
 */
EOAPI void efl_canvas_text_cursor_line_jump_by(Eo *obj, int by);

/**
 * @brief Compares two cursors.
 *
 * Note that 0 will be returned if they are cursors of different textblock
 * objects.
 *
 * See @ref efl_canvas_text_cursor_equal for a faster equality check.
 *
 * @param[in] cur The second cursor.
 *
 * @return -1 if cur1 < cur2, 0 if cur1 == cur2, 1 otherwise.
 *
 * @ingroup Efl_Canvas_Text_Cursor
 */
EOAPI int efl_canvas_text_cursor_compare(const Eo *obj, const Efl_Canvas_Text_Cursor *cur);

/**
 * @brief Checks if two cursors are equal
 *
 * This is faster than @ref efl_canvas_text_cursor_compare so it should be used
 * if all we care about is equality.
 *
 * @param[in] cur The second cursor.
 *
 * @return @c true if equal
 *
 * @since 1.18
 *
 * @ingroup Efl_Canvas_Text_Cursor
 */
EOAPI Eina_Bool efl_canvas_text_cursor_equal(const Eo *obj, const Efl_Canvas_Text_Cursor *cur);

/**
 * @brief Sets the position of the cursor according to the X and Y coordinates.
 *
 * @param[in] x x coord to set by.
 * @param[in] y y coord to set by.
 *
 * @ingroup Efl_Canvas_Text_Cursor
 */
EOAPI void efl_canvas_text_cursor_coord_set(Eo *obj, Evas_Coord x, Evas_Coord y);

/**
 * @brief Inserts a object item at specified position.
 *
 * This adds a placeholder to be queried by higher-level code, which in turn
 * place graphics on top of it. It essentially places an OBJECT REPLACEMENT
 * CHARACTER and set a special annotation to it.
 *
 * @param[in] format Format of the inserted item. See Format styles.
 *
 * @return The annotation handle of the inserted item.
 *
 * @ingroup Efl_Canvas_Text_Cursor
 */
EOAPI Efl_Canvas_Text_Annotation *efl_canvas_text_cursor_object_item_insert(Eo *obj, const char *format);

EOAPI extern const Eo_Event_Description _EFL_CANVAS_TEXT_CURSOR_EVENT_CHANGED;

/** The cursor position has changed
 *
 * @ingroup Efl_Canvas_Text_Cursor
 */
#define EFL_CANVAS_TEXT_CURSOR_EVENT_CHANGED (&(_EFL_CANVAS_TEXT_CURSOR_EVENT_CHANGED))

#endif
