'use strict';

Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.modifyUrl = modifyUrl;

var _url = require('url');

/**
 *  Takes a URL and a function that takes the meaningful parts
 *  of the URL as a key-value object, modifies some or all of
 *  the parts, and returns the modified parts formatted again
 *  as a url.
 *
 *  Url Parts sent:
 *    - protocol
 *    - slashes (does the url have the //)
 *    - auth
 *    - hostname (just the name of the host, no port or auth information)
 *    - port
 *    - pathmame (the path after the hostname, no query or hash, starts
 *        with a slash if there was a path)
 *    - query (always an object, even when no query on original url)
 *    - hash
 *
 *  Why?
 *    - The default url library in node produces several conflicting
 *      properties on the "parsed" output. Modifying any of these might
 *      lead to the modifications being ignored (depending on which
 *      property was modified)
 *    - It's not always clear wither to use path/pathname, host/hostname,
 *      so this trys to add helpful constraints
 *
 *  @param  {String} url - the url to parse
 *  @param  {Function<Object|undefined>} block - a function that will modify the parsed url, or return a new one
 *  @return {String} the modified and reformatted url
 */
function modifyUrl(url, block) {
  if (typeof block !== 'function') {
    throw new TypeError('You must pass a block to define the modifications desired');
  }

  const parsed = (0, _url.parse)(url, true);

  // copy over the most specific version of each
  // property. By default, the parsed url includes
  // several conflicting properties (like path and
  // pathname + search, or search and query) and keeping
  // track of which property is actually used when they
  // are formatted is harder than necessary
  const meaningfulParts = {
    protocol: parsed.protocol,
    slashes: parsed.slashes,
    auth: parsed.auth,
    hostname: parsed.hostname,
    port: parsed.port,
    pathname: parsed.pathname,
    query: parsed.query || {},
    hash: parsed.hash
  };

  // the block modifies the meaningfulParts object, or returns a new one
  const modifications = block(meaningfulParts) || meaningfulParts;

  // format the modified/replaced meaningfulParts back into a url
  return (0, _url.format)(modifications);
}
