#ifndef _EFL_CONFIG_GLOBAL_EO_H_
#define _EFL_CONFIG_GLOBAL_EO_H_

#ifndef _EFL_CONFIG_GLOBAL_EO_CLASS_TYPE
#define _EFL_CONFIG_GLOBAL_EO_CLASS_TYPE

typedef Eo Efl_Config_Global;

#endif

#ifndef _EFL_CONFIG_GLOBAL_EO_TYPES
#define _EFL_CONFIG_GLOBAL_EO_TYPES


#endif
/** This class is a singleton representing the global configuration for the
 * running application.
 *
 * @ingroup Efl_Config_Global
 */
#define EFL_CONFIG_GLOBAL_CLASS efl_config_global_class_get()

EWAPI const Eo_Class *efl_config_global_class_get(void);

/**
 * @brief The profile for the running application.
 *
 * Profiles are pre-set options that affect the whole look-and-feel of
 * Elementary-based applications. There are, for example, profiles aimed at
 * desktop computer applications and others aimed at mobile, touchscreen-based
 * ones. You most probably don't want to use the functions in this group unless
 * you're writing an elementary configuration manager.
 *
 * This gets or sets the global profile that is applied to all Elementary
 * applications.
 *
 * @param[in] profile
 *
 * @ingroup Efl_Config_Global
 */
EOAPI void efl_config_profile_set(Eo *obj, const char *profile);

/**
 * @brief The profile for the running application.
 *
 * Profiles are pre-set options that affect the whole look-and-feel of
 * Elementary-based applications. There are, for example, profiles aimed at
 * desktop computer applications and others aimed at mobile, touchscreen-based
 * ones. You most probably don't want to use the functions in this group unless
 * you're writing an elementary configuration manager.
 *
 * This gets or sets the global profile that is applied to all Elementary
 * applications.
 *
 * @ingroup Efl_Config_Global
 */
EOAPI const char *efl_config_profile_get(const Eo *obj);

#ifdef EFL_CONFIG_GLOBAL_PROTECTED
/**
 * @brief Saves Elementary configuration to disk.
 *
 * This function will take effect (does I/O) immediately. Use it when you want
 * to save all configuration changes at once. The current configuration set
 * will get saved onto the current profile configuration file.
 *
 * If @c profile is @c null, this will flush all settings to all applications
 * running on the same profile.
 *
 * If @c profile is not @c null, this will take the current in-memory config
 * and write it out to the named @c profile. This will not change profile for
 * the application or make other processes switch profile.
 *
 * @param[in] profile The profile name.
 *
 * @ingroup Efl_Config_Global
 */
EOAPI Eina_Bool efl_config_save(Eo *obj, const char *profile);
#endif

/**
 * @brief Returns the list of available profiles.
 *
 * @param[in] hidden If @c true, gets the full list of profiles, including
 * those stored in hidden files.
 *
 * @ingroup Efl_Config_Global
 */
EOAPI Eina_Iterator *efl_config_profile_iterate(Eo *obj, Eina_Bool hidden);

/**
 * @brief Returns whether a profile exists or not.
 *
 * @param[in] profile
 *
 * @ingroup Efl_Config_Global
 */
EOAPI Eina_Bool efl_config_profile_exists(Eo *obj, const char *profile);

/**
 * @brief Returns the directory where a profile is stored.
 *
 * @param[in] profile
 * @param[in] is_user @c true to lookup for a user profile or @c false for a
 * system one.
 *
 * @ingroup Efl_Config_Global
 */
EOAPI Eina_Stringshare *efl_config_profile_dir_get(Eo *obj, const char *profile, Eina_Bool is_user);

#ifdef EFL_CONFIG_GLOBAL_PROTECTED
/**
 * @brief Add a new profile of the given name to be derived from the current
 * profile.
 *
 * This creates a new profile of name @c profile that will be derived from the
 * currently used profile using the modification commands encoded in the
 * @c options string.
 *
 * At this point it is not expected that anyone would generally use this API
 * except if you are a destktop environment and so the user base of this API
 * will be enlightenment itself.
 *
 * @param[in] profile The new profile's name.
 * @param[in] options Derive options detailing how to modify.
 *
 * @since 1.17
 *
 * @ingroup Efl_Config_Global
 */
EOAPI void efl_config_profile_derived_add(Eo *obj, const char *profile, const char *options);
#endif

#ifdef EFL_CONFIG_GLOBAL_PROTECTED
/**
 * @brief Deletes a profile that is derived from the current one.
 *
 * This deletes a derived profile added by @ref efl_config_profile_derived_add.
 * This will delete the profile of the given name @c profile that is derived
 * from the current profile.
 *
 * At this point it is not expected that anyone would generally use this API
 * except if you are a destktop environment and so the user base of this API
 * will be enlightenment itself.
 *
 * @param[in] profile The name of the profile that is to be deleted.
 *
 * @since 1.17
 *
 * @ingroup Efl_Config_Global
 */
EOAPI void efl_config_profile_derived_del(Eo *obj, const char *profile);
#endif


#endif
