/*
 * Decompiled with CFR 0.152.
 */
package org.elasticsearch.snapshots;

import com.carrotsearch.hppc.cursors.ObjectCursor;
import com.carrotsearch.hppc.cursors.ObjectObjectCursor;
import java.io.IOException;
import java.util.ArrayList;
import java.util.Arrays;
import java.util.Collections;
import java.util.HashMap;
import java.util.HashSet;
import java.util.Iterator;
import java.util.List;
import java.util.Locale;
import java.util.Map;
import java.util.Objects;
import java.util.Optional;
import java.util.Set;
import java.util.concurrent.CopyOnWriteArrayList;
import java.util.stream.Collectors;
import org.apache.logging.log4j.message.ParameterizedMessage;
import org.apache.lucene.util.CollectionUtil;
import org.elasticsearch.ExceptionsHelper;
import org.elasticsearch.action.ActionListener;
import org.elasticsearch.action.support.IndicesOptions;
import org.elasticsearch.cluster.ClusterChangedEvent;
import org.elasticsearch.cluster.ClusterState;
import org.elasticsearch.cluster.ClusterStateApplier;
import org.elasticsearch.cluster.ClusterStateUpdateTask;
import org.elasticsearch.cluster.RestoreInProgress;
import org.elasticsearch.cluster.SnapshotDeletionsInProgress;
import org.elasticsearch.cluster.SnapshotsInProgress;
import org.elasticsearch.cluster.metadata.IndexMetaData;
import org.elasticsearch.cluster.metadata.IndexNameExpressionResolver;
import org.elasticsearch.cluster.metadata.MetaData;
import org.elasticsearch.cluster.metadata.RepositoriesMetaData;
import org.elasticsearch.cluster.node.DiscoveryNode;
import org.elasticsearch.cluster.node.DiscoveryNodes;
import org.elasticsearch.cluster.routing.IndexRoutingTable;
import org.elasticsearch.cluster.routing.IndexShardRoutingTable;
import org.elasticsearch.cluster.routing.RoutingTable;
import org.elasticsearch.cluster.routing.ShardRouting;
import org.elasticsearch.cluster.service.ClusterService;
import org.elasticsearch.common.Nullable;
import org.elasticsearch.common.Priority;
import org.elasticsearch.common.Strings;
import org.elasticsearch.common.UUIDs;
import org.elasticsearch.common.collect.ImmutableOpenMap;
import org.elasticsearch.common.collect.Tuple;
import org.elasticsearch.common.component.AbstractLifecycleComponent;
import org.elasticsearch.common.inject.Inject;
import org.elasticsearch.common.settings.Settings;
import org.elasticsearch.common.unit.TimeValue;
import org.elasticsearch.index.Index;
import org.elasticsearch.index.shard.ShardId;
import org.elasticsearch.index.snapshots.IndexShardSnapshotStatus;
import org.elasticsearch.repositories.IndexId;
import org.elasticsearch.repositories.RepositoriesService;
import org.elasticsearch.repositories.Repository;
import org.elasticsearch.repositories.RepositoryData;
import org.elasticsearch.repositories.RepositoryMissingException;
import org.elasticsearch.snapshots.ConcurrentSnapshotExecutionException;
import org.elasticsearch.snapshots.InvalidSnapshotNameException;
import org.elasticsearch.snapshots.Snapshot;
import org.elasticsearch.snapshots.SnapshotException;
import org.elasticsearch.snapshots.SnapshotId;
import org.elasticsearch.snapshots.SnapshotInfo;
import org.elasticsearch.snapshots.SnapshotMissingException;
import org.elasticsearch.snapshots.SnapshotShardFailure;
import org.elasticsearch.snapshots.SnapshotState;
import org.elasticsearch.threadpool.ThreadPool;

public class SnapshotsService
extends AbstractLifecycleComponent
implements ClusterStateApplier {
    private final ClusterService clusterService;
    private final IndexNameExpressionResolver indexNameExpressionResolver;
    private final RepositoriesService repositoriesService;
    private final ThreadPool threadPool;
    private final CopyOnWriteArrayList<SnapshotCompletionListener> snapshotCompletionListeners = new CopyOnWriteArrayList();

    @Inject
    public SnapshotsService(Settings settings, ClusterService clusterService, IndexNameExpressionResolver indexNameExpressionResolver, RepositoriesService repositoriesService, ThreadPool threadPool) {
        super(settings);
        this.clusterService = clusterService;
        this.indexNameExpressionResolver = indexNameExpressionResolver;
        this.repositoriesService = repositoriesService;
        this.threadPool = threadPool;
        if (DiscoveryNode.isMasterNode(settings)) {
            clusterService.addLowPriorityApplier(this);
        }
    }

    public RepositoryData getRepositoryData(String repositoryName) {
        Repository repository = this.repositoriesService.repository(repositoryName);
        assert (repository != null);
        return repository.getRepositoryData();
    }

    public SnapshotInfo snapshot(String repositoryName, SnapshotId snapshotId) {
        List<SnapshotsInProgress.Entry> entries = this.currentSnapshots(repositoryName, Arrays.asList(snapshotId.getName()));
        if (!entries.isEmpty()) {
            return this.inProgressSnapshot(entries.iterator().next());
        }
        return this.repositoriesService.repository(repositoryName).getSnapshotInfo(snapshotId);
    }

    public List<SnapshotInfo> snapshots(String repositoryName, List<SnapshotId> snapshotIds, Set<SnapshotId> incompatibleSnapshotIds, boolean ignoreUnavailable) {
        HashSet<SnapshotInfo> snapshotSet = new HashSet<SnapshotInfo>();
        HashSet<SnapshotId> snapshotIdsToIterate = new HashSet<SnapshotId>(snapshotIds);
        List<SnapshotsInProgress.Entry> entries = this.currentSnapshots(repositoryName, snapshotIdsToIterate.stream().map(SnapshotId::getName).collect(Collectors.toList()));
        for (SnapshotsInProgress.Entry entry : entries) {
            snapshotSet.add(this.inProgressSnapshot(entry));
            snapshotIdsToIterate.remove(entry.snapshot().getSnapshotId());
        }
        Repository repository = this.repositoriesService.repository(repositoryName);
        for (SnapshotId snapshotId : snapshotIdsToIterate) {
            try {
                if (incompatibleSnapshotIds.contains(snapshotId)) {
                    snapshotSet.add(SnapshotInfo.incompatible(snapshotId));
                    continue;
                }
                snapshotSet.add(repository.getSnapshotInfo(snapshotId));
            }
            catch (Exception ex) {
                if (ignoreUnavailable) {
                    this.logger.warn(() -> new ParameterizedMessage("failed to get snapshot [{}]", (Object)snapshotId), (Throwable)ex);
                    continue;
                }
                throw new SnapshotException(repositoryName, snapshotId, "Snapshot could not be read", (Throwable)ex);
            }
        }
        ArrayList arrayList = new ArrayList(snapshotSet);
        CollectionUtil.timSort(arrayList);
        return Collections.unmodifiableList(arrayList);
    }

    public List<SnapshotInfo> currentSnapshots(String repositoryName) {
        ArrayList<SnapshotInfo> snapshotList = new ArrayList<SnapshotInfo>();
        List<SnapshotsInProgress.Entry> entries = this.currentSnapshots(repositoryName, Collections.emptyList());
        for (SnapshotsInProgress.Entry entry : entries) {
            snapshotList.add(this.inProgressSnapshot(entry));
        }
        CollectionUtil.timSort(snapshotList);
        return Collections.unmodifiableList(snapshotList);
    }

    public void createSnapshot(final SnapshotRequest request, final CreateSnapshotListener listener) {
        final String repositoryName = request.repositoryName;
        final String snapshotName = request.snapshotName;
        SnapshotsService.validate(repositoryName, snapshotName);
        final SnapshotId snapshotId = new SnapshotId(snapshotName, UUIDs.randomBase64UUID());
        final RepositoryData repositoryData = this.repositoriesService.repository(repositoryName).getRepositoryData();
        this.clusterService.submitStateUpdateTask(request.cause(), new ClusterStateUpdateTask(){
            private SnapshotsInProgress.Entry newSnapshot = null;

            @Override
            public ClusterState execute(ClusterState currentState) {
                SnapshotsService.this.validate(request, currentState);
                SnapshotDeletionsInProgress deletionsInProgress = (SnapshotDeletionsInProgress)currentState.custom("snapshot_deletions");
                if (deletionsInProgress != null && deletionsInProgress.hasDeletionsInProgress()) {
                    throw new ConcurrentSnapshotExecutionException(repositoryName, snapshotName, "cannot snapshot while a snapshot deletion is in-progress");
                }
                SnapshotsInProgress snapshots = (SnapshotsInProgress)currentState.custom("snapshots");
                if (snapshots != null && !snapshots.entries().isEmpty()) {
                    throw new ConcurrentSnapshotExecutionException(repositoryName, snapshotName, "a snapshot is already running");
                }
                List<String> indices = Arrays.asList(SnapshotsService.this.indexNameExpressionResolver.concreteIndexNames(currentState, request.indicesOptions(), request.indices()));
                SnapshotsService.this.logger.trace("[{}][{}] creating snapshot for indices [{}]", (Object)repositoryName, (Object)snapshotName, indices);
                List<IndexId> snapshotIndices = repositoryData.resolveNewIndices(indices);
                this.newSnapshot = new SnapshotsInProgress.Entry(new Snapshot(repositoryName, snapshotId), request.includeGlobalState(), request.partial(), SnapshotsInProgress.State.INIT, snapshotIndices, System.currentTimeMillis(), repositoryData.getGenId(), null);
                snapshots = new SnapshotsInProgress(this.newSnapshot);
                return ClusterState.builder(currentState).putCustom("snapshots", snapshots).build();
            }

            @Override
            public void onFailure(String source, Exception e) {
                SnapshotsService.this.logger.warn(() -> new ParameterizedMessage("[{}][{}] failed to create snapshot", (Object)repositoryName, (Object)snapshotName), (Throwable)e);
                this.newSnapshot = null;
                listener.onFailure(e);
            }

            @Override
            public void clusterStateProcessed(String source, ClusterState oldState, ClusterState newState) {
                if (this.newSnapshot != null) {
                    SnapshotsService.this.threadPool.executor("snapshot").execute(() -> SnapshotsService.this.beginSnapshot(newState, this.newSnapshot, request.partial(), listener));
                }
            }

            @Override
            public TimeValue timeout() {
                return request.masterNodeTimeout();
            }
        });
    }

    private void validate(SnapshotRequest request, ClusterState state) {
        RepositoriesMetaData repositoriesMetaData = (RepositoriesMetaData)state.getMetaData().custom("repositories");
        String repository = request.repositoryName;
        if (repositoriesMetaData == null || repositoriesMetaData.repository(repository) == null) {
            throw new RepositoryMissingException(repository);
        }
        SnapshotsService.validate(repository, request.snapshotName);
    }

    private static void validate(String repositoryName, String snapshotName) {
        if (!Strings.hasLength(snapshotName)) {
            throw new InvalidSnapshotNameException(repositoryName, snapshotName, "cannot be empty");
        }
        if (snapshotName.contains(" ")) {
            throw new InvalidSnapshotNameException(repositoryName, snapshotName, "must not contain whitespace");
        }
        if (snapshotName.contains(",")) {
            throw new InvalidSnapshotNameException(repositoryName, snapshotName, "must not contain ','");
        }
        if (snapshotName.contains("#")) {
            throw new InvalidSnapshotNameException(repositoryName, snapshotName, "must not contain '#'");
        }
        if (snapshotName.charAt(0) == '_') {
            throw new InvalidSnapshotNameException(repositoryName, snapshotName, "must not start with '_'");
        }
        if (!snapshotName.toLowerCase(Locale.ROOT).equals(snapshotName)) {
            throw new InvalidSnapshotNameException(repositoryName, snapshotName, "must be lowercase");
        }
        if (!Strings.validFileName(snapshotName)) {
            throw new InvalidSnapshotNameException(repositoryName, snapshotName, "must not contain the following characters " + Strings.INVALID_FILENAME_CHARS);
        }
    }

    private void beginSnapshot(ClusterState clusterState, final SnapshotsInProgress.Entry snapshot, final boolean partial, final CreateSnapshotListener userCreateSnapshotListener) {
        boolean snapshotCreated = false;
        try {
            Repository repository = this.repositoriesService.repository(snapshot.snapshot().getRepository());
            MetaData metaData = clusterState.metaData();
            if (!snapshot.includeGlobalState()) {
                MetaData.Builder builder = MetaData.builder();
                for (IndexId index : snapshot.indices()) {
                    builder.put(metaData.index(index.getName()), false);
                }
                metaData = builder.build();
            }
            repository.initializeSnapshot(snapshot.snapshot().getSnapshotId(), snapshot.indices(), metaData);
            snapshotCreated = true;
            if (snapshot.indices().isEmpty()) {
                userCreateSnapshotListener.onResponse();
                this.endSnapshot(snapshot);
                return;
            }
            this.clusterService.submitStateUpdateTask("update_snapshot [" + snapshot.snapshot() + "]", new ClusterStateUpdateTask(){
                boolean accepted = false;
                SnapshotsInProgress.Entry updatedSnapshot;
                String failure = null;

                @Override
                public ClusterState execute(ClusterState currentState) {
                    SnapshotsInProgress snapshots = (SnapshotsInProgress)currentState.custom("snapshots");
                    ArrayList<SnapshotsInProgress.Entry> entries = new ArrayList<SnapshotsInProgress.Entry>();
                    for (SnapshotsInProgress.Entry entry : snapshots.entries()) {
                        if (entry.snapshot().equals(snapshot.snapshot())) {
                            ImmutableOpenMap shards = SnapshotsService.this.shards(currentState, entry.indices());
                            if (!partial) {
                                Tuple indicesWithMissingShards = SnapshotsService.this.indicesWithMissingShards(shards, currentState.metaData());
                                Set missing = (Set)indicesWithMissingShards.v1();
                                Set closed = (Set)indicesWithMissingShards.v2();
                                if (!missing.isEmpty() || !closed.isEmpty()) {
                                    StringBuilder failureMessage = new StringBuilder();
                                    this.updatedSnapshot = new SnapshotsInProgress.Entry(entry, SnapshotsInProgress.State.FAILED, shards);
                                    entries.add(this.updatedSnapshot);
                                    if (!missing.isEmpty()) {
                                        failureMessage.append("Indices don't have primary shards ");
                                        failureMessage.append(missing);
                                    }
                                    if (!closed.isEmpty()) {
                                        if (failureMessage.length() > 0) {
                                            failureMessage.append("; ");
                                        }
                                        failureMessage.append("Indices are closed ");
                                        failureMessage.append(closed);
                                    }
                                    this.failure = failureMessage.toString();
                                    continue;
                                }
                            }
                            this.updatedSnapshot = new SnapshotsInProgress.Entry(entry, SnapshotsInProgress.State.STARTED, shards);
                            entries.add(this.updatedSnapshot);
                            if (SnapshotsInProgress.completed(shards.values())) continue;
                            this.accepted = true;
                            continue;
                        }
                        entries.add(entry);
                    }
                    return ClusterState.builder(currentState).putCustom("snapshots", new SnapshotsInProgress(Collections.unmodifiableList(entries))).build();
                }

                @Override
                public void onFailure(String source, Exception e) {
                    SnapshotsService.this.logger.warn(() -> new ParameterizedMessage("[{}] failed to create snapshot", (Object)snapshot.snapshot().getSnapshotId()), (Throwable)e);
                    SnapshotsService.this.removeSnapshotFromClusterState(snapshot.snapshot(), null, e, new CleanupAfterErrorListener(snapshot, true, userCreateSnapshotListener, e));
                }

                @Override
                public void clusterStateProcessed(String source, ClusterState oldState, ClusterState newState) {
                    userCreateSnapshotListener.onResponse();
                    if (!this.accepted && this.updatedSnapshot != null) {
                        SnapshotsService.this.endSnapshot(this.updatedSnapshot, this.failure);
                    }
                }
            });
        }
        catch (Exception e) {
            this.logger.warn(() -> new ParameterizedMessage("failed to create snapshot [{}]", (Object)snapshot.snapshot().getSnapshotId()), (Throwable)e);
            this.removeSnapshotFromClusterState(snapshot.snapshot(), null, e, new CleanupAfterErrorListener(snapshot, snapshotCreated, userCreateSnapshotListener, e));
        }
    }

    private SnapshotInfo inProgressSnapshot(SnapshotsInProgress.Entry entry) {
        return new SnapshotInfo(entry.snapshot().getSnapshotId(), entry.indices().stream().map(IndexId::getName).collect(Collectors.toList()), entry.startTime());
    }

    public List<SnapshotsInProgress.Entry> currentSnapshots(String repository, List<String> snapshots) {
        SnapshotsInProgress snapshotsInProgress = (SnapshotsInProgress)this.clusterService.state().custom("snapshots");
        if (snapshotsInProgress == null || snapshotsInProgress.entries().isEmpty()) {
            return Collections.emptyList();
        }
        if ("_all".equals(repository)) {
            return snapshotsInProgress.entries();
        }
        if (snapshotsInProgress.entries().size() == 1) {
            SnapshotsInProgress.Entry entry = snapshotsInProgress.entries().get(0);
            if (!entry.snapshot().getRepository().equals(repository)) {
                return Collections.emptyList();
            }
            if (!snapshots.isEmpty()) {
                for (String snapshot : snapshots) {
                    if (!entry.snapshot().getSnapshotId().getName().equals(snapshot)) continue;
                    return snapshotsInProgress.entries();
                }
                return Collections.emptyList();
            }
            return snapshotsInProgress.entries();
        }
        ArrayList<SnapshotsInProgress.Entry> builder = new ArrayList<SnapshotsInProgress.Entry>();
        block1: for (SnapshotsInProgress.Entry entry : snapshotsInProgress.entries()) {
            if (!entry.snapshot().getRepository().equals(repository)) continue;
            if (!snapshots.isEmpty()) {
                for (String snapshot : snapshots) {
                    if (!entry.snapshot().getSnapshotId().getName().equals(snapshot)) continue;
                    builder.add(entry);
                    continue block1;
                }
                continue;
            }
            builder.add(entry);
        }
        return Collections.unmodifiableList(builder);
    }

    public Map<ShardId, IndexShardSnapshotStatus> snapshotShards(String repositoryName, SnapshotInfo snapshotInfo) throws IOException {
        HashMap<ShardId, IndexShardSnapshotStatus> shardStatus = new HashMap<ShardId, IndexShardSnapshotStatus>();
        Repository repository = this.repositoriesService.repository(repositoryName);
        RepositoryData repositoryData = repository.getRepositoryData();
        MetaData metaData = repository.getSnapshotMetaData(snapshotInfo, repositoryData.resolveIndices(snapshotInfo.indices()));
        for (String index : snapshotInfo.indices()) {
            IndexId indexId = repositoryData.resolveIndexId(index);
            IndexMetaData indexMetaData = metaData.indices().get(index);
            if (indexMetaData == null) continue;
            int numberOfShards = indexMetaData.getNumberOfShards();
            for (int i = 0; i < numberOfShards; ++i) {
                IndexShardSnapshotStatus shardSnapshotStatus;
                ShardId shardId = new ShardId(indexMetaData.getIndex(), i);
                SnapshotShardFailure shardFailure = this.findShardFailure(snapshotInfo.shardFailures(), shardId);
                if (shardFailure != null) {
                    shardSnapshotStatus = new IndexShardSnapshotStatus();
                    shardSnapshotStatus.updateStage(IndexShardSnapshotStatus.Stage.FAILURE);
                    shardSnapshotStatus.failure(shardFailure.reason());
                    shardStatus.put(shardId, shardSnapshotStatus);
                    continue;
                }
                if (snapshotInfo.state() == SnapshotState.FAILED) {
                    shardSnapshotStatus = new IndexShardSnapshotStatus();
                    shardSnapshotStatus.updateStage(IndexShardSnapshotStatus.Stage.FAILURE);
                    shardSnapshotStatus.failure("skipped");
                } else {
                    shardSnapshotStatus = repository.getShardSnapshotStatus(snapshotInfo.snapshotId(), snapshotInfo.version(), indexId, shardId);
                }
                shardStatus.put(shardId, shardSnapshotStatus);
            }
        }
        return Collections.unmodifiableMap(shardStatus);
    }

    private SnapshotShardFailure findShardFailure(List<SnapshotShardFailure> shardFailures, ShardId shardId) {
        for (SnapshotShardFailure shardFailure : shardFailures) {
            if (!shardId.getIndexName().equals(shardFailure.index()) || shardId.getId() != shardFailure.shardId()) continue;
            return shardFailure;
        }
        return null;
    }

    @Override
    public void applyClusterState(ClusterChangedEvent event) {
        try {
            if (event.localNodeMaster()) {
                if (event.nodesRemoved()) {
                    this.processSnapshotsOnRemovedNodes(event);
                }
                if (event.routingTableChanged()) {
                    this.processStartedShards(event);
                }
                this.removeFinishedSnapshotFromClusterState(event);
                this.finalizeSnapshotDeletionFromPreviousMaster(event);
            }
        }
        catch (Exception e) {
            this.logger.warn("Failed to update snapshot state ", (Throwable)e);
        }
    }

    private void finalizeSnapshotDeletionFromPreviousMaster(ClusterChangedEvent event) {
        SnapshotDeletionsInProgress deletionsInProgress;
        if (event.localNodeMaster() && !event.previousState().nodes().isLocalNodeElectedMaster() && (deletionsInProgress = (SnapshotDeletionsInProgress)event.state().custom("snapshot_deletions")) != null && deletionsInProgress.hasDeletionsInProgress()) {
            assert (deletionsInProgress.getEntries().size() == 1) : "only one in-progress deletion allowed per cluster";
            SnapshotDeletionsInProgress.Entry entry = deletionsInProgress.getEntries().get(0);
            this.deleteSnapshotFromRepository(entry.getSnapshot(), null, entry.getRepositoryStateId());
        }
    }

    private void removeFinishedSnapshotFromClusterState(ClusterChangedEvent event) {
        SnapshotsInProgress snapshotsInProgress;
        if (event.localNodeMaster() && !event.previousState().nodes().isLocalNodeElectedMaster() && (snapshotsInProgress = (SnapshotsInProgress)event.state().custom("snapshots")) != null && !snapshotsInProgress.entries().isEmpty()) {
            for (SnapshotsInProgress.Entry entry : snapshotsInProgress.entries()) {
                if (!entry.state().completed()) continue;
                this.endSnapshot(entry);
            }
        }
    }

    private void processSnapshotsOnRemovedNodes(ClusterChangedEvent event) {
        if (this.removedNodesCleanupNeeded(event)) {
            final boolean newMaster = !event.previousState().nodes().isLocalNodeElectedMaster();
            this.clusterService.submitStateUpdateTask("update snapshot state after node removal", new ClusterStateUpdateTask(){

                @Override
                public ClusterState execute(ClusterState currentState) throws Exception {
                    DiscoveryNodes nodes = currentState.nodes();
                    SnapshotsInProgress snapshots = (SnapshotsInProgress)currentState.custom("snapshots");
                    if (snapshots == null) {
                        return currentState;
                    }
                    boolean changed = false;
                    ArrayList<SnapshotsInProgress.Entry> entries = new ArrayList<SnapshotsInProgress.Entry>();
                    Iterator<SnapshotsInProgress.Entry> iterator = snapshots.entries().iterator();
                    while (iterator.hasNext()) {
                        SnapshotsInProgress.Entry snapshot;
                        SnapshotsInProgress.Entry updatedSnapshot = snapshot = iterator.next();
                        boolean snapshotChanged = false;
                        if (snapshot.state() == SnapshotsInProgress.State.STARTED || snapshot.state() == SnapshotsInProgress.State.ABORTED) {
                            ImmutableOpenMap.Builder<Object, Object> shards = ImmutableOpenMap.builder();
                            for (ObjectObjectCursor<ShardId, SnapshotsInProgress.ShardSnapshotStatus> objectObjectCursor : snapshot.shards()) {
                                SnapshotsInProgress.ShardSnapshotStatus shardStatus = (SnapshotsInProgress.ShardSnapshotStatus)objectObjectCursor.value;
                                if (shardStatus.state().completed() || shardStatus.nodeId() == null) continue;
                                if (nodes.nodeExists(shardStatus.nodeId())) {
                                    shards.put(objectObjectCursor.key, objectObjectCursor.value);
                                    continue;
                                }
                                snapshotChanged = true;
                                SnapshotsService.this.logger.warn("failing snapshot of shard [{}] on closed node [{}]", objectObjectCursor.key, (Object)shardStatus.nodeId());
                                shards.put(objectObjectCursor.key, new SnapshotsInProgress.ShardSnapshotStatus(shardStatus.nodeId(), SnapshotsInProgress.State.FAILED, "node shutdown"));
                            }
                            if (snapshotChanged) {
                                changed = true;
                                ImmutableOpenMap<ShardId, SnapshotsInProgress.ShardSnapshotStatus> shardsMap = shards.build();
                                if (!snapshot.state().completed() && SnapshotsInProgress.completed(shardsMap.values())) {
                                    updatedSnapshot = new SnapshotsInProgress.Entry(snapshot, SnapshotsInProgress.State.SUCCESS, shardsMap);
                                    SnapshotsService.this.endSnapshot(updatedSnapshot);
                                } else {
                                    updatedSnapshot = new SnapshotsInProgress.Entry(snapshot, snapshot.state(), shardsMap);
                                }
                            }
                            entries.add(updatedSnapshot);
                            continue;
                        }
                        if (snapshot.state() != SnapshotsInProgress.State.INIT || !newMaster) continue;
                        SnapshotsService.this.deleteSnapshot(snapshot.snapshot(), new DeleteSnapshotListener(){

                            @Override
                            public void onResponse() {
                                SnapshotsService.this.logger.debug("cleaned up abandoned snapshot {} in INIT state", (Object)snapshot.snapshot());
                            }

                            @Override
                            public void onFailure(Exception e) {
                                SnapshotsService.this.logger.warn("failed to clean up abandoned snapshot {} in INIT state", (Object)snapshot.snapshot());
                            }
                        }, updatedSnapshot.getRepositoryStateId(), false);
                    }
                    if (changed) {
                        snapshots = new SnapshotsInProgress(entries.toArray(new SnapshotsInProgress.Entry[entries.size()]));
                        return ClusterState.builder(currentState).putCustom("snapshots", snapshots).build();
                    }
                    return currentState;
                }

                @Override
                public void onFailure(String source, Exception e) {
                    SnapshotsService.this.logger.warn("failed to update snapshot state after node removal");
                }
            });
        }
    }

    private void processStartedShards(ClusterChangedEvent event) {
        if (this.waitingShardsStartedOrUnassigned(event)) {
            this.clusterService.submitStateUpdateTask("update snapshot state after shards started", new ClusterStateUpdateTask(){

                @Override
                public ClusterState execute(ClusterState currentState) throws Exception {
                    RoutingTable routingTable = currentState.routingTable();
                    SnapshotsInProgress snapshots = (SnapshotsInProgress)currentState.custom("snapshots");
                    if (snapshots != null) {
                        boolean changed = false;
                        ArrayList<SnapshotsInProgress.Entry> entries = new ArrayList<SnapshotsInProgress.Entry>();
                        Iterator<SnapshotsInProgress.Entry> iterator = snapshots.entries().iterator();
                        while (iterator.hasNext()) {
                            SnapshotsInProgress.Entry snapshot;
                            SnapshotsInProgress.Entry updatedSnapshot = snapshot = iterator.next();
                            if (snapshot.state() != SnapshotsInProgress.State.STARTED) continue;
                            ImmutableOpenMap shards = SnapshotsService.this.processWaitingShards(snapshot.shards(), routingTable);
                            if (shards != null) {
                                changed = true;
                                if (!snapshot.state().completed() && SnapshotsInProgress.completed(shards.values())) {
                                    updatedSnapshot = new SnapshotsInProgress.Entry(snapshot, SnapshotsInProgress.State.SUCCESS, shards);
                                    SnapshotsService.this.endSnapshot(updatedSnapshot);
                                } else {
                                    updatedSnapshot = new SnapshotsInProgress.Entry(snapshot, shards);
                                }
                            }
                            entries.add(updatedSnapshot);
                        }
                        if (changed) {
                            snapshots = new SnapshotsInProgress(entries.toArray(new SnapshotsInProgress.Entry[entries.size()]));
                            return ClusterState.builder(currentState).putCustom("snapshots", snapshots).build();
                        }
                    }
                    return currentState;
                }

                @Override
                public void onFailure(String source, Exception e) {
                    SnapshotsService.this.logger.warn(() -> new ParameterizedMessage("failed to update snapshot state after shards started from [{}] ", (Object)source), (Throwable)e);
                }
            });
        }
    }

    private ImmutableOpenMap<ShardId, SnapshotsInProgress.ShardSnapshotStatus> processWaitingShards(ImmutableOpenMap<ShardId, SnapshotsInProgress.ShardSnapshotStatus> snapshotShards, RoutingTable routingTable) {
        boolean snapshotChanged = false;
        ImmutableOpenMap.Builder<ShardId, SnapshotsInProgress.ShardSnapshotStatus> shards = ImmutableOpenMap.builder();
        for (ObjectObjectCursor<ShardId, SnapshotsInProgress.ShardSnapshotStatus> objectObjectCursor : snapshotShards) {
            SnapshotsInProgress.ShardSnapshotStatus shardStatus = (SnapshotsInProgress.ShardSnapshotStatus)objectObjectCursor.value;
            ShardId shardId = (ShardId)objectObjectCursor.key;
            if (shardStatus.state() == SnapshotsInProgress.State.WAITING) {
                IndexShardRoutingTable shardRouting;
                IndexRoutingTable indexShardRoutingTable = routingTable.index(shardId.getIndex());
                if (indexShardRoutingTable != null && (shardRouting = indexShardRoutingTable.shard(shardId.id())) != null && shardRouting.primaryShard() != null) {
                    if (shardRouting.primaryShard().started()) {
                        snapshotChanged = true;
                        this.logger.trace("starting shard that we were waiting for [{}] on node [{}]", (Object)shardId, (Object)shardStatus.nodeId());
                        shards.put(shardId, new SnapshotsInProgress.ShardSnapshotStatus(shardRouting.primaryShard().currentNodeId()));
                        continue;
                    }
                    if (shardRouting.primaryShard().initializing() || shardRouting.primaryShard().relocating()) {
                        shards.put(shardId, shardStatus);
                        continue;
                    }
                }
                snapshotChanged = true;
                this.logger.warn("failing snapshot of shard [{}] on unassigned shard [{}]", (Object)shardId, (Object)shardStatus.nodeId());
                shards.put(shardId, new SnapshotsInProgress.ShardSnapshotStatus(shardStatus.nodeId(), SnapshotsInProgress.State.FAILED, "shard is unassigned"));
                continue;
            }
            shards.put(shardId, shardStatus);
        }
        if (snapshotChanged) {
            return shards.build();
        }
        return null;
    }

    private boolean waitingShardsStartedOrUnassigned(ClusterChangedEvent event) {
        SnapshotsInProgress curr = (SnapshotsInProgress)event.state().custom("snapshots");
        if (curr != null) {
            for (SnapshotsInProgress.Entry entry : curr.entries()) {
                if (entry.state() != SnapshotsInProgress.State.STARTED || entry.waitingIndices().isEmpty()) continue;
                for (ObjectCursor index : entry.waitingIndices().keys()) {
                    if (!event.indexRoutingTableChanged((String)index.value)) continue;
                    IndexRoutingTable indexShardRoutingTable = event.state().getRoutingTable().index((String)index.value);
                    for (ShardId shardId : entry.waitingIndices().get((String)index.value)) {
                        ShardRouting shardRouting = indexShardRoutingTable.shard(shardId.id()).primaryShard();
                        if (shardRouting == null || !shardRouting.started() && !shardRouting.unassigned()) continue;
                        return true;
                    }
                }
            }
        }
        return false;
    }

    private boolean removedNodesCleanupNeeded(ClusterChangedEvent event) {
        SnapshotsInProgress snapshotsInProgress = (SnapshotsInProgress)event.state().custom("snapshots");
        if (snapshotsInProgress == null) {
            return false;
        }
        boolean newMaster = !event.previousState().nodes().isLocalNodeElectedMaster();
        for (SnapshotsInProgress.Entry snapshot : snapshotsInProgress.entries()) {
            if (newMaster && (snapshot.state() == SnapshotsInProgress.State.SUCCESS || snapshot.state() == SnapshotsInProgress.State.INIT)) {
                return true;
            }
            for (DiscoveryNode node : event.nodesDelta().removedNodes()) {
                for (ObjectCursor shardStatus : snapshot.shards().values()) {
                    if (((SnapshotsInProgress.ShardSnapshotStatus)shardStatus.value).state().completed() || !node.getId().equals(((SnapshotsInProgress.ShardSnapshotStatus)shardStatus.value).nodeId())) continue;
                    return true;
                }
            }
        }
        return false;
    }

    private Tuple<Set<String>, Set<String>> indicesWithMissingShards(ImmutableOpenMap<ShardId, SnapshotsInProgress.ShardSnapshotStatus> shards, MetaData metaData) {
        HashSet<String> missing = new HashSet<String>();
        HashSet<String> closed = new HashSet<String>();
        for (ObjectObjectCursor<ShardId, SnapshotsInProgress.ShardSnapshotStatus> objectObjectCursor : shards) {
            if (((SnapshotsInProgress.ShardSnapshotStatus)objectObjectCursor.value).state() != SnapshotsInProgress.State.MISSING) continue;
            if (metaData.hasIndex(((ShardId)objectObjectCursor.key).getIndex().getName()) && metaData.getIndexSafe(((ShardId)objectObjectCursor.key).getIndex()).getState() == IndexMetaData.State.CLOSE) {
                closed.add(((ShardId)objectObjectCursor.key).getIndex().getName());
                continue;
            }
            missing.add(((ShardId)objectObjectCursor.key).getIndex().getName());
        }
        return new Tuple<Set<String>, Set<String>>(missing, closed);
    }

    void endSnapshot(SnapshotsInProgress.Entry entry) {
        this.endSnapshot(entry, null);
    }

    private void endSnapshot(final SnapshotsInProgress.Entry entry, final String failure) {
        this.threadPool.executor("snapshot").execute(new Runnable(){

            @Override
            public void run() {
                Snapshot snapshot = entry.snapshot();
                try {
                    Repository repository = SnapshotsService.this.repositoriesService.repository(snapshot.getRepository());
                    SnapshotsService.this.logger.trace("[{}] finalizing snapshot in repository, state: [{}], failure[{}]", (Object)snapshot, (Object)entry.state(), (Object)failure);
                    ArrayList<SnapshotShardFailure> shardFailures = new ArrayList<SnapshotShardFailure>();
                    for (ObjectObjectCursor<ShardId, SnapshotsInProgress.ShardSnapshotStatus> objectObjectCursor : entry.shards()) {
                        ShardId shardId = (ShardId)objectObjectCursor.key;
                        SnapshotsInProgress.ShardSnapshotStatus status = (SnapshotsInProgress.ShardSnapshotStatus)objectObjectCursor.value;
                        if (!status.state().failed()) continue;
                        shardFailures.add(new SnapshotShardFailure(status.nodeId(), shardId, status.reason()));
                    }
                    SnapshotInfo snapshotInfo = repository.finalizeSnapshot(snapshot.getSnapshotId(), entry.indices(), entry.startTime(), failure, entry.shards().size(), Collections.unmodifiableList(shardFailures), entry.getRepositoryStateId());
                    SnapshotsService.this.removeSnapshotFromClusterState(snapshot, snapshotInfo, null);
                }
                catch (Exception e) {
                    SnapshotsService.this.logger.warn(() -> new ParameterizedMessage("[{}] failed to finalize snapshot", (Object)snapshot), (Throwable)e);
                    SnapshotsService.this.removeSnapshotFromClusterState(snapshot, null, e);
                }
            }
        });
    }

    private void removeSnapshotFromClusterState(Snapshot snapshot, SnapshotInfo snapshotInfo, Exception e) {
        this.removeSnapshotFromClusterState(snapshot, snapshotInfo, e, null);
    }

    private void removeSnapshotFromClusterState(final Snapshot snapshot, final SnapshotInfo snapshotInfo, final Exception failure, final @Nullable ActionListener<SnapshotInfo> listener) {
        this.clusterService.submitStateUpdateTask("remove snapshot metadata", new ClusterStateUpdateTask(){

            @Override
            public ClusterState execute(ClusterState currentState) {
                SnapshotsInProgress snapshots = (SnapshotsInProgress)currentState.custom("snapshots");
                if (snapshots != null) {
                    boolean changed = false;
                    ArrayList<SnapshotsInProgress.Entry> entries = new ArrayList<SnapshotsInProgress.Entry>();
                    for (SnapshotsInProgress.Entry entry : snapshots.entries()) {
                        if (entry.snapshot().equals(snapshot)) {
                            changed = true;
                            continue;
                        }
                        entries.add(entry);
                    }
                    if (changed) {
                        snapshots = new SnapshotsInProgress(entries.toArray(new SnapshotsInProgress.Entry[entries.size()]));
                        return ClusterState.builder(currentState).putCustom("snapshots", snapshots).build();
                    }
                }
                return currentState;
            }

            @Override
            public void onFailure(String source, Exception e) {
                SnapshotsService.this.logger.warn(() -> new ParameterizedMessage("[{}] failed to remove snapshot metadata", (Object)snapshot), (Throwable)e);
                if (listener != null) {
                    listener.onFailure(e);
                }
            }

            @Override
            public void clusterStateProcessed(String source, ClusterState oldState, ClusterState newState) {
                for (SnapshotCompletionListener listener2 : SnapshotsService.this.snapshotCompletionListeners) {
                    try {
                        if (snapshotInfo != null) {
                            listener2.onSnapshotCompletion(snapshot, snapshotInfo);
                            continue;
                        }
                        listener2.onSnapshotFailure(snapshot, failure);
                    }
                    catch (Exception t) {
                        SnapshotsService.this.logger.warn(() -> new ParameterizedMessage("failed to notify listener [{}]", (Object)listener2), (Throwable)t);
                    }
                }
                if (listener != null) {
                    listener.onResponse(snapshotInfo);
                }
            }
        });
    }

    public void deleteSnapshot(String repositoryName, String snapshotName, DeleteSnapshotListener listener, boolean immediatePriority) {
        Repository repository = this.repositoriesService.repository(repositoryName);
        RepositoryData repositoryData = repository.getRepositoryData();
        Optional<SnapshotId> incompatibleSnapshotId = repositoryData.getIncompatibleSnapshotIds().stream().filter(s -> snapshotName.equals(s.getName())).findFirst();
        if (incompatibleSnapshotId.isPresent()) {
            throw new SnapshotException(repositoryName, snapshotName, "cannot delete incompatible snapshot");
        }
        Optional<SnapshotId> matchedEntry = repositoryData.getSnapshotIds().stream().filter(s -> s.getName().equals(snapshotName)).findFirst();
        if (!matchedEntry.isPresent()) {
            matchedEntry = this.currentSnapshots(repositoryName, Collections.emptyList()).stream().map(e -> e.snapshot().getSnapshotId()).filter(s -> s.getName().equals(snapshotName)).findFirst();
        }
        if (!matchedEntry.isPresent()) {
            throw new SnapshotMissingException(repositoryName, snapshotName);
        }
        this.deleteSnapshot(new Snapshot(repositoryName, matchedEntry.get()), listener, repositoryData.getGenId(), immediatePriority);
    }

    private void deleteSnapshot(final Snapshot snapshot, final DeleteSnapshotListener listener, final long repositoryStateId, boolean immediatePriority) {
        Priority priority = immediatePriority ? Priority.IMMEDIATE : Priority.NORMAL;
        this.clusterService.submitStateUpdateTask("delete snapshot", new ClusterStateUpdateTask(priority){
            boolean waitForSnapshot;
            {
                super(priority);
                this.waitForSnapshot = false;
            }

            @Override
            public ClusterState execute(ClusterState currentState) throws Exception {
                SnapshotsInProgress.Entry snapshotEntry;
                SnapshotDeletionsInProgress deletionsInProgress = (SnapshotDeletionsInProgress)currentState.custom("snapshot_deletions");
                if (deletionsInProgress != null && deletionsInProgress.hasDeletionsInProgress()) {
                    throw new ConcurrentSnapshotExecutionException(snapshot, "cannot delete - another snapshot is currently being deleted");
                }
                RestoreInProgress restoreInProgress = (RestoreInProgress)currentState.custom("restore");
                if (restoreInProgress != null && !restoreInProgress.entries().isEmpty()) {
                    throw new ConcurrentSnapshotExecutionException(snapshot, "cannot delete snapshot during a restore");
                }
                ClusterState.Builder clusterStateBuilder = ClusterState.builder(currentState);
                SnapshotsInProgress snapshots = (SnapshotsInProgress)currentState.custom("snapshots");
                SnapshotsInProgress.Entry entry = snapshotEntry = snapshots != null ? snapshots.snapshot(snapshot) : null;
                if (snapshotEntry == null) {
                    if (snapshots != null && !snapshots.entries().isEmpty()) {
                        throw new ConcurrentSnapshotExecutionException(snapshot, "another snapshot is currently running cannot delete");
                    }
                    SnapshotDeletionsInProgress.Entry entry2 = new SnapshotDeletionsInProgress.Entry(snapshot, System.currentTimeMillis(), repositoryStateId);
                    deletionsInProgress = deletionsInProgress != null ? deletionsInProgress.withAddedEntry(entry2) : SnapshotDeletionsInProgress.newInstance(entry2);
                    clusterStateBuilder.putCustom("snapshot_deletions", deletionsInProgress);
                } else {
                    ImmutableOpenMap<ShardId, SnapshotsInProgress.ShardSnapshotStatus> shards;
                    this.waitForSnapshot = true;
                    if (snapshotEntry.state() == SnapshotsInProgress.State.STARTED && snapshotEntry.shards() != null) {
                        ImmutableOpenMap.Builder<Object, SnapshotsInProgress.ShardSnapshotStatus> shardsBuilder = ImmutableOpenMap.builder();
                        for (ObjectObjectCursor<ShardId, SnapshotsInProgress.ShardSnapshotStatus> objectObjectCursor : snapshotEntry.shards()) {
                            SnapshotsInProgress.ShardSnapshotStatus status = (SnapshotsInProgress.ShardSnapshotStatus)objectObjectCursor.value;
                            if (!status.state().completed()) {
                                shardsBuilder.put(objectObjectCursor.key, new SnapshotsInProgress.ShardSnapshotStatus(status.nodeId(), SnapshotsInProgress.State.ABORTED));
                                continue;
                            }
                            shardsBuilder.put(objectObjectCursor.key, status);
                        }
                        shards = shardsBuilder.build();
                    } else if (snapshotEntry.state() == SnapshotsInProgress.State.INIT) {
                        shards = snapshotEntry.shards();
                        SnapshotsService.this.endSnapshot(snapshotEntry);
                    } else {
                        boolean hasUncompletedShards = false;
                        for (ObjectCursor objectCursor : snapshotEntry.shards().values()) {
                            if (((SnapshotsInProgress.ShardSnapshotStatus)objectCursor.value).state().completed() || ((SnapshotsInProgress.ShardSnapshotStatus)objectCursor.value).nodeId() == null || currentState.nodes().get(((SnapshotsInProgress.ShardSnapshotStatus)objectCursor.value).nodeId()) == null) continue;
                            hasUncompletedShards = true;
                            break;
                        }
                        if (hasUncompletedShards) {
                            SnapshotsService.this.logger.debug("trying to delete completed snapshot - should wait for shards to finalize on all nodes");
                            return currentState;
                        }
                        SnapshotsService.this.logger.debug("trying to delete completed snapshot with no finalizing shards - can delete immediately");
                        shards = snapshotEntry.shards();
                        SnapshotsService.this.endSnapshot(snapshotEntry);
                    }
                    SnapshotsInProgress.Entry newSnapshot = new SnapshotsInProgress.Entry(snapshotEntry, SnapshotsInProgress.State.ABORTED, shards);
                    snapshots = new SnapshotsInProgress(newSnapshot);
                    clusterStateBuilder.putCustom("snapshots", snapshots);
                }
                return clusterStateBuilder.build();
            }

            @Override
            public void onFailure(String source, Exception e) {
                listener.onFailure(e);
            }

            @Override
            public void clusterStateProcessed(String source, ClusterState oldState, ClusterState newState) {
                if (this.waitForSnapshot) {
                    SnapshotsService.this.logger.trace("adding snapshot completion listener to wait for deleted snapshot to finish");
                    SnapshotsService.this.addListener(new SnapshotCompletionListener(){

                        @Override
                        public void onSnapshotCompletion(Snapshot completedSnapshot, SnapshotInfo snapshotInfo) {
                            if (completedSnapshot.equals(snapshot)) {
                                SnapshotsService.this.logger.debug("deleted snapshot completed - deleting files");
                                SnapshotsService.this.removeListener(this);
                                SnapshotsService.this.threadPool.executor("snapshot").execute(() -> SnapshotsService.this.deleteSnapshot(completedSnapshot.getRepository(), completedSnapshot.getSnapshotId().getName(), listener, true));
                            }
                        }

                        @Override
                        public void onSnapshotFailure(Snapshot failedSnapshot, Exception e) {
                            if (failedSnapshot.equals(snapshot)) {
                                SnapshotsService.this.logger.warn("deleted snapshot failed - deleting files", (Throwable)e);
                                SnapshotsService.this.removeListener(this);
                                SnapshotsService.this.threadPool.executor("snapshot").execute(() -> {
                                    try {
                                        SnapshotsService.this.deleteSnapshot(failedSnapshot.getRepository(), failedSnapshot.getSnapshotId().getName(), listener, true);
                                    }
                                    catch (SnapshotMissingException smex) {
                                        SnapshotsService.this.logger.info(() -> new ParameterizedMessage("Tried deleting in-progress snapshot [{}], but it could not be found after failing to abort.", (Object)smex.getSnapshotName()), (Throwable)e);
                                        listener.onFailure(new SnapshotException(snapshot, "Tried deleting in-progress snapshot [{}], but it could not be found after failing to abort.", smex));
                                    }
                                });
                            }
                        }
                    });
                } else {
                    SnapshotsService.this.logger.debug("deleted snapshot is not running - deleting files");
                    SnapshotsService.this.deleteSnapshotFromRepository(snapshot, listener, repositoryStateId);
                }
            }
        });
    }

    public static boolean isRepositoryInUse(ClusterState clusterState, String repository) {
        SnapshotDeletionsInProgress deletionsInProgress;
        SnapshotsInProgress snapshots = (SnapshotsInProgress)clusterState.custom("snapshots");
        if (snapshots != null) {
            for (SnapshotsInProgress.Entry snapshot : snapshots.entries()) {
                if (!repository.equals(snapshot.snapshot().getRepository())) continue;
                return true;
            }
        }
        if ((deletionsInProgress = (SnapshotDeletionsInProgress)clusterState.custom("snapshot_deletions")) != null) {
            for (SnapshotDeletionsInProgress.Entry entry : deletionsInProgress.getEntries()) {
                if (!entry.getSnapshot().getRepository().equals(repository)) continue;
                return true;
            }
        }
        return false;
    }

    private void deleteSnapshotFromRepository(Snapshot snapshot, @Nullable DeleteSnapshotListener listener, long repositoryStateId) {
        this.threadPool.executor("snapshot").execute(() -> {
            try {
                Repository repository = this.repositoriesService.repository(snapshot.getRepository());
                repository.deleteSnapshot(snapshot.getSnapshotId(), repositoryStateId);
                this.removeSnapshotDeletionFromClusterState(snapshot, null, listener);
            }
            catch (Exception ex) {
                this.removeSnapshotDeletionFromClusterState(snapshot, ex, listener);
            }
        });
    }

    private void removeSnapshotDeletionFromClusterState(final Snapshot snapshot, final @Nullable Exception failure, final @Nullable DeleteSnapshotListener listener) {
        this.clusterService.submitStateUpdateTask("remove snapshot deletion metadata", new ClusterStateUpdateTask(){

            @Override
            public ClusterState execute(ClusterState currentState) {
                SnapshotDeletionsInProgress deletions = (SnapshotDeletionsInProgress)currentState.custom("snapshot_deletions");
                if (deletions != null) {
                    boolean changed = false;
                    if (deletions.hasDeletionsInProgress()) {
                        assert (deletions.getEntries().size() == 1) : "should have exactly one deletion in progress";
                        SnapshotDeletionsInProgress.Entry entry = deletions.getEntries().get(0);
                        deletions = deletions.withRemovedEntry(entry);
                        changed = true;
                    }
                    if (changed) {
                        return ClusterState.builder(currentState).putCustom("snapshot_deletions", deletions).build();
                    }
                }
                return currentState;
            }

            @Override
            public void onFailure(String source, Exception e) {
                SnapshotsService.this.logger.warn(() -> new ParameterizedMessage("[{}] failed to remove snapshot deletion metadata", (Object)snapshot), (Throwable)e);
                if (listener != null) {
                    listener.onFailure(e);
                }
            }

            @Override
            public void clusterStateProcessed(String source, ClusterState oldState, ClusterState newState) {
                if (listener != null) {
                    if (failure != null) {
                        listener.onFailure(failure);
                    } else {
                        listener.onResponse();
                    }
                }
            }
        });
    }

    private ImmutableOpenMap<ShardId, SnapshotsInProgress.ShardSnapshotStatus> shards(ClusterState clusterState, List<IndexId> indices) {
        ImmutableOpenMap.Builder<ShardId, SnapshotsInProgress.ShardSnapshotStatus> builder = ImmutableOpenMap.builder();
        MetaData metaData = clusterState.metaData();
        for (IndexId index : indices) {
            String indexName = index.getName();
            IndexMetaData indexMetaData = metaData.index(indexName);
            if (indexMetaData == null) {
                builder.put(new ShardId(indexName, "_na_", 0), new SnapshotsInProgress.ShardSnapshotStatus(null, SnapshotsInProgress.State.MISSING, "missing index"));
                continue;
            }
            if (indexMetaData.getState() == IndexMetaData.State.CLOSE) {
                for (int i = 0; i < indexMetaData.getNumberOfShards(); ++i) {
                    ShardId shardId = new ShardId(indexMetaData.getIndex(), i);
                    builder.put(shardId, new SnapshotsInProgress.ShardSnapshotStatus(null, SnapshotsInProgress.State.MISSING, "index is closed"));
                }
                continue;
            }
            IndexRoutingTable indexRoutingTable = clusterState.getRoutingTable().index(indexName);
            for (int i = 0; i < indexMetaData.getNumberOfShards(); ++i) {
                ShardId shardId = new ShardId(indexMetaData.getIndex(), i);
                if (indexRoutingTable != null) {
                    ShardRouting primary = indexRoutingTable.shard(i).primaryShard();
                    if (primary == null || !primary.assignedToNode()) {
                        builder.put(shardId, new SnapshotsInProgress.ShardSnapshotStatus(null, SnapshotsInProgress.State.MISSING, "primary shard is not allocated"));
                        continue;
                    }
                    if (primary.relocating() || primary.initializing()) {
                        builder.put(shardId, new SnapshotsInProgress.ShardSnapshotStatus(primary.currentNodeId(), SnapshotsInProgress.State.WAITING));
                        continue;
                    }
                    if (!primary.started()) {
                        builder.put(shardId, new SnapshotsInProgress.ShardSnapshotStatus(primary.currentNodeId(), SnapshotsInProgress.State.MISSING, "primary shard hasn't been started yet"));
                        continue;
                    }
                    builder.put(shardId, new SnapshotsInProgress.ShardSnapshotStatus(primary.currentNodeId()));
                    continue;
                }
                builder.put(shardId, new SnapshotsInProgress.ShardSnapshotStatus(null, SnapshotsInProgress.State.MISSING, "missing routing table"));
            }
        }
        return builder.build();
    }

    public static void checkIndexDeletion(ClusterState currentState, Set<IndexMetaData> indices) {
        Set<Index> indicesToFail = SnapshotsService.indicesToFailForCloseOrDeletion(currentState, indices);
        if (indicesToFail != null) {
            throw new IllegalArgumentException("Cannot delete indices that are being snapshotted: " + indicesToFail + ". Try again after snapshot finishes or cancel the currently running snapshot.");
        }
    }

    public static void checkIndexClosing(ClusterState currentState, Set<IndexMetaData> indices) {
        Set<Index> indicesToFail = SnapshotsService.indicesToFailForCloseOrDeletion(currentState, indices);
        if (indicesToFail != null) {
            throw new IllegalArgumentException("Cannot close indices that are being snapshotted: " + indicesToFail + ". Try again after snapshot finishes or cancel the currently running snapshot.");
        }
    }

    private static Set<Index> indicesToFailForCloseOrDeletion(ClusterState currentState, Set<IndexMetaData> indices) {
        SnapshotsInProgress snapshots = (SnapshotsInProgress)currentState.custom("snapshots");
        HashSet<Index> indicesToFail = null;
        if (snapshots != null) {
            for (SnapshotsInProgress.Entry entry : snapshots.entries()) {
                IndexMetaData indexMetaData;
                if (entry.partial()) continue;
                if (entry.state() == SnapshotsInProgress.State.INIT) {
                    for (IndexId indexId : entry.indices()) {
                        indexMetaData = currentState.metaData().index(indexId.getName());
                        if (indexMetaData == null || !indices.contains(indexMetaData)) continue;
                        if (indicesToFail == null) {
                            indicesToFail = new HashSet<Index>();
                        }
                        indicesToFail.add(indexMetaData.getIndex());
                    }
                    continue;
                }
                for (ObjectObjectCursor objectObjectCursor : entry.shards()) {
                    if (((SnapshotsInProgress.ShardSnapshotStatus)objectObjectCursor.value).state().completed() || (indexMetaData = currentState.metaData().index(((ShardId)objectObjectCursor.key).getIndex())) == null || !indices.contains(indexMetaData)) continue;
                    if (indicesToFail == null) {
                        indicesToFail = new HashSet();
                    }
                    indicesToFail.add(((ShardId)objectObjectCursor.key).getIndex());
                }
            }
        }
        return indicesToFail;
    }

    public void addListener(SnapshotCompletionListener listener) {
        this.snapshotCompletionListeners.add(listener);
    }

    public void removeListener(SnapshotCompletionListener listener) {
        this.snapshotCompletionListeners.remove(listener);
    }

    @Override
    protected void doStart() {
    }

    @Override
    protected void doStop() {
    }

    @Override
    protected void doClose() {
        this.clusterService.removeApplier(this);
    }

    public RepositoriesService getRepositoriesService() {
        return this.repositoriesService;
    }

    public static class SnapshotRequest {
        private final String cause;
        private final String repositoryName;
        private final String snapshotName;
        private String[] indices;
        private IndicesOptions indicesOptions = IndicesOptions.strictExpandOpen();
        private boolean partial;
        private Settings settings;
        private boolean includeGlobalState;
        private TimeValue masterNodeTimeout;

        public SnapshotRequest(String repositoryName, String snapshotName, String cause) {
            this.repositoryName = Objects.requireNonNull(repositoryName);
            this.snapshotName = Objects.requireNonNull(snapshotName);
            this.cause = Objects.requireNonNull(cause);
        }

        public SnapshotRequest indices(String[] indices) {
            this.indices = indices;
            return this;
        }

        public SnapshotRequest settings(Settings settings) {
            this.settings = settings;
            return this;
        }

        public SnapshotRequest includeGlobalState(boolean includeGlobalState) {
            this.includeGlobalState = includeGlobalState;
            return this;
        }

        public SnapshotRequest masterNodeTimeout(TimeValue masterNodeTimeout) {
            this.masterNodeTimeout = masterNodeTimeout;
            return this;
        }

        public SnapshotRequest indicesOptions(IndicesOptions indicesOptions) {
            this.indicesOptions = indicesOptions;
            return this;
        }

        public SnapshotRequest partial(boolean partial) {
            this.partial = partial;
            return this;
        }

        public String cause() {
            return this.cause;
        }

        public String repositoryName() {
            return this.repositoryName;
        }

        public String snapshotName() {
            return this.snapshotName;
        }

        public String[] indices() {
            return this.indices;
        }

        public IndicesOptions indicesOptions() {
            return this.indicesOptions;
        }

        public Settings settings() {
            return this.settings;
        }

        public boolean includeGlobalState() {
            return this.includeGlobalState;
        }

        public boolean partial() {
            return this.partial;
        }

        public TimeValue masterNodeTimeout() {
            return this.masterNodeTimeout;
        }
    }

    public static interface SnapshotCompletionListener {
        public void onSnapshotCompletion(Snapshot var1, SnapshotInfo var2);

        public void onSnapshotFailure(Snapshot var1, Exception var2);
    }

    public static interface DeleteSnapshotListener {
        public void onResponse();

        public void onFailure(Exception var1);
    }

    public static interface CreateSnapshotListener {
        public void onResponse();

        public void onFailure(Exception var1);
    }

    private class CleanupAfterErrorListener
    implements ActionListener<SnapshotInfo> {
        private final SnapshotsInProgress.Entry snapshot;
        private final boolean snapshotCreated;
        private final CreateSnapshotListener userCreateSnapshotListener;
        private final Exception e;

        CleanupAfterErrorListener(SnapshotsInProgress.Entry snapshot, boolean snapshotCreated, CreateSnapshotListener userCreateSnapshotListener, Exception e) {
            this.snapshot = snapshot;
            this.snapshotCreated = snapshotCreated;
            this.userCreateSnapshotListener = userCreateSnapshotListener;
            this.e = e;
        }

        @Override
        public void onResponse(SnapshotInfo snapshotInfo) {
            this.cleanupAfterError(this.e);
        }

        @Override
        public void onFailure(Exception e) {
            e.addSuppressed(this.e);
            this.cleanupAfterError(e);
        }

        private void cleanupAfterError(Exception exception) {
            if (this.snapshotCreated) {
                try {
                    SnapshotsService.this.repositoriesService.repository(this.snapshot.snapshot().getRepository()).finalizeSnapshot(this.snapshot.snapshot().getSnapshotId(), this.snapshot.indices(), this.snapshot.startTime(), ExceptionsHelper.detailedMessage(exception), 0, Collections.emptyList(), this.snapshot.getRepositoryStateId());
                }
                catch (Exception inner) {
                    inner.addSuppressed(exception);
                    SnapshotsService.this.logger.warn(() -> new ParameterizedMessage("[{}] failed to close snapshot in repository", (Object)this.snapshot.snapshot()), (Throwable)inner);
                }
            }
            this.userCreateSnapshotListener.onFailure(this.e);
        }
    }
}

