var glancesApp = angular.module('glancesApp', ['glances.config', 'cfp.hotkeys'])

.value('CONFIG', {})
.value('ARGUMENTS', {})

.config(["hotkeysProvider", function (hotkeysProvider) {
    hotkeysProvider.useNgRoute = false;
    hotkeysProvider.includeCheatSheet = false;
}])

.run(["$rootScope", "GlancesStats", function ($rootScope, GlancesStats) {
    $rootScope.title = "Glances";

    $rootScope.$on('data_refreshed', function (event, data) {
        $rootScope.title = data.stats.system.hostname + ' - Glances';
    });

    GlancesStats.init();
}]);

glancesApp.directive("sortableTh", function () {
    return {
        restrict: 'A',
        scope: {
            sorter: '='
        },
        link: function (scope, element, attrs) {

            element.addClass('sortable');

            scope.$watch(function () {
                return scope.sorter.column;
            }, function (newValue, oldValue) {

                if (angular.isArray(newValue)) {
                    if (newValue.indexOf(attrs.column) !== -1) {
                        element.addClass('sort');
                    } else {
                        element.removeClass('sort');
                    }
                } else {
                    if (attrs.column === newValue) {
                        element.addClass('sort');
                    } else {
                        element.removeClass('sort');
                    }
                }

            });

            element.on('click', function () {

                scope.sorter.column = attrs.column;

                scope.$apply();
            });
        }
    };
});
glancesApp.filter('min_size', function () {
    return function (input, max) {
        var max = max || 8;
        if (input.length > max) {
            return "_" + input.substring(input.length - max + 1)
        }
        return input
    };
});
glancesApp.filter('exclamation', function () {
    return function (input) {
        if (input === undefined || input === '') {
            return '?';
        }
        return input;
    };
});

glancesApp.filter('bytes', function () {
    return function (bytes, low_precision) {
        low_precision = low_precision || false;
        if (isNaN(parseFloat(bytes)) || !isFinite(bytes) || bytes == 0) {
            return bytes;
        }

        var symbols = ['K', 'M', 'G', 'T', 'P', 'E', 'Z', 'Y'];
        var prefix = {
            'Y': 1208925819614629174706176,
            'Z': 1180591620717411303424,
            'E': 1152921504606846976,
            'P': 1125899906842624,
            'T': 1099511627776,
            'G': 1073741824,
            'M': 1048576,
            'K': 1024
        };

        var reverseSymbols = _(symbols).reverse().value();
        for (var i = 0; i < reverseSymbols.length; i++) {
            var symbol = reverseSymbols[i];
            var value = bytes / prefix[symbol];

            if (value > 1) {
                var decimal_precision = 0;

                if (value < 10) {
                    decimal_precision = 2;
                }
                else if (value < 100) {
                    decimal_precision = 1;
                }

                if (low_precision) {
                    if (symbol == 'MK') {
                        decimal_precision = 0;
                    }
                    else {
                        decimal_precision = _.min([1, decimal_precision]);
                    }
                }
                else if (symbol == 'K') {
                    decimal_precision = 0;
                }

                return parseFloat(value).toFixed(decimal_precision) + symbol;
            }
        }

        return bytes.toFixed(0);
    }
});

glancesApp.filter('bits', ["$filter", function ($filter) {
    return function (bits, low_precision) {
        bits = Math.round(bits) * 8;
        return $filter('bytes')(bits, low_precision) + 'b';
    }
}]);

glancesApp.filter('leftPad', function () {
    return function (value, length, chars) {
        length = length || 0;
        chars = chars || ' ';
        return _.padStart(value, length, chars);
    }
});

glancesApp.filter('timemillis', function () {
    return function (array) {
        var sum = 0.0;
        for (var i = 0; i < array.length; i++) {
            sum += array[i] * 1000.0;
        }
        return sum;
    }
});

glancesApp.filter('timedelta', ["$filter", function ($filter) {
    return function (value) {
        var sum = $filter('timemillis')(value);
        var d = new Date(sum);

        return {
            hours: d.getUTCHours(), // TODO : multiple days ( * (d.getDay() * 24)))
            minutes: d.getUTCMinutes(),
            seconds: d.getUTCSeconds(),
            milliseconds: parseInt("" + d.getUTCMilliseconds() / 10)
        };
    }
}]);

glancesApp.service('favicoService', function () {

    var favico = new Favico({
        animation: 'none'
    });

    this.badge = function (nb) {
        favico.badge(nb);
    };

    this.reset = function () {
        favico.reset();
    };
});

glancesApp.service('GlancesPluginHelper', function () {

    var plugin = {
        'limits': {},
        'limitSuffix': ['critical', 'careful', 'warning']
    };

    plugin.setLimits = function (limits) {
        this.limits = limits;
    };

    plugin.getAlert = function (pluginName, limitNamePrefix, current, maximum, log) {
        current = current || 0;
        maximum = maximum || 100;
        log = log || false;

        var log_str = log ? '_log' : '';
        var value = (current * 100) / maximum;

        if (this.limits[pluginName] != undefined) {
            for (var i = 0; i < this.limitSuffix.length; i++) {
                var limitName = limitNamePrefix + this.limitSuffix[i];
                var limit = this.limits[pluginName][limitName];

                if (value >= limit) {
                    var pos = limitName.lastIndexOf("_");
                    var className = limitName.substring(pos + 1);

                    return className + log_str;
                }
            }
        }

        return "ok" + log_str;
    };

    plugin.getAlertLog = function (pluginName, limitNamePrefix, current, maximum) {
        return this.getAlert(pluginName, limitNamePrefix, current, maximum, true);
    };

    return plugin;
});

glancesApp.service('GlancesStats', ["$http", "$q", "$rootScope", "$timeout", "GlancesPluginHelper", "REFRESH_TIME", "CONFIG", "ARGUMENTS", function ($http, $q, $rootScope, $timeout, GlancesPluginHelper, REFRESH_TIME, CONFIG, ARGUMENTS) {

    var _data = false;

    this.getData = function () {
        return _data;
    }

    // load config/limit/arguments and execute stats/views auto refresh
    this.init = function () {
        var refreshData = function () {
            return $q.all([
                getAllStats(),
                getAllViews()
            ]).then(function (results) {
                _data = {
                    'stats': results[0],
                    'views': results[1],
                    'isBsd': results[0]['system']['os_name'] === 'FreeBSD',
                    'isLinux': results[0]['system']['os_name'] === 'Linux',
                    'isMac': results[0]['system']['os_name'] === 'Darwin',
                    'isWindows': results[0]['system']['os_name'] === 'Windows'
                };

                $rootScope.$broadcast('data_refreshed', _data);
                nextLoad();
            }, function () {
                $rootScope.$broadcast('is_disconnected');
                nextLoad();
            });
        };

        // load limits to init GlancePlugin helper
        $http.get('api/2/all/limits').then(function (response) {
            GlancesPluginHelper.setLimits(response.data);
        });
        $http.get('api/2/config').then(function (response) {
            angular.extend(CONFIG, response.data);
        });
        $http.get('api/2/args').then(function (response) {
            angular.extend(ARGUMENTS, response.data);
        });

        var loadPromise;
        var cancelNextLoad = function () {
            $timeout.cancel(loadPromise);
        };

        var nextLoad = function () {
            cancelNextLoad();
            loadPromise = $timeout(refreshData, REFRESH_TIME * 1000); // in milliseconds
        };

        refreshData();
    }

    var getAllStats = function () {
        return $http.get('api/2/all').then(function (response) {
            return response.data;
        });
    };

    var getAllViews = function () {
        return $http.get('api/2/all/views').then(function (response) {
            return response.data;
        });
    };
}]);

'use strict';

glancesApp.component('glances', {
    controller: GlancesController,
    controllerAs: 'vm',
    templateUrl: 'components/glances/view.html'
});

'use strict';

function GlancesController($scope, GlancesStats, hotkeys, ARGUMENTS) {
    var vm = this;
    vm.dataLoaded = false;
    vm.arguments = ARGUMENTS;

    $scope.$on('data_refreshed', function (event, data) {
        vm.hasGpu = data.stats.gpu.length > 0;
        vm.dataLoaded = true;
    });

    // A => Enable/disable AMPs
    hotkeys.add({
        combo: 'A',
        callback: function () {
            ARGUMENTS.disable_amps = !ARGUMENTS.disable_amps;
        }
    });

    // d => Show/hide disk I/O stats
    hotkeys.add({
        combo: 'd',
        callback: function () {
            ARGUMENTS.disable_diskio = !ARGUMENTS.disable_diskio;
        }
    });

    // Q => Show/hide IRQ
    hotkeys.add({
        combo: 'Q',
        callback: function () {
            ARGUMENTS.enable_irq = !ARGUMENTS.enable_irq;
        }
    });

    // f => Show/hide filesystem stats
    hotkeys.add({
        combo: 'f',
        callback: function () {
            ARGUMENTS.disable_fs = !ARGUMENTS.disable_fs;
        }
    });

    // n => Show/hide network stats
    hotkeys.add({
        combo: 'n',
        callback: function () {
            ARGUMENTS.disable_network = !ARGUMENTS.disable_network;
        }
    });

    // s => Show/hide sensors stats
    hotkeys.add({
        combo: 's',
        callback: function () {
            ARGUMENTS.disable_sensors = !ARGUMENTS.disable_sensors;
        }
    });

    // 2 => Show/hide left sidebar
    hotkeys.add({
        combo: '2',
        callback: function () {
            ARGUMENTS.disable_left_sidebar = !ARGUMENTS.disable_left_sidebar;
        }
    });

    // z => Enable/disable processes stats
    hotkeys.add({
        combo: 'z',
        callback: function () {
            ARGUMENTS.disable_process = !ARGUMENTS.disable_process;
        }
    });

    // SLASH => Enable/disable short processes name
    hotkeys.add({
        combo: '/',
        callback: function () {
            ARGUMENTS.process_short_name = !ARGUMENTS.process_short_name;
        }
    });

    // D => Enable/disable Docker stats
    hotkeys.add({
        combo: 'D',
        callback: function () {
            ARGUMENTS.disable_docker = !ARGUMENTS.disable_docker;
        }
    });

    // b => Bytes or bits for network I/O
    hotkeys.add({
        combo: 'b',
        callback: function () {
            ARGUMENTS.byte = !ARGUMENTS.byte;
        }
    });

    // 'B' => Switch between bit/s and IO/s for Disk IO
    hotkeys.add({
        combo: 'B',
        callback: function () {
            ARGUMENTS.diskio_iops = !ARGUMENTS.diskio_iops;
        }
    });

    // l => Show/hide alert logs
    hotkeys.add({
        combo: 'l',
        callback: function () {
            ARGUMENTS.disable_alert = !ARGUMENTS.disable_alert;
        }
    });

    // 1 => Global CPU or per-CPU stats
    hotkeys.add({
        combo: '1',
        callback: function () {
            ARGUMENTS.percpu = !ARGUMENTS.percpu;
        }
    });

    // h => Show/hide this help screen
    hotkeys.add({
        combo: 'h',
        callback: function () {
            ARGUMENTS.help_tag = !ARGUMENTS.help_tag;
        }
    });

    // T => View network I/O as combination
    hotkeys.add({
        combo: 'T',
        callback: function () {
            ARGUMENTS.network_sum = !ARGUMENTS.network_sum;
        }
    });

    // U => View cumulative network I/O
    hotkeys.add({
        combo: 'U',
        callback: function () {
            ARGUMENTS.network_cumul = !ARGUMENTS.network_cumul;
        }
    });

    // F => Show filesystem free space
    hotkeys.add({
        combo: 'F',
        callback: function () {
            ARGUMENTS.fs_free_space = !ARGUMENTS.fs_free_space;
        }
    });

    // 3 => Enable/disable quick look plugin
    hotkeys.add({
        combo: '3',
        callback: function () {
            ARGUMENTS.disable_quicklook = !ARGUMENTS.disable_quicklook;
        }
    });

    // 6 => Enable/disable mean gpu
    hotkeys.add({
        combo: '6',
        callback: function () {
            ARGUMENTS.meangpu = !ARGUMENTS.meangpu;
        }
    });

    // G => Enable/disable gpu
    hotkeys.add({
        combo: 'G',
        callback: function () {
            ARGUMENTS.disable_gpu = !ARGUMENTS.disable_gpu;
        }
    });

    hotkeys.add({
        combo: '5',
        callback: function () {
            ARGUMENTS.disable_quicklook = !ARGUMENTS.disable_quicklook;
            ARGUMENTS.disable_cpu = !ARGUMENTS.disable_cpu;
            ARGUMENTS.disable_mem = !ARGUMENTS.disable_mem;
            ARGUMENTS.disable_memswap = !ARGUMENTS.disable_memswap;
            ARGUMENTS.disable_load = !ARGUMENTS.disable_load;
            ARGUMENTS.disable_gpu = !ARGUMENTS.disable_gpu;
        }
    });

    // I => Show/hide IP module
    hotkeys.add({
        combo: 'I',
        callback: function () {
            ARGUMENTS.disable_ip = !ARGUMENTS.disable_ip;
        }
    });

    // P => Enable/disable ports module
    hotkeys.add({
        combo: 'P',
        callback: function () {
            ARGUMENTS.disable_ports = !ARGUMENTS.disable_ports;
        }
    });

    // 'W' > Enable/Disable Wifi plugin
    hotkeys.add({
        combo: 'W',
        callback: function () {
            ARGUMENTS.disable_wifi = !ARGUMENTS.disable_wifi;
        }
    });
}

'use strict';

glancesApp.component('glancesHelp', {
    controller: GlancesHelpController,
    controllerAs: 'vm',
    templateUrl: 'components/help/view.html'
});

'use strict';

function GlancesHelpController($http) {
    var vm = this;

    $http.get('api/2/help').then(function (response) {
        vm.help = response.data;
    });
}

'use strict';

glancesApp.component('glancesPluginAlert', {
    controller: GlancesPluginAlertController,
    controllerAs: 'vm',
    templateUrl: 'components/plugin-alert/view.html'
});

'use strict';

function GlancesPluginAlertController($scope, favicoService) {
    var vm = this;
    var _alerts = [];

    $scope.$on('data_refreshed', function (event, data) {
        var alertStats = data.stats['alert'];
        if (!_.isArray(alertStats)) {
            alertStats = [];
        }

        _alerts = [];
        for (var i = 0; i < alertStats.length; i++) {
            var alertalertStats = alertStats[i];
            var alert = {};

            alert.name = alertalertStats[3];
            alert.level = alertalertStats[2];
            alert.begin = alertalertStats[0] * 1000;
            alert.end = alertalertStats[1] * 1000;
            alert.ongoing = alertalertStats[1] == -1;
            alert.min = alertalertStats[6];
            alert.mean = alertalertStats[5];
            alert.max = alertalertStats[4];

            if (!alert.ongoing) {
                var duration = alert.end - alert.begin;
                var seconds = parseInt((duration / 1000) % 60)
                    , minutes = parseInt((duration / (1000 * 60)) % 60)
                    , hours = parseInt((duration / (1000 * 60 * 60)) % 24);

                alert.duration = _.padStart(hours, 2, '0') + ":" + _.padStart(minutes, 2, '0') + ":" + _.padStart(seconds, 2, '0');
            }

            _alerts.push(alert);
        }

        if (vm.hasOngoingAlerts()) {
            favicoService.badge(vm.countOngoingAlerts());
        } else {
            favicoService.reset();
        }
    });

    vm.hasAlerts = function () {
        return _alerts.length > 0;
    };

    vm.getAlerts = function () {
        return _alerts;
    };

    vm.count = function () {
        return _alerts.length;
    };

    vm.hasOngoingAlerts = function () {
        return _.filter(_alerts, {'ongoing': true}).length > 0;
    };

    vm.countOngoingAlerts = function () {
        return _.filter(_alerts, {'ongoing': true}).length;
    }
}

'use strict';

glancesApp.component('glancesPluginAmps', {
    controller: GlancesPluginAmpsController,
    controllerAs: 'vm',
    templateUrl: 'components/plugin-amps/view.html'
});

'use strict';

function GlancesPluginAmpsController($scope, GlancesStats, favicoService) {
    var vm = this;
    vm.processes = [];

    vm.$onInit = function () {
        loadData(GlancesStats.getData());
    };

    $scope.$on('data_refreshed', function (event, data) {
        loadData(data);
    });

    var loadData = function (data) {
        var processes = data.stats['amps'];

        vm.processes = [];
        angular.forEach(processes, function (process) {
            if (process.result !== null) {
                vm.processes.push(process);
            }
        }, this);
    };

    vm.getDescriptionDecoration = function (process) {
        var count = process.count;
        var countMin = process.countmin;
        var countMax = process.countmax;
        var decoration = "ok";

        if (count > 0) {
            if ((countMin === null || count >= countMin) && (countMax === null || count <= countMax)) {
                decoration = 'ok';
            } else {
                decoration = 'careful';
            }
        } else {
            decoration = countMin === null ? 'ok' : 'critical';
        }

        return decoration;
    }
}

'use strict';

glancesApp.component('glancesPluginCloud', {
    controller: GlancesPluginCloudController,
    controllerAs: 'vm',
    templateUrl: 'components/plugin-cloud/view.html'
});

'use strict';

function GlancesPluginCloudController($scope, GlancesStats) {
    var vm = this;

    vm.provider = null;
    vm.instance = null;

    vm.$onInit = function () {
        loadData(GlancesStats.getData());
    };

    $scope.$on('data_refreshed', function (event, data) {
        loadData(data);
    });

    var loadData = function (data) {
        var stats = data.stats['cloud'];

        if (stats['ami-id'] !== undefined) {
            vm.provider = 'AWS EC2';
            vm.instance = stats['instance-type'] + ' instance ' + stats['instance-id'] + ' (' + stats['region'] + ')';
        }
    }
}

'use strict';

glancesApp.component('glancesPluginCpu', {
    controller: GlancesPluginCpuController,
    controllerAs: 'vm',
    templateUrl: 'components/plugin-cpu/view.html'
});

'use strict';

function GlancesPluginCpuController($scope, GlancesStats) {
    var vm = this;
    var _view = {};

    vm.total = null;
    vm.user = null;
    vm.system = null;
    vm.idle = null;
    vm.nice = null;
    vm.irq = null;
    vm.iowait = null;
    vm.steal = null;
    vm.ctx_switches = null;
    vm.interrupts = null;
    vm.soft_interrupts = null;
    vm.syscalls = null;

    vm.$onInit = function () {
        loadData(GlancesStats.getData());
    };

    $scope.$on('data_refreshed', function (event, data) {
        loadData(data);
    });

    var loadData = function (data) {
        var stats = data.stats['cpu'];
        _view = data.views['cpu'];

        vm.total = stats.total;
        vm.user = stats.user;
        vm.system = stats.system;
        vm.idle = stats.idle;
        vm.nice = stats.nice;
        vm.irq = stats.irq;
        vm.iowait = stats.iowait;
        vm.steal = stats.steal;

        if (stats.ctx_switches) {
            vm.ctx_switches = Math.floor(stats.ctx_switches / stats.time_since_update);
        }

        if (stats.interrupts) {
            vm.interrupts = Math.floor(stats.interrupts / stats.time_since_update);
        }

        if (stats.soft_interrupts) {
            vm.soft_interrupts = Math.floor(stats.soft_interrupts / stats.time_since_update);
        }

        if (stats.syscalls) {
            vm.syscalls = Math.floor(stats.syscalls / stats.time_since_update);
        }
    }

    vm.getDecoration = function (value) {
        if (_view[value] === undefined) {
            return;
        }

        return _view[value].decoration.toLowerCase();
    };
}

'use strict';

glancesApp.component('glancesPluginDiskio', {
    controller: GlancesPluginDiskioController,
    controllerAs: 'vm',
    templateUrl: 'components/plugin-diskio/view.html'
});

'use strict';

function GlancesPluginDiskioController($scope, $filter, GlancesStats, ARGUMENTS) {
    var vm = this;
    vm.arguments = ARGUMENTS;
    vm.disks = [];

    vm.$onInit = function () {
        loadData(GlancesStats.getData());
    };

    $scope.$on('data_refreshed', function (event, data) {
        loadData(data);
    });

    var loadData = function (data) {
        var stats = data.stats['diskio'] || [];
        stats = $filter('orderBy')(stats, 'disk_name');

        vm.disks = stats.map(function(diskioData) {
            var timeSinceUpdate = diskioData['time_since_update'];

            return {
                'name': diskioData['disk_name'],
                'bitrate': {
                    'txps': $filter('bytes')(diskioData['read_bytes'] / timeSinceUpdate),
                    'rxps': $filter('bytes')(diskioData['write_bytes'] / timeSinceUpdate)
                },
                'count': {
                    'txps': $filter('bytes')(diskioData['read_count'] / timeSinceUpdate),
                    'rxps': $filter('bytes')(diskioData['write_count'] / timeSinceUpdate)
                },
                'alias': diskioData['alias'] !== undefined ? diskioData['alias'] : null
            };
        });
    }
}

'use strict';

glancesApp.component('glancesPluginDocker', {
    controller: GlancesPluginDockerController,
    controllerAs: 'vm',
    templateUrl: 'components/plugin-docker/view.html'
});

'use strict';

function GlancesPluginDockerController($scope, GlancesStats) {
    var vm = this;
    vm.containers = [];
    vm.version = null;

    vm.$onInit = function () {
        loadData(GlancesStats.getData());
    };

    $scope.$on('data_refreshed', function (event, data) {
        loadData(data);
    });

    var loadData = function (data) {
        var stats = data.stats['docker'];
        vm.containers = [];

        if (_.isEmpty(stats)) {
            return;
        }

        vm.containers = stats['containers'].map(function(containerData) {
            return {
                'id': containerData.Id,
                'name': containerData.Names[0].split('/').splice(-1)[0],
                'status': containerData.Status,
                'cpu': containerData.cpu.total,
                'memory': containerData.memory.usage != undefined ? containerData.memory.usage : '?',
                'ior': containerData.io.ior != undefined ? containerData.io.ior : '?',
                'iow': containerData.io.iow != undefined ? containerData.io.iow : '?',
                'io_time_since_update': containerData.io.time_since_update,
                'rx': containerData.network.rx != undefined ? containerData.network.rx : '?',
                'tx': containerData.network.tx != undefined ? containerData.network.tx : '?',
                'net_time_since_update': containerData.network.time_since_update,
                'command': containerData.Command,
                'image': containerData.Image
            };
        });

        vm.version = stats['version']['Version'];
    }
}

'use strict';

glancesApp.component('glancesPluginFolders', {
    controller: GlancesPluginFsController,
    controllerAs: 'vm',
    templateUrl: 'components/plugin-folders/view.html'
});

'use strict';

function GlancesPluginFoldersController($scope, GlancesStats) {
    var vm = this;
    vm.folders = [];

    vm.$onInit = function () {
        loadData(GlancesStats.getData());
    };

    $scope.$on('data_refreshed', function (event, data) {
        loadData(data);
    });

    var loadData = function (data) {
        var stats = data.stats['folders'];
        vm.folders = [];

        for (var i = 0; i < stats.length; i++) {
            var folderData = stats[i];

            var folder = {
                'path': folderData['path'],
                'size': folderData['size'],
                'careful': folderData['careful'],
                'warning': folderData['warning'],
                'critical': folderData['critical']
            };

            vm.folders.push(folder);
        }
    }

    vm.getDecoration = function (folder) {

        if (!Number.isInteger(folder.size)) {
            return;
        }

        if (folder.critical !== null && folder.size > (folder.critical * 1000000)) {
            return 'critical';
        } else if (folder.warning !== null && folder.size > (folder.warning * 1000000)) {
            return 'warning';
        } else if (folder.careful !== null && folder.size > (folder.careful * 1000000)) {
            return 'careful';
        }

        return 'ok';
    };
}

'use strict';

glancesApp.component('glancesPluginFs', {
    controller: GlancesPluginFsController,
    controllerAs: 'vm',
    templateUrl: 'components/plugin-fs/view.html'
});

'use strict';

function GlancesPluginFsController($scope, $filter, GlancesStats, ARGUMENTS) {
    var vm = this;
    var _view = {};
    vm.arguments = ARGUMENTS;
    vm.fileSystems = [];

    vm.$onInit = function () {
        loadData(GlancesStats.getData());
    };

    $scope.$on('data_refreshed', function (event, data) {
        loadData(data);
    });

    var loadData = function (data) {
        var stats = data.stats['fs'];
        _view = data.views['fs'];

        vm.fileSystems = [];
        for (var i = 0; i < stats.length; i++) {
            var fsData = stats[i];

            var shortMountPoint = fsData['mnt_point'];
            if (shortMountPoint.length > 9) {
                shortMountPoint = '_' + fsData['mnt_point'].slice(-8);
            }

            vm.fileSystems.push(fs = {
                'name': fsData['device_name'],
                'mountPoint': fsData['mnt_point'],
                'shortMountPoint': shortMountPoint,
                'percent': fsData['percent'],
                'size': fsData['size'],
                'used': fsData['used'],
                'free': fsData['free']
            });
        }

        vm.fileSystems = $filter('orderBy')(vm.fileSystems, 'mnt_point');
    };

    vm.getDecoration = function (mountPoint, field) {
        if (_view[mountPoint][field] == undefined) {
            return;
        }

        return _view[mountPoint][field].decoration.toLowerCase();
    };
}

'use strict';

glancesApp.component('glancesPluginGpu', {
    controller: GlancesPluginGpuController,
    controllerAs: 'vm',
    templateUrl: 'components/plugin-gpu/view.html'
});

'use strict';

function GlancesPluginGpuController($scope, GlancesStats, ARGUMENTS) {
    var vm = this;
    vm.arguments = ARGUMENTS;
    var _view = {};
    vm.gpus = [];
    vm.name = "GPU";
    vm.mean = {};

    vm.$onInit = function () {
        loadData(GlancesStats.getData());
    };

    $scope.$on('data_refreshed', function (event, data) {
        loadData(data);
    });

    var loadData = function (data) {
        var stats = data.stats['gpu'];
        _view = data.views['gpu'];

        if (stats.length === 0) {
            return;
        }

        vm.gpus = [];
        vm.name = "GPU";
        vm.mean = {
            proc: null,
            mem: null
        };
        var sameName = true;

        for (var i = 0; i < stats.length; i++) {
            var gpuData = stats[i];

            var gpu = gpuData;

            vm.mean.proc += gpu.proc;
            vm.mean.mem += gpu.mem;

            vm.gpus.push(gpu);
        }

        if (stats.length === 1) {
            vm.name = stats[0].name;
        } else if (sameName) {
            vm.name = stats.length + ' GPU ' + stats[0].name;
        }

        vm.mean.proc = vm.mean.proc / stats.length;
        vm.mean.mem = vm.mean.mem / stats.length;
    }

    vm.getDecoration = function (gpuId, value) {
        if (_view[gpuId][value] == undefined) {
            return;
        }

        return _view[gpuId][value].decoration.toLowerCase();
    };

    vm.getMeanDecoration = function (value) {
        return vm.getDecoration(0, value);
    };
}

'use strict';

glancesApp.component('glancesPluginIp', {
    controller: GlancesPluginIpController,
    controllerAs: 'vm',
    templateUrl: 'components/plugin-ip/view.html'
});

'use strict';

function GlancesPluginIpController($scope, GlancesStats, ARGUMENTS) {
    var vm = this;
    vm.arguments = ARGUMENTS;

    vm.address = null;
    vm.gateway = null;
    vm.mask = null;
    vm.maskCidr = null;
    vm.publicAddress = null;

    vm.$onInit = function () {
        loadData(GlancesStats.getData());
    };

    $scope.$on('data_refreshed', function (event, data) {
        loadData(data);
    });

    var loadData = function (data) {
        var ipStats = data.stats['ip'];

        vm.address = ipStats.address;
        vm.gateway = ipStats.gateway;
        vm.mask = ipStats.mask;
        vm.maskCidr = ipStats.mask_cidr;
        vm.publicAddress = ipStats.public_address
    }
}

'use strict';

glancesApp.component('glancesPluginIrq', {
    controller: GlancesPluginIrqController,
    controllerAs: 'vm',
    templateUrl: 'components/plugin-irq/view.html'
});

'use strict';

function GlancesPluginIrqController($scope, GlancesStats) {
    var vm = this;
    vm.irqs = [];

    vm.$onInit = function () {
        loadData(GlancesStats.getData());
    };

    $scope.$on('data_refreshed', function (event, data) {
        loadData(data);
    });

    var loadData = function (data) {
        var stats = data.stats['irq'];
        vm.irqs = [];

        for (var i = 0; i < stats.length; i++) {
            var IrqData = stats[i];

            var irq = {
                'irq_line': IrqData['irq_line'],
                'irq_rate': IrqData['irq_rate']
            };

            vm.irqs.push(irq);
        }
    }
}

'use strict';

glancesApp.component('glancesPluginLoad', {
    controller: GlancesPluginLoadController,
    controllerAs: 'vm',
    templateUrl: 'components/plugin-load/view.html'
});

'use strict';

function GlancesPluginLoadController($scope, GlancesStats) {
    var vm = this;
    var _view = {};

    vm.cpucore = null;
    vm.min1 = null;
    vm.min5 = null;
    vm.min15 = null;

    vm.$onInit = function () {
        loadData(GlancesStats.getData());
    };

    $scope.$on('data_refreshed', function (event, data) {
        loadData(data);
    });

    var loadData = function (data) {
        var stats = data.stats['load'];
        _view = data.views['load'];

        vm.cpucore = stats['cpucore'];
        vm.min1 = stats['min1'];
        vm.min5 = stats['min5'];
        vm.min15 = stats['min15'];
    };

    vm.getDecoration = function (value) {
        if (_view[value] === undefined) {
            return;
        }

        return _view[value].decoration.toLowerCase();
    };
}

'use strict';

glancesApp.component('glancesPluginMem', {
    controller: GlancesPluginMemController,
    controllerAs: 'vm',
    templateUrl: 'components/plugin-mem/view.html'
});

'use strict';

function GlancesPluginMemController($scope, GlancesStats) {
    var vm = this;
    var _view = {};

    vm.percent = null;
    vm.total = null;
    vm.used = null;
    vm.free = null;

    vm.$onInit = function () {
        loadData(GlancesStats.getData());
    };

    $scope.$on('data_refreshed', function (event, data) {
        loadData(data);
    });

    var loadData = function (data) {
        var stats = data.stats['mem'];
        _view = data.views['mem'];

        vm.percent = stats['percent'];
        vm.total = stats['total'];
        vm.used = stats['used'];
        vm.free = stats['free'];
    }

    vm.getDecoration = function (value) {
        if (_view[value] === undefined) {
            return;
        }

        return _view[value].decoration.toLowerCase();
    };
}

'use strict';

glancesApp.component('glancesPluginMemMore', {
    controller: GlancesPluginMemMoreController,
    controllerAs: 'vm',
    templateUrl: 'components/plugin-mem-more/view.html'
});

'use strict';

function GlancesPluginMemMoreController($scope, GlancesStats) {
    var vm = this;

    vm.active = null;
    vm.inactive = null;
    vm.buffers = null;
    vm.cached = null;

    vm.$onInit = function () {
        loadData(GlancesStats.getData());
    };

    $scope.$on('data_refreshed', function (event, data) {
        loadData(data);
    });

    var loadData = function (data) {
        var stats = data.stats['mem'];

        vm.active = stats['active'];
        vm.inactive = stats['inactive'];
        vm.buffers = stats['buffers'];
        vm.cached = stats['cached'];
    }
}

'use strict';

glancesApp.component('glancesPluginMemswap', {
    controller: GlancesPluginMemswapController,
    controllerAs: 'vm',
    templateUrl: 'components/plugin-memswap/view.html'
});

'use strict';

function GlancesPluginMemswapController($scope, GlancesStats) {
    var vm = this;
    var _view = {};

    vm.percent = null;
    vm.total = null;
    vm.used = null;
    vm.free = null;

    vm.$onInit = function () {
        loadData(GlancesStats.getData());
    };

    $scope.$on('data_refreshed', function (event, data) {
        loadData(data);
    });

    var loadData = function (data) {
        var stats = data.stats['memswap'];
        _view = data.views['memswap'];

        vm.percent = stats['percent'];
        vm.total = stats['total'];
        vm.used = stats['used'];
        vm.free = stats['free'];
    };

    vm.getDecoration = function (value) {
        if (_view[value] === undefined) {
            return;
        }

        return _view[value].decoration.toLowerCase();
    };
}

'use strict';

glancesApp.component('glancesPluginPercpu', {
    controller: GlancesPluginPercpuController,
    controllerAs: 'vm',
    templateUrl: 'components/plugin-percpu/view.html'
});

'use strict';

function GlancesPluginPercpuController($scope, GlancesStats, GlancesPluginHelper) {
    var vm = this;
    vm.cpus = [];

    vm.$onInit = function () {
        loadData(GlancesStats.getData());
    };

    $scope.$on('data_refreshed', function (event, data) {
        loadData(data);
    });

    var loadData = function (data) {
        var percpuStats = data.stats['percpu'];

        vm.cpus = [];

        for (var i = 0; i < percpuStats.length; i++) {
            var cpuData = percpuStats[i];

            vm.cpus.push({
                'number': cpuData.cpu_number,
                'total': cpuData.total,
                'user': cpuData.user,
                'system': cpuData.system,
                'idle': cpuData.idle,
                'iowait': cpuData.iowait,
                'steal': cpuData.steal
            });
        }
    }

    vm.getUserAlert = function (cpu) {
        return GlancesPluginHelper.getAlert('percpu', 'percpu_user_', cpu.user)
    };

    vm.getSystemAlert = function (cpu) {
        return GlancesPluginHelper.getAlert('percpu', 'percpu_system_', cpu.system);
    };
}

'use strict';

glancesApp.component('glancesPluginNetwork', {
    controller: GlancesPluginNetworkController,
    controllerAs: 'vm',
    templateUrl: 'components/plugin-network/view.html'
});

'use strict';

function GlancesPluginNetworkController($scope, $filter, GlancesStats, ARGUMENTS) {
    var vm = this;
    vm.arguments = ARGUMENTS;
    vm.networks = [];

    vm.$onInit = function () {
        loadData(GlancesStats.getData());
    };

    $scope.$on('data_refreshed', function (event, data) {
        loadData(data);
    });

    var loadData = function (data) {
        var networkStats = data.stats['network'];

        vm.networks = [];
        for (var i = 0; i < networkStats.length; i++) {
            var networkData = networkStats[i];

            var network = {
                'interfaceName': networkData['interface_name'],
                'rx': networkData['rx'],
                'tx': networkData['tx'],
                'cx': networkData['cx'],
                'time_since_update': networkData['time_since_update'],
                'cumulativeRx': networkData['cumulative_rx'],
                'cumulativeTx': networkData['cumulative_tx'],
                'cumulativeCx': networkData['cumulative_cx']
            };

            vm.networks.push(network);
        }

        vm.networks = $filter('orderBy')(vm.networks, 'interfaceName');
    }
}

'use strict';

glancesApp.component('glancesPluginPorts', {
    controller: GlancesPluginPortsController,
    controllerAs: 'vm',
    templateUrl: 'components/plugin-ports/view.html'
});

'use strict';

function GlancesPluginPortsController($scope, GlancesStats) {
    var vm = this;
    vm.ports = [];

    vm.$onInit = function () {
        loadData(GlancesStats.getData());
    };

    $scope.$on('data_refreshed', function (event, data) {
        loadData(data);
    });

    var loadData = function (data) {
        var stats = data.stats['ports'];

        vm.ports = [];
        angular.forEach(stats, function (port) {
            vm.ports.push(port);
        }, this);
    }

    vm.getDecoration = function (port) {
        if (port.status === null) {
            return 'careful';
        }

        if (port.status === false) {
            return 'critical';
        }

        if (port.rtt_warning !== null && port.status > port.rtt_warning) {
            return 'warning';
        }

        return 'ok';
    };
}

'use strict';

glancesApp.component('glancesPluginProcess', {
    controller: GlancesPluginProcessController,
    controllerAs: 'vm',
    templateUrl: 'components/plugin-process/view.html'
});

'use strict';

function GlancesPluginProcessController(ARGUMENTS, hotkeys) {
    var vm = this;
    vm.arguments = ARGUMENTS;

    vm.sorter = {
        column: "cpu_percent",
        auto: true,
        isReverseColumn: function (column) {
            return !(column === 'username' || column === 'name');
        },
        getColumnLabel: function (column) {
            if (_.isEqual(column, ['io_read', 'io_write'])) {
                return 'io_counters';
            } else {
                return column;
            }
        }
    };

    // a => Sort processes automatically
    hotkeys.add({
        combo: 'a',
        callback: function () {
            vm.sorter.column = "cpu_percent";
            vm.sorter.auto = true;
        }
    });

    // c => Sort processes by CPU%
    hotkeys.add({
        combo: 'c',
        callback: function () {
            vm.sorter.column = "cpu_percent";
            vm.sorter.auto = false;
        }
    });

    // m => Sort processes by MEM%
    hotkeys.add({
        combo: 'm',
        callback: function () {
            vm.sorter.column = "memory_percent";
            vm.sorter.auto = false;
        }
    });

    // u => Sort processes by user
    hotkeys.add({
        combo: 'u',
        callback: function () {
            vm.sorter.column = "username";
            vm.sorter.auto = false;
        }
    });

    // p => Sort processes by name
    hotkeys.add({
        combo: 'p',
        callback: function () {
            vm.sorter.column = "name";
            vm.sorter.auto = false;
        }
    });

    // i => Sort processes by I/O rate
    hotkeys.add({
        combo: 'i',
        callback: function () {
            vm.sorter.column = ['io_read', 'io_write'];
            vm.sorter.auto = false;
        }
    });

    // t => Sort processes by time
    hotkeys.add({
        combo: 't',
        callback: function () {
            vm.sorter.column = "timemillis";
            vm.sorter.auto = false;
        }
    });
}

'use strict';

glancesApp.component('glancesPluginProcesscount', {
    controller: GlancesPluginProcesscountController,
    controllerAs: 'vm',
    bindings: {
        sorter: '<'
    },
    templateUrl: 'components/plugin-processcount/view.html'
});

'use strict';

function GlancesPluginProcesscountController($scope, GlancesStats) {
    var vm = this;

    vm.total = null;
    vm.running = null;
    vm.sleeping = null;
    vm.stopped = null;
    vm.thread = null;

    vm.$onInit = function () {
        loadData(GlancesStats.getData());
    };

    $scope.$on('data_refreshed', function (event, data) {
        loadData(data);
    });

    var loadData = function (data) {
        var processcountStats = data.stats['processcount'];

        vm.total = processcountStats['total'] || 0;
        vm.running = processcountStats['running'] || 0;
        vm.sleeping = processcountStats['sleeping'] || 0;
        vm.stopped = processcountStats['stopped'] || 0;
        vm.thread = processcountStats['thread'] || 0;
    }
}

'use strict';

glancesApp.component('glancesPluginProcesslist', {
    controller: GlancesPluginProcesslistController,
    controllerAs: 'vm',
    bindings: {
        sorter: '<'
    },
    templateUrl: 'components/plugin-processlist/view.html'
});

'use strict';

function GlancesPluginProcesslistController($scope, GlancesStats, GlancesPluginHelper, $filter, CONFIG, ARGUMENTS) {
    var vm = this;
    vm.arguments = ARGUMENTS;
    vm.processes = [];
    vm.ioReadWritePresent = false;

    vm.$onInit = function () {
        loadData(GlancesStats.getData());
    };

    $scope.$on('data_refreshed', function (event, data) {
        loadData(data);
    });

    var loadData = function (data) {
        var processlistStats = data.stats['processlist'] || [];

        vm.processes = [];
        vm.ioReadWritePresent = false;

        for (var i = 0; i < processlistStats.length; i++) {
            var process = processlistStats[i];

            process.memvirt = process.memory_info[1];
            process.memres = process.memory_info[0];
            process.timeplus = $filter('timedelta')(process.cpu_times);
            process.timemillis = $filter('timemillis')(process.cpu_times);

            process.ioRead = null;
            process.ioWrite = null;

            if (process.io_counters) {
                vm.ioReadWritePresent = true;

                process.ioRead = (process.io_counters[0] - process.io_counters[2]) / process.time_since_update;

                if (process.ioRead != 0) {
                    process.ioRead = $filter('bytes')(process.ioRead);
                }

                process.ioWrite = (process.io_counters[1] - process.io_counters[3]) / process.time_since_update;

                if (process.ioWrite != 0) {
                    process.ioWrite = $filter('bytes')(process.ioWrite);
                }
            }

            process.isNice = process.nice !== undefined && ((data.stats.isWindows && process.nice != 32) || (!data.stats.isWindows && process.nice != 0));

            if (Array.isArray(process.cmdline)) {
                process.cmdline = process.cmdline.join(' ');
            }

            if (data.isWindows) {
                process.username = _.last(process.username.split('\\'));
            }

            vm.processes.push(process);
        }
    }

    vm.getCpuPercentAlert = function (process) {
        return GlancesPluginHelper.getAlert('processlist', 'processlist_cpu_', process.cpu_percent);
    };

    vm.getMemoryPercentAlert = function (process) {
        return GlancesPluginHelper.getAlert('processlist', 'processlist_mem_', process.cpu_percent);
    };

    vm.getLimit = function () {
        return CONFIG.outputs !== undefined ? CONFIG.outputs.max_processes_display : undefined;
    };
}

'use strict';

glancesApp.component('glancesPluginQuicklook', {
    controller: GlancesPluginQuicklookController,
    controllerAs: 'vm',
    templateUrl: 'components/plugin-quicklook/view.html'
});

'use strict';

function GlancesPluginQuicklookController($scope, GlancesStats, ARGUMENTS) {
    var vm = this;
    vm.arguments = ARGUMENTS;
    var _view = {};

    vm.mem = null;
    vm.cpu = null;
    vm.cpu_name = null;
    vm.cpu_hz_current = null;
    vm.cpu_hz = null;
    vm.swap = null;
    vm.percpus = [];

    vm.$onInit = function () {
        loadData(GlancesStats.getData());
    };

    $scope.$on('data_refreshed', function (event, data) {
        loadData(data);
    });

    var loadData = function (data) {
        var stats = data.stats['quicklook'];
        _view = data.views['quicklook'];

        vm.mem = stats.mem;
        vm.cpu = stats.cpu;
        vm.cpu_name = stats.cpu_name;
        vm.cpu_hz_current = stats.cpu_hz_current;
        vm.cpu_hz = stats.cpu_hz;
        vm.swap = stats.swap;
        vm.percpus = [];

        angular.forEach(stats.percpu, function (cpu) {
            vm.percpus.push({
                'number': cpu.cpu_number,
                'total': cpu.total
            });
        }, this);
    };

    vm.getDecoration = function (value) {
        if (_view[value] === undefined) {
            return;
        }

        return _view[value].decoration.toLowerCase();
    };
}

'use strict';

glancesApp.component('glancesPluginRaid', {
    controller: GlancesPluginRaidController,
    controllerAs: 'vm',
    templateUrl: 'components/plugin-raid/view.html'
});

'use strict';

function GlancesPluginRaidController($scope, GlancesStats) {
    var vm = this;
    vm.disks = [];

    vm.$onInit = function () {
        loadData(GlancesStats.getData());
    };

    $scope.$on('data_refreshed', function (event, data) {
        loadData(data);
    });

    var loadData = function (data) {
        var disks = [];
        var stats = data.stats['raid'];

        _.forIn(stats, function (diskData, diskKey) {
            var disk = {
                'name': diskKey,
                'type': diskData.type == null ? 'UNKNOWN' : diskData.type,
                'used': diskData.used,
                'available': diskData.available,
                'status': diskData.status,
                'degraded': diskData.used < diskData.available,
                'config': diskData.config == null ? '' : diskData.config.replace('_', 'A'),
                'inactive': diskData.status == 'inactive',
                'components': []
            };

            _.forEach(diskData.components, function (number, name) {
                disk.components.push({
                    'number': number,
                    'name': name
                });
            });

            disks.push(disk);
        });

        vm.disks = disks;
    };

    vm.hasDisks = function () {
        return vm.disks.length > 0;
    };

    vm.getAlert = function (disk) {
        if (disk.inactive) {
            return 'critical';
        }

        if (disk.degraded) {
            return 'warning';
        }

        return 'ok'
    };
}

'use strict';

glancesApp.component('glancesPluginSensors', {
    controller: GlancesPluginSensorsController,
    controllerAs: 'vm',
    templateUrl: 'components/plugin-sensors/view.html'
});

'use strict';

function GlancesPluginSensorsController($scope, GlancesStats, GlancesPluginHelper) {
    var vm = this;
    vm.sensors = [];

    vm.$onInit = function () {
        loadData(GlancesStats.getData());
    };

    $scope.$on('data_refreshed', function (event, data) {
        loadData(data);
    });

    var loadData = function (data) {
        var stats = data.stats['sensors'];

        _.remove(stats, function (sensor) {
            return (_.isArray(sensor.value) && _.isEmpty(sensor.value)) || sensor.value === 0;
        });

        vm.sensors = data;
    };

    vm.getAlert = function (sensor) {
        var current = sensor.type == 'battery' ? 100 - sensor.value : sensor.value;

        return GlancesPluginHelper.getAlert('sensors', 'sensors_' + sensor.type + '_', current);
    };
}

'use strict';

glancesApp.component('glancesPluginSystem', {
    controller: GlancesPluginSystemController,
    controllerAs: 'vm',
    templateUrl: 'components/plugin-system/view.html'
});

'use strict';

function GlancesPluginSystemController($scope, GlancesStats) {
    var vm = this;

    vm.hostname = null;
    vm.platform = null;
    vm.humanReadableName = null;
    vm.os = {
        'name': null,
        'version': null
    };

    vm.$onInit = function () {
        loadData(GlancesStats.getData());
    };

    $scope.$on('data_refreshed', function (event, data) {
        loadData(data);
    });

    $scope.$on('is_disconnected', function () {
        vm.isDisconnected = true;
    });

    var loadData = function (data) {
        var stats = data.stats['system'];

        vm.hostname = stats['hostname'];
        vm.platform = stats['platform'];
        vm.os.name = stats['os_name'];
        vm.os.version = stats['os_version'];
        vm.humanReadableName = stats['hr_name'];
        vm.isDisconnected = false;
    }
}

'use strict';

glancesApp.component('glancesPluginUptime', {
    controller: GlancesPluginUptimeController,
    controllerAs: 'vm',
    templateUrl: 'components/plugin-uptime/view.html'
});

'use strict';

function GlancesPluginUptimeController($scope, GlancesStats) {
    var vm = this;
    vm.value = null;

    vm.$onInit = function () {
        loadData(GlancesStats.getData());
    };

    $scope.$on('data_refreshed', function (event, data) {
        loadData(data);
    });

    var loadData = function (data) {
        vm.value = data.stats['uptime'];
    }
}

'use strict';

glancesApp.component('glancesPluginWifi', {
    controller: GlancesPluginWifiController,
    controllerAs: 'vm',
    templateUrl: 'components/plugin-wifi/view.html'
});

'use strict';

function GlancesPluginWifiController($scope, $filter, GlancesStats) {
    var vm = this;
    var _view = {};

    vm.hotspots = [];

    vm.$onInit = function () {
        loadData(GlancesStats.getData());
    };

    $scope.$on('data_refreshed', function (event, data) {
        loadData(data);
    });

    var loadData = function (data) {
        var stats = data.stats['wifi'];
        _view = data.views['wifi'];
        //stats = [{"ssid": "Freebox-40A258", "encrypted": true, "signal": -45, "key": "ssid", "encryption_type": "wpa2", "quality": "65/70"}];

        vm.hotspots = [];
        for (var i = 0; i < stats.length; i++) {
            var hotspotData = stats[i];

            if (hotspotData['ssid'] === '') {
                continue;
            }

            vm.hotspots.push({
                'ssid': hotspotData['ssid'],
                'encrypted': hotspotData['encrypted'],
                'signal': hotspotData['signal'],
                'encryption_type': hotspotData['encryption_type']
            });
        }

        vm.hotspots = $filter('orderBy')(vm.hotspots, 'ssid');
    };

    vm.getDecoration = function (hotpost, field) {
        if (_view[hotpost.ssid][field] === undefined) {
            return;
        }

        return _view[hotpost.ssid][field].decoration.toLowerCase();
    };
}
