# page.py - Copyright (C) 2012 Red Hat, Inc.
# Written by Darryl L. Pierce <dpierce@redhat.com>
#
# This program is free software; you can redistribute it and/or modify
# it under the terms of the GNU General Public License as published by
# the Free Software Foundation; version 2 of the License.
#
# This program is distributed in the hope that it will be useful,
# but WITHOUT ANY WARRANTY; without even the implied warranty of
# MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
# GNU General Public License for more details.
#
# You should have received a copy of the GNU General Public License
# along with this program; if not, write to the Free Software
# Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston,
# MA  02110-1301, USA.  A copy of the GNU General Public License is
# also available at http://www.gnu.org/copyleft/gpl.html.

class Page:
    '''
    A Page represents the set of fields for a single page of an application,
    along with hooks for validating the data presented and also for
    navigating to other pages based on that data.
    '''

    def __init__(self, app, title, next = False, prev = False, finish = False, cancel = True):
        '''
        Creates the page and gives it the specified title.

        Children should assign the list of page elements at this time
        as well.

        Keyword arguments:
        app    -- the Application object
        title  -- the page's title
        next   -- True if there is a next page (def. False)
        prev   -- True if there is a previous page (def.False)
        finish -- True if this page can finish the app (def. False)
        cancel -- True if this page can cancel the app (def. True)
        '''
        self.app = app
        self.title = title
        self.next = next
        self.prev = prev
        self.finish = finish
        self.cancel = cancel
        self.page_elements = None

    def get_title(self):
        '''
        Returns the title for this page.
        '''
        return self.title

    def get_elements(self, screen):
        '''
        Returns the elements for this page as an array of Newt widgets.

        If the fields in a page may change between displays (fields are
        hidden or revealed based on previous pages) then you should
        override this method and recreate the array each time it's
        requested.

        NOTE: The self.page_elements array should be updated even when the
        list of elements is defined dynamically.
        '''
        return self.page_elements

    def get_element_count(self):
        '''
        Returns the number of elements on the page current.
        '''
        return len(self.page_elements)

    def get_valid(self):
        '''
        Called after the user has selected to navigate forward in the
        page flow. It returns True if the page content is valid.
        '''
        return True

    def has_next_button(self):
        '''
        Returns True if there is a page after the current page.
        '''
        return self.next

    def has_previous_button(self):
        '''
        Returns True if there is a page before the current page.
        '''
        return self.prev

    def has_finish_button(self):
        '''
        Returns True if the application can finish on this page.
        '''
        return self.finish

    def has_cancel_button(self):
        '''
        Returns True if the application can be cancelled on this page.
        '''
        return self.cancel

