
/*
 Copyright 2012, Kovid Goyal <kovid at kovidgoyal.net>
 Released under the GPLv3 License
 */

(function() {
  var MathJax, log,
    bind = function(fn, me){ return function(){ return fn.apply(me, arguments); }; };

  log = window.calibre_utils.log;

  MathJax = (function() {
    function MathJax() {
      this.load_finished = bind(this.load_finished, this);
      if (!this instanceof arguments.callee) {
        throw new Error('MathJax constructor called as function');
      }
      this.base = null;
      this.math_present = false;
      this.math_loaded = false;
      this.pending_cfi = null;
      this.hub = null;
    }

    MathJax.prototype.load_mathjax = function(user_config, is_windows) {
      var base, scale, script;
      if (this.base === null) {
        log('You must specify the path to the MathJax installation before trying to load MathJax');
        return null;
      }
      script = document.createElement('script');
      scale = is_windows ? 160 : 100;
      script.type = 'text/javascript';
      script.onerror = function(ev) {
        return console.log('Failed to load MathJax script: ' + ev.target.src);
      };
      base = this.base;
      if (base.substr(base.length - 1) !== '/') {
        base += '/';
      }
      script.src = base + 'MathJax.js';
      script.text = user_config + 'MathJax.Hub.signal.Interest(function (message) {if (String(message).match(/error/i)) {console.log(message)}});\nMathJax.Hub.Config({\n    positionToHash: false,\n    showMathMenu: false,\n    extensions: ["tex2jax.js", "asciimath2jax.js", "mml2jax.js"],\n    jax: ["input/TeX","input/MathML","input/AsciiMath","output/SVG"],\n    SVG : { linebreaks : { automatic : true }, scale: __scale__ },\n    TeX: {\n        extensions: ["AMSmath.js","AMSsymbols.js","noErrors.js","noUndefined.js"]\n    }\n        });\nMathJax.Hub.Startup.onload();\nMathJax.Hub.Register.StartupHook("End", window.mathjax.load_finished);\nwindow.mathjax.hub = MathJax.Hub'.replace('__scale__', scale);
      return document.head.appendChild(script);
    };

    MathJax.prototype.load_finished = function() {
      var callback, cfi, ref;
      log('MathJax load finished!');
      this.math_loaded = true;
      if (this.pending_cfi !== null) {
        ref = this.pending_cfi, cfi = ref[0], callback = ref[1];
        this.pending_cfi = null;
        return window.cfi.scroll_to(cfi, callback);
      }
    };

    MathJax.prototype.check_for_math = function(is_windows) {
      var c, i, len, ref, script, user_config;
      script = null;
      this.math_present = false;
      this.math_loaded = false;
      this.pending_cfi = null;
      user_config = '';
      ref = document.getElementsByTagName('script');
      for (i = 0, len = ref.length; i < len; i++) {
        c = ref[i];
        if (c.getAttribute('type') === 'text/x-mathjax-config') {
          if (c.text) {
            user_config += c.text;
          }
          script = c;
          c.parentNode.removeChild(c);
        }
      }
      if (script !== null || document.getElementsByTagName('math').length > 0) {
        this.math_present = true;
        this.remove_math_fallbacks();
        this.load_mathjax(user_config, is_windows);
      }
      return this.math_present;
    };

    MathJax.prototype.remove_math_fallbacks = function() {
      var c, found_math, i, len, non_math, ref, results, sw;
      ref = document.getElementsByTagName("epub:switch");
      results = [];
      for (i = 0, len = ref.length; i < len; i++) {
        sw = ref[i];
        non_math = [];
        found_math = false;
        c = sw.firstChild;
        while (c) {
          if (c.localName === 'epub:case') {
            if (c.getAttribute('required-namespace') === "http://www.w3.org/1998/Math/MathML") {
              found_math = c;
            } else {
              non_math.push(c);
            }
          } else if (c.localName === 'epub:default') {
            non_math.push(c);
          }
          c = c.nextSibling;
        }
        if (found_math) {
          results.push((function() {
            var j, len1, results1;
            results1 = [];
            for (j = 0, len1 = non_math.length; j < len1; j++) {
              c = non_math[j];
              results1.push(c.style.display = 'none');
            }
            return results1;
          })());
        } else {
          results.push(void 0);
        }
      }
      return results;
    };

    MathJax.prototype.after_resize = function() {
      if (!this.math_present || this.hub === null) {
        return;
      }
      return this.hub.Queue(["Rerender", this.hub]);
    };

    return MathJax;

  })();

  if (typeof window !== "undefined" && window !== null) {
    window.mathjax = new MathJax();
  }

}).call(this);
