/*
 * Decompiled with CFR 0.152.
 */
package org.elasticsearch.client;

import java.io.Closeable;
import java.io.IOException;
import java.net.URI;
import java.net.URISyntaxException;
import java.util.ArrayList;
import java.util.Collection;
import java.util.Collections;
import java.util.Comparator;
import java.util.HashMap;
import java.util.HashSet;
import java.util.Iterator;
import java.util.Locale;
import java.util.Map;
import java.util.Objects;
import java.util.Set;
import java.util.concurrent.ConcurrentHashMap;
import java.util.concurrent.ConcurrentMap;
import java.util.concurrent.CountDownLatch;
import java.util.concurrent.ExecutionException;
import java.util.concurrent.TimeUnit;
import java.util.concurrent.atomic.AtomicInteger;
import java.util.concurrent.atomic.AtomicReference;
import org.apache.commons.logging.Log;
import org.apache.commons.logging.LogFactory;
import org.apache.http.Header;
import org.apache.http.HttpEntity;
import org.apache.http.HttpHost;
import org.apache.http.HttpRequest;
import org.apache.http.HttpResponse;
import org.apache.http.auth.AuthScheme;
import org.apache.http.client.AuthCache;
import org.apache.http.client.methods.HttpEntityEnclosingRequestBase;
import org.apache.http.client.methods.HttpHead;
import org.apache.http.client.methods.HttpOptions;
import org.apache.http.client.methods.HttpPatch;
import org.apache.http.client.methods.HttpPost;
import org.apache.http.client.methods.HttpPut;
import org.apache.http.client.methods.HttpRequestBase;
import org.apache.http.client.methods.HttpTrace;
import org.apache.http.client.methods.HttpUriRequest;
import org.apache.http.client.protocol.HttpClientContext;
import org.apache.http.client.utils.URIBuilder;
import org.apache.http.concurrent.FutureCallback;
import org.apache.http.impl.auth.BasicScheme;
import org.apache.http.impl.client.BasicAuthCache;
import org.apache.http.impl.nio.client.CloseableHttpAsyncClient;
import org.apache.http.nio.client.methods.HttpAsyncMethods;
import org.apache.http.nio.protocol.HttpAsyncRequestProducer;
import org.apache.http.nio.protocol.HttpAsyncResponseConsumer;
import org.apache.http.protocol.HttpContext;
import org.elasticsearch.client.DeadHostState;
import org.elasticsearch.client.HttpAsyncResponseConsumerFactory;
import org.elasticsearch.client.HttpDeleteWithEntity;
import org.elasticsearch.client.HttpGetWithEntity;
import org.elasticsearch.client.RequestLogger;
import org.elasticsearch.client.Response;
import org.elasticsearch.client.ResponseException;
import org.elasticsearch.client.ResponseListener;
import org.elasticsearch.client.RestClientBuilder;

public class RestClient
implements Closeable {
    private static final Log logger = LogFactory.getLog(RestClient.class);
    private final CloseableHttpAsyncClient client;
    private final Header[] defaultHeaders;
    private final long maxRetryTimeoutMillis;
    private final String pathPrefix;
    private final AtomicInteger lastHostIndex = new AtomicInteger(0);
    private volatile HostTuple<Set<HttpHost>> hostTuple;
    private final ConcurrentMap<HttpHost, DeadHostState> blacklist = new ConcurrentHashMap<HttpHost, DeadHostState>();
    private final FailureListener failureListener;

    RestClient(CloseableHttpAsyncClient client, long maxRetryTimeoutMillis, Header[] defaultHeaders, HttpHost[] hosts, String pathPrefix, FailureListener failureListener) {
        this.client = client;
        this.maxRetryTimeoutMillis = maxRetryTimeoutMillis;
        this.defaultHeaders = defaultHeaders;
        this.failureListener = failureListener;
        this.pathPrefix = pathPrefix;
        this.setHosts(hosts);
    }

    public static RestClientBuilder builder(HttpHost ... hosts) {
        return new RestClientBuilder(hosts);
    }

    public synchronized void setHosts(HttpHost ... hosts) {
        if (hosts == null || hosts.length == 0) {
            throw new IllegalArgumentException("hosts must not be null nor empty");
        }
        HashSet<HttpHost> httpHosts = new HashSet<HttpHost>();
        BasicAuthCache authCache = new BasicAuthCache();
        for (HttpHost host : hosts) {
            Objects.requireNonNull(host, "host cannot be null");
            httpHosts.add(host);
            authCache.put(host, (AuthScheme)new BasicScheme());
        }
        this.hostTuple = new HostTuple(Collections.unmodifiableSet(httpHosts), (AuthCache)authCache);
        this.blacklist.clear();
    }

    public Response performRequest(String method, String endpoint, Header ... headers) throws IOException {
        return this.performRequest(method, endpoint, Collections.emptyMap(), (HttpEntity)null, headers);
    }

    public Response performRequest(String method, String endpoint, Map<String, String> params, Header ... headers) throws IOException {
        return this.performRequest(method, endpoint, params, (HttpEntity)null, headers);
    }

    public Response performRequest(String method, String endpoint, Map<String, String> params, HttpEntity entity, Header ... headers) throws IOException {
        return this.performRequest(method, endpoint, params, entity, HttpAsyncResponseConsumerFactory.DEFAULT, headers);
    }

    public Response performRequest(String method, String endpoint, Map<String, String> params, HttpEntity entity, HttpAsyncResponseConsumerFactory httpAsyncResponseConsumerFactory, Header ... headers) throws IOException {
        SyncResponseListener listener = new SyncResponseListener(this.maxRetryTimeoutMillis);
        this.performRequestAsync(method, endpoint, params, entity, httpAsyncResponseConsumerFactory, listener, headers);
        return listener.get();
    }

    public void performRequestAsync(String method, String endpoint, ResponseListener responseListener, Header ... headers) {
        this.performRequestAsync(method, endpoint, Collections.emptyMap(), null, responseListener, headers);
    }

    public void performRequestAsync(String method, String endpoint, Map<String, String> params, ResponseListener responseListener, Header ... headers) {
        this.performRequestAsync(method, endpoint, params, null, responseListener, headers);
    }

    public void performRequestAsync(String method, String endpoint, Map<String, String> params, HttpEntity entity, ResponseListener responseListener, Header ... headers) {
        this.performRequestAsync(method, endpoint, params, entity, HttpAsyncResponseConsumerFactory.DEFAULT, responseListener, headers);
    }

    public void performRequestAsync(String method, String endpoint, Map<String, String> params, HttpEntity entity, HttpAsyncResponseConsumerFactory httpAsyncResponseConsumerFactory, ResponseListener responseListener, Header ... headers) {
        try {
            Set<Object> ignoreErrorCodes;
            Objects.requireNonNull(params, "params must not be null");
            HashMap<String, String> requestParams = new HashMap<String, String>(params);
            String ignoreString = (String)requestParams.remove("ignore");
            if (ignoreString == null) {
                ignoreErrorCodes = "HEAD".equals(method) ? Collections.singleton(404) : Collections.emptySet();
            } else {
                String[] ignoresArray = ignoreString.split(",");
                ignoreErrorCodes = new HashSet();
                if ("HEAD".equals(method)) {
                    ignoreErrorCodes.add(404);
                }
                for (String ignoreCode : ignoresArray) {
                    try {
                        ignoreErrorCodes.add(Integer.valueOf(ignoreCode));
                    }
                    catch (NumberFormatException e) {
                        throw new IllegalArgumentException("ignore value should be a number, found [" + ignoreString + "] instead", e);
                    }
                }
            }
            URI uri = RestClient.buildUri(this.pathPrefix, endpoint, requestParams);
            HttpRequestBase request = RestClient.createHttpRequest(method, uri, entity);
            this.setHeaders((HttpRequest)request, headers);
            FailureTrackingResponseListener failureTrackingResponseListener = new FailureTrackingResponseListener(responseListener);
            long startTime = System.nanoTime();
            this.performRequestAsync(startTime, this.nextHost(), request, ignoreErrorCodes, httpAsyncResponseConsumerFactory, failureTrackingResponseListener);
        }
        catch (Exception e) {
            responseListener.onFailure(e);
        }
    }

    private void performRequestAsync(final long startTime, final HostTuple<Iterator<HttpHost>> hostTuple, final HttpRequestBase request, final Set<Integer> ignoreErrorCodes, final HttpAsyncResponseConsumerFactory httpAsyncResponseConsumerFactory, final FailureTrackingResponseListener listener) {
        final HttpHost host = (HttpHost)((Iterator)hostTuple.hosts).next();
        HttpAsyncRequestProducer requestProducer = HttpAsyncMethods.create((HttpHost)host, (HttpRequest)request);
        HttpAsyncResponseConsumer<HttpResponse> asyncResponseConsumer = httpAsyncResponseConsumerFactory.createHttpAsyncResponseConsumer();
        HttpClientContext context = HttpClientContext.create();
        context.setAuthCache(hostTuple.authCache);
        this.client.execute(requestProducer, asyncResponseConsumer, (HttpContext)context, (FutureCallback)new FutureCallback<HttpResponse>(){

            public void completed(HttpResponse httpResponse) {
                try {
                    RequestLogger.logResponse(logger, (HttpUriRequest)request, host, httpResponse);
                    int statusCode = httpResponse.getStatusLine().getStatusCode();
                    Response response = new Response(request.getRequestLine(), host, httpResponse);
                    if (RestClient.isSuccessfulResponse(statusCode) || ignoreErrorCodes.contains(response.getStatusLine().getStatusCode())) {
                        RestClient.this.onResponse(host);
                        listener.onSuccess(response);
                    } else {
                        ResponseException responseException = new ResponseException(response);
                        if (RestClient.isRetryStatus(statusCode)) {
                            RestClient.this.onFailure(host);
                            this.retryIfPossible(responseException);
                        } else {
                            RestClient.this.onResponse(host);
                            listener.onDefinitiveFailure(responseException);
                        }
                    }
                }
                catch (Exception e) {
                    listener.onDefinitiveFailure(e);
                }
            }

            public void failed(Exception failure) {
                try {
                    RequestLogger.logFailedRequest(logger, (HttpUriRequest)request, host, failure);
                    RestClient.this.onFailure(host);
                    this.retryIfPossible(failure);
                }
                catch (Exception e) {
                    listener.onDefinitiveFailure(e);
                }
            }

            private void retryIfPossible(Exception exception) {
                if (((Iterator)hostTuple.hosts).hasNext()) {
                    long timeElapsedMillis = TimeUnit.NANOSECONDS.toMillis(System.nanoTime() - startTime);
                    long timeout = RestClient.this.maxRetryTimeoutMillis - timeElapsedMillis;
                    if (timeout <= 0L) {
                        IOException retryTimeoutException = new IOException("request retries exceeded max retry timeout [" + RestClient.this.maxRetryTimeoutMillis + "]");
                        listener.onDefinitiveFailure(retryTimeoutException);
                    } else {
                        listener.trackFailure(exception);
                        request.reset();
                        RestClient.this.performRequestAsync(startTime, hostTuple, request, ignoreErrorCodes, httpAsyncResponseConsumerFactory, listener);
                    }
                } else {
                    listener.onDefinitiveFailure(exception);
                }
            }

            public void cancelled() {
                listener.onDefinitiveFailure(new ExecutionException("request was cancelled", null));
            }
        });
    }

    private void setHeaders(HttpRequest httpRequest, Header[] requestHeaders) {
        Objects.requireNonNull(requestHeaders, "request headers must not be null");
        HashSet<String> requestNames = new HashSet<String>(requestHeaders.length);
        for (Header requestHeader : requestHeaders) {
            Objects.requireNonNull(requestHeader, "request header must not be null");
            httpRequest.addHeader(requestHeader);
            requestNames.add(requestHeader.getName());
        }
        for (Header defaultHeader : this.defaultHeaders) {
            if (requestNames.contains(defaultHeader.getName())) continue;
            httpRequest.addHeader(defaultHeader);
        }
    }

    private HostTuple<Iterator<HttpHost>> nextHost() {
        HostTuple<Set<HttpHost>> hostTuple = this.hostTuple;
        Collection<Object> nextHosts = Collections.emptySet();
        do {
            HashSet filteredHosts = new HashSet((Collection)hostTuple.hosts);
            for (Map.Entry entry : this.blacklist.entrySet()) {
                if (System.nanoTime() - ((DeadHostState)entry.getValue()).getDeadUntilNanos() >= 0L) continue;
                filteredHosts.remove(entry.getKey());
            }
            if (filteredHosts.isEmpty()) {
                ArrayList sortedHosts = new ArrayList(this.blacklist.entrySet());
                if (sortedHosts.size() <= 0) continue;
                Collections.sort(sortedHosts, new Comparator<Map.Entry<HttpHost, DeadHostState>>(){

                    @Override
                    public int compare(Map.Entry<HttpHost, DeadHostState> o1, Map.Entry<HttpHost, DeadHostState> o2) {
                        return Long.compare(o1.getValue().getDeadUntilNanos(), o2.getValue().getDeadUntilNanos());
                    }
                });
                HttpHost deadHost = (HttpHost)((Map.Entry)sortedHosts.get(0)).getKey();
                logger.trace((Object)("resurrecting host [" + deadHost + "]"));
                nextHosts = Collections.singleton(deadHost);
                continue;
            }
            ArrayList rotatedHosts = new ArrayList(filteredHosts);
            Collections.rotate(rotatedHosts, rotatedHosts.size() - this.lastHostIndex.getAndIncrement());
            nextHosts = rotatedHosts;
        } while (nextHosts.isEmpty());
        return new HostTuple<Iterator<HttpHost>>(nextHosts.iterator(), hostTuple.authCache);
    }

    private void onResponse(HttpHost host) {
        DeadHostState removedHost = (DeadHostState)this.blacklist.remove(host);
        if (logger.isDebugEnabled() && removedHost != null) {
            logger.debug((Object)("removed host [" + host + "] from blacklist"));
        }
    }

    private void onFailure(HttpHost host) throws IOException {
        block1: {
            DeadHostState previousDeadHostState;
            do {
                if ((previousDeadHostState = this.blacklist.putIfAbsent(host, DeadHostState.INITIAL_DEAD_STATE)) != null) continue;
                logger.debug((Object)("added host [" + host + "] to blacklist"));
                break block1;
            } while (!this.blacklist.replace(host, previousDeadHostState, new DeadHostState(previousDeadHostState)));
            logger.debug((Object)("updated host [" + host + "] already in blacklist"));
        }
        this.failureListener.onFailure(host);
    }

    @Override
    public void close() throws IOException {
        this.client.close();
    }

    private static boolean isSuccessfulResponse(int statusCode) {
        return statusCode < 300;
    }

    private static boolean isRetryStatus(int statusCode) {
        switch (statusCode) {
            case 502: 
            case 503: 
            case 504: {
                return true;
            }
        }
        return false;
    }

    private static Exception addSuppressedException(Exception suppressedException, Exception currentException) {
        if (suppressedException != null) {
            currentException.addSuppressed(suppressedException);
        }
        return currentException;
    }

    private static HttpRequestBase createHttpRequest(String method, URI uri, HttpEntity entity) {
        switch (method.toUpperCase(Locale.ROOT)) {
            case "DELETE": {
                return RestClient.addRequestBody((HttpRequestBase)new HttpDeleteWithEntity(uri), entity);
            }
            case "GET": {
                return RestClient.addRequestBody((HttpRequestBase)new HttpGetWithEntity(uri), entity);
            }
            case "HEAD": {
                return RestClient.addRequestBody((HttpRequestBase)new HttpHead(uri), entity);
            }
            case "OPTIONS": {
                return RestClient.addRequestBody((HttpRequestBase)new HttpOptions(uri), entity);
            }
            case "PATCH": {
                return RestClient.addRequestBody((HttpRequestBase)new HttpPatch(uri), entity);
            }
            case "POST": {
                HttpPost httpPost = new HttpPost(uri);
                RestClient.addRequestBody((HttpRequestBase)httpPost, entity);
                return httpPost;
            }
            case "PUT": {
                return RestClient.addRequestBody((HttpRequestBase)new HttpPut(uri), entity);
            }
            case "TRACE": {
                return RestClient.addRequestBody((HttpRequestBase)new HttpTrace(uri), entity);
            }
        }
        throw new UnsupportedOperationException("http method not supported: " + method);
    }

    private static HttpRequestBase addRequestBody(HttpRequestBase httpRequest, HttpEntity entity) {
        if (entity != null) {
            if (httpRequest instanceof HttpEntityEnclosingRequestBase) {
                ((HttpEntityEnclosingRequestBase)httpRequest).setEntity(entity);
            } else {
                throw new UnsupportedOperationException(httpRequest.getMethod() + " with body is not supported");
            }
        }
        return httpRequest;
    }

    static URI buildUri(String pathPrefix, String path, Map<String, String> params) {
        Objects.requireNonNull(path, "path must not be null");
        try {
            String fullPath = pathPrefix != null ? (path.startsWith("/") ? pathPrefix + path : pathPrefix + "/" + path) : path;
            URIBuilder uriBuilder = new URIBuilder(fullPath);
            for (Map.Entry<String, String> param : params.entrySet()) {
                uriBuilder.addParameter(param.getKey(), param.getValue());
            }
            return uriBuilder.build();
        }
        catch (URISyntaxException e) {
            throw new IllegalArgumentException(e.getMessage(), e);
        }
    }

    private static class HostTuple<T> {
        final T hosts;
        final AuthCache authCache;

        HostTuple(T hosts, AuthCache authCache) {
            this.hosts = hosts;
            this.authCache = authCache;
        }
    }

    public static class FailureListener {
        public void onFailure(HttpHost host) {
        }
    }

    static class SyncResponseListener
    implements ResponseListener {
        private final CountDownLatch latch = new CountDownLatch(1);
        private final AtomicReference<Response> response = new AtomicReference();
        private final AtomicReference<Exception> exception = new AtomicReference();
        private final long timeout;

        SyncResponseListener(long timeout) {
            assert (timeout > 0L);
            this.timeout = timeout;
        }

        @Override
        public void onSuccess(Response response) {
            Objects.requireNonNull(response, "response must not be null");
            boolean wasResponseNull = this.response.compareAndSet(null, response);
            if (!wasResponseNull) {
                throw new IllegalStateException("response is already set");
            }
            this.latch.countDown();
        }

        @Override
        public void onFailure(Exception exception) {
            Objects.requireNonNull(exception, "exception must not be null");
            boolean wasExceptionNull = this.exception.compareAndSet(null, exception);
            if (!wasExceptionNull) {
                throw new IllegalStateException("exception is already set");
            }
            this.latch.countDown();
        }

        Response get() throws IOException {
            try {
                if (!this.latch.await(this.timeout, TimeUnit.MILLISECONDS)) {
                    throw new IOException("listener timeout after waiting for [" + this.timeout + "] ms");
                }
            }
            catch (InterruptedException e) {
                throw new RuntimeException("thread waiting for the response was interrupted", e);
            }
            Exception exception = this.exception.get();
            Response response = this.response.get();
            if (exception != null) {
                if (response != null) {
                    IllegalStateException e = new IllegalStateException("response and exception are unexpectedly set at the same time");
                    e.addSuppressed(exception);
                    throw e;
                }
                if (exception instanceof IOException) {
                    throw (IOException)exception;
                }
                if (exception instanceof RuntimeException) {
                    throw (RuntimeException)exception;
                }
                throw new RuntimeException("error while performing request", exception);
            }
            if (response == null) {
                throw new IllegalStateException("response not set and no exception caught either");
            }
            return response;
        }
    }

    static class FailureTrackingResponseListener {
        private final ResponseListener responseListener;
        private volatile Exception exception;

        FailureTrackingResponseListener(ResponseListener responseListener) {
            this.responseListener = responseListener;
        }

        void onSuccess(Response response) {
            this.responseListener.onSuccess(response);
        }

        void onDefinitiveFailure(Exception exception) {
            this.trackFailure(exception);
            this.responseListener.onFailure(this.exception);
        }

        void trackFailure(Exception exception) {
            this.exception = RestClient.addSuppressedException(this.exception, exception);
        }
    }
}

