#!/usr/bin/env python3
# vim: ft=python fileencoding=utf-8 sts=4 sw=4 et:

# Copyright 2014-2017 Claude (longneck) <longneck@scratchbook.ch>
# Copyright 2014-2017 Florian Bruhin (The Compiler) <mail@qutebrowser.org>

# This file is part of qutebrowser.
#
# qutebrowser is free software: you can redistribute it and/or modify
# it under the terms of the GNU General Public License as published by
# the Free Software Foundation, either version 3 of the License, or
# (at your option) any later version.
#
# qutebrowser is distributed in the hope that it will be useful,
# but WITHOUT ANY WARRANTY; without even the implied warranty of
# MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
# GNU General Public License for more details.
#
# You should have received a copy of the GNU General Public License
# along with qutebrowser.  If not, see <http://www.gnu.org/licenses/>.


"""Tool to import data from other browsers.

Currently only importing bookmarks from Netscape Bookmark files is supported.
"""


import argparse


def main():
    args = get_args()
    if args.browser in ['chromium', 'firefox', 'ie']:
        import_netscape_bookmarks(args.bookmarks, args.bookmark_format)


def get_args():
    """Get the argparse parser."""
    parser = argparse.ArgumentParser(
        epilog="To import bookmarks from Chromium, Firefox or IE, "
               "export them to HTML in your browsers bookmark manager. "
               "By default, this script will output in a quickmarks format.")
    parser.add_argument('browser', help="Which browser? (chromium, firefox)",
                        choices=['chromium', 'firefox', 'ie'],
                        metavar='browser')
    parser.add_argument('-b', help="Output in bookmark format.",
                        dest='bookmark_format', action='store_true',
                        default=False, required=False)
    parser.add_argument('bookmarks', help="Bookmarks file (html format)")
    args = parser.parse_args()
    return args


def import_netscape_bookmarks(bookmarks_file, is_bookmark_format):
    """Import bookmarks from a NETSCAPE-Bookmark-file v1.

    Generated by Chromium, Firefox, IE and possibly more browsers
    """
    import bs4
    with open(bookmarks_file, encoding='utf-8') as f:
        soup = bs4.BeautifulSoup(f, 'html.parser')

    html_tags = soup.findAll('a')
    if is_bookmark_format:
        output_template = '{tag[href]} {tag.string}'
    else:
        output_template = '{tag.string} {tag[href]}'

    bookmarks = []
    for tag in html_tags:
        if tag['href'] not in bookmarks:
            bookmarks.append(output_template.format(tag=tag))

    for bookmark in bookmarks:
        print(bookmark)


if __name__ == '__main__':
    main()
