/*
 * Decompiled with CFR 0.152.
 */
package org.elasticsearch.monitor.jvm;

import java.util.Collections;
import java.util.HashMap;
import java.util.Locale;
import java.util.Map;
import java.util.Objects;
import java.util.concurrent.TimeUnit;
import java.util.function.BiFunction;
import org.apache.logging.log4j.Logger;
import org.elasticsearch.common.component.AbstractLifecycleComponent;
import org.elasticsearch.common.settings.Setting;
import org.elasticsearch.common.settings.Settings;
import org.elasticsearch.common.unit.ByteSizeValue;
import org.elasticsearch.common.unit.TimeValue;
import org.elasticsearch.monitor.jvm.JvmInfo;
import org.elasticsearch.monitor.jvm.JvmStats;
import org.elasticsearch.threadpool.Scheduler;
import org.elasticsearch.threadpool.ThreadPool;

public class JvmGcMonitorService
extends AbstractLifecycleComponent {
    private final ThreadPool threadPool;
    private final boolean enabled;
    private final TimeValue interval;
    private final Map<String, GcThreshold> gcThresholds;
    private final GcOverheadThreshold gcOverheadThreshold;
    private volatile Scheduler.Cancellable scheduledFuture;
    public static final Setting<Boolean> ENABLED_SETTING = Setting.boolSetting("monitor.jvm.gc.enabled", true, Setting.Property.NodeScope);
    public static final Setting<TimeValue> REFRESH_INTERVAL_SETTING = Setting.timeSetting("monitor.jvm.gc.refresh_interval", TimeValue.timeValueSeconds(1L), TimeValue.timeValueSeconds(1L), Setting.Property.NodeScope);
    private static String GC_COLLECTOR_PREFIX = "monitor.jvm.gc.collector.";
    public static final Setting<Settings> GC_SETTING = Setting.groupSetting(GC_COLLECTOR_PREFIX, Setting.Property.NodeScope);
    public static final Setting<Integer> GC_OVERHEAD_WARN_SETTING = Setting.intSetting("monitor.jvm.gc.overhead.warn", 50, 0, 100, Setting.Property.NodeScope);
    public static final Setting<Integer> GC_OVERHEAD_INFO_SETTING = Setting.intSetting("monitor.jvm.gc.overhead.info", 25, 0, 100, Setting.Property.NodeScope);
    public static final Setting<Integer> GC_OVERHEAD_DEBUG_SETTING = Setting.intSetting("monitor.jvm.gc.overhead.debug", 10, 0, 100, Setting.Property.NodeScope);
    private static final String SLOW_GC_LOG_MESSAGE = "[gc][{}][{}][{}] duration [{}], collections [{}]/[{}], total [{}]/[{}], memory [{}]->[{}]/[{}], all_pools {}";
    private static final String OVERHEAD_LOG_MESSAGE = "[gc][{}] overhead, spent [{}] collecting in the last [{}]";

    public JvmGcMonitorService(Settings settings, ThreadPool threadPool) {
        super(settings);
        String message;
        this.threadPool = threadPool;
        this.enabled = ENABLED_SETTING.get(settings);
        this.interval = REFRESH_INTERVAL_SETTING.get(settings);
        HashMap<String, GcThreshold> gcThresholds = new HashMap<String, GcThreshold>();
        Map<String, Settings> gcThresholdGroups = GC_SETTING.get(settings).getAsGroups();
        for (Map.Entry<String, Settings> entry : gcThresholdGroups.entrySet()) {
            String name = entry.getKey();
            TimeValue warn = JvmGcMonitorService.getValidThreshold(entry.getValue(), entry.getKey(), "warn");
            TimeValue info = JvmGcMonitorService.getValidThreshold(entry.getValue(), entry.getKey(), "info");
            TimeValue debug = JvmGcMonitorService.getValidThreshold(entry.getValue(), entry.getKey(), "debug");
            gcThresholds.put(name, new GcThreshold(name, warn.millis(), info.millis(), debug.millis()));
        }
        gcThresholds.putIfAbsent("young", new GcThreshold("young", 1000L, 700L, 400L));
        gcThresholds.putIfAbsent("old", new GcThreshold("old", 10000L, 5000L, 2000L));
        gcThresholds.putIfAbsent("default", new GcThreshold("default", 10000L, 5000L, 2000L));
        this.gcThresholds = Collections.unmodifiableMap(gcThresholds);
        if (GC_OVERHEAD_WARN_SETTING.get(settings) <= GC_OVERHEAD_INFO_SETTING.get(settings)) {
            message = String.format(Locale.ROOT, "[%s] must be greater than [%s] [%d] but was [%d]", GC_OVERHEAD_WARN_SETTING.getKey(), GC_OVERHEAD_INFO_SETTING.getKey(), GC_OVERHEAD_INFO_SETTING.get(settings), GC_OVERHEAD_WARN_SETTING.get(settings));
            throw new IllegalArgumentException(message);
        }
        if (GC_OVERHEAD_INFO_SETTING.get(settings) <= GC_OVERHEAD_DEBUG_SETTING.get(settings)) {
            message = String.format(Locale.ROOT, "[%s] must be greater than [%s] [%d] but was [%d]", GC_OVERHEAD_INFO_SETTING.getKey(), GC_OVERHEAD_DEBUG_SETTING.getKey(), GC_OVERHEAD_DEBUG_SETTING.get(settings), GC_OVERHEAD_INFO_SETTING.get(settings));
            throw new IllegalArgumentException(message);
        }
        this.gcOverheadThreshold = new GcOverheadThreshold(GC_OVERHEAD_WARN_SETTING.get(settings), GC_OVERHEAD_INFO_SETTING.get(settings), GC_OVERHEAD_DEBUG_SETTING.get(settings));
        this.logger.debug("enabled [{}], interval [{}], gc_threshold [{}], overhead [{}, {}, {}]", (Object)this.enabled, (Object)this.interval, this.gcThresholds, (Object)this.gcOverheadThreshold.warnThreshold, (Object)this.gcOverheadThreshold.infoThreshold, (Object)this.gcOverheadThreshold.debugThreshold);
    }

    private static TimeValue getValidThreshold(Settings settings, String key, String level) {
        TimeValue threshold = settings.getAsTime(level, null);
        if (threshold == null) {
            throw new IllegalArgumentException("missing gc_threshold for [" + JvmGcMonitorService.getThresholdName(key, level) + "]");
        }
        if (threshold.nanos() <= 0L) {
            throw new IllegalArgumentException("invalid gc_threshold [" + threshold + "] for [" + JvmGcMonitorService.getThresholdName(key, level) + "]");
        }
        return threshold;
    }

    private static String getThresholdName(String key, String level) {
        return GC_COLLECTOR_PREFIX + key + "." + level;
    }

    @Override
    protected void doStart() {
        if (!this.enabled) {
            return;
        }
        this.scheduledFuture = this.threadPool.scheduleWithFixedDelay(new JvmMonitor(this.gcThresholds, this.gcOverheadThreshold){

            @Override
            void onMonitorFailure(Exception e) {
                JvmGcMonitorService.this.logger.debug("failed to monitor", (Throwable)e);
            }

            @Override
            void onSlowGc(JvmMonitor.Threshold threshold, long seq, JvmMonitor.SlowGcEvent slowGcEvent) {
                JvmGcMonitorService.logSlowGc(JvmGcMonitorService.this.logger, threshold, seq, slowGcEvent, JvmGcMonitorService::buildPools);
            }

            @Override
            void onGcOverhead(JvmMonitor.Threshold threshold, long current, long elapsed, long seq) {
                JvmGcMonitorService.logGcOverhead(JvmGcMonitorService.this.logger, threshold, current, elapsed, seq);
            }
        }, this.interval, "same");
    }

    static void logSlowGc(Logger logger, JvmMonitor.Threshold threshold, long seq, JvmMonitor.SlowGcEvent slowGcEvent, BiFunction<JvmStats, JvmStats, String> pools) {
        String name = slowGcEvent.currentGc.getName();
        long elapsed = slowGcEvent.elapsed;
        long totalGcCollectionCount = slowGcEvent.currentGc.getCollectionCount();
        long currentGcCollectionCount = slowGcEvent.collectionCount;
        TimeValue totalGcCollectionTime = slowGcEvent.currentGc.getCollectionTime();
        TimeValue currentGcCollectionTime = slowGcEvent.collectionTime;
        JvmStats lastJvmStats = slowGcEvent.lastJvmStats;
        JvmStats currentJvmStats = slowGcEvent.currentJvmStats;
        ByteSizeValue maxHeapUsed = slowGcEvent.maxHeapUsed;
        switch (threshold) {
            case WARN: {
                if (!logger.isWarnEnabled()) break;
                logger.warn(SLOW_GC_LOG_MESSAGE, new Object[]{name, seq, totalGcCollectionCount, currentGcCollectionTime, currentGcCollectionCount, TimeValue.timeValueMillis(elapsed), currentGcCollectionTime, totalGcCollectionTime, lastJvmStats.getMem().getHeapUsed(), currentJvmStats.getMem().getHeapUsed(), maxHeapUsed, pools.apply(lastJvmStats, currentJvmStats)});
                break;
            }
            case INFO: {
                if (!logger.isInfoEnabled()) break;
                logger.info(SLOW_GC_LOG_MESSAGE, new Object[]{name, seq, totalGcCollectionCount, currentGcCollectionTime, currentGcCollectionCount, TimeValue.timeValueMillis(elapsed), currentGcCollectionTime, totalGcCollectionTime, lastJvmStats.getMem().getHeapUsed(), currentJvmStats.getMem().getHeapUsed(), maxHeapUsed, pools.apply(lastJvmStats, currentJvmStats)});
                break;
            }
            case DEBUG: {
                if (!logger.isDebugEnabled()) break;
                logger.debug(SLOW_GC_LOG_MESSAGE, new Object[]{name, seq, totalGcCollectionCount, currentGcCollectionTime, currentGcCollectionCount, TimeValue.timeValueMillis(elapsed), currentGcCollectionTime, totalGcCollectionTime, lastJvmStats.getMem().getHeapUsed(), currentJvmStats.getMem().getHeapUsed(), maxHeapUsed, pools.apply(lastJvmStats, currentJvmStats)});
            }
        }
    }

    static String buildPools(JvmStats last, JvmStats current) {
        StringBuilder sb = new StringBuilder();
        for (JvmStats.MemoryPool currentPool : current.getMem()) {
            JvmStats.MemoryPool prevPool = null;
            for (JvmStats.MemoryPool pool : last.getMem()) {
                if (!pool.getName().equals(currentPool.getName())) continue;
                prevPool = pool;
                break;
            }
            sb.append("{[").append(currentPool.getName()).append("] [").append(prevPool == null ? "?" : prevPool.getUsed()).append("]->[").append(currentPool.getUsed()).append("]/[").append(currentPool.getMax()).append("]}");
        }
        return sb.toString();
    }

    static void logGcOverhead(Logger logger, JvmMonitor.Threshold threshold, long current, long elapsed, long seq) {
        switch (threshold) {
            case WARN: {
                if (!logger.isWarnEnabled()) break;
                logger.warn(OVERHEAD_LOG_MESSAGE, (Object)seq, (Object)TimeValue.timeValueMillis(current), (Object)TimeValue.timeValueMillis(elapsed));
                break;
            }
            case INFO: {
                if (!logger.isInfoEnabled()) break;
                logger.info(OVERHEAD_LOG_MESSAGE, (Object)seq, (Object)TimeValue.timeValueMillis(current), (Object)TimeValue.timeValueMillis(elapsed));
                break;
            }
            case DEBUG: {
                if (!logger.isDebugEnabled()) break;
                logger.debug(OVERHEAD_LOG_MESSAGE, (Object)seq, (Object)TimeValue.timeValueMillis(current), (Object)TimeValue.timeValueMillis(elapsed));
            }
        }
    }

    @Override
    protected void doStop() {
        if (!this.enabled) {
            return;
        }
        this.scheduledFuture.cancel();
    }

    @Override
    protected void doClose() {
    }

    static abstract class JvmMonitor
    implements Runnable {
        private long lastTime = this.now();
        private JvmStats lastJvmStats = this.jvmStats();
        private long seq = 0L;
        private final Map<String, GcThreshold> gcThresholds;
        final GcOverheadThreshold gcOverheadThreshold;

        JvmMonitor(Map<String, GcThreshold> gcThresholds, GcOverheadThreshold gcOverheadThreshold) {
            this.gcThresholds = Objects.requireNonNull(gcThresholds);
            this.gcOverheadThreshold = Objects.requireNonNull(gcOverheadThreshold);
        }

        @Override
        public void run() {
            try {
                this.monitorGc();
            }
            catch (Exception e) {
                this.onMonitorFailure(e);
            }
        }

        abstract void onMonitorFailure(Exception var1);

        synchronized void monitorGc() {
            ++this.seq;
            long currentTime = this.now();
            JvmStats currentJvmStats = this.jvmStats();
            long elapsed = TimeUnit.NANOSECONDS.toMillis(currentTime - this.lastTime);
            this.monitorSlowGc(currentJvmStats, elapsed);
            this.monitorGcOverhead(currentJvmStats, elapsed);
            this.lastTime = currentTime;
            this.lastJvmStats = currentJvmStats;
        }

        final void monitorSlowGc(JvmStats currentJvmStats, long elapsed) {
            for (int i = 0; i < currentJvmStats.getGc().getCollectors().length; ++i) {
                long collectionTime;
                JvmStats.GarbageCollector gc = currentJvmStats.getGc().getCollectors()[i];
                JvmStats.GarbageCollector prevGc = this.lastJvmStats.getGc().getCollectors()[i];
                long collections = gc.getCollectionCount() - prevGc.getCollectionCount();
                if (collections == 0L || (collectionTime = gc.getCollectionTime().millis() - prevGc.getCollectionTime().millis()) == 0L) continue;
                GcThreshold gcThreshold = this.gcThresholds.get(gc.getName());
                if (gcThreshold == null) {
                    gcThreshold = this.gcThresholds.get("default");
                }
                long avgCollectionTime = collectionTime / collections;
                Threshold threshold = null;
                if (avgCollectionTime > gcThreshold.warnThreshold) {
                    threshold = Threshold.WARN;
                } else if (avgCollectionTime > gcThreshold.infoThreshold) {
                    threshold = Threshold.INFO;
                } else if (avgCollectionTime > gcThreshold.debugThreshold) {
                    threshold = Threshold.DEBUG;
                }
                if (threshold == null) continue;
                this.onSlowGc(threshold, this.seq, new SlowGcEvent(gc, collections, TimeValue.timeValueMillis(collectionTime), elapsed, this.lastJvmStats, currentJvmStats, JvmInfo.jvmInfo().getMem().getHeapMax()));
            }
        }

        final void monitorGcOverhead(JvmStats currentJvmStats, long elapsed) {
            long current = 0L;
            for (int i = 0; i < currentJvmStats.getGc().getCollectors().length; ++i) {
                JvmStats.GarbageCollector gc = currentJvmStats.getGc().getCollectors()[i];
                JvmStats.GarbageCollector prevGc = this.lastJvmStats.getGc().getCollectors()[i];
                current += gc.getCollectionTime().millis() - prevGc.getCollectionTime().millis();
            }
            this.checkGcOverhead(current, elapsed, this.seq);
        }

        void checkGcOverhead(long current, long elapsed, long seq) {
            int fraction = (int)((double)(100L * current) / (double)elapsed);
            Threshold overheadThreshold = null;
            if (fraction >= this.gcOverheadThreshold.warnThreshold) {
                overheadThreshold = Threshold.WARN;
            } else if (fraction >= this.gcOverheadThreshold.infoThreshold) {
                overheadThreshold = Threshold.INFO;
            } else if (fraction >= this.gcOverheadThreshold.debugThreshold) {
                overheadThreshold = Threshold.DEBUG;
            }
            if (overheadThreshold != null) {
                this.onGcOverhead(overheadThreshold, current, elapsed, seq);
            }
        }

        JvmStats jvmStats() {
            return JvmStats.jvmStats();
        }

        long now() {
            return System.nanoTime();
        }

        abstract void onSlowGc(Threshold var1, long var2, SlowGcEvent var4);

        abstract void onGcOverhead(Threshold var1, long var2, long var4, long var6);

        static class SlowGcEvent {
            final JvmStats.GarbageCollector currentGc;
            final long collectionCount;
            final TimeValue collectionTime;
            final long elapsed;
            final JvmStats lastJvmStats;
            final JvmStats currentJvmStats;
            final ByteSizeValue maxHeapUsed;

            SlowGcEvent(JvmStats.GarbageCollector currentGc, long collectionCount, TimeValue collectionTime, long elapsed, JvmStats lastJvmStats, JvmStats currentJvmStats, ByteSizeValue maxHeapUsed) {
                this.currentGc = currentGc;
                this.collectionCount = collectionCount;
                this.collectionTime = collectionTime;
                this.elapsed = elapsed;
                this.lastJvmStats = lastJvmStats;
                this.currentJvmStats = currentJvmStats;
                this.maxHeapUsed = maxHeapUsed;
            }
        }

        static enum Threshold {
            DEBUG,
            INFO,
            WARN;

        }
    }

    static class GcThreshold {
        public final String name;
        public final long warnThreshold;
        public final long infoThreshold;
        public final long debugThreshold;

        GcThreshold(String name, long warnThreshold, long infoThreshold, long debugThreshold) {
            this.name = name;
            this.warnThreshold = warnThreshold;
            this.infoThreshold = infoThreshold;
            this.debugThreshold = debugThreshold;
        }

        public String toString() {
            return "GcThreshold{name='" + this.name + '\'' + ", warnThreshold=" + this.warnThreshold + ", infoThreshold=" + this.infoThreshold + ", debugThreshold=" + this.debugThreshold + '}';
        }
    }

    static class GcOverheadThreshold {
        final int warnThreshold;
        final int infoThreshold;
        final int debugThreshold;

        GcOverheadThreshold(int warnThreshold, int infoThreshold, int debugThreshold) {
            this.warnThreshold = warnThreshold;
            this.infoThreshold = infoThreshold;
            this.debugThreshold = debugThreshold;
        }
    }
}

