/**
 * Nextcloud - contacts
 *
 * This file is licensed under the Affero General Public License version 3 or
 * later. See the COPYING file.
 *
 * @author Hendrik Leppelsack <hendrik@leppelsack.de>
 * @copyright Hendrik Leppelsack 2015
 */

angular.module('contactsApp', ['uuid4', 'angular-cache', 'ngRoute', 'ui.bootstrap', 'ui.select', 'ngSanitize', 'angular-click-outside', 'ngclipboard'])
.config(['$routeProvider', function($routeProvider) {

	$routeProvider.when('/:gid', {
		template: '<contactdetails></contactdetails>'
	});

	$routeProvider.when('/contact/:uid', {
		redirectTo: function(parameters) {
			return '/' + t('contacts', 'All contacts') + '/' + parameters.uid;
		}
	});

	$routeProvider.when('/:gid/:uid', {
		template: '<contactdetails></contactdetails>'
	});

	$routeProvider.otherwise('/' + t('contacts', 'All contacts'));

}]);

angular.module('contactsApp')
.directive('datepicker', ['$timeout', function($timeout) {
	var loadDatepicker = function (scope, element, attrs, ngModelCtrl) {
		$timeout(function() {
			element.datepicker({
				dateFormat:'yy-mm-dd',
				minDate: null,
				maxDate: null,
				constrainInput: false,
				onSelect:function (date, dp) {
					if (dp.selectedYear < 1000) {
						date = '0' + date;
					}
					if (dp.selectedYear < 100) {
						date = '0' + date;
					}
					if (dp.selectedYear < 10) {
						date = '0' + date;
					}
					ngModelCtrl.$setViewValue(date);
					scope.$apply();
				}
			});
		});
	};
	return {
		restrict: 'A',
		require : 'ngModel',
		transclude: true,
		link : loadDatepicker
	};
}]);

angular.module('contactsApp')
.directive('focusExpression', ['$timeout', function ($timeout) {
	return {
		restrict: 'A',
		link: {
			post: function postLink(scope, element, attrs) {
				scope.$watch(attrs.focusExpression, function () {
					if (attrs.focusExpression) {
						if (scope.$eval(attrs.focusExpression)) {
							$timeout(function () {
								if (element.is('input')) {
									element.focus();
								} else {
									element.find('input').focus();
								}
							}, 100); //need some delay to work with ng-disabled
						}
					}
				});
			}
		}
	};
}]);

angular.module('contactsApp')
.directive('inputresize', function() {
	return {
		restrict: 'A',
		link : function (scope, element) {
			var elInput = element.val();
			element.bind('keydown keyup load focus', function() {
				elInput = element.val();
				// If set to 0, the min-width css data is ignored
				var length = elInput.length > 1 ? elInput.length : 1;
				element.attr('size', length);
			});
		}
	};
});

angular.module('contactsApp')
.directive('selectExpression', ['$timeout', function ($timeout) {
	return {
		restrict: 'A',
		link: {
			post: function postLink(scope, element, attrs) {
				scope.$watch(attrs.selectExpression, function () {
					if (attrs.selectExpression) {
						if (scope.$eval(attrs.selectExpression)) {
							$timeout(function () {
								if (element.is('input')) {
									element.select();
								} else {
									element.find('input').select();
								}
							}, 100); //need some delay to work with ng-disabled
						}
					}
				});
			}
		}
	};
}]);

angular.module('contactsApp')
.controller('addressbookCtrl', ['$scope', 'AddressBookService', function($scope, AddressBookService) {
	var ctrl = this;

	ctrl.t = {
		download: t('contacts', 'Download'),
		copyURL: t('contacts', 'Copy link'),
		clickToCopy: t('contacts', 'Click to copy the link to your clipboard'),
		shareAddressbook: t('contacts', 'Toggle sharing'),
		deleteAddressbook: t('contacts', 'Delete'),
		renameAddressbook: t('contacts', 'Rename'),
		shareInputPlaceHolder: t('contacts', 'Share with users or groups'),
		delete: t('contacts', 'Delete'),
		canEdit: t('contacts', 'can edit'),
		close: t('contacts', 'Close')
	};

	ctrl.editing = false;

	ctrl.tooltipIsOpen = false;
	ctrl.tooltipTitle = ctrl.t.clickToCopy;
	ctrl.showInputUrl = false;

	ctrl.clipboardSuccess = function() {
		ctrl.tooltipIsOpen = true;
		ctrl.tooltipTitle = t('core', 'Copied!');
		_.delay(function() {
			ctrl.tooltipIsOpen = false;
			ctrl.tooltipTitle = ctrl.t.clickToCopy;
		}, 3000);
	};

	ctrl.clipboardError = function() {
		ctrl.showInputUrl = true;
		if (/iPhone|iPad/i.test(navigator.userAgent)) {
			ctrl.InputUrlTooltip = t('core', 'Not supported!');
		} else if (/Mac/i.test(navigator.userAgent)) {
			ctrl.InputUrlTooltip = t('core', 'Press ⌘-C to copy.');
		} else {
			ctrl.InputUrlTooltip = t('core', 'Press Ctrl-C to copy.');
		}
		$('#addressBookUrl_'+ctrl.addressBook.ctag).select();
	};

	ctrl.renameAddressBook = function() {
		AddressBookService.rename(ctrl.addressBook, ctrl.addressBook.displayName);
		ctrl.editing = false;
	};

	ctrl.edit = function() {
		ctrl.editing = true;
	};

	ctrl.closeMenus = function() {
		$scope.$parent.ctrl.openedMenu = false;
	};

	ctrl.openMenu = function(index) {
		ctrl.closeMenus();
		$scope.$parent.ctrl.openedMenu = index;
	};

	ctrl.toggleMenu = function(index) {
		if ($scope.$parent.ctrl.openedMenu === index) {
			ctrl.closeMenus();
		} else {
			ctrl.openMenu(index);
		}
	};

	ctrl.toggleSharesEditor = function() {
		ctrl.editingShares = !ctrl.editingShares;
		ctrl.selectedSharee = null;
	};

	/* From Calendar-Rework - js/app/controllers/calendarlistcontroller.js */
	ctrl.findSharee = function (val) {
		return $.get(
			OC.linkToOCS('apps/files_sharing/api/v1') + 'sharees',
			{
				format: 'json',
				search: val.trim(),
				perPage: 200,
				itemType: 'principals'
			}
		).then(function(result) {
			// Todo - filter out current user, existing sharees
			var users   = result.ocs.data.exact.users.concat(result.ocs.data.users);
			var groups  = result.ocs.data.exact.groups.concat(result.ocs.data.groups);

			var userShares = ctrl.addressBook.sharedWith.users;
			var userSharesLength = userShares.length;
			var i, j;

			// Filter out current user
			var usersLength = users.length;
			for (i = 0 ; i < usersLength; i++) {
				if (users[i].value.shareWith === OC.currentUser) {
					users.splice(i, 1);
					break;
				}
			}

			// Now filter out all sharees that are already shared with
			for (i = 0; i < userSharesLength; i++) {
				var share = userShares[i];
				usersLength = users.length;
				for (j = 0; j < usersLength; j++) {
					if (users[j].value.shareWith === share.id) {
						users.splice(j, 1);
						break;
					}
				}
			}

			// Combine users and groups
			users = users.map(function(item) {
				return {
					display: item.value.shareWith,
					type: OC.Share.SHARE_TYPE_USER,
					identifier: item.value.shareWith
				};
			});

			groups = groups.map(function(item) {
				return {
					display: item.value.shareWith + ' (group)',
					type: OC.Share.SHARE_TYPE_GROUP,
					identifier: item.value.shareWith
				};
			});

			return groups.concat(users);
		});
	};

	ctrl.onSelectSharee = function (item) {
		// Prevent settings to slide down
		$('#app-settings-header > button').data('apps-slide-toggle', false);
		_.delay(function() {
			$('#app-settings-header > button').data('apps-slide-toggle', '#app-settings-content');
		}, 500);

		ctrl.selectedSharee = null;
		AddressBookService.share(ctrl.addressBook, item.type, item.identifier, false, false).then(function() {
			$scope.$apply();
		});

	};

	ctrl.updateExistingUserShare = function(userId, writable) {
		AddressBookService.share(ctrl.addressBook, OC.Share.SHARE_TYPE_USER, userId, writable, true).then(function() {
			$scope.$apply();
		});
	};

	ctrl.updateExistingGroupShare = function(groupId, writable) {
		AddressBookService.share(ctrl.addressBook, OC.Share.SHARE_TYPE_GROUP, groupId, writable, true).then(function() {
			$scope.$apply();
		});
	};

	ctrl.unshareFromUser = function(userId) {
		AddressBookService.unshare(ctrl.addressBook, OC.Share.SHARE_TYPE_USER, userId).then(function() {
			$scope.$apply();
		});
	};

	ctrl.unshareFromGroup = function(groupId) {
		AddressBookService.unshare(ctrl.addressBook, OC.Share.SHARE_TYPE_GROUP, groupId).then(function() {
			$scope.$apply();
		});
	};

	ctrl.deleteAddressBook = function() {
		AddressBookService.delete(ctrl.addressBook).then(function() {
			$scope.$apply();
		});
	};

}]);

angular.module('contactsApp')
.directive('addressbook', function() {
	return {
		restrict: 'A', // has to be an attribute to work with core css
		scope: {},
		controller: 'addressbookCtrl',
		controllerAs: 'ctrl',
		bindToController: {
			addressBook: '=data',
			list: '='
		},
		templateUrl: OC.linkTo('contacts', 'templates/addressBook.html')
	};
});

angular.module('contactsApp')
.controller('addressbooklistCtrl', ['$scope', 'AddressBookService', function($scope, AddressBookService) {
	var ctrl = this;

	ctrl.loading = true;
	ctrl.openedMenu = false;
	ctrl.addressBookRegex = /^[a-zA-Z0-9À-ÿ\s-_.!?#|()]+$/i;

	AddressBookService.getAll().then(function(addressBooks) {
		ctrl.addressBooks = addressBooks;
		ctrl.loading = false;
		if(ctrl.addressBooks.length === 0) {
			AddressBookService.create(t('contacts', 'Contacts')).then(function() {
				AddressBookService.getAddressBook(t('contacts', 'Contacts')).then(function(addressBook) {
					ctrl.addressBooks.push(addressBook);
					$scope.$apply();
				});
			});
		}
	});

	ctrl.t = {
		addressBookName : t('contacts', 'Address book name'),
		regexError : t('contacts', 'Only these special characters are allowed: -_.!?#|()')
	};

	ctrl.createAddressBook = function() {
		if(ctrl.newAddressBookName) {
			AddressBookService.create(ctrl.newAddressBookName).then(function() {
				AddressBookService.getAddressBook(ctrl.newAddressBookName).then(function(addressBook) {
					ctrl.addressBooks.push(addressBook);
					$scope.$apply();
				});
			}).catch(function() {
				OC.Notification.showTemporary(t('contacts', 'Address book could not be created.'));
			});
		}
	};
}]);

angular.module('contactsApp')
.directive('addressbooklist', function() {
	return {
		restrict: 'EA', // has to be an attribute to work with core css
		scope: {},
		controller: 'addressbooklistCtrl',
		controllerAs: 'ctrl',
		bindToController: {},
		templateUrl: OC.linkTo('contacts', 'templates/addressBookList.html')
	};
});

angular.module('contactsApp')
.controller('avatarCtrl', ['ContactService', function(ContactService) {
	var ctrl = this;

	ctrl.import = ContactService.import.bind(ContactService);

	ctrl.removePhoto = function() {
		ctrl.contact.removeProperty('photo', ctrl.contact.getProperty('photo'));
		ContactService.update(ctrl.contact);
		$('avatar').removeClass('maximized');
	};

	ctrl.downloadPhoto = function() {
		/* globals ArrayBuffer, Uint8Array */
		var img = document.getElementById('contact-avatar');
		// atob to base64_decode the data-URI
		var imageSplit = img.src.split(',');
		// "data:image/png;base64" -> "png"
		var extension = '.' + imageSplit[0].split(';')[0].split('/')[1];
		var imageData = atob(imageSplit[1]);
		// Use typed arrays to convert the binary data to a Blob
		var arrayBuffer = new ArrayBuffer(imageData.length);
		var view = new Uint8Array(arrayBuffer);
		for (var i=0; i<imageData.length; i++) {
			view[i] = imageData.charCodeAt(i) & 0xff;
		}
		var blob = new Blob([arrayBuffer], {type: 'application/octet-stream'});

		// Use the URL object to create a temporary URL
		var url = (window.webkitURL || window.URL).createObjectURL(blob);

		var a = document.createElement('a');
		document.body.appendChild(a);
		a.style = 'display: none';
		a.href = url;
		a.download = ctrl.contact.uid() + extension;
		a.click();
		window.URL.revokeObjectURL(url);
		a.remove();
	};

	ctrl.openPhoto = function() {
		$('avatar').toggleClass('maximized');
	};

	// Quit avatar preview
	$('avatar').click(function() {
		$('avatar').removeClass('maximized');
	});
	$('avatar img, avatar .avatar-options').click(function(e) {
		e.stopPropagation();
	});
	$(document).keyup(function(e) {
		if (e.keyCode === 27) {
			$('avatar').removeClass('maximized');
		}
	});

}]);

angular.module('contactsApp')
.directive('avatar', ['ContactService', function(ContactService) {
	return {
		scope: {
			contact: '=data'
		},
		controller: 'avatarCtrl',
		controllerAs: 'ctrl',
		bindToController: {
			contact: '=data'
		},
		link: function(scope, element) {
			var input = element.find('input');
			input.bind('change', function() {
				var file = input.get(0).files[0];
				if (file.size > 1024*1024) { // 1 MB
					OC.Notification.showTemporary(t('contacts', 'The selected image is too big (max 1MB)'));
				} else {
					var reader = new FileReader();

					reader.addEventListener('load', function () {
						scope.$apply(function() {
							scope.contact.photo(reader.result);
							ContactService.update(scope.contact);
						});
					}, false);

					if (file) {
						reader.readAsDataURL(file);
					}
				}
			});
		},
		templateUrl: OC.linkTo('contacts', 'templates/avatar.html')
	};
}]);

angular.module('contactsApp')
.controller('contactCtrl', ['$route', '$routeParams', 'SortByService', function($route, $routeParams, SortByService) {
	var ctrl = this;

	ctrl.t = {
		errorMessage : t('contacts', 'This card is corrupted and has been fixed. Please check the data and trigger a save to make the changes permanent.'),
	};

	ctrl.openContact = function() {
		$route.updateParams({
			gid: $routeParams.gid,
			uid: ctrl.contact.uid()});
	};

	ctrl.getName = function() {
		// If lastName equals to firstName then none of them is set
		if (ctrl.contact.lastName() === ctrl.contact.firstName()) {
			return ctrl.contact.displayName();
		}

		if (SortByService.getSortBy() === 'sortLastName') {
			return (
				ctrl.contact.lastName() + ', '
				+ ctrl.contact.firstName() + ' '
				+ ctrl.contact.additionalNames()
			).trim();
		}

		if (SortByService.getSortBy() === 'sortFirstName') {
			return (
				ctrl.contact.firstName() + ' '
				+ ctrl.contact.additionalNames() + ' '
				+ ctrl.contact.lastName()
			).trim();
		}

		return ctrl.contact.displayName();
	};
}]);

angular.module('contactsApp')
.directive('contact', function() {
	return {
		scope: {},
		controller: 'contactCtrl',
		controllerAs: 'ctrl',
		bindToController: {
			contact: '=data'
		},
		templateUrl: OC.linkTo('contacts', 'templates/contact.html')
	};
});

angular.module('contactsApp')
.controller('contactdetailsCtrl', ['ContactService', 'AddressBookService', 'vCardPropertiesService', '$route', '$routeParams', '$scope', function(ContactService, AddressBookService, vCardPropertiesService, $route, $routeParams, $scope) {

	var ctrl = this;

	ctrl.init = true;
	ctrl.loading = false;
	ctrl.show = false;

	ctrl.clearContact = function() {
		$route.updateParams({
			gid: $routeParams.gid,
			uid: undefined
		});
		ctrl.show = false;
		ctrl.contact = undefined;
	};

	ctrl.uid = $routeParams.uid;
	ctrl.t = {
		noContacts : t('contacts', 'No contacts in here'),
		placeholderName : t('contacts', 'Name'),
		placeholderOrg : t('contacts', 'Organization'),
		placeholderTitle : t('contacts', 'Title'),
		selectField : t('contacts', 'Add field …'),
		download : t('contacts', 'Download'),
		delete : t('contacts', 'Delete'),
		save : t('contacts', 'Save changes'),
		addressBook : t('contacts', 'Address book'),
		loading : t('contacts', 'Loading contacts …')
	};

	ctrl.fieldDefinitions = vCardPropertiesService.fieldDefinitions;
	ctrl.focus = undefined;
	ctrl.field = undefined;
	ctrl.addressBooks = [];

	AddressBookService.getAll().then(function(addressBooks) {
		ctrl.addressBooks = addressBooks;

		if (!_.isUndefined(ctrl.contact)) {
			ctrl.addressBook = _.find(ctrl.addressBooks, function(book) {
				return book.displayName === ctrl.contact.addressBookId;
			});
		}
		ctrl.init = false;
		// Start watching for ctrl.uid when we have addressBooks, as they are needed for fetching
		// full details.
		$scope.$watch('ctrl.uid', function(newValue) {
			ctrl.changeContact(newValue);
		});
	});


	ctrl.changeContact = function(uid) {
		if (typeof uid === 'undefined') {
			ctrl.show = false;
			$('#app-navigation-toggle').removeClass('showdetails');
			return;
		}
		ctrl.loading = true;
		ContactService.getById(ctrl.addressBooks, uid).then(function(contact) {
			if (angular.isUndefined(contact)) {
				ctrl.clearContact();
				return;
			}
			ctrl.contact = contact;
			ctrl.show = true;
			ctrl.loading = false;
			$('#app-navigation-toggle').addClass('showdetails');

			ctrl.addressBook = _.find(ctrl.addressBooks, function(book) {
				return book.displayName === ctrl.contact.addressBookId;
			});
		});
	};

	ctrl.updateContact = function() {
		ContactService.update(ctrl.contact);
	};

	ctrl.deleteContact = function() {
		ContactService.delete(ctrl.contact);
	};

	ctrl.addField = function(field) {
		var defaultValue = vCardPropertiesService.getMeta(field).defaultValue || {value: ''};
		ctrl.contact.addProperty(field, defaultValue);
		ctrl.focus = field;
		ctrl.field = '';
	};

	ctrl.deleteField = function (field, prop) {
		ctrl.contact.removeProperty(field, prop);
		ctrl.focus = undefined;
	};

	ctrl.changeAddressBook = function (addressBook) {
		ContactService.moveContact(ctrl.contact, addressBook);
	};
}]);

angular.module('contactsApp')
.directive('contactdetails', function() {
	return {
		priority: 1,
		scope: {},
		controller: 'contactdetailsCtrl',
		controllerAs: 'ctrl',
		bindToController: {},
		templateUrl: OC.linkTo('contacts', 'templates/contactDetails.html')
	};
});

angular.module('contactsApp')
.controller('contactimportCtrl', ['ContactService', 'AddressBookService', function(ContactService, AddressBookService) {
	var ctrl = this;

	ctrl.t = {
		importText : t('contacts', 'Import into'),
		importingText : t('contacts', 'Importing...'),
		selectAddressbook : t('contacts', 'Select your addressbook')
	};

	ctrl.import = ContactService.import.bind(ContactService);
	ctrl.loading = true;
	ctrl.importText = ctrl.t.importText;
	ctrl.importing = false;
	ctrl.loadingClass = 'icon-upload';

	AddressBookService.getAll().then(function(addressBooks) {
		ctrl.addressBooks = addressBooks;
		ctrl.loading = false;
		ctrl.selectedAddressBook = AddressBookService.getDefaultAddressBook();
	});

	ctrl.stopHideMenu = function(isOpen) {
		if(isOpen) {
			// disabling settings bind
			$('#app-settings-header > button').data('apps-slide-toggle', false);
		} else {
			// reenabling it
			$('#app-settings-header > button').data('apps-slide-toggle', '#app-settings-content');
		}
	};

}]);

angular.module('contactsApp')
.directive('contactimport', ['ContactService', 'ImportService', '$rootScope', function(ContactService, ImportService, $rootScope) {
	return {
		link: function(scope, element, attrs, ctrl) {
			var input = element.find('input');
			input.bind('change', function() {
				angular.forEach(input.get(0).files, function(file) {
					var reader = new FileReader();

					reader.addEventListener('load', function () {
						scope.$apply(function () {
							// Indicate the user we started something
							ctrl.importText = ctrl.t.importingText;
							ctrl.loadingClass = 'icon-loading-small';
							ctrl.importing = true;
							$rootScope.importing = true;

							ContactService.import.call(ContactService, reader.result, file.type, ctrl.selectedAddressBook, function (progress, user) {
								if (progress === 1) {
									ctrl.importText = ctrl.t.importText;
									ctrl.loadingClass = 'icon-upload';
									ctrl.importing = false;
									$rootScope.importing = false;
									ImportService.importPercent = 0;
									ImportService.importing = false;
									ImportService.importedUser = '';
									ImportService.selectedAddressBook = '';
								} else {
									// Ugly hack, hide sidebar on import & mobile
									// Simulate click since we can't directly access snapper
									if($(window).width() <= 768 && $('body').hasClass('snapjs-left')) {
										$('#app-navigation-toggle').click();
										$('body').removeClass('snapjs-left');
									}

									ImportService.importPercent = parseInt(Math.floor(progress * 100));
									ImportService.importing = true;
									ImportService.importedUser = user;
									ImportService.selectedAddressBook = ctrl.selectedAddressBook.displayName;
								}
								scope.$apply();

								/* Broadcast service update */
								$rootScope.$broadcast('importing', true);
							});
						});
					}, false);

					if (file) {
						reader.readAsText(file);
					}
				});
				input.get(0).value = '';
			});
		},
		templateUrl: OC.linkTo('contacts', 'templates/contactImport.html'),
		controller: 'contactimportCtrl',
		controllerAs: 'ctrl'
	};
}]);

angular.module('contactsApp')
.controller('contactlistCtrl', ['$scope', '$filter', '$route', '$routeParams', '$timeout', 'AddressBookService', 'ContactService', 'SortByService', 'vCardPropertiesService', 'SearchService', function($scope, $filter, $route, $routeParams, $timeout, AddressBookService, ContactService, SortByService, vCardPropertiesService, SearchService) {
	var ctrl = this;

	ctrl.routeParams = $routeParams;

	ctrl.filteredContacts = []; // the displayed contacts list
	ctrl.searchTerm = '';
	ctrl.show = true;
	ctrl.invalid = false;
	ctrl.limitTo = 25;

	ctrl.sortBy = SortByService.getSortBy();

	ctrl.t = {
		emptySearch : t('contacts', 'No search result for {query}', {query: ctrl.searchTerm})
	};

	ctrl.resetLimitTo = function () {
		ctrl.limitTo = 25;
		clearInterval(ctrl.intervalId);
		ctrl.intervalId = setInterval(
			function () {
				if (!ctrl.loading && ctrl.contactList && ctrl.contactList.length > ctrl.limitTo) {
					ctrl.limitTo += 25;
					$scope.$apply();
				}
			}, 300);
	};

	$scope.query = function(contact) {
		return contact.matches(SearchService.getSearchTerm());
	};

	SortByService.subscribe(function(newValue) {
		ctrl.sortBy = newValue;
	});

	SearchService.registerObserverCallback(function(ev) {
		if (ev.event === 'submitSearch') {
			var uid = !_.isEmpty(ctrl.filteredContacts) ? ctrl.filteredContacts[0].uid() : undefined;
			ctrl.setSelectedId(uid);
			$scope.$apply();
		}
		if (ev.event === 'changeSearch') {
			ctrl.resetLimitTo();
			ctrl.searchTerm = ev.searchTerm;
			ctrl.t.emptySearch = t('contacts',
								   'No search result for {query}',
								   {query: ctrl.searchTerm}
								  );
			$scope.$apply();
		}
	});

	ctrl.loading = true;

	ContactService.registerObserverCallback(function(ev) {
		/* after import at first refresh the contactList */
		if (ev.event === 'importend') {
			$scope.$apply(function() {
				ctrl.contactList = ev.contacts;
			});
		}
		/* update route parameters */
		$timeout(function() {
			$scope.$apply(function() {
				switch(ev.event) {
				case 'delete':
					ctrl.selectNearestContact(ev.uid);
					break;
				case 'create':
					$route.updateParams({
						gid: $routeParams.gid,
						uid: ev.uid
					});
					break;
				case 'importend':
					/* after import select 'All contacts' group and first contact */
					$route.updateParams({
						gid: t('contacts', 'All contacts'),
						uid: ctrl.filteredContacts.length !== 0 ? ctrl.filteredContacts[0].uid() : undefined
					});
					return;
				case 'getFullContacts' || 'update':
					break;
				default:
					// unknown event -> leave callback without action
					return;
				}
				ctrl.contactList = ev.contacts;
			});
		});
	});

	AddressBookService.registerObserverCallback(function(ev) {
		$timeout(function() {
			$scope.$apply(function() {
				if (ev.event === 'delete') {
					// Get contacts
					ctrl.loading = true;
					ContactService.updateDeletedAddressbook(function() {
						ContactService.getAll().then(function(contacts) {
							ctrl.contactList = contacts;
							ctrl.loading = false;
							ctrl.selectNearestContact(ctrl.getSelectedId());
						});
					});
				}
			});
		});
	});

	// Get contacts
	ContactService.getAll().then(function(contacts) {
		if(contacts.length>0) {
			$scope.$apply(function() {
				ctrl.contactList = contacts;
			});
		} else {
			ctrl.loading = false;
		}
	});

	var getVisibleContacts = function() {
		var scrolled = $('.app-content-list').scrollTop();
		var elHeight = $('.contacts-list').children().outerHeight(true);
		var listHeight = $('.app-content-list').height();

		var topContact = Math.round(scrolled/elHeight);
		var contactsCount = Math.round(listHeight/elHeight);

		return ctrl.filteredContacts.slice(topContact-1, topContact+contactsCount+1);
	};

	var timeoutId = null;
	document.querySelector('.app-content-list').addEventListener('scroll', function () {
		clearTimeout(timeoutId);
		timeoutId = setTimeout(function () {
			var contacts = getVisibleContacts();
			ContactService.getFullContacts(contacts);
		}, 250);
	});

	// Wait for ctrl.filteredContacts to be updated, load the contact requested in the URL if any, and
	// load full details for the probably initially visible contacts.
	// Then kill the watch.
	var unbindListWatch = $scope.$watch('ctrl.filteredContacts', function() {
		if(ctrl.filteredContacts && ctrl.filteredContacts.length > 0) {
			// Check if a specific uid is requested
			if($routeParams.uid && $routeParams.gid) {
				ctrl.filteredContacts.forEach(function(contact) {
					if(contact.uid() === $routeParams.uid) {
						ctrl.setSelectedId($routeParams.uid);
						ctrl.loading = false;
					}
				});
			}
			// No contact previously loaded, let's load the first of the list if not in mobile mode
			if(ctrl.loading && $(window).width() > 768) {
				ctrl.setSelectedId(ctrl.filteredContacts[0].uid());
			}
			// Get full data for the first 20 contacts of the list
			ContactService.getFullContacts(ctrl.filteredContacts.slice(0, 20));
			ctrl.loading = false;
			unbindListWatch();
		}
	});

	$scope.$watch('ctrl.routeParams.uid', function(newValue, oldValue) {
		// Used for mobile view to clear the url
		if(typeof oldValue != 'undefined' && typeof newValue == 'undefined' && $(window).width() <= 768) {
			// no contact selected
			ctrl.show = true;
			return;
		}
		if(newValue === undefined) {
			// we might have to wait until ng-repeat filled the contactList
			if(ctrl.filteredContacts && ctrl.filteredContacts.length > 0) {
				$route.updateParams({
					gid: $routeParams.gid,
					uid: ctrl.filteredContacts[0].uid()
				});
			} else {
				// watch for next contactList update
				var unbindWatch = $scope.$watch('ctrl.filteredContacts', function() {
					if(ctrl.filteredContacts && ctrl.filteredContacts.length > 0) {
						$route.updateParams({
							gid: $routeParams.gid,
							uid: ctrl.filteredContacts[0].uid()
						});
					}
					unbindWatch(); // unbind as we only want one update
				});
			}
		} else {
			// displaying contact details
			ctrl.show = false;
		}
	});

	$scope.$watch('ctrl.routeParams.gid', function() {
		// we might have to wait until ng-repeat filled the contactList
		ctrl.filteredContacts = [];
		ctrl.resetLimitTo();
		// not in mobile mode
		if($(window).width() > 768) {
			// watch for next contactList update
			var unbindWatch = $scope.$watch('ctrl.filteredContacts', function() {
				if(ctrl.filteredContacts && ctrl.filteredContacts.length > 0) {
					$route.updateParams({
						gid: $routeParams.gid,
						uid: $routeParams.uid || ctrl.filteredContacts[0].uid()
					});
				}
				unbindWatch(); // unbind as we only want one update
			});
		}
	});

	// Watch if we have an invalid contact
	$scope.$watch('ctrl.filteredContacts[0].displayName()', function(displayName) {
		ctrl.invalid = (displayName === '');
	});

	ctrl.hasContacts = function () {
		if (!ctrl.contactList) {
			return false;
		}
		return ctrl.contactList.length > 0;
	};

	ctrl.setSelectedId = function (contactId) {
		$route.updateParams({
			uid: contactId
		});
	};

	ctrl.getSelectedId = function() {
		return $routeParams.uid;
	};

	ctrl.selectNearestContact = function(contactId) {
		if (ctrl.filteredContacts.length === 1) {
			$route.updateParams({
				gid: $routeParams.gid,
				uid: undefined
			});
		} else {
			for (var i = 0, length = ctrl.filteredContacts.length; i < length; i++) {
				// Get nearest contact
				if (ctrl.filteredContacts[i].uid() === contactId) {
					$route.updateParams({
						gid: $routeParams.gid,
						uid: (ctrl.filteredContacts[i+1]) ? ctrl.filteredContacts[i+1].uid() : ctrl.filteredContacts[i-1].uid()
					});
					break;
				}
			}
		}
	};

}]);

angular.module('contactsApp')
.directive('contactlist', function() {
	return {
		priority: 1,
		scope: {},
		controller: 'contactlistCtrl',
		controllerAs: 'ctrl',
		bindToController: {
			addressbook: '=adrbook'
		},
		templateUrl: OC.linkTo('contacts', 'templates/contactList.html')
	};
});

angular.module('contactsApp')
.controller('detailsItemCtrl', ['$templateRequest', 'vCardPropertiesService', 'ContactService', function($templateRequest, vCardPropertiesService, ContactService) {
	var ctrl = this;

	ctrl.meta = vCardPropertiesService.getMeta(ctrl.name);
	ctrl.type = undefined;
	ctrl.isPreferred = false;
	ctrl.t = {
		poBox : t('contacts', 'Post office box'),
		postalCode : t('contacts', 'Postal code'),
		city : t('contacts', 'City'),
		state : t('contacts', 'State or province'),
		country : t('contacts', 'Country'),
		address: t('contacts', 'Address'),
		newGroup: t('contacts', '(new group)'),
		familyName: t('contacts', 'Last name'),
		firstName: t('contacts', 'First name'),
		additionalNames: t('contacts', 'Additional names'),
		honorificPrefix: t('contacts', 'Prefix'),
		honorificSuffix: t('contacts', 'Suffix'),
		delete: t('contacts', 'Delete')
	};

	ctrl.availableOptions = ctrl.meta.options || [];
	if (!_.isUndefined(ctrl.data) && !_.isUndefined(ctrl.data.meta) && !_.isUndefined(ctrl.data.meta.type)) {
		// parse type of the property
		var array = ctrl.data.meta.type[0].split(',');
		array = array.map(function (elem) {
			return elem.trim().replace(/\/+$/, '').replace(/\\+$/, '').trim().toUpperCase();
		});
		// the pref value is handled on its own so that we can add some favorite icon to the ui if we want
		if (array.indexOf('PREF') >= 0) {
			ctrl.isPreferred = true;
			array.splice(array.indexOf('PREF'), 1);
		}
		// simply join the upper cased types together as key
		ctrl.type = array.join(',');
		var displayName = array.map(function (element) {
			return element.charAt(0).toUpperCase() + element.slice(1).toLowerCase();
		}).join(' ');

		// in case the type is not yet in the default list of available options we add it
		if (!ctrl.availableOptions.some(function(e) { return e.id === ctrl.type; } )) {
			ctrl.availableOptions = ctrl.availableOptions.concat([{id: ctrl.type, name: displayName}]);
		}
	}
	if (!_.isUndefined(ctrl.data) && !_.isUndefined(ctrl.data.namespace)) {
		if (!_.isUndefined(ctrl.model.contact.props['X-ABLABEL'])) {
			var val = _.find(this.model.contact.props['X-ABLABEL'], function(x) { return x.namespace === ctrl.data.namespace; });
			ctrl.type = val.value;
			if (!_.isUndefined(val)) {
				// in case the type is not yet in the default list of available options we add it
				if (!ctrl.availableOptions.some(function(e) { return e.id === val.value; } )) {
					ctrl.availableOptions = ctrl.availableOptions.concat([{id: val.value, name: val.value}]);
				}
			}
		}
	}
	ctrl.availableGroups = [];

	ContactService.getGroups().then(function(groups) {
		ctrl.availableGroups = _.unique(groups);
	});

	ctrl.changeType = function (val) {
		if (ctrl.isPreferred) {
			val += ',PREF';
		}
		ctrl.data.meta = ctrl.data.meta || {};
		ctrl.data.meta.type = ctrl.data.meta.type || [];
		ctrl.data.meta.type[0] = val;
		ctrl.model.updateContact();
	};

	ctrl.dateInputChanged = function () {
		ctrl.data.meta = ctrl.data.meta || {};

		var match = ctrl.data.value.match(/^(\d{4})-(\d{2})-(\d{2})$/);
		if (match) {
			ctrl.data.meta.value = [];
		} else {
			ctrl.data.meta.value = ctrl.data.meta.value || [];
			ctrl.data.meta.value[0] = 'text';
		}
		ctrl.model.updateContact();
	};

	ctrl.updateDetailedName = function () {
		var fn = '';
		if (ctrl.data.value[3]) {
			fn += ctrl.data.value[3] + ' ';
		}
		if (ctrl.data.value[1]) {
			fn += ctrl.data.value[1] + ' ';
		}
		if (ctrl.data.value[2]) {
			fn += ctrl.data.value[2] + ' ';
		}
		if (ctrl.data.value[0]) {
			fn += ctrl.data.value[0] + ' ';
		}
		if (ctrl.data.value[4]) {
			fn += ctrl.data.value[4];
		}

		ctrl.model.contact.fullName(fn);
		ctrl.model.updateContact();
	};

	ctrl.getTemplate = function() {
		var templateUrl = OC.linkTo('contacts', 'templates/detailItems/' + ctrl.meta.template + '.html');
		return $templateRequest(templateUrl);
	};

	ctrl.deleteField = function () {
		ctrl.model.deleteField(ctrl.name, ctrl.data);
		ctrl.model.updateContact();
	};
}]);

angular.module('contactsApp')
.directive('detailsitem', ['$compile', function($compile) {
	return {
		scope: {},
		controller: 'detailsItemCtrl',
		controllerAs: 'ctrl',
		bindToController: {
			name: '=',
			data: '=',
			model: '=',
			index: '='
		},
		link: function(scope, element, attrs, ctrl) {
			ctrl.getTemplate().then(function(html) {
				var template = angular.element(html);
				element.append(template);
				$compile(template)(scope);
			});
		}
	};
}]);

angular.module('contactsApp')
.controller('groupCtrl', function() {
	// eslint-disable-next-line no-unused-vars
	var ctrl = this;
});

angular.module('contactsApp')
.directive('group', function() {
	return {
		restrict: 'A', // has to be an attribute to work with core css
		scope: {},
		controller: 'groupCtrl',
		controllerAs: 'ctrl',
		bindToController: {
			group: '=groupName',
			groupCount: '=groupCount'
		},
		templateUrl: OC.linkTo('contacts', 'templates/group.html')
	};
});

angular.module('contactsApp')
.controller('grouplistCtrl', ['$scope', '$timeout', 'ContactService', 'SearchService', '$routeParams', function($scope, $timeout, ContactService, SearchService, $routeParams) {
	var ctrl = this;

	ctrl.groups = [];

	ContactService.getGroupList().then(function(groups) {
		ctrl.groups = groups;
	});

	ctrl.getSelected = function() {
		return $routeParams.gid;
	};

	// Update groupList on contact add/delete/update/groupsUpdate
	ContactService.registerObserverCallback(function(ev) {
		if (ev.event !== 'getFullContacts') {
			$timeout(function () {
				$scope.$apply(function() {
					ContactService.getGroupList().then(function(groups) {
						ctrl.groups = groups;
					});
				});
			});
		}
	});

	ctrl.setSelected = function (selectedGroup) {
		SearchService.cleanSearch();
		$routeParams.gid = selectedGroup;
	};
}]);

angular.module('contactsApp')
.directive('grouplist', function() {
	return {
		restrict: 'EA', // has to be an attribute to work with core css
		scope: {},
		controller: 'grouplistCtrl',
		controllerAs: 'ctrl',
		bindToController: {},
		templateUrl: OC.linkTo('contacts', 'templates/groupList.html')
	};
});

angular.module('contactsApp')
.controller('importscreenCtrl', ['$scope', 'ImportService', function($scope, ImportService) {
	var ctrl = this;

	ctrl.t = {
		importingTo : t('contacts', 'Importing into'),
		selectAddressbook : t('contacts', 'Select your addressbook')
	};

	// Broadcast update
	$scope.$on('importing', function () {
		ctrl.selectedAddressBook = ImportService.selectedAddressBook;
		ctrl.importedUser = ImportService.importedUser;
		ctrl.importing = ImportService.importing;
		ctrl.importPercent = ImportService.importPercent;
	});

}]);

angular.module('contactsApp')
.directive('importscreen', function() {
	return {
		restrict: 'EA', // has to be an attribute to work with core css
		scope: {},
		controller: 'importscreenCtrl',
		controllerAs: 'ctrl',
		bindToController: {},
		templateUrl: OC.linkTo('contacts', 'templates/importScreen.html')
	};
});

angular.module('contactsApp')
.controller('newContactButtonCtrl', ['$scope', 'ContactService', '$routeParams', 'vCardPropertiesService', function($scope, ContactService, $routeParams, vCardPropertiesService) {
	var ctrl = this;

	ctrl.t = {
		addContact : t('contacts', 'New contact')
	};

	ctrl.createContact = function() {
		ContactService.create().then(function(contact) {
			['tel', 'adr', 'email'].forEach(function(field) {
				var defaultValue = vCardPropertiesService.getMeta(field).defaultValue || {value: ''};
				contact.addProperty(field, defaultValue);
			} );
			if ([t('contacts', 'All contacts'), t('contacts', 'Not grouped')].indexOf($routeParams.gid) === -1) {
				contact.categories([ $routeParams.gid ]);
			} else {
				contact.categories([]);
			}
			$('#details-fullName').focus();
		});
	};
}]);

angular.module('contactsApp')
.directive('newcontactbutton', function() {
	return {
		restrict: 'EA', // has to be an attribute to work with core css
		scope: {},
		controller: 'newContactButtonCtrl',
		controllerAs: 'ctrl',
		bindToController: {},
		templateUrl: OC.linkTo('contacts', 'templates/newContactButton.html')
	};
});

angular.module('contactsApp')
.directive('telModel', function() {
	return{
		restrict: 'A',
		require: 'ngModel',
		link: function(scope, element, attr, ngModel) {
			ngModel.$formatters.push(function(value) {
				return value;
			});
			ngModel.$parsers.push(function(value) {
				return value;
			});
		}
	};
});

angular.module('contactsApp')
.controller('sortbyCtrl', ['SortByService', function(SortByService) {
	var ctrl = this;

	var sortText = t('contacts', 'Sort by');
	ctrl.sortText = sortText;

	var sortList = SortByService.getSortByList();
	ctrl.sortList = sortList;

	ctrl.defaultOrder = SortByService.getSortBy();

	ctrl.updateSortBy = function() {
		SortByService.setSortBy(ctrl.defaultOrder);
	};
}]);

angular.module('contactsApp')
.directive('sortby', function() {
	return {
		priority: 1,
		scope: {},
		controller: 'sortbyCtrl',
		controllerAs: 'ctrl',
		bindToController: {},
		templateUrl: OC.linkTo('contacts', 'templates/sortBy.html')
	};
});

angular.module('contactsApp')
.factory('AddressBook', function()
{
	return function AddressBook(data) {
		angular.extend(this, {

			displayName: '',
			contacts: [],
			groups: data.data.props.groups,
			readOnly: data.data.props.readOnly === '1',

			getContact: function(uid) {
				for(var i in this.contacts) {
					if(this.contacts[i].uid() === uid) {
						return this.contacts[i];
					}
				}
				return undefined;
			},

			sharedWith: {
				users: [],
				groups: []
			}

		});
		angular.extend(this, data);
		angular.extend(this, {
			owner: data.data.props.owner.split('/').slice(-2, -1)[0]
		});

		var shares = this.data.props.invite;
		if (typeof shares !== 'undefined') {
			for (var j = 0; j < shares.length; j++) {
				var href = shares[j].href;
				if (href.length === 0) {
					continue;
				}
				var access = shares[j].access;
				if (access.length === 0) {
					continue;
				}

				var readWrite = (typeof access.readWrite !== 'undefined');

				if (href.startsWith('principal:principals/users/')) {
					this.sharedWith.users.push({
						id: href.substr(27),
						displayname: href.substr(27),
						writable: readWrite
					});
				} else if (href.startsWith('principal:principals/groups/')) {
					this.sharedWith.groups.push({
						id: href.substr(28),
						displayname: href.substr(28),
						writable: readWrite
					});
				}
			}
		}
	};
});

angular.module('contactsApp')
.factory('Contact', ['$filter', 'MimeService', function($filter, MimeService) {
	return function Contact(addressBook, vCard) {
		angular.extend(this, {

			data: {},
			props: {},
			failedProps: [],

			dateProperties: ['bday', 'anniversary', 'deathdate'],

			addressBookId: addressBook.displayName,

			version: function() {
				var property = this.getProperty('version');
				if(property) {
					return property.value;
				}

				return undefined;
			},

			uid: function(value) {
				var model = this;
				if (angular.isDefined(value)) {
					// setter
					return model.setProperty('uid', { value: value });
				} else {
					// getter
					return model.getProperty('uid').value;
				}
			},

			sortFirstName: function() {
				return [this.firstName(), this.lastName()];
			},

			sortLastName: function() {
				return [this.lastName(), this.firstName()];
			},

			sortDisplayName: function() {
				return this.displayName();
			},

			displayName: function() {
				var displayName = this.fullName() || this.org() || '';
				if(angular.isArray(displayName)) {
					return displayName.join(' ');
				}
				return displayName;
			},

			readableFilename: function() {
				if(this.displayName()) {
					return (this.displayName()) + '.vcf';
				} else {
					// fallback to default filename (see download attribute)
					return '';
				}

			},

			firstName: function() {
				var property = this.getProperty('n');
				if (property) {
					return property.value[1];
				} else {
					return this.displayName();
				}
			},

			lastName: function() {
				var property = this.getProperty('n');
				if (property) {
					return property.value[0];
				} else {
					return this.displayName();
				}
			},

			additionalNames: function() {
				var property = this.getProperty('n');
				if (property) {
					return property.value[2];
				} else {
					return '';
				}
			},

			fullName: function(value) {
				var model = this;
				if (angular.isDefined(value)) {
					// setter
					return this.setProperty('fn', { value: value });
				} else {
					// getter
					var property = model.getProperty('fn');
					if(property) {
						return property.value;
					}
					property = model.getProperty('n');
					if(property) {
						return property.value.filter(function(elem) {
							return elem;
						}).join(' ');
					}
					return undefined;
				}
			},

			title: function(value) {
				if (angular.isDefined(value)) {
					// setter
					return this.setProperty('title', { value: value });
				} else {
					// getter
					var property = this.getProperty('title');
					if(property) {
						return property.value;
					} else {
						return undefined;
					}
				}
			},

			org: function(value) {
				var property = this.getProperty('org');
				if (angular.isDefined(value)) {
					var val = value;
					// setter
					if(property && Array.isArray(property.value)) {
						val = property.value;
						val[0] = value;
					}
					return this.setProperty('org', { value: val });
				} else {
					// getter
					if(property) {
						if (Array.isArray(property.value)) {
							return property.value[0];
						}
						return property.value;
					} else {
						return undefined;
					}
				}
			},

			email: function() {
				// getter
				var property = this.getProperty('email');
				if(property) {
					return property.value;
				} else {
					return undefined;
				}
			},

			photo: function(value) {
				if (angular.isDefined(value)) {
					// setter
					// splits image data into "data:image/jpeg" and base 64 encoded image
					var imageData = value.split(';base64,');
					var imageType = imageData[0].slice('data:'.length);
					if (!imageType.startsWith('image/')) {
						return;
					}
					imageType = imageType.substring(6).toUpperCase();

					return this.setProperty('photo', { value: imageData[1], meta: {type: [imageType], encoding: ['b']} });
				} else {
					var property = this.getProperty('photo');
					if(property) {
						var type = property.meta.type;
						if (angular.isArray(type)) {
							type = type[0];
						}
						if (!type.startsWith('image/')) {
							type = 'image/' + type.toLowerCase();
						}
						return 'data:' + type + ';base64,' + property.value;
					} else {
						return undefined;
					}
				}
			},

			categories: function(value) {
				if (angular.isDefined(value)) {
					// setter
					if (angular.isString(value)) {
						/* check for empty string */
						this.setProperty('categories', { value: !value.length ? [] : [value] });
					} else if (angular.isArray(value)) {
						this.setProperty('categories', { value: value });
					}
				} else {
					// getter
					var property = this.getProperty('categories');
					if(!property) {
						return [];
					}
					if (angular.isArray(property.value)) {
						return property.value;
					}
					return [property.value];
				}
			},

			formatDateAsRFC6350: function(name, data) {
				if (angular.isUndefined(data) || angular.isUndefined(data.value)) {
					return data;
				}
				if (this.dateProperties.indexOf(name) !== -1) {
					var match = data.value.match(/^(\d{4})-(\d{2})-(\d{2})$/);
					if (match) {
						data.value = match[1] + match[2] + match[3];
					}
				}

				return data;
			},

			formatDateForDisplay: function(name, data) {
				if (angular.isUndefined(data) || angular.isUndefined(data.value)) {
					return data;
				}
				if (this.dateProperties.indexOf(name) !== -1) {
					var match = data.value.match(/^(\d{4})(\d{2})(\d{2})$/);
					if (match) {
						data.value = match[1] + '-' + match[2] + '-' + match[3];
					}
				}

				return data;
			},

			getProperty: function(name) {
				if (this.props[name]) {
					return this.formatDateForDisplay(name, this.validate(name, this.props[name][0]));
				} else {
					return undefined;
				}
			},
			addProperty: function(name, data) {
				data = angular.copy(data);
				data = this.formatDateAsRFC6350(name, data);
				if(!this.props[name]) {
					this.props[name] = [];
				}
				var idx = this.props[name].length;
				this.props[name][idx] = data;

				// keep vCard in sync
				this.data.addressData = $filter('JSON2vCard')(this.props);
				return idx;
			},
			setProperty: function(name, data) {
				if(!this.props[name]) {
					this.props[name] = [];
				}
				data = this.formatDateAsRFC6350(name, data);
				this.props[name][0] = data;

				// keep vCard in sync
				this.data.addressData = $filter('JSON2vCard')(this.props);
			},
			removeProperty: function (name, prop) {
				angular.copy(_.without(this.props[name], prop), this.props[name]);
				this.data.addressData = $filter('JSON2vCard')(this.props);
			},
			setETag: function(etag) {
				this.data.etag = etag;
			},
			setUrl: function(addressBook, uid) {
				this.data.url = addressBook.url + uid + '.vcf';
			},
			setAddressBook: function(addressBook) {
				this.addressBook = addressBook;
				this.addressBookId = addressBook.displayName;
				this.data.url = addressBook.url + this.uid() + '.vcf';
			},

			getISODate: function(date) {
				function pad(number) {
					if (number < 10) {
						return '0' + number;
					}
					return '' + number;
				}

				return date.getUTCFullYear() + '' +
						pad(date.getUTCMonth() + 1) +
						pad(date.getUTCDate()) +
						'T' + pad(date.getUTCHours()) +
						pad(date.getUTCMinutes()) +
						pad(date.getUTCSeconds()) + 'Z';
			},

			syncVCard: function() {

				this.setProperty('rev', { value: this.getISODate(new Date()) });
				var self = this;

				_.each(this.dateProperties, function(name) {
					if (!angular.isUndefined(self.props[name]) && !angular.isUndefined(self.props[name][0])) {
						// Set dates again to make sure they are in RFC-6350 format
						self.setProperty(name, self.props[name][0]);
					}
				});
				// force fn to be set
				this.fullName(this.fullName());

				// keep vCard in sync
				self.data.addressData = $filter('JSON2vCard')(self.props);

				// Revalidate all props
				_.each(self.failedProps, function(name, index) {
					if (!angular.isUndefined(self.props[name]) && !angular.isUndefined(self.props[name][0])) {
						// Reset previously failed properties
						self.failedProps.splice(index, 1);
						// And revalidate them again
						self.validate(name, self.props[name][0]);

					} else if(angular.isUndefined(self.props[name]) || angular.isUndefined(self.props[name][0])) {
						// Property has been removed
						self.failedProps.splice(index, 1);
					}
				});

			},

			matches: function(pattern) {
				if (angular.isUndefined(pattern) || pattern.length === 0) {
					return true;
				}
				var model = this;
				var matchingProps = ['fn', 'title', 'org', 'email', 'nickname', 'note', 'url', 'cloud', 'adr', 'impp', 'tel', 'gender', 'relationship'].filter(function (propName) {
					if (model.props[propName]) {
						return model.props[propName].filter(function (property) {
							if (!property.value) {
								return false;
							}
							if (angular.isString(property.value)) {
								return property.value.toLowerCase().indexOf(pattern.toLowerCase()) !== -1;
							}
							if (angular.isArray(property.value)) {
								return property.value.filter(function(v) {
									return v.toLowerCase().indexOf(pattern.toLowerCase()) !== -1;
								}).length > 0;
							}
							return false;
						}).length > 0;
					}
					return false;
				});
				return matchingProps.length > 0;
			},

			/* eslint-disable no-console */
			validate: function(prop, property) {
				switch(prop) {
				case 'rev':
				case 'prodid':
				case 'version':
					if (!angular.isUndefined(this.props[prop]) && this.props[prop].length > 1) {
						this.props[prop] = [this.props[prop][0]];
						console.warn(this.uid()+': Too many '+prop+' fields. Saving this one only: ' + this.props[prop][0].value);
						this.failedProps.push(prop);
					}
					break;

				case 'categories':
					// Avoid unescaped commas
					if (angular.isArray(property.value)) {
						if(property.value.join(';').indexOf(',') !== -1) {
							this.failedProps.push(prop);
							property.value = property.value.join(',').split(',');
							//console.warn(this.uid()+': Categories split: ' + property.value);
						}
					} else if (angular.isString(property.value)) {
						if(property.value.indexOf(',') !== -1) {
							this.failedProps.push(prop);
							property.value = property.value.split(',');
							//console.warn(this.uid()+': Categories split: ' + property.value);
						}
					}
					// Remove duplicate categories on array
					if(property.value.length !== 0 && angular.isArray(property.value)) {
						var uniqueCategories = _.unique(property.value);
						if(!angular.equals(uniqueCategories, property.value)) {
							this.failedProps.push(prop);
							property.value = uniqueCategories;
							//console.warn(this.uid()+': Categories duplicate: ' + property.value);
						}
					}
					break;
				case 'photo':
					// Avoid undefined photo type
					if (angular.isDefined(property)) {
						if (angular.isUndefined(property.meta.type)) {
							var mime = MimeService.b64mime(property.value);
							if (mime) {
								this.failedProps.push(prop);
								property.meta.type=[mime];
								this.setProperty('photo', {
									value:property.value,
									meta: {
										type:property.meta.type,
										encoding:property.meta.encoding
									}
								});
								console.warn(this.uid()+': Photo detected as ' + property.meta.type);
							} else {
								this.failedProps.push(prop);
								this.removeProperty('photo', property);
								property = undefined;
								console.warn(this.uid()+': Photo removed');
							}
						}
					}
					break;
				}
				return property;
			},
			/* eslint-enable no-console */

			fix: function() {
				this.validate('rev');
				this.validate('version');
				this.validate('prodid');
				return this.failedProps.indexOf('rev') !== -1
					|| this.failedProps.indexOf('prodid') !== -1
					|| this.failedProps.indexOf('version') !== -1;
			}

		});

		if(angular.isDefined(vCard)) {
			angular.extend(this.data, vCard);
			angular.extend(this.props, $filter('vCard2JSON')(this.data.addressData));
		} else {
			angular.extend(this.props, {
				version: [{value: '3.0'}],
				fn: [{value: t('contacts', 'New contact')}]
			});
			this.data.addressData = $filter('JSON2vCard')(this.props);
		}

		var property = this.getProperty('categories');
		if(!property) {
			// categories should always have the same type (an array)
			this.categories([]);
		} else {
			if (angular.isString(property.value)) {
				this.categories([property.value]);
			}
		}
	};
}]);

angular.module('contactsApp')
.factory('AddressBookService', ['DavClient', 'DavService', 'SettingsService', 'AddressBook', '$q', function(DavClient, DavService, SettingsService, AddressBook, $q) {

	var addressBooks = [];
	var loadPromise = undefined;

	var observerCallbacks = [];

	var notifyObservers = function(eventName) {
		var ev = {
			event: eventName,
			addressBooks: addressBooks
		};
		angular.forEach(observerCallbacks, function(callback) {
			callback(ev);
		});
	};

	var loadAll = function() {
		if (addressBooks.length > 0) {
			return $q.when(addressBooks);
		}
		if (_.isUndefined(loadPromise)) {
			loadPromise = DavService.then(function(account) {
				loadPromise = undefined;
				addressBooks = account.addressBooks.map(function(addressBook) {
					return new AddressBook(addressBook);
				});
			});
		}
		return loadPromise;
	};

	return {
		registerObserverCallback: function(callback) {
			observerCallbacks.push(callback);
		},

		getAll: function() {
			return loadAll().then(function() {
				return addressBooks;
			});
		},

		getGroups: function () {
			return this.getAll().then(function(addressBooks) {
				return addressBooks.map(function (element) {
					return element.groups;
				}).reduce(function(a, b) {
					return a.concat(b);
				});
			});
		},

		getDefaultAddressBook: function() {
			return addressBooks[0];
		},

		getAddressBook: function(displayName) {
			return DavService.then(function(account) {
				return DavClient.getAddressBook({displayName:displayName, url:account.homeUrl}).then(function(res) {
					var addressBook = new AddressBook({
						account: account,
						ctag: res[0].props.getctag,
						url: account.homeUrl+displayName+'/',
						data: res[0],
						displayName: res[0].props.displayname,
						resourcetype: res[0].props.resourcetype,
						syncToken: res[0].props.syncToken
					});
					return addressBook;
				});
			});
		},

		create: function(displayName) {
			return DavService.then(function(account) {
				return DavClient.createAddressBook({displayName:displayName, url:account.homeUrl});
			});
		},

		delete: function(addressBook) {
			return DavService.then(function() {
				return DavClient.deleteAddressBook(addressBook).then(function() {
					var index = addressBooks.indexOf(addressBook);
					addressBooks.splice(index, 1);
					notifyObservers('delete');
				});
			});
		},

		rename: function(addressBook, displayName) {
			return DavService.then(function(account) {
				return DavClient.renameAddressBook(addressBook, {displayName:displayName, url:account.homeUrl});
			});
		},

		get: function(displayName) {
			return this.getAll().then(function(addressBooks) {
				return addressBooks.filter(function (element) {
					return element.displayName === displayName;
				})[0];
			});
		},

		sync: function(addressBook) {
			return DavClient.syncAddressBook(addressBook);
		},

		share: function(addressBook, shareType, shareWith, writable, existingShare) {
			var xmlDoc = document.implementation.createDocument('', '', null);
			var oShare = xmlDoc.createElement('o:share');
			oShare.setAttribute('xmlns:d', 'DAV:');
			oShare.setAttribute('xmlns:o', 'http://owncloud.org/ns');
			xmlDoc.appendChild(oShare);

			var oSet = xmlDoc.createElement('o:set');
			oShare.appendChild(oSet);

			var dHref = xmlDoc.createElement('d:href');
			if (shareType === OC.Share.SHARE_TYPE_USER) {
				dHref.textContent = 'principal:principals/users/';
			} else if (shareType === OC.Share.SHARE_TYPE_GROUP) {
				dHref.textContent = 'principal:principals/groups/';
			}
			dHref.textContent += shareWith;
			oSet.appendChild(dHref);

			var oSummary = xmlDoc.createElement('o:summary');
			oSummary.textContent = t('contacts', '{addressbook} shared by {owner}', {
				addressbook: addressBook.displayName,
				owner: addressBook.owner
			});
			oSet.appendChild(oSummary);

			if (writable) {
				var oRW = xmlDoc.createElement('o:read-write');
				oSet.appendChild(oRW);
			}

			var body = oShare.outerHTML;

			return DavClient.xhr.send(
				dav.request.basic({method: 'POST', data: body}),
				addressBook.url
			).then(function(response) {
				if (response.status === 200) {
					if (!existingShare) {
						if (shareType === OC.Share.SHARE_TYPE_USER) {
							addressBook.sharedWith.users.push({
								id: shareWith,
								displayname: shareWith,
								writable: writable
							});
						} else if (shareType === OC.Share.SHARE_TYPE_GROUP) {
							addressBook.sharedWith.groups.push({
								id: shareWith,
								displayname: shareWith,
								writable: writable
							});
						}
					}
				}
			});

		},

		unshare: function(addressBook, shareType, shareWith) {
			var xmlDoc = document.implementation.createDocument('', '', null);
			var oShare = xmlDoc.createElement('o:share');
			oShare.setAttribute('xmlns:d', 'DAV:');
			oShare.setAttribute('xmlns:o', 'http://owncloud.org/ns');
			xmlDoc.appendChild(oShare);

			var oRemove = xmlDoc.createElement('o:remove');
			oShare.appendChild(oRemove);

			var dHref = xmlDoc.createElement('d:href');
			if (shareType === OC.Share.SHARE_TYPE_USER) {
				dHref.textContent = 'principal:principals/users/';
			} else if (shareType === OC.Share.SHARE_TYPE_GROUP) {
				dHref.textContent = 'principal:principals/groups/';
			}
			dHref.textContent += shareWith;
			oRemove.appendChild(dHref);
			var body = oShare.outerHTML;


			return DavClient.xhr.send(
				dav.request.basic({method: 'POST', data: body}),
				addressBook.url
			).then(function(response) {
				if (response.status === 200) {
					if (shareType === OC.Share.SHARE_TYPE_USER) {
						addressBook.sharedWith.users = addressBook.sharedWith.users.filter(function(user) {
							return user.id !== shareWith;
						});
					} else if (shareType === OC.Share.SHARE_TYPE_GROUP) {
						addressBook.sharedWith.groups = addressBook.sharedWith.groups.filter(function(groups) {
							return groups.id !== shareWith;
						});
					}
					//todo - remove entry from addressbook object
					return true;
				} else {
					return false;
				}
			});

		}


	};

}]);

angular.module('contactsApp')
.service('ContactService', ['DavClient', 'AddressBookService', 'Contact', '$q', 'CacheFactory', 'uuid4', function(DavClient, AddressBookService, Contact, $q, CacheFactory, uuid4) {

	var contactService = this;

	var cacheFilled = false;
	var contactsCache = CacheFactory('contacts');
	var observerCallbacks = [];
	var loadPromise = undefined;

	this.registerObserverCallback = function(callback) {
		observerCallbacks.push(callback);
	};

	var notifyObservers = function(eventName, uid) {
		var ev = {
			event: eventName,
			uid: uid,
			contacts: contactsCache.values()
		};
		angular.forEach(observerCallbacks, function(callback) {
			callback(ev);
		});
	};

	this.getFullContacts = function(contacts) {
		AddressBookService.getAll().then(function (enabledAddressBooks) {
			var promises = [];
			var xhrAddressBooks = [];
			contacts.forEach(function (contact) {
				// Regroup urls by addressbooks
				if(enabledAddressBooks.indexOf(contact.data.addressBook) !== -1) {
					// Initiate array if no exists
					xhrAddressBooks[contact.addressBookId] = xhrAddressBooks[contact.addressBookId] || [];
					xhrAddressBooks[contact.addressBookId].push(contact.data.url);
				}
			});
			// Get our full vCards
			enabledAddressBooks.forEach(function(addressBook) {
				if(angular.isArray(xhrAddressBooks[addressBook.displayName])) {
					var promise = DavClient.getContacts(addressBook, {}, xhrAddressBooks[addressBook.displayName]).then(
						function (vcards) {
							return vcards.map(function (vcard) {
								return new Contact(addressBook, vcard);
							});
						}).then(function (contacts_) {
							contacts_.map(function (contact) {
								// Validate some fields
								if(contact.fix()) {
									// Can't use this in those nested functions
									contactService.update(contact);
								}
								contactsCache.put(contact.uid(), contact);
							});
						});
					promises.push(promise);
				}
			});
			$q.all(promises).then(function () {
				notifyObservers('getFullContacts', '');
			});
		});
	};

	this.fillCache = function() {
		if (_.isUndefined(loadPromise)) {
			loadPromise = AddressBookService.getAll().then(function (enabledAddressBooks) {
				var promises = [];
				enabledAddressBooks.forEach(function (addressBook) {
					promises.push(
						AddressBookService.sync(addressBook).then(function (addressBook) {
							addressBook.objects.forEach(function(vcard) {
								try {
									var contact = new Contact(addressBook, vcard);
									contactsCache.put(contact.uid(), contact);
								} catch(error) {
									// eslint-disable-next-line no-console
									console.log('Invalid contact received: ', vcard);
								}
							});
						})
					);
				});
				return $q.all(promises).then(function () {
					cacheFilled = true;
				});
			});
		}
		return loadPromise;
	};

	this.getAll = function() {
		if(cacheFilled === false) {
			return this.fillCache().then(function() {
				return contactsCache.values();
			});
		} else {
			return $q.when(contactsCache.values());
		}
	};

	// get list of groups and the count of contacts in said groups
	this.getGroupList = function () {
		return this.getAll().then(function(contacts) {
			// the translated names for all and not-grouped are used in filtering, they must be exactly like this
			var allContacts = [t('contacts', 'All contacts'), contacts.length];
			var notGrouped =
				[t('contacts', 'Not grouped'),
					contacts.filter(
						function (contact) {
							 return contact.categories().length === 0;
						}).length
				];

			// allow groups with names such as toString
			var otherGroups = Object.create(null);

			// collect categories and their associated counts
			contacts.forEach(function (contact) {
				contact.categories().forEach(function (category) {
					otherGroups[category] = otherGroups[category] ? otherGroups[category] + 1 : 1;
				});
			});

			var priorityGroups = [allContacts];
			// Only have Not Grouped if at least one contact in it
			if(notGrouped[1] !== 0) {
				priorityGroups.push(notGrouped);
			}

			return priorityGroups.concat(_.keys(otherGroups).map(
					function (key) {
						return [key, otherGroups[key]];
					}));


		});
	};

	this.getGroups = function () {
		return this.getAll().then(function(contacts) {
			return _.uniq(contacts.map(function (element) {
				return element.categories();
			}).reduce(function(a, b) {
				return a.concat(b);
			}, []).sort(), true);
		});
	};

	this.getById = function(addressBooks, uid) {
		return (function () {
			if(cacheFilled === false) {
				return this.fillCache().then(function() {
					return contactsCache.get(uid);
				});
			} else {
				return $q.when(contactsCache.get(uid));
			}
		}).call(this)
			.then(function (contact) {
				if(angular.isUndefined(contact)) {
					OC.Notification.showTemporary(t('contacts', 'Contact not found.'));
					return;
				} else {
					var addressBook = _.find(addressBooks, function(book) {
						return book.displayName === contact.addressBookId;
					});
					return addressBook
						? DavClient.getContacts(addressBook, {}, [ contact.data.url ]).then(
							function (vcards) { return new Contact(addressBook, vcards[0]); }
						).then(function (contact) {
							contactsCache.put(contact.uid(), contact);
							notifyObservers('getFullContacts', contact.uid());
							return contact;
						}) : contact;
				}
			});
	};

	this.create = function(newContact, addressBook, uid, fromImport) {
		addressBook = addressBook || AddressBookService.getDefaultAddressBook();
		if(addressBook.readOnly) {
			OC.Notification.showTemporary(t('contacts', 'You don\'t have permission to write to this addressbook.'));
			return;
		}
		try {
			newContact = newContact || new Contact(addressBook);
		} catch(error) {
			OC.Notification.showTemporary(t('contacts', 'Contact could not be created.'));
			return;
		}
		var newUid = '';
		if(uuid4.validate(uid)) {
			newUid = uid;
		} else {
			newUid = uuid4.generate();
		}
		newContact.uid(newUid);
		newContact.setUrl(addressBook, newUid);
		newContact.addressBookId = addressBook.displayName;
		if (_.isUndefined(newContact.fullName()) || newContact.fullName() === '') {
			newContact.fullName(newContact.displayName());
		}

		return DavClient.createCard(
			addressBook,
			{
				data: newContact.data.addressData,
				filename: newUid + '.vcf'
			}
		).then(function(xhr) {
			newContact.setETag(xhr.getResponseHeader('ETag'));
			contactsCache.put(newUid, newContact);
			if (fromImport !== true) {
				notifyObservers('create', newUid);
				$('#details-fullName').select();
			}
			return newContact;
		}).catch(function() {
			OC.Notification.showTemporary(t('contacts', 'Contact could not be created.'));
			return false;
		});
	};

	this.import = function(data, type, addressBook, progressCallback) {
		addressBook = addressBook || AddressBookService.getDefaultAddressBook();

		var regexp = /BEGIN:VCARD[\s\S]*?END:VCARD/mgi;
		var singleVCards = data.match(regexp);

		if (!singleVCards) {
			OC.Notification.showTemporary(t('contacts', 'No contacts in file. Only vCard files are allowed.'));
			if (progressCallback) {
				progressCallback(1);
			}
			return;
		}

		notifyObservers('importstart');

		var num = 1;
		for(var i in singleVCards) {
			var newContact = new Contact(addressBook, {addressData: singleVCards[i]});
			if (['3.0', '4.0'].indexOf(newContact.version()) < 0) {
				if (progressCallback) {
					progressCallback(num / singleVCards.length);
				}
				OC.Notification.showTemporary(t('contacts', 'Only vCard version 4.0 (RFC6350) or version 3.0 (RFC2426) are supported.'));
				num++;
				continue;
			}
			// eslint-disable-next-line no-loop-func
			this.create(newContact, addressBook, '', true).then(function(xhrContact) {
				if (xhrContact !== false) {
					var xhrContactName = xhrContact.displayName();
				}
				// Update the progress indicator
				if (progressCallback) {
					progressCallback(num / singleVCards.length, xhrContactName);
				}
				num++;
				/* Import is over, let's notify */
				if (num === singleVCards.length + 1) {
					notifyObservers('importend');
				}
			});
		}
	};

	this.moveContact = function (contact, addressBook) {
		if (contact.addressBookId === addressBook.displayName) {
			return;
		}
		if(addressBook.readOnly) {
			OC.Notification.showTemporary(t('contacts', 'You don\'t have permission to write to this addressbook.'));
			return;
		}
		contact.syncVCard();
		var uid = contact.uid();

		// Delete on server
		DavClient.deleteCard(contact.data).then(function() {
			// Create new on server
			DavClient.createCard(
				addressBook,
				{
					data: contact.data.addressData,
					filename: uid + '.vcf'
				}
			).then(function(xhr) {
				// Edit local cached contact
				contact.setETag(xhr.getResponseHeader('ETag'));
				contact.setAddressBook(addressBook);
			});
		});
	};

	this.update = function(contact) {
		// update rev field
		contact.syncVCard();

		// update contact on server
		return DavClient.updateCard(contact.data, {json: true}).then(function(xhr) {
			var newEtag = xhr.getResponseHeader('ETag');
			contact.setETag(newEtag);
			notifyObservers('update', contact.uid());
		}).catch(function() {
			OC.Notification.showTemporary(t('contacts', 'Contact could not be saved.'));
		});
	};

	this.delete = function(contact) {
		// delete contact from server
		return DavClient.deleteCard(contact.data).then(function() {
			contactsCache.remove(contact.uid());
			notifyObservers('delete', contact.uid());
		});
	};

	this.updateDeletedAddressbook = function(callback) {
		// Delete contacts which addressbook has been removed from cache
		AddressBookService.getAll().then(function (enabledAddressBooks) {
			var addressBooksIds = [];
			angular.forEach(enabledAddressBooks, function(addressBook) {
				addressBooksIds.push(addressBook.displayName);
			});
			angular.forEach(contactsCache.values(), function(contact) {
				if (addressBooksIds.indexOf(contact.addressBookId) === -1) {
					contactsCache.remove(contact.uid());
				}
			});
			callback();
			notifyObservers('groupsUpdate');
		});
	};

}]);

angular.module('contactsApp')
.service('DavClient', function() {
	var xhr = new dav.transport.Basic(
		new dav.Credentials()
	);
	return new dav.Client(xhr);
});

angular.module('contactsApp')
.service('DavService', ['DavClient', function(DavClient) {
	return DavClient.createAccount({
		server: OC.linkToRemote('dav/addressbooks'),
		accountType: 'carddav',
		useProvidedPath: true
	});
}]);

angular.module('contactsApp')
.service('ImportService', function() {

	this.importing = false;
	this.selectedAddressBook = t('contacts', 'Import into');
	this.importedUser = t('contacts', 'Waiting for the server to be ready...');
	this.importPercent = 0;

	this.t = {
		importText : t('contacts', 'Import into'),
		importingText : t('contacts', 'Importing...')
	};

});

angular.module('contactsApp')
	.service('MimeService', function() {
		var magicNumbers = {
			'/9j/' : 'JPEG',
			'R0lGOD' : 'GIF',
			'iVBORw0KGgo' : 'PNG'
		};

		this.b64mime = function(b64string) {
			for (var mn in magicNumbers) {
				if(b64string.startsWith(mn)) return magicNumbers[mn];
			}
			return null;
		};
	});

angular.module('contactsApp')
.service('SearchService', function() {
	var searchTerm = '';

	var observerCallbacks = [];

	this.registerObserverCallback = function(callback) {
		observerCallbacks.push(callback);
	};

	var notifyObservers = function(eventName) {
		var ev = {
			event:eventName,
			searchTerm:searchTerm
		};
		angular.forEach(observerCallbacks, function(callback) {
			callback(ev);
		});
	};

	var SearchProxy = {
		attach: function(search) {
			search.setFilter('contacts', this.filterProxy);
		},
		filterProxy: function(query) {
			searchTerm = query;
			notifyObservers('changeSearch');
		}
	};

	this.getSearchTerm = function() {
		return searchTerm;
	};

	this.cleanSearch = function() {
		if (!_.isUndefined($('.searchbox'))) {
			$('.searchbox')[0].reset();
		}
		searchTerm = '';
	};

	if (!_.isUndefined(OC.Plugins)) {
		OC.Plugins.register('OCA.Search', SearchProxy);
		if (!_.isUndefined(OCA.Search)) {
			OC.Search = new OCA.Search($('#searchbox'), $('#searchresults'));
			$('#searchbox').show();
		}
	}

	if (!_.isUndefined($('.searchbox'))) {
		$('.searchbox')[0].addEventListener('keypress', function(e) {
			if(e.keyCode === 13) {
				notifyObservers('submitSearch');
			}
		});
	}
});

angular.module('contactsApp')
.service('SettingsService', function() {
	var settings = {
		addressBooks: [
			'testAddr'
		]
	};

	this.set = function(key, value) {
		settings[key] = value;
	};

	this.get = function(key) {
		return settings[key];
	};

	this.getAll = function() {
		return settings;
	};
});

angular.module('contactsApp')
.service('SortByService', function () {
	var subscriptions = [];
	var sortBy = 'sortDisplayName';

	var defaultOrder = window.localStorage.getItem('contacts_default_order');
	if (defaultOrder) {
		sortBy = defaultOrder;
	}

	function notifyObservers () {
		angular.forEach(subscriptions, function (subscription) {
			if (typeof subscription === 'function') {
				subscription(sortBy);
			}
		});
	}

	return {
		subscribe: function (callback) {
			subscriptions.push (callback);
		},
		setSortBy: function (value) {
			sortBy = value;
			window.localStorage.setItem ('contacts_default_order', value);
			notifyObservers ();
		},
		getSortBy: function () {
			return sortBy;
		},
		getSortByList: function () {
			return {
				sortDisplayName: t('contacts', 'Display name'),
				sortFirstName: t('contacts', 'First name'),
				sortLastName: t('contacts', 'Last name')
			};
		}
	};
});

angular.module('contactsApp')
.service('vCardPropertiesService', function() {
	/**
	 * map vCard attributes to internal attributes
	 *
	 * propName: {
	 * 		multiple: [Boolean], // is this prop allowed more than once? (default = false)
	 * 		readableName: [String], // internationalized readable name of prop
	 * 		template: [String], // template name found in /templates/detailItems
	 * 		[...] // optional additional information which might get used by the template
	 * }
	 */
	this.vCardMeta = {
		nickname: {
			readableName: t('contacts', 'Nickname'),
			template: 'text'
		},
		n: {
			readableName: t('contacts', 'Detailed name'),
			defaultValue: {
				value:['', '', '', '', '']
			},
			template: 'n'
		},
		note: {
			readableName: t('contacts', 'Notes'),
			template: 'textarea'
		},
		url: {
			multiple: true,
			readableName: t('contacts', 'Website'),
			template: 'url'
		},
		cloud: {
			multiple: true,
			readableName: t('contacts', 'Federated Cloud ID'),
			template: 'text',
			defaultValue: {
				value:[''],
				meta:{type:['HOME']}
			},
			options: [
				{id: 'HOME', name: t('contacts', 'Home')},
				{id: 'WORK', name: t('contacts', 'Work')},
				{id: 'OTHER', name: t('contacts', 'Other')}
			]		},
		adr: {
			multiple: true,
			readableName: t('contacts', 'Address'),
			template: 'adr',
			defaultValue: {
				value:['', '', '', '', '', '', ''],
				meta:{type:['HOME']}
			},
			options: [
				{id: 'HOME', name: t('contacts', 'Home')},
				{id: 'WORK', name: t('contacts', 'Work')},
				{id: 'OTHER', name: t('contacts', 'Other')}
			]
		},
		categories: {
			readableName: t('contacts', 'Groups'),
			template: 'groups'
		},
		bday: {
			readableName: t('contacts', 'Birthday'),
			template: 'date'
		},
		anniversary: {
			readableName: t('contacts', 'Anniversary'),
			template: 'date'
		},
		deathdate: {
			readableName: t('contacts', 'Date of death'),
			template: 'date'
		},
		email: {
			multiple: true,
			readableName: t('contacts', 'Email'),
			template: 'email',
			defaultValue: {
				value:'',
				meta:{type:['HOME']}
			},
			options: [
				{id: 'HOME', name: t('contacts', 'Home')},
				{id: 'WORK', name: t('contacts', 'Work')},
				{id: 'OTHER', name: t('contacts', 'Other')}
			]
		},
		impp: {
			multiple: true,
			readableName: t('contacts', 'Instant messaging'),
			template: 'username',
			defaultValue: {
				value:[''],
				meta:{type:['SKYPE']}
			},
			options: [
				{id: 'IRC', name: 'IRC'},
				{id: 'KIK', name: 'KiK'},
				{id: 'SKYPE', name: 'Skype'},
				{id: 'TELEGRAM', name: 'Telegram'},
				{id: 'XMPP', name:'XMPP'}
			]
		},
		tel: {
			multiple: true,
			readableName: t('contacts', 'Phone'),
			template: 'tel',
			defaultValue: {
				value:'',
				meta:{type:['HOME,VOICE']}
			},
			options: [
				{id: 'HOME,VOICE', name: t('contacts', 'Home')},
				{id: 'WORK,VOICE', name: t('contacts', 'Work')},
				{id: 'CELL', name: t('contacts', 'Mobile')},
				{id: 'FAX', name: t('contacts', 'Fax')},
				{id: 'HOME,FAX', name: t('contacts', 'Fax home')},
				{id: 'WORK,FAX', name: t('contacts', 'Fax work')},
				{id: 'PAGER', name: t('contacts', 'Pager')},
				{id: 'VOICE', name: t('contacts', 'Voice')}
			]
		},
		'X-SOCIALPROFILE': {
			multiple: true,
			readableName: t('contacts', 'Social network'),
			template: 'username',
			defaultValue: {
				value:[''],
				meta:{type:['facebook']}
			},
			options: [
				{id: 'FACEBOOK', name: 'Facebook'},
				{id: 'GITHUB', name: 'GitHub'},
				{id: 'GOOGLEPLUS', name: 'Google+'},
				{id: 'INSTAGRAM', name: 'Instagram'},
				{id: 'LINKEDIN', name: 'LinkedIn'},
				{id: 'PINTEREST', name: 'Pinterest'},
				{id: 'QZONE', name: 'QZone'},
				{id: 'TUMBLR', name: 'Tumblr'},
				{id: 'TWITTER', name: 'Twitter'},
				{id: 'WECHAT', name: 'WeChat'},
				{id: 'YOUTUBE', name: 'YouTube'}


			]
		},
		relationship: {
			readableName: t('contacts', 'Relationship'),
			template: 'select',
			options: [
				{id: 'SPOUSE', name: t('contacts', 'Spouse')},
				{id: 'CHILD', name: t('contacts', 'Child')},
				{id: 'MOTHER', name: t('contacts', 'Mother')},
				{id: 'FATHER', name: t('contacts', 'Father')},
				{id: 'PARENT', name: t('contacts', 'Parent')},
				{id: 'BROTHER', name: t('contacts', 'Brother')},
				{id: 'SISTER', name: t('contacts', 'Sister')},
				{id: 'RELATIVE', name: t('contacts', 'Relative')},
				{id: 'FRIEND', name: t('contacts', 'Friend')},
				{id: 'COLLEAGUE', name: t('contacts', 'Colleague')},
				{id: 'MANAGER', name: t('contacts', 'Manager')},
				{id: 'ASSISTANT', name: t('contacts', 'Assistant')},
			]
		},
		gender: {
			readableName: t('contacts', 'Gender'),
			template: 'select',
			options: [
				{id: 'F', name: t('contacts', 'Female')},
				{id: 'M', name: t('contacts', 'Male')},
				{id: 'O', name: t('contacts', 'Other')}
			]
		}
	};

	this.fieldOrder = [
		'org',
		'title',
		'tel',
		'email',
		'adr',
		'impp',
		'nick',
		'bday',
		'anniversary',
		'deathdate',
		'url',
		'X-SOCIALPROFILE',
		'relationship',
		'note',
		'categories',
		'role',
		'gender'
	];

	this.fieldDefinitions = [];
	for (var prop in this.vCardMeta) {
		this.fieldDefinitions.push({id: prop, name: this.vCardMeta[prop].readableName, multiple: !!this.vCardMeta[prop].multiple});
	}

	this.fallbackMeta = function(property) {
		function capitalize(string) { return string.charAt(0).toUpperCase() + string.slice(1); }
		return {
			name: 'unknown-' + property,
			readableName: capitalize(property),
			template: 'hidden',
			necessity: 'optional'
		};
	};

	this.getMeta = function(property) {
		return this.vCardMeta[property] || this.fallbackMeta(property);
	};

});

angular.module('contactsApp')
.filter('JSON2vCard', function() {
	return function(input) {
		return vCard.generate(input);
	};
});

angular.module('contactsApp')
.filter('contactColor', function() {
	return function(input) {
		// Check if core has the new color generator
		if(typeof input.toHsl === 'function') {
			var hsl = input.toHsl();
			return 'hsl('+hsl[0]+', '+hsl[1]+'%, '+hsl[2]+'%)';
		} else {
			// If not, we use the old one
			/* global md5 */
			var hash = md5(input).substring(0, 4),
				maxRange = parseInt('ffff', 16),
				hue = parseInt(hash, 16) / maxRange * 256;
			return 'hsl(' + hue + ', 90%, 65%)';
		}
	};
});
angular.module('contactsApp')
.filter('contactGroupFilter', function() {
	'use strict';
	return function (contacts, group) {
		if (typeof contacts === 'undefined') {
			return contacts;
		}
		if (typeof group === 'undefined' || group.toLowerCase() === t('contacts', 'All contacts').toLowerCase()) {
			return contacts;
		}
		var filter = [];
		if (contacts.length > 0) {
			for (var i = 0; i < contacts.length; i++) {
				if (group.toLowerCase() === t('contacts', 'Not grouped').toLowerCase()) {
					if (contacts[i].categories().length === 0) {
						filter.push(contacts[i]);
					}
				} else {
					if (contacts[i].categories().indexOf(group) >= 0) {
						filter.push(contacts[i]);
					}
				}
			}
		}
		return filter;
	};
});

// from https://docs.nextcloud.com/server/11/developer_manual/app/css.html#menus
angular.module('contactsApp')
.filter('counterFormatter', function () {
	'use strict';
	return function (count) {
		if (count > 999) {
			return '999+';
		}
		if (count === 0) {
			return '';
		}
		return count;
	};
});


angular.module('contactsApp')
.filter('fieldFilter', function() {
	'use strict';
	return function (fields, contact) {
		if (typeof fields === 'undefined') {
			return fields;
		}
		if (typeof contact === 'undefined') {
			return fields;
		}
		var filter = [];
		if (fields.length > 0) {
			for (var i = 0; i < fields.length; i++) {
				if (fields[i].multiple ) {
					filter.push(fields[i]);
					continue;
				}
				if (_.isUndefined(contact.getProperty(fields[i].id))) {
					filter.push(fields[i]);
				}
			}
		}
		return filter;
	};
});

angular.module('contactsApp')
.filter('firstCharacter', function() {
	return function(input) {
		return input.charAt(0);
	};
});

angular.module('contactsApp')
.filter('localeOrderBy', [function () {
	return function (array, sortPredicate, reverseOrder) {
		if (!Array.isArray(array)) return array;
		if (!sortPredicate) return array;

		var arrayCopy = [];
		angular.forEach(array, function (item) {
			arrayCopy.push(item);
		});

		arrayCopy.sort(function (a, b) {
			var valueA = a[sortPredicate];
			if (angular.isFunction(valueA)) {
				valueA = a[sortPredicate]();
			}
			var valueB = b[sortPredicate];
			if (angular.isFunction(valueB)) {
				valueB = b[sortPredicate]();
			}

			if (angular.isString(valueA)) {
				return !reverseOrder ? valueA.localeCompare(valueB) : valueB.localeCompare(valueA);
			}

			if (angular.isNumber(valueA) || typeof valueA === 'boolean') {
				return !reverseOrder ? valueA - valueB : valueB - valueA;
			}

			if (angular.isArray(valueA)) {
				if (valueA[0] === valueB[0]) {
					return !reverseOrder ? valueA[1].localeCompare(valueB[1]) : valueB[1].localeCompare(valueA[1]);
				}
				return !reverseOrder ? valueA[0].localeCompare(valueB[0]) : valueB[0].localeCompare(valueA[0]);
			}

			return 0;
		});

		return arrayCopy;
	};
}]);

angular.module('contactsApp')
.filter('newContact', function() {
	return function(input) {
		return input !== '' ? input : t('contacts', 'New contact');
	};
});

angular.module('contactsApp')
.filter('orderDetailItems', ['vCardPropertiesService', function(vCardPropertiesService) {
	'use strict';
	return function(items, field, reverse) {

		var filtered = [];
		angular.forEach(items, function(item) {
			filtered.push(item);
		});

		var fieldOrder = angular.copy(vCardPropertiesService.fieldOrder);
		// reverse to move custom items to the end (indexOf == -1)
		fieldOrder.reverse();

		filtered.sort(function (a, b) {
			if(fieldOrder.indexOf(a[field]) < fieldOrder.indexOf(b[field])) {
				return 1;
			}
			if(fieldOrder.indexOf(a[field]) > fieldOrder.indexOf(b[field])) {
				return -1;
			}
			return 0;
		});

		if(reverse) filtered.reverse();
		return filtered;
	};
}]);

angular.module('contactsApp')
.filter('toArray', function() {
	return function(obj) {
		if (!(obj instanceof Object)) return obj;
		return _.map(obj, function(val, key) {
			return Object.defineProperty(val, '$key', {value: key});
		});
	};
});

angular.module('contactsApp')
.filter('vCard2JSON', function() {
	return function(input) {
		return vCard.parse(input);
	};
});

//# sourceMappingURL=data:application/json;charset=utf8;base64,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
