package Net::DNS::SEC::Private;

#
# $Id: Private.pm 1397 2015-09-15 18:45:17Z willem $
#
use vars qw($VERSION);
$VERSION = (qw$LastChangedRevision: 1397 $)[1];


=head1 NAME

Net::DNS::SEC::Private - DNSSEC Private key object


=head1 SYNOPSIS

    use Net::DNS::SEC::Private;

    $private = new Net::DNS::SEC::Private( $keypath );

    $private = new Net::DNS::SEC::Private(
	'algorithm'  => '13',
	'keytag'     => '26512',
	'privatekey' => 'h/mc+iq9VDUbNAjQgi8S8JzlEX29IALchwJmNM3QYKk=',
	'signame'    => 'example.com.'
	);


=head1 DESCRIPTION

Class representing private keys as read from a keyfile generated by BIND
dnssec-keygen. The class is written to be used only in the context of the
Net::DNS::RR::RRSIG create method. This class is not designed to interact
with any other system.

=cut


use strict;
use integer;
use warnings;
use Carp;
use File::Spec;
use FileHandle;
use MIME::Base64;


sub new { scalar(@_) > 2 ? &_new_params : &_new_keyfile }

sub _new_keyfile {
	my ( $class, $file ) = @_;

	my ( $vol, $dir, $keyname ) = File::Spec->splitpath($file);

	# Format something like: /Kbla.foo.+001+60114.private'
	# assuming proper file name.
	# We determine the algorithm from the filename.
	croak "$keyname does not appear to be a private key"
			unless $keyname =~ /^K(.*\.)\+(\d+)\+(\d+)\.private$/;
	my @identifier = ( signame => $1, algorithm => 0 + $2, keytag => $3 );

	my $handle = new FileHandle($file) or croak qq(open: "$file" $!);

	my @content;
	local $_;
	while (<$handle>) {
		chomp;
		next if /^$/;
		next if /^\s*[;]/;
		s/\(.+\)//;
		my ( $name, $value ) = split;
		push @content, $name, $value;
	}

	return $class->_new_params( @content, @identifier );
}


sub _new_params {
	my $class = shift;
	my $param = {};
	while (@_) {
		my $name = shift @_;
		$name =~ tr/A-Za-z0-9\000-\377/a-za-z0-9/d;
		$param->{$name} = shift @_;
	}

	my $self = bless sub { $param->{shift()} }, $class;
	croak 'algorithm not specified' unless defined $self->algorithm;
	croak 'signame not specified'	unless defined $self->signame;
	return $self;
}


sub generate_rsa {
	my ( $class, $domain, $flags, $keysize, $entropy, $algcode ) = @_;

	$keysize ||= 1024;

	require Crypt::OpenSSL::RSA;

	if ( defined($entropy) ) {
		require Crypt::OpenSSL::Random;
		Crypt::OpenSSL::Random::random_seed($entropy);
		Crypt::OpenSSL::RSA->import_random_seed();
	}

	my $rsa = Crypt::OpenSSL::RSA->generate_key($keysize);

	my $keyblob = $rsa->get_private_key_string;

	return $class->new_rsa_priv( $keyblob, $domain, $flags, $algcode );
}


sub new_rsa_priv {
	my $class   = shift;
	my $keyblob = shift;
	my $domain  = shift;
	my $flags   = shift || 256;
	my $algcode = shift || 5;

	require Crypt::OpenSSL::RSA;

	my $rsa = Crypt::OpenSSL::RSA->new_private_key($keyblob);

	my @params = map $_->to_bin, $rsa->get_key_parameters;
	my @base64 = map encode_base64( $_, '' ), @params;	# private key

	my ( $modulus, $exponent ) = @params;			# public key
	my $explen = length $exponent;
	# my $format = $explen < 256 ? 'C a* a*' : 'xn a* a*';	# RFC3110, section 2
	my $format = 'C a* a*';					# real world
	my $keybin = pack $format, $explen, $exponent, $modulus;

	my $dnskey = new Net::DNS::RR(
		name	  => $domain,
		type	  => 'DNSKEY',
		algorithm => $algcode,
		flags	  => $flags,
		keybin	  => $keybin,
		);

	my $algorithm = $dnskey->algorithm;
	my $mnemonic  = $dnskey->algorithm('MNEMONIC');
	my $keytag    = $dnskey->keytag;

	my $private_key = <<"END";
Private-key-format: v1.2
Algorithm: $algorithm ($mnemonic)
Modulus: $base64[0]
PublicExponent: $base64[1]
PrivateExponent: $base64[2]
Prime1: $base64[3]
Prime2: $base64[4]
Exponent1: $base64[5]
Exponent2: $base64[6]
Coefficient: $base64[7]
END
	return $class->_new_params(
		algorithm	     => $algorithm,
		keytag		     => $keytag,
		signame		     => $dnskey->signame,
		privatekeyname	     => $dnskey->privatekeyname,
		flags		     => $dnskey->flags,
		Modulus		     => $base64[0],
		PublicExponent	     => $base64[1],
		PrivateExponent	     => $base64[2],
		Prime1		     => $base64[3],
		Prime2		     => $base64[4],
		Exponent1	     => $base64[5],
		Exponent2	     => $base64[6],
		Coefficient	     => $base64[7],
		dump_rsa_priv	     => $private_key,
		dump_rsa_pub	     => $dnskey->key,
		dump_rsa_keytag	     => $keytag,
		dump_rsa_private_pem => $rsa->get_private_key_string,
		dump_rsa_private_der => $rsa->get_private_key_string,	# historical
		);
}


use vars qw($AUTOLOAD);

sub AUTOLOAD {				## Default method
	my ($self) = @_;
	no strict q/refs/;

	my ($attribute) = $AUTOLOAD =~ m/::([^:]*)$/;
	$attribute =~ tr/A-Za-z0-9\000-\377/a-za-z0-9/d;

	# Build a method in the class
	*{$AUTOLOAD} = sub {
		my $self = shift;
		&$self($attribute);
	};

carp <<"END" if $attribute =~ /^dumprsa/;
Warning: $AUTOLOAD is deprecated
and may be removed in a future release
END
	# and jump to it
	goto &{$AUTOLOAD};
}


1;
__END__


=head1 METHODS

=head2 new (from private keyfile)

    $keypath = '/home/foo/Kexample.com.+013+26512.private';
    $private = new Net::DNS::SEC::Private( $keypath );

The argument is the full path to a private key file generated by the
BIND dnssec-keygen tool.  Note that the filename contains information
about the algorithm and keytag.


=head2 new (from private key parameters)

    $private = new Net::DNS::SEC::Private(
	'algorithm'  => '13',
	'keytag'     => '26512',
	'privatekey' => 'h/mc+iq9VDUbNAjQgi8S8JzlEX29IALchwJmNM3QYKk=',
	'signame'    => 'example.com.'
	);

The arguments define the private key parameters as (name,value) pairs.
The name and data representation are identical to that used in a BIND
private keyfile.


=head2 private_key_format

    $format = $private->private_key_format;

Returns a string which identifies the format of the private key file.


=head2 algorithm, keytag, signame
 
    $algorithm = $private->algorithm;
    $keytag    = $private->keytag;
    $signame   = $private->signame;

Returns the corresponding attribute determined from the filename.


=head2 Private key attributes

    $attribute = $private->attribute;

Returns the value as it appears in the private key file.
The attribute names correspond to the tag in the key file, modified
to form an acceptable Perl subroutine name.


=head2 created, publish, activate

    $created  = $private->created;
    $publish  = $private->publish;
    $activate = $private->activate;

Returns a string which represents a date in the form 20141212123456.
Returns undefined value for key formats older than v1.3.


=head1 RSA SPECIFIC HELPER FUNCTIONS

These functions may be useful to generate RSA private keys and
import PEM format RSA private keys.

=head2 new_rsa_priv

    $private = Net::DNS::SEC::Private->new_rsa_priv( $keyblob,
						     $domain,
						     $flag,
						     $algorithm
						);

Constructor method which creates a Net::DNS::SEC::Private object from
the supplied PEM keyblob.

The second argument specifies the domain name for which this key will
be used.

The flag argument should be either 257 or 256 for SEP and non-SEP key
respectively.

The keyblob should include the -----BEGIN...----- and -----END...----- lines.
The padding is set to PKCS1_OAEP.


=head2 dump_rsa_priv

    $BIND_private_key = $private->dump_rsa_priv();
  
Returns the content of a BIND private keyfile (Private-key-format: v1.2).


=head2 dump_rsa_pub

    $public_key = $private->dump_rsa_pub();

Returns the public key field of the DNSKEY resource record.


=head2 dump_rsa_keytag
    
    $keytag = $private->dump_rsa_keytag();

Returns the keytag field of the DNSKEY resource record.


=head2 dump_rsa_private_pem

    $keyblob = $private->dump_rsa_private_pem();

Return the PEM-encoded representation of the private key.
(Same format that can be read with the new_rsa_priv method.)


=head2 generate_rsa

    $newkey = Net::DNS::SEC::Private->generate_rsa( "example.com",
					256, 1024, $random, $algorithm );
    print $newkey->dump_rsa_priv();
    print $newkey->dump_rsa_pub();


Uses Crypt::OpenSSL::RSA generate_key to create a keypair.

The first argument is the name of the key.

The flag field takes the value of 257 for key-signing keys and ther
value of 256 for zone signing keys.

The 3rd argument is the keysize (default 1024).

The 4th argument, if defined, is passed to the Crypt::OpenSSL::Random
random_seed method (see Crypt::OpenSSL::RSA for details), not needed
with a proper /dev/random.

The 5th argument specifies the algorithm if not RSASHA1 (the default).


=head1 COPYRIGHT

Copyright (c)2014 Dick Franks

All Rights Reserved


=head1 LICENSE

Permission to use, copy, modify, and distribute this software and its
documentation for any purpose and without fee is hereby granted, provided
that the above copyright notice appear in all copies and that both that
copyright notice and this permission notice appear in supporting
documentation, and that the name of the author not be used in advertising
or publicity pertaining to distribution of the software without specific
prior written permission.

THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR
IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY,
FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT. IN NO EVENT SHALL
THE AUTHORS OR COPYRIGHT HOLDERS BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER
LIABILITY, WHETHER IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING
FROM, OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER
DEALINGS IN THE SOFTWARE.


=head1 SEE ALSO

L<perl>, L<Net::DNS>, L<Net::DNS::SEC>,
L<Net::DNS::RR::DNSKEY>, L<Net::DNS::RR::KEY>,
L<Net::DNS::RR::RRSIG>, L<Net::DNS::RR::SIG>

=cut

