# Licensed under the Apache License, Version 2.0 (the "License"); you may
# not use this file except in compliance with the License. You may obtain
# a copy of the License at
#
#      http://www.apache.org/licenses/LICENSE-2.0
#
# Unless required by applicable law or agreed to in writing, software
# distributed under the License is distributed on an "AS IS" BASIS, WITHOUT
# WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied. See the
# License for the specific language governing permissions and limitations
# under the License.

from openstack.load_balancer import load_balancer_service as lb_service
from openstack import resource2 as resource


class Listener(resource.Resource):
    resource_key = 'listener'
    resources_key = 'listeners'
    base_path = '/v2.0/lbaas/listeners'
    service = lb_service.LoadBalancerService()

    # capabilities
    allow_create = True
    allow_get = True
    allow_update = True
    allow_delete = True
    allow_list = True

    _query_mapping = resource.QueryParameters(
        'connection_limit', 'default_pool_id', 'default_tls_container_ref',
        'description', 'name', 'project_id', 'protocol', 'protocol_port',
        'created_at', 'updated_at', 'provisioning_status', 'operating_status',
        'sni_container_refs', 'insert_headers', 'load_balancer_id',
        is_admin_state_up='admin_state_up',
    )

    # Properties
    #: The maximum number of connections permitted for this load balancer.
    #: Default is infinite.
    connection_limit = resource.Body('connection_limit')
    #: Timestamp when the listener was created.
    created_at = resource.Body('created_at')
    #: Default pool to which the requests will be routed.
    default_pool = resource.Body('default_pool')
    #: ID of default pool. Must have compatible protocol with listener.
    default_pool_id = resource.Body('default_pool_id')
    #: A reference to a container of TLS secrets.
    default_tls_container_ref = resource.Body('default_tls_container_ref')
    #: Description for the listener.
    description = resource.Body('description')
    #: Dictionary of additional headers insertion into HTTP header.
    insert_headers = resource.Body('insert_headers', type=dict)
    #: The administrative state of the listener, which is up
    #: ``True`` or down ``False``. *Type: bool*
    is_admin_state_up = resource.Body('admin_state_up', type=bool)
    #: List of l7policies associated with this listener.
    l7_policies = resource.Body('l7policies', type=list)
    #: The ID of the parent load balancer.
    load_balancer_id = resource.Body('loadbalancer_id')
    #: List of load balancers associated with this listener.
    #: *Type: list of dicts which contain the load balancer IDs*
    load_balancers = resource.Body('loadbalancers', type=list)
    #: Name of the listener
    name = resource.Body('name')
    #: Operating status of the listener.
    operating_status = resource.Body('operating_status')
    #: The ID of the project this listener is associated with.
    project_id = resource.Body('project_id')
    #: The protocol of the listener, which is TCP, HTTP, HTTPS
    #: or TERMINATED_HTTPS.
    protocol = resource.Body('protocol')
    #: Port the listener will listen to, e.g. 80.
    protocol_port = resource.Body('protocol_port', type=int)
    #: The provisioning status of this listener.
    provisioning_status = resource.Body('provisioning_status')
    #: A list of references to TLS secrets.
    #: *Type: list*
    sni_container_refs = resource.Body('sni_container_refs')
    #: Timestamp when the listener was last updated.
    updated_at = resource.Body('updated_at')
