/*
 * Decompiled with CFR 0.152.
 */
package org.eclipse.jgit.internal.storage.file;

import java.io.IOException;
import java.lang.ref.ReferenceQueue;
import java.lang.ref.SoftReference;
import java.util.Random;
import java.util.concurrent.atomic.AtomicInteger;
import java.util.concurrent.atomic.AtomicLong;
import java.util.concurrent.atomic.AtomicReferenceArray;
import java.util.concurrent.locks.ReentrantLock;
import org.eclipse.jgit.internal.JGitText;
import org.eclipse.jgit.internal.storage.file.ByteWindow;
import org.eclipse.jgit.internal.storage.file.DeltaBaseCache;
import org.eclipse.jgit.internal.storage.file.PackFile;
import org.eclipse.jgit.storage.file.WindowCacheConfig;

public class WindowCache {
    private static final Random rng = new Random();
    private static volatile WindowCache cache;
    private static volatile int streamFileThreshold;
    private final ReferenceQueue<ByteWindow> queue;
    private final int tableSize;
    private final AtomicLong clock;
    private final AtomicReferenceArray<Entry> table;
    private final Lock[] locks;
    private final ReentrantLock evictLock;
    private final int evictBatch;
    private final int maxFiles;
    private final long maxBytes;
    private final boolean mmap;
    private final int windowSizeShift;
    private final int windowSize;
    private final AtomicInteger openFiles;
    private final AtomicLong openBytes;

    private static final int bits(int newSize) {
        if (newSize < 4096) {
            throw new IllegalArgumentException(JGitText.get().invalidWindowSize);
        }
        if (Integer.bitCount(newSize) != 1) {
            throw new IllegalArgumentException(JGitText.get().windowSizeMustBePowerOf2);
        }
        return Integer.numberOfTrailingZeros(newSize);
    }

    @Deprecated
    public static void reconfigure(WindowCacheConfig cfg) {
        WindowCache nc = new WindowCache(cfg);
        WindowCache oc = cache;
        if (oc != null) {
            oc.removeAll();
        }
        cache = nc;
        streamFileThreshold = cfg.getStreamFileThreshold();
        DeltaBaseCache.reconfigure(cfg);
    }

    static int getStreamFileThreshold() {
        return streamFileThreshold;
    }

    static WindowCache getInstance() {
        return cache;
    }

    static final ByteWindow get(PackFile pack, long offset) throws IOException {
        WindowCache c = cache;
        ByteWindow r = c.getOrLoad(pack, c.toStart(offset));
        if (c != cache) {
            c.removeAll();
        }
        return r;
    }

    static final void purge(PackFile pack) {
        cache.removeAll(pack);
    }

    private WindowCache(WindowCacheConfig cfg) {
        this.tableSize = WindowCache.tableSize(cfg);
        int lockCount = WindowCache.lockCount(cfg);
        if (this.tableSize < 1) {
            throw new IllegalArgumentException(JGitText.get().tSizeMustBeGreaterOrEqual1);
        }
        if (lockCount < 1) {
            throw new IllegalArgumentException(JGitText.get().lockCountMustBeGreaterOrEqual1);
        }
        this.queue = new ReferenceQueue();
        this.clock = new AtomicLong(1L);
        this.table = new AtomicReferenceArray(this.tableSize);
        this.locks = new Lock[lockCount];
        for (int i = 0; i < this.locks.length; ++i) {
            this.locks[i] = new Lock();
        }
        this.evictLock = new ReentrantLock();
        int eb = (int)((double)this.tableSize * 0.1);
        if (64 < eb) {
            eb = 64;
        } else if (eb < 4) {
            eb = 4;
        }
        if (this.tableSize < eb) {
            eb = this.tableSize;
        }
        this.evictBatch = eb;
        this.maxFiles = cfg.getPackedGitOpenFiles();
        this.maxBytes = cfg.getPackedGitLimit();
        this.mmap = cfg.isPackedGitMMAP();
        this.windowSizeShift = WindowCache.bits(cfg.getPackedGitWindowSize());
        this.windowSize = 1 << this.windowSizeShift;
        this.openFiles = new AtomicInteger();
        this.openBytes = new AtomicLong();
        if (this.maxFiles < 1) {
            throw new IllegalArgumentException(JGitText.get().openFilesMustBeAtLeast1);
        }
        if (this.maxBytes < (long)this.windowSize) {
            throw new IllegalArgumentException(JGitText.get().windowSizeMustBeLesserThanLimit);
        }
    }

    int getOpenFiles() {
        return this.openFiles.get();
    }

    long getOpenBytes() {
        return this.openBytes.get();
    }

    private int hash(int packHash, long off) {
        return packHash + (int)(off >>> this.windowSizeShift);
    }

    private ByteWindow load(PackFile pack, long offset) throws IOException {
        if (pack.beginWindowCache()) {
            this.openFiles.incrementAndGet();
        }
        try {
            if (this.mmap) {
                return pack.mmap(offset, this.windowSize);
            }
            return pack.read(offset, this.windowSize);
        }
        catch (IOException e) {
            this.close(pack);
            throw e;
        }
        catch (RuntimeException e) {
            this.close(pack);
            throw e;
        }
        catch (Error e) {
            this.close(pack);
            throw e;
        }
    }

    private Ref createRef(PackFile p, long o, ByteWindow v) {
        Ref ref = new Ref(p, o, v, this.queue);
        this.openBytes.addAndGet(ref.size);
        return ref;
    }

    private void clear(Ref ref) {
        this.openBytes.addAndGet(-ref.size);
        this.close(ref.pack);
    }

    private void close(PackFile pack) {
        if (pack.endWindowCache()) {
            this.openFiles.decrementAndGet();
        }
    }

    private boolean isFull() {
        return this.maxFiles < this.openFiles.get() || this.maxBytes < this.openBytes.get();
    }

    private long toStart(long offset) {
        return offset >>> this.windowSizeShift << this.windowSizeShift;
    }

    private static int tableSize(WindowCacheConfig cfg) {
        int wsz = cfg.getPackedGitWindowSize();
        long limit = cfg.getPackedGitLimit();
        if (wsz <= 0) {
            throw new IllegalArgumentException(JGitText.get().invalidWindowSize);
        }
        if (limit < (long)wsz) {
            throw new IllegalArgumentException(JGitText.get().windowSizeMustBeLesserThanLimit);
        }
        return (int)Math.min(5L * (limit / (long)wsz) / 2L, 2000000000L);
    }

    private static int lockCount(WindowCacheConfig cfg) {
        return Math.max(cfg.getPackedGitOpenFiles(), 32);
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    private ByteWindow getOrLoad(PackFile pack, long position) throws IOException {
        int slot = this.slot(pack, position);
        Entry e1 = this.table.get(slot);
        ByteWindow v = this.scan(e1, pack, position);
        if (v != null) {
            return v;
        }
        Lock lock = this.lock(pack, position);
        synchronized (lock) {
            Entry n;
            Entry e2 = this.table.get(slot);
            if (e2 != e1 && (v = this.scan(e2, pack, position)) != null) {
                return v;
            }
            v = this.load(pack, position);
            Ref ref = this.createRef(pack, position, v);
            this.hit(ref);
            while (!this.table.compareAndSet(slot, e2, n = new Entry(WindowCache.clean(e2), ref))) {
                e2 = this.table.get(slot);
            }
        }
        if (this.evictLock.tryLock()) {
            try {
                this.gc();
                this.evict();
            }
            finally {
                this.evictLock.unlock();
            }
        }
        return v;
    }

    private ByteWindow scan(Entry n, PackFile pack, long position) {
        while (n != null) {
            Ref r = n.ref;
            if (r.pack == pack && r.position == position) {
                ByteWindow v = (ByteWindow)r.get();
                if (v != null) {
                    this.hit(r);
                    return v;
                }
                n.kill();
                break;
            }
            n = n.next;
        }
        return null;
    }

    private void hit(Ref r) {
        long c = this.clock.get();
        this.clock.compareAndSet(c, c + 1L);
        r.lastAccess = c;
    }

    private void evict() {
        while (this.isFull()) {
            int ptr = rng.nextInt(this.tableSize);
            Entry old = null;
            int slot = 0;
            int b = this.evictBatch - 1;
            while (b >= 0) {
                if (this.tableSize <= ptr) {
                    ptr = 0;
                }
                Entry e = this.table.get(ptr);
                while (e != null) {
                    if (!(e.dead || old != null && e.ref.lastAccess >= old.ref.lastAccess)) {
                        old = e;
                        slot = ptr;
                    }
                    e = e.next;
                }
                --b;
                ++ptr;
            }
            if (old == null) continue;
            old.kill();
            this.gc();
            Entry e1 = this.table.get(slot);
            this.table.compareAndSet(slot, e1, WindowCache.clean(e1));
        }
    }

    private void removeAll() {
        for (int s = 0; s < this.tableSize; ++s) {
            Entry e1;
            do {
                Entry e = e1 = this.table.get(s);
                while (e != null) {
                    e.kill();
                    e = e.next;
                }
            } while (!this.table.compareAndSet(s, e1, null));
        }
        this.gc();
    }

    private void removeAll(PackFile pack) {
        for (int s = 0; s < this.tableSize; ++s) {
            Entry e1 = this.table.get(s);
            boolean hasDead = false;
            Entry e = e1;
            while (e != null) {
                if (e.ref.pack == pack) {
                    e.kill();
                    hasDead = true;
                } else if (e.dead) {
                    hasDead = true;
                }
                e = e.next;
            }
            if (!hasDead) continue;
            this.table.compareAndSet(s, e1, WindowCache.clean(e1));
        }
        this.gc();
    }

    private void gc() {
        Ref r;
        while ((r = (Ref)this.queue.poll()) != null) {
            Entry e1;
            if (!r.canClear()) continue;
            this.clear(r);
            boolean found = false;
            int s = this.slot(r.pack, r.position);
            Entry n = e1 = this.table.get(s);
            while (n != null) {
                if (n.ref == r) {
                    n.dead = true;
                    found = true;
                    break;
                }
                n = n.next;
            }
            if (!found) continue;
            this.table.compareAndSet(s, e1, WindowCache.clean(e1));
        }
    }

    private int slot(PackFile pack, long position) {
        return (this.hash(pack.hash, position) >>> 1) % this.tableSize;
    }

    private Lock lock(PackFile pack, long position) {
        return this.locks[(this.hash(pack.hash, position) >>> 1) % this.locks.length];
    }

    private static Entry clean(Entry top) {
        while (top != null && top.dead) {
            top.ref.enqueue();
            top = top.next;
        }
        if (top == null) {
            return null;
        }
        Entry n = WindowCache.clean(top.next);
        return n == top.next ? top : new Entry(n, top.ref);
    }

    static {
        WindowCache.reconfigure(new WindowCacheConfig());
    }

    private static final class Lock {
        private Lock() {
        }
    }

    private static class Ref
    extends SoftReference<ByteWindow> {
        final PackFile pack;
        final long position;
        final int size;
        long lastAccess;
        private boolean cleared;

        protected Ref(PackFile pack, long position, ByteWindow v, ReferenceQueue<ByteWindow> queue) {
            super(v, queue);
            this.pack = pack;
            this.position = position;
            this.size = v.size();
        }

        final synchronized boolean canClear() {
            if (this.cleared) {
                return false;
            }
            this.cleared = true;
            return true;
        }
    }

    private static class Entry {
        final Entry next;
        final Ref ref;
        volatile boolean dead;

        Entry(Entry n, Ref r) {
            this.next = n;
            this.ref = r;
        }

        final void kill() {
            this.dead = true;
            this.ref.enqueue();
        }
    }
}

