/****************************************************************************
**
*W  records.h                   GAP source                   Martin Schönert
**
**
*Y  Copyright (C)  1996,  Lehrstuhl D für Mathematik,  RWTH Aachen,  Germany
*Y  (C) 1998 School Math and Comp. Sci., University of St Andrews, Scotland
*Y  Copyright (C) 2002 The GAP Group
**
**  This file declares the functions of the generic record package.
**
**  This package  provides a uniform  interface to  the functions that access
**  records and the elements for the other packages in the GAP kernel.
*/

#ifndef libGAP_GAP_RECORDS_H
#define libGAP_GAP_RECORDS_H


/****************************************************************************
**
*F  NAME_RNAM(<rnam>) . . . . . . . . . . .name for a record name as C string
*F  NAME_OBJ_RNAM(<rnam>) . . . . . . . . .name for a record name as an Obj
**
**  'NAME_RNAM' returns the name (as a C string) for the record name <rnam>.
**  'NAME_OBJ_RNAM' returns the name (as an Obj) for the record name <rnam>.
**
**  Note that 'NAME_RNAM' and 'NAME_OBJ_RNAM' are macros, so do not call them
**  with arguments that have side effects.
*/
#define libGAP_NAME_RNAM(rnam) libGAP_CSTR_STRING( libGAP_ELM_PLIST( libGAP_NamesRNam, rnam ) )
#define libGAP_NAME_OBJ_RNAM(rnam) libGAP_ELM_PLIST( libGAP_NamesRNam, rnam )

extern  libGAP_Obj             libGAP_NamesRNam;


/****************************************************************************
**
*F  RNamName(<name>)  . . . . . . . . . . . . convert a name to a record name
**
**  'RNamName' returns  the record name with the  name  <name> (which is  a C
**  string).
*/
extern  libGAP_UInt            libGAP_RNamName (
            const libGAP_Char *        name );


/****************************************************************************
**
*F  RNamIntg(<intg>)  . . . . . . . . . . convert an integer to a record name
**
**  'RNamIntg' returns the record name corresponding to the integer <intg>.
*/
extern  libGAP_UInt            libGAP_RNamIntg (
            libGAP_Int                 intg );


/****************************************************************************
**
*F  RNamObj(<obj>)  . . . . . . . . . . .  convert an object to a record name
**
**  'RNamObj' returns the record name  corresponding  to  the  object  <obj>,
**  which currently must be a string or an integer.
*/
extern  libGAP_UInt            libGAP_RNamObj (
            libGAP_Obj                 obj );


/****************************************************************************
**
*F  IS_REC(<obj>) . . . . . . . . . . . . . . . . . . . is an object a record
*V  IsRecFuncs[<type>]  . . . . . . . . . . . . . . . . table of record tests
**
**  'IS_REC' returns a nonzero value if the object <obj> is a  record  and  0
**  otherwise.
**
**  Note that 'IS_REC' is a record, so do not call  it  with  arguments  that
**  side effects.
*/
#define libGAP_IS_REC(obj)     ((*libGAP_IsRecFuncs[ libGAP_TNUM_OBJ(obj) ])( obj ))

extern  libGAP_Int             (*libGAP_IsRecFuncs[libGAP_LAST_REAL_TNUM+1]) ( libGAP_Obj obj );


/****************************************************************************
**
*F  ELM_REC(<rec>,<rnam>) . . . . . . . . . . select an element from a record
**
**  'ELM_REC' returns the element, i.e., the value of the component, with the
**  record name <rnam> in  the record <rec>.   An error is signalled if <rec>
**  is not a record or if <rec> has no component with the record name <rnam>.
**
**  Note that 'ELM_REC' is  a macro, so do   not call it with arguments  that
**  have side effects.
*/
#define libGAP_ELM_REC(rec,rnam) \
                        ((*libGAP_ElmRecFuncs[ libGAP_TNUM_OBJ(rec) ])( rec, rnam ))

extern  libGAP_Obj             (*libGAP_ElmRecFuncs[libGAP_LAST_REAL_TNUM+1]) ( libGAP_Obj rec, libGAP_UInt rnam );


/****************************************************************************
**
*F  ISB_REC(<rec>,<rnam>) . . . . . . . . . test for an element from a record
**
**  'ISB_REC' returns 1 if the record <rec> has a component with  the  record
**  name <rnam> and 0 otherwise.  An error is signalled if  <rec>  is  not  a
**  record.
**
**  Note  that 'ISB_REC'  is a macro,  so do not call  it with arguments that
**  have side effects.
*/
#define libGAP_ISB_REC(rec,rnam) \
                        ((*libGAP_IsbRecFuncs[ libGAP_TNUM_OBJ(rec) ])( rec, rnam ))

extern  libGAP_Int             (*libGAP_IsbRecFuncs[libGAP_LAST_REAL_TNUM+1]) ( libGAP_Obj rec, libGAP_UInt rnam );


/****************************************************************************
**
*F  ASS_REC(<rec>,<rnam>,<obj>) . . . . . . . . . . . . .  assign to a record
**
**  'ASS_REC' assigns the object <obj>  to  the  record  component  with  the
**  record name <rnam> in the record <rec>.  An error is signalled  if  <rec>
**  is not a record.
**
**  Note that  'ASS_REC' is a  macro, so do not call   it with arguments that
**  have side effects.
*/
#define libGAP_ASS_REC(rec,rnam,obj) \
                        ((*libGAP_AssRecFuncs[ libGAP_TNUM_OBJ(rec) ])( rec, rnam, obj ))

extern  void            (*libGAP_AssRecFuncs[libGAP_LAST_REAL_TNUM+1]) ( libGAP_Obj rec, libGAP_UInt rnam, libGAP_Obj obj );


/****************************************************************************
**
*F  UNB_REC(<rec>,<rnam>) . . . . . . unbind a record component from a record
**
**  'UNB_REC' removes the record component  with the record name <rnam>  from
**  the record <rec>.
**
**  Note that 'UNB_REC' is  a macro, so  do  not call it with  arguments that
**  have side effects.
*/
#define libGAP_UNB_REC(rec,rnam) \
                        ((*libGAP_UnbRecFuncs[ libGAP_TNUM_OBJ(rec) ])( rec, rnam ))

extern  void            (*libGAP_UnbRecFuncs[libGAP_LAST_REAL_TNUM+1]) ( libGAP_Obj rec, libGAP_UInt rnam );


/****************************************************************************
**
*F  iscomplete_rnam( <name>, <len> )  . . . . . . . . . . . . .  check <name>
*/
extern libGAP_UInt libGAP_iscomplete_rnam (
            libGAP_Char *              name,
            libGAP_UInt                len );


/****************************************************************************
**
*F  completion_rnam( <name>, <len> )  . . . . . . . . . . . . find completion
*/
extern libGAP_UInt libGAP_completion_rnam (
            libGAP_Char *              name,
            libGAP_UInt                len );


/****************************************************************************
**

*F * * * * * * * * * * * * * initialize package * * * * * * * * * * * * * * *
*/

/****************************************************************************
**

*F  InitInfoRecords() . . . . . . . . . . . . . . . . table of init functions
*/
libGAP_StructInitInfo * libGAP_InitInfoRecords ( void );


#endif // GAP_RECORDS_H

/****************************************************************************
**

*E  records.h . . . . . . . . . . . . . . . . . . . . . . . . . . . ends here
*/
