# -*- coding: utf-8 -*-

# Copyright (c) 2014 - 2018 Detlev Offenbach <detlev@die-offenbachs.de>
#

"""
Module implementing the virtualenv interface plug-in.
"""

from __future__ import unicode_literals

from PyQt5.QtCore import QObject
from PyQt5.QtWidgets import QDialog

from E5Gui.E5Application import e5App
from E5Gui.E5Action import E5Action

import UI.Info

# Start-of-Header
name = "virtualenv Configurator Plug-in"
author = "Detlev Offenbach <detlev@die-offenbachs.de>"
autoactivate = True
deactivateable = True
version = UI.Info.VersionOnly
className = "VirtualenvInterfacePlugin"
packageName = "__core__"
shortDescription = "Configurator for Python virtual environments."
longDescription = \
    """This plug-in implements a configuration interface to create Python""" \
    """ virtual environments using 'virtualenv' or 'pyvenv."""
needsRestart = False
pyqtApi = 2
python2Compatible = True
# End-of-Header

error = ""


class VirtualenvInterfacePlugin(QObject):
    """
    Class implementing the virtualenv wizard plug-in.
    """
    def __init__(self, ui):
        """
        Constructor
        
        @param ui reference to the user interface object (UI.UserInterface)
        """
        super(VirtualenvInterfacePlugin, self).__init__(ui)
        self.__ui = ui
        self.__action = None
        
        self.__initAction()
    
    def activate(self):
        """
        Public method to activate this plug-in.
        
        @return tuple of None and activation status (boolean)
        """
        e5App().getObject("ToolbarManager").addAction(self.__action, "Tools")
        
        menu = self.__ui.getMenu("extras")
        menu.addAction(self.__action)

        return None, True
    
    def deactivate(self):
        """
        Public method to deactivate this plug-in.
        """
        e5App().getObject("ToolbarManager").removeAction(self.__action)
        
        menu = self.__ui.getMenu("extras")
        menu.removeAction(self.__action)
    
    def __initAction(self):
        """
        Private method to initialize the action.
        """
        self.__action = E5Action(
            self.tr('Virtualenv Configurator'),
            self.tr('&Virtualenv Configurator...'),
            0, 0, self,
            'virtualenv_configurator')
        self.__action.setStatusTip(self.tr('Virtualenv Configurator'))
        self.__action.setWhatsThis(self.tr(
            """<b>Virtualenv Configurator</b>"""
            """<p>This opens a dialog for entering all the parameters"""
            """ needed to create a Python virtual environment using"""
            """ virtualenv or pyvenv.</p>"""
        ))
        self.__action.triggered.connect(self.__handle)
    
    def __handle(self):
        """
        Private method to handle the creation of a virtual environment.
        """
        from UiExtensionPlugins.VirtualenvInterface\
            .VirtualenvConfigurationDialog import VirtualenvConfigurationDialog
        
        dlg = VirtualenvConfigurationDialog()
        if dlg.exec_() == QDialog.Accepted:
            (pyvenv, args, openTarget, createLog, createScript, targetDir,
             interpreter) = dlg.getData()
            
            # now do the call
            from UiExtensionPlugins.VirtualenvInterface.VirtualenvExecDialog \
                import VirtualenvExecDialog
            dia = VirtualenvExecDialog(pyvenv, targetDir, openTarget,
                                       createLog, createScript, interpreter)
            dia.show()
            dia.start(args)
            dia.exec_()

#
# eflag: noqa = M801
