#------------------------------------------------------------------------------
# File:         JPEGDigest.pm
#
# Description:  Calculate JPEGDigest and JPEGQualityEstimate
#
# Revisions:    2008/09/15 - P. Harvey Created
#               2016/01/05 - PH Added calculation of JPEGQualityEstimate
#
# References:   JD) Jens Duttke
#               2) Franz Buchinger private communication
#               3) https://github.com/ImageMagick/ImageMagick/blob/master/coders/jpeg.c
#------------------------------------------------------------------------------

package Image::ExifTool::JPEGDigest;
use strict;
use vars qw($VERSION);

$VERSION = '1.06';

# the print conversion for the JPEGDigest tag
my %PrintConv = ( #JD
    # No DQT defined
    'd41d8cd98f00b204e9800998ecf8427e' => 'No DQT defined',

    # Tested with:
    # - Independent JPEG Group library (used by many applications) X3 (Win)
    #   - Different subsamplings possible
    # - Dynamic Photo HDR 3.0 (Win)
    #   - Fixed to quality 92?
    #   - Fixed subsampling of 111111?
    # - FixFoto (Win)
    #   - These DQTs are only used for the Quality settings 24 to 100
    #   - Different subsamplings possible
    #   - Subsampling of 221111 is default
    # - GraphicConverter 4.4.4 (Mac)
    #   - Using the JPEG 6.0 library
    #   - Fixed subsampling to 221111
    # - IrfanView 4.10 (Win)
    #   - Subsampling 111111 with option "Disable subsampling", otherwise 221111
    #   - Quality mode 0 doesn't exist here
    # - Jasc Paint Shop Pro Version 5.01 (Win)
    #   - Fixed subsampling of 221111
    #   - Use reversed Quality values (0 or 1 = 99; 2 = 98; 3 = 97 etc.)
    # - Microsoft GDI+ Version 5.1.3102.2180 (Win)
    #   - Fixed subsampling of 221111 (or is that just the default subsampling?)
    # - Photomatix pro 3.0 (Win)
    #   - Fixed subsampling to 221111

    # IJG library
    'f83c5bc303fa1f74265863c2c6844edf' => 'Independent JPEG Group library (used by many applications), Quality 0 or 1',
    '1e81ee25c96cdf46f44a9b930780f8c0' => 'Independent JPEG Group library (used by many applications), Quality 2',
    '205d4a597e68f2da78137e52f39d2728' => 'Independent JPEG Group library (used by many applications), Quality 3',
    '81a936d8371a7d59da428fcfc349850f' => 'Independent JPEG Group library (used by many applications), Quality 4',
    '610772f3cf75c2fd89214fafbd7617a6' => 'Independent JPEG Group library (used by many applications), Quality 5',
    '577ec8895d884612d064771e84cf231f' => 'Independent JPEG Group library (used by many applications), Quality 6',
    '7dc25cc528116e25dd1aeb590dd7cb66' => 'Independent JPEG Group library (used by many applications), Quality 7',
    'e3cc8a85db1a32e81650b9668b98644a' => 'Independent JPEG Group library (used by many applications), Quality 8',
    'e2408846813c1f5c7f5ce3cf69e741c4' => 'Independent JPEG Group library (used by many applications), Quality 9',
    'e800426d2ef8d3cda13a0b41f1b2cc5a' => 'Independent JPEG Group library (used by many applications), Quality 10',
    '8ef467e72e5006d1b48209e7b5d94541' => 'Independent JPEG Group library (used by many applications), Quality 11',
    'af6a08d0742aa8ed6bae2f1c374e7931' => 'Independent JPEG Group library (used by many applications), Quality 12',
    'e002d9728c73f60e4d0509e1cea9af43' => 'Independent JPEG Group library (used by many applications), Quality 13',
    '1eb8b98bef8b062e4bc10cffea2d8372' => 'Independent JPEG Group library (used by many applications), Quality 14',
    '99d933a8a9ece6f2ee65757aa81ef5bd' => 'Independent JPEG Group library (used by many applications), Quality 15',
    'c7d0f7dee5631d01bc55b7c598805d98' => 'Independent JPEG Group library (used by many applications), Quality 16',
    '71555bee8d64c9dfca3cefa9dd332472' => 'Independent JPEG Group library (used by many applications), Quality 17',
    '5009778c2a1df1deeb040c85fb9d0db2' => 'Independent JPEG Group library (used by many applications), Quality 18',
    '81339d89e8294729c69fc096f1a448f3' => 'Independent JPEG Group library (used by many applications), Quality 19',
    '82028a770d9e45d64d6aa26faee97e72' => 'Independent JPEG Group library (used by many applications), Quality 20',
    '3ebb7aa9113b3e1628f55732de7dae7f' => 'Independent JPEG Group library (used by many applications), Quality 21',
    '704c1868fe865f1038aa2bd0697f71a0' => 'Independent JPEG Group library (used by many applications), Quality 22',
    '555fd108ab97e641fcfefee4bda6a306' => 'Independent JPEG Group library (used by many applications), Quality 23',
    'f1d9a0410c5ea11613569783625f5cf3' => 'Independent JPEG Group library (used by many applications), Quality 24',
    'ce378a14d52ee5752ded23b5a444f75e' => 'Independent JPEG Group library (used by many applications), Quality 25',
    '3f243aacd371617b69f1e1eacadbbe2e' => 'Independent JPEG Group library (used by many applications), Quality 26',
    'c70a1df73760a88deb890003cdab3bfe' => 'Independent JPEG Group library (used by many applications), Quality 27',
    'aa5e05f96c53f6bc498f5016b1113651' => 'Independent JPEG Group library (used by many applications), Quality 28',
    '740d4a06f4d0f774c6aac95719673793' => 'Independent JPEG Group library (used by many applications), Quality 29',
    '1426dd9c1c8098936f395e201f1eb56d' => 'Independent JPEG Group library (used by many applications), Quality 30',
    '891a53bb6a5261a2c076cf8931c3660e' => 'Independent JPEG Group library (used by many applications), Quality 31',
    '1c87cb13f4b7a5ed45b09fc4f0e52d68' => 'Independent JPEG Group library (used by many applications), Quality 32',
    '2457f78378c5efdac8e1ef619a4285cd' => 'Independent JPEG Group library (used by many applications), Quality 33',
    '20f79a557d4edb1917d243d00a7a9ba8' => 'Independent JPEG Group library (used by many applications), Quality 34',
    'e06d44ffef23792c88f215a5b2ed9478' => 'Independent JPEG Group library (used by many applications), Quality 35',
    'df1b50991b82b66c82dc856cf82383da' => 'Independent JPEG Group library (used by many applications), Quality 36',
    'ffd6e30af6d99997d38ec3e303687e25' => 'Independent JPEG Group library (used by many applications), Quality 37',
    'ed923fdb1e5009215a49c0d92061c3b0' => 'Independent JPEG Group library (used by many applications), Quality 38',
    'aaf1ebc6949569327f95cf7da78ee7bc' => 'Independent JPEG Group library (used by many applications), Quality 39',
    '0fe5225afaf055efd8453426c00e81e1' => 'Independent JPEG Group library (used by many applications), Quality 40',
    'a79e66299883be78b02ceaaf9159c320' => 'Independent JPEG Group library (used by many applications), Quality 41',
    'c9f10fb6d352cc8a8967e7e96a64862c' => 'Independent JPEG Group library (used by many applications), Quality 42',
    '6dc3abbdd52b2f26b790ddb33b82099a' => 'Independent JPEG Group library (used by many applications), Quality 43',
    '29572be1991c210fabacaeb658a74844' => 'Independent JPEG Group library (used by many applications), Quality 44',
    '0c9f89a3728234e0e85645c968d1d84a' => 'Independent JPEG Group library (used by many applications), Quality 45',
    'a5e5355ae60c569dec306eb971a49276' => 'Independent JPEG Group library (used by many applications), Quality 46',
    '4a2605b7c5b1bc99b6715342de7b6562' => 'Independent JPEG Group library (used by many applications), Quality 47',
    '6af4053465275c1b1b2d5c97f4d841aa' => 'Independent JPEG Group library (used by many applications), Quality 48',
    '905a0b1688644902f6a65d872d68a9db' => 'Independent JPEG Group library (used by many applications), Quality 49',
    'ffcd5eab8daeced571d59d4cdcc002c4' => 'Independent JPEG Group library (used by many applications), Quality 50',
    '76806382fa57818da2e406a0dc23ce20' => 'Independent JPEG Group library (used by many applications), Quality 51',
    '7c71a5eb9408b93be3ea6cf9be1d31ea' => 'Independent JPEG Group library (used by many applications), Quality 52',
    'ccf3f63196092667f97c2ff723481a21' => 'Independent JPEG Group library (used by many applications), Quality 53',
    'acc5e596cc4eb156b83eb89190b289a7' => 'Independent JPEG Group library (used by many applications), Quality 54',
    'a16e53aa41aa66124557c0b976d73734' => 'Independent JPEG Group library (used by many applications), Quality 55',
    '2101fdf5c6f6e92943bc16ccf8aa46a8' => 'Independent JPEG Group library (used by many applications), Quality 56',
    'aaa2026d750590e466d9198f20b888e4' => 'Independent JPEG Group library (used by many applications), Quality 57',
    'fcd8c97cf6004230444ce21dab8a167f' => 'Independent JPEG Group library (used by many applications), Quality 58',
    '318081abf5329c90d984c2214d69f097' => 'Independent JPEG Group library (used by many applications), Quality 59',
    '95348adff2bf5a88f3967e9237fc571e' => 'Independent JPEG Group library (used by many applications), Quality 60; Pentax Better',
    'e22b26930415798b7ecf2b060c1cdc2a' => 'Independent JPEG Group library (used by many applications), Quality 61',
    '3e646bfadb62d25ae8404b179e93e74e' => 'Independent JPEG Group library (used by many applications), Quality 62',
    '34bf5c46342f2a514f8ae562e520ece0' => 'Independent JPEG Group library (used by many applications), Quality 63',
    'fc9f2bd278075ea89d482e1f9e66738f' => 'Independent JPEG Group library (used by many applications), Quality 64',
    'b55f27c368f119f25957c8e0036b27f8' => 'Independent JPEG Group library (used by many applications), Quality 65',
    '5f7b9af6d66eaf12874aa680e7b0d31b' => 'Independent JPEG Group library (used by many applications), Quality 66',
    'a78fd0f1183b268b2fdfa23308c3ad44' => 'Independent JPEG Group library (used by many applications), Quality 67',
    '61814a2eca233e10b0ba26881551fb50' => 'Independent JPEG Group library (used by many applications), Quality 68',
    'ce0f36a3d870b24f9816634000ea2d2e' => 'Independent JPEG Group library (used by many applications), Quality 69',
    'f7cba1affebd362a322abd45ce580e56' => 'Independent JPEG Group library (used by many applications), Quality 70',
    '7b85fffdf97680de53e49788712c50de' => 'Independent JPEG Group library (used by many applications), Quality 71',
    '4abd7dbca7735c987eb899b0f8646ce4' => 'Independent JPEG Group library (used by many applications), Quality 72',
    '167959a11aff7940df84ed9f3379ed0a' => 'Independent JPEG Group library (used by many applications), Quality 73',
    'd61807da54a72c4651466049c9f67541' => 'Independent JPEG Group library (used by many applications), Quality 74',
    '2851eea5e15f1b977c1496a77c884b4f' => 'Independent JPEG Group library (used by many applications), Quality 75',
    'b095631a0665a515d9aa290639f6672b' => 'Independent JPEG Group library (used by many applications), Quality 76',
    '0ee4ea4687d3c57d060e3afd2559b7bb' => 'Independent JPEG Group library (used by many applications), Quality 77',
    '37e999828e5bc43ee4a470bf29ea97b7' => 'Independent JPEG Group library (used by many applications), Quality 78',
    'dd9b9d09a624deab730d9bd5b8825baa' => 'Independent JPEG Group library (used by many applications), Quality 79',
    '5319a82dc93c1ee5c8d2265e4f1fb60e' => 'Independent JPEG Group library (used by many applications), Quality 80',
    '6fa04ba1184e986c4da3df8141e05a42' => 'Independent JPEG Group library (used by many applications), Quality 81',
    '1d099901cfe37674e4aeb2cdbddf0703' => 'Independent JPEG Group library (used by many applications), Quality 82',
    '8cc04e05813c028683e3cb8e6eab79eb' => 'Independent JPEG Group library (used by many applications), Quality 83',
    '3eafbf05c0dd233b385856065e456c11' => 'Independent JPEG Group library (used by many applications), Quality 84',
    '98dec36fe95ed7c1772d9ed67a67e260' => 'Independent JPEG Group library (used by many applications), Quality 85',
    'a1ee06d19dcb62c7467768d1ba73cf12' => 'Independent JPEG Group library (used by many applications), Quality 86',
    '643ad95753c261eacc5ea3f4c9e4d469' => 'Independent JPEG Group library (used by many applications), Quality 87',
    '2557828257dd798ad636df350c685a26' => 'Independent JPEG Group library (used by many applications), Quality 88',
    'a5b4bbf018828a3d00e54ab72a175fdc' => 'Independent JPEG Group library (used by many applications), Quality 89',
    '50325a7b6e0a9b7f9aea8f5a6f7f31aa' => 'Independent JPEG Group library (used by many applications), Quality 90',
    'c0cf6b0c508a35a13acd8c912548a269' => 'Independent JPEG Group library (used by many applications), Quality 91',
    'ff0aa07a220cd8973a4b86f3ecd4325b' => 'Independent JPEG Group library (used by many applications), Quality 92',
    'f251d4554524d22f94a34668ab17957c' => 'Independent JPEG Group library (used by many applications), Quality 93',
    'cd993be55bad60bb539df0cc2d7f9f6f' => 'Independent JPEG Group library (used by many applications), Quality 94',
    'adf507352f9ce218a4605700459d597f' => 'Independent JPEG Group library (used by many applications), Quality 95',
    '9e9e3e22af4e41ea4ec1b8f656f28f42' => 'Independent JPEG Group library (used by many applications), Quality 96',
    'b9f5c003ef62cbd8fc93be6679c1c3bc' => 'Independent JPEG Group library (used by many applications), Quality 97',
    '95b6a316836182441b12039279872ec3' => 'Independent JPEG Group library (used by many applications), Quality 98',
    'b9b1ce23552e2f82b95b48de20065ed3' => 'Independent JPEG Group library (used by many applications), Quality 99',
    'a97591462e9b2339efd4f88ca97bb471' => 'Independent JPEG Group library (used by many applications), Quality 100',

    # Independent JPEG Group library (Grayscale)
    #
    # Tested with:
    # - Corel PhotoImpact X3 (Win)
    # - IrfanView (Win)
    #   - Quality mode 0 doesn't exist here
    '185893c53196f6156d458a84e1135c43:11' => 'Independent JPEG Group library (used by many applications), Quality 0 or 1 (Grayscale)',
    'b41ccbe66e41a05de5e68832c07969a7:11' => 'Independent JPEG Group library (used by many applications), Quality 2 (Grayscale)',
    'efa024d741ecc5204e7edd4f590a7a25:11' => 'Independent JPEG Group library (used by many applications), Quality 3 (Grayscale)',
    '3396344724a1868ada2330ebaeb9448e:11' => 'Independent JPEG Group library (used by many applications), Quality 4 (Grayscale)',
    '14276fffb98deb42b7dbce30abb8425f:11' => 'Independent JPEG Group library (used by many applications), Quality 5 (Grayscale)',
    'a99e2826c10d0922ce8942c5437f53a6:11' => 'Independent JPEG Group library (used by many applications), Quality 6 (Grayscale)',
    '0d3de456aa5cbb8a2578208250aa9b88:11' => 'Independent JPEG Group library (used by many applications), Quality 7 (Grayscale)',
    'fa987940fdedbe883cc0e9fcc907f89e:11' => 'Independent JPEG Group library (used by many applications), Quality 8 (Grayscale)',
    '1c9bb67190ee64e82d3c67f7943bf4a4:11' => 'Independent JPEG Group library (used by many applications), Quality 9 (Grayscale)',
    '57d20578d190b04c7667b10d3df241bb:11' => 'Independent JPEG Group library (used by many applications), Quality 10 (Grayscale)',
    '619fd49197f0403ce13d86cffec46419:11' => 'Independent JPEG Group library (used by many applications), Quality 11 (Grayscale)',
    '327f47dd8f999b2bbb3bb25c43cf5be5:11' => 'Independent JPEG Group library (used by many applications), Quality 12 (Grayscale)',
    'e4e5bc705c40cfaffff6565f16fe98a9:11' => 'Independent JPEG Group library (used by many applications), Quality 13 (Grayscale)',
    '6c64fa9ad302624a826f04ecc80459be:11' => 'Independent JPEG Group library (used by many applications), Quality 14 (Grayscale)',
    '039a3f0e101f1bcdb6bb81478cf7ae6b:11' => 'Independent JPEG Group library (used by many applications), Quality 15 (Grayscale)',
    'c23b08c94d7537c9447691d54ae1080c:11' => 'Independent JPEG Group library (used by many applications), Quality 16 (Grayscale)',
    '200107bc0174104bbf1d4653c4b05058:11' => 'Independent JPEG Group library (used by many applications), Quality 17 (Grayscale)',
    '72abfdc6e65b32ded2cd7ac77a04f447:11' => 'Independent JPEG Group library (used by many applications), Quality 18 (Grayscale)',
    '1799a236c36da0b30729d9005ca7c7f9:11' => 'Independent JPEG Group library (used by many applications), Quality 19 (Grayscale)',
    'c33a667bff7f590655d196010c5e39f3:11' => 'Independent JPEG Group library (used by many applications), Quality 20 (Grayscale)',
    'b1dc98f6a2f8828f8432872da43e7d94:11' => 'Independent JPEG Group library (used by many applications), Quality 21 (Grayscale)',
    '07318a0acfebe9086f0e04a4c4f5398a:11' => 'Independent JPEG Group library (used by many applications), Quality 22 (Grayscale)',
    'a295b7163305f327a5a45ae177a0a19c:11' => 'Independent JPEG Group library (used by many applications), Quality 23 (Grayscale)',
    'c741c1b134cf81ab69acc81f15a67137:11' => 'Independent JPEG Group library (used by many applications), Quality 24 (Grayscale)',
    'a68893776502a591548c7b5bece13e1b:11' => 'Independent JPEG Group library (used by many applications), Quality 25 (Grayscale)',
    '111848d9e41f6f408ef70841f90c0519:11' => 'Independent JPEG Group library (used by many applications), Quality 26 (Grayscale)',
    '886374ceebcfd4dfed200b0b34b4baca:11' => 'Independent JPEG Group library (used by many applications), Quality 27 (Grayscale)',
    '666dd95fd0e20f5c20bc44d78d528869:11' => 'Independent JPEG Group library (used by many applications), Quality 28 (Grayscale)',
    '1aa58cb85dda84de2ddf436667124dcd:11' => 'Independent JPEG Group library (used by many applications), Quality 29 (Grayscale)',
    '9d321ab2bdda6f3cb76d2d88838aa8c3:11' => 'Independent JPEG Group library (used by many applications), Quality 30 (Grayscale)',
    '6ad87d648101c268f83fa379d4c773f2:11' => 'Independent JPEG Group library (used by many applications), Quality 31 (Grayscale)',
    'cdf8e921300f27a4af7661a2de16e91a:11' => 'Independent JPEG Group library (used by many applications), Quality 32 (Grayscale)',
    '3f48672e37b6dd2e571b222e4b7ff97d:11' => 'Independent JPEG Group library (used by many applications), Quality 33 (Grayscale)',
    'a53a7d4cc86d01f4c1b867270c9c078f:11' => 'Independent JPEG Group library (used by many applications), Quality 34 (Grayscale)',
    '09ec03f5096df106c692123f3fd34296:11' => 'Independent JPEG Group library (used by many applications), Quality 35 (Grayscale)',
    'a946498fd1902c9de87a1f5182966742:11' => 'Independent JPEG Group library (used by many applications), Quality 36 (Grayscale)',
    '5d650a1d38108fd79d4f336ba8e254c2:11' => 'Independent JPEG Group library (used by many applications), Quality 37 (Grayscale)',
    '81d620f1b470fd535b26544b4ea20643:11' => 'Independent JPEG Group library (used by many applications), Quality 38 (Grayscale)',
    '892788bdf8cbef5c6fbd7019a079bf8e:11' => 'Independent JPEG Group library (used by many applications), Quality 39 (Grayscale)',
    'cf3929fd4c1e5c28b7f137f982178ad1:11' => 'Independent JPEG Group library (used by many applications), Quality 40 (Grayscale)',
    '31f288945896ed839f1d936bff06fb03:11' => 'Independent JPEG Group library (used by many applications), Quality 41 (Grayscale)',
    'e0c38f0c5e6562445d4e92bae51713be:11' => 'Independent JPEG Group library (used by many applications), Quality 42 (Grayscale)',
    '18fa29d1164984883a6af76377b60d5a:11' => 'Independent JPEG Group library (used by many applications), Quality 43 (Grayscale)',
    'eff737b226fbce48c42625c5bf9dabb6:11' => 'Independent JPEG Group library (used by many applications), Quality 44 (Grayscale)',
    'b900f91ee8697255d5daebce858caaeb:11' => 'Independent JPEG Group library (used by many applications), Quality 45 (Grayscale)',
    'ab2f8513823067af242f7e3c04a88a9c:11' => 'Independent JPEG Group library (used by many applications), Quality 46 (Grayscale)',
    '60b682c4d412f5255efbaa32787c46ca:11' => 'Independent JPEG Group library (used by many applications), Quality 47 (Grayscale)',
    'ea50813e06203c2ad1165252bcb99a1d:11' => 'Independent JPEG Group library (used by many applications), Quality 48 (Grayscale)',
    'f6308a717437d3653b0751ebf511db0f:11' => 'Independent JPEG Group library (used by many applications), Quality 49 (Grayscale)',
    '7c8242581553e818ef243fc680879a19:11' => 'Independent JPEG Group library (used by many applications), Quality 50 (Grayscale)',
    'fc41ab8251718977bc6676f502f457e0:11' => 'Independent JPEG Group library (used by many applications), Quality 51 (Grayscale)',
    '606c4c78c0226646bf4d3c5a5898fb17:11' => 'Independent JPEG Group library (used by many applications), Quality 52 (Grayscale)',
    '0e6c6a5440d33d25f1c25836a45cfa69:11' => 'Independent JPEG Group library (used by many applications), Quality 53 (Grayscale)',
    '7464b2361e5b5f5a9ba74a87475dda91:11' => 'Independent JPEG Group library (used by many applications), Quality 54 (Grayscale)',
    'aeaa2ca48eabb3088ebb713b3c4e1a67:11' => 'Independent JPEG Group library (used by many applications), Quality 55 (Grayscale)',
    '3f36450b0ba074578391e77f7340cef0:11' => 'Independent JPEG Group library (used by many applications), Quality 56 (Grayscale)',
    'be232444027e83db6f8d8b79d078442e:11' => 'Independent JPEG Group library (used by many applications), Quality 57 (Grayscale)',
    '712c145d6472a2b315b2ecfb916d1590:11' => 'Independent JPEG Group library (used by many applications), Quality 58 (Grayscale)',
    'ae3dd4568cc71c47d068cf831c66b59d:11' => 'Independent JPEG Group library (used by many applications), Quality 59 (Grayscale)',
    'b290e52c21a435fede4586636ef5e287:11' => 'Independent JPEG Group library (used by many applications), Quality 60 (Grayscale)',
    'a09ca4c4391e0221396a08f229a65f9d:11' => 'Independent JPEG Group library (used by many applications), Quality 61 (Grayscale)',
    '0818578fc5fc571b4f8d5ffefc9dc0d8:11' => 'Independent JPEG Group library (used by many applications), Quality 62 (Grayscale)',
    '7c685e2916555eda34cb37a1e71adc6a:11' => 'Independent JPEG Group library (used by many applications), Quality 63 (Grayscale)',
    '69c6b9440342adfc0db89a6c91aba332:11' => 'Independent JPEG Group library (used by many applications), Quality 64 (Grayscale)',
    'd5d484b68e25b44288e67e699829695c:11' => 'Independent JPEG Group library (used by many applications), Quality 65 (Grayscale)',
    'de8310d09116a7a62965f3e0e43ef525:11' => 'Independent JPEG Group library (used by many applications), Quality 66 (Grayscale)',
    'e4735f63e88baf04599afc034e690845:11' => 'Independent JPEG Group library (used by many applications), Quality 67 (Grayscale)',
    'b4ef810b14dee9c6d6d8cace98f799a6:11' => 'Independent JPEG Group library (used by many applications), Quality 68 (Grayscale)',
    '52886ef80147c9a136e20b2bc3b76f52:11' => 'Independent JPEG Group library (used by many applications), Quality 69 (Grayscale)',
    '9c62dbc848be82ef91219ba9843998be:11' => 'Independent JPEG Group library (used by many applications), Quality 70 (Grayscale)',
    'bfe8c1c73eea84b85673487a82f67627:11' => 'Independent JPEG Group library (used by many applications), Quality 71 (Grayscale)',
    'ea445840d29c51009a2a8cd49b96ccee:11' => 'Independent JPEG Group library (used by many applications), Quality 72 (Grayscale)',
    '71c1a56890fff9b0a095fa5a1c96132b:11' => 'Independent JPEG Group library (used by many applications), Quality 73 (Grayscale)',
    'f080b02331ac8adf03de2281042d2b49:11' => 'Independent JPEG Group library (used by many applications), Quality 74 (Grayscale)',
    'd0eaa368737f17f6037757d393a22599:11' => 'Independent JPEG Group library (used by many applications), Quality 75 (Grayscale)',
    '303663905d055b77bb547fe0b0beb9c5:11' => 'Independent JPEG Group library (used by many applications), Quality 76 (Grayscale)',
    '5cdf1d5bbe19375ad5c7237273dddede:11' => 'Independent JPEG Group library (used by many applications), Quality 77 (Grayscale)',
    'd64e7ff8292fd77131932864d3c9ce7c:11' => 'Independent JPEG Group library (used by many applications), Quality 78 (Grayscale)',
    '12b4cc13891c5aef3dadb3405b6fa65d:11' => 'Independent JPEG Group library (used by many applications), Quality 79 (Grayscale)',
    'b008cd63591f8fd366f77d2b224b9c9c:11' => 'Independent JPEG Group library (used by many applications), Quality 80 (Grayscale)',
    '49b6e472c7d5ecead593c6009768e765:11' => 'Independent JPEG Group library (used by many applications), Quality 81 (Grayscale)',
    'bce6fa61623ad4f65ff3fec1528cb026:11' => 'Independent JPEG Group library (used by many applications), Quality 82 (Grayscale)',
    'c2b037bf9f5e5baba804d7bbbb2dc73b:11' => 'Independent JPEG Group library (used by many applications), Quality 83 (Grayscale)',
    '7fe7b339c6ffc62b984eeab4b0df9168:11' => 'Independent JPEG Group library (used by many applications), Quality 84 (Grayscale)',
    '274bbeb0ac3939f90c578ebb1f5a9eef:11' => 'Independent JPEG Group library (used by many applications), Quality 85 (Grayscale)',
    '0a0268c655d616b0e4af2851533aa3af:11' => 'Independent JPEG Group library (used by many applications), Quality 86 (Grayscale)',
    '52318e260c0d6b3dbee85c87f9b94e63:11' => 'Independent JPEG Group library (used by many applications), Quality 87 (Grayscale)',
    'b64cc19a0f81a506ed5bcfb9c131c8fe:11' => 'Independent JPEG Group library (used by many applications), Quality 88 (Grayscale)',
    'd8c54333eb475b8db9f32f11fe96337e:11' => 'Independent JPEG Group library (used by many applications), Quality 89 (Grayscale)',
    '12fe6b9bfd20f4d7f0ac2a221c566c45:11' => 'Independent JPEG Group library (used by many applications), Quality 90 (Grayscale)',
    '12aefbf7689633c83da714c9f0e90e05:11' => 'Independent JPEG Group library (used by many applications), Quality 91 (Grayscale)',
    'a3a96add050fc51a2b3ce59a9a491034:11' => 'Independent JPEG Group library (used by many applications), Quality 92 (Grayscale)',
    '7b0242bd9aaeab4962f5d5b39b9a4027:11' => 'Independent JPEG Group library (used by many applications), Quality 93 (Grayscale)',
    '12fc29c1d8940c93a47ee9d927a17561:11' => 'Independent JPEG Group library (used by many applications), Quality 94 (Grayscale)',
    'e1fedef5184beeb7b0f5c055c7ae1d31:11' => 'Independent JPEG Group library (used by many applications), Quality 95 (Grayscale)',
    'ae9202355f603776794d3e62c43578d6:11' => 'Independent JPEG Group library (used by many applications), Quality 96 (Grayscale)',
    '36da00bae6cd81d1f97e32748c07e33f:11' => 'Independent JPEG Group library (used by many applications), Quality 97 (Grayscale)',
    '54dc50b16e7cc9bc383eb9e73e85e199:11' => 'Independent JPEG Group library (used by many applications), Quality 98 (Grayscale)',
    '23a59c4f9ec045faf9f8379b3ca302bb:11' => 'Independent JPEG Group library (used by many applications), Quality 99 (Grayscale)',
    'bbd2dbcfe20b59e981e9a42cd1eb6ece:11' => 'Independent JPEG Group library (used by many applications), Quality 100 (Grayscale)',

    # Tested with Adobe Photoshop Lightroom 1.4.1 (Win) - "Export"
    '683506a889c78d9bc230a0c7ee5f62f3:221111' => 'Adobe Lightroom, Quality 0% - 7%',
    'bc490651af6592cd1dbbbc4fa2cfa1fb:221111' => 'Adobe Lightroom, Quality 8% - 15%',
    'ce4286d9f07999524c3c7472b065c5ab:221111' => 'Adobe Lightroom, Quality 16% - 23%',
    'cbfbfef12aead8841585ef605c789b9f:221111' => 'Adobe Lightroom, Quality 24% - 30%',
    'a0772e73dec2bdc4057c27da47bff376:221111' => 'Adobe Lightroom, Quality 31% - 38%',
    '7ef06dbde538346b8b01c6b538ca70c6:221111' => 'Adobe Lightroom, Quality 39% - 46%',
    '0ff225f58a214f79d1d85d78f6f5dab8:221111' => 'Adobe Lightroom, Quality 47% - 53%',
    '8a8603650fa5ae5fdcf4b2eaf0b23638:111111' => 'Adobe Lightroom, Quality 54% - 61%',
    '44f583ed6b65cb8ba915ec5df051616c:111111' => 'Adobe Lightroom, Quality 62% - 69%',
    'de94c5591bafc7456ccaef430271b907:111111' => 'Adobe Lightroom, Quality 70% - 76%',
    'a6841b35e9ffefa5d83a0445dddd2621:111111' => 'Adobe Lightroom, Quality 77% - 84%',
    '6e3f6a3a5a1eae6155331d42d6f968dd:111111' => 'Adobe Lightroom, Quality 85% - 92%',
    '5379e0133d4439b6f7c7039fc7f7734f:111111' => 'Adobe Lightroom, Quality 93% - 100%',

    # Adobe Photoshop Lightroom 2.1 (Win) - "Export"
    # (0-23% equal to Lightroom 1.4.1)
    '8453391d3adf377c46a1a0cee08c35c3:221111' => 'Adobe Lightroom, Quality 24% - 30%',
    'f8ede291b1272576d1580e333d30103e:221111' => 'Adobe Lightroom, Quality 31% - 38%',
    'de0fb6d13e12e8df26140dd74691bf0f:221111' => 'Adobe Lightroom, Quality 39% - 46%',
    '0d5b0af72561f68c671731f22d9e41e2:221111' => 'Adobe Lightroom, Quality 47% - 53%',
    'b9fd15fd52408af5ea2a5045227233d8:111111' => 'Adobe Lightroom, Quality 54% - 61%',
    '27472e3714251402d5509438505611c3:111111' => 'Adobe Lightroom, Quality 62% - 69%',
    '34a599dff2b6aaed12143938b7374f2f:111111' => 'Adobe Lightroom, Quality 70% - 76%',
    '5c508e529d045b6f0c800e29ba2d6ab5:111111' => 'Adobe Lightroom, Quality 77% - 84%',
    '42bfe52476bf07f1ed0e6451903cc9ee:111111' => 'Adobe Lightroom, Quality 85% - 92%',
    '4c3c425b4024b68c0de03904a825bc35:111111' => 'Adobe Lightroom, Quality 93% - 100%',

    # Tested with Adobe Photoshop CS2 Version 9.0.2 (Win) - "Save as..." (RGB/CYMK)
    '683506a889c78d9bc230a0c7ee5f62f3:221111' => 'Adobe Photoshop, Quality 0',
    'bc490651af6592cd1dbbbc4fa2cfa1fb:221111' => 'Adobe Photoshop, Quality 1',
    'ce4286d9f07999524c3c7472b065c5ab:221111' => 'Adobe Photoshop, Quality 2',
    'cbfbfef12aead8841585ef605c789b9f:221111' => 'Adobe Photoshop, Quality 3',
    'a0772e73dec2bdc4057c27da47bff376:221111' => 'Adobe Photoshop, Quality 4',
    '7ef06dbde538346b8b01c6b538ca70c6:221111' => 'Adobe Photoshop, Quality 5',
    '0ff225f58a214f79d1d85d78f6f5dab8:221111' => 'Adobe Photoshop, Quality 6',
    '8a8603650fa5ae5fdcf4b2eaf0b23638:111111' => 'Adobe Photoshop, Quality 7',
    '44f583ed6b65cb8ba915ec5df051616c:111111' => 'Adobe Photoshop, Quality 8',
    'de94c5591bafc7456ccaef430271b907:111111' => 'Adobe Photoshop, Quality 9',
    'a6841b35e9ffefa5d83a0445dddd2621:111111' => 'Adobe Photoshop, Quality 10',
    '6e3f6a3a5a1eae6155331d42d6f968dd:111111' => 'Adobe Photoshop, Quality 11',
    '5379e0133d4439b6f7c7039fc7f7734f:111111' => 'Adobe Photoshop, Quality 12',

    # Tested with Adobe Photoshop CS2 Version 9.0.2 (Win) - "Save as..." (Grayscale)
    '3b0b5975a0e1c9d732c93e1b37a6978b:11' => 'Adobe Photoshop, Quality 0',
    'f4d19ed563e2d0519d6a547088771ddb:11' => 'Adobe Photoshop, Quality 1',
    'e9ef286567fd84a1f479b35ca00db43c:11' => 'Adobe Photoshop, Quality 2',
    'b39cafdb459a42749be3f6459a596677:11' => 'Adobe Photoshop, Quality 3',
    'b3f215deea48e982e205619af279205f:11' => 'Adobe Photoshop, Quality 4',
    'fccd63ce166e198065eaae05c8d78407:11' => 'Adobe Photoshop, Quality 5',
    '0a50266ad8d1dff11c90cd1480c0a2be:11' => 'Adobe Photoshop, Quality 6',
    '6579941db0216f41f0a20de9b626538a:11' => 'Adobe Photoshop, Quality 7',
    '57aa47876e10c6b4f35ecb8889e55ad9:11' => 'Adobe Photoshop, Quality 8',
    '076598d43c5186f6d7a1020b64b93625:11' => 'Adobe Photoshop, Quality 9',
    '37132e8ea81137fdf26ce30926ab8100:11' => 'Adobe Photoshop, Quality 10',
    '46f55ee294723cee9faa816549b3cfa7:11' => 'Adobe Photoshop, Quality 11',
    '7b83284f61decf47ab3f8f7361c18943:11' => 'Adobe Photoshop, Quality 12',

    # Tested with Adobe Photoshop CS2 Version 9.0.2 (Win) - "Save for web..."
    '9ac881c536e509675e5cf3795a85d9de:221111' => 'Adobe Photoshop, Save for web, Quality 0',
    '3521d793fd9d2d9aac85dc4f0be40290:221111' => 'Adobe Photoshop, Save for web, Quality 1',
    '041c9e3cf0d34a8b89539e3115bca31b:221111' => 'Adobe Photoshop, Save for web, Quality 2',
    '029b3a6f0b92af6786d753788eafabfe:221111' => 'Adobe Photoshop, Save for web, Quality 3',
    '6cdd3762e346b16a59af4bddb213b07a:221111' => 'Adobe Photoshop, Save for web, Quality 4',
    '84a69c0b43505dd0cbc25d640873b5b9:221111' => 'Adobe Photoshop, Save for web, Quality 5',
    '7254c012821f2bc866d7d6dd7906c92d:221111' => 'Adobe Photoshop, Save for web, Quality 6',
    '428ba2c747ea4e495ff3c7ff44a988d2:221111' => 'Adobe Photoshop, Save for web, Quality 7',
    '42cb001aea7e24d239f6c2fcbd861862:221111' => 'Adobe Photoshop, Save for web, Quality 8',
    'a3698813ce90772a30b6eb9a7deb3f4a:221111' => 'Adobe Photoshop, Save for web, Quality 9',
    '301158b292e3232856a765486da26fa6:221111' => 'Adobe Photoshop, Save for web, Quality 10',
    '8d9edea9287aa919e433b620f61468dc:221111' => 'Adobe Photoshop, Save for web, Quality 11',
    'c1e0554d8a6ed003eb98e068429b56b9:221111' => 'Adobe Photoshop, Save for web, Quality 12',
    '0e0a151e0a52097cbd7683c9385e3a7c:221111' => 'Adobe Photoshop, Save for web, Quality 13',
    '911e66f21fe242cc74e0a5738b0330bd:221111' => 'Adobe Photoshop, Save for web, Quality 14',
    '028fafd94aa66ee269f58d800c89d838:221111' => 'Adobe Photoshop, Save for web, Quality 15',
    '866b8adb1ce7c9dc0e58b7c1e013280f:221111' => 'Adobe Photoshop, Save for web, Quality 16',
    '7f712aecf513621f635a007aadda61af:221111' => 'Adobe Photoshop, Save for web, Quality 17',
    '38f26622a54ba22accac05f7c0a3b307:221111' => 'Adobe Photoshop, Save for web, Quality 18',
    'd241d5165e64e98024b47dfbf76be88c:221111' => 'Adobe Photoshop, Save for web, Quality 19',
    'afb31cfed194d4e125bde8fd4755bb8a:221111' => 'Adobe Photoshop, Save for web, Quality 20',
    '0d501a036c984d2caf49fd298b2d0d16:221111' => 'Adobe Photoshop, Save for web, Quality 21',
    '9e992f35767c4aa023b8afd243b247bf:221111' => 'Adobe Photoshop, Save for web, Quality 22',
    '0a80e5bf01a9c5650384dfe1a428f61d:221111' => 'Adobe Photoshop, Save for web, Quality 23',
    '2501aad23cdf94b25c6df0ab6984b6e0:221111' => 'Adobe Photoshop, Save for web, Quality 24',
    '09c168d2e075070d3a2535e7f2e455df:221111' => 'Adobe Photoshop, Save for web, Quality 25',
    '63190207beeb805306f7d0bcc3898cb3:221111' => 'Adobe Photoshop, Save for web, Quality 26',
    'e47902bc7ba3037921010c568648c8c3:221111' => 'Adobe Photoshop, Save for web, Quality 27',
    '410ed63b6e5225d8b99da6272fd6069b:221111' => 'Adobe Photoshop, Save for web, Quality 28',
    'b40f3f3c46d70a560e2033fadd8c7bb5:221111' => 'Adobe Photoshop, Save for web, Quality 29',
    '45148ae63b12ccaa6fb5a487ca7620e9:221111' => 'Adobe Photoshop, Save for web, Quality 30',
    '5180e51bd58432c7b51a305ed0c24d1b:221111' => 'Adobe Photoshop, Save for web, Quality 31',
    'c5c472d899462bbe31da9aa8c072c0bc:221111' => 'Adobe Photoshop, Save for web, Quality 32',
    '28cdbc95898e02dd0ffc45ba48596ca7:221111' => 'Adobe Photoshop, Save for web, Quality 33',
    '42cd88e0eb3c14a705b952550ec2eacd:221111' => 'Adobe Photoshop, Save for web, Quality 34',
    '78a2a442aac5cca7fa2ef5a8bd96219e:221111' => 'Adobe Photoshop, Save for web, Quality 35',
    '96bce854134a2fccfcb68dca6687cd51:221111' => 'Adobe Photoshop, Save for web, Quality 36',
    'fefd00ec4610895e4294de690f5977e9:221111' => 'Adobe Photoshop, Save for web, Quality 37',
    'd71c8ddb9117920304d83a6f8b7832a4:221111' => 'Adobe Photoshop, Save for web, Quality 38',
    '1727e720300403e5f315b5e17ef84d3f:221111' => 'Adobe Photoshop, Save for web, Quality 39',
    '8fb05e3c3b0a7404ff6ca54f952d2a5e:221111' => 'Adobe Photoshop, Save for web, Quality 40',
    '328ab751ea48f5a8bc7c4b8628138ce0:221111' => 'Adobe Photoshop, Save for web, Quality 41',
    'd9653333a3af8842dd4b72856ac4ef4e:221111' => 'Adobe Photoshop, Save for web, Quality 42',
    '276da99e50e1b39134e13826789d655e:221111' => 'Adobe Photoshop, Save for web, Quality 43',
    '23f2bd2d96ec531815609503dae4a2b0:221111' => 'Adobe Photoshop, Save for web, Quality 44',
    'bafe2a89809f23bc7367e9a819570728:221111' => 'Adobe Photoshop, Save for web, Quality 45',
    '6bfdcd36327406f801be86d0e8ca6b60:221111' => 'Adobe Photoshop, Save for web, Quality 46',
    'eb8e5c42d31b916737ac21dffd6f012b:221111' => 'Adobe Photoshop, Save for web, Quality 47',
    'e57a9878be74473990343573c6585f79:221111' => 'Adobe Photoshop, Save for web, Quality 48',
    '91dfacd928ce717cb135c6da03afd907:221111' => 'Adobe Photoshop, Save for web, Quality 49',
    '16c443478b9417d44893f8748d49b790:221111' => 'Adobe Photoshop, Save for web, Quality 50',
    '84de345dcf710f937a39a0b631b87fc4:111111' => 'Adobe Photoshop, Save for web, Quality 51',
    'bdd6043e7f5a5f1512b99b2394a075e2:111111' => 'Adobe Photoshop, Save for web, Quality 52',
    'c7614d3d384a02630721be335062ef75:111111' => 'Adobe Photoshop, Save for web, Quality 53',
    '42d6f71aace3de2ccfdd8348b0198704:111111' => 'Adobe Photoshop, Save for web, Quality 54',
    '84d5f059ce3e1b78d91355e1e86e2d1a:111111' => 'Adobe Photoshop, Save for web, Quality 55',
    '5881004f575752d77ee00e767d848e51:111111' => 'Adobe Photoshop, Save for web, Quality 56',
    '0cb697537acde3d2e85078377461a8e0:111111' => 'Adobe Photoshop, Save for web, Quality 57',
    'b2762ffa5c0a1799fb2e9ad6dfd2171a:111111' => 'Adobe Photoshop, Save for web, Quality 58',
    '2b7a6a83259aa9967e098d3e70f1ee09:111111' => 'Adobe Photoshop, Save for web, Quality 59',
    '6123a3685e1012af5a0d024de1ce0304:111111' => 'Adobe Photoshop, Save for web, Quality 60',
    'd08c8435de33f2c186aa2dd9cba3e874:111111' => 'Adobe Photoshop, Save for web, Quality 61',
    'e69be2174dbbfb952e54576fbdfe6c14:111111' => 'Adobe Photoshop, Save for web, Quality 62',
    '2ec2d5c10641952fce5c435b331b8872:111111' => 'Adobe Photoshop, Save for web, Quality 63',
    '98201e1185b7069f1247ac3cdc56c824:111111' => 'Adobe Photoshop, Save for web, Quality 64',
    '8fc0325d05c9199bc1e2dec417c3a55e:111111' => 'Adobe Photoshop, Save for web, Quality 65',
    '016600f44a61cc5a5673c9bad85e23a3:111111' => 'Adobe Photoshop, Save for web, Quality 66',
    '91d7b4300c98c726aff7b19cbe098a3e:111111' => 'Adobe Photoshop, Save for web, Quality 67',
    'f9b83ba21b86a3d4ddb507e3edce490c:111111' => 'Adobe Photoshop, Save for web, Quality 68',
    'd312a23c8ecb3bf59bc11bbe17d79e55:111111' => 'Adobe Photoshop, Save for web, Quality 69',
    '240fffe5f8e2d8f3345b8175f9cb0a40:111111' => 'Adobe Photoshop, Save for web, Quality 70',
    'ba60a642bfb1a184c11e5561581d7115:111111' => 'Adobe Photoshop, Save for web, Quality 71',
    'c901580e589f58d309f8b50590cfe214:111111' => 'Adobe Photoshop, Save for web, Quality 72',
    'c244f94d84a016840c6ef06250c58ade:111111' => 'Adobe Photoshop, Save for web, Quality 73',
    '3589227bdd85f880f3337b492e895c5d:111111' => 'Adobe Photoshop, Save for web, Quality 74',
    'cae6fd91a423ff181d50bb9c26a0d392:111111' => 'Adobe Photoshop, Save for web, Quality 75',
    '7d8ee11ca66d2c22ff9ed1f778b5dbac:111111' => 'Adobe Photoshop, Save for web, Quality 76',
    'a16371762ce48953d42dfb5b77d1bfc6:111111' => 'Adobe Photoshop, Save for web, Quality 77',
    '204b111d4aaa85b430e86273a63fd004:111111' => 'Adobe Photoshop, Save for web, Quality 78',
    '6a243ac0b8575c2ed962070cd7d39e04:111111' => 'Adobe Photoshop, Save for web, Quality 79',
    '51879d6e5178d2282d5e8276ed4e2439:111111' => 'Adobe Photoshop, Save for web, Quality 80',
    'ca683ab6caaa3132bf661a0ebf32ef4e:111111' => 'Adobe Photoshop, Save for web, Quality 81',
    '5399adc3f21ecb30c96d6a94b38ab74c:111111' => 'Adobe Photoshop, Save for web, Quality 82',
    '43eb3b161279ccc1fb4f9cbe7b92398f:111111' => 'Adobe Photoshop, Save for web, Quality 83',
    '2d387641f4e94b6986908b3770fb762e:111111' => 'Adobe Photoshop, Save for web, Quality 84',
    '75ee5a0fd61559c6bf8e6ebc920c93b0:111111' => 'Adobe Photoshop, Save for web, Quality 85',
    '60d17e041a23d47b96c5aac86180a022:111111' => 'Adobe Photoshop, Save for web, Quality 86',
    '8e5290b1d12832ad259de92a53e1ef4e:111111' => 'Adobe Photoshop, Save for web, Quality 87',
    'dc19a48af9051bbdc54cf7e88c03f13e:111111' => 'Adobe Photoshop, Save for web, Quality 88',
    'c3fbc85c803ddc81c8882c03330b5b15:111111' => 'Adobe Photoshop, Save for web, Quality 89',
    '5e016a2d28f8ad3e7e27e4e2981031d2:111111' => 'Adobe Photoshop, Save for web, Quality 90',
    'ef4fa43f4d548e0687c4d4151a0bf1bd:111111' => 'Adobe Photoshop, Save for web, Quality 91',
    '00f03e367cd316b71de360c6e7af0e6b:111111' => 'Adobe Photoshop, Save for web, Quality 92',
    '982fc46fd167df238fbf23494a1ce761:111111' => 'Adobe Photoshop, Save for web, Quality 93',
    '6fd7b56ac6b58dc861e6021815fb5704:111111' => 'Adobe Photoshop, Save for web, Quality 94',
    'c6d9120293c8435cf6b40574b45756bb:111111' => 'Adobe Photoshop, Save for web, Quality 95',
    '1e133f4bf9f7c7c1e0accf44c0b1107d:111111' => 'Adobe Photoshop, Save for web, Quality 96',
    'fb91d6a8a1b72388d68130f551698865:111111' => 'Adobe Photoshop, Save for web, Quality 97',
    '4ea4e07900e04a3bd7572d4b59aa7a74:111111' => 'Adobe Photoshop, Save for web, Quality 98',
    '15e1d2321b96b355d4ad109a8d2fe882:111111' => 'Adobe Photoshop, Save for web, Quality 99',
    '234d8f310d75effc9f77beb1d3847f49:111111' => 'Adobe Photoshop, Save for web, Quality 100',

    # Adobe Photoshop Express (Release build number: 481589)
    'aeb34eb083acc888770d65e691497bcf:111111' => 'Adobe Photoshop Express, Original Size',

    # Apple QuickTime (Color)
    #
    # Tested with:
    # - QuickTime 7.5.0 (Win)
    # - QuickTime 7.5.0 (Mac)
    # - QuickTime 7.4.1 (Mac)
    # - GraphicConverter 4.4.4 (Mac)
    #   - Using the QuickTime library
    #   - Fixed subsampling to 221111
    #   - Use a subset of 101 of the 1024 values
    'dbdc6a6f3c9ffff19e67cfad2cc51ae4:221111' => 'Apple QuickTime, Quality 0-63 (0-6%)',
    '23bd5edb6224e03e2f7c282e04986553:221111' => 'Apple QuickTime, Quality 64-80 (6-8%)',
    '69747272d4079b78c2ee2ef0f5e63f30:221111' => 'Apple QuickTime, Quality 81-92 (8-9%)',
    '4f50903ec9314f739e460c79552a20c5:221111' => 'Apple QuickTime, Quality 93-101 (9-10%)',
    '2535e621267a9ff2e2d09148643e3389:221111' => 'Apple QuickTime, Quality 102-109 (10-11%)',
    '46a125048b572576eed271816b2cbad2:221111' => 'Apple QuickTime, Quality 110-116 (11%)',
    'ce05e4b45c53a6e321d9cf1061c62128:221111' => 'Apple QuickTime, Quality 117-122 (11-12%)',
    'c3b80241282d06ac5114f2750089589a:221111' => 'Apple QuickTime, Quality 123-127 (12%)',
    '9d91481900305fb9ad09339f0924f690:221111' => 'Apple QuickTime, Quality 128-133 (13%)',
    '348f4874d57ae6aae04ef96132374913:221111' => 'Apple QuickTime, Quality 134-137 (13%)',
    '4e3daadebe0517955b1c86fb1fbc1dc1:221111' => 'Apple QuickTime, Quality 138-142 (13-14%)',
    '599b6ad93f32b9d5ce67e1622338c379:221111' => 'Apple QuickTime, Quality 143-146 (14%)',
    '66aeb0f03343673eeed6462e0ce9e1aa:221111' => 'Apple QuickTime, Quality 147-150 (14-15%)',
    '13570e05da917fe51235ef66a295bc78:221111' => 'Apple QuickTime, Quality 151-154 (15%)',
    '25e399a9cf70fe7a13867b40ac2c3416:221111' => 'Apple QuickTime, Quality 155-157 (15%)',
    '25eb3d27e65659435a89e401edfab65f:221111' => 'Apple QuickTime, Quality 158-161 (15-16%)',
    'c3bb3c557e70b56a890b07236348518b:221111' => 'Apple QuickTime, Quality 162-164 (16%)',
    '792e93c41ac63451068b887b11ad0c2e:221111' => 'Apple QuickTime, Quality 165-167 (16%)',
    '35af99d11406974cf2ffa6676801b10c:221111' => 'Apple QuickTime, Quality 168-170 (16-17%)',
    '606e5652fc33c6a02328f0bd23ee9751:221111' => 'Apple QuickTime, Quality 171-173 (17%)',
    '9b62a9e4544cbc1033c67732ea0bbb08:221111' => 'Apple QuickTime, Quality 174-176 (17%)',
    '93ef48999d5659763a33c45a2a0fa784:221111' => 'Apple QuickTime, Quality 177-179 (17%)',
    '01b48291bfeccf2fadab996816225b9b:221111' => 'Apple QuickTime, Quality 180-182 (18%)',
    '613ef896fc4af5baad36e2680968a7ba:221111' => 'Apple QuickTime, Quality 183-184 (18%)',
    '758d37a9d3b91c0ba383d23f5a080d8f:221111' => 'Apple QuickTime, Quality 185-187 (18%)',
    '34457d32b9531f04696a52969e02dc1a:221111' => 'Apple QuickTime, Quality 188-189 (18%)',
    '6634cdad61e7a8e6fb3a4ba1a0416256:221111' => 'Apple QuickTime, Quality 190-191 (19%)',
    'e5b0739f8e02c6d481e0cdafe7326ae2:221111' => 'Apple QuickTime, Quality 192-194 (19%)',
    'fca91c73d4275748587f97b472b59280:221111' => 'Apple QuickTime, Quality 195-196 (19%)',
    '7dc9316230c4f197fb5d1b36f09cd883:221111' => 'Apple QuickTime, Quality 197-198 (19%)',
    '13d6536913342860ab993be8b141f644:221111' => 'Apple QuickTime, Quality 199-201 (19-20%)',
    'd835580b2be669d4aa6c68ead27c0c2f:221111' => 'Apple QuickTime, Quality 202-203 (20%)',
    '1e91365abfe1d9f7a008c363c834a66e:221111' => 'Apple QuickTime, Quality 204-205 (20%)',
    '6f7825365b673f9eb2ac050d27a21d1b:221111' => 'Apple QuickTime, Quality 206-207 (20%)',
    '2a98c0b884281080eefcdf98dd33fd6b:221111' => 'Apple QuickTime, Quality 208-209 (20%)',
    '0ffe7e9fc17393a338b95c345052b7c5:221111' => 'Apple QuickTime, Quality 210-211 (21%)',
    '9dd5c9717d0fd45486af4d26e59ebb35:221111' => 'Apple QuickTime, Quality 212-213 (21%)',
    '0ef85155f08194f8fed3f4e7197126e6:221111' => 'Apple QuickTime, Quality 214-215 (21%)',
    '35886289b5c8921f7932f895d7f1855d:221111' => 'Apple QuickTime, Quality 216-217 (21%)',
    '1d7ac617d70b1880be9c7ba16f96a3ec:221111' => 'Apple QuickTime, Quality 218 (21%)',
    'b26a53dce1477ac3970335df110bb240:221111' => 'Apple QuickTime, Quality 219-220 (21%)',
    'a333be9f2b13b53bfdf64bf5665f8e55:221111' => 'Apple QuickTime, Quality 221-222 (22%)',
    'f1e8d9b3d66fa34ec9a51a987b48a159:221111' => 'Apple QuickTime, Quality 223-224 (22%)',
    '20f37f34a9fd18089aa58fe77493a7b7:221111' => 'Apple QuickTime, Quality 225 (22%)',
    'b80e56d3ed0c4a8e1c6bb0c5a1d45ca9:221111' => 'Apple QuickTime, Quality 226-227 (22%)',
    'd14411ab659ac68209ee8c75b941cb48:221111' => 'Apple QuickTime, Quality 228-229 (22%)',
    'a5da49ac5bfe27aafda44bae107ae1c5:221111' => 'Apple QuickTime, Quality 230 (22%)',
    '9aa9359126240c0712610121371f870c:221111' => 'Apple QuickTime, Quality 231-232 (23%)',
    '5975500a23ab9a547ba149bf1aaa1893:221111' => 'Apple QuickTime, Quality 233-234 (23%)',
    'd03a4790dd96a862113b1a2408103ad6:221111' => 'Apple QuickTime, Quality 235 (23%)',
    '44d2a7baaf1e3f8c3d45e4e6272a39b1:221111' => 'Apple QuickTime, Quality 236-237 (23%)',
    '7755ba223679105c184be0ada8c99f92:221111' => 'Apple QuickTime, Quality 238 (23%)',
    '1ea3f373d0adf989e8416ecb11c38608:221111' => 'Apple QuickTime, Quality 239-240 (23%)',
    '12196a46c697fbb88d8bef279b52b106:221111' => 'Apple QuickTime, Quality 241 (24%)',
    '2183a6f77fe72f5c70726244dcabc963:221111' => 'Apple QuickTime, Quality 242-243 (24%)',
    'd1ca3a3723c1385d2989b199a7a30557:221111' => 'Apple QuickTime, Quality 244 (24%)',
    'd83207842d60965f9d194d89f3281ccd:221111' => 'Apple QuickTime, Quality 245-246 (24%)',
    '6ae041573525edd42e800e1b61d4313c:221111' => 'Apple QuickTime, Quality 247 (24%)',
    '6c42d12564d1c5706653a8ddb5375192:221111' => 'Apple QuickTime, Quality 248-249 (24%)',
    '671a071a1b17f49a774da3893f7199c7:221111' => 'Apple QuickTime, Quality 250 (24%)',
    'fb3c0cc15ad21b6c19576dd8d7d29a0e:221111' => 'Apple QuickTime, Quality 251 (25%)',
    'c9ce3dc3d0567f631e463cc3ff1b2e30:221111' => 'Apple QuickTime, Quality 252-253 (25%)',
    'b309a0dc90b16ac01f0798a04c3127e8:221111' => 'Apple QuickTime, Quality 254 (25%)',
    'cd15038a76bd8752c3afd14669816c2e:221111' => 'Apple QuickTime, Quality 255 (25%)',
    'd275e9aebd39cf411496caf6e54d0c5f:221111' => 'Apple QuickTime, Quality 256-257 (25%)',
    '5e75328df5dadca132bb83e0883ce522:221111' => 'Apple QuickTime, Quality 258 (25%)',
    'b04cbc1812939770d59461982cd9d32d:221111' => 'Apple QuickTime, Quality 259 (25%)',
    'ddf1f3b922ea51f6f4ca3cb6863eeae0:221111' => 'Apple QuickTime, Quality 260-261 (25%)',
    '5532e398abb0a455b528659e59c7cfd7:221111' => 'Apple QuickTime, Quality 262 (26%)',
    'f9bff3eeb4e94fb9ab4820184b0b6058:221111' => 'Apple QuickTime, Quality 263 (26%)',
    '081da80ed314194b571ff9880a7c11d3:221111' => 'Apple QuickTime, Quality 264-265 (26%)',
    '178aa0138d7a08be081aeff794956a71:221111' => 'Apple QuickTime, Quality 266 (26%)',
    '78bb04e3ced3eee51c78e94b421ecc26:221111' => 'Apple QuickTime, Quality 267 (26%)',
    '17782e930dc2cba42da909d95278fe9b:221111' => 'Apple QuickTime, Quality 268 (26%)',
    '093b011ce21ae794d3eca7c64eecf5b6:221111' => 'Apple QuickTime, Quality 269 (26%)',
    '4bf515768d1a06e4c529ebae3e03b4b5:221111' => 'Apple QuickTime, Quality 270-271 (26%)',
    '3dd79429ada0455422ff6605c1727456:221111' => 'Apple QuickTime, Quality 272 (27%)',
    '4be1504b9732d1d9f6265d0616bad21b:221111' => 'Apple QuickTime, Quality 273 (27%)',
    'b2118dc8e8b1762cc634e135a2a1893c:221111' => 'Apple QuickTime, Quality 274 (27%)',
    '70d843457698f46db30181ac616deb75:221111' => 'Apple QuickTime, Quality 275 (27%)',
    'f3c42f077883313db21c72bd240de05f:221111' => 'Apple QuickTime, Quality 276 (27%)',
    '56e9a02eb25508a9f71ad1a7cb9f9f4d:221111' => 'Apple QuickTime, Quality 277-278 (27%)',
    '1d956197da5eb19ffe8855a0e2a52c98:221111' => 'Apple QuickTime, Quality 279 (27%)',
    '3e37de5c00962684feba769939fce685:221111' => 'Apple QuickTime, Quality 280 (27%)',
    '5628aeb29bb04d9c5073bc1caf371f01:221111' => 'Apple QuickTime, Quality 281 (27%)',
    'a2e7b219d18177294378485759215f72:221111' => 'Apple QuickTime, Quality 282 (28%)',
    '537f40d0aae588fbce4cde9ba148604d:221111' => 'Apple QuickTime, Quality 283 (28%)',
    '14b7d58b539ad8d6f1c4f8fd82c91358:221111' => 'Apple QuickTime, Quality 284 (28%)',
    '58b302794024b9842657bbe7cb667577:221111' => 'Apple QuickTime, Quality 285 (28%)',
    '4be64c2782cbb36b757cdcadd756498a:221111' => 'Apple QuickTime, Quality 286 (28%)',
    '6c947f09bc02f87b257a26f9f5c77a77:221111' => 'Apple QuickTime, Quality 287 (28%)',
    '663b8a9dbd00efa78281f5028b35c503:221111' => 'Apple QuickTime, Quality 288 (28%)',
    'e5deb190a5e17492a01e8136afdfd6c1:221111' => 'Apple QuickTime, Quality 289 (28%)',
    'ba36e1298ce7fed908ee3e02b83ae7c3:221111' => 'Apple QuickTime, Quality 290 (28%)',
    'bc4abc4600f2efc0bdead1e4be78801b:221111' => 'Apple QuickTime, Quality 291-292 (28-29%)',
    '43887ad276efb9ca8e8110498b38d814:221111' => 'Apple QuickTime, Quality 293 (29%)',
    '3050624718ce9acc06f85c2fa0208cc7:221111' => 'Apple QuickTime, Quality 294 (29%)',
    'f6d9c8699e54823040b66c4b8e1361aa:221111' => 'Apple QuickTime, Quality 295 (29%)',
    'a9b2875fc3c21e7b998969c57f74fa7a:221111' => 'Apple QuickTime, Quality 296 (29%)',
    'b9be740b8a374a52808ad5ef6db2bfe7:221111' => 'Apple QuickTime, Quality 297 (29%)',
    '63f9786c6a9b8ef87c791818ddaba058:221111' => 'Apple QuickTime, Quality 298 (29%)',
    'a65113fd3b66ef137f9b1144367f731b:221111' => 'Apple QuickTime, Quality 299 (29%)',
    'ee58773aa7b774040d650365937cf173:221111' => 'Apple QuickTime, Quality 300 (29%)',
    '29a9ee0cae41784d90fa74d7cd240a3e:221111' => 'Apple QuickTime, Quality 301 (29%)',
    '369e1cfc338b45a239cb7db09778037e:221111' => 'Apple QuickTime, Quality 302 (29%)',
    '131ddd6eec5f51e825cf7afd9c7ab3b2:221111' => 'Apple QuickTime, Quality 303 (30%)',
    '88a2772be7b74a5a9b7ebbea28ddde47:221111' => 'Apple QuickTime, Quality 304 (30%)',
    '83e206dafb515f20a4b9a0c16f770940:221111' => 'Apple QuickTime, Quality 305 (30%)',
    'dddc3adae44a64457b05416affc2502e:221111' => 'Apple QuickTime, Quality 306 (30%)',
    'ff0758d87a0cbdb323fb93bf9ed1fdff:221111' => 'Apple QuickTime, Quality 307 (30%)',
    'b1b1a08ebaf13142b731c95771d97226:221111' => 'Apple QuickTime, Quality 308 (30%)',
    'd6e206f8224d6a3582fb1066b511437b:221111' => 'Apple QuickTime, Quality 309 (30%)',
    '2926d6bf5a27174bd9057bd6198413cd:221111' => 'Apple QuickTime, Quality 310 (30%)',
    '5d1b5e80f9777a636d1d5cb402fcfc32:221111' => 'Apple QuickTime, Quality 311 (30%)',
    'ae39c8775a10e34accdf2bba3bffc483:221111' => 'Apple QuickTime, Quality 312 (30%)',
    'bc6d3a9f349a97c5cde3f8fa4e1b5beb:221111' => 'Apple QuickTime, Quality 313 (31%)',
    '91bd468ca96fe548a7df9646b51880d1:221111' => 'Apple QuickTime, Quality 314 (31%)',
    '104c3b63e4ca667a4ee2e4250340052c:221111' => 'Apple QuickTime, Quality 315 (31%)',
    '64b80be38604eaecc99236b1f74a99f8:221111' => 'Apple QuickTime, Quality 316 (31%)',
    '284efada45882694778e65969f761478:221111' => 'Apple QuickTime, Quality 317 (31%)',
    'd8bd88390c27b2b05a0784eafd4b31ef:221111' => 'Apple QuickTime, Quality 318 (31%)',
    '99bf8158a4060d354b521f3d6f5648ac:221111' => 'Apple QuickTime, Quality 319 (31%)',
    '96c9e3cd827097ec03edc458fc1053e4:221111' => 'Apple QuickTime, Quality 320 (31%)',
    '272b5b12f7701be4cceba51e9d5dbf13:221111' => 'Apple QuickTime, Quality 321 (31%)',
    'dcc3ffcda228ab283d53e1dc2cb739ef:221111' => 'Apple QuickTime, Quality 322-324 (31-32%)',
    'c44701e8185306f5e6d09be16a2b0fbd:221111' => 'Apple QuickTime, Quality 325 (32%)',
    '476a1ebd043ed59e56d18dd6d08777d7:221111' => 'Apple QuickTime, Quality 326 (32%)',
    '26831dfc8d0dc1d202d50d6cf7b4f4a4:221111' => 'Apple QuickTime, Quality 327 (32%)',
    '00f929d549fdd9f89fbb10303445cc2c:221111' => 'Apple QuickTime, Quality 328 (32%)',
    '030736cda242f0583a7064cb60cc026e:221111' => 'Apple QuickTime, Quality 329 (32%)',
    '825fb58744c6c2432d232f5fb83a9597:221111' => 'Apple QuickTime, Quality 330 (32%)',
    '0bac94d5b6ef090da7875e294a7f8040:221111' => 'Apple QuickTime, Quality 331 (32%)',
    '16de07616490b8439576d837c74aefbe:221111' => 'Apple QuickTime, Quality 332 (32%)',
    '8190e844832ee8ea97492b509c728de4:221111' => 'Apple QuickTime, Quality 333 (33%)',
    '0bb6bf7365676f75d285bb38a40b8e3f:221111' => 'Apple QuickTime, Quality 334 (33%)',
    '04710d4ba5233b4f82bd260263f9e992:221111' => 'Apple QuickTime, Quality 335 (33%)',
    'fcb49e821b83f8436d450b03f1b1f182:221111' => 'Apple QuickTime, Quality 336 (33%)',
    '1bca645051a125cd2c3af262074f70e7:221111' => 'Apple QuickTime, Quality 337 (33%)',
    '7a5c04b63f9fe6af176efef387ba1f03:221111' => 'Apple QuickTime, Quality 338 (33%)',
    '24fff8dcfdc8640225fff020ad869c18:221111' => 'Apple QuickTime, Quality 339 (33%)',
    '08549fa433585b86d6eab75b6dcb1fe3:221111' => 'Apple QuickTime, Quality 340 (33%)',
    'caabe462a50217592c74902def037c07:221111' => 'Apple QuickTime, Quality 341 (33%)',
    '757e97f3490ebc5b74fd63792fb23992:221111' => 'Apple QuickTime, Quality 342 (33%)',
    'aa6072a632f7bae361c8d371aa022c57:221111' => 'Apple QuickTime, Quality 343 (33%)',
    '03809d08372d3a9fd86ff254854f45b7:221111' => 'Apple QuickTime, Quality 344 (34%)',
    'bdebcafc7b5f6b7fea114943e042df5e:221111' => 'Apple QuickTime, Quality 345 (34%)',
    '555dc90fb10df448f37c67ee7ec31bc2:221111' => 'Apple QuickTime, Quality 346 (34%)',
    'a2f2a404cd1c2278ef65f2a27c0365e0:221111' => 'Apple QuickTime, Quality 347 (34%)',
    'aee867276d6dc4ed4b682a454815acd1:221111' => 'Apple QuickTime, Quality 348 (34%)',
    '8f2f9e8433104cedb50c3e54577fcd00:221111' => 'Apple QuickTime, Quality 349 (34%)',
    '84c2067991afbb6851204f21f5d132ea:221111' => 'Apple QuickTime, Quality 350 (34%)',
    '2d2a77bf6078ab4f07261c76b637b597:221111' => 'Apple QuickTime, Quality 351 (34%)',
    'fc9d0f82571701e8b4cf764125ac0d2e:221111' => 'Apple QuickTime, Quality 352 (34%)',
    '75d4ffdc6c10675cb1b5bd002d4e0e41:221111' => 'Apple QuickTime, Quality 353 (34%)',
    '4b96c3457701c201f90d56af1a82d43b:221111' => 'Apple QuickTime, Quality 354 (35%)',
    'a56edeb9e571dc790a429c26ebc59976:221111' => 'Apple QuickTime, Quality 355 (35%)',
    '3673ce9ec4f6f916009d39282ff3a8d7:221111' => 'Apple QuickTime, Quality 356 (35%)',
    'b7279ade733ff1c88073971cebe6edd8:221111' => 'Apple QuickTime, Quality 357 (35%)',
    'b35f3358027aa4d2cca0c64425aa8f1b:221111' => 'Apple QuickTime, Quality 358 (35%)',
    '854d2e536bc92a9e2e3db3ff2c18e138:221111' => 'Apple QuickTime, Quality 359 (35%)',
    '48b6aa4f0258162cceb9d43e19c96043:221111' => 'Apple QuickTime, Quality 360 (35%)',
    '1dfb48b5955cf2a50011f52b9a05f1a4:221111' => 'Apple QuickTime, Quality 361 (35%)',
    'd67da3fcbac8975acffe7f1ab088f646:221111' => 'Apple QuickTime, Quality 362 (35%)',
    '7be7bded72d0ade6f907e3adcf62b391:221111' => 'Apple QuickTime, Quality 363 (35%)',
    '02c0554e4a004ceaddd0d7772e68a38b:221111' => 'Apple QuickTime, Quality 364 (36%)',
    'e7b9303f785f78a2cb27f83616c18726:221111' => 'Apple QuickTime, Quality 365 (36%)',
    '4a4a154781db3f5f500e8cf177a4b446:221111' => 'Apple QuickTime, Quality 366 (36%)',
    '7e4b44f2900a405e7b85090af7d40298:221111' => 'Apple QuickTime, Quality 367 (36%)',
    '76aa290370382de8a3516f73389f9350:221111' => 'Apple QuickTime, Quality 368 (36%)',
    '912779b5b7c935f2b533af0f400402f3:221111' => 'Apple QuickTime, Quality 369 (36%)',
    '91c7f694fbf07321037a838c3a4d6e7d:221111' => 'Apple QuickTime, Quality 370 (36%)',
    'f98a4286abf1cbe8bf46fba1e78cec61:221111' => 'Apple QuickTime, Quality 371 (36%)',
    '2fbbcce5a035d6215e4851a0ae63481f:221111' => 'Apple QuickTime, Quality 372 (36%)',
    '8d8fab3b6b7386a4e81f10c15a7abaa5:221111' => 'Apple QuickTime, Quality 373 (36%)',
    '82e672854d7e00d47a988855b95d2f7f:221111' => 'Apple QuickTime, Quality 374 (37%)',
    '9fffe6b2fbbce23598c19e6cd177adb0:221111' => 'Apple QuickTime, Quality 375 (37%)',
    '3f69293a4abeb2201004e7241fe22c75:221111' => 'Apple QuickTime, Quality 376 (37%)',
    'a01d3a7766c7c593a79ff6c63433860a:221111' => 'Apple QuickTime, Quality 377 (37%)',
    '787ed74c3d5570c03f98804bc9d0c448:221111' => 'Apple QuickTime, Quality 378 (37%)',
    '12239aa16bb4091d8f873f9536e40371:221111' => 'Apple QuickTime, Quality 379 (37%)',
    'f45d495d3b470eadba70bcca888042b3:221111' => 'Apple QuickTime, Quality 380 (37%)',
    '97346edee67c3afea7823c72e57cb6c5:221111' => 'Apple QuickTime, Quality 381 (37%)',
    '5b35e4bc9cbbc353b8e4b73132324088:221111' => 'Apple QuickTime, Quality 382 (37%)',
    'a43b370edaaee853bb16e46ee4a002e8:221111' => 'Apple QuickTime, Quality 383 (37%)',
    '7f3d110973a4d7d5824724c4e577b407:221111' => 'Apple QuickTime, Quality 384 (38%)',
    '0e36104efe90a5a77e9b686d0a6528ab:221111' => 'Apple QuickTime, Quality 385 (38%)',
    '0bc0941c2a59d9a12b66d1d34117cfd7:221111' => 'Apple QuickTime, Quality 386 (38%)',
    '342e3bddb81140ea9df00400a46461d7:221111' => 'Apple QuickTime, Quality 387 (38%)',
    'ad8bbd6b23b87950d1b76278fbb7de87:221111' => 'Apple QuickTime, Quality 388 (38%)',
    'd58f5d339b69e1296911a3387cc664a4:221111' => 'Apple QuickTime, Quality 389 (38%)',
    'c740804ef8493bb467744e1cdb8882c1:221111' => 'Apple QuickTime, Quality 390 (38%)',
    'ab975404bdb713bb6a58ac560330aaf1:221111' => 'Apple QuickTime, Quality 391 (38%)',
    'eb68a0ff9c83267e5fb5e998365b4480:221111' => 'Apple QuickTime, Quality 392 (38%)',
    '9c1865e7fdc0289dc5fe8f4c1f65577e:221111' => 'Apple QuickTime, Quality 393 (38%)',
    '21db4122ad8183006542018e53e0c653:221111' => 'Apple QuickTime, Quality 394 (38%)',
    '734255167cbd052200fb4c474f05bcd9:221111' => 'Apple QuickTime, Quality 395 (39%)',
    'e5dcd017a9734f9f0e18b515c7fa1787:221111' => 'Apple QuickTime, Quality 396 (39%)',
    'c58a47c2e3dc737b9591420812b9cc27:221111' => 'Apple QuickTime, Quality 397 (39%)',
    'ea25d0beaa91434a14348fb60f5cff31:221111' => 'Apple QuickTime, Quality 398 (39%)',
    '21f8a4a67742edcde0ac854522028c9f:221111' => 'Apple QuickTime, Quality 399 (39%)',
    '8b8dc34912d8b18742a7670be4b1c867:221111' => 'Apple QuickTime, Quality 400 (39%)',
    'f7adac1fb54bb1fc566b66822122a9c6:221111' => 'Apple QuickTime, Quality 401 (39%)',
    '0fe0c7e65c0696d9e76ad819d61e44ae:221111' => 'Apple QuickTime, Quality 402 (39%)',
    '6dabf05ddc213b650ff08aa9a8cb9f50:221111' => 'Apple QuickTime, Quality 403 (39%)',
    '226788e417078cdcc5aa989379b9e824:221111' => 'Apple QuickTime, Quality 404 (39%)',
    '9366dde6c37f4cac36a8e8cea4d5f51c:221111' => 'Apple QuickTime, Quality 405 (40%)',
    '3ce5057aaf0ff155ee69d66591c8290d:221111' => 'Apple QuickTime, Quality 406 (40%)',
    '58fe81014a9ee26a7bd393c8e31f4011:221111' => 'Apple QuickTime, Quality 407 (40%)',
    '41e44bae14ab49d1b0f06438d34cb316:221111' => 'Apple QuickTime, Quality 408 (40%)',
    '80ccbe5645cc62ebd4ae7b2128b42d91:221111' => 'Apple QuickTime, Quality 409 (40%)',
    'b08af3cffd1904e8a8cfbbba71077069:221111' => 'Apple QuickTime, Quality 410 (40%)',
    'abb56efe234d4b8fdf50016a19c63684:221111' => 'Apple QuickTime, Quality 411 (40%)',
    'e41806d0928fbb5552225e10db7b55d0:221111' => 'Apple QuickTime, Quality 412 (40%)',
    'ae9de0a8343d730e2e6a358849c29a4e:221111' => 'Apple QuickTime, Quality 413 (40%)',
    'b83146f54d17b2c8e242f7f36dc36f19:221111' => 'Apple QuickTime, Quality 414 (40%)',
    'a748fdfe8d6dc9493253908410e517eb:221111' => 'Apple QuickTime, Quality 415 (41%)',
    'd4bb4c59b5284630a4c716a0290d9091:221111' => 'Apple QuickTime, Quality 416 (41%)',
    'de93dd8ab7918b25f191923f4a43a5c2:221111' => 'Apple QuickTime, Quality 417 (41%)',
    '0807f8b3b41b01054509858fa74dcf4d:221111' => 'Apple QuickTime, Quality 418 (41%)',
    'efd780e10dcd0ab8ca0a0f4f3cb215d3:221111' => 'Apple QuickTime, Quality 419 (41%)',
    '2ec3d0ec37690e40f009b7a9f9b17c49:221111' => 'Apple QuickTime, Quality 420 (41%)',
    'a4680e71907e5c6f7b18e20e46286412:221111' => 'Apple QuickTime, Quality 421 (41%)',
    '8e54abf2320cca661b6dd67b7658c9f3:221111' => 'Apple QuickTime, Quality 422 (41%)',
    '00f76480eafd05aa5267053aec3aa122:221111' => 'Apple QuickTime, Quality 423 (41%)',
    '849bce1254b14d44e24a6b419c385597:221111' => 'Apple QuickTime, Quality 424 (41%)',
    '8a91452f2df82874183be50601242106:221111' => 'Apple QuickTime, Quality 425 (42%)',
    'a670182cd48f37dd16652db878791a7a:221111' => 'Apple QuickTime, Quality 426 (42%)',
    'c2afe9aca67de0276a6fb507861c3e80:221111' => 'Apple QuickTime, Quality 427 (42%)',
    '18836b72e5399e2a19cd6420562ab1ff:221111' => 'Apple QuickTime, Quality 428 (42%)',
    'cb207af75faf8ee1ef0ca3caa593bb69:221111' => 'Apple QuickTime, Quality 429 (42%)',
    '7a318965f27e3c09d11f53cbb10a872b:221111' => 'Apple QuickTime, Quality 430 (42%)',
    'b0a501129cb83e54f97006610ec9ed64:221111' => 'Apple QuickTime, Quality 431 (42%)',
    'f1a8af8c0abe4b3423d5ac8c6273a7ca:221111' => 'Apple QuickTime, Quality 432 (42%)',
    '56224ea0ac2fccb92cbe9702896f9796:221111' => 'Apple QuickTime, Quality 433 (42%)',
    '2f2101a8450c617a09ccad472c275b88:221111' => 'Apple QuickTime, Quality 434 (42%)',
    '13e218420429e2c94d4b9474ab03f8e4:221111' => 'Apple QuickTime, Quality 435 (42%)',
    '41061e1cdb97926ed5bded3da11af209:221111' => 'Apple QuickTime, Quality 436 (43%)',
    'cc3e4dc4e190d00a12bd03199efdcc6d:221111' => 'Apple QuickTime, Quality 437 (43%)',
    '36b2371ec6df13143af12d600232c2ab:221111' => 'Apple QuickTime, Quality 438 (43%)',
    '71d0e3444a4c82cf39048ba8cf7b1d5f:221111' => 'Apple QuickTime, Quality 439 (43%)',
    '9f48e71f610caa47b94d3e474608cb3d:221111' => 'Apple QuickTime, Quality 440 (43%)',
    'dcecd4f366d521e118e94d87ef915caa:221111' => 'Apple QuickTime, Quality 441 (43%)',
    'cf0a070dc9b4a8983b50f8e3f105b857:221111' => 'Apple QuickTime, Quality 442 (43%)',
    '5504a428191bc87e5c1ba4b5e9984a37:221111' => 'Apple QuickTime, Quality 443 (43%)',
    'de6a322383022ee8d966e848a2df4f28:221111' => 'Apple QuickTime, Quality 444 (43%)',
    '163be99e863436e9b3d32615785ec8e1:221111' => 'Apple QuickTime, Quality 445 (43%)',
    '958185c48000065b5b8d03b0f975d95b:221111' => 'Apple QuickTime, Quality 446 (44%)',
    '2d719cb263e284fc8621bbec1fe52cd5:221111' => 'Apple QuickTime, Quality 447 (44%)',
    '8f0f54955cf19689f38df36715908b76:221111' => 'Apple QuickTime, Quality 448 (44%)',
    '737c61e006222488645fa2e007f83f3c:221111' => 'Apple QuickTime, Quality 449 (44%)',
    'bb342113b57cf66ce0cf3a09fae5fd16:221111' => 'Apple QuickTime, Quality 450 (44%)',
    'cd3ed5b396580d8e9f0cb7b78baed8b8:221111' => 'Apple QuickTime, Quality 451 (44%)',
    'e7914fbf6c9b2a127af3676726e6bd8b:221111' => 'Apple QuickTime, Quality 452 (44%)',
    '377a7b50c2d7484255bbbf537bf9fa86:221111' => 'Apple QuickTime, Quality 453 (44%)',
    '09f9009406d2fe8dfa1b35236f8b1bdb:221111' => 'Apple QuickTime, Quality 454 (44%)',
    '78b0e590ea36cb11c495097049022d2e:221111' => 'Apple QuickTime, Quality 455 (44%)',
    '984c0f34636e6197b508265f17cbd6c9:221111' => 'Apple QuickTime, Quality 456 (45%)',
    '7fca22065811c0efe6599a15ca38f05e:221111' => 'Apple QuickTime, Quality 457 (45%)',
    'd55d9744065708d7b6fa7fb6e8eb2453:221111' => 'Apple QuickTime, Quality 458 (45%)',
    '63ef900bf59d41003a0e0602baa60681:221111' => 'Apple QuickTime, Quality 459 (45%)',
    'e8b31dbd18c91229a3c40356efeb2622:221111' => 'Apple QuickTime, Quality 460 (45%)',
    '8cb1bb16c6fa524199dad5513386d225:221111' => 'Apple QuickTime, Quality 461 (45%)',
    'c53438ece0552cedb1ec0d50ad2d5dbe:221111' => 'Apple QuickTime, Quality 462 (45%)',
    '5db6302d7e68c1a274139033681b8fcc:221111' => 'Apple QuickTime, Quality 463 (45%)',
    'acd1d5ec1787c9d346d87c281a7b6da0:221111' => 'Apple QuickTime, Quality 464-465 (45%)',
    'e66c03f97b19213f385136f014c78ac1:221111' => 'Apple QuickTime, Quality 466-467 (46%)',
    '23816ed847127a41e3c7f52e04072e41:221111' => 'Apple QuickTime, Quality 468 (46%)',
    'ebce337e9ef5a07775cebe40d7623862:221111' => 'Apple QuickTime, Quality 469 (46%)',
    '09563e47ab174b05fb19f722e9aa43c3:221111' => 'Apple QuickTime, Quality 470 (46%)',
    '545e14e832fb81f032526a9efcbf2450:221111' => 'Apple QuickTime, Quality 471 (46%)',
    'edda5d6ae456d4cdccec80e390ac9279:221111' => 'Apple QuickTime, Quality 472 (46%)',
    '9a43fc0aa6223673c32a49fb76d6525c:221111' => 'Apple QuickTime, Quality 473 (46%)',
    'bdaf3e68c2925cbaad3864359fdbbb77:221111' => 'Apple QuickTime, Quality 474 (46%)',
    '4f83a1a8338a8e6e70eaa58cd236f62a:221111' => 'Apple QuickTime, Quality 475 (46%)',
    '879b320cfd6e27b2e283b573483bda81:221111' => 'Apple QuickTime, Quality 476 (46%)',
    '2748fa249a86361b1b5f0662a88abdb3:221111' => 'Apple QuickTime, Quality 477 (47%)',
    '32eb803f68d72719267a1313548e7180:221111' => 'Apple QuickTime, Quality 478 (47%)',
    '771eeb43856b1821a271b0aa8398a243:221111' => 'Apple QuickTime, Quality 479 (47%)',
    '653c5006512bb3aaa1e6a4e77078b630:221111' => 'Apple QuickTime, Quality 480 (47%)',
    'c5c102ba5f004d49656f424d89e9773c:221111' => 'Apple QuickTime, Quality 481 (47%)',
    'd5220fcfa99764e440684fbac6273cff:221111' => 'Apple QuickTime, Quality 482 (47%)',
    'd300f18258f46060d89c994dbc370131:221111' => 'Apple QuickTime, Quality 483 (47%)',
    '60258dcc1e3a81858d176080ef774730:221111' => 'Apple QuickTime, Quality 484 (47%)',
    '8bda9fb1ed75249ac5b2feaad7b51d2f:221111' => 'Apple QuickTime, Quality 485 (47%)',
    '9e334af92d75ab7d4ea1a9816840ea73:221111' => 'Apple QuickTime, Quality 486 (47%)',
    'c40a38c96832a6042c6ddfc9754c1d6d:221111' => 'Apple QuickTime, Quality 487 (48%)',
    '30be130aa27d0b91d6f55ed9b1cd6c84:221111' => 'Apple QuickTime, Quality 488 (48%)',
    '3601e95d6cd507065d46b3f058229d91:221111' => 'Apple QuickTime, Quality 489 (48%)',
    '9127f8ddd20e583523bc848e99061126:221111' => 'Apple QuickTime, Quality 490 (48%)',
    '0d605d279c48a74ef71a24e89ca426a8:221111' => 'Apple QuickTime, Quality 491 (48%)',
    'a5cd2d8592e1c45b67cfb3009d07fb49:221111' => 'Apple QuickTime, Quality 492 (48%)',
    'e346ce6e3bee6abff16420f5ba95ceb9:221111' => 'Apple QuickTime, Quality 493 (48%)',
    '03295b26893cab9c7dea4ec15ed56d08:221111' => 'Apple QuickTime, Quality 494 (48%)',
    '76af24fe94edf8f3992e38c1dd6eebce:221111' => 'Apple QuickTime, Quality 495 (48%)',
    '3c58e82299d87346d37023ea015f3e80:221111' => 'Apple QuickTime, Quality 496 (48%)',
    'aa8940194463b7adc14f20dbee9c6a75:221111' => 'Apple QuickTime, Quality 497 (49%)',
    '68a808b23bfa8096e04006171926b72c:221111' => 'Apple QuickTime, Quality 498 (49%)',
    '34c0043b98d09193beda0cf5d1ada274:221111' => 'Apple QuickTime, Quality 499 (49%)',
    '5881bb3c6e7e2ac43983b4b1e947a6c3:221111' => 'Apple QuickTime, Quality 500 (49%)',
    'a0a7061bc100f051a3c5470559661138:221111' => 'Apple QuickTime, Quality 501 (49%)',
    '62e6812d1f7935adddd1a69227cdf626:221111' => 'Apple QuickTime, Quality 502 (49%)',
    '94f6dbd754fb4ba3c92698d5f08084f9:221111' => 'Apple QuickTime, Quality 503 (49%)',
    'b7257ba67e4b38b7ccdca2a65d60c970:221111' => 'Apple QuickTime, Quality 504 (49%)',
    '5a19d6130b03080dfedef45b6415f4f8:221111' => 'Apple QuickTime, Quality 505 (49%)',
    '2b3262e10b1563600a5f0738fec342ed:221111' => 'Apple QuickTime, Quality 506 (49%)',
    '295cb1e2772312ba5cd546966d1aa70d:221111' => 'Apple QuickTime, Quality 507 (50%)',
    'c910bcb7b9e8967b87cfa08229d9ca34:221111' => 'Apple QuickTime, Quality 508 (50%)',
    '61c8506b490d5e596151b951ffa7a14f:221111' => 'Apple QuickTime, Quality 509 (50%)',
    'ae15629cecc940fef9f24ad9f207fa10:221111' => 'Apple QuickTime, Quality 510 (50%)',
    'c8ef3c50ca99c44ea13f1692ac1190dc:221111' => 'Apple QuickTime, Quality 511 (50%)',
    '50500b1272433ef5c9c96f16069fbdf1:221111' => 'Apple QuickTime, Quality 512 (50%)',
    'e41e5416e21dbfb5a41f006b3485f5bb:221111' => 'Apple QuickTime, Quality 513 (50%)',
    'd606add3e7590885ac8978af6d09a2aa:221111' => 'Apple QuickTime, Quality 514 (50%)',
    'cc3936c39c298ef67d9196d0254b0c19:221111' => 'Apple QuickTime, Quality 515 (50%)',
    'e83f8505dc3f5f46b37e22b590f71b98:221111' => 'Apple QuickTime, Quality 516 (50%)',
    '2a8e27e03b6e1555335c91231c452bba:221111' => 'Apple QuickTime, Quality 517 (50%)',
    '52ab880d25db7b36137e2a3c04987c9a:221111' => 'Apple QuickTime, Quality 518 (51%)',
    'ce2335cc1f8289deda620877f50fd90d:221111' => 'Apple QuickTime, Quality 519 (51%)',
    'bbad0e19b252268530df19c563aa9176:221111' => 'Apple QuickTime, Quality 520 (51%)',
    '3184b71ca26bfe0c80811cf10423fa92:221111' => 'Apple QuickTime, Quality 521 (51%)',
    '37802f44dab089a35e03b94a298b19da:221111' => 'Apple QuickTime, Quality 522 (51%)',
    '5e528bd6778792490c6cf292cf9ba8df:221111' => 'Apple QuickTime, Quality 523 (51%)',
    'fbe0f5b89f266ff382f2b14c70a83097:221111' => 'Apple QuickTime, Quality 524 (51%)',
    'a4bfd80e0c8b9ae7a1114d79a7b63ad6:221111' => 'Apple QuickTime, Quality 525 (51%)',
    'abdf532dc2005805db7d8d0214227146:221111' => 'Apple QuickTime, Quality 526 (51%)',
    '2a7ec778642b15b8bce238f7b63ef537:221111' => 'Apple QuickTime, Quality 527 (51%)',
    '70073f02f04ee893510bceb09e411d53:221111' => 'Apple QuickTime, Quality 528 (52%)',
    '345d210b180a45bd23b0c7931c59c263:221111' => 'Apple QuickTime, Quality 529 (52%)',
    'ddbc4e6566bbcc74b6205526393ef468:221111' => 'Apple QuickTime, Quality 530 (52%)',
    '420af34c4f718cc0a10de5285140b6e0:221111' => 'Apple QuickTime, Quality 531 (52%)',
    '16b6c2d8688113b1a28afbbc57f46f80:221111' => 'Apple QuickTime, Quality 532-533 (52%)',
    '4a2a0e381fed49e5d5ba074998652561:221111' => 'Apple QuickTime, Quality 534 (52%)',
    '50f1255f2424b2de5b930751ddf24842:221111' => 'Apple QuickTime, Quality 535 (52%)',
    '80f2c05e2ad3524f18dd55bac10ee2e3:221111' => 'Apple QuickTime, Quality 536 (52%)',
    '6c0916ab5aa02602cc682bcdbc22369e:221111' => 'Apple QuickTime, Quality 537 (52%)',
    'f7e5656e1f2cf036e9a57a6c02373398:221111' => 'Apple QuickTime, Quality 538 (53%)',
    '92044affd220e31ee953aff021144b29:221111' => 'Apple QuickTime, Quality 539 (53%)',
    '2bafe4b75b8a105d72e981b21fe3b6cf:221111' => 'Apple QuickTime, Quality 540 (53%)',
    'ca0e84028714f19cf20cb868d1cd346c:221111' => 'Apple QuickTime, Quality 541 (53%)',
    'db1ae392a31d30cd5564dc7bbea24019:221111' => 'Apple QuickTime, Quality 542 (53%)',
    '4417e739b9244781987769c2177abc6f:221111' => 'Apple QuickTime, Quality 543 (53%)',
    'f1de58c1c6a48dc36ce7e8c69636539c:221111' => 'Apple QuickTime, Quality 544 (53%)',
    '10c931d7bff7bfcc20e37f0868887228:221111' => 'Apple QuickTime, Quality 545 (53%)',
    'e082971717023e667f3d922bbccf089b:221111' => 'Apple QuickTime, Quality 546 (53%)',
    '5a285190351b16fee0eb14778280d74f:221111' => 'Apple QuickTime, Quality 547 (53%)',
    'aad0e2cd42c5adaec41080a05be4ffdc:221111' => 'Apple QuickTime, Quality 548 (54%)',
    'de802b8c64d7f854081c7df6ed345b43:221111' => 'Apple QuickTime, Quality 549 (54%)',
    '82f45d11d651d93a67995965b94aa649:221111' => 'Apple QuickTime, Quality 550 (54%)',
    '617c4c853344ef079f4a1f1062672e8c:221111' => 'Apple QuickTime, Quality 551-552 (54%)',
    '90d96923be1883e6ee15a9d0d32a114c:221111' => 'Apple QuickTime, Quality 553 (54%)',
    '307c47179fdad179b5f962228c115db8:221111' => 'Apple QuickTime, Quality 554 (54%)',
    'a4683813bdf6e2bd429c4c5676128384:221111' => 'Apple QuickTime, Quality 555 (54%)',
    '8e3cfc2fc9cfbba0f6aed9850504ebb6:221111' => 'Apple QuickTime, Quality 556 (54%)',
    'a8f8928c72b69049e1da7639e977c9c7:221111' => 'Apple QuickTime, Quality 557 (54%)',
    '1fb4c8af2d70cdeecab3fd9fc882e0ce:221111' => 'Apple QuickTime, Quality 558 (54%)',
    '98ddda3b0ada32ce919b9af9df4054dd:221111' => 'Apple QuickTime, Quality 559 (55%)',
    'b43ab5c404469c416e853e52497b3f0d:221111' => 'Apple QuickTime, Quality 560 (55%)',
    'd9696efa02b9de813caf8d684b06346f:221111' => 'Apple QuickTime, Quality 561 (55%)',
    'be63c4e967eff819bd8a052a561a4576:221111' => 'Apple QuickTime, Quality 562 (55%)',
    'f984581f90913e44f3898fffd8fce8b0:221111' => 'Apple QuickTime, Quality 563 (55%)',
    '082779cf55f6b922036f11b74df54110:221111' => 'Apple QuickTime, Quality 564 (55%)',
    '066fd6cb3a5dd994fc6159987afde581:221111' => 'Apple QuickTime, Quality 565 (55%)',
    '133351a0f39427f1199312585cd6c997:221111' => 'Apple QuickTime, Quality 566 (55%)',
    '6c23da63c864f1433ec198ae202e56f0:221111' => 'Apple QuickTime, Quality 567 (55%)',
    '7d1819ccce2756fcf6dfbb67565c2552:221111' => 'Apple QuickTime, Quality 568 (55%)',
    'd9794fa54e2ef47be48b972cdca910c2:221111' => 'Apple QuickTime, Quality 569 (56%)',
    '085db73bd47194c8fdf567fc619c3b62:221111' => 'Apple QuickTime, Quality 570 (56%)',
    '9866add6e1d251e1d4c40793f4300dce:221111' => 'Apple QuickTime, Quality 571 (56%)',
    '1b9e7e39831b05b058025ae0a7482d44:221111' => 'Apple QuickTime, Quality 572 (56%)',
    'a8b52e666bd3d81404c0f8915ac18b43:221111' => 'Apple QuickTime, Quality 573 (56%)',
    'e34d11f979458a87492b57eabfd4f4ea:221111' => 'Apple QuickTime, Quality 574 (56%)',
    '4295c1330dec60585760cbb05b79662d:221111' => 'Apple QuickTime, Quality 575 (56%)',
    '9205cc28769d94d6d00c25804ac70a88:221111' => 'Apple QuickTime, Quality 576 (56%)',
    'b5648f13228d20fd7ae81965394f7515:221111' => 'Apple QuickTime, Quality 577 (56%)',
    'c8f02bf550c40daa39b28911a4ef5a69:221111' => 'Apple QuickTime, Quality 578 (56%)',
    'c2b23a91d377ce2d99ac4109f2740069:221111' => 'Apple QuickTime, Quality 579 (57%)',
    'c0cecb47363aff00a2764a915f95cd35:221111' => 'Apple QuickTime, Quality 580 (57%)',
    'fc0d8f17be060220464fe7bc0a2d754e:221111' => 'Apple QuickTime, Quality 581 (57%)',
    '63b59904874e5e427ddecb37e12f90c7:221111' => 'Apple QuickTime, Quality 582 (57%)',
    'df6535865562ce7cbf08e9368e991a95:221111' => 'Apple QuickTime, Quality 583 (57%)',
    '74523ad3424dcff6aa697c3ce433ad4e:221111' => 'Apple QuickTime, Quality 584 (57%)',
    'cbdec670ec6d9105277434b304226920:221111' => 'Apple QuickTime, Quality 585 (57%)',
    '68783ed0a7956cf0b7a1b2787e756213:221111' => 'Apple QuickTime, Quality 586 (57%)',
    'a431976a61e281e7b9d808f094b74d2e:221111' => 'Apple QuickTime, Quality 587 (57%)',
    '8e2a66454fb149552d4538d53ec033aa:221111' => 'Apple QuickTime, Quality 588 (57%)',
    '49222c4a3be01e93baad695bba63b254:221111' => 'Apple QuickTime, Quality 589 (58%)',
    'ab8fe796c87f9f61cedbfa64af9f5dec:221111' => 'Apple QuickTime, Quality 590 (58%)',
    '9bfe788e7ae4bc9cbe76d36f9a2b1b5e:221111' => 'Apple QuickTime, Quality 591 (58%)',
    '22b5f11b635ea5484469708cd7e6e3d9:221111' => 'Apple QuickTime, Quality 592 (58%)',
    '44e36eb25c6f9e313ef2a8f4c520c335:221111' => 'Apple QuickTime, Quality 593 (58%)',
    'c71aa81fb12b378dd31a1ca128942f76:221111' => 'Apple QuickTime, Quality 594 (58%)',
    'e56ca8f4da20395ec1f87d380198fa0a:221111' => 'Apple QuickTime, Quality 595 (58%)',
    '38f4d508dcf9c82d9488b42a2487b191:221111' => 'Apple QuickTime, Quality 596 (58%)',
    '3da1e7270e0900a17a0a4ff8d3c9a488:221111' => 'Apple QuickTime, Quality 597 (58%)',
    '900fee18a5f6d1dc3fd856d3d92f5414:221111' => 'Apple QuickTime, Quality 598 (58%)',
    '3c724d4b5d8cbe203ebbf92ea8e22808:221111' => 'Apple QuickTime, Quality 599 (58%)',
    'e7e7befa282a985a0532634f360df7db:221111' => 'Apple QuickTime, Quality 600 (59%)',
    '1f21bf5b7e0e79c229ef4d06fc9d3cc8:221111' => 'Apple QuickTime, Quality 601 (59%)',
    '3bb09b202acd618286d26a33f688f7c7:221111' => 'Apple QuickTime, Quality 602 (59%)',
    '9717c5a17cbffdfaa2e5d3769b87fbc5:221111' => 'Apple QuickTime, Quality 603 (59%)',
    'ffa7874d293c62ecc55c098b8f305ae1:221111' => 'Apple QuickTime, Quality 604 (59%)',
    'e68841bf28d33d749d0031bfe3a5219c:221111' => 'Apple QuickTime, Quality 605 (59%)',
    '5862c8c2b241a9760f6804d970eefd66:221111' => 'Apple QuickTime, Quality 606 (59%)',
    '1d069604250e871bd92a4a24c7be2bd5:221111' => 'Apple QuickTime, Quality 607 (59%)',
    '3806bcbefd350e8791be95dfc62bab27:221111' => 'Apple QuickTime, Quality 608 (59%)',
    '490b035a665ef80c7b48804461d55b7f:221111' => 'Apple QuickTime, Quality 609 (59%)',
    'bd6943a8c92a14e74d2b24052a19400a:221111' => 'Apple QuickTime, Quality 610 (60%)',
    '93e725418f46b2a70723523bef0979fe:221111' => 'Apple QuickTime, Quality 611 (60%)',
    '5b66fa5c0c1ba746289747229193cfb0:221111' => 'Apple QuickTime, Quality 612 (60%)',
    '1bdac971e8cddd198ad3123849370037:221111' => 'Apple QuickTime, Quality 613 (60%)',
    '20f7b70185f4b324a8451ac4657c1d66:221111' => 'Apple QuickTime, Quality 614 (60%)',
    '86ab18d6c1359a424f303fcfd0930df2:221111' => 'Apple QuickTime, Quality 615 (60%)',
    '0d70031c9962dba7c39da59ada2f1660:221111' => 'Apple QuickTime, Quality 616-617 (60%)',
    '9cb9a256b6deb481cf13e5230fe87dbb:221111' => 'Apple QuickTime, Quality 618-619 (60%)',
    'cb34e1a0e18a4dd7ffe823f9c92b3622:221111' => 'Apple QuickTime, Quality 620 (61%)',
    'ac015afc1d80314edd832aebfb495d25:221111' => 'Apple QuickTime, Quality 621 (61%)',
    '4974cc7044768888244b324449a238ab:221111' => 'Apple QuickTime, Quality 622 (61%)',
    'bc4541f5bc4d58b99b53d24f3f520b32:221111' => 'Apple QuickTime, Quality 623 (61%)',
    'ec2fd56a50df0e42498018d441a3aa75:221111' => 'Apple QuickTime, Quality 624 (61%)',
    '153a6f0994d16003aa4f1112e6757467:221111' => 'Apple QuickTime, Quality 625 (61%)',
    'ed6b90ca62ed648d1102e1c506a0af26:221111' => 'Apple QuickTime, Quality 626 (61%)',
    '18593e50c21c8ad521b30933ef7479b1:221111' => 'Apple QuickTime, Quality 627 (61%)',
    '6f96ed52a987d67e8d950b2627d3fbc2:221111' => 'Apple QuickTime, Quality 628 (61%)',
    'e76d86e8de4f0bf9e58cd389e0a8c117:221111' => 'Apple QuickTime, Quality 629 (61%)',
    '4fa27c83741226576ac6359cd4f6248e:221111' => 'Apple QuickTime, Quality 630 (62%)',
    'be010732a7783ee345548a1eb95d024a:221111' => 'Apple QuickTime, Quality 631 (62%)',
    '6700663d4ebaeb394bfd3c85597347b5:221111' => 'Apple QuickTime, Quality 632 (62%)',
    '34dba33043aa5ee317b7649242e702b1:221111' => 'Apple QuickTime, Quality 633 (62%)',
    '821d7e59bcf756171b7644ec5736266e:221111' => 'Apple QuickTime, Quality 634 (62%)',
    'd81683c0458d9ad72751530d6fbc1389:221111' => 'Apple QuickTime, Quality 635 (62%)',
    '4c04d6fe904a4b6ff8b25c9f0e9f0a16:221111' => 'Apple QuickTime, Quality 636 (62%)',
    '8efda55d6186d9867189c5cb572c5413:221111' => 'Apple QuickTime, Quality 637 (62%)',
    '25497c83113bd738e89d91bd48d7086c:221111' => 'Apple QuickTime, Quality 638 (62%)',
    '84c8e142e6d27734b126f76653b9199d:221111' => 'Apple QuickTime, Quality 639 (62%)',
    'bdaf13038b56b5701f60300528f8a89c:221111' => 'Apple QuickTime, Quality 640-641 (63%)',
    'b015ada43293b8d5bd2a8f288f8fb928:221111' => 'Apple QuickTime, Quality 642 (63%)',
    '8134ff0c4713cc1ef4a25ff60b49ac54:221111' => 'Apple QuickTime, Quality 643 (63%)',
    '967fc5c3ece2b69662257c76397416c9:221111' => 'Apple QuickTime, Quality 644 (63%)',
    'd01a38e0f568d2a7b6b71f8fa63b8bcc:221111' => 'Apple QuickTime, Quality 645 (63%)',
    '5229288e448311401bb284133ac7d48c:221111' => 'Apple QuickTime, Quality 646 (63%)',
    '76d22de881d1b95b491689b589743b7a:221111' => 'Apple QuickTime, Quality 647 (63%)',
    'c81b03b0291d2277461a551ed6861252:221111' => 'Apple QuickTime, Quality 648-649 (63%)',
    'ebb774b4e106d1a9df5824958d4e5a95:221111' => 'Apple QuickTime, Quality 650 (63%)',
    '9ed53fb5bc8e397daf9409251c0a0a6c:221111' => 'Apple QuickTime, Quality 651 (64%)',
    '87d40f2e4dad34fa435c62af6817dc18:221111' => 'Apple QuickTime, Quality 652 (64%)',
    '0b933cf90e62682da926267d6356ac2b:221111' => 'Apple QuickTime, Quality 653 (64%)',
    '2c4a4cb841ee92aa3a2b4c93467ba7a8:221111' => 'Apple QuickTime, Quality 654 (64%)',
    '8a6ba56597670b7adb70901eca278049:221111' => 'Apple QuickTime, Quality 655 (64%)',
    'd1ef25928fd4eefe131ffcfc249b9f8a:221111' => 'Apple QuickTime, Quality 656 (64%)',
    '8c85462b5a01db09bcbf304d7be1d543:221111' => 'Apple QuickTime, Quality 657 (64%)',
    '19c03533b9b2e3304a0b02d9b1054497:221111' => 'Apple QuickTime, Quality 658 (64%)',
    'd2baa8fbc56f0970f820c376c6065d41:221111' => 'Apple QuickTime, Quality 659 (64%)',
    'e6a0a679a13a99de16e13c6ea2829deb:221111' => 'Apple QuickTime, Quality 660-661 (64-65%)',
    '8e4f695afcf2a06254561e5e22b7a80b:221111' => 'Apple QuickTime, Quality 662 (65%)',
    'd053fd2c67ce96b0ecf9ffc4b7f7775d:221111' => 'Apple QuickTime, Quality 663 (65%)',
    '326c33f64f96592487d2bfdd198738bf:221111' => 'Apple QuickTime, Quality 664 (65%)',
    '1ff8f5ff33353a3ee0b6dc8fbb6321a0:221111' => 'Apple QuickTime, Quality 665 (65%)',
    '14a5534e4216458662a43101d56d84c8:221111' => 'Apple QuickTime, Quality 666 (65%)',
    '0643b87475939754c8d56825cd96242f:221111' => 'Apple QuickTime, Quality 667 (65%)',
    '45c46a02a434d8ea759742907bfa0ee5:221111' => 'Apple QuickTime, Quality 668 (65%)',
    '2916d9453b885ee4123e6e3ee94ccbc7:221111' => 'Apple QuickTime, Quality 669-671 (65-66%)',
    'b8fca611f92cbc459fe21e11f0214328:221111' => 'Apple QuickTime, Quality 672-673 (66%)',
    'f40fb322c4bde68a2902c86c613af841:221111' => 'Apple QuickTime, Quality 674 (66%)',
    '1cbd419717a2916b53f9f504ec1167ca:221111' => 'Apple QuickTime, Quality 675 (66%)',
    '79f546689b548868a904f50214928aa1:221111' => 'Apple QuickTime, Quality 676 (66%)',
    '8ff6f2d4369155b0474417b00c3c4ac9:221111' => 'Apple QuickTime, Quality 677 (66%)',
    'c60dbbefd4f215b9359dd004f4fb0fd3:221111' => 'Apple QuickTime, Quality 678 (66%)',
    'c192d5847d1146a31db621263a9ce2f5:221111' => 'Apple QuickTime, Quality 679 (66%)',
    '4a2361c48a583f6df779d1e6088ed83c:221111' => 'Apple QuickTime, Quality 680 (66%)',
    '6c6260b84a3a588614d65133430289ea:221111' => 'Apple QuickTime, Quality 681 (67%)',
    '6773f3db56ae831012dbe43c1650571a:221111' => 'Apple QuickTime, Quality 682 (67%)',
    '813b89236cfe429fe534361f28ace015:221111' => 'Apple QuickTime, Quality 683 (67%)',
    '363b54d38094e5f2e2d63c50870ae76c:221111' => 'Apple QuickTime, Quality 684 (67%)',
    '7b17607b9954c37e525b1fbc35271553:221111' => 'Apple QuickTime, Quality 685 (67%)',
    '8b1138e2d88033d42698a386a2e8605b:221111' => 'Apple QuickTime, Quality 686 (67%)',
    'c029b8a48e3c93f7c0367f2a149491c7:221111' => 'Apple QuickTime, Quality 687 (67%)',
    'a36199f5a090de94b10a32fbe05f2916:221111' => 'Apple QuickTime, Quality 688-689 (67%)',
    'a873e49b871c32bcaf8e3c6622744e70:221111' => 'Apple QuickTime, Quality 690 (67%)',
    'b9d16f36087d4cca70eef1512c4be569:221111' => 'Apple QuickTime, Quality 691 (67%)',
    'd5994dbe056ea3544b3256a7a6b53749:221111' => 'Apple QuickTime, Quality 692 (68%)',
    '0106cf02dcf4109cc6f02fa4ec0e2700:221111' => 'Apple QuickTime, Quality 693 (68%)',
    '7d71776416a8771d10e3c2e6dc6a5f21:221111' => 'Apple QuickTime, Quality 694-695 (68%)',
    'c1557f789acc622c8858be4dfbc53c31:221111' => 'Apple QuickTime, Quality 696 (68%)',
    '5a54f085c1780cadb13a7dea8347c7c6:221111' => 'Apple QuickTime, Quality 697 (68%)',
    '04a5bb959bc203221e72e6575ff39602:221111' => 'Apple QuickTime, Quality 698 (68%)',
    '116e3d5fee4e3a695c0f79c09c89ff84:221111' => 'Apple QuickTime, Quality 699 (68%)',
    '145bfd5481e99e18c4c3707228557fa5:221111' => 'Apple QuickTime, Quality 700 (68%)',
    '19ceef79e864691318beea6502ddc3e1:221111' => 'Apple QuickTime, Quality 701 (68%)',
    'dd0a023941d7bfd118d272f4f925e6e2:221111' => 'Apple QuickTime, Quality 702 (69%)',
    '81f039d6a0ded8227dc51273d153b295:221111' => 'Apple QuickTime, Quality 703-704 (69%)',
    '9a7ebf265afce16abaa6ca2fbb550b63:221111' => 'Apple QuickTime, Quality 705 (69%)',
    '8d0fed09156984328f90f9f19fb5a079:221111' => 'Apple QuickTime, Quality 706 (69%)',
    'be7c72e09c46622b0d2b93e170a03e17:221111' => 'Apple QuickTime, Quality 707 (69%)',
    '50a510968effffab80bed1d08c6c5ccc:221111' => 'Apple QuickTime, Quality 708 (69%)',
    'da2501a6f59b2256adb0833b58b504f2:221111' => 'Apple QuickTime, Quality 709 (69%)',
    '2a1b83345108443a090cdab4c83143fb:221111' => 'Apple QuickTime, Quality 710 (69%)',
    'e7f293f640878b53fe95a7cb0b1dcc83:221111' => 'Apple QuickTime, Quality 711-712 (69-70%)',
    'd9e0a4c08ef5d7f72eecce74c94c054d:221111' => 'Apple QuickTime, Quality 713 (70%)',
    'c5774ffb4573926fd03d4175818c0e5d:221111' => 'Apple QuickTime, Quality 714 (70%)',
    'e2b368a164b67e15598683f9f184bd77:221111' => 'Apple QuickTime, Quality 715 (70%)',
    'f30792e8fad278c3e1677b5f5b74c682:221111' => 'Apple QuickTime, Quality 716-717 (70%)',
    '6f6bfc10750e6717cc3791a9ea1d7569:221111' => 'Apple QuickTime, Quality 718-719 (70%)',
    '5e3981a937c61480451d5bdc253e5472:221111' => 'Apple QuickTime, Quality 720 (70%)',
    'c4fb82f47a7b002d5cab421592ae4972:221111' => 'Apple QuickTime, Quality 721 (70%)',
    'a6c4a173169d168e003839e51f035661:221111' => 'Apple QuickTime, Quality 722 (71%)',
    '72df283a5c07671eba341500a3fc18f1:221111' => 'Apple QuickTime, Quality 723 (71%)',
    'dfb203555c34fe146c526350e11309eb:221111' => 'Apple QuickTime, Quality 724 (71%)',
    '033472a8a855fab8cd8f6a5788dd07c8:221111' => 'Apple QuickTime, Quality 725 (71%)',
    '5b8a79eec9b7eb7755deb7f2c189e94a:221111' => 'Apple QuickTime, Quality 726 (71%)',
    'ad3aad027e3829959ebeb6288bfab268:221111' => 'Apple QuickTime, Quality 727 (71%)',
    '2234156f0550a047700c2a08459c8242:221111' => 'Apple QuickTime, Quality 728 (71%)',
    '0c0351c3a444b851cd105dd5cc4db59c:221111' => 'Apple QuickTime, Quality 729 (71%)',
    'c10fca5e6f66238ab09f7e8105f54e39:221111' => 'Apple QuickTime, Quality 730 (71%)',
    '74cc07bbb7049d59aff0c4965d4d5084:221111' => 'Apple QuickTime, Quality 731 (71%)',
    'ce9ad8466ffd84b91039326e8688c44a:221111' => 'Apple QuickTime, Quality 732-733 (71-72%)',
    'e4b0c56d41f4af9e10971876ad7ad56d:221111' => 'Apple QuickTime, Quality 734 (72%)',
    '96076425ecc546ec028d0eab48332756:221111' => 'Apple QuickTime, Quality 735 (72%)',
    'ba49b0656894f3c76d852223721b3b1f:221111' => 'Apple QuickTime, Quality 736 (72%)',
    'ae5c6eab0d57249acbcb8b1990b2602f:221111' => 'Apple QuickTime, Quality 737-738 (72%)',
    '72f08842473a6c504469d341259e5cd7:221111' => 'Apple QuickTime, Quality 739-740 (72%)',
    'c9f953acdfc1f5afdbb9e9f74692d23e:221111' => 'Apple QuickTime, Quality 741 (72%)',
    '5a1b57a2583acf5c2428cd62fe24b773:221111' => 'Apple QuickTime, Quality 742 (72%)',
    '6a9ead8b2339567482a172a581e86c15:221111' => 'Apple QuickTime, Quality 743-744 (73%)',
    '513d9e9dabbb480eb60f7ef76b1d755e:221111' => 'Apple QuickTime, Quality 745 (73%)',
    'b99bdcd0145833d52b916e71f2c20a04:221111' => 'Apple QuickTime, Quality 746 (73%)',
    'ff084566430a3ed4733cd59aec26a55d:221111' => 'Apple QuickTime, Quality 747 (73%)',
    '76bcc27918d8f12b343e6e5a41108781:221111' => 'Apple QuickTime, Quality 748 (73%)',
    '7bf7022a7c12b3b7ea085b46158253e6:221111' => 'Apple QuickTime, Quality 749 (73%)',
    '4baf3b1df2426fbdac3d0aaa0503ee94:221111' => 'Apple QuickTime, Quality 750 (73%)',
    'bb0180b9eda074c3f913c8ada3d4c1ad:221111' => 'Apple QuickTime, Quality 751 (73%)',
    'c5fcb1748f616ac97794d34b1b93616e:221111' => 'Apple QuickTime, Quality 752-753 (73-74%)',
    '0da77ccec22a9cff9a049a47e86d3502:221111' => 'Apple QuickTime, Quality 754 (74%)',
    'c31f71de437dc301d34f847d95267d9e:221111' => 'Apple QuickTime, Quality 755 (74%)',
    '01137dc7ef90f0aee15362c221f7b1d3:221111' => 'Apple QuickTime, Quality 756-758 (74%)',
    '026780f2172c289bc1ff73a34c6aee57:221111' => 'Apple QuickTime, Quality 759-760 (74%)',
    '3fab8f2b141f95a989fc4b046ad825cb:221111' => 'Apple QuickTime, Quality 761 (74%)',
    'cd091eeb9d27d9dc7cdb5bff73572679:221111' => 'Apple QuickTime, Quality 762 (74%)',
    'ad2221ee8bb94a3558ed16766efaec4f:221111' => 'Apple QuickTime, Quality 763 (75%)',
    '8a4ff70dce3efc9312ff7239e79b6bc9:221111' => 'Apple QuickTime, Quality 764 (75%)',
    'b8d1fcda3a19d00788c2be73fd4c2c8e:221111' => 'Apple QuickTime, Quality 765-766 (75%)',
    '3af16b87c33bb2e48152e249beb9147b:221111' => 'Apple QuickTime, Quality 767 (75%)',
    '3af16b87c33bb2e48152e249beb9147b:211111' => 'Apple QuickTime, Quality 768 (75%)',
    '683270dbffdc5cd2d4e6cb841f17b206:211111' => 'Apple QuickTime, Quality 769 (75%)',
    '285bdd58fac87b174a22d2a93d69cd7c:211111' => 'Apple QuickTime, Quality 770 (75%)',
    '312e047b5d9076cd1e126f3dbce928e5:211111' => 'Apple QuickTime, Quality 771 (75%)',
    '99458d7a01a39fe126592d9afb1402ce:211111' => 'Apple QuickTime, Quality 772 (75%)',
    'b4633256b0e0d5e2a5021f01ebabc105:211111' => 'Apple QuickTime, Quality 773 (75%)',
    '90d39fd222f9114f613a315a894283ca:211111' => 'Apple QuickTime, Quality 774 (76%)',
    '61b1d4a02498b7467f2c8e8cfebdfae9:211111' => 'Apple QuickTime, Quality 775-776 (76%)',
    '987ebcbd20b633b40241fcd30266e986:211111' => 'Apple QuickTime, Quality 777 (76%)',
    '31e214243395b008048469d4bc4dc780:211111' => 'Apple QuickTime, Quality 778 (76%)',
    'db5b3a078a942131b5d86bc189baac24:211111' => 'Apple QuickTime, Quality 779 (76%)',
    'ec440a2ffcbce8895beb663b36975073:211111' => 'Apple QuickTime, Quality 780-781 (76%)',
    '93d7ac97a931be74c7fe849edc482ea1:211111' => 'Apple QuickTime, Quality 782 (76%)',
    '3974d72e6831171ec970bbb09b9cc506:211111' => 'Apple QuickTime, Quality 783 (76%)',
    '07bd22218437079a86ce0b93ffa9cc90:211111' => 'Apple QuickTime, Quality 784 (77%)',
    '32757023bb5e7f703acf737a5a29c9d6:211111' => 'Apple QuickTime, Quality 785-786 (77%)',
    '6096eb584b99a587f5527e20473aa9d1:211111' => 'Apple QuickTime, Quality 787 (77%)',
    'c6d134475eb85bd454f2ee5153366c51:211111' => 'Apple QuickTime, Quality 788 (77%)',
    'a3ba20f325ff36f874d633919185f92d:211111' => 'Apple QuickTime, Quality 789 (77%)',
    'af10133169e143a2b3634c48dede9440:211111' => 'Apple QuickTime, Quality 790 (77%)',
    '9dbb8223620e7f25ca3292849f7aa025:211111' => 'Apple QuickTime, Quality 791-792 (77%)',
    '5071640a38c5898dd5d2043346fd23e1:211111' => 'Apple QuickTime, Quality 793-795 (77-78%)',
    '60a0ca27f3e7289d97c033ca217899cc:211111' => 'Apple QuickTime, Quality 796-798 (78%)',
    'f7a5ea485a254cba0d39cdeaf89ad344:211111' => 'Apple QuickTime, Quality 799 (78%)',
    'd8cd0ca367d9afaf9a1aca0415da5361:211111' => 'Apple QuickTime, Quality 800 (78%)',
    '87fb3c7402ba4edcda34b71696d2b0e3:211111' => 'Apple QuickTime, Quality 801 (78%)',
    'f8df76525f7f97d2e89173989e6786af:211111' => 'Apple QuickTime, Quality 802-804 (78-79%)',
    '64677161baed1c47d2fdd6eefd779583:211111' => 'Apple QuickTime, Quality 805 (79%)',
    'ca39dde8e9b4ccd6261b28e089181639:211111' => 'Apple QuickTime, Quality 806-807 (79%)',
    'e6c99d520b86fd6f5eb513d1a084324e:211111' => 'Apple QuickTime, Quality 808 (79%)',
    '9e048c787b12b9ab47d6166e81bc8bda:211111' => 'Apple QuickTime, Quality 809 (79%)',
    '03c035b39889356e0b10805d8549a1f7:211111' => 'Apple QuickTime, Quality 810 (79%)',
    '15de51ede231cfbe123daa42a1a46070:211111' => 'Apple QuickTime, Quality 811-813 (79%)',
    '725bcc59a6f5a1436dfa0dfd96cdcf44:211111' => 'Apple QuickTime, Quality 814 (79%)',
    'd00103d50108e8be370a78d47f51aba0:211111' => 'Apple QuickTime, Quality 815 (80%)',
    '1c4c74ccc581b11050cfe18792246e5e:211111' => 'Apple QuickTime, Quality 816 (80%)',
    'b63e97c56859f2476ed3f15f40775fb5:211111' => 'Apple QuickTime, Quality 817 (80%)',
    'aac2510e3cd617eb2cd60e7dc6f5d252:211111' => 'Apple QuickTime, Quality 818 (80%)',
    '028caa124d0837dd9b1a64028e4f2965:211111' => 'Apple QuickTime, Quality 819 (80%)',
    '0440231d1a4a1187bffaa5b5576827f9:211111' => 'Apple QuickTime, Quality 820 (80%)',
    '6f879b2b5642ee3d01faf3410a721e2d:211111' => 'Apple QuickTime, Quality 821 (80%)',
    '4642245b427d5dd5c1c3766c323204ac:211111' => 'Apple QuickTime, Quality 822-824 (80%)',
    '8bd486eb557ae8f39948775aba222731:211111' => 'Apple QuickTime, Quality 825 (81%)',
    '8bc4e4bec8e9b193c11ad90c7f8bfaf3:211111' => 'Apple QuickTime, Quality 826 (81%)',
    'f54c2ea8437408238f6c181a355af6cb:211111' => 'Apple QuickTime, Quality 827-829 (81%)',
    '0f58458f2b9959dbc57b4868200c0432:211111' => 'Apple QuickTime, Quality 830-832 (81%)',
    '24f95056dce30d11bad39b33ab271262:211111' => 'Apple QuickTime, Quality 833-834 (81%)',
    'bad6fdd8761fb9d0921384013acf783f:211111' => 'Apple QuickTime, Quality 835 (82%)',
    '8c482fe6aef2a59a94cb779e6795e512:211111' => 'Apple QuickTime, Quality 836 (82%)',
    '50b309f18bcf477742aa491ea55af777:211111' => 'Apple QuickTime, Quality 837 (82%)',
    '92a9e0d027a1b2e5f7e49f7ffd96277e:211111' => 'Apple QuickTime, Quality 838-839 (82%)',
    'a6df2748a4972d4323f0386820ce35a4:211111' => 'Apple QuickTime, Quality 840 (82%)',
    '9ffb80389e2eed2301e6b07860c2fbd7:211111' => 'Apple QuickTime, Quality 841 (82%)',
    '6042038094d7f4ad72c61c2a2e7a467f:211111' => 'Apple QuickTime, Quality 842 (82%)',
    '78d004490e822405acded09846135e50:211111' => 'Apple QuickTime, Quality 843 (82%)',
    'a589d880de576ed888c57814ccea47a0:211111' => 'Apple QuickTime, Quality 844 (82%)',
    'beee113eea5950b8211cdc49e5a04099:211111' => 'Apple QuickTime, Quality 845 (83%)',
    'f04fed79cdc47709d649187cfcc7e342:211111' => 'Apple QuickTime, Quality 846-849 (83%)',
    'edb0be7fcce943c28d02ff78ae600afb:211111' => 'Apple QuickTime, Quality 850 (83%)',
    '0e9648c1f28b99a377dcf7deec6450e6:211111' => 'Apple QuickTime, Quality 851-852 (83%)',
    '2dffe433bbb9c81b05e569afd3d9b585:211111' => 'Apple QuickTime, Quality 853 (83%)',
    '0ef4f8fa922f87f1be646fccaa0ef42e:211111' => 'Apple QuickTime, Quality 854 (83%)',
    '4b799df6fc9476102f890343080e66f5:211111' => 'Apple QuickTime, Quality 855-856 (83-84%)',
    '53a66cb32deb83c855f36b26527f4c10:211111' => 'Apple QuickTime, Quality 857 (84%)',
    '151d7cd5a95929d45c6790beb87705fe:211111' => 'Apple QuickTime, Quality 858 (84%)',
    'bc2afe0a9c7c68b8d84bd231209be3e2:211111' => 'Apple QuickTime, Quality 859-860 (84%)',
    'b9d66564ab9c4bb0910eb228aa9a48e1:211111' => 'Apple QuickTime, Quality 861-863 (84%)',
    '6808ca55a29fcb9c15db1925a84370c3:211111' => 'Apple QuickTime, Quality 864 (84%)',
    'ee5b4ed7f04821d1e3a509d7565cb10d:211111' => 'Apple QuickTime, Quality 865 (84%)',
    'd38be79f7c8c6c27a3268275b144add6:211111' => 'Apple QuickTime, Quality 866 (85%)',
    '59eedef87f255db058b5ba0b1d3a4ce8:211111' => 'Apple QuickTime, Quality 867 (85%)',
    '5e5530c45def7006a7f672ce5778513d:211111' => 'Apple QuickTime, Quality 868 (85%)',
    'b09abfa40fc6607dc26d8b5df48c72fc:211111' => 'Apple QuickTime, Quality 869 (85%)',
    'cfc78404529f2b81b16d3f25fc96e8f4:211111' => 'Apple QuickTime, Quality 870 (85%)',
    '14c62682032efe8dc2de80c9330c6206:211111' => 'Apple QuickTime, Quality 871-872 (85%)',
    'ffadac945c3420537e21e67ab3a843d6:211111' => 'Apple QuickTime, Quality 873-875 (85%)',
    'e67a8a7e92a9f03413e9a67b99624b8b:211111' => 'Apple QuickTime, Quality 876-877 (86%)',
    'ba4af3bb30dda0a7be4c04ff1ebbd9ef:211111' => 'Apple QuickTime, Quality 878-880 (86%)',
    '3eedb8a357141ff5ae765fd3be2b232f:211111' => 'Apple QuickTime, Quality 881-886 (86-87%)',
    '127b0599fc6804909a33832be7a9dd36:211111' => 'Apple QuickTime, Quality 887 (87%)',
    'b697448eec21ef07f3111b62d592c423:211111' => 'Apple QuickTime, Quality 888-890 (87%)',
    'a08a6b6535f292518b5ff6d0d05ae187:211111' => 'Apple QuickTime, Quality 891-893 (87%)',
    'a439b365c2d0cf1fbaad2e42d331d759:211111' => 'Apple QuickTime, Quality 894 (87%)',
    'bf0c20b20af6473b7c4a338ba57d1a96:211111' => 'Apple QuickTime, Quality 895 (87%)',
    '09cf94311753aa9796ffd720749c51f7:211111' => 'Apple QuickTime, Quality 896 (88%)',
    'a60bbd6538af00192c411020d7494a1d:211111' => 'Apple QuickTime, Quality 897-898 (88%)',
    'df8ea903695e76e4b1466bdd3a3480c7:211111' => 'Apple QuickTime, Quality 899 (88%)',
    'eb4eb617beaa4f23acf41167742806fc:211111' => 'Apple QuickTime, Quality 900-901 (88%)',
    '591c923a44c635c33769704c9cfa6ab7:211111' => 'Apple QuickTime, Quality 902 (88%)',
    '960caf85ef273541ac2e76c9554dc860:211111' => 'Apple QuickTime, Quality 903 (88%)',
    '22c77ec6f4e8f75d48f98473abe62e59:211111' => 'Apple QuickTime, Quality 904-905 (88%)',
    'fc8d384969030e7bc0255d34a7a5c0b0:211111' => 'Apple QuickTime, Quality 906 (88%)',
    '42e7323506b113685e82e6d42664626f:211111' => 'Apple QuickTime, Quality 907-908 (89%)',
    '7163b345b90553e246296a48b46cc0b3:211111' => 'Apple QuickTime, Quality 909-910 (89%)',
    '52f25cf8c4d610dffcc45681def8fb49:211111' => 'Apple QuickTime, Quality 911-914 (89%)',
    '5554cfd817a2713a690b957145b088ed:211111' => 'Apple QuickTime, Quality 915-917 (89-90%)',
    '6ef0b71a5676c4645a3166b9c34744fa:211111' => 'Apple QuickTime, Quality 918-920 (90%)',
    '1228da2b97793a88a41542ddcfca7ad2:211111' => 'Apple QuickTime, Quality 921-922 (90%)',
    '9060906039e9ff37171ba48d908f6ad5:211111' => 'Apple QuickTime, Quality 923-924 (90%)',
    '6eb301fb89e7d625129b77a53fe30dcc:211111' => 'Apple QuickTime, Quality 925-926 (90%)',
    'ad5399708089baad5891319303ba92df:211111' => 'Apple QuickTime, Quality 927 (91%)',
    'afd16e145464c7c5a3cd703017b4ef7a:211111' => 'Apple QuickTime, Quality 928 (91%)',
    '4271405c840705072a102d7e18b374d9:211111' => 'Apple QuickTime, Quality 929 (91%)',
    '72a91837a63fa7444416bc00a05d988b:211111' => 'Apple QuickTime, Quality 930 (91%)',
    '8fe3845bafb06ee4de1a6f75c2a42e9b:211111' => 'Apple QuickTime, Quality 931 (91%)',
    '8d3b678651ec71f27e3727718123f354:211111' => 'Apple QuickTime, Quality 932-933 (91%)',
    '36d42b031eea0c9f626f15533e72162a:211111' => 'Apple QuickTime, Quality 934-936 (91%)',
    '789076781ff1e18154091f2460c1bab5:211111' => 'Apple QuickTime, Quality 937-938 (92%)',
    '07464723ecfd8e5ed8fd6904e9d15a23:211111' => 'Apple QuickTime, Quality 939 (92%)',
    '0efd0d9423b440cfc8efacf2e4dfcb7f:211111' => 'Apple QuickTime, Quality 940 (92%)',
    '80409b38f84336548b62e337a850e9cb:211111' => 'Apple QuickTime, Quality 941-943 (92%)',
    '5fa6bb26309d43ca6c89d6cc776a68a4:211111' => 'Apple QuickTime, Quality 944-946 (92%)',
    '705064f644ac4b24884500a40ad0f7cf:211111' => 'Apple QuickTime, Quality 947-948 (92-93%)',
    'c181c79bc41cf5fe11e6f253242ce2c4:211111' => 'Apple QuickTime, Quality 949-951 (93%)',
    '1a7da03994ee019a30dbd37117761467:211111' => 'Apple QuickTime, Quality 952-954 (93%)',
    '070620a25578b4a38ed0c09d6d512de8:211111' => 'Apple QuickTime, Quality 955 (93%)',
    '6a092d8fd56ca0e852d74bd86cfc4f47:211111' => 'Apple QuickTime, Quality 956-957 (93%)',
    '66e85870faf72f4f3fe25486409b286a:211111' => 'Apple QuickTime, Quality 958 (94%)',
    '31365833a4d7d0ef2c1db9b90e515f7f:211111' => 'Apple QuickTime, Quality 959-961 (94%)',
    '2edccd94198ab5a459a8396d9a0be4aa:211111' => 'Apple QuickTime, Quality 962 (94%)',
    'ca0bf66c467278f9d5ca5301840e7a7f:211111' => 'Apple QuickTime, Quality 963-967 (94%)',
    '261bdba7fe6d8bca5302e4e93b52c1fb:211111' => 'Apple QuickTime, Quality 968-970 (95%)',
    '762f9501e83d58307d1e102ddb343207:211111' => 'Apple QuickTime, Quality 971 (95%)',
    'ba18a8f4175bdedfea7af9bf5fe8dd9c:211111' => 'Apple QuickTime, Quality 972-973 (95%)',
    'd1a8052e7152e0c35d167e9e56418eb7:211111' => 'Apple QuickTime, Quality 974 (95%)',
    '32682ece28c3bee7754fde6fec109b47:211111' => 'Apple QuickTime, Quality 975-977 (95%)',
    'a8780d0f85eef638c6a448e57b157378:211111' => 'Apple QuickTime, Quality 978-979 (96%)',
    'b79ff1a16807a48a31d457ad7e0b94f2:211111' => 'Apple QuickTime, Quality 980-984 (96%)',
    '2bf80ea6a878f7ecb88ea827b58c98f8:211111' => 'Apple QuickTime, Quality 985-987 (96%)',
    'add779ad00786bd2ccb9dcc226386b1a:211111' => 'Apple QuickTime, Quality 988-991 (96-97%)',
    '56c4efb597cc30275229486199e60f70:211111' => 'Apple QuickTime, Quality 992-993 (97%)',
    'c2df556e8ede9fb199b9a16e01279c6b:211111' => 'Apple QuickTime, Quality 994-996 (97%)',
    '6af868a0eececd267495f749a38b4f95:211111' => 'Apple QuickTime, Quality 997-998 (97%)',
    'c92c755320e7ce8f46f644b90b7907e8:211111' => 'Apple QuickTime, Quality 999-1000 (98%)',
    '6fcbaaa11108d1712bad5410b3db5b91:211111' => 'Apple QuickTime, Quality 1001-1002 (98%)',
    'f7d803e16f0c66df7d46747715b1ae24:211111' => 'Apple QuickTime, Quality 1003 (98%)',
    '7f51ebf21174bcd3b027ae3cc77c4459:211111' => 'Apple QuickTime, Quality 1004 (98%)',
    'f2423a8ae68a49cc6191a2ec80367893:211111' => 'Apple QuickTime, Quality 1005-1006 (98%)',
    'f97cd4c7b1125556dc3eb57fc494e6b5:211111' => 'Apple QuickTime, Quality 1007-1009 (98-99%)',
    '389e1ca056b1bd05dd29ecaecae5b4ae:211111' => 'Apple QuickTime, Quality 1010-1013 (99%)',
    '43f9929d00af93968662983b891364d8:211111' => 'Apple QuickTime, Quality 1014-1016 (99%)',
    '7e1453eec55a8c40166b2d8985ad6bdc:211111' => 'Apple QuickTime, Quality 1017 (99%)',
    '31697e4b294a13e35ab8d55d3a9612ca:211111' => 'Apple QuickTime, Quality 1018-1020 (99-100%)',
    'ec76274ff22c07e53299ad34633ba88f:211111' => 'Apple QuickTime, Quality 1021-1023 (100%)',
    '7f8b33a26e7f35a6eaf2e95df81e1cca:111111' => 'Apple QuickTime, Quality 1024 (Lossless)',

    # Apple QuickTime (Grayscale)
    #
    # Tested with:
    # - QuickTime 7.5.0 (Win)
    '7e6246d9be5273b979beb680b284e7b8:11' => 'Apple QuickTime, Quality 0-63 (0-6%)',
    'caf33ddc94762bf60a8c5e5024550b21:11' => 'Apple QuickTime, Quality 64-80 (6-8%)',
    '042ae0dbef2b1e91c4eb36e66a39b5b9:11' => 'Apple QuickTime, Quality 81-92 (8-9%)',
    'bdcc7abca09941326c079bb3bc30de4d:11' => 'Apple QuickTime, Quality 93-101 (9-10%)',
    '8edf0677ca6be750511593fad835bbb5:11' => 'Apple QuickTime, Quality 102-109 (10-11%)',
    'dd54b4e3d8801f3a7969be542d165c6b:11' => 'Apple QuickTime, Quality 110-116 (11%)',
    'c00374dece11c3cab5f2c3bf9621d365:11' => 'Apple QuickTime, Quality 117-122 (11-12%)',
    'a2e3baa02454492ef811619ac18c65da:11' => 'Apple QuickTime, Quality 123-127 (12%)',
    '8f699e4439175f5f0cf0f903040fb3c5:11' => 'Apple QuickTime, Quality 128-133 (13%)',
    '50f9224c87a32486851bdbd3e686fd5b:11' => 'Apple QuickTime, Quality 134-137 (13%)',
    'cccd5f36920fbe8ad77da2214f8ab6ed:11' => 'Apple QuickTime, Quality 138-142 (13-14%)',
    '8bc7e3b8f24507e284075ebeb272c3f4:11' => 'Apple QuickTime, Quality 143-146 (14%)',
    'bc156b933365b88e5ba9f1bd4b2fee4e:11' => 'Apple QuickTime, Quality 147-150 (14-15%)',
    'a50ff29c6c2a7e73f742ca94678956ba:11' => 'Apple QuickTime, Quality 151-154 (15%)',
    'd0a67359275cf9e2e8f35de79d2e28ae:11' => 'Apple QuickTime, Quality 155-157 (15%)',
    '37914b5d31e7f0f13066e5292c07c305:11' => 'Apple QuickTime, Quality 158-161 (15-16%)',
    '340aeb15b2b6c05968bb2c6e3d85cbed:11' => 'Apple QuickTime, Quality 162-164 (16%)',
    'bc93228921ec863e90850325cfd90dd2:11' => 'Apple QuickTime, Quality 165-167 (16%)',
    'd5c95455812515ad4855ed725d5bf2d9:11' => 'Apple QuickTime, Quality 168-170 (16-17%)',
    'd018c811df2390446b43cc702888864c:11' => 'Apple QuickTime, Quality 171-173 (17%)',
    '824a9788f50aad6ca26ada301cae5c72:11' => 'Apple QuickTime, Quality 174-176 (17%)',
    'd35254d58224b1b6babda94d7f1a5ffe:11' => 'Apple QuickTime, Quality 177-179 (17%)',
    '4177be1c82543b32bf6578dc3a78d49d:11' => 'Apple QuickTime, Quality 180-182 (18%)',
    '88b94edfd7a6c7aadac520905e6cfa0a:11' => 'Apple QuickTime, Quality 183-184 (18%)',
    '589b1ef8cc8bece150218e4646d9dfd6:11' => 'Apple QuickTime, Quality 185-187 (18%)',
    'a6c15a75ab70e28c78e6084f909523bf:11' => 'Apple QuickTime, Quality 188-189 (18%)',
    'd052e48078f986c715e68f502d371ccc:11' => 'Apple QuickTime, Quality 190-191 (19%)',
    '7d4205e3d4e0b6c7071a418c9b5840cb:11' => 'Apple QuickTime, Quality 192-194 (19%)',
    '2200d1873e51bf812bdcb57c10c6c14b:11' => 'Apple QuickTime, Quality 195-196 (19%)',
    '395ef59782311cd2081887c78c40c4bc:11' => 'Apple QuickTime, Quality 197-198 (19%)',
    'd33d12dc779097bee959fefac6de9a3e:11' => 'Apple QuickTime, Quality 199-201 (19-20%)',
    '64ff54dc33f610e3705cae31428ce43d:11' => 'Apple QuickTime, Quality 202-203 (20%)',
    'dbee605b07dfe30c992622877dffb049:11' => 'Apple QuickTime, Quality 204-205 (20%)',
    'aa5a427657696f05da789e1516b8c2ff:11' => 'Apple QuickTime, Quality 206-207 (20%)',
    'cf8fce0d4bde00a2feb680bb52667c8f:11' => 'Apple QuickTime, Quality 208-209 (20%)',
    '81ac42cc63416f7c66cd2a51a8801cbd:11' => 'Apple QuickTime, Quality 210-211 (21%)',
    '5e55cc3328e61e88b9f2a49af4ec2268:11' => 'Apple QuickTime, Quality 212-213 (21%)',
    'ef938a0533502fe19f311d46c43fa86c:11' => 'Apple QuickTime, Quality 214-215 (21%)',
    'f841cbd6a77d64924ab19845219f3399:11' => 'Apple QuickTime, Quality 216-217 (21%)',
    '98b684f30055c84ba5734e29f7b98b5f:11' => 'Apple QuickTime, Quality 218 (21%)',
    '8d4f697b3a2baaecc8765f31f54a76ae:11' => 'Apple QuickTime, Quality 219-220 (21%)',
    '6fc283989bb3a8c91f6c4384df2fa25d:11' => 'Apple QuickTime, Quality 221-222 (22%)',
    'dccca51d261b315120f069697872377d:11' => 'Apple QuickTime, Quality 223-224 (22%)',
    '5869e4a9592a7900e740b09fe19261a1:11' => 'Apple QuickTime, Quality 225 (22%)',
    'ebd575cf069eb906d2f2b2e202f67247:11' => 'Apple QuickTime, Quality 226-227 (22%)',
    '7ed52852c280b97fd44def8434d84051:11' => 'Apple QuickTime, Quality 228-229 (22%)',
    '984d291debac8a0caeaccccea5fbfbdf:11' => 'Apple QuickTime, Quality 230 (22%)',
    '9b2247e0f55b4485e7c55a04ee6a801c:11' => 'Apple QuickTime, Quality 231-232 (23%)',
    'd2a1887cf45aecd63d838e585dbb5794:11' => 'Apple QuickTime, Quality 233-234 (23%)',
    '6bb5ab15f80beebcb73fae0ef089fa61:11' => 'Apple QuickTime, Quality 235 (23%)',
    '281f65a19e5de33d9ff5f3afeda06973:11' => 'Apple QuickTime, Quality 236-237 (23%)',
    '563f732877b2c654d571c269bbb36a40:11' => 'Apple QuickTime, Quality 238 (23%)',
    'fa11118bb9f90b1464e34c785d0da357:11' => 'Apple QuickTime, Quality 239-240 (23%)',
    '8dc4cff27c3c5196b4bc8905ef32f119:11' => 'Apple QuickTime, Quality 241 (24%)',
    '2ac28889e4ad4724d49f8b4c36b0cece:11' => 'Apple QuickTime, Quality 242-243 (24%)',
    '5cbeb8f83a47b6a5e8711fe0ea7c42d7:11' => 'Apple QuickTime, Quality 244 (24%)',
    'd5d329f5687d154e4ceeb48697b848ba:11' => 'Apple QuickTime, Quality 245-246 (24%)',
    'd3eaad34ae4fc8a3ac6330c1c9dceb28:11' => 'Apple QuickTime, Quality 247 (24%)',
    '29b74834ce7570b9c175d0200e75316e:11' => 'Apple QuickTime, Quality 248-249 (24%)',
    '850f2b2aaa99ad390bc9443be1b587dc:11' => 'Apple QuickTime, Quality 250 (24%)',
    '036d2395718f99bf916486e1af42cb92:11' => 'Apple QuickTime, Quality 251 (25%)',
    'e9275719ef4cb335f9dfed63c3737f0e:11' => 'Apple QuickTime, Quality 252-253 (25%)',
    'dfcbd3df5c6b96106e6348b77f89c56a:11' => 'Apple QuickTime, Quality 254 (25%)',
    'f9ef906cd67c9f9b62514a6ac1f8bd3f:11' => 'Apple QuickTime, Quality 255 (25%)',
    '469d14cef27dbb7c1f6f49324c077852:11' => 'Apple QuickTime, Quality 256-257 (25%)',
    '65d7471a913f6cc87e9dc65ea594606b:11' => 'Apple QuickTime, Quality 258 (25%)',
    '6b590185b5d6ecbc1d79c2624a0d5319:11' => 'Apple QuickTime, Quality 259 (25%)',
    'fbf3d8d87f68077aa95e5e40047c1607:11' => 'Apple QuickTime, Quality 260-261 (25%)',
    '7e3999424de8a8f6bb84e3cfc07628e8:11' => 'Apple QuickTime, Quality 262 (26%)',
    '6c0476ba4b3fcc4675cfab20d3c96368:11' => 'Apple QuickTime, Quality 263 (26%)',
    '0bb76dd0e08175a90343a9c7dab48bfa:11' => 'Apple QuickTime, Quality 264-265 (26%)',
    '83c8ceab43dedde06d8068e5b8ccdc2b:11' => 'Apple QuickTime, Quality 266 (26%)',
    '5aee693372b77c9721dba9d3596e371c:11' => 'Apple QuickTime, Quality 267 (26%)',
    '1312bc5c7456856400f43749d407fb9f:11' => 'Apple QuickTime, Quality 268 (26%)',
    '3f01645e33791ef09fbeb6c0e63db6a9:11' => 'Apple QuickTime, Quality 269 (26%)',
    '785c36a6aa2bedd207cb1fa450a5e6d4:11' => 'Apple QuickTime, Quality 270-271 (26%)',
    '84788c494352a07ab54f360f4a2a3d34:11' => 'Apple QuickTime, Quality 272 (27%)',
    'bf29abfdf0086437452e2ca220e69cae:11' => 'Apple QuickTime, Quality 273 (27%)',
    'c7da2e951711b8b1314a7c531e09cbdc:11' => 'Apple QuickTime, Quality 274 (27%)',
    'fb549f21b7ad3b556bc91165b3067a77:11' => 'Apple QuickTime, Quality 275 (27%)',
    '0a7497e67acef345c655f79fd00b26de:11' => 'Apple QuickTime, Quality 276 (27%)',
    '1e2be0dde2c5d2216bca879a3f89c565:11' => 'Apple QuickTime, Quality 277-278 (27%)',
    'd09f78b68290ff6b470720ead4d79b15:11' => 'Apple QuickTime, Quality 279 (27%)',
    '5b54396a8a725e49e9bd4c9883b151df:11' => 'Apple QuickTime, Quality 280 (27%)',
    'f60d4afe566a641f0187a42ca6462560:11' => 'Apple QuickTime, Quality 281 (27%)',
    'b69090d1ab951e6355ab193b1f20bf48:11' => 'Apple QuickTime, Quality 282 (28%)',
    '7ce7d00283cada911c3ebc347680bc7d:11' => 'Apple QuickTime, Quality 283 (28%)',
    '2502314c1b957a0e4f911d17db770a01:11' => 'Apple QuickTime, Quality 284 (28%)',
    '100f3392aa8292fb78548513a619671a:11' => 'Apple QuickTime, Quality 285 (28%)',
    'af683de3118ab595c41b5796b57a9540:11' => 'Apple QuickTime, Quality 286 (28%)',
    '1228f1572d76b53658f4042bda8e99a2:11' => 'Apple QuickTime, Quality 287 (28%)',
    'e9c647b8bf2d7535d259eed6fbabe206:11' => 'Apple QuickTime, Quality 288 (28%)',
    '9ab0afefad0e6bb7c3c1a8bca0c3f987:11' => 'Apple QuickTime, Quality 289 (28%)',
    'b6900aafebce0e59136abb701eacb1e5:11' => 'Apple QuickTime, Quality 290 (28%)',
    'a3c40b635e584c8f49d6b6b110846fee:11' => 'Apple QuickTime, Quality 291-292 (28-29%)',
    '80175a9dbb871d045c738fdeb6fcbdc7:11' => 'Apple QuickTime, Quality 293 (29%)',
    'c79231716eff96853fe03a26c1c38120:11' => 'Apple QuickTime, Quality 294 (29%)',
    '8f96a0f2af7f1f1b0b2d4895bced1326:11' => 'Apple QuickTime, Quality 295 (29%)',
    'ba3c103d00719e795b093ac7a75e6fac:11' => 'Apple QuickTime, Quality 296 (29%)',
    '42fd2864197991a38b3f80374a69d4e9:11' => 'Apple QuickTime, Quality 297 (29%)',
    '76bc1d777c94b680683610218732eb11:11' => 'Apple QuickTime, Quality 298 (29%)',
    '8b9e19fe69d7c7e1989018aca76c0aea:11' => 'Apple QuickTime, Quality 299 (29%)',
    'd3c0e7437c630f3bed0867737c5f1921:11' => 'Apple QuickTime, Quality 300 (29%)',
    'a8ecf55a88fd0e1b29646207aff8c36f:11' => 'Apple QuickTime, Quality 301 (29%)',
    'eaffe0714878be5fb67a914f5bb79fef:11' => 'Apple QuickTime, Quality 302 (29%)',
    'a6a49ea0300157ecb401ce45d7f1f850:11' => 'Apple QuickTime, Quality 303 (30%)',
    'fc28ca358af7cd55dc78853e4288f26d:11' => 'Apple QuickTime, Quality 304 (30%)',
    '61cb5e93e3e69f6929d97653824733b0:11' => 'Apple QuickTime, Quality 305 (30%)',
    '2cba6ba1aede8c791ada1acaba8c162e:11' => 'Apple QuickTime, Quality 306 (30%)',
    'c1bcc3db9f417dc52595f2bb224e30d7:11' => 'Apple QuickTime, Quality 307 (30%)',
    '2273274a8d695da4bebff145cbcbafcc:11' => 'Apple QuickTime, Quality 308 (30%)',
    'e8bdbff8c7908e36c51e1344c0e99746:11' => 'Apple QuickTime, Quality 309 (30%)',
    'ffb8ea8efdb22c5c8256cc4e4008f11c:11' => 'Apple QuickTime, Quality 310 (30%)',
    'a1a8f92dc00c42877eb9a1d7462f8408:11' => 'Apple QuickTime, Quality 311 (30%)',
    '723c2a2de195391f2db06456e9345c5b:11' => 'Apple QuickTime, Quality 312 (30%)',
    '916225049ab8d411a5e0138ea9087e37:11' => 'Apple QuickTime, Quality 313 (31%)',
    '3b0315316de45b649bd8ba5b5471ab81:11' => 'Apple QuickTime, Quality 314 (31%)',
    '6cf948e65c9d32279c757394a4f5b77e:11' => 'Apple QuickTime, Quality 315 (31%)',
    '4c4b7fc28e54a2bbdccd90d3618f01e8:11' => 'Apple QuickTime, Quality 316 (31%)',
    '767c20d7d54970b0974f205c790d7d04:11' => 'Apple QuickTime, Quality 317 (31%)',
    '81c1ce1c7d15394d95eaf2d6bd1495e3:11' => 'Apple QuickTime, Quality 318 (31%)',
    '6547daee398d39f773742be92ef2d0d0:11' => 'Apple QuickTime, Quality 319 (31%)',
    '65edf81f975f01a7b3ad1c16a1af64cb:11' => 'Apple QuickTime, Quality 320 (31%)',
    'f8948967aeda9fb6ca1637a082ed04db:11' => 'Apple QuickTime, Quality 321 (31%)',
    '2444e1c407a9965fb5ea2dafd269911f:11' => 'Apple QuickTime, Quality 322-324 (31-32%)',
    '7c8242581553e818ef243fc680879a19:11' => 'Apple QuickTime, Quality 325 (32%)',
    'e2fe91d57078586f15b09e3b9c8cd3fa:11' => 'Apple QuickTime, Quality 326 (32%)',
    '0740db8af7951c1363f2c8d75462d378:11' => 'Apple QuickTime, Quality 327 (32%)',
    '3c1ff7ebab192163b4578e7dfcf63ce6:11' => 'Apple QuickTime, Quality 328 (32%)',
    '705ae76b905302bd9f3b78cc8d1cb28f:11' => 'Apple QuickTime, Quality 329 (32%)',
    '9438633929a283aac168f415d8ca44d6:11' => 'Apple QuickTime, Quality 330 (32%)',
    '68799ccfa08e2f55b5be79264d3ca58a:11' => 'Apple QuickTime, Quality 331 (32%)',
    '9bda57f21c56ea0dc971164b8dc56394:11' => 'Apple QuickTime, Quality 332 (32%)',
    'f9988c61ae580fcfc8bf929134b07c2e:11' => 'Apple QuickTime, Quality 333 (33%)',
    '59faa8c6fb70d4cf42765a92c1c7afc1:11' => 'Apple QuickTime, Quality 334 (33%)',
    '277982593a55786fe424c80a17224cd7:11' => 'Apple QuickTime, Quality 335 (33%)',
    'c2a8a67d050b22a0673ee9ad6685a540:11' => 'Apple QuickTime, Quality 336 (33%)',
    '040e09f495355470a44c580bca654693:11' => 'Apple QuickTime, Quality 337 (33%)',
    '93173762094b6b506aa495e022ced65f:11' => 'Apple QuickTime, Quality 338 (33%)',
    '281c39340554f672ff62c65e0bf1036b:11' => 'Apple QuickTime, Quality 339 (33%)',
    '82d40afcb23ac10dba01bbab101da176:11' => 'Apple QuickTime, Quality 340 (33%)',
    'f56a4679494e5af4692381caa63b9062:11' => 'Apple QuickTime, Quality 341 (33%)',
    '78787c9f0aae4ab8d15ab47eaea5035c:11' => 'Apple QuickTime, Quality 342 (33%)',
    'a1664b510ce4c6aa3588cdbc327a6f57:11' => 'Apple QuickTime, Quality 343 (33%)',
    'f6150beda200179d9744527637e52baa:11' => 'Apple QuickTime, Quality 344 (34%)',
    '620244f053fef313466fbcb232077aca:11' => 'Apple QuickTime, Quality 345 (34%)',
    '91c1b36d4411306ba3afaea0658f1ad8:11' => 'Apple QuickTime, Quality 346 (34%)',
    '0381b4e34e700adecd618afdcfb5513e:11' => 'Apple QuickTime, Quality 347 (34%)',
    '4d8f909ee8cb53e0386eb09c1591099b:11' => 'Apple QuickTime, Quality 348 (34%)',
    'f7425d5d0a0207e6dfaa0ee7c35d4ec6:11' => 'Apple QuickTime, Quality 349 (34%)',
    '8d0663f8149a308365e18bdeb8c867e8:11' => 'Apple QuickTime, Quality 350 (34%)',
    '4fa58542b5953534072b6dc1085deadf:11' => 'Apple QuickTime, Quality 351 (34%)',
    '2358594d2a85b48dc0bd03e024dec9bd:11' => 'Apple QuickTime, Quality 352 (34%)',
    'b9594c8100236f288cdc01e6488cbc41:11' => 'Apple QuickTime, Quality 353 (34%)',
    '3542444d51fa859ed5af78a1f5fc4f36:11' => 'Apple QuickTime, Quality 354 (35%)',
    '7c95c94440f652232530fe4c411be1a2:11' => 'Apple QuickTime, Quality 355 (35%)',
    '8361a9dbb5d93ad098a0ce2091b0bdf5:11' => 'Apple QuickTime, Quality 356 (35%)',
    '44c8e4d0d7678034cb206609652ffeef:11' => 'Apple QuickTime, Quality 357 (35%)',
    '4694896b11fb898106e30fd4ed50cded:11' => 'Apple QuickTime, Quality 358 (35%)',
    '9ddc6134fe65ea64048fdfd27c82bed7:11' => 'Apple QuickTime, Quality 359 (35%)',
    'c60f75f7e09f0454db9cc48392a7eeed:11' => 'Apple QuickTime, Quality 360 (35%)',
    '151731e5cd38be847f4dad794c023a69:11' => 'Apple QuickTime, Quality 361 (35%)',
    '0468ecbf6fc1303467adfdcab8edfe6d:11' => 'Apple QuickTime, Quality 362 (35%)',
    'debd5adf671e3b907c10155cc910dcc1:11' => 'Apple QuickTime, Quality 363 (35%)',
    '6385ee79b090ea430190dbe1ee93ddca:11' => 'Apple QuickTime, Quality 364 (36%)',
    '67ed20f2fe283549dae4ba40860c3777:11' => 'Apple QuickTime, Quality 365 (36%)',
    'e168523157ee45551ba30378d597dfd6:11' => 'Apple QuickTime, Quality 366 (36%)',
    'd0cbe6c7372724a802d0183c6de66f8b:11' => 'Apple QuickTime, Quality 367 (36%)',
    '38a1f9d86241eb3b96d5d42bc6587598:11' => 'Apple QuickTime, Quality 368 (36%)',
    '4dc4b433113acbde9d77a4cbad69bb14:11' => 'Apple QuickTime, Quality 369 (36%)',
    '186948d91ea43a64f874ebb9dee44564:11' => 'Apple QuickTime, Quality 370 (36%)',
    '786aa4e46172ac65e10b230f3dcaadb2:11' => 'Apple QuickTime, Quality 371 (36%)',
    'ac76c6ebb64c843736fc765a03674d94:11' => 'Apple QuickTime, Quality 372 (36%)',
    '38a60cdb8033a9f90027895eab0c40ba:11' => 'Apple QuickTime, Quality 373 (36%)',
    'ce48f7fb2ba9edee46c3f4839b40ef60:11' => 'Apple QuickTime, Quality 374 (37%)',
    '7b3058792db9876a86c65ec44c0261b3:11' => 'Apple QuickTime, Quality 375 (37%)',
    '5e983407295808e244f6bdece469c8be:11' => 'Apple QuickTime, Quality 376 (37%)',
    '68ff8bfc0e15c93586ef6b4cf347469c:11' => 'Apple QuickTime, Quality 377 (37%)',
    '9d4a8c44917390e56bca2352a8a4b1be:11' => 'Apple QuickTime, Quality 378 (37%)',
    '36e7560256c5ffd285a1ca0f6d4bf97d:11' => 'Apple QuickTime, Quality 379 (37%)',
    'a88bad671d80cf6a70bd6e37be9c95c9:11' => 'Apple QuickTime, Quality 380 (37%)',
    '23ab27876006666358e95d9c1104bcd0:11' => 'Apple QuickTime, Quality 381 (37%)',
    'b87750acf49940bf1f01f6a134a600b1:11' => 'Apple QuickTime, Quality 382 (37%)',
    '731fa7404c090db157030e40804604b6:11' => 'Apple QuickTime, Quality 383 (37%)',
    '442c2664c07af1ec15d86581f43aab0b:11' => 'Apple QuickTime, Quality 384 (38%)',
    '145c52a48a9b2e954e785c3f8df5c27e:11' => 'Apple QuickTime, Quality 385 (38%)',
    '55d37ee1e3c8d12a70e67206fa1c9b0c:11' => 'Apple QuickTime, Quality 386 (38%)',
    '60880ff1f7bfe6a85cd80c2d4582395b:11' => 'Apple QuickTime, Quality 387 (38%)',
    '85fc5daf51e6cbb04352016c817e5714:11' => 'Apple QuickTime, Quality 388 (38%)',
    'd7c835210eec5a8bedb3a18d32cbe066:11' => 'Apple QuickTime, Quality 389 (38%)',
    '315e7fee22864b37b1b7670957f259fe:11' => 'Apple QuickTime, Quality 390 (38%)',
    '46dd3917c1473ed0f8fc3f1e6f08416d:11' => 'Apple QuickTime, Quality 391 (38%)',
    '1e93645e6163af46937c35a18b55c601:11' => 'Apple QuickTime, Quality 392 (38%)',
    'ba1a32697c0ae4e76a78f4b5624a8ce0:11' => 'Apple QuickTime, Quality 393 (38%)',
    '68a0d6250be9df2c05556ff59988c499:11' => 'Apple QuickTime, Quality 394 (38%)',
    '13b1310840627eddaf435e9feffebebe:11' => 'Apple QuickTime, Quality 395 (39%)',
    '2e420a34dcf01dab91fd8509d4dbaab5:11' => 'Apple QuickTime, Quality 396 (39%)',
    'eef3afec34329517513541a8509b7aab:11' => 'Apple QuickTime, Quality 397 (39%)',
    '80bdd75a2fc87b5288bc77763481df83:11' => 'Apple QuickTime, Quality 398 (39%)',
    'e230e3ac7c740f3e8fe6bc74fff72c10:11' => 'Apple QuickTime, Quality 399 (39%)',
    '68b07a219cda4b9fc9a8507b788d8230:11' => 'Apple QuickTime, Quality 400 (39%)',
    'f1c23475d19d9e950dbc4086902365a3:11' => 'Apple QuickTime, Quality 401 (39%)',
    'dc2af4340202aa481491b86539888720:11' => 'Apple QuickTime, Quality 402 (39%)',
    '495aeee0f43a596938c98c5364feb2ee:11' => 'Apple QuickTime, Quality 403 (39%)',
    '8319dfe3caedea6988e5024b0196d317:11' => 'Apple QuickTime, Quality 404 (39%)',
    'd62cd17e8e04ebd568a8f5abc38cad4a:11' => 'Apple QuickTime, Quality 405 (40%)',
    '047e1711c44262f352034452d0b0d07b:11' => 'Apple QuickTime, Quality 406 (40%)',
    '552bf986ae119444955ded5f485d5dc4:11' => 'Apple QuickTime, Quality 407 (40%)',
    'd39329b38fdcabe9e1ae5f1b205c825a:11' => 'Apple QuickTime, Quality 408 (40%)',
    'bf2904a3e3870a2b4d060e0863530d92:11' => 'Apple QuickTime, Quality 409 (40%)',
    '1acceb7ae4f9edbb835006d97ca30094:11' => 'Apple QuickTime, Quality 410 (40%)',
    '5de50c687a6e885634bf16adfd75e6bc:11' => 'Apple QuickTime, Quality 411 (40%)',
    'd92c5bba7cfd1bfbb8c662c1a27ca413:11' => 'Apple QuickTime, Quality 412 (40%)',
    'cea44b5645cd1dbf469c8ae5600e4ff5:11' => 'Apple QuickTime, Quality 413 (40%)',
    '5caf3989a757842c716220e4e426bde2:11' => 'Apple QuickTime, Quality 414 (40%)',
    'fd2f7a6518c12848a9ecdb1c3beb1fa8:11' => 'Apple QuickTime, Quality 415 (41%)',
    'de0547c872fed9c9c75c8fec2fe010e6:11' => 'Apple QuickTime, Quality 416 (41%)',
    '6cf4dfbe3df89d9728e0f34b7b145223:11' => 'Apple QuickTime, Quality 417 (41%)',
    'd94d79b70686d3e2568d61d07e5819eb:11' => 'Apple QuickTime, Quality 418 (41%)',
    '9ec82f50503769a9bb17e876594833b6:11' => 'Apple QuickTime, Quality 419 (41%)',
    '4a39b0ae55f0eaa5672f00015cae2d40:11' => 'Apple QuickTime, Quality 420 (41%)',
    '42bae7ef4a41562b2e98d74248f4f22e:11' => 'Apple QuickTime, Quality 421 (41%)',
    '8c0ea132cfacf212c518ad297229be34:11' => 'Apple QuickTime, Quality 422 (41%)',
    'c1af7f1a3716bef087124306b068605c:11' => 'Apple QuickTime, Quality 423 (41%)',
    'd3ea3c519f92dd870fed03f63cabf05e:11' => 'Apple QuickTime, Quality 424 (41%)',
    '0442196d850319833f27df632e92f064:11' => 'Apple QuickTime, Quality 425 (42%)',
    '8c372e99fa96d2598e431f8137e47da6:11' => 'Apple QuickTime, Quality 426 (42%)',
    '9bade640c3fcb807ed1322479f9e7f1c:11' => 'Apple QuickTime, Quality 427 (42%)',
    '8015ad9fa22d6565ca61ce9979f3663f:11' => 'Apple QuickTime, Quality 428 (42%)',
    '09a13f94022839a24065b82d5f4ffdbd:11' => 'Apple QuickTime, Quality 429 (42%)',
    '0e570bc627acaee0962472a1a646816b:11' => 'Apple QuickTime, Quality 430 (42%)',
    'be7e7114f08e1775ca9676d2feeeccca:11' => 'Apple QuickTime, Quality 431 (42%)',
    '44be972c54cd64be7524a133a7395401:11' => 'Apple QuickTime, Quality 432 (42%)',
    'b2f7b5e3007387aa22df74e82e916195:11' => 'Apple QuickTime, Quality 433 (42%)',
    '5c13c8db3a0b590f4fa3ec462b8890c3:11' => 'Apple QuickTime, Quality 434 (42%)',
    '2733a3cb2e0a2313b74d686437fa3ae2:11' => 'Apple QuickTime, Quality 435 (42%)',
    '66309175abaa59d6246237a77ce9eb76:11' => 'Apple QuickTime, Quality 436 (43%)',
    '6a2eb9f07f3c96365a06d91da171e673:11' => 'Apple QuickTime, Quality 437 (43%)',
    'e395118c42b6492dd4d9d30754f0a697:11' => 'Apple QuickTime, Quality 438 (43%)',
    '7a75abc5c5ec8cc0fa43f239ab048c08:11' => 'Apple QuickTime, Quality 439 (43%)',
    'c558f3407dc549c902efad68c54920de:11' => 'Apple QuickTime, Quality 440 (43%)',
    '49cd1849b501868260d8a3b1e96d8625:11' => 'Apple QuickTime, Quality 441 (43%)',
    'f7d70cfab7ff888c97078d277fa01307:11' => 'Apple QuickTime, Quality 442 (43%)',
    'f06ddea698cebe653bdd0c208c3d8c95:11' => 'Apple QuickTime, Quality 443 (43%)',
    '73478bb7714d1d2342bbf22c5fdc04d6:11' => 'Apple QuickTime, Quality 444 (43%)',
    '24406aee81b89ea50881ae71f878d0ec:11' => 'Apple QuickTime, Quality 445 (43%)',
    'e3e88302627b6743725cace74ddb17f9:11' => 'Apple QuickTime, Quality 446 (44%)',
    '3fb9c046dff30dcb4128df984532d6ba:11' => 'Apple QuickTime, Quality 447 (44%)',
    'c9a4b04bc8e580608014b6f3111322d7:11' => 'Apple QuickTime, Quality 448 (44%)',
    '7749ec06b1f1b1be30aa58dbef838d49:11' => 'Apple QuickTime, Quality 449 (44%)',
    'c0a54e87a2ef1c163311bcc1abf85214:11' => 'Apple QuickTime, Quality 450 (44%)',
    'f20a253d2513f4d8f2cfeea980852820:11' => 'Apple QuickTime, Quality 451 (44%)',
    '6538fc6f5f1744b40c0b8b5bc7179983:11' => 'Apple QuickTime, Quality 452 (44%)',
    '68a4a67af696f82bbbb7db15a16c0c46:11' => 'Apple QuickTime, Quality 453 (44%)',
    'e477932560b308940ac7439eed9f63da:11' => 'Apple QuickTime, Quality 454 (44%)',
    'fd732b0493e7ff16da4bde7faa88e22d:11' => 'Apple QuickTime, Quality 455 (44%)',
    '61884dc8b93e63c07bb487a6e29d6fb7:11' => 'Apple QuickTime, Quality 456 (45%)',
    '67515a725833d40535a54b4ef9551e05:11' => 'Apple QuickTime, Quality 457 (45%)',
    'cc6bb734b742b0631ab6562a329e1603:11' => 'Apple QuickTime, Quality 458 (45%)',
    '1801686a97836f690ce3d5523ffcfa9a:11' => 'Apple QuickTime, Quality 459 (45%)',
    'f73f690cacd5d4e247f59964ad0f43b9:11' => 'Apple QuickTime, Quality 460 (45%)',
    '853946ede6a624136546ec5b68ecdc49:11' => 'Apple QuickTime, Quality 461 (45%)',
    'f05c48d79edbefdb4d260dc23cf258e6:11' => 'Apple QuickTime, Quality 462 (45%)',
    '8dc8361e94137f5466c8dd1f9aa06781:11' => 'Apple QuickTime, Quality 463 (45%)',
    '6c121faf4784a5a93fbf7fff4470dea4:11' => 'Apple QuickTime, Quality 464-465 (45%)',
    'd0eaa368737f17f6037757d393a22599:11' => 'Apple QuickTime, Quality 466-467 (46%)',
    '9c6f5faa1009cafe8bc3060fe18d4b60:11' => 'Apple QuickTime, Quality 468 (46%)',
    'ac47d493602dddace7844a9bc962e5ed:11' => 'Apple QuickTime, Quality 469 (46%)',
    '24784b5651e1790242c01de522a6e05b:11' => 'Apple QuickTime, Quality 470 (46%)',
    'adbb56f1f0e0392392f9c7a38351a9ec:11' => 'Apple QuickTime, Quality 471 (46%)',
    '6e0952a44c37bc2d98dbede4ec429c99:11' => 'Apple QuickTime, Quality 472 (46%)',
    '20c7942ddec30475a182cb281f12bc03:11' => 'Apple QuickTime, Quality 473 (46%)',
    '4080277d75b20871d00ebc01ffbdb848:11' => 'Apple QuickTime, Quality 474 (46%)',
    '4f15f7e4c56e7a75c0fe5454ab7e8f72:11' => 'Apple QuickTime, Quality 475 (46%)',
    'ff82adb92189413246aee9a992eb2013:11' => 'Apple QuickTime, Quality 476 (46%)',
    '16df79eb7c5f062aeebde385fbce1553:11' => 'Apple QuickTime, Quality 477 (47%)',
    'df02b0ea9dab7d291950b6cfc65c4bb1:11' => 'Apple QuickTime, Quality 478 (47%)',
    '4ca8ec2a0c651e0508aab3b153cfee23:11' => 'Apple QuickTime, Quality 479 (47%)',
    'bbbae155e558e9d37686ec34bd065a53:11' => 'Apple QuickTime, Quality 480 (47%)',
    '9bf86a5ec6e5382f214e07364a62b1b3:11' => 'Apple QuickTime, Quality 481 (47%)',
    '41d873034f29b298d899b48cd321c93f:11' => 'Apple QuickTime, Quality 482 (47%)',
    '52092035b4e3fd45de3298c4d641385a:11' => 'Apple QuickTime, Quality 483 (47%)',
    '70e5babe9507bae6725e401a36903070:11' => 'Apple QuickTime, Quality 484 (47%)',
    '3cfa966dde2536c83c921aa250b978b3:11' => 'Apple QuickTime, Quality 485 (47%)',
    'b0144b1d2671d145d29812ebcebd863d:11' => 'Apple QuickTime, Quality 486 (47%)',
    '2a6a136faaf1f13c2b80dcb4786d90b2:11' => 'Apple QuickTime, Quality 487 (48%)',
    '3a6eac793d818f378e7b24826c9115cc:11' => 'Apple QuickTime, Quality 488 (48%)',
    '4a78c6570fc84378e3334bfcd8a5680f:11' => 'Apple QuickTime, Quality 489 (48%)',
    '0709c0afc0eae932a50903e56ec95ad2:11' => 'Apple QuickTime, Quality 490 (48%)',
    'b013b5c9b7bafc9dcad9a1e87fc629ff:11' => 'Apple QuickTime, Quality 491 (48%)',
    '7cb380e582317b8387037450cc68db5e:11' => 'Apple QuickTime, Quality 492 (48%)',
    'f94618c1a011209cb3b060887c7e244e:11' => 'Apple QuickTime, Quality 493 (48%)',
    '649a90949cab8f45d3ecef78068165d1:11' => 'Apple QuickTime, Quality 494 (48%)',
    '70e105a22b036f7c1ce0b5d02fa1c34e:11' => 'Apple QuickTime, Quality 495 (48%)',
    '18c10ea6fe5918e09daf1a3a7a74e678:11' => 'Apple QuickTime, Quality 496 (48%)',
    '23822cafcc61ce2a52691f1fc963ff18:11' => 'Apple QuickTime, Quality 497 (49%)',
    '5bb2cf3e6721c2dd8eb3341f9bff4159:11' => 'Apple QuickTime, Quality 498 (49%)',
    'e55b5345d9668d1b11b657537f707072:11' => 'Apple QuickTime, Quality 499 (49%)',
    '4cbebcb06d1003e29429e9d5c9445919:11' => 'Apple QuickTime, Quality 500 (49%)',
    '916b16f020b2b21e4c8114da8c05d584:11' => 'Apple QuickTime, Quality 501 (49%)',
    '98abef3366c7f451e44f5c2799e2be6d:11' => 'Apple QuickTime, Quality 502 (49%)',
    'f8e99ed03828752f16c51bb8c9887e9e:11' => 'Apple QuickTime, Quality 503 (49%)',
    'eef05c558c1aba5cf2891fb13ee07167:11' => 'Apple QuickTime, Quality 504 (49%)',
    'cb5fc7927d88ac99f556b2dd7985eaf9:11' => 'Apple QuickTime, Quality 505 (49%)',
    'f0e3f635bbcf96654812e8c78b227701:11' => 'Apple QuickTime, Quality 506 (49%)',
    '234c9cf6d7fe671b52c3ec5a20046ec8:11' => 'Apple QuickTime, Quality 507 (50%)',
    '9dfcc9ae3baee4bb4ad63abf2f740275:11' => 'Apple QuickTime, Quality 508 (50%)',
    'b1f1b6519991ac7696b233dd9b9de6b5:11' => 'Apple QuickTime, Quality 509 (50%)',
    'dba2f5203ffecada66a8bf9b1272f1eb:11' => 'Apple QuickTime, Quality 510 (50%)',
    '367b3d63cddc0cd27e58030c2b8f1aaa:11' => 'Apple QuickTime, Quality 511 (50%)',
    'c28ab3fd6480c92028327957228c0a11:11' => 'Apple QuickTime, Quality 512 (50%)',
    '6bc9ebaf9f3ed62ec8818076f6f81c7f:11' => 'Apple QuickTime, Quality 513 (50%)',
    '3bbfcd817441d2267a49bf76b48c5f47:11' => 'Apple QuickTime, Quality 514 (50%)',
    '27e6ed2cecfebe31eb3d66128c926562:11' => 'Apple QuickTime, Quality 515 (50%)',
    '3cf112c5843f98410599ea2a197e5cf6:11' => 'Apple QuickTime, Quality 516 (50%)',
    '2821aae8108df4bd98e5eaa451a351d2:11' => 'Apple QuickTime, Quality 517 (50%)',
    '798f48b6dbe3f1cd7b40b03fae8d2611:11' => 'Apple QuickTime, Quality 518 (51%)',
    '67a7c8896d03a030b56130e1f9c5caad:11' => 'Apple QuickTime, Quality 519 (51%)',
    '4bf1b53c292dec3f7cf3c020a3a9d911:11' => 'Apple QuickTime, Quality 520 (51%)',
    '2bfe0ace876b80be6f601a1703187d94:11' => 'Apple QuickTime, Quality 521 (51%)',
    '7590bc1a40090163a101bfd28daa3fc2:11' => 'Apple QuickTime, Quality 522 (51%)',
    '118c7b118b1df404c90cfb1d10cf2a77:11' => 'Apple QuickTime, Quality 523 (51%)',
    '86cfac24ca9f4ab254f882ad399ea758:11' => 'Apple QuickTime, Quality 524 (51%)',
    '0268c3e9e3e1c3e6eb25fe0d31940c7f:11' => 'Apple QuickTime, Quality 525 (51%)',
    '9beb1b7c55129a34c850c359d7263457:11' => 'Apple QuickTime, Quality 526 (51%)',
    '0cec2c8f96c092bd6e7cf0f7ea294c99:11' => 'Apple QuickTime, Quality 527 (51%)',
    'd8c5179c2419775f43e9a7899bacddd7:11' => 'Apple QuickTime, Quality 528 (52%)',
    '9ce50f6e0b00d2e601f2fcc151abc4d8:11' => 'Apple QuickTime, Quality 529 (52%)',
    '80db52b1671d32d8bd3126bf1d7db8ec:11' => 'Apple QuickTime, Quality 530 (52%)',
    'f3ef06f90579eaf1008e07b94e818a40:11' => 'Apple QuickTime, Quality 531 (52%)',
    '3c85026793f58eb45141847a27854fe2:11' => 'Apple QuickTime, Quality 532-533 (52%)',
    'e3042cbd43d2067ae92e1a8ce3f2c5a1:11' => 'Apple QuickTime, Quality 534 (52%)',
    'dd71fdab3d46341a9b6ca0b6c6929d23:11' => 'Apple QuickTime, Quality 535 (52%)',
    'a451c79ccddcd543a80e1ce0449dcb0d:11' => 'Apple QuickTime, Quality 536 (52%)',
    '2da67fe5f0bb3c8b10403295895fb154:11' => 'Apple QuickTime, Quality 537 (52%)',
    '4f72f3cdc82d433e7f749be8036d4ce0:11' => 'Apple QuickTime, Quality 538 (53%)',
    'da29cc9a4d5fd7e0dc36a2dd0c70e84f:11' => 'Apple QuickTime, Quality 539 (53%)',
    '63f61a0d3c4f1ace8ebe5b6ae23e3f25:11' => 'Apple QuickTime, Quality 540 (53%)',
    'fde14219617069bbf6b26dcb42036de7:11' => 'Apple QuickTime, Quality 541 (53%)',
    'c84313bc621c6d05999510fa57c56d05:11' => 'Apple QuickTime, Quality 542 (53%)',
    '92658d4c879d6e48bfda1a6e9f49ef8d:11' => 'Apple QuickTime, Quality 543 (53%)',
    '3deffd01a1c03929873dddd86a5339f1:11' => 'Apple QuickTime, Quality 544 (53%)',
    'e76c1b26bbd196efe2e793e27727704d:11' => 'Apple QuickTime, Quality 545 (53%)',
    '9f3289994c790a10ecb2d93021677840:11' => 'Apple QuickTime, Quality 546 (53%)',
    'f7493b01895b7880c651841c73678d33:11' => 'Apple QuickTime, Quality 547 (53%)',
    '3c9d094741c995c2c0ac9daf14c4e683:11' => 'Apple QuickTime, Quality 548 (54%)',
    '8c0e1d4cd6138817963af6ca149cb5d5:11' => 'Apple QuickTime, Quality 549 (54%)',
    'd781cc6f686fc7c7b9b6eef90fab4d87:11' => 'Apple QuickTime, Quality 550 (54%)',
    '7dd6377a907070b1ca7e05f770ca2aab:11' => 'Apple QuickTime, Quality 551-552 (54%)',
    'c2d5e2e93ec191015d8181c9e25387d8:11' => 'Apple QuickTime, Quality 553 (54%)',
    '65d20361f4ba0725cf150c7ae2033776:11' => 'Apple QuickTime, Quality 554 (54%)',
    '2d7eb8eb9df9f4831a843626f4fc7e19:11' => 'Apple QuickTime, Quality 555 (54%)',
    'ce0e14187fec73f57242becd633a89a3:11' => 'Apple QuickTime, Quality 556 (54%)',
    '140cc5a99ef865e318a217ea069aa84d:11' => 'Apple QuickTime, Quality 557 (54%)',
    '0db59bd18beb49f9beb901f3435e22a5:11' => 'Apple QuickTime, Quality 558 (54%)',
    '78d19da8de8095644aa31fb409033fe7:11' => 'Apple QuickTime, Quality 559 (55%)',
    '00687c4e4852ed1cd446c09a3764e505:11' => 'Apple QuickTime, Quality 560 (55%)',
    'f3795398903c82e1beababf95d3a8413:11' => 'Apple QuickTime, Quality 561 (55%)',
    'b749b90354443bf17da7a67a5ad53397:11' => 'Apple QuickTime, Quality 562 (55%)',
    'c722656df4bb0651821cd90880953a20:11' => 'Apple QuickTime, Quality 563 (55%)',
    '4d17c873e65b9d398f27735b0020c777:11' => 'Apple QuickTime, Quality 564 (55%)',
    '067a76c2e5386ae85f9187e3e2134621:11' => 'Apple QuickTime, Quality 565 (55%)',
    '09b689f7d0c1d4bb0d96d06c02b8dcf8:11' => 'Apple QuickTime, Quality 566 (55%)',
    'ffc0192eb5a182370a641cffe9b1d71f:11' => 'Apple QuickTime, Quality 567 (55%)',
    '3af2163438180050bbcf123d4f4587d3:11' => 'Apple QuickTime, Quality 568 (55%)',
    '3bdb097a9791f3ce6d7bbc4d6a194aa4:11' => 'Apple QuickTime, Quality 569 (56%)',
    '8c1fead15819016583650eff5a4f5bda:11' => 'Apple QuickTime, Quality 570 (56%)',
    '967aeb5bc4d75a0d5c0998bbfb282982:11' => 'Apple QuickTime, Quality 571 (56%)',
    'cdd7bd689f14d5f7c3ea790f6f09ae64:11' => 'Apple QuickTime, Quality 572 (56%)',
    '6a20041beb5b67d38525bb7507ffeb49:11' => 'Apple QuickTime, Quality 573 (56%)',
    '2bf57fbe54370c4d917f259631af033e:11' => 'Apple QuickTime, Quality 574 (56%)',
    'd2e983c44eae2983f48f526992fbbfb4:11' => 'Apple QuickTime, Quality 575 (56%)',
    'c5bb48d86e26ac496bb4b4bc888cc06a:11' => 'Apple QuickTime, Quality 576 (56%)',
    'a595c40b5dbd45557c3c8d23ebee5e24:11' => 'Apple QuickTime, Quality 577 (56%)',
    'f2d84e1114ef85682818b96720d439b5:11' => 'Apple QuickTime, Quality 578 (56%)',
    '332d9c49c5b32ae2addbb06a1e32fd49:11' => 'Apple QuickTime, Quality 579 (57%)',
    '19c5c7c0270bd36c49f695475a62c293:11' => 'Apple QuickTime, Quality 580 (57%)',
    '6516e60b3995e21b6750ebca1ddcfee5:11' => 'Apple QuickTime, Quality 581 (57%)',
    '8daf4a87b28106876529a549cf1040b8:11' => 'Apple QuickTime, Quality 582 (57%)',
    '6b37d6acc52259bf972a41e84dea7754:11' => 'Apple QuickTime, Quality 583 (57%)',
    '588666e111892f10ca3f17bc362d9276:11' => 'Apple QuickTime, Quality 584 (57%)',
    '9ec2859c370f557783903608748e7fb1:11' => 'Apple QuickTime, Quality 585 (57%)',
    '18e3ac85da74fe92ab5da3d5f7614e09:11' => 'Apple QuickTime, Quality 586 (57%)',
    'da4c88f145393972fbe9d3f40838cab9:11' => 'Apple QuickTime, Quality 587 (57%)',
    '8ae7d2b569c437904a20a10bbd21fe89:11' => 'Apple QuickTime, Quality 588 (57%)',
    'fc6dfb9669566b249cb03228aeb020c3:11' => 'Apple QuickTime, Quality 589 (58%)',
    '198f3a32e4036d7c37fbc0c343d883af:11' => 'Apple QuickTime, Quality 590 (58%)',
    '876ec039f82e49b925b232843b4703d4:11' => 'Apple QuickTime, Quality 591 (58%)',
    '673b05962b8255cbc9bdbbc48965b4b7:11' => 'Apple QuickTime, Quality 592 (58%)',
    '02e9a58edf45d75be000dee144316c66:11' => 'Apple QuickTime, Quality 593 (58%)',
    '2abebf7a61009c5c1aa9516539b9084e:11' => 'Apple QuickTime, Quality 594 (58%)',
    'fe44d8625d6242f4b5deb82be8ccaacf:11' => 'Apple QuickTime, Quality 595 (58%)',
    '66529cc8ef9694e6a37e8787d0f160fd:11' => 'Apple QuickTime, Quality 596 (58%)',
    '2129ee2bff47bfa8a8bb79ea9fb67b92:11' => 'Apple QuickTime, Quality 597 (58%)',
    'ccb55ec1549a51212859495e104c626b:11' => 'Apple QuickTime, Quality 598 (58%)',
    '05e53fb216ba4a1734eefaccd249d8e2:11' => 'Apple QuickTime, Quality 599 (58%)',
    'f4923d7b7dedd365646169e720eee427:11' => 'Apple QuickTime, Quality 600 (59%)',
    'c0f2265630bdf5f29e8c95df25c89edb:11' => 'Apple QuickTime, Quality 601 (59%)',
    '0fdd23e8274090da3c925a3db7303adf:11' => 'Apple QuickTime, Quality 602 (59%)',
    'a5bcbd80472fdf697db770ac78d6a4e3:11' => 'Apple QuickTime, Quality 603 (59%)',
    'b157c4aabba2816f391c8f76ca3d4072:11' => 'Apple QuickTime, Quality 604 (59%)',
    'b8ef26dabc2d81a8ba13b1f49ea711d3:11' => 'Apple QuickTime, Quality 605 (59%)',
    '8a983844f9b0aec26fc8ac75a258e3ac:11' => 'Apple QuickTime, Quality 606 (59%)',
    '44ced96f11e3a410201beed353a864cf:11' => 'Apple QuickTime, Quality 607 (59%)',
    '4a14a3e37c89e5a7570f672b1970ca55:11' => 'Apple QuickTime, Quality 608 (59%)',
    'e5ec78e112e3ba6463de24b3518347eb:11' => 'Apple QuickTime, Quality 609 (59%)',
    '29b7cdc7a570b950457d20541c22c4ce:11' => 'Apple QuickTime, Quality 610 (60%)',
    '470c0c761e2bb5e314a7112f3d64b277:11' => 'Apple QuickTime, Quality 611 (60%)',
    'ced483058f2abf19df0f7935dafd217a:11' => 'Apple QuickTime, Quality 612 (60%)',
    '8a202c89c57e77f50e1df27a3be7d5b7:11' => 'Apple QuickTime, Quality 613 (60%)',
    '7eb9fe0338a7b802860a60e0088418fd:11' => 'Apple QuickTime, Quality 614 (60%)',
    'ac25112c596d62f95518af109457975c:11' => 'Apple QuickTime, Quality 615 (60%)',
    'f57e9a5f1d8dea7fd83a1b5840243686:11' => 'Apple QuickTime, Quality 616-617 (60%)',
    '0eef5c6ff8ba65ff799081a9c96a2297:11' => 'Apple QuickTime, Quality 618-619 (60%)',
    '05f98e12bfa14ba6347fb43f2241ba43:11' => 'Apple QuickTime, Quality 620 (61%)',
    '91885755c780ebe16b1278a0359eda83:11' => 'Apple QuickTime, Quality 621 (61%)',
    'b44cb1ca15fc9e3a27420df2ddae5879:11' => 'Apple QuickTime, Quality 622 (61%)',
    'ae6c2112dd560530b7bacc8bfa9fb7f6:11' => 'Apple QuickTime, Quality 623 (61%)',
    'd3b05f3cd78e0ac3ab37a02152e22831:11' => 'Apple QuickTime, Quality 624 (61%)',
    'b0c2c3f76d848ee2e8f47a9a90131a21:11' => 'Apple QuickTime, Quality 625 (61%)',
    '752fbc15f77a8c2149f5ae6bf49204b8:11' => 'Apple QuickTime, Quality 626 (61%)',
    '9c34d3dedfe47d95edabdcbc5568a2a8:11' => 'Apple QuickTime, Quality 627 (61%)',
    '252ba8a31aeb601e23b3a70f9af7abc1:11' => 'Apple QuickTime, Quality 628 (61%)',
    '359038cd7c45242e96e176e91210922d:11' => 'Apple QuickTime, Quality 629 (61%)',
    '8ab83dc2e7ca8b9db1f3b0ab500f3aca:11' => 'Apple QuickTime, Quality 630 (62%)',
    '21c9f9a0ff71d4528ef7a19d2cfd0b6c:11' => 'Apple QuickTime, Quality 631 (62%)',
    'a62fb887c209e0fab99fcb7ac81137a2:11' => 'Apple QuickTime, Quality 632 (62%)',
    'e454ca92aca849d59b873d9be817baea:11' => 'Apple QuickTime, Quality 633 (62%)',
    '9ad6008e7b4f8478043bfa54e1d9e48e:11' => 'Apple QuickTime, Quality 634 (62%)',
    '45b20f0b0d7d88d8330354212af2e087:11' => 'Apple QuickTime, Quality 635 (62%)',
    'b0a00ffee2e55457cd999bba2d07f63e:11' => 'Apple QuickTime, Quality 636 (62%)',
    '5dfcb96d9a1186f662c6adb38bb9520a:11' => 'Apple QuickTime, Quality 637 (62%)',
    'd973f38c501796adff40c4f70cbd8885:11' => 'Apple QuickTime, Quality 638 (62%)',
    'dcfe5898ec101a8b2bf98330445dd1bf:11' => 'Apple QuickTime, Quality 639 (62%)',
    '2b81eecf0fecd33679adac27e79ef3f4:11' => 'Apple QuickTime, Quality 640-641 (63%)',
    '5aad44c55bf4f6dc538eaca006cafac2:11' => 'Apple QuickTime, Quality 642 (63%)',
    '7079c2d71ff33e7c4e8efdece23c307b:11' => 'Apple QuickTime, Quality 643 (63%)',
    'a5f724e9d7148f1a84ee597f45c33141:11' => 'Apple QuickTime, Quality 644 (63%)',
    '3ae705fae9e895eda345d482525215e3:11' => 'Apple QuickTime, Quality 645 (63%)',
    '7cbd635e5fee8bbd290b0d383b03da5a:11' => 'Apple QuickTime, Quality 646 (63%)',
    '0f125e2c5ee6b123cf67b586ea23d422:11' => 'Apple QuickTime, Quality 647 (63%)',
    '87eac5c1375cca9aa16eba0704616a7b:11' => 'Apple QuickTime, Quality 648-649 (63%)',
    'aa83fd556c569ddcd81e0cc1ba866373:11' => 'Apple QuickTime, Quality 650 (63%)',
    'dab4fa97da49aa37889185c5b43917c1:11' => 'Apple QuickTime, Quality 651 (64%)',
    '51ad55cb254f36748123ca83f43556f4:11' => 'Apple QuickTime, Quality 652 (64%)',
    '86e707c017682fe08213216d064b1b51:11' => 'Apple QuickTime, Quality 653 (64%)',
    '3730182602996b4a1d540eb3fd970072:11' => 'Apple QuickTime, Quality 654 (64%)',
    '1bf7a5d7477ad75b9c7b281de622d53b:11' => 'Apple QuickTime, Quality 655 (64%)',
    '82b4bc7c4a832b620e810311a33c9771:11' => 'Apple QuickTime, Quality 656 (64%)',
    '6502d634e5bf3f849e9d382886fc32fe:11' => 'Apple QuickTime, Quality 657 (64%)',
    'a10e87fa030f8177a4f59f8d16a20afd:11' => 'Apple QuickTime, Quality 658 (64%)',
    '338a78a7658ff3c1de33d88aa0ab7c74:11' => 'Apple QuickTime, Quality 659 (64%)',
    '4c30c4399ef4bb2920601d940ed404eb:11' => 'Apple QuickTime, Quality 660-661 (64-65%)',
    'ab7cfb73667875854893982a8cfcfab9:11' => 'Apple QuickTime, Quality 662 (65%)',
    '7bff346b97abf46ca005af4e74b560fa:11' => 'Apple QuickTime, Quality 663 (65%)',
    '2eae93ed601a50284ee31c20651584cb:11' => 'Apple QuickTime, Quality 664 (65%)',
    'e566eaef7eacd6c7161feebf4cec79e8:11' => 'Apple QuickTime, Quality 665 (65%)',
    'a9f461d3bca42dfab57c834fa5f34419:11' => 'Apple QuickTime, Quality 666 (65%)',
    '2b394105d4dd418e79b32e66496679d4:11' => 'Apple QuickTime, Quality 667 (65%)',
    'c65677d79e37baf57767e10d7b7f1ce8:11' => 'Apple QuickTime, Quality 668 (65%)',
    '688c7dca6b12c22a21e0caf1a0336c80:11' => 'Apple QuickTime, Quality 669-671 (65-66%)',
    '20d4b3c9e3c292c68181974704fe5048:11' => 'Apple QuickTime, Quality 672-673 (66%)',
    '3bc3948025869f25b143aa517b2154ac:11' => 'Apple QuickTime, Quality 674 (66%)',
    '8a2ae82991917070de49cc48d104446f:11' => 'Apple QuickTime, Quality 675 (66%)',
    'b5424d9dce37dd5c9e0e38bcc775f48e:11' => 'Apple QuickTime, Quality 676 (66%)',
    '28f718af4edb0069a1fdab00f6ea978d:11' => 'Apple QuickTime, Quality 677 (66%)',
    '910416483a49503202cbe3ecee33afc9:11' => 'Apple QuickTime, Quality 678 (66%)',
    '153196d4f99569f9bbd3fe0e96d2909c:11' => 'Apple QuickTime, Quality 679 (66%)',
    '919a38ebb9fc0bcba388643a9b3ef27c:11' => 'Apple QuickTime, Quality 680 (66%)',
    'ef511ac2c9d7153c16e3e1846325b727:11' => 'Apple QuickTime, Quality 681 (67%)',
    'e8b4ef8f94d89c59c855758a73ec451f:11' => 'Apple QuickTime, Quality 682 (67%)',
    'a09fe2e60a7ff12e1e5ca00afa9719ef:11' => 'Apple QuickTime, Quality 683 (67%)',
    'e56cf84423c16869a9a4fd6605b15ba4:11' => 'Apple QuickTime, Quality 684 (67%)',
    'f64d88456f97a65fe562eb69e619782a:11' => 'Apple QuickTime, Quality 685 (67%)',
    '8cbf6cda8ae0249fb91c1ff5ab788a04:11' => 'Apple QuickTime, Quality 686 (67%)',
    'f0ebdd8d44ac1a80727041a087553847:11' => 'Apple QuickTime, Quality 687 (67%)',
    'c8f917220d6285cda6428a2cf6a9a1b3:11' => 'Apple QuickTime, Quality 688-689 (67%)',
    '83ff61ebceff5f888b9615b250aa7b76:11' => 'Apple QuickTime, Quality 690 (67%)',
    'a2d1de53a0882047287a954f86bc783d:11' => 'Apple QuickTime, Quality 691 (67%)',
    'c78717ac2705274888912f349eeb2c8e:11' => 'Apple QuickTime, Quality 692 (68%)',
    'e032225ecdcf1d91d85626df59a2d0c6:11' => 'Apple QuickTime, Quality 693 (68%)',
    'ebf82c50697d66a6913727095299f192:11' => 'Apple QuickTime, Quality 694-695 (68%)',
    'bcc29022955cc7532b08640ab118259c:11' => 'Apple QuickTime, Quality 696 (68%)',
    '6a87dd29703c2b3ef80f1b5d2cc8d26a:11' => 'Apple QuickTime, Quality 697 (68%)',
    '91c9d96e11d96e10b328a5f18960247b:11' => 'Apple QuickTime, Quality 698 (68%)',
    '6eeaaacec8edc933b68602b01d16204e:11' => 'Apple QuickTime, Quality 699 (68%)',
    'c638d9151dc650993b56f4effc0fe19c:11' => 'Apple QuickTime, Quality 700 (68%)',
    'c67d246229fcc0dd1b974f7df556d247:11' => 'Apple QuickTime, Quality 701 (68%)',
    '7a52e3960831057d58e9b1ba03b87cf3:11' => 'Apple QuickTime, Quality 702 (69%)',
    '907e599e3c462b498e936dfc35b20bb9:11' => 'Apple QuickTime, Quality 703-704 (69%)',
    '65bf1ddc176fe4002a7a2ecaac60e58c:11' => 'Apple QuickTime, Quality 705 (69%)',
    '40e59fdb430180502ceacf7b4034eff8:11' => 'Apple QuickTime, Quality 706 (69%)',
    '4205aec34791d70be03b90ab1e54ef8c:11' => 'Apple QuickTime, Quality 707 (69%)',
    '5c0a83e613d3bdd9ec7e86983f75b5be:11' => 'Apple QuickTime, Quality 708 (69%)',
    'b3ebdf0376c9c48cca51ea8b550f6c51:11' => 'Apple QuickTime, Quality 709 (69%)',
    '72161116404ed3cb449674760d0e4776:11' => 'Apple QuickTime, Quality 710 (69%)',
    '35b3697c265e35185e9463aac6ce9b2d:11' => 'Apple QuickTime, Quality 711-712 (69-70%)',
    '85adcc8c52c25334a9e7ea9d79433f8d:11' => 'Apple QuickTime, Quality 713 (70%)',
    '8f72f67948f264bdbd33107c33b1e0a0:11' => 'Apple QuickTime, Quality 714 (70%)',
    'c25ed5069735a3f9677ee494108a52bc:11' => 'Apple QuickTime, Quality 715 (70%)',
    'fd3167b1cdcfa1bdd37a4841d37b1624:11' => 'Apple QuickTime, Quality 716-717 (70%)',
    'a02b6b8286cf6d036961911e98bd8f89:11' => 'Apple QuickTime, Quality 718-719 (70%)',
    'd876e8934da14a985abda0ebe722bbee:11' => 'Apple QuickTime, Quality 720 (70%)',
    '31852b7659883eade6e273ac61ef0262:11' => 'Apple QuickTime, Quality 721 (70%)',
    'de3aa6ed96eaf3ed3cd3ea70a2f75002:11' => 'Apple QuickTime, Quality 722 (71%)',
    'f178977e9e0133711f395943816d26aa:11' => 'Apple QuickTime, Quality 723 (71%)',
    '18fce3103e312ce26252ec4af6cd1350:11' => 'Apple QuickTime, Quality 724 (71%)',
    'fdf6b04a2d8ac06d3fe64d1dceba4cd9:11' => 'Apple QuickTime, Quality 725 (71%)',
    '6a37c2572bb47dff514aa4b343c104b5:11' => 'Apple QuickTime, Quality 726 (71%)',
    'fdbe851f6e559bc17ce3610b91e7fead:11' => 'Apple QuickTime, Quality 727 (71%)',
    '9fee7fc42e670d6e30a5e9fcf696241d:11' => 'Apple QuickTime, Quality 728 (71%)',
    '5aedf3816a813ed63b0521b0c384a677:11' => 'Apple QuickTime, Quality 729 (71%)',
    'c1278992838bdd62e71bf41c20126a5f:11' => 'Apple QuickTime, Quality 730 (71%)',
    '791e3897f6ac92fb5e708b28dbd361b1:11' => 'Apple QuickTime, Quality 731 (71%)',
    'd5b2902ae3fcd87e1521da86585e7b3a:11' => 'Apple QuickTime, Quality 732-733 (71-72%)',
    'a99810db58e835fe4b213d707dc0b754:11' => 'Apple QuickTime, Quality 734 (72%)',
    'e8032085aa55f664f7f74201ac10bb99:11' => 'Apple QuickTime, Quality 735 (72%)',
    '488e5d04f779b15c53f76e67cccdb2ed:11' => 'Apple QuickTime, Quality 736 (72%)',
    '548a841c0a8c3b2beeb134c6c3b922fc:11' => 'Apple QuickTime, Quality 737-738 (72%)',
    'cfe4549eb2dd81684920aa32b598260c:11' => 'Apple QuickTime, Quality 739-740 (72%)',
    'b6ff9215f87e4b053aaee36381f59005:11' => 'Apple QuickTime, Quality 741 (72%)',
    'd1c8c1e1fc2bfb776d2ee1aace3fc6f9:11' => 'Apple QuickTime, Quality 742 (72%)',
    '365693ebd558aebc31a79a1abff9709d:11' => 'Apple QuickTime, Quality 743-744 (73%)',
    '0233ba670d2891e75da3ce5e7664cb67:11' => 'Apple QuickTime, Quality 745 (73%)',
    'a91f5f8e8d743e52169d965992c5021e:11' => 'Apple QuickTime, Quality 746 (73%)',
    '1a0487e7e1a8f4ade97b2e8a4cab46ec:11' => 'Apple QuickTime, Quality 747 (73%)',
    '5a2311c438c7183f6cd1f45c10e5783a:11' => 'Apple QuickTime, Quality 748 (73%)',
    '2750f3df7a97d6007d6a17f5dd27790a:11' => 'Apple QuickTime, Quality 749 (73%)',
    '74fddf6faaf251b28a00b4d0cd9e5621:11' => 'Apple QuickTime, Quality 750 (73%)',
    'ea65bdd87f78f7507fe6098473cbe0c9:11' => 'Apple QuickTime, Quality 751 (73%)',
    '26d087368a13e3aca9ca13a54bcc648f:11' => 'Apple QuickTime, Quality 752-753 (73-74%)',
    '5664c6ca557bc75526f59bea6aebde51:11' => 'Apple QuickTime, Quality 754 (74%)',
    'ff7a6007b6aab26f3c72c715ce487d72:11' => 'Apple QuickTime, Quality 755 (74%)',
    '912804912a3914f0b470b29495810d8c:11' => 'Apple QuickTime, Quality 756-758 (74%)',
    '6537be61d21f1b6ded3253fdd84f599b:11' => 'Apple QuickTime, Quality 759-760 (74%)',
    '72d947637340246a35ff3ee969fd613f:11' => 'Apple QuickTime, Quality 761 (74%)',
    '8c5c788bd53945222fc98f1a6155004c:11' => 'Apple QuickTime, Quality 762 (74%)',
    '9fa9c3d1041911322544aef0298695ba:11' => 'Apple QuickTime, Quality 763 (75%)',
    '33113dc71a90e8db06b43dadfe36b020:11' => 'Apple QuickTime, Quality 764 (75%)',
    '55e0cf02a898abf8e224e2e9cf6e6ca5:11' => 'Apple QuickTime, Quality 765-766 (75%)',
    '4f00127e7931d668a7b472c8a669925a:11' => 'Apple QuickTime, Quality 767-768 (75%)',
    '32aa33dc6de46b7c5c5948b0ae06cb0e:11' => 'Apple QuickTime, Quality 769 (75%)',
    'a2b6067d9e5731be8029e17c00d7e043:11' => 'Apple QuickTime, Quality 770 (75%)',
    'cb6bc96131c4a5f762b5e5f79e7c4b66:11' => 'Apple QuickTime, Quality 771 (75%)',
    'e582a5f93f66cf34facfba5918a1d9e2:11' => 'Apple QuickTime, Quality 772 (75%)',
    '54ab2e8fbd7c4ecac9eba5fb02a5ccd9:11' => 'Apple QuickTime, Quality 773 (75%)',
    'c71131cb485b59faf920d11982f7d454:11' => 'Apple QuickTime, Quality 774 (76%)',
    'c18239304e22bd19e3c8a21f9875ba39:11' => 'Apple QuickTime, Quality 775-776 (76%)',
    '8278e4f14c7bf6efd2a847ef40f392e3:11' => 'Apple QuickTime, Quality 777 (76%)',
    '6c0e396c705a59ec610a22f11466621b:11' => 'Apple QuickTime, Quality 778 (76%)',
    'a0c4d0114c04c89c879d9dc03463b347:11' => 'Apple QuickTime, Quality 779 (76%)',
    'c804929d3963c7427fa143e0d1e8c94e:11' => 'Apple QuickTime, Quality 780-781 (76%)',
    'e331a0dd2c53616c2881bb381fc4c1e2:11' => 'Apple QuickTime, Quality 782 (76%)',
    '24ec8f0996e5e8d4dd7019d2b6063290:11' => 'Apple QuickTime, Quality 783 (76%)',
    '36e9cb02d3ef245a3e15272c5071b0ee:11' => 'Apple QuickTime, Quality 784 (77%)',
    'b691578c1d6b16687fe2df12843d0642:11' => 'Apple QuickTime, Quality 785-786 (77%)',
    'ecbf939a145939d5aa48fc3c9e19cfe8:11' => 'Apple QuickTime, Quality 787 (77%)',
    '9440b11ee5eb970a3ea6de9353099761:11' => 'Apple QuickTime, Quality 788 (77%)',
    '282914c43bab6e5c62f3caaf549f1510:11' => 'Apple QuickTime, Quality 789-790 (77%)',
    '040c8ed8b19485d8677b964b03bc9929:11' => 'Apple QuickTime, Quality 791-792 (77%)',
    'd9e503dc2dd4f6493be988ecb0f44f2c:11' => 'Apple QuickTime, Quality 793-795 (77-78%)',
    '64d056c9e5e558d6c04d07d9d21aa7a3:11' => 'Apple QuickTime, Quality 796-798 (78%)',
    '498f446de17202060a4752434df1ed7b:11' => 'Apple QuickTime, Quality 799 (78%)',
    '5519e1c07692f51d0ee421ede78fb907:11' => 'Apple QuickTime, Quality 800 (78%)',
    '511c5bddc18566a2544732291920caf3:11' => 'Apple QuickTime, Quality 801 (78%)',
    '659f7466f80a8034f74a00ba07b8c3fb:11' => 'Apple QuickTime, Quality 802-804 (78-79%)',
    'fb64b35fe4021c34f16cf5bb1e59c0e8:11' => 'Apple QuickTime, Quality 805 (79%)',
    '54fcb6649f5ba51c32c68970797e41ea:11' => 'Apple QuickTime, Quality 806-807 (79%)',
    '80ef22ca4475af1bbb8963ac511144d7:11' => 'Apple QuickTime, Quality 808 (79%)',
    'ccdf1a403ec068ad21ee78686a86dd10:11' => 'Apple QuickTime, Quality 809 (79%)',
    '1487f0cfc64949393aee2eab71b6b72c:11' => 'Apple QuickTime, Quality 810 (79%)',
    'a8be3b3791e958d092b3e37286802e0c:11' => 'Apple QuickTime, Quality 811-813 (79%)',
    'f9176b3ef0b4038c6c52b30ba033eb7f:11' => 'Apple QuickTime, Quality 814 (79%)',
    '60264e35250325032bf7866ca8beaf58:11' => 'Apple QuickTime, Quality 815 (80%)',
    '5efe038d405e029badcea4c8ee2bfc88:11' => 'Apple QuickTime, Quality 816 (80%)',
    '717cbe19ae1dc9f72c86ef518aefea16:11' => 'Apple QuickTime, Quality 817 (80%)',
    '9f490145dcc00db3e57014d41d2f99f2:11' => 'Apple QuickTime, Quality 818 (80%)',
    '120b93a6ab4a1e8c78578e86e3ef837f:11' => 'Apple QuickTime, Quality 819 (80%)',
    '9d765f3947408b2c6f26163d7b072895:11' => 'Apple QuickTime, Quality 820 (80%)',
    'a9c018e06868989776a81650300bcfce:11' => 'Apple QuickTime, Quality 821 (80%)',
    '2c6581e20fa5393b3dd4d58f0df01957:11' => 'Apple QuickTime, Quality 822-824 (80%)',
    '925b6581f0ae2f288530b00168152b80:11' => 'Apple QuickTime, Quality 825 (81%)',
    'c94e09eec2df4a41b2806c23d9939cb6:11' => 'Apple QuickTime, Quality 826 (81%)',
    '68d5ce8ce1a9e337ee9630dadad0650e:11' => 'Apple QuickTime, Quality 827-829 (81%)',
    'f73704219e174961963f0fcd832b09a8:11' => 'Apple QuickTime, Quality 830-832 (81%)',
    'b783df92ec8a787b1eae4e05888b184b:11' => 'Apple QuickTime, Quality 833-834 (81%)',
    '55cd2cad99821382b1bd78355980b1d1:11' => 'Apple QuickTime, Quality 835 (82%)',
    '9ebd96ea70c2bcf4f377a175c71baf2c:11' => 'Apple QuickTime, Quality 836 (82%)',
    '12d303b2e6a467b4f20a34e695f9da7e:11' => 'Apple QuickTime, Quality 837 (82%)',
    '8711d6e1c56049c0e643bc6cf19a735c:11' => 'Apple QuickTime, Quality 838-839 (82%)',
    '88558d6e9a513ff713945bd60ed19cc7:11' => 'Apple QuickTime, Quality 840 (82%)',
    '25b45d6c1668613a61b81c6b60fa158a:11' => 'Apple QuickTime, Quality 841 (82%)',
    'b2f95d6a0eec4de39e0964a3b7303e9f:11' => 'Apple QuickTime, Quality 842 (82%)',
    '631548841b70e871ee16009737dd4b9c:11' => 'Apple QuickTime, Quality 843-844 (82%)',
    '7fac641c5b795e68ca8cfae4466a19c7:11' => 'Apple QuickTime, Quality 845 (83%)',
    'a15a74418a924874c5f3d2ca20e7af90:11' => 'Apple QuickTime, Quality 846-849 (83%)',
    'e4d76c3cd4d36b72537a2234a3597933:11' => 'Apple QuickTime, Quality 850 (83%)',
    '36f1ec430bae8e5c72af6388e2a4d807:11' => 'Apple QuickTime, Quality 851-852 (83%)',
    'b558f097ed59f547d2b370a73145caf9:11' => 'Apple QuickTime, Quality 853 (83%)',
    'ceecacb651d0e01e9b1c78cde2d7835a:11' => 'Apple QuickTime, Quality 854 (83%)',
    '82a9cce34fb9c7c3c0ab908533a9a9bf:11' => 'Apple QuickTime, Quality 855-856 (83-84%)',
    'e93244fdb14bb27a2c30ee133b3e9f5e:11' => 'Apple QuickTime, Quality 857 (84%)',
    '7a12ccd01bfdd2cf3b8ee2df498b8ae0:11' => 'Apple QuickTime, Quality 858 (84%)',
    'd0d6f781130c0fecd985df78c15c5c16:11' => 'Apple QuickTime, Quality 859-860 (84%)',
    '1c19e4fde384a33f208074c73775d990:11' => 'Apple QuickTime, Quality 861-863 (84%)',
    '8fb281fcf4d481e59e1c15ed51ef8f19:11' => 'Apple QuickTime, Quality 864 (84%)',
    'c3f615274e58eb887a2aa75acad436ff:11' => 'Apple QuickTime, Quality 865 (84%)',
    '9146c5df73615b0f2a470521bab7e4c4:11' => 'Apple QuickTime, Quality 866 (85%)',
    'ac2d99dd3ff609760c207419312e7b30:11' => 'Apple QuickTime, Quality 867 (85%)',
    '7c7d225760bce3b4e479aca8bcd2b850:11' => 'Apple QuickTime, Quality 868 (85%)',
    '9da4a7e310cb44578b009e78458d3b19:11' => 'Apple QuickTime, Quality 869 (85%)',
    '1ae782c12797f5e5c9b083099148e43a:11' => 'Apple QuickTime, Quality 870 (85%)',
    '193ace8d9e274bb9188b1a9a7bdee777:11' => 'Apple QuickTime, Quality 871-872 (85%)',
    '9160a8eeb898a05dcb02206603a45a65:11' => 'Apple QuickTime, Quality 873-877 (85-86%)',
    '0443da2c934e95fca0df8a0a1433eea4:11' => 'Apple QuickTime, Quality 878-880 (86%)',
    'fa99ff1ecc29c5caa95fa62189fca670:11' => 'Apple QuickTime, Quality 881-886 (86-87%)',
    'd9a914baea5468bb52c09d1d0b5bd131:11' => 'Apple QuickTime, Quality 887 (87%)',
    '6c0ac535ec30285b609e0b8ca18e4dc0:11' => 'Apple QuickTime, Quality 888-890 (87%)',
    'dffb0a244fa54783569693edf84d1cda:11' => 'Apple QuickTime, Quality 891-893 (87%)',
    '507b49a59dce17c02dc16fcb329352eb:11' => 'Apple QuickTime, Quality 894 (87%)',
    'dd757185a44c3d6222e9d16a0c2ee890:11' => 'Apple QuickTime, Quality 895 (87%)',
    'f3526abe33de71ad0eb728c9d446b545:11' => 'Apple QuickTime, Quality 896 (88%)',
    '34466385e7bf9b2708adf19be1eb3c2d:11' => 'Apple QuickTime, Quality 897-898 (88%)',
    'e43438348a912a2210261472d1a747ab:11' => 'Apple QuickTime, Quality 899 (88%)',
    '184d01e77f6239f63fd6ab6d36761e64:11' => 'Apple QuickTime, Quality 900-901 (88%)',
    '26005cdf9397dcce883660aeecb0426b:11' => 'Apple QuickTime, Quality 902 (88%)',
    '09321d810a54503da7ad7b8e883227ea:11' => 'Apple QuickTime, Quality 903 (88%)',
    'd6438ea5a93b6d4d0ba26de7c56f2634:11' => 'Apple QuickTime, Quality 904-905 (88%)',
    'ddc90333cb2279aa533a339710bd81ef:11' => 'Apple QuickTime, Quality 906 (88%)',
    '14a352b80a350e2df6b2bc444ccc74f6:11' => 'Apple QuickTime, Quality 907-908 (89%)',
    '91a7f0747ed633f481918a83b1a7c77c:11' => 'Apple QuickTime, Quality 909-914 (89%)',
    'a3bb9728b3dfa7002364659db0c420fc:11' => 'Apple QuickTime, Quality 915-917 (89-90%)',
    '9df94f927ffc1f3345923232691fab3b:11' => 'Apple QuickTime, Quality 918-920 (90%)',
    '259764d950c1f1e399cdb27e159c8985:11' => 'Apple QuickTime, Quality 921-922 (90%)',
    '73313d816524749545292ed2284c804c:11' => 'Apple QuickTime, Quality 923-924 (90%)',
    '449ed4370a849e0f736b57ee7ccab942:11' => 'Apple QuickTime, Quality 925-926 (90%)',
    'bd1f81bd50cf9859eb0ae6d2dbf75d09:11' => 'Apple QuickTime, Quality 927 (91%)',
    '820db98d1ee91bb648e7a05498a340b0:11' => 'Apple QuickTime, Quality 928 (91%)',
    '93c1ba0af1f50d889cb5e2364be3a056:11' => 'Apple QuickTime, Quality 929 (91%)',
    'cc96e1c8906353c4023bc7e6b72bb684:11' => 'Apple QuickTime, Quality 930 (91%)',
    'eb728dc3105ddb5a0597384b54ed948c:11' => 'Apple QuickTime, Quality 931 (91%)',
    'dc0278d78fa5c8daf84f8c2672f582c6:11' => 'Apple QuickTime, Quality 932-933 (91%)',
    '1a603cba63dbba0d0815fc7319271b93:11' => 'Apple QuickTime, Quality 934-936 (91%)',
    '3d8f4957eab9756993a78efe08ba3798:11' => 'Apple QuickTime, Quality 937-938 (92%)',
    'cef5a49e1834c316a4a9e7dca8d69157:11' => 'Apple QuickTime, Quality 939 (92%)',
    'ca21386b17f3866a235fca4b6e72b124:11' => 'Apple QuickTime, Quality 940 (92%)',
    'ec662935c494c5abd6f6c907f77be65c:11' => 'Apple QuickTime, Quality 941-943 (92%)',
    '190a54eb1127ee87231795bc3d661b5a:11' => 'Apple QuickTime, Quality 944-946 (92%)',
    'e78229129afca214a07ad978262c545e:11' => 'Apple QuickTime, Quality 947-951 (92-93%)',
    'c31cb63954b137b62c5fe35379235e2e:11' => 'Apple QuickTime, Quality 952-954 (93%)',
    '25ca43baa972ad9c82128606cd383805:11' => 'Apple QuickTime, Quality 955 (93%)',
    '334c67d739adf957d1620201cb7a521c:11' => 'Apple QuickTime, Quality 956-957 (93-93%)',
    '29c42d951dc84d62bd134bec71bf731b:11' => 'Apple QuickTime, Quality 958 (94%)',
    'fa6f80883480ab3ddea8ee2f6796a14b:11' => 'Apple QuickTime, Quality 959-961 (94%)',
    'e40f6a9a3daf4bfc42aedcb9107076ea:11' => 'Apple QuickTime, Quality 962 (94%)',
    'd26a06f6114d83714e3b64b6dbe97e6f:11' => 'Apple QuickTime, Quality 963-967 (94%)',
    'c40848be9d1d1018747630cdac2d7294:11' => 'Apple QuickTime, Quality 968-970 (95%)',
    'ab2b6d0624f294bf4e53e34208caeaa6:11' => 'Apple QuickTime, Quality 971 (95%)',
    'b6ea2f838fa1942a21e41f6bba417782:11' => 'Apple QuickTime, Quality 972-973 (95%)',
    'ff765e75c06c9db34f66ae7ee0202d05:11' => 'Apple QuickTime, Quality 974 (95%)',
    'c33573208ef877f1bc9a64f595e00c4d:11' => 'Apple QuickTime, Quality 975-977 (95%)',
    '6528da6df208ce35fd84dccabc81d4da:11' => 'Apple QuickTime, Quality 978-979 (96%)',
    '420d094d00a4a8aec94c5667254d2053:11' => 'Apple QuickTime, Quality 980-984 (96%)',
    '427271dd1e2a3d7a7ce54af73d9e6c77:11' => 'Apple QuickTime, Quality 985-987 (96%)',
    '99589fb7d66a29f15d9ff0f37235e7a2:11' => 'Apple QuickTime, Quality 988-991 (96-97%)',
    '98fddd9e5862e06385b46a016597c02f:11' => 'Apple QuickTime, Quality 992-993 (97%)',
    '381d1ca1d61986f28cfd6da0fca348da:11' => 'Apple QuickTime, Quality 994-996 (97%)',
    '11a232fa9de634fadde1869007baab9c:11' => 'Apple QuickTime, Quality 997-998 (97%)',
    '02374d6405239e8fe5ab939b92f4fd03:11' => 'Apple QuickTime, Quality 999-1000 (98%)',
    'f91cfb708c99c2fef0f7148976514f44:11' => 'Apple QuickTime, Quality 1001-1002 (98%)',
    '83f3452abc7906930b228c29a4320089:11' => 'Apple QuickTime, Quality 1003 (98%)',
    '4379016ba81dc331ffd5f9a8ab5b6399:11' => 'Apple QuickTime, Quality 1004 (98%)',
    'e2cccecebc01c7d8a4fca2dab682ba8f:11' => 'Apple QuickTime, Quality 1005-1009 (98-99%)',
    '8d4b2a14a176d63e509684aa4246dabb:11' => 'Apple QuickTime, Quality 1010-1013 (99%)',
    '40a7c1d8f58612f4470c2531768d93b5:11' => 'Apple QuickTime, Quality 1014-1016 (99%)',
    '8392742f8f5971ed08c7520d0e9c81f3:11' => 'Apple QuickTime, Quality 1017 (99%)',
    '14040c7711b6fa62383da3edc9ade1b7:11' => 'Apple QuickTime, Quality 1018-1020 (99-100%)',
    '22ccf9c976b36da34f48385a09b1951b:11' => 'Apple QuickTime, Quality 1021-1023 (100%)',
    '4aa632db3be6b6e85565c1fe66cb22d1:11' => 'Apple QuickTime, Quality 1024 (lossless)',

    # Apple Aperture 2.1.3 (ref 2)
    '60cb2afa0cfa7395635a9360fc690b46:221111' => 'Apple Aperture Quality 0',
    '6b9be09d6ec6491a20c2827dbeb678c0:221111' => 'Apple Aperture Quality 1',
    'dbb17a02e661f2475411fc1dc37902ef:221111' => 'Apple Aperture Quality 2',
    '8a5df2b5337bf8251c3f66f6adbb5262:221111' => 'Apple Aperture Quality 3',
    '3841f0f3be30520a1a57f41c449588ee:221111' => 'Apple Aperture Quality 4',
    '2b1dba266c728a9f46d06e6e5c247953:221111' => 'Apple Aperture Quality 5',
  # 'Independent JPEG Group library (used by many applications), Quality 91' => 'Apple Aperture Quality 6',
    '93818f3a0e6d491500cb62e1f683da22:221111' => 'Apple Aperture Quality 7',
    '8c0c36696a99fd889e0f0c7d64824f3c:221111' => 'Apple Aperture Quality 8',
    '043645382c79035b6f2afc62d373a37f:221111' => 'Apple Aperture Quality 9',
    '558d017ce6d5b5282ce76727fe99b91e:221111' => 'Apple Aperture Quality 10',
    '0b52b82694040193aee10e8074cd7ad5:221111' => 'Apple Aperture Quality 11',
  # 'Independent JPEG Group library (used by many applications), Quality 100' => 'Apple Aperture Quality 12',

    # Tested with Corel Paint Shop Pro PHOTO X2 (Win) - Different subsamplings possible
    '1c78c0daaa0bbfd4a1678b5569b0fa13' => 'Corel Paint Shop Pro PHOTO, Quality 1',
    '5ffdd2e918ec293efc79083703737290' => 'Corel Paint Shop Pro PHOTO, Quality 2',
    '4ed4751d772933938600c4e7560bf19c' => 'Corel Paint Shop Pro PHOTO, Quality 3',
    'f647f0fb4320c61f52e2a79d12bbc8cc' => 'Corel Paint Shop Pro PHOTO, Quality 4',
    '6194167174dfcb4a769cf26f5c7a018d' => 'Corel Paint Shop Pro PHOTO, Quality 5',
    '6120ded86d4cc42cd7ca2131b1f51fad' => 'Corel Paint Shop Pro PHOTO, Quality 6',
    'c07a6430e56ef16a0526673398e87ac6' => 'Corel Paint Shop Pro PHOTO, Quality 7',
    '507cc511e561916efa3b49228ffc8c9a' => 'Corel Paint Shop Pro PHOTO, Quality 8',
    '612941a50f2c0992938bc13106caf228' => 'Corel Paint Shop Pro PHOTO, Quality 9',
    '7624f08396d811fdb6f1ead575e67e58' => 'Corel Paint Shop Pro PHOTO, Quality 10',
    'e215df38e258b3d8bceb57aa64388d26' => 'Corel Paint Shop Pro PHOTO, Quality 11',
    '78f66ee0bc442950808e25daa02a2b02' => 'Corel Paint Shop Pro PHOTO, Quality 12',
    '14efb0bb5124910a37bcbd5f06de9aa9' => 'Corel Paint Shop Pro PHOTO, Quality 13',
    'd61168238621bd221ef1eb3dcbe270a3' => 'Corel Paint Shop Pro PHOTO, Quality 14',
    'e2d2755891b4e4bc5f7c8d76dcbb0d53' => 'Corel Paint Shop Pro PHOTO, Quality 15',
    'f6c4502144a2e5c82c07994d3cd01665' => 'Corel Paint Shop Pro PHOTO, Quality 16',
    '78801638505e95827c2f7cc0c7ef78f4' => 'Corel Paint Shop Pro PHOTO, Quality 17',
    'e8ff3d165b4c028c18ec8a8f940a12a1' => 'Corel Paint Shop Pro PHOTO, Quality 18',
    '984c359b9fbcc4d6f805946aa23ae708' => 'Corel Paint Shop Pro PHOTO, Quality 19',
    'd1dc48d911055bc533779d6e086f7242' => 'Corel Paint Shop Pro PHOTO, Quality 20',
    'd7437a18e86ac2832d73204acd82aa89' => 'Corel Paint Shop Pro PHOTO, Quality 21',
    'bceaee6c1a150006b3643de6942ccfa3' => 'Corel Paint Shop Pro PHOTO, Quality 22',
    'c448e6817efa9acdad225e60ed0013f9' => 'Corel Paint Shop Pro PHOTO, Quality 23',
    '904f231c98f390400ba7ae17c252813f' => 'Corel Paint Shop Pro PHOTO, Quality 24',
    'ccd6708ca1dbd66a23d40cee635a0f76' => 'Corel Paint Shop Pro PHOTO, Quality 25',
    '10d87624d888b75b29e156be8dad35f4' => 'Corel Paint Shop Pro PHOTO, Quality 26',
    '8558c6d41f03db192198dceefbd1e89b' => 'Corel Paint Shop Pro PHOTO, Quality 27',
    '058fc759cff9d615f91d9ffb4b46436a' => 'Corel Paint Shop Pro PHOTO, Quality 28',
    '5c606e0f7168a78fd8d0c91646c801a3' => 'Corel Paint Shop Pro PHOTO, Quality 29',
    'e9555e593a6fd9aeee399de16080cd61' => 'Corel Paint Shop Pro PHOTO, Quality 30',
    '2c2726484978a15d3d756d43b0baa290' => 'Corel Paint Shop Pro PHOTO, Quality 31',
    '8b1d11d31bc9445278cf9af55b0c156b' => 'Corel Paint Shop Pro PHOTO, Quality 32',
    'aa4a5528ae18ecd36ec052014b91f651' => 'Corel Paint Shop Pro PHOTO, Quality 33',
    '9a26194b114b7db253601ff80b03da9a' => 'Corel Paint Shop Pro PHOTO, Quality 34',
    '3fa780a3dff1d787f7d883585a46dcfb' => 'Corel Paint Shop Pro PHOTO, Quality 35',
    '0a899361ed0d51e224dc535ceb02f9a1' => 'Corel Paint Shop Pro PHOTO, Quality 36',
    '3a2ab96a6ad9612e1377ddc822f02ddd' => 'Corel Paint Shop Pro PHOTO, Quality 37',
    '315f4faadd967e72d730155091c4912f' => 'Corel Paint Shop Pro PHOTO, Quality 38',
    '5f6e3a66672d6e4c41b1689996ca57d3' => 'Corel Paint Shop Pro PHOTO, Quality 39',
    '9503a86793e86d1fca3d8797548fa243' => 'Corel Paint Shop Pro PHOTO, Quality 40',
    '3b95f11bd77cb8af977c09d5851131f8' => 'Corel Paint Shop Pro PHOTO, Quality 41',
    'ececf8dfa473110534b506db58d98f15' => 'Corel Paint Shop Pro PHOTO, Quality 42',
    'cfe3144d4f8048a0507269a9d8a85993' => 'Corel Paint Shop Pro PHOTO, Quality 43',
    'eb9d48d135b2c61c51fc3f23b0001b4d' => 'Corel Paint Shop Pro PHOTO, Quality 44',
    'b08313a6919d308e50b806f138a8a2a1' => 'Corel Paint Shop Pro PHOTO, Quality 45',
    '7c34e6e7fe2cc760fa5c3ed812a8b74c' => 'Corel Paint Shop Pro PHOTO, Quality 46',
    '90ece7123e8d614d9aab55eaba6dd7da' => 'Corel Paint Shop Pro PHOTO, Quality 47',
    '6d79fe623c4c5320bdbe4d3026f4e71a' => 'Corel Paint Shop Pro PHOTO, Quality 48',
    'a7e85552c3e5e40288891d225f308590' => 'Corel Paint Shop Pro PHOTO, Quality 49',
    '67b9a678d9f669167c5b4bf12422ad50' => 'Corel Paint Shop Pro PHOTO, Quality 50',
    '1fab112b17e94f53e94a9208e9091b7b' => 'Corel Paint Shop Pro PHOTO, Quality 51',
    '4971237e046795a030a99a0e8d2c5acb' => 'Corel Paint Shop Pro PHOTO, Quality 52',
    'f3e1672b93ff159231c51b1b157e45fd' => 'Corel Paint Shop Pro PHOTO, Quality 53',
    '6e9cfb8131373c3d1873e3f497e46b64' => 'Corel Paint Shop Pro PHOTO, Quality 54',
    '9155c8acf8322e8af898272c694fa1d6' => 'Corel Paint Shop Pro PHOTO, Quality 55',
    '52b20edc779f206f2aed50610971f181' => 'Corel Paint Shop Pro PHOTO, Quality 56',
    'ad801813f822ef9774801ab4d9145a61' => 'Corel Paint Shop Pro PHOTO, Quality 57',
    '07259679e2a842478df97c7f0ddd4df3' => 'Corel Paint Shop Pro PHOTO, Quality 58',
    '67db25c57803c34b065736f46f6afadb' => 'Corel Paint Shop Pro PHOTO, Quality 59',
    'c7498fc4b3802b290a452631dd1e1b63' => 'Corel Paint Shop Pro PHOTO, Quality 60',
    '3f7b04c7952f96d2624813ed9896f128' => 'Corel Paint Shop Pro PHOTO, Quality 61',
    'd5ec901d20f3887007d0f4cfb7d1460d' => 'Corel Paint Shop Pro PHOTO, Quality 62',
    '61bb38e23040b6a8b0e8721e6d6eff66' => 'Corel Paint Shop Pro PHOTO, Quality 63',
    '48fac53d9d168eab3ce9b6edc4b9fcb1' => 'Corel Paint Shop Pro PHOTO, Quality 64',
    '8cb101a5ae986e45cc31a9e19a35535d' => 'Corel Paint Shop Pro PHOTO, Quality 65',
    '0e08dc629e883530cb2ae78c90f125b3' => 'Corel Paint Shop Pro PHOTO, Quality 66',
    '5134762d2d4baac8711a52e76730591c' => 'Corel Paint Shop Pro PHOTO, Quality 67',
    '14b57dc6d5381fd0a743c7bd8b28bed1' => 'Corel Paint Shop Pro PHOTO, Quality 68',
    '9d398f1b1f40b7aaec1bd9cdb6922530' => 'Corel Paint Shop Pro PHOTO, Quality 69',
    'c7e68d88bee5c2ee4b61a11bc2e68c80' => 'Corel Paint Shop Pro PHOTO, Quality 70',
    '917fe67f6ded5decac1820642239622c' => 'Corel Paint Shop Pro PHOTO, Quality 71',
    '362c3e0c08f6951018cde7b412cd513f' => 'Corel Paint Shop Pro PHOTO, Quality 72',
    'd91cd4a2dcd1a29e6ef652ebcfdd58d7' => 'Corel Paint Shop Pro PHOTO, Quality 73',
    '11f5fbd5e74e5c5e305b95dbbc4356a8' => 'Corel Paint Shop Pro PHOTO, Quality 74',
    'bf010771f909049fc5fceedcaa0f917c' => 'Corel Paint Shop Pro PHOTO, Quality 75',
    'a455a3149812ba6951a016ee6114f9da' => 'Corel Paint Shop Pro PHOTO, Quality 76',
    '42e0c4082ec4d026c77d19a053a983f4' => 'Corel Paint Shop Pro PHOTO, Quality 77',
    '326bd5938e2db7de9250a9fb0efc6692' => 'Corel Paint Shop Pro PHOTO, Quality 78',
    'a3e2cc4ea95cda49501bc73c494e9420' => 'Corel Paint Shop Pro PHOTO, Quality 79',
    '8c89043f00678bb5c68ee90390c1b43b' => 'Corel Paint Shop Pro PHOTO, Quality 80',
    'fc5812ad9a4cd0122eb1c63f0ac3b5a3' => 'Corel Paint Shop Pro PHOTO, Quality 81',
    '84dbe33962674aab86e03681ac3bd35f' => 'Corel Paint Shop Pro PHOTO, Quality 82',
    'b6b80a78472dca05c9135702e96fdad9' => 'Corel Paint Shop Pro PHOTO, Quality 83',
    '01f997907a4c1dfd1e6b00aca9ff5d80' => 'Corel Paint Shop Pro PHOTO, Quality 84',
    '8431e86434062b325c519fd836353cd0' => 'Corel Paint Shop Pro PHOTO, Quality 85',
    '15f375a620952738ff21ff4aa496b8f7' => 'Corel Paint Shop Pro PHOTO, Quality 86',
    '7b0f02aa96271376d3f81658d98fb1df' => 'Corel Paint Shop Pro PHOTO, Quality 87',
    '86e7666b05bd1fc130fbf4b48f854288' => 'Corel Paint Shop Pro PHOTO, Quality 88',
    '6af05d547e8911fe2d1f2b4d968a477e' => 'Corel Paint Shop Pro PHOTO, Quality 89',
    '8baa876790518bf509dd09093759331d' => 'Corel Paint Shop Pro PHOTO, Quality 90',
    'eb7d90d291044d1bd8f40ca1b3ce0ddf' => 'Corel Paint Shop Pro PHOTO, Quality 91',
    '6f338385a8f2cd2dd3420a4f6138a206' => 'Corel Paint Shop Pro PHOTO, Quality 92',
    'b0a0fd1ec2dd366ad00d3e83d6dedec2' => 'Corel Paint Shop Pro PHOTO, Quality 93',
    'e09026128c9880b44ac71224f477cd3b' => 'Corel Paint Shop Pro PHOTO, Quality 94',
    'd0a8f50ff547da69a57eeb892e194cff' => 'Corel Paint Shop Pro PHOTO, Quality 95',
    '7849ba902d96273b5ac7b6eb98f4d009' => 'Corel Paint Shop Pro PHOTO, Quality 96',
    '379f9f196d4190298a732ab9a7031001' => 'Corel Paint Shop Pro PHOTO, Quality 97',
    'c3d1601f84ec3adfbc8ca17883ef6378' => 'Corel Paint Shop Pro PHOTO, Quality 98',
    '1f5e87bec674bdd7dff166c2ea9ca004' => 'Corel Paint Shop Pro PHOTO, Quality 99',

    # Tested with FixFoto Version 2.90 Build 136 (Win) - different subsamplings possible
    '866dd04cb0fe2e00cda7395162480117' => 'FixFoto, Quality 0 or 1',
    '1e400ba25fa835e2771772bbfb15b94b' => 'FixFoto, Quality 2',
    '302ff1ad1a50d0f01a82cc88f286c649' => 'FixFoto, Quality 3',
    '1343a117f5fab26d556a3e7558366591' => 'FixFoto, Quality 4',
    '8fbb8cc5368224625689df80bf4d2a04' => 'FixFoto, Quality 5',
    'a371d1ffc8d85d502854a356f3b0ea74' => 'FixFoto, Quality 6',
    'a9a0a5000cd6fb322960a4c45cf1d032' => 'FixFoto, Quality 7',
    'aaac84043224d33e1d3a1723b653b0cd' => 'FixFoto, Quality 8',
    '701e4820f6d0b68e67b6a2b90a7baa0c' => 'FixFoto, Quality 9',
    '877d03a5abf5b6c4ad03c39afd97f4a2' => 'FixFoto, Quality 10',
    'b3d9bdc2090200537fb42f4d69631150' => 'FixFoto, Quality 11',
    '3cf156d54120b53057f56e9f38ee2896' => 'FixFoto, Quality 12',
    '69fe5c29b9d5e4c823f8a082ab7b3285' => 'FixFoto, Quality 13',
    'cf8573af40ced1793dcbc2346f969240' => 'FixFoto, Quality 14',
    '22944c3bc03d6adea8d6819f914452c3' => 'FixFoto, Quality 15',
    'd768df38fb51c4b9977e5d7185f97a6c' => 'FixFoto, Quality 16',
    '7ef2cd2b66d51fe80d94d5db427ee9ef' => 'FixFoto, Quality 17',
    'ed3d3b9ff9faf0009e44b9803f6295d7' => 'FixFoto, Quality 18',
    '70a0b15e2e5f97e0a9333a2011afe5cd' => 'FixFoto, Quality 19',
    'd798b707a6b83eb54664abe0833b46aa' => 'FixFoto, Quality 20',
    'bf68d1866b75cea8f99cf2fc46f9d686' => 'FixFoto, Quality 21',
    'b98b8adb8f1f78b65800efe6c329ceab' => 'FixFoto, Quality 22',
    'c063344185079018af9fcf161a3fdf98' => 'FixFoto, Quality 23',

    # Tested with Nikon Capture NX Version 2.0.0 (Win)
    '0ef9d9f62ab68807eedf6cb8c2ec120b:221111' => 'Nikon Capture NX, Quality 0',
    'efbc50df45bc1d1fbbbd29c3e5de04b2:221111' => 'Nikon Capture NX, Quality 1',
    'cbde745c78fd546d6e83dd7512ebe863:221111' => 'Nikon Capture NX, Quality 2',
    '33731f743fc28e9d81e542f0ed7cdfba:221111' => 'Nikon Capture NX, Quality 3',
    '866fcb1296d7da02b4ad31afb242f25f:221111' => 'Nikon Capture NX, Quality 4',
    'cfbe44397240092d3a67241a23342528:221111' => 'Nikon Capture NX, Quality 5',
    'a4b8b3408ae302ae81f125e972901131:221111' => 'Nikon Capture NX, Quality 6',
    '3a6cefd4f43c513fdf0858f26afeab5a:221111' => 'Nikon Capture NX, Quality 7',
    '1e861ce223babf95bc795e18cbdb49d1:221111' => 'Nikon Capture NX, Quality 8',
    '4d5b512d8bc173f14e6a3cf8574f670a:221111' => 'Nikon Capture NX, Quality 9',
    '9b1e6d379d3030dfa313bcaedc1ef3c7:221111' => 'Nikon Capture NX, Quality 10',
    'e39b60fcecf3221d14c62dc13ddf4726:221111' => 'Nikon Capture NX, Quality 11',
    '3654bbf4a45e0c0758a82a075b3f77cc:221111' => 'Nikon Capture NX, Quality 12',
    '4f5889173779409ec604622a1894ab4a:221111' => 'Nikon Capture NX, Quality 13',
    '738685b86b80ff0e8b562102d1b58f71:221111' => 'Nikon Capture NX, Quality 14',
    '48a53035374c08e6490893d8113ed6b3:221111' => 'Nikon Capture NX, Quality 15',
    '03651ac1d15043f77949a63ac3762584:221111' => 'Nikon Capture NX, Quality 16',
    '27811b28d02bd417857904f0a9e1ed58:221111' => 'Nikon Capture NX, Quality 17',
    '03201bd5642a451d99b99bfd10fc42df:221111' => 'Nikon Capture NX, Quality 18',
    '67d5eb5f55c9a5baa0a67d42a841d77b:221111' => 'Nikon Capture NX, Quality 19',
    '18392b08bf8cf788a579f376297c3334:221111' => 'Nikon Capture NX, Quality 20',
    'de0c784b75953851dc370f4daecfa1a9:221111' => 'Nikon Capture NX, Quality 21',
    '75f260644b87a9779188126da8709e7f:221111' => 'Nikon Capture NX, Quality 22',
    'c44701e8185306f5e6d09be16a2b0fbd:221111' => 'Nikon Capture NX, Quality 23',
    '086e5ce1149e14efd9e424956734fe05:221111' => 'Nikon Capture NX, Quality 24',
    'aad1109d9c49b8170feac125148b2a50:221111' => 'Nikon Capture NX, Quality 25',
    'c97965ce5392623f668a386b30e41cee:221111' => 'Nikon Capture NX, Quality 26',
    'd9dadfb6f0a25765abe00e69857c5520:221111' => 'Nikon Capture NX, Quality 27',
    '0ee9ca02a1fe8a17b6e50a2e86d19a7c:221111' => 'Nikon Capture NX, Quality 28',
    '88b1726a20759f29eecfa2b129773127:221111' => 'Nikon Capture NX, Quality 29',
    '70a311935ed066da954897fad5079377:221111' => 'Nikon Capture NX, Quality 30',
    'aa2d374bbab2a30e00c1863264588a42:221111' => 'Nikon Capture NX, Quality 31',
    '097b684846696b3a8bbdf2bd2f9ded9c:221111' => 'Nikon Capture NX, Quality 32',
    'bb313d5398065376c7765092fc8ea0f0:221111' => 'Nikon Capture NX, Quality 33',
    'aa049fdc1387851a664143df0408f55c:221111' => 'Nikon Capture NX, Quality 34',
    '087c1c1a368adc82900d83235f432d3f:221111' => 'Nikon Capture NX, Quality 35',
    '7dec6568dbad7a70622c994a326957e2:221111' => 'Nikon Capture NX, Quality 36',
    'd2e14d8ba7d38f7544b569eea7221255:221111' => 'Nikon Capture NX, Quality 37',
    'ce6bcb98c5f9358594f5934e64b4ecc3:221111' => 'Nikon Capture NX, Quality 38',
    '4785aafc8471873402819e423b8969a9:221111' => 'Nikon Capture NX, Quality 39',
    '66ae78a749b520b35d4daf4531df8ae5:221111' => 'Nikon Capture NX, Quality 40',
    '946d9f9346a0c65eec478945ad3d6143:221111' => 'Nikon Capture NX, Quality 41',
    'f46e96afa026233c1662c9114feb61e9:221111' => 'Nikon Capture NX, Quality 42',
    '96a267e050b6d8a13439f8a9bb89722c:221111' => 'Nikon Capture NX, Quality 43',
    '27c301566e155f700b01906a43473ffe:221111' => 'Nikon Capture NX, Quality 44',
    'ceff136f6dd88242500bfd639cb0c003:221111' => 'Nikon Capture NX, Quality 45',
    '939b804eefc95158a934bb48e3f3b545:221111' => 'Nikon Capture NX, Quality 46',
    '06186292fe0ccaaeb5999319a366c4b4:221111' => 'Nikon Capture NX, Quality 47',
    'e456c998dc126c1efad013eb7b0186c1:221111' => 'Nikon Capture NX, Quality 48',
    'ef0cd1902fb1afe284468a67eaffd078:221111' => 'Nikon Capture NX, Quality 49',
    'f4693035f8db19e0788f41255c3c052e:221111' => 'Nikon Capture NX, Quality 50',
    '40c6f2886cdca8f19a654ce321ea993e:221111' => 'Nikon Capture NX, Quality 51',
    'e9387b4065bba8570375d6535ab2124b:221111' => 'Nikon Capture NX, Quality 52',
    'f3a55e422a4ab829b2c1f5a1784ce9f6:221111' => 'Nikon Capture NX, Quality 53',
    '2fff3c6e48247992d1543d9e5c679759:221111' => 'Nikon Capture NX, Quality 54',
    '5a1849b49122ff09949f1d355b4f9eaa:221111' => 'Nikon Capture NX, Quality 55',
    'a582968bb1890620ffbae916ebafcb64:221111' => 'Nikon Capture NX, Quality 56',
    '81597eb992e32e186d2b5565bbe4ae3a:221111' => 'Nikon Capture NX, Quality 57',
    '7364416ce4f2a9282efdbe052574527b:221111' => 'Nikon Capture NX, Quality 58',
    '5301c2bcae09fd4305e47ffc56b2c8a7:221111' => 'Nikon Capture NX, Quality 59',
    '5a1849b49122ff09949f1d355b4f9eaa:211111' => 'Nikon Capture NX, Quality 60',
    '9be2446f168941ff42d9fc7441f2429b:211111' => 'Nikon Capture NX, Quality 61',
    'bbba80e58afae43278e287021d4f1499:211111' => 'Nikon Capture NX, Quality 62',
    '2a9ae394dc32a418960522cbe9c6df24:211111' => 'Nikon Capture NX, Quality 63',
    '67fbe0dce139b6db1813e30bbbceccf3:211111' => 'Nikon Capture NX, Quality 64',
    '17bce376f588ebf2b3e9002a337c239d:211111' => 'Nikon Capture NX, Quality 65',
    'cd2c6c01d8eb8d985086b54e2269278a:211111' => 'Nikon Capture NX, Quality 66',
    '34b25782fc089616807bbbe7f7cd8413:211111' => 'Nikon Capture NX, Quality 67',
    '37b8bbab382a228eabb0dc64c0edcb0f:211111' => 'Nikon Capture NX, Quality 68',
    'b163f35baed567d70aa2536695558724:211111' => 'Nikon Capture NX, Quality 69',
    '251eb2d7903f63b168348ec483ba499a:211111' => 'Nikon Capture NX, Quality 70',
    '42e7cdf33b9067a7124dd27020704f9a:211111' => 'Nikon Capture NX, Quality 71',
    '032678d9de74e5530896c28079f666af:211111' => 'Nikon Capture NX, Quality 72',
    '30d7b6db02954dfc4ce47a089d0f40d9:211111' => 'Nikon Capture NX, Quality 73',
    '5c1a40094128ac76eab0405dcb4ae3c7:211111' => 'Nikon Capture NX, Quality 74',
    '2706b8b0cf6686148e285b6d3e44dd72:211111' => 'Nikon Capture NX, Quality 75',
    '6ca4a27cb36f35ab84b0e2df06bb32f4:211111' => 'Nikon Capture NX, Quality 76',
    '6f9cae52d3f47f514f7c927314455a5a:211111' => 'Nikon Capture NX, Quality 77',
    'c0204862b8aafa2c286c7b58d755c31f:211111' => 'Nikon Capture NX, Quality 78',
    'd8ef40736b072f09bead5f73f5ec1372:211111' => 'Nikon Capture NX, Quality 79',
    '8c389c29eca238b3b331f65f7e124a27:111111' => 'Nikon Capture NX, Quality 80',
    '6f9cae52d3f47f514f7c927314455a5a:111111' => 'Nikon Capture NX, Quality 81',
    '8e1ceace8fafe31282393d8677e76994:111111' => 'Nikon Capture NX, Quality 82',
    '60f75a915647ed50d1724179d50a35d2:111111' => 'Nikon Capture NX, Quality 83',
    'df54eb20ec90f41f1e6c37e241ee381c:111111' => 'Nikon Capture NX, Quality 84',
    '5522213c915e2af3ad01ee2ec27ee3ed:111111' => 'Nikon Capture NX, Quality 85',
    '08c063f0997262d9977df4b44e682d82:111111' => 'Nikon Capture NX, Quality 86',
    'd2e34c70872ac119dda6bdeeb36bf229:111111' => 'Nikon Capture NX, Quality 87',
    'e5abf48ce0cc2b4a3db7eca3a1112b7a:111111' => 'Nikon Capture NX, Quality 88',
    'b69dcb672088f296323d891219464ad8:111111' => 'Nikon Capture NX, Quality 89',
    'b6d1c6efb27ea721577888b5f981ad7b:111111' => 'Nikon Capture NX, Quality 90',
    'b023f424f81c8cbbab20119c06163dce:111111' => 'Nikon Capture NX, Quality 91',
    '77f680490d08697cb0f11ff3fe76b7e8:111111' => 'Nikon Capture NX, Quality 92',
    '1860106097672532e7ebc2026d7f9681:111111' => 'Nikon Capture NX, Quality 93',
    '0c7d4861b3bee5d766a93f2d34027bfa:111111' => 'Nikon Capture NX, Quality 94',
    '3adf9a0b85a4000243bbf833cd8e6966:111111' => 'Nikon Capture NX, Quality 95',
    '9530dfffc5574606841a597212ec25b4:111111' => 'Nikon Capture NX, Quality 96',
    'c7294290fe26155147072f9041705cfb:111111' => 'Nikon Capture NX, Quality 97',
    'c24c44a4dadd77c15e0b4c741a2d4bd5:111111' => 'Nikon Capture NX, Quality 98',
    '36016cd5527c505ef3bbba8b3e22f9db:111111' => 'Nikon Capture NX, Quality 99',
    'c9309ab058680151be5f97e6c54dc687:111111' => 'Nikon Capture NX, Quality 100',

    # Tested with ACDsee PhotoEditor 4.0.208c-de (Win)
    '2ab2f6a116ca6fc0bbf188b19b9de967' => 'ACD Systems Digital Imaging, Quality 0 or 1',
    'f4f9d5c07c78e8700a6f3def0782a18e' => 'ACD Systems Digital Imaging, Quality 2',
    '66fc410ab8f71a7fdef86fd70b742dc1' => 'ACD Systems Digital Imaging, Quality 3',
    '8e763b5b9255df1f4cb7b9732e99c210' => 'ACD Systems Digital Imaging, Quality 4',
    'fd3eed19f6667ab0bedfa3263390ce25' => 'ACD Systems Digital Imaging, Quality 5 or 6',
    'dc0dc92085037072e27247f64af0f22d' => 'ACD Systems Digital Imaging, Quality 7',
    '233ed690eb7e9008c20ed16e79aa3eb5' => 'ACD Systems Digital Imaging, Quality 8',
    '684649f6c1590f5a912a827a6d8bfc6b' => 'ACD Systems Digital Imaging, Quality 9',
    'ed6aec096e8776b483b2c2b3d7e15d76' => 'ACD Systems Digital Imaging, Quality 10 or 11',
    '9cd85933ddb1101d9b859a19e9a30334' => 'ACD Systems Digital Imaging, Quality 12',
    '222a8769205a592ec834b6f5fc654a21' => 'ACD Systems Digital Imaging, Quality 13',
    '29f957e2a0af0f44d271c3c4e27eec4b' => 'ACD Systems Digital Imaging, Quality 14',
    'c46c764191f9c3db2bfe8d134512bcd8' => 'ACD Systems Digital Imaging, Quality 15 or 16',
    '56caa684ce7eb0b1cf662e1c88ed1614' => 'ACD Systems Digital Imaging, Quality 17',
    'cedc5208c6e1cbffd8be0e47bfd76698' => 'ACD Systems Digital Imaging, Quality 18',
    'dec0717305bae8309a934e1d6a251d88' => 'ACD Systems Digital Imaging, Quality 19',
    '8c85e0e8f41257e2cd739a5b158ec218' => 'ACD Systems Digital Imaging, Quality 20 or 21',
    '6ae7ab4e6d5e0e67006cca59c70f843c' => 'ACD Systems Digital Imaging, Quality 22',
    '840be626ed18db6cdef3c5c357e24d34' => 'ACD Systems Digital Imaging, Quality 23',
    'd48c2b9e514e25fcc4b3f2408d168d72' => 'ACD Systems Digital Imaging, Quality 24',
    'b9eb63b89c80c71f4eac8c6e27d272f1' => 'ACD Systems Digital Imaging, Quality 25 or 26',
    'bcd4d36a9db91a51d1a571f71f8230d4' => 'ACD Systems Digital Imaging, Quality 27',
    'ac2f66ab2559019fcf021b9a32b049ab' => 'ACD Systems Digital Imaging, Quality 28',
    '4208fca702ec702bd5d41c8231883057' => 'ACD Systems Digital Imaging, Quality 29',
    'fa620c67ab09a4c0d1c5b8e65ade361e' => 'ACD Systems Digital Imaging, Quality 30 or 31',
    '679dea81c8d4563e07efac4fab6b89ca' => 'ACD Systems Digital Imaging, Quality 32',
    '43ceb0c1a5d94d55ee20dc3a168498b2' => 'ACD Systems Digital Imaging, Quality 33',
    'a9cc8a19ae25bc024c3d92d84c13c7a5' => 'ACD Systems Digital Imaging, Quality 34',
    'e3e7280c8a9e82d31e22d24d5b733580' => 'ACD Systems Digital Imaging, Quality 35 or 36',
    'a06d250213e349005897bd6fa5bebca8' => 'ACD Systems Digital Imaging, Quality 37',
    '40d08b823fa60b838dd9998d1e2b550a' => 'ACD Systems Digital Imaging, Quality 38',
    '4998abefc838e35cf0180395309e2e33' => 'ACD Systems Digital Imaging, Quality 39',
    '280205c47c8d3706c2f36b1986e9b149' => 'ACD Systems Digital Imaging, Quality 40 or 41',
    '8534b67f8115ddc0296623a1ed3fc8ec' => 'ACD Systems Digital Imaging, Quality 42',
    '292b83b37765408b65f496cddd3f96ea' => 'ACD Systems Digital Imaging, Quality 43',
    'cae0c8eb9a11a1f6eb7eca9651d8dbc0' => 'ACD Systems Digital Imaging, Quality 44',
    '078db0d0bffafa44def2e8b85eec26f6' => 'ACD Systems Digital Imaging, Quality 45 or 46',
    '6a26a11cc28df00e01d5979e2e0fb4f7' => 'ACD Systems Digital Imaging, Quality 47',
    'b41b3d226ba21244b8070ba719ec721a' => 'ACD Systems Digital Imaging, Quality 48',
    '9a8a54328e297faa0a546c46145c9aa8' => 'ACD Systems Digital Imaging, Quality 49',
    '256e617be51dade18503fcbbe87cd4a6' => 'ACD Systems Digital Imaging, Quality 50 or 51',
    '064f160a8504465551738c9071f3850f' => 'ACD Systems Digital Imaging, Quality 52',
    '5aef4c0bc6a5c8f1baded29946a56310' => 'ACD Systems Digital Imaging, Quality 53',
    'c20f4841a1ff7e393af8f6ea4124403c' => 'ACD Systems Digital Imaging, Quality 54',
    '14afe9b58e0eacef42db61e1d7fdd09c' => 'ACD Systems Digital Imaging, Quality 55 or 56',
    '147598404233439485574200e253f88e' => 'ACD Systems Digital Imaging, Quality 57',
    '17479c1e73d2c062872c871db80d949b' => 'ACD Systems Digital Imaging, Quality 58',
    'd237b1202f88ba8183bc1cb69dd4be66' => 'ACD Systems Digital Imaging, Quality 59',
    'fc923f2d38e0e549134e1ec86f58149a' => 'ACD Systems Digital Imaging, Quality 60 or 61',
    '93b4929d4a3b955f4996ab7e3b6fbe53' => 'ACD Systems Digital Imaging, Quality 62',
    '054f418c24a6a733186a27aa739dc93a' => 'ACD Systems Digital Imaging, Quality 63',
    '0df2be705ae86e5de1e508db95efb182' => 'ACD Systems Digital Imaging, Quality 64',
    'c1978a445de1173b5039b0cf8d8a91fe' => 'ACD Systems Digital Imaging, Quality 65 or 66',
    '3d8e25b74d0d9be662f26ec5fed6fe94' => 'ACD Systems Digital Imaging, Quality 67',
    '8887b718c97e0d80ed8d9a198387e2eb' => 'ACD Systems Digital Imaging, Quality 68',
    '8cdb9100cfbb246d440d469e72ce37a6' => 'ACD Systems Digital Imaging, Quality 69',
    '379efafa6e71a90ccfcb57073d0bc5c8' => 'ACD Systems Digital Imaging, Quality 70 or 71',
    'e1e122ebb2733a5ccdb5ff1cdce86d4d' => 'ACD Systems Digital Imaging, Quality 72',
    '41dd47887a2b87e22ad3bbacc022374e' => 'ACD Systems Digital Imaging, Quality 73',
    'a0a30c816d5d47a91c66e5645eb5fdb8' => 'ACD Systems Digital Imaging, Quality 74',
    '731f7ffedba80407d039c1db5a785f95' => 'ACD Systems Digital Imaging, Quality 75 or 76',
    '7cfd092a41a0e1c029e82467cb4c034f' => 'ACD Systems Digital Imaging, Quality 77',
    'f1b005980104aac41b49973beed9c8c2' => 'ACD Systems Digital Imaging, Quality 78',
    'ba12dbfbd652c9cde69822996bdb2139' => 'ACD Systems Digital Imaging, Quality 79',
    'd3784280d08a8df51e607bde8c8b5ead' => 'ACD Systems Digital Imaging, Quality 80 or 81',
    '7ed560efea0b44168d910a73fab9204c' => 'ACD Systems Digital Imaging, Quality 82',
    'deaa8bbd7c5414b93d8029aa14a76d3a' => 'ACD Systems Digital Imaging, Quality 83',
    '9ae3a57ce98290176c4700baaff5661f' => 'ACD Systems Digital Imaging, Quality 84',
    'cb99b9bd30ae36929755fee9208ab36b' => 'ACD Systems Digital Imaging, Quality 85 or 86',
    '75ff62bbf17aa1762dd15677e961ce67' => 'ACD Systems Digital Imaging, Quality 87',
    'd4f1922c71a6c96a530a9a8268fbc63b' => 'ACD Systems Digital Imaging, Quality 88',
    'ec994ef421efd6bc78671858b9f942ad' => 'ACD Systems Digital Imaging, Quality 89',
    '5ca52e1ffe2c84660d7377c33c88ad53' => 'ACD Systems Digital Imaging, Quality 90 or 91',
    '5522213c915e2af3ad01ee2ec27ee3ed' => 'ACD Systems Digital Imaging, Quality 92',
    '21aa1a0036251eecfffd24e37d7ce3dd' => 'ACD Systems Digital Imaging, Quality 93',
    '3233b63fc39fbbaa9af364e8a33862ff' => 'ACD Systems Digital Imaging, Quality 94',
    '1860106097672532e7ebc2026d7f9681' => 'ACD Systems Digital Imaging, Quality 95 or 96',
    '0c7d4861b3bee5d766a93f2d34027bfa' => 'ACD Systems Digital Imaging, Quality 97',
    'c9309ab058680151be5f97e6c54dc687' => 'ACD Systems Digital Imaging, Quality 98',
    'ffe6bb565b2c9008ab917c57ba94cd67' => 'ACD Systems Digital Imaging, Quality 99',
    'd6390cc36d2f03c1d2dd13d6910ca46b' => 'ACD Systems Digital Imaging, Quality 100; Pentax K20D/OptioE60 Premium',

    # StereoPhoto Maker Version 3.25 - Option "No compression ghosting" activated
    '185893c53196f6156d458a84e1135c43:111111' => 'StereoPhoto Maker, No compression ghosting, Quality 1',
    'b41ccbe66e41a05de5e68832c07969a7:111111' => 'StereoPhoto Maker, No compression ghosting, Quality 2',
    'efa024d741ecc5204e7edd4f590a7a25:111111' => 'StereoPhoto Maker, No compression ghosting, Quality 3',
    '3396344724a1868ada2330ebaeb9448e:111111' => 'StereoPhoto Maker, No compression ghosting, Quality 4',
    '14276fffb98deb42b7dbce30abb8425f:111111' => 'StereoPhoto Maker, No compression ghosting, Quality 5',
    'a99e2826c10d0922ce8942c5437f53a6:111111' => 'StereoPhoto Maker, No compression ghosting, Quality 6',
    '0d3de456aa5cbb8a2578208250aa9b88:111111' => 'StereoPhoto Maker, No compression ghosting, Quality 7',
    'fa987940fdedbe883cc0e9fcc907f89e:111111' => 'StereoPhoto Maker, No compression ghosting, Quality 8',
    '1c9bb67190ee64e82d3c67f7943bf4a4:111111' => 'StereoPhoto Maker, No compression ghosting, Quality 9',
    '57d20578d190b04c7667b10d3df241bb:111111' => 'StereoPhoto Maker, No compression ghosting, Quality 10',
    '619fd49197f0403ce13d86cffec46419:111111' => 'StereoPhoto Maker, No compression ghosting, Quality 11',
    '327f47dd8f999b2bbb3bb25c43cf5be5:111111' => 'StereoPhoto Maker, No compression ghosting, Quality 12',
    'e4e5bc705c40cfaffff6565f16fe98a9:111111' => 'StereoPhoto Maker, No compression ghosting, Quality 13',
    '6c64fa9ad302624a826f04ecc80459be:111111' => 'StereoPhoto Maker, No compression ghosting, Quality 14',
    '039a3f0e101f1bcdb6bb81478cf7ae6b:111111' => 'StereoPhoto Maker, No compression ghosting, Quality 15',
    'c23b08c94d7537c9447691d54ae1080c:111111' => 'StereoPhoto Maker, No compression ghosting, Quality 16',
    '200107bc0174104bbf1d4653c4b05058:111111' => 'StereoPhoto Maker, No compression ghosting, Quality 17',
    '72abfdc6e65b32ded2cd7ac77a04f447:111111' => 'StereoPhoto Maker, No compression ghosting, Quality 18',
    '1799a236c36da0b30729d9005ca7c7f9:111111' => 'StereoPhoto Maker, No compression ghosting, Quality 19',
    'c33a667bff7f590655d196010c5e39f3:111111' => 'StereoPhoto Maker, No compression ghosting, Quality 20',
    'b1dc98f6a2f8828f8432872da43e7d94:111111' => 'StereoPhoto Maker, No compression ghosting, Quality 21',
    '07318a0acfebe9086f0e04a4c4f5398a:111111' => 'StereoPhoto Maker, No compression ghosting, Quality 22',
    'a295b7163305f327a5a45ae177a0a19c:111111' => 'StereoPhoto Maker, No compression ghosting, Quality 23',
    'c741c1b134cf81ab69acc81f15a67137:111111' => 'StereoPhoto Maker, No compression ghosting, Quality 24',
    'a68893776502a591548c7b5bece13e1b:111111' => 'StereoPhoto Maker, No compression ghosting, Quality 25',
    '111848d9e41f6f408ef70841f90c0519:111111' => 'StereoPhoto Maker, No compression ghosting, Quality 26',
    '886374ceebcfd4dfed200b0b34b4baca:111111' => 'StereoPhoto Maker, No compression ghosting, Quality 27',
    '666dd95fd0e20f5c20bc44d78d528869:111111' => 'StereoPhoto Maker, No compression ghosting, Quality 28',
    '1aa58cb85dda84de2ddf436667124dcd:111111' => 'StereoPhoto Maker, No compression ghosting, Quality 29',
    '9d321ab2bdda6f3cb76d2d88838aa8c3:111111' => 'StereoPhoto Maker, No compression ghosting, Quality 30',
    '6ad87d648101c268f83fa379d4c773f2:111111' => 'StereoPhoto Maker, No compression ghosting, Quality 31',
    'cdf8e921300f27a4af7661a2de16e91a:111111' => 'StereoPhoto Maker, No compression ghosting, Quality 32',
    '3f48672e37b6dd2e571b222e4b7ff97d:111111' => 'StereoPhoto Maker, No compression ghosting, Quality 33',
    'a53a7d4cc86d01f4c1b867270c9c078f:111111' => 'StereoPhoto Maker, No compression ghosting, Quality 34',
    '09ec03f5096df106c692123f3fd34296:111111' => 'StereoPhoto Maker, No compression ghosting, Quality 35',
    'a946498fd1902c9de87a1f5182966742:111111' => 'StereoPhoto Maker, No compression ghosting, Quality 36',
    '5d650a1d38108fd79d4f336ba8e254c2:111111' => 'StereoPhoto Maker, No compression ghosting, Quality 37',
    '81d620f1b470fd535b26544b4ea20643:111111' => 'StereoPhoto Maker, No compression ghosting, Quality 38',
    '892788bdf8cbef5c6fbd7019a079bf8e:111111' => 'StereoPhoto Maker, No compression ghosting, Quality 39',
    'cf3929fd4c1e5c28b7f137f982178ad1:111111' => 'StereoPhoto Maker, No compression ghosting, Quality 40',
    '31f288945896ed839f1d936bff06fb03:111111' => 'StereoPhoto Maker, No compression ghosting, Quality 41',
    'e0c38f0c5e6562445d4e92bae51713be:111111' => 'StereoPhoto Maker, No compression ghosting, Quality 42',
    '18fa29d1164984883a6af76377b60d5a:111111' => 'StereoPhoto Maker, No compression ghosting, Quality 43',
    'eff737b226fbce48c42625c5bf9dabb6:111111' => 'StereoPhoto Maker, No compression ghosting, Quality 44',
    'b900f91ee8697255d5daebce858caaeb:111111' => 'StereoPhoto Maker, No compression ghosting, Quality 45',
    'ab2f8513823067af242f7e3c04a88a9c:111111' => 'StereoPhoto Maker, No compression ghosting, Quality 46',
    '60b682c4d412f5255efbaa32787c46ca:111111' => 'StereoPhoto Maker, No compression ghosting, Quality 47',
    'ea50813e06203c2ad1165252bcb99a1d:111111' => 'StereoPhoto Maker, No compression ghosting, Quality 48',
    'f6308a717437d3653b0751ebf511db0f:111111' => 'StereoPhoto Maker, No compression ghosting, Quality 49',
    '7c8242581553e818ef243fc680879a19:111111' => 'StereoPhoto Maker, No compression ghosting, Quality 50',
    'fc41ab8251718977bc6676f502f457e0:111111' => 'StereoPhoto Maker, No compression ghosting, Quality 51',
    '606c4c78c0226646bf4d3c5a5898fb17:111111' => 'StereoPhoto Maker, No compression ghosting, Quality 52',
    '0e6c6a5440d33d25f1c25836a45cfa69:111111' => 'StereoPhoto Maker, No compression ghosting, Quality 53',
    '7464b2361e5b5f5a9ba74a87475dda91:111111' => 'StereoPhoto Maker, No compression ghosting, Quality 54',
    'aeaa2ca48eabb3088ebb713b3c4e1a67:111111' => 'StereoPhoto Maker, No compression ghosting, Quality 55',
    '3f36450b0ba074578391e77f7340cef0:111111' => 'StereoPhoto Maker, No compression ghosting, Quality 56',
    'be232444027e83db6f8d8b79d078442e:111111' => 'StereoPhoto Maker, No compression ghosting, Quality 57',
    '712c145d6472a2b315b2ecfb916d1590:111111' => 'StereoPhoto Maker, No compression ghosting, Quality 58',
    'ae3dd4568cc71c47d068cf831c66b59d:111111' => 'StereoPhoto Maker, No compression ghosting, Quality 59',
    'b290e52c21a435fede4586636ef5e287:111111' => 'StereoPhoto Maker, No compression ghosting, Quality 60',
    'a09ca4c4391e0221396a08f229a65f9d:111111' => 'StereoPhoto Maker, No compression ghosting, Quality 61',
    '0818578fc5fc571b4f8d5ffefc9dc0d8:111111' => 'StereoPhoto Maker, No compression ghosting, Quality 62',
    '7c685e2916555eda34cb37a1e71adc6a:111111' => 'StereoPhoto Maker, No compression ghosting, Quality 63',
    '69c6b9440342adfc0db89a6c91aba332:111111' => 'StereoPhoto Maker, No compression ghosting, Quality 64',
    'd5d484b68e25b44288e67e699829695c:111111' => 'StereoPhoto Maker, No compression ghosting, Quality 65',
    'de8310d09116a7a62965f3e0e43ef525:111111' => 'StereoPhoto Maker, No compression ghosting, Quality 66',
    'e4735f63e88baf04599afc034e690845:111111' => 'StereoPhoto Maker, No compression ghosting, Quality 67',
    'b4ef810b14dee9c6d6d8cace98f799a6:111111' => 'StereoPhoto Maker, No compression ghosting, Quality 68',
    '52886ef80147c9a136e20b2bc3b76f52:111111' => 'StereoPhoto Maker, No compression ghosting, Quality 69',
    '9c62dbc848be82ef91219ba9843998be:111111' => 'StereoPhoto Maker, No compression ghosting, Quality 70',
    'bfe8c1c73eea84b85673487a82f67627:111111' => 'StereoPhoto Maker, No compression ghosting, Quality 71',
    'ea445840d29c51009a2a8cd49b96ccee:111111' => 'StereoPhoto Maker, No compression ghosting, Quality 72',
    '71c1a56890fff9b0a095fa5a1c96132b:111111' => 'StereoPhoto Maker, No compression ghosting, Quality 73',
    'f080b02331ac8adf03de2281042d2b49:111111' => 'StereoPhoto Maker, No compression ghosting, Quality 74',
    'd0eaa368737f17f6037757d393a22599:111111' => 'StereoPhoto Maker, No compression ghosting, Quality 75',
    '303663905d055b77bb547fe0b0beb9c5:111111' => 'StereoPhoto Maker, No compression ghosting, Quality 76',
    '5cdf1d5bbe19375ad5c7237273dddede:111111' => 'StereoPhoto Maker, No compression ghosting, Quality 77',
    'd64e7ff8292fd77131932864d3c9ce7c:111111' => 'StereoPhoto Maker, No compression ghosting, Quality 78',
    '12b4cc13891c5aef3dadb3405b6fa65d:111111' => 'StereoPhoto Maker, No compression ghosting, Quality 79',
    'b008cd63591f8fd366f77d2b224b9c9c:111111' => 'StereoPhoto Maker, No compression ghosting, Quality 80',
    '49b6e472c7d5ecead593c6009768e765:111111' => 'StereoPhoto Maker, No compression ghosting, Quality 81',
    'bce6fa61623ad4f65ff3fec1528cb026:111111' => 'StereoPhoto Maker, No compression ghosting, Quality 82',
    'c2b037bf9f5e5baba804d7bbbb2dc73b:111111' => 'StereoPhoto Maker, No compression ghosting, Quality 83',
    '7fe7b339c6ffc62b984eeab4b0df9168:111111' => 'StereoPhoto Maker, No compression ghosting, Quality 84',
    '274bbeb0ac3939f90c578ebb1f5a9eef:111111' => 'StereoPhoto Maker, No compression ghosting, Quality 85',
    '0a0268c655d616b0e4af2851533aa3af:111111' => 'StereoPhoto Maker, No compression ghosting, Quality 86',
    '52318e260c0d6b3dbee85c87f9b94e63:111111' => 'StereoPhoto Maker, No compression ghosting, Quality 87',
    'b64cc19a0f81a506ed5bcfb9c131c8fe:111111' => 'StereoPhoto Maker, No compression ghosting, Quality 88',
    'd8c54333eb475b8db9f32f11fe96337e:111111' => 'StereoPhoto Maker, No compression ghosting, Quality 89',
    '12fe6b9bfd20f4d7f0ac2a221c566c45:111111' => 'StereoPhoto Maker, No compression ghosting, Quality 90',
    '12aefbf7689633c83da714c9f0e90e05:111111' => 'StereoPhoto Maker, No compression ghosting, Quality 91',
    'a3a96add050fc51a2b3ce59a9a491034:111111' => 'StereoPhoto Maker, No compression ghosting, Quality 92',
    '7b0242bd9aaeab4962f5d5b39b9a4027:111111' => 'StereoPhoto Maker, No compression ghosting, Quality 93',
    '12fc29c1d8940c93a47ee9d927a17561:111111' => 'StereoPhoto Maker, No compression ghosting, Quality 94',
    'e1fedef5184beeb7b0f5c055c7ae1d31:111111' => 'StereoPhoto Maker, No compression ghosting, Quality 95',
    'ae9202355f603776794d3e62c43578d6:111111' => 'StereoPhoto Maker, No compression ghosting, Quality 96',
    '36da00bae6cd81d1f97e32748c07e33f:111111' => 'StereoPhoto Maker, No compression ghosting, Quality 97',

    # Tested with FujiFilm FinePixViewer Ver 5.4.11G (Win)
    '3a8a34631e388e39d13616d003f05957:211111' => 'FinePixViewer, Basic',
    'b6a2598792fd87b7eb0c094cbd52862f:211111' => 'FinePixViewer, Fine',
    '4ee61c39b97558a273f310e085d0bdd2:211111' => 'FinePixViewer, Normal',

    # Tested with Canon Digital Photo Professional Version 3.4.1.1 (Win)
    '252482232ff1c8cf77db4f0c6402f858:211111' => 'Canon Digital Photo Professional, Quality 1',
    'ec6c55677b94970bc09f70265f1d5b55:211111' => 'Canon Digital Photo Professional, Quality 2',
    'a1085c167f1cd610258fe38c8a84a8b9:211111' => 'Canon Digital Photo Professional, Quality 3',
    '8ab1119f4ed4941736cb8ec1796f5674:211111' => 'Canon Digital Photo Professional, Quality 4',
    'e66c03f97b19213f385136f014c78ac1:211111' => 'Canon Digital Photo Professional, Quality 5',
    'a2f4b6ac52f87791380bdfe38ae333e1:211111' => 'Canon Digital Photo Professional, Quality 6',
    'fe85b802c5779dcf45ea4bb7749ee886:211111' => 'Canon Digital Photo Professional, Quality 7',
    '35686967efa5fb333fb8f4844efc33a3:211111' => 'Canon Digital Photo Professional, Quality 8',
    'a5894172d7ec5f0c1550934c9e9385c9:211111' => 'Canon Digital Photo Professional, Quality 9',
    'd6390cc36d2f03c1d2dd13d6910ca46b:211111' => 'Canon Digital Photo Professional, Quality 10',

    # Tested with Canon ZoomBrowser EX 6.1.1.21 (Win)
    'e66c03f97b19213f385136f014c78ac1:211111' => 'Canon ZoomBrowser, Low',
    'bf72e4d4aacbdaeb86fd3f67c8df2667:211111' => 'Canon ZoomBrowser, Medium',
    'd6390cc36d2f03c1d2dd13d6910ca46b:211111' => 'Canon ZoomBrowser, Highest',
    'aeb34eb083acc888770d65e691497bcf:211111' => 'Canon ZoomBrowser, High',

    # Tested with PENTAX PHOTO Laboratory 3.51 (Win)
    '76d958276bf2cac3c36b7d9a677094a7:211111' => 'PENTAX PHOTO Laboratory, Highest compression',
    'bf72e4d4aacbdaeb86fd3f67c8df2667:211111' => 'PENTAX PHOTO Laboratory, High compression',
    'fa8720d025f2a164542b6a8e31112991:211111' => 'PENTAX PHOTO Laboratory, Medium quality',
    'f3235a7d187d083b7b7ead949653f730:211111' => 'PENTAX PHOTO Laboratory, High quality',
    'd6390cc36d2f03c1d2dd13d6910ca46b:211111' => 'PENTAX PHOTO Laboratory, Highest quality',

    # Tested with Sony Image Data Converter SR Version 2.0.00.08150 (Win)
    'd6390cc36d2f03c1d2dd13d6910ca46b:211111' => 'Sony Image Data Suite, Quality 1 (high quality)',
    'aeb34eb083acc888770d65e691497bcf:211111' => 'Sony Image Data Suite, Quality 2',
    '524742ca0cff64ecc0c7d7413e7d4b8d:211111' => 'Sony Image Data Suite, Quality 3',
    'c44701e8185306f5e6d09be16a2b0fbd:211111' => 'Sony Image Data Suite, Quality 4 (high compression)',

    # Devices

    # Canon
    '0147c5088beb16642f9754f8671f13b3:211111' => 'Canon PowerShot, Fine',
    '586b40c7d4b95e11309636703e81fbe9:211111' => 'Canon EOS 20D, Fine',
    '6640ae3bb6f646013769b182c74931b5:211111' => 'Canon PowerShot, Normal',
    '83d6d7dd7ace56feeeb65b88accae1bc:211111' => 'Canon PowerShot, Normal Small',
    'b8548a302585d78a0c269b54bff86541:211111' => 'Canon PowerShot, Fine Small',
    '9d125046484461bbc155d8eff6d4e8f0:211111' => 'Canon PowerShot, Superfine (A430/A460)',
    '0e618a0e79b4d540da1f6e07fcdce354:211111' => 'Canon PowerShot, Superfine Small',
    'd255f70a910a2d0039f4e792d2c01210:211111' => 'Canon PowerShot, Superfine Medium2',
    '8bc267b04a54c02fdee1f4fdf0bcce83:211111' => 'Canon EOS 1DmkIII/5DmkII/40D/1000D, Fine',
    '17cb779485969589a5c7eb07a5d53247:211111' => 'Canon EOS 1DmkIII, Fine (pre-production)',
    'ee1c033afaf4cd5263ff2b1c1ff8966c:211111' => 'Canon PowerShot, Superfine',
    'a92912eb3c81e5c873d49433264af842:211111' => 'Canon EOS 30D/40D/50D/300D, Normal',
    '0cec88a0cd8fe35720e78cdcdbdadef6:121111' => 'Canon EOS 1DmkII, Fine (A)',
    '72cdcc91e3ddc2c3d17c20173b75c5ef:211111' => 'Canon EOS 1DmkII, Fine (B)',
    '483b5288e4256aa8ff96d6ccb96eba43:211111' => 'Canon EOS 1DmkII, Fine (C)',
    'ea2f997a0261bab501bf122b04cbc859:211111' => 'Canon EOS 1DSmkII, Fine',
    '98af13526b7e4bbf73a9fb11a8fa789d:121111' => 'Canon EOS 1DSmkII, Fine (vertical)',
    '9e6abfb26d3b95b8cd2f710e78def947:121111' => 'Canon EOS 300D, Fine (vertical)',
    '4d6b36e81fe30c67dd53edb4d7c05422:121111' => 'Canon EOS 40D, Fine (vertical)',
    '92c1557deaa14f1cdaf92cf0531487f1:121111' => 'Canon EOS 1D/1DS, Fine',
    'db8d4df12405d0d69eb25f06a963ac5b:211111' => 'Canon DV',
    'eaead98bbdfde35210f48286662e8ad2:211111' => 'Canon DV Hi-Res',
    '74f0ef9476707be45f06951ca9a809ba:211111' => 'Canon DV/Optura/Elura, Superfine',

    # HTC
    'bf72e4d4aacbdaeb86fd3f67c8df2667:221111' => 'HTC Touch Diamond P3700, Quality Unknown',

    # Konica/Minolta
    'b5c213a3785c4c62b25d8f8c30758593:211111' => 'Konica/Minolta DYNAX 7D, Fine',

    # Nikon
    '118a60a90c56bcb363fdd93b911a3371:211111' => 'Nikon D50 / D80, Fine',
    '457b05fd0787a8e29bd43cd65911d6ca:211111' => 'Nikon D80, Basic',
    '5701582a0da2e9e8dcd923a5cf877494:211111' => 'Nikon D50, Fine',
    '662bd7fb9dff6426e310f9261a3703d0:211111' => 'Nikon D50, Fine',
    'e06eb7848ec8766239ff014aa8b62e49:211111' => 'Nikon D80, Normal',
    '9e201a496a3700a77d9102c0dd0f8dbf:211111' => 'Nikon D300, Basic',

    # Panasonic
    '07d3cd227395b060a132411cbfc22593:211111' => 'Panasonic DMC-FZ50, High (A)',
    '118a60a90c56bcb363fdd93b911a3371:211111' => 'Panasonic DMC-FZ50/TZ3, High (A)',
    '1b8d04b1d56a4c0c811a0d3a68e86d06:211111' => 'Panasonic DMC-FZ50, High (B)',
    '1e619cbdee1f8ff196d34dad9140876f:211111' => 'Panasonic DMC-FZ50, High (C)',
    '493abc7f4b392a0341bfcac091edb8f8:211111' => 'Panasonic DMC-FZ30, High (B)',
    '4aa883c43840de7f0d090284120c69bc:211111' => 'Panasonic DMC-FZ50, High (D)',
    '7eafb9874384d391836e64911e912295:211111' => 'Panasonic DMC-FZ50, High (E)',
    '82b56237e4eccde035edff4a5abdba44:211111' => 'Panasonic DMC-FZ50, High (F)',
    '8335023e5a1ee8df80d52327b0556c44:211111' => 'Panasonic DMC-FZ30, High (C)',
    '8c105b3669931607853fa5ba4fffb839:211111' => 'Panasonic DMC-FZ30, High (D)',
    '8ecfb959bc76e5d6703f3f3bba2c5529:211111' => 'Panasonic DMC-FZ30, High (E)',
    '96eda111b2153648b3f27d6c1a9ec48f:211111' => 'Panasonic DMC-FZ50/TZ3, High (B)',
    '99f76923cfbd774febea883b603b8103:211111' => 'Panasonic DMC-FZ30, High (F)',
    '9b3475b865b9d31e433538460b75a588:211111' => 'Panasonic DMC-FZ10, High',
    '9eb7cdfd07099c1bb8e2c6c04b20b8ba:211111' => 'Panasonic DMC-FZ30, High (G)',
    '9fc030294fa5c4044dbb0cb461b0cf93:211111' => 'Panasonic DMC-TZ5, High (A)',
    'a8779af4cb8afa2def1d346a9b16e81a:211111' => 'Panasonic DMC-TZ5, High (B)',
    'bebd334aca511e2a2b6c60f43f9e6cf1:211111' => 'Panasonic DMC-FZ30, High (H)',
    'c871ce0851d4647f226b2dcfd49fe9a9:211111' => 'Panasonic DMC-L1, Very High',
    'eb625c64e32314f51dc4286564a71f7b:211111' => 'Panasonic DMC-FZ10, High',

    # Pentax
    # (K10D uses same DQT tables for different qualities)
    '1027a4af6a2a07e58bbd6df5b197d44e:211111' => 'Pentax K10D (A)',
    '17a77c2574ff5b72b3284f57977187f3:211111' => 'Pentax K10D (B)',
    '1aee684c7eb75320d988f6296c4c16ea:211111' => 'Pentax K10D (C)',
    '32386501afff88b45432b23fe41593e8:211111' => 'Pentax K10D (D)',
    '35ad02c3d8237a074b67423c39d3d61c:211111' => 'Pentax K10D (E)',
    '39d929c095f37a90e7d083db40e8642d:211111' => 'Pentax K10D (F)',
    '4127433151f74654762b1ef3293781f4:211111' => 'Pentax K10D (G)',
    '599a7794c32b9d60e80426909ed40a09:211111' => 'Pentax K10D (H)',
    '641812174c82d5b62ec86c33bd852204:211111' => 'Pentax K10D (I)',
    '76d958276bf2cac3c36b7d9a677094a7:211111' => 'Pentax K10D (J)',
    '79b07131be4827795315bf42c65212f2:211111' => 'Pentax K10D (K)',
    '836448ef538366adb50202927b53808a:211111' => 'Pentax K10D (L)',
    '8f70e4a31ad4584043ddc655eca17e89:211111' => 'Pentax K10D (M)',
    '90d3c964eaf6e4bd12cf5ca791a7d753:211111' => 'Pentax K10D (N)',
    '994a9f2060976d95719ca7064be3a99c:211111' => 'Pentax K10D (0)',
    '994a9f2060976d95719ca7064be3a99c:211111' => 'Pentax K10D/K20D (P)',
    '9971f02a466c47d640e8f20a2e4b55b9:211111' => 'Pentax K10D (Q)',
    'a16626c285e5a2290d331f99f4eec774:211111' => 'Pentax K10D (R)',
    'a64569d6387a118992e44e41aaeac27e:211111' => 'Pentax K10D (S)',
    'a8055a53fda7f9a0e387026c81960aa4:211111' => 'Pentax K10D (T)',
    'ab50a9f53a44ffecc54efe1cb7c6620a:211111' => 'Pentax K10D (U)',
    'aeb34eb083acc888770d65e691497bcf:211111' => 'Pentax K10D (V)',
    'af2a112c30fa29213a402dbd3c2b2d3a:211111' => 'Pentax K10D (W)',
    'bb4475a9e14464eb4682fd81cceb1f91:211111' => 'Pentax K10D (X)',
    'bf72e4d4aacbdaeb86fd3f67c8df2667:211111' => 'Pentax K10D (Y)',
    '0a953ba56b59fa0bbbdac0162ea1c96b:211111' => 'Pentax K10D (Z)',
    '387354b46b9726f33da5c0c1a0c383a0:211111' => 'Pentax K10D/K20D (AA)',
    '4e7f4e5cd15f4fc089ab25890619dc60:211111' => 'Pentax K10D (AB)',
    '6518270228fd20730740a08cc8a171f6:211111' => 'Pentax K10D (AC)',
    '72bce7df55635509eb6468fc6406941d:211111' => 'Pentax K10D (AD)',
    '7cafc25f204fc4ddf39d86e2f0f07b62:211111' => 'Pentax K10D (AE)',
    '811e5b0229f0e8baf4b40cd2d8777550:211111' => 'Pentax K10D (AF)',
    '9282a1cec6bbd1232b3673091164d43d:211111' => 'Pentax K10D (AG)',
    'c59a4cf0beedbfd1b102dc3d3c8e73ac:211111' => 'Pentax K10D (AH)',
    'd97b27b45fdbe82a79364e0939adbf90:121111' => 'Pentax K10D (AI)',
    'db87a4c5c1d4e03dc6645bcf0535a930:211111' => 'Pentax K10D (AJ)',
    'f9a93cb70da7bbe87e35cd9980a5fd47:211111' => 'Pentax K10D (AK)',
    'ff6a158f803e42bfbf9f702c016b84b3:211111' => 'Pentax K10D (AL)',
    'ff6d4a4a60a1c5e032e7fb7d9c91f817:211111' => 'Pentax K10D (AM)',
    'dca5476d81d0ceca97f480fecd09b23c:211111' => 'Pentax K10D (AN)',
    'efbe7634221900639b3c072395c61bef:211111' => 'Pentax K10D (AO)',
    'f4dba22dd251350a21f8122f2777e7b0:211111' => 'Pentax K10D (AP)',
    'f90135fcff0e1720dda86e9ad718c0c0:211111' => 'Pentax K10D (AQ)',
    'fa3d7753be7b329ab9961657cbc65386:211111' => 'Pentax K10D (AR)',
    'fa8720d025f2a164542b6a8e31112991:211111' => 'Pentax K10D (AS)',
    '2941d12ef34511d96b659ba30d682cd1:211111' => 'Pentax K10D (AT)',
    '2aa82b6717f1cdfe6b6d60a4486b5671:211111' => 'Pentax K10D (AU)',
    '2aa82b6717f1cdfe6b6d60a4486b5671:211111' => 'Pentax K10D (AV)',
    '3527616df6f26a3ab36b80a8d885fc07:211111' => 'Pentax K10D (AW)',
    '3527616df6f26a3ab36b80a8d885fc07:211111' => 'Pentax K10D (AX)',
    '3527616df6f26a3ab36b80a8d885fc07:211111' => 'Pentax K10D (AY)',
    '5ea9e766888399a41f3f1a3c5c15cd90:211111' => 'Pentax K10D (AZ)',
    'f83d978290d0699054eabb0a7811c7a4:211111' => 'Pentax K10D (BA)',
    'f83d978290d0699054eabb0a7811c7a4:211111' => 'Pentax K10D (BB)',
    'fa8720d025f2a164542b6a8e31112991:211111' => 'Pentax K10D/K100D',
    '586b40c7d4b95e11309636703e81fbe9:211111' => 'Pentax K20D/K200D/Optio 230, Best; Canon EOS 10D/300D/350D, Fine',
    'b73481179da895f3b9ecea1737054a9c:211111' => 'Pentax K20D, Best (B)',
    'b8fce00f93108e7db57a012c51fad341:211111' => 'Pentax K20D, Best (C)',
    '5ee766b90badc8fed5a5386e78a80783:211111' => 'Pentax *istDS, Good (edit in camera)',
    'd528fac9b63536ff52041745945dcb09:211111' => 'Pentax *istDS, Better (edit in camera)',
    'd6390cc36d2f03c1d2dd13d6910ca46b:211111' => 'Pentax *istDS, Best (edit in camera)',
    'dc149d41f08d16cb9d52a5bdd487a67e:121111' => 'Pentax *istD/K100Dsuper/Optio300GS, Best',
    'e10030f09a14acdd647eff13c0bf333a:211111' => 'Pentax *istD/DS/DS2/K100D/Optio330GS/33L, Best',
    'ef0cd1902fb1afe284468a67eaffd078:211111' => 'Pentax *istDS/K100D/K100Dsuper, Good',
    'f1262dfcada6e6c2cd4b9fa7e881233b:211111' => 'Pentax *istDL/DS, Better',
    'f3235a7d187d083b7b7ead949653f730:211111' => 'Pentax K20D/K200D, Best (D)',
    '6686cddc46088f0987e7476861fbfb47:211111' => 'Pentax K2000, Best (A)',
    '5910b8431fdd8ab93ce258f366c4b867:211111' => 'Pentax K2000, Best (B)',
    'c8bfcc60aeec937300405f59373be4ef:211111' => 'Pentax K2000, Best (C)',
    '689a0e3511f2aea75637f46e6af9fd9f:211111' => 'Pentax Optio A40, Best (edit in camera)',
    '8d14598ae9cc1b7f5357424a19d05a71:211111' => 'Pentax Optio A30/A40, Good',
    'a4cb8a3a000484b37c4373cde1170091:211111' => 'Pentax Optio A30/A40/S10/S12, Best',
    '0ac5cb651c496369d0e924ae070b7c53:211111' => 'Pentax Optio A40, Better (edit in camera)',
    '1068be028c278941bd8abf3b0021655e:211111' => 'Pentax Optio A40, Good (edit in camera)',
    '336eeeb78e386bf66fe6325b4a0fcfa6:211111' => 'Pentax Optio A40, Better',
    'ae2efaf1a96a4fdcfa9003b9aa963ae4:221111' => 'Pentax Optio 330, Best (vertical)',
    '3803d7f6b7aed64c658c21dbb2bc0797:221111' => 'Pentax Optio 330, Best',
    '353bf09900feb764885329e7bebfd95e:211111' => 'Pentax Optio 330GS, Good',
    '6c2bc41a4b6ad1e20655ffcc0dfd2c41:221111' => 'Pentax Optio 330RS, Fine',
    'e9206045838e9f5f9bd207744254e96d:221111' => 'Pentax Optio 430, Best',
    '759fb7011e13fa5f975bb668f5b94d8b:211111' => 'Pentax Optio 550/750Z/M60/X, Best',
    '637103ef9d8e84f8345f8218f158fc3c:211111' => 'Pentax Optio 550/M10/T30/W30, Best',
    '23f2a5970523c5f7fd2ab7fa3b09dff9:211111' => 'Pentax Optio 550/555/M20/M30/W10/W20, Best',
    '8d2f02a07bad6b5cec48466036fef319:121111' => 'Pentax Optio 550, Better',
    '27297008a89ee49804f0859ea6435878:211111' => 'Pentax Optio MX, Best',
    '6cfe3833aadd87487afc11129d8cb2aa:221111' => 'Pentax Optio S, Better',
    'fcef35c97674aeb26c67e539b726057f:221111' => 'Pentax Optio S, Best (A)',
    '13b2644cdad6f75767667e8ea5c218a3:221111' => 'Pentax Optio S, Best (B)',
    '310b70bc4fac884f64a07040a4b87468:221111' => 'Pentax Optio S, Best (C)',
    'aa05fbe795d86a1063c55865e8613536:221111' => 'Pentax Optio S, Best (D)',
    'd57ac6956e4fe86c386f0eef00a5e021:221111' => 'Pentax Optio S, Best (E)',
    '28782f5ee24fe983fe90b9438b39ae2e:221111' => 'Pentax Optio S4, Best',
    '804bd63907214e005f01fb65a2bb00e6:221111' => 'Pentax Optio S4i, Best',
    '84285f5b3248884488e5142b8c7210e2:211111' => 'Pentax Optio S6, Good',
    'e97694f0093de13987a335e131b30eb0:221111' => 'Pentax Optio SVi, Best',
    '037d043c8a8d5332c28d59f71a0dcfd2:211111' => 'Pentax Optio E35',
    'dd8ad8ce688c4248f924022c38d3228c:211111' => 'Pentax Optio 43WR, Good',
    'e55e0c1adbbca8b9d100881248050eb5:211111' => 'Pentax Optio 43WR, Better',
    '7770d784d852b3333f9213713e481125:211111' => 'Pentax Optio 450, Best',
    '61d311bde22762ae0e88b768e835eced:211111' => 'Pentax Optio 33WR/M50, Best',
    'bc066ff3fbea8a290c6f9882687945e0:221111' => 'Pentax Optio 430RS, Fine',
    'b6bd9f956309a20e3a56294077536391:211111' => 'Pentax Optio A10/S7, Best',
    'a4ecd6b77f06671530942783c3595aca:211111' => 'Pentax Optio A20, Best',
    '40f66b0a209f24716320636b776dda94:211111' => 'Pentax Optio E30/E40, Best',
    '59a868b3d11d9cdc87859c02757e13bb:211111' => 'Pentax Optio E50, Best',
    '9570584f017ed2c4f0fb91782b51faa9:211111' => 'Pentax Optio M40/Z10, Best',
    '5a74f09fb2586fa000c42e98e3b9f2d8:211111' => 'Pentax Optio T10',
    '0867bdf854d1fbb141411de518a66ba6:211111' => 'Pentax Optio T20 (A)',
    'f74b3853185743c111ccb13e6febdc21:211111' => 'Pentax Optio T20 (B)',
    'b6640d3879f9922708d23e6adb3d61c9:211111' => 'Pentax Optio V10, Best',
    '253467dc35dfbb32cb3d619fc635d689:211111' => 'Pentax Optio V20/W60, Best',

    # Sony
    '6bd350bf5df27ed1b5bf1d83fa9d021f:211111' => 'Sony DSLR-A700, Fine',
);

#------------------------------------------------------------------------------
# Estimate JPEG quality from quantization tables (ref 3)
# Inputs: 0) 1) DQT segments array ref
# Returns: JPEG quality, or undefined if it can't be calculated
sub EstimateQuality($)
{
    local $_;
    my $dqtList = shift;
    my ($i, $dqt, @qtbl, $quality, @hash, @sums);

    # unpack DQT segments and sum quantization tables
    my $sum=0;
DQT: foreach $dqt (@$dqtList) {
        next unless defined $dqt;
        for ($i=1; $i+64<=length($dqt); $i+=65) {
            my @qt = unpack("x$i C64", $dqt);
            $sum += $_ foreach @qt;
            push @qtbl, \@qt;
            last DQT if @qtbl >= 4;
        }
    }
    return undef unless @qtbl;

    my $qval = $qtbl[0][2] + $qtbl[0][53];
    if (@qtbl > 1) {
        # color JPEG
        $qval += $qtbl[1][0] + $qtbl[1][63];
        @hash =(
            1020, 1015,  932,  848,  780,  735,  702,  679,  660,  645,
             632,  623,  613,  607,  600,  594,  589,  585,  581,  571,
             555,  542,  529,  514,  494,  474,  457,  439,  424,  410,
             397,  386,  373,  364,  351,  341,  334,  324,  317,  309,
             299,  294,  287,  279,  274,  267,  262,  257,  251,  247,
             243,  237,  232,  227,  222,  217,  213,  207,  202,  198,
             192,  188,  183,  177,  173,  168,  163,  157,  153,  148,
             143,  139,  132,  128,  125,  119,  115,  108,  104,   99,
              94,   90,   84,   79,   74,   70,   64,   59,   55,   49,
              45,   40,   34,   30,   25,   20,   15,   11,    6,    4
        );
        @sums = (
            32640, 32635, 32266, 31495, 30665, 29804, 29146, 28599, 28104, 27670,
            27225, 26725, 26210, 25716, 25240, 24789, 24373, 23946, 23572, 22846,
            21801, 20842, 19949, 19121, 18386, 17651, 16998, 16349, 15800, 15247,
            14783, 14321, 13859, 13535, 13081, 12702, 12423, 12056, 11779, 11513,
            11135, 10955, 10676, 10392, 10208,  9928,  9747,  9564,  9369,  9193,
             9017,  8822,  8639,  8458,  8270,  8084,  7896,  7710,  7527,  7347,
             7156,  6977,  6788,  6607,  6422,  6236,  6054,  5867,  5684,  5495,
             5305,  5128,  4945,  4751,  4638,  4442,  4248,  4065,  3888,  3698,
             3509,  3326,  3139,  2957,  2775,  2586,  2405,  2216,  2037,  1846,
             1666,  1483,  1297,  1109,   927,   735,   554,   375,   201,   128
        );
    } else {
        # greyscale JPEG
        @hash = (
            510,  505,  422,  380,  355,  338,  326,  318,  311,  305,
            300,  297,  293,  291,  288,  286,  284,  283,  281,  280,
            279,  278,  277,  273,  262,  251,  243,  233,  225,  218,
            211,  205,  198,  193,  186,  181,  177,  172,  168,  164,
            158,  156,  152,  148,  145,  142,  139,  136,  133,  131,
            129,  126,  123,  120,  118,  115,  113,  110,  107,  105,
            102,  100,   97,   94,   92,   89,   87,   83,   81,   79,
             76,   74,   70,   68,   66,   63,   61,   57,   55,   52,
             50,   48,   44,   42,   39,   37,   34,   31,   29,   26,
             24,   21,   18,   16,   13,   11,    8,    6,    3,    2
        );
        @sums = (
            16320, 16315, 15946, 15277, 14655, 14073, 13623, 13230, 12859, 12560,
            12240, 11861, 11456, 11081, 10714, 10360, 10027,  9679,  9368,  9056,
             8680,  8331,  7995,  7668,  7376,  7084,  6823,  6562,  6345,  6125,
             5939,  5756,  5571,  5421,  5240,  5086,  4976,  4829,  4719,  4616,
             4463,  4393,  4280,  4166,  4092,  3980,  3909,  3835,  3755,  3688,
             3621,  3541,  3467,  3396,  3323,  3247,  3170,  3096,  3021,  2952,
             2874,  2804,  2727,  2657,  2583,  2509,  2437,  2362,  2290,  2211,
             2136,  2068,  1996,  1915,  1858,  1773,  1692,  1620,  1552,  1477,
             1398,  1326,  1251,  1179,  1109,  1031,   961,   884,   814,   736,
              667,   592,   518,   441,   369,   292,   221,   151,    86,    64
        );
    }
    for ($i=0; $i<100; ++$i) {
        next if $qval < $hash[$i] and $sum < $sums[$i];
        $quality = $i + 1 if ($qval <= $hash[$i] and $sum <= $sums[$i]) or $i >= 50;
        last;
    }
    return $quality;
}

#------------------------------------------------------------------------------
# Calculate JPEGDigest and/or JPEGQualityEstimate
# Inputs: 0) ExifTool object ref, 1) DQT segments array ref, 2) subsampling string
sub Calculate($$$)
{
    my ($et, $dqtList, $subSampling) = @_;

    # estimate JPEG quality if requested
    my $all = ($$et{OPTIONS}{RequestAll} and $$et{OPTIONS}{RequestAll} > 2);
    if ($all or $$et{REQ_TAG_LOOKUP}{jpegqualityestimate}) {
        my $quality = EstimateQuality($dqtList);
        $quality = '<unknown>' unless defined $quality;
        $et->FoundTag('JPEGQualityEstimate', $quality);
    }
    return unless ($all or $$et{REQ_TAG_LOOKUP}{jpegdigest}) and $subSampling;

    unless (eval { require Digest::MD5 }) {
        $et->Warn('Digest::MD5 must be installed to calculate JPEGDigest');
        return;
    }
    # create a string of DQT tables (in indexed order), separated by zero bytes
    my $dqt = '';
    my $dat;
    foreach $dat (@$dqtList) {
        next unless $dat;
        $dqt .= "\0" if $dqt;
        $dqt .= $dat;
    }
    # generate ASCII-hex string of DQT MD5 digest
    my $md5 = unpack 'H*', Digest::MD5::md5($dqt);

    # add sub-sampling string unless we get a match without it
    $md5 .= ':' . $subSampling unless $PrintConv{$md5};

    # add print conversion for JPEGDigest dynamically so it doesn't
    # bulk up the documentation and slow down loading unnecessarily
    $Image::ExifTool::Extra{JPEGDigest}{PrintConv} = \%PrintConv;
    $et->FoundTag('JPEGDigest', $md5);
}


1;  # end

__END__

=head1 NAME

Image::ExifTool::JPEGDigest - Calculate JPEGDigest and JPEGQualityEstimate

=head1 SYNOPSIS

This module is used by Image::ExifTool

=head1 DESCRIPTION

This module contains a lookup for values of the JPEG DQT digest, allowing
some image identification from JPEG data alone.  It also calculates an
estimated JPEG quality if requested.

=head1 AUTHOR

Copyright 2003-2017, Phil Harvey (phil at owl.phy.queensu.ca)

This library is free software; you can redistribute it and/or modify it
under the same terms as Perl itself.

=head1 REFERENCES

=over 4

=item L<https://github.com/ImageMagick/ImageMagick/blob/master/coders/jpeg.c>

=back

=head1 ACKNOWLEDGEMENTS

Thanks to Jens Duttke for most of the work that went into this module, and
to Franz Buchinger for the values he added.

=head1 SEE ALSO

L<Image::ExifTool::TagNames/JPEG Tags>,
L<Image::ExifTool(3pm)|Image::ExifTool>

=cut

