/*
 * Decompiled with CFR 0.152.
 */
package org.elasticsearch.percolator;

import java.util.ArrayList;
import java.util.Arrays;
import java.util.Collections;
import java.util.HashMap;
import java.util.HashSet;
import java.util.List;
import java.util.Map;
import java.util.Objects;
import java.util.Set;
import java.util.function.BiFunction;
import java.util.stream.Collectors;
import org.apache.lucene.index.PrefixCodedTerms;
import org.apache.lucene.index.Term;
import org.apache.lucene.queries.BlendedTermQuery;
import org.apache.lucene.queries.CommonTermsQuery;
import org.apache.lucene.search.BooleanClause;
import org.apache.lucene.search.BooleanQuery;
import org.apache.lucene.search.BoostQuery;
import org.apache.lucene.search.ConstantScoreQuery;
import org.apache.lucene.search.DisjunctionMaxQuery;
import org.apache.lucene.search.IndexOrDocValuesQuery;
import org.apache.lucene.search.MatchAllDocsQuery;
import org.apache.lucene.search.MatchNoDocsQuery;
import org.apache.lucene.search.MultiPhraseQuery;
import org.apache.lucene.search.PhraseQuery;
import org.apache.lucene.search.PointRangeQuery;
import org.apache.lucene.search.Query;
import org.apache.lucene.search.SynonymQuery;
import org.apache.lucene.search.TermInSetQuery;
import org.apache.lucene.search.TermQuery;
import org.apache.lucene.search.spans.SpanFirstQuery;
import org.apache.lucene.search.spans.SpanNearQuery;
import org.apache.lucene.search.spans.SpanNotQuery;
import org.apache.lucene.search.spans.SpanOrQuery;
import org.apache.lucene.search.spans.SpanQuery;
import org.apache.lucene.search.spans.SpanTermQuery;
import org.apache.lucene.util.BytesRef;
import org.apache.lucene.util.NumericUtils;
import org.elasticsearch.Version;
import org.elasticsearch.common.logging.LoggerMessageFormat;
import org.elasticsearch.common.lucene.search.function.FunctionScoreQuery;
import org.elasticsearch.index.search.ESToParentBlockJoinQuery;

final class QueryAnalyzer {
    private static final Map<Class<? extends Query>, BiFunction<Query, Version, Result>> queryProcessors;

    private QueryAnalyzer() {
    }

    static Result analyze(Query query, Version indexVersion) {
        BiFunction<Query, Version, Result> queryProcessor;
        Class<?> queryClass = query.getClass();
        if (queryClass.isAnonymousClass()) {
            queryClass = queryClass.getSuperclass();
        }
        if ((queryProcessor = queryProcessors.get(queryClass)) != null) {
            return queryProcessor.apply(query, indexVersion);
        }
        throw new UnsupportedQueryException(query);
    }

    private static BiFunction<Query, Version, Result> matchNoDocsQuery() {
        return (query, version) -> new Result(true, Collections.emptySet(), 1);
    }

    private static BiFunction<Query, Version, Result> matchAllDocsQuery() {
        return (query, version) -> new Result(true, true);
    }

    private static BiFunction<Query, Version, Result> constantScoreQuery() {
        return (query, boosts) -> {
            Query wrappedQuery = ((ConstantScoreQuery)query).getQuery();
            return QueryAnalyzer.analyze(wrappedQuery, boosts);
        };
    }

    private static BiFunction<Query, Version, Result> boostQuery() {
        return (query, version) -> {
            Query wrappedQuery = ((BoostQuery)query).getQuery();
            return QueryAnalyzer.analyze(wrappedQuery, version);
        };
    }

    private static BiFunction<Query, Version, Result> termQuery() {
        return (query, version) -> {
            TermQuery termQuery = (TermQuery)query;
            return new Result(true, Collections.singleton(new QueryExtraction(termQuery.getTerm())), 1);
        };
    }

    private static BiFunction<Query, Version, Result> termInSetQuery() {
        return (query, version) -> {
            TermInSetQuery termInSetQuery = (TermInSetQuery)query;
            HashSet<QueryExtraction> terms = new HashSet<QueryExtraction>();
            PrefixCodedTerms.TermIterator iterator = termInSetQuery.getTermData().iterator();
            BytesRef term = iterator.next();
            while (term != null) {
                terms.add(new QueryExtraction(new Term(iterator.field(), term)));
                term = iterator.next();
            }
            return new Result(true, terms, 1);
        };
    }

    private static BiFunction<Query, Version, Result> synonymQuery() {
        return (query, version) -> {
            Set<QueryExtraction> terms = ((SynonymQuery)query).getTerms().stream().map(QueryExtraction::new).collect(Collectors.toSet());
            return new Result(true, terms, 1);
        };
    }

    private static BiFunction<Query, Version, Result> commonTermsQuery() {
        return (query, version) -> {
            Set<QueryExtraction> terms = ((CommonTermsQuery)query).getTerms().stream().map(QueryExtraction::new).collect(Collectors.toSet());
            return new Result(false, terms, 1);
        };
    }

    private static BiFunction<Query, Version, Result> blendedTermQuery() {
        return (query, version) -> {
            Set<QueryExtraction> terms = ((BlendedTermQuery)query).getTerms().stream().map(QueryExtraction::new).collect(Collectors.toSet());
            return new Result(true, terms, 1);
        };
    }

    private static BiFunction<Query, Version, Result> phraseQuery() {
        return (query, version) -> {
            Term[] terms = ((PhraseQuery)query).getTerms();
            if (terms.length == 0) {
                return new Result(true, Collections.emptySet(), 1);
            }
            if (version.onOrAfter(Version.V_6_1_0)) {
                Set<QueryExtraction> extractions = Arrays.stream(terms).map(QueryExtraction::new).collect(Collectors.toSet());
                return new Result(false, extractions, extractions.size());
            }
            Term longestTerm = terms[0];
            for (Term term : terms) {
                if (longestTerm.bytes().length >= term.bytes().length) continue;
                longestTerm = term;
            }
            return new Result(false, Collections.singleton(new QueryExtraction(longestTerm)), 1);
        };
    }

    private static BiFunction<Query, Version, Result> multiPhraseQuery() {
        return (query, version) -> {
            Term[][] terms = ((MultiPhraseQuery)query).getTermArrays();
            if (terms.length == 0) {
                return new Result(true, Collections.emptySet(), 1);
            }
            if (version.onOrAfter(Version.V_6_1_0)) {
                HashSet<QueryExtraction> extractions = new HashSet<QueryExtraction>();
                for (Term[] termArr : terms) {
                    extractions.addAll(Arrays.stream(termArr).map(QueryExtraction::new).collect(Collectors.toSet()));
                }
                return new Result(false, extractions, terms.length);
            }
            Set<QueryExtraction> bestTermArr = null;
            for (Term[] termArr : terms) {
                Set<QueryExtraction> queryExtractions = Arrays.stream(termArr).map(QueryExtraction::new).collect(Collectors.toSet());
                bestTermArr = QueryAnalyzer.selectBestExtraction(bestTermArr, queryExtractions);
            }
            return new Result(false, bestTermArr, 1);
        };
    }

    private static BiFunction<Query, Version, Result> spanTermQuery() {
        return (query, version) -> {
            Term term = ((SpanTermQuery)query).getTerm();
            return new Result(true, Collections.singleton(new QueryExtraction(term)), 1);
        };
    }

    private static BiFunction<Query, Version, Result> spanNearQuery() {
        return (query, version) -> {
            SpanNearQuery spanNearQuery = (SpanNearQuery)query;
            if (version.onOrAfter(Version.V_6_1_0)) {
                Set results = Arrays.stream(spanNearQuery.getClauses()).map(clause -> QueryAnalyzer.analyze((Query)clause, version)).collect(Collectors.toSet());
                int msm = 0;
                HashSet<QueryExtraction> extractions = new HashSet<QueryExtraction>();
                HashSet<String> seenRangeFields = new HashSet<String>();
                for (Result result : results) {
                    QueryExtraction[] t = result.extractions.toArray(new QueryExtraction[1]);
                    if (result.extractions.size() == 1 && t[0].range != null) {
                        if (seenRangeFields.add(t[0].range.fieldName)) {
                            ++msm;
                        }
                    } else {
                        msm += result.minimumShouldMatch;
                    }
                    extractions.addAll(result.extractions);
                }
                return new Result(false, extractions, msm);
            }
            Set<QueryExtraction> bestClauses = null;
            for (SpanQuery clause2 : spanNearQuery.getClauses()) {
                Result temp = QueryAnalyzer.analyze((Query)clause2, version);
                bestClauses = QueryAnalyzer.selectBestExtraction(temp.extractions, bestClauses);
            }
            return new Result(false, bestClauses, 1);
        };
    }

    private static BiFunction<Query, Version, Result> spanOrQuery() {
        return (query, version) -> {
            HashSet<QueryExtraction> terms = new HashSet<QueryExtraction>();
            SpanOrQuery spanOrQuery = (SpanOrQuery)query;
            for (SpanQuery clause : spanOrQuery.getClauses()) {
                terms.addAll(QueryAnalyzer.analyze((Query)clause, (Version)version).extractions);
            }
            return new Result(false, terms, 1);
        };
    }

    private static BiFunction<Query, Version, Result> spanNotQuery() {
        return (query, version) -> {
            Result result = QueryAnalyzer.analyze((Query)((SpanNotQuery)query).getInclude(), version);
            return new Result(false, result.extractions, result.minimumShouldMatch);
        };
    }

    private static BiFunction<Query, Version, Result> spanFirstQuery() {
        return (query, version) -> {
            Result result = QueryAnalyzer.analyze((Query)((SpanFirstQuery)query).getMatch(), version);
            return new Result(false, result.extractions, result.minimumShouldMatch);
        };
    }

    private static BiFunction<Query, Version, Result> booleanQuery() {
        return (query, version) -> {
            BooleanQuery bq = (BooleanQuery)query;
            List clauses = bq.clauses();
            int minimumShouldMatch = bq.getMinimumNumberShouldMatch();
            int numRequiredClauses = 0;
            int numOptionalClauses = 0;
            int numProhibitedClauses = 0;
            for (BooleanClause clause : clauses) {
                if (clause.isRequired()) {
                    ++numRequiredClauses;
                }
                if (clause.isProhibited()) {
                    ++numProhibitedClauses;
                }
                if (clause.getOccur() != BooleanClause.Occur.SHOULD) continue;
                ++numOptionalClauses;
            }
            if (numRequiredClauses > 0) {
                if (version.onOrAfter(Version.V_6_1_0)) {
                    UnsupportedQueryException uqe = null;
                    ArrayList<Result> results = new ArrayList<Result>(numRequiredClauses);
                    for (BooleanClause clause : clauses) {
                        if (!clause.isRequired()) continue;
                        try {
                            results.add(QueryAnalyzer.analyze(clause.getQuery(), version));
                        }
                        catch (UnsupportedQueryException e) {
                            uqe = e;
                        }
                    }
                    if (results.isEmpty()) {
                        if (uqe != null) {
                            throw uqe;
                        }
                        return new Result(true, Collections.emptySet(), 1);
                    }
                    int msm = 0;
                    boolean requiredShouldClauses = minimumShouldMatch > 0 && numOptionalClauses > 0;
                    boolean verified = uqe == null && numProhibitedClauses == 0 && !requiredShouldClauses;
                    boolean matchAllDocs = true;
                    HashSet<QueryExtraction> extractions = new HashSet<QueryExtraction>();
                    HashSet<String> seenRangeFields = new HashSet<String>();
                    for (Result result : results) {
                        QueryExtraction[] t = result.extractions.toArray(new QueryExtraction[1]);
                        if (result.extractions.size() == 1 && t[0].range != null) {
                            if (seenRangeFields.add(t[0].range.fieldName)) {
                                ++msm;
                            }
                        } else {
                            msm += result.minimumShouldMatch;
                        }
                        verified &= result.verified;
                        matchAllDocs &= result.matchAllDocs;
                        extractions.addAll(result.extractions);
                    }
                    if (matchAllDocs) {
                        return new Result(matchAllDocs, verified);
                    }
                    return new Result(verified, extractions, msm);
                }
                Set<QueryExtraction> bestClause = null;
                UnsupportedQueryException uqe = null;
                for (BooleanClause clause : clauses) {
                    Result temp;
                    if (!clause.isRequired()) continue;
                    try {
                        temp = QueryAnalyzer.analyze(clause.getQuery(), version);
                    }
                    catch (UnsupportedQueryException e) {
                        uqe = e;
                        continue;
                    }
                    bestClause = QueryAnalyzer.selectBestExtraction(temp.extractions, bestClause);
                }
                if (bestClause != null) {
                    return new Result(false, bestClause, 1);
                }
                if (uqe != null) {
                    throw uqe;
                }
                return new Result(true, Collections.emptySet(), 1);
            }
            ArrayList<Query> disjunctions = new ArrayList<Query>(numOptionalClauses);
            for (BooleanClause clause : clauses) {
                if (clause.getOccur() != BooleanClause.Occur.SHOULD) continue;
                disjunctions.add(clause.getQuery());
            }
            return QueryAnalyzer.handleDisjunction(disjunctions, minimumShouldMatch, numProhibitedClauses > 0, version);
        };
    }

    private static BiFunction<Query, Version, Result> disjunctionMaxQuery() {
        return (query, version) -> {
            List disjuncts = ((DisjunctionMaxQuery)query).getDisjuncts();
            return QueryAnalyzer.handleDisjunction(disjuncts, 1, false, version);
        };
    }

    private static BiFunction<Query, Version, Result> functionScoreQuery() {
        return (query, version) -> {
            FunctionScoreQuery functionScoreQuery = (FunctionScoreQuery)query;
            Result result = QueryAnalyzer.analyze(functionScoreQuery.getSubQuery(), version);
            boolean verified = functionScoreQuery.getMinScore() == null;
            return new Result(verified, result.extractions, result.minimumShouldMatch);
        };
    }

    private static BiFunction<Query, Version, Result> pointRangeQuery() {
        return (query, version) -> {
            byte[] upperPoint;
            PointRangeQuery pointRangeQuery = (PointRangeQuery)query;
            if (pointRangeQuery.getNumDims() != 1) {
                throw new UnsupportedQueryException((Query)query);
            }
            byte[] lowerPoint = pointRangeQuery.getLowerPoint();
            if (new BytesRef(lowerPoint).compareTo(new BytesRef(upperPoint = pointRangeQuery.getUpperPoint())) > 0) {
                return new Result(true, Collections.emptySet(), 1);
            }
            byte[] interval = new byte[16];
            NumericUtils.subtract((int)16, (int)0, (byte[])QueryAnalyzer.prepad(upperPoint), (byte[])QueryAnalyzer.prepad(lowerPoint), (byte[])interval);
            return new Result(false, Collections.singleton(new QueryExtraction(new Range(pointRangeQuery.getField(), lowerPoint, upperPoint, interval))), 1);
        };
    }

    private static byte[] prepad(byte[] original) {
        int offset = 16 - original.length;
        byte[] result = new byte[16];
        System.arraycopy(original, 0, result, offset, original.length);
        return result;
    }

    private static BiFunction<Query, Version, Result> indexOrDocValuesQuery() {
        return (query, version) -> {
            IndexOrDocValuesQuery indexOrDocValuesQuery = (IndexOrDocValuesQuery)query;
            return QueryAnalyzer.analyze(indexOrDocValuesQuery.getIndexQuery(), version);
        };
    }

    private static BiFunction<Query, Version, Result> toParentBlockJoinQuery() {
        return (query, version) -> {
            ESToParentBlockJoinQuery toParentBlockJoinQuery = (ESToParentBlockJoinQuery)query;
            Result result = QueryAnalyzer.analyze(toParentBlockJoinQuery.getChildQuery(), version);
            return new Result(false, result.extractions, result.minimumShouldMatch);
        };
    }

    private static Result handleDisjunction(List<Query> disjunctions, int requiredShouldClauses, boolean otherClauses, Version version) {
        boolean verified;
        int[] msmPerClause = new int[disjunctions.size()];
        String[] rangeFieldNames = new String[disjunctions.size()];
        boolean bl = verified = !otherClauses;
        if (version.before(Version.V_6_1_0)) {
            verified &= requiredShouldClauses <= 1;
        }
        int numMatchAllClauses = 0;
        HashSet<QueryExtraction> terms = new HashSet<QueryExtraction>();
        for (int i = 0; i < disjunctions.size(); ++i) {
            Query disjunct = disjunctions.get(i);
            Result subResult = QueryAnalyzer.analyze(disjunct, version);
            verified &= subResult.verified;
            if (subResult.matchAllDocs) {
                ++numMatchAllClauses;
            }
            terms.addAll(subResult.extractions);
            QueryExtraction[] t = subResult.extractions.toArray(new QueryExtraction[1]);
            msmPerClause[i] = subResult.minimumShouldMatch;
            if (subResult.extractions.size() != 1 || t[0].range == null) continue;
            rangeFieldNames[i] = t[0].range.fieldName;
        }
        boolean matchAllDocs = numMatchAllClauses > 0 && numMatchAllClauses >= requiredShouldClauses;
        int msm = 0;
        if (version.onOrAfter(Version.V_6_1_0)) {
            HashSet<String> seenRangeFields = new HashSet<String>();
            Arrays.sort(msmPerClause);
            int limit = Math.min(msmPerClause.length, Math.max(1, requiredShouldClauses));
            for (int i = 0; i < limit; ++i) {
                if (rangeFieldNames[i] != null) {
                    if (!seenRangeFields.add(rangeFieldNames[i])) continue;
                    ++msm;
                    continue;
                }
                msm += msmPerClause[i];
            }
        } else {
            msm = 1;
        }
        if (matchAllDocs) {
            return new Result(matchAllDocs, verified);
        }
        return new Result(verified, terms, msm);
    }

    static Set<QueryExtraction> selectBestExtraction(Set<QueryExtraction> extractions1, Set<QueryExtraction> extractions2) {
        int extraction2ShortestTerm;
        assert (extractions1 != null || extractions2 != null);
        if (extractions1 == null) {
            return extractions2;
        }
        if (extractions2 == null) {
            return extractions1;
        }
        boolean onlyRangeBasedExtractions = true;
        for (QueryExtraction clause : extractions1) {
            if (clause.term == null) continue;
            onlyRangeBasedExtractions = false;
            break;
        }
        for (QueryExtraction clause : extractions2) {
            if (clause.term == null) continue;
            onlyRangeBasedExtractions = false;
            break;
        }
        if (onlyRangeBasedExtractions) {
            BytesRef extraction1SmallestRange = QueryAnalyzer.smallestRange(extractions1);
            BytesRef extraction2SmallestRange = QueryAnalyzer.smallestRange(extractions2);
            if (extraction1SmallestRange == null) {
                return extractions2;
            }
            if (extraction2SmallestRange == null) {
                return extractions1;
            }
            if (extraction1SmallestRange.compareTo(extraction2SmallestRange) <= 0) {
                return extractions1;
            }
            return extractions2;
        }
        int extraction1ShortestTerm = QueryAnalyzer.minTermLength(extractions1);
        if (extraction1ShortestTerm >= (extraction2ShortestTerm = QueryAnalyzer.minTermLength(extractions2))) {
            return extractions1;
        }
        return extractions2;
    }

    private static int minTermLength(Set<QueryExtraction> extractions) {
        if (extractions.stream().filter(queryExtraction -> queryExtraction.term != null).count() == 0L && extractions.stream().filter(queryExtraction -> queryExtraction.range != null).count() > 0L) {
            return Integer.MIN_VALUE;
        }
        int min = Integer.MAX_VALUE;
        for (QueryExtraction qt : extractions) {
            if (qt.term == null) continue;
            min = Math.min(min, qt.bytes().length);
        }
        return min;
    }

    private static BytesRef smallestRange(Set<QueryExtraction> terms) {
        BytesRef min = null;
        for (QueryExtraction qt : terms) {
            if (qt.range == null || min != null && qt.range.interval.compareTo(min) >= 0) continue;
            min = qt.range.interval;
        }
        return min;
    }

    static {
        HashMap<Class, BiFunction<Query, Version, Result>> map = new HashMap<Class, BiFunction<Query, Version, Result>>();
        map.put(MatchNoDocsQuery.class, QueryAnalyzer.matchNoDocsQuery());
        map.put(MatchAllDocsQuery.class, QueryAnalyzer.matchAllDocsQuery());
        map.put(ConstantScoreQuery.class, QueryAnalyzer.constantScoreQuery());
        map.put(BoostQuery.class, QueryAnalyzer.boostQuery());
        map.put(TermQuery.class, QueryAnalyzer.termQuery());
        map.put(TermInSetQuery.class, QueryAnalyzer.termInSetQuery());
        map.put(CommonTermsQuery.class, QueryAnalyzer.commonTermsQuery());
        map.put(BlendedTermQuery.class, QueryAnalyzer.blendedTermQuery());
        map.put(PhraseQuery.class, QueryAnalyzer.phraseQuery());
        map.put(MultiPhraseQuery.class, QueryAnalyzer.multiPhraseQuery());
        map.put(SpanTermQuery.class, QueryAnalyzer.spanTermQuery());
        map.put(SpanNearQuery.class, QueryAnalyzer.spanNearQuery());
        map.put(SpanOrQuery.class, QueryAnalyzer.spanOrQuery());
        map.put(SpanFirstQuery.class, QueryAnalyzer.spanFirstQuery());
        map.put(SpanNotQuery.class, QueryAnalyzer.spanNotQuery());
        map.put(BooleanQuery.class, QueryAnalyzer.booleanQuery());
        map.put(DisjunctionMaxQuery.class, QueryAnalyzer.disjunctionMaxQuery());
        map.put(SynonymQuery.class, QueryAnalyzer.synonymQuery());
        map.put(FunctionScoreQuery.class, QueryAnalyzer.functionScoreQuery());
        map.put(PointRangeQuery.class, QueryAnalyzer.pointRangeQuery());
        map.put(IndexOrDocValuesQuery.class, QueryAnalyzer.indexOrDocValuesQuery());
        map.put(ESToParentBlockJoinQuery.class, QueryAnalyzer.toParentBlockJoinQuery());
        queryProcessors = Collections.unmodifiableMap(map);
    }

    static class Range {
        final String fieldName;
        final byte[] lowerPoint;
        final byte[] upperPoint;
        final BytesRef interval;

        Range(String fieldName, byte[] lowerPoint, byte[] upperPoint, byte[] interval) {
            this.fieldName = fieldName;
            this.lowerPoint = lowerPoint;
            this.upperPoint = upperPoint;
            this.interval = new BytesRef(interval);
        }

        public boolean equals(Object o) {
            if (this == o) {
                return true;
            }
            if (o == null || this.getClass() != o.getClass()) {
                return false;
            }
            Range range = (Range)o;
            return Objects.equals(this.fieldName, range.fieldName) && Arrays.equals(this.lowerPoint, range.lowerPoint) && Arrays.equals(this.upperPoint, range.upperPoint);
        }

        public int hashCode() {
            int result = 1;
            result += 31 * this.fieldName.hashCode();
            result += Arrays.hashCode(this.lowerPoint);
            return result += Arrays.hashCode(this.upperPoint);
        }

        public String toString() {
            return "Range{, fieldName='" + this.fieldName + '\'' + ", interval=" + this.interval + '}';
        }
    }

    static class UnsupportedQueryException
    extends RuntimeException {
        private final Query unsupportedQuery;

        UnsupportedQueryException(Query unsupportedQuery) {
            super(LoggerMessageFormat.format((String)"no query terms can be extracted from query [{}]", (Object[])new Object[]{unsupportedQuery}));
            this.unsupportedQuery = unsupportedQuery;
        }

        Query getUnsupportedQuery() {
            return this.unsupportedQuery;
        }
    }

    static class QueryExtraction {
        final Term term;
        final Range range;

        QueryExtraction(Term term) {
            this.term = term;
            this.range = null;
        }

        QueryExtraction(Range range) {
            this.term = null;
            this.range = range;
        }

        String field() {
            return this.term != null ? this.term.field() : null;
        }

        BytesRef bytes() {
            return this.term != null ? this.term.bytes() : null;
        }

        String text() {
            return this.term != null ? this.term.text() : null;
        }

        public boolean equals(Object o) {
            if (this == o) {
                return true;
            }
            if (o == null || this.getClass() != o.getClass()) {
                return false;
            }
            QueryExtraction queryExtraction = (QueryExtraction)o;
            return Objects.equals(this.term, queryExtraction.term) && Objects.equals(this.range, queryExtraction.range);
        }

        public int hashCode() {
            return Objects.hash(this.term, this.range);
        }

        public String toString() {
            return "QueryExtraction{term=" + this.term + ",range=" + this.range + '}';
        }
    }

    static class Result {
        final Set<QueryExtraction> extractions;
        final boolean verified;
        final int minimumShouldMatch;
        final boolean matchAllDocs;

        Result(boolean verified, Set<QueryExtraction> extractions, int minimumShouldMatch) {
            this.extractions = extractions;
            this.verified = verified;
            this.minimumShouldMatch = minimumShouldMatch;
            this.matchAllDocs = false;
        }

        Result(boolean matchAllDocs, boolean verified) {
            this.extractions = Collections.emptySet();
            this.verified = verified;
            this.minimumShouldMatch = 0;
            this.matchAllDocs = matchAllDocs;
        }
    }
}

