/**
 * Nextcloud - contacts
 *
 * This file is licensed under the Affero General Public License version 3 or
 * later. See the COPYING file.
 *
 * @author Hendrik Leppelsack <hendrik@leppelsack.de>
 * @copyright Hendrik Leppelsack 2015
 */

angular.module('contactsApp', ['uuid4', 'angular-cache', 'ngRoute', 'ui.bootstrap', 'ui.select', 'ngSanitize', 'angular-click-outside', 'ngclipboard'])
.config(['$routeProvider', function($routeProvider) {

	$routeProvider.when('/:gid', {
		template: '<contactdetails></contactdetails>'
	});

	$routeProvider.when('/contact/:uid', {
		redirectTo: function(parameters) {
			return '/' + t('contacts', 'All contacts') + '/' + parameters.uid;
		}
	});

	$routeProvider.when('/:gid/:uid', {
		template: '<contactdetails></contactdetails>'
	});

	$routeProvider.otherwise('/' + t('contacts', 'All contacts'));

}]);

angular.module('contactsApp')
.directive('datepicker', ['$timeout', function($timeout) {
	var loadDatepicker = function (scope, element, attrs, ngModelCtrl) {
		$timeout(function() {
			element.datepicker({
				dateFormat:'yy-mm-dd',
				minDate: null,
				maxDate: null,
				constrainInput: false,
				onSelect:function (date, dp) {
					if (dp.selectedYear < 1000) {
						date = '0' + date;
					}
					if (dp.selectedYear < 100) {
						date = '0' + date;
					}
					if (dp.selectedYear < 10) {
						date = '0' + date;
					}
					ngModelCtrl.$setViewValue(date);
					scope.$apply();
				}
			});
		});
	};
	return {
		restrict: 'A',
		require : 'ngModel',
		transclude: true,
		link : loadDatepicker
	};
}]);

angular.module('contactsApp')
.directive('focusExpression', ['$timeout', function ($timeout) {
	return {
		restrict: 'A',
		link: {
			post: function postLink(scope, element, attrs) {
				scope.$watch(attrs.focusExpression, function () {
					if (attrs.focusExpression) {
						if (scope.$eval(attrs.focusExpression)) {
							$timeout(function () {
								if (element.is('input')) {
									element.focus();
								} else {
									element.find('input').focus();
								}
							}, 100); //need some delay to work with ng-disabled
						}
					}
				});
			}
		}
	};
}]);

angular.module('contactsApp')
.directive('inputresize', function() {
	return {
		restrict: 'A',
		link : function (scope, element) {
			var elInput = element.val();
			element.bind('keydown keyup load focus', function() {
				elInput = element.val();
				// If set to 0, the min-width css data is ignored
				var length = elInput.length > 1 ? elInput.length : 1;
				element.attr('size', length);
			});
		}
	};
});

angular.module('contactsApp')
.directive('selectExpression', ['$timeout', function ($timeout) {
	return {
		restrict: 'A',
		link: {
			post: function postLink(scope, element, attrs) {
				scope.$watch(attrs.selectExpression, function () {
					if (attrs.selectExpression) {
						if (scope.$eval(attrs.selectExpression)) {
							$timeout(function () {
								if (element.is('input')) {
									element.select();
								} else {
									element.find('input').select();
								}
							}, 100); //need some delay to work with ng-disabled
						}
					}
				});
			}
		}
	};
}]);

angular.module('contactsApp')
.controller('addressbookCtrl', ['$scope', 'AddressBookService', function($scope, AddressBookService) {
	var ctrl = this;

	ctrl.t = {
		download: t('contacts', 'Download'),
		copyURL: t('contacts', 'Copy link'),
		clickToCopy: t('contacts', 'Click to copy the link to your clipboard'),
		shareAddressbook: t('contacts', 'Toggle sharing'),
		deleteAddressbook: t('contacts', 'Delete'),
		renameAddressbook: t('contacts', 'Rename'),
		shareInputPlaceHolder: t('contacts', 'Share with users or groups'),
		delete: t('contacts', 'Delete'),
		canEdit: t('contacts', 'can edit'),
		close: t('contacts', 'Close'),
		enabled: t('contacts', 'Enabled'),
		disabled: t('contacts', 'Disabled')
	};

	ctrl.editing = false;
	ctrl.enabled = ctrl.addressBook.enabled;

	ctrl.tooltipIsOpen = false;
	ctrl.tooltipTitle = ctrl.t.clickToCopy;
	ctrl.showInputUrl = false;

	ctrl.clipboardSuccess = function() {
		ctrl.tooltipIsOpen = true;
		ctrl.tooltipTitle = t('core', 'Copied!');
		_.delay(function() {
			ctrl.tooltipIsOpen = false;
			ctrl.tooltipTitle = ctrl.t.clickToCopy;
		}, 3000);
	};

	ctrl.clipboardError = function() {
		ctrl.showInputUrl = true;
		if (/iPhone|iPad/i.test(navigator.userAgent)) {
			ctrl.InputUrlTooltip = t('core', 'Not supported!');
		} else if (/Mac/i.test(navigator.userAgent)) {
			ctrl.InputUrlTooltip = t('core', 'Press ⌘-C to copy.');
		} else {
			ctrl.InputUrlTooltip = t('core', 'Press Ctrl-C to copy.');
		}
		$('#addressBookUrl_'+ctrl.addressBook.ctag).select();
	};

	ctrl.renameAddressBook = function() {
		AddressBookService.rename(ctrl.addressBook, ctrl.addressBook.displayName);
		ctrl.editing = false;
	};

	ctrl.edit = function() {
		ctrl.editing = true;
	};

	ctrl.closeMenus = function() {
		$scope.$parent.ctrl.openedMenu = false;
	};

	ctrl.openMenu = function(index) {
		ctrl.closeMenus();
		$scope.$parent.ctrl.openedMenu = index;
	};

	ctrl.toggleMenu = function(index) {
		if ($scope.$parent.ctrl.openedMenu === index) {
			ctrl.closeMenus();
		} else {
			ctrl.openMenu(index);
		}
	};

	ctrl.toggleSharesEditor = function() {
		ctrl.editingShares = !ctrl.editingShares;
		ctrl.selectedSharee = null;
	};

	/* From Calendar-Rework - js/app/controllers/calendarlistcontroller.js */
	ctrl.findSharee = function (val) {
		return $.get(
			OC.linkToOCS('apps/files_sharing/api/v1') + 'sharees',
			{
				format: 'json',
				search: val.trim(),
				perPage: 200,
				itemType: 'principals'
			}
		).then(function(result) {
			var users   = result.ocs.data.exact.users.concat(result.ocs.data.users);
			var groups  = result.ocs.data.exact.groups.concat(result.ocs.data.groups);

			var userShares = ctrl.addressBook.sharedWith.users;
			var userSharesLength = userShares.length;

			var groupsShares = ctrl.addressBook.sharedWith.groups;
			var groupsSharesLength = groupsShares.length;
			var i, j;

			// Filter out current user
			for (i = 0 ; i < users.length; i++) {
				if (users[i].value.shareWith === OC.currentUser) {
					users.splice(i, 1);
					break;
				}
			}

			// Now filter out all sharees that are already shared with
			for (i = 0; i < userSharesLength; i++) {
				var shareUser = userShares[i];
				for (j = 0; j < users.length; j++) {
					if (users[j].value.shareWith === shareUser.id) {
						users.splice(j, 1);
						break;
					}
				}
			}

			// Now filter out all groups that are already shared with
			for (i = 0; i < groupsSharesLength; i++) {
				var sharedGroup = groupsShares[i];
				for (j = 0; j < groups.length; j++) {
					if (groups[j].value.shareWith === sharedGroup.id) {
						groups.splice(j, 1);
						break;
					}
				}
			}

			// Combine users and groups
			users = users.map(function(item) {
				return {
					display: item.value.shareWith,
					type: OC.Share.SHARE_TYPE_USER,
					identifier: item.value.shareWith
				};
			});

			groups = groups.map(function(item) {
				return {
					display: item.value.shareWith + ' (group)',
					type: OC.Share.SHARE_TYPE_GROUP,
					identifier: item.value.shareWith
				};
			});

			return groups.concat(users);
		});
	};

	ctrl.onSelectSharee = function (item) {
		// Prevent settings to slide down
		$('#app-settings-header > button').data('apps-slide-toggle', false);
		_.delay(function() {
			$('#app-settings-header > button').data('apps-slide-toggle', '#app-settings-content');
		}, 500);

		ctrl.selectedSharee = null;
		AddressBookService.share(ctrl.addressBook, item.type, item.identifier, false, false).then(function() {
			$scope.$apply();
		});

	};

	ctrl.updateExistingUserShare = function(userId, writable) {
		AddressBookService.share(ctrl.addressBook, OC.Share.SHARE_TYPE_USER, userId, writable, true).then(function() {
			$scope.$apply();
		});
	};

	ctrl.updateExistingGroupShare = function(groupId, writable) {
		AddressBookService.share(ctrl.addressBook, OC.Share.SHARE_TYPE_GROUP, groupId, writable, true).then(function() {
			$scope.$apply();
		});
	};

	ctrl.unshareFromUser = function(userId) {
		AddressBookService.unshare(ctrl.addressBook, OC.Share.SHARE_TYPE_USER, userId).then(function() {
			$scope.$apply();
		});
	};

	ctrl.unshareFromGroup = function(groupId) {
		AddressBookService.unshare(ctrl.addressBook, OC.Share.SHARE_TYPE_GROUP, groupId).then(function() {
			$scope.$apply();
		});
	};

	ctrl.deleteAddressBook = function() {
		AddressBookService.delete(ctrl.addressBook).then(function() {
			$scope.$apply();
		});
	};

	ctrl.toggleState = function() {
		AddressBookService.toggleState(ctrl.addressBook).then(function(addressBook) {
			ctrl.enabled = addressBook.enabled;
			$scope.$apply();
		});
	};

}]);

angular.module('contactsApp')
.directive('addressbook', function() {
	return {
		restrict: 'A', // has to be an attribute to work with core css
		scope: {},
		controller: 'addressbookCtrl',
		controllerAs: 'ctrl',
		bindToController: {
			addressBook: '=data',
			list: '='
		},
		templateUrl: OC.linkTo('contacts', 'templates/addressBook.html')
	};
});

angular.module('contactsApp')
.controller('addressbooklistCtrl', ['$scope', 'AddressBookService', function($scope, AddressBookService) {
	var ctrl = this;

	ctrl.loading = true;
	ctrl.openedMenu = false;
	ctrl.addressBookRegex = /^[a-zA-Z0-9À-ÿ\s-_.!?#|()]+$/i;

	AddressBookService.getAll().then(function(addressBooks) {
		ctrl.addressBooks = addressBooks;
		ctrl.loading = false;
		if(ctrl.addressBooks.length === 0) {
			AddressBookService.create(t('contacts', 'Contacts')).then(function() {
				AddressBookService.getAddressBook(t('contacts', 'Contacts')).then(function(addressBook) {
					ctrl.addressBooks.push(addressBook);
					$scope.$apply();
				});
			});
		}
	});

	ctrl.t = {
		addressBookName : t('contacts', 'Address book name'),
		regexError : t('contacts', 'Only these special characters are allowed: -_.!?#|()')
	};

	ctrl.createAddressBook = function() {
		if(ctrl.newAddressBookName) {
			AddressBookService.create(ctrl.newAddressBookName).then(function() {
				AddressBookService.getAddressBook(ctrl.newAddressBookName).then(function(addressBook) {
					ctrl.addressBooks.push(addressBook);
					$scope.$apply();
				});
			}).catch(function() {
				OC.Notification.showTemporary(t('contacts', 'Address book could not be created.'));
			});
		}
	};
}]);

angular.module('contactsApp')
.directive('addressbooklist', function() {
	return {
		restrict: 'EA', // has to be an attribute to work with core css
		scope: {},
		controller: 'addressbooklistCtrl',
		controllerAs: 'ctrl',
		bindToController: {},
		templateUrl: OC.linkTo('contacts', 'templates/addressBookList.html')
	};
});

angular.module('contactsApp')
.controller('avatarCtrl', ['ContactService', function(ContactService) {
	var ctrl = this;

	ctrl.import = ContactService.import.bind(ContactService);

	ctrl.removePhoto = function() {
		ctrl.contact.removeProperty('photo', ctrl.contact.getProperty('photo'));
		ContactService.update(ctrl.contact);
		$('avatar').removeClass('maximized');
	};

	ctrl.downloadPhoto = function() {
		/* globals ArrayBuffer, Uint8Array */
		var img = document.getElementById('contact-avatar');
		// atob to base64_decode the data-URI
		var imageSplit = img.src.split(',');
		// "data:image/png;base64" -> "png"
		var extension = '.' + imageSplit[0].split(';')[0].split('/')[1];
		var imageData = atob(imageSplit[1]);
		// Use typed arrays to convert the binary data to a Blob
		var arrayBuffer = new ArrayBuffer(imageData.length);
		var view = new Uint8Array(arrayBuffer);
		for (var i=0; i<imageData.length; i++) {
			view[i] = imageData.charCodeAt(i) & 0xff;
		}
		var blob = new Blob([arrayBuffer], {type: 'application/octet-stream'});

		// Use the URL object to create a temporary URL
		var url = (window.webkitURL || window.URL).createObjectURL(blob);

		var a = document.createElement('a');
		document.body.appendChild(a);
		a.style = 'display: none';
		a.href = url;
		a.download = ctrl.contact.uid() + extension;
		a.click();
		window.URL.revokeObjectURL(url);
		a.remove();
	};

	ctrl.openPhoto = function() {
		$('avatar').toggleClass('maximized');
	};

	// Quit avatar preview
	$('avatar').click(function() {
		$('avatar').removeClass('maximized');
	});
	$('avatar img, avatar .avatar-options').click(function(e) {
		e.stopPropagation();
	});
	$(document).keyup(function(e) {
		if (e.keyCode === 27) {
			$('avatar').removeClass('maximized');
		}
	});

}]);

angular.module('contactsApp')
.directive('avatar', ['ContactService', function(ContactService) {
	return {
		scope: {
			contact: '=data'
		},
		controller: 'avatarCtrl',
		controllerAs: 'ctrl',
		bindToController: {
			contact: '=data'
		},
		link: function(scope, element) {
			var input = element.find('input');
			input.bind('change', function() {
				var file = input.get(0).files[0];
				if (file.size > 1024*1024) { // 1 MB
					OC.Notification.showTemporary(t('contacts', 'The selected image is too big (max 1MB)'));
				} else {
					var reader = new FileReader();

					reader.addEventListener('load', function () {
						scope.$apply(function() {
							scope.contact.photo(reader.result);
							ContactService.update(scope.contact);
						});
					}, false);

					if (file) {
						reader.readAsDataURL(file);
					}
				}
			});
		},
		templateUrl: OC.linkTo('contacts', 'templates/avatar.html')
	};
}]);

angular.module('contactsApp')
.controller('contactCtrl', ['$route', '$routeParams', 'SortByService', function($route, $routeParams, SortByService) {
	var ctrl = this;

	ctrl.t = {
		errorMessage : t('contacts', 'This card is corrupted and has been fixed. Please check the data and trigger a save to make the changes permanent.'),
	};

	ctrl.getName = function() {
		// If lastName equals to firstName then none of them is set
		if (ctrl.contact.lastName() === ctrl.contact.firstName()) {
			return ctrl.contact.displayName();
		}

		if (SortByService.getSortByKey() === 'sortLastName') {
			return (
				ctrl.contact.lastName() + ', '
				+ ctrl.contact.firstName() + ' '
				+ ctrl.contact.additionalNames()
			).trim();
		}

		if (SortByService.getSortByKey() === 'sortFirstName') {
			return (
				ctrl.contact.firstName() + ' '
				+ ctrl.contact.additionalNames() + ' '
				+ ctrl.contact.lastName()
			).trim();
		}

		return ctrl.contact.displayName();
	};
}]);

angular.module('contactsApp')
.directive('contact', function() {
	return {
		scope: {},
		controller: 'contactCtrl',
		controllerAs: 'ctrl',
		bindToController: {
			contact: '=data'
		},
		templateUrl: OC.linkTo('contacts', 'templates/contact.html')
	};
});

angular.module('contactsApp')
.controller('contactdetailsCtrl', ['ContactService', 'AddressBookService', 'vCardPropertiesService', '$route', '$routeParams', '$scope', function(ContactService, AddressBookService, vCardPropertiesService, $route, $routeParams, $scope) {

	var ctrl = this;

	ctrl.init = true;
	ctrl.loading = false;
	ctrl.show = false;

	ctrl.clearContact = function() {
		$route.updateParams({
			gid: $routeParams.gid,
			uid: undefined
		});
		ctrl.show = false;
		ctrl.contact = undefined;
	};

	ctrl.uid = $routeParams.uid;
	ctrl.t = {
		noContacts : t('contacts', 'No contacts in here'),
		placeholderName : t('contacts', 'Name'),
		placeholderOrg : t('contacts', 'Organization'),
		placeholderTitle : t('contacts', 'Title'),
		selectField : t('contacts', 'Add field …'),
		download : t('contacts', 'Download'),
		delete : t('contacts', 'Delete'),
		save : t('contacts', 'Save changes'),
		addressBook : t('contacts', 'Address book'),
		loading : t('contacts', 'Loading contacts …')
	};

	ctrl.fieldDefinitions = vCardPropertiesService.fieldDefinitions;
	ctrl.focus = undefined;
	ctrl.field = undefined;
	ctrl.addressBooks = [];

	AddressBookService.getAll().then(function(addressBooks) {
		ctrl.addressBooks = addressBooks;

		if (!angular.isUndefined(ctrl.contact)) {
			ctrl.addressBook = _.find(ctrl.addressBooks, function(book) {
				return book.displayName === ctrl.contact.addressBookId;
			});
		}
		ctrl.init = false;
		// Start watching for ctrl.uid when we have addressBooks, as they are needed for fetching
		// full details.
		$scope.$watch('ctrl.uid', function(newValue) {
			ctrl.changeContact(newValue);
		});
	});


	ctrl.changeContact = function(uid) {
		if (typeof uid === 'undefined') {
			ctrl.show = false;
			$('#app-navigation-toggle').removeClass('showdetails');
			return;
		}
		ctrl.loading = true;
		ContactService.getById(ctrl.addressBooks, uid).then(function(contact) {
			if (angular.isUndefined(contact)) {
				ctrl.clearContact();
				return;
			}
			ctrl.contact = contact;
			ctrl.show = true;
			ctrl.loading = false;
			$('#app-navigation-toggle').addClass('showdetails');

			ctrl.addressBook = _.find(ctrl.addressBooks, function(book) {
				return book.displayName === ctrl.contact.addressBookId;
			});
		});
	};

	ctrl.deleteContact = function() {
		ContactService.delete(ctrl.addressBook, ctrl.contact);
	};

	ctrl.addField = function(field) {
		var defaultValue = vCardPropertiesService.getMeta(field).defaultValue || {value: ''};
		ctrl.contact.addProperty(field, defaultValue);
		ctrl.focus = field;
		ctrl.field = '';
	};

	ctrl.deleteField = function (field, prop) {
		ctrl.contact.removeProperty(field, prop);
		ctrl.focus = undefined;
	};

	ctrl.changeAddressBook = function (addressBook, oldAddressBook) {
		ContactService.moveContact(ctrl.contact, addressBook, oldAddressBook);
	};
}]);

angular.module('contactsApp')
.directive('contactdetails', function() {
	return {
		priority: 1,
		scope: {},
		controller: 'contactdetailsCtrl',
		controllerAs: 'ctrl',
		bindToController: {},
		templateUrl: OC.linkTo('contacts', 'templates/contactDetails.html')
	};
});

angular.module('contactsApp')
.controller('contactfilterCtrl', function() {
	// eslint-disable-next-line no-unused-vars
	var ctrl = this;
});

angular.module('contactsApp')
.directive('contactFilter', function() {
	return {
		restrict: 'A', // has to be an attribute to work with core css
		scope: {},
		controller: 'contactfilterCtrl',
		controllerAs: 'ctrl',
		bindToController: {
			contactFilter: '=contactFilter'
		},
		templateUrl: OC.linkTo('contacts', 'templates/contactFilter.html')
	};
});

angular.module('contactsApp')
.controller('contactimportCtrl', ['ContactService', 'AddressBookService', '$timeout', '$scope', function(ContactService, AddressBookService, $timeout, $scope) {
	var ctrl = this;

	ctrl.t = {
		importText : t('contacts', 'Import into'),
		importingText : t('contacts', 'Importing...'),
		selectAddressbook : t('contacts', 'Select your addressbook'),
		importdisabled : t('contacts', 'Import is disabled because no writable address book had been found.')
	};

	ctrl.import = ContactService.import.bind(ContactService);
	ctrl.loading = true;
	ctrl.importText = ctrl.t.importText;
	ctrl.importing = false;
	ctrl.loadingClass = 'icon-upload';

	AddressBookService.getAll().then(function(addressBooks) {
		ctrl.addressBooks = addressBooks;
		ctrl.loading = false;
		ctrl.selectedAddressBook = AddressBookService.getDefaultAddressBook();
	});

	AddressBookService.registerObserverCallback(function() {
		$timeout(function() {
			$scope.$apply(function() {
				ctrl.selectedAddressBook = AddressBookService.getDefaultAddressBook();
			});
		});
	});

	ctrl.stopHideMenu = function(isOpen) {
		if(isOpen) {
			// disabling settings bind
			$('#app-settings-header > button').data('apps-slide-toggle', false);
		} else {
			// reenabling it
			$('#app-settings-header > button').data('apps-slide-toggle', '#app-settings-content');
		}
	};

}]);

angular.module('contactsApp')
.directive('contactimport', ['ContactService', 'ImportService', '$rootScope', function(ContactService, ImportService, $rootScope) {
	return {
		link: function(scope, element, attrs, ctrl) {
			var input = element.find('input');
			input.bind('change', function() {
				angular.forEach(input.get(0).files, function(file) {
					var reader = new FileReader();

					reader.addEventListener('load', function () {
						scope.$apply(function () {
							// Indicate the user we started something
							ctrl.importText = ctrl.t.importingText;
							ctrl.loadingClass = 'icon-loading-small';
							ctrl.importing = true;
							$rootScope.importing = true;

							ContactService.import.call(ContactService, reader.result, file.type, ctrl.selectedAddressBook, function (progress, user) {
								if (progress === 1) {
									ctrl.importText = ctrl.t.importText;
									ctrl.loadingClass = 'icon-upload';
									ctrl.importing = false;
									$rootScope.importing = false;
									ImportService.importPercent = 0;
									ImportService.importing = false;
									ImportService.importedUser = '';
									ImportService.selectedAddressBook = '';
								} else {
									// Ugly hack, hide sidebar on import & mobile
									// Simulate click since we can't directly access snapper
									if($(window).width() <= 768 && $('body').hasClass('snapjs-left')) {
										$('#app-navigation-toggle').click();
										$('body').removeClass('snapjs-left');
									}

									ImportService.importPercent = parseInt(Math.floor(progress * 100));
									ImportService.importing = true;
									ImportService.importedUser = user;
									ImportService.selectedAddressBook = ctrl.selectedAddressBook.displayName;
								}
								scope.$apply();

								/* Broadcast service update */
								$rootScope.$broadcast('importing', true);
							});
						});
					}, false);

					if (file) {
						reader.readAsText(file);
					}
				});
				input.get(0).value = '';
			});
		},
		templateUrl: OC.linkTo('contacts', 'templates/contactImport.html'),
		controller: 'contactimportCtrl',
		controllerAs: 'ctrl'
	};
}]);

angular.module('contactsApp')
.controller('contactlistCtrl', ['$scope', '$filter', '$route', '$routeParams', '$timeout', 'AddressBookService', 'ContactService', 'SortByService', 'vCardPropertiesService', 'SearchService', function($scope, $filter, $route, $routeParams, $timeout, AddressBookService, ContactService, SortByService, vCardPropertiesService, SearchService) {
	var ctrl = this;

	ctrl.routeParams = $routeParams;

	ctrl.filteredContacts = []; // the displayed contacts list
	ctrl.searchTerm = '';
	ctrl.show = true;
	ctrl.invalid = false;
	ctrl.limitTo = 25;

	ctrl.sortBy = SortByService.getSortBy();

	ctrl.t = {
		emptySearch : t('contacts', 'No search result for {query}', {query: ctrl.searchTerm})
	};

	ctrl.resetLimitTo = function () {
		ctrl.limitTo = 25;
		clearInterval(ctrl.intervalId);
		ctrl.intervalId = setInterval(
			function () {
				if (!ctrl.loading && ctrl.contactList && ctrl.contactList.length > ctrl.limitTo) {
					ctrl.limitTo += 25;
					$scope.$apply();
				}
			}, 300);
	};

	$scope.query = function(contact) {
		return contact.matches(SearchService.getSearchTerm());
	};

	SortByService.subscribe(function(newValue) {
		ctrl.sortBy = newValue;
	});

	SearchService.registerObserverCallback(function(ev) {
		if (ev.event === 'submitSearch') {
			var uid = !_.isEmpty(ctrl.filteredContacts) ? ctrl.filteredContacts[0].uid() : undefined;
			ctrl.setSelectedId(uid);
			$scope.$apply();
		}
		if (ev.event === 'changeSearch') {
			ctrl.resetLimitTo();
			ctrl.searchTerm = ev.searchTerm;
			ctrl.t.emptySearch = t('contacts',
								   'No search result for {query}',
								   {query: ctrl.searchTerm}
								  );
			$scope.$apply();
		}
	});

	ctrl.loading = true;

	ContactService.registerObserverCallback(function(ev) {
		/* after import at first refresh the contactList */
		if (ev.event === 'importend') {
			$scope.$apply(function() {
				ctrl.contactList = ev.contacts;
			});
		}
		/* update route parameters */
		$timeout(function() {
			$scope.$apply(function() {
				switch(ev.event) {
				case 'delete':
					ctrl.selectNearestContact(ev.uid);
					break;
				case 'create':
					$route.updateParams({
						gid: $routeParams.gid,
						uid: ev.uid
					});
					break;
				case 'importend':
					/* after import select 'All contacts' group and first contact */
					$route.updateParams({
						gid: t('contacts', 'All contacts'),
						uid: ctrl.filteredContacts.length !== 0 ? ctrl.filteredContacts[0].uid() : undefined
					});
					return;
				case 'getFullContacts' || 'update':
					break;
				default:
					// unknown event -> leave callback without action
					return;
				}
				ctrl.contactList = ev.contacts;
			});
		});
	});

	AddressBookService.registerObserverCallback(function(ev) {
		$timeout(function() {
			$scope.$apply(function() {
				switch (ev.event) {
				case 'delete':
				case 'disable':
					ctrl.loading = true;
					ContactService.removeContactsFromAddressbook(ev.addressBook, function() {
						ContactService.getAll().then(function(contacts) {
							ctrl.contactList = contacts;
							ctrl.loading = false;
							// Only change contact if the selectd one is not in the list anymore
							if(ctrl.contactList.findIndex(function(contact) {
								return contact.uid() === ctrl.getSelectedId();
							}) === -1) {
								ctrl.selectNearestContact(ctrl.getSelectedId());
							}
						});
					});
					break;
				case 'enable':
					ctrl.loading = true;
					ContactService.appendContactsFromAddressbook(ev.addressBook, function() {
						ContactService.getAll().then(function(contacts) {
							ctrl.contactList = contacts;
							ctrl.loading = false;
						});
					});
					break;
				default:
						// unknown event -> leave callback without action
					return;

				}
			});
		});
	});

	// Get contacts
	ContactService.getAll().then(function(contacts) {
		if(contacts.length>0) {
			$scope.$apply(function() {
				ctrl.contactList = contacts;
			});
		} else {
			ctrl.loading = false;
		}
	});

	var getVisibleContacts = function() {
		var scrolled = $('.app-content-list').scrollTop();
		var elHeight = $('.contacts-list').children().outerHeight(true);
		var listHeight = $('.app-content-list').height();

		var topContact = Math.round(scrolled/elHeight);
		var contactsCount = Math.round(listHeight/elHeight);

		return ctrl.filteredContacts.slice(topContact-1, topContact+contactsCount+1);
	};

	var timeoutId = null;
	document.querySelector('.app-content-list').addEventListener('scroll', function () {
		clearTimeout(timeoutId);
		timeoutId = setTimeout(function () {
			var contacts = getVisibleContacts();
			ContactService.getFullContacts(contacts);
		}, 250);
	});

	// Wait for ctrl.filteredContacts to be updated, load the contact requested in the URL if any, and
	// load full details for the probably initially visible contacts.
	// Then kill the watch.
	var unbindListWatch = $scope.$watch('ctrl.filteredContacts', function() {
		if(ctrl.filteredContacts && ctrl.filteredContacts.length > 0) {
			// Check if a specific uid is requested
			if($routeParams.uid && $routeParams.gid) {
				ctrl.filteredContacts.forEach(function(contact) {
					if(contact.uid() === $routeParams.uid) {
						ctrl.setSelectedId($routeParams.uid);
						ctrl.loading = false;
					}
				});
			}
			// No contact previously loaded, let's load the first of the list if not in mobile mode
			if(ctrl.loading && $(window).width() > 768) {
				ctrl.setSelectedId(ctrl.filteredContacts[0].uid());
			}
			// Get full data for the first 20 contacts of the list
			ContactService.getFullContacts(ctrl.filteredContacts.slice(0, 20));
			ctrl.loading = false;
			unbindListWatch();
		}
	});

	$scope.$watch('ctrl.routeParams.uid', function(newValue, oldValue) {
		// Used for mobile view to clear the url
		if(typeof oldValue != 'undefined' && typeof newValue == 'undefined' && $(window).width() <= 768) {
			// no contact selected
			ctrl.show = true;
			return;
		}
		if(newValue === undefined) {
			// we might have to wait until ng-repeat filled the contactList
			if(ctrl.filteredContacts && ctrl.filteredContacts.length > 0) {
				$route.updateParams({
					gid: $routeParams.gid,
					uid: ctrl.filteredContacts[0].uid()
				});
			} else {
				// watch for next contactList update
				var unbindWatch = $scope.$watch('ctrl.filteredContacts', function() {
					if(ctrl.filteredContacts && ctrl.filteredContacts.length > 0) {
						$route.updateParams({
							gid: $routeParams.gid,
							uid: ctrl.filteredContacts[0].uid()
						});
					}
					unbindWatch(); // unbind as we only want one update
				});
			}
		} else {
			// displaying contact details
			ctrl.show = false;
		}
	});

	$scope.$watch('ctrl.routeParams.gid', function() {
		// we might have to wait until ng-repeat filled the contactList
		ctrl.filteredContacts = [];
		ctrl.resetLimitTo();
		// not in mobile mode
		if($(window).width() > 768) {
			// watch for next contactList update
			var unbindWatch = $scope.$watch('ctrl.filteredContacts', function() {
				if(ctrl.filteredContacts && ctrl.filteredContacts.length > 0) {
					$route.updateParams({
						gid: $routeParams.gid,
						uid: $routeParams.uid || ctrl.filteredContacts[0].uid()
					});
				}
				unbindWatch(); // unbind as we only want one update
			});
		}
	});

	// Watch if we have an invalid contact
	$scope.$watch('ctrl.filteredContacts[0].displayName()', function(displayName) {
		ctrl.invalid = (displayName === '');
	});

	ctrl.hasContacts = function () {
		if (!ctrl.contactList) {
			return false;
		}
		return ctrl.contactList.length > 0;
	};

	ctrl.setSelectedId = function (contactId) {
		$route.updateParams({
			uid: contactId
		});
	};

	ctrl.getSelectedId = function() {
		return $routeParams.uid;
	};

	ctrl.selectNearestContact = function(contactId) {
		if (ctrl.filteredContacts.length === 1) {
			$route.updateParams({
				gid: $routeParams.gid,
				uid: undefined
			});
		} else {
			for (var i = 0, length = ctrl.filteredContacts.length; i < length; i++) {
				// Get nearest contact
				if (ctrl.filteredContacts[i].uid() === contactId) {
					$route.updateParams({
						gid: $routeParams.gid,
						uid: (ctrl.filteredContacts[i+1]) ? ctrl.filteredContacts[i+1].uid() : ctrl.filteredContacts[i-1].uid()
					});
					break;
				}
			}
		}
	};

}]);

angular.module('contactsApp')
.directive('contactlist', function() {
	return {
		priority: 1,
		scope: {},
		controller: 'contactlistCtrl',
		controllerAs: 'ctrl',
		bindToController: {
			addressbook: '=adrbook'
		},
		templateUrl: OC.linkTo('contacts', 'templates/contactList.html')
	};
});

angular.module('contactsApp')
.controller('detailsItemCtrl', ['$templateRequest', '$filter', 'vCardPropertiesService', 'ContactService', function($templateRequest, $filter, vCardPropertiesService, ContactService) {
	var ctrl = this;

	ctrl.meta = vCardPropertiesService.getMeta(ctrl.name);
	ctrl.type = undefined;
	ctrl.isPreferred = false;
	ctrl.t = {
		poBox : t('contacts', 'Post office box'),
		postalCode : t('contacts', 'Postal code'),
		city : t('contacts', 'City'),
		state : t('contacts', 'State or province'),
		country : t('contacts', 'Country'),
		address: t('contacts', 'Address'),
		newGroup: t('contacts', '(new group)'),
		familyName: t('contacts', 'Last name'),
		firstName: t('contacts', 'First name'),
		additionalNames: t('contacts', 'Additional names'),
		honorificPrefix: t('contacts', 'Prefix'),
		honorificSuffix: t('contacts', 'Suffix'),
		delete: t('contacts', 'Delete')
	};

	ctrl.availableOptions = ctrl.meta.options || [];
	if (!_.isUndefined(ctrl.data) && !_.isUndefined(ctrl.data.meta) && !_.isUndefined(ctrl.data.meta.type)) {
		// parse type of the property
		var array = ctrl.data.meta.type[0].split(',');
		array = array.map(function (elem) {
			return elem.trim().replace(/\/+$/, '').replace(/\\+$/, '').trim().toUpperCase();
		});
		// the pref value is handled on its own so that we can add some favorite icon to the ui if we want
		if (array.indexOf('PREF') >= 0) {
			ctrl.isPreferred = true;
			array.splice(array.indexOf('PREF'), 1);
		}
		// simply join the upper cased types together as key
		ctrl.type = array.join(',');
		var displayName = array.map(function (element) {
			return element.charAt(0).toUpperCase() + element.slice(1).toLowerCase();
		}).join(' ');
		// in case the type is not yet in the default list of available options we add it
		if (!ctrl.availableOptions.some(function(e) { return e.id === ctrl.type; } )) {
			ctrl.availableOptions = ctrl.availableOptions.concat([{id: ctrl.type, name: displayName}]);
		}

		// Remove duplicate entry
		ctrl.availableOptions = _.uniq(ctrl.availableOptions, function(option) { return option.name; });
		if (ctrl.availableOptions.filter(function(option) { return option.id === ctrl.type; }).length === 0) {
			// Our default value has been thrown out by the uniq function, let's find a replacement
			var optionName = ctrl.meta.options.filter(function(option) { return option.id === ctrl.type; })[0].name;
			ctrl.type = ctrl.availableOptions.filter(function(option) { return option.name === optionName; })[0].id;
			// We don't want to override the default keys. Compatibility > standardization
			// ctrl.data.meta.type[0] = ctrl.type;
			// ctrl.model.updateContact();
		}
	}
	if (!_.isUndefined(ctrl.data) && !_.isUndefined(ctrl.data.namespace)) {
		if (!_.isUndefined(ctrl.contact.props['X-ABLABEL'])) {
			var val = _.find(this.contact.props['X-ABLABEL'], function(x) { return x.namespace === ctrl.data.namespace; });
			ctrl.type = val.value.toUpperCase();
			if (!_.isUndefined(val)) {
				// in case the type is not yet in the default list of available options we add it
				if (!ctrl.availableOptions.some(function(e) { return e.id === val.value; } )) {
					ctrl.availableOptions = ctrl.availableOptions.concat([{id: val.value.toUpperCase(), name: val.value.toUpperCase()}]);
				}
			}
		}
	}

	ctrl.availableGroups = [];

	ContactService.getGroups().then(function(groups) {
		ctrl.availableGroups = _.unique(groups);
	});

	ctrl.changeType = function (val) {
		if (ctrl.isPreferred) {
			val += ',PREF';
		}
		ctrl.data.meta = ctrl.data.meta || {};
		ctrl.data.meta.type = ctrl.data.meta.type || [];
		ctrl.data.meta.type[0] = val;
		ContactService.queueUpdate(ctrl.contact);
	};

	ctrl.dateInputChanged = function () {
		ctrl.data.meta = ctrl.data.meta || {};

		var match = ctrl.data.value.match(/^(\d{4})-(\d{2})-(\d{2})$/);
		if (match) {
			ctrl.data.meta.value = [];
		} else {
			ctrl.data.meta.value = ctrl.data.meta.value || [];
			ctrl.data.meta.value[0] = 'text';
		}
		ContactService.queueUpdate(ctrl.contact);
	};

	ctrl.updateDetailedName = function () {
		var fn = '';
		if (ctrl.data.value[3]) {
			fn += ctrl.data.value[3] + ' ';
		}
		if (ctrl.data.value[1]) {
			fn += ctrl.data.value[1] + ' ';
		}
		if (ctrl.data.value[2]) {
			fn += ctrl.data.value[2] + ' ';
		}
		if (ctrl.data.value[0]) {
			fn += ctrl.data.value[0] + ' ';
		}
		if (ctrl.data.value[4]) {
			fn += ctrl.data.value[4];
		}

		ctrl.contact.fullName(fn);
		ContactService.queueUpdate(ctrl.contact);
	};

	ctrl.updateContact = function() {
		ContactService.queueUpdate(ctrl.contact);
	};

	ctrl.getTemplate = function() {
		var templateUrl = OC.linkTo('contacts', 'templates/detailItems/' + ctrl.meta.template + '.html');
		return $templateRequest(templateUrl);
	};

	ctrl.deleteField = function () {
		ctrl.contact.removeProperty(ctrl.name, ctrl.data);
		ContactService.queueUpdate(ctrl.contact);
	};
}]);

angular.module('contactsApp')
.directive('detailsitem', ['$compile', function($compile) {
	return {
		scope: {},
		controller: 'detailsItemCtrl',
		controllerAs: 'ctrl',
		bindToController: {
			name: '=',
			data: '=',
			contact: '=model',
			index: '='
		},
		link: function(scope, element, attrs, ctrl) {
			ctrl.getTemplate().then(function(html) {
				var template = angular.element(html);
				element.append(template);
				$compile(template)(scope);
			});
		}
	};
}]);

angular.module('contactsApp')
.controller('groupCtrl', function() {
	// eslint-disable-next-line no-unused-vars
	var ctrl = this;
});

angular.module('contactsApp')
.directive('group', function() {
	return {
		restrict: 'A', // has to be an attribute to work with core css
		scope: {},
		controller: 'groupCtrl',
		controllerAs: 'ctrl',
		bindToController: {
			group: '=group'
		},
		templateUrl: OC.linkTo('contacts', 'templates/group.html')
	};
});

angular.module('contactsApp')
.controller('grouplistCtrl', ['$scope', '$timeout', 'ContactService', 'SearchService', '$routeParams', function($scope, $timeout, ContactService, SearchService, $routeParams) {
	var ctrl = this;

	ctrl.groups = [];
	ctrl.contactFilters = [];

	ContactService.getGroupList().then(function(groups) {
		ctrl.groups = groups;
	});

	ContactService.getContactFilters().then(function(contactFilters) {
		ctrl.contactFilters = contactFilters;
	});

	ctrl.getSelected = function() {
		return $routeParams.gid;
	};

	// Update groupList on contact add/delete/update/groupsUpdate
	ContactService.registerObserverCallback(function(ev) {
		if (ev.event !== 'getFullContacts') {
			$timeout(function () {
				$scope.$apply(function() {
					ContactService.getGroupList().then(function(groups) {
						ctrl.groups = groups;
					});
					ContactService.getContactFilters().then(function(contactFilters) {
						ctrl.contactFilters = contactFilters;
					});
				});
			});
		}
	});

	ctrl.setSelected = function (selectedGroup) {
		SearchService.cleanSearch();
		$routeParams.gid = selectedGroup;
	};
}]);

angular.module('contactsApp')
.directive('grouplist', function() {
	return {
		restrict: 'EA', // has to be an attribute to work with core css
		scope: {},
		controller: 'grouplistCtrl',
		controllerAs: 'ctrl',
		bindToController: {},
		templateUrl: OC.linkTo('contacts', 'templates/groupList.html')
	};
});

angular.module('contactsApp')
.controller('importscreenCtrl', ['$scope', 'ImportService', function($scope, ImportService) {
	var ctrl = this;

	ctrl.t = {
		importingTo : t('contacts', 'Importing into'),
		selectAddressbook : t('contacts', 'Select your addressbook')
	};

	// Broadcast update
	$scope.$on('importing', function () {
		ctrl.selectedAddressBook = ImportService.selectedAddressBook;
		ctrl.importedUser = ImportService.importedUser;
		ctrl.importing = ImportService.importing;
		ctrl.importPercent = ImportService.importPercent;
	});

}]);

angular.module('contactsApp')
.directive('importscreen', function() {
	return {
		restrict: 'EA', // has to be an attribute to work with core css
		scope: {},
		controller: 'importscreenCtrl',
		controllerAs: 'ctrl',
		bindToController: {},
		templateUrl: OC.linkTo('contacts', 'templates/importScreen.html')
	};
});

angular.module('contactsApp')
.controller('newContactButtonCtrl', ['$scope', 'ContactService', '$routeParams', 'vCardPropertiesService', function($scope, ContactService, $routeParams, vCardPropertiesService) {
	var ctrl = this;

	ctrl.t = {
		addContact : t('contacts', 'New contact')
	};

	ctrl.createContact = function() {
		ContactService.create().then(function(contact) {
			['tel', 'adr', 'email'].forEach(function(field) {
				var defaultValue = vCardPropertiesService.getMeta(field).defaultValue || {value: ''};
				contact.addProperty(field, defaultValue);
			} );
			if ([t('contacts', 'All contacts'), t('contacts', 'Not grouped')].indexOf($routeParams.gid) === -1) {
				contact.categories([ $routeParams.gid ]);
			} else {
				contact.categories([]);
			}
			$('#details-fullName').focus();
		});
	};
}]);

angular.module('contactsApp')
.directive('newcontactbutton', function() {
	return {
		restrict: 'EA', // has to be an attribute to work with core css
		scope: {},
		controller: 'newContactButtonCtrl',
		controllerAs: 'ctrl',
		bindToController: {},
		templateUrl: OC.linkTo('contacts', 'templates/newContactButton.html')
	};
});

angular.module('contactsApp')
.directive('telModel', function() {
	return{
		restrict: 'A',
		require: 'ngModel',
		link: function(scope, element, attr, ngModel) {
			ngModel.$formatters.push(function(value) {
				return value;
			});
			ngModel.$parsers.push(function(value) {
				return value;
			});
		}
	};
});

angular.module('contactsApp')
.controller('propertyGroupCtrl', ['vCardPropertiesService', function(vCardPropertiesService) {
	var ctrl = this;

	ctrl.meta = vCardPropertiesService.getMeta(ctrl.name);

	this.isHidden = function() {
		return ctrl.meta.hasOwnProperty('hidden') && ctrl.meta.hidden === true;
	};

	this.getIconClass = function() {
		return ctrl.meta.icon || 'icon-contacts-dark';
	};

	this.getReadableName = function() {
		return ctrl.meta.readableName;
	};
}]);

angular.module('contactsApp')
.directive('propertygroup', function() {
	return {
		scope: {},
		controller: 'propertyGroupCtrl',
		controllerAs: 'ctrl',
		bindToController: {
			properties: '=data',
			name: '=',
			contact: '=model'
		},
		templateUrl: OC.linkTo('contacts', 'templates/propertyGroup.html'),
		link: function(scope, element, attrs, ctrl) {
			if(ctrl.isHidden()) {
				// TODO replace with class
				element.css('display', 'none');
			}
		}
	};
});

angular.module('contactsApp')
.controller('sortbyCtrl', ['SortByService', function(SortByService) {
	var ctrl = this;

	var sortText = t('contacts', 'Sort by');
	ctrl.sortText = sortText;

	var sortList = SortByService.getSortByList();
	ctrl.sortList = sortList;

	ctrl.defaultOrder = SortByService.getSortByKey();

	ctrl.updateSortBy = function() {
		SortByService.setSortBy(ctrl.defaultOrder);
	};
}]);

angular.module('contactsApp')
.directive('sortby', function() {
	return {
		priority: 1,
		scope: {},
		controller: 'sortbyCtrl',
		controllerAs: 'ctrl',
		bindToController: {},
		templateUrl: OC.linkTo('contacts', 'templates/sortBy.html')
	};
});

angular.module('contactsApp')
.factory('AddressBook', function()
{
	return function AddressBook(data) {
		angular.extend(this, {

			displayName: '',
			contacts: [],
			groups: data.data.props.groups,
			readOnly: data.data.props.readOnly === '1',
			// In case of not defined
			enabled: data.data.props.enabled !== '0',

			sharedWith: {
				users: [],
				groups: []
			}

		});
		angular.extend(this, data);
		angular.extend(this, {
			owner: data.data.props.owner.split('/').slice(-2, -1)[0]
		});

		var shares = this.data.props.invite;
		if (typeof shares !== 'undefined') {
			for (var j = 0; j < shares.length; j++) {
				var href = shares[j].href;
				if (href.length === 0) {
					continue;
				}
				var access = shares[j].access;
				if (access.length === 0) {
					continue;
				}

				var readWrite = (typeof access.readWrite !== 'undefined');

				if (href.startsWith('principal:principals/users/')) {
					this.sharedWith.users.push({
						id: href.substr(27),
						displayname: href.substr(27),
						writable: readWrite
					});
				} else if (href.startsWith('principal:principals/groups/')) {
					this.sharedWith.groups.push({
						id: href.substr(28),
						displayname: href.substr(28),
						writable: readWrite
					});
				}
			}
		}
	};
});

angular.module('contactsApp')
	.factory('ContactFilter', function()
	{
		return function ContactFilter(data) {
			angular.extend(this, {
				name: '',
				count: 0
			});

			angular.extend(this, data);
		};
	});

angular.module('contactsApp')
.factory('Contact', ['$filter', 'MimeService', function($filter, MimeService) {
	return function Contact(addressBook, vCard) {
		angular.extend(this, {

			data: {},
			props: {},
			failedProps: [],

			dateProperties: ['bday', 'anniversary', 'deathdate'],

			addressBookId: addressBook.displayName,
			readOnly: addressBook.readOnly,

			version: function() {
				var property = this.getProperty('version');
				if(property) {
					return property.value;
				}

				return undefined;
			},

			uid: function(value) {
				var model = this;
				if (angular.isDefined(value)) {
					// setter
					return model.setProperty('uid', { value: value });
				} else {
					// getter
					return model.getProperty('uid').value;
				}
			},

			displayName: function() {
				var displayName = this.fullName() || this.org() || '';
				if(angular.isArray(displayName)) {
					return displayName.join(' ');
				}
				return displayName;
			},

			readableFilename: function() {
				if(this.displayName()) {
					return (this.displayName()) + '.vcf';
				} else {
					// fallback to default filename (see download attribute)
					return '';
				}

			},

			firstName: function() {
				var property = this.getProperty('n');
				if (property) {
					return property.value[1];
				} else {
					return this.displayName();
				}
			},

			lastName: function() {
				var property = this.getProperty('n');
				if (property) {
					return property.value[0];
				} else {
					return this.displayName();
				}
			},

			additionalNames: function() {
				var property = this.getProperty('n');
				if (property) {
					return property.value[2];
				} else {
					return '';
				}
			},

			fullName: function(value) {
				var model = this;
				if (angular.isDefined(value)) {
					// setter
					return this.setProperty('fn', { value: value });
				} else {
					// getter
					var property = model.getProperty('fn');
					if(property) {
						return property.value;
					}
					property = model.getProperty('n');
					if(property) {
						return property.value.filter(function(elem) {
							return elem;
						}).join(' ');
					}
					return undefined;
				}
			},

			title: function(value) {
				if (angular.isDefined(value)) {
					// setter
					return this.setProperty('title', { value: value });
				} else {
					// getter
					var property = this.getProperty('title');
					if(property) {
						return property.value;
					} else {
						return undefined;
					}
				}
			},

			org: function(value) {
				var property = this.getProperty('org');
				if (angular.isDefined(value)) {
					var val = value;
					// setter
					if(property && Array.isArray(property.value)) {
						val = property.value;
						val[0] = value;
					}
					return this.setProperty('org', { value: val });
				} else {
					// getter
					if(property) {
						if (Array.isArray(property.value)) {
							return property.value[0];
						}
						return property.value;
					} else {
						return undefined;
					}
				}
			},

			email: function() {
				// getter
				var property = this.getProperty('email');
				if(property) {
					return property.value;
				} else {
					return undefined;
				}
			},

			photo: function(value) {
				if (angular.isDefined(value)) {
					// setter
					// splits image data into "data:image/jpeg" and base 64 encoded image
					var imageData = value.split(';base64,');
					var imageType = imageData[0].slice('data:'.length);
					if (!imageType.startsWith('image/')) {
						return;
					}
					imageType = imageType.substring(6).toUpperCase();

					return this.setProperty('photo', { value: imageData[1], meta: {type: [imageType], encoding: ['b']} });
				} else {
					var property = this.getProperty('photo');
					if(property) {
						var type = property.meta.type;
						if (angular.isArray(type)) {
							type = type[0];
						}
						if (!type.startsWith('image/')) {
							type = 'image/' + type.toLowerCase();
						}
						return 'data:' + type + ';base64,' + property.value;
					} else {
						return undefined;
					}
				}
			},

			categories: function(value) {
				if (angular.isDefined(value)) {
					// setter
					if (angular.isString(value)) {
						/* check for empty string */
						this.setProperty('categories', { value: !value.length ? [] : [value] });
					} else if (angular.isArray(value)) {
						this.setProperty('categories', { value: value });
					}
				} else {
					// getter
					var property = this.getProperty('categories');
					if(!property) {
						return [];
					}
					if (angular.isArray(property.value)) {
						return property.value;
					}
					return [property.value];
				}
			},

			formatDateAsRFC6350: function(name, data) {
				if (angular.isUndefined(data) || angular.isUndefined(data.value)) {
					return data;
				}
				if (this.dateProperties.indexOf(name) !== -1) {
					var match = data.value.match(/^(\d{4})-(\d{2})-(\d{2})$/);
					if (match) {
						data.value = match[1] + match[2] + match[3];
					}
				}

				return data;
			},

			formatDateForDisplay: function(name, data) {
				if (angular.isUndefined(data) || angular.isUndefined(data.value)) {
					return data;
				}
				if (this.dateProperties.indexOf(name) !== -1) {
					var match = data.value.match(/^(\d{4})(\d{2})(\d{2})$/);
					if (match) {
						data.value = match[1] + '-' + match[2] + '-' + match[3];
					}
				}

				return data;
			},

			getProperty: function(name) {
				if (this.props[name]) {
					return this.formatDateForDisplay(name, this.validate(name, this.props[name][0]));
				} else {
					return undefined;
				}
			},
			addProperty: function(name, data) {
				data = angular.copy(data);
				data = this.formatDateAsRFC6350(name, data);
				if(!this.props[name]) {
					this.props[name] = [];
				}
				var idx = this.props[name].length;
				this.props[name][idx] = data;

				// keep vCard in sync
				this.data.addressData = $filter('JSON2vCard')(this.props);
				return idx;
			},
			setProperty: function(name, data) {
				if(!this.props[name]) {
					this.props[name] = [];
				}
				data = this.formatDateAsRFC6350(name, data);
				this.props[name][0] = data;

				// keep vCard in sync
				this.data.addressData = $filter('JSON2vCard')(this.props);
			},
			removeProperty: function (name, prop) {
				angular.copy(_.without(this.props[name], prop), this.props[name]);
				if(this.props[name].length === 0) {
					delete this.props[name];
				}
				this.data.addressData = $filter('JSON2vCard')(this.props);
			},
			setETag: function(etag) {
				this.data.etag = etag;
			},
			setUrl: function(addressBook, uid) {
				this.data.url = addressBook.url + uid + '.vcf';
			},
			setAddressBook: function(addressBook) {
				this.addressBookId = addressBook.displayName;
				this.data.url = addressBook.url + this.uid() + '.vcf';
			},

			getISODate: function(date) {
				function pad(number) {
					if (number < 10) {
						return '0' + number;
					}
					return '' + number;
				}

				return date.getUTCFullYear() + '' +
						pad(date.getUTCMonth() + 1) +
						pad(date.getUTCDate()) +
						'T' + pad(date.getUTCHours()) +
						pad(date.getUTCMinutes()) +
						pad(date.getUTCSeconds()) + 'Z';
			},

			syncVCard: function() {

				this.setProperty('rev', { value: this.getISODate(new Date()) });
				var self = this;

				_.each(this.dateProperties, function(name) {
					if (!angular.isUndefined(self.props[name]) && !angular.isUndefined(self.props[name][0])) {
						// Set dates again to make sure they are in RFC-6350 format
						self.setProperty(name, self.props[name][0]);
					}
				});
				// force fn to be set
				this.fullName(this.fullName());

				// keep vCard in sync
				self.data.addressData = $filter('JSON2vCard')(self.props);

				// Revalidate all props
				_.each(self.failedProps, function(name, index) {
					if (!angular.isUndefined(self.props[name]) && !angular.isUndefined(self.props[name][0])) {
						// Reset previously failed properties
						self.failedProps.splice(index, 1);
						// And revalidate them again
						self.validate(name, self.props[name][0]);

					} else if(angular.isUndefined(self.props[name]) || angular.isUndefined(self.props[name][0])) {
						// Property has been removed
						self.failedProps.splice(index, 1);
					}
				});

			},

			matches: function(pattern) {
				if (angular.isUndefined(pattern) || pattern.length === 0) {
					return true;
				}
				var model = this;
				var matchingProps = ['fn', 'title', 'org', 'email', 'nickname', 'note', 'url', 'cloud', 'adr', 'impp', 'tel', 'gender', 'relationship'].filter(function (propName) {
					if (model.props[propName]) {
						return model.props[propName].filter(function (property) {
							if (!property.value) {
								return false;
							}
							if (angular.isString(property.value)) {
								return property.value.toLowerCase().indexOf(pattern.toLowerCase()) !== -1;
							}
							if (angular.isArray(property.value)) {
								return property.value.filter(function(v) {
									return v.toLowerCase().indexOf(pattern.toLowerCase()) !== -1;
								}).length > 0;
							}
							return false;
						}).length > 0;
					}
					return false;
				});
				return matchingProps.length > 0;
			},

			/* eslint-disable no-console */
			validate: function(prop, property) {
				switch(prop) {
				case 'rev':
				case 'prodid':
				case 'version':
					if (!angular.isUndefined(this.props[prop]) && this.props[prop].length > 1) {
						this.props[prop] = [this.props[prop][0]];
						console.warn(this.uid()+': Too many '+prop+' fields. Saving this one only: ' + this.props[prop][0].value);
						this.failedProps.push(prop);
					}
					break;

				case 'categories':
					// Avoid unescaped commas
					if (angular.isArray(property.value)) {
						if(property.value.join(';').indexOf(',') !== -1) {
							this.failedProps.push(prop);
							property.value = property.value.join(',').split(',');
							//console.warn(this.uid()+': Categories split: ' + property.value);
						}
					} else if (angular.isString(property.value)) {
						if(property.value.indexOf(',') !== -1) {
							this.failedProps.push(prop);
							property.value = property.value.split(',');
							//console.warn(this.uid()+': Categories split: ' + property.value);
						}
					}
					// Remove duplicate categories on array
					if(property.value.length !== 0 && angular.isArray(property.value)) {
						var uniqueCategories = _.unique(property.value);
						if(!angular.equals(uniqueCategories, property.value)) {
							this.failedProps.push(prop);
							property.value = uniqueCategories;
							//console.warn(this.uid()+': Categories duplicate: ' + property.value);
						}
					}
					break;
				case 'photo':
					// Avoid undefined photo type
					if (angular.isDefined(property)) {
						if (angular.isUndefined(property.meta.type)) {
							var mime = MimeService.b64mime(property.value);
							if (mime) {
								this.failedProps.push(prop);
								property.meta.type=[mime];
								this.setProperty('photo', {
									value:property.value,
									meta: {
										type:property.meta.type,
										encoding:property.meta.encoding
									}
								});
								console.warn(this.uid()+': Photo detected as ' + property.meta.type);
							} else {
								this.failedProps.push(prop);
								this.removeProperty('photo', property);
								property = undefined;
								console.warn(this.uid()+': Photo removed');
							}
						}
					}
					break;
				}
				return property;
			},
			/* eslint-enable no-console */

			fix: function() {
				this.validate('rev');
				this.validate('version');
				this.validate('prodid');
				return this.failedProps.indexOf('rev') !== -1
					|| this.failedProps.indexOf('prodid') !== -1
					|| this.failedProps.indexOf('version') !== -1;
			}

		});

		if(angular.isDefined(vCard)) {
			angular.extend(this.data, vCard);
			angular.extend(this.props, $filter('vCard2JSON')(this.data.addressData));
			// We do not want to store our addressbook within contacts
			delete this.data.addressBook;
		} else {
			angular.extend(this.props, {
				version: [{value: '3.0'}],
				fn: [{value: t('contacts', 'New contact')}]
			});
			this.data.addressData = $filter('JSON2vCard')(this.props);
		}

		var property = this.getProperty('categories');
		if(!property) {
			// categories should always have the same type (an array)
			this.categories([]);
		} else {
			if (angular.isString(property.value)) {
				this.categories([property.value]);
			}
		}
	};
}]);

angular.module('contactsApp')
	.factory('Group', function()
	{
		return function Group(data) {
			angular.extend(this, {
				name: '',
				count: 0
			});

			angular.extend(this, data);
		};
	});

angular.module('contactsApp')
.factory('AddressBookService', ['DavClient', 'DavService', 'SettingsService', 'AddressBook', '$q', function(DavClient, DavService, SettingsService, AddressBook, $q) {

	var addressBooks = [];
	var loadPromise = undefined;

	var observerCallbacks = [];

	var notifyObservers = function(eventName, addressBook) {
		var ev = {
			event: eventName,
			addressBooks: addressBooks,
			addressBook: addressBook,
		};
		angular.forEach(observerCallbacks, function(callback) {
			callback(ev);
		});
	};

	var loadAll = function() {
		if (addressBooks.length > 0) {
			return $q.when(addressBooks);
		}
		if (_.isUndefined(loadPromise)) {
			loadPromise = DavService.then(function(account) {
				loadPromise = undefined;
				addressBooks = account.addressBooks.map(function(addressBook) {
					return new AddressBook(addressBook);
				});
			});
		}
		return loadPromise;
	};

	return {
		registerObserverCallback: function(callback) {
			observerCallbacks.push(callback);
		},

		getAll: function() {
			return loadAll().then(function() {
				return addressBooks;
			});
		},

		getGroups: function() {
			return this.getAll().then(function(addressBooks) {
				return addressBooks.map(function (element) {
					return element.groups;
				}).reduce(function(a, b) {
					return a.concat(b);
				});
			});
		},

		getDefaultAddressBook: function(throwOC) {
			var i = addressBooks.findIndex(function(addressBook) {
				return addressBook.enabled && !addressBook.readOnly;
			});
			if (i !== -1) {
				return addressBooks[i];
			} else if(throwOC) {
				OC.Notification.showTemporary(t('contacts', 'There is no address book available to create a contact.'));
			}
			return false;
		},

		getAddressBook: function(displayName) {
			return DavService.then(function(account) {
				return DavClient.getAddressBook({displayName:displayName, url:account.homeUrl}).then(function(res) {
					var addressBook = new AddressBook({
						account: account,
						ctag: res[0].props.getctag,
						url: account.homeUrl+displayName+'/',
						data: res[0],
						displayName: res[0].props.displayname,
						resourcetype: res[0].props.resourcetype,
						syncToken: res[0].props.syncToken
					});
					notifyObservers('create', addressBook);
					return addressBook;
				});
			});
		},

		create: function(displayName) {
			return DavService.then(function(account) {
				return DavClient.createAddressBook({displayName:displayName, url:account.homeUrl});
			});
		},

		delete: function(addressBook) {
			return DavService.then(function() {
				return DavClient.deleteAddressBook(addressBook).then(function() {
					var index = addressBooks.indexOf(addressBook);
					addressBooks.splice(index, 1);
					notifyObservers('delete', addressBook);
				});
			});
		},

		rename: function(addressBook, displayName) {
			return DavService.then(function(account) {
				return DavClient.renameAddressBook(addressBook, {displayName:displayName, url:account.homeUrl});
			});
		},

		get: function(displayName) {
			return this.getAll().then(function(addressBooks) {
				return addressBooks.filter(function (element) {
					return element.displayName === displayName;
				})[0];
			});
		},

		sync: function(addressBook) {
			return DavClient.syncAddressBook(addressBook);
		},

		addContact: function(addressBook, contact) {
			// We don't want to add the same contact again
			if (addressBook.contacts.indexOf(contact) === -1) {
				return addressBook.contacts.push(contact);
			}
		},

		removeContact: function(addressBook, contact) {
			// We can't remove an undefined object
			if (addressBook.contacts.indexOf(contact) !== -1) {
				return addressBook.contacts.splice(addressBook.contacts.indexOf(contact), 1);
			}
		},

		toggleState: function(addressBook) {
			var xmlDoc = document.implementation.createDocument('', '', null);
			var dPropUpdate = xmlDoc.createElement('d:propertyupdate');
			dPropUpdate.setAttribute('xmlns:d', 'DAV:');
			dPropUpdate.setAttribute('xmlns:o', 'http://owncloud.org/ns');
			xmlDoc.appendChild(dPropUpdate);

			var dSet = xmlDoc.createElement('d:set');
			dPropUpdate.appendChild(dSet);

			var dProp = xmlDoc.createElement('d:prop');
			dSet.appendChild(dProp);

			var oEnabled = xmlDoc.createElement('o:enabled');
			// Revert state to toggle
			oEnabled.textContent = !addressBook.enabled ? '1' : '0';
			dProp.appendChild(oEnabled);

			var body = dPropUpdate.outerHTML;

			return DavClient.xhr.send(
				dav.request.basic({method: 'PROPPATCH', data: body}),
				addressBook.url
			).then(function(response) {
				if (response.status === 207) {
					addressBook.enabled = !addressBook.enabled;
					notifyObservers(
						addressBook.enabled ? 'enable' : 'disable',
						addressBook
					);
				}
				return addressBook;
			});
		},

		share: function(addressBook, shareType, shareWith, writable, existingShare) {
			var xmlDoc = document.implementation.createDocument('', '', null);
			var oShare = xmlDoc.createElement('o:share');
			oShare.setAttribute('xmlns:d', 'DAV:');
			oShare.setAttribute('xmlns:o', 'http://owncloud.org/ns');
			xmlDoc.appendChild(oShare);

			var oSet = xmlDoc.createElement('o:set');
			oShare.appendChild(oSet);

			var dHref = xmlDoc.createElement('d:href');
			if (shareType === OC.Share.SHARE_TYPE_USER) {
				dHref.textContent = 'principal:principals/users/';
			} else if (shareType === OC.Share.SHARE_TYPE_GROUP) {
				dHref.textContent = 'principal:principals/groups/';
			}
			dHref.textContent += shareWith;
			oSet.appendChild(dHref);

			var oSummary = xmlDoc.createElement('o:summary');
			oSummary.textContent = t('contacts', '{addressbook} shared by {owner}', {
				addressbook: addressBook.displayName,
				owner: addressBook.owner
			});
			oSet.appendChild(oSummary);

			if (writable) {
				var oRW = xmlDoc.createElement('o:read-write');
				oSet.appendChild(oRW);
			}

			var body = oShare.outerHTML;

			return DavClient.xhr.send(
				dav.request.basic({method: 'POST', data: body}),
				addressBook.url
			).then(function(response) {
				if (response.status === 200) {
					if (!existingShare) {
						if (shareType === OC.Share.SHARE_TYPE_USER) {
							addressBook.sharedWith.users.push({
								id: shareWith,
								displayname: shareWith,
								writable: writable
							});
						} else if (shareType === OC.Share.SHARE_TYPE_GROUP) {
							addressBook.sharedWith.groups.push({
								id: shareWith,
								displayname: shareWith,
								writable: writable
							});
						}
					}
				}
			});

		},

		unshare: function(addressBook, shareType, shareWith) {
			var xmlDoc = document.implementation.createDocument('', '', null);
			var oShare = xmlDoc.createElement('o:share');
			oShare.setAttribute('xmlns:d', 'DAV:');
			oShare.setAttribute('xmlns:o', 'http://owncloud.org/ns');
			xmlDoc.appendChild(oShare);

			var oRemove = xmlDoc.createElement('o:remove');
			oShare.appendChild(oRemove);

			var dHref = xmlDoc.createElement('d:href');
			if (shareType === OC.Share.SHARE_TYPE_USER) {
				dHref.textContent = 'principal:principals/users/';
			} else if (shareType === OC.Share.SHARE_TYPE_GROUP) {
				dHref.textContent = 'principal:principals/groups/';
			}
			dHref.textContent += shareWith;
			oRemove.appendChild(dHref);
			var body = oShare.outerHTML;


			return DavClient.xhr.send(
				dav.request.basic({method: 'POST', data: body}),
				addressBook.url
			).then(function(response) {
				if (response.status === 200) {
					if (shareType === OC.Share.SHARE_TYPE_USER) {
						addressBook.sharedWith.users = addressBook.sharedWith.users.filter(function(user) {
							return user.id !== shareWith;
						});
					} else if (shareType === OC.Share.SHARE_TYPE_GROUP) {
						addressBook.sharedWith.groups = addressBook.sharedWith.groups.filter(function(groups) {
							return groups.id !== shareWith;
						});
					}
					//todo - remove entry from addressbook object
					return true;
				} else {
					return false;
				}
			});

		}


	};

}]);

angular.module('contactsApp')
.service('ContactService', ['DavClient', 'AddressBookService', 'Contact', 'Group', 'ContactFilter', '$q', 'CacheFactory', 'uuid4', function(DavClient, AddressBookService, Contact, Group, ContactFilter, $q, CacheFactory, uuid4) {

	var contactService = this;

	var cacheFilled = false;
	var contactsCache = CacheFactory('contacts');
	var observerCallbacks = [];
	var loadPromise = undefined;

	var allUpdates = $q.when();
	this.queueUpdate = function(contact) {
		allUpdates = allUpdates.then(function() {
			return contactService.update(contact);
		});
	};

	this.registerObserverCallback = function(callback) {
		observerCallbacks.push(callback);
	};

	var notifyObservers = function(eventName, uid) {
		var ev = {
			event: eventName,
			uid: uid,
			contacts: contactsCache.values()
		};
		angular.forEach(observerCallbacks, function(callback) {
			callback(ev);
		});
	};

	this.getFullContacts = function(contacts) {
		AddressBookService.getAll().then(function(addressBooks) {
			var promises = [];
			var xhrAddressBooks = [];
			contacts.forEach(function(contact) {
				// Regroup urls by addressbooks
				if(addressBooks.indexOf(contact.addressBook) !== -1) {
					// Initiate array if no exists
					xhrAddressBooks[contact.addressBookId] = xhrAddressBooks[contact.addressBookId] || [];
					xhrAddressBooks[contact.addressBookId].push(contact.data.url);
				}
			});
			// Get our full vCards
			addressBooks.forEach(function(addressBook) {
				// Only go through enabled addressbooks
				// Though xhrAddressBooks does not contains contacts from disabled ones
				if(addressBook.enabled) {
					if(angular.isArray(xhrAddressBooks[addressBook.displayName])) {
						var promise = DavClient.getContacts(addressBook, {}, xhrAddressBooks[addressBook.displayName]).then(
							function(vcards) {
								return vcards.map(function(vcard) {
									return new Contact(addressBook, vcard);
								});
							}).then(function(contacts_) {
								contacts_.map(function(contact) {
									// Validate some fields
									if(contact.fix()) {
										// Can't use `this` in those nested functions
										contactService.update(contact);
									}
									contactsCache.put(contact.uid(), contact);
									addressBook.contacts.push(contact);
								});
							});
						promises.push(promise);
					}
				}
			});
			$q.all(promises).then(function() {
				notifyObservers('getFullContacts', '');
			});
		});
	};

	this.fillCache = function() {
		if (_.isUndefined(loadPromise)) {
			loadPromise = AddressBookService.getAll().then(function(addressBooks) {
				var promises = [];
				addressBooks.forEach(function(addressBook) {
					// Only go through enabled addressbooks
					if(addressBook.enabled) {
						promises.push(
							AddressBookService.sync(addressBook).then(function(addressBook) {
								contactService.appendContactsFromAddressbook(addressBook);
							})
						);
					}
				});
				return $q.all(promises).then(function() {
					cacheFilled = true;
				});
			});
		}
		return loadPromise;
	};

	this.getAll = function() {
		if(cacheFilled === false) {
			return this.fillCache().then(function() {
				return contactsCache.values();
			});
		} else {
			return $q.when(contactsCache.values());
		}
	};

	this.getContactFilters = function() {
		return this.getAll().then(function(contacts) {
			var allContacts = new ContactFilter({
				name: t('contacts', 'All contacts'),
				count: contacts.length
			});
			var notGrouped = new ContactFilter({
				name: t('contacts', 'Not grouped'),
				count: contacts.filter(
					function(contact) {
						return contact.categories().length === 0;
					}).length
			});
			var filters = [allContacts];
			// Only have Not Grouped if at least one contact in it
			if(notGrouped.count !== 0) {
				filters.push(notGrouped);
			}

			return filters;
		});
	};

	// get list of groups and the count of contacts in said groups
	this.getGroupList = function() {
		return this.getAll().then(function(contacts) {
			// allow groups with names such as toString
			var groups = Object.create(null);

			// collect categories and their associated counts
			contacts.forEach(function(contact) {
				contact.categories().forEach(function(category) {
					groups[category] = groups[category] ? groups[category] + 1 : 1;
				});
			});
			return _.keys(groups).map(
				function(key) {
					return new Group({
						name: key,
						count: groups[key]
					});
				});
		});
	};

	this.getGroups = function() {
		return this.getAll().then(function(contacts) {
			return _.uniq(contacts.map(function(element) {
				return element.categories();
			}).reduce(function(a, b) {
				return a.concat(b);
			}, []).sort(), true);
		});
	};

	this.getById = function(addressBooks, uid) {
		return (function() {
			if(cacheFilled === false) {
				return this.fillCache().then(function() {
					return contactsCache.get(uid);
				});
			} else {
				return $q.when(contactsCache.get(uid));
			}
		}).call(this)
			.then(function(contact) {
				if(angular.isUndefined(contact)) {
					OC.Notification.showTemporary(t('contacts', 'Contact not found.'));
					return;
				} else {
					var addressBook = addressBooks.find(function(book) {
						return book.displayName === contact.addressBookId;
					});
					// Fetch and return full contact vcard
					return addressBook
						? DavClient.getContacts(addressBook, {}, [ contact.data.url ]).then(function(vcards) {
							return new Contact(addressBook, vcards[0]);
						}).then(function(newContact) {
							contactsCache.put(contact.uid(), newContact);
							var contactIndex = addressBook.contacts.findIndex(function(testedContact) {
								return testedContact.uid() === contact.uid();
							});
							addressBook.contacts[contactIndex] = newContact;
							notifyObservers('getFullContacts', contact.uid());
							return newContact;
						}) : contact;
				}
			});
	};

	this.create = function(newContact, addressBook, uid, fromImport) {
		addressBook = addressBook || AddressBookService.getDefaultAddressBook(true);

		// No addressBook available
		if(!addressBook) {
			return;
		}

		if(addressBook.readOnly) {
			OC.Notification.showTemporary(t('contacts', 'You don\'t have permission to write to this addressbook.'));
			return;
		}
		try {
			newContact = newContact || new Contact(addressBook);
		} catch(error) {
			OC.Notification.showTemporary(t('contacts', 'Contact could not be created.'));
			return;
		}
		var newUid = '';
		if(uuid4.validate(uid)) {
			newUid = uid;
		} else {
			newUid = uuid4.generate();
		}
		newContact.uid(newUid);
		newContact.setUrl(addressBook, newUid);
		newContact.addressBookId = addressBook.displayName;
		if (_.isUndefined(newContact.fullName()) || newContact.fullName() === '') {
			newContact.fullName(newContact.displayName());
		}

		return DavClient.createCard(
			addressBook,
			{
				data: newContact.data.addressData,
				filename: newUid + '.vcf'
			}
		).then(function(xhr) {
			newContact.setETag(xhr.getResponseHeader('ETag'));
			contactsCache.put(newUid, newContact);
			AddressBookService.addContact(addressBook, newContact);
			if (fromImport !== true) {
				notifyObservers('create', newUid);
				$('#details-fullName').select();
			}
			return newContact;
		}).catch(function() {
			OC.Notification.showTemporary(t('contacts', 'Contact could not be created.'));
			return false;
		});
	};

	this.import = function(data, type, addressBook, progressCallback) {
		addressBook = addressBook || AddressBookService.getDefaultAddressBook(true);

		// No addressBook available
		if(!addressBook) {
			return;
		}

		var regexp = /BEGIN:VCARD[\s\S]*?END:VCARD/mgi;
		var singleVCards = data.match(regexp);

		if (!singleVCards) {
			OC.Notification.showTemporary(t('contacts', 'No contacts in file. Only vCard files are allowed.'));
			if (progressCallback) {
				progressCallback(1);
			}
			return;
		}

		notifyObservers('importstart');

		var num = 1;
		for(var i in singleVCards) {
			var newContact = new Contact(addressBook, {addressData: singleVCards[i]});
			if (['3.0', '4.0'].indexOf(newContact.version()) < 0) {
				if (progressCallback) {
					progressCallback(num / singleVCards.length);
				}
				OC.Notification.showTemporary(t('contacts', 'Only vCard version 4.0 (RFC6350) or version 3.0 (RFC2426) are supported.'));
				num++;
				continue;
			}
			// eslint-disable-next-line no-loop-func
			this.create(newContact, addressBook, '', true).then(function(xhrContact) {
				if (xhrContact !== false) {
					var xhrContactName = xhrContact.displayName();
				}
				// Update the progress indicator
				if (progressCallback) {
					progressCallback(num / singleVCards.length, xhrContactName);
				}
				num++;
				/* Import is over, let's notify */
				if (num === singleVCards.length + 1) {
					notifyObservers('importend');
				}
			});
		}
	};

	this.moveContact = function(contact, addressBook, oldAddressBook) {
		if (addressBook !== null && contact.addressBookId === addressBook.displayName) {
			return;
		}
		if (addressBook.readOnly) {
			OC.Notification.showTemporary(t('contacts', 'You don\'t have permission to write to this addressbook.'));
			return;
		}
		contact.syncVCard();

		DavClient.xhr.send(
			dav.request.basic({method: 'MOVE', destination: addressBook.url + contact.data.url.split('/').pop(-1)}),
			contact.data.url
		).then(function(response) {
			if (response.status === 201 || response.status === 204) {
				contact.setAddressBook(addressBook);
				AddressBookService.addContact(addressBook, contact);
				AddressBookService.removeContact(oldAddressBook, contact);
				notifyObservers('groupsUpdate');
			} else {
				OC.Notification.showTemporary(t('contacts', 'Contact could not be moved.'));
			}
		});
	};

	this.update = function(contact) {
		// update rev field
		contact.syncVCard();

		// update contact on server
		return DavClient.updateCard(contact.data, {json: true}).then(function(xhr) {
			var newEtag = xhr.getResponseHeader('ETag');
			contact.setETag(newEtag);
			notifyObservers('update', contact.uid());
		}).catch(function() {
			OC.Notification.showTemporary(t('contacts', 'Contact could not be saved.'));
		});
	};

	this.delete = function(addressBook, contact) {
		// delete contact from server
		return DavClient.deleteCard(contact.data).then(function() {
			contactsCache.remove(contact.uid());
			AddressBookService.removeContact(addressBook, contact);
			notifyObservers('delete', contact.uid());
		});
	};

	/*
	 * Delete all contacts present in the addressBook from the cache
	 */
	this.removeContactsFromAddressbook = function(addressBook, callback) {
		angular.forEach(addressBook.contacts, function(contact) {
			contactsCache.remove(contact.uid());
		});
		callback();
		notifyObservers('groupsUpdate');
	};

	/*
	 * Create and append contacts to the addressBook
	 */
	this.appendContactsFromAddressbook = function(addressBook, callback) {
		// Addressbook has been initiated but contacts have not been fetched
		if (addressBook.objects === null) {
			AddressBookService.sync(addressBook).then(function(addressBook) {
				contactService.appendContactsFromAddressbook(addressBook, callback);
			});
		} else if (addressBook.contacts.length === 0) {
			// Only add contact if the addressBook doesn't already have it
			addressBook.objects.forEach(function(vcard) {
				try {
					// Only add contact if the addressBook doesn't already have it
					var contact = new Contact(addressBook, vcard);
					contactsCache.put(contact.uid(), contact);
					AddressBookService.addContact(addressBook, contact);
				} catch(error) {
					// eslint-disable-next-line no-console
					console.log('Invalid contact received: ', vcard, error);
				}
			});
		} else {
			// Contact are already present in the addressBook
			angular.forEach(addressBook.contacts, function(contact) {
				contactsCache.put(contact.uid(), contact);
			});
		}
		notifyObservers('groupsUpdate');
		if (typeof callback === 'function') {
			callback();
		}
	};

}]);

angular.module('contactsApp')
.service('DavClient', function() {
	var xhr = new dav.transport.Basic(
		new dav.Credentials()
	);
	return new dav.Client(xhr);
});

angular.module('contactsApp')
.service('DavService', ['DavClient', function(DavClient) {
	return DavClient.createAccount({
		server: OC.linkToRemote('dav/addressbooks'),
		accountType: 'carddav',
		useProvidedPath: true
	});
}]);

angular.module('contactsApp')
.service('ImportService', function() {

	this.importing = false;
	this.selectedAddressBook = t('contacts', 'Import into');
	this.importedUser = t('contacts', 'Waiting for the server to be ready…');
	this.importPercent = 0;

	this.t = {
		importText : t('contacts', 'Import into'),
		importingText : t('contacts', 'Importing…')
	};

});

angular.module('contactsApp')
	.service('MimeService', function() {
		var magicNumbers = {
			'/9j/' : 'JPEG',
			'R0lGOD' : 'GIF',
			'iVBORw0KGgo' : 'PNG'
		};

		this.b64mime = function(b64string) {
			for (var mn in magicNumbers) {
				if(b64string.startsWith(mn)) return magicNumbers[mn];
			}
			return null;
		};
	});

angular.module('contactsApp')
.service('SearchService', function() {
	var searchTerm = '';

	var observerCallbacks = [];

	this.registerObserverCallback = function(callback) {
		observerCallbacks.push(callback);
	};

	var notifyObservers = function(eventName) {
		var ev = {
			event:eventName,
			searchTerm:searchTerm
		};
		angular.forEach(observerCallbacks, function(callback) {
			callback(ev);
		});
	};

	var SearchProxy = {
		attach: function(search) {
			search.setFilter('contacts', this.filterProxy);
		},
		filterProxy: function(query) {
			searchTerm = query;
			notifyObservers('changeSearch');
		}
	};

	this.getSearchTerm = function() {
		return searchTerm;
	};

	this.cleanSearch = function() {
		if (!_.isUndefined($('.searchbox'))) {
			$('.searchbox')[0].reset();
		}
		searchTerm = '';
	};

	if (!_.isUndefined(OC.Plugins)) {
		OC.Plugins.register('OCA.Search', SearchProxy);
		if (!_.isUndefined(OCA.Search)) {
			OC.Search = new OCA.Search($('#searchbox'), $('#searchresults'));
			$('#searchbox').show();
		}
	}

	if (!_.isUndefined($('.searchbox'))) {
		$('.searchbox')[0].addEventListener('keypress', function(e) {
			if(e.keyCode === 13) {
				notifyObservers('submitSearch');
			}
		});
	}
});

angular.module('contactsApp')
.service('SettingsService', function() {
	var settings = {
		addressBooks: [
			'testAddr'
		]
	};

	this.set = function(key, value) {
		settings[key] = value;
	};

	this.get = function(key) {
		return settings[key];
	};

	this.getAll = function() {
		return settings;
	};
});

angular.module('contactsApp')
.service('SortByService', function () {
	var subscriptions = [];

	// Array of keys to sort by. Ordered by priorities.
	var sortOptions = {
		sortFirstName: ['firstName', 'lastName', 'uid'],
		sortLastName: ['lastName', 'firstName', 'uid'],
		sortDisplayName: ['displayName', 'uid']
	};

	// Key
	var sortBy = 'sortDisplayName';

	var defaultOrder = window.localStorage.getItem('contacts_default_order');
	if (defaultOrder) {
		sortBy = defaultOrder;
	}

	function notifyObservers() {
		angular.forEach(subscriptions, function (subscription) {
			if (typeof subscription === 'function') {
				subscription(sortOptions[sortBy]);
			}
		});
	}

	return {
		subscribe: function (callback) {
			subscriptions.push(callback);
		},
		setSortBy: function (value) {
			sortBy = value;
			window.localStorage.setItem('contacts_default_order', value);
			notifyObservers();
		},
		getSortBy: function () {
			return sortOptions[sortBy];
		},
		getSortByKey: function () {
			return sortBy;
		},
		getSortByList: function () {
			return {
				sortDisplayName: t('contacts', 'Display name'),
				sortFirstName: t('contacts', 'First name'),
				sortLastName: t('contacts', 'Last name')
			};
		}
	};
});

angular.module('contactsApp')
.service('vCardPropertiesService', function() {
	/**
	 * map vCard attributes to internal attributes
	 *
	 * propName: {
	 * 		multiple: [Boolean], // is this prop allowed more than once? (default = false)
	 * 		readableName: [String], // internationalized readable name of prop
	 * 		template: [String], // template name found in /templates/detailItems
	 * 		[...] // optional additional information which might get used by the template
	 *
	 *		options: If multiple options have the same name, the first will be used as default.
	 *				 Others will be merge, but still supported. Order is important!
	 * }
	 */
	this.vCardMeta = {
		nickname: {
			readableName: t('contacts', 'Nickname'),
			template: 'text',
			icon: 'icon-user'
		},
		n: {
			readableName: t('contacts', 'Detailed name'),
			defaultValue: {
				value:['', '', '', '', '']
			},
			template: 'n',
			icon: 'icon-user'
		},
		note: {
			readableName: t('contacts', 'Notes'),
			template: 'textarea',
			icon: 'icon-rename'
		},
		url: {
			multiple: true,
			readableName: t('contacts', 'Website'),
			template: 'url',
			icon: 'icon-public'
		},
		cloud: {
			multiple: true,
			readableName: t('contacts', 'Federated Cloud ID'),
			template: 'text',
			defaultValue: {
				value:[''],
				meta:{type:['HOME']}
			},
			options: [
				{id: 'HOME', name: t('contacts', 'Home')},
				{id: 'WORK', name: t('contacts', 'Work')},
				{id: 'OTHER', name: t('contacts', 'Other')}
			]		},
		adr: {
			multiple: true,
			readableName: t('contacts', 'Address'),
			template: 'adr',
			icon: 'icon-address',
			defaultValue: {
				value:['', '', '', '', '', '', ''],
				meta:{type:['HOME']}
			},
			options: [
				{id: 'HOME', name: t('contacts', 'Home')},
				{id: 'WORK', name: t('contacts', 'Work')},
				{id: 'OTHER', name: t('contacts', 'Other')}
			]
		},
		categories: {
			readableName: t('contacts', 'Groups'),
			template: 'groups'
		},
		bday: {
			readableName: t('contacts', 'Birthday'),
			template: 'date',
			icon: 'icon-calendar-dark'
		},
		anniversary: {
			readableName: t('contacts', 'Anniversary'),
			template: 'date',
			icon: 'icon-calendar-dark'
		},
		deathdate: {
			readableName: t('contacts', 'Date of death'),
			template: 'date',
			icon: 'icon-calendar-dark'
		},
		email: {
			multiple: true,
			readableName: t('contacts', 'Email'),
			template: 'email',
			icon: 'icon-mail',
			defaultValue: {
				value:'',
				meta:{type:['HOME']}
			},
			options: [
				{id: 'HOME', name: t('contacts', 'Home')},
				{id: 'WORK', name: t('contacts', 'Work')},
				{id: 'OTHER', name: t('contacts', 'Other')}
			]
		},
		impp: {
			multiple: true,
			readableName: t('contacts', 'Instant messaging'),
			template: 'username',
			icon: 'icon-comment',
			defaultValue: {
				value:[''],
				meta:{type:['SKYPE']}
			},
			options: [
				{id: 'IRC', name: 'IRC'},
				{id: 'KIK', name: 'KiK'},
				{id: 'SKYPE', name: 'Skype'},
				{id: 'TELEGRAM', name: 'Telegram'},
				{id: 'XMPP', name:'XMPP'}
			]
		},
		tel: {
			multiple: true,
			readableName: t('contacts', 'Phone'),
			template: 'tel',
			icon: 'icon-comment',
			defaultValue: {
				value:'',
				meta:{type:['HOME,VOICE']}
			},
			options: [
				{id: 'HOME,VOICE', name: t('contacts', 'Home')},
				{id: 'HOME', name: t('contacts', 'Home')},
				{id: 'WORK,VOICE', name: t('contacts', 'Work')},
				{id: 'WORK', name: t('contacts', 'Work')},
				{id: 'CELL', name: t('contacts', 'Mobile')},
				{id: 'CELL,VOICE', name: t('contacts', 'Mobile')},
				{id: 'WORK,CELL', name: t('contacts', 'Work mobile')},
				{id: 'FAX', name: t('contacts', 'Fax')},
				{id: 'HOME,FAX', name: t('contacts', 'Fax home')},
				{id: 'WORK,FAX', name: t('contacts', 'Fax work')},
				{id: 'PAGER', name: t('contacts', 'Pager')},
				{id: 'VOICE', name: t('contacts', 'Voice')},
				{id: 'CAR', name: t('contacts', 'Car')},
				{id: 'PAGER', name: t('contacts', 'Pager')},
				{id: 'WORK,PAGER', name: t('contacts', 'Work pager')}
			]
		},
		'X-SOCIALPROFILE': {
			multiple: true,
			readableName: t('contacts', 'Social network'),
			template: 'username',
			defaultValue: {
				value:[''],
				meta:{type:['facebook']}
			},
			options: [
				{id: 'FACEBOOK', name: 'Facebook'},
				{id: 'GITHUB', name: 'GitHub'},
				{id: 'GOOGLEPLUS', name: 'Google+'},
				{id: 'INSTAGRAM', name: 'Instagram'},
				{id: 'LINKEDIN', name: 'LinkedIn'},
				{id: 'PINTEREST', name: 'Pinterest'},
				{id: 'QZONE', name: 'QZone'},
				{id: 'TUMBLR', name: 'Tumblr'},
				{id: 'TWITTER', name: 'Twitter'},
				{id: 'WECHAT', name: 'WeChat'},
				{id: 'YOUTUBE', name: 'YouTube'}


			]
		},
		relationship: {
			readableName: t('contacts', 'Relationship'),
			template: 'select',
			options: [
				{id: 'SPOUSE', name: t('contacts', 'Spouse')},
				{id: 'CHILD', name: t('contacts', 'Child')},
				{id: 'MOTHER', name: t('contacts', 'Mother')},
				{id: 'FATHER', name: t('contacts', 'Father')},
				{id: 'PARENT', name: t('contacts', 'Parent')},
				{id: 'BROTHER', name: t('contacts', 'Brother')},
				{id: 'SISTER', name: t('contacts', 'Sister')},
				{id: 'RELATIVE', name: t('contacts', 'Relative')},
				{id: 'FRIEND', name: t('contacts', 'Friend')},
				{id: 'COLLEAGUE', name: t('contacts', 'Colleague')},
				{id: 'MANAGER', name: t('contacts', 'Manager')},
				{id: 'ASSISTANT', name: t('contacts', 'Assistant')},
			]
		},
		gender: {
			readableName: t('contacts', 'Gender'),
			template: 'select',
			options: [
				{id: 'F', name: t('contacts', 'Female')},
				{id: 'M', name: t('contacts', 'Male')},
				{id: 'O', name: t('contacts', 'Other')}
			]
		}
	};

	this.fieldOrder = [
		'org',
		'title',
		'tel',
		'email',
		'adr',
		'impp',
		'nick',
		'bday',
		'anniversary',
		'deathdate',
		'url',
		'X-SOCIALPROFILE',
		'relationship',
		'note',
		'categories',
		'role',
		'gender'
	];

	this.fieldDefinitions = [];
	for (var prop in this.vCardMeta) {
		this.fieldDefinitions.push({id: prop, name: this.vCardMeta[prop].readableName, multiple: !!this.vCardMeta[prop].multiple});
	}

	this.fallbackMeta = function(property) {
		function capitalize(string) { return string.charAt(0).toUpperCase() + string.slice(1); }
		return {
			name: 'unknown-' + property,
			readableName: capitalize(property),
			template: 'hidden',
			necessity: 'optional',
			hidden: true
		};
	};

	this.getMeta = function(property) {
		return this.vCardMeta[property] || this.fallbackMeta(property);
	};

});

angular.module('contactsApp')
.filter('JSON2vCard', function() {
	return function(input) {
		return vCard.generate(input);
	};
});

angular.module('contactsApp')
.filter('contactColor', function() {
	return function(input) {
		// Check if core has the new color generator
		if(typeof input.toHsl === 'function') {
			var hsl = input.toHsl();
			return 'hsl('+hsl[0]+', '+hsl[1]+'%, '+hsl[2]+'%)';
		} else {
			// If not, we use the old one
			/* global md5 */
			var hash = md5(input).substring(0, 4),
				maxRange = parseInt('ffff', 16),
				hue = parseInt(hash, 16) / maxRange * 256;
			return 'hsl(' + hue + ', 90%, 65%)';
		}
	};
});
angular.module('contactsApp')
.filter('contactGroupFilter', function() {
	'use strict';
	return function (contacts, group) {
		if (typeof contacts === 'undefined') {
			return contacts;
		}
		if (typeof group === 'undefined' || group.toLowerCase() === t('contacts', 'All contacts').toLowerCase()) {
			return contacts;
		}
		var filter = [];
		if (contacts.length > 0) {
			for (var i = 0; i < contacts.length; i++) {
				if (group.toLowerCase() === t('contacts', 'Not grouped').toLowerCase()) {
					if (contacts[i].categories().length === 0) {
						filter.push(contacts[i]);
					}
				} else {
					if (contacts[i].categories().indexOf(group) >= 0) {
						filter.push(contacts[i]);
					}
				}
			}
		}
		return filter;
	};
});

// from https://docs.nextcloud.com/server/11/developer_manual/app/css.html#menus
angular.module('contactsApp')
.filter('counterFormatter', function () {
	'use strict';
	return function (count) {
		if (count > 999) {
			return '999+';
		}
		if (count === 0) {
			return '';
		}
		return count;
	};
});


angular.module('contactsApp')
.filter('fieldFilter', function() {
	'use strict';
	return function (fields, contact) {
		if (typeof fields === 'undefined') {
			return fields;
		}
		if (typeof contact === 'undefined') {
			return fields;
		}
		var filter = [];
		if (fields.length > 0) {
			for (var i = 0; i < fields.length; i++) {
				if (fields[i].multiple ) {
					filter.push(fields[i]);
					continue;
				}
				if (_.isUndefined(contact.getProperty(fields[i].id))) {
					filter.push(fields[i]);
				}
			}
		}
		return filter;
	};
});

angular.module('contactsApp')
.filter('firstCharacter', function() {
	return function(input) {
		return input.charAt(0);
	};
});

angular.module('contactsApp')
.filter('localeOrderBy', [function () {
	return function (array, sortPredicate, reverseOrder) {
		if (!Array.isArray(array)) return array;
		if (!sortPredicate) return array;

		var arrayCopy = [];
		angular.forEach(array, function (item) {
			arrayCopy.push(item);
		});

		arrayCopy.sort(function (a, b) {


			// Did we pass multiple sorting options? If not, create an array anyway.
			sortPredicate = angular.isArray(sortPredicate) ? sortPredicate: [sortPredicate];
			// Let's test the first sort and continue if no sort occured
			for(var i=0; i<sortPredicate.length; i++) {
				var sortBy = sortPredicate[i];

				var valueA = a[sortBy];
				if (angular.isFunction(valueA)) {
					valueA = a[sortBy]();
				}
				var valueB = b[sortBy];
				if (angular.isFunction(valueB)) {
					valueB = b[sortBy]();
				}

				// Start sorting
				if (angular.isString(valueA)) {
					if(valueA !== valueB) {
						return reverseOrder ? valueB.localeCompare(valueA) : valueA.localeCompare(valueB);
					}
				}

				if (angular.isNumber(valueA) || typeof valueA === 'boolean') {
					if(valueA !== valueB) {
						return reverseOrder ? valueB - valueA : valueA - valueB;
					}
				}
			}

			return 0;
		});

		return arrayCopy;
	};
}]);

angular.module('contactsApp')
.filter('newContact', function() {
	return function(input) {
		return input !== '' ? input : t('contacts', 'New contact');
	};
});

angular.module('contactsApp')
.filter('orderDetailItems', ['vCardPropertiesService', function(vCardPropertiesService) {
	'use strict';
	return function(items, field, reverse) {

		var filtered = [];
		angular.forEach(items, function(item) {
			filtered.push(item);
		});

		var fieldOrder = angular.copy(vCardPropertiesService.fieldOrder);
		// reverse to move custom items to the end (indexOf == -1)
		fieldOrder.reverse();

		filtered.sort(function (a, b) {
			if(fieldOrder.indexOf(a[field]) < fieldOrder.indexOf(b[field])) {
				return 1;
			}
			if(fieldOrder.indexOf(a[field]) > fieldOrder.indexOf(b[field])) {
				return -1;
			}
			return 0;
		});

		if(reverse) filtered.reverse();
		return filtered;
	};
}]);

angular.module('contactsApp')
.filter('toArray', function() {
	return function(obj) {
		if (!(obj instanceof Object)) return obj;
		return _.map(obj, function(val, key) {
			return Object.defineProperty(val, '$key', {value: key});
		});
	};
});

angular.module('contactsApp')
.filter('vCard2JSON', function() {
	return function(input) {
		return vCard.parse(input);
	};
});

//# sourceMappingURL=data:application/json;charset=utf8;base64,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
